/**
@license
Copyright (c) 2017 Vaadin Ltd.
This program is available under Apache License Version 2.0, available at https://vaadin.com/license/
*/
import { PolymerElement } from '@polymer/polymer/polymer-element.js';

import { TextFieldMixin } from './vaadin-text-field-mixin.js';
import { ControlStateMixin } from '@vaadin/vaadin-control-state-mixin/vaadin-control-state-mixin.js';
import { ThemableMixin } from '@vaadin/vaadin-themable-mixin/vaadin-themable-mixin.js';
import { ElementMixin } from '@vaadin/vaadin-element-mixin/vaadin-element-mixin.js';
import { html } from '@polymer/polymer/lib/utils/html-tag.js';
/**
 * `<vaadin-text-area>` is a Web Component for text area control in forms.
 *
 * ```html
 * <vaadin-text-area label="Add description">
 * </vaadin-text-area>
 * ```
 *
 * ### Prefixes and suffixes
 *
 * These are child elements of a `<vaadin-text-area>` that are displayed
 * inline with the input, before or after.
 * In order for an element to be considered as a prefix, it must have the slot
 * attribute set to `prefix` (and similarly for `suffix`).
 *
 * ```html
 * <vaadin-text-area label="Add description">
 *   <div slot="prefix">Details:</div>
 *   <div slot="suffix">The end!</div>
 * </vaadin-text-area>
 * ```
 *
 * ### Styling
 *
 * The following shadow DOM parts are available for styling:
 *
 * Part name | Description
 * ----------------|----------------
 * `label` | The label element
 * `input-field` | The element that wraps prefix, value and suffix
 * `value` | The text value element inside the `input-field` element
 * `error-message` | The error message element
 *
 * The following state attributes are available for styling:
 *
 * Attribute    | Description | Part name
 * -------------|-------------|------------
 * `disabled` | Set to a disabled text field | :host
 * `has-value` | Set when the element has a value | :host
 * `has-label` | Set when the element has a label | :host
 * `has-helper` | Set when the element has helper text | :host
 * `has-error-message` | Set when the element has an error message | :host
 * `invalid` | Set when the element is invalid | :host
 * `focused` | Set when the element is focused | :host
 * `focus-ring` | Set when the element is keyboard focused | :host
 * `readonly` | Set to a readonly text field | :host
 *
 * See [ThemableMixin – how to apply styles for shadow parts](https://github.com/vaadin/vaadin-themable-mixin/wiki)
 *
 * @extends PolymerElement
 * @mixes TextFieldMixin
 * @mixes ControlStateMixin
 * @mixes ElementMixin
 * @mixes ThemableMixin
 * @demo demo/index.html
 */
class TextAreaElement extends
  ElementMixin(
    TextFieldMixin(
      ControlStateMixin(
        ThemableMixin(PolymerElement)))) {
  static get template() {
    return html`
    <style include="vaadin-text-field-shared-styles">
      .vaadin-text-area-container {
        flex: auto;
        max-height: inherit; /* MSIE 11 */
        min-height: inherit; /* MSIE 11 */
      }

      /* The label, helper text and the error message should neither grow nor shrink. */
      [part="label"],
      [part="helper-text"],
      [part="error-message"] {
        flex: none;
      }

      [part="input-field"] {
        overflow: auto;
        -webkit-overflow-scrolling: touch;
      }

      [part="value"] {
        resize: none;
      }

      [part="value"],
      [part="input-field"] ::slotted(*) {
        align-self: flex-start;
      }

      @keyframes vaadin-text-area-appear {
        to {
          opacity: 1;
        }
      }

      :host {
        animation: 1ms vaadin-text-area-appear;
      }

      /* Workaround https://bugzilla.mozilla.org/show_bug.cgi?id=1739079 */
      :host([disabled]) [part='value'] {
        user-select: none;
      }
    </style>

    <div class="vaadin-text-area-container">

      <label part="label" on-click="focus" id="[[_labelId]]">[[label]]</label>

      <div part="input-field" id="[[_inputId]]" on-scroll="__scrollPositionUpdated">

        <slot name="prefix"></slot>

        <slot name="textarea">
          <textarea part="value"></textarea>
        </slot>

        <div part="clear-button" id="clearButton" role="button" aria-label\$="[[i18n.clear]]"></div>
        <slot name="suffix"></slot>

      </div>

      <div part="helper-text" id="[[_helperTextId]]">
        <slot name="helper">[[helperText]]</slot>
      </div>

      <div part="error-message" id="[[_errorId]]" aria-live="assertive" aria-hidden\$="[[_getErrorMessageAriaHidden(invalid, errorMessage, _errorId)]]">[[errorMessage]]</div>

    </div>
`;
  }

  static get is() {
    return 'vaadin-text-area';
  }

  static get version() {
    return '2.10.0';
  }

  static get properties() {
    return {
      /**
       * A regular expression that the value is checked against.
       * The pattern must match the entire value, not just some subset.
       */
      pattern: {
        type: String
      }
    };
  }

  static get observers() {
    return [
      '_textAreaValueChanged(value)'
    ];
  }

  /** @protected */
  ready() {
    super.ready();
    this._updateHeight();
    this.addEventListener('animationend', this._onAnimationEnd);

    this._inputField = this.root.querySelector('[part=input-field]');
    // Wheel scrolling results in async scroll events. Preventing the wheel
    // event, scrolling manually and then synchronously updating the scroll position CSS variable
    // allows us to avoid some jumpy behavior that would occur on wheel otherwise.
    this._inputField.addEventListener('wheel', (e) => {
      const scrollTopBefore = this._inputField.scrollTop;
      this._inputField.scrollTop += e.deltaY;

      if (scrollTopBefore !== this._inputField.scrollTop) {
        e.preventDefault();
        this.__scrollPositionUpdated();
      }
    });

    this.__scrollPositionUpdated();
  }

  /** @private */
  __scrollPositionUpdated() {
    this.updateStyles({
      '--_text-area-vertical-scroll-position': this._inputField.scrollTop + 'px',
    });
  }

  /** @private */
  _onAnimationEnd(e) {
    if (e.animationName.indexOf('vaadin-text-area-appear') === 0) {
      this._updateHeight();
    }
  }

  /**
   * @return {string}
   * @protected
   */
  get _slottedTagName() {
    return 'textarea';
  }

  /** @private */
  _textAreaValueChanged(value) {
    this._updateHeight();
  }

  /** @private */
  _updateHeight() {
    const inputField = this.root.querySelector('[part=input-field]');
    const scrollTop = inputField.scrollTop;
    const input = this.inputElement;

    // Only clear the height when the content shortens to minimize scrollbar flickering.
    const valueLength = this.value ? this.value.length : 0;

    if (this._oldValueLength >= valueLength) {
      const inputFieldHeight = getComputedStyle(inputField).height;
      const inputWidth = getComputedStyle(input).width;

      // Temporarily fix the height of the wrapping input field container to prevent changing the browsers scroll
      // position while resetting the textareas height. If the textarea had a large height, then removing its height
      // will reset its height to the default of two rows. That might reduce the height of the page, and the
      // browser might adjust the scroll position before we can restore the measured height of the textarea.
      inputField.style.display = 'block';
      inputField.style.height = inputFieldHeight;

      // Fix the input element width so its scroll height isn't affected by host's disappearing scrollbars
      input.style.maxWidth = inputWidth;

      // Clear the height of the textarea to allow measuring a reduced scroll height
      input.style.height = 'auto';
    }
    this._oldValueLength = valueLength;

    const inputHeight = input.scrollHeight;
    if (inputHeight > input.clientHeight) {
      input.style.height = inputHeight + 'px';
    }

    // Restore
    input.style.removeProperty('max-width');
    inputField.style.removeProperty('display');
    inputField.style.removeProperty('height');
    inputField.scrollTop = scrollTop;

    this._dispatchIronResizeEventIfNeeded('InputHeight', inputHeight);
  }

  /**
   * Returns true if the current textarea value satisfies all constraints (if any).
   * @return {boolean}
   */
  checkValidity() {
    if (!super.checkValidity()) {
      return false;
    }

    // Native <textarea> does not support pattern attribute, so we have a custom logic
    // according to WHATWG spec for <input>, with tests inspired by web-platform-tests
    // https://html.spec.whatwg.org/multipage/input.html#the-pattern-attribute

    if (!this.pattern || !this._inputElementValue) {
      // Mark as valid if there is no pattern, or the value is empty
      return true;
    }

    try {
      const match = this._inputElementValue.match(this.pattern);
      return match ? match[0] === match.input : false;
    } catch (_) {
      // If the pattern can not be compiled, then report as valid
      return true;
    }
  }

  /**
   * Fired when the text-area height changes.
   *
   * @event iron-resize
   */
}

customElements.define(TextAreaElement.is, TextAreaElement);

export { TextAreaElement };
