import { createVNode as _createVNode } from "vue";
import { computed } from 'vue'; // Utils

import { addUnit, createNamespace, preventDefault } from '../utils'; // Composition

import { useRefs } from '../composables/use-refs';
import { useTouch } from '../composables/use-touch';
import { useLinkField } from '../composables/use-link-field'; // Components

import Icon from '../icon';

var _createNamespace = createNamespace('rate'),
    createComponent = _createNamespace[0],
    bem = _createNamespace[1];

function getRateStatus(value, index, allowHalf) {
  if (value >= index) {
    return 'full';
  }

  if (value + 0.5 >= index && allowHalf) {
    return 'half';
  }

  return 'void';
}

export default createComponent({
  props: {
    size: [Number, String],
    color: String,
    gutter: [Number, String],
    readonly: Boolean,
    disabled: Boolean,
    allowHalf: Boolean,
    voidColor: String,
    iconPrefix: String,
    disabledColor: String,
    modelValue: {
      type: Number,
      default: 0
    },
    icon: {
      type: String,
      default: 'star'
    },
    voidIcon: {
      type: String,
      default: 'star-o'
    },
    count: {
      type: [Number, String],
      default: 5
    },
    touchable: {
      type: Boolean,
      default: true
    }
  },
  emits: ['change', 'update:modelValue'],
  setup: function setup(props, _ref) {
    var emit = _ref.emit;
    var ranges;
    var touch = useTouch();

    var _useRefs = useRefs(),
        itemRefs = _useRefs[0],
        setItemRefs = _useRefs[1];

    var untouchable = function untouchable() {
      return props.readonly || props.disabled || !props.touchable;
    };

    var list = computed(function () {
      var list = [];

      for (var i = 1; i <= props.count; i++) {
        list.push(getRateStatus(props.modelValue, i, props.allowHalf));
      }

      return list;
    });

    var select = function select(index) {
      if (!props.disabled && !props.readonly && index !== props.modelValue) {
        emit('update:modelValue', index);
        emit('change', index);
      }
    };

    var getScoreByPosition = function getScoreByPosition(x) {
      for (var i = ranges.length - 1; i > 0; i--) {
        if (x > ranges[i].left) {
          return ranges[i].score;
        }
      }

      return props.allowHalf ? 0.5 : 1;
    };

    var onTouchStart = function onTouchStart(event) {
      if (untouchable()) {
        return;
      }

      touch.start(event);
      var rects = itemRefs.value.map(function (item) {
        return item.getBoundingClientRect();
      });
      ranges = [];
      rects.forEach(function (rect, index) {
        if (props.allowHalf) {
          ranges.push({
            score: index + 0.5,
            left: rect.left
          }, {
            score: index + 1,
            left: rect.left + rect.width / 2
          });
        } else {
          ranges.push({
            score: index + 1,
            left: rect.left
          });
        }
      });
    };

    var onTouchMove = function onTouchMove(event) {
      if (untouchable()) {
        return;
      }

      touch.move(event);

      if (touch.isHorizontal()) {
        var clientX = event.touches[0].clientX;
        preventDefault(event);
        select(getScoreByPosition(clientX));
      }
    };

    var renderStar = function renderStar(status, index) {
      var icon = props.icon,
          size = props.size,
          color = props.color,
          count = props.count,
          gutter = props.gutter,
          voidIcon = props.voidIcon,
          disabled = props.disabled,
          voidColor = props.voidColor,
          allowHalf = props.allowHalf,
          iconPrefix = props.iconPrefix,
          disabledColor = props.disabledColor;
      var score = index + 1;
      var isFull = status === 'full';
      var isVoid = status === 'void';
      var style;

      if (gutter && score !== +count) {
        style = {
          paddingRight: addUnit(gutter)
        };
      }

      return _createVNode("div", {
        "key": index,
        "ref": setItemRefs(index),
        "role": "radio",
        "style": style,
        "class": bem('item'),
        "tabindex": "0",
        "aria-setsize": count,
        "aria-posinset": score,
        "aria-checked": String(!isVoid)
      }, [_createVNode(Icon, {
        "size": size,
        "name": isFull ? icon : voidIcon,
        "class": bem('icon', {
          disabled: disabled,
          full: isFull
        }),
        "color": disabled ? disabledColor : isFull ? color : voidColor,
        "classPrefix": iconPrefix,
        "data-score": score,
        "onClick": function onClick() {
          select(score);
        }
      }, null), allowHalf && _createVNode(Icon, {
        "size": size,
        "name": isVoid ? voidIcon : icon,
        "class": bem('icon', ['half', {
          disabled: disabled,
          full: !isVoid
        }]),
        "color": disabled ? disabledColor : isVoid ? voidColor : color,
        "classPrefix": iconPrefix,
        "data-score": score - 0.5,
        "onClick": function onClick() {
          select(score - 0.5);
        }
      }, null)]);
    };

    useLinkField(function () {
      return props.modelValue;
    });
    return function () {
      return _createVNode("div", {
        "role": "radiogroup",
        "class": bem({
          readonly: props.readonly,
          disabled: props.disabled
        }),
        "tabindex": "0",
        "onTouchstart": onTouchStart,
        "onTouchmove": onTouchMove
      }, [list.value.map(renderStar)]);
    };
  }
});