"use strict";
var chai_1 = require('chai');
var util_1 = require('../util');
var axis = require('../../src/compile/axis');
var channel_1 = require('../../src/channel');
describe('Axis', function () {
    describe('= true', function () {
        it('should produce default properties for axis', function () {
            var model1 = util_1.parseUnitModel({
                "mark": "bar",
                "encoding": {
                    "y": { "type": "quantitative", "field": 'US_Gross', "aggregate": "sum", "axis": true }
                },
                "data": { "url": "data/movies.json" }
            });
            var model2 = util_1.parseUnitModel({
                "mark": "bar",
                "encoding": {
                    "y": { "type": "quantitative", "field": 'US_Gross', "aggregate": "sum" }
                },
                "data": { "url": "data/movies.json" }
            });
            chai_1.assert.deepEqual(model1.axis(channel_1.Y), model2.axis(channel_1.Y));
        });
    });
    describe('parseInnerAxis', function () {
        it('should produce a Vega inner axis object with correct type, scale and grid properties', function () {
            var model = util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a",
                        type: "quantitative",
                        axis: { grid: true, gridColor: "blue", gridWidth: 20 }
                    }
                }
            });
            var def = axis.parseInnerAxis(channel_1.X, model);
            chai_1.assert.isObject(def);
            chai_1.assert.equal(def.type, 'x');
            chai_1.assert.equal(def.scale, 'x');
            chai_1.assert.deepEqual(def.properties.grid, { stroke: { value: "blue" }, strokeWidth: { value: 20 } });
        });
    });
    describe('parseAxis', function () {
        it('should produce a Vega axis object with correct type and scale', function () {
            var model = util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "quantitative" }
                }
            });
            var def = axis.parseAxis(channel_1.X, model);
            chai_1.assert.isObject(def);
            chai_1.assert.equal(def.type, 'x');
            chai_1.assert.equal(def.scale, 'x');
        });
    });
    describe('grid()', function () {
        it('should return specified orient', function () {
            var grid = axis.grid(util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: 'a', axis: { grid: false } }
                }
            }), channel_1.X);
            chai_1.assert.deepEqual(grid, false);
        });
        it('should return true by default', function () {
            var grid = axis.grid(util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: 'a' }
                }
            }), channel_1.X);
            chai_1.assert.deepEqual(grid, true);
        });
        it('should return undefined for COLUMN', function () {
            var grid = axis.grid(util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: 'a' }
                }
            }), channel_1.COLUMN);
            chai_1.assert.deepEqual(grid, undefined);
        });
        it('should return undefined for ROW', function () {
            var grid = axis.grid(util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: 'a' }
                }
            }), channel_1.ROW);
            chai_1.assert.deepEqual(grid, undefined);
        });
    });
    describe('layer()', function () {
        it('should return undefined by default without grid defined', function () {
            var layer = axis.layer(util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: 'a' }
                }
            }), channel_1.X, channel_1.Y);
            chai_1.assert.deepEqual(layer, undefined);
        });
        it('should return back by default with grid defined', function () {
            var layer = axis.layer(util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: 'a' }
                }
            }), channel_1.X, { grid: true });
            chai_1.assert.deepEqual(layer, "back");
        });
        it('should return specified layer', function () {
            var layer = axis.layer(util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: 'a', axis: { layer: "front" } }
                }
            }), channel_1.X, { grid: true });
            chai_1.assert.deepEqual(layer, "front");
        });
    });
    describe('orient()', function () {
        it('should return specified orient', function () {
            var orient = axis.orient(util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: 'a', axis: { orient: 'bottom' } }
                }
            }), channel_1.X);
            chai_1.assert.deepEqual(orient, 'bottom');
        });
        it('should return undefined by default', function () {
            var orient = axis.orient(util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: 'a' }
                }
            }), channel_1.X);
            chai_1.assert.deepEqual(orient, undefined);
        });
        it('should return top for COL', function () {
            var orient = axis.orient(util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: 'a' },
                    column: { field: 'a' }
                }
            }), channel_1.COLUMN);
            chai_1.assert.deepEqual(orient, 'top');
        });
    });
    describe('ticks', function () {
        it('should return undefined by default', function () {
            var ticks = axis.ticks(util_1.parseModel({
                mark: "point",
                encoding: {
                    y: { field: 'a' }
                }
            }), channel_1.Y);
            chai_1.assert.deepEqual(ticks, undefined);
        });
        it('should return 5 by default', function () {
            var ticks = axis.ticks(util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: 'a' }
                }
            }), channel_1.X);
            chai_1.assert.deepEqual(ticks, 5);
        });
        it('should return specified ticks', function () {
            var ticks = axis.ticks(util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: 'a', axis: { ticks: 10 } }
                }
            }), channel_1.X);
            chai_1.assert.deepEqual(ticks, 10);
        });
    });
    describe('tickSize', function () {
        it('should return undefined by default', function () {
            var tickSize = axis.tickSize(util_1.parseModel({
                mark: "point",
                encoding: {
                    y: { field: 'a' }
                }
            }), channel_1.Y);
            chai_1.assert.deepEqual(tickSize, undefined);
        });
        it('should return specified ticks', function () {
            var tickSize = axis.tickSize(util_1.parseModel({
                mark: "point",
                encoding: {
                    y: { field: 'a', axis: { tickSize: 10 } }
                }
            }), channel_1.Y);
            chai_1.assert.deepEqual(tickSize, 10);
        });
    });
    describe('tickSizeEnd', function () {
        it('should return undefined by default', function () {
            var tickSizeEnd = axis.tickSizeEnd(util_1.parseModel({
                mark: "point",
                encoding: {
                    y: { field: 'a' }
                }
            }), channel_1.Y);
            chai_1.assert.deepEqual(tickSizeEnd, undefined);
        });
        it('should return specified ticks', function () {
            var tickSizeEnd = axis.tickSizeEnd(util_1.parseModel({
                mark: "point",
                encoding: {
                    y: { field: 'a', axis: { tickSizeEnd: 5 } }
                }
            }), channel_1.Y);
            chai_1.assert.deepEqual(tickSizeEnd, 5);
        });
    });
    describe('title()', function () {
        it('should add explicitly specified title', function () {
            var title = axis.title(util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: 'a', axis: { title: 'Custom' } }
                }
            }), channel_1.X);
            chai_1.assert.deepEqual(title, 'Custom');
        });
        it('should add return fieldTitle by default', function () {
            var title = axis.title(util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: 'a', type: "quantitative", axis: { titleMaxLength: 3 } }
                }
            }), channel_1.X);
            chai_1.assert.deepEqual(title, 'a');
        });
        it('should add return fieldTitle by default', function () {
            var title = axis.title(util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: 'a', type: "quantitative", aggregate: 'sum', axis: { titleMaxLength: 10 } }
                }
            }), channel_1.X);
            chai_1.assert.deepEqual(title, 'SUM(a)');
        });
        it('should add return fieldTitle by default and truncate', function () {
            var title = axis.title(util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: 'a', type: "quantitative", aggregate: 'sum', axis: { titleMaxLength: 3 } }
                }
            }), channel_1.X);
            chai_1.assert.deepEqual(title, 'SU…');
        });
        it('should add return fieldTitle by default and truncate', function () {
            var title = axis.title(util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: 'abcdefghijkl' }
                },
                config: {
                    cell: { width: 60 }
                }
            }), channel_1.X);
            chai_1.assert.deepEqual(title, 'abcdefghi…');
        });
        it('should add return fieldTitle by default and truncate', function () {
            var title = axis.title(util_1.parseUnitModel({
                height: 60,
                mark: "point",
                encoding: {
                    y: { field: 'abcdefghijkl' }
                }
            }), channel_1.Y);
            chai_1.assert.deepEqual(title, 'abcdefghi…');
        });
    });
    describe('titleOffset', function () {
        it('should return undefined by default', function () {
            var titleOffset = axis.titleOffset(util_1.parseModel({
                mark: "point",
                encoding: {
                    y: { field: 'a' }
                }
            }), channel_1.Y);
            chai_1.assert.deepEqual(titleOffset, undefined);
        });
        it('should return specified ticks', function () {
            var titleOffset = axis.titleOffset(util_1.parseModel({
                mark: "point",
                encoding: {
                    y: { field: 'a', axis: { tickSize: 10, titleOffset: 15 } }
                }
            }), channel_1.Y);
            chai_1.assert.deepEqual(titleOffset, 15);
        });
    });
    describe('values', function () {
        it('should return correct timestamp values for DateTimes', function () {
            var values = axis.values(util_1.parseModel({
                mark: "point",
                encoding: {
                    y: { field: 'a', type: 'temporal', axis: { values: [{ year: 1970 }, { year: 1980 }] } }
                }
            }), channel_1.Y);
            chai_1.assert.deepEqual(values, [
                new Date(1970, 0, 1).getTime(),
                new Date(1980, 0, 1).getTime()
            ]);
        });
        it('should simply return values for non-DateTime', function () {
            var values = axis.values(util_1.parseModel({
                mark: "point",
                encoding: {
                    y: { field: 'a', type: 'quantitative', axis: { values: [1, 2, 3, 4] } }
                }
            }), channel_1.Y);
            chai_1.assert.deepEqual(values, [1, 2, 3, 4]);
        });
    });
    describe('properties.axis()', function () {
        it('axisColor should change axis\'s color', function () {
            var model = util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "quantitative", axis: { axisColor: '#fff' } }
                }
            });
            var axes = axis.properties.axis(model, channel_1.X, {});
            chai_1.assert.equal(axes.stroke.value, '#fff');
        });
        it('axisWidth should change axis\'s width', function () {
            var model = util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "quantitative", axis: { axisWidth: 2 } }
                }
            });
            var axes = axis.properties.axis(model, channel_1.X, {});
            chai_1.assert.equal(axes.strokeWidth.value, 2);
        });
    });
    describe('properties.grid()', function () {
        it('gridColor should change grid\'s color', function () {
            var model = util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "quantitative", axis: { gridColor: '#fff' } }
                }
            });
            var axes = axis.properties.grid(model, channel_1.X, {});
            chai_1.assert.equal(axes.stroke.value, '#fff');
        });
        it('gridOpacity should change grid\'s opacity', function () {
            var model = util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "quantitative", axis: { grid: true, gridOpacity: 0.6 } }
                }
            });
            var axes = axis.properties.grid(model, channel_1.X, {});
            chai_1.assert.equal(axes.strokeOpacity.value, 0.6);
        });
        it('gridWidth should change grid\'s width', function () {
            var model = util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "quantitative", axis: { grid: true, gridWidth: 2 } }
                }
            });
            var axes = axis.properties.grid(model, channel_1.X, {});
            chai_1.assert.equal(axes.strokeWidth.value, 2);
        });
        it('gridDash should change grid\'s dash offset', function () {
            var model = util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "quantitative", axis: { grid: true, gridDash: [2] } }
                }
            });
            var axes = axis.properties.grid(model, channel_1.X, {});
            chai_1.assert.deepEqual(axes.strokeDashOffset.value, [2]);
        });
    });
    describe('properties.labels()', function () {
        it('should show labels by default', function () {
            var labels = axis.properties.labels(util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: 'a', type: "ordinal" }
                }
            }), channel_1.X, {}, { orient: 'top' });
            chai_1.assert.deepEqual(labels.text.template, '{{ datum["data"] | truncate:25 }}');
        });
        it('should hide labels if labels are set to false', function () {
            var labels = axis.properties.labels(util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: 'a', type: "ordinal", axis: { labels: false } }
                }
            }), channel_1.X, {}, null);
            chai_1.assert.deepEqual(labels.text, '');
        });
        it('should rotate labels if labelAngle is defined', function () {
            var model = util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "quantitative", axis: { labelAngle: -45 } }
                }
            });
            var labels = axis.properties.labels(model, channel_1.X, {}, {});
            chai_1.assert.equal(labels.angle.value, -45);
        });
        it('should rotate label', function () {
            var model = util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "temporal", timeUnit: "month" }
                }
            });
            var labels = axis.properties.labels(model, channel_1.X, {}, { type: 'x' });
            chai_1.assert.equal(labels.angle.value, 270);
            chai_1.assert.equal(labels.baseline.value, 'middle');
        });
        it('should have correct text.template for quarter timeUnits', function () {
            var model = util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "temporal", timeUnit: "quarter" }
                }
            });
            var labels = axis.properties.labels(model, channel_1.X, {}, { type: 'x' });
            var quarterPrefix = 'Q';
            var expected = quarterPrefix + '{{datum["data"] | quarter}}';
            chai_1.assert.deepEqual(labels.text.template, expected);
        });
        it('should have correct text.template for yearquartermonth timeUnits', function () {
            var model = util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "temporal", timeUnit: "yearquartermonth" }
                }
            });
            var labels = axis.properties.labels(model, channel_1.X, {}, { type: 'x' });
            var quarterPrefix = 'Q';
            var expected = quarterPrefix + '{{datum["data"] | quarter}} {{datum["data"] | time:\'%b %Y\'}}';
            chai_1.assert.deepEqual(labels.text.template, expected);
        });
        it('tickLabelColor should change with axis\'s label\' color', function () {
            var model = util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: "a", axis: { tickLabelColor: "blue" } }
                }
            });
            var labels = axis.properties.labels(model, channel_1.X, {}, {});
            chai_1.assert.equal(labels.fill.value, "blue");
        });
        it('tickLabelFont should change with axis\'s label\'s font', function () {
            var model = util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: "a", axis: { tickLabelFont: "Helvetica Neue" } }
                }
            });
            var labels = axis.properties.labels(model, channel_1.X, {}, {});
            chai_1.assert.equal(labels.font.value, "Helvetica Neue");
        });
        it('tickLabelFontSize should change with axis\'s label\'s font size', function () {
            var model = util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: "a", axis: { tickLabelFontSize: 20 } }
                }
            });
            var labels = axis.properties.labels(model, channel_1.X, {}, {});
            chai_1.assert.equal(labels.fontSize.value, 20);
        });
    });
    describe('properties.ticks()', function () {
        it('tickColor should change axis\'s ticks\'s color', function () {
            var model = util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "quantitative", axis: { tickColor: '#123' } }
                }
            });
            var axes = axis.properties.ticks(model, channel_1.X, {});
            chai_1.assert.equal(axes.stroke.value, '#123');
        });
        it('tickWidth should change axis\'s ticks\'s color', function () {
            var model = util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "quantitative", axis: { tickWidth: 13 } }
                }
            });
            var axes = axis.properties.ticks(model, channel_1.X, {});
            chai_1.assert.equal(axes.strokeWidth.value, 13);
        });
    });
    describe('properties.title()', function () {
        it('titleColor should change axis\'s title\'s color', function () {
            var model = util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "quantitative", axis: { titleColor: '#abc' } }
                }
            });
            var axes = axis.properties.title(model, channel_1.X, {});
            chai_1.assert.equal(axes.fill.value, '#abc');
        });
        it('titleFont should change axis\'s title\'s font', function () {
            var model = util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "quantitative", axis: { titleFont: 'anything' } }
                }
            });
            var axes = axis.properties.title(model, channel_1.X, {});
            chai_1.assert.equal(axes.font.value, 'anything');
        });
        it('titleFontSize should change axis\'s title\'s font size', function () {
            var model = util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "quantitative", axis: { titleFontSize: 56 } }
                }
            });
            var axes = axis.properties.title(model, channel_1.X, {});
            chai_1.assert.equal(axes.fontSize.value, 56);
        });
        it('titleFontWeight should change axis\'s title\'s font weight', function () {
            var model = util_1.parseModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "quantitative", axis: { titleFontWeight: 'bold' } }
                }
            });
            var axes = axis.properties.title(model, channel_1.X, {});
            chai_1.assert.equal(axes.fontWeight.value, 'bold');
        });
    });
});
//# sourceMappingURL=axis.test.js.map