"use strict";
var chai_1 = require('chai');
var util_1 = require('../util');
var channel_1 = require('../../src/channel');
var config_1 = require('../../src/config');
var timeunit_1 = require('../../src/timeunit');
var fielddef_1 = require('../../src/fielddef');
var type_1 = require('../../src/type');
var common_1 = require('../../src/compile/common');
describe('Common', function () {
    describe('timeFormat()', function () {
        it('should get the right time template for month with shortTimeLabels=true', function () {
            var fieldDef = { timeUnit: timeunit_1.TimeUnit.MONTH, field: 'a', type: type_1.TEMPORAL };
            var template = common_1.timeTemplate(fielddef_1.field(fieldDef, { datum: true }), timeunit_1.TimeUnit.MONTH, undefined, true, config_1.defaultConfig);
            chai_1.assert.equal(template, '{{datum["month_a"] | time:\'%b\'}}');
        });
        it('should get the right time template for month with shortTimeLabels=false', function () {
            var fieldDef = { timeUnit: timeunit_1.TimeUnit.MONTH, field: 'a', type: type_1.TEMPORAL };
            var template = common_1.timeTemplate(fielddef_1.field(fieldDef, { datum: true }), timeunit_1.TimeUnit.MONTH, undefined, false, config_1.defaultConfig);
            chai_1.assert.equal(template, '{{datum["month_a"] | time:\'%B\'}}');
        });
        it('should get the right time template for yearmonth with custom format', function () {
            var fieldDef = { timeUnit: timeunit_1.TimeUnit.YEARMONTH, field: 'a', type: type_1.TEMPORAL };
            var template = common_1.timeTemplate(fielddef_1.field(fieldDef, { datum: true }), timeunit_1.TimeUnit.MONTH, '%Y', true, config_1.defaultConfig);
            chai_1.assert.equal(template, '{{datum["yearmonth_a"] | time:\'%Y\'}}');
        });
        it('should get the right time template for quarter', function () {
            var fieldDef = { timeUnit: timeunit_1.TimeUnit.QUARTER, field: 'a', type: type_1.TEMPORAL };
            var template = common_1.timeTemplate(fielddef_1.field(fieldDef, { datum: true }), timeunit_1.TimeUnit.QUARTER, undefined, true, config_1.defaultConfig);
            chai_1.assert.equal(template, 'Q{{datum["quarter_a"] | quarter}}');
        });
        it('should get the right time template for yearquarter', function () {
            var template = common_1.timeTemplate('datum["data"]', timeunit_1.TimeUnit.YEARQUARTER, undefined, true, config_1.defaultConfig);
            chai_1.assert.equal(template, 'Q{{datum["data"] | quarter}} {{datum["data"] | time:\'%y\'}}');
        });
    });
    describe('numberFormat()', function () {
        it('should use number format for quantitative scale', function () {
            chai_1.assert.equal(common_1.numberFormat({ field: 'a', type: type_1.QUANTITATIVE }, undefined, { numberFormat: 'd' }, channel_1.X), 'd');
        });
        it('should support empty number format', function () {
            chai_1.assert.equal(common_1.numberFormat({ field: 'a', type: type_1.QUANTITATIVE }, undefined, { numberFormat: '' }, channel_1.X), '');
        });
        it('should use format if provided', function () {
            chai_1.assert.equal(common_1.numberFormat({ field: 'a', type: type_1.QUANTITATIVE }, 'a', 'd', channel_1.X), 'a');
        });
        it('should not use number format for binned quantitative scale', function () {
            chai_1.assert.equal(common_1.numberFormat({ bin: true, field: 'a', type: type_1.QUANTITATIVE }, undefined, 'd', channel_1.X), undefined);
        });
        it('should not use number format for non-quantitative scale', function () {
            for (var _i = 0, _a = [type_1.TEMPORAL, type_1.NOMINAL, type_1.ORDINAL]; _i < _a.length; _i++) {
                var type = _a[_i];
                chai_1.assert.equal(common_1.numberFormat({ bin: true, field: 'a', type: type }, undefined, 'd', channel_1.X), undefined);
            }
        });
    });
    describe('applyColorAndOpacity()', function () {
        it('opacity should be mapped to a field if specified', function () {
            var model = util_1.parseUnitModel({
                "mark": "bar",
                "encoding": {
                    "y": { "type": "quantitative", "field": 'US_Gross', "aggregate": "sum", "axis": true },
                    "opacity": { "field": "US_Gross", "type": "quantitative" }
                },
                "data": { "url": "data/movies.json" }
            });
            var p = {};
            common_1.applyColorAndOpacity(p, model);
            chai_1.assert.deepEqual(p.opacity.field, 'US_Gross');
        });
        it('opacity should be mapped to a value if specified', function () {
            var model = util_1.parseUnitModel({
                "mark": "bar",
                "encoding": {
                    "y": { "type": "quantitative", "field": 'US_Gross', "aggregate": "sum", "axis": true },
                    "opacity": { "value": 0.5 }
                },
                "data": { "url": "data/movies.json" }
            });
            var p = {};
            common_1.applyColorAndOpacity(p, model);
            chai_1.assert.deepEqual(p.opacity.value, 0.5);
        });
    });
});
//# sourceMappingURL=common.test.js.map