"use strict";
var chai_1 = require('chai');
var data_1 = require('../../src/compile/data/data');
var bin_1 = require('../../src/compile/data/bin');
var filter_1 = require('../../src/compile/data/filter');
var nullfilter_1 = require('../../src/compile/data/nullfilter');
var source_1 = require('../../src/compile/data/source');
var stackscale_1 = require('../../src/compile/data/stackscale');
var summary_1 = require('../../src/compile/data/summary');
var timeunit_1 = require('../../src/compile/data/timeunit');
var timeunitdomain_1 = require('../../src/compile/data/timeunitdomain');
var formatparse_1 = require('../../src/compile/data/formatparse');
var nonpositivenullfilter_1 = require('../../src/compile/data/nonpositivenullfilter');
var util_1 = require('../util');
var util_2 = require('../../src/util');
function compileAssembleData(model) {
    model.parseData();
    return data_1.assembleData(model, []);
}
describe('data', function () {
    describe('compileData & assembleData', function () {
        describe('for aggregate encoding', function () {
            it('should contain 2 tables', function () {
                var model = util_1.parseUnitModel({
                    mark: "point",
                    encoding: {
                        x: { field: 'a', type: "temporal" },
                        y: { field: 'b', type: "quantitative", scale: { type: 'log' }, aggregate: 'sum' }
                    }
                });
                var data = compileAssembleData(model);
                chai_1.assert.equal(data.length, 2);
            });
        });
        describe('when contains log in non-aggregate', function () {
            var model = util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: 'a', type: "temporal" },
                    y: { field: 'b', type: "quantitative", scale: { type: 'log' } }
                }
            });
            var data = compileAssembleData(model);
            it('should contains 1 table', function () {
                chai_1.assert.equal(data.length, 1);
            });
            it('should have filter non-positive in source', function () {
                var sourceTransform = data[0].transform;
                chai_1.assert.deepEqual(sourceTransform[sourceTransform.length - 1], {
                    type: 'filter',
                    test: 'datum["b"] > 0'
                });
            });
        });
    });
    describe('assemble', function () {
        it('should have correct order of transforms (null filter, timeUnit, bin then filter)', function () {
            var model = util_1.parseUnitModel({
                transform: {
                    calculate: [{
                            field: 'b2',
                            expr: '2 * datum["b"]'
                        }],
                    filter: 'datum["a"] > datum["b"] && datum["c"] === datum["d"]'
                },
                mark: "point",
                encoding: {
                    x: { field: 'a', type: "temporal", timeUnit: 'year' },
                    y: {
                        bin: {
                            min: 0,
                            max: 100
                        },
                        'field': 'Acceleration',
                        'type': "quantitative"
                    },
                    size: { field: 'b2', type: 'quantitative' }
                }
            });
            var transform = compileAssembleData(model)[0].transform;
            chai_1.assert.deepEqual(transform[0].type, 'formula');
            chai_1.assert.deepEqual(transform[1].type, 'filter');
            chai_1.assert.deepEqual(transform[2].type, 'filter');
            chai_1.assert.deepEqual(transform[3].type, 'bin');
            chai_1.assert.deepEqual(transform[4].type, 'formula');
        });
    });
});
describe('data: source', function () {
    describe('compileUnit', function () {
        describe('with explicit values', function () {
            var model = util_1.parseUnitModel({
                data: {
                    values: [{ a: 1, b: 2, c: 3 }, { a: 4, b: 5, c: 6 }]
                }
            });
            var sourceComponent = source_1.source.parseUnit(model);
            it('should have values', function () {
                chai_1.assert.equal(sourceComponent.name, 'source');
                chai_1.assert.deepEqual(sourceComponent.values, [{ a: 1, b: 2, c: 3 }, { a: 4, b: 5, c: 6 }]);
            });
            it('should have source.format.type', function () {
                chai_1.assert.deepEqual(sourceComponent.format.type, 'json');
            });
        });
        describe('with link to url', function () {
            var model = util_1.parseUnitModel({
                data: {
                    url: 'http://foo.bar',
                }
            });
            var sourceComponent = source_1.source.parseUnit(model);
            it('should have format.type json', function () {
                chai_1.assert.equal(sourceComponent.name, 'source');
                chai_1.assert.equal(sourceComponent.format.type, 'json');
            });
            it('should have correct url', function () {
                chai_1.assert.equal(sourceComponent.url, 'http://foo.bar');
            });
        });
        describe('with no data specified', function () {
            var model = util_1.parseUnitModel({});
            var sourceComponent = source_1.source.parseUnit(model);
            it('should provide placeholder source data', function () {
                chai_1.assert.deepEqual(sourceComponent, { name: 'source' });
            });
        });
        describe('data format', function () {
            describe('json', function () {
                it('should include property if specified', function () {
                    var model = util_1.parseUnitModel({
                        data: {
                            url: 'http://foo.bar',
                            format: { type: 'json', property: 'baz' }
                        }
                    });
                    var sourceComponent = source_1.source.parseUnit(model);
                    chai_1.assert.equal(sourceComponent.format.property, 'baz');
                });
            });
            describe('topojson', function () {
                describe('feature property is specified', function () {
                    var model = util_1.parseUnitModel({
                        data: {
                            url: 'http://foo.bar',
                            format: { type: 'topojson', feature: 'baz' }
                        }
                    });
                    var sourceComponent = source_1.source.parseUnit(model);
                    it('should have format.type topojson', function () {
                        chai_1.assert.equal(sourceComponent.name, 'source');
                        chai_1.assert.equal(sourceComponent.format.type, 'topojson');
                    });
                    it('should have format.feature baz', function () {
                        chai_1.assert.equal(sourceComponent.format.feature, 'baz');
                    });
                });
                describe('mesh property is specified', function () {
                    var model = util_1.parseUnitModel({
                        data: {
                            url: 'http://foo.bar',
                            format: { type: 'topojson', mesh: 'baz' }
                        }
                    });
                    var sourceComponent = source_1.source.parseUnit(model);
                    it('should have format.type topojson', function () {
                        chai_1.assert.equal(sourceComponent.name, 'source');
                        chai_1.assert.equal(sourceComponent.format.type, 'topojson');
                    });
                    it('should have format.mesh baz', function () {
                        chai_1.assert.equal(sourceComponent.format.mesh, 'baz');
                    });
                });
            });
        });
    });
});
describe('data: formatParse', function () {
    describe('compileUnit', function () {
        it('should include parse for all applicable fields, and exclude calculated fields', function () {
            var model = util_1.parseUnitModel({
                transform: {
                    calculate: [
                        { field: 'b2', expr: 'datum["b"] * 2' }
                    ]
                },
                mark: "point",
                encoding: {
                    x: { field: 'a', type: "temporal" },
                    y: { field: 'b', type: "quantitative" },
                    color: { field: '*', type: "quantitative", aggregate: 'count' },
                    size: { field: 'b2', type: "quantitative" },
                }
            });
            var formatParseComponent = formatparse_1.formatParse.parseUnit(model);
            chai_1.assert.deepEqual(formatParseComponent, {
                'a': 'date',
                'b': 'number'
            });
        });
    });
    describe('assemble', function () {
    });
});
describe('data: bin', function () {
    describe('compileUnit', function () {
        var model = util_1.parseUnitModel({
            mark: "point",
            encoding: {
                y: {
                    bin: { min: 0, max: 100 },
                    'field': 'Acceleration',
                    'type': "quantitative"
                }
            }
        });
        it('should add bin transform and correctly apply bin', function () {
            var transform = util_2.vals(bin_1.bin.parseUnit(model))[0];
            chai_1.assert.deepEqual(transform[0], {
                type: 'bin',
                field: 'Acceleration',
                output: {
                    start: 'bin_Acceleration_start',
                    mid: 'bin_Acceleration_mid',
                    end: 'bin_Acceleration_end'
                },
                maxbins: 10,
                min: 0,
                max: 100
            });
        });
    });
    describe('assemble', function () {
    });
});
describe('data: nullFilter', function () {
    describe('compileUnit', function () {
        var spec = {
            mark: "point",
            encoding: {
                y: { field: 'qq', type: "quantitative" },
                x: { field: 'tt', type: "temporal" },
                color: { field: 'oo', type: "ordinal" }
            }
        };
        it('should add filterNull for Q and T by default', function () {
            var model = util_1.parseUnitModel(spec);
            chai_1.assert.deepEqual(nullfilter_1.nullFilter.parseUnit(model), {
                qq: { field: 'qq', type: "quantitative" },
                tt: { field: 'tt', type: "temporal" },
                oo: null
            });
        });
        it('should add filterNull for O when specified', function () {
            var model = util_1.parseUnitModel(util_2.mergeDeep(spec, {
                transform: {
                    filterNull: true
                }
            }));
            chai_1.assert.deepEqual(nullfilter_1.nullFilter.parseUnit(model), {
                qq: { field: 'qq', type: "quantitative" },
                tt: { field: 'tt', type: "temporal" },
                oo: { field: 'oo', type: "ordinal" }
            });
        });
        it('should add no null filter if filterInvalid is false', function () {
            var model = util_1.parseUnitModel(util_2.mergeDeep(spec, {
                transform: {
                    filterInvalid: false
                }
            }));
            chai_1.assert.deepEqual(nullfilter_1.nullFilter.parseUnit(model), {
                qq: null,
                tt: null,
                oo: null
            });
        });
        it('should add no null filter if filterNull is false', function () {
            var model = util_1.parseUnitModel(util_2.mergeDeep(spec, {
                transform: {
                    filterNull: false
                }
            }));
            chai_1.assert.deepEqual(nullfilter_1.nullFilter.parseUnit(model), {
                qq: null,
                tt: null,
                oo: null
            });
        });
        it('should add no null filter for count field', function () {
            var model = util_1.parseUnitModel({
                transform: {
                    filterNull: true
                },
                mark: "point",
                encoding: {
                    y: { aggregate: 'count', field: '*', type: "quantitative" }
                }
            });
            chai_1.assert.deepEqual(nullfilter_1.nullFilter.parseUnit(model), {});
        });
        describe('assemble', function () {
        });
    });
    describe('compileFacet', function () {
        it('should produce child\'s filter if child has no source and the facet has no filter', function () {
        });
        it('should produce child\'s filter and its own filter if child has no source and the facet has filter', function () {
        });
    });
    describe('assemble', function () {
    });
});
describe('data: filter', function () {
    describe('compileUnit', function () {
        var model = util_1.parseUnitModel({
            transform: {
                filter: 'datum["a"] > datum["b"] && datum["c"] === datum["d"]'
            }
        });
        it('should return array that contains a filter transform', function () {
            chai_1.assert.deepEqual(filter_1.filter.parseUnit(model), 'datum["a"] > datum["b"] && datum["c"] === datum["d"]');
        });
    });
    describe('assemble', function () {
    });
});
describe('data: formula', function () {
    describe('unit', function () {
    });
    describe('facet', function () {
    });
});
describe('data: timeUnit', function () {
    describe('compileUnit', function () {
        var model = util_1.parseUnitModel({
            mark: "point",
            encoding: {
                x: { field: 'a', type: "temporal", timeUnit: 'year' }
            }
        });
        it('should add formula transform', function () {
            var transform = util_2.vals(timeunit_1.timeUnit.parseUnit(model));
            chai_1.assert.deepEqual(transform[0], {
                type: 'formula',
                field: 'year_a',
                expr: 'datetime(year(datum["a"]), 0, 1, 0, 0, 0, 0)'
            });
        });
    });
    describe('compileFacet', function () {
    });
    describe('assemble', function () {
    });
});
describe('data: timeUnitDomain', function () {
    describe('unit: day', function () {
        var model = util_1.parseUnitModel({
            mark: "point",
            encoding: {
                'y': {
                    'aggregate': 'sum',
                    'field': 'Acceleration',
                    'type': "quantitative"
                },
                'x': {
                    'field': 'date',
                    'type': "temporal",
                    'timeUnit': 'day'
                }
            }
        });
        it('should be compiled into correct string set', function () {
            model.component.data = {};
            model.component.data.timeUnitDomain = timeunitdomain_1.timeUnitDomain.parseUnit(model);
            chai_1.assert.deepEqual(model.component.data.timeUnitDomain, { day: true });
        });
        it('should assemble data source with raw domain data', function () {
            var defs = timeunitdomain_1.timeUnitDomain.assemble(model.component.data);
            chai_1.assert.deepEqual(defs, [{
                    name: 'day',
                    transform: [
                        {
                            expr: 'datetime(2006, 0, datum["data"]+1, 0, 0, 0, 0)',
                            field: 'date',
                            type: 'formula'
                        }
                    ],
                    values: [0, 1, 2, 3, 4, 5, 6]
                }]);
        });
    });
    describe('unit: day', function () {
    });
    describe('facet', function () {
    });
});
describe('data: colorRank', function () {
});
describe('data: nonPositiveFilter', function () {
    describe('unit (with log scale)', function () {
        var model = util_1.parseUnitModel({
            mark: "point",
            encoding: {
                x: { field: 'a', type: "temporal" },
                y: { field: 'b', type: "quantitative", scale: { type: 'log' } }
            }
        });
        it('should produce the correct nonPositiveFilter component', function () {
            model.component.data = {};
            model.component.data.nonPositiveFilter = nonpositivenullfilter_1.nonPositiveFilter.parseUnit(model);
            chai_1.assert.deepEqual(model.component.data.nonPositiveFilter, {
                b: true,
                a: false
            });
        });
        it('should assemble the correct filter transform', function () {
            var filterTransform = nonpositivenullfilter_1.nonPositiveFilter.assemble(model.component.data)[0];
            chai_1.assert.deepEqual(filterTransform, {
                type: 'filter',
                test: 'datum["b"] > 0'
            });
        });
    });
    describe('unit (with aggregated log scale)', function () {
    });
    describe('facet', function () {
    });
});
describe('data: stack', function () {
    describe('unit without stack', function () {
        var model = util_1.parseUnitModel({
            "mark": "point",
            "encoding": {}
        });
        it('should not produce stack component', function () {
            model.component.data = {};
            model.component.data.stackScale = stackscale_1.stackScale.parseUnit(model);
            chai_1.assert.equal(model.component.data.stackScale, null);
        });
    });
    describe('unit with color and binned x', function () {
        var model = util_1.parseUnitModel({
            "mark": "bar",
            "encoding": {
                "x": { "type": "quantitative", "field": "Cost__Other", "aggregate": "sum" },
                "y": { "bin": true, "type": "quantitative", "field": "Cost__Total_$" },
                "color": { "type": "ordinal", "field": "Effect__Amount_of_damage" }
            }
        });
        model.component.data = {};
        model.component.data.stackScale = stackscale_1.stackScale.parseUnit(model);
        it('should produce the correct stack component', function () {
            var stackedData = model.component.data.stackScale;
            chai_1.assert.equal(stackedData.transform[0].groupby[0], 'bin_Cost__Total_$_start');
        });
        it('should assemble stack summary data correctly', function () {
            var summaryData = stackscale_1.stackScale.assemble(model.component.data);
            chai_1.assert.deepEqual(summaryData, model.component.data.stackScale);
        });
    });
    describe('unit with color and binned y', function () {
        var model = util_1.parseUnitModel({
            "mark": "bar",
            "encoding": {
                "y": { "type": "quantitative", "field": "Cost__Other", "aggregate": "sum" },
                "x": { "bin": true, "type": "quantitative", "field": "Cost__Total_$" },
                "color": { "type": "ordinal", "field": "Effect__Amount_of_damage" }
            }
        });
        model.component.data = {};
        model.component.data.stackScale = stackscale_1.stackScale.parseUnit(model);
        it('should produce the correct stack component', function () {
            var stackedData = model.component.data.stackScale;
            chai_1.assert.equal(stackedData.transform[0].groupby[0], 'bin_Cost__Total_$_start');
        });
        it('should assemble stack summary data correctly', function () {
            var summaryData = stackscale_1.stackScale.assemble(model.component.data);
            chai_1.assert.deepEqual(summaryData, model.component.data.stackScale);
        });
    });
    describe('facet', function () {
    });
});
describe('data: summary', function () {
    var identity = {
        dataName: function (data) {
            return 'source';
        }
    };
    describe('unit (aggregated)', function () {
        var model = util_1.parseUnitModel({
            mark: "point",
            encoding: {
                'y': {
                    'aggregate': 'sum',
                    'field': 'Acceleration',
                    'type': "quantitative"
                },
                'x': {
                    'field': 'Origin',
                    'type': "ordinal"
                },
                color: { field: '*', type: "quantitative", aggregate: 'count' }
            }
        });
        model.component.data = {};
        model.component.data.summary = summary_1.summary.parseUnit(model);
        it('should produce the correct summary component', function () {
            chai_1.assert.deepEqual(model.component.data.summary, [{
                    name: 'summary',
                    dimensions: { Origin: true },
                    measures: { '*': { count: true }, Acceleration: { sum: true } }
                }]);
        });
        it('should assemble the correct aggregate transform', function () {
            var summaryData = summary_1.summary.assemble(model.component.data, identity)[0];
            chai_1.assert.deepEqual(summaryData, {
                'name': "summary",
                'source': 'source',
                'transform': [{
                        'type': 'aggregate',
                        'groupby': ['Origin'],
                        'summarize': {
                            '*': ['count'],
                            'Acceleration': ['sum']
                        }
                    }]
            });
        });
    });
    describe('unit (aggregated with detail arrays)', function () {
        var model = util_1.parseUnitModel({
            mark: "point",
            encoding: {
                'x': { 'aggregate': 'mean', 'field': 'Displacement', 'type': "quantitative" },
                'detail': [
                    { 'field': 'Origin', 'type': "ordinal" },
                    { 'field': 'Cylinders', 'type': "quantitative" }
                ]
            }
        });
        it('should produce the correct summary component', function () {
            model.component.data = {};
            model.component.data.summary = summary_1.summary.parseUnit(model);
            chai_1.assert.deepEqual(model.component.data.summary, [{
                    name: 'summary',
                    dimensions: { Origin: true, Cylinders: true },
                    measures: { Displacement: { mean: true } }
                }]);
        });
        it('should assemble the correct summary data', function () {
            var summaryData = summary_1.summary.assemble(model.component.data, identity)[0];
            chai_1.assert.deepEqual(summaryData, {
                'name': "summary",
                'source': 'source',
                'transform': [{
                        'type': 'aggregate',
                        'groupby': ['Origin', 'Cylinders'],
                        'summarize': {
                            'Displacement': ['mean']
                        }
                    }]
            });
        });
    });
});
//# sourceMappingURL=data.test.js.map