"use strict";
var chai_1 = require('chai');
var util_1 = require('../util');
var channel_1 = require('../../src/channel');
var layout_1 = require('../../src/compile/layout');
describe('compile/layout', function () {
    describe('cardinalityExpr', function () {
        it('should return correct cardinality expr by default', function () {
            var model = util_1.parseUnitModel({
                mark: 'point',
                encoding: {
                    x: { field: 'a', type: 'ordinal' }
                }
            });
            var expr = layout_1.cardinalityExpr(model, channel_1.X);
            chai_1.assert.equal(expr, 'datum["distinct_a"]');
        });
        it('should return domain length if custom domain is provided', function () {
            var model = util_1.parseUnitModel({
                mark: 'point',
                encoding: {
                    x: { field: 'a', type: 'ordinal', scale: { domain: ['a', 'b'] } }
                }
            });
            var expr = layout_1.cardinalityExpr(model, channel_1.X);
            chai_1.assert.equal(expr, '2');
        });
        it('should return complete domain of timeUnit if applicable', function () {
            var model = util_1.parseUnitModel({
                mark: 'point',
                encoding: {
                    x: { field: 'a', type: 'temporal', timeUnit: 'month' }
                }
            });
            var expr = layout_1.cardinalityExpr(model, channel_1.X);
            chai_1.assert.equal(expr, '12');
        });
    });
    describe('unitSizeExpr', function () {
        it('should return correct formula for ordinal scale', function () {
            var model = util_1.parseUnitModel({
                mark: 'point',
                encoding: {
                    x: { field: 'a', type: 'ordinal' }
                }
            });
            var sizeExpr = layout_1.unitSizeExpr(model, channel_1.X);
            chai_1.assert.equal(sizeExpr, '(datum["distinct_a"] + 1) * 21');
        });
        it('should return static cell size for ordinal x-scale with fit', function () {
            var model = util_1.parseUnitModel({
                mark: 'point',
                encoding: {
                    x: { field: 'a', type: 'ordinal', scale: { bandSize: 'fit' } }
                }
            });
            var sizeExpr = layout_1.unitSizeExpr(model, channel_1.X);
            chai_1.assert.equal(sizeExpr, '200');
        });
        it('should return static cell size for ordinal y-scale with fit', function () {
            var model = util_1.parseUnitModel({
                mark: 'point',
                encoding: {
                    y: { field: 'a', type: 'ordinal', scale: { bandSize: 'fit' } }
                }
            });
            var sizeExpr = layout_1.unitSizeExpr(model, channel_1.Y);
            chai_1.assert.equal(sizeExpr, '200');
        });
        it('should return static cell size for ordinal scale with top-level width', function () {
            var model = util_1.parseUnitModel({
                width: 205,
                mark: 'point',
                encoding: {
                    x: { field: 'a', type: 'ordinal' }
                }
            });
            var sizeExpr = layout_1.unitSizeExpr(model, channel_1.X);
            chai_1.assert.equal(sizeExpr, '205');
        });
        it('should return static cell size for ordinal scale with top-level width even if there is numeric bandSize', function () {
            var model = util_1.parseUnitModel({
                width: 205,
                mark: 'point',
                encoding: {
                    x: { field: 'a', type: 'ordinal', scale: { bandSize: 21 } }
                }
            });
            var sizeExpr = layout_1.unitSizeExpr(model, channel_1.X);
            chai_1.assert.equal(sizeExpr, '205');
        });
        it('should return static cell width for non-ordinal x-scale', function () {
            var model = util_1.parseUnitModel({
                mark: 'point',
                encoding: {
                    x: { field: 'a', type: 'quantitative' }
                }
            });
            var sizeExpr = layout_1.unitSizeExpr(model, channel_1.X);
            chai_1.assert.equal(sizeExpr, '200');
        });
        it('should return static cell size for non-ordinal y-scale', function () {
            var model = util_1.parseUnitModel({
                mark: 'point',
                encoding: {
                    y: { field: 'a', type: 'quantitative' }
                }
            });
            var sizeExpr = layout_1.unitSizeExpr(model, channel_1.Y);
            chai_1.assert.equal(sizeExpr, '200');
        });
        it('should return default bandSize if axis is not mapped', function () {
            var model = util_1.parseUnitModel({
                mark: 'point',
                encoding: {},
                config: { scale: { bandSize: 17 } }
            });
            var sizeExpr = layout_1.unitSizeExpr(model, channel_1.X);
            chai_1.assert.equal(sizeExpr, '17');
        });
        it('should return textBandWidth if axis is not mapped for X of text mark', function () {
            var model = util_1.parseUnitModel({
                mark: 'text',
                encoding: {},
                config: { scale: { textBandWidth: 91 } }
            });
            var sizeExpr = layout_1.unitSizeExpr(model, channel_1.X);
            chai_1.assert.equal(sizeExpr, '91');
        });
    });
});
//# sourceMappingURL=layout.test.js.map