"use strict";
var chai_1 = require('chai');
var util_1 = require('../util');
var channel_1 = require('../../src/channel');
var config_1 = require('../../src/config');
var legend = require('../../src/compile/legend');
var timeunit_1 = require('../../src/timeunit');
var type_1 = require('../../src/type');
describe('Legend', function () {
    describe('parseLegend()', function () {
        it('should output explicitly specified properties', function () {
            it('should produce a Vega axis object with correct type and scale', function () {
                var model = util_1.parseUnitModel({
                    mark: "point",
                    encoding: {
                        x: { field: "a", type: "nominal" },
                        color: { field: "a", type: "nominal" }
                    }
                });
                var def = legend.parseLegend(model, channel_1.X);
                chai_1.assert.isObject(def);
                chai_1.assert.equal(def.title, "a");
            });
        });
    });
    describe('title()', function () {
        it('should add explicitly specified title', function () {
            var title = legend.title({ title: 'Custom' }, { field: 'a' }, config_1.defaultConfig);
            chai_1.assert.deepEqual(title, 'Custom');
        });
        it('should add return fieldTitle by default', function () {
            var title = legend.title({}, { field: 'a' }, config_1.defaultConfig);
            chai_1.assert.deepEqual(title, 'a');
        });
    });
    describe('values()', function () {
        it('should return correct timestamp values for DateTimes', function () {
            var values = legend.values({ values: [{ year: 1970 }, { year: 1980 }] });
            chai_1.assert.deepEqual(values, [
                new Date(1970, 0, 1).getTime(),
                new Date(1980, 0, 1).getTime()
            ]);
        });
        it('should simply return values for non-DateTime', function () {
            var values = legend.values({ values: [1, 2, 3, 4] });
            chai_1.assert.deepEqual(values, [1, 2, 3, 4]);
        });
    });
    describe('properties.symbols', function () {
        it('should initialize if filled', function () {
            var symbol = legend.properties.symbols({ field: 'a' }, {}, util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "nominal" },
                    color: { field: "a", type: "nominal" }
                }
            }), channel_1.COLOR);
            chai_1.assert.deepEqual(symbol.strokeWidth.value, 2);
        });
        it('should not have strokeDash and strokeDashOffset', function () {
            var symbol = legend.properties.symbols({ field: 'a' }, {}, util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "nominal" },
                    color: { field: "a", type: "nominal" }
                }
            }), channel_1.COLOR);
            chai_1.assert.isUndefined(symbol.strokeDash);
            chai_1.assert.isUndefined(symbol.strokeDashOffset);
        });
        it('should return specific color value', function () {
            var symbol = legend.properties.symbols({ field: 'a' }, {}, util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "nominal" },
                    color: { field: "a", type: "nominal", legend: { "symbolColor": "red" } }
                }
            }), channel_1.COLOR);
            chai_1.assert.deepEqual(symbol.fill.value, "red");
        });
        it('should return specific shape value', function () {
            var symbol = legend.properties.symbols({ field: 'a' }, {}, util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "nominal" },
                    color: { field: "a", type: "nominal", legend: { "symbolShape": "diamond" } }
                }
            }), channel_1.COLOR);
            chai_1.assert.deepEqual(symbol.shape.value, "diamond");
        });
        it('should return specific size of the symbol', function () {
            var symbol = legend.properties.symbols({ field: 'a' }, {}, util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "nominal" },
                    color: { field: "a", type: "nominal", legend: { "symbolSize": 20 } } }
            }), channel_1.COLOR);
            chai_1.assert.deepEqual(symbol.size.value, 20);
        });
        it('should return not override size of the symbol for size channel', function () {
            var symbol = legend.properties.symbols({ field: 'a' }, {}, util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "nominal" },
                    size: { field: "b", type: "quantitative", legend: { "symbolSize": 20 } } }
            }), channel_1.SIZE);
            chai_1.assert.isUndefined(symbol.size);
        });
        it('should return not override size of the symbol for shape channel', function () {
            var symbol = legend.properties.symbols({ field: 'a' }, {}, util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "nominal" },
                    shape: { field: "b", type: "nominal", legend: { "shape": "circle" } } }
            }), channel_1.SHAPE);
            chai_1.assert.isUndefined(symbol.size);
        });
        it('should return specific width of the symbol', function () {
            var symbol = legend.properties.symbols({ field: 'a' }, {}, util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "nominal" },
                    color: { field: "a", type: "nominal", legend: { "symbolStrokeWidth": 20 } } }
            }), channel_1.COLOR);
            chai_1.assert.deepEqual(symbol.strokeWidth.value, 20);
        });
        it('should create legend for SVG path', function () {
            var symbol = legend.properties.symbols({ field: 'a' }, {}, util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "nominal" },
                    color: { field: "a", type: "nominal" }
                },
                config: {
                    mark: {
                        shape: "M0,0.2L0.2351,0.3236 0.1902,0.0618 0.3804,-0.1236 0.1175,-0.1618 0,-0.4 -0.1175,-0.1618 -0.3804,-0.1236 -0.1902,0.0618 -0.2351,0.3236 0,0.2Z"
                    }
                }
            }), channel_1.COLOR);
            chai_1.assert.deepEqual(symbol.shape.value, "M0,0.2L0.2351,0.3236 0.1902,0.0618 0.3804,-0.1236 0.1175,-0.1618 0,-0.4 -0.1175,-0.1618 -0.3804,-0.1236 -0.1902,0.0618 -0.2351,0.3236 0,0.2Z");
        });
    });
    describe('properties.labels', function () {
        it('should return alignment value of the label', function () {
            var label = legend.properties.labels({ field: 'a' }, {}, util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "nominal" },
                    color: { field: "a", type: "nominal", legend: { "labelAlign": "left" } } }
            }), channel_1.COLOR);
            chai_1.assert.deepEqual(label.align.value, "left");
        });
        it('should return color value of the label', function () {
            var label = legend.properties.labels({ field: 'a' }, {}, util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "nominal" },
                    color: { field: "a", type: "nominal", legend: { "labelColor": "blue" } } }
            }), channel_1.COLOR);
            chai_1.assert.deepEqual(label.fill.value, "blue");
        });
        it('should return font of the label', function () {
            var label = legend.properties.labels({ field: 'a' }, {}, util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "nominal" },
                    color: { field: "a", type: "nominal", legend: { "labelFont": "what" } } }
            }), channel_1.COLOR);
            chai_1.assert.deepEqual(label.font.value, "what");
        });
        it('should return font size of the label', function () {
            var label = legend.properties.labels({ field: 'a' }, {}, util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "nominal" },
                    color: { field: "a", type: "nominal", legend: { "labelFontSize": 20 } } }
            }), channel_1.COLOR);
            chai_1.assert.deepEqual(label.fontSize.value, 20);
        });
        it('should return baseline of the label', function () {
            var label = legend.properties.labels({ field: 'a' }, {}, util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "nominal" },
                    color: { field: "a", type: "nominal", legend: { "labelBaseline": "middle" } } }
            }), channel_1.COLOR);
            chai_1.assert.deepEqual(label.baseline.value, "middle");
        });
        it('should return correct template for the timeUnit: TimeUnit.MONTH', function () {
            var model = util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "temporal" },
                    color: { field: "a", type: "temporal", timeUnit: "month" } }
            });
            var fieldDef = { field: 'a', type: type_1.TEMPORAL, timeUnit: timeunit_1.TimeUnit.MONTH };
            var label = legend.properties.labels(fieldDef, {}, model, channel_1.COLOR);
            var expected = "{{datum[\"data\"] | time:'%b'}}";
            chai_1.assert.deepEqual(label.text.template, expected);
        });
        it('should return correct template for the timeUnit: TimeUnit.QUARTER', function () {
            var model = util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "temporal" },
                    color: { field: "a", type: "temporal", timeUnit: "quarter" } }
            });
            var fieldDef = { field: 'a', type: type_1.TEMPORAL, timeUnit: timeunit_1.TimeUnit.QUARTER };
            var label = legend.properties.labels(fieldDef, {}, model, channel_1.COLOR);
            var quarterPrefix = 'Q';
            var expected = quarterPrefix + "{{datum[\"data\"] | quarter}}";
            chai_1.assert.deepEqual(label.text.template, expected);
        });
    });
    describe('properties.title', function () {
        it('should return color of the title', function () {
            var title = legend.properties.title({ field: 'a' }, {}, util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "nominal" },
                    color: { field: "a", type: "nominal", legend: { "titleColor": "black" } } }
            }), channel_1.COLOR);
            chai_1.assert.deepEqual(title.fill.value, "black");
        });
        it('should return font of the title', function () {
            var title = legend.properties.title({ field: 'a' }, {}, util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "nominal" },
                    color: { field: "a", type: "nominal", legend: { "titleFont": "abcd" } } }
            }), channel_1.COLOR);
            chai_1.assert.deepEqual(title.font.value, "abcd");
        });
        it('should return font size of the title', function () {
            var title = legend.properties.title({ field: 'a' }, {}, util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "nominal" },
                    color: { field: "a", type: "nominal", legend: { "titleFontSize": 22 } } }
            }), channel_1.COLOR);
            chai_1.assert.deepEqual(title.fontSize.value, 22);
        });
        it('should return font weight of the title', function () {
            var title = legend.properties.title({ field: 'a' }, {}, util_1.parseUnitModel({
                mark: "point",
                encoding: {
                    x: { field: "a", type: "nominal" },
                    color: { field: "a", type: "nominal", legend: { "titleFontWeight": 5 } } }
            }), channel_1.COLOR);
            chai_1.assert.deepEqual(title.fontWeight.value, 5);
        });
    });
});
//# sourceMappingURL=legend.test.js.map