"use strict";
var chai_1 = require('chai');
var util_1 = require('../../util');
var util_2 = require('../../../src/util');
var channel_1 = require('../../../src/channel');
var point_1 = require('../../../src/compile/mark/point');
describe('Mark: Point', function () {
    it('should return the correct mark type', function () {
        chai_1.assert.equal(point_1.point.markType(), 'symbol');
    });
    function pointXY(moreEncoding) {
        if (moreEncoding === void 0) { moreEncoding = {}; }
        var spec = {
            "mark": "point",
            "encoding": util_2.extend({
                "x": { "field": "year", "type": "ordinal" },
                "y": { "field": "yield", "type": "quantitative" }
            }, moreEncoding),
            "data": { "url": "data/barley.json" }
        };
        return spec;
    }
    describe('with x', function () {
        var model = util_1.parseUnitModel({
            "mark": "point",
            "encoding": { "x": { "field": "year", "type": "ordinal" } },
            "data": { "url": "data/barley.json" }
        });
        var props = point_1.point.properties(model);
        it('should be centered on y', function () {
            chai_1.assert.deepEqual(props.y, { value: 21 / 2 });
        });
        it('should scale on x', function () {
            chai_1.assert.deepEqual(props.x, { scale: channel_1.X, field: 'year' });
        });
    });
    describe('with stacked x', function () {
        var model = util_1.parseUnitModel({
            "mark": "point",
            "encoding": {
                "x": { "aggregate": "sum", "field": "a", "type": "quantitative" },
                "color": { "field": "b", "type": "ordinal" }
            },
            "data": { "url": "data/barley.json" },
            "config": { "mark": { "stacked": "zero" } }
        });
        var props = point_1.point.properties(model);
        it('should use stack_end on x', function () {
            chai_1.assert.deepEqual(props.x, { scale: channel_1.X, field: 'sum_a_end' });
        });
    });
    describe('with y', function () {
        var model = util_1.parseUnitModel({
            "mark": "point",
            "encoding": { "y": { "field": "year", "type": "ordinal" } },
            "data": { "url": "data/barley.json" }
        });
        var props = point_1.point.properties(model);
        it('should be centered on x', function () {
            chai_1.assert.deepEqual(props.x, { value: 21 / 2 });
        });
        it('should scale on y', function () {
            chai_1.assert.deepEqual(props.y, { scale: channel_1.Y, field: 'year' });
        });
    });
    describe('with stacked y', function () {
        var model = util_1.parseUnitModel({
            "mark": "point",
            "encoding": {
                "y": { "aggregate": "sum", "field": "a", "type": "quantitative" },
                "color": { "field": "b", "type": "ordinal" }
            },
            "data": { "url": "data/barley.json" },
            "config": { "mark": { "stacked": "zero" } }
        });
        var props = point_1.point.properties(model);
        it('should use stack_end on y', function () {
            chai_1.assert.deepEqual(props.y, { scale: channel_1.Y, field: 'sum_a_end' });
        });
    });
    describe('with x and y', function () {
        var model = util_1.parseUnitModel(pointXY());
        var props = point_1.point.properties(model);
        it('should scale on x', function () {
            chai_1.assert.deepEqual(props.x, { scale: channel_1.X, field: 'year' });
        });
        it('should scale on y', function () {
            chai_1.assert.deepEqual(props.y, { scale: channel_1.Y, field: 'yield' });
        });
    });
    describe('with x, y, size', function () {
        var model = util_1.parseUnitModel(pointXY({
            "size": { "field": "*", "type": "quantitative", "aggregate": "count" }
        }));
        var props = point_1.point.properties(model);
        it('should have scale for size', function () {
            chai_1.assert.deepEqual(props.size, { scale: channel_1.SIZE, field: 'count' });
        });
    });
    describe('with x, y, color', function () {
        var model = util_1.parseUnitModel(pointXY({
            "color": { "field": "yield", "type": "quantitative" }
        }));
        var props = point_1.point.properties(model);
        it('should have scale for color', function () {
            chai_1.assert.deepEqual(props.stroke, { scale: channel_1.COLOR, field: 'yield' });
        });
    });
    describe('with x, y, shape', function () {
        var model = util_1.parseUnitModel(pointXY({
            "shape": { "bin": { "maxbins": 15 }, "field": "yield", "type": "quantitative" }
        }));
        var props = point_1.point.properties(model);
        it('should have scale for shape', function () {
            chai_1.assert.deepEqual(props.shape, { scale: channel_1.SHAPE, field: 'bin_yield_range' });
        });
    });
    describe('with constant color, shape, and size', function () {
        var model = util_1.parseUnitModel(pointXY({
            "shape": { "value": "circle" },
            "color": { "value": "red" },
            "size": { "value": 23 }
        }));
        var props = point_1.point.properties(model);
        it('should correct shape, color and size', function () {
            chai_1.assert.deepEqual(props.shape, { value: "circle" });
            chai_1.assert.deepEqual(props.stroke, { value: "red" });
            chai_1.assert.deepEqual(props.size, { value: 23 });
        });
    });
    describe('with configs', function () {
        it('should apply stroke config over color config', function () {
            var model = util_1.parseUnitModel({
                "mark": "point",
                "encoding": {
                    "x": { "field": "Horsepower", "type": "quantitative" },
                    "y": { "field": "Miles_per_Gallon", "type": "quantitative" }
                },
                "config": { "mark": { "color": "red", "stroke": "blue" } }
            });
            var props = point_1.point.properties(model);
            chai_1.assert.deepEqual(props.stroke, { value: "blue" });
        });
        it('should apply color config', function () {
            var model = util_1.parseUnitModel({
                "mark": "point",
                "encoding": {
                    "x": { "field": "Horsepower", "type": "quantitative" },
                    "y": { "field": "Miles_per_Gallon", "type": "quantitative" }
                },
                "config": { "mark": { "color": "red" } }
            });
            var props = point_1.point.properties(model);
            chai_1.assert.deepEqual(props.stroke, { value: "red" });
        });
    });
});
describe('Mark: Square', function () {
    it('should return the correct mark type', function () {
        chai_1.assert.equal(point_1.square.markType(), 'symbol');
    });
    it('should have correct shape', function () {
        var model = util_1.parseUnitModel({
            "mark": "square",
            "encoding": {
                "color": { "value": "blue" }
            }
        });
        var props = point_1.square.properties(model);
        chai_1.assert.equal(props.shape.value, 'square');
    });
    it('should be filled by default', function () {
        var model = util_1.parseUnitModel({
            "mark": "square",
            "encoding": {
                "color": { "value": "blue" }
            }
        });
        var props = point_1.square.properties(model);
        chai_1.assert.equal(props.fill.value, 'blue');
    });
    it('with config.mark.filled:false should have transparent fill', function () {
        var model = util_1.parseUnitModel({
            "mark": "square",
            "encoding": {
                "color": { "value": "blue" }
            },
            "config": {
                "mark": {
                    "filled": false
                }
            }
        });
        var props = point_1.square.properties(model);
        chai_1.assert.equal(props.stroke.value, 'blue');
        chai_1.assert.equal(props.fill.value, 'transparent');
    });
});
describe('Mark: Circle', function () {
    it('should return the correct mark type', function () {
        chai_1.assert.equal(point_1.circle.markType(), 'symbol');
    });
    var model = util_1.parseUnitModel({
        "mark": "circle",
        "encoding": {
            "color": { "value": "blue" }
        }
    });
    var props = point_1.circle.properties(model);
    it('should have correct shape', function () {
        chai_1.assert.equal(props.shape.value, 'circle');
    });
    it('should be filled by default', function () {
        chai_1.assert.equal(props.fill.value, 'blue');
    });
    it('with config.mark.filled:false should have transparent fill', function () {
        var filledCircleModel = util_1.parseUnitModel({
            "mark": "circle",
            "encoding": {
                "color": { "value": "blue" }
            },
            "config": {
                "mark": {
                    "filled": false
                }
            }
        });
        var filledCircleProps = point_1.circle.properties(filledCircleModel);
        chai_1.assert.equal(filledCircleProps.stroke.value, 'blue');
        chai_1.assert.equal(filledCircleProps.fill.value, 'transparent');
    });
});
//# sourceMappingURL=point.test.js.map