"use strict";
var chai_1 = require('chai');
var util_1 = require('../../util');
var text_1 = require('../../../src/compile/mark/text');
var channel_1 = require('../../../src/channel');
describe('Mark: Text', function () {
    it('should return correct marktype', function () {
        chai_1.assert.equal(text_1.text.markType(), 'text');
    });
    describe('with nothing', function () {
        var spec = {
            "mark": "text",
            "encoding": {},
            "data": { "url": "data/cars.json" }
        };
        var model = util_1.parseUnitModel(spec);
        var props = text_1.text.properties(model);
        it('should have placeholder text', function () {
            chai_1.assert.deepEqual(props.text, { value: "Abc" });
        });
    });
    describe('with stacked x', function () {
        var model = util_1.parseUnitModel({
            "mark": "text",
            "encoding": {
                "x": { "aggregate": "sum", "field": "a", "type": "quantitative" },
                "color": { "field": "b", "type": "ordinal" }
            },
            "data": { "url": "data/barley.json" },
            "config": { "mark": { "stacked": "zero" } }
        });
        var props = text_1.text.properties(model);
        it('should use stack_end on x', function () {
            chai_1.assert.deepEqual(props.x, { scale: channel_1.X, field: 'sum_a_end' });
        });
    });
    describe('with stacked y', function () {
        var model = util_1.parseUnitModel({
            "mark": "text",
            "encoding": {
                "y": { "aggregate": "sum", "field": "a", "type": "quantitative" },
                "color": { "field": "b", "type": "ordinal" }
            },
            "data": { "url": "data/barley.json" },
            "config": { "mark": { "stacked": "zero" } }
        });
        var props = text_1.text.properties(model);
        it('should use stack_end on y', function () {
            chai_1.assert.deepEqual(props.y, { scale: channel_1.Y, field: 'sum_a_end' });
        });
    });
    describe('with quantitative and format', function () {
        var spec = {
            "mark": "text",
            "encoding": {
                "text": { "field": "foo", "type": "quantitative" }
            },
            "config": {
                "mark": {
                    "format": "d"
                }
            }
        };
        var model = util_1.parseUnitModel(spec);
        var props = text_1.text.properties(model);
        it('should use number template', function () {
            chai_1.assert.deepEqual(props.text, { template: '{{datum["foo"] | number:\'d\'}}' });
        });
    });
    describe('with temporal', function () {
        var spec = {
            "mark": "text",
            "encoding": {
                "text": { "field": "foo", "type": "temporal" }
            }
        };
        var model = util_1.parseUnitModel(spec);
        var props = text_1.text.properties(model);
        it('should use date template', function () {
            chai_1.assert.deepEqual(props.text, { template: '{{datum["foo"] | time:\'%b %d, %Y\'}}' });
        });
    });
    describe('with x, y, text (ordinal)', function () {
        var spec = {
            "mark": "text",
            "encoding": {
                "x": { "field": "Acceleration", "type": "ordinal" },
                "y": { "field": "Displacement", "type": "quantitative" },
                "text": { "field": "Origin", "type": "ordinal" },
            },
            "data": { "url": "data/cars.json" }
        };
        var model = util_1.parseUnitModel(spec);
        var props = text_1.text.properties(model);
        it('should scale on x', function () {
            chai_1.assert.deepEqual(props.x, { scale: channel_1.X, field: 'Acceleration' });
        });
        it('should scale on y', function () {
            chai_1.assert.deepEqual(props.y, { scale: channel_1.Y, field: 'Displacement' });
        });
        it('should be centered', function () {
            chai_1.assert.deepEqual(props.align, { value: "center" });
        });
        it('should map text without template', function () {
            chai_1.assert.deepEqual(props.text, { field: "Origin" });
        });
    });
    describe('with row, column, text, and color', function () {
        var spec = {
            "mark": "text",
            "encoding": {
                "row": { "field": "Origin", "type": "ordinal" },
                "column": { "field": "Cylinders", "type": "ordinal" },
                "text": { "field": "Acceleration", "type": "quantitative", "aggregate": "mean" },
                "color": { "field": "Acceleration", "type": "quantitative", "aggregate": "mean" },
                "size": { "field": "Acceleration", "type": "quantitative", "aggregate": "mean" }
            },
            "data": { "url": "data/cars.json" }
        };
        var model = util_1.parseUnitModel(spec);
        var props = text_1.text.properties(model);
        it('should fit cell on x', function () {
            chai_1.assert.deepEqual(props.x, { field: { group: 'width' }, offset: -5 });
        });
        it('should center on y', function () {
            chai_1.assert.deepEqual(props.y, { value: 10.5 });
        });
        it('should map text to template', function () {
            chai_1.assert.deepEqual(props.text, {
                template: "{{datum[\"mean_Acceleration\"] | number:'s'}}"
            });
        });
        it('should map color to fill', function () {
            chai_1.assert.deepEqual(props.fill, {
                scale: 'color',
                field: 'mean_Acceleration'
            });
        });
        it('should map size to fontSize', function () {
            chai_1.assert.deepEqual(props.fontSize, {
                scale: 'size',
                field: 'mean_Acceleration'
            });
        });
    });
    describe('with row, column, text, and color and mark configs(applyColorToBackground, opacity)', function () {
        var spec = {
            "mark": "text",
            "encoding": {
                "row": { "field": "Origin", "type": "ordinal" },
                "column": { "field": "Cylinders", "type": "ordinal" },
                "text": { "field": "Acceleration", "type": "quantitative", "aggregate": "mean" },
                "color": { "field": "Acceleration", "type": "quantitative", "aggregate": "mean" },
                "size": { "field": "Acceleration", "type": "quantitative", "aggregate": "mean" }
            },
            "config": {
                "mark": {
                    "applyColorToBackground": true,
                    "opacity": 0.8
                }
            },
            "data": { "url": "data/cars.json" }
        };
        var model = util_1.parseUnitModel(spec);
        var props = text_1.text.properties(model);
        it('should fill black', function () {
            chai_1.assert.deepEqual(props.fill, { value: 'black' });
        });
        var bg = text_1.text.background(model);
        it('should map color to background', function () {
            chai_1.assert.deepEqual(bg.fill, {
                scale: 'color',
                field: 'mean_Acceleration'
            });
        });
    });
});
//# sourceMappingURL=text.test.js.map