"use strict";
var chai_1 = require('chai');
var util_1 = require('../../util');
var channel_1 = require('../../../src/channel');
var tick_1 = require('../../../src/compile/mark/tick');
describe('Mark: Tick', function () {
    it('should return the correct mark type', function () {
        chai_1.assert.equal(tick_1.tick.markType(), 'rect');
    });
    describe('with stacked x', function () {
        var model = util_1.parseUnitModel({
            "mark": "tick",
            "encoding": {
                "x": { "aggregate": "sum", "field": "a", "type": "quantitative" },
                "color": { "field": "b", "type": "ordinal" }
            },
            "data": { "url": "data/barley.json" },
            "config": { "mark": { "stacked": "zero" } }
        });
        var props = tick_1.tick.properties(model);
        it('should use stack_end on x', function () {
            chai_1.assert.deepEqual(props.xc, { scale: channel_1.X, field: 'sum_a_end' });
        });
    });
    describe('with stacked y', function () {
        var model = util_1.parseUnitModel({
            "mark": "tick",
            "encoding": {
                "y": { "aggregate": "sum", "field": "a", "type": "quantitative" },
                "color": { "field": "b", "type": "ordinal" }
            },
            "data": { "url": "data/barley.json" },
            "config": { "mark": { "stacked": "zero" } }
        });
        var props = tick_1.tick.properties(model);
        it('should use stack_end on y', function () {
            chai_1.assert.deepEqual(props.yc, { scale: channel_1.Y, field: 'sum_a_end' });
        });
    });
    describe('with quantitative x', function () {
        var model = util_1.parseUnitModel({
            'mark': 'tick',
            'encoding': { 'x': { 'field': 'Horsepower', 'type': 'quantitative' } },
            'data': { 'url': 'data/cars.json' }
        });
        var props = tick_1.tick.properties(model);
        it('should be centered on y', function () {
            chai_1.assert.deepEqual(props.yc, { value: 10.5 });
        });
        it('should scale on x', function () {
            chai_1.assert.deepEqual(props.xc, { scale: channel_1.X, field: 'Horsepower' });
        });
        it('width should tick thickness with orient vertical', function () {
            chai_1.assert.deepEqual(props.width, { value: 1 });
        });
    });
    describe('with quantitative y', function () {
        var model = util_1.parseUnitModel({
            'mark': 'tick',
            'encoding': { 'y': { 'field': 'Cylinders', 'type': 'quantitative' } },
            'data': { 'url': 'data/cars.json' }
        });
        var props = tick_1.tick.properties(model);
        it('should be centered on x', function () {
            chai_1.assert.deepEqual(props.xc, { value: 10.5 });
        });
        it('should scale on y', function () {
            chai_1.assert.deepEqual(props.yc, { scale: channel_1.Y, field: 'Cylinders' });
        });
        it('height should tick thickness with orient horizontal', function () {
            chai_1.assert.deepEqual(props.height, { value: 1 });
        });
    });
    describe('with quantitative x and ordinal y', function () {
        var model = util_1.parseUnitModel({
            'mark': 'tick',
            'encoding': {
                'x': { 'field': 'Horsepower', 'type': 'quantitative' },
                'y': { 'field': 'Cylinders', 'type': 'ordinal' }
            },
            'data': { 'url': 'data/cars.json' }
        });
        var props = tick_1.tick.properties(model);
        it('should scale on x', function () {
            chai_1.assert.deepEqual(props.xc, { scale: channel_1.X, field: 'Horsepower' });
        });
        it('should scale on y', function () {
            chai_1.assert.deepEqual(props.yc, { scale: channel_1.Y, field: 'Cylinders' });
        });
        it('wiidth should be tick thickness with default orient vertical', function () {
            chai_1.assert.deepEqual(props.width, { value: 1 });
        });
        it('height should be matched to field with default orient vertical', function () {
            chai_1.assert.deepEqual(props.height, { value: 14 });
        });
    });
    describe('width should be mapped to size', function () {
        var model = util_1.parseUnitModel({
            'mark': 'tick',
            'config': { 'mark': { 'orient': 'vertical' } },
            'encoding': {
                'x': { 'field': 'Horsepower', 'type': 'quantitative' },
                'y': { 'field': 'Cylinders', 'type': 'ordinal' },
                'size': { 'field': 'Acceleration' }
            },
            'data': { 'url': 'data/cars.json' },
        });
        var props = tick_1.tick.properties(model);
        it('width should change with size field', function () {
            chai_1.assert.deepEqual(props.height, { 'field': 'Acceleration', 'scale': channel_1.SIZE });
        });
    });
    describe('height should be mapped to size', function () {
        var model = util_1.parseUnitModel({
            'mark': 'tick',
            'config': { 'mark': { 'orient': 'horizontal' } },
            'encoding': {
                'x': { 'field': 'Horsepower', 'type': 'quantitative' },
                'y': { 'field': 'Cylinders', 'type': 'ordinal' },
                'size': { 'field': 'Acceleration' }
            },
            'data': { 'url': 'data/cars.json' },
        });
        var props = tick_1.tick.properties(model);
        it('height should change with size field', function () {
            chai_1.assert.deepEqual(props.width, { 'field': 'Acceleration', 'scale': channel_1.SIZE });
        });
    });
});
//# sourceMappingURL=tick.test.js.map