"use strict";
var chai_1 = require('chai');
var scale_1 = require('../../src/compile/scale');
var data_1 = require('../../src/data');
var util_1 = require('../util');
var channel_1 = require('../../src/channel');
var scale_2 = require('../../src/scale');
var mark_1 = require('../../src/mark');
describe('Scale', function () {
    describe('scaleType()', function () {
        it('should return null for channel without scale', function () {
            var model = util_1.parseUnitModel({
                mark: 'point',
                encoding: {
                    detail: {
                        field: 'a',
                        type: 'temporal',
                        timeUnit: 'yearMonth'
                    }
                }
            });
            var fieldDef = model.encoding().detail;
            chai_1.assert.deepEqual(scale_1.scaleType(null, fieldDef, channel_1.DETAIL, model.mark()), null);
        });
        it('should return time for yearmonth', function () {
            var model = util_1.parseUnitModel({
                mark: 'point',
                encoding: {
                    y: {
                        field: 'a',
                        type: 'temporal',
                        timeUnit: 'yearMonth'
                    }
                }
            });
            var fieldDef = model.encoding().y;
            var scale = model.scale(channel_1.Y);
            chai_1.assert.deepEqual(scale_1.scaleType(scale, fieldDef, channel_1.Y, model.mark()), scale_2.ScaleType.TIME);
        });
        it('should return ordinal for month', function () {
            var model = util_1.parseUnitModel({
                mark: 'point',
                encoding: {
                    y: {
                        field: 'a',
                        type: 'temporal',
                        timeUnit: 'month'
                    }
                }
            });
            var fieldDef = model.encoding().y;
            var scale = model.scale(channel_1.Y);
            chai_1.assert.deepEqual(scale_1.scaleType(scale, fieldDef, channel_1.Y, model.mark()), scale_2.ScaleType.ORDINAL);
        });
        it('should return ordinal for row', function () {
            var model = util_1.parseUnitModel({
                mark: 'point',
                encoding: {
                    row: {
                        field: 'a',
                        type: 'temporal',
                        timeUnit: 'yearMonth'
                    }
                }
            });
            var fieldDef = model.encoding().row;
            var scale = model.scale(channel_1.ROW);
            chai_1.assert.deepEqual(scale_1.scaleType(scale, fieldDef, channel_1.ROW, model.mark()), scale_2.ScaleType.ORDINAL);
        });
    });
    describe('scaleBandSize()', function () {
        it('should return undefined for non-ordinal scale.', function () {
            chai_1.assert.equal(scale_1.scaleBandSize(scale_2.ScaleType.LINEAR, undefined, scale_2.defaultScaleConfig, 180, mark_1.POINT, channel_1.X), undefined);
            chai_1.assert.equal(scale_1.scaleBandSize(scale_2.ScaleType.LINEAR, 21, scale_2.defaultScaleConfig, undefined, mark_1.POINT, channel_1.X), undefined);
        });
        it('should return "fit" if top-level size is provided for ordinal scale', function () {
            var bandSize = scale_1.scaleBandSize(scale_2.ScaleType.ORDINAL, undefined, scale_2.defaultScaleConfig, 180, mark_1.POINT, channel_1.X);
            chai_1.assert.deepEqual(bandSize, scale_2.BANDSIZE_FIT);
        });
        it('should return provided bandSize for ordinal scale', function () {
            var bandSize = scale_1.scaleBandSize(scale_2.ScaleType.ORDINAL, 21, scale_2.defaultScaleConfig, undefined, mark_1.POINT, channel_1.X);
            chai_1.assert.deepEqual(bandSize, 21);
        });
    });
    describe('domain()', function () {
        it('should return domain for stack', function () {
            var model = util_1.parseUnitModel({
                mark: "bar",
                encoding: {
                    y: {
                        aggregate: 'sum',
                        field: 'origin'
                    },
                    x: { field: 'x', type: "ordinal" },
                    color: { field: 'color', type: "ordinal" },
                    row: { field: 'row' }
                }
            });
            var _domain = scale_1.domain(model.scale(channel_1.Y), model, channel_1.Y);
            chai_1.assert.deepEqual(_domain, {
                data: 'stacked_scale',
                field: 'sum_sum_origin'
            });
        });
        describe('for quantitative', function () {
            it('should return the right domain for binned Q', function () {
                var model = util_1.parseUnitModel({
                    mark: "point",
                    encoding: {
                        y: {
                            bin: { maxbins: 15 },
                            field: 'origin',
                            scale: { useRawDomain: true },
                            type: "quantitative"
                        }
                    }
                });
                var _domain = scale_1.domain(model.scale(channel_1.Y), model, channel_1.Y);
                chai_1.assert.deepEqual(_domain, {
                    data: data_1.SOURCE,
                    field: [
                        'bin_origin_start',
                        'bin_origin_end'
                    ]
                });
            });
            it('should return the raw domain if useRawDomain is true for non-bin, non-sum Q', function () {
                var model = util_1.parseUnitModel({
                    mark: "point",
                    encoding: {
                        y: {
                            aggregate: 'mean',
                            field: 'origin',
                            scale: { useRawDomain: true },
                            type: "quantitative"
                        }
                    }
                });
                var _domain = scale_1.domain(model.scale(channel_1.Y), model, channel_1.Y);
                chai_1.assert.deepEqual(_domain.data, data_1.SOURCE);
            });
            it('should return the aggregate domain for sum Q', function () {
                var model = util_1.parseUnitModel({
                    mark: "point",
                    encoding: {
                        y: {
                            aggregate: 'sum',
                            field: 'origin',
                            scale: { useRawDomain: true },
                            type: "quantitative"
                        }
                    }
                });
                var _domain = scale_1.domain(model.scale(channel_1.Y), model, channel_1.Y);
                chai_1.assert.deepEqual(_domain.data, data_1.SUMMARY);
            });
            it('should return the right custom domain', function () {
                var model = util_1.parseUnitModel({
                    mark: "point",
                    encoding: {
                        y: {
                            field: 'horsepower',
                            type: "quantitative",
                            scale: { domain: [0, 200] }
                        }
                    }
                });
                var _domain = scale_1.domain(model.scale(channel_1.Y), model, channel_1.Y);
                chai_1.assert.deepEqual(_domain, [0, 200]);
            });
            it('should return the aggregated domain if useRawDomain is false', function () {
                var model = util_1.parseUnitModel({
                    mark: "point",
                    encoding: {
                        y: {
                            aggregate: 'min',
                            field: 'origin',
                            scale: { useRawDomain: false },
                            type: "quantitative"
                        }
                    }
                });
                var _domain = scale_1.domain(model.scale(channel_1.Y), model, channel_1.Y);
                chai_1.assert.deepEqual(_domain.data, data_1.SUMMARY);
            });
        });
        describe('for time', function () {
            it('should return the raw domain if useRawDomain is true for raw T', function () {
                var model = util_1.parseUnitModel({
                    mark: "point",
                    encoding: {
                        y: {
                            field: 'origin',
                            scale: { useRawDomain: true },
                            type: "temporal"
                        }
                    }
                });
                var _domain = scale_1.domain(model.scale(channel_1.Y), model, channel_1.Y);
                chai_1.assert.deepEqual(_domain.data, data_1.SOURCE);
            });
            it('should return the raw domain if useRawDomain is true for year T', function () {
                var model = util_1.parseUnitModel({
                    mark: "point",
                    encoding: {
                        y: {
                            field: 'origin',
                            scale: { useRawDomain: true },
                            type: "temporal",
                            timeUnit: 'year'
                        }
                    }
                });
                var _domain = scale_1.domain(model.scale(channel_1.Y), model, channel_1.Y);
                chai_1.assert.deepEqual(_domain.data, data_1.SOURCE);
                chai_1.assert.operator(_domain.field.indexOf('year'), '>', -1);
            });
            it('should return the correct domain for month T', function () {
                var model = util_1.parseUnitModel({
                    mark: "point",
                    encoding: {
                        y: {
                            field: 'origin',
                            scale: { useRawDomain: true },
                            type: "temporal",
                            timeUnit: 'month'
                        }
                    }
                });
                var _domain = scale_1.domain(model.scale(channel_1.Y), model, channel_1.Y);
                chai_1.assert.deepEqual(_domain, { data: 'month', field: 'date' });
            });
            it('should return the correct domain for yearmonth T', function () {
                var model = util_1.parseUnitModel({
                    mark: "point",
                    encoding: {
                        y: {
                            field: 'origin',
                            scale: { useRawDomain: true },
                            type: "temporal",
                            timeUnit: 'yearmonth'
                        }
                    }
                });
                var _domain = scale_1.domain(model.scale(channel_1.Y), model, channel_1.Y);
                chai_1.assert.deepEqual(_domain, {
                    data: 'source', field: 'yearmonth_origin',
                    sort: { field: 'yearmonth_origin', op: 'min' }
                });
            });
            it('should return the right custom domain with DateTime objects', function () {
                var model = util_1.parseUnitModel({
                    mark: "point",
                    encoding: {
                        y: {
                            field: 'year',
                            type: "temporal",
                            scale: { domain: [{ year: 1970 }, { year: 1980 }] }
                        }
                    }
                });
                var _domain = scale_1.domain(model.scale(channel_1.Y), model, channel_1.Y);
                chai_1.assert.deepEqual(_domain, [
                    new Date(1970, 0, 1).getTime(),
                    new Date(1980, 0, 1).getTime()
                ]);
            });
        });
        describe('for ordinal', function () {
            it('should return correct domain with the provided sort property', function () {
                var sortDef = { op: 'min', field: 'Acceleration' };
                var model = util_1.parseUnitModel({
                    mark: "point",
                    encoding: {
                        y: { field: 'origin', type: "ordinal", sort: sortDef }
                    }
                });
                chai_1.assert.deepEqual(scale_1.domain(model.scale(channel_1.Y), model, channel_1.Y), {
                    data: "source",
                    field: 'origin',
                    sort: sortDef
                });
            });
            it('should return correct domain without sort if sort is not provided', function () {
                var model = util_1.parseUnitModel({
                    mark: "point",
                    encoding: {
                        y: { field: 'origin', type: "ordinal" }
                    }
                });
                chai_1.assert.deepEqual(scale_1.domain(model.scale(channel_1.Y), model, channel_1.Y), {
                    data: "source",
                    field: 'origin',
                    sort: true
                });
            });
        });
    });
    describe('ordinal with color', function () {
        var model = util_1.parseUnitModel({
            mark: "point",
            encoding: {
                color: { field: 'origin', type: "ordinal" }
            }
        });
        var scales = scale_1.parseScaleComponent(model)['color'];
        it('should create color and inverse scales', function () {
            chai_1.assert.equal(scales.main.name, 'color');
            chai_1.assert.equal(scales.colorLegend.name, 'color_legend');
            chai_1.assert.equal(scales.binColorLegend, undefined);
        });
        it('should create correct inverse scale', function () {
            chai_1.assert.equal(scales.colorLegend.type, 'ordinal');
            chai_1.assert.deepEqual(scales.colorLegend.domain, {
                data: 'source',
                field: 'rank_origin',
                sort: true
            });
            chai_1.assert.deepEqual(scales.colorLegend.range, {
                data: 'source',
                field: 'origin',
                sort: true
            });
        });
        it('should create correct color scale', function () {
            chai_1.assert.equal(scales.main.type, 'linear');
            chai_1.assert.deepEqual(scales.main.domain, {
                data: 'source',
                field: 'rank_origin'
            });
        });
    });
    describe('color with bin', function () {
        var model = util_1.parseUnitModel({
            mark: "point",
            encoding: {
                color: { field: 'origin', type: "quantitative", bin: true }
            }
        });
        var scales = scale_1.parseScaleComponent(model)['color'];
        it('should add correct scales', function () {
            chai_1.assert.equal(scales.main.name, 'color');
            chai_1.assert.equal(scales.colorLegend.name, 'color_legend');
            chai_1.assert.equal(scales.binColorLegend.name, 'color_legend_label');
        });
        it('should create correct identity scale', function () {
            chai_1.assert.equal(scales.colorLegend.type, 'ordinal');
            chai_1.assert.deepEqual(scales.colorLegend.domain, {
                data: 'source',
                field: 'bin_origin_start',
                sort: true
            });
            chai_1.assert.deepEqual(scales.colorLegend.range, {
                data: 'source',
                field: 'bin_origin_start',
                sort: true
            });
        });
        it('should sort range of color labels', function () {
            chai_1.assert.deepEqual(scales.binColorLegend.domain, {
                data: 'source',
                field: 'bin_origin_start',
                sort: true
            });
            chai_1.assert.deepEqual(scales.binColorLegend.range, {
                data: 'source',
                field: 'bin_origin_range',
                sort: { "field": "bin_origin_start", "op": "min" }
            });
        });
    });
    describe('color with time unit', function () {
        var model = util_1.parseUnitModel({
            mark: "point",
            encoding: {
                color: { field: 'origin', type: "temporal", timeUnit: "year" }
            }
        });
        var scales = scale_1.parseScaleComponent(model)['color'];
        it('should add correct scales', function () {
            chai_1.assert.equal(scales.main.name, 'color');
            chai_1.assert.equal(scales.colorLegend.name, 'color_legend');
            chai_1.assert.equal(scales.binColorLegend, undefined);
        });
        it('should create correct identity scale', function () {
            chai_1.assert.equal(scales.colorLegend.type, 'ordinal');
            chai_1.assert.deepEqual(scales.colorLegend.domain, {
                data: 'source',
                field: 'year_origin',
                sort: true
            });
            chai_1.assert.deepEqual(scales.colorLegend.range, {
                data: 'source',
                field: 'year_origin',
                sort: true
            });
        });
    });
    describe('rangeMixins()', function () {
        describe('row', function () {
        });
        describe('column', function () {
        });
        describe('x', function () {
        });
        describe('y', function () {
        });
        describe('size', function () {
            describe('bar', function () {
            });
            describe('text', function () {
            });
            describe('rule', function () {
            });
            describe('point, square, circle', function () {
            });
        });
        describe('shape', function () {
        });
        describe('color', function () {
        });
    });
    describe('bandSize()', function () {
    });
    describe('nice()', function () {
    });
    describe('outerPadding()', function () {
    });
    describe('points()', function () {
    });
    describe('reverse()', function () {
    });
    describe('zero()', function () {
    });
});
//# sourceMappingURL=scale.test.js.map