"use strict";
var chai_1 = require('chai');
var aggregate_1 = require('../src/aggregate');
var channel_1 = require('../src/channel');
var mark_1 = require('../src/mark');
var scale_1 = require('../src/scale');
var stack_1 = require('../src/stack');
var spec_1 = require('../src/spec');
describe('stack', function () {
    var STACKABLE_MARKS = [mark_1.BAR, mark_1.AREA];
    var NON_STACKABLE_MARKS = [mark_1.RULE];
    it('should be disabled for non-stackable marks with at least of of the stack channel', function () {
        [undefined, stack_1.StackOffset.CENTER, stack_1.StackOffset.NONE, stack_1.StackOffset.ZERO, stack_1.StackOffset.NORMALIZE].forEach(function (stacked) {
            NON_STACKABLE_MARKS.forEach(function (nonStackableMark) {
                var spec = {
                    "data": { "url": "data/barley.json" },
                    "mark": nonStackableMark,
                    "encoding": {
                        "x": { "aggregate": "sum", "field": "yield", "type": "quantitative" },
                        "y": { "field": "variety", "type": "nominal" },
                        "color": { "field": "site", "type": "nominal" }
                    },
                    "config": {
                        "mark": { "stacked": stacked }
                    }
                };
                chai_1.assert.isNull(stack_1.stack(spec.mark, spec.encoding, spec.config.mark.stacked));
                chai_1.assert.isFalse(spec_1.isStacked(spec));
            });
        });
    });
    it('should always be disabled for raw plot', function () {
        [undefined, stack_1.StackOffset.CENTER, stack_1.StackOffset.NONE, stack_1.StackOffset.ZERO, stack_1.StackOffset.NORMALIZE].forEach(function (stacked) {
            mark_1.PRIMITIVE_MARKS.forEach(function (mark) {
                var spec = {
                    "data": { "url": "data/barley.json" },
                    "mark": mark,
                    "encoding": {
                        "x": { "field": "yield", "type": "quantitative" },
                        "y": { "field": "variety", "type": "nominal" },
                        "color": { "field": "site", "type": "nominal" }
                    },
                    "config": {
                        "mark": { "stacked": stacked }
                    }
                };
                chai_1.assert.isNull(stack_1.stack(spec.mark, spec.encoding, spec.config.mark.stacked));
                chai_1.assert.isFalse(spec_1.isStacked(spec));
            });
        });
    });
    it('should always be disabled if there is no stackby channel', function () {
        [undefined, stack_1.StackOffset.CENTER, stack_1.StackOffset.NONE, stack_1.StackOffset.ZERO, stack_1.StackOffset.NORMALIZE].forEach(function (stacked) {
            mark_1.PRIMITIVE_MARKS.forEach(function (mark) {
                var spec = {
                    "data": { "url": "data/barley.json" },
                    "mark": mark,
                    "encoding": {
                        "x": { "aggregate": "sum", "field": "yield", "type": "quantitative" },
                        "y": { "field": "variety", "type": "nominal" }
                    },
                    "config": {
                        "mark": { "stacked": stacked }
                    }
                };
                chai_1.assert.isNull(stack_1.stack(spec.mark, spec.encoding, spec.config.mark.stacked));
                chai_1.assert.isFalse(spec_1.isStacked(spec));
            });
        });
    });
    it('should always be disabled if the stackby channel is aggregated', function () {
        [undefined, stack_1.StackOffset.CENTER, stack_1.StackOffset.NONE, stack_1.StackOffset.ZERO, stack_1.StackOffset.NORMALIZE].forEach(function (stacked) {
            mark_1.PRIMITIVE_MARKS.forEach(function (mark) {
                var spec = {
                    "data": { "url": "data/barley.json" },
                    "mark": mark,
                    "encoding": {
                        "x": { "aggregate": "sum", "field": "yield", "type": "quantitative" },
                        "y": { "field": "variety", "type": "nominal" },
                        "color": { "aggregate": "count", "field": "*", "type": "quantitative" }
                    },
                    "config": {
                        "mark": { "stacked": stacked }
                    }
                };
                chai_1.assert.isNull(stack_1.stack(spec.mark, spec.encoding, spec.config.mark.stacked));
                chai_1.assert.isFalse(spec_1.isStacked(spec));
            });
        });
    });
    it('can enabled if one of the stackby channels is not aggregated', function () {
        [undefined, stack_1.StackOffset.CENTER, stack_1.StackOffset.ZERO, stack_1.StackOffset.NORMALIZE].forEach(function (stacked) {
            STACKABLE_MARKS.forEach(function (mark) {
                var spec = {
                    "data": { "url": "data/barley.json" },
                    "mark": mark,
                    "encoding": {
                        "x": { "aggregate": "sum", "field": "yield", "type": "quantitative" },
                        "y": { "field": "variety", "type": "nominal" },
                        "color": { "aggregate": "count", "field": "*", "type": "quantitative" },
                        "detail": { "field": "site", "type": "nominal" }
                    },
                    "config": {
                        "mark": { "stacked": stacked }
                    }
                };
                var _stack = stack_1.stack(spec.mark, spec.encoding, spec.config.mark.stacked);
                chai_1.assert.isOk(_stack);
                chai_1.assert.isTrue(spec_1.isStacked(spec));
                chai_1.assert.equal(_stack.stackByChannels[0], channel_1.DETAIL);
            });
        });
    });
    it('should always be disabled if both x and y are aggregate', function () {
        [undefined, stack_1.StackOffset.CENTER, stack_1.StackOffset.NONE, stack_1.StackOffset.ZERO, stack_1.StackOffset.NORMALIZE].forEach(function (stacked) {
            mark_1.PRIMITIVE_MARKS.forEach(function (mark) {
                var spec = {
                    "data": { "url": "data/barley.json" },
                    "mark": mark,
                    "encoding": {
                        "x": { "aggregate": "sum", "field": "yield", "type": "quantitative" },
                        "y": { "aggregate": "count", "field": "*", "type": "quantitative" },
                        "color": { "field": "site", "type": "nominal" }
                    },
                    "config": {
                        "mark": { "stacked": stacked }
                    }
                };
                chai_1.assert.isNull(stack_1.stack(spec.mark, spec.encoding, spec.config.mark.stacked));
                chai_1.assert.isFalse(spec_1.isStacked(spec));
            });
        });
    });
    it('should always be disabled if neither x nor y is aggregate', function () {
        [undefined, stack_1.StackOffset.CENTER, stack_1.StackOffset.NONE, stack_1.StackOffset.ZERO, stack_1.StackOffset.NORMALIZE].forEach(function (stacked) {
            mark_1.PRIMITIVE_MARKS.forEach(function (mark) {
                var spec = {
                    "data": { "url": "data/barley.json" },
                    "mark": mark,
                    "encoding": {
                        "x": { "field": "variety", "type": "nominal" },
                        "y": { "field": "variety", "type": "nominal" },
                        "color": { "field": "site", "type": "nominal" }
                    },
                    "config": {
                        "mark": { "stacked": stacked }
                    }
                };
                chai_1.assert.isNull(stack_1.stack(spec.mark, spec.encoding, spec.config.mark.stacked));
                chai_1.assert.isFalse(spec_1.isStacked(spec));
            });
        });
    });
    it('should always be disabled if the aggregated axis has non-linear scale', function () {
        [undefined, stack_1.StackOffset.CENTER, stack_1.StackOffset.NONE, stack_1.StackOffset.ZERO, stack_1.StackOffset.NORMALIZE].forEach(function (stacked) {
            [scale_1.ScaleType.LOG, scale_1.ScaleType.POW, scale_1.ScaleType.SQRT].forEach(function (scaleType) {
                mark_1.PRIMITIVE_MARKS.forEach(function (mark) {
                    var spec = {
                        "data": { "url": "data/barley.json" },
                        "mark": mark,
                        "encoding": {
                            "x": { "field": "a", "type": "quantitative", "aggregate": "sum", "scale": { "type": scaleType } },
                            "y": { "field": "variety", "type": "nominal" },
                            "color": { "field": "site", "type": "nominal" }
                        },
                        "config": {
                            "mark": { "stacked": stacked }
                        }
                    };
                    chai_1.assert.isNull(stack_1.stack(spec.mark, spec.encoding, spec.config.mark.stacked));
                    chai_1.assert.isFalse(spec_1.isStacked(spec));
                });
            });
        });
    });
    it('should always be disabled if the aggregated axis has non-summative aggregate', function () {
        [undefined, stack_1.StackOffset.CENTER, stack_1.StackOffset.NONE, stack_1.StackOffset.ZERO, stack_1.StackOffset.NORMALIZE].forEach(function (stacked) {
            [aggregate_1.AggregateOp.AVERAGE, aggregate_1.AggregateOp.VARIANCE, aggregate_1.AggregateOp.Q3].forEach(function (aggregate) {
                mark_1.PRIMITIVE_MARKS.forEach(function (mark) {
                    var spec = {
                        "data": { "url": "data/barley.json" },
                        "mark": mark,
                        "encoding": {
                            "x": { "field": "a", "type": "quantitative", "aggregate": aggregate },
                            "y": { "field": "variety", "type": "nominal" },
                            "color": { "field": "site", "type": "nominal" }
                        },
                        "config": {
                            "mark": { "stacked": stacked }
                        }
                    };
                    chai_1.assert.isNull(stack_1.stack(spec.mark, spec.encoding, spec.config.mark.stacked));
                    chai_1.assert.isFalse(spec_1.isStacked(spec));
                });
            });
        });
    });
    describe('stack().groupbyChannel, .fieldChannel', function () {
        it('should be correct for horizontal', function () {
            [mark_1.BAR, mark_1.AREA].forEach(function (stackableMark) {
                var spec = {
                    "data": { "url": "data/barley.json" },
                    "mark": stackableMark,
                    "encoding": {
                        "x": { "aggregate": "sum", "field": "yield", "type": "quantitative" },
                        "y": { "field": "variety", "type": "nominal" },
                        "color": { "field": "site", "type": "nominal" }
                    }
                };
                var _stack = stack_1.stack(spec.mark, spec.encoding, undefined);
                chai_1.assert.equal(_stack.fieldChannel, channel_1.X);
                chai_1.assert.equal(_stack.groupbyChannel, channel_1.Y);
                chai_1.assert.isTrue(spec_1.isStacked(spec));
            });
        });
        it('should be correct for horizontal (single)', function () {
            [mark_1.BAR, mark_1.AREA].forEach(function (stackableMark) {
                var spec = {
                    "data": { "url": "data/barley.json" },
                    "mark": stackableMark,
                    "encoding": {
                        "x": { "aggregate": "sum", "field": "yield", "type": "quantitative" },
                        "color": { "field": "site", "type": "nominal" }
                    }
                };
                var _stack = stack_1.stack(spec.mark, spec.encoding, undefined);
                chai_1.assert.equal(_stack.fieldChannel, channel_1.X);
                chai_1.assert.equal(_stack.groupbyChannel, null);
                chai_1.assert.isTrue(spec_1.isStacked(spec));
            });
        });
        it('should be correct for vertical', function () {
            [mark_1.BAR, mark_1.AREA].forEach(function (stackableMark) {
                var spec = {
                    "data": { "url": "data/barley.json" },
                    "mark": stackableMark,
                    "encoding": {
                        "y": { "aggregate": "sum", "field": "yield", "type": "quantitative" },
                        "x": { "field": "variety", "type": "nominal" },
                        "color": { "field": "site", "type": "nominal" }
                    }
                };
                var _stack = stack_1.stack(spec.mark, spec.encoding, undefined);
                chai_1.assert.equal(_stack.fieldChannel, channel_1.Y);
                chai_1.assert.equal(_stack.groupbyChannel, channel_1.X);
                chai_1.assert.isTrue(spec_1.isStacked(spec));
            });
        });
        it('should be correct for vertical (single)', function () {
            [mark_1.BAR, mark_1.AREA].forEach(function (stackableMark) {
                var spec = {
                    "data": { "url": "data/barley.json" },
                    "mark": stackableMark,
                    "encoding": {
                        "y": { "aggregate": "sum", "field": "yield", "type": "quantitative" },
                        "color": { "field": "site", "type": "nominal" }
                    }
                };
                var _stack = stack_1.stack(spec.mark, spec.encoding, undefined);
                chai_1.assert.equal(_stack.fieldChannel, channel_1.Y);
                chai_1.assert.equal(_stack.groupbyChannel, null);
                chai_1.assert.isTrue(spec_1.isStacked(spec));
            });
        });
    });
    describe('stack().offset', function () {
        it('should be zero for stackable marks with at least of of the stack channel if stacked is unspecified', function () {
            [mark_1.BAR, mark_1.AREA].forEach(function (stackableMark) {
                var spec = {
                    "data": { "url": "data/barley.json" },
                    "mark": stackableMark,
                    "encoding": {
                        "x": { "aggregate": "sum", "field": "yield", "type": "quantitative" },
                        "y": { "field": "variety", "type": "nominal" },
                        "color": { "field": "site", "type": "nominal" }
                    }
                };
                chai_1.assert.equal(stack_1.stack(spec.mark, spec.encoding, undefined).offset, stack_1.StackOffset.ZERO);
                chai_1.assert.isTrue(spec_1.isStacked(spec));
            });
        });
        it('should be the specified stacked for stackable marks with at least one of the stack channel', function () {
            [stack_1.StackOffset.CENTER, stack_1.StackOffset.ZERO, stack_1.StackOffset.NORMALIZE].forEach(function (stacked) {
                [mark_1.BAR, mark_1.AREA].forEach(function (stackableMark) {
                    var spec = {
                        "data": { "url": "data/barley.json" },
                        "mark": stackableMark,
                        "encoding": {
                            "x": { "aggregate": "sum", "field": "yield", "type": "quantitative" },
                            "y": { "field": "variety", "type": "nominal" },
                            "color": { "field": "site", "type": "nominal" }
                        },
                        "config": {
                            "mark": { "stacked": stacked }
                        }
                    };
                    chai_1.assert.equal(stack_1.stack(spec.mark, spec.encoding, spec.config.mark.stacked).offset, stacked);
                    chai_1.assert.equal(spec_1.isStacked(spec), stack_1.StackOffset.NONE !== stacked);
                });
            });
        });
    });
});
//# sourceMappingURL=stack.test.js.map