"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

var _lodashFunctionBind = require("lodash/function/bind");

var _lodashFunctionBind2 = _interopRequireDefault(_lodashFunctionBind);

var _lodashCollectionIncludes = require("lodash/collection/includes");

var _lodashCollectionIncludes2 = _interopRequireDefault(_lodashCollectionIncludes);

var _lodashLangIsFunction = require("lodash/lang/isFunction");

var _lodashLangIsFunction2 = _interopRequireDefault(_lodashLangIsFunction);

var _react = require("react");

var _type = require("./type");

/**
 * Return a new validator based on `validator` but with the option to chain
 * `isRequired` onto the validation. This is nearly identical to how React
 * does it internally, but they don't expose their helper for us to use.
 * @param {Function} validator Validation function.
 * @returns {Function} Validator with `isRequired` option.
 */
var makeChainable = function makeChainable(validator) {
  /* eslint-disable max-params */
  var _chainable = function _chainable(isRequired, props, propName, componentName) {
    var value = props[propName];
    if (typeof value === "undefined" || value === null) {
      if (isRequired) {
        return new Error("Required `" + propName + "` was not specified in `" + componentName + "`.");
      }
      return null;
    }
    return validator(props, propName, componentName);
  };
  var chainable = (0, _lodashFunctionBind2["default"])(_chainable, null, false);
  chainable.isRequired = (0, _lodashFunctionBind2["default"])(_chainable, null, true);
  return chainable;
};

exports.makeChainable = makeChainable;
/**
 * Return a new validator which returns true
 * if and only if all validators passed as arguments return true.
 * Like React.propTypes.oneOfType, except "all" instead of "any"
 * @param {Array} validators Validation functions.
 * @returns {Function} Combined validator function
 */
var allOfType = function allOfType(validators) {
  return makeChainable(function (props, propName, componentName) {
    var error = validators.reduce(function (result, validator) {
      return result || validator(props, propName, componentName);
    }, undefined);
    if (error) {
      return error;
    }
  });
};

exports.allOfType = allOfType;
/**
 * Check that the value is a non-negative number.
 */
var nonNegative = makeChainable(function (props, propName, componentName) {
  var error = _react.PropTypes.number(props, propName, componentName);
  if (error) {
    return error;
  }
  var value = props[propName];
  if (value < 0) {
    return new Error("`" + propName + "` in `" + componentName + "` must be non-negative.");
  }
});

exports.nonNegative = nonNegative;
/**
 * Check that the value is an integer.
 */
var integer = makeChainable(function (props, propName, componentName) {
  var error = _react.PropTypes.number(props, propName, componentName);
  if (error) {
    return error;
  }
  var value = props[propName];
  if (value % 1 !== 0) {
    return new Error("`" + propName + "` in `" + componentName + "` must be an integer.");
  }
});

exports.integer = integer;
/**
 * Check that the value is an Array of two unique values.
 */
var domain = makeChainable(function (props, propName, componentName) {
  var error = _react.PropTypes.array(props, propName, componentName);
  if (error) {
    return error;
  }
  var value = props[propName];
  if (value.length !== 2 || value[1] === value[0]) {
    return new Error("`" + propName + "` in `" + componentName + "` must be an array of two unique numeric values.");
  }
});

exports.domain = domain;
/**
 * Check that the value looks like a d3 `scale` function.
 */
var scale = makeChainable(function (props, propName, componentName) {
  var supportedScaleStrings = ["linear", "time", "log", "sqrt"];
  var validScale = function validScale(scl) {
    if ((0, _lodashLangIsFunction2["default"])(scl)) {
      return (0, _lodashLangIsFunction2["default"])(scl.copy) && (0, _lodashLangIsFunction2["default"])(scl.domain) && (0, _lodashLangIsFunction2["default"])(scl.range);
    } else if (typeof scl === "string") {
      return (0, _lodashCollectionIncludes2["default"])(supportedScaleStrings, scl);
    }
    return false;
  };

  var value = props[propName];
  if (!validScale(value)) {
    return new Error("`" + propName + "` in `" + componentName + "` must be a d3 scale.");
  }
});

exports.scale = scale;
/**
 * Check that an array contains items of the same type.
 */
var homogeneousArray = makeChainable(function (props, propName, componentName) {
  var error = _react.PropTypes.array(props, propName, componentName);
  if (error) {
    return error;
  }
  var value = props[propName];
  if (value.length > 1) {
    var _constructor = (0, _type.getConstructor)(value[0]);
    for (var i = 1; i < value.length; i++) {
      var otherConstructor = (0, _type.getConstructor)(value[i]);
      if (_constructor !== otherConstructor) {
        var constructorName = (0, _type.getConstructorName)(value[0]);
        var otherConstructorName = (0, _type.getConstructorName)(value[i]);
        return new Error("Expected `" + propName + "` in `" + componentName + "` to be a " + ("homogeneous array, but found types `" + constructorName + "` and ") + ("`" + otherConstructorName + "`."));
      }
    }
  }
});
exports.homogeneousArray = homogeneousArray;