// Maps for number <-> hex string conversion
var byteToHex = [];
var hexToByte = {};

for (var i = 0; i < 256; i++) {
  byteToHex[i] = (i + 0x100).toString(16).substr(1);
  hexToByte[byteToHex[i]] = i;
}
/**
 * Parse a string UUID representation into it's component bytes.
 *
 * @param str - String UUID.
 * @param buf - Buffer to be filled with the bytes.
 * @param offset - Offset from the start of the buffer where the UUID bytes will be saved.
 *
 * @returns An array (or Uint8Array if supplied) of bytes.
 */


function parseUUID(str) {
  var buf = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : [];
  var offset = arguments.length > 2 ? arguments[2] : undefined;
  var i = buf && offset || 0;
  var ii = 0;
  str.toLowerCase().replace(/[0-9a-f]{2}/g, function (oct) {
    if (ii < 16) {
      // Don't overflow!
      buf[i + ii++] = hexToByte[oct];
    }

    return '';
  }); // Zero out remaining bytes if string was short

  while (ii < 16) {
    buf[i + ii++] = 0;
  }

  return buf;
}
/**
 * Represent binary UUID into it's string representation.
 *
 * @param buf - Buffer containing UUID bytes.
 * @param offset - Offset from the start of the buffer where the UUID is saved (not needed if the buffer starts with the UUID).
 *
 * @returns String representation of the UUID.
 */

function stringifyUUID(buf, offset) {
  var i = offset || 0;
  var bth = byteToHex;
  return bth[buf[i++]] + bth[buf[i++]] + bth[buf[i++]] + bth[buf[i++]] + '-' + bth[buf[i++]] + bth[buf[i++]] + '-' + bth[buf[i++]] + bth[buf[i++]] + '-' + bth[buf[i++]] + bth[buf[i++]] + '-' + bth[buf[i++]] + bth[buf[i++]] + bth[buf[i++]] + bth[buf[i++]] + bth[buf[i++]] + bth[buf[i++]];
}

/**
 * Generate 16 random bytes to be used as a base for UUID.
 *
 * @ignore
 */
var random = function () {
  if (typeof crypto !== 'undefined' && crypto.getRandomValues) {
    // WHATWG crypto-based RNG - http://wiki.whatwg.org/wiki/Crypto
    // Moderately fast, high quality
    var _rnds8 = new Uint8Array(16);

    return function whatwgRNG() {
      crypto.getRandomValues(_rnds8);
      return _rnds8;
    };
  } // Math.random()-based (RNG)
  //
  // If all else fails, use Math.random().
  // It's fast, but is of unspecified quality.


  var _rnds = new Array(16);

  return function () {
    for (var i = 0, r; i < 16; i++) {
      if ((i & 0x03) === 0) {
        r = Math.random() * 0x100000000;
      }

      _rnds[i] = r >>> ((i & 0x03) << 3) & 0xff;
    }

    return _rnds;
  }; //     uuid.js
  //
  //     Copyright (c) 2010-2012 Robert Kieffer
  //     MIT License - http://opensource.org/licenses/mit-license.php
  // Unique ID creation requires a high quality random # generator.  We feature
  // detect to determine the best RNG source, normalizing to a function that
  // returns 128-bits of randomness, since that's what's usually required
  // return require('./rng');
}();

var byteToHex$1 = [];

for (var i$1 = 0; i$1 < 256; i$1++) {
  byteToHex$1[i$1] = (i$1 + 0x100).toString(16).substr(1);
} // **`v1()` - Generate time-based UUID**
//
// Inspired by https://github.com/LiosK/UUID.js
// and http://docs.python.org/library/uuid.html
// random #'s we need to init node and clockseq


var seedBytes = random(); // Per 4.5, create and 48-bit node id, (47 random bits + multicast bit = 1)

var defaultNodeId = [seedBytes[0] | 0x01, seedBytes[1], seedBytes[2], seedBytes[3], seedBytes[4], seedBytes[5]]; // Per 4.2.2, randomize (14 bit) clockseq

var defaultClockseq = (seedBytes[6] << 8 | seedBytes[7]) & 0x3fff; // Previous uuid creation time

var lastMSecs = 0;
var lastNSecs = 0;
/**
 * UUIDv1 options.
 */

/**
 * Generate UUIDv1
 *
 * @param options - Options to be used instead of default values.
 * @param buf - If present the buffer will be filled with the generated UUID.
 * @param offset - Offset of the UUID from the start of the buffer.
 *
 * @returns UUIDv1
 */
function uuid1() {
  var options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
  var buf = arguments.length > 1 ? arguments[1] : undefined;
  var offset = arguments.length > 2 ? arguments[2] : undefined;
  var i = buf && offset || 0;
  var b = buf || [];
  var clockseq = options.clockseq !== undefined ? options.clockseq : defaultClockseq; // UUID timestamps are 100 nano-second units since the Gregorian epoch,
  // (1582-10-15 00:00).  JSNumbers aren't precise enough for this, so
  // time is handled internally as 'msecs' (integer milliseconds) and 'nsecs'
  // (100-nanoseconds offset from msecs) since unix epoch, 1970-01-01 00:00.

  var msecs = options.msecs !== undefined ? options.msecs : new Date().getTime(); // Per 4.2.1.2, use count of uuid's generated during the current clock
  // cycle to simulate higher resolution clock

  var nsecs = options.nsecs !== undefined ? options.nsecs : lastNSecs + 1; // Time since last uuid creation (in msecs)

  var dt = msecs - lastMSecs + (nsecs - lastNSecs) / 10000; // Per 4.2.1.2, Bump clockseq on clock regression

  if (dt < 0 && options.clockseq === undefined) {
    clockseq = clockseq + 1 & 0x3fff;
  } // Reset nsecs if clock regresses (new clockseq) or we've moved onto a new
  // time interval


  if ((dt < 0 || msecs > lastMSecs) && options.nsecs === undefined) {
    nsecs = 0;
  } // Per 4.2.1.2 Throw error if too many uuids are requested


  if (nsecs >= 10000) {
    throw new Error("uuid.v1(): Can't create more than 10M uuids/sec");
  }

  lastMSecs = msecs;
  lastNSecs = nsecs;
  defaultClockseq = clockseq; // Per 4.1.4 - Convert from unix epoch to Gregorian epoch

  msecs += 12219292800000; // `time_low`

  var tl = ((msecs & 0xfffffff) * 10000 + nsecs) % 0x100000000;
  b[i++] = tl >>> 24 & 0xff;
  b[i++] = tl >>> 16 & 0xff;
  b[i++] = tl >>> 8 & 0xff;
  b[i++] = tl & 0xff; // `time_mid`

  var tmh = msecs / 0x100000000 * 10000 & 0xfffffff;
  b[i++] = tmh >>> 8 & 0xff;
  b[i++] = tmh & 0xff; // `time_high_and_version`

  b[i++] = tmh >>> 24 & 0xf | 0x10; // include version

  b[i++] = tmh >>> 16 & 0xff; // `clock_seq_hi_and_reserved` (Per 4.2.2 - include variant)

  b[i++] = clockseq >>> 8 | 0x80; // `clock_seq_low`

  b[i++] = clockseq & 0xff; // `node`

  var node = options.node || defaultNodeId;

  for (var n = 0; n < 6; n++) {
    b[i + n] = node[n];
  }

  return buf ? buf : stringifyUUID(b);
}

/**
 * UUIDv4 options.
 */

/**
 * Generate UUIDv4
 *
 * @param options - Options to be used instead of default generated values.
 * String 'binary' is a shorthand for uuid4({}, new Array(16)).
 * @param buf - If present the buffer will be filled with the generated UUID.
 * @param offset - Offset of the UUID from the start of the buffer.
 *
 * @returns UUIDv4
 */
function uuid4() {
  var options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
  var buf = arguments.length > 1 ? arguments[1] : undefined;
  var offset = arguments.length > 2 ? arguments[2] : undefined;
  // Deprecated - 'format' argument, as supported in v1.2
  var i = buf && offset || 0;

  if (typeof options === 'string') {
    buf = options === 'binary' ? new Array(16) : undefined;
    options = {};
  }

  var rnds = options.random || (options.rng || random)(); // Per 4.4, set bits for version and `clock_seq_hi_and_reserved`

  rnds[6] = rnds[6] & 0x0f | 0x40;
  rnds[8] = rnds[8] & 0x3f | 0x80; // Copy bytes to buffer, if provided

  if (buf) {
    for (var ii = 0; ii < 16; ii++) {
      buf[i + ii] = rnds[ii];
    }
  }

  return buf || stringifyUUID(rnds);
}

// Rollup will complain about mixing default and named exports in UMD build,
// but since they both implement the same interface, there won't be any problems.

/**
 * API properties as used before ES2015 modules and TypeScript.
 */

var oldAPI = function oldAPI() {
  return uuid4.apply(void 0, arguments);
};

oldAPI.v1 = uuid1;
oldAPI.v4 = uuid4;
oldAPI.parse = parseUUID;
oldAPI.unparse = stringifyUUID;

export default oldAPI;
export { parseUUID as parse, parseUUID, stringifyUUID, stringifyUUID as unparse, uuid1, uuid4, uuid1 as v1, uuid4 as v4 };
