import chalk from "chalk";
import { spawn } from "child_process";
import pick from "lodash.pick";
import npmRunPath from "npm-run-path";
import path from "path";
import { Checker } from "./Checker.js";
import { RUNTIME_PUBLIC_PATH, runtimeCode, WS_CHECKER_RECONNECT_EVENT } from "./client/index.js";
import {
  ACTION_TYPES
} from "./types.js";
const sharedConfigKeys = ["enableBuild", "overlay"];
const buildInCheckerKeys = [
  "typescript",
  "vueTsc",
  "vls",
  "eslint",
  "stylelint"
];
async function createCheckers(userConfig, env) {
  const serveAndBuildCheckers = [];
  const sharedConfig = pick(userConfig, sharedConfigKeys);
  for (const name of buildInCheckerKeys) {
    if (!userConfig[name])
      continue;
    const { createServeAndBuild } = await import(`./checkers/${name}/main.js`);
    serveAndBuildCheckers.push(
      createServeAndBuild({ [name]: userConfig[name], ...sharedConfig }, env)
    );
  }
  return serveAndBuildCheckers;
}
function checker(userConfig) {
  const enableBuild = (userConfig == null ? void 0 : userConfig.enableBuild) ?? true;
  const enableOverlay = (userConfig == null ? void 0 : userConfig.overlay) !== false;
  const enableTerminal = (userConfig == null ? void 0 : userConfig.terminal) !== false;
  const overlayConfig = typeof (userConfig == null ? void 0 : userConfig.overlay) === "object" ? userConfig == null ? void 0 : userConfig.overlay : {};
  let resolvedRuntimePath = RUNTIME_PUBLIC_PATH;
  let checkers = [];
  let viteMode;
  let resolvedConfig;
  return {
    name: "vite-plugin-checker",
    __internal__checker: Checker,
    config: async (config, env) => {
      viteMode = env.command;
      checkers = await createCheckers(userConfig || {}, env);
      if (viteMode !== "serve")
        return;
      checkers.forEach((checker2) => {
        const workerConfig = checker2.serve.config;
        workerConfig({
          enableOverlay,
          enableTerminal,
          env
        });
      });
    },
    configResolved(config) {
      resolvedConfig = config;
      resolvedRuntimePath = config.base + RUNTIME_PUBLIC_PATH.slice(1);
    },
    buildEnd() {
      if (viteMode === "serve") {
        checkers.forEach((checker2) => {
          const { worker } = checker2.serve;
          worker.terminate();
        });
      }
    },
    resolveId(id) {
      if (viteMode === "serve") {
        if (id === RUNTIME_PUBLIC_PATH) {
          return id;
        }
      }
      return;
    },
    load(id) {
      if (viteMode === "serve") {
        if (id === RUNTIME_PUBLIC_PATH) {
          return runtimeCode;
        }
      }
      return;
    },
    transform(code, id, options) {
      if (id === RUNTIME_PUBLIC_PATH) {
        if (!resolvedConfig)
          return;
        const devBase = resolvedConfig.base;
        const hmrConfig = isObject(resolvedConfig.server.hmr) ? resolvedConfig.server.hmr : {};
        const host = hmrConfig.host || null;
        const protocol = hmrConfig.protocol || null;
        let port = (hmrConfig == null ? void 0 : hmrConfig.clientPort) || (hmrConfig == null ? void 0 : hmrConfig.port) || null;
        if (resolvedConfig.server.middlewareMode) {
          port || (port = 24678);
        }
        let hmrBase = devBase;
        if (hmrConfig == null ? void 0 : hmrConfig.path) {
          hmrBase = path.posix.join(hmrBase, hmrConfig.path);
        }
        return code.replace(/__HMR_PROTOCOL__/g, JSON.stringify(protocol)).replace(/__HMR_HOSTNAME__/g, JSON.stringify(host)).replace(/__HMR_PORT__/g, JSON.stringify(port)).replace(/__HMR_BASE__/g, JSON.stringify(hmrBase));
      }
      return null;
    },
    transformIndexHtml() {
      if (viteMode === "serve") {
        return [
          {
            tag: "script",
            attrs: { type: "module" },
            children: `import { inject } from "${resolvedRuntimePath}";
inject({
  overlayConfig: ${JSON.stringify(overlayConfig)},
  base: "${resolvedConfig == null ? void 0 : resolvedConfig.base}",
});`
          }
        ];
      }
      return;
    },
    buildStart: () => {
      if (viteMode !== "build" || !enableBuild)
        return;
      const localEnv = npmRunPath.env({
        env: process.env,
        cwd: process.cwd(),
        execPath: process.execPath
      });
      (async () => {
        const exitCodes = await Promise.all(
          checkers.map((checker2) => spawnChecker(checker2, userConfig, localEnv))
        );
        const exitCode = exitCodes.find((code) => code !== 0) ?? 0;
        if (exitCode !== 0 && !(resolvedConfig == null ? void 0 : resolvedConfig.build.watch)) {
          process.exit(exitCode);
        }
      })();
    },
    configureServer(server) {
      let connectedTimes = 0;
      let latestOverlayErrors = new Array(checkers.length);
      checkers.forEach((checker2, index) => {
        const { worker, configureServer: workerConfigureServer } = checker2.serve;
        workerConfigureServer({ root: server.config.root });
        worker.on("message", (action) => {
          if (action.type === ACTION_TYPES.overlayError) {
            latestOverlayErrors[index] = action.payload;
            if (action.payload) {
              server.ws.send(action.payload);
            }
          } else if (action.type === ACTION_TYPES.console) {
            Checker.log(action);
          }
        });
      });
      return () => {
        if (server.ws.on) {
          server.ws.on("connection", () => {
            connectedTimes++;
            if (connectedTimes > 1) {
              server.ws.send({
                type: "custom",
                event: WS_CHECKER_RECONNECT_EVENT,
                data: latestOverlayErrors.filter(Boolean)
              });
            }
          });
        } else {
          setTimeout(() => {
            console.warn(
              chalk.yellow(
                "[vite-plugin-checker]: `server.ws.on` is introduced to Vite in 2.6.8, see [PR](https://github.com/vitejs/vite/pull/5273) and [changelog](https://github.com/vitejs/vite/blob/main/packages/vite/CHANGELOG.md#268-2021-10-18). \nvite-plugin-checker relies on `server.ws.on` to bring diagnostics back after a full reload and it' not available for you now due to the old version of Vite. You can upgrade Vite to latest version to eliminate this warning."
              )
            );
          }, 5e3);
        }
        server.middlewares.use((req, res, next) => {
          next();
        });
      };
    }
  };
}
function spawnChecker(checker2, userConfig, localEnv) {
  return new Promise((resolve) => {
    const buildBin = checker2.build.buildBin;
    const finalBin = typeof buildBin === "function" ? buildBin(userConfig) : buildBin;
    const proc = spawn(...finalBin, {
      cwd: process.cwd(),
      stdio: "inherit",
      env: localEnv,
      shell: true
    });
    proc.on("exit", (code) => {
      if (code !== null && code !== 0) {
        resolve(code);
      } else {
        resolve(0);
      }
    });
  });
}
function isObject(value) {
  return Object.prototype.toString.call(value) === "[object Object]";
}
var main_default = checker;
export {
  checker,
  main_default as default
};
//# sourceMappingURL=main.js.map