// src/utils.ts
import { readFileSync } from "fs";
import TauriCli from "@tauri-apps/cli";
import fg from "fast-glob";
function getTauriConfPath() {
  const tauriDepthEnv = process.env.TAURI_PATH_DEPTH;
  const deep = tauriDepthEnv ? parseInt(tauriDepthEnv) : 3;
  return fg.sync("**/tauri.conf.(json|json5)", {
    absolute: true,
    unique: true,
    ignore: ["node_modules/**", "target/**"],
    deep
  })[0];
}
function getPackageJson() {
  return JSON.parse(readFileSync("package.json", "utf8"));
}
var GREEN = "\x1B[32m";
var BOLD = "\x1B[1m";
var DIM = "\x1B[2m";
var RESET = "\x1B[0m";
function confirm(msg) {
  return new Promise((resolve, reject) => {
    const question = `${GREEN}? ${RESET}${BOLD}${msg}${RESET} ${DIM}(Y/n)${RESET}`;
    process.stdout.write(question);
    process.stdin.setRawMode(true);
    process.stdin.once("data", (data) => {
      process.stdout.write(data.toString());
      process.stdout.write("\n");
      process.stdin.setRawMode(false);
      const key = data.toString();
      if (key === "y" || key === "Y") {
        resolve(true);
      } else if (key === "n" || key === "N") {
        resolve(false);
      } else {
        process.exit(1);
      }
    });
  });
}
async function initTauri(args) {
  const confirmed = await confirm(
    "Couldn't find a Tauri project in current directory, would you like to initialize a new one?"
  );
  if (!confirmed)
    process.exit(0);
  console.log("Initializing Tauri...");
  const pkgName = getPackageJson().name;
  await TauriCli.run(
    [
      "init",
      "--app-name",
      pkgName ?? "tauri-app",
      "--window-title",
      (pkgName ?? "tauri-app") + " window",
      "--dist-dir",
      `Injected by vite-plugin-tauri, you can change this if you want to use tauri cli directly`,
      "--dev-path",
      `Injected by vite-plugin-tauri, you can change this if you want to use tauri cli directly`,
      "--before-build-command",
      "",
      "--before-dev-command",
      "",
      ...args ?? []
    ],
    "vite-tauri"
  );
  console.log("Tauri initialized.");
}

// src/index.ts
import TauriCli2 from "@tauri-apps/cli";
import path, { dirname } from "path";
function tauri(options) {
  let viteConfig;
  return [
    {
      name: "vite-plugin-tauri:serve",
      apply: "serve",
      enforce: "post",
      configResolved(config) {
        viteConfig = config;
      },
      async configureServer(server) {
        var _a;
        if (!getTauriConfPath())
          await initTauri();
        (_a = server.httpServer) == null ? void 0 : _a.once("listening", () => {
          var _a2;
          const localhosts = [
            "localhost",
            "127.0.0.1",
            "::1",
            "0000:0000:0000:0000:0000:0000:0000:0001"
          ];
          const address = (_a2 = server.httpServer) == null ? void 0 : _a2.address();
          if (!address || typeof address === "string") {
            console.error("Unexpected dev server address", address);
            process.exit(1);
          }
          const protocol = server.config.server.https ? "https" : "http";
          const host = localhosts.includes(address.address) ? "localhost" : address.address;
          const port = address.port;
          TauriCli2.run(
            [
              "dev",
              "--config",
              JSON.stringify({
                build: {
                  devPath: `${protocol}://${host}:${port}`
                }
              }),
              ...(options == null ? void 0 : options.debug) ? ["--debug"] : []
            ],
            "vite-plugin-tauri"
          );
        });
      }
    },
    {
      name: "vite-plugin-tauri:build",
      apply: "build",
      enforce: "post",
      configResolved(config) {
        viteConfig = config;
      },
      async closeBundle() {
        let tauriConfPath = getTauriConfPath();
        if (!tauriConfPath) {
          await initTauri();
          tauriConfPath = getTauriConfPath();
        }
        await TauriCli2.run(
          [
            "build",
            "--config",
            JSON.stringify({
              build: {
                distDir: path.relative(
                  dirname(tauriConfPath),
                  path.resolve(viteConfig.build.outDir)
                )
              }
            }),
            ...(options == null ? void 0 : options.debug) ? ["--debug"] : []
          ],
          "vite-plugin-tauri"
        );
      }
    }
  ];
}
export {
  tauri
};
