import * as vite from 'vite';
import { UserConfig as UserConfig$1, AliasOptions, ServerOptions, BuildOptions } from 'vite';
import { Options } from '@vitejs/plugin-vue';
import MarkdownIt from 'markdown-it';
import { Theme } from 'shiki';
import anchor from 'markdown-it-anchor';

declare namespace DefaultTheme {
  export interface Config {
    /**
     * The logo file of the site.
     *
     * @example '/logo.svg'
     */
    logo?: string

    /**
     * The nav items.
     */
    nav?: NavItem[]

    /**
     * The sidebar items.
     */
    sidebar?: Sidebar

    /**
     * Info for the edit link. If it's undefined, the edit link feature will
     * be disabled.
     */
    editLink?: EditLink

    /**
     * The social links to be displayed at the end of the nav bar. Perfect for
     * placing links to social services such as GitHub, Twitter, Facebook, etc.
     */
    socialLinks?: SocialLink[]

    /**
     * The footer configuration.
     */
    footer?: Footer

    /**
     * Adds locale menu to the nav. This option should be used when you have
     * your translated sites outside of the project.
     */
    localeLinks?: LocaleLinks

    /**
     * The algolia options. Leave it undefined to disable the search feature.
     */
    algolia?: AlgoliaSearchOptions

    /**
     * The carbon ads options. Leave it undefined to disable the ads feature.
     */
    carbonAds?: CarbonAdsOptions
  }

  // nav -----------------------------------------------------------------------

  export type NavItem = NavItemWithLink | NavItemWithChildren

  export type NavItemWithLink = {
    text: string
    link: string

    /**
     * `activeMatch` is expected to be a regex string. We can't use actual
     * RegExp object here because it isn't serializable
     */
    activeMatch?: string
  }

  export interface NavItemWithChildren {
    text?: string
    items: NavItemWithLink[]
  }

  // sidebar -------------------------------------------------------------------

  export type Sidebar = SidebarGroup[] | SidebarMulti

  export interface SidebarMulti {
    [path: string]: SidebarGroup[]
  }

  export interface SidebarGroup {
    text: string
    items: SidebarItem[]

    /**
     * If `true`, toggle button is shown.
     *
     * @default false
     */
    collapsible?: boolean

    /**
     * If `true`, collapsible group is collapsed by default.
     *
     * @default false
     */
    collapsed?: boolean
  }

  export interface SidebarItem {
    text: string
    link: string
  }

  // edit link -----------------------------------------------------------------

  export interface EditLink {
    /**
     * Repo of the site.
     *
     * @example 'vuejs/docs'
     */
    repo: string

    /**
     * Branch of the repo.
     *
     * @default 'main'
     */
    branch?: string

    /**
     * If your docs are not at the root of the repo.
     *
     * @example 'docs'
     */
    dir?: string

    /**
     * Custom text for edit link.
     *
     * @default 'Edit this page'
     */
    text?: string
  }

  // social link ---------------------------------------------------------------

  export interface SocialLink {
    icon: SocialLinkIcon
    link: string
  }

  export type SocialLinkIcon =
    | 'discord'
    | 'facebook'
    | 'github'
    | 'instagram'
    | 'linkedin'
    | 'slack'
    | 'twitter'
    | 'youtube'

  // footer --------------------------------------------------------------------

  export interface Footer {
    message?: string
    copyright?: string
  }

  // locales -------------------------------------------------------------------

  export interface LocaleLinks {
    text: string
    items: LocaleLink[]
  }

  export interface LocaleLink {
    text: string
    link: string
  }

  // algolia ------------------------------------------------------------------

  /**
   * The Algolia search options. Partially copied from
   * `@docsearch/react/dist/esm/DocSearch.d.ts`
   */
  export interface AlgoliaSearchOptions {
    appId: string
    apiKey: string
    indexName: string
    placeholder?: string
    searchParameters?: any
    disableUserPersonalization?: boolean
    initialQuery?: string
  }

  // carbon ads ----------------------------------------------------------------

  export interface CarbonAdsOptions {
    code: string
    placement: string
  }
}

// types shared between server and client


interface Header {
  level: number
  title: string
  slug: string
}

interface SiteData<ThemeConfig = any> {
  base: string

  /**
   * Language of the site as it should be set on the `html` element.
   *
   * @example `en-US`, `zh-CN`
   */
  lang: string

  title: string
  titleTemplate?: string | boolean
  description: string
  head: HeadConfig[]
  appearance: boolean
  themeConfig: ThemeConfig
  scrollOffset: number | string
  locales: Record<string, LocaleConfig>

  /**
   * Available locales for the site when it has defined `locales` in its
   * `themeConfig`. This object is otherwise empty. Keys are paths like `/` or
   * `/zh/`.
   */
  langs: Record<
    string,
    {
      /**
       * Lang attribute as set on the `<html>` element.
       * @example `en-US`, `zh-CN`
       */
      lang: string
      /**
       * Label to display in the language menu.
       * @example `English`, `简体中文`
       */
      label: string
    }
  >
}

type HeadConfig =
  | [string, Record<string, string>]
  | [string, Record<string, string>, string]

interface LocaleConfig {
  lang: string
  title?: string
  titleTemplate?: string | boolean
  description?: string
  head?: HeadConfig[]
  label?: string
  selectText?: string
}

declare function resolveSiteDataByRoute(siteData: SiteData, route: string): SiteData;

interface MarkdownOptions extends MarkdownIt.Options {
    lineNumbers?: boolean;
    config?: (md: MarkdownIt) => void;
    anchor?: {
        permalink?: anchor.AnchorOptions['permalink'];
    };
    attrs?: {
        leftDelimiter?: string;
        rightDelimiter?: string;
        allowedAttributes?: string[];
        disable?: boolean;
    };
    theme?: Theme;
    toc?: any;
    externalLinks?: Record<string, string>;
}
interface MarkdownParsedData {
    hoistedTags?: string[];
    links?: string[];
    headers?: Header[];
}
interface MarkdownRenderer extends MarkdownIt {
    __path: string;
    __relativePath: string;
    __data: MarkdownParsedData;
}

declare const createMarkdownRenderer: (srcDir: string, options: MarkdownOptions | undefined, base: string) => Promise<MarkdownRenderer>;

interface UserConfig<ThemeConfig = any> {
    extends?: RawConfigExports<ThemeConfig>;
    base?: string;
    lang?: string;
    title?: string;
    titleTemplate?: string | boolean;
    description?: string;
    head?: HeadConfig[];
    appearance?: boolean;
    themeConfig?: ThemeConfig;
    locales?: Record<string, LocaleConfig>;
    markdown?: MarkdownOptions;
    lastUpdated?: boolean;
    /**
     * Options to pass on to `@vitejs/plugin-vue`
     */
    vue?: Options;
    /**
     * Vite config
     */
    vite?: UserConfig$1;
    srcDir?: string;
    srcExclude?: string[];
    outDir?: string;
    shouldPreload?: (link: string, page: string) => boolean;
    /**
     * Configure the scroll offset when the theme has a sticky header.
     * Can be a number or a selector element to get the offset from.
     */
    scrollOffset?: number | string;
    /**
     * Enable MPA / zero-JS mode
     * @experimental
     */
    mpa?: boolean;
}
declare type RawConfigExports<ThemeConfig = any> = UserConfig<ThemeConfig> | Promise<UserConfig<ThemeConfig>> | (() => UserConfig<ThemeConfig> | Promise<UserConfig<ThemeConfig>>);
interface SiteConfig<ThemeConfig = any> extends Pick<UserConfig, 'markdown' | 'vue' | 'vite' | 'shouldPreload' | 'mpa' | 'lastUpdated'> {
    root: string;
    srcDir: string;
    site: SiteData<ThemeConfig>;
    configPath: string | undefined;
    themeDir: string;
    outDir: string;
    tempDir: string;
    alias: AliasOptions;
    pages: string[];
}
/**
 * Type config helper
 */
declare function defineConfig(config: UserConfig<DefaultTheme.Config>): UserConfig<DefaultTheme.Config>;
/**
 * Type config helper for custom theme config
 */
declare function defineConfigWithTheme<ThemeConfig>(config: UserConfig<ThemeConfig>): UserConfig<ThemeConfig>;
declare function resolveConfig(root?: string, command?: 'serve' | 'build', mode?: string): Promise<SiteConfig>;
declare function resolveSiteData(root: string, userConfig?: UserConfig, command?: 'serve' | 'build', mode?: string): Promise<SiteData>;

declare function createServer(root?: string, serverOptions?: ServerOptions): Promise<vite.ViteDevServer>;

declare function build(root: string, buildOptions?: BuildOptions & {
    mpa?: string;
}): Promise<void>;

interface ServeOptions {
    root?: string;
    port?: number;
}
declare function serve(options?: ServeOptions): Promise<void>;

export { DefaultTheme, HeadConfig, Header, LocaleConfig, MarkdownOptions, MarkdownParsedData, MarkdownRenderer, RawConfigExports, ServeOptions, SiteConfig, SiteData, UserConfig, build, createMarkdownRenderer, createServer, defineConfig, defineConfigWithTheme, resolveConfig, resolveSiteData, resolveSiteDataByRoute, serve };
