'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = codePointAt;

var _surrogate_pair = require('./helper/string/surrogate_pair');

var _coerce_to_number = require('./helper/number/coerce_to_number');

var _coerce_to_number2 = _interopRequireDefault(_coerce_to_number);

var _coerce_to_string = require('./helper/string/coerce_to_string');

var _coerce_to_string2 = _interopRequireDefault(_coerce_to_string);

var _nan_default = require('./helper/number/nan_default');

var _nan_default2 = _interopRequireDefault(_nan_default);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Get the Unicode code point value of the character at `position`. <br/>
 * If a valid UTF-16 <a href="https://rainsoft.io/what-every-javascript-developer-should-know-about-unicode/#24surrogatepairs">
 * surrogate pair</a> starts at `position`, the
 * <a href="https://rainsoft.io/what-every-javascript-developer-should-know-about-unicode/#astralplanes">astral code point</a>
 * value at `position` is returned.
 *
 * @function codePointAt
 * @static
 * @since 1.0.0
 * @memberOf Chop
 * @param  {string} [subject=''] The string to extract from.
 * @param  {number} position The position to get the code point number.
 * @return {number} Returns a non-negative number less than or equal to `0x10FFFF`.
 * @example
 * v.codePointAt('rain', 1);
 * // => 97, or 0x0061
 *
 * v.codePointAt('\uD83D\uDE00 is smile', 0); // or '😀 is smile'
 * // => 128512, or 0x1F600
 */
function codePointAt(subject, position) {
  var subjectString = (0, _coerce_to_string2.default)(subject);
  var subjectStringLength = subjectString.length;
  var positionNumber = (0, _coerce_to_number2.default)(position);
  positionNumber = (0, _nan_default2.default)(positionNumber, 0);
  if (positionNumber < 0 || positionNumber >= subjectStringLength) {
    return undefined;
  }
  var firstCodePoint = subjectString.charCodeAt(positionNumber);
  var secondCodePoint = void 0;
  if ((0, _surrogate_pair.isHighSurrogate)(firstCodePoint) && subjectStringLength > positionNumber + 1) {
    secondCodePoint = subjectString.charCodeAt(positionNumber + 1);
    if ((0, _surrogate_pair.isLowSurrogate)(secondCodePoint)) {
      return (0, _surrogate_pair.getAstralNumberFromSurrogatePair)(firstCodePoint, secondCodePoint);
    }
  }
  return firstCodePoint;
}
module.exports = exports['default'];