"use strict";
/* --------------------------------------------------------------------------------------------
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 * ------------------------------------------------------------------------------------------ */
/// <reference path="../../typings/vscode.proposed.tabs.d.ts" />
/// <reference path="../../typings/vscode.proposed.openEditors.d.ts" />
Object.defineProperty(exports, "__esModule", { value: true });
exports.DiagnosticFeature = exports.vsdiag = void 0;
const vscode_1 = require("vscode");
const vscode_languageserver_protocol_1 = require("vscode-languageserver-protocol");
const uuid_1 = require("./utils/uuid");
const client_1 = require("./client");
function ensure(target, key) {
    if (target[key] === void 0) {
        target[key] = {};
    }
    return target[key];
}
var vsdiag;
(function (vsdiag) {
    let DocumentDiagnosticReportKind;
    (function (DocumentDiagnosticReportKind) {
        DocumentDiagnosticReportKind["full"] = "full";
        DocumentDiagnosticReportKind["unChanged"] = "unChanged";
    })(DocumentDiagnosticReportKind = vsdiag.DocumentDiagnosticReportKind || (vsdiag.DocumentDiagnosticReportKind = {}));
})(vsdiag = exports.vsdiag || (exports.vsdiag = {}));
var RequestStateKind;
(function (RequestStateKind) {
    RequestStateKind["active"] = "open";
    RequestStateKind["reschedule"] = "reschedule";
    RequestStateKind["outDated"] = "drop";
})(RequestStateKind || (RequestStateKind = {}));
class EditorTracker {
    constructor() {
        this.open = new Set();
        const openTabsHandler = () => {
            this.open.clear();
            // New API
            if (vscode_1.window.tabs !== undefined) {
                for (const tab of vscode_1.window.tabs) {
                    if (tab.resource !== undefined) {
                        this.open.add(tab.resource.toString());
                    }
                }
                // Old pre 1.61 API
            }
            else if (vscode_1.window.openEditors !== undefined) {
                for (const info of vscode_1.window.openEditors) {
                    if (info.resource !== undefined) {
                        this.open.add(info.resource.toString());
                    }
                }
            }
        };
        openTabsHandler();
        if (vscode_1.window.onDidChangeTabs !== undefined) {
            this.disposable = vscode_1.window.onDidChangeTabs(openTabsHandler);
        }
        else if (vscode_1.window.onDidChangeOpenEditors !== undefined) {
            this.disposable = vscode_1.window.onDidChangeOpenEditors(openTabsHandler);
        }
        else {
            this.disposable = { dispose: () => { } };
        }
    }
    dispose() {
        this.disposable.dispose();
    }
    isActive(textDocument) {
        return vscode_1.window.activeTextEditor?.document === textDocument;
    }
    isVisible(textDocument) {
        return this.open.has(textDocument.uri.toString());
    }
}
var PullState;
(function (PullState) {
    PullState[PullState["document"] = 1] = "document";
    PullState[PullState["workspace"] = 2] = "workspace";
})(PullState || (PullState = {}));
class DocumentPullStateTracker {
    constructor() {
        this.documentPullStates = new Map();
        this.workspacePullStates = new Map();
    }
    track(kind, document, arg1) {
        const states = kind === PullState.document ? this.documentPullStates : this.workspacePullStates;
        const [key, uri, version] = typeof document === 'string'
            ? [document, vscode_1.Uri.parse(document), arg1]
            : [document.uri.toString(), document.uri, document.version];
        let state = states.get(key);
        if (state === undefined) {
            state = { document: uri, pulledVersion: version, resultId: undefined };
            states.set(key, state);
        }
        return state;
    }
    update(kind, document, arg1, arg2) {
        const states = kind === PullState.document ? this.documentPullStates : this.workspacePullStates;
        const [key, uri, version, resultId] = typeof document === 'string'
            ? [document, vscode_1.Uri.parse(document), arg1, arg2]
            : [document.uri.toString(), document.uri, document.version, arg1];
        let state = states.get(key);
        if (state === undefined) {
            state = { document: uri, pulledVersion: version, resultId };
            states.set(key, state);
        }
        else {
            state.pulledVersion = version;
            state.resultId = resultId;
        }
    }
    unTrack(kind, textDocument) {
        const states = kind === PullState.document ? this.documentPullStates : this.workspacePullStates;
        states.delete(textDocument.uri.toString());
    }
    tracks(kind, document) {
        const key = typeof document === 'string' ? document : document.uri.toString();
        const states = kind === PullState.document ? this.documentPullStates : this.workspacePullStates;
        return states.has(key);
    }
    getResultId(kind, textDocument) {
        const states = kind === PullState.document ? this.documentPullStates : this.workspacePullStates;
        return states.get(textDocument.uri.toString())?.resultId;
    }
    getAllResultIds() {
        const result = [];
        for (let [uri, value] of this.workspacePullStates) {
            if (this.documentPullStates.has(uri)) {
                value = this.documentPullStates.get(uri);
            }
            if (value.resultId !== undefined) {
                result.push({ uri, value: value.resultId });
            }
        }
        return result;
    }
}
class DiagnosticRequestor {
    constructor(client, editorTracker, options) {
        this.client = client;
        this.editorTracker = editorTracker;
        this.options = options;
        this.isDisposed = false;
        this.onDidChangeDiagnosticsEmitter = new vscode_1.EventEmitter();
        this.provider = this.createProvider();
        this.diagnostics = vscode_1.languages.createDiagnosticCollection(options.identifier);
        this.openRequests = new Map();
        this.documentStates = new DocumentPullStateTracker();
        this.workspaceErrorCounter = 0;
    }
    knows(kind, textDocument) {
        return this.documentStates.tracks(kind, textDocument);
    }
    pull(textDocument, cb) {
        this.pullAsync(textDocument).then(() => {
            if (cb) {
                cb();
            }
        }, (error) => {
            this.client.error(`Document pull failed for text document ${textDocument.uri.toString()}`, error, false);
        });
    }
    async pullAsync(textDocument) {
        const key = textDocument.uri.toString();
        const version = textDocument.version;
        const currentRequestState = this.openRequests.get(key);
        const documentState = this.documentStates.track(PullState.document, textDocument);
        if (currentRequestState === undefined) {
            const tokenSource = new vscode_1.CancellationTokenSource();
            this.openRequests.set(key, { state: RequestStateKind.active, version: version, textDocument, tokenSource });
            let report;
            let afterState;
            try {
                report = await this.provider.provideDiagnostics(textDocument, documentState.resultId, tokenSource.token) ?? { kind: vsdiag.DocumentDiagnosticReportKind.full, items: [] };
            }
            catch (error) {
                if (error instanceof client_1.LSPCancellationError && vscode_languageserver_protocol_1.Proposed.DiagnosticServerCancellationData.is(error.data) && error.data.retriggerRequest === false) {
                    afterState = { state: RequestStateKind.outDated, textDocument };
                }
                if (afterState === undefined && error instanceof vscode_1.CancellationError) {
                    afterState = { state: RequestStateKind.reschedule, textDocument };
                }
                else {
                    throw error;
                }
            }
            afterState = afterState ?? this.openRequests.get(key);
            if (afterState === undefined) {
                // This shouldn't happen. Log it
                this.client.error(`Lost request state in diagnostic pull model. Clearing diagnostics for ${key}`);
                this.diagnostics.delete(textDocument.uri);
                return;
            }
            this.openRequests.delete(key);
            if (!this.editorTracker.isVisible(textDocument)) {
                this.documentStates.unTrack(PullState.document, textDocument);
                return;
            }
            if (afterState.state === RequestStateKind.outDated) {
                return;
            }
            // report is only undefined if the request has thrown.
            if (report !== undefined) {
                if (report.kind === vsdiag.DocumentDiagnosticReportKind.full) {
                    this.diagnostics.set(textDocument.uri, report.items);
                }
                documentState.pulledVersion = version;
                documentState.resultId = report.resultId;
            }
            if (afterState.state === RequestStateKind.reschedule) {
                this.pull(textDocument);
            }
        }
        else {
            if (currentRequestState.state === RequestStateKind.active) {
                // Cancel the current request and reschedule a new one when the old one returned.
                currentRequestState.tokenSource.cancel();
                this.openRequests.set(key, { state: RequestStateKind.reschedule, textDocument: currentRequestState.textDocument });
            }
            else if (currentRequestState.state === RequestStateKind.outDated) {
                this.openRequests.set(key, { state: RequestStateKind.reschedule, textDocument: currentRequestState.textDocument });
            }
        }
    }
    cleanupPull(textDocument) {
        const key = textDocument.uri.toString();
        const request = this.openRequests.get(key);
        if (this.options.workspaceDiagnostics || this.options.interFileDependencies) {
            if (request !== undefined) {
                this.openRequests.set(key, { state: RequestStateKind.reschedule, textDocument: textDocument });
            }
            else {
                this.pull(textDocument);
            }
        }
        else {
            if (request !== undefined) {
                if (request.state === RequestStateKind.active) {
                    request.tokenSource.cancel();
                }
                this.openRequests.set(key, { state: RequestStateKind.outDated, textDocument: textDocument });
            }
            this.diagnostics.delete(textDocument.uri);
        }
    }
    pullWorkspace() {
        this.pullWorkspaceAsync().then(() => {
            this.workspaceTimeout = (0, vscode_languageserver_protocol_1.RAL)().timer.setTimeout(() => {
                this.pullWorkspace();
            }, 2000);
        }, (error) => {
            if (!(error instanceof client_1.LSPCancellationError) && !vscode_languageserver_protocol_1.Proposed.DiagnosticServerCancellationData.is(error.data)) {
                this.client.error(`Workspace diagnostic pull failed.`, error, false);
                this.workspaceErrorCounter++;
            }
            if (this.workspaceErrorCounter <= 5) {
                this.workspaceTimeout = (0, vscode_languageserver_protocol_1.RAL)().timer.setTimeout(() => {
                    this.pullWorkspace();
                }, 2000);
            }
        });
    }
    async pullWorkspaceAsync() {
        if (!this.provider.provideWorkspaceDiagnostics) {
            return;
        }
        if (this.workspaceCancellation !== undefined) {
            this.workspaceCancellation.cancel();
            this.workspaceCancellation = undefined;
        }
        this.workspaceCancellation = new vscode_1.CancellationTokenSource();
        const previousResultIds = this.documentStates.getAllResultIds().map((item) => {
            return {
                uri: this.client.protocol2CodeConverter.asUri(item.uri),
                value: item.value
            };
        });
        await this.provider.provideWorkspaceDiagnostics(previousResultIds, this.workspaceCancellation.token, (chunk) => {
            if (!chunk || this.isDisposed) {
                return;
            }
            for (const item of chunk.items) {
                if (item.kind === vsdiag.DocumentDiagnosticReportKind.full) {
                    // Favour document pull result over workspace results. So skip if it is tracked
                    // as a document result.
                    if (!this.documentStates.tracks(PullState.document, item.uri.toString())) {
                        this.diagnostics.set(item.uri, item.items);
                    }
                }
                this.documentStates.update(PullState.workspace, item.uri.toString(), item.version ?? undefined, item.resultId);
            }
        });
    }
    createProvider() {
        const result = {
            onDidChangeDiagnostics: this.onDidChangeDiagnosticsEmitter.event,
            provideDiagnostics: (textDocument, previousResultId, token) => {
                const provideDiagnostics = (textDocument, previousResultId, token) => {
                    const params = {
                        identifier: this.options.identifier,
                        textDocument: { uri: this.client.code2ProtocolConverter.asUri(textDocument.uri) },
                        previousResultId: previousResultId
                    };
                    return this.client.sendRequest(vscode_languageserver_protocol_1.Proposed.DocumentDiagnosticRequest.type, params, token).then((result) => {
                        if (result === undefined || result === null || this.isDisposed) {
                            return { kind: vsdiag.DocumentDiagnosticReportKind.full, items: [] };
                        }
                        if (result.kind === vscode_languageserver_protocol_1.Proposed.DocumentDiagnosticReportKind.full) {
                            return { kind: vsdiag.DocumentDiagnosticReportKind.full, resultId: result.resultId, items: this.client.protocol2CodeConverter.asDiagnostics(result.items) };
                        }
                        else {
                            return { kind: vsdiag.DocumentDiagnosticReportKind.unChanged, resultId: result.resultId };
                        }
                    }, (error) => {
                        return this.client.handleFailedRequest(vscode_languageserver_protocol_1.Proposed.DocumentDiagnosticRequest.type, token, error, { kind: vsdiag.DocumentDiagnosticReportKind.full, items: [] });
                    });
                };
                const middleware = this.client.clientOptions.middleware;
                return middleware.provideDiagnostics
                    ? middleware.provideDiagnostics(textDocument, previousResultId, token, provideDiagnostics)
                    : provideDiagnostics(textDocument, previousResultId, token);
            }
        };
        if (this.options.workspaceDiagnostics) {
            result.provideWorkspaceDiagnostics = (resultIds, token, resultReporter) => {
                const convertReport = (report) => {
                    if (report.kind === vscode_languageserver_protocol_1.Proposed.DocumentDiagnosticReportKind.full) {
                        return {
                            kind: vsdiag.DocumentDiagnosticReportKind.full,
                            uri: this.client.protocol2CodeConverter.asUri(report.uri),
                            resultId: report.resultId,
                            version: report.version,
                            items: this.client.protocol2CodeConverter.asDiagnostics(report.items)
                        };
                    }
                    else {
                        return {
                            kind: vsdiag.DocumentDiagnosticReportKind.unChanged,
                            uri: this.client.protocol2CodeConverter.asUri(report.uri),
                            resultId: report.resultId,
                            version: report.version
                        };
                    }
                };
                const convertPreviousResultIds = (resultIds) => {
                    const converted = [];
                    for (const item of resultIds) {
                        converted.push({ uri: this.client.code2ProtocolConverter.asUri(item.uri), value: item.value });
                    }
                    return converted;
                };
                const provideDiagnostics = (resultIds, token) => {
                    const partialResultToken = (0, uuid_1.generateUuid)();
                    const disposable = this.client.onProgress(vscode_languageserver_protocol_1.Proposed.WorkspaceDiagnosticRequest.partialResult, partialResultToken, (partialResult) => {
                        if (partialResult === undefined || partialResult === null) {
                            resultReporter(null);
                            return;
                        }
                        const converted = {
                            items: []
                        };
                        for (const item of partialResult.items) {
                            converted.items.push(convertReport(item));
                        }
                        resultReporter(converted);
                    });
                    const params = {
                        identifier: this.options.identifier,
                        previousResultIds: convertPreviousResultIds(resultIds),
                        partialResultToken: partialResultToken
                    };
                    return this.client.sendRequest(vscode_languageserver_protocol_1.Proposed.WorkspaceDiagnosticRequest.type, params, token).then((result) => {
                        const converted = {
                            items: []
                        };
                        for (const item of result.items) {
                            converted.items.push(convertReport(item));
                        }
                        disposable.dispose();
                        resultReporter(converted);
                        return { items: [] };
                    }, (error) => {
                        disposable.dispose();
                        return this.client.handleFailedRequest(vscode_languageserver_protocol_1.Proposed.DocumentDiagnosticRequest.type, token, error, { items: [] });
                    });
                };
                const middleware = this.client.clientOptions.middleware;
                return middleware.provideWorkspaceDiagnostics
                    ? middleware.provideWorkspaceDiagnostics(resultIds, token, resultReporter, provideDiagnostics)
                    : provideDiagnostics(resultIds, token, resultReporter);
            };
        }
        return result;
    }
    dispose() {
        this.isDisposed = true;
        // Cancel and clear workspace pull if present.
        this.workspaceCancellation?.cancel();
        this.workspaceTimeout?.dispose();
        // Cancel all request and mark open requests as outdated.
        for (const [key, request] of this.openRequests) {
            if (request.state === RequestStateKind.active) {
                request.tokenSource.cancel();
            }
            this.openRequests.set(key, { state: RequestStateKind.outDated, textDocument: request.textDocument });
        }
    }
}
class BackgroundScheduler {
    constructor(diagnosticRequestor) {
        this.diagnosticRequestor = diagnosticRequestor;
        this.documents = new vscode_languageserver_protocol_1.LinkedMap();
    }
    add(textDocument) {
        const key = textDocument.uri.toString();
        if (this.documents.has(key)) {
            return;
        }
        this.documents.set(textDocument.uri.toString(), textDocument, vscode_languageserver_protocol_1.Touch.Last);
        this.trigger();
    }
    remove(textDocument) {
        const key = textDocument.uri.toString();
        if (this.documents.has(key)) {
            this.documents.delete(key);
            // Do a last pull
            this.diagnosticRequestor.pull(textDocument);
        }
        // No more documents. Stop background activity.
        if (this.documents.size === 0) {
            this.stop();
        }
        else if (textDocument === this.endDocument) {
            // Make sure we have a correct last document. It could have
            this.endDocument = this.documents.last;
        }
    }
    trigger() {
        // We have a round running. So simply make sure we run up to the
        // last document
        if (this.intervalHandle !== undefined) {
            this.endDocument = this.documents.last;
            return;
        }
        this.endDocument = this.documents.last;
        this.intervalHandle = (0, vscode_languageserver_protocol_1.RAL)().timer.setInterval(() => {
            const document = this.documents.first;
            if (document !== undefined) {
                this.diagnosticRequestor.pull(document);
                this.documents.set(document.uri.toString(), document, vscode_languageserver_protocol_1.Touch.Last);
                if (document === this.endDocument) {
                    this.stop();
                }
            }
        }, 200);
    }
    dispose() {
        this.stop();
        this.documents.clear();
    }
    stop() {
        this.intervalHandle?.dispose();
        this.intervalHandle = undefined;
        this.endDocument = undefined;
    }
}
class DiagnosticFeatureProviderImpl {
    constructor(client, editorTracker, options) {
        const diagnosticPullOptions = client.clientOptions.diagnosticPullOptions ?? { onChange: true, onSave: false };
        const documentSelector = options.documentSelector;
        const disposables = [];
        const matches = (textDocument) => {
            return client_1.$DocumentSelector.matchForProvider(documentSelector, textDocument) && editorTracker.isVisible(textDocument);
        };
        this.diagnosticRequestor = new DiagnosticRequestor(client, editorTracker, options);
        this.backgroundScheduler = new BackgroundScheduler(this.diagnosticRequestor);
        const addToBackgroundIfNeeded = (textDocument) => {
            if (!matches(textDocument) || !options.interFileDependencies || this.activeTextDocument === textDocument) {
                return;
            }
            this.backgroundScheduler.add(textDocument);
        };
        this.activeTextDocument = vscode_1.window.activeTextEditor?.document;
        vscode_1.window.onDidChangeActiveTextEditor((editor) => {
            const oldActive = this.activeTextDocument;
            this.activeTextDocument = editor?.document;
            if (oldActive !== undefined) {
                addToBackgroundIfNeeded(oldActive);
            }
            if (this.activeTextDocument !== undefined) {
                this.backgroundScheduler.remove(this.activeTextDocument);
            }
        });
        // We always pull on open.
        const openFeature = client.getFeature(vscode_languageserver_protocol_1.DidOpenTextDocumentNotification.method);
        disposables.push(openFeature.onNotificationSent((event) => {
            const textDocument = event.original;
            if (matches(textDocument)) {
                this.diagnosticRequestor.pull(textDocument, () => { addToBackgroundIfNeeded(textDocument); });
            }
        }));
        // Pull all diagnostics for documents that are already open
        for (const textDocument of vscode_1.workspace.textDocuments) {
            if (matches(textDocument)) {
                this.diagnosticRequestor.pull(textDocument, () => { addToBackgroundIfNeeded(textDocument); });
            }
        }
        if (diagnosticPullOptions.onChange) {
            const changeFeature = client.getFeature(vscode_languageserver_protocol_1.DidChangeTextDocumentNotification.method);
            disposables.push(changeFeature.onNotificationSent(async (event) => {
                const textDocument = event.original.document;
                if ((diagnosticPullOptions.filter === undefined || !diagnosticPullOptions.filter(textDocument, client_1.DiagnosticPullMode.onType)) && this.diagnosticRequestor.knows(PullState.document, textDocument) && event.original.contentChanges.length > 0) {
                    this.diagnosticRequestor.pull(textDocument, () => { this.backgroundScheduler.trigger(); });
                }
            }));
        }
        if (diagnosticPullOptions.onSave) {
            const saveFeature = client.getFeature(vscode_languageserver_protocol_1.DidSaveTextDocumentNotification.method);
            disposables.push(saveFeature.onNotificationSent((event) => {
                const textDocument = event.original;
                if ((diagnosticPullOptions.filter === undefined || !diagnosticPullOptions.filter(textDocument, client_1.DiagnosticPullMode.onSave)) && this.diagnosticRequestor.knows(PullState.document, textDocument)) {
                    this.diagnosticRequestor.pull(event.original, () => { this.backgroundScheduler.trigger(); });
                }
            }));
        }
        // When the document closes clear things up
        const closeFeature = client.getFeature(vscode_languageserver_protocol_1.DidCloseTextDocumentNotification.method);
        disposables.push(closeFeature.onNotificationSent((event) => {
            const textDocument = event.original;
            this.diagnosticRequestor.cleanupPull(textDocument);
            this.backgroundScheduler.remove(textDocument);
        }));
        // We received a did change from the server.
        this.diagnosticRequestor.onDidChangeDiagnosticsEmitter.event(() => {
            for (const textDocument of vscode_1.workspace.textDocuments) {
                if (matches(textDocument)) {
                    this.diagnosticRequestor.pull(textDocument);
                }
            }
        });
        // da348dc5-c30a-4515-9d98-31ff3be38d14 is the test UUID to test the middle ware. So don't auto trigger pulls.
        if (options.workspaceDiagnostics === true && options.identifier !== 'da348dc5-c30a-4515-9d98-31ff3be38d14') {
            this.diagnosticRequestor.pullWorkspace();
        }
        this.disposable = vscode_1.Disposable.from(...disposables, this.backgroundScheduler, this.diagnosticRequestor);
    }
    get onDidChangeDiagnosticsEmitter() {
        return this.diagnosticRequestor.onDidChangeDiagnosticsEmitter;
    }
    get diagnostics() {
        return this.diagnosticRequestor.provider;
    }
}
class DiagnosticFeature extends client_1.TextDocumentFeature {
    constructor(client) {
        super(client, vscode_languageserver_protocol_1.Proposed.DocumentDiagnosticRequest.type);
        this.editorTracker = new EditorTracker();
    }
    fillClientCapabilities(capabilities) {
        let capability = ensure(ensure(capabilities, 'textDocument'), 'diagnostic');
        capability.dynamicRegistration = true;
        // We first need to decide how a UI will look with related documents.
        // An easy implementation would be to only show related diagnostics for
        // the active editor.
        capability.relatedDocumentSupport = false;
    }
    initialize(capabilities, documentSelector) {
        const client = this._client;
        client.onRequest(vscode_languageserver_protocol_1.Proposed.DiagnosticRefreshRequest.type, async () => {
            for (const provider of this.getAllProviders()) {
                provider.onDidChangeDiagnosticsEmitter.fire();
            }
        });
        let [id, options] = this.getRegistration(documentSelector, capabilities.diagnosticProvider);
        if (!id || !options) {
            return;
        }
        this.register({ id: id, registerOptions: options });
    }
    dispose() {
        this.editorTracker.dispose();
        super.dispose();
    }
    registerLanguageProvider(options) {
        const provider = new DiagnosticFeatureProviderImpl(this._client, this.editorTracker, options);
        return [provider.disposable, provider];
    }
}
exports.DiagnosticFeature = DiagnosticFeature;
//# sourceMappingURL=proposed.diagnostic.js.map