"use strict";
/* --------------------------------------------------------------------------------------------
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 * ------------------------------------------------------------------------------------------ */
Object.defineProperty(exports, "__esModule", { value: true });
exports.InlineValueFeature = void 0;
const vscode_1 = require("vscode");
const vscode_languageserver_protocol_1 = require("vscode-languageserver-protocol");
const client_1 = require("./client");
function ensure(target, key) {
    if (target[key] === void 0) {
        target[key] = Object.create(null);
    }
    return target[key];
}
class InlineValueFeature extends client_1.TextDocumentFeature {
    constructor(client) {
        super(client, vscode_languageserver_protocol_1.Proposed.InlineValuesRequest.type);
    }
    fillClientCapabilities(capabilities) {
        ensure(ensure(capabilities, 'textDocument'), 'inlineValues').dynamicRegistration = true;
        ensure(ensure(capabilities, 'workspace'), 'codeLens').refreshSupport = true;
    }
    initialize(capabilities, documentSelector) {
        this._client.onRequest(vscode_languageserver_protocol_1.Proposed.InlineValuesRefreshRequest.type, async () => {
            for (const provider of this.getAllProviders()) {
                provider.onDidChangeInlineValues.fire();
            }
        });
        const [id, options] = this.getRegistration(documentSelector, capabilities.inlineValuesProvider);
        if (!id || !options) {
            return;
        }
        this.register({ id: id, registerOptions: options });
    }
    registerLanguageProvider(options) {
        const selector = options.documentSelector;
        const eventEmitter = new vscode_1.EventEmitter();
        const provider = {
            onDidChangeInlineValues: eventEmitter.event,
            provideInlineValues: (document, viewPort, context, token) => {
                if (client_1.$DocumentSelector.skipCellTextDocument(selector, document)) {
                    return undefined;
                }
                const client = this._client;
                const provideInlineValues = (document, viewPort, context, token) => {
                    const requestParams = {
                        textDocument: client.code2ProtocolConverter.asTextDocumentIdentifier(document),
                        viewPort: client.code2ProtocolConverter.asRange(viewPort),
                        context: client.code2ProtocolConverter.asInlineValuesContext(context)
                    };
                    return client.sendRequest(vscode_languageserver_protocol_1.Proposed.InlineValuesRequest.type, requestParams, token).then((values) => client.protocol2CodeConverter.asInlineValues(values), (error) => {
                        return client.handleFailedRequest(vscode_languageserver_protocol_1.Proposed.InlineValuesRequest.type, token, error, null);
                    });
                };
                const middleware = client.clientOptions.middleware;
                return middleware.provideInlineValues
                    ? middleware.provideInlineValues(document, viewPort, context, token, provideInlineValues)
                    : provideInlineValues(document, viewPort, context, token);
            }
        };
        return [vscode_1.languages.registerInlineValuesProvider(client_1.$DocumentSelector.asTextDocumentFilters(selector), provider), { provider: provider, onDidChangeInlineValues: eventEmitter }];
    }
}
exports.InlineValueFeature = InlineValueFeature;
//# sourceMappingURL=proposed.inlineValues.js.map