"use strict";
/* --------------------------------------------------------------------------------------------
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 * ------------------------------------------------------------------------------------------ */
/// <reference path="../../typings/vscode.proposed.notebookEditor.d.ts" />
Object.defineProperty(exports, "__esModule", { value: true });
exports.NotebookDocumentSyncFeature = void 0;
const vscode = require("vscode");
const minimatch = require("minimatch");
const proto = require("vscode-languageserver-protocol");
const vscode_languageserver_protocol_1 = require("vscode-languageserver-protocol");
const client_1 = require("./client");
const UUID = require("./utils/uuid");
function ensure(target, key) {
    if (target[key] === void 0) {
        target[key] = {};
    }
    return target[key];
}
var Converter;
(function (Converter) {
    let c2p;
    (function (c2p) {
        function asVersionedNotebookDocumentIdentifier(notebookDocument, base) {
            return {
                version: notebookDocument.version,
                uri: base.asUri(notebookDocument.uri)
            };
        }
        c2p.asVersionedNotebookDocumentIdentifier = asVersionedNotebookDocumentIdentifier;
        function asNotebookDocument(notebookDocument, cells, base) {
            const result = proto.Proposed.NotebookDocument.create(base.asUri(notebookDocument.uri), notebookDocument.notebookType, notebookDocument.version, asNotebookCells(cells, base));
            if (Object.keys(notebookDocument.metadata).length > 0) {
                result.metadata = asMetadata(notebookDocument.metadata);
            }
            return result;
        }
        c2p.asNotebookDocument = asNotebookDocument;
        function asNotebookCells(cells, base) {
            return cells.map(cell => asNotebookCell(cell, base));
        }
        c2p.asNotebookCells = asNotebookCells;
        function asMetadata(metadata) {
            const seen = new Set();
            return deepCopy(seen, metadata);
        }
        c2p.asMetadata = asMetadata;
        function asNotebookCell(cell, base) {
            const result = proto.Proposed.NotebookCell.create(asNotebookCellKind(cell.kind), base.asUri(cell.document.uri));
            if (Object.keys(cell.metadata).length > 0) {
                result.metadata = asMetadata(cell.metadata);
            }
            return result;
        }
        c2p.asNotebookCell = asNotebookCell;
        function asNotebookCellKind(kind) {
            switch (kind) {
                case vscode.NotebookCellKind.Markup:
                    return proto.Proposed.NotebookCellKind.Markup;
                case vscode.NotebookCellKind.Code:
                    return proto.Proposed.NotebookCellKind.Code;
            }
        }
        function deepCopy(seen, value) {
            if (seen.has(value)) {
                throw new Error(`Can't deep copy cyclic structures.`);
            }
            if (Array.isArray(value)) {
                const result = [];
                for (const elem of value) {
                    if (elem !== null && typeof elem === 'object' || Array.isArray(elem)) {
                        result.push(deepCopy(seen, elem));
                    }
                    else {
                        if (elem instanceof RegExp) {
                            throw new Error(`Can't transfer regular expressions to the server`);
                        }
                        result.push(elem);
                    }
                }
                return result;
            }
            else {
                const props = Object.keys(value);
                const result = Object.create(null);
                for (const prop of props) {
                    const elem = value[prop];
                    if (elem !== null && typeof elem === 'object' || Array.isArray(elem)) {
                        result[prop] = deepCopy(seen, elem);
                    }
                    else {
                        if (elem instanceof RegExp) {
                            throw new Error(`Can't transfer regular expressions to the server`);
                        }
                        result[prop] = elem;
                    }
                }
                return result;
            }
        }
    })(c2p = Converter.c2p || (Converter.c2p = {}));
})(Converter || (Converter = {}));
var NotebookCell;
(function (NotebookCell) {
    function computeDiff(originalCells, modifiedCells, compareMetadata = false) {
        const originalLength = originalCells.length;
        const modifiedLength = modifiedCells.length;
        let startIndex = 0;
        while (startIndex < modifiedLength && startIndex < originalLength && proto.Proposed.NotebookCell.equals(originalCells[startIndex], modifiedCells[startIndex], compareMetadata)) {
            startIndex++;
        }
        if (startIndex < modifiedLength && startIndex < originalLength) {
            let originalEndIndex = originalLength - 1;
            let modifiedEndIndex = modifiedLength - 1;
            while (originalEndIndex >= 0 && modifiedEndIndex >= 0 && proto.Proposed.NotebookCell.equals(originalCells[originalEndIndex], modifiedCells[modifiedEndIndex], compareMetadata)) {
                originalEndIndex--;
                modifiedEndIndex--;
            }
            const deleteCount = (originalEndIndex + 1) - startIndex;
            const newCells = startIndex === modifiedEndIndex + 1 ? undefined : modifiedCells.slice(startIndex, modifiedEndIndex + 1);
            return newCells !== undefined ? { start: startIndex, deleteCount, cells: newCells } : { start: startIndex, deleteCount };
        }
        else if (startIndex < modifiedLength) {
            return { start: startIndex, deleteCount: 0, cells: modifiedCells.slice(startIndex) };
        }
        else if (startIndex < originalLength) {
            return { start: startIndex, deleteCount: originalLength - startIndex };
        }
        else {
            // The two arrays are the same.
            return undefined;
        }
    }
    NotebookCell.computeDiff = computeDiff;
})(NotebookCell || (NotebookCell = {}));
var $NotebookDocumentFilter;
(function ($NotebookDocumentFilter) {
    function matchNotebook(filter, notebookDocument) {
        if (filter.notebookType !== undefined && notebookDocument.notebookType !== filter.notebookType) {
            return false;
        }
        const uri = notebookDocument.uri;
        if (filter.scheme !== undefined && uri.scheme !== filter.scheme) {
            return false;
        }
        if (filter.pattern !== undefined) {
            const matcher = new minimatch.Minimatch(filter.pattern, { noext: true });
            if (!matcher.makeRe()) {
                return false;
            }
            if (!matcher.match(uri.fsPath)) {
                return false;
            }
        }
        return true;
    }
    $NotebookDocumentFilter.matchNotebook = matchNotebook;
})($NotebookDocumentFilter || ($NotebookDocumentFilter = {}));
var $NotebookDocumentSyncOptions;
(function ($NotebookDocumentSyncOptions) {
    function match(options, cell, mode) {
        if (mode === 'cellContent' && !client_1.$DocumentSelector.matchForDocumentSync(options.cellDocumentSelector, cell.document)) {
            return false;
        }
        if (mode === 'notebook' && !client_1.$DocumentSelector.matchForProvider(options.cellDocumentSelector, cell.document)) {
            return false;
        }
        const notebook = cell.notebook;
        for (const filter of options.notebookDocumentSelector) {
            if (filter.notebookDocumentFilter !== undefined && $NotebookDocumentFilter.matchNotebook(filter.notebookDocumentFilter, notebook)) {
                return true;
            }
        }
        return false;
    }
    $NotebookDocumentSyncOptions.match = match;
})($NotebookDocumentSyncOptions || ($NotebookDocumentSyncOptions = {}));
var SyncInfo;
(function (SyncInfo) {
    function create(proto, code) {
        return {
            cells: proto,
            uris: new Set(code.map(cell => cell.document.uri.toString()))
        };
    }
    SyncInfo.create = create;
})(SyncInfo || (SyncInfo = {}));
class NotebookDocumentSyncFeatureProvider {
    constructor(client, options) {
        this.client = client;
        this.options = options;
        this.notebookSyncInfo = new Map();
        this.notebookDidOpen = new Set();
        this.disposables = [];
        // open
        vscode.workspace.onDidOpenNotebookDocument((notebookDocument) => {
            this.notebookDidOpen.add(notebookDocument.uri.toString());
            this.didOpen(notebookDocument);
        }, undefined, this.disposables);
        for (const notebookDocument of vscode.workspace.notebookDocuments) {
            this.notebookDidOpen.add(notebookDocument.uri.toString());
            this.didOpen(notebookDocument);
        }
        // notebook document meta data changed
        vscode.notebooks.onDidChangeNotebookDocumentMetadata(event => this.notebookDocumentMetadataChanged(event.document), undefined, this.disposables);
        // cell add, remove, reorder
        vscode.notebooks.onDidChangeNotebookCells(event => this.cellStructureChanged(event.document), undefined, this.disposables);
        // The metadata of the cell has changed.
        vscode.notebooks.onDidChangeCellMetadata(event => this.cellMetaDataChanged(event.cell.notebook, event.cell), undefined, this.disposables);
        //save
        if (this.options.save === true) {
            vscode.notebooks.onDidSaveNotebookDocument(notebookDocument => this.didSave(notebookDocument), undefined, this.disposables);
        }
        // close
        vscode.workspace.onDidCloseNotebookDocument((notebookDocument) => {
            this.didClose(notebookDocument);
            this.notebookDidOpen.delete(notebookDocument.uri.toString());
        }, undefined, this.disposables);
    }
    get mode() {
        return 'notebook';
    }
    handles(notebookCell) {
        return $NotebookDocumentSyncOptions.match(this.options, notebookCell, this.mode);
    }
    didOpenNotebookCellDocument(notebookDocument, cell) {
        if (!this.notebookDidOpen.has(notebookDocument.uri.toString())) {
            // We have never received an open notification for the notebook document.
            // VS Code guarantees that we first get cell document open and then
            // notebook open. So simply wait for the notebook open.
            return;
        }
        const syncInfo = this.notebookSyncInfo.get(notebookDocument.uri.toString());
        // In VS Code we receive a notebook open before a cell document open.
        // The document and the cell is synced.
        const cells = this.getMatchingCells(notebookDocument, [cell]);
        const cellMatches = cells !== undefined && cells[0] === cell;
        if (syncInfo !== undefined) {
            const cellIsSynced = syncInfo.uris.has(cell.document.uri.toString());
            // The notebook document is synced
            if (cellMatches && cellIsSynced) {
                // Cell matches and is synced.
                return;
            }
            this.cellStructureChanged(notebookDocument, syncInfo);
        }
        else {
            // No sync info
            if (!cellMatches) {
                // Cell doesn't match. Everything OK
                return;
            }
            this.cellStructureChanged(notebookDocument, syncInfo);
        }
    }
    didChangeCellTextDocument(notebookDocument, event) {
        if (!client_1.$DocumentSelector.matchForProvider(this.options.cellDocumentSelector, event.document)) {
            return;
        }
        this.doSendChange(notebookDocument, {
            cellTextDocuments: [this.client.code2ProtocolConverter.asChangeTextDocumentParams(event)]
        }).catch(() => { });
    }
    dispose() {
        for (const disposable of this.disposables) {
            disposable.dispose();
        }
    }
    didOpen(notebookDocument, optionalCells = null, syncInfo = this.notebookSyncInfo.get(notebookDocument.uri.toString())) {
        if (syncInfo !== undefined) {
            // The notebook document got synced because a cell document got opened.
            this.cellStructureChanged(notebookDocument, syncInfo);
        }
        else {
            // Check if we need to sync the notebook document.
            const cells = optionalCells ?? this.getMatchingCells(notebookDocument);
            if (cells === undefined) {
                return;
            }
            this.doSendOpen(notebookDocument, cells, (nb) => {
                this.notebookSyncInfo.set(notebookDocument.uri.toString(), SyncInfo.create(nb.cells, cells));
            }).catch(() => { });
        }
    }
    notebookDocumentMetadataChanged(notebookDocument) {
        const syncInfo = this.notebookSyncInfo.get(notebookDocument.uri.toString());
        if (syncInfo === undefined) {
            return;
        }
        this.doSendChange(notebookDocument, {
            metadata: Converter.c2p.asMetadata(notebookDocument.metadata)
        }).catch(() => { });
    }
    cellStructureChanged(notebookDocument, syncInfo = this.notebookSyncInfo.get(notebookDocument.uri.toString())) {
        if (syncInfo === undefined) {
            // The notebook has not been synced. Could be it never matched or some
            // cells didn't match. So check if it would match now.
            const cells = this.getMatchingCells(notebookDocument);
            if (cells === undefined) {
                return;
            }
            this.didOpen(notebookDocument, cells, syncInfo);
        }
        else {
            // It is synced. Could be no cells match anymore. If this is the
            // case we close the notebook document. Otherwise we send a change event.
            const cells = this.getMatchingCells(notebookDocument);
            if (cells === undefined) {
                this.didClose(notebookDocument, syncInfo);
                return;
            }
            const oldCells = syncInfo.cells;
            const newCells = Converter.c2p.asNotebookCells(cells, this.client.code2ProtocolConverter);
            // meta data changes are reported using a different event. So we can ignore comparing the meta data which
            // has a positive impact on performance.
            const diff = NotebookCell.computeDiff(syncInfo.cells, newCells, false);
            if (diff === undefined) {
                return;
            }
            const deletedCells = diff.deleteCount === 0
                ? new Set()
                : new Set(oldCells.slice(diff.start, diff.start + diff.deleteCount).map(cell => cell.document));
            const insertedCells = diff.cells === undefined
                ? new Set()
                : new Set(diff.cells.map(cell => cell.document));
            // Remove the onces that got deleted and inserted again.
            for (const key of Array.from(deletedCells.values())) {
                if (insertedCells.has(key)) {
                    deletedCells.delete(key);
                    insertedCells.delete(key);
                }
            }
            const didOpen = [];
            const didClose = [];
            if (deletedCells.size > 0 || insertedCells.size > 0) {
                const codeCells = new Map(cells.map(cell => [this.client.code2ProtocolConverter.asUri(cell.document.uri), cell]));
                for (const document of insertedCells.values()) {
                    const cell = codeCells.get(document);
                    if (cell !== undefined) {
                        didOpen.push(this.client.code2ProtocolConverter.asTextDocumentItem(cell.document));
                    }
                }
                for (const document of deletedCells.values()) {
                    didClose.push({ uri: document });
                }
            }
            this.doSendChange(notebookDocument, {
                cellStructure: {
                    array: diff,
                    didClose: didClose.length > 0 ? didClose : undefined,
                    didOpen: didOpen.length > 0 ? didOpen : undefined
                }
            }).catch(() => { });
            this.notebookSyncInfo.set(notebookDocument.uri.toString(), SyncInfo.create(newCells, cells));
        }
    }
    cellMetaDataChanged(notebookDocument, cell) {
        const syncInfo = this.notebookSyncInfo.get(notebookDocument.uri.toString());
        if (syncInfo === undefined) {
            // No sync info. Since the cells meta data change it can have no impact on the
            // sync in general (not filter on metadata). So nothing to do
            return;
        }
        if (!syncInfo.uris.has(cell.document.uri.toString())) {
            // The cell is not sync. So ignore as well.
            return;
        }
        const pc = Converter.c2p.asNotebookCell(cell, this.client.code2ProtocolConverter);
        this.doSendChange(notebookDocument, { cellData: [pc] }).catch(() => { });
    }
    didSave(notebookDocument) {
        const syncInfo = this.notebookSyncInfo.get(notebookDocument.uri.toString());
        if (syncInfo === undefined) {
            return;
        }
        this.doSendSave(notebookDocument).catch(() => { });
    }
    didClose(notebookDocument, syncInfo = this.notebookSyncInfo.get(notebookDocument.uri.toString())) {
        if (syncInfo === undefined) {
            return;
        }
        const syncedCells = notebookDocument.getCells().filter(cell => syncInfo.uris.has(cell.document.uri.toString()));
        this.doSendClose(notebookDocument, syncedCells).catch(() => { });
        this.notebookSyncInfo.delete(notebookDocument.uri.toString());
    }
    getMatchingCells(notebookDocument, cells = notebookDocument.getCells()) {
        if (this.options.notebookDocumentSelector === undefined) {
            return undefined;
        }
        for (const item of this.options.notebookDocumentSelector) {
            if (item.notebookDocumentFilter === undefined) {
                if (item.cellSelector === undefined) {
                    return undefined;
                }
                const filtered = this.filterCells(cells, item.cellSelector);
                return filtered.length === 0 ? undefined : filtered;
            }
            else if ($NotebookDocumentFilter.matchNotebook(item.notebookDocumentFilter, notebookDocument)) {
                return item.cellSelector === undefined ? cells : this.filterCells(cells, item.cellSelector);
            }
        }
        return undefined;
    }
    filterCells(cells, cellSelector) {
        return cells.filter((cell) => {
            const cellLanguage = cell.document.languageId;
            return cellSelector.some((filter => cellLanguage === filter.language));
        });
    }
    async sendOpen(notebookDocument) {
        const cells = this.getMatchingCells(notebookDocument);
        if (cells === undefined) {
            return;
        }
        return this.doSendOpen(notebookDocument, cells);
    }
    async doSendOpen(notebookDocument, cells, cb) {
        const send = (notebookDocument, cells) => {
            const nb = Converter.c2p.asNotebookDocument(notebookDocument, cells, this.client.code2ProtocolConverter);
            const cellDocuments = cells.map(cell => this.client.code2ProtocolConverter.asTextDocumentItem(cell.document));
            const result = this.client.sendNotification(proto.Proposed.DidOpenNotebookDocumentNotification.type, {
                notebookDocument: nb,
                cellTextDocuments: cellDocuments
            }).catch((error) => {
                this.client.error('Sending DidOpenNotebookDocumentNotification failed', error);
                throw error;
            });
            cb && cb(nb);
            return result;
        };
        const middleware = this.client.clientOptions.middleware?.notebooks;
        return middleware?.didOpen !== undefined ? middleware.didOpen(notebookDocument, cells, send) : send(notebookDocument, cells);
    }
    async sendChange(notebookDocument, changeData) {
        const changeEvent = Object.create(null);
        if (changeData.metadata) {
            changeEvent.metadata = Converter.c2p.asMetadata(notebookDocument.metadata);
        }
        if (changeData.cellStructure) {
            changeEvent.cellStructure = {
                array: {
                    start: changeData.cellStructure.array.start,
                    deleteCount: changeData.cellStructure.array.deleteCount,
                    cells: changeData.cellStructure.array.cells !== undefined ? changeData.cellStructure.array.cells.map(cell => Converter.c2p.asNotebookCell(cell, this.client.code2ProtocolConverter)) : undefined
                },
                didOpen: changeData.cellStructure.didOpen !== undefined
                    ? changeData.cellStructure.didOpen.map(cell => this.client.code2ProtocolConverter.asOpenTextDocumentParams(cell.document).textDocument)
                    : undefined,
                didClose: changeData.cellStructure.didClose !== undefined
                    ? changeData.cellStructure.didClose.map(cell => this.client.code2ProtocolConverter.asCloseTextDocumentParams(cell.document).textDocument)
                    : undefined
            };
        }
        if (changeData.cellData !== undefined) {
            changeEvent.cellData = changeData.cellData.map(cell => Converter.c2p.asNotebookCell(cell, this.client.code2ProtocolConverter));
        }
        if (changeData.cellTextContent !== undefined) {
            changeEvent.cellTextDocuments = changeData.cellTextContent.map(event => this.client.code2ProtocolConverter.asChangeTextDocumentParams(event));
        }
        if (Object.keys(changeEvent).length > 0) {
            return this.doSendChange(notebookDocument, changeEvent);
        }
    }
    async doSendChange(notebookDocument, change) {
        const send = (notebookDocument, change) => {
            return this.client.sendNotification(proto.Proposed.DidChangeNotebookDocumentNotification.type, {
                notebookDocument: Converter.c2p.asVersionedNotebookDocumentIdentifier(notebookDocument, this.client.code2ProtocolConverter),
                change: change
            }).catch((error) => {
                this.client.error('Sending DidChangeNotebookDocumentNotification failed', error);
            });
        };
        const middleware = this.client.clientOptions.middleware?.notebooks;
        return middleware?.didChange !== undefined ? middleware?.didChange(notebookDocument, change, send) : send(notebookDocument, change);
    }
    async sendSave(notebookDocument) {
        return this.doSendSave(notebookDocument);
    }
    async doSendSave(notebookDocument) {
        const send = (notebookDocument) => {
            return this.client.sendNotification(proto.Proposed.DidSaveNotebookDocumentNotification.type, {
                notebookDocument: { uri: this.client.code2ProtocolConverter.asUri(notebookDocument.uri) }
            }).catch((error) => {
                this.client.error('Sending DidSaveNotebookDocumentNotification failed', error);
                throw error;
            });
        };
        const middleware = this.client.clientOptions.middleware?.notebooks;
        return middleware?.didSave !== undefined ? middleware.didSave(notebookDocument, send) : send(notebookDocument);
    }
    async sendClose(notebookDocument) {
        return this.doSendClose(notebookDocument, this.getMatchingCells(notebookDocument) ?? []);
    }
    async doSendClose(notebookDocument, cells) {
        const send = (notebookDocument, cells) => {
            return this.client.sendNotification(proto.Proposed.DidCloseNotebookDocumentNotification.type, {
                notebookDocument: { uri: this.client.code2ProtocolConverter.asUri(notebookDocument.uri) },
                cellTextDocuments: cells.map(cell => this.client.code2ProtocolConverter.asTextDocumentIdentifier(cell.document))
            }).catch((error) => {
                this.client.error('Sending DidCloseNotebookDocumentNotification failed', error);
                throw error;
            });
        };
        const middleware = this.client.clientOptions.middleware?.notebooks;
        return middleware?.didClose !== undefined ? middleware.didClose(notebookDocument, cells, send) : send(notebookDocument, cells);
    }
}
class NotebookCellTextDocumentSyncFeatureProvider {
    constructor(client, options) {
        this.client = client;
        this.options = options;
        if (options.cellDocumentSelector.length > 0) {
            const openId = UUID.generateUuid();
            this.client.getFeature(vscode_languageserver_protocol_1.DidOpenTextDocumentNotification.method).register({
                id: openId, registerOptions: { documentSelector: options.cellDocumentSelector }
            });
            const changeId = UUID.generateUuid();
            this.client.getFeature(vscode_languageserver_protocol_1.DidChangeTextDocumentNotification.method).register({
                id: changeId, registerOptions: { documentSelector: options.cellDocumentSelector, syncKind: vscode_languageserver_protocol_1.TextDocumentSyncKind.Incremental }
            });
            const closeId = UUID.generateUuid();
            this.client.getFeature(vscode_languageserver_protocol_1.DidCloseTextDocumentNotification.method).register({
                id: closeId, registerOptions: { documentSelector: options.cellDocumentSelector }
            });
            this.registrations = { open: openId, change: changeId, close: closeId };
        }
    }
    get mode() {
        return 'cellContent';
    }
    handles(notebookCell) {
        return $NotebookDocumentSyncOptions.match(this.options, notebookCell, this.mode);
    }
    dispose() {
        if (this.registrations !== undefined) {
            this.client.getFeature(vscode_languageserver_protocol_1.DidOpenTextDocumentNotification.method).unregister(this.registrations.open);
            this.client.getFeature(vscode_languageserver_protocol_1.DidChangeTextDocumentNotification.method).unregister(this.registrations.change);
            this.client.getFeature(vscode_languageserver_protocol_1.DidCloseTextDocumentNotification.method).unregister(this.registrations.close);
        }
    }
    async sendOpen(textDocument) {
        const provider = this.client.getFeature(vscode_languageserver_protocol_1.DidOpenTextDocumentNotification.method).getProvider(textDocument);
        return provider !== undefined ? provider.send(textDocument) : Promise.reject(new Error(`No open provider found for notebook cell document ${textDocument.uri.toString()}`));
    }
    async sendChange(event) {
        const provider = this.client.getFeature(vscode_languageserver_protocol_1.DidChangeTextDocumentNotification.method).getProvider(event.document);
        return provider !== undefined ? provider.send(event) : Promise.reject(new Error(`No change provider found for notebook cell document ${event.document.uri.toString()}`));
    }
    async sendClose(textDocument) {
        const provider = this.client.getFeature(vscode_languageserver_protocol_1.DidCloseTextDocumentNotification.method).getProvider(textDocument);
        return provider !== undefined ? provider.send(textDocument) : Promise.reject(new Error(`No close provider found for notebook cell document ${textDocument.uri.toString()}`));
    }
}
class NotebookDocumentSyncFeature {
    constructor(client) {
        this.client = client;
        this.registrations = new Map();
        this.registrationType = proto.Proposed.NotebookDocumentSyncRegistrationType.type;
        // We don't receive an event for cells where the document changes its language mode
        // Since we allow servers to filter on the language mode we fire such an event ourselves.
        vscode.workspace.onDidOpenTextDocument((textDocument) => {
            if (textDocument.uri.scheme !== NotebookDocumentSyncFeature.CellScheme) {
                return;
            }
            const [notebookDocument, notebookCell] = this.getNotebookDocument(textDocument);
            if (notebookDocument === undefined || notebookCell === undefined) {
                return;
            }
            for (const provider of this.registrations.values()) {
                if (provider instanceof NotebookDocumentSyncFeatureProvider) {
                    provider.didOpenNotebookCellDocument(notebookDocument, notebookCell);
                }
            }
        });
        vscode.workspace.onDidChangeTextDocument((event) => {
            if (event.contentChanges.length === 0) {
                return;
            }
            const textDocument = event.document;
            if (textDocument.uri.scheme !== NotebookDocumentSyncFeature.CellScheme) {
                return;
            }
            const [notebookDocument,] = this.getNotebookDocument(textDocument);
            if (notebookDocument === undefined) {
                return;
            }
            for (const provider of this.registrations.values()) {
                if (provider instanceof NotebookDocumentSyncFeatureProvider) {
                    provider.didChangeCellTextDocument(notebookDocument, event);
                }
            }
        });
    }
    fillClientCapabilities(capabilities) {
        const synchronization = ensure(ensure(capabilities, 'notebookDocument'), 'synchronization');
        synchronization.dynamicRegistration = true;
    }
    initialize(capabilities) {
        const options = capabilities.notebookDocumentSync;
        if (options === undefined) {
            return;
        }
        const id = options.id ?? UUID.generateUuid();
        this.register({ id, registerOptions: options });
    }
    register(data) {
        if (data.registerOptions.mode === 'cellContent') {
            const options = Object.assign({}, data.registerOptions, { cellDocumentSelector: this.getNotebookCellTextDocumentFilter(data.registerOptions, true) });
            const provider = new NotebookCellTextDocumentSyncFeatureProvider(this.client, options);
            this.registrations.set(data.id, provider);
        }
        else {
            const options = Object.assign({}, data.registerOptions, { cellDocumentSelector: this.getNotebookCellTextDocumentFilter(data.registerOptions) });
            const provider = new NotebookDocumentSyncFeatureProvider(this.client, options);
            this.registrations.set(data.id, provider);
        }
    }
    unregister(id) {
        const provider = this.registrations.get(id);
        provider && provider.dispose();
    }
    dispose() {
        for (const provider of this.registrations.values()) {
            provider.dispose();
        }
        this.registrations.clear();
    }
    getProvider(notebookCell) {
        for (const provider of this.registrations.values()) {
            if (provider.handles(notebookCell)) {
                return provider;
            }
        }
        return undefined;
    }
    getNotebookDocument(textDocument) {
        const uri = textDocument.uri.toString();
        for (const notebookDocument of vscode.workspace.notebookDocuments) {
            for (const cell of notebookDocument.getCells()) {
                if (cell.document.uri.toString() === uri) {
                    return [notebookDocument, cell];
                }
            }
        }
        return [undefined, undefined];
    }
    getNotebookCellTextDocumentFilter(options, sync) {
        const documentSelector = [];
        for (const item of options.notebookDocumentSelector) {
            let nf;
            if (item.notebookDocumentFilter !== undefined) {
                nf = sync === true ? { notebookDocument: Object.assign({}, item.notebookDocumentFilter, { sync: true }) } : { notebookDocument: Object.assign({}, item.notebookDocumentFilter) };
            }
            if (item.cellSelector !== undefined) {
                for (const cell of item.cellSelector) {
                    if (nf === undefined) {
                        documentSelector.push((sync === true ? { cellLanguage: cell.language, sync: true } : { cellLanguage: cell.language }));
                    }
                    else {
                        documentSelector.push(Object.assign({}, nf, { cellLanguage: cell.language }));
                    }
                }
                nf = undefined;
            }
            else if (nf !== undefined) {
                documentSelector.push(nf);
            }
        }
        return documentSelector;
    }
}
exports.NotebookDocumentSyncFeature = NotebookDocumentSyncFeature;
NotebookDocumentSyncFeature.CellScheme = 'vscode-notebook-cell';
//# sourceMappingURL=proposed.notebooks.js.map