"use strict";
/* --------------------------------------------------------------------------------------------
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 * ------------------------------------------------------------------------------------------ */
Object.defineProperty(exports, "__esModule", { value: true });
exports.TypeHierarchyFeature = void 0;
const vscode_1 = require("vscode");
const vscode_languageserver_protocol_1 = require("vscode-languageserver-protocol");
const client_1 = require("./client");
function ensure(target, key) {
    if (target[key] === void 0) {
        target[key] = {};
    }
    return target[key];
}
class TypeHierarchyProvider {
    constructor(client, options) {
        this.client = client;
        this.options = options;
        this.middleware = client.clientOptions.middleware;
    }
    prepareTypeHierarchy(document, position, token) {
        if (client_1.$DocumentSelector.skipCellTextDocument(this.options.documentSelector, document)) {
            return undefined;
        }
        const client = this.client;
        const middleware = this.middleware;
        const prepareTypeHierarchy = (document, position, token) => {
            const params = client.code2ProtocolConverter.asTextDocumentPositionParams(document, position);
            return client.sendRequest(vscode_languageserver_protocol_1.Proposed.TypeHierarchyPrepareRequest.type, params, token).then((result) => {
                return client.protocol2CodeConverter.asTypeHierarchyItems(result);
            }, (error) => {
                return client.handleFailedRequest(vscode_languageserver_protocol_1.Proposed.TypeHierarchyPrepareRequest.type, token, error, null);
            });
        };
        return middleware.prepareTypeHierarchy
            ? middleware.prepareTypeHierarchy(document, position, token, prepareTypeHierarchy)
            : prepareTypeHierarchy(document, position, token);
    }
    provideTypeHierarchySupertypes(item, token) {
        const client = this.client;
        const middleware = this.middleware;
        const provideTypeHierarchySupertypes = (item, token) => {
            const params = {
                item: client.code2ProtocolConverter.asTypeHierarchyItem(item)
            };
            return client.sendRequest(vscode_languageserver_protocol_1.Proposed.TypeHierarchySupertypesRequest.type, params, token).then((result) => {
                return client.protocol2CodeConverter.asTypeHierarchyItems(result);
            }, (error) => {
                return client.handleFailedRequest(vscode_languageserver_protocol_1.Proposed.TypeHierarchySupertypesRequest.type, token, error, null);
            });
        };
        return middleware.provideTypeHierarchySupertypes
            ? middleware.provideTypeHierarchySupertypes(item, token, provideTypeHierarchySupertypes)
            : provideTypeHierarchySupertypes(item, token);
    }
    provideTypeHierarchySubtypes(item, token) {
        const client = this.client;
        const middleware = this.middleware;
        const provideTypeHierarchySubtypes = (item, token) => {
            const params = {
                item: client.code2ProtocolConverter.asTypeHierarchyItem(item)
            };
            return client.sendRequest(vscode_languageserver_protocol_1.Proposed.TypeHierarchySubtypesRequest.type, params, token).then((result) => {
                return client.protocol2CodeConverter.asTypeHierarchyItems(result);
            }, (error) => {
                return client.handleFailedRequest(vscode_languageserver_protocol_1.Proposed.TypeHierarchySubtypesRequest.type, token, error, null);
            });
        };
        return middleware.provideTypeHierarchySubtypes
            ? middleware.provideTypeHierarchySubtypes(item, token, provideTypeHierarchySubtypes)
            : provideTypeHierarchySubtypes(item, token);
    }
}
class TypeHierarchyFeature extends client_1.TextDocumentFeature {
    constructor(client) {
        super(client, vscode_languageserver_protocol_1.Proposed.TypeHierarchyPrepareRequest.type);
    }
    fillClientCapabilities(cap) {
        const capabilities = cap;
        const capability = ensure(ensure(capabilities, 'textDocument'), 'typeHierarchy');
        capability.dynamicRegistration = true;
    }
    initialize(capabilities, documentSelector) {
        const [id, options] = this.getRegistration(documentSelector, capabilities.typeHierarchyProvider);
        if (!id || !options) {
            return;
        }
        this.register({ id: id, registerOptions: options });
    }
    registerLanguageProvider(options) {
        const client = this._client;
        const provider = new TypeHierarchyProvider(client, options);
        return [vscode_1.languages.registerTypeHierarchyProvider(client_1.$DocumentSelector.asTextDocumentFilters(options.documentSelector), provider), provider];
    }
}
exports.TypeHierarchyFeature = TypeHierarchyFeature;
//# sourceMappingURL=proposed.typeHierarchy.js.map