import macro                from 'vtk.js/Sources/macro';
import vtkCameraManipulator from 'vtk.js/Sources/Interaction/Manipulators/CameraManipulator';
import vtkMath              from 'vtk.js/Sources/Common/Core/Math';

// ----------------------------------------------------------------------------
// vtkTrackballPan methods
// ----------------------------------------------------------------------------

function vtkTrackballPan(publicAPI, model) {
  // Set our className
  model.classHierarchy.push('vtkTrackballPan');

  publicAPI.onAnimation = (interactor, renderer) => {
    const lastPtr = interactor.getPointerIndex();
    const pos = interactor.getAnimationEventPosition(lastPtr);
    const lastPos = interactor.getLastAnimationEventPosition(lastPtr);

    if (!pos || !lastPos || !renderer) {
      return;
    }

    const camera = renderer.getActiveCamera();
    const camPos = camera.getPosition();
    const fp = camera.getFocalPoint();

    if (camera.getParallelProjection()) {
      camera.orthogonalizeViewUp();

      const up = camera.getViewUp();
      const vpn = camera.getViewPlaneNormal();

      const right = [0, 0, 0];

      vtkMath.cross(vpn, up, right);

      // These are different because y is flipped.
      const size = interactor.getView().getSize();
      let dx = (pos.x - lastPos.x) / size[1];
      let dy = (lastPos.y - pos.y) / size[1];

      const scale = camera.getParallelScale();
      dx *= scale * 2.0;
      dy *= scale * 2.0;

      let tmp = (right[0] * dx) + (up[0] * dy);
      camPos[0] += tmp;
      fp[0] += tmp;
      tmp = (right[1] * dx) + (up[1] * dy);
      camPos[1] += tmp;
      fp[1] += tmp;
      tmp = (right[2] * dx) + (up[2] * dy);
      camPos[2] += tmp;
      fp[2] += tmp;
      camera.setPosition(camPos);
      camera.setFocalPoint(fp);
    } else {
      const center = model.center;
      const style = interactor.getInteractorStyle();
      const focalDepth = style.computeWorldToDisplay(center[0], center[1], center[2])[2];
      const worldPoint = style.computeDisplayToWorld(pos.x, pos.y, focalDepth);
      const lastWorldPoint = style.computeDisplayToWorld(lastPos.x, lastPos.y, focalDepth);

      const newCamPos = [
        camPos[0] + (lastWorldPoint[0] - worldPoint[0]),
        camPos[1] + (lastWorldPoint[1] - worldPoint[1]),
        camPos[2] + (lastWorldPoint[2] - worldPoint[2]),
      ];

      const newFp = [
        fp[0] + (lastWorldPoint[0] - worldPoint[0]),
        fp[1] + (lastWorldPoint[1] - worldPoint[1]),
        fp[2] + (lastWorldPoint[2] - worldPoint[2]),
      ];

      camera.setPosition(newCamPos[0], newCamPos[1], newCamPos[2]);
      camera.setFocalPoint(newFp[0], newFp[1], newFp[2]);
    }

    renderer.resetCameraClippingRange();
  };
}

// ----------------------------------------------------------------------------
// Object factory
// ----------------------------------------------------------------------------

const DEFAULT_VALUES = {};

// ----------------------------------------------------------------------------

export function extend(publicAPI, model, initialValues = {}) {
  Object.assign(model, DEFAULT_VALUES, initialValues);

  // Inheritance
  vtkCameraManipulator.extend(publicAPI, model, initialValues);

  // Object specific methods
  vtkTrackballPan(publicAPI, model);
}

// ----------------------------------------------------------------------------

export const newInstance = macro.newInstance(extend, 'vtkTrackballPan');

// ----------------------------------------------------------------------------

export default Object.assign({ newInstance, extend });
