import { App } from 'vue';
import { ComponentInternalInstance } from 'vue';
import { ComputedRef } from '@vue/reactivity';
import { WritableComputedRef } from '@vue/reactivity';

export declare function baseCompile(source: string, options?: CompileOptions): CompileResult;

export declare interface BaseFormatProps {
    tag?: string;
    locale?: Locale;
    scope?: ComponetI18nScope;
}

export declare type Choice = number;

export declare function clearCompileCache(): void;

export declare function clearDateTimeFormat(context: RuntimeContext, locale: Locale, format: DateTimeFormat): void;

export declare function clearNumberFormat(context: RuntimeContext, locale: Locale, format: NumberFormat): void;

declare type CodeGenOptions = {
    mode?: 'normal' | 'arrow';
    onError?: CompileErrorHandler;
};

export declare function compile(source: string, options?: CompileOptions): MessageFunction;

declare type CompileCacheKeyHandler = (source: string) => string;

export declare type CompileDomain = 'tokenizer' | 'parser' | 'generator' | 'transformer' | 'compiler';

export declare interface CompileError extends SyntaxError {
    code: number;
    domain?: CompileDomain;
    location?: SourceLocation;
}

export declare const enum CompileErrorCodes {
    EXPECTED_TOKEN = 0,
    INVALID_TOKEN_IN_PLACEHOLDER = 1,
    UNTERMINATED_SINGLE_QUOTE_IN_PLACEHOLDER = 2,
    UNKNOWN_ESCAPE_SEQUENCE = 3,
    INVALID_UNICODE_ESCAPE_SEQUENCE = 4,
    UNBALANCED_CLOSING_BRACE = 5,
    UNTERMINATED_CLOSING_BRACE = 6,
    EMPTY_PLACEHOLDER = 7,
    NOT_ALLOW_NEST_PLACEHOLDER = 8,
    INVALID_LINKED_FORMAT = 9,
    MUST_HAVE_MESSAGES_IN_PLURAL = 10,
    UNEXPECTED_LEXICAL_ANALYSIS = 11,
    __EXTEND_POINT__ = 12
}

export declare type CompileErrorHandler = (error: CompileError) => void;

export declare type CompileOptions = {
    warnHtmlMessage?: boolean;
    onCacheKey?: CompileCacheKeyHandler;
} & TransformOptions & CodeGenOptions & ParserOptions & TokenizeOptions;

declare type CompileResult = {
    code: string;
    ast: ResourceNode;
};

export declare type ComponetI18nScope = Exclude<I18nScope, 'local'>;

/**
 * Composer Interfaces
 *
 * @remarks
 * This is the interface for being used for Vue 3 Composition API.
 */
export declare interface Composer {
    /*!
     * properties
     */
    locale: WritableComputedRef<Locale>;
    fallbackLocale: WritableComputedRef<FallbackLocale>;
    inheritLocale: boolean;
    readonly availableLocales: Locale[];
    readonly messages: ComputedRef<LocaleMessages>;
    readonly datetimeFormats: ComputedRef<DateTimeFormats>;
    readonly numberFormats: ComputedRef<NumberFormats>;
    readonly modifiers: LinkedModifiers;
    readonly pluralRules?: PluralizationRules;
    readonly isGlobal: boolean;
    missingWarn: boolean | RegExp;
    fallbackWarn: boolean | RegExp;
    fallbackRoot: boolean;
    fallbackFormat: boolean;
    warnHtmlMessage: boolean;
    /*!
     * methods
     */
    t(key: Path): string;
    t(key: Path, plural: number): string;
    t(key: Path, plural: number, options: TranslateOptions): string;
    t(key: Path, defaultMsg: string): string;
    t(key: Path, defaultMsg: string, options: TranslateOptions): string;
    t(key: Path, list: unknown[]): string;
    t(key: Path, list: unknown[], plural: number): string;
    t(key: Path, list: unknown[], defaultMsg: string): string;
    t(key: Path, list: unknown[], options: TranslateOptions): string;
    t(key: Path, named: NamedValue): string;
    t(key: Path, named: NamedValue, plural: number): string;
    t(key: Path, named: NamedValue, defaultMsg: string): string;
    t(key: Path, named: NamedValue, options: TranslateOptions): string;
    t(...args: unknown[]): string;
    d(value: number | Date): string;
    d(value: number | Date, key: string): string;
    d(value: number | Date, key: string, locale: Locale): string;
    d(value: number | Date, options: DateTimeOptions): string;
    d(...args: unknown[]): string;
    n(value: number): string;
    n(value: number, key: string): string;
    n(value: number, key: string, locale: Locale): string;
    n(value: number, options: NumberOptions): string;
    n(...args: unknown[]): string;
    getLocaleMessage(locale: Locale): LocaleMessage;
    setLocaleMessage(locale: Locale, message: LocaleMessage): void;
    mergeLocaleMessage(locale: Locale, message: LocaleMessage): void;
    getDateTimeFormat(locale: Locale): DateTimeFormat;
    setDateTimeFormat(locale: Locale, format: DateTimeFormat): void;
    mergeDateTimeFormat(locale: Locale, format: DateTimeFormat): void;
    getNumberFormat(locale: Locale): NumberFormat;
    setNumberFormat(locale: Locale, format: NumberFormat): void;
    mergeNumberFormat(locale: Locale, format: NumberFormat): void;
    getPostTranslationHandler(): PostTranslationHandler | null;
    setPostTranslationHandler(handler: PostTranslationHandler | null): void;
    getMissingHandler(): MissingHandler | null;
    setMissingHandler(handler: MissingHandler | null): void;
}

/**
 * Composer additional options for `useI18n`
 *
 * @remarks
 * `ComposerAdditionalOptions` is extend for {@link ComposerOptions}, so you can specify these options.
 */
export declare interface ComposerAdditionalOptions {
    useScope?: I18nScope;
}

/**
 * Composer Options
 *
 * @remarks
 * This is options to create composer.
 */
export declare interface ComposerOptions {
    locale?: Locale;
    fallbackLocale?: FallbackLocale;
    inheritLocale?: boolean;
    messages?: LocaleMessages;
    datetimeFormats?: DateTimeFormats;
    numberFormats?: NumberFormats;
    modifiers?: LinkedModifiers;
    pluralRules?: PluralizationRules;
    missing?: MissingHandler;
    missingWarn?: boolean | RegExp;
    fallbackWarn?: boolean | RegExp;
    fallbackRoot?: boolean;
    fallbackFormat?: boolean;
    postTranslation?: PostTranslationHandler;
    warnHtmlMessage?: boolean;
}

/**
 * I18n factory function
 *
 * @param options - see the {@link I18nOptions}
 * @returns {@link I18n} object
 *
 * @remarks
 * When you use Composable API, you need to specify options of {@link ComposerOptions}.
 * When you use Legacy API, you need toto specify options of {@link VueI18nOptions} and `legacy: true` option.
 *
 * @example
 * case: for Composable API
 * ```js
 * import { createApp } from 'vue'
 * import { createI18n, useI18n } from 'vue-i18n'
 *
 * // call with I18n option
 * const i18n = createI18n({
 *   locale: 'ja',
 *   messages: {
 *     en: { ... },
 *     ja: { ... }
 *   }
 * })
 *
 * const App = {
 *   setup() {
 *     // ...
 *     const { t } = useI18n({ ... })
 *     return { ... , t }
 *   }
 * }
 *
 * const app = createApp(App)
 *
 * // install!
 * app.use(i18n)
 * app.mount('#app')
 * ```
 *
 * @example
 * case: for Legacy API
 * ```js
 * import { createApp } from 'vue'
 * import { createI18n } from 'vue-i18n'
 *
 * // call with I18n option
 * const i18n = createI18n({
 *   legacy: true, // you must specify 'lagacy: true' option
 *   locale: 'ja',
 *   messages: {
 *     en: { ... },
 *     ja: { ... }
 *   }
 * })
 *
 * const App = {
 *   // ...
 * }
 *
 * const app = createApp(App)
 *
 * // install!
 * app.use(i18n)
 * app.mount('#app')
 * ```
 */
export declare function createI18n(options?: I18nOptions): I18n;

export declare function createParser(options?: ParserOptions): Parser;

export declare function createRuntimeContext(options?: RuntimeOptions): RuntimeContext;

/**
 *  number
 */
declare type CurrencyDisplay = 'symbol' | 'code' | 'name';

declare interface CurrencyNumberFormatOptions extends Intl.NumberFormatOptions {
    style: 'currency';
    currency: string;
    currencyDisplay?: CurrencyDisplay;
    localeMatcher?: 'lookup' | 'best-fit';
    formatMatcher?: 'basic' | 'best-fit';
}

export declare type CustomBlocks = string[] | PreCompileHandler;

export declare function datetime(context: RuntimeContext, value: number | Date): string | number | Intl.DateTimeFormatPart[];

export declare function datetime(context: RuntimeContext, value: number | Date, key: string): string | number | Intl.DateTimeFormatPart[];

export declare function datetime(context: RuntimeContext, value: number | Date, key: string, locale: Locale): string | number | Intl.DateTimeFormatPart[];

export declare function datetime(context: RuntimeContext, value: number | Date, options: DateTimeOptions): string | number | Intl.DateTimeFormatPart[];

export declare function datetime(context: RuntimeContext, ...args: unknown[]): string | number | Intl.DateTimeFormatPart[];

declare type DateTimeDigital = 'numeric' | '2-digit';

declare type DateTimeFormat = {
    [key: string]: DateTimeFormatOptions;
};

declare type DateTimeFormatOptions = Intl.DateTimeFormatOptions | SpecificDateTimeFormatOptions;

export declare type DatetimeFormatProps = FormattableProps<number | Date, Intl.DateTimeFormatOptions>;

export declare type DateTimeFormatResult = string;

declare type DateTimeFormats = {
    [locale: string]: DateTimeFormat;
};

declare type DateTimeHumanReadable = 'long' | 'short' | 'narrow';

/**
 *  # datetime
 *
 *  ## usages:
 *    // for example `context.datetimeFormats` below
 *    'en-US': {
 *      short: {
 *        year: 'numeric', month: '2-digit', day: '2-digit',
 *        hour: '2-digit', minute: '2-digit'
 *      }
 *    },
 *    'ja-JP': { ... }
 *
 *    // datetimeable value only
 *    datetime(context, value)
 *
 *    // key argument
 *    datetime(context, value, 'short')
 *
 *    // key & locale argument
 *    datetime(context, value, 'short', 'ja-JP')
 *
 *    // object sytle argument
 *    datetime(context, value, { key: 'short', locale: 'ja-JP' })
 *
 *    // suppress localize miss warning option, override context.missingWarn
 *    datetime(context, value, { key: 'short', locale: 'ja-JP', missingWarn: false })
 *
 *    // suppress localize fallback warning option, override context.fallbackWarn
 *    datetime(context, value, { key: 'short', locale: 'ja-JP', fallbackWarn: false })
 *
 *    // if you specify `part` options, you can get an array of objects containing the formatted datetime in parts
 *    datetime(context, value, { key: 'short', part: true })
 *
 *    // orverride context.datetimeFormats[locale] options with functino options
 *    datetime(cnotext, value, 'short', { currency: 'EUR' })
 *    datetime(cnotext, value, 'short', 'ja-JP', { currency: 'EUR' })
 *    datetime(context, value, { key: 'short', part: true }, { currency: 'EUR'})
 */
export declare type DateTimeOptions = {
    key?: string;
    locale?: Locale;
    missingWarn?: boolean;
    fallbackWarn?: boolean;
    part?: boolean;
};

export declare type FallbackLocale = Locale | Locale[] | {
    [locale in string]: Locale[];
} | false;

export declare interface FormattableProps<Value, Format> extends BaseFormatProps {
    value: Value;
    format?: string | Format;
}

export declare interface Formatter {
    interpolate(message: string, values: any, path: string): Array<any> | null;
}

export declare const friendlyJSONstringify: (json: unknown) => string;

export declare const generateFormatCacheKey: (locale: string, key: string, source: string) => string;

export declare function getLocaleChain(context: RuntimeContext, fallback: FallbackLocale, start?: Locale): Locale[];

export declare function handleMissing(context: RuntimeContext, key: Path, locale: Locale, missingWarn: boolean | RegExp, type: string): unknown;

/**
 * I18n interface
 */
export declare interface I18n {
    /**
     * I18n API mode
     *
     * @remarks
     * if you specified `legacy: true` option in `createI18n`, return `legacy`,
     * else `composable`
     *
     * @defaultValue `composable`
     */
    readonly mode: I18nMode;
    /**
     * Global composer
     */
    readonly global: Composer;
    /**
     * @internal
     */
    install(app: App, ...options: unknown[]): void;
}

/**
 * I18n Additional Options for `createI18n`
 */
export declare interface I18nAdditionalOptions {
    /**
     * Whether vue-i18n legacy API use on your Vue App.
     *
     * @defaultValue `false`
     */
    legacy?: boolean;
}

/**
 * I18n API mode
 */
export declare type I18nMode = 'legacy' | 'composable';

/**
 * I18n Options for `createI18n`
 *
 * @remarks
 * `I18nOptions` is inherited {@link I18nAdditionalOptions}, {@link ComposerOptions} and {@link VueI18nOptions},
 * so you can specify these options.
 *
 */
export declare type I18nOptions = I18nAdditionalOptions & (ComposerOptions | VueI18nOptions);

/**
 * I18n plugin options
 *
 * @remarks
 * An options specified when installing vue-i18n as Vue plugin with using `app.use`.
 */
export declare interface I18nPluginOptions {
    useI18nComponentName?: boolean;
}

/**
 * I18n Scope
 */
export declare type I18nScope = 'local' | 'parent' | 'global';

declare type Identifier = string;

export declare function isTranslateMissingWarn(missing: boolean | RegExp, key: Path): boolean;

export declare function isTrarnslateFallbackWarn(fallback: boolean | RegExp, key: string): boolean;

declare interface LinkedKeyNode extends Node_2 {
    type: NodeTypes.LinkedKey;
    value: string;
}

export declare type LinkedModifiers = {
    [key: string]: LinkedModify;
};

declare type LinkedModify = (value: unknown, type: string) => unknown;

declare interface LinkedModitierNode extends Node_2 {
    type: NodeTypes.LinkedModifier;
    value: Identifier;
}

declare interface LinkedNode extends Node_2 {
    type: NodeTypes.Linked;
    modifier?: LinkedModitierNode;
    key: LinkedKeyNode | NamedNode | ListNode | LiteralNode;
}

declare interface ListNode extends Node_2 {
    type: NodeTypes.List;
    index: number;
}

declare interface LiteralNode extends Node_2 {
    type: NodeTypes.Literal;
    value: string;
}

export declare type Locale = string;

export declare type LocaleMessage = string | MessageFunction | LocaleMessageDictionary | LocaleMessage[];

export declare type LocaleMessageDictionary = {
    [property: string]: LocaleMessage;
};

export declare type LocaleMessageObject = LocaleMessageDictionary;

export declare type LocaleMessages = Record<Locale, LocaleMessage>;

export declare type MessageCompiler = (source: string, options?: CompileOptions) => MessageFunction;

declare type MessageContext = {
    list: (index: number) => unknown;
    named: (key: string) => unknown;
    pluralIndex: number;
    pluralRule: PluralizationRule;
    orgPluralRule?: PluralizationRule;
    modifier: (name: string) => LinkedModify;
    message: (name: string) => MessageFunction;
    type: string;
    interpolate: MessageInterpolate;
    normalize: MessageNormalize;
};

declare type MessageElementNode = TextNode | NamedNode | ListNode | LiteralNode | LinkedNode;

export declare type MessageFunction = {
    (ctx: MessageContext): unknown;
    key?: string;
    locale?: string;
    source?: string;
};

export declare type MessageFunctions = Record<string, MessageFunction>;

declare type MessageInterpolate = (val: unknown) => unknown;

declare interface MessageNode extends Node_2 {
    type: NodeTypes.Message;
    items: MessageElementNode[];
}

declare type MessageNormalize = (values: unknown[]) => unknown;

declare interface MessageProcessor {
    type?: string;
    interpolate?: MessageInterpolate;
    normalize?: MessageNormalize;
}

export declare const MISSING_RESOLVE_VALUE = "";

export declare type MissingHandler = (locale: Locale, key: Path, insttance?: ComponentInternalInstance, type?: string) => string | void;

declare interface NamedNode extends Node_2 {
    type: NodeTypes.Named;
    key: Identifier;
}

declare type NamedValue<T = {}> = T & Record<string, unknown>;

declare interface Node_2 {
    type: NodeTypes;
    start: number;
    end: number;
    loc?: SourceLocation;
}

declare const enum NodeTypes {
    Resource = 0,
    Plural = 1,
    Message = 2,
    Text = 3,
    Named = 4,
    List = 5,
    Linked = 6,
    LinkedKey = 7,
    LinkedModifier = 8,
    Literal = 9
}

export declare const NOT_REOSLVED = -1;

export declare function number(context: RuntimeContext, value: number): string | number | Intl.NumberFormatPart[];

export declare function number(context: RuntimeContext, value: number, key: string): string | number | Intl.NumberFormatPart[];

export declare function number(context: RuntimeContext, value: number, key: string, locale: Locale): string | number | Intl.NumberFormatPart[];

export declare function number(context: RuntimeContext, value: number, options: NumberOptions): string | number | Intl.NumberFormatPart[];

export declare function number(context: RuntimeContext, ...args: unknown[]): string | number | Intl.NumberFormatPart[];

declare type NumberFormat = {
    [key: string]: NumberFormatOptions;
};

declare type NumberFormatOptions = Intl.NumberFormatOptions | SpecificNumberFormatOptions | CurrencyNumberFormatOptions;

export declare type NumberFormatProps = FormattableProps<number, Intl.NumberFormatOptions>;

export declare type NumberFormatResult = string;

declare type NumberFormats = {
    [locale: string]: NumberFormat;
};

/**
 *  # number
 *
 *  ## usages
 *    // for example `context.numberFormats` below
 *    'en-US': {
 *      'currency': {
 *        style: 'currency', currency: 'USD', currencyDisplay: 'symbol'
 *      }
 *    },
 *    'ja-JP: { ... }
 *
 *    // value only
 *    number(context, value)
 *
 *    // key argument
 *    number(context, value, 'currency')
 *
 *    // key & locale argument
 *    number(context, value, 'currency', 'ja-JP')
 *
 *    // object sytle argument
 *    number(context, value, { key: 'currency', locale: 'ja-JP' })
 *
 *    // suppress localize miss warning option, override context.missingWarn
 *    number(context, value, { key: 'currency', locale: 'ja-JP', missingWarn: false })
 *
 *    // suppress localize fallback warning option, override context.fallbackWarn
 *    number(context, value, { key: 'currency', locale: 'ja-JP', fallbackWarn: false })
 *
 *    // if you specify `part` options, you can get an array of objects containing the formatted number in parts
 *    number(context, value, { key: 'currenty', part: true })
 *
 *    // orverride context.numberFormats[locale] options with functino options
 *    number(cnotext, value, 'currency', { year: '2-digit' })
 *    number(cnotext, value, 'currency', 'ja-JP', { year: '2-digit' })
 *    number(context, value, { key: 'currenty', part: true }, { year: '2-digit'})
 */
export declare type NumberOptions = {
    key?: string;
    locale?: Locale;
    missingWarn?: boolean;
    fallbackWarn?: boolean;
    part?: boolean;
};

export declare function parseDateTimeArgs(...args: unknown[]): [string, number | Date, DateTimeOptions, Intl.DateTimeFormatOptions];

export declare function parseNumberArgs(...args: unknown[]): [string, number, NumberOptions, Intl.NumberFormatOptions];

export declare type Parser = Readonly<{
    parse: (source: string) => ResourceNode;
}>;

declare type ParserOptions = {
    location?: boolean;
    onError?: CompileErrorHandler;
};

export declare function parseTranslateArgs(...args: unknown[]): [Path, TranslateOptions];

export declare type Path = string;

export declare type PathValue = string | number | boolean | Function | null | {
    [key: string]: PathValue;
} | PathValue[];

export declare type PluralizationRule = (choice: number, choicesLength: number, orgRule?: PluralizationRule) => number;

declare type PluralizationRules = {
    [locale: string]: PluralizationRule;
};

export declare type PluralizationRulesMap = {
    [locale: string]: PluralizationRule;
};

declare interface PluralNode extends Node_2 {
    type: NodeTypes.Plural;
    cases: MessageNode[];
}

declare interface Position_2 {
    offset: number;
    line: number;
    column: number;
}

export declare type PostTranslationHandler = (translated: unknown) => unknown;

declare type PreCompileHandler = () => {
    messages: LocaleMessages;
    functions: MessageFunctions;
};

declare interface ResourceNode extends Node_2 {
    type: NodeTypes.Resource;
    body: MessageNode | PluralNode;
    helpers?: string[];
}

export declare type RuntimeContext = {
    locale: Locale;
    fallbackLocale: FallbackLocale;
    messages: LocaleMessages;
    datetimeFormats: DateTimeFormats;
    numberFormats: NumberFormats;
    modifiers: LinkedModifiers;
    pluralRules?: PluralizationRules;
    missing: RuntimeMissingHandler | null;
    missingWarn: boolean | RegExp;
    fallbackWarn: boolean | RegExp;
    fallbackFormat: boolean;
    unresolving: boolean;
    postTranslation: PostTranslationHandler | null;
    processor: MessageProcessor | null;
    warnHtmlMessage: boolean;
    messageCompiler: MessageCompiler;
    onWarn: (msg: string, err?: Error) => void;
    _datetimeFormatters: Map<string, Intl.DateTimeFormat>;
    _numberFormatters: Map<string, Intl.NumberFormat>;
    _fallbackLocaleStack?: Locale[];
    _localeChainCache?: Map<Locale, Locale[]>;
};

export declare type RuntimeMissingHandler = (context: RuntimeContext, locale: Locale, key: Path, type: string, ...values: unknown[]) => string | void;

export declare type RuntimeOptions = {
    locale?: Locale;
    fallbackLocale?: FallbackLocale;
    messages?: LocaleMessages;
    datetimeFormats?: DateTimeFormats;
    numberFormats?: NumberFormats;
    modifiers?: LinkedModifiers;
    pluralRules?: PluralizationRules;
    missing?: RuntimeMissingHandler;
    missingWarn?: boolean | RegExp;
    fallbackWarn?: boolean | RegExp;
    fallbackFormat?: boolean;
    unresolving?: boolean;
    postTranslation?: PostTranslationHandler;
    processor?: MessageProcessor;
    warnHtmlMessage?: boolean;
    messageCompiler?: MessageCompiler;
    onWarn?: (msg: string, err?: Error) => void;
    _datetimeFormatters?: Map<string, Intl.DateTimeFormat>;
    _numberFormatters?: Map<string, Intl.NumberFormat>;
};

declare interface SourceLocation {
    start: Position_2;
    end: Position_2;
    source?: string;
}

declare interface SpecificDateTimeFormatOptions extends Intl.DateTimeFormatOptions {
    year?: DateTimeDigital;
    month?: DateTimeDigital | DateTimeHumanReadable;
    day?: DateTimeDigital;
    hour?: DateTimeDigital;
    minute?: DateTimeDigital;
    second?: DateTimeDigital;
    weekday?: DateTimeHumanReadable;
    era?: DateTimeHumanReadable;
    timeZoneName?: 'long' | 'short';
    localeMatcher?: 'lookup' | 'best-fit';
    formatMatcher?: 'basic' | 'best-fit';
}

declare interface SpecificNumberFormatOptions extends Intl.NumberFormatOptions {
    style?: 'decimal' | 'percent';
    currency?: string;
    currencyDisplay?: CurrencyDisplay;
    localeMatcher?: 'lookup' | 'best-fit';
    formatMatcher?: 'basic' | 'best-fit';
}

declare interface TextNode extends Node_2 {
    type: NodeTypes.Text;
    value: string;
}

declare type TokenizeOptions = {
    location?: boolean;
    onError?: CompileErrorHandler;
};

declare type TransformOptions = {
    onError?: CompileErrorHandler;
};

export declare function translate(context: RuntimeContext, key: Path): unknown;

export declare function translate(context: RuntimeContext, key: Path, plural: number): unknown;

export declare function translate(context: RuntimeContext, key: Path, plural: number, options: TranslateOptions): unknown;

export declare function translate(context: RuntimeContext, key: Path, defaultMsg: string): unknown;

export declare function translate(context: RuntimeContext, key: Path, defaultMsg: string, options: TranslateOptions): unknown;

export declare function translate(context: RuntimeContext, key: Path, list: unknown[]): unknown;

export declare function translate(context: RuntimeContext, key: Path, list: unknown[], plural: number): unknown;

export declare function translate(context: RuntimeContext, key: Path, list: unknown[], defaultMsg: string): unknown;

export declare function translate(context: RuntimeContext, key: Path, list: unknown[], options: TranslateOptions): unknown;

export declare function translate(context: RuntimeContext, key: Path, named: NamedValue): unknown;

export declare function translate(context: RuntimeContext, key: Path, named: NamedValue, plural: number): unknown;

export declare function translate(context: RuntimeContext, key: Path, named: NamedValue, defaultMsg: string): unknown;

export declare function translate(context: RuntimeContext, key: Path, named: NamedValue, options: TranslateOptions): unknown;

export declare function translate(context: RuntimeContext, ...args: unknown[]): unknown;

/**
 *  # translate
 *
 *  ## usages:
 *    // for example, locale messages key
 *    { 'foo.bar': 'hi {0} !' or 'hi {name} !' }
 *
 *    // no argument, context & path only
 *    translate(context, 'foo.bar')
 *
 *    // list argument
 *    translate(context, 'foo.bar', ['kazupon'])
 *
 *    // named argument
 *    translate(context, 'foo.bar', { name: 'kazupon' })
 *
 *    // plural choice number
 *    translate(context, 'foo.bar', 2)
 *
 *    // plural choice number with name argument
 *    translate(context, 'foo.bar', { name: 'kazupon' }, 2)
 *
 *    // default message argument
 *    translate(context, 'foo.bar', 'this is default message')
 *
 *    // default message with named argument
 *    translate(context, 'foo.bar', { name: 'kazupon' }, 'Hello {name} !')
 *
 *    // use key as default message
 *    translate(context, 'hi {0} !', ['kazupon'], { default: true })
 *
 *    // locale option, override context.locale
 *    translate(context, 'foo.bar', { name: 'kazupon' }, { locale: 'ja' })
 *
 *    // suppress localize miss warning option, override context.missingWarn
 *    translate(context, 'foo.bar', { name: 'kazupon' }, { missingWarn: false })
 *
 *    // suppress localize fallback warning option, override context.fallbackWarn
 *    translate(context, 'foo.bar', { name: 'kazupon' }, { fallbackWarn: false })
 */
export declare type TranslateOptions = {
    list?: unknown[];
    named?: NamedValue;
    plural?: number;
    default?: string | boolean;
    locale?: Locale;
    missingWarn?: boolean;
    fallbackWarn?: boolean;
};

export declare type TranslateResult = string;

export declare interface TranslationProps extends BaseFormatProps {
    keypath: string;
    plural?: number | string;
}

export declare function updateFallbackLocale(context: RuntimeContext, locale: Locale, fallback: FallbackLocale): void;

/**
 * Use Composable API starting function
 *
 * @param options - See {@link UseI18nOptions}
 * @returns {@link Composer} object
 *
 * @remarks
 * This function is mainly used by `setup`.
 * If options are specified, Composer object is created for each component and you can be localized on the component.
 * If options are not specified, you can be localized using the global Composer.
 *
 * @example
 * case: Component resource base localization
 * ```html
 * <template>
 *   <form>
 *     <label>{{ t('language') }}</label>
 *     <select v-model="locale">
 *       <option value="en">en</option>
 *       <option value="ja">ja</option>
 *     </select>
 *   </form>
 *   <p>message: {{ t('hello') }}</p>
 * </template>
 *
 * <script>
 * import { useI18n } from 'vue-i18n'
 *
 * export default {
 *  setup() {
 *    const { t, locale } = useI18n({
 *      locale: 'ja',
 *      messages: {
 *        en: { ... },
 *        ja: { ... }
 *      }
 *    })
 *    // Something to do ...
 *
 *    return { ..., t, locale }
 *  }
 * }
 * </script>
 * ```
 */
export declare function useI18n(options?: UseI18nOptions): Composer;

/**
 * I18n Options for `useI18n`
 *
 * @remarks
 * `UseI18nOptions` is inherited {@link ComposerAdditionalOptions} and {@link ComposerOptions},
 * so you can specify these options.
 */
export declare type UseI18nOptions = ComposerAdditionalOptions & ComposerOptions;

/**
 * vue-i18n version
 */
export declare const VERSION: string;

/**
 *  VueI18n Interfaces
 *
 *  @remarks
 *  This interface is compatible with interface of `VueI18n` class (offered with vue-i18n@8.x).
 */
export declare interface VueI18n {
    /*!
     * properties
     */
    locale: Locale;
    fallbackLocale: FallbackLocale;
    readonly availableLocales: Locale[];
    readonly messages: LocaleMessages;
    readonly datetimeFormats: DateTimeFormats;
    readonly numberFormats: NumberFormats;
    formatter: Formatter;
    missing: MissingHandler | null;
    postTranslation: PostTranslationHandler | null;
    silentTranslationWarn: boolean | RegExp;
    silentFallbackWarn: boolean | RegExp;
    formatFallbackMessages: boolean;
    sync: boolean;
    warnHtmlInMessage: WarnHtmlInMessageLevel;
    preserveDirectiveContent: boolean;
    /*!
     * methods
     */
    t(key: Path): TranslateResult;
    t(key: Path, locale: Locale): TranslateResult;
    t(key: Path, locale: Locale, list: unknown[]): TranslateResult;
    t(key: Path, locale: Locale, named: object): TranslateResult;
    t(key: Path, list: unknown[]): TranslateResult;
    t(key: Path, named: Record<string, unknown>): TranslateResult;
    t(...args: unknown[]): TranslateResult;
    tc(key: Path): TranslateResult;
    tc(key: Path, locale: Locale): TranslateResult;
    tc(key: Path, list: unknown[]): TranslateResult;
    tc(key: Path, named: Record<string, unknown>): TranslateResult;
    tc(key: Path, choice: number): TranslateResult;
    tc(key: Path, choice: number, locale: Locale): TranslateResult;
    tc(key: Path, choice: number, list: unknown[]): TranslateResult;
    tc(key: Path, choice: number, named: Record<string, unknown>): TranslateResult;
    tc(...args: unknown[]): TranslateResult;
    te(key: Path, locale?: Locale): boolean;
    getLocaleMessage(locale: Locale): LocaleMessage;
    setLocaleMessage(locale: Locale, message: LocaleMessage): void;
    mergeLocaleMessage(locale: Locale, message: LocaleMessage): void;
    d(value: number | Date): DateTimeFormatResult;
    d(value: number | Date, key: string): DateTimeFormatResult;
    d(value: number | Date, key: string, locale: Locale): DateTimeFormatResult;
    d(value: number | Date, args: {
        [key: string]: string;
    }): DateTimeFormatResult;
    d(...args: unknown[]): DateTimeFormatResult;
    getDateTimeFormat(locale: Locale): DateTimeFormat;
    setDateTimeFormat(locale: Locale, format: DateTimeFormat): void;
    mergeDateTimeFormat(locale: Locale, format: DateTimeFormat): void;
    n(value: number): NumberFormatResult;
    n(value: number, key: string): NumberFormatResult;
    n(value: number, key: string, locale: Locale): NumberFormatResult;
    n(value: number, args: {
        [key: string]: string;
    }): NumberFormatResult;
    n(...args: unknown[]): NumberFormatResult;
    getNumberFormat(locale: Locale): NumberFormat;
    setNumberFormat(locale: Locale, format: NumberFormat): void;
    mergeNumberFormat(locale: Locale, format: NumberFormat): void;
    getChoiceIndex: (choice: Choice, choicesLength: number) => number;
}

/**
 *  VueI18n Options
 *
 *  @remarks
 *  This option is compatible with the constructor options of `VueI18n` class (offered with vue-i18n@8.x).
 */
export declare interface VueI18nOptions {
    locale?: Locale;
    fallbackLocale?: FallbackLocale;
    messages?: LocaleMessages;
    datetimeFormats?: DateTimeFormats;
    numberFormats?: NumberFormats;
    availableLocales?: Locale[];
    modifiers?: LinkedModifiers;
    formatter?: Formatter;
    missing?: MissingHandler;
    fallbackRoot?: boolean;
    silentTranslationWarn?: boolean | RegExp;
    silentFallbackWarn?: boolean | RegExp;
    formatFallbackMessages?: boolean;
    preserveDirectiveContent?: boolean;
    warnHtmlInMessage?: WarnHtmlInMessageLevel;
    sharedMessages?: LocaleMessages;
    pluralizationRules?: PluralizationRules;
    postTranslation?: PostTranslationHandler;
    sync?: boolean;
}

export declare type WarnHtmlInMessageLevel = 'off' | 'warn' | 'error';

export { }
