(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
  typeof define === 'function' && define.amd ? define(['exports'], factory) :
  (factory((global.VueTimeago = {})));
}(this, (function (exports) { 'use strict';

  /**
   * @category Common Helpers
   * @summary Is the given argument an instance of Date?
   *
   * @description
   * Is the given argument an instance of Date?
   *
   * @param {*} argument - the argument to check
   * @returns {Boolean} the given argument is an instance of Date
   *
   * @example
   * // Is 'mayonnaise' a Date?
   * var result = isDate('mayonnaise')
   * //=> false
   */
  function isDate (argument) {
    return argument instanceof Date
  }

  var is_date = isDate;

  var MILLISECONDS_IN_HOUR = 3600000;
  var MILLISECONDS_IN_MINUTE = 60000;
  var DEFAULT_ADDITIONAL_DIGITS = 2;

  var parseTokenDateTimeDelimeter = /[T ]/;
  var parseTokenPlainTime = /:/;

  // year tokens
  var parseTokenYY = /^(\d{2})$/;
  var parseTokensYYY = [
    /^([+-]\d{2})$/, // 0 additional digits
    /^([+-]\d{3})$/, // 1 additional digit
    /^([+-]\d{4})$/ // 2 additional digits
  ];

  var parseTokenYYYY = /^(\d{4})/;
  var parseTokensYYYYY = [
    /^([+-]\d{4})/, // 0 additional digits
    /^([+-]\d{5})/, // 1 additional digit
    /^([+-]\d{6})/ // 2 additional digits
  ];

  // date tokens
  var parseTokenMM = /^-(\d{2})$/;
  var parseTokenDDD = /^-?(\d{3})$/;
  var parseTokenMMDD = /^-?(\d{2})-?(\d{2})$/;
  var parseTokenWww = /^-?W(\d{2})$/;
  var parseTokenWwwD = /^-?W(\d{2})-?(\d{1})$/;

  // time tokens
  var parseTokenHH = /^(\d{2}([.,]\d*)?)$/;
  var parseTokenHHMM = /^(\d{2}):?(\d{2}([.,]\d*)?)$/;
  var parseTokenHHMMSS = /^(\d{2}):?(\d{2}):?(\d{2}([.,]\d*)?)$/;

  // timezone tokens
  var parseTokenTimezone = /([Z+-].*)$/;
  var parseTokenTimezoneZ = /^(Z)$/;
  var parseTokenTimezoneHH = /^([+-])(\d{2})$/;
  var parseTokenTimezoneHHMM = /^([+-])(\d{2}):?(\d{2})$/;

  /**
   * @category Common Helpers
   * @summary Convert the given argument to an instance of Date.
   *
   * @description
   * Convert the given argument to an instance of Date.
   *
   * If the argument is an instance of Date, the function returns its clone.
   *
   * If the argument is a number, it is treated as a timestamp.
   *
   * If an argument is a string, the function tries to parse it.
   * Function accepts complete ISO 8601 formats as well as partial implementations.
   * ISO 8601: http://en.wikipedia.org/wiki/ISO_8601
   *
   * If all above fails, the function passes the given argument to Date constructor.
   *
   * @param {Date|String|Number} argument - the value to convert
   * @param {Object} [options] - the object with options
   * @param {0 | 1 | 2} [options.additionalDigits=2] - the additional number of digits in the extended year format
   * @returns {Date} the parsed date in the local time zone
   *
   * @example
   * // Convert string '2014-02-11T11:30:30' to date:
   * var result = parse('2014-02-11T11:30:30')
   * //=> Tue Feb 11 2014 11:30:30
   *
   * @example
   * // Parse string '+02014101',
   * // if the additional number of digits in the extended year format is 1:
   * var result = parse('+02014101', {additionalDigits: 1})
   * //=> Fri Apr 11 2014 00:00:00
   */
  function parse (argument, dirtyOptions) {
    if (is_date(argument)) {
      // Prevent the date to lose the milliseconds when passed to new Date() in IE10
      return new Date(argument.getTime())
    } else if (typeof argument !== 'string') {
      return new Date(argument)
    }

    var options = dirtyOptions || {};
    var additionalDigits = options.additionalDigits;
    if (additionalDigits == null) {
      additionalDigits = DEFAULT_ADDITIONAL_DIGITS;
    } else {
      additionalDigits = Number(additionalDigits);
    }

    var dateStrings = splitDateString(argument);

    var parseYearResult = parseYear(dateStrings.date, additionalDigits);
    var year = parseYearResult.year;
    var restDateString = parseYearResult.restDateString;

    var date = parseDate(restDateString, year);

    if (date) {
      var timestamp = date.getTime();
      var time = 0;
      var offset;

      if (dateStrings.time) {
        time = parseTime(dateStrings.time);
      }

      if (dateStrings.timezone) {
        offset = parseTimezone(dateStrings.timezone);
      } else {
        // get offset accurate to hour in timezones that change offset
        offset = new Date(timestamp + time).getTimezoneOffset();
        offset = new Date(timestamp + time + offset * MILLISECONDS_IN_MINUTE).getTimezoneOffset();
      }

      return new Date(timestamp + time + offset * MILLISECONDS_IN_MINUTE)
    } else {
      return new Date(argument)
    }
  }

  function splitDateString (dateString) {
    var dateStrings = {};
    var array = dateString.split(parseTokenDateTimeDelimeter);
    var timeString;

    if (parseTokenPlainTime.test(array[0])) {
      dateStrings.date = null;
      timeString = array[0];
    } else {
      dateStrings.date = array[0];
      timeString = array[1];
    }

    if (timeString) {
      var token = parseTokenTimezone.exec(timeString);
      if (token) {
        dateStrings.time = timeString.replace(token[1], '');
        dateStrings.timezone = token[1];
      } else {
        dateStrings.time = timeString;
      }
    }

    return dateStrings
  }

  function parseYear (dateString, additionalDigits) {
    var parseTokenYYY = parseTokensYYY[additionalDigits];
    var parseTokenYYYYY = parseTokensYYYYY[additionalDigits];

    var token;

    // YYYY or ±YYYYY
    token = parseTokenYYYY.exec(dateString) || parseTokenYYYYY.exec(dateString);
    if (token) {
      var yearString = token[1];
      return {
        year: parseInt(yearString, 10),
        restDateString: dateString.slice(yearString.length)
      }
    }

    // YY or ±YYY
    token = parseTokenYY.exec(dateString) || parseTokenYYY.exec(dateString);
    if (token) {
      var centuryString = token[1];
      return {
        year: parseInt(centuryString, 10) * 100,
        restDateString: dateString.slice(centuryString.length)
      }
    }

    // Invalid ISO-formatted year
    return {
      year: null
    }
  }

  function parseDate (dateString, year) {
    // Invalid ISO-formatted year
    if (year === null) {
      return null
    }

    var token;
    var date;
    var month;
    var week;

    // YYYY
    if (dateString.length === 0) {
      date = new Date(0);
      date.setUTCFullYear(year);
      return date
    }

    // YYYY-MM
    token = parseTokenMM.exec(dateString);
    if (token) {
      date = new Date(0);
      month = parseInt(token[1], 10) - 1;
      date.setUTCFullYear(year, month);
      return date
    }

    // YYYY-DDD or YYYYDDD
    token = parseTokenDDD.exec(dateString);
    if (token) {
      date = new Date(0);
      var dayOfYear = parseInt(token[1], 10);
      date.setUTCFullYear(year, 0, dayOfYear);
      return date
    }

    // YYYY-MM-DD or YYYYMMDD
    token = parseTokenMMDD.exec(dateString);
    if (token) {
      date = new Date(0);
      month = parseInt(token[1], 10) - 1;
      var day = parseInt(token[2], 10);
      date.setUTCFullYear(year, month, day);
      return date
    }

    // YYYY-Www or YYYYWww
    token = parseTokenWww.exec(dateString);
    if (token) {
      week = parseInt(token[1], 10) - 1;
      return dayOfISOYear(year, week)
    }

    // YYYY-Www-D or YYYYWwwD
    token = parseTokenWwwD.exec(dateString);
    if (token) {
      week = parseInt(token[1], 10) - 1;
      var dayOfWeek = parseInt(token[2], 10) - 1;
      return dayOfISOYear(year, week, dayOfWeek)
    }

    // Invalid ISO-formatted date
    return null
  }

  function parseTime (timeString) {
    var token;
    var hours;
    var minutes;

    // hh
    token = parseTokenHH.exec(timeString);
    if (token) {
      hours = parseFloat(token[1].replace(',', '.'));
      return (hours % 24) * MILLISECONDS_IN_HOUR
    }

    // hh:mm or hhmm
    token = parseTokenHHMM.exec(timeString);
    if (token) {
      hours = parseInt(token[1], 10);
      minutes = parseFloat(token[2].replace(',', '.'));
      return (hours % 24) * MILLISECONDS_IN_HOUR +
        minutes * MILLISECONDS_IN_MINUTE
    }

    // hh:mm:ss or hhmmss
    token = parseTokenHHMMSS.exec(timeString);
    if (token) {
      hours = parseInt(token[1], 10);
      minutes = parseInt(token[2], 10);
      var seconds = parseFloat(token[3].replace(',', '.'));
      return (hours % 24) * MILLISECONDS_IN_HOUR +
        minutes * MILLISECONDS_IN_MINUTE +
        seconds * 1000
    }

    // Invalid ISO-formatted time
    return null
  }

  function parseTimezone (timezoneString) {
    var token;
    var absoluteOffset;

    // Z
    token = parseTokenTimezoneZ.exec(timezoneString);
    if (token) {
      return 0
    }

    // ±hh
    token = parseTokenTimezoneHH.exec(timezoneString);
    if (token) {
      absoluteOffset = parseInt(token[2], 10) * 60;
      return (token[1] === '+') ? -absoluteOffset : absoluteOffset
    }

    // ±hh:mm or ±hhmm
    token = parseTokenTimezoneHHMM.exec(timezoneString);
    if (token) {
      absoluteOffset = parseInt(token[2], 10) * 60 + parseInt(token[3], 10);
      return (token[1] === '+') ? -absoluteOffset : absoluteOffset
    }

    return 0
  }

  function dayOfISOYear (isoYear, week, day) {
    week = week || 0;
    day = day || 0;
    var date = new Date(0);
    date.setUTCFullYear(isoYear, 0, 4);
    var fourthOfJanuaryDay = date.getUTCDay() || 7;
    var diff = week * 7 + day + 1 - fourthOfJanuaryDay;
    date.setUTCDate(date.getUTCDate() + diff);
    return date
  }

  var parse_1 = parse;

  /**
   * @category Common Helpers
   * @summary Compare the two dates reverse chronologically and return -1, 0 or 1.
   *
   * @description
   * Compare the two dates and return -1 if the first date is after the second,
   * 1 if the first date is before the second or 0 if dates are equal.
   *
   * @param {Date|String|Number} dateLeft - the first date to compare
   * @param {Date|String|Number} dateRight - the second date to compare
   * @returns {Number} the result of the comparison
   *
   * @example
   * // Compare 11 February 1987 and 10 July 1989 reverse chronologically:
   * var result = compareDesc(
   *   new Date(1987, 1, 11),
   *   new Date(1989, 6, 10)
   * )
   * //=> 1
   *
   * @example
   * // Sort the array of dates in reverse chronological order:
   * var result = [
   *   new Date(1995, 6, 2),
   *   new Date(1987, 1, 11),
   *   new Date(1989, 6, 10)
   * ].sort(compareDesc)
   * //=> [
   * //   Sun Jul 02 1995 00:00:00,
   * //   Mon Jul 10 1989 00:00:00,
   * //   Wed Feb 11 1987 00:00:00
   * // ]
   */
  function compareDesc (dirtyDateLeft, dirtyDateRight) {
    var dateLeft = parse_1(dirtyDateLeft);
    var timeLeft = dateLeft.getTime();
    var dateRight = parse_1(dirtyDateRight);
    var timeRight = dateRight.getTime();

    if (timeLeft > timeRight) {
      return -1
    } else if (timeLeft < timeRight) {
      return 1
    } else {
      return 0
    }
  }

  var compare_desc = compareDesc;

  /**
   * @category Millisecond Helpers
   * @summary Get the number of milliseconds between the given dates.
   *
   * @description
   * Get the number of milliseconds between the given dates.
   *
   * @param {Date|String|Number} dateLeft - the later date
   * @param {Date|String|Number} dateRight - the earlier date
   * @returns {Number} the number of milliseconds
   *
   * @example
   * // How many milliseconds are between
   * // 2 July 2014 12:30:20.600 and 2 July 2014 12:30:21.700?
   * var result = differenceInMilliseconds(
   *   new Date(2014, 6, 2, 12, 30, 21, 700),
   *   new Date(2014, 6, 2, 12, 30, 20, 600)
   * )
   * //=> 1100
   */
  function differenceInMilliseconds (dirtyDateLeft, dirtyDateRight) {
    var dateLeft = parse_1(dirtyDateLeft);
    var dateRight = parse_1(dirtyDateRight);
    return dateLeft.getTime() - dateRight.getTime()
  }

  var difference_in_milliseconds = differenceInMilliseconds;

  /**
   * @category Second Helpers
   * @summary Get the number of seconds between the given dates.
   *
   * @description
   * Get the number of seconds between the given dates.
   *
   * @param {Date|String|Number} dateLeft - the later date
   * @param {Date|String|Number} dateRight - the earlier date
   * @returns {Number} the number of seconds
   *
   * @example
   * // How many seconds are between
   * // 2 July 2014 12:30:07.999 and 2 July 2014 12:30:20.000?
   * var result = differenceInSeconds(
   *   new Date(2014, 6, 2, 12, 30, 20, 0),
   *   new Date(2014, 6, 2, 12, 30, 7, 999)
   * )
   * //=> 12
   */
  function differenceInSeconds (dirtyDateLeft, dirtyDateRight) {
    var diff = difference_in_milliseconds(dirtyDateLeft, dirtyDateRight) / 1000;
    return diff > 0 ? Math.floor(diff) : Math.ceil(diff)
  }

  var difference_in_seconds = differenceInSeconds;

  /**
   * @category Month Helpers
   * @summary Get the number of calendar months between the given dates.
   *
   * @description
   * Get the number of calendar months between the given dates.
   *
   * @param {Date|String|Number} dateLeft - the later date
   * @param {Date|String|Number} dateRight - the earlier date
   * @returns {Number} the number of calendar months
   *
   * @example
   * // How many calendar months are between 31 January 2014 and 1 September 2014?
   * var result = differenceInCalendarMonths(
   *   new Date(2014, 8, 1),
   *   new Date(2014, 0, 31)
   * )
   * //=> 8
   */
  function differenceInCalendarMonths (dirtyDateLeft, dirtyDateRight) {
    var dateLeft = parse_1(dirtyDateLeft);
    var dateRight = parse_1(dirtyDateRight);

    var yearDiff = dateLeft.getFullYear() - dateRight.getFullYear();
    var monthDiff = dateLeft.getMonth() - dateRight.getMonth();

    return yearDiff * 12 + monthDiff
  }

  var difference_in_calendar_months = differenceInCalendarMonths;

  /**
   * @category Common Helpers
   * @summary Compare the two dates and return -1, 0 or 1.
   *
   * @description
   * Compare the two dates and return 1 if the first date is after the second,
   * -1 if the first date is before the second or 0 if dates are equal.
   *
   * @param {Date|String|Number} dateLeft - the first date to compare
   * @param {Date|String|Number} dateRight - the second date to compare
   * @returns {Number} the result of the comparison
   *
   * @example
   * // Compare 11 February 1987 and 10 July 1989:
   * var result = compareAsc(
   *   new Date(1987, 1, 11),
   *   new Date(1989, 6, 10)
   * )
   * //=> -1
   *
   * @example
   * // Sort the array of dates:
   * var result = [
   *   new Date(1995, 6, 2),
   *   new Date(1987, 1, 11),
   *   new Date(1989, 6, 10)
   * ].sort(compareAsc)
   * //=> [
   * //   Wed Feb 11 1987 00:00:00,
   * //   Mon Jul 10 1989 00:00:00,
   * //   Sun Jul 02 1995 00:00:00
   * // ]
   */
  function compareAsc (dirtyDateLeft, dirtyDateRight) {
    var dateLeft = parse_1(dirtyDateLeft);
    var timeLeft = dateLeft.getTime();
    var dateRight = parse_1(dirtyDateRight);
    var timeRight = dateRight.getTime();

    if (timeLeft < timeRight) {
      return -1
    } else if (timeLeft > timeRight) {
      return 1
    } else {
      return 0
    }
  }

  var compare_asc = compareAsc;

  /**
   * @category Month Helpers
   * @summary Get the number of full months between the given dates.
   *
   * @description
   * Get the number of full months between the given dates.
   *
   * @param {Date|String|Number} dateLeft - the later date
   * @param {Date|String|Number} dateRight - the earlier date
   * @returns {Number} the number of full months
   *
   * @example
   * // How many full months are between 31 January 2014 and 1 September 2014?
   * var result = differenceInMonths(
   *   new Date(2014, 8, 1),
   *   new Date(2014, 0, 31)
   * )
   * //=> 7
   */
  function differenceInMonths (dirtyDateLeft, dirtyDateRight) {
    var dateLeft = parse_1(dirtyDateLeft);
    var dateRight = parse_1(dirtyDateRight);

    var sign = compare_asc(dateLeft, dateRight);
    var difference = Math.abs(difference_in_calendar_months(dateLeft, dateRight));
    dateLeft.setMonth(dateLeft.getMonth() - sign * difference);

    // Math.abs(diff in full months - diff in calendar months) === 1 if last calendar month is not full
    // If so, result must be decreased by 1 in absolute value
    var isLastMonthNotFull = compare_asc(dateLeft, dateRight) === -sign;
    return sign * (difference - isLastMonthNotFull)
  }

  var difference_in_months = differenceInMonths;

  function buildDistanceInWordsLocale () {
    var distanceInWordsLocale = {
      lessThanXSeconds: {
        one: 'less than a second',
        other: 'less than {{count}} seconds'
      },

      xSeconds: {
        one: '1 second',
        other: '{{count}} seconds'
      },

      halfAMinute: 'half a minute',

      lessThanXMinutes: {
        one: 'less than a minute',
        other: 'less than {{count}} minutes'
      },

      xMinutes: {
        one: '1 minute',
        other: '{{count}} minutes'
      },

      aboutXHours: {
        one: 'about 1 hour',
        other: 'about {{count}} hours'
      },

      xHours: {
        one: '1 hour',
        other: '{{count}} hours'
      },

      xDays: {
        one: '1 day',
        other: '{{count}} days'
      },

      aboutXMonths: {
        one: 'about 1 month',
        other: 'about {{count}} months'
      },

      xMonths: {
        one: '1 month',
        other: '{{count}} months'
      },

      aboutXYears: {
        one: 'about 1 year',
        other: 'about {{count}} years'
      },

      xYears: {
        one: '1 year',
        other: '{{count}} years'
      },

      overXYears: {
        one: 'over 1 year',
        other: 'over {{count}} years'
      },

      almostXYears: {
        one: 'almost 1 year',
        other: 'almost {{count}} years'
      }
    };

    function localize (token, count, options) {
      options = options || {};

      var result;
      if (typeof distanceInWordsLocale[token] === 'string') {
        result = distanceInWordsLocale[token];
      } else if (count === 1) {
        result = distanceInWordsLocale[token].one;
      } else {
        result = distanceInWordsLocale[token].other.replace('{{count}}', count);
      }

      if (options.addSuffix) {
        if (options.comparison > 0) {
          return 'in ' + result
        } else {
          return result + ' ago'
        }
      }

      return result
    }

    return {
      localize: localize
    }
  }

  var build_distance_in_words_locale = buildDistanceInWordsLocale;

  var commonFormatterKeys = [
    'M', 'MM', 'Q', 'D', 'DD', 'DDD', 'DDDD', 'd',
    'E', 'W', 'WW', 'YY', 'YYYY', 'GG', 'GGGG',
    'H', 'HH', 'h', 'hh', 'm', 'mm',
    's', 'ss', 'S', 'SS', 'SSS',
    'Z', 'ZZ', 'X', 'x'
  ];

  function buildFormattingTokensRegExp (formatters) {
    var formatterKeys = [];
    for (var key in formatters) {
      if (formatters.hasOwnProperty(key)) {
        formatterKeys.push(key);
      }
    }

    var formattingTokens = commonFormatterKeys
      .concat(formatterKeys)
      .sort()
      .reverse();
    var formattingTokensRegExp = new RegExp(
      '(\\[[^\\[]*\\])|(\\\\)?' + '(' + formattingTokens.join('|') + '|.)', 'g'
    );

    return formattingTokensRegExp
  }

  var build_formatting_tokens_reg_exp = buildFormattingTokensRegExp;

  function buildFormatLocale () {
    // Note: in English, the names of days of the week and months are capitalized.
    // If you are making a new locale based on this one, check if the same is true for the language you're working on.
    // Generally, formatted dates should look like they are in the middle of a sentence,
    // e.g. in Spanish language the weekdays and months should be in the lowercase.
    var months3char = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
    var monthsFull = ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'];
    var weekdays2char = ['Su', 'Mo', 'Tu', 'We', 'Th', 'Fr', 'Sa'];
    var weekdays3char = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
    var weekdaysFull = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
    var meridiemUppercase = ['AM', 'PM'];
    var meridiemLowercase = ['am', 'pm'];
    var meridiemFull = ['a.m.', 'p.m.'];

    var formatters = {
      // Month: Jan, Feb, ..., Dec
      'MMM': function (date) {
        return months3char[date.getMonth()]
      },

      // Month: January, February, ..., December
      'MMMM': function (date) {
        return monthsFull[date.getMonth()]
      },

      // Day of week: Su, Mo, ..., Sa
      'dd': function (date) {
        return weekdays2char[date.getDay()]
      },

      // Day of week: Sun, Mon, ..., Sat
      'ddd': function (date) {
        return weekdays3char[date.getDay()]
      },

      // Day of week: Sunday, Monday, ..., Saturday
      'dddd': function (date) {
        return weekdaysFull[date.getDay()]
      },

      // AM, PM
      'A': function (date) {
        return (date.getHours() / 12) >= 1 ? meridiemUppercase[1] : meridiemUppercase[0]
      },

      // am, pm
      'a': function (date) {
        return (date.getHours() / 12) >= 1 ? meridiemLowercase[1] : meridiemLowercase[0]
      },

      // a.m., p.m.
      'aa': function (date) {
        return (date.getHours() / 12) >= 1 ? meridiemFull[1] : meridiemFull[0]
      }
    };

    // Generate ordinal version of formatters: M -> Mo, D -> Do, etc.
    var ordinalFormatters = ['M', 'D', 'DDD', 'd', 'Q', 'W'];
    ordinalFormatters.forEach(function (formatterToken) {
      formatters[formatterToken + 'o'] = function (date, formatters) {
        return ordinal(formatters[formatterToken](date))
      };
    });

    return {
      formatters: formatters,
      formattingTokensRegExp: build_formatting_tokens_reg_exp(formatters)
    }
  }

  function ordinal (number) {
    var rem100 = number % 100;
    if (rem100 > 20 || rem100 < 10) {
      switch (rem100 % 10) {
        case 1:
          return number + 'st'
        case 2:
          return number + 'nd'
        case 3:
          return number + 'rd'
      }
    }
    return number + 'th'
  }

  var build_format_locale = buildFormatLocale;

  /**
   * @category Locales
   * @summary English locale.
   */
  var en = {
    distanceInWords: build_distance_in_words_locale(),
    format: build_format_locale()
  };

  var MINUTES_IN_DAY = 1440;
  var MINUTES_IN_ALMOST_TWO_DAYS = 2520;
  var MINUTES_IN_MONTH = 43200;
  var MINUTES_IN_TWO_MONTHS = 86400;

  /**
   * @category Common Helpers
   * @summary Return the distance between the given dates in words.
   *
   * @description
   * Return the distance between the given dates in words.
   *
   * | Distance between dates                                            | Result              |
   * |-------------------------------------------------------------------|---------------------|
   * | 0 ... 30 secs                                                     | less than a minute  |
   * | 30 secs ... 1 min 30 secs                                         | 1 minute            |
   * | 1 min 30 secs ... 44 mins 30 secs                                 | [2..44] minutes     |
   * | 44 mins ... 30 secs ... 89 mins 30 secs                           | about 1 hour        |
   * | 89 mins 30 secs ... 23 hrs 59 mins 30 secs                        | about [2..24] hours |
   * | 23 hrs 59 mins 30 secs ... 41 hrs 59 mins 30 secs                 | 1 day               |
   * | 41 hrs 59 mins 30 secs ... 29 days 23 hrs 59 mins 30 secs         | [2..30] days        |
   * | 29 days 23 hrs 59 mins 30 secs ... 44 days 23 hrs 59 mins 30 secs | about 1 month       |
   * | 44 days 23 hrs 59 mins 30 secs ... 59 days 23 hrs 59 mins 30 secs | about 2 months      |
   * | 59 days 23 hrs 59 mins 30 secs ... 1 yr                           | [2..12] months      |
   * | 1 yr ... 1 yr 3 months                                            | about 1 year        |
   * | 1 yr 3 months ... 1 yr 9 month s                                  | over 1 year         |
   * | 1 yr 9 months ... 2 yrs                                           | almost 2 years      |
   * | N yrs ... N yrs 3 months                                          | about N years       |
   * | N yrs 3 months ... N yrs 9 months                                 | over N years        |
   * | N yrs 9 months ... N+1 yrs                                        | almost N+1 years    |
   *
   * With `options.includeSeconds == true`:
   * | Distance between dates | Result               |
   * |------------------------|----------------------|
   * | 0 secs ... 5 secs      | less than 5 seconds  |
   * | 5 secs ... 10 secs     | less than 10 seconds |
   * | 10 secs ... 20 secs    | less than 20 seconds |
   * | 20 secs ... 40 secs    | half a minute        |
   * | 40 secs ... 60 secs    | less than a minute   |
   * | 60 secs ... 90 secs    | 1 minute             |
   *
   * @param {Date|String|Number} dateToCompare - the date to compare with
   * @param {Date|String|Number} date - the other date
   * @param {Object} [options] - the object with options
   * @param {Boolean} [options.includeSeconds=false] - distances less than a minute are more detailed
   * @param {Boolean} [options.addSuffix=false] - result indicates if the second date is earlier or later than the first
   * @param {Object} [options.locale=enLocale] - the locale object
   * @returns {String} the distance in words
   *
   * @example
   * // What is the distance between 2 July 2014 and 1 January 2015?
   * var result = distanceInWords(
   *   new Date(2014, 6, 2),
   *   new Date(2015, 0, 1)
   * )
   * //=> '6 months'
   *
   * @example
   * // What is the distance between 1 January 2015 00:00:15
   * // and 1 January 2015 00:00:00, including seconds?
   * var result = distanceInWords(
   *   new Date(2015, 0, 1, 0, 0, 15),
   *   new Date(2015, 0, 1, 0, 0, 0),
   *   {includeSeconds: true}
   * )
   * //=> 'less than 20 seconds'
   *
   * @example
   * // What is the distance from 1 January 2016
   * // to 1 January 2015, with a suffix?
   * var result = distanceInWords(
   *   new Date(2016, 0, 1),
   *   new Date(2015, 0, 1),
   *   {addSuffix: true}
   * )
   * //=> 'about 1 year ago'
   *
   * @example
   * // What is the distance between 1 August 2016 and 1 January 2015 in Esperanto?
   * var eoLocale = require('date-fns/locale/eo')
   * var result = distanceInWords(
   *   new Date(2016, 7, 1),
   *   new Date(2015, 0, 1),
   *   {locale: eoLocale}
   * )
   * //=> 'pli ol 1 jaro'
   */
  function distanceInWords (dirtyDateToCompare, dirtyDate, dirtyOptions) {
    var options = dirtyOptions || {};

    var comparison = compare_desc(dirtyDateToCompare, dirtyDate);

    var locale = options.locale;
    var localize = en.distanceInWords.localize;
    if (locale && locale.distanceInWords && locale.distanceInWords.localize) {
      localize = locale.distanceInWords.localize;
    }

    var localizeOptions = {
      addSuffix: Boolean(options.addSuffix),
      comparison: comparison
    };

    var dateLeft, dateRight;
    if (comparison > 0) {
      dateLeft = parse_1(dirtyDateToCompare);
      dateRight = parse_1(dirtyDate);
    } else {
      dateLeft = parse_1(dirtyDate);
      dateRight = parse_1(dirtyDateToCompare);
    }

    var seconds = difference_in_seconds(dateRight, dateLeft);
    var offset = dateRight.getTimezoneOffset() - dateLeft.getTimezoneOffset();
    var minutes = Math.round(seconds / 60) - offset;
    var months;

    // 0 up to 2 mins
    if (minutes < 2) {
      if (options.includeSeconds) {
        if (seconds < 5) {
          return localize('lessThanXSeconds', 5, localizeOptions)
        } else if (seconds < 10) {
          return localize('lessThanXSeconds', 10, localizeOptions)
        } else if (seconds < 20) {
          return localize('lessThanXSeconds', 20, localizeOptions)
        } else if (seconds < 40) {
          return localize('halfAMinute', null, localizeOptions)
        } else if (seconds < 60) {
          return localize('lessThanXMinutes', 1, localizeOptions)
        } else {
          return localize('xMinutes', 1, localizeOptions)
        }
      } else {
        if (minutes === 0) {
          return localize('lessThanXMinutes', 1, localizeOptions)
        } else {
          return localize('xMinutes', minutes, localizeOptions)
        }
      }

    // 2 mins up to 0.75 hrs
    } else if (minutes < 45) {
      return localize('xMinutes', minutes, localizeOptions)

    // 0.75 hrs up to 1.5 hrs
    } else if (minutes < 90) {
      return localize('aboutXHours', 1, localizeOptions)

    // 1.5 hrs up to 24 hrs
    } else if (minutes < MINUTES_IN_DAY) {
      var hours = Math.round(minutes / 60);
      return localize('aboutXHours', hours, localizeOptions)

    // 1 day up to 1.75 days
    } else if (minutes < MINUTES_IN_ALMOST_TWO_DAYS) {
      return localize('xDays', 1, localizeOptions)

    // 1.75 days up to 30 days
    } else if (minutes < MINUTES_IN_MONTH) {
      var days = Math.round(minutes / MINUTES_IN_DAY);
      return localize('xDays', days, localizeOptions)

    // 1 month up to 2 months
    } else if (minutes < MINUTES_IN_TWO_MONTHS) {
      months = Math.round(minutes / MINUTES_IN_MONTH);
      return localize('aboutXMonths', months, localizeOptions)
    }

    months = difference_in_months(dateRight, dateLeft);

    // 2 months up to 12 months
    if (months < 12) {
      var nearestMonth = Math.round(minutes / MINUTES_IN_MONTH);
      return localize('xMonths', nearestMonth, localizeOptions)

    // 1 year up to max Date
    } else {
      var monthsSinceStartOfYear = months % 12;
      var years = Math.floor(months / 12);

      // N years up to 1 years 3 months
      if (monthsSinceStartOfYear < 3) {
        return localize('aboutXYears', years, localizeOptions)

      // N years 3 months up to N years 9 months
      } else if (monthsSinceStartOfYear < 9) {
        return localize('overXYears', years, localizeOptions)

      // N years 9 months up to N year 12 months
      } else {
        return localize('almostXYears', years + 1, localizeOptions)
      }
    }
  }

  var distance_in_words = distanceInWords;

  /**
   * @category Common Helpers
   * @summary Return the distance between the given date and now in words.
   *
   * @description
   * Return the distance between the given date and now in words.
   *
   * | Distance to now                                                   | Result              |
   * |-------------------------------------------------------------------|---------------------|
   * | 0 ... 30 secs                                                     | less than a minute  |
   * | 30 secs ... 1 min 30 secs                                         | 1 minute            |
   * | 1 min 30 secs ... 44 mins 30 secs                                 | [2..44] minutes     |
   * | 44 mins ... 30 secs ... 89 mins 30 secs                           | about 1 hour        |
   * | 89 mins 30 secs ... 23 hrs 59 mins 30 secs                        | about [2..24] hours |
   * | 23 hrs 59 mins 30 secs ... 41 hrs 59 mins 30 secs                 | 1 day               |
   * | 41 hrs 59 mins 30 secs ... 29 days 23 hrs 59 mins 30 secs         | [2..30] days        |
   * | 29 days 23 hrs 59 mins 30 secs ... 44 days 23 hrs 59 mins 30 secs | about 1 month       |
   * | 44 days 23 hrs 59 mins 30 secs ... 59 days 23 hrs 59 mins 30 secs | about 2 months      |
   * | 59 days 23 hrs 59 mins 30 secs ... 1 yr                           | [2..12] months      |
   * | 1 yr ... 1 yr 3 months                                            | about 1 year        |
   * | 1 yr 3 months ... 1 yr 9 month s                                  | over 1 year         |
   * | 1 yr 9 months ... 2 yrs                                           | almost 2 years      |
   * | N yrs ... N yrs 3 months                                          | about N years       |
   * | N yrs 3 months ... N yrs 9 months                                 | over N years        |
   * | N yrs 9 months ... N+1 yrs                                        | almost N+1 years    |
   *
   * With `options.includeSeconds == true`:
   * | Distance to now     | Result               |
   * |---------------------|----------------------|
   * | 0 secs ... 5 secs   | less than 5 seconds  |
   * | 5 secs ... 10 secs  | less than 10 seconds |
   * | 10 secs ... 20 secs | less than 20 seconds |
   * | 20 secs ... 40 secs | half a minute        |
   * | 40 secs ... 60 secs | less than a minute   |
   * | 60 secs ... 90 secs | 1 minute             |
   *
   * @param {Date|String|Number} date - the given date
   * @param {Object} [options] - the object with options
   * @param {Boolean} [options.includeSeconds=false] - distances less than a minute are more detailed
   * @param {Boolean} [options.addSuffix=false] - result specifies if the second date is earlier or later than the first
   * @param {Object} [options.locale=enLocale] - the locale object
   * @returns {String} the distance in words
   *
   * @example
   * // If today is 1 January 2015, what is the distance to 2 July 2014?
   * var result = distanceInWordsToNow(
   *   new Date(2014, 6, 2)
   * )
   * //=> '6 months'
   *
   * @example
   * // If now is 1 January 2015 00:00:00,
   * // what is the distance to 1 January 2015 00:00:15, including seconds?
   * var result = distanceInWordsToNow(
   *   new Date(2015, 0, 1, 0, 0, 15),
   *   {includeSeconds: true}
   * )
   * //=> 'less than 20 seconds'
   *
   * @example
   * // If today is 1 January 2015,
   * // what is the distance to 1 January 2016, with a suffix?
   * var result = distanceInWordsToNow(
   *   new Date(2016, 0, 1),
   *   {addSuffix: true}
   * )
   * //=> 'in about 1 year'
   *
   * @example
   * // If today is 1 January 2015,
   * // what is the distance to 1 August 2016 in Esperanto?
   * var eoLocale = require('date-fns/locale/eo')
   * var result = distanceInWordsToNow(
   *   new Date(2016, 7, 1),
   *   {locale: eoLocale}
   * )
   * //=> 'pli ol 1 jaro'
   */
  function distanceInWordsToNow (dirtyDate, dirtyOptions) {
    return distance_in_words(Date.now(), dirtyDate, dirtyOptions)
  }

  var distance_in_words_to_now = distanceInWordsToNow;

  var defaultConverter = (function (date, locale, converterOptions) {
    var includeSeconds = converterOptions.includeSeconds,
        _converterOptions$add = converterOptions.addSuffix,
        addSuffix = _converterOptions$add === void 0 ? true : _converterOptions$add;
    return distance_in_words_to_now(date, {
      locale: locale,
      includeSeconds: includeSeconds,
      addSuffix: addSuffix
    });
  });

  var createTimeago = function createTimeago() {
    var opts = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    var locales = opts.locales || {};
    var name = opts.name || 'Timeago';
    return {
      name: name,
      props: {
        datetime: {
          required: true
        },
        title: {
          type: [String, Boolean]
        },
        locale: {
          type: String
        },
        autoUpdate: {
          type: [Number, Boolean]
        },
        converter: {
          type: Function
        },
        converterOptions: {
          type: Object
        }
      },
      data: function data() {
        return {
          timeago: this.getTimeago()
        };
      },
      mounted: function mounted() {
        this.startUpdater();
      },
      beforeDestroy: function beforeDestroy() {
        this.stopUpdater();
      },
      render: function render(h) {
        return h('time', {
          attrs: {
            datetime: new Date(this.datetime),
            title: typeof this.title === 'string' ? this.title : this.title === false ? null : this.timeago
          }
        }, [this.timeago]);
      },
      methods: {
        getTimeago: function getTimeago(datetime) {
          var converter = this.converter || opts.converter || defaultConverter;
          return converter(datetime || this.datetime, locales[this.locale || opts.locale], this.converterOptions || {});
        },
        convert: function convert(datetime) {
          this.timeago = this.getTimeago(datetime);
        },
        startUpdater: function startUpdater() {
          var _this = this;

          if (this.autoUpdate) {
            var autoUpdaye = this.autoUpdate === true ? 60 : this.autoUpdate;
            this.updater = setInterval(function () {
              _this.convert();
            }, autoUpdaye * 1000);
          }
        },
        stopUpdater: function stopUpdater() {
          if (this.updater) {
            clearInterval(this.updater);
            this.updater = null;
          }
        }
      },
      watch: {
        autoUpdate: function autoUpdate(newValue) {
          this.stopUpdater();

          if (newValue) {
            this.startUpdater();
          }
        },
        datetime: function datetime() {
          this.convert();
        },
        locale: function locale() {
          this.convert();
        },
        converter: function converter() {
          this.convert();
        },
        converterOptions: {
          handler: function handler() {
            this.convert();
          },
          deep: true
        }
      }
    };
  };
  var install = function install(Vue, opts) {
    var Component = createTimeago(opts);
    Vue.component(Component.name, Component);
  };
  var converter = defaultConverter;

  exports.createTimeago = createTimeago;
  exports.install = install;
  exports.converter = converter;
  exports.default = install;

  Object.defineProperty(exports, '__esModule', { value: true });

})));
