"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = exports.VXETablePluginElement = void 0;

var _xeUtils = _interopRequireDefault(require("xe-utils"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

function isEmptyValue(cellValue) {
  return cellValue === null || cellValue === undefined || cellValue === '';
}

function getModelProp(renderOpts) {
  return 'value';
}

function getModelEvent(renderOpts) {
  return 'input';
}

function getChangeEvent(renderOpts) {
  var type = 'change';

  switch (renderOpts.name) {
    case 'ElAutocomplete':
      type = 'select';
      break;

    case 'ElInput':
    case 'ElInputNumber':
      type = 'input';
      break;
  }

  return type;
}

function parseDate(value, props) {
  return value && props.valueFormat ? _xeUtils["default"].toStringDate(value, props.valueFormat) : value;
}

function getFormatDate(value, props, defaultFormat) {
  return _xeUtils["default"].toDateString(parseDate(value, props), props.format || defaultFormat);
}

function getFormatDates(values, props, separator, defaultFormat) {
  return _xeUtils["default"].map(values, function (date) {
    return getFormatDate(date, props, defaultFormat);
  }).join(separator);
}

function equalDaterange(cellValue, data, props, defaultFormat) {
  cellValue = getFormatDate(cellValue, props, defaultFormat);
  return cellValue >= getFormatDate(data[0], props, defaultFormat) && cellValue <= getFormatDate(data[1], props, defaultFormat);
}

function getCellEditFilterProps(renderOpts, params, value, defaultProps) {
  var vSize = params.$table.vSize;
  return _xeUtils["default"].assign(vSize ? {
    size: vSize
  } : {}, defaultProps, renderOpts.props, _defineProperty({}, getModelProp(renderOpts), value));
}

function getItemProps(renderOpts, params, value, defaultProps) {
  var vSize = params.$form.vSize;
  return _xeUtils["default"].assign(vSize ? {
    size: vSize
  } : {}, defaultProps, renderOpts.props, _defineProperty({}, getModelProp(renderOpts), value));
}

function formatText(cellValue) {
  return '' + (isEmptyValue(cellValue) ? '' : cellValue);
}

function getCellLabelVNs(h, renderOpts, params, cellLabel) {
  var placeholder = renderOpts.placeholder;
  return [h('span', {
    "class": 'vxe-cell--label'
  }, placeholder && isEmptyValue(cellLabel) ? [h('span', {
    "class": 'vxe-cell--placeholder'
  }, formatText(placeholder))] : formatText(cellLabel))];
}

function getNativeOns(renderOpts, params) {
  var nativeEvents = renderOpts.nativeEvents;
  var nativeOns = {};

  _xeUtils["default"].objectEach(nativeEvents, function (func, key) {
    nativeOns[key] = function () {
      for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
        args[_key] = arguments[_key];
      }

      func.apply(void 0, [params].concat(args));
    };
  });

  return nativeOns;
}

function getOns(renderOpts, params, inputFunc, changeFunc) {
  var events = renderOpts.events;
  var modelEvent = getModelEvent(renderOpts);
  var changeEvent = getChangeEvent(renderOpts);
  var isSameEvent = changeEvent === modelEvent;
  var ons = {};

  _xeUtils["default"].objectEach(events, function (func, key) {
    ons[key] = function () {
      for (var _len2 = arguments.length, args = new Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
        args[_key2] = arguments[_key2];
      }

      func.apply(void 0, [params].concat(args));
    };
  });

  if (inputFunc) {
    ons[modelEvent] = function (targetEvnt) {
      inputFunc(targetEvnt);

      if (events && events[modelEvent]) {
        events[modelEvent](params, targetEvnt);
      }

      if (isSameEvent && changeFunc) {
        changeFunc(targetEvnt);
      }
    };
  }

  if (!isSameEvent && changeFunc) {
    ons[changeEvent] = function () {
      for (var _len3 = arguments.length, args = new Array(_len3), _key3 = 0; _key3 < _len3; _key3++) {
        args[_key3] = arguments[_key3];
      }

      changeFunc.apply(void 0, args);

      if (events && events[changeEvent]) {
        events[changeEvent].apply(events, [params].concat(args));
      }
    };
  }

  return ons;
}

function getEditOns(renderOpts, params) {
  var $table = params.$table,
      row = params.row,
      column = params.column;
  return getOns(renderOpts, params, function (value) {
    // 处理 model 值双向绑定
    _xeUtils["default"].set(row, column.property, value);
  }, function () {
    // 处理 change 事件相关逻辑
    $table.updateStatus(params);
  });
}

function getFilterOns(renderOpts, params, option, changeFunc) {
  return getOns(renderOpts, params, function (value) {
    // 处理 model 值双向绑定
    option.data = value;
  }, changeFunc);
}

function getItemOns(renderOpts, params) {
  var $form = params.$form,
      data = params.data,
      property = params.property;
  return getOns(renderOpts, params, function (value) {
    // 处理 model 值双向绑定
    _xeUtils["default"].set(data, property, value);
  }, function () {
    // 处理 change 事件相关逻辑
    $form.updateStatus(params);
  });
}

function matchCascaderData(index, list, values, labels) {
  var val = values[index];

  if (list && values.length > index) {
    _xeUtils["default"].each(list, function (item) {
      if (item.value === val) {
        labels.push(item.label);
        matchCascaderData(++index, item.children, values, labels);
      }
    });
  }
}

function getSelectCellValue(renderOpts, params) {
  var _renderOpts$options = renderOpts.options,
      options = _renderOpts$options === void 0 ? [] : _renderOpts$options,
      optionGroups = renderOpts.optionGroups,
      _renderOpts$props = renderOpts.props,
      props = _renderOpts$props === void 0 ? {} : _renderOpts$props,
      _renderOpts$optionPro = renderOpts.optionProps,
      optionProps = _renderOpts$optionPro === void 0 ? {} : _renderOpts$optionPro,
      _renderOpts$optionGro = renderOpts.optionGroupProps,
      optionGroupProps = _renderOpts$optionGro === void 0 ? {} : _renderOpts$optionGro;
  var row = params.row,
      column = params.column;
  var $table = params.$table;
  var labelProp = optionProps.label || 'label';
  var valueProp = optionProps.value || 'value';
  var groupOptions = optionGroupProps.options || 'options';

  var cellValue = _xeUtils["default"].get(row, column.property);

  var colid = column.id;
  var rest;
  var cellData;

  if (props.filterable) {
    var fullAllDataRowMap = $table.fullAllDataRowMap;
    var cacheCell = fullAllDataRowMap.has(row);

    if (cacheCell) {
      rest = fullAllDataRowMap.get(row);
      cellData = rest.cellData;

      if (!cellData) {
        cellData = fullAllDataRowMap.get(row).cellData = {};
      }
    }

    if (rest && cellData[colid] && cellData[colid].value === cellValue) {
      return cellData[colid].label;
    }
  }

  if (!isEmptyValue(cellValue)) {
    var selectlabel = _xeUtils["default"].map(props.multiple ? cellValue : [cellValue], optionGroups ? function (value) {
      var selectItem;

      for (var index = 0; index < optionGroups.length; index++) {
        selectItem = _xeUtils["default"].find(optionGroups[index][groupOptions], function (item) {
          return item[valueProp] === value;
        });

        if (selectItem) {
          break;
        }
      }

      return selectItem ? selectItem[labelProp] : value;
    } : function (value) {
      var selectItem = _xeUtils["default"].find(options, function (item) {
        return item[valueProp] === value;
      });

      return selectItem ? selectItem[labelProp] : value;
    }).join(', ');

    if (cellData && options && options.length) {
      cellData[colid] = {
        value: cellValue,
        label: selectlabel
      };
    }

    return selectlabel;
  }

  return '';
}

function getCascaderCellValue(renderOpts, params) {
  var _renderOpts$props2 = renderOpts.props,
      props = _renderOpts$props2 === void 0 ? {} : _renderOpts$props2;
  var row = params.row,
      column = params.column;

  var cellValue = _xeUtils["default"].get(row, column.property);

  var values = cellValue || [];
  var labels = [];
  matchCascaderData(0, props.options, values, labels);
  return (props.showAllLevels === false ? labels.slice(labels.length - 1, labels.length) : labels).join(" ".concat(props.separator || '/', " "));
}

function getDatePickerCellValue(renderOpts, params) {
  var _renderOpts$props3 = renderOpts.props,
      props = _renderOpts$props3 === void 0 ? {} : _renderOpts$props3;
  var row = params.row,
      column = params.column;
  var _props$rangeSeparator = props.rangeSeparator,
      rangeSeparator = _props$rangeSeparator === void 0 ? '-' : _props$rangeSeparator;

  var cellValue = _xeUtils["default"].get(row, column.property);

  switch (props.type) {
    case 'week':
      cellValue = getFormatDate(cellValue, props, 'yyyywWW');
      break;

    case 'month':
      cellValue = getFormatDate(cellValue, props, 'yyyy-MM');
      break;

    case 'year':
      cellValue = getFormatDate(cellValue, props, 'yyyy');
      break;

    case 'dates':
      cellValue = getFormatDates(cellValue, props, ', ', 'yyyy-MM-dd');
      break;

    case 'daterange':
      cellValue = getFormatDates(cellValue, props, " ".concat(rangeSeparator, " "), 'yyyy-MM-dd');
      break;

    case 'datetimerange':
      cellValue = getFormatDates(cellValue, props, " ".concat(rangeSeparator, " "), 'yyyy-MM-dd HH:ss:mm');
      break;

    case 'monthrange':
      cellValue = getFormatDates(cellValue, props, " ".concat(rangeSeparator, " "), 'yyyy-MM');
      break;

    default:
      cellValue = getFormatDate(cellValue, props, 'yyyy-MM-dd');
  }

  return cellValue;
}

function getTimePickerCellValue(renderOpts, params) {
  var _renderOpts$props4 = renderOpts.props,
      props = _renderOpts$props4 === void 0 ? {} : _renderOpts$props4;
  var row = params.row,
      column = params.column;
  var isRange = props.isRange,
      _props$format = props.format,
      format = _props$format === void 0 ? 'hh:mm:ss' : _props$format,
      _props$rangeSeparator2 = props.rangeSeparator,
      rangeSeparator = _props$rangeSeparator2 === void 0 ? '-' : _props$rangeSeparator2;

  var cellValue = _xeUtils["default"].get(row, column.property);

  if (cellValue && isRange) {
    cellValue = _xeUtils["default"].map(cellValue, function (date) {
      return _xeUtils["default"].toDateString(parseDate(date, props), format);
    }).join(" ".concat(rangeSeparator, " "));
  }

  return _xeUtils["default"].toDateString(parseDate(cellValue, props), format);
}

function createEditRender(defaultProps) {
  return function (h, renderOpts, params) {
    var row = params.row,
        column = params.column;
    var attrs = renderOpts.attrs;

    var cellValue = _xeUtils["default"].get(row, column.property);

    return [h(renderOpts.name, {
      attrs: attrs,
      props: getCellEditFilterProps(renderOpts, params, cellValue, defaultProps),
      on: getEditOns(renderOpts, params),
      nativeOn: getNativeOns(renderOpts, params)
    })];
  };
}

function defaultButtonEditRender(h, renderOpts, params) {
  var attrs = renderOpts.attrs;
  return [h('el-button', {
    attrs: attrs,
    props: getCellEditFilterProps(renderOpts, params, null),
    on: getOns(renderOpts, params),
    nativeOn: getNativeOns(renderOpts, params)
  }, cellText(h, renderOpts.content))];
}

function defaultButtonsEditRender(h, renderOpts, params) {
  return renderOpts.children.map(function (childRenderOpts) {
    return defaultButtonEditRender(h, childRenderOpts, params)[0];
  });
}

function createFilterRender(defaultProps) {
  return function (h, renderOpts, params) {
    var column = params.column;
    var name = renderOpts.name,
        attrs = renderOpts.attrs;
    var nativeOn = getNativeOns(renderOpts, params);
    return [h('div', {
      "class": 'vxe-table--filter-element-wrapper'
    }, column.filters.map(function (option, oIndex) {
      var optionValue = option.data;
      return h(name, {
        key: oIndex,
        attrs: attrs,
        props: getCellEditFilterProps(renderOpts, params, optionValue, defaultProps),
        on: getFilterOns(renderOpts, params, option, function () {
          // 处理 change 事件相关逻辑
          handleConfirmFilter(params, !!option.data, option);
        }),
        nativeOn: nativeOn
      });
    }))];
  };
}

function handleConfirmFilter(params, checked, option) {
  var $panel = params.$panel;
  $panel.changeOption({}, checked, option);
}
/**
 * 模糊匹配
 * @param params
 */


function defaultFuzzyFilterMethod(params) {
  var option = params.option,
      row = params.row,
      column = params.column;
  var data = option.data;

  var cellValue = _xeUtils["default"].get(row, column.property);

  return _xeUtils["default"].toString(cellValue).indexOf(data) > -1;
}
/**
 * 精确匹配
 * @param params
 */


function defaultExactFilterMethod(params) {
  var option = params.option,
      row = params.row,
      column = params.column;
  var data = option.data;

  var cellValue = _xeUtils["default"].get(row, column.property);
  /* eslint-disable eqeqeq */


  return cellValue === data;
}

function renderOptions(h, options, optionProps) {
  var labelProp = optionProps.label || 'label';
  var valueProp = optionProps.value || 'value';
  return _xeUtils["default"].map(options, function (item, oIndex) {
    return h('el-option', {
      key: oIndex,
      props: {
        value: item[valueProp],
        label: item[labelProp],
        disabled: item.disabled
      }
    });
  });
}

function cellText(h, cellValue) {
  return [formatText(cellValue)];
}

function createFormItemRender(defaultProps) {
  return function (h, renderOpts, params) {
    var data = params.data,
        property = params.property;
    var name = renderOpts.name;
    var attrs = renderOpts.attrs;

    var itemValue = _xeUtils["default"].get(data, property);

    return [h(name, {
      attrs: attrs,
      props: getItemProps(renderOpts, params, itemValue, defaultProps),
      on: getItemOns(renderOpts, params),
      nativeOn: getNativeOns(renderOpts, params)
    })];
  };
}

function defaultButtonItemRender(h, renderOpts, params) {
  var attrs = renderOpts.attrs;
  var props = getItemProps(renderOpts, params, null);
  return [h('el-button', {
    attrs: attrs,
    props: props,
    on: getOns(renderOpts, params),
    nativeOn: getNativeOns(renderOpts, params)
  }, cellText(h, renderOpts.content || props.content))];
}

function defaultButtonsItemRender(h, renderOpts, params) {
  return renderOpts.children.map(function (childRenderOpts) {
    return defaultButtonItemRender(h, childRenderOpts, params)[0];
  });
}

function createExportMethod(getExportCellValue) {
  return function (params) {
    var row = params.row,
        column = params.column,
        options = params.options;
    return options && options.original ? _xeUtils["default"].get(row, column.property) : getExportCellValue(column.editRender || column.cellRender, params);
  };
}

function createFormItemRadioAndCheckboxRender() {
  return function (h, renderOpts, params) {
    var name = renderOpts.name,
        _renderOpts$options2 = renderOpts.options,
        options = _renderOpts$options2 === void 0 ? [] : _renderOpts$options2,
        _renderOpts$optionPro2 = renderOpts.optionProps,
        optionProps = _renderOpts$optionPro2 === void 0 ? {} : _renderOpts$optionPro2,
        attrs = renderOpts.attrs;
    var data = params.data,
        property = params.property;
    var labelProp = optionProps.label || 'label';
    var valueProp = optionProps.value || 'value';

    var itemValue = _xeUtils["default"].get(data, property);

    return [h("".concat(name, "Group"), {
      attrs: attrs,
      props: getItemProps(renderOpts, params, itemValue),
      on: getItemOns(renderOpts, params),
      nativeOn: getNativeOns(renderOpts, params)
    }, options.map(function (option, oIndex) {
      return h(name, {
        key: oIndex,
        props: {
          label: option[valueProp],
          disabled: option.disabled
        }
      }, option[labelProp]);
    }))];
  };
}
/**
 * 检查触发源是否属于目标节点
 */


function getEventTargetNode(evnt, container, className) {
  var targetElem;
  var target = evnt.target;

  while (target && target.nodeType && target !== document) {
    if (className && target.className && target.className.split && target.className.split(' ').indexOf(className) > -1) {
      targetElem = target;
    } else if (target === container) {
      return {
        flag: className ? !!targetElem : true,
        container: container,
        targetElem: targetElem
      };
    }

    target = target.parentNode;
  }

  return {
    flag: false
  };
}
/**
 * 事件兼容性处理
 */


function handleClearEvent(params, e) {
  var bodyElem = document.body;
  var evnt = params.$event || e;

  if ( // 远程搜索
  getEventTargetNode(evnt, bodyElem, 'el-autocomplete-suggestion').flag || // 下拉框
  getEventTargetNode(evnt, bodyElem, 'el-select-dropdown').flag || // 级联
  getEventTargetNode(evnt, bodyElem, 'el-cascader__dropdown').flag || getEventTargetNode(evnt, bodyElem, 'el-cascader-menus').flag || // 日期
  getEventTargetNode(evnt, bodyElem, 'el-time-panel').flag || getEventTargetNode(evnt, bodyElem, 'el-picker-panel').flag || // 颜色
  getEventTargetNode(evnt, bodyElem, 'el-color-dropdown').flag) {
    return false;
  }
}
/**
 * 基于 vxe-table 表格的适配插件，用于兼容 element-ui 组件库
 */


var VXETablePluginElement = {
  install: function install(_ref) {
    var interceptor = _ref.interceptor,
        renderer = _ref.renderer;
    renderer.mixin({
      ElAutocomplete: {
        autofocus: 'input.el-input__inner',
        renderDefault: createEditRender(),
        renderEdit: createEditRender(),
        renderFilter: createFilterRender(),
        filterMethod: defaultExactFilterMethod,
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender()
      },
      ElInput: {
        autofocus: 'input.el-input__inner',
        renderDefault: createEditRender(),
        renderEdit: createEditRender(),
        renderFilter: createFilterRender(),
        filterMethod: defaultFuzzyFilterMethod,
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender()
      },
      ElInputNumber: {
        autofocus: 'input.el-input__inner',
        renderDefault: createEditRender(),
        renderEdit: createEditRender(),
        renderFilter: createFilterRender(),
        filterMethod: defaultFuzzyFilterMethod,
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender()
      },
      ElSelect: {
        renderEdit: function renderEdit(h, renderOpts, params) {
          var _renderOpts$options3 = renderOpts.options,
              options = _renderOpts$options3 === void 0 ? [] : _renderOpts$options3,
              optionGroups = renderOpts.optionGroups,
              _renderOpts$optionPro3 = renderOpts.optionProps,
              optionProps = _renderOpts$optionPro3 === void 0 ? {} : _renderOpts$optionPro3,
              _renderOpts$optionGro2 = renderOpts.optionGroupProps,
              optionGroupProps = _renderOpts$optionGro2 === void 0 ? {} : _renderOpts$optionGro2;
          var row = params.row,
              column = params.column;
          var attrs = renderOpts.attrs;

          var cellValue = _xeUtils["default"].get(row, column.property);

          var props = getCellEditFilterProps(renderOpts, params, cellValue);
          var on = getEditOns(renderOpts, params);
          var nativeOn = getNativeOns(renderOpts, params);

          if (optionGroups) {
            var groupOptions = optionGroupProps.options || 'options';
            var groupLabel = optionGroupProps.label || 'label';
            return [h('el-select', {
              attrs: attrs,
              props: props,
              on: on,
              nativeOn: nativeOn
            }, _xeUtils["default"].map(optionGroups, function (group, gIndex) {
              return h('el-option-group', {
                key: gIndex,
                props: {
                  label: group[groupLabel]
                }
              }, renderOptions(h, group[groupOptions], optionProps));
            }))];
          }

          return [h('el-select', {
            props: props,
            attrs: attrs,
            on: on,
            nativeOn: nativeOn
          }, renderOptions(h, options, optionProps))];
        },
        renderCell: function renderCell(h, renderOpts, params) {
          return getCellLabelVNs(h, renderOpts, params, getSelectCellValue(renderOpts, params));
        },
        renderFilter: function renderFilter(h, renderOpts, params) {
          var _renderOpts$options4 = renderOpts.options,
              options = _renderOpts$options4 === void 0 ? [] : _renderOpts$options4,
              optionGroups = renderOpts.optionGroups,
              _renderOpts$optionPro4 = renderOpts.optionProps,
              optionProps = _renderOpts$optionPro4 === void 0 ? {} : _renderOpts$optionPro4,
              _renderOpts$optionGro3 = renderOpts.optionGroupProps,
              optionGroupProps = _renderOpts$optionGro3 === void 0 ? {} : _renderOpts$optionGro3;
          var groupOptions = optionGroupProps.options || 'options';
          var groupLabel = optionGroupProps.label || 'label';
          var column = params.column;
          var attrs = renderOpts.attrs;
          var nativeOn = getNativeOns(renderOpts, params);
          return [h('div', {
            "class": 'vxe-table--filter-element-wrapper'
          }, optionGroups ? column.filters.map(function (option, oIndex) {
            var optionValue = option.data;
            var props = getCellEditFilterProps(renderOpts, params, optionValue);
            return h('el-select', {
              key: oIndex,
              attrs: attrs,
              props: props,
              on: getFilterOns(renderOpts, params, option, function () {
                // 处理 change 事件相关逻辑
                handleConfirmFilter(params, props.multiple ? option.data && option.data.length > 0 : !_xeUtils["default"].eqNull(option.data), option);
              }),
              nativeOn: nativeOn
            }, _xeUtils["default"].map(optionGroups, function (group, gIndex) {
              return h('el-option-group', {
                key: gIndex,
                props: {
                  label: group[groupLabel]
                }
              }, renderOptions(h, group[groupOptions], optionProps));
            }));
          }) : column.filters.map(function (option, oIndex) {
            var optionValue = option.data;
            var props = getCellEditFilterProps(renderOpts, params, optionValue);
            return h('el-select', {
              key: oIndex,
              attrs: attrs,
              props: props,
              on: getFilterOns(renderOpts, params, option, function () {
                // 处理 change 事件相关逻辑
                handleConfirmFilter(params, props.multiple ? option.data && option.data.length > 0 : !_xeUtils["default"].eqNull(option.data), option);
              }),
              nativeOn: nativeOn
            }, renderOptions(h, options, optionProps));
          }))];
        },
        filterMethod: function filterMethod(params) {
          var option = params.option,
              row = params.row,
              column = params.column;
          var data = option.data;
          var property = column.property,
              renderOpts = column.filterRender;
          var _renderOpts$props5 = renderOpts.props,
              props = _renderOpts$props5 === void 0 ? {} : _renderOpts$props5;

          var cellValue = _xeUtils["default"].get(row, property);

          if (props.multiple) {
            if (_xeUtils["default"].isArray(cellValue)) {
              return _xeUtils["default"].includeArrays(cellValue, data);
            }

            return data.indexOf(cellValue) > -1;
          }
          /* eslint-disable eqeqeq */


          return cellValue == data;
        },
        renderItem: function renderItem(h, renderOpts, params) {
          var _renderOpts$options5 = renderOpts.options,
              options = _renderOpts$options5 === void 0 ? [] : _renderOpts$options5,
              optionGroups = renderOpts.optionGroups,
              _renderOpts$optionPro5 = renderOpts.optionProps,
              optionProps = _renderOpts$optionPro5 === void 0 ? {} : _renderOpts$optionPro5,
              _renderOpts$optionGro4 = renderOpts.optionGroupProps,
              optionGroupProps = _renderOpts$optionGro4 === void 0 ? {} : _renderOpts$optionGro4;
          var data = params.data,
              property = params.property;
          var attrs = renderOpts.attrs;

          var itemValue = _xeUtils["default"].get(data, property);

          var props = getItemProps(renderOpts, params, itemValue);
          var on = getItemOns(renderOpts, params);
          var nativeOn = getNativeOns(renderOpts, params);

          if (optionGroups) {
            var groupOptions = optionGroupProps.options || 'options';
            var groupLabel = optionGroupProps.label || 'label';
            return [h('el-select', {
              attrs: attrs,
              props: props,
              on: on,
              nativeOn: nativeOn
            }, _xeUtils["default"].map(optionGroups, function (group, gIndex) {
              return h('el-option-group', {
                props: {
                  label: group[groupLabel]
                },
                key: gIndex
              }, renderOptions(h, group[groupOptions], optionProps));
            }))];
          }

          return [h('el-select', {
            attrs: attrs,
            props: props,
            on: on,
            nativeOn: nativeOn
          }, renderOptions(h, options, optionProps))];
        },
        renderItemContent: function renderItemContent(h, renderOpts, params) {
          var _renderOpts$options6 = renderOpts.options,
              options = _renderOpts$options6 === void 0 ? [] : _renderOpts$options6,
              optionGroups = renderOpts.optionGroups,
              _renderOpts$optionPro6 = renderOpts.optionProps,
              optionProps = _renderOpts$optionPro6 === void 0 ? {} : _renderOpts$optionPro6,
              _renderOpts$optionGro5 = renderOpts.optionGroupProps,
              optionGroupProps = _renderOpts$optionGro5 === void 0 ? {} : _renderOpts$optionGro5;
          var data = params.data,
              property = params.property;
          var attrs = renderOpts.attrs;

          var itemValue = _xeUtils["default"].get(data, property);

          var props = getItemProps(renderOpts, params, itemValue);
          var on = getItemOns(renderOpts, params);
          var nativeOn = getNativeOns(renderOpts, params);

          if (optionGroups) {
            var groupOptions = optionGroupProps.options || 'options';
            var groupLabel = optionGroupProps.label || 'label';
            return [h('el-select', {
              attrs: attrs,
              props: props,
              on: on,
              nativeOn: nativeOn
            }, _xeUtils["default"].map(optionGroups, function (group, gIndex) {
              return h('el-option-group', {
                props: {
                  label: group[groupLabel]
                },
                key: gIndex
              }, renderOptions(h, group[groupOptions], optionProps));
            }))];
          }

          return [h('el-select', {
            attrs: attrs,
            props: props,
            on: on,
            nativeOn: nativeOn
          }, renderOptions(h, options, optionProps))];
        },
        cellExportMethod: createExportMethod(getSelectCellValue),
        exportMethod: createExportMethod(getSelectCellValue)
      },
      ElCascader: {
        renderEdit: createEditRender(),
        renderCell: function renderCell(h, renderOpts, params) {
          return getCellLabelVNs(h, renderOpts, params, getCascaderCellValue(renderOpts, params));
        },
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender(),
        cellExportMethod: createExportMethod(getCascaderCellValue),
        exportMethod: createExportMethod(getCascaderCellValue)
      },
      ElDatePicker: {
        renderEdit: createEditRender(),
        renderCell: function renderCell(h, renderOpts, params) {
          return getCellLabelVNs(h, renderOpts, params, getDatePickerCellValue(renderOpts, params));
        },
        renderFilter: function renderFilter(h, renderOpts, params) {
          var column = params.column;
          var attrs = renderOpts.attrs;
          var nativeOn = getNativeOns(renderOpts, params);
          return [h('div', {
            "class": 'vxe-table--filter-element-wrapper'
          }, column.filters.map(function (option, oIndex) {
            var optionValue = option.data;
            return h(renderOpts.name, {
              key: oIndex,
              attrs: attrs,
              props: getCellEditFilterProps(renderOpts, params, optionValue),
              on: getFilterOns(renderOpts, params, option, function () {
                // 处理 change 事件相关逻辑
                handleConfirmFilter(params, !!option.data, option);
              }),
              nativeOn: nativeOn
            });
          }))];
        },
        filterMethod: function filterMethod(params) {
          var option = params.option,
              row = params.row,
              column = params.column;
          var data = option.data;
          var renderOpts = column.filterRender;
          var _renderOpts$props6 = renderOpts.props,
              props = _renderOpts$props6 === void 0 ? {} : _renderOpts$props6;

          var cellValue = _xeUtils["default"].get(row, column.property);

          if (data) {
            switch (props.type) {
              case 'daterange':
                return equalDaterange(cellValue, data, props, 'yyyy-MM-dd');

              case 'datetimerange':
                return equalDaterange(cellValue, data, props, 'yyyy-MM-dd HH:ss:mm');

              case 'monthrange':
                return equalDaterange(cellValue, data, props, 'yyyy-MM');

              default:
                return cellValue === data;
            }
          }

          return false;
        },
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender(),
        cellExportMethod: createExportMethod(getDatePickerCellValue),
        exportMethod: createExportMethod(getDatePickerCellValue)
      },
      ElTimePicker: {
        renderEdit: createEditRender(),
        renderCell: function renderCell(h, renderOpts, params) {
          return getCellLabelVNs(h, renderOpts, params, getTimePickerCellValue(renderOpts, params));
        },
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender(),
        cellExportMethod: createExportMethod(getTimePickerCellValue),
        exportMethod: createExportMethod(getTimePickerCellValue)
      },
      ElTimeSelect: {
        renderEdit: createEditRender(),
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender()
      },
      ElRate: {
        renderDefault: createEditRender(),
        renderEdit: createEditRender(),
        renderFilter: createFilterRender(),
        filterMethod: defaultExactFilterMethod,
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender()
      },
      ElSwitch: {
        renderDefault: createEditRender(),
        renderEdit: createEditRender(),
        renderFilter: function renderFilter(h, renderOpts, params) {
          var column = params.column;
          var name = renderOpts.name,
              attrs = renderOpts.attrs;
          var nativeOn = getNativeOns(renderOpts, params);
          return [h('div', {
            "class": 'vxe-table--filter-element-wrapper'
          }, column.filters.map(function (option, oIndex) {
            var optionValue = option.data;
            return h(name, {
              key: oIndex,
              attrs: attrs,
              props: getCellEditFilterProps(renderOpts, params, optionValue),
              on: getFilterOns(renderOpts, params, option, function () {
                // 处理 change 事件相关逻辑
                handleConfirmFilter(params, _xeUtils["default"].isBoolean(option.data), option);
              }),
              nativeOn: nativeOn
            });
          }))];
        },
        filterMethod: defaultExactFilterMethod,
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender()
      },
      ElSlider: {
        renderDefault: createEditRender(),
        renderEdit: createEditRender(),
        renderFilter: createFilterRender(),
        filterMethod: defaultExactFilterMethod,
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender()
      },
      ElRadio: {
        renderItem: createFormItemRadioAndCheckboxRender(),
        renderItemContent: createFormItemRadioAndCheckboxRender()
      },
      ElCheckbox: {
        renderItem: createFormItemRadioAndCheckboxRender(),
        renderItemContent: createFormItemRadioAndCheckboxRender()
      },
      ElButton: {
        renderDefault: defaultButtonEditRender,
        renderItem: defaultButtonItemRender,
        renderItemContent: defaultButtonItemRender
      },
      ElButtons: {
        renderDefault: defaultButtonsEditRender,
        renderItem: defaultButtonsItemRender,
        renderItemContent: defaultButtonsItemRender
      }
    });
    interceptor.add('event.clearFilter', handleClearEvent);
    interceptor.add('event.clearActived', handleClearEvent);
    interceptor.add('event.clearAreas', handleClearEvent);
  }
};
exports.VXETablePluginElement = VXETablePluginElement;

if (typeof window !== 'undefined' && window.VXETable && window.VXETable.use) {
  window.VXETable.use(VXETablePluginElement);
}

var _default = VXETablePluginElement;
exports["default"] = _default;
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
