"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = exports.VXETablePluginElement = void 0;

var _xeUtils = _interopRequireDefault(require("xe-utils"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

function isEmptyValue(cellValue) {
  return cellValue === null || cellValue === undefined || cellValue === '';
}

function getModelProp(renderOpts) {
  return 'value';
}

function getModelEvent(renderOpts) {
  return 'input';
}

function getChangeEvent(renderOpts) {
  var type = 'change';

  switch (renderOpts.name) {
    case 'ElAutocomplete':
      type = 'select';
      break;

    case 'ElInput':
    case 'ElInputNumber':
      type = 'input';
      break;
  }

  return type;
}

function parseDate(value, props) {
  return value && props.valueFormat ? _xeUtils["default"].toStringDate(value, props.valueFormat) : value;
}

function getFormatDate(value, props, defaultFormat) {
  return _xeUtils["default"].toDateString(parseDate(value, props), props.format || defaultFormat);
}

function getFormatDates(values, props, separator, defaultFormat) {
  return _xeUtils["default"].map(values, function (date) {
    return getFormatDate(date, props, defaultFormat);
  }).join(separator);
}

function equalDaterange(cellValue, data, props, defaultFormat) {
  cellValue = getFormatDate(cellValue, props, defaultFormat);
  return cellValue >= getFormatDate(data[0], props, defaultFormat) && cellValue <= getFormatDate(data[1], props, defaultFormat);
}

function getCellEditFilterProps(renderOpts, params, value, defaultProps) {
  var vSize = params.$table.vSize;
  return _xeUtils["default"].assign(vSize ? {
    size: vSize
  } : {}, defaultProps, renderOpts.props, _defineProperty({}, getModelProp(renderOpts), value));
}

function getItemProps(renderOpts, params, value, defaultProps) {
  var vSize = params.$form.vSize;
  return _xeUtils["default"].assign(vSize ? {
    size: vSize
  } : {}, defaultProps, renderOpts.props, _defineProperty({}, getModelProp(renderOpts), value));
}

function formatText(cellValue) {
  return '' + (isEmptyValue(cellValue) ? '' : cellValue);
}

function getCellLabelVNs(h, renderOpts, params, cellLabel) {
  var placeholder = renderOpts.placeholder;
  return [h('span', {
    "class": 'vxe-cell--label'
  }, placeholder && isEmptyValue(cellLabel) ? [h('span', {
    "class": 'vxe-cell--placeholder'
  }, formatText(placeholder))] : formatText(cellLabel))];
}

function getNativeOns(renderOpts, params) {
  var nativeEvents = renderOpts.nativeEvents;
  var nativeOns = {};

  _xeUtils["default"].objectEach(nativeEvents, function (func, key) {
    nativeOns[key] = function () {
      for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
        args[_key] = arguments[_key];
      }

      func.apply(void 0, [params].concat(args));
    };
  });

  return nativeOns;
}

function getOns(renderOpts, params, inputFunc, changeFunc) {
  var events = renderOpts.events;
  var modelEvent = getModelEvent(renderOpts);
  var changeEvent = getChangeEvent(renderOpts);
  var isSameEvent = changeEvent === modelEvent;
  var ons = {};

  _xeUtils["default"].objectEach(events, function (func, key) {
    ons[key] = function () {
      for (var _len2 = arguments.length, args = new Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
        args[_key2] = arguments[_key2];
      }

      func.apply(void 0, [params].concat(args));
    };
  });

  if (inputFunc) {
    ons[modelEvent] = function (targetEvnt) {
      inputFunc(targetEvnt);

      if (events && events[modelEvent]) {
        events[modelEvent](params, targetEvnt);
      }

      if (isSameEvent && changeFunc) {
        changeFunc(targetEvnt);
      }
    };
  }

  if (!isSameEvent && changeFunc) {
    ons[changeEvent] = function () {
      for (var _len3 = arguments.length, args = new Array(_len3), _key3 = 0; _key3 < _len3; _key3++) {
        args[_key3] = arguments[_key3];
      }

      changeFunc.apply(void 0, args);

      if (events && events[changeEvent]) {
        events[changeEvent].apply(events, [params].concat(args));
      }
    };
  }

  return ons;
}

function getEditOns(renderOpts, params) {
  var $table = params.$table,
      row = params.row,
      column = params.column;
  return getOns(renderOpts, params, function (value) {
    // 处理 model 值双向绑定
    _xeUtils["default"].set(row, column.property, value);
  }, function () {
    // 处理 change 事件相关逻辑
    $table.updateStatus(params);
  });
}

function getFilterOns(renderOpts, params, option, changeFunc) {
  return getOns(renderOpts, params, function (value) {
    // 处理 model 值双向绑定
    option.data = value;
  }, changeFunc);
}

function getItemOns(renderOpts, params) {
  var $form = params.$form,
      data = params.data,
      property = params.property;
  return getOns(renderOpts, params, function (value) {
    // 处理 model 值双向绑定
    _xeUtils["default"].set(data, property, value);
  }, function () {
    // 处理 change 事件相关逻辑
    $form.updateStatus(params);
  });
}

function matchCascaderData(index, list, values, labels) {
  var val = values[index];

  if (list && values.length > index) {
    _xeUtils["default"].each(list, function (item) {
      if (item.value === val) {
        labels.push(item.label);
        matchCascaderData(++index, item.children, values, labels);
      }
    });
  }
}

function getSelectCellValue(renderOpts, params) {
  var _renderOpts$options = renderOpts.options,
      options = _renderOpts$options === void 0 ? [] : _renderOpts$options,
      optionGroups = renderOpts.optionGroups,
      _renderOpts$props = renderOpts.props,
      props = _renderOpts$props === void 0 ? {} : _renderOpts$props,
      _renderOpts$optionPro = renderOpts.optionProps,
      optionProps = _renderOpts$optionPro === void 0 ? {} : _renderOpts$optionPro,
      _renderOpts$optionGro = renderOpts.optionGroupProps,
      optionGroupProps = _renderOpts$optionGro === void 0 ? {} : _renderOpts$optionGro;
  var row = params.row,
      column = params.column;
  var $table = params.$table;
  var labelProp = optionProps.label || 'label';
  var valueProp = optionProps.value || 'value';
  var groupOptions = optionGroupProps.options || 'options';

  var cellValue = _xeUtils["default"].get(row, column.property);

  var colid = column.id;
  var rest;
  var cellData;

  if (props.filterable) {
    var fullAllDataRowMap = $table.fullAllDataRowMap;
    var cacheCell = fullAllDataRowMap.has(row);

    if (cacheCell) {
      rest = fullAllDataRowMap.get(row);
      cellData = rest.cellData;

      if (!cellData) {
        cellData = fullAllDataRowMap.get(row).cellData = {};
      }
    }

    if (rest && cellData[colid] && cellData[colid].value === cellValue) {
      return cellData[colid].label;
    }
  }

  if (!isEmptyValue(cellValue)) {
    var selectlabel = _xeUtils["default"].map(props.multiple ? cellValue : [cellValue], optionGroups ? function (value) {
      var selectItem;

      for (var index = 0; index < optionGroups.length; index++) {
        selectItem = _xeUtils["default"].find(optionGroups[index][groupOptions], function (item) {
          return item[valueProp] === value;
        });

        if (selectItem) {
          break;
        }
      }

      return selectItem ? selectItem[labelProp] : value;
    } : function (value) {
      var selectItem = _xeUtils["default"].find(options, function (item) {
        return item[valueProp] === value;
      });

      return selectItem ? selectItem[labelProp] : value;
    }).join(', ');

    if (cellData && options && options.length) {
      cellData[colid] = {
        value: cellValue,
        label: selectlabel
      };
    }

    return selectlabel;
  }

  return '';
}

function getCascaderCellValue(renderOpts, params) {
  var _renderOpts$props2 = renderOpts.props,
      props = _renderOpts$props2 === void 0 ? {} : _renderOpts$props2;
  var row = params.row,
      column = params.column;

  var cellValue = _xeUtils["default"].get(row, column.property);

  var values = cellValue || [];
  var labels = [];
  matchCascaderData(0, props.options, values, labels);
  return (props.showAllLevels === false ? labels.slice(labels.length - 1, labels.length) : labels).join(" ".concat(props.separator || '/', " "));
}

function getDatePickerCellValue(renderOpts, params) {
  var _renderOpts$props3 = renderOpts.props,
      props = _renderOpts$props3 === void 0 ? {} : _renderOpts$props3;
  var row = params.row,
      column = params.column;
  var _props$rangeSeparator = props.rangeSeparator,
      rangeSeparator = _props$rangeSeparator === void 0 ? '-' : _props$rangeSeparator;

  var cellValue = _xeUtils["default"].get(row, column.property);

  switch (props.type) {
    case 'week':
      cellValue = getFormatDate(cellValue, props, 'yyyywWW');
      break;

    case 'month':
      cellValue = getFormatDate(cellValue, props, 'yyyy-MM');
      break;

    case 'year':
      cellValue = getFormatDate(cellValue, props, 'yyyy');
      break;

    case 'dates':
      cellValue = getFormatDates(cellValue, props, ', ', 'yyyy-MM-dd');
      break;

    case 'daterange':
      cellValue = getFormatDates(cellValue, props, " ".concat(rangeSeparator, " "), 'yyyy-MM-dd');
      break;

    case 'datetimerange':
      cellValue = getFormatDates(cellValue, props, " ".concat(rangeSeparator, " "), 'yyyy-MM-dd HH:ss:mm');
      break;

    case 'monthrange':
      cellValue = getFormatDates(cellValue, props, " ".concat(rangeSeparator, " "), 'yyyy-MM');
      break;

    default:
      cellValue = getFormatDate(cellValue, props, 'yyyy-MM-dd');
  }

  return cellValue;
}

function getTimePickerCellValue(renderOpts, params) {
  var _renderOpts$props4 = renderOpts.props,
      props = _renderOpts$props4 === void 0 ? {} : _renderOpts$props4;
  var row = params.row,
      column = params.column;
  var isRange = props.isRange,
      _props$format = props.format,
      format = _props$format === void 0 ? 'hh:mm:ss' : _props$format,
      _props$rangeSeparator2 = props.rangeSeparator,
      rangeSeparator = _props$rangeSeparator2 === void 0 ? '-' : _props$rangeSeparator2;

  var cellValue = _xeUtils["default"].get(row, column.property);

  if (cellValue && isRange) {
    cellValue = _xeUtils["default"].map(cellValue, function (date) {
      return _xeUtils["default"].toDateString(parseDate(date, props), format);
    }).join(" ".concat(rangeSeparator, " "));
  }

  return _xeUtils["default"].toDateString(parseDate(cellValue, props), format);
}

function createEditRender(defaultProps) {
  return function (h, renderOpts, params) {
    var row = params.row,
        column = params.column;
    var attrs = renderOpts.attrs;

    var cellValue = _xeUtils["default"].get(row, column.property);

    return [h(renderOpts.name, {
      attrs: attrs,
      props: getCellEditFilterProps(renderOpts, params, cellValue, defaultProps),
      on: getEditOns(renderOpts, params),
      nativeOn: getNativeOns(renderOpts, params)
    })];
  };
}

function defaultButtonEditRender(h, renderOpts, params) {
  var attrs = renderOpts.attrs;
  return [h('el-button', {
    attrs: attrs,
    props: getCellEditFilterProps(renderOpts, params, null),
    on: getOns(renderOpts, params),
    nativeOn: getNativeOns(renderOpts, params)
  }, cellText(h, renderOpts.content))];
}

function defaultButtonsEditRender(h, renderOpts, params) {
  return renderOpts.children.map(function (childRenderOpts) {
    return defaultButtonEditRender(h, childRenderOpts, params)[0];
  });
}

function createFilterRender(defaultProps) {
  return function (h, renderOpts, params) {
    var column = params.column;
    var name = renderOpts.name,
        attrs = renderOpts.attrs;
    var nativeOn = getNativeOns(renderOpts, params);
    return [h('div', {
      "class": 'vxe-table--filter-element-wrapper'
    }, column.filters.map(function (option, oIndex) {
      var optionValue = option.data;
      return h(name, {
        key: oIndex,
        attrs: attrs,
        props: getCellEditFilterProps(renderOpts, params, optionValue, defaultProps),
        on: getFilterOns(renderOpts, params, option, function () {
          // 处理 change 事件相关逻辑
          handleConfirmFilter(params, !!option.data, option);
        }),
        nativeOn: nativeOn
      });
    }))];
  };
}

function handleConfirmFilter(params, checked, option) {
  var $panel = params.$panel;
  $panel.changeOption({}, checked, option);
}
/**
 * 模糊匹配
 * @param params
 */


function defaultFuzzyFilterMethod(params) {
  var option = params.option,
      row = params.row,
      column = params.column;
  var data = option.data;

  var cellValue = _xeUtils["default"].get(row, column.property);

  return _xeUtils["default"].toValueString(cellValue).indexOf(data) > -1;
}
/**
 * 精确匹配
 * @param params
 */


function defaultExactFilterMethod(params) {
  var option = params.option,
      row = params.row,
      column = params.column;
  var data = option.data;

  var cellValue = _xeUtils["default"].get(row, column.property);
  /* eslint-disable eqeqeq */


  return cellValue === data;
}

function renderOptions(h, options, optionProps) {
  var labelProp = optionProps.label || 'label';
  var valueProp = optionProps.value || 'value';
  return _xeUtils["default"].map(options, function (item, oIndex) {
    return h('el-option', {
      key: oIndex,
      props: {
        value: item[valueProp],
        label: item[labelProp],
        disabled: item.disabled
      }
    });
  });
}

function cellText(h, cellValue) {
  return [formatText(cellValue)];
}

function createFormItemRender(defaultProps) {
  return function (h, renderOpts, params) {
    var data = params.data,
        property = params.property;
    var name = renderOpts.name;
    var attrs = renderOpts.attrs;

    var itemValue = _xeUtils["default"].get(data, property);

    return [h(name, {
      attrs: attrs,
      props: getItemProps(renderOpts, params, itemValue, defaultProps),
      on: getItemOns(renderOpts, params),
      nativeOn: getNativeOns(renderOpts, params)
    })];
  };
}

function defaultButtonItemRender(h, renderOpts, params) {
  var attrs = renderOpts.attrs;
  var props = getItemProps(renderOpts, params, null);
  return [h('el-button', {
    attrs: attrs,
    props: props,
    on: getOns(renderOpts, params),
    nativeOn: getNativeOns(renderOpts, params)
  }, cellText(h, renderOpts.content || props.content))];
}

function defaultButtonsItemRender(h, renderOpts, params) {
  return renderOpts.children.map(function (childRenderOpts) {
    return defaultButtonItemRender(h, childRenderOpts, params)[0];
  });
}

function createExportMethod(getExportCellValue) {
  return function (params) {
    var row = params.row,
        column = params.column,
        options = params.options;
    return options && options.original ? _xeUtils["default"].get(row, column.property) : getExportCellValue(column.editRender || column.cellRender, params);
  };
}

function createFormItemRadioAndCheckboxRender() {
  return function (h, renderOpts, params) {
    var name = renderOpts.name,
        _renderOpts$options2 = renderOpts.options,
        options = _renderOpts$options2 === void 0 ? [] : _renderOpts$options2,
        _renderOpts$optionPro2 = renderOpts.optionProps,
        optionProps = _renderOpts$optionPro2 === void 0 ? {} : _renderOpts$optionPro2,
        attrs = renderOpts.attrs;
    var data = params.data,
        property = params.property;
    var labelProp = optionProps.label || 'label';
    var valueProp = optionProps.value || 'value';

    var itemValue = _xeUtils["default"].get(data, property);

    return [h("".concat(name, "Group"), {
      attrs: attrs,
      props: getItemProps(renderOpts, params, itemValue),
      on: getItemOns(renderOpts, params),
      nativeOn: getNativeOns(renderOpts, params)
    }, options.map(function (option, oIndex) {
      return h(name, {
        key: oIndex,
        props: {
          label: option[valueProp],
          disabled: option.disabled
        }
      }, option[labelProp]);
    }))];
  };
}
/**
 * 检查触发源是否属于目标节点
 */


function getEventTargetNode(evnt, container, className) {
  var targetElem;
  var target = evnt.target;

  while (target && target.nodeType && target !== document) {
    if (className && target.className && target.className.split && target.className.split(' ').indexOf(className) > -1) {
      targetElem = target;
    } else if (target === container) {
      return {
        flag: className ? !!targetElem : true,
        container: container,
        targetElem: targetElem
      };
    }

    target = target.parentNode;
  }

  return {
    flag: false
  };
}
/**
 * 事件兼容性处理
 */


function handleClearEvent(params, e) {
  var bodyElem = document.body;
  var evnt = params.$event || e;

  if ( // 远程搜索
  getEventTargetNode(evnt, bodyElem, 'el-autocomplete-suggestion').flag || // 下拉框
  getEventTargetNode(evnt, bodyElem, 'el-select-dropdown').flag || // 级联
  getEventTargetNode(evnt, bodyElem, 'el-cascader__dropdown').flag || getEventTargetNode(evnt, bodyElem, 'el-cascader-menus').flag || // 日期
  getEventTargetNode(evnt, bodyElem, 'el-time-panel').flag || getEventTargetNode(evnt, bodyElem, 'el-picker-panel').flag || // 颜色
  getEventTargetNode(evnt, bodyElem, 'el-color-dropdown').flag) {
    return false;
  }
}
/**
 * 基于 vxe-table 表格的适配插件，用于兼容 element-ui 组件库
 */


var VXETablePluginElement = {
  install: function install(_ref) {
    var interceptor = _ref.interceptor,
        renderer = _ref.renderer;
    renderer.mixin({
      ElAutocomplete: {
        autofocus: 'input.el-input__inner',
        renderDefault: createEditRender(),
        renderEdit: createEditRender(),
        renderFilter: createFilterRender(),
        defaultFilterMethod: defaultExactFilterMethod,
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender()
      },
      ElInput: {
        autofocus: 'input.el-input__inner',
        renderDefault: createEditRender(),
        renderEdit: createEditRender(),
        renderFilter: createFilterRender(),
        defaultFilterMethod: defaultFuzzyFilterMethod,
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender()
      },
      ElInputNumber: {
        autofocus: 'input.el-input__inner',
        renderDefault: createEditRender(),
        renderEdit: createEditRender(),
        renderFilter: createFilterRender(),
        defaultFilterMethod: defaultFuzzyFilterMethod,
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender()
      },
      ElSelect: {
        renderEdit: function renderEdit(h, renderOpts, params) {
          var _renderOpts$options3 = renderOpts.options,
              options = _renderOpts$options3 === void 0 ? [] : _renderOpts$options3,
              optionGroups = renderOpts.optionGroups,
              _renderOpts$optionPro3 = renderOpts.optionProps,
              optionProps = _renderOpts$optionPro3 === void 0 ? {} : _renderOpts$optionPro3,
              _renderOpts$optionGro2 = renderOpts.optionGroupProps,
              optionGroupProps = _renderOpts$optionGro2 === void 0 ? {} : _renderOpts$optionGro2;
          var row = params.row,
              column = params.column;
          var attrs = renderOpts.attrs;

          var cellValue = _xeUtils["default"].get(row, column.property);

          var props = getCellEditFilterProps(renderOpts, params, cellValue);
          var on = getEditOns(renderOpts, params);
          var nativeOn = getNativeOns(renderOpts, params);

          if (optionGroups) {
            var groupOptions = optionGroupProps.options || 'options';
            var groupLabel = optionGroupProps.label || 'label';
            return [h('el-select', {
              attrs: attrs,
              props: props,
              on: on,
              nativeOn: nativeOn
            }, _xeUtils["default"].map(optionGroups, function (group, gIndex) {
              return h('el-option-group', {
                key: gIndex,
                props: {
                  label: group[groupLabel]
                }
              }, renderOptions(h, group[groupOptions], optionProps));
            }))];
          }

          return [h('el-select', {
            props: props,
            attrs: attrs,
            on: on,
            nativeOn: nativeOn
          }, renderOptions(h, options, optionProps))];
        },
        renderCell: function renderCell(h, renderOpts, params) {
          return getCellLabelVNs(h, renderOpts, params, getSelectCellValue(renderOpts, params));
        },
        renderFilter: function renderFilter(h, renderOpts, params) {
          var _renderOpts$options4 = renderOpts.options,
              options = _renderOpts$options4 === void 0 ? [] : _renderOpts$options4,
              optionGroups = renderOpts.optionGroups,
              _renderOpts$optionPro4 = renderOpts.optionProps,
              optionProps = _renderOpts$optionPro4 === void 0 ? {} : _renderOpts$optionPro4,
              _renderOpts$optionGro3 = renderOpts.optionGroupProps,
              optionGroupProps = _renderOpts$optionGro3 === void 0 ? {} : _renderOpts$optionGro3;
          var groupOptions = optionGroupProps.options || 'options';
          var groupLabel = optionGroupProps.label || 'label';
          var column = params.column;
          var attrs = renderOpts.attrs;
          var nativeOn = getNativeOns(renderOpts, params);
          return [h('div', {
            "class": 'vxe-table--filter-element-wrapper'
          }, optionGroups ? column.filters.map(function (option, oIndex) {
            var optionValue = option.data;
            var props = getCellEditFilterProps(renderOpts, params, optionValue);
            return h('el-select', {
              key: oIndex,
              attrs: attrs,
              props: props,
              on: getFilterOns(renderOpts, params, option, function () {
                // 处理 change 事件相关逻辑
                handleConfirmFilter(params, props.multiple ? option.data && option.data.length > 0 : !_xeUtils["default"].eqNull(option.data), option);
              }),
              nativeOn: nativeOn
            }, _xeUtils["default"].map(optionGroups, function (group, gIndex) {
              return h('el-option-group', {
                key: gIndex,
                props: {
                  label: group[groupLabel]
                }
              }, renderOptions(h, group[groupOptions], optionProps));
            }));
          }) : column.filters.map(function (option, oIndex) {
            var optionValue = option.data;
            var props = getCellEditFilterProps(renderOpts, params, optionValue);
            return h('el-select', {
              key: oIndex,
              attrs: attrs,
              props: props,
              on: getFilterOns(renderOpts, params, option, function () {
                // 处理 change 事件相关逻辑
                handleConfirmFilter(params, props.multiple ? option.data && option.data.length > 0 : !_xeUtils["default"].eqNull(option.data), option);
              }),
              nativeOn: nativeOn
            }, renderOptions(h, options, optionProps));
          }))];
        },
        defaultFilterMethod: function defaultFilterMethod(params) {
          var option = params.option,
              row = params.row,
              column = params.column;
          var data = option.data;
          var property = column.property,
              renderOpts = column.filterRender;
          var _renderOpts$props5 = renderOpts.props,
              props = _renderOpts$props5 === void 0 ? {} : _renderOpts$props5;

          var cellValue = _xeUtils["default"].get(row, property);

          if (props.multiple) {
            if (_xeUtils["default"].isArray(cellValue)) {
              return _xeUtils["default"].includeArrays(cellValue, data);
            }

            return data.indexOf(cellValue) > -1;
          }
          /* eslint-disable eqeqeq */


          return cellValue == data;
        },
        renderItem: function renderItem(h, renderOpts, params) {
          var _renderOpts$options5 = renderOpts.options,
              options = _renderOpts$options5 === void 0 ? [] : _renderOpts$options5,
              optionGroups = renderOpts.optionGroups,
              _renderOpts$optionPro5 = renderOpts.optionProps,
              optionProps = _renderOpts$optionPro5 === void 0 ? {} : _renderOpts$optionPro5,
              _renderOpts$optionGro4 = renderOpts.optionGroupProps,
              optionGroupProps = _renderOpts$optionGro4 === void 0 ? {} : _renderOpts$optionGro4;
          var data = params.data,
              property = params.property;
          var attrs = renderOpts.attrs;

          var itemValue = _xeUtils["default"].get(data, property);

          var props = getItemProps(renderOpts, params, itemValue);
          var on = getItemOns(renderOpts, params);
          var nativeOn = getNativeOns(renderOpts, params);

          if (optionGroups) {
            var groupOptions = optionGroupProps.options || 'options';
            var groupLabel = optionGroupProps.label || 'label';
            return [h('el-select', {
              attrs: attrs,
              props: props,
              on: on,
              nativeOn: nativeOn
            }, _xeUtils["default"].map(optionGroups, function (group, gIndex) {
              return h('el-option-group', {
                props: {
                  label: group[groupLabel]
                },
                key: gIndex
              }, renderOptions(h, group[groupOptions], optionProps));
            }))];
          }

          return [h('el-select', {
            attrs: attrs,
            props: props,
            on: on,
            nativeOn: nativeOn
          }, renderOptions(h, options, optionProps))];
        },
        renderItemContent: function renderItemContent(h, renderOpts, params) {
          var _renderOpts$options6 = renderOpts.options,
              options = _renderOpts$options6 === void 0 ? [] : _renderOpts$options6,
              optionGroups = renderOpts.optionGroups,
              _renderOpts$optionPro6 = renderOpts.optionProps,
              optionProps = _renderOpts$optionPro6 === void 0 ? {} : _renderOpts$optionPro6,
              _renderOpts$optionGro5 = renderOpts.optionGroupProps,
              optionGroupProps = _renderOpts$optionGro5 === void 0 ? {} : _renderOpts$optionGro5;
          var data = params.data,
              property = params.property;
          var attrs = renderOpts.attrs;

          var itemValue = _xeUtils["default"].get(data, property);

          var props = getItemProps(renderOpts, params, itemValue);
          var on = getItemOns(renderOpts, params);
          var nativeOn = getNativeOns(renderOpts, params);

          if (optionGroups) {
            var groupOptions = optionGroupProps.options || 'options';
            var groupLabel = optionGroupProps.label || 'label';
            return [h('el-select', {
              attrs: attrs,
              props: props,
              on: on,
              nativeOn: nativeOn
            }, _xeUtils["default"].map(optionGroups, function (group, gIndex) {
              return h('el-option-group', {
                props: {
                  label: group[groupLabel]
                },
                key: gIndex
              }, renderOptions(h, group[groupOptions], optionProps));
            }))];
          }

          return [h('el-select', {
            attrs: attrs,
            props: props,
            on: on,
            nativeOn: nativeOn
          }, renderOptions(h, options, optionProps))];
        },
        cellExportMethod: createExportMethod(getSelectCellValue),
        exportMethod: createExportMethod(getSelectCellValue)
      },
      ElCascader: {
        renderEdit: createEditRender(),
        renderCell: function renderCell(h, renderOpts, params) {
          return getCellLabelVNs(h, renderOpts, params, getCascaderCellValue(renderOpts, params));
        },
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender(),
        cellExportMethod: createExportMethod(getCascaderCellValue),
        exportMethod: createExportMethod(getCascaderCellValue)
      },
      ElDatePicker: {
        renderEdit: createEditRender(),
        renderCell: function renderCell(h, renderOpts, params) {
          return getCellLabelVNs(h, renderOpts, params, getDatePickerCellValue(renderOpts, params));
        },
        renderFilter: function renderFilter(h, renderOpts, params) {
          var column = params.column;
          var attrs = renderOpts.attrs;
          var nativeOn = getNativeOns(renderOpts, params);
          return [h('div', {
            "class": 'vxe-table--filter-element-wrapper'
          }, column.filters.map(function (option, oIndex) {
            var optionValue = option.data;
            return h(renderOpts.name, {
              key: oIndex,
              attrs: attrs,
              props: getCellEditFilterProps(renderOpts, params, optionValue),
              on: getFilterOns(renderOpts, params, option, function () {
                // 处理 change 事件相关逻辑
                handleConfirmFilter(params, !!option.data, option);
              }),
              nativeOn: nativeOn
            });
          }))];
        },
        defaultFilterMethod: function defaultFilterMethod(params) {
          var option = params.option,
              row = params.row,
              column = params.column;
          var data = option.data;
          var renderOpts = column.filterRender;
          var _renderOpts$props6 = renderOpts.props,
              props = _renderOpts$props6 === void 0 ? {} : _renderOpts$props6;

          var cellValue = _xeUtils["default"].get(row, column.property);

          if (data) {
            switch (props.type) {
              case 'daterange':
                return equalDaterange(cellValue, data, props, 'yyyy-MM-dd');

              case 'datetimerange':
                return equalDaterange(cellValue, data, props, 'yyyy-MM-dd HH:ss:mm');

              case 'monthrange':
                return equalDaterange(cellValue, data, props, 'yyyy-MM');

              default:
                return cellValue === data;
            }
          }

          return false;
        },
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender(),
        cellExportMethod: createExportMethod(getDatePickerCellValue),
        exportMethod: createExportMethod(getDatePickerCellValue)
      },
      ElTimePicker: {
        renderEdit: createEditRender(),
        renderCell: function renderCell(h, renderOpts, params) {
          return getCellLabelVNs(h, renderOpts, params, getTimePickerCellValue(renderOpts, params));
        },
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender(),
        cellExportMethod: createExportMethod(getTimePickerCellValue),
        exportMethod: createExportMethod(getTimePickerCellValue)
      },
      ElTimeSelect: {
        renderEdit: createEditRender(),
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender()
      },
      ElRate: {
        renderDefault: createEditRender(),
        renderEdit: createEditRender(),
        renderFilter: createFilterRender(),
        defaultFilterMethod: defaultExactFilterMethod,
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender()
      },
      ElSwitch: {
        renderDefault: createEditRender(),
        renderEdit: createEditRender(),
        renderFilter: function renderFilter(h, renderOpts, params) {
          var column = params.column;
          var name = renderOpts.name,
              attrs = renderOpts.attrs;
          var nativeOn = getNativeOns(renderOpts, params);
          return [h('div', {
            "class": 'vxe-table--filter-element-wrapper'
          }, column.filters.map(function (option, oIndex) {
            var optionValue = option.data;
            return h(name, {
              key: oIndex,
              attrs: attrs,
              props: getCellEditFilterProps(renderOpts, params, optionValue),
              on: getFilterOns(renderOpts, params, option, function () {
                // 处理 change 事件相关逻辑
                handleConfirmFilter(params, _xeUtils["default"].isBoolean(option.data), option);
              }),
              nativeOn: nativeOn
            });
          }))];
        },
        defaultFilterMethod: defaultExactFilterMethod,
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender()
      },
      ElSlider: {
        renderDefault: createEditRender(),
        renderEdit: createEditRender(),
        renderFilter: createFilterRender(),
        defaultFilterMethod: defaultExactFilterMethod,
        renderItem: createFormItemRender(),
        renderItemContent: createFormItemRender()
      },
      ElRadio: {
        renderItem: createFormItemRadioAndCheckboxRender(),
        renderItemContent: createFormItemRadioAndCheckboxRender()
      },
      ElCheckbox: {
        renderItem: createFormItemRadioAndCheckboxRender(),
        renderItemContent: createFormItemRadioAndCheckboxRender()
      },
      ElButton: {
        renderDefault: defaultButtonEditRender,
        renderItem: defaultButtonItemRender,
        renderItemContent: defaultButtonItemRender
      },
      ElButtons: {
        renderDefault: defaultButtonsEditRender,
        renderItem: defaultButtonsItemRender,
        renderItemContent: defaultButtonsItemRender
      }
    });
    interceptor.add('event.clearFilter', handleClearEvent);
    interceptor.add('event.clearActived', handleClearEvent);
    interceptor.add('event.clearAreas', handleClearEvent);
  }
};
exports.VXETablePluginElement = VXETablePluginElement;

if (typeof window !== 'undefined' && window.VXETable && window.VXETable.use) {
  window.VXETable.use(VXETablePluginElement);
}

var _default = VXETablePluginElement;
exports["default"] = _default;
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
