"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = exports.VXETablePluginMenus = void 0;

var _xeUtils = _interopRequireDefault(require("xe-utils"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

var vxetable;
var handleCopy;

var columnIndexOf = function columnIndexOf(cols, column) {
  return _xeUtils["default"].findIndexOf(cols, function (item) {
    return item.id === column.id;
  });
};

var rowIndexOf = function rowIndexOf($table, rows, row) {
  return $table.findRowIndexOf(rows, row);
};

function handleFixedColumn(fixed) {
  return function (params) {
    var $table = params.$table,
        column = params.column;

    _xeUtils["default"].eachTree([column], function (column) {
      column.fixed = fixed;
    });

    $table.refreshColumn();
  };
}

function handleCopyOrCut(params, isCut) {
  var $event = params.$event,
      $table = params.$table,
      row = params.row,
      column = params.column;

  if (row && column) {
    var props = $table.props;
    var mouseConfig = props.mouseConfig;

    var _$table$getComputeMap = $table.getComputeMaps(),
        computeMouseOpts = _$table$getComputeMap.computeMouseOpts;

    var mouseOpts = computeMouseOpts.value;
    var text = '';

    if (mouseConfig && mouseOpts.area) {
      if (isCut) {
        $table.triggerCutCellAreaEvent($event);
      } else {
        $table.triggerCopyCellAreaEvent($event);
      }

      text = vxetable.config.clipboard.text;
    } else {
      text = _xeUtils["default"].toValueString(_xeUtils["default"].get(row, column.property)); // 操作内置剪贴板

      vxetable.config.clipboard = {
        text: text,
        html: ''
      };
    } // 开始复制操作


    if (_xeUtils["default"].isFunction(handleCopy)) {
      handleCopy(text);
    } else {
      console.warn('Copy function does not exist, copy to clipboard failed.');
    }
  }
}

function checkCellOverlay(params, cellAreas) {
  var $table = params.$table;

  var _$table$getTableData = $table.getTableData(),
      visibleData = _$table$getTableData.visibleData;

  var _$table$getTableColum = $table.getTableColumn(),
      visibleColumn = _$table$getTableColum.visibleColumn;

  var indexMaps = {};

  for (var aIndex = 0, areaSize = cellAreas.length; aIndex < areaSize; aIndex++) {
    var areaItem = cellAreas[aIndex];
    var rows = areaItem.rows,
        cols = areaItem.cols;

    for (var rIndex = 0, rowSize = rows.length; rIndex < rowSize; rIndex++) {
      var offsetRow = rows[rIndex];
      var orIndex = rowIndexOf($table, visibleData, offsetRow);

      for (var cIndex = 0, colSize = cols.length; cIndex < colSize; cIndex++) {
        var offsetColumn = cols[cIndex];
        var ocIndex = columnIndexOf(visibleColumn, offsetColumn);
        var key = orIndex + ':' + ocIndex;

        if (indexMaps[key]) {
          return false;
        }

        indexMaps[key] = true;
      }
    }
  }

  return true;
}

function getBeenMerges(params) {
  var $table = params.$table;
  var props = $table.props;
  var mouseConfig = props.mouseConfig;

  var _$table$getComputeMap2 = $table.getComputeMaps(),
      computeMouseOpts = _$table$getComputeMap2.computeMouseOpts;

  var mouseOpts = computeMouseOpts.value;

  var _$table$getTableData2 = $table.getTableData(),
      visibleData = _$table$getTableData2.visibleData;

  var _$table$getTableColum2 = $table.getTableColumn(),
      visibleColumn = _$table$getTableColum2.visibleColumn;

  var cellAreas = mouseConfig && mouseOpts.area ? $table.getCellAreas() : [];
  var mergeList = $table.getMergeCells();
  return mergeList.filter(function (_ref) {
    var mergeRowIndex = _ref.row,
        mergeColIndex = _ref.col,
        mergeRowspan = _ref.rowspan,
        mergeColspan = _ref.colspan;
    return cellAreas.some(function (areaItem) {
      var rows = areaItem.rows,
          cols = areaItem.cols;
      var startRowIndex = rowIndexOf($table, visibleData, rows[0]);
      var endRowIndex = rowIndexOf($table, visibleData, rows[rows.length - 1]);
      var startColIndex = columnIndexOf(visibleColumn, cols[0]);
      var endColIndex = columnIndexOf(visibleColumn, cols[cols.length - 1]);
      return mergeRowIndex >= startRowIndex && mergeRowIndex + mergeRowspan - 1 <= endRowIndex && mergeColIndex >= startColIndex && mergeColIndex + mergeColspan - 1 <= endColIndex;
    });
  });
}

function handleClearMergeCells(params) {
  var $table = params.$table;
  var beenMerges = getBeenMerges(params);

  if (beenMerges.length) {
    $table.removeMergeCells(beenMerges);
  }

  return beenMerges;
}

function checkPrivilege(item, params) {
  var code = item.code;
  var $table = params.$table,
      row = params.row,
      column = params.column;
  var props = $table.props;
  var editConfig = props.editConfig,
      mouseConfig = props.mouseConfig;

  switch (code) {
    case 'CLEAR_ALL_SORT':
      {
        var sortList = $table.getSortColumns();
        item.disabled = !sortList.length;
        break;
      }

    case 'CLEAR_ALL_FILTER':
      {
        var filterList = $table.getCheckedFilters();
        item.disabled = !filterList.length;
        break;
      }

    case 'CLEAR_ALL_MERGE':
      {
        var mergeCells = $table.getMergeCells();
        var mergeFooterItems = $table.getMergeFooterItems();
        item.disabled = !mergeCells.length && !mergeFooterItems.length;
        break;
      }

    case 'CLEAR_MERGE_CELL':
      {
        var beenMerges = getBeenMerges(params);
        item.disabled = !beenMerges.length;
        break;
      }

    case 'EDIT_CELL':
    case 'CLEAR_CELL':
    case 'CLEAR_ROW':
    case 'COPY_CELL':
    case 'CUT_CELL':
    case 'PASTE_CELL':
    case 'MERGE_OR_CLEAR':
    case 'MERGE_CELL':
    case 'REVERT_CELL':
    case 'REVERT_ROW':
    case 'INSERT_AT_ROW':
    case 'INSERT_AT_ACTIVED_ROW':
    case 'DELETE_ROW':
    case 'CLEAR_SORT':
    case 'SORT_ASC':
    case 'SORT_DESC':
    case 'CLEAR_FILTER':
    case 'FILTER_CELL':
    case 'EXPORT_ROW':
    case 'OPEN_FIND':
    case 'OPEN_REPLACE':
    case 'HIDDEN_COLUMN':
    case 'FIXED_LEFT_COLUMN':
    case 'FIXED_RIGHT_COLUMN':
    case 'CLEAR_FIXED_COLUMN':
      {
        item.disabled = !column;

        if (column) {
          var _$table$getComputeMap3 = $table.getComputeMaps(),
              computeMouseOpts = _$table$getComputeMap3.computeMouseOpts;

          var mouseOpts = computeMouseOpts.value;
          var isChildCol = !!column.parentId;

          switch (code) {
            case 'CLEAR_SORT':
              {
                item.disabled = !column.sortable || !column.order;
                break;
              }

            case 'SORT_ASC':
            case 'SORT_DESC':
              item.disabled = !column.sortable;
              break;

            case 'FILTER_CELL':
            case 'CLEAR_FILTER':
              item.disabled = !column.filters || !column.filters.length;

              if (!item.disabled) {
                switch (code) {
                  case 'CLEAR_FILTER':
                    item.disabled = !column.filters.some(function (option) {
                      return option.checked;
                    });
                    break;
                }
              }

              break;

            case 'REVERT_CELL':
              {
                item.disabled = !row || !column.property || !$table.isUpdateByRow(row, column.property);
                break;
              }

            case 'REVERT_ROW':
              {
                item.disabled = !row || !column.property || !$table.isUpdateByRow(row);
                break;
              }

            case 'OPEN_FIND':
            case 'OPEN_REPLACE':
              {
                item.disabled = !(mouseConfig && mouseOpts.area);
                break;
              }

            case 'EDIT_CELL':
              {
                item.disabled = !editConfig || !column.editRender;
                break;
              }

            case 'COPY_CELL':
            case 'CUT_CELL':
            case 'PASTE_CELL':
              {
                var cellAreas = mouseConfig && mouseOpts.area ? $table.getCellAreas() : [];
                item.disabled = cellAreas.length > 1;

                if (!item.disabled) {
                  switch (code) {
                    case 'PASTE_CELL':
                      var clipboard = vxetable.config.clipboard;
                      item.disabled = !clipboard || !clipboard.text;
                      break;
                  }
                }

                break;
              }

            case 'MERGE_OR_CLEAR':
            case 'MERGE_CELL':
              {
                var _cellAreas = mouseConfig && mouseOpts.area ? $table.getCellAreas() : [];

                item.disabled = !_cellAreas.length || _cellAreas.length === 1 && _cellAreas[0].rows.length === 1 && _cellAreas[0].cols.length === 1 || !checkCellOverlay(params, _cellAreas);
                break;
              }

            case 'FIXED_LEFT_COLUMN':
              item.disabled = isChildCol || column.fixed === 'left';
              break;

            case 'FIXED_RIGHT_COLUMN':
              item.disabled = isChildCol || column.fixed === 'right';
              break;

            case 'CLEAR_FIXED_COLUMN':
              item.disabled = isChildCol || !column.fixed;
              break;
          }
        }

        break;
      }
  }
}

function handlePrivilegeEvent(params) {
  params.options.forEach(function (list) {
    list.forEach(function (item) {
      checkPrivilege(item, params);

      if (item.children) {
        item.children.forEach(function (child) {
          checkPrivilege(child, params);
        });
      }
    });
  });
  return true;
}

function pluginSetup(options) {
  if (options && options.copy) {
    handleCopy = options.copy;
  }
}
/**
 * 基于 vxe-table 表格的增强插件，提供实用的快捷菜单集
 */


var VXETablePluginMenus = {
  setup: pluginSetup,
  install: function install(vxetablecore, options) {
    var interceptor = vxetablecore.interceptor,
        menus = vxetablecore.menus;
    vxetable = vxetablecore;

    if (window.XEClipboard) {
      handleCopy = window.XEClipboard.copy;
    }

    pluginSetup(options);
    menus.mixin({
      /**
       * 清除单元格数据的值；如果启用 mouse-config.area 功能，则清除区域范围内的单元格数据
       */
      CLEAR_CELL: function CLEAR_CELL(params) {
        var $table = params.$table,
            row = params.row,
            column = params.column;

        if (row && column) {
          var props = $table.props;
          var mouseConfig = props.mouseConfig;

          var _$table$getComputeMap4 = $table.getComputeMaps(),
              computeMouseOpts = _$table$getComputeMap4.computeMouseOpts;

          var mouseOpts = computeMouseOpts.value;

          if (mouseConfig && mouseOpts.area) {
            var cellAreas = $table.getCellAreas();

            if (cellAreas && cellAreas.length) {
              cellAreas.forEach(function (areaItem) {
                var rows = areaItem.rows,
                    cols = areaItem.cols;
                cols.forEach(function (column) {
                  rows.forEach(function (row) {
                    $table.clearData(row, column.property);
                  });
                });
              });
            }
          } else {
            $table.clearData(row, column.property);
          }
        }
      },

      /**
       * 清除行数据的值
       */
      CLEAR_ROW: function CLEAR_ROW(params) {
        var $table = params.$table,
            row = params.row;

        if (row) {
          $table.clearData(row);
        }
      },

      /**
       * 清除复选框选中行数据的值
       */
      CLEAR_CHECKBOX_ROW: function CLEAR_CHECKBOX_ROW(params) {
        var $table = params.$table;
        $table.clearData($table.getCheckboxRecords());
      },

      /**
       * 清除所有数据的值
       */
      CLEAR_ALL: function CLEAR_ALL(params) {
        var $table = params.$table;
        $table.clearData();
      },

      /**
       * 还原单元格数据的值；如果启用 mouse-config.area 功能，则还原区域范围内的单元格数据
       */
      REVERT_CELL: function REVERT_CELL(params) {
        var $table = params.$table,
            row = params.row,
            column = params.column;

        if (row && column) {
          var props = $table.props;
          var mouseConfig = props.mouseConfig;

          var _$table$getComputeMap5 = $table.getComputeMaps(),
              computeMouseOpts = _$table$getComputeMap5.computeMouseOpts;

          var mouseOpts = computeMouseOpts.value;

          if (mouseConfig && mouseOpts.area) {
            var cellAreas = $table.getCellAreas();

            if (cellAreas && cellAreas.length) {
              cellAreas.forEach(function (areaItem) {
                var rows = areaItem.rows,
                    cols = areaItem.cols;
                cols.forEach(function (column) {
                  rows.forEach(function (row) {
                    $table.revertData(row, column.property);
                  });
                });
              });
            }
          } else {
            $table.revertData(row, column.property);
          }
        }
      },

      /**
       * 还原行数据的值
       */
      REVERT_ROW: function REVERT_ROW(params) {
        var $table = params.$table,
            row = params.row;

        if (row) {
          $table.revertData(row);
        }
      },

      /**
       * 还原复选框选中行数据的值
       */
      REVERT_CHECKBOX_ROW: function REVERT_CHECKBOX_ROW(params) {
        var $table = params.$table;
        $table.revertData($table.getCheckboxRecords());
      },

      /**
       * 还原所有数据的值
       */
      REVERT_ALL: function REVERT_ALL(params) {
        var $table = params.$table;
        $table.revertData();
      },

      /**
       * 复制单元格数据的值；如果启用 mouse-config.area 功能，则复制区域范围内的单元格数据，支持 Excel 和 WPS
       */
      COPY_CELL: function COPY_CELL(params) {
        handleCopyOrCut(params);
      },

      /**
       * 剪贴单元格数据的值；如果启用 mouse-config.area 功能，则剪贴区域范围内的单元格数据，支持 Excel 和 WPS
       */
      CUT_CELL: function CUT_CELL(params) {
        handleCopyOrCut(params, true);
      },

      /**
       * 粘贴从表格中被复制的数据；如果启用 mouse-config.area 功能，则粘贴区域范围内的单元格数据，不支持读取剪贴板
       */
      PASTE_CELL: function PASTE_CELL(params) {
        var $event = params.$event,
            $table = params.$table,
            row = params.row,
            column = params.column;
        var props = $table.props;
        var mouseConfig = props.mouseConfig;

        var _$table$getComputeMap6 = $table.getComputeMaps(),
            computeMouseOpts = _$table$getComputeMap6.computeMouseOpts;

        var mouseOpts = computeMouseOpts.value;

        if (mouseConfig && mouseOpts.area) {
          $table.triggerPasteCellAreaEvent($event);
        } else {
          var clipboard = vxetable.config.clipboard; // 读取内置剪贴板

          if (clipboard && clipboard.text) {
            _xeUtils["default"].set(row, column.property, clipboard.text);
          }
        }
      },

      /**
       * 如果启用 mouse-config.area 功能，如果所选区域内已存在合并单元格，则取消临时合并，否则临时合并
       */
      MERGE_OR_CLEAR: function MERGE_OR_CLEAR(params) {
        var $event = params.$event,
            $table = params.$table;
        var cellAreas = $table.getCellAreas();
        var beenMerges = getBeenMerges(params);
        var status = false;

        if (beenMerges.length) {
          $table.removeMergeCells(beenMerges);
        } else {
          status = true;
          $table.setMergeCells(cellAreas.map(function (_ref2) {
            var rows = _ref2.rows,
                cols = _ref2.cols;
            return {
              row: rows[0],
              col: cols[0],
              rowspan: rows.length,
              colspan: cols.length
            };
          }));
        }

        var targetAreas = cellAreas.map(function (_ref3) {
          var rows = _ref3.rows,
              cols = _ref3.cols;
          return {
            rows: rows,
            cols: cols
          };
        });
        $table.dispatchEvent('cell-area-merge', {
          status: status,
          targetAreas: targetAreas
        }, $event);
      },

      /**
       * 如果启用 mouse-config.area 功能，临时合并区域范围内的单元格，不管是否存在已合并
       */
      MERGE_CELL: function MERGE_CELL(params) {
        var $event = params.$event,
            $table = params.$table;

        var _$table$getTableData3 = $table.getTableData(),
            visibleData = _$table$getTableData3.visibleData;

        var _$table$getTableColum3 = $table.getTableColumn(),
            visibleColumn = _$table$getTableColum3.visibleColumn;

        var cellAreas = $table.getCellAreas();
        handleClearMergeCells(params);

        if (cellAreas.some(function (_ref4) {
          var rows = _ref4.rows,
              cols = _ref4.cols;
          return rows.length === visibleData.length || cols.length === visibleColumn.length;
        })) {
          if (vxetable.modal) {
            vxetable.modal.message({
              content: vxetable.t('vxe.pro.area.mergeErr'),
              status: 'error',
              id: 'operErr'
            });
          }

          return;
        }

        $table.setMergeCells(cellAreas.map(function (_ref5) {
          var rows = _ref5.rows,
              cols = _ref5.cols;
          return {
            row: rows[0],
            col: cols[0],
            rowspan: rows.length,
            colspan: cols.length
          };
        }));
        var targetAreas = cellAreas.map(function (_ref6) {
          var rows = _ref6.rows,
              cols = _ref6.cols;
          return {
            rows: rows,
            cols: cols
          };
        });
        $table.dispatchEvent('cell-area-merge', {
          status: true,
          targetAreas: targetAreas
        }, $event);
      },

      /**
       * 如果启用 mouse-config.area 功能，清除区域范围内单元格的临时合并状态
       */
      CLEAR_MERGE_CELL: function CLEAR_MERGE_CELL(params) {
        var $event = params.$event,
            $table = params.$table;
        var beenMerges = handleClearMergeCells(params);

        if (beenMerges.length) {
          $table.dispatchEvent('clear-cell-area-merge', {
            mergeCells: beenMerges
          }, $event);
        }
      },

      /**
       * 清除所有单元格及表尾的临时合并状态
       */
      CLEAR_ALL_MERGE: function CLEAR_ALL_MERGE(params) {
        var $event = params.$event,
            $table = params.$table;
        var mergeCells = $table.getMergeCells();
        var mergeFooterItems = $table.getMergeFooterItems();
        $table.clearMergeCells();
        $table.clearMergeFooterItems();
        $table.dispatchEvent('clear-merge', {
          mergeCells: mergeCells,
          mergeFooterItems: mergeFooterItems
        }, $event);
      },

      /**
       * 编辑单元格
       */
      EDIT_CELL: function EDIT_CELL(params) {
        var $table = params.$table,
            row = params.row,
            column = params.column;
        $table.setActiveCell(row, column.property);
      },

      /**
       * 编辑行
       */
      EDIT_ROW: function EDIT_ROW(params) {
        var $table = params.$table,
            row = params.row;
        $table.setActiveRow(row);
      },

      /**
       * 插入数据
       */
      INSERT_ROW: function INSERT_ROW(params) {
        var $table = params.$table,
            menu = params.menu;
        $table.insert(menu.params);
      },

      /**
       * 插入数据并激活编辑状态
       */
      INSERT_ACTIVED_ROW: function INSERT_ACTIVED_ROW(params) {
        var $table = params.$table,
            menu = params.menu,
            column = params.column;
        var args = menu.params || []; // [{}, 'field']

        $table.insert(args[0]).then(function (_ref7) {
          var row = _ref7.row;
          return $table.setActiveCell(row, args[1] || column.property);
        });
      },

      /**
       * 插入数据到指定位置
       */
      INSERT_AT_ROW: function INSERT_AT_ROW(params) {
        var $table = params.$table,
            menu = params.menu,
            row = params.row;

        if (row) {
          $table.insertAt(menu.params, row);
        }
      },

      /**
       * 插入数据到指定位置并激活编辑状态
       */
      INSERT_AT_ACTIVED_ROW: function INSERT_AT_ACTIVED_ROW(params) {
        var $table = params.$table,
            menu = params.menu,
            row = params.row,
            column = params.column;

        if (row) {
          var args = menu.params || []; // [{}, 'field']

          $table.insertAt(args[0], row).then(function (_ref8) {
            var row = _ref8.row;
            return $table.setActiveCell(row, args[1] || column.property);
          });
        }
      },

      /**
       * 移除行数据
       */
      DELETE_ROW: function DELETE_ROW(params) {
        var $table = params.$table,
            row = params.row;

        if (row) {
          $table.remove(row);
        }
      },

      /**
       * 移除复选框选中行数据
       */
      DELETE_CHECKBOX_ROW: function DELETE_CHECKBOX_ROW(params) {
        var $table = params.$table;
        $table.removeCheckboxRow();
      },

      /**
       * 移除所有行数据
       */
      DELETE_ALL: function DELETE_ALL(params) {
        var $table = params.$table;
        $table.remove();
      },

      /**
       * 清除所选列排序条件
       */
      CLEAR_SORT: function CLEAR_SORT(params) {
        var $event = params.$event,
            $table = params.$table,
            column = params.column;

        if (column) {
          $table.triggerSortEvent($event, column, null);
        }
      },

      /**
       * 清除所有排序条件
       */
      CLEAR_ALL_SORT: function CLEAR_ALL_SORT(params) {
        var $event = params.$event,
            $table = params.$table;
        var sortList = $table.getSortColumns();

        if (sortList.length) {
          $table.clearSort();
          $table.dispatchEvent('clear-sort', {
            sortList: sortList
          }, $event);
        }
      },

      /**
       * 按所选列的值升序
       */
      SORT_ASC: function SORT_ASC(params) {
        var $event = params.$event,
            $table = params.$table,
            column = params.column;

        if (column) {
          $table.triggerSortEvent($event, column, 'asc');
        }
      },

      /**
       * 按所选列的值倒序
       */
      SORT_DESC: function SORT_DESC(params) {
        var $event = params.$event,
            $table = params.$table,
            column = params.column;

        if (column) {
          $table.triggerSortEvent($event, column, 'desc');
        }
      },

      /**
       * 清除复选框选中列的筛选条件
       */
      CLEAR_FILTER: function CLEAR_FILTER(params) {
        var $event = params.$event,
            $table = params.$table,
            column = params.column;

        if (column) {
          $table.handleClearFilter(column);
          $table.confirmFilterEvent($event);
        }
      },

      /**
       * 清除所有列筛选条件
       */
      CLEAR_ALL_FILTER: function CLEAR_ALL_FILTER(params) {
        var $event = params.$event,
            $table = params.$table;
        var filterList = $table.getCheckedFilters();

        if (filterList.length) {
          $table.clearFilter();
          $table.dispatchEvent('clear-filter', {
            filterList: filterList
          }, $event);
        }
      },

      /**
       * 根据单元格值筛选
       */
      FILTER_CELL: function FILTER_CELL(params) {
        var $table = params.$table,
            row = params.row,
            column = params.column;

        if (row && column) {
          var property = column.property,
              filters = column.filters;

          if (filters.length) {
            var option = filters[0];
            option.data = _xeUtils["default"].get(row, property);
            option.checked = true;
            $table.updateData();
          }
        }
      },

      /**
       * 导出行数据
       */
      EXPORT_ROW: function EXPORT_ROW(params) {
        var $table = params.$table,
            menu = params.menu,
            row = params.row;

        if (row) {
          var opts = {
            data: [row]
          };
          $table.exportData(_xeUtils["default"].assign(opts, menu.params[0]));
        }
      },

      /**
       * 导出复选框选中行数据
       */
      EXPORT_CHECKBOX_ROW: function EXPORT_CHECKBOX_ROW(params) {
        var $table = params.$table,
            menu = params.menu;
        var opts = {
          data: $table.getCheckboxRecords()
        };
        $table.exportData(_xeUtils["default"].assign(opts, menu.params[0]));
      },

      /**
       * 导出所有行数据
       */
      EXPORT_ALL: function EXPORT_ALL(params) {
        var $table = params.$table,
            menu = params.menu;
        $table.exportData(menu.params);
      },

      /**
       * 打印所有行数据
       */
      PRINT_ALL: function PRINT_ALL(params) {
        var $table = params.$table,
            menu = params.menu;
        $table.print(menu.params);
      },

      /**
       * 打印复选框选中行
       */
      PRINT_CHECKBOX_ROW: function PRINT_CHECKBOX_ROW(params) {
        var $table = params.$table,
            menu = params.menu;
        var opts = {
          data: $table.getCheckboxRecords()
        };
        $table.print(_xeUtils["default"].assign(opts, menu.params));
      },

      /**
       * 打开查找功能
       */
      OPEN_FIND: function OPEN_FIND(params) {
        var $event = params.$event,
            $table = params.$table;
        $table.triggerFNROpenEvent($event, 'find');
      },

      /**
       * 打开替换功能
       */
      OPEN_REPLACE: function OPEN_REPLACE(params) {
        var $event = params.$event,
            $table = params.$table;
        $table.triggerFNROpenEvent($event, 'replace');
      },

      /**
       * 隐藏当前列
       */
      HIDDEN_COLUMN: function HIDDEN_COLUMN(params) {
        var $table = params.$table,
            column = params.column;

        if (column) {
          $table.hideColumn(column);
        }
      },

      /**
       * 将列固定到左侧
       */
      FIXED_LEFT_COLUMN: handleFixedColumn('left'),

      /**
       * 将列固定到右侧
       */
      FIXED_RIGHT_COLUMN: handleFixedColumn('right'),

      /**
       * 清除固定列
       */
      CLEAR_FIXED_COLUMN: handleFixedColumn(null),

      /**
       * 重置列的可视状态
       */
      RESET_COLUMN: function RESET_COLUMN(params) {
        var $table = params.$table;
        $table.resetColumn({
          visible: true,
          resizable: false
        });
      },

      /**
       * 重置列宽状态
       */
      RESET_RESIZABLE: function RESET_RESIZABLE(params) {
        var $table = params.$table;
        $table.resetColumn({
          visible: false,
          resizable: true
        });
      },

      /**
       * 重置列的所有状态
       */
      RESET_ALL: function RESET_ALL(params) {
        var $table = params.$table;
        $table.resetColumn(true);
      }
    });
    interceptor.add('event.showMenu', handlePrivilegeEvent);
  }
};
exports.VXETablePluginMenus = VXETablePluginMenus;

if (typeof window !== 'undefined' && window.VXETable && window.VXETable.use) {
  window.VXETable.use(VXETablePluginMenus);
}

var _default = VXETablePluginMenus;
exports["default"] = _default;
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
