"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = exports.VXETablePluginRenderer = void 0;

var _xeUtils = _interopRequireDefault(require("xe-utils"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

var defaultColors = ['#2F4554', '#C23531', '#61A0A8', '#D48265', '#91C7AE', '#749F83', '#CA8622', '#006699', '#BDA29A', '#546570'];
var tmplOpts = {
  tmplRE: /\{([.\w[\]\s]+)\}/g
};

function getDefaultColor(index) {
  return defaultColors[index % 10];
}

function toRGBLight(color, level) {
  var rgbs = color.match(/(\d{1,3}),\s?(\d{1,3}),\s?(\d{1,3})(,\s?(\d{1,3}))?/);

  if (rgbs) {
    var r = parseInt(rgbs[1]);
    var g = parseInt(rgbs[2]);
    var b = parseInt(rgbs[3]);
    var alpha = parseInt(rgbs[5]);
    return "rgb(".concat(r + level, ",").concat(g + level, ",").concat(b + level).concat(alpha ? ",".concat(alpha) : '', ")");
  }

  return null;
}

function getStyleUnit(val) {
  return _xeUtils["default"].isNumber(val) ? "".concat(val, "px") : val;
}

function showTooltip(elem, params, formatter, value) {
  var row = params.row,
      column = params.column,
      $table = params.$table;
  var content = _xeUtils["default"].isString(formatter) ? _xeUtils["default"].template(formatter, {
    value: value,
    row: row,
    column: column
  }, tmplOpts) : null;
  $table.openTooltip(elem, content);
}

function hideTooltip(elem, params) {
  var $table = params.$table;

  if ($table) {
    $table.closeTooltip();
  }
}

function createBarVNs(h, params, renderOpts) {
  var row = params.row,
      column = params.column;
  var _renderOpts$props = renderOpts.props,
      props = _renderOpts$props === void 0 ? {} : _renderOpts$props;
  var margin = props.margin,
      _props$colors = props.colors,
      colors = _props$colors === void 0 ? [] : _props$colors,
      _props$bar = props.bar,
      bar = _props$bar === void 0 ? {} : _props$bar,
      _props$label = props.label,
      barLabel = _props$label === void 0 ? {} : _props$label,
      _props$tooltip = props.tooltip,
      tooltip = _props$tooltip === void 0 ? {} : _props$tooltip;
  var max = bar.max;
  var barHeight = getStyleUnit(bar.width);
  var cellValue = row[column.property];

  if (!_xeUtils["default"].isArray(cellValue)) {
    cellValue = [cellValue];
  }

  var numList = [];
  var maxVal = 0;
  cellValue.forEach(function (num) {
    num = _xeUtils["default"].toNumber(num);
    maxVal = Math.max(maxVal, num);
    numList.push(num);
  });
  var ratio = Math.max(_xeUtils["default"].toNumber(max), maxVal) / 100;
  var barList = numList.map(function (num) {
    return parseInt("".concat(num / ratio));
  });
  return barList.map(function (barValue, index) {
    var labelPosition;

    if (barValue < 30) {
      labelPosition = 'outer';
    } else if (barValue > 70) {
      labelPosition = 'inner';
    }

    return h('span', {
      "class": ['vxe-renderer-bar', _defineProperty({}, "label--".concat(labelPosition), labelPosition)],
      style: {
        margin: getStyleUnit(margin),
        height: barHeight,
        lineHeight: barHeight
      }
    }, [h('span', {
      "class": 'vxe-renderer-bar--chart',
      style: {
        width: "".concat(barValue, "%"),
        backgroundColor: colors[index] || getDefaultColor(index)
      },
      on: {
        mouseenter: function mouseenter(evnt) {
          var elem = evnt.currentTarget;
          var hoverColor = toRGBLight(elem.style.backgroundColor, 10);

          if (hoverColor) {
            elem.style.backgroundColor = hoverColor;
          }

          if (tooltip.formatter) {
            showTooltip(elem, params, tooltip.formatter, numList[index]);
          }
        },
        mouseleave: function mouseleave(evnt) {
          var elem = evnt.currentTarget;
          var reColor = colors[index] || getDefaultColor(index);
          elem.style.backgroundColor = reColor;
          hideTooltip(elem, params);
        }
      }
    }), h('span', {
      "class": 'vxe-renderer-bar--label',
      style: {
        color: barLabel.color
      }
    }, _xeUtils["default"].isString(barLabel.formatter) ? _xeUtils["default"].template(barLabel.formatter, {
      value: numList[index],
      row: row,
      column: column
    }, tmplOpts) : null)]);
  });
}

function parsePieAreas(blockList, total) {
  var prves = [];
  var nexts = [];
  var countDeg = 0;
  var ratio = total / 360;
  blockList.forEach(function (item) {
    item.deg = countDeg;
    countDeg += parseInt("".concat(item.val / ratio));

    if (countDeg > 180 && item.deg <= 180) {
      var repairItem = Object.assign({}, item);
      prves.push(repairItem);
      item.deg = 180;
    }

    if (countDeg > 180) {
      nexts.push(item);
    } else {
      prves.push(item);
    }
  });
  return {
    prves: prves,
    nexts: nexts
  };
}

function createPieVNs(h, params, renderOptList, cellValue) {
  if (!_xeUtils["default"].isArray(cellValue)) {
    cellValue = [cellValue];
  }

  return renderOptList.map(function (renderOpts, renderIndex) {
    var row = params.row,
        column = params.column;
    var _renderOpts$props2 = renderOpts.props,
        props = _renderOpts$props2 === void 0 ? {} : _renderOpts$props2;
    var margin = props.margin,
        _props$colors2 = props.colors,
        colors = _props$colors2 === void 0 ? [] : _props$colors2,
        _props$ring = props.ring,
        ring = _props$ring === void 0 ? {} : _props$ring,
        _props$label2 = props.label,
        ringLabel = _props$label2 === void 0 ? {} : _props$label2,
        _props$tooltip2 = props.tooltip,
        tooltip = _props$tooltip2 === void 0 ? {} : _props$tooltip2;
    var pieVals = cellValue[renderIndex];
    var pieDiameter = getStyleUnit(props.diameter);
    var ringDiameter = getStyleUnit(ring.diameter);
    var blockList = [];
    var countVal = 0;

    if (!_xeUtils["default"].isArray(pieVals)) {
      pieVals = [pieVals];
    }

    pieVals.forEach(function (val, index) {
      val = _xeUtils["default"].toNumber(val);
      countVal += val;
      blockList.push({
        val: val,
        deg: 0,
        index: index
      });
    });

    var _parsePieAreas = parsePieAreas(blockList, countVal),
        prveList = _parsePieAreas.prves,
        nextList = _parsePieAreas.nexts;

    var blockOns = {
      mouseenter: function mouseenter(evnt) {
        var elem = evnt.currentTarget;

        var index = _xeUtils["default"].toNumber(elem.getAttribute('block'));

        var hoverColor = toRGBLight(elem.style.backgroundColor, 10);

        if (hoverColor) {
          _xeUtils["default"].arrayEach(elem.parentNode.parentNode.querySelectorAll(".block-".concat(index)), function (elem) {
            elem.style.backgroundColor = hoverColor;
          });
        }

        if (tooltip.formatter) {
          showTooltip(elem, params, tooltip.formatter, blockList[index].val);
        }
      },
      mouseleave: function mouseleave(evnt) {
        var elem = evnt.currentTarget;

        var index = _xeUtils["default"].toNumber(elem.getAttribute('block'));

        var reColor = colors[index] || getDefaultColor(index);

        _xeUtils["default"].arrayEach(elem.parentNode.parentNode.querySelectorAll(".block-".concat(index)), function (elem) {
          elem.style.backgroundColor = reColor;
        });

        hideTooltip(elem, params);
      }
    };
    var pieVNs = [h('span', {
      "class": 'vxe-renderer-pie--next-half'
    }, nextList.map(function (item) {
      return h('span', {
        "class": ['vxe-renderer-pie--block', "block-".concat(item.index)],
        style: {
          backgroundColor: colors[item.index] || getDefaultColor(item.index),
          transform: "rotate(".concat(item.deg - 180, "deg)")
        },
        attrs: {
          block: item.index
        },
        on: blockOns
      });
    })), h('span', {
      "class": 'vxe-renderer-pie--prve-half'
    }, prveList.map(function (item) {
      return h('span', {
        "class": ['vxe-renderer-pie--block', "block-".concat(item.index)],
        style: {
          backgroundColor: colors[item.index] || getDefaultColor(item.index),
          transform: "rotate(".concat(item.deg, "deg)")
        },
        attrs: {
          block: item.index
        },
        on: blockOns
      });
    }))];

    if (ringDiameter) {
      pieVNs.push(h('span', {
        "class": 'vxe-renderer-pie--ring-bg',
        style: {
          width: ringDiameter,
          height: ringDiameter,
          backgroundColor: ring.color
        }
      }), h('span', {
        "class": 'vxe-renderer-pie--ring-label',
        style: {
          color: ringLabel.color
        }
      }, _xeUtils["default"].isString(ringLabel.formatter) ? _xeUtils["default"].template(ringLabel.formatter, {
        value: row[column.property] || [],
        row: row,
        column: column
      }, tmplOpts) : null));
    }

    return h('span', {
      "class": 'vxe-renderer-pie',
      style: {
        margin: getStyleUnit(margin),
        width: pieDiameter,
        height: pieDiameter
      }
    }, pieVNs);
  });
}
/**
 * 基于 vxe-table 表格的增强插件，提供一些常用的渲染器
 */


var VXETablePluginRenderer = {
  install: function install(xtable) {
    xtable.renderer.mixin({
      bar: {
        renderDefault: function renderDefault(h, renderOpts, params) {
          return createBarVNs(h, params, renderOpts);
        }
      },
      pie: {
        renderDefault: function renderDefault(h, renderOpts, params) {
          var row = params.row,
              column = params.column;
          var cellValue = row[column.property];
          return createPieVNs(h, params, [renderOpts], cellValue ? [cellValue] : []);
        }
      },
      pies: {
        renderDefault: function renderDefault(h, renderOpts, params) {
          var row = params.row,
              column = params.column;
          var cellValue = row[column.property];
          return createPieVNs(h, params, renderOpts.children || [], cellValue);
        }
      },
      rate: {
        renderDefault: function renderDefault(h, renderOpts, params) {
          var row = params.row,
              column = params.column;
          var _renderOpts$props3 = renderOpts.props,
              props = _renderOpts$props3 === void 0 ? {} : _renderOpts$props3;
          var _props$colors3 = props.colors,
              colors = _props$colors3 === void 0 ? [] : _props$colors3;

          var cellValue = _xeUtils["default"].toNumber(row[column.property]);

          var rateVNs = [];
          var lastColor;

          _xeUtils["default"].range(0, _xeUtils["default"].toNumber(props.count) || 5).forEach(function (obj, index) {
            var itemIndex = index + 1;
            var isActive = cellValue >= itemIndex;
            var itemColor;

            if (isActive) {
              if (colors[itemIndex]) {
                lastColor = colors[itemIndex];
              }

              itemColor = lastColor || '#F7BA2A';
            } else {
              itemColor = colors[0] || '#E9E9E9';
            }

            var itemOns = {
              mouseenter: function mouseenter(evnt) {
                var elem = evnt.currentTarget;
                var hoverColor = toRGBLight(elem.style.color, 10);

                if (hoverColor) {
                  elem.style.color = hoverColor;
                }
              },
              mouseleave: function mouseleave(evnt) {
                var elem = evnt.currentTarget;
                elem.style.color = itemColor;
              }
            };
            rateVNs.push(h('span', {
              "class": 'vxe-renderer-rate-item',
              style: {
                color: itemColor
              },
              on: isActive ? itemOns : {}
            }));
          });

          return [h('div', {
            "class": 'vxe-renderer-rate'
          }, rateVNs)];
        }
      }
    });
  }
};
exports.VXETablePluginRenderer = VXETablePluginRenderer;

if (typeof window !== 'undefined' && window.VXETable && window.VXETable.use) {
  window.VXETable.use(VXETablePluginRenderer);
}

var _default = VXETablePluginRenderer;
exports["default"] = _default;
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
