/*!-----------------------------------------------------------------------------
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Version: 0.44.0(3e047efd345ff102c8c61b5398fb30845aaac166)
 * Released under the MIT license
 * https://github.com/microsoft/monaco-editor/blob/main/LICENSE.txt
 *-----------------------------------------------------------------------------*/

// src/language/css/css.worker.ts
import * as worker from "../../editor/editor.worker.js";

// node_modules/vscode-css-languageservice/lib/esm/parser/cssScanner.js
var TokenType;
(function(TokenType2) {
  TokenType2[TokenType2["Ident"] = 0] = "Ident";
  TokenType2[TokenType2["AtKeyword"] = 1] = "AtKeyword";
  TokenType2[TokenType2["String"] = 2] = "String";
  TokenType2[TokenType2["BadString"] = 3] = "BadString";
  TokenType2[TokenType2["UnquotedString"] = 4] = "UnquotedString";
  TokenType2[TokenType2["Hash"] = 5] = "Hash";
  TokenType2[TokenType2["Num"] = 6] = "Num";
  TokenType2[TokenType2["Percentage"] = 7] = "Percentage";
  TokenType2[TokenType2["Dimension"] = 8] = "Dimension";
  TokenType2[TokenType2["UnicodeRange"] = 9] = "UnicodeRange";
  TokenType2[TokenType2["CDO"] = 10] = "CDO";
  TokenType2[TokenType2["CDC"] = 11] = "CDC";
  TokenType2[TokenType2["Colon"] = 12] = "Colon";
  TokenType2[TokenType2["SemiColon"] = 13] = "SemiColon";
  TokenType2[TokenType2["CurlyL"] = 14] = "CurlyL";
  TokenType2[TokenType2["CurlyR"] = 15] = "CurlyR";
  TokenType2[TokenType2["ParenthesisL"] = 16] = "ParenthesisL";
  TokenType2[TokenType2["ParenthesisR"] = 17] = "ParenthesisR";
  TokenType2[TokenType2["BracketL"] = 18] = "BracketL";
  TokenType2[TokenType2["BracketR"] = 19] = "BracketR";
  TokenType2[TokenType2["Whitespace"] = 20] = "Whitespace";
  TokenType2[TokenType2["Includes"] = 21] = "Includes";
  TokenType2[TokenType2["Dashmatch"] = 22] = "Dashmatch";
  TokenType2[TokenType2["SubstringOperator"] = 23] = "SubstringOperator";
  TokenType2[TokenType2["PrefixOperator"] = 24] = "PrefixOperator";
  TokenType2[TokenType2["SuffixOperator"] = 25] = "SuffixOperator";
  TokenType2[TokenType2["Delim"] = 26] = "Delim";
  TokenType2[TokenType2["EMS"] = 27] = "EMS";
  TokenType2[TokenType2["EXS"] = 28] = "EXS";
  TokenType2[TokenType2["Length"] = 29] = "Length";
  TokenType2[TokenType2["Angle"] = 30] = "Angle";
  TokenType2[TokenType2["Time"] = 31] = "Time";
  TokenType2[TokenType2["Freq"] = 32] = "Freq";
  TokenType2[TokenType2["Exclamation"] = 33] = "Exclamation";
  TokenType2[TokenType2["Resolution"] = 34] = "Resolution";
  TokenType2[TokenType2["Comma"] = 35] = "Comma";
  TokenType2[TokenType2["Charset"] = 36] = "Charset";
  TokenType2[TokenType2["EscapedJavaScript"] = 37] = "EscapedJavaScript";
  TokenType2[TokenType2["BadEscapedJavaScript"] = 38] = "BadEscapedJavaScript";
  TokenType2[TokenType2["Comment"] = 39] = "Comment";
  TokenType2[TokenType2["SingleLineComment"] = 40] = "SingleLineComment";
  TokenType2[TokenType2["EOF"] = 41] = "EOF";
  TokenType2[TokenType2["CustomToken"] = 42] = "CustomToken";
})(TokenType || (TokenType = {}));
var MultiLineStream = function() {
  function MultiLineStream2(source) {
    this.source = source;
    this.len = source.length;
    this.position = 0;
  }
  MultiLineStream2.prototype.substring = function(from, to) {
    if (to === void 0) {
      to = this.position;
    }
    return this.source.substring(from, to);
  };
  MultiLineStream2.prototype.eos = function() {
    return this.len <= this.position;
  };
  MultiLineStream2.prototype.pos = function() {
    return this.position;
  };
  MultiLineStream2.prototype.goBackTo = function(pos) {
    this.position = pos;
  };
  MultiLineStream2.prototype.goBack = function(n) {
    this.position -= n;
  };
  MultiLineStream2.prototype.advance = function(n) {
    this.position += n;
  };
  MultiLineStream2.prototype.nextChar = function() {
    return this.source.charCodeAt(this.position++) || 0;
  };
  MultiLineStream2.prototype.peekChar = function(n) {
    if (n === void 0) {
      n = 0;
    }
    return this.source.charCodeAt(this.position + n) || 0;
  };
  MultiLineStream2.prototype.lookbackChar = function(n) {
    if (n === void 0) {
      n = 0;
    }
    return this.source.charCodeAt(this.position - n) || 0;
  };
  MultiLineStream2.prototype.advanceIfChar = function(ch) {
    if (ch === this.source.charCodeAt(this.position)) {
      this.position++;
      return true;
    }
    return false;
  };
  MultiLineStream2.prototype.advanceIfChars = function(ch) {
    if (this.position + ch.length > this.source.length) {
      return false;
    }
    var i = 0;
    for (; i < ch.length; i++) {
      if (this.source.charCodeAt(this.position + i) !== ch[i]) {
        return false;
      }
    }
    this.advance(i);
    return true;
  };
  MultiLineStream2.prototype.advanceWhileChar = function(condition) {
    var posNow = this.position;
    while (this.position < this.len && condition(this.source.charCodeAt(this.position))) {
      this.position++;
    }
    return this.position - posNow;
  };
  return MultiLineStream2;
}();
var _a = "a".charCodeAt(0);
var _f = "f".charCodeAt(0);
var _z = "z".charCodeAt(0);
var _u = "u".charCodeAt(0);
var _A = "A".charCodeAt(0);
var _F = "F".charCodeAt(0);
var _Z = "Z".charCodeAt(0);
var _0 = "0".charCodeAt(0);
var _9 = "9".charCodeAt(0);
var _TLD = "~".charCodeAt(0);
var _HAT = "^".charCodeAt(0);
var _EQS = "=".charCodeAt(0);
var _PIP = "|".charCodeAt(0);
var _MIN = "-".charCodeAt(0);
var _USC = "_".charCodeAt(0);
var _PRC = "%".charCodeAt(0);
var _MUL = "*".charCodeAt(0);
var _LPA = "(".charCodeAt(0);
var _RPA = ")".charCodeAt(0);
var _LAN = "<".charCodeAt(0);
var _RAN = ">".charCodeAt(0);
var _ATS = "@".charCodeAt(0);
var _HSH = "#".charCodeAt(0);
var _DLR = "$".charCodeAt(0);
var _BSL = "\\".charCodeAt(0);
var _FSL = "/".charCodeAt(0);
var _NWL = "\n".charCodeAt(0);
var _CAR = "\r".charCodeAt(0);
var _LFD = "\f".charCodeAt(0);
var _DQO = '"'.charCodeAt(0);
var _SQO = "'".charCodeAt(0);
var _WSP = " ".charCodeAt(0);
var _TAB = "	".charCodeAt(0);
var _SEM = ";".charCodeAt(0);
var _COL = ":".charCodeAt(0);
var _CUL = "{".charCodeAt(0);
var _CUR = "}".charCodeAt(0);
var _BRL = "[".charCodeAt(0);
var _BRR = "]".charCodeAt(0);
var _CMA = ",".charCodeAt(0);
var _DOT = ".".charCodeAt(0);
var _BNG = "!".charCodeAt(0);
var _QSM = "?".charCodeAt(0);
var _PLS = "+".charCodeAt(0);
var staticTokenTable = {};
staticTokenTable[_SEM] = TokenType.SemiColon;
staticTokenTable[_COL] = TokenType.Colon;
staticTokenTable[_CUL] = TokenType.CurlyL;
staticTokenTable[_CUR] = TokenType.CurlyR;
staticTokenTable[_BRR] = TokenType.BracketR;
staticTokenTable[_BRL] = TokenType.BracketL;
staticTokenTable[_LPA] = TokenType.ParenthesisL;
staticTokenTable[_RPA] = TokenType.ParenthesisR;
staticTokenTable[_CMA] = TokenType.Comma;
var staticUnitTable = {};
staticUnitTable["em"] = TokenType.EMS;
staticUnitTable["ex"] = TokenType.EXS;
staticUnitTable["px"] = TokenType.Length;
staticUnitTable["cm"] = TokenType.Length;
staticUnitTable["mm"] = TokenType.Length;
staticUnitTable["in"] = TokenType.Length;
staticUnitTable["pt"] = TokenType.Length;
staticUnitTable["pc"] = TokenType.Length;
staticUnitTable["deg"] = TokenType.Angle;
staticUnitTable["rad"] = TokenType.Angle;
staticUnitTable["grad"] = TokenType.Angle;
staticUnitTable["ms"] = TokenType.Time;
staticUnitTable["s"] = TokenType.Time;
staticUnitTable["hz"] = TokenType.Freq;
staticUnitTable["khz"] = TokenType.Freq;
staticUnitTable["%"] = TokenType.Percentage;
staticUnitTable["fr"] = TokenType.Percentage;
staticUnitTable["dpi"] = TokenType.Resolution;
staticUnitTable["dpcm"] = TokenType.Resolution;
var Scanner = function() {
  function Scanner2() {
    this.stream = new MultiLineStream("");
    this.ignoreComment = true;
    this.ignoreWhitespace = true;
    this.inURL = false;
  }
  Scanner2.prototype.setSource = function(input) {
    this.stream = new MultiLineStream(input);
  };
  Scanner2.prototype.finishToken = function(offset, type, text) {
    return {
      offset,
      len: this.stream.pos() - offset,
      type,
      text: text || this.stream.substring(offset)
    };
  };
  Scanner2.prototype.substring = function(offset, len) {
    return this.stream.substring(offset, offset + len);
  };
  Scanner2.prototype.pos = function() {
    return this.stream.pos();
  };
  Scanner2.prototype.goBackTo = function(pos) {
    this.stream.goBackTo(pos);
  };
  Scanner2.prototype.scanUnquotedString = function() {
    var offset = this.stream.pos();
    var content = [];
    if (this._unquotedString(content)) {
      return this.finishToken(offset, TokenType.UnquotedString, content.join(""));
    }
    return null;
  };
  Scanner2.prototype.scan = function() {
    var triviaToken = this.trivia();
    if (triviaToken !== null) {
      return triviaToken;
    }
    var offset = this.stream.pos();
    if (this.stream.eos()) {
      return this.finishToken(offset, TokenType.EOF);
    }
    return this.scanNext(offset);
  };
  Scanner2.prototype.tryScanUnicode = function() {
    var offset = this.stream.pos();
    if (!this.stream.eos() && this._unicodeRange()) {
      return this.finishToken(offset, TokenType.UnicodeRange);
    }
    this.stream.goBackTo(offset);
    return void 0;
  };
  Scanner2.prototype.scanNext = function(offset) {
    if (this.stream.advanceIfChars([_LAN, _BNG, _MIN, _MIN])) {
      return this.finishToken(offset, TokenType.CDO);
    }
    if (this.stream.advanceIfChars([_MIN, _MIN, _RAN])) {
      return this.finishToken(offset, TokenType.CDC);
    }
    var content = [];
    if (this.ident(content)) {
      return this.finishToken(offset, TokenType.Ident, content.join(""));
    }
    if (this.stream.advanceIfChar(_ATS)) {
      content = ["@"];
      if (this._name(content)) {
        var keywordText = content.join("");
        if (keywordText === "@charset") {
          return this.finishToken(offset, TokenType.Charset, keywordText);
        }
        return this.finishToken(offset, TokenType.AtKeyword, keywordText);
      } else {
        return this.finishToken(offset, TokenType.Delim);
      }
    }
    if (this.stream.advanceIfChar(_HSH)) {
      content = ["#"];
      if (this._name(content)) {
        return this.finishToken(offset, TokenType.Hash, content.join(""));
      } else {
        return this.finishToken(offset, TokenType.Delim);
      }
    }
    if (this.stream.advanceIfChar(_BNG)) {
      return this.finishToken(offset, TokenType.Exclamation);
    }
    if (this._number()) {
      var pos = this.stream.pos();
      content = [this.stream.substring(offset, pos)];
      if (this.stream.advanceIfChar(_PRC)) {
        return this.finishToken(offset, TokenType.Percentage);
      } else if (this.ident(content)) {
        var dim = this.stream.substring(pos).toLowerCase();
        var tokenType_1 = staticUnitTable[dim];
        if (typeof tokenType_1 !== "undefined") {
          return this.finishToken(offset, tokenType_1, content.join(""));
        } else {
          return this.finishToken(offset, TokenType.Dimension, content.join(""));
        }
      }
      return this.finishToken(offset, TokenType.Num);
    }
    content = [];
    var tokenType = this._string(content);
    if (tokenType !== null) {
      return this.finishToken(offset, tokenType, content.join(""));
    }
    tokenType = staticTokenTable[this.stream.peekChar()];
    if (typeof tokenType !== "undefined") {
      this.stream.advance(1);
      return this.finishToken(offset, tokenType);
    }
    if (this.stream.peekChar(0) === _TLD && this.stream.peekChar(1) === _EQS) {
      this.stream.advance(2);
      return this.finishToken(offset, TokenType.Includes);
    }
    if (this.stream.peekChar(0) === _PIP && this.stream.peekChar(1) === _EQS) {
      this.stream.advance(2);
      return this.finishToken(offset, TokenType.Dashmatch);
    }
    if (this.stream.peekChar(0) === _MUL && this.stream.peekChar(1) === _EQS) {
      this.stream.advance(2);
      return this.finishToken(offset, TokenType.SubstringOperator);
    }
    if (this.stream.peekChar(0) === _HAT && this.stream.peekChar(1) === _EQS) {
      this.stream.advance(2);
      return this.finishToken(offset, TokenType.PrefixOperator);
    }
    if (this.stream.peekChar(0) === _DLR && this.stream.peekChar(1) === _EQS) {
      this.stream.advance(2);
      return this.finishToken(offset, TokenType.SuffixOperator);
    }
    this.stream.nextChar();
    return this.finishToken(offset, TokenType.Delim);
  };
  Scanner2.prototype.trivia = function() {
    while (true) {
      var offset = this.stream.pos();
      if (this._whitespace()) {
        if (!this.ignoreWhitespace) {
          return this.finishToken(offset, TokenType.Whitespace);
        }
      } else if (this.comment()) {
        if (!this.ignoreComment) {
          return this.finishToken(offset, TokenType.Comment);
        }
      } else {
        return null;
      }
    }
  };
  Scanner2.prototype.comment = function() {
    if (this.stream.advanceIfChars([_FSL, _MUL])) {
      var success_1 = false, hot_1 = false;
      this.stream.advanceWhileChar(function(ch) {
        if (hot_1 && ch === _FSL) {
          success_1 = true;
          return false;
        }
        hot_1 = ch === _MUL;
        return true;
      });
      if (success_1) {
        this.stream.advance(1);
      }
      return true;
    }
    return false;
  };
  Scanner2.prototype._number = function() {
    var npeek = 0, ch;
    if (this.stream.peekChar() === _DOT) {
      npeek = 1;
    }
    ch = this.stream.peekChar(npeek);
    if (ch >= _0 && ch <= _9) {
      this.stream.advance(npeek + 1);
      this.stream.advanceWhileChar(function(ch2) {
        return ch2 >= _0 && ch2 <= _9 || npeek === 0 && ch2 === _DOT;
      });
      return true;
    }
    return false;
  };
  Scanner2.prototype._newline = function(result) {
    var ch = this.stream.peekChar();
    switch (ch) {
      case _CAR:
      case _LFD:
      case _NWL:
        this.stream.advance(1);
        result.push(String.fromCharCode(ch));
        if (ch === _CAR && this.stream.advanceIfChar(_NWL)) {
          result.push("\n");
        }
        return true;
    }
    return false;
  };
  Scanner2.prototype._escape = function(result, includeNewLines) {
    var ch = this.stream.peekChar();
    if (ch === _BSL) {
      this.stream.advance(1);
      ch = this.stream.peekChar();
      var hexNumCount = 0;
      while (hexNumCount < 6 && (ch >= _0 && ch <= _9 || ch >= _a && ch <= _f || ch >= _A && ch <= _F)) {
        this.stream.advance(1);
        ch = this.stream.peekChar();
        hexNumCount++;
      }
      if (hexNumCount > 0) {
        try {
          var hexVal = parseInt(this.stream.substring(this.stream.pos() - hexNumCount), 16);
          if (hexVal) {
            result.push(String.fromCharCode(hexVal));
          }
        } catch (e) {
        }
        if (ch === _WSP || ch === _TAB) {
          this.stream.advance(1);
        } else {
          this._newline([]);
        }
        return true;
      }
      if (ch !== _CAR && ch !== _LFD && ch !== _NWL) {
        this.stream.advance(1);
        result.push(String.fromCharCode(ch));
        return true;
      } else if (includeNewLines) {
        return this._newline(result);
      }
    }
    return false;
  };
  Scanner2.prototype._stringChar = function(closeQuote, result) {
    var ch = this.stream.peekChar();
    if (ch !== 0 && ch !== closeQuote && ch !== _BSL && ch !== _CAR && ch !== _LFD && ch !== _NWL) {
      this.stream.advance(1);
      result.push(String.fromCharCode(ch));
      return true;
    }
    return false;
  };
  Scanner2.prototype._string = function(result) {
    if (this.stream.peekChar() === _SQO || this.stream.peekChar() === _DQO) {
      var closeQuote = this.stream.nextChar();
      result.push(String.fromCharCode(closeQuote));
      while (this._stringChar(closeQuote, result) || this._escape(result, true)) {
      }
      if (this.stream.peekChar() === closeQuote) {
        this.stream.nextChar();
        result.push(String.fromCharCode(closeQuote));
        return TokenType.String;
      } else {
        return TokenType.BadString;
      }
    }
    return null;
  };
  Scanner2.prototype._unquotedChar = function(result) {
    var ch = this.stream.peekChar();
    if (ch !== 0 && ch !== _BSL && ch !== _SQO && ch !== _DQO && ch !== _LPA && ch !== _RPA && ch !== _WSP && ch !== _TAB && ch !== _NWL && ch !== _LFD && ch !== _CAR) {
      this.stream.advance(1);
      result.push(String.fromCharCode(ch));
      return true;
    }
    return false;
  };
  Scanner2.prototype._unquotedString = function(result) {
    var hasContent = false;
    while (this._unquotedChar(result) || this._escape(result)) {
      hasContent = true;
    }
    return hasContent;
  };
  Scanner2.prototype._whitespace = function() {
    var n = this.stream.advanceWhileChar(function(ch) {
      return ch === _WSP || ch === _TAB || ch === _NWL || ch === _LFD || ch === _CAR;
    });
    return n > 0;
  };
  Scanner2.prototype._name = function(result) {
    var matched = false;
    while (this._identChar(result) || this._escape(result)) {
      matched = true;
    }
    return matched;
  };
  Scanner2.prototype.ident = function(result) {
    var pos = this.stream.pos();
    var hasMinus = this._minus(result);
    if (hasMinus) {
      if (this._minus(result) || this._identFirstChar(result) || this._escape(result)) {
        while (this._identChar(result) || this._escape(result)) {
        }
        return true;
      }
    } else if (this._identFirstChar(result) || this._escape(result)) {
      while (this._identChar(result) || this._escape(result)) {
      }
      return true;
    }
    this.stream.goBackTo(pos);
    return false;
  };
  Scanner2.prototype._identFirstChar = function(result) {
    var ch = this.stream.peekChar();
    if (ch === _USC || ch >= _a && ch <= _z || ch >= _A && ch <= _Z || ch >= 128 && ch <= 65535) {
      this.stream.advance(1);
      result.push(String.fromCharCode(ch));
      return true;
    }
    return false;
  };
  Scanner2.prototype._minus = function(result) {
    var ch = this.stream.peekChar();
    if (ch === _MIN) {
      this.stream.advance(1);
      result.push(String.fromCharCode(ch));
      return true;
    }
    return false;
  };
  Scanner2.prototype._identChar = function(result) {
    var ch = this.stream.peekChar();
    if (ch === _USC || ch === _MIN || ch >= _a && ch <= _z || ch >= _A && ch <= _Z || ch >= _0 && ch <= _9 || ch >= 128 && ch <= 65535) {
      this.stream.advance(1);
      result.push(String.fromCharCode(ch));
      return true;
    }
    return false;
  };
  Scanner2.prototype._unicodeRange = function() {
    if (this.stream.advanceIfChar(_PLS)) {
      var isHexDigit = function(ch) {
        return ch >= _0 && ch <= _9 || ch >= _a && ch <= _f || ch >= _A && ch <= _F;
      };
      var codePoints = this.stream.advanceWhileChar(isHexDigit) + this.stream.advanceWhileChar(function(ch) {
        return ch === _QSM;
      });
      if (codePoints >= 1 && codePoints <= 6) {
        if (this.stream.advanceIfChar(_MIN)) {
          var digits = this.stream.advanceWhileChar(isHexDigit);
          if (digits >= 1 && digits <= 6) {
            return true;
          }
        } else {
          return true;
        }
      }
    }
    return false;
  };
  return Scanner2;
}();

// node_modules/vscode-css-languageservice/lib/esm/utils/strings.js
function startsWith(haystack, needle) {
  if (haystack.length < needle.length) {
    return false;
  }
  for (var i = 0; i < needle.length; i++) {
    if (haystack[i] !== needle[i]) {
      return false;
    }
  }
  return true;
}
function endsWith(haystack, needle) {
  var diff = haystack.length - needle.length;
  if (diff > 0) {
    return haystack.lastIndexOf(needle) === diff;
  } else if (diff === 0) {
    return haystack === needle;
  } else {
    return false;
  }
}
function difference(first, second, maxLenDelta) {
  if (maxLenDelta === void 0) {
    maxLenDelta = 4;
  }
  var lengthDifference = Math.abs(first.length - second.length);
  if (lengthDifference > maxLenDelta) {
    return 0;
  }
  var LCS = [];
  var zeroArray = [];
  var i, j;
  for (i = 0; i < second.length + 1; ++i) {
    zeroArray.push(0);
  }
  for (i = 0; i < first.length + 1; ++i) {
    LCS.push(zeroArray);
  }
  for (i = 1; i < first.length + 1; ++i) {
    for (j = 1; j < second.length + 1; ++j) {
      if (first[i - 1] === second[j - 1]) {
        LCS[i][j] = LCS[i - 1][j - 1] + 1;
      } else {
        LCS[i][j] = Math.max(LCS[i - 1][j], LCS[i][j - 1]);
      }
    }
  }
  return LCS[first.length][second.length] - Math.sqrt(lengthDifference);
}
function getLimitedString(str, ellipsis) {
  if (ellipsis === void 0) {
    ellipsis = true;
  }
  if (!str) {
    return "";
  }
  if (str.length < 140) {
    return str;
  }
  return str.slice(0, 140) + (ellipsis ? "\u2026" : "");
}
function trim(str, regexp) {
  var m = regexp.exec(str);
  if (m && m[0].length) {
    return str.substr(0, str.length - m[0].length);
  }
  return str;
}
function repeat(value, count) {
  var s = "";
  while (count > 0) {
    if ((count & 1) === 1) {
      s += value;
    }
    value += value;
    count = count >>> 1;
  }
  return s;
}

// node_modules/vscode-css-languageservice/lib/esm/parser/cssNodes.js
var __extends = function() {
  var extendStatics = function(d, b) {
    extendStatics = Object.setPrototypeOf || { __proto__: [] } instanceof Array && function(d2, b2) {
      d2.__proto__ = b2;
    } || function(d2, b2) {
      for (var p in b2)
        if (Object.prototype.hasOwnProperty.call(b2, p))
          d2[p] = b2[p];
    };
    return extendStatics(d, b);
  };
  return function(d, b) {
    if (typeof b !== "function" && b !== null)
      throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
    extendStatics(d, b);
    function __() {
      this.constructor = d;
    }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
  };
}();
var NodeType;
(function(NodeType2) {
  NodeType2[NodeType2["Undefined"] = 0] = "Undefined";
  NodeType2[NodeType2["Identifier"] = 1] = "Identifier";
  NodeType2[NodeType2["Stylesheet"] = 2] = "Stylesheet";
  NodeType2[NodeType2["Ruleset"] = 3] = "Ruleset";
  NodeType2[NodeType2["Selector"] = 4] = "Selector";
  NodeType2[NodeType2["SimpleSelector"] = 5] = "SimpleSelector";
  NodeType2[NodeType2["SelectorInterpolation"] = 6] = "SelectorInterpolation";
  NodeType2[NodeType2["SelectorCombinator"] = 7] = "SelectorCombinator";
  NodeType2[NodeType2["SelectorCombinatorParent"] = 8] = "SelectorCombinatorParent";
  NodeType2[NodeType2["SelectorCombinatorSibling"] = 9] = "SelectorCombinatorSibling";
  NodeType2[NodeType2["SelectorCombinatorAllSiblings"] = 10] = "SelectorCombinatorAllSiblings";
  NodeType2[NodeType2["SelectorCombinatorShadowPiercingDescendant"] = 11] = "SelectorCombinatorShadowPiercingDescendant";
  NodeType2[NodeType2["Page"] = 12] = "Page";
  NodeType2[NodeType2["PageBoxMarginBox"] = 13] = "PageBoxMarginBox";
  NodeType2[NodeType2["ClassSelector"] = 14] = "ClassSelector";
  NodeType2[NodeType2["IdentifierSelector"] = 15] = "IdentifierSelector";
  NodeType2[NodeType2["ElementNameSelector"] = 16] = "ElementNameSelector";
  NodeType2[NodeType2["PseudoSelector"] = 17] = "PseudoSelector";
  NodeType2[NodeType2["AttributeSelector"] = 18] = "AttributeSelector";
  NodeType2[NodeType2["Declaration"] = 19] = "Declaration";
  NodeType2[NodeType2["Declarations"] = 20] = "Declarations";
  NodeType2[NodeType2["Property"] = 21] = "Property";
  NodeType2[NodeType2["Expression"] = 22] = "Expression";
  NodeType2[NodeType2["BinaryExpression"] = 23] = "BinaryExpression";
  NodeType2[NodeType2["Term"] = 24] = "Term";
  NodeType2[NodeType2["Operator"] = 25] = "Operator";
  NodeType2[NodeType2["Value"] = 26] = "Value";
  NodeType2[NodeType2["StringLiteral"] = 27] = "StringLiteral";
  NodeType2[NodeType2["URILiteral"] = 28] = "URILiteral";
  NodeType2[NodeType2["EscapedValue"] = 29] = "EscapedValue";
  NodeType2[NodeType2["Function"] = 30] = "Function";
  NodeType2[NodeType2["NumericValue"] = 31] = "NumericValue";
  NodeType2[NodeType2["HexColorValue"] = 32] = "HexColorValue";
  NodeType2[NodeType2["RatioValue"] = 33] = "RatioValue";
  NodeType2[NodeType2["MixinDeclaration"] = 34] = "MixinDeclaration";
  NodeType2[NodeType2["MixinReference"] = 35] = "MixinReference";
  NodeType2[NodeType2["VariableName"] = 36] = "VariableName";
  NodeType2[NodeType2["VariableDeclaration"] = 37] = "VariableDeclaration";
  NodeType2[NodeType2["Prio"] = 38] = "Prio";
  NodeType2[NodeType2["Interpolation"] = 39] = "Interpolation";
  NodeType2[NodeType2["NestedProperties"] = 40] = "NestedProperties";
  NodeType2[NodeType2["ExtendsReference"] = 41] = "ExtendsReference";
  NodeType2[NodeType2["SelectorPlaceholder"] = 42] = "SelectorPlaceholder";
  NodeType2[NodeType2["Debug"] = 43] = "Debug";
  NodeType2[NodeType2["If"] = 44] = "If";
  NodeType2[NodeType2["Else"] = 45] = "Else";
  NodeType2[NodeType2["For"] = 46] = "For";
  NodeType2[NodeType2["Each"] = 47] = "Each";
  NodeType2[NodeType2["While"] = 48] = "While";
  NodeType2[NodeType2["MixinContentReference"] = 49] = "MixinContentReference";
  NodeType2[NodeType2["MixinContentDeclaration"] = 50] = "MixinContentDeclaration";
  NodeType2[NodeType2["Media"] = 51] = "Media";
  NodeType2[NodeType2["Keyframe"] = 52] = "Keyframe";
  NodeType2[NodeType2["FontFace"] = 53] = "FontFace";
  NodeType2[NodeType2["Import"] = 54] = "Import";
  NodeType2[NodeType2["Namespace"] = 55] = "Namespace";
  NodeType2[NodeType2["Invocation"] = 56] = "Invocation";
  NodeType2[NodeType2["FunctionDeclaration"] = 57] = "FunctionDeclaration";
  NodeType2[NodeType2["ReturnStatement"] = 58] = "ReturnStatement";
  NodeType2[NodeType2["MediaQuery"] = 59] = "MediaQuery";
  NodeType2[NodeType2["MediaCondition"] = 60] = "MediaCondition";
  NodeType2[NodeType2["MediaFeature"] = 61] = "MediaFeature";
  NodeType2[NodeType2["FunctionParameter"] = 62] = "FunctionParameter";
  NodeType2[NodeType2["FunctionArgument"] = 63] = "FunctionArgument";
  NodeType2[NodeType2["KeyframeSelector"] = 64] = "KeyframeSelector";
  NodeType2[NodeType2["ViewPort"] = 65] = "ViewPort";
  NodeType2[NodeType2["Document"] = 66] = "Document";
  NodeType2[NodeType2["AtApplyRule"] = 67] = "AtApplyRule";
  NodeType2[NodeType2["CustomPropertyDeclaration"] = 68] = "CustomPropertyDeclaration";
  NodeType2[NodeType2["CustomPropertySet"] = 69] = "CustomPropertySet";
  NodeType2[NodeType2["ListEntry"] = 70] = "ListEntry";
  NodeType2[NodeType2["Supports"] = 71] = "Supports";
  NodeType2[NodeType2["SupportsCondition"] = 72] = "SupportsCondition";
  NodeType2[NodeType2["NamespacePrefix"] = 73] = "NamespacePrefix";
  NodeType2[NodeType2["GridLine"] = 74] = "GridLine";
  NodeType2[NodeType2["Plugin"] = 75] = "Plugin";
  NodeType2[NodeType2["UnknownAtRule"] = 76] = "UnknownAtRule";
  NodeType2[NodeType2["Use"] = 77] = "Use";
  NodeType2[NodeType2["ModuleConfiguration"] = 78] = "ModuleConfiguration";
  NodeType2[NodeType2["Forward"] = 79] = "Forward";
  NodeType2[NodeType2["ForwardVisibility"] = 80] = "ForwardVisibility";
  NodeType2[NodeType2["Module"] = 81] = "Module";
  NodeType2[NodeType2["UnicodeRange"] = 82] = "UnicodeRange";
})(NodeType || (NodeType = {}));
var ReferenceType;
(function(ReferenceType2) {
  ReferenceType2[ReferenceType2["Mixin"] = 0] = "Mixin";
  ReferenceType2[ReferenceType2["Rule"] = 1] = "Rule";
  ReferenceType2[ReferenceType2["Variable"] = 2] = "Variable";
  ReferenceType2[ReferenceType2["Function"] = 3] = "Function";
  ReferenceType2[ReferenceType2["Keyframe"] = 4] = "Keyframe";
  ReferenceType2[ReferenceType2["Unknown"] = 5] = "Unknown";
  ReferenceType2[ReferenceType2["Module"] = 6] = "Module";
  ReferenceType2[ReferenceType2["Forward"] = 7] = "Forward";
  ReferenceType2[ReferenceType2["ForwardVisibility"] = 8] = "ForwardVisibility";
})(ReferenceType || (ReferenceType = {}));
function getNodeAtOffset(node, offset) {
  var candidate = null;
  if (!node || offset < node.offset || offset > node.end) {
    return null;
  }
  node.accept(function(node2) {
    if (node2.offset === -1 && node2.length === -1) {
      return true;
    }
    if (node2.offset <= offset && node2.end >= offset) {
      if (!candidate) {
        candidate = node2;
      } else if (node2.length <= candidate.length) {
        candidate = node2;
      }
      return true;
    }
    return false;
  });
  return candidate;
}
function getNodePath(node, offset) {
  var candidate = getNodeAtOffset(node, offset);
  var path = [];
  while (candidate) {
    path.unshift(candidate);
    candidate = candidate.parent;
  }
  return path;
}
function getParentDeclaration(node) {
  var decl = node.findParent(NodeType.Declaration);
  var value = decl && decl.getValue();
  if (value && value.encloses(node)) {
    return decl;
  }
  return null;
}
var Node = function() {
  function Node2(offset, len, nodeType) {
    if (offset === void 0) {
      offset = -1;
    }
    if (len === void 0) {
      len = -1;
    }
    this.parent = null;
    this.offset = offset;
    this.length = len;
    if (nodeType) {
      this.nodeType = nodeType;
    }
  }
  Object.defineProperty(Node2.prototype, "end", {
    get: function() {
      return this.offset + this.length;
    },
    enumerable: false,
    configurable: true
  });
  Object.defineProperty(Node2.prototype, "type", {
    get: function() {
      return this.nodeType || NodeType.Undefined;
    },
    set: function(type) {
      this.nodeType = type;
    },
    enumerable: false,
    configurable: true
  });
  Node2.prototype.getTextProvider = function() {
    var node = this;
    while (node && !node.textProvider) {
      node = node.parent;
    }
    if (node) {
      return node.textProvider;
    }
    return function() {
      return "unknown";
    };
  };
  Node2.prototype.getText = function() {
    return this.getTextProvider()(this.offset, this.length);
  };
  Node2.prototype.matches = function(str) {
    return this.length === str.length && this.getTextProvider()(this.offset, this.length) === str;
  };
  Node2.prototype.startsWith = function(str) {
    return this.length >= str.length && this.getTextProvider()(this.offset, str.length) === str;
  };
  Node2.prototype.endsWith = function(str) {
    return this.length >= str.length && this.getTextProvider()(this.end - str.length, str.length) === str;
  };
  Node2.prototype.accept = function(visitor) {
    if (visitor(this) && this.children) {
      for (var _i = 0, _a2 = this.children; _i < _a2.length; _i++) {
        var child = _a2[_i];
        child.accept(visitor);
      }
    }
  };
  Node2.prototype.acceptVisitor = function(visitor) {
    this.accept(visitor.visitNode.bind(visitor));
  };
  Node2.prototype.adoptChild = function(node, index) {
    if (index === void 0) {
      index = -1;
    }
    if (node.parent && node.parent.children) {
      var idx = node.parent.children.indexOf(node);
      if (idx >= 0) {
        node.parent.children.splice(idx, 1);
      }
    }
    node.parent = this;
    var children = this.children;
    if (!children) {
      children = this.children = [];
    }
    if (index !== -1) {
      children.splice(index, 0, node);
    } else {
      children.push(node);
    }
    return node;
  };
  Node2.prototype.attachTo = function(parent, index) {
    if (index === void 0) {
      index = -1;
    }
    if (parent) {
      parent.adoptChild(this, index);
    }
    return this;
  };
  Node2.prototype.collectIssues = function(results) {
    if (this.issues) {
      results.push.apply(results, this.issues);
    }
  };
  Node2.prototype.addIssue = function(issue) {
    if (!this.issues) {
      this.issues = [];
    }
    this.issues.push(issue);
  };
  Node2.prototype.hasIssue = function(rule) {
    return Array.isArray(this.issues) && this.issues.some(function(i) {
      return i.getRule() === rule;
    });
  };
  Node2.prototype.isErroneous = function(recursive) {
    if (recursive === void 0) {
      recursive = false;
    }
    if (this.issues && this.issues.length > 0) {
      return true;
    }
    return recursive && Array.isArray(this.children) && this.children.some(function(c) {
      return c.isErroneous(true);
    });
  };
  Node2.prototype.setNode = function(field, node, index) {
    if (index === void 0) {
      index = -1;
    }
    if (node) {
      node.attachTo(this, index);
      this[field] = node;
      return true;
    }
    return false;
  };
  Node2.prototype.addChild = function(node) {
    if (node) {
      if (!this.children) {
        this.children = [];
      }
      node.attachTo(this);
      this.updateOffsetAndLength(node);
      return true;
    }
    return false;
  };
  Node2.prototype.updateOffsetAndLength = function(node) {
    if (node.offset < this.offset || this.offset === -1) {
      this.offset = node.offset;
    }
    var nodeEnd = node.end;
    if (nodeEnd > this.end || this.length === -1) {
      this.length = nodeEnd - this.offset;
    }
  };
  Node2.prototype.hasChildren = function() {
    return !!this.children && this.children.length > 0;
  };
  Node2.prototype.getChildren = function() {
    return this.children ? this.children.slice(0) : [];
  };
  Node2.prototype.getChild = function(index) {
    if (this.children && index < this.children.length) {
      return this.children[index];
    }
    return null;
  };
  Node2.prototype.addChildren = function(nodes) {
    for (var _i = 0, nodes_1 = nodes; _i < nodes_1.length; _i++) {
      var node = nodes_1[_i];
      this.addChild(node);
    }
  };
  Node2.prototype.findFirstChildBeforeOffset = function(offset) {
    if (this.children) {
      var current = null;
      for (var i = this.children.length - 1; i >= 0; i--) {
        current = this.children[i];
        if (current.offset <= offset) {
          return current;
        }
      }
    }
    return null;
  };
  Node2.prototype.findChildAtOffset = function(offset, goDeep) {
    var current = this.findFirstChildBeforeOffset(offset);
    if (current && current.end >= offset) {
      if (goDeep) {
        return current.findChildAtOffset(offset, true) || current;
      }
      return current;
    }
    return null;
  };
  Node2.prototype.encloses = function(candidate) {
    return this.offset <= candidate.offset && this.offset + this.length >= candidate.offset + candidate.length;
  };
  Node2.prototype.getParent = function() {
    var result = this.parent;
    while (result instanceof Nodelist) {
      result = result.parent;
    }
    return result;
  };
  Node2.prototype.findParent = function(type) {
    var result = this;
    while (result && result.type !== type) {
      result = result.parent;
    }
    return result;
  };
  Node2.prototype.findAParent = function() {
    var types = [];
    for (var _i = 0; _i < arguments.length; _i++) {
      types[_i] = arguments[_i];
    }
    var result = this;
    while (result && !types.some(function(t) {
      return result.type === t;
    })) {
      result = result.parent;
    }
    return result;
  };
  Node2.prototype.setData = function(key, value) {
    if (!this.options) {
      this.options = {};
    }
    this.options[key] = value;
  };
  Node2.prototype.getData = function(key) {
    if (!this.options || !this.options.hasOwnProperty(key)) {
      return null;
    }
    return this.options[key];
  };
  return Node2;
}();
var Nodelist = function(_super) {
  __extends(Nodelist2, _super);
  function Nodelist2(parent, index) {
    if (index === void 0) {
      index = -1;
    }
    var _this = _super.call(this, -1, -1) || this;
    _this.attachTo(parent, index);
    _this.offset = -1;
    _this.length = -1;
    return _this;
  }
  return Nodelist2;
}(Node);
var UnicodeRange = function(_super) {
  __extends(UnicodeRange2, _super);
  function UnicodeRange2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(UnicodeRange2.prototype, "type", {
    get: function() {
      return NodeType.UnicodeRange;
    },
    enumerable: false,
    configurable: true
  });
  UnicodeRange2.prototype.setRangeStart = function(rangeStart) {
    return this.setNode("rangeStart", rangeStart);
  };
  UnicodeRange2.prototype.getRangeStart = function() {
    return this.rangeStart;
  };
  UnicodeRange2.prototype.setRangeEnd = function(rangeEnd) {
    return this.setNode("rangeEnd", rangeEnd);
  };
  UnicodeRange2.prototype.getRangeEnd = function() {
    return this.rangeEnd;
  };
  return UnicodeRange2;
}(Node);
var Identifier = function(_super) {
  __extends(Identifier2, _super);
  function Identifier2(offset, length) {
    var _this = _super.call(this, offset, length) || this;
    _this.isCustomProperty = false;
    return _this;
  }
  Object.defineProperty(Identifier2.prototype, "type", {
    get: function() {
      return NodeType.Identifier;
    },
    enumerable: false,
    configurable: true
  });
  Identifier2.prototype.containsInterpolation = function() {
    return this.hasChildren();
  };
  return Identifier2;
}(Node);
var Stylesheet = function(_super) {
  __extends(Stylesheet2, _super);
  function Stylesheet2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(Stylesheet2.prototype, "type", {
    get: function() {
      return NodeType.Stylesheet;
    },
    enumerable: false,
    configurable: true
  });
  return Stylesheet2;
}(Node);
var Declarations = function(_super) {
  __extends(Declarations2, _super);
  function Declarations2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(Declarations2.prototype, "type", {
    get: function() {
      return NodeType.Declarations;
    },
    enumerable: false,
    configurable: true
  });
  return Declarations2;
}(Node);
var BodyDeclaration = function(_super) {
  __extends(BodyDeclaration2, _super);
  function BodyDeclaration2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  BodyDeclaration2.prototype.getDeclarations = function() {
    return this.declarations;
  };
  BodyDeclaration2.prototype.setDeclarations = function(decls) {
    return this.setNode("declarations", decls);
  };
  return BodyDeclaration2;
}(Node);
var RuleSet = function(_super) {
  __extends(RuleSet2, _super);
  function RuleSet2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(RuleSet2.prototype, "type", {
    get: function() {
      return NodeType.Ruleset;
    },
    enumerable: false,
    configurable: true
  });
  RuleSet2.prototype.getSelectors = function() {
    if (!this.selectors) {
      this.selectors = new Nodelist(this);
    }
    return this.selectors;
  };
  RuleSet2.prototype.isNested = function() {
    return !!this.parent && this.parent.findParent(NodeType.Declarations) !== null;
  };
  return RuleSet2;
}(BodyDeclaration);
var Selector = function(_super) {
  __extends(Selector2, _super);
  function Selector2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(Selector2.prototype, "type", {
    get: function() {
      return NodeType.Selector;
    },
    enumerable: false,
    configurable: true
  });
  return Selector2;
}(Node);
var SimpleSelector = function(_super) {
  __extends(SimpleSelector2, _super);
  function SimpleSelector2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(SimpleSelector2.prototype, "type", {
    get: function() {
      return NodeType.SimpleSelector;
    },
    enumerable: false,
    configurable: true
  });
  return SimpleSelector2;
}(Node);
var AtApplyRule = function(_super) {
  __extends(AtApplyRule2, _super);
  function AtApplyRule2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(AtApplyRule2.prototype, "type", {
    get: function() {
      return NodeType.AtApplyRule;
    },
    enumerable: false,
    configurable: true
  });
  AtApplyRule2.prototype.setIdentifier = function(node) {
    return this.setNode("identifier", node, 0);
  };
  AtApplyRule2.prototype.getIdentifier = function() {
    return this.identifier;
  };
  AtApplyRule2.prototype.getName = function() {
    return this.identifier ? this.identifier.getText() : "";
  };
  return AtApplyRule2;
}(Node);
var AbstractDeclaration = function(_super) {
  __extends(AbstractDeclaration2, _super);
  function AbstractDeclaration2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  return AbstractDeclaration2;
}(Node);
var CustomPropertySet = function(_super) {
  __extends(CustomPropertySet2, _super);
  function CustomPropertySet2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(CustomPropertySet2.prototype, "type", {
    get: function() {
      return NodeType.CustomPropertySet;
    },
    enumerable: false,
    configurable: true
  });
  return CustomPropertySet2;
}(BodyDeclaration);
var Declaration = function(_super) {
  __extends(Declaration2, _super);
  function Declaration2(offset, length) {
    var _this = _super.call(this, offset, length) || this;
    _this.property = null;
    return _this;
  }
  Object.defineProperty(Declaration2.prototype, "type", {
    get: function() {
      return NodeType.Declaration;
    },
    enumerable: false,
    configurable: true
  });
  Declaration2.prototype.setProperty = function(node) {
    return this.setNode("property", node);
  };
  Declaration2.prototype.getProperty = function() {
    return this.property;
  };
  Declaration2.prototype.getFullPropertyName = function() {
    var propertyName = this.property ? this.property.getName() : "unknown";
    if (this.parent instanceof Declarations && this.parent.getParent() instanceof NestedProperties) {
      var parentDecl = this.parent.getParent().getParent();
      if (parentDecl instanceof Declaration2) {
        return parentDecl.getFullPropertyName() + propertyName;
      }
    }
    return propertyName;
  };
  Declaration2.prototype.getNonPrefixedPropertyName = function() {
    var propertyName = this.getFullPropertyName();
    if (propertyName && propertyName.charAt(0) === "-") {
      var vendorPrefixEnd = propertyName.indexOf("-", 1);
      if (vendorPrefixEnd !== -1) {
        return propertyName.substring(vendorPrefixEnd + 1);
      }
    }
    return propertyName;
  };
  Declaration2.prototype.setValue = function(value) {
    return this.setNode("value", value);
  };
  Declaration2.prototype.getValue = function() {
    return this.value;
  };
  Declaration2.prototype.setNestedProperties = function(value) {
    return this.setNode("nestedProperties", value);
  };
  Declaration2.prototype.getNestedProperties = function() {
    return this.nestedProperties;
  };
  return Declaration2;
}(AbstractDeclaration);
var CustomPropertyDeclaration = function(_super) {
  __extends(CustomPropertyDeclaration2, _super);
  function CustomPropertyDeclaration2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(CustomPropertyDeclaration2.prototype, "type", {
    get: function() {
      return NodeType.CustomPropertyDeclaration;
    },
    enumerable: false,
    configurable: true
  });
  CustomPropertyDeclaration2.prototype.setPropertySet = function(value) {
    return this.setNode("propertySet", value);
  };
  CustomPropertyDeclaration2.prototype.getPropertySet = function() {
    return this.propertySet;
  };
  return CustomPropertyDeclaration2;
}(Declaration);
var Property = function(_super) {
  __extends(Property2, _super);
  function Property2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(Property2.prototype, "type", {
    get: function() {
      return NodeType.Property;
    },
    enumerable: false,
    configurable: true
  });
  Property2.prototype.setIdentifier = function(value) {
    return this.setNode("identifier", value);
  };
  Property2.prototype.getIdentifier = function() {
    return this.identifier;
  };
  Property2.prototype.getName = function() {
    return trim(this.getText(), /[_\+]+$/);
  };
  Property2.prototype.isCustomProperty = function() {
    return !!this.identifier && this.identifier.isCustomProperty;
  };
  return Property2;
}(Node);
var Invocation = function(_super) {
  __extends(Invocation2, _super);
  function Invocation2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(Invocation2.prototype, "type", {
    get: function() {
      return NodeType.Invocation;
    },
    enumerable: false,
    configurable: true
  });
  Invocation2.prototype.getArguments = function() {
    if (!this.arguments) {
      this.arguments = new Nodelist(this);
    }
    return this.arguments;
  };
  return Invocation2;
}(Node);
var Function = function(_super) {
  __extends(Function2, _super);
  function Function2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(Function2.prototype, "type", {
    get: function() {
      return NodeType.Function;
    },
    enumerable: false,
    configurable: true
  });
  Function2.prototype.setIdentifier = function(node) {
    return this.setNode("identifier", node, 0);
  };
  Function2.prototype.getIdentifier = function() {
    return this.identifier;
  };
  Function2.prototype.getName = function() {
    return this.identifier ? this.identifier.getText() : "";
  };
  return Function2;
}(Invocation);
var FunctionParameter = function(_super) {
  __extends(FunctionParameter2, _super);
  function FunctionParameter2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(FunctionParameter2.prototype, "type", {
    get: function() {
      return NodeType.FunctionParameter;
    },
    enumerable: false,
    configurable: true
  });
  FunctionParameter2.prototype.setIdentifier = function(node) {
    return this.setNode("identifier", node, 0);
  };
  FunctionParameter2.prototype.getIdentifier = function() {
    return this.identifier;
  };
  FunctionParameter2.prototype.getName = function() {
    return this.identifier ? this.identifier.getText() : "";
  };
  FunctionParameter2.prototype.setDefaultValue = function(node) {
    return this.setNode("defaultValue", node, 0);
  };
  FunctionParameter2.prototype.getDefaultValue = function() {
    return this.defaultValue;
  };
  return FunctionParameter2;
}(Node);
var FunctionArgument = function(_super) {
  __extends(FunctionArgument2, _super);
  function FunctionArgument2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(FunctionArgument2.prototype, "type", {
    get: function() {
      return NodeType.FunctionArgument;
    },
    enumerable: false,
    configurable: true
  });
  FunctionArgument2.prototype.setIdentifier = function(node) {
    return this.setNode("identifier", node, 0);
  };
  FunctionArgument2.prototype.getIdentifier = function() {
    return this.identifier;
  };
  FunctionArgument2.prototype.getName = function() {
    return this.identifier ? this.identifier.getText() : "";
  };
  FunctionArgument2.prototype.setValue = function(node) {
    return this.setNode("value", node, 0);
  };
  FunctionArgument2.prototype.getValue = function() {
    return this.value;
  };
  return FunctionArgument2;
}(Node);
var IfStatement = function(_super) {
  __extends(IfStatement2, _super);
  function IfStatement2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(IfStatement2.prototype, "type", {
    get: function() {
      return NodeType.If;
    },
    enumerable: false,
    configurable: true
  });
  IfStatement2.prototype.setExpression = function(node) {
    return this.setNode("expression", node, 0);
  };
  IfStatement2.prototype.setElseClause = function(elseClause) {
    return this.setNode("elseClause", elseClause);
  };
  return IfStatement2;
}(BodyDeclaration);
var ForStatement = function(_super) {
  __extends(ForStatement2, _super);
  function ForStatement2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(ForStatement2.prototype, "type", {
    get: function() {
      return NodeType.For;
    },
    enumerable: false,
    configurable: true
  });
  ForStatement2.prototype.setVariable = function(node) {
    return this.setNode("variable", node, 0);
  };
  return ForStatement2;
}(BodyDeclaration);
var EachStatement = function(_super) {
  __extends(EachStatement2, _super);
  function EachStatement2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(EachStatement2.prototype, "type", {
    get: function() {
      return NodeType.Each;
    },
    enumerable: false,
    configurable: true
  });
  EachStatement2.prototype.getVariables = function() {
    if (!this.variables) {
      this.variables = new Nodelist(this);
    }
    return this.variables;
  };
  return EachStatement2;
}(BodyDeclaration);
var WhileStatement = function(_super) {
  __extends(WhileStatement2, _super);
  function WhileStatement2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(WhileStatement2.prototype, "type", {
    get: function() {
      return NodeType.While;
    },
    enumerable: false,
    configurable: true
  });
  return WhileStatement2;
}(BodyDeclaration);
var ElseStatement = function(_super) {
  __extends(ElseStatement2, _super);
  function ElseStatement2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(ElseStatement2.prototype, "type", {
    get: function() {
      return NodeType.Else;
    },
    enumerable: false,
    configurable: true
  });
  return ElseStatement2;
}(BodyDeclaration);
var FunctionDeclaration = function(_super) {
  __extends(FunctionDeclaration2, _super);
  function FunctionDeclaration2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(FunctionDeclaration2.prototype, "type", {
    get: function() {
      return NodeType.FunctionDeclaration;
    },
    enumerable: false,
    configurable: true
  });
  FunctionDeclaration2.prototype.setIdentifier = function(node) {
    return this.setNode("identifier", node, 0);
  };
  FunctionDeclaration2.prototype.getIdentifier = function() {
    return this.identifier;
  };
  FunctionDeclaration2.prototype.getName = function() {
    return this.identifier ? this.identifier.getText() : "";
  };
  FunctionDeclaration2.prototype.getParameters = function() {
    if (!this.parameters) {
      this.parameters = new Nodelist(this);
    }
    return this.parameters;
  };
  return FunctionDeclaration2;
}(BodyDeclaration);
var ViewPort = function(_super) {
  __extends(ViewPort2, _super);
  function ViewPort2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(ViewPort2.prototype, "type", {
    get: function() {
      return NodeType.ViewPort;
    },
    enumerable: false,
    configurable: true
  });
  return ViewPort2;
}(BodyDeclaration);
var FontFace = function(_super) {
  __extends(FontFace2, _super);
  function FontFace2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(FontFace2.prototype, "type", {
    get: function() {
      return NodeType.FontFace;
    },
    enumerable: false,
    configurable: true
  });
  return FontFace2;
}(BodyDeclaration);
var NestedProperties = function(_super) {
  __extends(NestedProperties2, _super);
  function NestedProperties2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(NestedProperties2.prototype, "type", {
    get: function() {
      return NodeType.NestedProperties;
    },
    enumerable: false,
    configurable: true
  });
  return NestedProperties2;
}(BodyDeclaration);
var Keyframe = function(_super) {
  __extends(Keyframe2, _super);
  function Keyframe2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(Keyframe2.prototype, "type", {
    get: function() {
      return NodeType.Keyframe;
    },
    enumerable: false,
    configurable: true
  });
  Keyframe2.prototype.setKeyword = function(keyword) {
    return this.setNode("keyword", keyword, 0);
  };
  Keyframe2.prototype.getKeyword = function() {
    return this.keyword;
  };
  Keyframe2.prototype.setIdentifier = function(node) {
    return this.setNode("identifier", node, 0);
  };
  Keyframe2.prototype.getIdentifier = function() {
    return this.identifier;
  };
  Keyframe2.prototype.getName = function() {
    return this.identifier ? this.identifier.getText() : "";
  };
  return Keyframe2;
}(BodyDeclaration);
var KeyframeSelector = function(_super) {
  __extends(KeyframeSelector2, _super);
  function KeyframeSelector2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(KeyframeSelector2.prototype, "type", {
    get: function() {
      return NodeType.KeyframeSelector;
    },
    enumerable: false,
    configurable: true
  });
  return KeyframeSelector2;
}(BodyDeclaration);
var Import = function(_super) {
  __extends(Import2, _super);
  function Import2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(Import2.prototype, "type", {
    get: function() {
      return NodeType.Import;
    },
    enumerable: false,
    configurable: true
  });
  Import2.prototype.setMedialist = function(node) {
    if (node) {
      node.attachTo(this);
      return true;
    }
    return false;
  };
  return Import2;
}(Node);
var Use = function(_super) {
  __extends(Use2, _super);
  function Use2() {
    return _super !== null && _super.apply(this, arguments) || this;
  }
  Object.defineProperty(Use2.prototype, "type", {
    get: function() {
      return NodeType.Use;
    },
    enumerable: false,
    configurable: true
  });
  Use2.prototype.getParameters = function() {
    if (!this.parameters) {
      this.parameters = new Nodelist(this);
    }
    return this.parameters;
  };
  Use2.prototype.setIdentifier = function(node) {
    return this.setNode("identifier", node, 0);
  };
  Use2.prototype.getIdentifier = function() {
    return this.identifier;
  };
  return Use2;
}(Node);
var ModuleConfiguration = function(_super) {
  __extends(ModuleConfiguration2, _super);
  function ModuleConfiguration2() {
    return _super !== null && _super.apply(this, arguments) || this;
  }
  Object.defineProperty(ModuleConfiguration2.prototype, "type", {
    get: function() {
      return NodeType.ModuleConfiguration;
    },
    enumerable: false,
    configurable: true
  });
  ModuleConfiguration2.prototype.setIdentifier = function(node) {
    return this.setNode("identifier", node, 0);
  };
  ModuleConfiguration2.prototype.getIdentifier = function() {
    return this.identifier;
  };
  ModuleConfiguration2.prototype.getName = function() {
    return this.identifier ? this.identifier.getText() : "";
  };
  ModuleConfiguration2.prototype.setValue = function(node) {
    return this.setNode("value", node, 0);
  };
  ModuleConfiguration2.prototype.getValue = function() {
    return this.value;
  };
  return ModuleConfiguration2;
}(Node);
var Forward = function(_super) {
  __extends(Forward2, _super);
  function Forward2() {
    return _super !== null && _super.apply(this, arguments) || this;
  }
  Object.defineProperty(Forward2.prototype, "type", {
    get: function() {
      return NodeType.Forward;
    },
    enumerable: false,
    configurable: true
  });
  Forward2.prototype.setIdentifier = function(node) {
    return this.setNode("identifier", node, 0);
  };
  Forward2.prototype.getIdentifier = function() {
    return this.identifier;
  };
  Forward2.prototype.getMembers = function() {
    if (!this.members) {
      this.members = new Nodelist(this);
    }
    return this.members;
  };
  Forward2.prototype.getParameters = function() {
    if (!this.parameters) {
      this.parameters = new Nodelist(this);
    }
    return this.parameters;
  };
  return Forward2;
}(Node);
var ForwardVisibility = function(_super) {
  __extends(ForwardVisibility2, _super);
  function ForwardVisibility2() {
    return _super !== null && _super.apply(this, arguments) || this;
  }
  Object.defineProperty(ForwardVisibility2.prototype, "type", {
    get: function() {
      return NodeType.ForwardVisibility;
    },
    enumerable: false,
    configurable: true
  });
  ForwardVisibility2.prototype.setIdentifier = function(node) {
    return this.setNode("identifier", node, 0);
  };
  ForwardVisibility2.prototype.getIdentifier = function() {
    return this.identifier;
  };
  return ForwardVisibility2;
}(Node);
var Namespace = function(_super) {
  __extends(Namespace2, _super);
  function Namespace2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(Namespace2.prototype, "type", {
    get: function() {
      return NodeType.Namespace;
    },
    enumerable: false,
    configurable: true
  });
  return Namespace2;
}(Node);
var Media = function(_super) {
  __extends(Media2, _super);
  function Media2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(Media2.prototype, "type", {
    get: function() {
      return NodeType.Media;
    },
    enumerable: false,
    configurable: true
  });
  return Media2;
}(BodyDeclaration);
var Supports = function(_super) {
  __extends(Supports2, _super);
  function Supports2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(Supports2.prototype, "type", {
    get: function() {
      return NodeType.Supports;
    },
    enumerable: false,
    configurable: true
  });
  return Supports2;
}(BodyDeclaration);
var Document = function(_super) {
  __extends(Document2, _super);
  function Document2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(Document2.prototype, "type", {
    get: function() {
      return NodeType.Document;
    },
    enumerable: false,
    configurable: true
  });
  return Document2;
}(BodyDeclaration);
var Medialist = function(_super) {
  __extends(Medialist2, _super);
  function Medialist2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Medialist2.prototype.getMediums = function() {
    if (!this.mediums) {
      this.mediums = new Nodelist(this);
    }
    return this.mediums;
  };
  return Medialist2;
}(Node);
var MediaQuery = function(_super) {
  __extends(MediaQuery2, _super);
  function MediaQuery2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(MediaQuery2.prototype, "type", {
    get: function() {
      return NodeType.MediaQuery;
    },
    enumerable: false,
    configurable: true
  });
  return MediaQuery2;
}(Node);
var MediaCondition = function(_super) {
  __extends(MediaCondition2, _super);
  function MediaCondition2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(MediaCondition2.prototype, "type", {
    get: function() {
      return NodeType.MediaCondition;
    },
    enumerable: false,
    configurable: true
  });
  return MediaCondition2;
}(Node);
var MediaFeature = function(_super) {
  __extends(MediaFeature2, _super);
  function MediaFeature2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(MediaFeature2.prototype, "type", {
    get: function() {
      return NodeType.MediaFeature;
    },
    enumerable: false,
    configurable: true
  });
  return MediaFeature2;
}(Node);
var SupportsCondition = function(_super) {
  __extends(SupportsCondition2, _super);
  function SupportsCondition2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(SupportsCondition2.prototype, "type", {
    get: function() {
      return NodeType.SupportsCondition;
    },
    enumerable: false,
    configurable: true
  });
  return SupportsCondition2;
}(Node);
var Page = function(_super) {
  __extends(Page2, _super);
  function Page2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(Page2.prototype, "type", {
    get: function() {
      return NodeType.Page;
    },
    enumerable: false,
    configurable: true
  });
  return Page2;
}(BodyDeclaration);
var PageBoxMarginBox = function(_super) {
  __extends(PageBoxMarginBox2, _super);
  function PageBoxMarginBox2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(PageBoxMarginBox2.prototype, "type", {
    get: function() {
      return NodeType.PageBoxMarginBox;
    },
    enumerable: false,
    configurable: true
  });
  return PageBoxMarginBox2;
}(BodyDeclaration);
var Expression = function(_super) {
  __extends(Expression2, _super);
  function Expression2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(Expression2.prototype, "type", {
    get: function() {
      return NodeType.Expression;
    },
    enumerable: false,
    configurable: true
  });
  return Expression2;
}(Node);
var BinaryExpression = function(_super) {
  __extends(BinaryExpression2, _super);
  function BinaryExpression2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(BinaryExpression2.prototype, "type", {
    get: function() {
      return NodeType.BinaryExpression;
    },
    enumerable: false,
    configurable: true
  });
  BinaryExpression2.prototype.setLeft = function(left) {
    return this.setNode("left", left);
  };
  BinaryExpression2.prototype.getLeft = function() {
    return this.left;
  };
  BinaryExpression2.prototype.setRight = function(right) {
    return this.setNode("right", right);
  };
  BinaryExpression2.prototype.getRight = function() {
    return this.right;
  };
  BinaryExpression2.prototype.setOperator = function(value) {
    return this.setNode("operator", value);
  };
  BinaryExpression2.prototype.getOperator = function() {
    return this.operator;
  };
  return BinaryExpression2;
}(Node);
var Term = function(_super) {
  __extends(Term2, _super);
  function Term2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(Term2.prototype, "type", {
    get: function() {
      return NodeType.Term;
    },
    enumerable: false,
    configurable: true
  });
  Term2.prototype.setOperator = function(value) {
    return this.setNode("operator", value);
  };
  Term2.prototype.getOperator = function() {
    return this.operator;
  };
  Term2.prototype.setExpression = function(value) {
    return this.setNode("expression", value);
  };
  Term2.prototype.getExpression = function() {
    return this.expression;
  };
  return Term2;
}(Node);
var AttributeSelector = function(_super) {
  __extends(AttributeSelector2, _super);
  function AttributeSelector2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(AttributeSelector2.prototype, "type", {
    get: function() {
      return NodeType.AttributeSelector;
    },
    enumerable: false,
    configurable: true
  });
  AttributeSelector2.prototype.setNamespacePrefix = function(value) {
    return this.setNode("namespacePrefix", value);
  };
  AttributeSelector2.prototype.getNamespacePrefix = function() {
    return this.namespacePrefix;
  };
  AttributeSelector2.prototype.setIdentifier = function(value) {
    return this.setNode("identifier", value);
  };
  AttributeSelector2.prototype.getIdentifier = function() {
    return this.identifier;
  };
  AttributeSelector2.prototype.setOperator = function(operator) {
    return this.setNode("operator", operator);
  };
  AttributeSelector2.prototype.getOperator = function() {
    return this.operator;
  };
  AttributeSelector2.prototype.setValue = function(value) {
    return this.setNode("value", value);
  };
  AttributeSelector2.prototype.getValue = function() {
    return this.value;
  };
  return AttributeSelector2;
}(Node);
var Operator = function(_super) {
  __extends(Operator2, _super);
  function Operator2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(Operator2.prototype, "type", {
    get: function() {
      return NodeType.Operator;
    },
    enumerable: false,
    configurable: true
  });
  return Operator2;
}(Node);
var HexColorValue = function(_super) {
  __extends(HexColorValue2, _super);
  function HexColorValue2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(HexColorValue2.prototype, "type", {
    get: function() {
      return NodeType.HexColorValue;
    },
    enumerable: false,
    configurable: true
  });
  return HexColorValue2;
}(Node);
var RatioValue = function(_super) {
  __extends(RatioValue2, _super);
  function RatioValue2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(RatioValue2.prototype, "type", {
    get: function() {
      return NodeType.RatioValue;
    },
    enumerable: false,
    configurable: true
  });
  return RatioValue2;
}(Node);
var _dot = ".".charCodeAt(0);
var _02 = "0".charCodeAt(0);
var _92 = "9".charCodeAt(0);
var NumericValue = function(_super) {
  __extends(NumericValue2, _super);
  function NumericValue2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(NumericValue2.prototype, "type", {
    get: function() {
      return NodeType.NumericValue;
    },
    enumerable: false,
    configurable: true
  });
  NumericValue2.prototype.getValue = function() {
    var raw = this.getText();
    var unitIdx = 0;
    var code;
    for (var i = 0, len = raw.length; i < len; i++) {
      code = raw.charCodeAt(i);
      if (!(_02 <= code && code <= _92 || code === _dot)) {
        break;
      }
      unitIdx += 1;
    }
    return {
      value: raw.substring(0, unitIdx),
      unit: unitIdx < raw.length ? raw.substring(unitIdx) : void 0
    };
  };
  return NumericValue2;
}(Node);
var VariableDeclaration = function(_super) {
  __extends(VariableDeclaration2, _super);
  function VariableDeclaration2(offset, length) {
    var _this = _super.call(this, offset, length) || this;
    _this.variable = null;
    _this.value = null;
    _this.needsSemicolon = true;
    return _this;
  }
  Object.defineProperty(VariableDeclaration2.prototype, "type", {
    get: function() {
      return NodeType.VariableDeclaration;
    },
    enumerable: false,
    configurable: true
  });
  VariableDeclaration2.prototype.setVariable = function(node) {
    if (node) {
      node.attachTo(this);
      this.variable = node;
      return true;
    }
    return false;
  };
  VariableDeclaration2.prototype.getVariable = function() {
    return this.variable;
  };
  VariableDeclaration2.prototype.getName = function() {
    return this.variable ? this.variable.getName() : "";
  };
  VariableDeclaration2.prototype.setValue = function(node) {
    if (node) {
      node.attachTo(this);
      this.value = node;
      return true;
    }
    return false;
  };
  VariableDeclaration2.prototype.getValue = function() {
    return this.value;
  };
  return VariableDeclaration2;
}(AbstractDeclaration);
var Interpolation = function(_super) {
  __extends(Interpolation2, _super);
  function Interpolation2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(Interpolation2.prototype, "type", {
    get: function() {
      return NodeType.Interpolation;
    },
    enumerable: false,
    configurable: true
  });
  return Interpolation2;
}(Node);
var Variable = function(_super) {
  __extends(Variable2, _super);
  function Variable2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(Variable2.prototype, "type", {
    get: function() {
      return NodeType.VariableName;
    },
    enumerable: false,
    configurable: true
  });
  Variable2.prototype.getName = function() {
    return this.getText();
  };
  return Variable2;
}(Node);
var ExtendsReference = function(_super) {
  __extends(ExtendsReference2, _super);
  function ExtendsReference2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(ExtendsReference2.prototype, "type", {
    get: function() {
      return NodeType.ExtendsReference;
    },
    enumerable: false,
    configurable: true
  });
  ExtendsReference2.prototype.getSelectors = function() {
    if (!this.selectors) {
      this.selectors = new Nodelist(this);
    }
    return this.selectors;
  };
  return ExtendsReference2;
}(Node);
var MixinContentReference = function(_super) {
  __extends(MixinContentReference2, _super);
  function MixinContentReference2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(MixinContentReference2.prototype, "type", {
    get: function() {
      return NodeType.MixinContentReference;
    },
    enumerable: false,
    configurable: true
  });
  MixinContentReference2.prototype.getArguments = function() {
    if (!this.arguments) {
      this.arguments = new Nodelist(this);
    }
    return this.arguments;
  };
  return MixinContentReference2;
}(Node);
var MixinContentDeclaration = function(_super) {
  __extends(MixinContentDeclaration2, _super);
  function MixinContentDeclaration2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(MixinContentDeclaration2.prototype, "type", {
    get: function() {
      return NodeType.MixinContentReference;
    },
    enumerable: false,
    configurable: true
  });
  MixinContentDeclaration2.prototype.getParameters = function() {
    if (!this.parameters) {
      this.parameters = new Nodelist(this);
    }
    return this.parameters;
  };
  return MixinContentDeclaration2;
}(BodyDeclaration);
var MixinReference = function(_super) {
  __extends(MixinReference2, _super);
  function MixinReference2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(MixinReference2.prototype, "type", {
    get: function() {
      return NodeType.MixinReference;
    },
    enumerable: false,
    configurable: true
  });
  MixinReference2.prototype.getNamespaces = function() {
    if (!this.namespaces) {
      this.namespaces = new Nodelist(this);
    }
    return this.namespaces;
  };
  MixinReference2.prototype.setIdentifier = function(node) {
    return this.setNode("identifier", node, 0);
  };
  MixinReference2.prototype.getIdentifier = function() {
    return this.identifier;
  };
  MixinReference2.prototype.getName = function() {
    return this.identifier ? this.identifier.getText() : "";
  };
  MixinReference2.prototype.getArguments = function() {
    if (!this.arguments) {
      this.arguments = new Nodelist(this);
    }
    return this.arguments;
  };
  MixinReference2.prototype.setContent = function(node) {
    return this.setNode("content", node);
  };
  MixinReference2.prototype.getContent = function() {
    return this.content;
  };
  return MixinReference2;
}(Node);
var MixinDeclaration = function(_super) {
  __extends(MixinDeclaration2, _super);
  function MixinDeclaration2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(MixinDeclaration2.prototype, "type", {
    get: function() {
      return NodeType.MixinDeclaration;
    },
    enumerable: false,
    configurable: true
  });
  MixinDeclaration2.prototype.setIdentifier = function(node) {
    return this.setNode("identifier", node, 0);
  };
  MixinDeclaration2.prototype.getIdentifier = function() {
    return this.identifier;
  };
  MixinDeclaration2.prototype.getName = function() {
    return this.identifier ? this.identifier.getText() : "";
  };
  MixinDeclaration2.prototype.getParameters = function() {
    if (!this.parameters) {
      this.parameters = new Nodelist(this);
    }
    return this.parameters;
  };
  MixinDeclaration2.prototype.setGuard = function(node) {
    if (node) {
      node.attachTo(this);
      this.guard = node;
    }
    return false;
  };
  return MixinDeclaration2;
}(BodyDeclaration);
var UnknownAtRule = function(_super) {
  __extends(UnknownAtRule2, _super);
  function UnknownAtRule2(offset, length) {
    return _super.call(this, offset, length) || this;
  }
  Object.defineProperty(UnknownAtRule2.prototype, "type", {
    get: function() {
      return NodeType.UnknownAtRule;
    },
    enumerable: false,
    configurable: true
  });
  UnknownAtRule2.prototype.setAtRuleName = function(atRuleName) {
    this.atRuleName = atRuleName;
  };
  UnknownAtRule2.prototype.getAtRuleName = function() {
    return this.atRuleName;
  };
  return UnknownAtRule2;
}(BodyDeclaration);
var ListEntry = function(_super) {
  __extends(ListEntry2, _super);
  function ListEntry2() {
    return _super !== null && _super.apply(this, arguments) || this;
  }
  Object.defineProperty(ListEntry2.prototype, "type", {
    get: function() {
      return NodeType.ListEntry;
    },
    enumerable: false,
    configurable: true
  });
  ListEntry2.prototype.setKey = function(node) {
    return this.setNode("key", node, 0);
  };
  ListEntry2.prototype.setValue = function(node) {
    return this.setNode("value", node, 1);
  };
  return ListEntry2;
}(Node);
var LessGuard = function(_super) {
  __extends(LessGuard2, _super);
  function LessGuard2() {
    return _super !== null && _super.apply(this, arguments) || this;
  }
  LessGuard2.prototype.getConditions = function() {
    if (!this.conditions) {
      this.conditions = new Nodelist(this);
    }
    return this.conditions;
  };
  return LessGuard2;
}(Node);
var GuardCondition = function(_super) {
  __extends(GuardCondition2, _super);
  function GuardCondition2() {
    return _super !== null && _super.apply(this, arguments) || this;
  }
  GuardCondition2.prototype.setVariable = function(node) {
    return this.setNode("variable", node);
  };
  return GuardCondition2;
}(Node);
var Module = function(_super) {
  __extends(Module3, _super);
  function Module3() {
    return _super !== null && _super.apply(this, arguments) || this;
  }
  Object.defineProperty(Module3.prototype, "type", {
    get: function() {
      return NodeType.Module;
    },
    enumerable: false,
    configurable: true
  });
  Module3.prototype.setIdentifier = function(node) {
    return this.setNode("identifier", node, 0);
  };
  Module3.prototype.getIdentifier = function() {
    return this.identifier;
  };
  return Module3;
}(Node);
var Level;
(function(Level2) {
  Level2[Level2["Ignore"] = 1] = "Ignore";
  Level2[Level2["Warning"] = 2] = "Warning";
  Level2[Level2["Error"] = 4] = "Error";
})(Level || (Level = {}));
var Marker = function() {
  function Marker2(node, rule, level, message, offset, length) {
    if (offset === void 0) {
      offset = node.offset;
    }
    if (length === void 0) {
      length = node.length;
    }
    this.node = node;
    this.rule = rule;
    this.level = level;
    this.message = message || rule.message;
    this.offset = offset;
    this.length = length;
  }
  Marker2.prototype.getRule = function() {
    return this.rule;
  };
  Marker2.prototype.getLevel = function() {
    return this.level;
  };
  Marker2.prototype.getOffset = function() {
    return this.offset;
  };
  Marker2.prototype.getLength = function() {
    return this.length;
  };
  Marker2.prototype.getNode = function() {
    return this.node;
  };
  Marker2.prototype.getMessage = function() {
    return this.message;
  };
  return Marker2;
}();
var ParseErrorCollector = function() {
  function ParseErrorCollector2() {
    this.entries = [];
  }
  ParseErrorCollector2.entries = function(node) {
    var visitor = new ParseErrorCollector2();
    node.acceptVisitor(visitor);
    return visitor.entries;
  };
  ParseErrorCollector2.prototype.visitNode = function(node) {
    if (node.isErroneous()) {
      node.collectIssues(this.entries);
    }
    return true;
  };
  return ParseErrorCollector2;
}();

// build/fillers/vscode-nls.ts
function format(message, args) {
  let result;
  if (args.length === 0) {
    result = message;
  } else {
    result = message.replace(/\{(\d+)\}/g, (match, rest) => {
      let index = rest[0];
      return typeof args[index] !== "undefined" ? args[index] : match;
    });
  }
  return result;
}
function localize(key, message, ...args) {
  return format(message, args);
}
function loadMessageBundle(file) {
  return localize;
}

// node_modules/vscode-css-languageservice/lib/esm/parser/cssErrors.js
var localize2 = loadMessageBundle();
var CSSIssueType = function() {
  function CSSIssueType2(id, message) {
    this.id = id;
    this.message = message;
  }
  return CSSIssueType2;
}();
var ParseError = {
  NumberExpected: new CSSIssueType("css-numberexpected", localize2("expected.number", "number expected")),
  ConditionExpected: new CSSIssueType("css-conditionexpected", localize2("expected.condt", "condition expected")),
  RuleOrSelectorExpected: new CSSIssueType("css-ruleorselectorexpected", localize2("expected.ruleorselector", "at-rule or selector expected")),
  DotExpected: new CSSIssueType("css-dotexpected", localize2("expected.dot", "dot expected")),
  ColonExpected: new CSSIssueType("css-colonexpected", localize2("expected.colon", "colon expected")),
  SemiColonExpected: new CSSIssueType("css-semicolonexpected", localize2("expected.semicolon", "semi-colon expected")),
  TermExpected: new CSSIssueType("css-termexpected", localize2("expected.term", "term expected")),
  ExpressionExpected: new CSSIssueType("css-expressionexpected", localize2("expected.expression", "expression expected")),
  OperatorExpected: new CSSIssueType("css-operatorexpected", localize2("expected.operator", "operator expected")),
  IdentifierExpected: new CSSIssueType("css-identifierexpected", localize2("expected.ident", "identifier expected")),
  PercentageExpected: new CSSIssueType("css-percentageexpected", localize2("expected.percentage", "percentage expected")),
  URIOrStringExpected: new CSSIssueType("css-uriorstringexpected", localize2("expected.uriorstring", "uri or string expected")),
  URIExpected: new CSSIssueType("css-uriexpected", localize2("expected.uri", "URI expected")),
  VariableNameExpected: new CSSIssueType("css-varnameexpected", localize2("expected.varname", "variable name expected")),
  VariableValueExpected: new CSSIssueType("css-varvalueexpected", localize2("expected.varvalue", "variable value expected")),
  PropertyValueExpected: new CSSIssueType("css-propertyvalueexpected", localize2("expected.propvalue", "property value expected")),
  LeftCurlyExpected: new CSSIssueType("css-lcurlyexpected", localize2("expected.lcurly", "{ expected")),
  RightCurlyExpected: new CSSIssueType("css-rcurlyexpected", localize2("expected.rcurly", "} expected")),
  LeftSquareBracketExpected: new CSSIssueType("css-rbracketexpected", localize2("expected.lsquare", "[ expected")),
  RightSquareBracketExpected: new CSSIssueType("css-lbracketexpected", localize2("expected.rsquare", "] expected")),
  LeftParenthesisExpected: new CSSIssueType("css-lparentexpected", localize2("expected.lparen", "( expected")),
  RightParenthesisExpected: new CSSIssueType("css-rparentexpected", localize2("expected.rparent", ") expected")),
  CommaExpected: new CSSIssueType("css-commaexpected", localize2("expected.comma", "comma expected")),
  PageDirectiveOrDeclarationExpected: new CSSIssueType("css-pagedirordeclexpected", localize2("expected.pagedirordecl", "page directive or declaraton expected")),
  UnknownAtRule: new CSSIssueType("css-unknownatrule", localize2("unknown.atrule", "at-rule unknown")),
  UnknownKeyword: new CSSIssueType("css-unknownkeyword", localize2("unknown.keyword", "unknown keyword")),
  SelectorExpected: new CSSIssueType("css-selectorexpected", localize2("expected.selector", "selector expected")),
  StringLiteralExpected: new CSSIssueType("css-stringliteralexpected", localize2("expected.stringliteral", "string literal expected")),
  WhitespaceExpected: new CSSIssueType("css-whitespaceexpected", localize2("expected.whitespace", "whitespace expected")),
  MediaQueryExpected: new CSSIssueType("css-mediaqueryexpected", localize2("expected.mediaquery", "media query expected")),
  IdentifierOrWildcardExpected: new CSSIssueType("css-idorwildcardexpected", localize2("expected.idorwildcard", "identifier or wildcard expected")),
  WildcardExpected: new CSSIssueType("css-wildcardexpected", localize2("expected.wildcard", "wildcard expected")),
  IdentifierOrVariableExpected: new CSSIssueType("css-idorvarexpected", localize2("expected.idorvar", "identifier or variable expected"))
};

// node_modules/vscode-languageserver-types/lib/esm/main.js
var integer;
(function(integer2) {
  integer2.MIN_VALUE = -2147483648;
  integer2.MAX_VALUE = 2147483647;
})(integer || (integer = {}));
var uinteger;
(function(uinteger2) {
  uinteger2.MIN_VALUE = 0;
  uinteger2.MAX_VALUE = 2147483647;
})(uinteger || (uinteger = {}));
var Position;
(function(Position2) {
  function create(line, character) {
    if (line === Number.MAX_VALUE) {
      line = uinteger.MAX_VALUE;
    }
    if (character === Number.MAX_VALUE) {
      character = uinteger.MAX_VALUE;
    }
    return { line, character };
  }
  Position2.create = create;
  function is(value) {
    var candidate = value;
    return Is.objectLiteral(candidate) && Is.uinteger(candidate.line) && Is.uinteger(candidate.character);
  }
  Position2.is = is;
})(Position || (Position = {}));
var Range;
(function(Range2) {
  function create(one, two, three, four) {
    if (Is.uinteger(one) && Is.uinteger(two) && Is.uinteger(three) && Is.uinteger(four)) {
      return { start: Position.create(one, two), end: Position.create(three, four) };
    } else if (Position.is(one) && Position.is(two)) {
      return { start: one, end: two };
    } else {
      throw new Error("Range#create called with invalid arguments[" + one + ", " + two + ", " + three + ", " + four + "]");
    }
  }
  Range2.create = create;
  function is(value) {
    var candidate = value;
    return Is.objectLiteral(candidate) && Position.is(candidate.start) && Position.is(candidate.end);
  }
  Range2.is = is;
})(Range || (Range = {}));
var Location;
(function(Location2) {
  function create(uri, range) {
    return { uri, range };
  }
  Location2.create = create;
  function is(value) {
    var candidate = value;
    return Is.defined(candidate) && Range.is(candidate.range) && (Is.string(candidate.uri) || Is.undefined(candidate.uri));
  }
  Location2.is = is;
})(Location || (Location = {}));
var LocationLink;
(function(LocationLink2) {
  function create(targetUri, targetRange, targetSelectionRange, originSelectionRange) {
    return { targetUri, targetRange, targetSelectionRange, originSelectionRange };
  }
  LocationLink2.create = create;
  function is(value) {
    var candidate = value;
    return Is.defined(candidate) && Range.is(candidate.targetRange) && Is.string(candidate.targetUri) && (Range.is(candidate.targetSelectionRange) || Is.undefined(candidate.targetSelectionRange)) && (Range.is(candidate.originSelectionRange) || Is.undefined(candidate.originSelectionRange));
  }
  LocationLink2.is = is;
})(LocationLink || (LocationLink = {}));
var Color;
(function(Color2) {
  function create(red, green, blue, alpha) {
    return {
      red,
      green,
      blue,
      alpha
    };
  }
  Color2.create = create;
  function is(value) {
    var candidate = value;
    return Is.numberRange(candidate.red, 0, 1) && Is.numberRange(candidate.green, 0, 1) && Is.numberRange(candidate.blue, 0, 1) && Is.numberRange(candidate.alpha, 0, 1);
  }
  Color2.is = is;
})(Color || (Color = {}));
var ColorInformation;
(function(ColorInformation2) {
  function create(range, color) {
    return {
      range,
      color
    };
  }
  ColorInformation2.create = create;
  function is(value) {
    var candidate = value;
    return Range.is(candidate.range) && Color.is(candidate.color);
  }
  ColorInformation2.is = is;
})(ColorInformation || (ColorInformation = {}));
var ColorPresentation;
(function(ColorPresentation2) {
  function create(label, textEdit, additionalTextEdits) {
    return {
      label,
      textEdit,
      additionalTextEdits
    };
  }
  ColorPresentation2.create = create;
  function is(value) {
    var candidate = value;
    return Is.string(candidate.label) && (Is.undefined(candidate.textEdit) || TextEdit.is(candidate)) && (Is.undefined(candidate.additionalTextEdits) || Is.typedArray(candidate.additionalTextEdits, TextEdit.is));
  }
  ColorPresentation2.is = is;
})(ColorPresentation || (ColorPresentation = {}));
var FoldingRangeKind;
(function(FoldingRangeKind2) {
  FoldingRangeKind2["Comment"] = "comment";
  FoldingRangeKind2["Imports"] = "imports";
  FoldingRangeKind2["Region"] = "region";
})(FoldingRangeKind || (FoldingRangeKind = {}));
var FoldingRange;
(function(FoldingRange2) {
  function create(startLine, endLine, startCharacter, endCharacter, kind) {
    var result = {
      startLine,
      endLine
    };
    if (Is.defined(startCharacter)) {
      result.startCharacter = startCharacter;
    }
    if (Is.defined(endCharacter)) {
      result.endCharacter = endCharacter;
    }
    if (Is.defined(kind)) {
      result.kind = kind;
    }
    return result;
  }
  FoldingRange2.create = create;
  function is(value) {
    var candidate = value;
    return Is.uinteger(candidate.startLine) && Is.uinteger(candidate.startLine) && (Is.undefined(candidate.startCharacter) || Is.uinteger(candidate.startCharacter)) && (Is.undefined(candidate.endCharacter) || Is.uinteger(candidate.endCharacter)) && (Is.undefined(candidate.kind) || Is.string(candidate.kind));
  }
  FoldingRange2.is = is;
})(FoldingRange || (FoldingRange = {}));
var DiagnosticRelatedInformation;
(function(DiagnosticRelatedInformation2) {
  function create(location, message) {
    return {
      location,
      message
    };
  }
  DiagnosticRelatedInformation2.create = create;
  function is(value) {
    var candidate = value;
    return Is.defined(candidate) && Location.is(candidate.location) && Is.string(candidate.message);
  }
  DiagnosticRelatedInformation2.is = is;
})(DiagnosticRelatedInformation || (DiagnosticRelatedInformation = {}));
var DiagnosticSeverity;
(function(DiagnosticSeverity2) {
  DiagnosticSeverity2.Error = 1;
  DiagnosticSeverity2.Warning = 2;
  DiagnosticSeverity2.Information = 3;
  DiagnosticSeverity2.Hint = 4;
})(DiagnosticSeverity || (DiagnosticSeverity = {}));
var DiagnosticTag;
(function(DiagnosticTag2) {
  DiagnosticTag2.Unnecessary = 1;
  DiagnosticTag2.Deprecated = 2;
})(DiagnosticTag || (DiagnosticTag = {}));
var CodeDescription;
(function(CodeDescription2) {
  function is(value) {
    var candidate = value;
    return candidate !== void 0 && candidate !== null && Is.string(candidate.href);
  }
  CodeDescription2.is = is;
})(CodeDescription || (CodeDescription = {}));
var Diagnostic;
(function(Diagnostic2) {
  function create(range, message, severity, code, source, relatedInformation) {
    var result = { range, message };
    if (Is.defined(severity)) {
      result.severity = severity;
    }
    if (Is.defined(code)) {
      result.code = code;
    }
    if (Is.defined(source)) {
      result.source = source;
    }
    if (Is.defined(relatedInformation)) {
      result.relatedInformation = relatedInformation;
    }
    return result;
  }
  Diagnostic2.create = create;
  function is(value) {
    var _a2;
    var candidate = value;
    return Is.defined(candidate) && Range.is(candidate.range) && Is.string(candidate.message) && (Is.number(candidate.severity) || Is.undefined(candidate.severity)) && (Is.integer(candidate.code) || Is.string(candidate.code) || Is.undefined(candidate.code)) && (Is.undefined(candidate.codeDescription) || Is.string((_a2 = candidate.codeDescription) === null || _a2 === void 0 ? void 0 : _a2.href)) && (Is.string(candidate.source) || Is.undefined(candidate.source)) && (Is.undefined(candidate.relatedInformation) || Is.typedArray(candidate.relatedInformation, DiagnosticRelatedInformation.is));
  }
  Diagnostic2.is = is;
})(Diagnostic || (Diagnostic = {}));
var Command;
(function(Command2) {
  function create(title, command) {
    var args = [];
    for (var _i = 2; _i < arguments.length; _i++) {
      args[_i - 2] = arguments[_i];
    }
    var result = { title, command };
    if (Is.defined(args) && args.length > 0) {
      result.arguments = args;
    }
    return result;
  }
  Command2.create = create;
  function is(value) {
    var candidate = value;
    return Is.defined(candidate) && Is.string(candidate.title) && Is.string(candidate.command);
  }
  Command2.is = is;
})(Command || (Command = {}));
var TextEdit;
(function(TextEdit2) {
  function replace(range, newText) {
    return { range, newText };
  }
  TextEdit2.replace = replace;
  function insert(position, newText) {
    return { range: { start: position, end: position }, newText };
  }
  TextEdit2.insert = insert;
  function del(range) {
    return { range, newText: "" };
  }
  TextEdit2.del = del;
  function is(value) {
    var candidate = value;
    return Is.objectLiteral(candidate) && Is.string(candidate.newText) && Range.is(candidate.range);
  }
  TextEdit2.is = is;
})(TextEdit || (TextEdit = {}));
var ChangeAnnotation;
(function(ChangeAnnotation2) {
  function create(label, needsConfirmation, description) {
    var result = { label };
    if (needsConfirmation !== void 0) {
      result.needsConfirmation = needsConfirmation;
    }
    if (description !== void 0) {
      result.description = description;
    }
    return result;
  }
  ChangeAnnotation2.create = create;
  function is(value) {
    var candidate = value;
    return candidate !== void 0 && Is.objectLiteral(candidate) && Is.string(candidate.label) && (Is.boolean(candidate.needsConfirmation) || candidate.needsConfirmation === void 0) && (Is.string(candidate.description) || candidate.description === void 0);
  }
  ChangeAnnotation2.is = is;
})(ChangeAnnotation || (ChangeAnnotation = {}));
var ChangeAnnotationIdentifier;
(function(ChangeAnnotationIdentifier2) {
  function is(value) {
    var candidate = value;
    return typeof candidate === "string";
  }
  ChangeAnnotationIdentifier2.is = is;
})(ChangeAnnotationIdentifier || (ChangeAnnotationIdentifier = {}));
var AnnotatedTextEdit;
(function(AnnotatedTextEdit2) {
  function replace(range, newText, annotation) {
    return { range, newText, annotationId: annotation };
  }
  AnnotatedTextEdit2.replace = replace;
  function insert(position, newText, annotation) {
    return { range: { start: position, end: position }, newText, annotationId: annotation };
  }
  AnnotatedTextEdit2.insert = insert;
  function del(range, annotation) {
    return { range, newText: "", annotationId: annotation };
  }
  AnnotatedTextEdit2.del = del;
  function is(value) {
    var candidate = value;
    return TextEdit.is(candidate) && (ChangeAnnotation.is(candidate.annotationId) || ChangeAnnotationIdentifier.is(candidate.annotationId));
  }
  AnnotatedTextEdit2.is = is;
})(AnnotatedTextEdit || (AnnotatedTextEdit = {}));
var TextDocumentEdit;
(function(TextDocumentEdit2) {
  function create(textDocument, edits) {
    return { textDocument, edits };
  }
  TextDocumentEdit2.create = create;
  function is(value) {
    var candidate = value;
    return Is.defined(candidate) && OptionalVersionedTextDocumentIdentifier.is(candidate.textDocument) && Array.isArray(candidate.edits);
  }
  TextDocumentEdit2.is = is;
})(TextDocumentEdit || (TextDocumentEdit = {}));
var CreateFile;
(function(CreateFile2) {
  function create(uri, options, annotation) {
    var result = {
      kind: "create",
      uri
    };
    if (options !== void 0 && (options.overwrite !== void 0 || options.ignoreIfExists !== void 0)) {
      result.options = options;
    }
    if (annotation !== void 0) {
      result.annotationId = annotation;
    }
    return result;
  }
  CreateFile2.create = create;
  function is(value) {
    var candidate = value;
    return candidate && candidate.kind === "create" && Is.string(candidate.uri) && (candidate.options === void 0 || (candidate.options.overwrite === void 0 || Is.boolean(candidate.options.overwrite)) && (candidate.options.ignoreIfExists === void 0 || Is.boolean(candidate.options.ignoreIfExists))) && (candidate.annotationId === void 0 || ChangeAnnotationIdentifier.is(candidate.annotationId));
  }
  CreateFile2.is = is;
})(CreateFile || (CreateFile = {}));
var RenameFile;
(function(RenameFile2) {
  function create(oldUri, newUri, options, annotation) {
    var result = {
      kind: "rename",
      oldUri,
      newUri
    };
    if (options !== void 0 && (options.overwrite !== void 0 || options.ignoreIfExists !== void 0)) {
      result.options = options;
    }
    if (annotation !== void 0) {
      result.annotationId = annotation;
    }
    return result;
  }
  RenameFile2.create = create;
  function is(value) {
    var candidate = value;
    return candidate && candidate.kind === "rename" && Is.string(candidate.oldUri) && Is.string(candidate.newUri) && (candidate.options === void 0 || (candidate.options.overwrite === void 0 || Is.boolean(candidate.options.overwrite)) && (candidate.options.ignoreIfExists === void 0 || Is.boolean(candidate.options.ignoreIfExists))) && (candidate.annotationId === void 0 || ChangeAnnotationIdentifier.is(candidate.annotationId));
  }
  RenameFile2.is = is;
})(RenameFile || (RenameFile = {}));
var DeleteFile;
(function(DeleteFile2) {
  function create(uri, options, annotation) {
    var result = {
      kind: "delete",
      uri
    };
    if (options !== void 0 && (options.recursive !== void 0 || options.ignoreIfNotExists !== void 0)) {
      result.options = options;
    }
    if (annotation !== void 0) {
      result.annotationId = annotation;
    }
    return result;
  }
  DeleteFile2.create = create;
  function is(value) {
    var candidate = value;
    return candidate && candidate.kind === "delete" && Is.string(candidate.uri) && (candidate.options === void 0 || (candidate.options.recursive === void 0 || Is.boolean(candidate.options.recursive)) && (candidate.options.ignoreIfNotExists === void 0 || Is.boolean(candidate.options.ignoreIfNotExists))) && (candidate.annotationId === void 0 || ChangeAnnotationIdentifier.is(candidate.annotationId));
  }
  DeleteFile2.is = is;
})(DeleteFile || (DeleteFile = {}));
var WorkspaceEdit;
(function(WorkspaceEdit2) {
  function is(value) {
    var candidate = value;
    return candidate && (candidate.changes !== void 0 || candidate.documentChanges !== void 0) && (candidate.documentChanges === void 0 || candidate.documentChanges.every(function(change) {
      if (Is.string(change.kind)) {
        return CreateFile.is(change) || RenameFile.is(change) || DeleteFile.is(change);
      } else {
        return TextDocumentEdit.is(change);
      }
    }));
  }
  WorkspaceEdit2.is = is;
})(WorkspaceEdit || (WorkspaceEdit = {}));
var TextEditChangeImpl = function() {
  function TextEditChangeImpl2(edits, changeAnnotations) {
    this.edits = edits;
    this.changeAnnotations = changeAnnotations;
  }
  TextEditChangeImpl2.prototype.insert = function(position, newText, annotation) {
    var edit;
    var id;
    if (annotation === void 0) {
      edit = TextEdit.insert(position, newText);
    } else if (ChangeAnnotationIdentifier.is(annotation)) {
      id = annotation;
      edit = AnnotatedTextEdit.insert(position, newText, annotation);
    } else {
      this.assertChangeAnnotations(this.changeAnnotations);
      id = this.changeAnnotations.manage(annotation);
      edit = AnnotatedTextEdit.insert(position, newText, id);
    }
    this.edits.push(edit);
    if (id !== void 0) {
      return id;
    }
  };
  TextEditChangeImpl2.prototype.replace = function(range, newText, annotation) {
    var edit;
    var id;
    if (annotation === void 0) {
      edit = TextEdit.replace(range, newText);
    } else if (ChangeAnnotationIdentifier.is(annotation)) {
      id = annotation;
      edit = AnnotatedTextEdit.replace(range, newText, annotation);
    } else {
      this.assertChangeAnnotations(this.changeAnnotations);
      id = this.changeAnnotations.manage(annotation);
      edit = AnnotatedTextEdit.replace(range, newText, id);
    }
    this.edits.push(edit);
    if (id !== void 0) {
      return id;
    }
  };
  TextEditChangeImpl2.prototype.delete = function(range, annotation) {
    var edit;
    var id;
    if (annotation === void 0) {
      edit = TextEdit.del(range);
    } else if (ChangeAnnotationIdentifier.is(annotation)) {
      id = annotation;
      edit = AnnotatedTextEdit.del(range, annotation);
    } else {
      this.assertChangeAnnotations(this.changeAnnotations);
      id = this.changeAnnotations.manage(annotation);
      edit = AnnotatedTextEdit.del(range, id);
    }
    this.edits.push(edit);
    if (id !== void 0) {
      return id;
    }
  };
  TextEditChangeImpl2.prototype.add = function(edit) {
    this.edits.push(edit);
  };
  TextEditChangeImpl2.prototype.all = function() {
    return this.edits;
  };
  TextEditChangeImpl2.prototype.clear = function() {
    this.edits.splice(0, this.edits.length);
  };
  TextEditChangeImpl2.prototype.assertChangeAnnotations = function(value) {
    if (value === void 0) {
      throw new Error("Text edit change is not configured to manage change annotations.");
    }
  };
  return TextEditChangeImpl2;
}();
var ChangeAnnotations = function() {
  function ChangeAnnotations2(annotations) {
    this._annotations = annotations === void 0 ? /* @__PURE__ */ Object.create(null) : annotations;
    this._counter = 0;
    this._size = 0;
  }
  ChangeAnnotations2.prototype.all = function() {
    return this._annotations;
  };
  Object.defineProperty(ChangeAnnotations2.prototype, "size", {
    get: function() {
      return this._size;
    },
    enumerable: false,
    configurable: true
  });
  ChangeAnnotations2.prototype.manage = function(idOrAnnotation, annotation) {
    var id;
    if (ChangeAnnotationIdentifier.is(idOrAnnotation)) {
      id = idOrAnnotation;
    } else {
      id = this.nextId();
      annotation = idOrAnnotation;
    }
    if (this._annotations[id] !== void 0) {
      throw new Error("Id " + id + " is already in use.");
    }
    if (annotation === void 0) {
      throw new Error("No annotation provided for id " + id);
    }
    this._annotations[id] = annotation;
    this._size++;
    return id;
  };
  ChangeAnnotations2.prototype.nextId = function() {
    this._counter++;
    return this._counter.toString();
  };
  return ChangeAnnotations2;
}();
var WorkspaceChange = function() {
  function WorkspaceChange2(workspaceEdit) {
    var _this = this;
    this._textEditChanges = /* @__PURE__ */ Object.create(null);
    if (workspaceEdit !== void 0) {
      this._workspaceEdit = workspaceEdit;
      if (workspaceEdit.documentChanges) {
        this._changeAnnotations = new ChangeAnnotations(workspaceEdit.changeAnnotations);
        workspaceEdit.changeAnnotations = this._changeAnnotations.all();
        workspaceEdit.documentChanges.forEach(function(change) {
          if (TextDocumentEdit.is(change)) {
            var textEditChange = new TextEditChangeImpl(change.edits, _this._changeAnnotations);
            _this._textEditChanges[change.textDocument.uri] = textEditChange;
          }
        });
      } else if (workspaceEdit.changes) {
        Object.keys(workspaceEdit.changes).forEach(function(key) {
          var textEditChange = new TextEditChangeImpl(workspaceEdit.changes[key]);
          _this._textEditChanges[key] = textEditChange;
        });
      }
    } else {
      this._workspaceEdit = {};
    }
  }
  Object.defineProperty(WorkspaceChange2.prototype, "edit", {
    get: function() {
      this.initDocumentChanges();
      if (this._changeAnnotations !== void 0) {
        if (this._changeAnnotations.size === 0) {
          this._workspaceEdit.changeAnnotations = void 0;
        } else {
          this._workspaceEdit.changeAnnotations = this._changeAnnotations.all();
        }
      }
      return this._workspaceEdit;
    },
    enumerable: false,
    configurable: true
  });
  WorkspaceChange2.prototype.getTextEditChange = function(key) {
    if (OptionalVersionedTextDocumentIdentifier.is(key)) {
      this.initDocumentChanges();
      if (this._workspaceEdit.documentChanges === void 0) {
        throw new Error("Workspace edit is not configured for document changes.");
      }
      var textDocument = { uri: key.uri, version: key.version };
      var result = this._textEditChanges[textDocument.uri];
      if (!result) {
        var edits = [];
        var textDocumentEdit = {
          textDocument,
          edits
        };
        this._workspaceEdit.documentChanges.push(textDocumentEdit);
        result = new TextEditChangeImpl(edits, this._changeAnnotations);
        this._textEditChanges[textDocument.uri] = result;
      }
      return result;
    } else {
      this.initChanges();
      if (this._workspaceEdit.changes === void 0) {
        throw new Error("Workspace edit is not configured for normal text edit changes.");
      }
      var result = this._textEditChanges[key];
      if (!result) {
        var edits = [];
        this._workspaceEdit.changes[key] = edits;
        result = new TextEditChangeImpl(edits);
        this._textEditChanges[key] = result;
      }
      return result;
    }
  };
  WorkspaceChange2.prototype.initDocumentChanges = function() {
    if (this._workspaceEdit.documentChanges === void 0 && this._workspaceEdit.changes === void 0) {
      this._changeAnnotations = new ChangeAnnotations();
      this._workspaceEdit.documentChanges = [];
      this._workspaceEdit.changeAnnotations = this._changeAnnotations.all();
    }
  };
  WorkspaceChange2.prototype.initChanges = function() {
    if (this._workspaceEdit.documentChanges === void 0 && this._workspaceEdit.changes === void 0) {
      this._workspaceEdit.changes = /* @__PURE__ */ Object.create(null);
    }
  };
  WorkspaceChange2.prototype.createFile = function(uri, optionsOrAnnotation, options) {
    this.initDocumentChanges();
    if (this._workspaceEdit.documentChanges === void 0) {
      throw new Error("Workspace edit is not configured for document changes.");
    }
    var annotation;
    if (ChangeAnnotation.is(optionsOrAnnotation) || ChangeAnnotationIdentifier.is(optionsOrAnnotation)) {
      annotation = optionsOrAnnotation;
    } else {
      options = optionsOrAnnotation;
    }
    var operation;
    var id;
    if (annotation === void 0) {
      operation = CreateFile.create(uri, options);
    } else {
      id = ChangeAnnotationIdentifier.is(annotation) ? annotation : this._changeAnnotations.manage(annotation);
      operation = CreateFile.create(uri, options, id);
    }
    this._workspaceEdit.documentChanges.push(operation);
    if (id !== void 0) {
      return id;
    }
  };
  WorkspaceChange2.prototype.renameFile = function(oldUri, newUri, optionsOrAnnotation, options) {
    this.initDocumentChanges();
    if (this._workspaceEdit.documentChanges === void 0) {
      throw new Error("Workspace edit is not configured for document changes.");
    }
    var annotation;
    if (ChangeAnnotation.is(optionsOrAnnotation) || ChangeAnnotationIdentifier.is(optionsOrAnnotation)) {
      annotation = optionsOrAnnotation;
    } else {
      options = optionsOrAnnotation;
    }
    var operation;
    var id;
    if (annotation === void 0) {
      operation = RenameFile.create(oldUri, newUri, options);
    } else {
      id = ChangeAnnotationIdentifier.is(annotation) ? annotation : this._changeAnnotations.manage(annotation);
      operation = RenameFile.create(oldUri, newUri, options, id);
    }
    this._workspaceEdit.documentChanges.push(operation);
    if (id !== void 0) {
      return id;
    }
  };
  WorkspaceChange2.prototype.deleteFile = function(uri, optionsOrAnnotation, options) {
    this.initDocumentChanges();
    if (this._workspaceEdit.documentChanges === void 0) {
      throw new Error("Workspace edit is not configured for document changes.");
    }
    var annotation;
    if (ChangeAnnotation.is(optionsOrAnnotation) || ChangeAnnotationIdentifier.is(optionsOrAnnotation)) {
      annotation = optionsOrAnnotation;
    } else {
      options = optionsOrAnnotation;
    }
    var operation;
    var id;
    if (annotation === void 0) {
      operation = DeleteFile.create(uri, options);
    } else {
      id = ChangeAnnotationIdentifier.is(annotation) ? annotation : this._changeAnnotations.manage(annotation);
      operation = DeleteFile.create(uri, options, id);
    }
    this._workspaceEdit.documentChanges.push(operation);
    if (id !== void 0) {
      return id;
    }
  };
  return WorkspaceChange2;
}();
var TextDocumentIdentifier;
(function(TextDocumentIdentifier2) {
  function create(uri) {
    return { uri };
  }
  TextDocumentIdentifier2.create = create;
  function is(value) {
    var candidate = value;
    return Is.defined(candidate) && Is.string(candidate.uri);
  }
  TextDocumentIdentifier2.is = is;
})(TextDocumentIdentifier || (TextDocumentIdentifier = {}));
var VersionedTextDocumentIdentifier;
(function(VersionedTextDocumentIdentifier2) {
  function create(uri, version) {
    return { uri, version };
  }
  VersionedTextDocumentIdentifier2.create = create;
  function is(value) {
    var candidate = value;
    return Is.defined(candidate) && Is.string(candidate.uri) && Is.integer(candidate.version);
  }
  VersionedTextDocumentIdentifier2.is = is;
})(VersionedTextDocumentIdentifier || (VersionedTextDocumentIdentifier = {}));
var OptionalVersionedTextDocumentIdentifier;
(function(OptionalVersionedTextDocumentIdentifier2) {
  function create(uri, version) {
    return { uri, version };
  }
  OptionalVersionedTextDocumentIdentifier2.create = create;
  function is(value) {
    var candidate = value;
    return Is.defined(candidate) && Is.string(candidate.uri) && (candidate.version === null || Is.integer(candidate.version));
  }
  OptionalVersionedTextDocumentIdentifier2.is = is;
})(OptionalVersionedTextDocumentIdentifier || (OptionalVersionedTextDocumentIdentifier = {}));
var TextDocumentItem;
(function(TextDocumentItem2) {
  function create(uri, languageId, version, text) {
    return { uri, languageId, version, text };
  }
  TextDocumentItem2.create = create;
  function is(value) {
    var candidate = value;
    return Is.defined(candidate) && Is.string(candidate.uri) && Is.string(candidate.languageId) && Is.integer(candidate.version) && Is.string(candidate.text);
  }
  TextDocumentItem2.is = is;
})(TextDocumentItem || (TextDocumentItem = {}));
var MarkupKind;
(function(MarkupKind2) {
  MarkupKind2.PlainText = "plaintext";
  MarkupKind2.Markdown = "markdown";
})(MarkupKind || (MarkupKind = {}));
(function(MarkupKind2) {
  function is(value) {
    var candidate = value;
    return candidate === MarkupKind2.PlainText || candidate === MarkupKind2.Markdown;
  }
  MarkupKind2.is = is;
})(MarkupKind || (MarkupKind = {}));
var MarkupContent;
(function(MarkupContent2) {
  function is(value) {
    var candidate = value;
    return Is.objectLiteral(value) && MarkupKind.is(candidate.kind) && Is.string(candidate.value);
  }
  MarkupContent2.is = is;
})(MarkupContent || (MarkupContent = {}));
var CompletionItemKind;
(function(CompletionItemKind2) {
  CompletionItemKind2.Text = 1;
  CompletionItemKind2.Method = 2;
  CompletionItemKind2.Function = 3;
  CompletionItemKind2.Constructor = 4;
  CompletionItemKind2.Field = 5;
  CompletionItemKind2.Variable = 6;
  CompletionItemKind2.Class = 7;
  CompletionItemKind2.Interface = 8;
  CompletionItemKind2.Module = 9;
  CompletionItemKind2.Property = 10;
  CompletionItemKind2.Unit = 11;
  CompletionItemKind2.Value = 12;
  CompletionItemKind2.Enum = 13;
  CompletionItemKind2.Keyword = 14;
  CompletionItemKind2.Snippet = 15;
  CompletionItemKind2.Color = 16;
  CompletionItemKind2.File = 17;
  CompletionItemKind2.Reference = 18;
  CompletionItemKind2.Folder = 19;
  CompletionItemKind2.EnumMember = 20;
  CompletionItemKind2.Constant = 21;
  CompletionItemKind2.Struct = 22;
  CompletionItemKind2.Event = 23;
  CompletionItemKind2.Operator = 24;
  CompletionItemKind2.TypeParameter = 25;
})(CompletionItemKind || (CompletionItemKind = {}));
var InsertTextFormat;
(function(InsertTextFormat2) {
  InsertTextFormat2.PlainText = 1;
  InsertTextFormat2.Snippet = 2;
})(InsertTextFormat || (InsertTextFormat = {}));
var CompletionItemTag;
(function(CompletionItemTag2) {
  CompletionItemTag2.Deprecated = 1;
})(CompletionItemTag || (CompletionItemTag = {}));
var InsertReplaceEdit;
(function(InsertReplaceEdit2) {
  function create(newText, insert, replace) {
    return { newText, insert, replace };
  }
  InsertReplaceEdit2.create = create;
  function is(value) {
    var candidate = value;
    return candidate && Is.string(candidate.newText) && Range.is(candidate.insert) && Range.is(candidate.replace);
  }
  InsertReplaceEdit2.is = is;
})(InsertReplaceEdit || (InsertReplaceEdit = {}));
var InsertTextMode;
(function(InsertTextMode2) {
  InsertTextMode2.asIs = 1;
  InsertTextMode2.adjustIndentation = 2;
})(InsertTextMode || (InsertTextMode = {}));
var CompletionItem;
(function(CompletionItem2) {
  function create(label) {
    return { label };
  }
  CompletionItem2.create = create;
})(CompletionItem || (CompletionItem = {}));
var CompletionList;
(function(CompletionList2) {
  function create(items, isIncomplete) {
    return { items: items ? items : [], isIncomplete: !!isIncomplete };
  }
  CompletionList2.create = create;
})(CompletionList || (CompletionList = {}));
var MarkedString;
(function(MarkedString2) {
  function fromPlainText(plainText) {
    return plainText.replace(/[\\`*_{}[\]()#+\-.!]/g, "\\$&");
  }
  MarkedString2.fromPlainText = fromPlainText;
  function is(value) {
    var candidate = value;
    return Is.string(candidate) || Is.objectLiteral(candidate) && Is.string(candidate.language) && Is.string(candidate.value);
  }
  MarkedString2.is = is;
})(MarkedString || (MarkedString = {}));
var Hover;
(function(Hover2) {
  function is(value) {
    var candidate = value;
    return !!candidate && Is.objectLiteral(candidate) && (MarkupContent.is(candidate.contents) || MarkedString.is(candidate.contents) || Is.typedArray(candidate.contents, MarkedString.is)) && (value.range === void 0 || Range.is(value.range));
  }
  Hover2.is = is;
})(Hover || (Hover = {}));
var ParameterInformation;
(function(ParameterInformation2) {
  function create(label, documentation) {
    return documentation ? { label, documentation } : { label };
  }
  ParameterInformation2.create = create;
})(ParameterInformation || (ParameterInformation = {}));
var SignatureInformation;
(function(SignatureInformation2) {
  function create(label, documentation) {
    var parameters = [];
    for (var _i = 2; _i < arguments.length; _i++) {
      parameters[_i - 2] = arguments[_i];
    }
    var result = { label };
    if (Is.defined(documentation)) {
      result.documentation = documentation;
    }
    if (Is.defined(parameters)) {
      result.parameters = parameters;
    } else {
      result.parameters = [];
    }
    return result;
  }
  SignatureInformation2.create = create;
})(SignatureInformation || (SignatureInformation = {}));
var DocumentHighlightKind;
(function(DocumentHighlightKind2) {
  DocumentHighlightKind2.Text = 1;
  DocumentHighlightKind2.Read = 2;
  DocumentHighlightKind2.Write = 3;
})(DocumentHighlightKind || (DocumentHighlightKind = {}));
var DocumentHighlight;
(function(DocumentHighlight2) {
  function create(range, kind) {
    var result = { range };
    if (Is.number(kind)) {
      result.kind = kind;
    }
    return result;
  }
  DocumentHighlight2.create = create;
})(DocumentHighlight || (DocumentHighlight = {}));
var SymbolKind;
(function(SymbolKind2) {
  SymbolKind2.File = 1;
  SymbolKind2.Module = 2;
  SymbolKind2.Namespace = 3;
  SymbolKind2.Package = 4;
  SymbolKind2.Class = 5;
  SymbolKind2.Method = 6;
  SymbolKind2.Property = 7;
  SymbolKind2.Field = 8;
  SymbolKind2.Constructor = 9;
  SymbolKind2.Enum = 10;
  SymbolKind2.Interface = 11;
  SymbolKind2.Function = 12;
  SymbolKind2.Variable = 13;
  SymbolKind2.Constant = 14;
  SymbolKind2.String = 15;
  SymbolKind2.Number = 16;
  SymbolKind2.Boolean = 17;
  SymbolKind2.Array = 18;
  SymbolKind2.Object = 19;
  SymbolKind2.Key = 20;
  SymbolKind2.Null = 21;
  SymbolKind2.EnumMember = 22;
  SymbolKind2.Struct = 23;
  SymbolKind2.Event = 24;
  SymbolKind2.Operator = 25;
  SymbolKind2.TypeParameter = 26;
})(SymbolKind || (SymbolKind = {}));
var SymbolTag;
(function(SymbolTag2) {
  SymbolTag2.Deprecated = 1;
})(SymbolTag || (SymbolTag = {}));
var SymbolInformation;
(function(SymbolInformation2) {
  function create(name, kind, range, uri, containerName) {
    var result = {
      name,
      kind,
      location: { uri, range }
    };
    if (containerName) {
      result.containerName = containerName;
    }
    return result;
  }
  SymbolInformation2.create = create;
})(SymbolInformation || (SymbolInformation = {}));
var DocumentSymbol;
(function(DocumentSymbol2) {
  function create(name, detail, kind, range, selectionRange, children) {
    var result = {
      name,
      detail,
      kind,
      range,
      selectionRange
    };
    if (children !== void 0) {
      result.children = children;
    }
    return result;
  }
  DocumentSymbol2.create = create;
  function is(value) {
    var candidate = value;
    return candidate && Is.string(candidate.name) && Is.number(candidate.kind) && Range.is(candidate.range) && Range.is(candidate.selectionRange) && (candidate.detail === void 0 || Is.string(candidate.detail)) && (candidate.deprecated === void 0 || Is.boolean(candidate.deprecated)) && (candidate.children === void 0 || Array.isArray(candidate.children)) && (candidate.tags === void 0 || Array.isArray(candidate.tags));
  }
  DocumentSymbol2.is = is;
})(DocumentSymbol || (DocumentSymbol = {}));
var CodeActionKind;
(function(CodeActionKind2) {
  CodeActionKind2.Empty = "";
  CodeActionKind2.QuickFix = "quickfix";
  CodeActionKind2.Refactor = "refactor";
  CodeActionKind2.RefactorExtract = "refactor.extract";
  CodeActionKind2.RefactorInline = "refactor.inline";
  CodeActionKind2.RefactorRewrite = "refactor.rewrite";
  CodeActionKind2.Source = "source";
  CodeActionKind2.SourceOrganizeImports = "source.organizeImports";
  CodeActionKind2.SourceFixAll = "source.fixAll";
})(CodeActionKind || (CodeActionKind = {}));
var CodeActionContext;
(function(CodeActionContext2) {
  function create(diagnostics, only) {
    var result = { diagnostics };
    if (only !== void 0 && only !== null) {
      result.only = only;
    }
    return result;
  }
  CodeActionContext2.create = create;
  function is(value) {
    var candidate = value;
    return Is.defined(candidate) && Is.typedArray(candidate.diagnostics, Diagnostic.is) && (candidate.only === void 0 || Is.typedArray(candidate.only, Is.string));
  }
  CodeActionContext2.is = is;
})(CodeActionContext || (CodeActionContext = {}));
var CodeAction;
(function(CodeAction2) {
  function create(title, kindOrCommandOrEdit, kind) {
    var result = { title };
    var checkKind = true;
    if (typeof kindOrCommandOrEdit === "string") {
      checkKind = false;
      result.kind = kindOrCommandOrEdit;
    } else if (Command.is(kindOrCommandOrEdit)) {
      result.command = kindOrCommandOrEdit;
    } else {
      result.edit = kindOrCommandOrEdit;
    }
    if (checkKind && kind !== void 0) {
      result.kind = kind;
    }
    return result;
  }
  CodeAction2.create = create;
  function is(value) {
    var candidate = value;
    return candidate && Is.string(candidate.title) && (candidate.diagnostics === void 0 || Is.typedArray(candidate.diagnostics, Diagnostic.is)) && (candidate.kind === void 0 || Is.string(candidate.kind)) && (candidate.edit !== void 0 || candidate.command !== void 0) && (candidate.command === void 0 || Command.is(candidate.command)) && (candidate.isPreferred === void 0 || Is.boolean(candidate.isPreferred)) && (candidate.edit === void 0 || WorkspaceEdit.is(candidate.edit));
  }
  CodeAction2.is = is;
})(CodeAction || (CodeAction = {}));
var CodeLens;
(function(CodeLens2) {
  function create(range, data) {
    var result = { range };
    if (Is.defined(data)) {
      result.data = data;
    }
    return result;
  }
  CodeLens2.create = create;
  function is(value) {
    var candidate = value;
    return Is.defined(candidate) && Range.is(candidate.range) && (Is.undefined(candidate.command) || Command.is(candidate.command));
  }
  CodeLens2.is = is;
})(CodeLens || (CodeLens = {}));
var FormattingOptions;
(function(FormattingOptions2) {
  function create(tabSize, insertSpaces) {
    return { tabSize, insertSpaces };
  }
  FormattingOptions2.create = create;
  function is(value) {
    var candidate = value;
    return Is.defined(candidate) && Is.uinteger(candidate.tabSize) && Is.boolean(candidate.insertSpaces);
  }
  FormattingOptions2.is = is;
})(FormattingOptions || (FormattingOptions = {}));
var DocumentLink;
(function(DocumentLink2) {
  function create(range, target, data) {
    return { range, target, data };
  }
  DocumentLink2.create = create;
  function is(value) {
    var candidate = value;
    return Is.defined(candidate) && Range.is(candidate.range) && (Is.undefined(candidate.target) || Is.string(candidate.target));
  }
  DocumentLink2.is = is;
})(DocumentLink || (DocumentLink = {}));
var SelectionRange;
(function(SelectionRange2) {
  function create(range, parent) {
    return { range, parent };
  }
  SelectionRange2.create = create;
  function is(value) {
    var candidate = value;
    return candidate !== void 0 && Range.is(candidate.range) && (candidate.parent === void 0 || SelectionRange2.is(candidate.parent));
  }
  SelectionRange2.is = is;
})(SelectionRange || (SelectionRange = {}));
var TextDocument;
(function(TextDocument3) {
  function create(uri, languageId, version, content) {
    return new FullTextDocument(uri, languageId, version, content);
  }
  TextDocument3.create = create;
  function is(value) {
    var candidate = value;
    return Is.defined(candidate) && Is.string(candidate.uri) && (Is.undefined(candidate.languageId) || Is.string(candidate.languageId)) && Is.uinteger(candidate.lineCount) && Is.func(candidate.getText) && Is.func(candidate.positionAt) && Is.func(candidate.offsetAt) ? true : false;
  }
  TextDocument3.is = is;
  function applyEdits(document, edits) {
    var text = document.getText();
    var sortedEdits = mergeSort2(edits, function(a2, b) {
      var diff = a2.range.start.line - b.range.start.line;
      if (diff === 0) {
        return a2.range.start.character - b.range.start.character;
      }
      return diff;
    });
    var lastModifiedOffset = text.length;
    for (var i = sortedEdits.length - 1; i >= 0; i--) {
      var e = sortedEdits[i];
      var startOffset = document.offsetAt(e.range.start);
      var endOffset = document.offsetAt(e.range.end);
      if (endOffset <= lastModifiedOffset) {
        text = text.substring(0, startOffset) + e.newText + text.substring(endOffset, text.length);
      } else {
        throw new Error("Overlapping edit");
      }
      lastModifiedOffset = startOffset;
    }
    return text;
  }
  TextDocument3.applyEdits = applyEdits;
  function mergeSort2(data, compare) {
    if (data.length <= 1) {
      return data;
    }
    var p = data.length / 2 | 0;
    var left = data.slice(0, p);
    var right = data.slice(p);
    mergeSort2(left, compare);
    mergeSort2(right, compare);
    var leftIdx = 0;
    var rightIdx = 0;
    var i = 0;
    while (leftIdx < left.length && rightIdx < right.length) {
      var ret = compare(left[leftIdx], right[rightIdx]);
      if (ret <= 0) {
        data[i++] = left[leftIdx++];
      } else {
        data[i++] = right[rightIdx++];
      }
    }
    while (leftIdx < left.length) {
      data[i++] = left[leftIdx++];
    }
    while (rightIdx < right.length) {
      data[i++] = right[rightIdx++];
    }
    return data;
  }
})(TextDocument || (TextDocument = {}));
var FullTextDocument = function() {
  function FullTextDocument3(uri, languageId, version, content) {
    this._uri = uri;
    this._languageId = languageId;
    this._version = version;
    this._content = content;
    this._lineOffsets = void 0;
  }
  Object.defineProperty(FullTextDocument3.prototype, "uri", {
    get: function() {
      return this._uri;
    },
    enumerable: false,
    configurable: true
  });
  Object.defineProperty(FullTextDocument3.prototype, "languageId", {
    get: function() {
      return this._languageId;
    },
    enumerable: false,
    configurable: true
  });
  Object.defineProperty(FullTextDocument3.prototype, "version", {
    get: function() {
      return this._version;
    },
    enumerable: false,
    configurable: true
  });
  FullTextDocument3.prototype.getText = function(range) {
    if (range) {
      var start = this.offsetAt(range.start);
      var end = this.offsetAt(range.end);
      return this._content.substring(start, end);
    }
    return this._content;
  };
  FullTextDocument3.prototype.update = function(event, version) {
    this._content = event.text;
    this._version = version;
    this._lineOffsets = void 0;
  };
  FullTextDocument3.prototype.getLineOffsets = function() {
    if (this._lineOffsets === void 0) {
      var lineOffsets = [];
      var text = this._content;
      var isLineStart = true;
      for (var i = 0; i < text.length; i++) {
        if (isLineStart) {
          lineOffsets.push(i);
          isLineStart = false;
        }
        var ch = text.charAt(i);
        isLineStart = ch === "\r" || ch === "\n";
        if (ch === "\r" && i + 1 < text.length && text.charAt(i + 1) === "\n") {
          i++;
        }
      }
      if (isLineStart && text.length > 0) {
        lineOffsets.push(text.length);
      }
      this._lineOffsets = lineOffsets;
    }
    return this._lineOffsets;
  };
  FullTextDocument3.prototype.positionAt = function(offset) {
    offset = Math.max(Math.min(offset, this._content.length), 0);
    var lineOffsets = this.getLineOffsets();
    var low = 0, high = lineOffsets.length;
    if (high === 0) {
      return Position.create(0, offset);
    }
    while (low < high) {
      var mid = Math.floor((low + high) / 2);
      if (lineOffsets[mid] > offset) {
        high = mid;
      } else {
        low = mid + 1;
      }
    }
    var line = low - 1;
    return Position.create(line, offset - lineOffsets[line]);
  };
  FullTextDocument3.prototype.offsetAt = function(position) {
    var lineOffsets = this.getLineOffsets();
    if (position.line >= lineOffsets.length) {
      return this._content.length;
    } else if (position.line < 0) {
      return 0;
    }
    var lineOffset = lineOffsets[position.line];
    var nextLineOffset = position.line + 1 < lineOffsets.length ? lineOffsets[position.line + 1] : this._content.length;
    return Math.max(Math.min(lineOffset + position.character, nextLineOffset), lineOffset);
  };
  Object.defineProperty(FullTextDocument3.prototype, "lineCount", {
    get: function() {
      return this.getLineOffsets().length;
    },
    enumerable: false,
    configurable: true
  });
  return FullTextDocument3;
}();
var Is;
(function(Is2) {
  var toString = Object.prototype.toString;
  function defined(value) {
    return typeof value !== "undefined";
  }
  Is2.defined = defined;
  function undefined2(value) {
    return typeof value === "undefined";
  }
  Is2.undefined = undefined2;
  function boolean(value) {
    return value === true || value === false;
  }
  Is2.boolean = boolean;
  function string(value) {
    return toString.call(value) === "[object String]";
  }
  Is2.string = string;
  function number(value) {
    return toString.call(value) === "[object Number]";
  }
  Is2.number = number;
  function numberRange(value, min, max) {
    return toString.call(value) === "[object Number]" && min <= value && value <= max;
  }
  Is2.numberRange = numberRange;
  function integer2(value) {
    return toString.call(value) === "[object Number]" && -2147483648 <= value && value <= 2147483647;
  }
  Is2.integer = integer2;
  function uinteger2(value) {
    return toString.call(value) === "[object Number]" && 0 <= value && value <= 2147483647;
  }
  Is2.uinteger = uinteger2;
  function func(value) {
    return toString.call(value) === "[object Function]";
  }
  Is2.func = func;
  function objectLiteral(value) {
    return value !== null && typeof value === "object";
  }
  Is2.objectLiteral = objectLiteral;
  function typedArray(value, check) {
    return Array.isArray(value) && value.every(check);
  }
  Is2.typedArray = typedArray;
})(Is || (Is = {}));

// node_modules/vscode-languageserver-textdocument/lib/esm/main.js
var FullTextDocument2 = class {
  constructor(uri, languageId, version, content) {
    this._uri = uri;
    this._languageId = languageId;
    this._version = version;
    this._content = content;
    this._lineOffsets = void 0;
  }
  get uri() {
    return this._uri;
  }
  get languageId() {
    return this._languageId;
  }
  get version() {
    return this._version;
  }
  getText(range) {
    if (range) {
      const start = this.offsetAt(range.start);
      const end = this.offsetAt(range.end);
      return this._content.substring(start, end);
    }
    return this._content;
  }
  update(changes, version) {
    for (let change of changes) {
      if (FullTextDocument2.isIncremental(change)) {
        const range = getWellformedRange(change.range);
        const startOffset = this.offsetAt(range.start);
        const endOffset = this.offsetAt(range.end);
        this._content = this._content.substring(0, startOffset) + change.text + this._content.substring(endOffset, this._content.length);
        const startLine = Math.max(range.start.line, 0);
        const endLine = Math.max(range.end.line, 0);
        let lineOffsets = this._lineOffsets;
        const addedLineOffsets = computeLineOffsets(change.text, false, startOffset);
        if (endLine - startLine === addedLineOffsets.length) {
          for (let i = 0, len = addedLineOffsets.length; i < len; i++) {
            lineOffsets[i + startLine + 1] = addedLineOffsets[i];
          }
        } else {
          if (addedLineOffsets.length < 1e4) {
            lineOffsets.splice(startLine + 1, endLine - startLine, ...addedLineOffsets);
          } else {
            this._lineOffsets = lineOffsets = lineOffsets.slice(0, startLine + 1).concat(addedLineOffsets, lineOffsets.slice(endLine + 1));
          }
        }
        const diff = change.text.length - (endOffset - startOffset);
        if (diff !== 0) {
          for (let i = startLine + 1 + addedLineOffsets.length, len = lineOffsets.length; i < len; i++) {
            lineOffsets[i] = lineOffsets[i] + diff;
          }
        }
      } else if (FullTextDocument2.isFull(change)) {
        this._content = change.text;
        this._lineOffsets = void 0;
      } else {
        throw new Error("Unknown change event received");
      }
    }
    this._version = version;
  }
  getLineOffsets() {
    if (this._lineOffsets === void 0) {
      this._lineOffsets = computeLineOffsets(this._content, true);
    }
    return this._lineOffsets;
  }
  positionAt(offset) {
    offset = Math.max(Math.min(offset, this._content.length), 0);
    let lineOffsets = this.getLineOffsets();
    let low = 0, high = lineOffsets.length;
    if (high === 0) {
      return { line: 0, character: offset };
    }
    while (low < high) {
      let mid = Math.floor((low + high) / 2);
      if (lineOffsets[mid] > offset) {
        high = mid;
      } else {
        low = mid + 1;
      }
    }
    let line = low - 1;
    return { line, character: offset - lineOffsets[line] };
  }
  offsetAt(position) {
    let lineOffsets = this.getLineOffsets();
    if (position.line >= lineOffsets.length) {
      return this._content.length;
    } else if (position.line < 0) {
      return 0;
    }
    let lineOffset = lineOffsets[position.line];
    let nextLineOffset = position.line + 1 < lineOffsets.length ? lineOffsets[position.line + 1] : this._content.length;
    return Math.max(Math.min(lineOffset + position.character, nextLineOffset), lineOffset);
  }
  get lineCount() {
    return this.getLineOffsets().length;
  }
  static isIncremental(event) {
    let candidate = event;
    return candidate !== void 0 && candidate !== null && typeof candidate.text === "string" && candidate.range !== void 0 && (candidate.rangeLength === void 0 || typeof candidate.rangeLength === "number");
  }
  static isFull(event) {
    let candidate = event;
    return candidate !== void 0 && candidate !== null && typeof candidate.text === "string" && candidate.range === void 0 && candidate.rangeLength === void 0;
  }
};
var TextDocument2;
(function(TextDocument3) {
  function create(uri, languageId, version, content) {
    return new FullTextDocument2(uri, languageId, version, content);
  }
  TextDocument3.create = create;
  function update(document, changes, version) {
    if (document instanceof FullTextDocument2) {
      document.update(changes, version);
      return document;
    } else {
      throw new Error("TextDocument.update: document must be created by TextDocument.create");
    }
  }
  TextDocument3.update = update;
  function applyEdits(document, edits) {
    let text = document.getText();
    let sortedEdits = mergeSort(edits.map(getWellformedEdit), (a2, b) => {
      let diff = a2.range.start.line - b.range.start.line;
      if (diff === 0) {
        return a2.range.start.character - b.range.start.character;
      }
      return diff;
    });
    let lastModifiedOffset = 0;
    const spans = [];
    for (const e of sortedEdits) {
      let startOffset = document.offsetAt(e.range.start);
      if (startOffset < lastModifiedOffset) {
        throw new Error("Overlapping edit");
      } else if (startOffset > lastModifiedOffset) {
        spans.push(text.substring(lastModifiedOffset, startOffset));
      }
      if (e.newText.length) {
        spans.push(e.newText);
      }
      lastModifiedOffset = document.offsetAt(e.range.end);
    }
    spans.push(text.substr(lastModifiedOffset));
    return spans.join("");
  }
  TextDocument3.applyEdits = applyEdits;
})(TextDocument2 || (TextDocument2 = {}));
function mergeSort(data, compare) {
  if (data.length <= 1) {
    return data;
  }
  const p = data.length / 2 | 0;
  const left = data.slice(0, p);
  const right = data.slice(p);
  mergeSort(left, compare);
  mergeSort(right, compare);
  let leftIdx = 0;
  let rightIdx = 0;
  let i = 0;
  while (leftIdx < left.length && rightIdx < right.length) {
    let ret = compare(left[leftIdx], right[rightIdx]);
    if (ret <= 0) {
      data[i++] = left[leftIdx++];
    } else {
      data[i++] = right[rightIdx++];
    }
  }
  while (leftIdx < left.length) {
    data[i++] = left[leftIdx++];
  }
  while (rightIdx < right.length) {
    data[i++] = right[rightIdx++];
  }
  return data;
}
function computeLineOffsets(text, isAtLineStart, textOffset = 0) {
  const result = isAtLineStart ? [textOffset] : [];
  for (let i = 0; i < text.length; i++) {
    let ch = text.charCodeAt(i);
    if (ch === 13 || ch === 10) {
      if (ch === 13 && i + 1 < text.length && text.charCodeAt(i + 1) === 10) {
        i++;
      }
      result.push(textOffset + i + 1);
    }
  }
  return result;
}
function getWellformedRange(range) {
  const start = range.start;
  const end = range.end;
  if (start.line > end.line || start.line === end.line && start.character > end.character) {
    return { start: end, end: start };
  }
  return range;
}
function getWellformedEdit(textEdit) {
  const range = getWellformedRange(textEdit.range);
  if (range !== textEdit.range) {
    return { newText: textEdit.newText, range };
  }
  return textEdit;
}

// node_modules/vscode-css-languageservice/lib/esm/cssLanguageTypes.js
var ClientCapabilities;
(function(ClientCapabilities2) {
  ClientCapabilities2.LATEST = {
    textDocument: {
      completion: {
        completionItem: {
          documentationFormat: [MarkupKind.Markdown, MarkupKind.PlainText]
        }
      },
      hover: {
        contentFormat: [MarkupKind.Markdown, MarkupKind.PlainText]
      }
    }
  };
})(ClientCapabilities || (ClientCapabilities = {}));
var FileType;
(function(FileType2) {
  FileType2[FileType2["Unknown"] = 0] = "Unknown";
  FileType2[FileType2["File"] = 1] = "File";
  FileType2[FileType2["Directory"] = 2] = "Directory";
  FileType2[FileType2["SymbolicLink"] = 64] = "SymbolicLink";
})(FileType || (FileType = {}));

// node_modules/vscode-css-languageservice/lib/esm/languageFacts/entry.js
var browserNames = {
  E: "Edge",
  FF: "Firefox",
  S: "Safari",
  C: "Chrome",
  IE: "IE",
  O: "Opera"
};
function getEntryStatus(status) {
  switch (status) {
    case "experimental":
      return "\u26A0\uFE0F Property is experimental. Be cautious when using it.\uFE0F\n\n";
    case "nonstandard":
      return "\u{1F6A8}\uFE0F Property is nonstandard. Avoid using it.\n\n";
    case "obsolete":
      return "\u{1F6A8}\uFE0F\uFE0F\uFE0F Property is obsolete. Avoid using it.\n\n";
    default:
      return "";
  }
}
function getEntryDescription(entry, doesSupportMarkdown, settings) {
  var result;
  if (doesSupportMarkdown) {
    result = {
      kind: "markdown",
      value: getEntryMarkdownDescription(entry, settings)
    };
  } else {
    result = {
      kind: "plaintext",
      value: getEntryStringDescription(entry, settings)
    };
  }
  if (result.value === "") {
    return void 0;
  }
  return result;
}
function textToMarkedString(text) {
  text = text.replace(/[\\`*_{}[\]()#+\-.!]/g, "\\$&");
  return text.replace(/</g, "&lt;").replace(/>/g, "&gt;");
}
function getEntryStringDescription(entry, settings) {
  if (!entry.description || entry.description === "") {
    return "";
  }
  if (typeof entry.description !== "string") {
    return entry.description.value;
  }
  var result = "";
  if ((settings === null || settings === void 0 ? void 0 : settings.documentation) !== false) {
    if (entry.status) {
      result += getEntryStatus(entry.status);
    }
    result += entry.description;
    var browserLabel = getBrowserLabel(entry.browsers);
    if (browserLabel) {
      result += "\n(" + browserLabel + ")";
    }
    if ("syntax" in entry) {
      result += "\n\nSyntax: ".concat(entry.syntax);
    }
  }
  if (entry.references && entry.references.length > 0 && (settings === null || settings === void 0 ? void 0 : settings.references) !== false) {
    if (result.length > 0) {
      result += "\n\n";
    }
    result += entry.references.map(function(r) {
      return "".concat(r.name, ": ").concat(r.url);
    }).join(" | ");
  }
  return result;
}
function getEntryMarkdownDescription(entry, settings) {
  if (!entry.description || entry.description === "") {
    return "";
  }
  var result = "";
  if ((settings === null || settings === void 0 ? void 0 : settings.documentation) !== false) {
    if (entry.status) {
      result += getEntryStatus(entry.status);
    }
    if (typeof entry.description === "string") {
      result += textToMarkedString(entry.description);
    } else {
      result += entry.description.kind === MarkupKind.Markdown ? entry.description.value : textToMarkedString(entry.description.value);
    }
    var browserLabel = getBrowserLabel(entry.browsers);
    if (browserLabel) {
      result += "\n\n(" + textToMarkedString(browserLabel) + ")";
    }
    if ("syntax" in entry && entry.syntax) {
      result += "\n\nSyntax: ".concat(textToMarkedString(entry.syntax));
    }
  }
  if (entry.references && entry.references.length > 0 && (settings === null || settings === void 0 ? void 0 : settings.references) !== false) {
    if (result.length > 0) {
      result += "\n\n";
    }
    result += entry.references.map(function(r) {
      return "[".concat(r.name, "](").concat(r.url, ")");
    }).join(" | ");
  }
  return result;
}
function getBrowserLabel(browsers) {
  if (browsers === void 0) {
    browsers = [];
  }
  if (browsers.length === 0) {
    return null;
  }
  return browsers.map(function(b) {
    var result = "";
    var matches2 = b.match(/([A-Z]+)(\d+)?/);
    var name = matches2[1];
    var version = matches2[2];
    if (name in browserNames) {
      result += browserNames[name];
    }
    if (version) {
      result += " " + version;
    }
    return result;
  }).join(", ");
}

// node_modules/vscode-css-languageservice/lib/esm/languageFacts/colors.js
var localize3 = loadMessageBundle();
var colorFunctions = [
  { func: "rgb($red, $green, $blue)", desc: localize3("css.builtin.rgb", "Creates a Color from red, green, and blue values.") },
  { func: "rgba($red, $green, $blue, $alpha)", desc: localize3("css.builtin.rgba", "Creates a Color from red, green, blue, and alpha values.") },
  { func: "hsl($hue, $saturation, $lightness)", desc: localize3("css.builtin.hsl", "Creates a Color from hue, saturation, and lightness values.") },
  { func: "hsla($hue, $saturation, $lightness, $alpha)", desc: localize3("css.builtin.hsla", "Creates a Color from hue, saturation, lightness, and alpha values.") },
  { func: "hwb($hue $white $black)", desc: localize3("css.builtin.hwb", "Creates a Color from hue, white and black.") }
];
var colors = {
  aliceblue: "#f0f8ff",
  antiquewhite: "#faebd7",
  aqua: "#00ffff",
  aquamarine: "#7fffd4",
  azure: "#f0ffff",
  beige: "#f5f5dc",
  bisque: "#ffe4c4",
  black: "#000000",
  blanchedalmond: "#ffebcd",
  blue: "#0000ff",
  blueviolet: "#8a2be2",
  brown: "#a52a2a",
  burlywood: "#deb887",
  cadetblue: "#5f9ea0",
  chartreuse: "#7fff00",
  chocolate: "#d2691e",
  coral: "#ff7f50",
  cornflowerblue: "#6495ed",
  cornsilk: "#fff8dc",
  crimson: "#dc143c",
  cyan: "#00ffff",
  darkblue: "#00008b",
  darkcyan: "#008b8b",
  darkgoldenrod: "#b8860b",
  darkgray: "#a9a9a9",
  darkgrey: "#a9a9a9",
  darkgreen: "#006400",
  darkkhaki: "#bdb76b",
  darkmagenta: "#8b008b",
  darkolivegreen: "#556b2f",
  darkorange: "#ff8c00",
  darkorchid: "#9932cc",
  darkred: "#8b0000",
  darksalmon: "#e9967a",
  darkseagreen: "#8fbc8f",
  darkslateblue: "#483d8b",
  darkslategray: "#2f4f4f",
  darkslategrey: "#2f4f4f",
  darkturquoise: "#00ced1",
  darkviolet: "#9400d3",
  deeppink: "#ff1493",
  deepskyblue: "#00bfff",
  dimgray: "#696969",
  dimgrey: "#696969",
  dodgerblue: "#1e90ff",
  firebrick: "#b22222",
  floralwhite: "#fffaf0",
  forestgreen: "#228b22",
  fuchsia: "#ff00ff",
  gainsboro: "#dcdcdc",
  ghostwhite: "#f8f8ff",
  gold: "#ffd700",
  goldenrod: "#daa520",
  gray: "#808080",
  grey: "#808080",
  green: "#008000",
  greenyellow: "#adff2f",
  honeydew: "#f0fff0",
  hotpink: "#ff69b4",
  indianred: "#cd5c5c",
  indigo: "#4b0082",
  ivory: "#fffff0",
  khaki: "#f0e68c",
  lavender: "#e6e6fa",
  lavenderblush: "#fff0f5",
  lawngreen: "#7cfc00",
  lemonchiffon: "#fffacd",
  lightblue: "#add8e6",
  lightcoral: "#f08080",
  lightcyan: "#e0ffff",
  lightgoldenrodyellow: "#fafad2",
  lightgray: "#d3d3d3",
  lightgrey: "#d3d3d3",
  lightgreen: "#90ee90",
  lightpink: "#ffb6c1",
  lightsalmon: "#ffa07a",
  lightseagreen: "#20b2aa",
  lightskyblue: "#87cefa",
  lightslategray: "#778899",
  lightslategrey: "#778899",
  lightsteelblue: "#b0c4de",
  lightyellow: "#ffffe0",
  lime: "#00ff00",
  limegreen: "#32cd32",
  linen: "#faf0e6",
  magenta: "#ff00ff",
  maroon: "#800000",
  mediumaquamarine: "#66cdaa",
  mediumblue: "#0000cd",
  mediumorchid: "#ba55d3",
  mediumpurple: "#9370d8",
  mediumseagreen: "#3cb371",
  mediumslateblue: "#7b68ee",
  mediumspringgreen: "#00fa9a",
  mediumturquoise: "#48d1cc",
  mediumvioletred: "#c71585",
  midnightblue: "#191970",
  mintcream: "#f5fffa",
  mistyrose: "#ffe4e1",
  moccasin: "#ffe4b5",
  navajowhite: "#ffdead",
  navy: "#000080",
  oldlace: "#fdf5e6",
  olive: "#808000",
  olivedrab: "#6b8e23",
  orange: "#ffa500",
  orangered: "#ff4500",
  orchid: "#da70d6",
  palegoldenrod: "#eee8aa",
  palegreen: "#98fb98",
  paleturquoise: "#afeeee",
  palevioletred: "#d87093",
  papayawhip: "#ffefd5",
  peachpuff: "#ffdab9",
  peru: "#cd853f",
  pink: "#ffc0cb",
  plum: "#dda0dd",
  powderblue: "#b0e0e6",
  purple: "#800080",
  red: "#ff0000",
  rebeccapurple: "#663399",
  rosybrown: "#bc8f8f",
  royalblue: "#4169e1",
  saddlebrown: "#8b4513",
  salmon: "#fa8072",
  sandybrown: "#f4a460",
  seagreen: "#2e8b57",
  seashell: "#fff5ee",
  sienna: "#a0522d",
  silver: "#c0c0c0",
  skyblue: "#87ceeb",
  slateblue: "#6a5acd",
  slategray: "#708090",
  slategrey: "#708090",
  snow: "#fffafa",
  springgreen: "#00ff7f",
  steelblue: "#4682b4",
  tan: "#d2b48c",
  teal: "#008080",
  thistle: "#d8bfd8",
  tomato: "#ff6347",
  turquoise: "#40e0d0",
  violet: "#ee82ee",
  wheat: "#f5deb3",
  white: "#ffffff",
  whitesmoke: "#f5f5f5",
  yellow: "#ffff00",
  yellowgreen: "#9acd32"
};
var colorKeywords = {
  "currentColor": "The value of the 'color' property. The computed value of the 'currentColor' keyword is the computed value of the 'color' property. If the 'currentColor' keyword is set on the 'color' property itself, it is treated as 'color:inherit' at parse time.",
  "transparent": "Fully transparent. This keyword can be considered a shorthand for rgba(0,0,0,0) which is its computed value."
};
function getNumericValue(node, factor) {
  var val = node.getText();
  var m = val.match(/^([-+]?[0-9]*\.?[0-9]+)(%?)$/);
  if (m) {
    if (m[2]) {
      factor = 100;
    }
    var result = parseFloat(m[1]) / factor;
    if (result >= 0 && result <= 1) {
      return result;
    }
  }
  throw new Error();
}
function getAngle(node) {
  var val = node.getText();
  var m = val.match(/^([-+]?[0-9]*\.?[0-9]+)(deg|rad|grad|turn)?$/);
  if (m) {
    switch (m[2]) {
      case "deg":
        return parseFloat(val) % 360;
      case "rad":
        return parseFloat(val) * 180 / Math.PI % 360;
      case "grad":
        return parseFloat(val) * 0.9 % 360;
      case "turn":
        return parseFloat(val) * 360 % 360;
      default:
        if ("undefined" === typeof m[2]) {
          return parseFloat(val) % 360;
        }
    }
  }
  throw new Error();
}
function isColorConstructor(node) {
  var name = node.getName();
  if (!name) {
    return false;
  }
  return /^(rgb|rgba|hsl|hsla|hwb)$/gi.test(name);
}
var Digit0 = 48;
var Digit9 = 57;
var A = 65;
var a = 97;
var f = 102;
function hexDigit(charCode) {
  if (charCode < Digit0) {
    return 0;
  }
  if (charCode <= Digit9) {
    return charCode - Digit0;
  }
  if (charCode < a) {
    charCode += a - A;
  }
  if (charCode >= a && charCode <= f) {
    return charCode - a + 10;
  }
  return 0;
}
function colorFromHex(text) {
  if (text[0] !== "#") {
    return null;
  }
  switch (text.length) {
    case 4:
      return {
        red: hexDigit(text.charCodeAt(1)) * 17 / 255,
        green: hexDigit(text.charCodeAt(2)) * 17 / 255,
        blue: hexDigit(text.charCodeAt(3)) * 17 / 255,
        alpha: 1
      };
    case 5:
      return {
        red: hexDigit(text.charCodeAt(1)) * 17 / 255,
        green: hexDigit(text.charCodeAt(2)) * 17 / 255,
        blue: hexDigit(text.charCodeAt(3)) * 17 / 255,
        alpha: hexDigit(text.charCodeAt(4)) * 17 / 255
      };
    case 7:
      return {
        red: (hexDigit(text.charCodeAt(1)) * 16 + hexDigit(text.charCodeAt(2))) / 255,
        green: (hexDigit(text.charCodeAt(3)) * 16 + hexDigit(text.charCodeAt(4))) / 255,
        blue: (hexDigit(text.charCodeAt(5)) * 16 + hexDigit(text.charCodeAt(6))) / 255,
        alpha: 1
      };
    case 9:
      return {
        red: (hexDigit(text.charCodeAt(1)) * 16 + hexDigit(text.charCodeAt(2))) / 255,
        green: (hexDigit(text.charCodeAt(3)) * 16 + hexDigit(text.charCodeAt(4))) / 255,
        blue: (hexDigit(text.charCodeAt(5)) * 16 + hexDigit(text.charCodeAt(6))) / 255,
        alpha: (hexDigit(text.charCodeAt(7)) * 16 + hexDigit(text.charCodeAt(8))) / 255
      };
  }
  return null;
}
function colorFromHSL(hue, sat, light, alpha) {
  if (alpha === void 0) {
    alpha = 1;
  }
  hue = hue / 60;
  if (sat === 0) {
    return { red: light, green: light, blue: light, alpha };
  } else {
    var hueToRgb = function(t12, t22, hue2) {
      while (hue2 < 0) {
        hue2 += 6;
      }
      while (hue2 >= 6) {
        hue2 -= 6;
      }
      if (hue2 < 1) {
        return (t22 - t12) * hue2 + t12;
      }
      if (hue2 < 3) {
        return t22;
      }
      if (hue2 < 4) {
        return (t22 - t12) * (4 - hue2) + t12;
      }
      return t12;
    };
    var t2 = light <= 0.5 ? light * (sat + 1) : light + sat - light * sat;
    var t1 = light * 2 - t2;
    return { red: hueToRgb(t1, t2, hue + 2), green: hueToRgb(t1, t2, hue), blue: hueToRgb(t1, t2, hue - 2), alpha };
  }
}
function hslFromColor(rgba) {
  var r = rgba.red;
  var g = rgba.green;
  var b = rgba.blue;
  var a2 = rgba.alpha;
  var max = Math.max(r, g, b);
  var min = Math.min(r, g, b);
  var h = 0;
  var s = 0;
  var l = (min + max) / 2;
  var chroma = max - min;
  if (chroma > 0) {
    s = Math.min(l <= 0.5 ? chroma / (2 * l) : chroma / (2 - 2 * l), 1);
    switch (max) {
      case r:
        h = (g - b) / chroma + (g < b ? 6 : 0);
        break;
      case g:
        h = (b - r) / chroma + 2;
        break;
      case b:
        h = (r - g) / chroma + 4;
        break;
    }
    h *= 60;
    h = Math.round(h);
  }
  return { h, s, l, a: a2 };
}
function colorFromHWB(hue, white, black, alpha) {
  if (alpha === void 0) {
    alpha = 1;
  }
  if (white + black >= 1) {
    var gray = white / (white + black);
    return { red: gray, green: gray, blue: gray, alpha };
  }
  var rgb = colorFromHSL(hue, 1, 0.5, alpha);
  var red = rgb.red;
  red *= 1 - white - black;
  red += white;
  var green = rgb.green;
  green *= 1 - white - black;
  green += white;
  var blue = rgb.blue;
  blue *= 1 - white - black;
  blue += white;
  return {
    red,
    green,
    blue,
    alpha
  };
}
function hwbFromColor(rgba) {
  var hsl = hslFromColor(rgba);
  var white = Math.min(rgba.red, rgba.green, rgba.blue);
  var black = 1 - Math.max(rgba.red, rgba.green, rgba.blue);
  return {
    h: hsl.h,
    w: white,
    b: black,
    a: hsl.a
  };
}
function getColorValue(node) {
  if (node.type === NodeType.HexColorValue) {
    var text = node.getText();
    return colorFromHex(text);
  } else if (node.type === NodeType.Function) {
    var functionNode = node;
    var name = functionNode.getName();
    var colorValues = functionNode.getArguments().getChildren();
    if (colorValues.length === 1) {
      var functionArg = colorValues[0].getChildren();
      if (functionArg.length === 1 && functionArg[0].type === NodeType.Expression) {
        colorValues = functionArg[0].getChildren();
        if (colorValues.length === 3) {
          var lastValue = colorValues[2];
          if (lastValue instanceof BinaryExpression) {
            var left = lastValue.getLeft(), right = lastValue.getRight(), operator = lastValue.getOperator();
            if (left && right && operator && operator.matches("/")) {
              colorValues = [colorValues[0], colorValues[1], left, right];
            }
          }
        }
      }
    }
    if (!name || colorValues.length < 3 || colorValues.length > 4) {
      return null;
    }
    try {
      var alpha = colorValues.length === 4 ? getNumericValue(colorValues[3], 1) : 1;
      if (name === "rgb" || name === "rgba") {
        return {
          red: getNumericValue(colorValues[0], 255),
          green: getNumericValue(colorValues[1], 255),
          blue: getNumericValue(colorValues[2], 255),
          alpha
        };
      } else if (name === "hsl" || name === "hsla") {
        var h = getAngle(colorValues[0]);
        var s = getNumericValue(colorValues[1], 100);
        var l = getNumericValue(colorValues[2], 100);
        return colorFromHSL(h, s, l, alpha);
      } else if (name === "hwb") {
        var h = getAngle(colorValues[0]);
        var w = getNumericValue(colorValues[1], 100);
        var b = getNumericValue(colorValues[2], 100);
        return colorFromHWB(h, w, b, alpha);
      }
    } catch (e) {
      return null;
    }
  } else if (node.type === NodeType.Identifier) {
    if (node.parent && node.parent.type !== NodeType.Term) {
      return null;
    }
    var term = node.parent;
    if (term && term.parent && term.parent.type === NodeType.BinaryExpression) {
      var expression = term.parent;
      if (expression.parent && expression.parent.type === NodeType.ListEntry && expression.parent.key === expression) {
        return null;
      }
    }
    var candidateColor = node.getText().toLowerCase();
    if (candidateColor === "none") {
      return null;
    }
    var colorHex = colors[candidateColor];
    if (colorHex) {
      return colorFromHex(colorHex);
    }
  }
  return null;
}

// node_modules/vscode-css-languageservice/lib/esm/languageFacts/builtinData.js
var positionKeywords = {
  "bottom": "Computes to \u2018100%\u2019 for the vertical position if one or two values are given, otherwise specifies the bottom edge as the origin for the next offset.",
  "center": "Computes to \u201850%\u2019 (\u2018left 50%\u2019) for the horizontal position if the horizontal position is not otherwise specified, or \u201850%\u2019 (\u2018top 50%\u2019) for the vertical position if it is.",
  "left": "Computes to \u20180%\u2019 for the horizontal position if one or two values are given, otherwise specifies the left edge as the origin for the next offset.",
  "right": "Computes to \u2018100%\u2019 for the horizontal position if one or two values are given, otherwise specifies the right edge as the origin for the next offset.",
  "top": "Computes to \u20180%\u2019 for the vertical position if one or two values are given, otherwise specifies the top edge as the origin for the next offset."
};
var repeatStyleKeywords = {
  "no-repeat": "Placed once and not repeated in this direction.",
  "repeat": "Repeated in this direction as often as needed to cover the background painting area.",
  "repeat-x": "Computes to \u2018repeat no-repeat\u2019.",
  "repeat-y": "Computes to \u2018no-repeat repeat\u2019.",
  "round": "Repeated as often as will fit within the background positioning area. If it doesn\u2019t fit a whole number of times, it is rescaled so that it does.",
  "space": "Repeated as often as will fit within the background positioning area without being clipped and then the images are spaced out to fill the area."
};
var lineStyleKeywords = {
  "dashed": "A series of square-ended dashes.",
  "dotted": "A series of round dots.",
  "double": "Two parallel solid lines with some space between them.",
  "groove": "Looks as if it were carved in the canvas.",
  "hidden": "Same as \u2018none\u2019, but has different behavior in the border conflict resolution rules for border-collapsed tables.",
  "inset": "Looks as if the content on the inside of the border is sunken into the canvas.",
  "none": "No border. Color and width are ignored.",
  "outset": "Looks as if the content on the inside of the border is coming out of the canvas.",
  "ridge": "Looks as if it were coming out of the canvas.",
  "solid": "A single line segment."
};
var lineWidthKeywords = ["medium", "thick", "thin"];
var boxKeywords = {
  "border-box": "The background is painted within (clipped to) the border box.",
  "content-box": "The background is painted within (clipped to) the content box.",
  "padding-box": "The background is painted within (clipped to) the padding box."
};
var geometryBoxKeywords = {
  "margin-box": "Uses the margin box as reference box.",
  "fill-box": "Uses the object bounding box as reference box.",
  "stroke-box": "Uses the stroke bounding box as reference box.",
  "view-box": "Uses the nearest SVG viewport as reference box."
};
var cssWideKeywords = {
  "initial": "Represents the value specified as the property\u2019s initial value.",
  "inherit": "Represents the computed value of the property on the element\u2019s parent.",
  "unset": "Acts as either `inherit` or `initial`, depending on whether the property is inherited or not."
};
var cssWideFunctions = {
  "var()": "Evaluates the value of a custom variable.",
  "calc()": "Evaluates an mathematical expression. The following operators can be used: + - * /."
};
var imageFunctions = {
  "url()": "Reference an image file by URL",
  "image()": "Provide image fallbacks and annotations.",
  "-webkit-image-set()": "Provide multiple resolutions. Remember to use unprefixed image-set() in addition.",
  "image-set()": "Provide multiple resolutions of an image and const the UA decide which is most appropriate in a given situation.",
  "-moz-element()": "Use an element in the document as an image. Remember to use unprefixed element() in addition.",
  "element()": "Use an element in the document as an image.",
  "cross-fade()": "Indicates the two images to be combined and how far along in the transition the combination is.",
  "-webkit-gradient()": "Deprecated. Use modern linear-gradient() or radial-gradient() instead.",
  "-webkit-linear-gradient()": "Linear gradient. Remember to use unprefixed version in addition.",
  "-moz-linear-gradient()": "Linear gradient. Remember to use unprefixed version in addition.",
  "-o-linear-gradient()": "Linear gradient. Remember to use unprefixed version in addition.",
  "linear-gradient()": "A linear gradient is created by specifying a straight gradient line, and then several colors placed along that line.",
  "-webkit-repeating-linear-gradient()": "Repeating Linear gradient. Remember to use unprefixed version in addition.",
  "-moz-repeating-linear-gradient()": "Repeating Linear gradient. Remember to use unprefixed version in addition.",
  "-o-repeating-linear-gradient()": "Repeating Linear gradient. Remember to use unprefixed version in addition.",
  "repeating-linear-gradient()": "Same as linear-gradient, except the color-stops are repeated infinitely in both directions, with their positions shifted by multiples of the difference between the last specified color-stop\u2019s position and the first specified color-stop\u2019s position.",
  "-webkit-radial-gradient()": "Radial gradient. Remember to use unprefixed version in addition.",
  "-moz-radial-gradient()": "Radial gradient. Remember to use unprefixed version in addition.",
  "radial-gradient()": "Colors emerge from a single point and smoothly spread outward in a circular or elliptical shape.",
  "-webkit-repeating-radial-gradient()": "Repeating radial gradient. Remember to use unprefixed version in addition.",
  "-moz-repeating-radial-gradient()": "Repeating radial gradient. Remember to use unprefixed version in addition.",
  "repeating-radial-gradient()": "Same as radial-gradient, except the color-stops are repeated infinitely in both directions, with their positions shifted by multiples of the difference between the last specified color-stop\u2019s position and the first specified color-stop\u2019s position."
};
var transitionTimingFunctions = {
  "ease": "Equivalent to cubic-bezier(0.25, 0.1, 0.25, 1.0).",
  "ease-in": "Equivalent to cubic-bezier(0.42, 0, 1.0, 1.0).",
  "ease-in-out": "Equivalent to cubic-bezier(0.42, 0, 0.58, 1.0).",
  "ease-out": "Equivalent to cubic-bezier(0, 0, 0.58, 1.0).",
  "linear": "Equivalent to cubic-bezier(0.0, 0.0, 1.0, 1.0).",
  "step-end": "Equivalent to steps(1, end).",
  "step-start": "Equivalent to steps(1, start).",
  "steps()": "The first parameter specifies the number of intervals in the function. The second parameter, which is optional, is either the value \u201Cstart\u201D or \u201Cend\u201D.",
  "cubic-bezier()": "Specifies a cubic-bezier curve. The four values specify points P1 and P2  of the curve as (x1, y1, x2, y2).",
  "cubic-bezier(0.6, -0.28, 0.735, 0.045)": "Ease-in Back. Overshoots.",
  "cubic-bezier(0.68, -0.55, 0.265, 1.55)": "Ease-in-out Back. Overshoots.",
  "cubic-bezier(0.175, 0.885, 0.32, 1.275)": "Ease-out Back. Overshoots.",
  "cubic-bezier(0.6, 0.04, 0.98, 0.335)": "Ease-in Circular. Based on half circle.",
  "cubic-bezier(0.785, 0.135, 0.15, 0.86)": "Ease-in-out Circular. Based on half circle.",
  "cubic-bezier(0.075, 0.82, 0.165, 1)": "Ease-out Circular. Based on half circle.",
  "cubic-bezier(0.55, 0.055, 0.675, 0.19)": "Ease-in Cubic. Based on power of three.",
  "cubic-bezier(0.645, 0.045, 0.355, 1)": "Ease-in-out Cubic. Based on power of three.",
  "cubic-bezier(0.215, 0.610, 0.355, 1)": "Ease-out Cubic. Based on power of three.",
  "cubic-bezier(0.95, 0.05, 0.795, 0.035)": "Ease-in Exponential. Based on two to the power ten.",
  "cubic-bezier(1, 0, 0, 1)": "Ease-in-out Exponential. Based on two to the power ten.",
  "cubic-bezier(0.19, 1, 0.22, 1)": "Ease-out Exponential. Based on two to the power ten.",
  "cubic-bezier(0.47, 0, 0.745, 0.715)": "Ease-in Sine.",
  "cubic-bezier(0.445, 0.05, 0.55, 0.95)": "Ease-in-out Sine.",
  "cubic-bezier(0.39, 0.575, 0.565, 1)": "Ease-out Sine.",
  "cubic-bezier(0.55, 0.085, 0.68, 0.53)": "Ease-in Quadratic. Based on power of two.",
  "cubic-bezier(0.455, 0.03, 0.515, 0.955)": "Ease-in-out Quadratic. Based on power of two.",
  "cubic-bezier(0.25, 0.46, 0.45, 0.94)": "Ease-out Quadratic. Based on power of two.",
  "cubic-bezier(0.895, 0.03, 0.685, 0.22)": "Ease-in Quartic. Based on power of four.",
  "cubic-bezier(0.77, 0, 0.175, 1)": "Ease-in-out Quartic. Based on power of four.",
  "cubic-bezier(0.165, 0.84, 0.44, 1)": "Ease-out Quartic. Based on power of four.",
  "cubic-bezier(0.755, 0.05, 0.855, 0.06)": "Ease-in Quintic. Based on power of five.",
  "cubic-bezier(0.86, 0, 0.07, 1)": "Ease-in-out Quintic. Based on power of five.",
  "cubic-bezier(0.23, 1, 0.320, 1)": "Ease-out Quintic. Based on power of five."
};
var basicShapeFunctions = {
  "circle()": "Defines a circle.",
  "ellipse()": "Defines an ellipse.",
  "inset()": "Defines an inset rectangle.",
  "polygon()": "Defines a polygon."
};
var units = {
  "length": ["em", "rem", "ex", "px", "cm", "mm", "in", "pt", "pc", "ch", "vw", "vh", "vmin", "vmax"],
  "angle": ["deg", "rad", "grad", "turn"],
  "time": ["ms", "s"],
  "frequency": ["Hz", "kHz"],
  "resolution": ["dpi", "dpcm", "dppx"],
  "percentage": ["%", "fr"]
};
var html5Tags = [
  "a",
  "abbr",
  "address",
  "area",
  "article",
  "aside",
  "audio",
  "b",
  "base",
  "bdi",
  "bdo",
  "blockquote",
  "body",
  "br",
  "button",
  "canvas",
  "caption",
  "cite",
  "code",
  "col",
  "colgroup",
  "data",
  "datalist",
  "dd",
  "del",
  "details",
  "dfn",
  "dialog",
  "div",
  "dl",
  "dt",
  "em",
  "embed",
  "fieldset",
  "figcaption",
  "figure",
  "footer",
  "form",
  "h1",
  "h2",
  "h3",
  "h4",
  "h5",
  "h6",
  "head",
  "header",
  "hgroup",
  "hr",
  "html",
  "i",
  "iframe",
  "img",
  "input",
  "ins",
  "kbd",
  "keygen",
  "label",
  "legend",
  "li",
  "link",
  "main",
  "map",
  "mark",
  "menu",
  "menuitem",
  "meta",
  "meter",
  "nav",
  "noscript",
  "object",
  "ol",
  "optgroup",
  "option",
  "output",
  "p",
  "param",
  "picture",
  "pre",
  "progress",
  "q",
  "rb",
  "rp",
  "rt",
  "rtc",
  "ruby",
  "s",
  "samp",
  "script",
  "section",
  "select",
  "small",
  "source",
  "span",
  "strong",
  "style",
  "sub",
  "summary",
  "sup",
  "table",
  "tbody",
  "td",
  "template",
  "textarea",
  "tfoot",
  "th",
  "thead",
  "time",
  "title",
  "tr",
  "track",
  "u",
  "ul",
  "const",
  "video",
  "wbr"
];
var svgElements = [
  "circle",
  "clipPath",
  "cursor",
  "defs",
  "desc",
  "ellipse",
  "feBlend",
  "feColorMatrix",
  "feComponentTransfer",
  "feComposite",
  "feConvolveMatrix",
  "feDiffuseLighting",
  "feDisplacementMap",
  "feDistantLight",
  "feDropShadow",
  "feFlood",
  "feFuncA",
  "feFuncB",
  "feFuncG",
  "feFuncR",
  "feGaussianBlur",
  "feImage",
  "feMerge",
  "feMergeNode",
  "feMorphology",
  "feOffset",
  "fePointLight",
  "feSpecularLighting",
  "feSpotLight",
  "feTile",
  "feTurbulence",
  "filter",
  "foreignObject",
  "g",
  "hatch",
  "hatchpath",
  "image",
  "line",
  "linearGradient",
  "marker",
  "mask",
  "mesh",
  "meshpatch",
  "meshrow",
  "metadata",
  "mpath",
  "path",
  "pattern",
  "polygon",
  "polyline",
  "radialGradient",
  "rect",
  "set",
  "solidcolor",
  "stop",
  "svg",
  "switch",
  "symbol",
  "text",
  "textPath",
  "tspan",
  "use",
  "view"
];
var pageBoxDirectives = [
  "@bottom-center",
  "@bottom-left",
  "@bottom-left-corner",
  "@bottom-right",
  "@bottom-right-corner",
  "@left-bottom",
  "@left-middle",
  "@left-top",
  "@right-bottom",
  "@right-middle",
  "@right-top",
  "@top-center",
  "@top-left",
  "@top-left-corner",
  "@top-right",
  "@top-right-corner"
];

// node_modules/vscode-css-languageservice/lib/esm/utils/objects.js
function values(obj) {
  return Object.keys(obj).map(function(key) {
    return obj[key];
  });
}
function isDefined(obj) {
  return typeof obj !== "undefined";
}

// node_modules/vscode-css-languageservice/lib/esm/parser/cssParser.js
var __spreadArray = function(to, from, pack) {
  if (pack || arguments.length === 2)
    for (var i = 0, l = from.length, ar; i < l; i++) {
      if (ar || !(i in from)) {
        if (!ar)
          ar = Array.prototype.slice.call(from, 0, i);
        ar[i] = from[i];
      }
    }
  return to.concat(ar || Array.prototype.slice.call(from));
};
var Parser = function() {
  function Parser2(scnr) {
    if (scnr === void 0) {
      scnr = new Scanner();
    }
    this.keyframeRegex = /^@(\-(webkit|ms|moz|o)\-)?keyframes$/i;
    this.scanner = scnr;
    this.token = { type: TokenType.EOF, offset: -1, len: 0, text: "" };
    this.prevToken = void 0;
  }
  Parser2.prototype.peekIdent = function(text) {
    return TokenType.Ident === this.token.type && text.length === this.token.text.length && text === this.token.text.toLowerCase();
  };
  Parser2.prototype.peekKeyword = function(text) {
    return TokenType.AtKeyword === this.token.type && text.length === this.token.text.length && text === this.token.text.toLowerCase();
  };
  Parser2.prototype.peekDelim = function(text) {
    return TokenType.Delim === this.token.type && text === this.token.text;
  };
  Parser2.prototype.peek = function(type) {
    return type === this.token.type;
  };
  Parser2.prototype.peekOne = function() {
    var types = [];
    for (var _i = 0; _i < arguments.length; _i++) {
      types[_i] = arguments[_i];
    }
    return types.indexOf(this.token.type) !== -1;
  };
  Parser2.prototype.peekRegExp = function(type, regEx) {
    if (type !== this.token.type) {
      return false;
    }
    return regEx.test(this.token.text);
  };
  Parser2.prototype.hasWhitespace = function() {
    return !!this.prevToken && this.prevToken.offset + this.prevToken.len !== this.token.offset;
  };
  Parser2.prototype.consumeToken = function() {
    this.prevToken = this.token;
    this.token = this.scanner.scan();
  };
  Parser2.prototype.acceptUnicodeRange = function() {
    var token = this.scanner.tryScanUnicode();
    if (token) {
      this.prevToken = token;
      this.token = this.scanner.scan();
      return true;
    }
    return false;
  };
  Parser2.prototype.mark = function() {
    return {
      prev: this.prevToken,
      curr: this.token,
      pos: this.scanner.pos()
    };
  };
  Parser2.prototype.restoreAtMark = function(mark) {
    this.prevToken = mark.prev;
    this.token = mark.curr;
    this.scanner.goBackTo(mark.pos);
  };
  Parser2.prototype.try = function(func) {
    var pos = this.mark();
    var node = func();
    if (!node) {
      this.restoreAtMark(pos);
      return null;
    }
    return node;
  };
  Parser2.prototype.acceptOneKeyword = function(keywords) {
    if (TokenType.AtKeyword === this.token.type) {
      for (var _i = 0, keywords_1 = keywords; _i < keywords_1.length; _i++) {
        var keyword = keywords_1[_i];
        if (keyword.length === this.token.text.length && keyword === this.token.text.toLowerCase()) {
          this.consumeToken();
          return true;
        }
      }
    }
    return false;
  };
  Parser2.prototype.accept = function(type) {
    if (type === this.token.type) {
      this.consumeToken();
      return true;
    }
    return false;
  };
  Parser2.prototype.acceptIdent = function(text) {
    if (this.peekIdent(text)) {
      this.consumeToken();
      return true;
    }
    return false;
  };
  Parser2.prototype.acceptKeyword = function(text) {
    if (this.peekKeyword(text)) {
      this.consumeToken();
      return true;
    }
    return false;
  };
  Parser2.prototype.acceptDelim = function(text) {
    if (this.peekDelim(text)) {
      this.consumeToken();
      return true;
    }
    return false;
  };
  Parser2.prototype.acceptRegexp = function(regEx) {
    if (regEx.test(this.token.text)) {
      this.consumeToken();
      return true;
    }
    return false;
  };
  Parser2.prototype._parseRegexp = function(regEx) {
    var node = this.createNode(NodeType.Identifier);
    do {
    } while (this.acceptRegexp(regEx));
    return this.finish(node);
  };
  Parser2.prototype.acceptUnquotedString = function() {
    var pos = this.scanner.pos();
    this.scanner.goBackTo(this.token.offset);
    var unquoted = this.scanner.scanUnquotedString();
    if (unquoted) {
      this.token = unquoted;
      this.consumeToken();
      return true;
    }
    this.scanner.goBackTo(pos);
    return false;
  };
  Parser2.prototype.resync = function(resyncTokens, resyncStopTokens) {
    while (true) {
      if (resyncTokens && resyncTokens.indexOf(this.token.type) !== -1) {
        this.consumeToken();
        return true;
      } else if (resyncStopTokens && resyncStopTokens.indexOf(this.token.type) !== -1) {
        return true;
      } else {
        if (this.token.type === TokenType.EOF) {
          return false;
        }
        this.token = this.scanner.scan();
      }
    }
  };
  Parser2.prototype.createNode = function(nodeType) {
    return new Node(this.token.offset, this.token.len, nodeType);
  };
  Parser2.prototype.create = function(ctor) {
    return new ctor(this.token.offset, this.token.len);
  };
  Parser2.prototype.finish = function(node, error, resyncTokens, resyncStopTokens) {
    if (!(node instanceof Nodelist)) {
      if (error) {
        this.markError(node, error, resyncTokens, resyncStopTokens);
      }
      if (this.prevToken) {
        var prevEnd = this.prevToken.offset + this.prevToken.len;
        node.length = prevEnd > node.offset ? prevEnd - node.offset : 0;
      }
    }
    return node;
  };
  Parser2.prototype.markError = function(node, error, resyncTokens, resyncStopTokens) {
    if (this.token !== this.lastErrorToken) {
      node.addIssue(new Marker(node, error, Level.Error, void 0, this.token.offset, this.token.len));
      this.lastErrorToken = this.token;
    }
    if (resyncTokens || resyncStopTokens) {
      this.resync(resyncTokens, resyncStopTokens);
    }
  };
  Parser2.prototype.parseStylesheet = function(textDocument) {
    var versionId = textDocument.version;
    var text = textDocument.getText();
    var textProvider = function(offset, length) {
      if (textDocument.version !== versionId) {
        throw new Error("Underlying model has changed, AST is no longer valid");
      }
      return text.substr(offset, length);
    };
    return this.internalParse(text, this._parseStylesheet, textProvider);
  };
  Parser2.prototype.internalParse = function(input, parseFunc, textProvider) {
    this.scanner.setSource(input);
    this.token = this.scanner.scan();
    var node = parseFunc.bind(this)();
    if (node) {
      if (textProvider) {
        node.textProvider = textProvider;
      } else {
        node.textProvider = function(offset, length) {
          return input.substr(offset, length);
        };
      }
    }
    return node;
  };
  Parser2.prototype._parseStylesheet = function() {
    var node = this.create(Stylesheet);
    while (node.addChild(this._parseStylesheetStart())) {
    }
    var inRecovery = false;
    do {
      var hasMatch = false;
      do {
        hasMatch = false;
        var statement = this._parseStylesheetStatement();
        if (statement) {
          node.addChild(statement);
          hasMatch = true;
          inRecovery = false;
          if (!this.peek(TokenType.EOF) && this._needsSemicolonAfter(statement) && !this.accept(TokenType.SemiColon)) {
            this.markError(node, ParseError.SemiColonExpected);
          }
        }
        while (this.accept(TokenType.SemiColon) || this.accept(TokenType.CDO) || this.accept(TokenType.CDC)) {
          hasMatch = true;
          inRecovery = false;
        }
      } while (hasMatch);
      if (this.peek(TokenType.EOF)) {
        break;
      }
      if (!inRecovery) {
        if (this.peek(TokenType.AtKeyword)) {
          this.markError(node, ParseError.UnknownAtRule);
        } else {
          this.markError(node, ParseError.RuleOrSelectorExpected);
        }
        inRecovery = true;
      }
      this.consumeToken();
    } while (!this.peek(TokenType.EOF));
    return this.finish(node);
  };
  Parser2.prototype._parseStylesheetStart = function() {
    return this._parseCharset();
  };
  Parser2.prototype._parseStylesheetStatement = function(isNested) {
    if (isNested === void 0) {
      isNested = false;
    }
    if (this.peek(TokenType.AtKeyword)) {
      return this._parseStylesheetAtStatement(isNested);
    }
    return this._parseRuleset(isNested);
  };
  Parser2.prototype._parseStylesheetAtStatement = function(isNested) {
    if (isNested === void 0) {
      isNested = false;
    }
    return this._parseImport() || this._parseMedia(isNested) || this._parsePage() || this._parseFontFace() || this._parseKeyframe() || this._parseSupports(isNested) || this._parseViewPort() || this._parseNamespace() || this._parseDocument() || this._parseUnknownAtRule();
  };
  Parser2.prototype._tryParseRuleset = function(isNested) {
    var mark = this.mark();
    if (this._parseSelector(isNested)) {
      while (this.accept(TokenType.Comma) && this._parseSelector(isNested)) {
      }
      if (this.accept(TokenType.CurlyL)) {
        this.restoreAtMark(mark);
        return this._parseRuleset(isNested);
      }
    }
    this.restoreAtMark(mark);
    return null;
  };
  Parser2.prototype._parseRuleset = function(isNested) {
    if (isNested === void 0) {
      isNested = false;
    }
    var node = this.create(RuleSet);
    var selectors = node.getSelectors();
    if (!selectors.addChild(this._parseSelector(isNested))) {
      return null;
    }
    while (this.accept(TokenType.Comma)) {
      if (!selectors.addChild(this._parseSelector(isNested))) {
        return this.finish(node, ParseError.SelectorExpected);
      }
    }
    return this._parseBody(node, this._parseRuleSetDeclaration.bind(this));
  };
  Parser2.prototype._parseRuleSetDeclarationAtStatement = function() {
    return this._parseUnknownAtRule();
  };
  Parser2.prototype._parseRuleSetDeclaration = function() {
    if (this.peek(TokenType.AtKeyword)) {
      return this._parseRuleSetDeclarationAtStatement();
    }
    return this._parseDeclaration();
  };
  Parser2.prototype._needsSemicolonAfter = function(node) {
    switch (node.type) {
      case NodeType.Keyframe:
      case NodeType.ViewPort:
      case NodeType.Media:
      case NodeType.Ruleset:
      case NodeType.Namespace:
      case NodeType.If:
      case NodeType.For:
      case NodeType.Each:
      case NodeType.While:
      case NodeType.MixinDeclaration:
      case NodeType.FunctionDeclaration:
      case NodeType.MixinContentDeclaration:
        return false;
      case NodeType.ExtendsReference:
      case NodeType.MixinContentReference:
      case NodeType.ReturnStatement:
      case NodeType.MediaQuery:
      case NodeType.Debug:
      case NodeType.Import:
      case NodeType.AtApplyRule:
      case NodeType.CustomPropertyDeclaration:
        return true;
      case NodeType.VariableDeclaration:
        return node.needsSemicolon;
      case NodeType.MixinReference:
        return !node.getContent();
      case NodeType.Declaration:
        return !node.getNestedProperties();
    }
    return false;
  };
  Parser2.prototype._parseDeclarations = function(parseDeclaration) {
    var node = this.create(Declarations);
    if (!this.accept(TokenType.CurlyL)) {
      return null;
    }
    var decl = parseDeclaration();
    while (node.addChild(decl)) {
      if (this.peek(TokenType.CurlyR)) {
        break;
      }
      if (this._needsSemicolonAfter(decl) && !this.accept(TokenType.SemiColon)) {
        return this.finish(node, ParseError.SemiColonExpected, [TokenType.SemiColon, TokenType.CurlyR]);
      }
      if (decl && this.prevToken && this.prevToken.type === TokenType.SemiColon) {
        decl.semicolonPosition = this.prevToken.offset;
      }
      while (this.accept(TokenType.SemiColon)) {
      }
      decl = parseDeclaration();
    }
    if (!this.accept(TokenType.CurlyR)) {
      return this.finish(node, ParseError.RightCurlyExpected, [TokenType.CurlyR, TokenType.SemiColon]);
    }
    return this.finish(node);
  };
  Parser2.prototype._parseBody = function(node, parseDeclaration) {
    if (!node.setDeclarations(this._parseDeclarations(parseDeclaration))) {
      return this.finish(node, ParseError.LeftCurlyExpected, [TokenType.CurlyR, TokenType.SemiColon]);
    }
    return this.finish(node);
  };
  Parser2.prototype._parseSelector = function(isNested) {
    var node = this.create(Selector);
    var hasContent = false;
    if (isNested) {
      hasContent = node.addChild(this._parseCombinator());
    }
    while (node.addChild(this._parseSimpleSelector())) {
      hasContent = true;
      node.addChild(this._parseCombinator());
    }
    return hasContent ? this.finish(node) : null;
  };
  Parser2.prototype._parseDeclaration = function(stopTokens) {
    var custonProperty = this._tryParseCustomPropertyDeclaration(stopTokens);
    if (custonProperty) {
      return custonProperty;
    }
    var node = this.create(Declaration);
    if (!node.setProperty(this._parseProperty())) {
      return null;
    }
    if (!this.accept(TokenType.Colon)) {
      return this.finish(node, ParseError.ColonExpected, [TokenType.Colon], stopTokens || [TokenType.SemiColon]);
    }
    if (this.prevToken) {
      node.colonPosition = this.prevToken.offset;
    }
    if (!node.setValue(this._parseExpr())) {
      return this.finish(node, ParseError.PropertyValueExpected);
    }
    node.addChild(this._parsePrio());
    if (this.peek(TokenType.SemiColon)) {
      node.semicolonPosition = this.token.offset;
    }
    return this.finish(node);
  };
  Parser2.prototype._tryParseCustomPropertyDeclaration = function(stopTokens) {
    if (!this.peekRegExp(TokenType.Ident, /^--/)) {
      return null;
    }
    var node = this.create(CustomPropertyDeclaration);
    if (!node.setProperty(this._parseProperty())) {
      return null;
    }
    if (!this.accept(TokenType.Colon)) {
      return this.finish(node, ParseError.ColonExpected, [TokenType.Colon]);
    }
    if (this.prevToken) {
      node.colonPosition = this.prevToken.offset;
    }
    var mark = this.mark();
    if (this.peek(TokenType.CurlyL)) {
      var propertySet = this.create(CustomPropertySet);
      var declarations = this._parseDeclarations(this._parseRuleSetDeclaration.bind(this));
      if (propertySet.setDeclarations(declarations) && !declarations.isErroneous(true)) {
        propertySet.addChild(this._parsePrio());
        if (this.peek(TokenType.SemiColon)) {
          this.finish(propertySet);
          node.setPropertySet(propertySet);
          node.semicolonPosition = this.token.offset;
          return this.finish(node);
        }
      }
      this.restoreAtMark(mark);
    }
    var expression = this._parseExpr();
    if (expression && !expression.isErroneous(true)) {
      this._parsePrio();
      if (this.peekOne.apply(this, __spreadArray(__spreadArray([], stopTokens || [], false), [TokenType.SemiColon, TokenType.EOF], false))) {
        node.setValue(expression);
        if (this.peek(TokenType.SemiColon)) {
          node.semicolonPosition = this.token.offset;
        }
        return this.finish(node);
      }
    }
    this.restoreAtMark(mark);
    node.addChild(this._parseCustomPropertyValue(stopTokens));
    node.addChild(this._parsePrio());
    if (isDefined(node.colonPosition) && this.token.offset === node.colonPosition + 1) {
      return this.finish(node, ParseError.PropertyValueExpected);
    }
    return this.finish(node);
  };
  Parser2.prototype._parseCustomPropertyValue = function(stopTokens) {
    var _this = this;
    if (stopTokens === void 0) {
      stopTokens = [TokenType.CurlyR];
    }
    var node = this.create(Node);
    var isTopLevel = function() {
      return curlyDepth === 0 && parensDepth === 0 && bracketsDepth === 0;
    };
    var onStopToken = function() {
      return stopTokens.indexOf(_this.token.type) !== -1;
    };
    var curlyDepth = 0;
    var parensDepth = 0;
    var bracketsDepth = 0;
    done:
      while (true) {
        switch (this.token.type) {
          case TokenType.SemiColon:
            if (isTopLevel()) {
              break done;
            }
            break;
          case TokenType.Exclamation:
            if (isTopLevel()) {
              break done;
            }
            break;
          case TokenType.CurlyL:
            curlyDepth++;
            break;
          case TokenType.CurlyR:
            curlyDepth--;
            if (curlyDepth < 0) {
              if (onStopToken() && parensDepth === 0 && bracketsDepth === 0) {
                break done;
              }
              return this.finish(node, ParseError.LeftCurlyExpected);
            }
            break;
          case TokenType.ParenthesisL:
            parensDepth++;
            break;
          case TokenType.ParenthesisR:
            parensDepth--;
            if (parensDepth < 0) {
              if (onStopToken() && bracketsDepth === 0 && curlyDepth === 0) {
                break done;
              }
              return this.finish(node, ParseError.LeftParenthesisExpected);
            }
            break;
          case TokenType.BracketL:
            bracketsDepth++;
            break;
          case TokenType.BracketR:
            bracketsDepth--;
            if (bracketsDepth < 0) {
              return this.finish(node, ParseError.LeftSquareBracketExpected);
            }
            break;
          case TokenType.BadString:
            break done;
          case TokenType.EOF:
            var error = ParseError.RightCurlyExpected;
            if (bracketsDepth > 0) {
              error = ParseError.RightSquareBracketExpected;
            } else if (parensDepth > 0) {
              error = ParseError.RightParenthesisExpected;
            }
            return this.finish(node, error);
        }
        this.consumeToken();
      }
    return this.finish(node);
  };
  Parser2.prototype._tryToParseDeclaration = function(stopTokens) {
    var mark = this.mark();
    if (this._parseProperty() && this.accept(TokenType.Colon)) {
      this.restoreAtMark(mark);
      return this._parseDeclaration(stopTokens);
    }
    this.restoreAtMark(mark);
    return null;
  };
  Parser2.prototype._parseProperty = function() {
    var node = this.create(Property);
    var mark = this.mark();
    if (this.acceptDelim("*") || this.acceptDelim("_")) {
      if (this.hasWhitespace()) {
        this.restoreAtMark(mark);
        return null;
      }
    }
    if (node.setIdentifier(this._parsePropertyIdentifier())) {
      return this.finish(node);
    }
    return null;
  };
  Parser2.prototype._parsePropertyIdentifier = function() {
    return this._parseIdent();
  };
  Parser2.prototype._parseCharset = function() {
    if (!this.peek(TokenType.Charset)) {
      return null;
    }
    var node = this.create(Node);
    this.consumeToken();
    if (!this.accept(TokenType.String)) {
      return this.finish(node, ParseError.IdentifierExpected);
    }
    if (!this.accept(TokenType.SemiColon)) {
      return this.finish(node, ParseError.SemiColonExpected);
    }
    return this.finish(node);
  };
  Parser2.prototype._parseImport = function() {
    if (!this.peekKeyword("@import")) {
      return null;
    }
    var node = this.create(Import);
    this.consumeToken();
    if (!node.addChild(this._parseURILiteral()) && !node.addChild(this._parseStringLiteral())) {
      return this.finish(node, ParseError.URIOrStringExpected);
    }
    if (!this.peek(TokenType.SemiColon) && !this.peek(TokenType.EOF)) {
      node.setMedialist(this._parseMediaQueryList());
    }
    return this.finish(node);
  };
  Parser2.prototype._parseNamespace = function() {
    if (!this.peekKeyword("@namespace")) {
      return null;
    }
    var node = this.create(Namespace);
    this.consumeToken();
    if (!node.addChild(this._parseURILiteral())) {
      node.addChild(this._parseIdent());
      if (!node.addChild(this._parseURILiteral()) && !node.addChild(this._parseStringLiteral())) {
        return this.finish(node, ParseError.URIExpected, [TokenType.SemiColon]);
      }
    }
    if (!this.accept(TokenType.SemiColon)) {
      return this.finish(node, ParseError.SemiColonExpected);
    }
    return this.finish(node);
  };
  Parser2.prototype._parseFontFace = function() {
    if (!this.peekKeyword("@font-face")) {
      return null;
    }
    var node = this.create(FontFace);
    this.consumeToken();
    return this._parseBody(node, this._parseRuleSetDeclaration.bind(this));
  };
  Parser2.prototype._parseViewPort = function() {
    if (!this.peekKeyword("@-ms-viewport") && !this.peekKeyword("@-o-viewport") && !this.peekKeyword("@viewport")) {
      return null;
    }
    var node = this.create(ViewPort);
    this.consumeToken();
    return this._parseBody(node, this._parseRuleSetDeclaration.bind(this));
  };
  Parser2.prototype._parseKeyframe = function() {
    if (!this.peekRegExp(TokenType.AtKeyword, this.keyframeRegex)) {
      return null;
    }
    var node = this.create(Keyframe);
    var atNode = this.create(Node);
    this.consumeToken();
    node.setKeyword(this.finish(atNode));
    if (atNode.matches("@-ms-keyframes")) {
      this.markError(atNode, ParseError.UnknownKeyword);
    }
    if (!node.setIdentifier(this._parseKeyframeIdent())) {
      return this.finish(node, ParseError.IdentifierExpected, [TokenType.CurlyR]);
    }
    return this._parseBody(node, this._parseKeyframeSelector.bind(this));
  };
  Parser2.prototype._parseKeyframeIdent = function() {
    return this._parseIdent([ReferenceType.Keyframe]);
  };
  Parser2.prototype._parseKeyframeSelector = function() {
    var node = this.create(KeyframeSelector);
    if (!node.addChild(this._parseIdent()) && !this.accept(TokenType.Percentage)) {
      return null;
    }
    while (this.accept(TokenType.Comma)) {
      if (!node.addChild(this._parseIdent()) && !this.accept(TokenType.Percentage)) {
        return this.finish(node, ParseError.PercentageExpected);
      }
    }
    return this._parseBody(node, this._parseRuleSetDeclaration.bind(this));
  };
  Parser2.prototype._tryParseKeyframeSelector = function() {
    var node = this.create(KeyframeSelector);
    var pos = this.mark();
    if (!node.addChild(this._parseIdent()) && !this.accept(TokenType.Percentage)) {
      return null;
    }
    while (this.accept(TokenType.Comma)) {
      if (!node.addChild(this._parseIdent()) && !this.accept(TokenType.Percentage)) {
        this.restoreAtMark(pos);
        return null;
      }
    }
    if (!this.peek(TokenType.CurlyL)) {
      this.restoreAtMark(pos);
      return null;
    }
    return this._parseBody(node, this._parseRuleSetDeclaration.bind(this));
  };
  Parser2.prototype._parseSupports = function(isNested) {
    if (isNested === void 0) {
      isNested = false;
    }
    if (!this.peekKeyword("@supports")) {
      return null;
    }
    var node = this.create(Supports);
    this.consumeToken();
    node.addChild(this._parseSupportsCondition());
    return this._parseBody(node, this._parseSupportsDeclaration.bind(this, isNested));
  };
  Parser2.prototype._parseSupportsDeclaration = function(isNested) {
    if (isNested === void 0) {
      isNested = false;
    }
    if (isNested) {
      return this._tryParseRuleset(true) || this._tryToParseDeclaration() || this._parseStylesheetStatement(true);
    }
    return this._parseStylesheetStatement(false);
  };
  Parser2.prototype._parseSupportsCondition = function() {
    var node = this.create(SupportsCondition);
    if (this.acceptIdent("not")) {
      node.addChild(this._parseSupportsConditionInParens());
    } else {
      node.addChild(this._parseSupportsConditionInParens());
      if (this.peekRegExp(TokenType.Ident, /^(and|or)$/i)) {
        var text = this.token.text.toLowerCase();
        while (this.acceptIdent(text)) {
          node.addChild(this._parseSupportsConditionInParens());
        }
      }
    }
    return this.finish(node);
  };
  Parser2.prototype._parseSupportsConditionInParens = function() {
    var node = this.create(SupportsCondition);
    if (this.accept(TokenType.ParenthesisL)) {
      if (this.prevToken) {
        node.lParent = this.prevToken.offset;
      }
      if (!node.addChild(this._tryToParseDeclaration([TokenType.ParenthesisR]))) {
        if (!this._parseSupportsCondition()) {
          return this.finish(node, ParseError.ConditionExpected);
        }
      }
      if (!this.accept(TokenType.ParenthesisR)) {
        return this.finish(node, ParseError.RightParenthesisExpected, [TokenType.ParenthesisR], []);
      }
      if (this.prevToken) {
        node.rParent = this.prevToken.offset;
      }
      return this.finish(node);
    } else if (this.peek(TokenType.Ident)) {
      var pos = this.mark();
      this.consumeToken();
      if (!this.hasWhitespace() && this.accept(TokenType.ParenthesisL)) {
        var openParentCount = 1;
        while (this.token.type !== TokenType.EOF && openParentCount !== 0) {
          if (this.token.type === TokenType.ParenthesisL) {
            openParentCount++;
          } else if (this.token.type === TokenType.ParenthesisR) {
            openParentCount--;
          }
          this.consumeToken();
        }
        return this.finish(node);
      } else {
        this.restoreAtMark(pos);
      }
    }
    return this.finish(node, ParseError.LeftParenthesisExpected, [], [TokenType.ParenthesisL]);
  };
  Parser2.prototype._parseMediaDeclaration = function(isNested) {
    if (isNested === void 0) {
      isNested = false;
    }
    if (isNested) {
      return this._tryParseRuleset(true) || this._tryToParseDeclaration() || this._parseStylesheetStatement(true);
    }
    return this._parseStylesheetStatement(false);
  };
  Parser2.prototype._parseMedia = function(isNested) {
    if (isNested === void 0) {
      isNested = false;
    }
    if (!this.peekKeyword("@media")) {
      return null;
    }
    var node = this.create(Media);
    this.consumeToken();
    if (!node.addChild(this._parseMediaQueryList())) {
      return this.finish(node, ParseError.MediaQueryExpected);
    }
    return this._parseBody(node, this._parseMediaDeclaration.bind(this, isNested));
  };
  Parser2.prototype._parseMediaQueryList = function() {
    var node = this.create(Medialist);
    if (!node.addChild(this._parseMediaQuery())) {
      return this.finish(node, ParseError.MediaQueryExpected);
    }
    while (this.accept(TokenType.Comma)) {
      if (!node.addChild(this._parseMediaQuery())) {
        return this.finish(node, ParseError.MediaQueryExpected);
      }
    }
    return this.finish(node);
  };
  Parser2.prototype._parseMediaQuery = function() {
    var node = this.create(MediaQuery);
    var pos = this.mark();
    this.acceptIdent("not");
    if (!this.peek(TokenType.ParenthesisL)) {
      if (this.acceptIdent("only")) {
      }
      if (!node.addChild(this._parseIdent())) {
        return null;
      }
      if (this.acceptIdent("and")) {
        node.addChild(this._parseMediaCondition());
      }
    } else {
      this.restoreAtMark(pos);
      node.addChild(this._parseMediaCondition());
    }
    return this.finish(node);
  };
  Parser2.prototype._parseRatio = function() {
    var pos = this.mark();
    var node = this.create(RatioValue);
    if (!this._parseNumeric()) {
      return null;
    }
    if (!this.acceptDelim("/")) {
      this.restoreAtMark(pos);
      return null;
    }
    if (!this._parseNumeric()) {
      return this.finish(node, ParseError.NumberExpected);
    }
    return this.finish(node);
  };
  Parser2.prototype._parseMediaCondition = function() {
    var node = this.create(MediaCondition);
    this.acceptIdent("not");
    var parseExpression = true;
    while (parseExpression) {
      if (!this.accept(TokenType.ParenthesisL)) {
        return this.finish(node, ParseError.LeftParenthesisExpected, [], [TokenType.CurlyL]);
      }
      if (this.peek(TokenType.ParenthesisL) || this.peekIdent("not")) {
        node.addChild(this._parseMediaCondition());
      } else {
        node.addChild(this._parseMediaFeature());
      }
      if (!this.accept(TokenType.ParenthesisR)) {
        return this.finish(node, ParseError.RightParenthesisExpected, [], [TokenType.CurlyL]);
      }
      parseExpression = this.acceptIdent("and") || this.acceptIdent("or");
    }
    return this.finish(node);
  };
  Parser2.prototype._parseMediaFeature = function() {
    var _this = this;
    var resyncStopToken = [TokenType.ParenthesisR];
    var node = this.create(MediaFeature);
    var parseRangeOperator = function() {
      if (_this.acceptDelim("<") || _this.acceptDelim(">")) {
        if (!_this.hasWhitespace()) {
          _this.acceptDelim("=");
        }
        return true;
      } else if (_this.acceptDelim("=")) {
        return true;
      }
      return false;
    };
    if (node.addChild(this._parseMediaFeatureName())) {
      if (this.accept(TokenType.Colon)) {
        if (!node.addChild(this._parseMediaFeatureValue())) {
          return this.finish(node, ParseError.TermExpected, [], resyncStopToken);
        }
      } else if (parseRangeOperator()) {
        if (!node.addChild(this._parseMediaFeatureValue())) {
          return this.finish(node, ParseError.TermExpected, [], resyncStopToken);
        }
        if (parseRangeOperator()) {
          if (!node.addChild(this._parseMediaFeatureValue())) {
            return this.finish(node, ParseError.TermExpected, [], resyncStopToken);
          }
        }
      } else {
      }
    } else if (node.addChild(this._parseMediaFeatureValue())) {
      if (!parseRangeOperator()) {
        return this.finish(node, ParseError.OperatorExpected, [], resyncStopToken);
      }
      if (!node.addChild(this._parseMediaFeatureName())) {
        return this.finish(node, ParseError.IdentifierExpected, [], resyncStopToken);
      }
      if (parseRangeOperator()) {
        if (!node.addChild(this._parseMediaFeatureValue())) {
          return this.finish(node, ParseError.TermExpected, [], resyncStopToken);
        }
      }
    } else {
      return this.finish(node, ParseError.IdentifierExpected, [], resyncStopToken);
    }
    return this.finish(node);
  };
  Parser2.prototype._parseMediaFeatureName = function() {
    return this._parseIdent();
  };
  Parser2.prototype._parseMediaFeatureValue = function() {
    return this._parseRatio() || this._parseTermExpression();
  };
  Parser2.prototype._parseMedium = function() {
    var node = this.create(Node);
    if (node.addChild(this._parseIdent())) {
      return this.finish(node);
    } else {
      return null;
    }
  };
  Parser2.prototype._parsePageDeclaration = function() {
    return this._parsePageMarginBox() || this._parseRuleSetDeclaration();
  };
  Parser2.prototype._parsePage = function() {
    if (!this.peekKeyword("@page")) {
      return null;
    }
    var node = this.create(Page);
    this.consumeToken();
    if (node.addChild(this._parsePageSelector())) {
      while (this.accept(TokenType.Comma)) {
        if (!node.addChild(this._parsePageSelector())) {
          return this.finish(node, ParseError.IdentifierExpected);
        }
      }
    }
    return this._parseBody(node, this._parsePageDeclaration.bind(this));
  };
  Parser2.prototype._parsePageMarginBox = function() {
    if (!this.peek(TokenType.AtKeyword)) {
      return null;
    }
    var node = this.create(PageBoxMarginBox);
    if (!this.acceptOneKeyword(pageBoxDirectives)) {
      this.markError(node, ParseError.UnknownAtRule, [], [TokenType.CurlyL]);
    }
    return this._parseBody(node, this._parseRuleSetDeclaration.bind(this));
  };
  Parser2.prototype._parsePageSelector = function() {
    if (!this.peek(TokenType.Ident) && !this.peek(TokenType.Colon)) {
      return null;
    }
    var node = this.create(Node);
    node.addChild(this._parseIdent());
    if (this.accept(TokenType.Colon)) {
      if (!node.addChild(this._parseIdent())) {
        return this.finish(node, ParseError.IdentifierExpected);
      }
    }
    return this.finish(node);
  };
  Parser2.prototype._parseDocument = function() {
    if (!this.peekKeyword("@-moz-document")) {
      return null;
    }
    var node = this.create(Document);
    this.consumeToken();
    this.resync([], [TokenType.CurlyL]);
    return this._parseBody(node, this._parseStylesheetStatement.bind(this));
  };
  Parser2.prototype._parseUnknownAtRule = function() {
    if (!this.peek(TokenType.AtKeyword)) {
      return null;
    }
    var node = this.create(UnknownAtRule);
    node.addChild(this._parseUnknownAtRuleName());
    var isTopLevel = function() {
      return curlyDepth === 0 && parensDepth === 0 && bracketsDepth === 0;
    };
    var curlyLCount = 0;
    var curlyDepth = 0;
    var parensDepth = 0;
    var bracketsDepth = 0;
    done:
      while (true) {
        switch (this.token.type) {
          case TokenType.SemiColon:
            if (isTopLevel()) {
              break done;
            }
            break;
          case TokenType.EOF:
            if (curlyDepth > 0) {
              return this.finish(node, ParseError.RightCurlyExpected);
            } else if (bracketsDepth > 0) {
              return this.finish(node, ParseError.RightSquareBracketExpected);
            } else if (parensDepth > 0) {
              return this.finish(node, ParseError.RightParenthesisExpected);
            } else {
              return this.finish(node);
            }
          case TokenType.CurlyL:
            curlyLCount++;
            curlyDepth++;
            break;
          case TokenType.CurlyR:
            curlyDepth--;
            if (curlyLCount > 0 && curlyDepth === 0) {
              this.consumeToken();
              if (bracketsDepth > 0) {
                return this.finish(node, ParseError.RightSquareBracketExpected);
              } else if (parensDepth > 0) {
                return this.finish(node, ParseError.RightParenthesisExpected);
              }
              break done;
            }
            if (curlyDepth < 0) {
              if (parensDepth === 0 && bracketsDepth === 0) {
                break done;
              }
              return this.finish(node, ParseError.LeftCurlyExpected);
            }
            break;
          case TokenType.ParenthesisL:
            parensDepth++;
            break;
          case TokenType.ParenthesisR:
            parensDepth--;
            if (parensDepth < 0) {
              return this.finish(node, ParseError.LeftParenthesisExpected);
            }
            break;
          case TokenType.BracketL:
            bracketsDepth++;
            break;
          case TokenType.BracketR:
            bracketsDepth--;
            if (bracketsDepth < 0) {
              return this.finish(node, ParseError.LeftSquareBracketExpected);
            }
            break;
        }
        this.consumeToken();
      }
    return node;
  };
  Parser2.prototype._parseUnknownAtRuleName = function() {
    var node = this.create(Node);
    if (this.accept(TokenType.AtKeyword)) {
      return this.finish(node);
    }
    return node;
  };
  Parser2.prototype._parseOperator = function() {
    if (this.peekDelim("/") || this.peekDelim("*") || this.peekDelim("+") || this.peekDelim("-") || this.peek(TokenType.Dashmatch) || this.peek(TokenType.Includes) || this.peek(TokenType.SubstringOperator) || this.peek(TokenType.PrefixOperator) || this.peek(TokenType.SuffixOperator) || this.peekDelim("=")) {
      var node = this.createNode(NodeType.Operator);
      this.consumeToken();
      return this.finish(node);
    } else {
      return null;
    }
  };
  Parser2.prototype._parseUnaryOperator = function() {
    if (!this.peekDelim("+") && !this.peekDelim("-")) {
      return null;
    }
    var node = this.create(Node);
    this.consumeToken();
    return this.finish(node);
  };
  Parser2.prototype._parseCombinator = function() {
    if (this.peekDelim(">")) {
      var node = this.create(Node);
      this.consumeToken();
      var mark = this.mark();
      if (!this.hasWhitespace() && this.acceptDelim(">")) {
        if (!this.hasWhitespace() && this.acceptDelim(">")) {
          node.type = NodeType.SelectorCombinatorShadowPiercingDescendant;
          return this.finish(node);
        }
        this.restoreAtMark(mark);
      }
      node.type = NodeType.SelectorCombinatorParent;
      return this.finish(node);
    } else if (this.peekDelim("+")) {
      var node = this.create(Node);
      this.consumeToken();
      node.type = NodeType.SelectorCombinatorSibling;
      return this.finish(node);
    } else if (this.peekDelim("~")) {
      var node = this.create(Node);
      this.consumeToken();
      node.type = NodeType.SelectorCombinatorAllSiblings;
      return this.finish(node);
    } else if (this.peekDelim("/")) {
      var node = this.create(Node);
      this.consumeToken();
      var mark = this.mark();
      if (!this.hasWhitespace() && this.acceptIdent("deep") && !this.hasWhitespace() && this.acceptDelim("/")) {
        node.type = NodeType.SelectorCombinatorShadowPiercingDescendant;
        return this.finish(node);
      }
      this.restoreAtMark(mark);
    }
    return null;
  };
  Parser2.prototype._parseSimpleSelector = function() {
    var node = this.create(SimpleSelector);
    var c = 0;
    if (node.addChild(this._parseElementName())) {
      c++;
    }
    while ((c === 0 || !this.hasWhitespace()) && node.addChild(this._parseSimpleSelectorBody())) {
      c++;
    }
    return c > 0 ? this.finish(node) : null;
  };
  Parser2.prototype._parseSimpleSelectorBody = function() {
    return this._parsePseudo() || this._parseHash() || this._parseClass() || this._parseAttrib();
  };
  Parser2.prototype._parseSelectorIdent = function() {
    return this._parseIdent();
  };
  Parser2.prototype._parseHash = function() {
    if (!this.peek(TokenType.Hash) && !this.peekDelim("#")) {
      return null;
    }
    var node = this.createNode(NodeType.IdentifierSelector);
    if (this.acceptDelim("#")) {
      if (this.hasWhitespace() || !node.addChild(this._parseSelectorIdent())) {
        return this.finish(node, ParseError.IdentifierExpected);
      }
    } else {
      this.consumeToken();
    }
    return this.finish(node);
  };
  Parser2.prototype._parseClass = function() {
    if (!this.peekDelim(".")) {
      return null;
    }
    var node = this.createNode(NodeType.ClassSelector);
    this.consumeToken();
    if (this.hasWhitespace() || !node.addChild(this._parseSelectorIdent())) {
      return this.finish(node, ParseError.IdentifierExpected);
    }
    return this.finish(node);
  };
  Parser2.prototype._parseElementName = function() {
    var pos = this.mark();
    var node = this.createNode(NodeType.ElementNameSelector);
    node.addChild(this._parseNamespacePrefix());
    if (!node.addChild(this._parseSelectorIdent()) && !this.acceptDelim("*")) {
      this.restoreAtMark(pos);
      return null;
    }
    return this.finish(node);
  };
  Parser2.prototype._parseNamespacePrefix = function() {
    var pos = this.mark();
    var node = this.createNode(NodeType.NamespacePrefix);
    if (!node.addChild(this._parseIdent()) && !this.acceptDelim("*")) {
    }
    if (!this.acceptDelim("|")) {
      this.restoreAtMark(pos);
      return null;
    }
    return this.finish(node);
  };
  Parser2.prototype._parseAttrib = function() {
    if (!this.peek(TokenType.BracketL)) {
      return null;
    }
    var node = this.create(AttributeSelector);
    this.consumeToken();
    node.setNamespacePrefix(this._parseNamespacePrefix());
    if (!node.setIdentifier(this._parseIdent())) {
      return this.finish(node, ParseError.IdentifierExpected);
    }
    if (node.setOperator(this._parseOperator())) {
      node.setValue(this._parseBinaryExpr());
      this.acceptIdent("i");
      this.acceptIdent("s");
    }
    if (!this.accept(TokenType.BracketR)) {
      return this.finish(node, ParseError.RightSquareBracketExpected);
    }
    return this.finish(node);
  };
  Parser2.prototype._parsePseudo = function() {
    var _this = this;
    var node = this._tryParsePseudoIdentifier();
    if (node) {
      if (!this.hasWhitespace() && this.accept(TokenType.ParenthesisL)) {
        var tryAsSelector = function() {
          var selectors = _this.create(Node);
          if (!selectors.addChild(_this._parseSelector(false))) {
            return null;
          }
          while (_this.accept(TokenType.Comma) && selectors.addChild(_this._parseSelector(false))) {
          }
          if (_this.peek(TokenType.ParenthesisR)) {
            return _this.finish(selectors);
          }
          return null;
        };
        node.addChild(this.try(tryAsSelector) || this._parseBinaryExpr());
        if (!this.accept(TokenType.ParenthesisR)) {
          return this.finish(node, ParseError.RightParenthesisExpected);
        }
      }
      return this.finish(node);
    }
    return null;
  };
  Parser2.prototype._tryParsePseudoIdentifier = function() {
    if (!this.peek(TokenType.Colon)) {
      return null;
    }
    var pos = this.mark();
    var node = this.createNode(NodeType.PseudoSelector);
    this.consumeToken();
    if (this.hasWhitespace()) {
      this.restoreAtMark(pos);
      return null;
    }
    this.accept(TokenType.Colon);
    if (this.hasWhitespace() || !node.addChild(this._parseIdent())) {
      return this.finish(node, ParseError.IdentifierExpected);
    }
    return this.finish(node);
  };
  Parser2.prototype._tryParsePrio = function() {
    var mark = this.mark();
    var prio = this._parsePrio();
    if (prio) {
      return prio;
    }
    this.restoreAtMark(mark);
    return null;
  };
  Parser2.prototype._parsePrio = function() {
    if (!this.peek(TokenType.Exclamation)) {
      return null;
    }
    var node = this.createNode(NodeType.Prio);
    if (this.accept(TokenType.Exclamation) && this.acceptIdent("important")) {
      return this.finish(node);
    }
    return null;
  };
  Parser2.prototype._parseExpr = function(stopOnComma) {
    if (stopOnComma === void 0) {
      stopOnComma = false;
    }
    var node = this.create(Expression);
    if (!node.addChild(this._parseBinaryExpr())) {
      return null;
    }
    while (true) {
      if (this.peek(TokenType.Comma)) {
        if (stopOnComma) {
          return this.finish(node);
        }
        this.consumeToken();
      } else if (!this.hasWhitespace()) {
        break;
      }
      if (!node.addChild(this._parseBinaryExpr())) {
        break;
      }
    }
    return this.finish(node);
  };
  Parser2.prototype._parseUnicodeRange = function() {
    if (!this.peekIdent("u")) {
      return null;
    }
    var node = this.create(UnicodeRange);
    if (!this.acceptUnicodeRange()) {
      return null;
    }
    return this.finish(node);
  };
  Parser2.prototype._parseNamedLine = function() {
    if (!this.peek(TokenType.BracketL)) {
      return null;
    }
    var node = this.createNode(NodeType.GridLine);
    this.consumeToken();
    while (node.addChild(this._parseIdent())) {
    }
    if (!this.accept(TokenType.BracketR)) {
      return this.finish(node, ParseError.RightSquareBracketExpected);
    }
    return this.finish(node);
  };
  Parser2.prototype._parseBinaryExpr = function(preparsedLeft, preparsedOper) {
    var node = this.create(BinaryExpression);
    if (!node.setLeft(preparsedLeft || this._parseTerm())) {
      return null;
    }
    if (!node.setOperator(preparsedOper || this._parseOperator())) {
      return this.finish(node);
    }
    if (!node.setRight(this._parseTerm())) {
      return this.finish(node, ParseError.TermExpected);
    }
    node = this.finish(node);
    var operator = this._parseOperator();
    if (operator) {
      node = this._parseBinaryExpr(node, operator);
    }
    return this.finish(node);
  };
  Parser2.prototype._parseTerm = function() {
    var node = this.create(Term);
    node.setOperator(this._parseUnaryOperator());
    if (node.setExpression(this._parseTermExpression())) {
      return this.finish(node);
    }
    return null;
  };
  Parser2.prototype._parseTermExpression = function() {
    return this._parseURILiteral() || this._parseUnicodeRange() || this._parseFunction() || this._parseIdent() || this._parseStringLiteral() || this._parseNumeric() || this._parseHexColor() || this._parseOperation() || this._parseNamedLine();
  };
  Parser2.prototype._parseOperation = function() {
    if (!this.peek(TokenType.ParenthesisL)) {
      return null;
    }
    var node = this.create(Node);
    this.consumeToken();
    node.addChild(this._parseExpr());
    if (!this.accept(TokenType.ParenthesisR)) {
      return this.finish(node, ParseError.RightParenthesisExpected);
    }
    return this.finish(node);
  };
  Parser2.prototype._parseNumeric = function() {
    if (this.peek(TokenType.Num) || this.peek(TokenType.Percentage) || this.peek(TokenType.Resolution) || this.peek(TokenType.Length) || this.peek(TokenType.EMS) || this.peek(TokenType.EXS) || this.peek(TokenType.Angle) || this.peek(TokenType.Time) || this.peek(TokenType.Dimension) || this.peek(TokenType.Freq)) {
      var node = this.create(NumericValue);
      this.consumeToken();
      return this.finish(node);
    }
    return null;
  };
  Parser2.prototype._parseStringLiteral = function() {
    if (!this.peek(TokenType.String) && !this.peek(TokenType.BadString)) {
      return null;
    }
    var node = this.createNode(NodeType.StringLiteral);
    this.consumeToken();
    return this.finish(node);
  };
  Parser2.prototype._parseURILiteral = function() {
    if (!this.peekRegExp(TokenType.Ident, /^url(-prefix)?$/i)) {
      return null;
    }
    var pos = this.mark();
    var node = this.createNode(NodeType.URILiteral);
    this.accept(TokenType.Ident);
    if (this.hasWhitespace() || !this.peek(TokenType.ParenthesisL)) {
      this.restoreAtMark(pos);
      return null;
    }
    this.scanner.inURL = true;
    this.consumeToken();
    node.addChild(this._parseURLArgument());
    this.scanner.inURL = false;
    if (!this.accept(TokenType.ParenthesisR)) {
      return this.finish(node, ParseError.RightParenthesisExpected);
    }
    return this.finish(node);
  };
  Parser2.prototype._parseURLArgument = function() {
    var node = this.create(Node);
    if (!this.accept(TokenType.String) && !this.accept(TokenType.BadString) && !this.acceptUnquotedString()) {
      return null;
    }
    return this.finish(node);
  };
  Parser2.prototype._parseIdent = function(referenceTypes) {
    if (!this.peek(TokenType.Ident)) {
      return null;
    }
    var node = this.create(Identifier);
    if (referenceTypes) {
      node.referenceTypes = referenceTypes;
    }
    node.isCustomProperty = this.peekRegExp(TokenType.Ident, /^--/);
    this.consumeToken();
    return this.finish(node);
  };
  Parser2.prototype._parseFunction = function() {
    var pos = this.mark();
    var node = this.create(Function);
    if (!node.setIdentifier(this._parseFunctionIdentifier())) {
      return null;
    }
    if (this.hasWhitespace() || !this.accept(TokenType.ParenthesisL)) {
      this.restoreAtMark(pos);
      return null;
    }
    if (node.getArguments().addChild(this._parseFunctionArgument())) {
      while (this.accept(TokenType.Comma)) {
        if (this.peek(TokenType.ParenthesisR)) {
          break;
        }
        if (!node.getArguments().addChild(this._parseFunctionArgument())) {
          this.markError(node, ParseError.ExpressionExpected);
        }
      }
    }
    if (!this.accept(TokenType.ParenthesisR)) {
      return this.finish(node, ParseError.RightParenthesisExpected);
    }
    return this.finish(node);
  };
  Parser2.prototype._parseFunctionIdentifier = function() {
    if (!this.peek(TokenType.Ident)) {
      return null;
    }
    var node = this.create(Identifier);
    node.referenceTypes = [ReferenceType.Function];
    if (this.acceptIdent("progid")) {
      if (this.accept(TokenType.Colon)) {
        while (this.accept(TokenType.Ident) && this.acceptDelim(".")) {
        }
      }
      return this.finish(node);
    }
    this.consumeToken();
    return this.finish(node);
  };
  Parser2.prototype._parseFunctionArgument = function() {
    var node = this.create(FunctionArgument);
    if (node.setValue(this._parseExpr(true))) {
      return this.finish(node);
    }
    return null;
  };
  Parser2.prototype._parseHexColor = function() {
    if (this.peekRegExp(TokenType.Hash, /^#([A-Fa-f0-9]{3}|[A-Fa-f0-9]{4}|[A-Fa-f0-9]{6}|[A-Fa-f0-9]{8})$/g)) {
      var node = this.create(HexColorValue);
      this.consumeToken();
      return this.finish(node);
    } else {
      return null;
    }
  };
  return Parser2;
}();

// node_modules/vscode-css-languageservice/lib/esm/utils/arrays.js
function findFirst(array, p) {
  var low = 0, high = array.length;
  if (high === 0) {
    return 0;
  }
  while (low < high) {
    var mid = Math.floor((low + high) / 2);
    if (p(array[mid])) {
      high = mid;
    } else {
      low = mid + 1;
    }
  }
  return low;
}
function includes(array, item) {
  return array.indexOf(item) !== -1;
}
function union() {
  var arrays = [];
  for (var _i = 0; _i < arguments.length; _i++) {
    arrays[_i] = arguments[_i];
  }
  var result = [];
  for (var _a2 = 0, arrays_1 = arrays; _a2 < arrays_1.length; _a2++) {
    var array = arrays_1[_a2];
    for (var _b = 0, array_1 = array; _b < array_1.length; _b++) {
      var item = array_1[_b];
      if (!includes(result, item)) {
        result.push(item);
      }
    }
  }
  return result;
}

// node_modules/vscode-css-languageservice/lib/esm/parser/cssSymbolScope.js
var __extends2 = function() {
  var extendStatics = function(d, b) {
    extendStatics = Object.setPrototypeOf || { __proto__: [] } instanceof Array && function(d2, b2) {
      d2.__proto__ = b2;
    } || function(d2, b2) {
      for (var p in b2)
        if (Object.prototype.hasOwnProperty.call(b2, p))
          d2[p] = b2[p];
    };
    return extendStatics(d, b);
  };
  return function(d, b) {
    if (typeof b !== "function" && b !== null)
      throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
    extendStatics(d, b);
    function __() {
      this.constructor = d;
    }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
  };
}();
var Scope = function() {
  function Scope2(offset, length) {
    this.offset = offset;
    this.length = length;
    this.symbols = [];
    this.parent = null;
    this.children = [];
  }
  Scope2.prototype.addChild = function(scope) {
    this.children.push(scope);
    scope.setParent(this);
  };
  Scope2.prototype.setParent = function(scope) {
    this.parent = scope;
  };
  Scope2.prototype.findScope = function(offset, length) {
    if (length === void 0) {
      length = 0;
    }
    if (this.offset <= offset && this.offset + this.length > offset + length || this.offset === offset && this.length === length) {
      return this.findInScope(offset, length);
    }
    return null;
  };
  Scope2.prototype.findInScope = function(offset, length) {
    if (length === void 0) {
      length = 0;
    }
    var end = offset + length;
    var idx = findFirst(this.children, function(s) {
      return s.offset > end;
    });
    if (idx === 0) {
      return this;
    }
    var res = this.children[idx - 1];
    if (res.offset <= offset && res.offset + res.length >= offset + length) {
      return res.findInScope(offset, length);
    }
    return this;
  };
  Scope2.prototype.addSymbol = function(symbol) {
    this.symbols.push(symbol);
  };
  Scope2.prototype.getSymbol = function(name, type) {
    for (var index = 0; index < this.symbols.length; index++) {
      var symbol = this.symbols[index];
      if (symbol.name === name && symbol.type === type) {
        return symbol;
      }
    }
    return null;
  };
  Scope2.prototype.getSymbols = function() {
    return this.symbols;
  };
  return Scope2;
}();
var GlobalScope = function(_super) {
  __extends2(GlobalScope2, _super);
  function GlobalScope2() {
    return _super.call(this, 0, Number.MAX_VALUE) || this;
  }
  return GlobalScope2;
}(Scope);
var Symbol2 = function() {
  function Symbol3(name, value, node, type) {
    this.name = name;
    this.value = value;
    this.node = node;
    this.type = type;
  }
  return Symbol3;
}();
var ScopeBuilder = function() {
  function ScopeBuilder2(scope) {
    this.scope = scope;
  }
  ScopeBuilder2.prototype.addSymbol = function(node, name, value, type) {
    if (node.offset !== -1) {
      var current = this.scope.findScope(node.offset, node.length);
      if (current) {
        current.addSymbol(new Symbol2(name, value, node, type));
      }
    }
  };
  ScopeBuilder2.prototype.addScope = function(node) {
    if (node.offset !== -1) {
      var current = this.scope.findScope(node.offset, node.length);
      if (current && (current.offset !== node.offset || current.length !== node.length)) {
        var newScope = new Scope(node.offset, node.length);
        current.addChild(newScope);
        return newScope;
      }
      return current;
    }
    return null;
  };
  ScopeBuilder2.prototype.addSymbolToChildScope = function(scopeNode, node, name, value, type) {
    if (scopeNode && scopeNode.offset !== -1) {
      var current = this.addScope(scopeNode);
      if (current) {
        current.addSymbol(new Symbol2(name, value, node, type));
      }
    }
  };
  ScopeBuilder2.prototype.visitNode = function(node) {
    switch (node.type) {
      case NodeType.Keyframe:
        this.addSymbol(node, node.getName(), void 0, ReferenceType.Keyframe);
        return true;
      case NodeType.CustomPropertyDeclaration:
        return this.visitCustomPropertyDeclarationNode(node);
      case NodeType.VariableDeclaration:
        return this.visitVariableDeclarationNode(node);
      case NodeType.Ruleset:
        return this.visitRuleSet(node);
      case NodeType.MixinDeclaration:
        this.addSymbol(node, node.getName(), void 0, ReferenceType.Mixin);
        return true;
      case NodeType.FunctionDeclaration:
        this.addSymbol(node, node.getName(), void 0, ReferenceType.Function);
        return true;
      case NodeType.FunctionParameter: {
        return this.visitFunctionParameterNode(node);
      }
      case NodeType.Declarations:
        this.addScope(node);
        return true;
      case NodeType.For:
        var forNode = node;
        var scopeNode = forNode.getDeclarations();
        if (scopeNode && forNode.variable) {
          this.addSymbolToChildScope(scopeNode, forNode.variable, forNode.variable.getName(), void 0, ReferenceType.Variable);
        }
        return true;
      case NodeType.Each: {
        var eachNode = node;
        var scopeNode_1 = eachNode.getDeclarations();
        if (scopeNode_1) {
          var variables = eachNode.getVariables().getChildren();
          for (var _i = 0, variables_1 = variables; _i < variables_1.length; _i++) {
            var variable = variables_1[_i];
            this.addSymbolToChildScope(scopeNode_1, variable, variable.getName(), void 0, ReferenceType.Variable);
          }
        }
        return true;
      }
    }
    return true;
  };
  ScopeBuilder2.prototype.visitRuleSet = function(node) {
    var current = this.scope.findScope(node.offset, node.length);
    if (current) {
      for (var _i = 0, _a2 = node.getSelectors().getChildren(); _i < _a2.length; _i++) {
        var child = _a2[_i];
        if (child instanceof Selector) {
          if (child.getChildren().length === 1) {
            current.addSymbol(new Symbol2(child.getChild(0).getText(), void 0, child, ReferenceType.Rule));
          }
        }
      }
    }
    return true;
  };
  ScopeBuilder2.prototype.visitVariableDeclarationNode = function(node) {
    var value = node.getValue() ? node.getValue().getText() : void 0;
    this.addSymbol(node, node.getName(), value, ReferenceType.Variable);
    return true;
  };
  ScopeBuilder2.prototype.visitFunctionParameterNode = function(node) {
    var scopeNode = node.getParent().getDeclarations();
    if (scopeNode) {
      var valueNode = node.getDefaultValue();
      var value = valueNode ? valueNode.getText() : void 0;
      this.addSymbolToChildScope(scopeNode, node, node.getName(), value, ReferenceType.Variable);
    }
    return true;
  };
  ScopeBuilder2.prototype.visitCustomPropertyDeclarationNode = function(node) {
    var value = node.getValue() ? node.getValue().getText() : "";
    this.addCSSVariable(node.getProperty(), node.getProperty().getName(), value, ReferenceType.Variable);
    return true;
  };
  ScopeBuilder2.prototype.addCSSVariable = function(node, name, value, type) {
    if (node.offset !== -1) {
      this.scope.addSymbol(new Symbol2(name, value, node, type));
    }
  };
  return ScopeBuilder2;
}();
var Symbols = function() {
  function Symbols2(node) {
    this.global = new GlobalScope();
    node.acceptVisitor(new ScopeBuilder(this.global));
  }
  Symbols2.prototype.findSymbolsAtOffset = function(offset, referenceType) {
    var scope = this.global.findScope(offset, 0);
    var result = [];
    var names = {};
    while (scope) {
      var symbols = scope.getSymbols();
      for (var i = 0; i < symbols.length; i++) {
        var symbol = symbols[i];
        if (symbol.type === referenceType && !names[symbol.name]) {
          result.push(symbol);
          names[symbol.name] = true;
        }
      }
      scope = scope.parent;
    }
    return result;
  };
  Symbols2.prototype.internalFindSymbol = function(node, referenceTypes) {
    var scopeNode = node;
    if (node.parent instanceof FunctionParameter && node.parent.getParent() instanceof BodyDeclaration) {
      scopeNode = node.parent.getParent().getDeclarations();
    }
    if (node.parent instanceof FunctionArgument && node.parent.getParent() instanceof Function) {
      var funcId = node.parent.getParent().getIdentifier();
      if (funcId) {
        var functionSymbol = this.internalFindSymbol(funcId, [ReferenceType.Function]);
        if (functionSymbol) {
          scopeNode = functionSymbol.node.getDeclarations();
        }
      }
    }
    if (!scopeNode) {
      return null;
    }
    var name = node.getText();
    var scope = this.global.findScope(scopeNode.offset, scopeNode.length);
    while (scope) {
      for (var index = 0; index < referenceTypes.length; index++) {
        var type = referenceTypes[index];
        var symbol = scope.getSymbol(name, type);
        if (symbol) {
          return symbol;
        }
      }
      scope = scope.parent;
    }
    return null;
  };
  Symbols2.prototype.evaluateReferenceTypes = function(node) {
    if (node instanceof Identifier) {
      var referenceTypes = node.referenceTypes;
      if (referenceTypes) {
        return referenceTypes;
      } else {
        if (node.isCustomProperty) {
          return [ReferenceType.Variable];
        }
        var decl = getParentDeclaration(node);
        if (decl) {
          var propertyName = decl.getNonPrefixedPropertyName();
          if ((propertyName === "animation" || propertyName === "animation-name") && decl.getValue() && decl.getValue().offset === node.offset) {
            return [ReferenceType.Keyframe];
          }
        }
      }
    } else if (node instanceof Variable) {
      return [ReferenceType.Variable];
    }
    var selector = node.findAParent(NodeType.Selector, NodeType.ExtendsReference);
    if (selector) {
      return [ReferenceType.Rule];
    }
    return null;
  };
  Symbols2.prototype.findSymbolFromNode = function(node) {
    if (!node) {
      return null;
    }
    while (node.type === NodeType.Interpolation) {
      node = node.getParent();
    }
    var referenceTypes = this.evaluateReferenceTypes(node);
    if (referenceTypes) {
      return this.internalFindSymbol(node, referenceTypes);
    }
    return null;
  };
  Symbols2.prototype.matchesSymbol = function(node, symbol) {
    if (!node) {
      return false;
    }
    while (node.type === NodeType.Interpolation) {
      node = node.getParent();
    }
    if (!node.matches(symbol.name)) {
      return false;
    }
    var referenceTypes = this.evaluateReferenceTypes(node);
    if (!referenceTypes || referenceTypes.indexOf(symbol.type) === -1) {
      return false;
    }
    var nodeSymbol = this.internalFindSymbol(node, referenceTypes);
    return nodeSymbol === symbol;
  };
  Symbols2.prototype.findSymbol = function(name, type, offset) {
    var scope = this.global.findScope(offset);
    while (scope) {
      var symbol = scope.getSymbol(name, type);
      if (symbol) {
        return symbol;
      }
      scope = scope.parent;
    }
    return null;
  };
  return Symbols2;
}();

// node_modules/vscode-uri/lib/esm/index.js
var LIB;
LIB = (() => {
  "use strict";
  var t = { 470: (t2) => {
    function e2(t3) {
      if ("string" != typeof t3)
        throw new TypeError("Path must be a string. Received " + JSON.stringify(t3));
    }
    function r2(t3, e3) {
      for (var r3, n2 = "", o = 0, i = -1, a2 = 0, h = 0; h <= t3.length; ++h) {
        if (h < t3.length)
          r3 = t3.charCodeAt(h);
        else {
          if (47 === r3)
            break;
          r3 = 47;
        }
        if (47 === r3) {
          if (i === h - 1 || 1 === a2)
            ;
          else if (i !== h - 1 && 2 === a2) {
            if (n2.length < 2 || 2 !== o || 46 !== n2.charCodeAt(n2.length - 1) || 46 !== n2.charCodeAt(n2.length - 2)) {
              if (n2.length > 2) {
                var s = n2.lastIndexOf("/");
                if (s !== n2.length - 1) {
                  -1 === s ? (n2 = "", o = 0) : o = (n2 = n2.slice(0, s)).length - 1 - n2.lastIndexOf("/"), i = h, a2 = 0;
                  continue;
                }
              } else if (2 === n2.length || 1 === n2.length) {
                n2 = "", o = 0, i = h, a2 = 0;
                continue;
              }
            }
            e3 && (n2.length > 0 ? n2 += "/.." : n2 = "..", o = 2);
          } else
            n2.length > 0 ? n2 += "/" + t3.slice(i + 1, h) : n2 = t3.slice(i + 1, h), o = h - i - 1;
          i = h, a2 = 0;
        } else
          46 === r3 && -1 !== a2 ? ++a2 : a2 = -1;
      }
      return n2;
    }
    var n = { resolve: function() {
      for (var t3, n2 = "", o = false, i = arguments.length - 1; i >= -1 && !o; i--) {
        var a2;
        i >= 0 ? a2 = arguments[i] : (void 0 === t3 && (t3 = process.cwd()), a2 = t3), e2(a2), 0 !== a2.length && (n2 = a2 + "/" + n2, o = 47 === a2.charCodeAt(0));
      }
      return n2 = r2(n2, !o), o ? n2.length > 0 ? "/" + n2 : "/" : n2.length > 0 ? n2 : ".";
    }, normalize: function(t3) {
      if (e2(t3), 0 === t3.length)
        return ".";
      var n2 = 47 === t3.charCodeAt(0), o = 47 === t3.charCodeAt(t3.length - 1);
      return 0 !== (t3 = r2(t3, !n2)).length || n2 || (t3 = "."), t3.length > 0 && o && (t3 += "/"), n2 ? "/" + t3 : t3;
    }, isAbsolute: function(t3) {
      return e2(t3), t3.length > 0 && 47 === t3.charCodeAt(0);
    }, join: function() {
      if (0 === arguments.length)
        return ".";
      for (var t3, r3 = 0; r3 < arguments.length; ++r3) {
        var o = arguments[r3];
        e2(o), o.length > 0 && (void 0 === t3 ? t3 = o : t3 += "/" + o);
      }
      return void 0 === t3 ? "." : n.normalize(t3);
    }, relative: function(t3, r3) {
      if (e2(t3), e2(r3), t3 === r3)
        return "";
      if ((t3 = n.resolve(t3)) === (r3 = n.resolve(r3)))
        return "";
      for (var o = 1; o < t3.length && 47 === t3.charCodeAt(o); ++o)
        ;
      for (var i = t3.length, a2 = i - o, h = 1; h < r3.length && 47 === r3.charCodeAt(h); ++h)
        ;
      for (var s = r3.length - h, c = a2 < s ? a2 : s, f2 = -1, u = 0; u <= c; ++u) {
        if (u === c) {
          if (s > c) {
            if (47 === r3.charCodeAt(h + u))
              return r3.slice(h + u + 1);
            if (0 === u)
              return r3.slice(h + u);
          } else
            a2 > c && (47 === t3.charCodeAt(o + u) ? f2 = u : 0 === u && (f2 = 0));
          break;
        }
        var l = t3.charCodeAt(o + u);
        if (l !== r3.charCodeAt(h + u))
          break;
        47 === l && (f2 = u);
      }
      var p = "";
      for (u = o + f2 + 1; u <= i; ++u)
        u !== i && 47 !== t3.charCodeAt(u) || (0 === p.length ? p += ".." : p += "/..");
      return p.length > 0 ? p + r3.slice(h + f2) : (h += f2, 47 === r3.charCodeAt(h) && ++h, r3.slice(h));
    }, _makeLong: function(t3) {
      return t3;
    }, dirname: function(t3) {
      if (e2(t3), 0 === t3.length)
        return ".";
      for (var r3 = t3.charCodeAt(0), n2 = 47 === r3, o = -1, i = true, a2 = t3.length - 1; a2 >= 1; --a2)
        if (47 === (r3 = t3.charCodeAt(a2))) {
          if (!i) {
            o = a2;
            break;
          }
        } else
          i = false;
      return -1 === o ? n2 ? "/" : "." : n2 && 1 === o ? "//" : t3.slice(0, o);
    }, basename: function(t3, r3) {
      if (void 0 !== r3 && "string" != typeof r3)
        throw new TypeError('"ext" argument must be a string');
      e2(t3);
      var n2, o = 0, i = -1, a2 = true;
      if (void 0 !== r3 && r3.length > 0 && r3.length <= t3.length) {
        if (r3.length === t3.length && r3 === t3)
          return "";
        var h = r3.length - 1, s = -1;
        for (n2 = t3.length - 1; n2 >= 0; --n2) {
          var c = t3.charCodeAt(n2);
          if (47 === c) {
            if (!a2) {
              o = n2 + 1;
              break;
            }
          } else
            -1 === s && (a2 = false, s = n2 + 1), h >= 0 && (c === r3.charCodeAt(h) ? -1 == --h && (i = n2) : (h = -1, i = s));
        }
        return o === i ? i = s : -1 === i && (i = t3.length), t3.slice(o, i);
      }
      for (n2 = t3.length - 1; n2 >= 0; --n2)
        if (47 === t3.charCodeAt(n2)) {
          if (!a2) {
            o = n2 + 1;
            break;
          }
        } else
          -1 === i && (a2 = false, i = n2 + 1);
      return -1 === i ? "" : t3.slice(o, i);
    }, extname: function(t3) {
      e2(t3);
      for (var r3 = -1, n2 = 0, o = -1, i = true, a2 = 0, h = t3.length - 1; h >= 0; --h) {
        var s = t3.charCodeAt(h);
        if (47 !== s)
          -1 === o && (i = false, o = h + 1), 46 === s ? -1 === r3 ? r3 = h : 1 !== a2 && (a2 = 1) : -1 !== r3 && (a2 = -1);
        else if (!i) {
          n2 = h + 1;
          break;
        }
      }
      return -1 === r3 || -1 === o || 0 === a2 || 1 === a2 && r3 === o - 1 && r3 === n2 + 1 ? "" : t3.slice(r3, o);
    }, format: function(t3) {
      if (null === t3 || "object" != typeof t3)
        throw new TypeError('The "pathObject" argument must be of type Object. Received type ' + typeof t3);
      return function(t4, e3) {
        var r3 = e3.dir || e3.root, n2 = e3.base || (e3.name || "") + (e3.ext || "");
        return r3 ? r3 === e3.root ? r3 + n2 : r3 + "/" + n2 : n2;
      }(0, t3);
    }, parse: function(t3) {
      e2(t3);
      var r3 = { root: "", dir: "", base: "", ext: "", name: "" };
      if (0 === t3.length)
        return r3;
      var n2, o = t3.charCodeAt(0), i = 47 === o;
      i ? (r3.root = "/", n2 = 1) : n2 = 0;
      for (var a2 = -1, h = 0, s = -1, c = true, f2 = t3.length - 1, u = 0; f2 >= n2; --f2)
        if (47 !== (o = t3.charCodeAt(f2)))
          -1 === s && (c = false, s = f2 + 1), 46 === o ? -1 === a2 ? a2 = f2 : 1 !== u && (u = 1) : -1 !== a2 && (u = -1);
        else if (!c) {
          h = f2 + 1;
          break;
        }
      return -1 === a2 || -1 === s || 0 === u || 1 === u && a2 === s - 1 && a2 === h + 1 ? -1 !== s && (r3.base = r3.name = 0 === h && i ? t3.slice(1, s) : t3.slice(h, s)) : (0 === h && i ? (r3.name = t3.slice(1, a2), r3.base = t3.slice(1, s)) : (r3.name = t3.slice(h, a2), r3.base = t3.slice(h, s)), r3.ext = t3.slice(a2, s)), h > 0 ? r3.dir = t3.slice(0, h - 1) : i && (r3.dir = "/"), r3;
    }, sep: "/", delimiter: ":", win32: null, posix: null };
    n.posix = n, t2.exports = n;
  }, 447: (t2, e2, r2) => {
    var n;
    if (r2.r(e2), r2.d(e2, { URI: () => d, Utils: () => P }), "object" == typeof process)
      n = "win32" === process.platform;
    else if ("object" == typeof navigator) {
      var o = navigator.userAgent;
      n = o.indexOf("Windows") >= 0;
    }
    var i, a2, h = (i = function(t3, e3) {
      return (i = Object.setPrototypeOf || { __proto__: [] } instanceof Array && function(t4, e4) {
        t4.__proto__ = e4;
      } || function(t4, e4) {
        for (var r3 in e4)
          Object.prototype.hasOwnProperty.call(e4, r3) && (t4[r3] = e4[r3]);
      })(t3, e3);
    }, function(t3, e3) {
      if ("function" != typeof e3 && null !== e3)
        throw new TypeError("Class extends value " + String(e3) + " is not a constructor or null");
      function r3() {
        this.constructor = t3;
      }
      i(t3, e3), t3.prototype = null === e3 ? Object.create(e3) : (r3.prototype = e3.prototype, new r3());
    }), s = /^\w[\w\d+.-]*$/, c = /^\//, f2 = /^\/\//;
    function u(t3, e3) {
      if (!t3.scheme && e3)
        throw new Error('[UriError]: Scheme is missing: {scheme: "", authority: "'.concat(t3.authority, '", path: "').concat(t3.path, '", query: "').concat(t3.query, '", fragment: "').concat(t3.fragment, '"}'));
      if (t3.scheme && !s.test(t3.scheme))
        throw new Error("[UriError]: Scheme contains illegal characters.");
      if (t3.path) {
        if (t3.authority) {
          if (!c.test(t3.path))
            throw new Error('[UriError]: If a URI contains an authority component, then the path component must either be empty or begin with a slash ("/") character');
        } else if (f2.test(t3.path))
          throw new Error('[UriError]: If a URI does not contain an authority component, then the path cannot begin with two slash characters ("//")');
      }
    }
    var l = "", p = "/", g = /^(([^:/?#]+?):)?(\/\/([^/?#]*))?([^?#]*)(\?([^#]*))?(#(.*))?/, d = function() {
      function t3(t4, e3, r3, n2, o2, i2) {
        void 0 === i2 && (i2 = false), "object" == typeof t4 ? (this.scheme = t4.scheme || l, this.authority = t4.authority || l, this.path = t4.path || l, this.query = t4.query || l, this.fragment = t4.fragment || l) : (this.scheme = function(t5, e4) {
          return t5 || e4 ? t5 : "file";
        }(t4, i2), this.authority = e3 || l, this.path = function(t5, e4) {
          switch (t5) {
            case "https":
            case "http":
            case "file":
              e4 ? e4[0] !== p && (e4 = p + e4) : e4 = p;
          }
          return e4;
        }(this.scheme, r3 || l), this.query = n2 || l, this.fragment = o2 || l, u(this, i2));
      }
      return t3.isUri = function(e3) {
        return e3 instanceof t3 || !!e3 && "string" == typeof e3.authority && "string" == typeof e3.fragment && "string" == typeof e3.path && "string" == typeof e3.query && "string" == typeof e3.scheme && "string" == typeof e3.fsPath && "function" == typeof e3.with && "function" == typeof e3.toString;
      }, Object.defineProperty(t3.prototype, "fsPath", { get: function() {
        return A2(this, false);
      }, enumerable: false, configurable: true }), t3.prototype.with = function(t4) {
        if (!t4)
          return this;
        var e3 = t4.scheme, r3 = t4.authority, n2 = t4.path, o2 = t4.query, i2 = t4.fragment;
        return void 0 === e3 ? e3 = this.scheme : null === e3 && (e3 = l), void 0 === r3 ? r3 = this.authority : null === r3 && (r3 = l), void 0 === n2 ? n2 = this.path : null === n2 && (n2 = l), void 0 === o2 ? o2 = this.query : null === o2 && (o2 = l), void 0 === i2 ? i2 = this.fragment : null === i2 && (i2 = l), e3 === this.scheme && r3 === this.authority && n2 === this.path && o2 === this.query && i2 === this.fragment ? this : new y(e3, r3, n2, o2, i2);
      }, t3.parse = function(t4, e3) {
        void 0 === e3 && (e3 = false);
        var r3 = g.exec(t4);
        return r3 ? new y(r3[2] || l, O(r3[4] || l), O(r3[5] || l), O(r3[7] || l), O(r3[9] || l), e3) : new y(l, l, l, l, l);
      }, t3.file = function(t4) {
        var e3 = l;
        if (n && (t4 = t4.replace(/\\/g, p)), t4[0] === p && t4[1] === p) {
          var r3 = t4.indexOf(p, 2);
          -1 === r3 ? (e3 = t4.substring(2), t4 = p) : (e3 = t4.substring(2, r3), t4 = t4.substring(r3) || p);
        }
        return new y("file", e3, t4, l, l);
      }, t3.from = function(t4) {
        var e3 = new y(t4.scheme, t4.authority, t4.path, t4.query, t4.fragment);
        return u(e3, true), e3;
      }, t3.prototype.toString = function(t4) {
        return void 0 === t4 && (t4 = false), w(this, t4);
      }, t3.prototype.toJSON = function() {
        return this;
      }, t3.revive = function(e3) {
        if (e3) {
          if (e3 instanceof t3)
            return e3;
          var r3 = new y(e3);
          return r3._formatted = e3.external, r3._fsPath = e3._sep === v ? e3.fsPath : null, r3;
        }
        return e3;
      }, t3;
    }(), v = n ? 1 : void 0, y = function(t3) {
      function e3() {
        var e4 = null !== t3 && t3.apply(this, arguments) || this;
        return e4._formatted = null, e4._fsPath = null, e4;
      }
      return h(e3, t3), Object.defineProperty(e3.prototype, "fsPath", { get: function() {
        return this._fsPath || (this._fsPath = A2(this, false)), this._fsPath;
      }, enumerable: false, configurable: true }), e3.prototype.toString = function(t4) {
        return void 0 === t4 && (t4 = false), t4 ? w(this, true) : (this._formatted || (this._formatted = w(this, false)), this._formatted);
      }, e3.prototype.toJSON = function() {
        var t4 = { $mid: 1 };
        return this._fsPath && (t4.fsPath = this._fsPath, t4._sep = v), this._formatted && (t4.external = this._formatted), this.path && (t4.path = this.path), this.scheme && (t4.scheme = this.scheme), this.authority && (t4.authority = this.authority), this.query && (t4.query = this.query), this.fragment && (t4.fragment = this.fragment), t4;
      }, e3;
    }(d), m = ((a2 = {})[58] = "%3A", a2[47] = "%2F", a2[63] = "%3F", a2[35] = "%23", a2[91] = "%5B", a2[93] = "%5D", a2[64] = "%40", a2[33] = "%21", a2[36] = "%24", a2[38] = "%26", a2[39] = "%27", a2[40] = "%28", a2[41] = "%29", a2[42] = "%2A", a2[43] = "%2B", a2[44] = "%2C", a2[59] = "%3B", a2[61] = "%3D", a2[32] = "%20", a2);
    function b(t3, e3) {
      for (var r3 = void 0, n2 = -1, o2 = 0; o2 < t3.length; o2++) {
        var i2 = t3.charCodeAt(o2);
        if (i2 >= 97 && i2 <= 122 || i2 >= 65 && i2 <= 90 || i2 >= 48 && i2 <= 57 || 45 === i2 || 46 === i2 || 95 === i2 || 126 === i2 || e3 && 47 === i2)
          -1 !== n2 && (r3 += encodeURIComponent(t3.substring(n2, o2)), n2 = -1), void 0 !== r3 && (r3 += t3.charAt(o2));
        else {
          void 0 === r3 && (r3 = t3.substr(0, o2));
          var a3 = m[i2];
          void 0 !== a3 ? (-1 !== n2 && (r3 += encodeURIComponent(t3.substring(n2, o2)), n2 = -1), r3 += a3) : -1 === n2 && (n2 = o2);
        }
      }
      return -1 !== n2 && (r3 += encodeURIComponent(t3.substring(n2))), void 0 !== r3 ? r3 : t3;
    }
    function C(t3) {
      for (var e3 = void 0, r3 = 0; r3 < t3.length; r3++) {
        var n2 = t3.charCodeAt(r3);
        35 === n2 || 63 === n2 ? (void 0 === e3 && (e3 = t3.substr(0, r3)), e3 += m[n2]) : void 0 !== e3 && (e3 += t3[r3]);
      }
      return void 0 !== e3 ? e3 : t3;
    }
    function A2(t3, e3) {
      var r3;
      return r3 = t3.authority && t3.path.length > 1 && "file" === t3.scheme ? "//".concat(t3.authority).concat(t3.path) : 47 === t3.path.charCodeAt(0) && (t3.path.charCodeAt(1) >= 65 && t3.path.charCodeAt(1) <= 90 || t3.path.charCodeAt(1) >= 97 && t3.path.charCodeAt(1) <= 122) && 58 === t3.path.charCodeAt(2) ? e3 ? t3.path.substr(1) : t3.path[1].toLowerCase() + t3.path.substr(2) : t3.path, n && (r3 = r3.replace(/\//g, "\\")), r3;
    }
    function w(t3, e3) {
      var r3 = e3 ? C : b, n2 = "", o2 = t3.scheme, i2 = t3.authority, a3 = t3.path, h2 = t3.query, s2 = t3.fragment;
      if (o2 && (n2 += o2, n2 += ":"), (i2 || "file" === o2) && (n2 += p, n2 += p), i2) {
        var c2 = i2.indexOf("@");
        if (-1 !== c2) {
          var f3 = i2.substr(0, c2);
          i2 = i2.substr(c2 + 1), -1 === (c2 = f3.indexOf(":")) ? n2 += r3(f3, false) : (n2 += r3(f3.substr(0, c2), false), n2 += ":", n2 += r3(f3.substr(c2 + 1), false)), n2 += "@";
        }
        -1 === (c2 = (i2 = i2.toLowerCase()).indexOf(":")) ? n2 += r3(i2, false) : (n2 += r3(i2.substr(0, c2), false), n2 += i2.substr(c2));
      }
      if (a3) {
        if (a3.length >= 3 && 47 === a3.charCodeAt(0) && 58 === a3.charCodeAt(2))
          (u2 = a3.charCodeAt(1)) >= 65 && u2 <= 90 && (a3 = "/".concat(String.fromCharCode(u2 + 32), ":").concat(a3.substr(3)));
        else if (a3.length >= 2 && 58 === a3.charCodeAt(1)) {
          var u2;
          (u2 = a3.charCodeAt(0)) >= 65 && u2 <= 90 && (a3 = "".concat(String.fromCharCode(u2 + 32), ":").concat(a3.substr(2)));
        }
        n2 += r3(a3, true);
      }
      return h2 && (n2 += "?", n2 += r3(h2, false)), s2 && (n2 += "#", n2 += e3 ? s2 : b(s2, false)), n2;
    }
    function x(t3) {
      try {
        return decodeURIComponent(t3);
      } catch (e3) {
        return t3.length > 3 ? t3.substr(0, 3) + x(t3.substr(3)) : t3;
      }
    }
    var _ = /(%[0-9A-Za-z][0-9A-Za-z])+/g;
    function O(t3) {
      return t3.match(_) ? t3.replace(_, function(t4) {
        return x(t4);
      }) : t3;
    }
    var P, j = r2(470), U = function(t3, e3, r3) {
      if (r3 || 2 === arguments.length)
        for (var n2, o2 = 0, i2 = e3.length; o2 < i2; o2++)
          !n2 && o2 in e3 || (n2 || (n2 = Array.prototype.slice.call(e3, 0, o2)), n2[o2] = e3[o2]);
      return t3.concat(n2 || Array.prototype.slice.call(e3));
    }, I = j.posix || j;
    !function(t3) {
      t3.joinPath = function(t4) {
        for (var e3 = [], r3 = 1; r3 < arguments.length; r3++)
          e3[r3 - 1] = arguments[r3];
        return t4.with({ path: I.join.apply(I, U([t4.path], e3, false)) });
      }, t3.resolvePath = function(t4) {
        for (var e3 = [], r3 = 1; r3 < arguments.length; r3++)
          e3[r3 - 1] = arguments[r3];
        var n2 = t4.path || "/";
        return t4.with({ path: I.resolve.apply(I, U([n2], e3, false)) });
      }, t3.dirname = function(t4) {
        var e3 = I.dirname(t4.path);
        return 1 === e3.length && 46 === e3.charCodeAt(0) ? t4 : t4.with({ path: e3 });
      }, t3.basename = function(t4) {
        return I.basename(t4.path);
      }, t3.extname = function(t4) {
        return I.extname(t4.path);
      };
    }(P || (P = {}));
  } }, e = {};
  function r(n) {
    if (e[n])
      return e[n].exports;
    var o = e[n] = { exports: {} };
    return t[n](o, o.exports, r), o.exports;
  }
  return r.d = (t2, e2) => {
    for (var n in e2)
      r.o(e2, n) && !r.o(t2, n) && Object.defineProperty(t2, n, { enumerable: true, get: e2[n] });
  }, r.o = (t2, e2) => Object.prototype.hasOwnProperty.call(t2, e2), r.r = (t2) => {
    "undefined" != typeof Symbol && Symbol.toStringTag && Object.defineProperty(t2, Symbol.toStringTag, { value: "Module" }), Object.defineProperty(t2, "__esModule", { value: true });
  }, r(447);
})();
var { URI, Utils } = LIB;

// node_modules/vscode-css-languageservice/lib/esm/utils/resources.js
var __spreadArray2 = function(to, from, pack) {
  if (pack || arguments.length === 2)
    for (var i = 0, l = from.length, ar; i < l; i++) {
      if (ar || !(i in from)) {
        if (!ar)
          ar = Array.prototype.slice.call(from, 0, i);
        ar[i] = from[i];
      }
    }
  return to.concat(ar || Array.prototype.slice.call(from));
};
function dirname(uriString) {
  return Utils.dirname(URI.parse(uriString)).toString();
}
function joinPath(uriString) {
  var paths = [];
  for (var _i = 1; _i < arguments.length; _i++) {
    paths[_i - 1] = arguments[_i];
  }
  return Utils.joinPath.apply(Utils, __spreadArray2([URI.parse(uriString)], paths, false)).toString();
}

// node_modules/vscode-css-languageservice/lib/esm/services/pathCompletion.js
var __awaiter = function(thisArg, _arguments, P, generator) {
  function adopt(value) {
    return value instanceof P ? value : new P(function(resolve) {
      resolve(value);
    });
  }
  return new (P || (P = Promise))(function(resolve, reject) {
    function fulfilled(value) {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    }
    function rejected(value) {
      try {
        step(generator["throw"](value));
      } catch (e) {
        reject(e);
      }
    }
    function step(result) {
      result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected);
    }
    step((generator = generator.apply(thisArg, _arguments || [])).next());
  });
};
var __generator = function(thisArg, body) {
  var _ = { label: 0, sent: function() {
    if (t[0] & 1)
      throw t[1];
    return t[1];
  }, trys: [], ops: [] }, f2, y, t, g;
  return g = { next: verb(0), "throw": verb(1), "return": verb(2) }, typeof Symbol === "function" && (g[Symbol.iterator] = function() {
    return this;
  }), g;
  function verb(n) {
    return function(v) {
      return step([n, v]);
    };
  }
  function step(op) {
    if (f2)
      throw new TypeError("Generator is already executing.");
    while (_)
      try {
        if (f2 = 1, y && (t = op[0] & 2 ? y["return"] : op[0] ? y["throw"] || ((t = y["return"]) && t.call(y), 0) : y.next) && !(t = t.call(y, op[1])).done)
          return t;
        if (y = 0, t)
          op = [op[0] & 2, t.value];
        switch (op[0]) {
          case 0:
          case 1:
            t = op;
            break;
          case 4:
            _.label++;
            return { value: op[1], done: false };
          case 5:
            _.label++;
            y = op[1];
            op = [0];
            continue;
          case 7:
            op = _.ops.pop();
            _.trys.pop();
            continue;
          default:
            if (!(t = _.trys, t = t.length > 0 && t[t.length - 1]) && (op[0] === 6 || op[0] === 2)) {
              _ = 0;
              continue;
            }
            if (op[0] === 3 && (!t || op[1] > t[0] && op[1] < t[3])) {
              _.label = op[1];
              break;
            }
            if (op[0] === 6 && _.label < t[1]) {
              _.label = t[1];
              t = op;
              break;
            }
            if (t && _.label < t[2]) {
              _.label = t[2];
              _.ops.push(op);
              break;
            }
            if (t[2])
              _.ops.pop();
            _.trys.pop();
            continue;
        }
        op = body.call(thisArg, _);
      } catch (e) {
        op = [6, e];
        y = 0;
      } finally {
        f2 = t = 0;
      }
    if (op[0] & 5)
      throw op[1];
    return { value: op[0] ? op[1] : void 0, done: true };
  }
};
var PathCompletionParticipant = function() {
  function PathCompletionParticipant2(readDirectory) {
    this.readDirectory = readDirectory;
    this.literalCompletions = [];
    this.importCompletions = [];
  }
  PathCompletionParticipant2.prototype.onCssURILiteralValue = function(context) {
    this.literalCompletions.push(context);
  };
  PathCompletionParticipant2.prototype.onCssImportPath = function(context) {
    this.importCompletions.push(context);
  };
  PathCompletionParticipant2.prototype.computeCompletions = function(document, documentContext) {
    return __awaiter(this, void 0, void 0, function() {
      var result, _i, _a2, literalCompletion, uriValue, fullValue, items, _b, items_1, item, _c, _d, importCompletion, pathValue, fullValue, suggestions, _e, suggestions_1, item;
      return __generator(this, function(_f2) {
        switch (_f2.label) {
          case 0:
            result = { items: [], isIncomplete: false };
            _i = 0, _a2 = this.literalCompletions;
            _f2.label = 1;
          case 1:
            if (!(_i < _a2.length))
              return [3, 5];
            literalCompletion = _a2[_i];
            uriValue = literalCompletion.uriValue;
            fullValue = stripQuotes(uriValue);
            if (!(fullValue === "." || fullValue === ".."))
              return [3, 2];
            result.isIncomplete = true;
            return [3, 4];
          case 2:
            return [4, this.providePathSuggestions(uriValue, literalCompletion.position, literalCompletion.range, document, documentContext)];
          case 3:
            items = _f2.sent();
            for (_b = 0, items_1 = items; _b < items_1.length; _b++) {
              item = items_1[_b];
              result.items.push(item);
            }
            _f2.label = 4;
          case 4:
            _i++;
            return [3, 1];
          case 5:
            _c = 0, _d = this.importCompletions;
            _f2.label = 6;
          case 6:
            if (!(_c < _d.length))
              return [3, 10];
            importCompletion = _d[_c];
            pathValue = importCompletion.pathValue;
            fullValue = stripQuotes(pathValue);
            if (!(fullValue === "." || fullValue === ".."))
              return [3, 7];
            result.isIncomplete = true;
            return [3, 9];
          case 7:
            return [4, this.providePathSuggestions(pathValue, importCompletion.position, importCompletion.range, document, documentContext)];
          case 8:
            suggestions = _f2.sent();
            if (document.languageId === "scss") {
              suggestions.forEach(function(s) {
                if (startsWith(s.label, "_") && endsWith(s.label, ".scss")) {
                  if (s.textEdit) {
                    s.textEdit.newText = s.label.slice(1, -5);
                  } else {
                    s.label = s.label.slice(1, -5);
                  }
                }
              });
            }
            for (_e = 0, suggestions_1 = suggestions; _e < suggestions_1.length; _e++) {
              item = suggestions_1[_e];
              result.items.push(item);
            }
            _f2.label = 9;
          case 9:
            _c++;
            return [3, 6];
          case 10:
            return [2, result];
        }
      });
    });
  };
  PathCompletionParticipant2.prototype.providePathSuggestions = function(pathValue, position, range, document, documentContext) {
    return __awaiter(this, void 0, void 0, function() {
      var fullValue, isValueQuoted, valueBeforeCursor, currentDocUri, fullValueRange, replaceRange, valueBeforeLastSlash, parentDir, result, infos, _i, infos_1, _a2, name, type, e_1;
      return __generator(this, function(_b) {
        switch (_b.label) {
          case 0:
            fullValue = stripQuotes(pathValue);
            isValueQuoted = startsWith(pathValue, "'") || startsWith(pathValue, '"');
            valueBeforeCursor = isValueQuoted ? fullValue.slice(0, position.character - (range.start.character + 1)) : fullValue.slice(0, position.character - range.start.character);
            currentDocUri = document.uri;
            fullValueRange = isValueQuoted ? shiftRange(range, 1, -1) : range;
            replaceRange = pathToReplaceRange(valueBeforeCursor, fullValue, fullValueRange);
            valueBeforeLastSlash = valueBeforeCursor.substring(0, valueBeforeCursor.lastIndexOf("/") + 1);
            parentDir = documentContext.resolveReference(valueBeforeLastSlash || ".", currentDocUri);
            if (!parentDir)
              return [3, 4];
            _b.label = 1;
          case 1:
            _b.trys.push([1, 3, , 4]);
            result = [];
            return [4, this.readDirectory(parentDir)];
          case 2:
            infos = _b.sent();
            for (_i = 0, infos_1 = infos; _i < infos_1.length; _i++) {
              _a2 = infos_1[_i], name = _a2[0], type = _a2[1];
              if (name.charCodeAt(0) !== CharCode_dot && (type === FileType.Directory || joinPath(parentDir, name) !== currentDocUri)) {
                result.push(createCompletionItem(name, type === FileType.Directory, replaceRange));
              }
            }
            return [2, result];
          case 3:
            e_1 = _b.sent();
            return [3, 4];
          case 4:
            return [2, []];
        }
      });
    });
  };
  return PathCompletionParticipant2;
}();
var CharCode_dot = ".".charCodeAt(0);
function stripQuotes(fullValue) {
  if (startsWith(fullValue, "'") || startsWith(fullValue, '"')) {
    return fullValue.slice(1, -1);
  } else {
    return fullValue;
  }
}
function pathToReplaceRange(valueBeforeCursor, fullValue, fullValueRange) {
  var replaceRange;
  var lastIndexOfSlash = valueBeforeCursor.lastIndexOf("/");
  if (lastIndexOfSlash === -1) {
    replaceRange = fullValueRange;
  } else {
    var valueAfterLastSlash = fullValue.slice(lastIndexOfSlash + 1);
    var startPos = shiftPosition(fullValueRange.end, -valueAfterLastSlash.length);
    var whitespaceIndex = valueAfterLastSlash.indexOf(" ");
    var endPos = void 0;
    if (whitespaceIndex !== -1) {
      endPos = shiftPosition(startPos, whitespaceIndex);
    } else {
      endPos = fullValueRange.end;
    }
    replaceRange = Range.create(startPos, endPos);
  }
  return replaceRange;
}
function createCompletionItem(name, isDir, replaceRange) {
  if (isDir) {
    name = name + "/";
    return {
      label: escapePath(name),
      kind: CompletionItemKind.Folder,
      textEdit: TextEdit.replace(replaceRange, escapePath(name)),
      command: {
        title: "Suggest",
        command: "editor.action.triggerSuggest"
      }
    };
  } else {
    return {
      label: escapePath(name),
      kind: CompletionItemKind.File,
      textEdit: TextEdit.replace(replaceRange, escapePath(name))
    };
  }
}
function escapePath(p) {
  return p.replace(/(\s|\(|\)|,|"|')/g, "\\$1");
}
function shiftPosition(pos, offset) {
  return Position.create(pos.line, pos.character + offset);
}
function shiftRange(range, startOffset, endOffset) {
  var start = shiftPosition(range.start, startOffset);
  var end = shiftPosition(range.end, endOffset);
  return Range.create(start, end);
}

// node_modules/vscode-css-languageservice/lib/esm/services/cssCompletion.js
var __awaiter2 = function(thisArg, _arguments, P, generator) {
  function adopt(value) {
    return value instanceof P ? value : new P(function(resolve) {
      resolve(value);
    });
  }
  return new (P || (P = Promise))(function(resolve, reject) {
    function fulfilled(value) {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    }
    function rejected(value) {
      try {
        step(generator["throw"](value));
      } catch (e) {
        reject(e);
      }
    }
    function step(result) {
      result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected);
    }
    step((generator = generator.apply(thisArg, _arguments || [])).next());
  });
};
var __generator2 = function(thisArg, body) {
  var _ = { label: 0, sent: function() {
    if (t[0] & 1)
      throw t[1];
    return t[1];
  }, trys: [], ops: [] }, f2, y, t, g;
  return g = { next: verb(0), "throw": verb(1), "return": verb(2) }, typeof Symbol === "function" && (g[Symbol.iterator] = function() {
    return this;
  }), g;
  function verb(n) {
    return function(v) {
      return step([n, v]);
    };
  }
  function step(op) {
    if (f2)
      throw new TypeError("Generator is already executing.");
    while (_)
      try {
        if (f2 = 1, y && (t = op[0] & 2 ? y["return"] : op[0] ? y["throw"] || ((t = y["return"]) && t.call(y), 0) : y.next) && !(t = t.call(y, op[1])).done)
          return t;
        if (y = 0, t)
          op = [op[0] & 2, t.value];
        switch (op[0]) {
          case 0:
          case 1:
            t = op;
            break;
          case 4:
            _.label++;
            return { value: op[1], done: false };
          case 5:
            _.label++;
            y = op[1];
            op = [0];
            continue;
          case 7:
            op = _.ops.pop();
            _.trys.pop();
            continue;
          default:
            if (!(t = _.trys, t = t.length > 0 && t[t.length - 1]) && (op[0] === 6 || op[0] === 2)) {
              _ = 0;
              continue;
            }
            if (op[0] === 3 && (!t || op[1] > t[0] && op[1] < t[3])) {
              _.label = op[1];
              break;
            }
            if (op[0] === 6 && _.label < t[1]) {
              _.label = t[1];
              t = op;
              break;
            }
            if (t && _.label < t[2]) {
              _.label = t[2];
              _.ops.push(op);
              break;
            }
            if (t[2])
              _.ops.pop();
            _.trys.pop();
            continue;
        }
        op = body.call(thisArg, _);
      } catch (e) {
        op = [6, e];
        y = 0;
      } finally {
        f2 = t = 0;
      }
    if (op[0] & 5)
      throw op[1];
    return { value: op[0] ? op[1] : void 0, done: true };
  }
};
var localize4 = loadMessageBundle();
var SnippetFormat = InsertTextFormat.Snippet;
var retriggerCommand = {
  title: "Suggest",
  command: "editor.action.triggerSuggest"
};
var SortTexts;
(function(SortTexts2) {
  SortTexts2["Enums"] = " ";
  SortTexts2["Normal"] = "d";
  SortTexts2["VendorPrefixed"] = "x";
  SortTexts2["Term"] = "y";
  SortTexts2["Variable"] = "z";
})(SortTexts || (SortTexts = {}));
var CSSCompletion = function() {
  function CSSCompletion2(variablePrefix, lsOptions, cssDataManager) {
    if (variablePrefix === void 0) {
      variablePrefix = null;
    }
    this.variablePrefix = variablePrefix;
    this.lsOptions = lsOptions;
    this.cssDataManager = cssDataManager;
    this.completionParticipants = [];
  }
  CSSCompletion2.prototype.configure = function(settings) {
    this.defaultSettings = settings;
  };
  CSSCompletion2.prototype.getSymbolContext = function() {
    if (!this.symbolContext) {
      this.symbolContext = new Symbols(this.styleSheet);
    }
    return this.symbolContext;
  };
  CSSCompletion2.prototype.setCompletionParticipants = function(registeredCompletionParticipants) {
    this.completionParticipants = registeredCompletionParticipants || [];
  };
  CSSCompletion2.prototype.doComplete2 = function(document, position, styleSheet, documentContext, completionSettings) {
    if (completionSettings === void 0) {
      completionSettings = this.defaultSettings;
    }
    return __awaiter2(this, void 0, void 0, function() {
      var participant, contributedParticipants, result, pathCompletionResult;
      return __generator2(this, function(_a2) {
        switch (_a2.label) {
          case 0:
            if (!this.lsOptions.fileSystemProvider || !this.lsOptions.fileSystemProvider.readDirectory) {
              return [2, this.doComplete(document, position, styleSheet, completionSettings)];
            }
            participant = new PathCompletionParticipant(this.lsOptions.fileSystemProvider.readDirectory);
            contributedParticipants = this.completionParticipants;
            this.completionParticipants = [participant].concat(contributedParticipants);
            result = this.doComplete(document, position, styleSheet, completionSettings);
            _a2.label = 1;
          case 1:
            _a2.trys.push([1, , 3, 4]);
            return [4, participant.computeCompletions(document, documentContext)];
          case 2:
            pathCompletionResult = _a2.sent();
            return [2, {
              isIncomplete: result.isIncomplete || pathCompletionResult.isIncomplete,
              items: pathCompletionResult.items.concat(result.items)
            }];
          case 3:
            this.completionParticipants = contributedParticipants;
            return [7];
          case 4:
            return [2];
        }
      });
    });
  };
  CSSCompletion2.prototype.doComplete = function(document, position, styleSheet, documentSettings) {
    this.offset = document.offsetAt(position);
    this.position = position;
    this.currentWord = getCurrentWord(document, this.offset);
    this.defaultReplaceRange = Range.create(Position.create(this.position.line, this.position.character - this.currentWord.length), this.position);
    this.textDocument = document;
    this.styleSheet = styleSheet;
    this.documentSettings = documentSettings;
    try {
      var result = { isIncomplete: false, items: [] };
      this.nodePath = getNodePath(this.styleSheet, this.offset);
      for (var i = this.nodePath.length - 1; i >= 0; i--) {
        var node = this.nodePath[i];
        if (node instanceof Property) {
          this.getCompletionsForDeclarationProperty(node.getParent(), result);
        } else if (node instanceof Expression) {
          if (node.parent instanceof Interpolation) {
            this.getVariableProposals(null, result);
          } else {
            this.getCompletionsForExpression(node, result);
          }
        } else if (node instanceof SimpleSelector) {
          var parentRef = node.findAParent(NodeType.ExtendsReference, NodeType.Ruleset);
          if (parentRef) {
            if (parentRef.type === NodeType.ExtendsReference) {
              this.getCompletionsForExtendsReference(parentRef, node, result);
            } else {
              var parentRuleSet = parentRef;
              this.getCompletionsForSelector(parentRuleSet, parentRuleSet && parentRuleSet.isNested(), result);
            }
          }
        } else if (node instanceof FunctionArgument) {
          this.getCompletionsForFunctionArgument(node, node.getParent(), result);
        } else if (node instanceof Declarations) {
          this.getCompletionsForDeclarations(node, result);
        } else if (node instanceof VariableDeclaration) {
          this.getCompletionsForVariableDeclaration(node, result);
        } else if (node instanceof RuleSet) {
          this.getCompletionsForRuleSet(node, result);
        } else if (node instanceof Interpolation) {
          this.getCompletionsForInterpolation(node, result);
        } else if (node instanceof FunctionDeclaration) {
          this.getCompletionsForFunctionDeclaration(node, result);
        } else if (node instanceof MixinReference) {
          this.getCompletionsForMixinReference(node, result);
        } else if (node instanceof Function) {
          this.getCompletionsForFunctionArgument(null, node, result);
        } else if (node instanceof Supports) {
          this.getCompletionsForSupports(node, result);
        } else if (node instanceof SupportsCondition) {
          this.getCompletionsForSupportsCondition(node, result);
        } else if (node instanceof ExtendsReference) {
          this.getCompletionsForExtendsReference(node, null, result);
        } else if (node.type === NodeType.URILiteral) {
          this.getCompletionForUriLiteralValue(node, result);
        } else if (node.parent === null) {
          this.getCompletionForTopLevel(result);
        } else if (node.type === NodeType.StringLiteral && this.isImportPathParent(node.parent.type)) {
          this.getCompletionForImportPath(node, result);
        } else {
          continue;
        }
        if (result.items.length > 0 || this.offset > node.offset) {
          return this.finalize(result);
        }
      }
      this.getCompletionsForStylesheet(result);
      if (result.items.length === 0) {
        if (this.variablePrefix && this.currentWord.indexOf(this.variablePrefix) === 0) {
          this.getVariableProposals(null, result);
        }
      }
      return this.finalize(result);
    } finally {
      this.position = null;
      this.currentWord = null;
      this.textDocument = null;
      this.styleSheet = null;
      this.symbolContext = null;
      this.defaultReplaceRange = null;
      this.nodePath = null;
    }
  };
  CSSCompletion2.prototype.isImportPathParent = function(type) {
    return type === NodeType.Import;
  };
  CSSCompletion2.prototype.finalize = function(result) {
    return result;
  };
  CSSCompletion2.prototype.findInNodePath = function() {
    var types = [];
    for (var _i = 0; _i < arguments.length; _i++) {
      types[_i] = arguments[_i];
    }
    for (var i = this.nodePath.length - 1; i >= 0; i--) {
      var node = this.nodePath[i];
      if (types.indexOf(node.type) !== -1) {
        return node;
      }
    }
    return null;
  };
  CSSCompletion2.prototype.getCompletionsForDeclarationProperty = function(declaration, result) {
    return this.getPropertyProposals(declaration, result);
  };
  CSSCompletion2.prototype.getPropertyProposals = function(declaration, result) {
    var _this = this;
    var triggerPropertyValueCompletion = this.isTriggerPropertyValueCompletionEnabled;
    var completePropertyWithSemicolon = this.isCompletePropertyWithSemicolonEnabled;
    var properties = this.cssDataManager.getProperties();
    properties.forEach(function(entry) {
      var range;
      var insertText;
      var retrigger = false;
      if (declaration) {
        range = _this.getCompletionRange(declaration.getProperty());
        insertText = entry.name;
        if (!isDefined(declaration.colonPosition)) {
          insertText += ": ";
          retrigger = true;
        }
      } else {
        range = _this.getCompletionRange(null);
        insertText = entry.name + ": ";
        retrigger = true;
      }
      if (!declaration && completePropertyWithSemicolon) {
        insertText += "$0;";
      }
      if (declaration && !declaration.semicolonPosition) {
        if (completePropertyWithSemicolon && _this.offset >= _this.textDocument.offsetAt(range.end)) {
          insertText += "$0;";
        }
      }
      var item = {
        label: entry.name,
        documentation: getEntryDescription(entry, _this.doesSupportMarkdown()),
        tags: isDeprecated(entry) ? [CompletionItemTag.Deprecated] : [],
        textEdit: TextEdit.replace(range, insertText),
        insertTextFormat: InsertTextFormat.Snippet,
        kind: CompletionItemKind.Property
      };
      if (!entry.restrictions) {
        retrigger = false;
      }
      if (triggerPropertyValueCompletion && retrigger) {
        item.command = retriggerCommand;
      }
      var relevance = typeof entry.relevance === "number" ? Math.min(Math.max(entry.relevance, 0), 99) : 50;
      var sortTextSuffix = (255 - relevance).toString(16);
      var sortTextPrefix = startsWith(entry.name, "-") ? SortTexts.VendorPrefixed : SortTexts.Normal;
      item.sortText = sortTextPrefix + "_" + sortTextSuffix;
      result.items.push(item);
    });
    this.completionParticipants.forEach(function(participant) {
      if (participant.onCssProperty) {
        participant.onCssProperty({
          propertyName: _this.currentWord,
          range: _this.defaultReplaceRange
        });
      }
    });
    return result;
  };
  Object.defineProperty(CSSCompletion2.prototype, "isTriggerPropertyValueCompletionEnabled", {
    get: function() {
      var _a2, _b;
      return (_b = (_a2 = this.documentSettings) === null || _a2 === void 0 ? void 0 : _a2.triggerPropertyValueCompletion) !== null && _b !== void 0 ? _b : true;
    },
    enumerable: false,
    configurable: true
  });
  Object.defineProperty(CSSCompletion2.prototype, "isCompletePropertyWithSemicolonEnabled", {
    get: function() {
      var _a2, _b;
      return (_b = (_a2 = this.documentSettings) === null || _a2 === void 0 ? void 0 : _a2.completePropertyWithSemicolon) !== null && _b !== void 0 ? _b : true;
    },
    enumerable: false,
    configurable: true
  });
  CSSCompletion2.prototype.getCompletionsForDeclarationValue = function(node, result) {
    var _this = this;
    var propertyName = node.getFullPropertyName();
    var entry = this.cssDataManager.getProperty(propertyName);
    var existingNode = node.getValue() || null;
    while (existingNode && existingNode.hasChildren()) {
      existingNode = existingNode.findChildAtOffset(this.offset, false);
    }
    this.completionParticipants.forEach(function(participant) {
      if (participant.onCssPropertyValue) {
        participant.onCssPropertyValue({
          propertyName,
          propertyValue: _this.currentWord,
          range: _this.getCompletionRange(existingNode)
        });
      }
    });
    if (entry) {
      if (entry.restrictions) {
        for (var _i = 0, _a2 = entry.restrictions; _i < _a2.length; _i++) {
          var restriction = _a2[_i];
          switch (restriction) {
            case "color":
              this.getColorProposals(entry, existingNode, result);
              break;
            case "position":
              this.getPositionProposals(entry, existingNode, result);
              break;
            case "repeat":
              this.getRepeatStyleProposals(entry, existingNode, result);
              break;
            case "line-style":
              this.getLineStyleProposals(entry, existingNode, result);
              break;
            case "line-width":
              this.getLineWidthProposals(entry, existingNode, result);
              break;
            case "geometry-box":
              this.getGeometryBoxProposals(entry, existingNode, result);
              break;
            case "box":
              this.getBoxProposals(entry, existingNode, result);
              break;
            case "image":
              this.getImageProposals(entry, existingNode, result);
              break;
            case "timing-function":
              this.getTimingFunctionProposals(entry, existingNode, result);
              break;
            case "shape":
              this.getBasicShapeProposals(entry, existingNode, result);
              break;
          }
        }
      }
      this.getValueEnumProposals(entry, existingNode, result);
      this.getCSSWideKeywordProposals(entry, existingNode, result);
      this.getUnitProposals(entry, existingNode, result);
    } else {
      var existingValues = collectValues(this.styleSheet, node);
      for (var _b = 0, _c = existingValues.getEntries(); _b < _c.length; _b++) {
        var existingValue = _c[_b];
        result.items.push({
          label: existingValue,
          textEdit: TextEdit.replace(this.getCompletionRange(existingNode), existingValue),
          kind: CompletionItemKind.Value
        });
      }
    }
    this.getVariableProposals(existingNode, result);
    this.getTermProposals(entry, existingNode, result);
    return result;
  };
  CSSCompletion2.prototype.getValueEnumProposals = function(entry, existingNode, result) {
    if (entry.values) {
      for (var _i = 0, _a2 = entry.values; _i < _a2.length; _i++) {
        var value = _a2[_i];
        var insertString = value.name;
        var insertTextFormat = void 0;
        if (endsWith(insertString, ")")) {
          var from = insertString.lastIndexOf("(");
          if (from !== -1) {
            insertString = insertString.substr(0, from) + "($1)";
            insertTextFormat = SnippetFormat;
          }
        }
        var sortText = SortTexts.Enums;
        if (startsWith(value.name, "-")) {
          sortText += SortTexts.VendorPrefixed;
        }
        var item = {
          label: value.name,
          documentation: getEntryDescription(value, this.doesSupportMarkdown()),
          tags: isDeprecated(entry) ? [CompletionItemTag.Deprecated] : [],
          textEdit: TextEdit.replace(this.getCompletionRange(existingNode), insertString),
          sortText,
          kind: CompletionItemKind.Value,
          insertTextFormat
        };
        result.items.push(item);
      }
    }
    return result;
  };
  CSSCompletion2.prototype.getCSSWideKeywordProposals = function(entry, existingNode, result) {
    for (var keywords in cssWideKeywords) {
      result.items.push({
        label: keywords,
        documentation: cssWideKeywords[keywords],
        textEdit: TextEdit.replace(this.getCompletionRange(existingNode), keywords),
        kind: CompletionItemKind.Value
      });
    }
    for (var func in cssWideFunctions) {
      var insertText = moveCursorInsideParenthesis(func);
      result.items.push({
        label: func,
        documentation: cssWideFunctions[func],
        textEdit: TextEdit.replace(this.getCompletionRange(existingNode), insertText),
        kind: CompletionItemKind.Function,
        insertTextFormat: SnippetFormat,
        command: startsWith(func, "var") ? retriggerCommand : void 0
      });
    }
    return result;
  };
  CSSCompletion2.prototype.getCompletionsForInterpolation = function(node, result) {
    if (this.offset >= node.offset + 2) {
      this.getVariableProposals(null, result);
    }
    return result;
  };
  CSSCompletion2.prototype.getVariableProposals = function(existingNode, result) {
    var symbols = this.getSymbolContext().findSymbolsAtOffset(this.offset, ReferenceType.Variable);
    for (var _i = 0, symbols_1 = symbols; _i < symbols_1.length; _i++) {
      var symbol = symbols_1[_i];
      var insertText = startsWith(symbol.name, "--") ? "var(".concat(symbol.name, ")") : symbol.name;
      var completionItem = {
        label: symbol.name,
        documentation: symbol.value ? getLimitedString(symbol.value) : symbol.value,
        textEdit: TextEdit.replace(this.getCompletionRange(existingNode), insertText),
        kind: CompletionItemKind.Variable,
        sortText: SortTexts.Variable
      };
      if (typeof completionItem.documentation === "string" && isColorString(completionItem.documentation)) {
        completionItem.kind = CompletionItemKind.Color;
      }
      if (symbol.node.type === NodeType.FunctionParameter) {
        var mixinNode = symbol.node.getParent();
        if (mixinNode.type === NodeType.MixinDeclaration) {
          completionItem.detail = localize4("completion.argument", "argument from '{0}'", mixinNode.getName());
        }
      }
      result.items.push(completionItem);
    }
    return result;
  };
  CSSCompletion2.prototype.getVariableProposalsForCSSVarFunction = function(result) {
    var allReferencedVariables = new Set();
    this.styleSheet.acceptVisitor(new VariableCollector(allReferencedVariables, this.offset));
    var symbols = this.getSymbolContext().findSymbolsAtOffset(this.offset, ReferenceType.Variable);
    for (var _i = 0, symbols_2 = symbols; _i < symbols_2.length; _i++) {
      var symbol = symbols_2[_i];
      if (startsWith(symbol.name, "--")) {
        var completionItem = {
          label: symbol.name,
          documentation: symbol.value ? getLimitedString(symbol.value) : symbol.value,
          textEdit: TextEdit.replace(this.getCompletionRange(null), symbol.name),
          kind: CompletionItemKind.Variable
        };
        if (typeof completionItem.documentation === "string" && isColorString(completionItem.documentation)) {
          completionItem.kind = CompletionItemKind.Color;
        }
        result.items.push(completionItem);
      }
      allReferencedVariables.remove(symbol.name);
    }
    for (var _a2 = 0, _b = allReferencedVariables.getEntries(); _a2 < _b.length; _a2++) {
      var name = _b[_a2];
      if (startsWith(name, "--")) {
        var completionItem = {
          label: name,
          textEdit: TextEdit.replace(this.getCompletionRange(null), name),
          kind: CompletionItemKind.Variable
        };
        result.items.push(completionItem);
      }
    }
    return result;
  };
  CSSCompletion2.prototype.getUnitProposals = function(entry, existingNode, result) {
    var currentWord = "0";
    if (this.currentWord.length > 0) {
      var numMatch = this.currentWord.match(/^-?\d[\.\d+]*/);
      if (numMatch) {
        currentWord = numMatch[0];
        result.isIncomplete = currentWord.length === this.currentWord.length;
      }
    } else if (this.currentWord.length === 0) {
      result.isIncomplete = true;
    }
    if (existingNode && existingNode.parent && existingNode.parent.type === NodeType.Term) {
      existingNode = existingNode.getParent();
    }
    if (entry.restrictions) {
      for (var _i = 0, _a2 = entry.restrictions; _i < _a2.length; _i++) {
        var restriction = _a2[_i];
        var units2 = units[restriction];
        if (units2) {
          for (var _b = 0, units_1 = units2; _b < units_1.length; _b++) {
            var unit = units_1[_b];
            var insertText = currentWord + unit;
            result.items.push({
              label: insertText,
              textEdit: TextEdit.replace(this.getCompletionRange(existingNode), insertText),
              kind: CompletionItemKind.Unit
            });
          }
        }
      }
    }
    return result;
  };
  CSSCompletion2.prototype.getCompletionRange = function(existingNode) {
    if (existingNode && existingNode.offset <= this.offset && this.offset <= existingNode.end) {
      var end = existingNode.end !== -1 ? this.textDocument.positionAt(existingNode.end) : this.position;
      var start = this.textDocument.positionAt(existingNode.offset);
      if (start.line === end.line) {
        return Range.create(start, end);
      }
    }
    return this.defaultReplaceRange;
  };
  CSSCompletion2.prototype.getColorProposals = function(entry, existingNode, result) {
    for (var color in colors) {
      result.items.push({
        label: color,
        documentation: colors[color],
        textEdit: TextEdit.replace(this.getCompletionRange(existingNode), color),
        kind: CompletionItemKind.Color
      });
    }
    for (var color in colorKeywords) {
      result.items.push({
        label: color,
        documentation: colorKeywords[color],
        textEdit: TextEdit.replace(this.getCompletionRange(existingNode), color),
        kind: CompletionItemKind.Value
      });
    }
    var colorValues = new Set();
    this.styleSheet.acceptVisitor(new ColorValueCollector(colorValues, this.offset));
    for (var _i = 0, _a2 = colorValues.getEntries(); _i < _a2.length; _i++) {
      var color = _a2[_i];
      result.items.push({
        label: color,
        textEdit: TextEdit.replace(this.getCompletionRange(existingNode), color),
        kind: CompletionItemKind.Color
      });
    }
    var _loop_1 = function(p2) {
      var tabStop = 1;
      var replaceFunction = function(_match, p1) {
        return "${" + tabStop++ + ":" + p1 + "}";
      };
      var insertText = p2.func.replace(/\[?\$(\w+)\]?/g, replaceFunction);
      result.items.push({
        label: p2.func.substr(0, p2.func.indexOf("(")),
        detail: p2.func,
        documentation: p2.desc,
        textEdit: TextEdit.replace(this_1.getCompletionRange(existingNode), insertText),
        insertTextFormat: SnippetFormat,
        kind: CompletionItemKind.Function
      });
    };
    var this_1 = this;
    for (var _b = 0, _c = colorFunctions; _b < _c.length; _b++) {
      var p = _c[_b];
      _loop_1(p);
    }
    return result;
  };
  CSSCompletion2.prototype.getPositionProposals = function(entry, existingNode, result) {
    for (var position in positionKeywords) {
      result.items.push({
        label: position,
        documentation: positionKeywords[position],
        textEdit: TextEdit.replace(this.getCompletionRange(existingNode), position),
        kind: CompletionItemKind.Value
      });
    }
    return result;
  };
  CSSCompletion2.prototype.getRepeatStyleProposals = function(entry, existingNode, result) {
    for (var repeat2 in repeatStyleKeywords) {
      result.items.push({
        label: repeat2,
        documentation: repeatStyleKeywords[repeat2],
        textEdit: TextEdit.replace(this.getCompletionRange(existingNode), repeat2),
        kind: CompletionItemKind.Value
      });
    }
    return result;
  };
  CSSCompletion2.prototype.getLineStyleProposals = function(entry, existingNode, result) {
    for (var lineStyle in lineStyleKeywords) {
      result.items.push({
        label: lineStyle,
        documentation: lineStyleKeywords[lineStyle],
        textEdit: TextEdit.replace(this.getCompletionRange(existingNode), lineStyle),
        kind: CompletionItemKind.Value
      });
    }
    return result;
  };
  CSSCompletion2.prototype.getLineWidthProposals = function(entry, existingNode, result) {
    for (var _i = 0, _a2 = lineWidthKeywords; _i < _a2.length; _i++) {
      var lineWidth = _a2[_i];
      result.items.push({
        label: lineWidth,
        textEdit: TextEdit.replace(this.getCompletionRange(existingNode), lineWidth),
        kind: CompletionItemKind.Value
      });
    }
    return result;
  };
  CSSCompletion2.prototype.getGeometryBoxProposals = function(entry, existingNode, result) {
    for (var box in geometryBoxKeywords) {
      result.items.push({
        label: box,
        documentation: geometryBoxKeywords[box],
        textEdit: TextEdit.replace(this.getCompletionRange(existingNode), box),
        kind: CompletionItemKind.Value
      });
    }
    return result;
  };
  CSSCompletion2.prototype.getBoxProposals = function(entry, existingNode, result) {
    for (var box in boxKeywords) {
      result.items.push({
        label: box,
        documentation: boxKeywords[box],
        textEdit: TextEdit.replace(this.getCompletionRange(existingNode), box),
        kind: CompletionItemKind.Value
      });
    }
    return result;
  };
  CSSCompletion2.prototype.getImageProposals = function(entry, existingNode, result) {
    for (var image in imageFunctions) {
      var insertText = moveCursorInsideParenthesis(image);
      result.items.push({
        label: image,
        documentation: imageFunctions[image],
        textEdit: TextEdit.replace(this.getCompletionRange(existingNode), insertText),
        kind: CompletionItemKind.Function,
        insertTextFormat: image !== insertText ? SnippetFormat : void 0
      });
    }
    return result;
  };
  CSSCompletion2.prototype.getTimingFunctionProposals = function(entry, existingNode, result) {
    for (var timing in transitionTimingFunctions) {
      var insertText = moveCursorInsideParenthesis(timing);
      result.items.push({
        label: timing,
        documentation: transitionTimingFunctions[timing],
        textEdit: TextEdit.replace(this.getCompletionRange(existingNode), insertText),
        kind: CompletionItemKind.Function,
        insertTextFormat: timing !== insertText ? SnippetFormat : void 0
      });
    }
    return result;
  };
  CSSCompletion2.prototype.getBasicShapeProposals = function(entry, existingNode, result) {
    for (var shape in basicShapeFunctions) {
      var insertText = moveCursorInsideParenthesis(shape);
      result.items.push({
        label: shape,
        documentation: basicShapeFunctions[shape],
        textEdit: TextEdit.replace(this.getCompletionRange(existingNode), insertText),
        kind: CompletionItemKind.Function,
        insertTextFormat: shape !== insertText ? SnippetFormat : void 0
      });
    }
    return result;
  };
  CSSCompletion2.prototype.getCompletionsForStylesheet = function(result) {
    var node = this.styleSheet.findFirstChildBeforeOffset(this.offset);
    if (!node) {
      return this.getCompletionForTopLevel(result);
    }
    if (node instanceof RuleSet) {
      return this.getCompletionsForRuleSet(node, result);
    }
    if (node instanceof Supports) {
      return this.getCompletionsForSupports(node, result);
    }
    return result;
  };
  CSSCompletion2.prototype.getCompletionForTopLevel = function(result) {
    var _this = this;
    this.cssDataManager.getAtDirectives().forEach(function(entry) {
      result.items.push({
        label: entry.name,
        textEdit: TextEdit.replace(_this.getCompletionRange(null), entry.name),
        documentation: getEntryDescription(entry, _this.doesSupportMarkdown()),
        tags: isDeprecated(entry) ? [CompletionItemTag.Deprecated] : [],
        kind: CompletionItemKind.Keyword
      });
    });
    this.getCompletionsForSelector(null, false, result);
    return result;
  };
  CSSCompletion2.prototype.getCompletionsForRuleSet = function(ruleSet, result) {
    var declarations = ruleSet.getDeclarations();
    var isAfter = declarations && declarations.endsWith("}") && this.offset >= declarations.end;
    if (isAfter) {
      return this.getCompletionForTopLevel(result);
    }
    var isInSelectors = !declarations || this.offset <= declarations.offset;
    if (isInSelectors) {
      return this.getCompletionsForSelector(ruleSet, ruleSet.isNested(), result);
    }
    return this.getCompletionsForDeclarations(ruleSet.getDeclarations(), result);
  };
  CSSCompletion2.prototype.getCompletionsForSelector = function(ruleSet, isNested, result) {
    var _this = this;
    var existingNode = this.findInNodePath(NodeType.PseudoSelector, NodeType.IdentifierSelector, NodeType.ClassSelector, NodeType.ElementNameSelector);
    if (!existingNode && this.hasCharacterAtPosition(this.offset - this.currentWord.length - 1, ":")) {
      this.currentWord = ":" + this.currentWord;
      if (this.hasCharacterAtPosition(this.offset - this.currentWord.length - 1, ":")) {
        this.currentWord = ":" + this.currentWord;
      }
      this.defaultReplaceRange = Range.create(Position.create(this.position.line, this.position.character - this.currentWord.length), this.position);
    }
    var pseudoClasses = this.cssDataManager.getPseudoClasses();
    pseudoClasses.forEach(function(entry2) {
      var insertText = moveCursorInsideParenthesis(entry2.name);
      var item = {
        label: entry2.name,
        textEdit: TextEdit.replace(_this.getCompletionRange(existingNode), insertText),
        documentation: getEntryDescription(entry2, _this.doesSupportMarkdown()),
        tags: isDeprecated(entry2) ? [CompletionItemTag.Deprecated] : [],
        kind: CompletionItemKind.Function,
        insertTextFormat: entry2.name !== insertText ? SnippetFormat : void 0
      };
      if (startsWith(entry2.name, ":-")) {
        item.sortText = SortTexts.VendorPrefixed;
      }
      result.items.push(item);
    });
    var pseudoElements = this.cssDataManager.getPseudoElements();
    pseudoElements.forEach(function(entry2) {
      var insertText = moveCursorInsideParenthesis(entry2.name);
      var item = {
        label: entry2.name,
        textEdit: TextEdit.replace(_this.getCompletionRange(existingNode), insertText),
        documentation: getEntryDescription(entry2, _this.doesSupportMarkdown()),
        tags: isDeprecated(entry2) ? [CompletionItemTag.Deprecated] : [],
        kind: CompletionItemKind.Function,
        insertTextFormat: entry2.name !== insertText ? SnippetFormat : void 0
      };
      if (startsWith(entry2.name, "::-")) {
        item.sortText = SortTexts.VendorPrefixed;
      }
      result.items.push(item);
    });
    if (!isNested) {
      for (var _i = 0, _a2 = html5Tags; _i < _a2.length; _i++) {
        var entry = _a2[_i];
        result.items.push({
          label: entry,
          textEdit: TextEdit.replace(this.getCompletionRange(existingNode), entry),
          kind: CompletionItemKind.Keyword
        });
      }
      for (var _b = 0, _c = svgElements; _b < _c.length; _b++) {
        var entry = _c[_b];
        result.items.push({
          label: entry,
          textEdit: TextEdit.replace(this.getCompletionRange(existingNode), entry),
          kind: CompletionItemKind.Keyword
        });
      }
    }
    var visited = {};
    visited[this.currentWord] = true;
    var docText = this.textDocument.getText();
    this.styleSheet.accept(function(n) {
      if (n.type === NodeType.SimpleSelector && n.length > 0) {
        var selector2 = docText.substr(n.offset, n.length);
        if (selector2.charAt(0) === "." && !visited[selector2]) {
          visited[selector2] = true;
          result.items.push({
            label: selector2,
            textEdit: TextEdit.replace(_this.getCompletionRange(existingNode), selector2),
            kind: CompletionItemKind.Keyword
          });
        }
        return false;
      }
      return true;
    });
    if (ruleSet && ruleSet.isNested()) {
      var selector = ruleSet.getSelectors().findFirstChildBeforeOffset(this.offset);
      if (selector && ruleSet.getSelectors().getChildren().indexOf(selector) === 0) {
        this.getPropertyProposals(null, result);
      }
    }
    return result;
  };
  CSSCompletion2.prototype.getCompletionsForDeclarations = function(declarations, result) {
    if (!declarations || this.offset === declarations.offset) {
      return result;
    }
    var node = declarations.findFirstChildBeforeOffset(this.offset);
    if (!node) {
      return this.getCompletionsForDeclarationProperty(null, result);
    }
    if (node instanceof AbstractDeclaration) {
      var declaration = node;
      if (!isDefined(declaration.colonPosition) || this.offset <= declaration.colonPosition) {
        return this.getCompletionsForDeclarationProperty(declaration, result);
      } else if (isDefined(declaration.semicolonPosition) && declaration.semicolonPosition < this.offset) {
        if (this.offset === declaration.semicolonPosition + 1) {
          return result;
        }
        return this.getCompletionsForDeclarationProperty(null, result);
      }
      if (declaration instanceof Declaration) {
        return this.getCompletionsForDeclarationValue(declaration, result);
      }
    } else if (node instanceof ExtendsReference) {
      this.getCompletionsForExtendsReference(node, null, result);
    } else if (this.currentWord && this.currentWord[0] === "@") {
      this.getCompletionsForDeclarationProperty(null, result);
    } else if (node instanceof RuleSet) {
      this.getCompletionsForDeclarationProperty(null, result);
    }
    return result;
  };
  CSSCompletion2.prototype.getCompletionsForVariableDeclaration = function(declaration, result) {
    if (this.offset && isDefined(declaration.colonPosition) && this.offset > declaration.colonPosition) {
      this.getVariableProposals(declaration.getValue(), result);
    }
    return result;
  };
  CSSCompletion2.prototype.getCompletionsForExpression = function(expression, result) {
    var parent = expression.getParent();
    if (parent instanceof FunctionArgument) {
      this.getCompletionsForFunctionArgument(parent, parent.getParent(), result);
      return result;
    }
    var declaration = expression.findParent(NodeType.Declaration);
    if (!declaration) {
      this.getTermProposals(void 0, null, result);
      return result;
    }
    var node = expression.findChildAtOffset(this.offset, true);
    if (!node) {
      return this.getCompletionsForDeclarationValue(declaration, result);
    }
    if (node instanceof NumericValue || node instanceof Identifier) {
      return this.getCompletionsForDeclarationValue(declaration, result);
    }
    return result;
  };
  CSSCompletion2.prototype.getCompletionsForFunctionArgument = function(arg, func, result) {
    var identifier = func.getIdentifier();
    if (identifier && identifier.matches("var")) {
      if (!func.getArguments().hasChildren() || func.getArguments().getChild(0) === arg) {
        this.getVariableProposalsForCSSVarFunction(result);
      }
    }
    return result;
  };
  CSSCompletion2.prototype.getCompletionsForFunctionDeclaration = function(decl, result) {
    var declarations = decl.getDeclarations();
    if (declarations && this.offset > declarations.offset && this.offset < declarations.end) {
      this.getTermProposals(void 0, null, result);
    }
    return result;
  };
  CSSCompletion2.prototype.getCompletionsForMixinReference = function(ref, result) {
    var _this = this;
    var allMixins = this.getSymbolContext().findSymbolsAtOffset(this.offset, ReferenceType.Mixin);
    for (var _i = 0, allMixins_1 = allMixins; _i < allMixins_1.length; _i++) {
      var mixinSymbol = allMixins_1[_i];
      if (mixinSymbol.node instanceof MixinDeclaration) {
        result.items.push(this.makeTermProposal(mixinSymbol, mixinSymbol.node.getParameters(), null));
      }
    }
    var identifierNode = ref.getIdentifier() || null;
    this.completionParticipants.forEach(function(participant) {
      if (participant.onCssMixinReference) {
        participant.onCssMixinReference({
          mixinName: _this.currentWord,
          range: _this.getCompletionRange(identifierNode)
        });
      }
    });
    return result;
  };
  CSSCompletion2.prototype.getTermProposals = function(entry, existingNode, result) {
    var allFunctions = this.getSymbolContext().findSymbolsAtOffset(this.offset, ReferenceType.Function);
    for (var _i = 0, allFunctions_1 = allFunctions; _i < allFunctions_1.length; _i++) {
      var functionSymbol = allFunctions_1[_i];
      if (functionSymbol.node instanceof FunctionDeclaration) {
        result.items.push(this.makeTermProposal(functionSymbol, functionSymbol.node.getParameters(), existingNode));
      }
    }
    return result;
  };
  CSSCompletion2.prototype.makeTermProposal = function(symbol, parameters, existingNode) {
    var decl = symbol.node;
    var params = parameters.getChildren().map(function(c) {
      return c instanceof FunctionParameter ? c.getName() : c.getText();
    });
    var insertText = symbol.name + "(" + params.map(function(p, index) {
      return "${" + (index + 1) + ":" + p + "}";
    }).join(", ") + ")";
    return {
      label: symbol.name,
      detail: symbol.name + "(" + params.join(", ") + ")",
      textEdit: TextEdit.replace(this.getCompletionRange(existingNode), insertText),
      insertTextFormat: SnippetFormat,
      kind: CompletionItemKind.Function,
      sortText: SortTexts.Term
    };
  };
  CSSCompletion2.prototype.getCompletionsForSupportsCondition = function(supportsCondition, result) {
    var child = supportsCondition.findFirstChildBeforeOffset(this.offset);
    if (child) {
      if (child instanceof Declaration) {
        if (!isDefined(child.colonPosition) || this.offset <= child.colonPosition) {
          return this.getCompletionsForDeclarationProperty(child, result);
        } else {
          return this.getCompletionsForDeclarationValue(child, result);
        }
      } else if (child instanceof SupportsCondition) {
        return this.getCompletionsForSupportsCondition(child, result);
      }
    }
    if (isDefined(supportsCondition.lParent) && this.offset > supportsCondition.lParent && (!isDefined(supportsCondition.rParent) || this.offset <= supportsCondition.rParent)) {
      return this.getCompletionsForDeclarationProperty(null, result);
    }
    return result;
  };
  CSSCompletion2.prototype.getCompletionsForSupports = function(supports, result) {
    var declarations = supports.getDeclarations();
    var inInCondition = !declarations || this.offset <= declarations.offset;
    if (inInCondition) {
      var child = supports.findFirstChildBeforeOffset(this.offset);
      if (child instanceof SupportsCondition) {
        return this.getCompletionsForSupportsCondition(child, result);
      }
      return result;
    }
    return this.getCompletionForTopLevel(result);
  };
  CSSCompletion2.prototype.getCompletionsForExtendsReference = function(extendsRef, existingNode, result) {
    return result;
  };
  CSSCompletion2.prototype.getCompletionForUriLiteralValue = function(uriLiteralNode, result) {
    var uriValue;
    var position;
    var range;
    if (!uriLiteralNode.hasChildren()) {
      uriValue = "";
      position = this.position;
      var emptyURIValuePosition = this.textDocument.positionAt(uriLiteralNode.offset + "url(".length);
      range = Range.create(emptyURIValuePosition, emptyURIValuePosition);
    } else {
      var uriValueNode = uriLiteralNode.getChild(0);
      uriValue = uriValueNode.getText();
      position = this.position;
      range = this.getCompletionRange(uriValueNode);
    }
    this.completionParticipants.forEach(function(participant) {
      if (participant.onCssURILiteralValue) {
        participant.onCssURILiteralValue({
          uriValue,
          position,
          range
        });
      }
    });
    return result;
  };
  CSSCompletion2.prototype.getCompletionForImportPath = function(importPathNode, result) {
    var _this = this;
    this.completionParticipants.forEach(function(participant) {
      if (participant.onCssImportPath) {
        participant.onCssImportPath({
          pathValue: importPathNode.getText(),
          position: _this.position,
          range: _this.getCompletionRange(importPathNode)
        });
      }
    });
    return result;
  };
  CSSCompletion2.prototype.hasCharacterAtPosition = function(offset, char) {
    var text = this.textDocument.getText();
    return offset >= 0 && offset < text.length && text.charAt(offset) === char;
  };
  CSSCompletion2.prototype.doesSupportMarkdown = function() {
    var _a2, _b, _c;
    if (!isDefined(this.supportsMarkdown)) {
      if (!isDefined(this.lsOptions.clientCapabilities)) {
        this.supportsMarkdown = true;
        return this.supportsMarkdown;
      }
      var documentationFormat = (_c = (_b = (_a2 = this.lsOptions.clientCapabilities.textDocument) === null || _a2 === void 0 ? void 0 : _a2.completion) === null || _b === void 0 ? void 0 : _b.completionItem) === null || _c === void 0 ? void 0 : _c.documentationFormat;
      this.supportsMarkdown = Array.isArray(documentationFormat) && documentationFormat.indexOf(MarkupKind.Markdown) !== -1;
    }
    return this.supportsMarkdown;
  };
  return CSSCompletion2;
}();
function isDeprecated(entry) {
  if (entry.status && (entry.status === "nonstandard" || entry.status === "obsolete")) {
    return true;
  }
  return false;
}
var Set = function() {
  function Set2() {
    this.entries = {};
  }
  Set2.prototype.add = function(entry) {
    this.entries[entry] = true;
  };
  Set2.prototype.remove = function(entry) {
    delete this.entries[entry];
  };
  Set2.prototype.getEntries = function() {
    return Object.keys(this.entries);
  };
  return Set2;
}();
function moveCursorInsideParenthesis(text) {
  return text.replace(/\(\)$/, "($1)");
}
function collectValues(styleSheet, declaration) {
  var fullPropertyName = declaration.getFullPropertyName();
  var entries = new Set();
  function visitValue(node) {
    if (node instanceof Identifier || node instanceof NumericValue || node instanceof HexColorValue) {
      entries.add(node.getText());
    }
    return true;
  }
  function matchesProperty(decl) {
    var propertyName = decl.getFullPropertyName();
    return fullPropertyName === propertyName;
  }
  function vistNode(node) {
    if (node instanceof Declaration && node !== declaration) {
      if (matchesProperty(node)) {
        var value = node.getValue();
        if (value) {
          value.accept(visitValue);
        }
      }
    }
    return true;
  }
  styleSheet.accept(vistNode);
  return entries;
}
var ColorValueCollector = function() {
  function ColorValueCollector2(entries, currentOffset) {
    this.entries = entries;
    this.currentOffset = currentOffset;
  }
  ColorValueCollector2.prototype.visitNode = function(node) {
    if (node instanceof HexColorValue || node instanceof Function && isColorConstructor(node)) {
      if (this.currentOffset < node.offset || node.end < this.currentOffset) {
        this.entries.add(node.getText());
      }
    }
    return true;
  };
  return ColorValueCollector2;
}();
var VariableCollector = function() {
  function VariableCollector2(entries, currentOffset) {
    this.entries = entries;
    this.currentOffset = currentOffset;
  }
  VariableCollector2.prototype.visitNode = function(node) {
    if (node instanceof Identifier && node.isCustomProperty) {
      if (this.currentOffset < node.offset || node.end < this.currentOffset) {
        this.entries.add(node.getText());
      }
    }
    return true;
  };
  return VariableCollector2;
}();
function getCurrentWord(document, offset) {
  var i = offset - 1;
  var text = document.getText();
  while (i >= 0 && ' 	\n\r":{[()]},*>+'.indexOf(text.charAt(i)) === -1) {
    i--;
  }
  return text.substring(i + 1, offset);
}
function isColorString(s) {
  return s.toLowerCase() in colors || /(^#[0-9A-F]{6}$)|(^#[0-9A-F]{3}$)/i.test(s);
}

// node_modules/vscode-css-languageservice/lib/esm/services/selectorPrinting.js
var __extends3 = function() {
  var extendStatics = function(d, b) {
    extendStatics = Object.setPrototypeOf || { __proto__: [] } instanceof Array && function(d2, b2) {
      d2.__proto__ = b2;
    } || function(d2, b2) {
      for (var p in b2)
        if (Object.prototype.hasOwnProperty.call(b2, p))
          d2[p] = b2[p];
    };
    return extendStatics(d, b);
  };
  return function(d, b) {
    if (typeof b !== "function" && b !== null)
      throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
    extendStatics(d, b);
    function __() {
      this.constructor = d;
    }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
  };
}();
var localize5 = loadMessageBundle();
var Element = function() {
  function Element3() {
    this.parent = null;
    this.children = null;
    this.attributes = null;
  }
  Element3.prototype.findAttribute = function(name) {
    if (this.attributes) {
      for (var _i = 0, _a2 = this.attributes; _i < _a2.length; _i++) {
        var attribute = _a2[_i];
        if (attribute.name === name) {
          return attribute.value;
        }
      }
    }
    return null;
  };
  Element3.prototype.addChild = function(child) {
    if (child instanceof Element3) {
      child.parent = this;
    }
    if (!this.children) {
      this.children = [];
    }
    this.children.push(child);
  };
  Element3.prototype.append = function(text) {
    if (this.attributes) {
      var last = this.attributes[this.attributes.length - 1];
      last.value = last.value + text;
    }
  };
  Element3.prototype.prepend = function(text) {
    if (this.attributes) {
      var first = this.attributes[0];
      first.value = text + first.value;
    }
  };
  Element3.prototype.findRoot = function() {
    var curr = this;
    while (curr.parent && !(curr.parent instanceof RootElement)) {
      curr = curr.parent;
    }
    return curr;
  };
  Element3.prototype.removeChild = function(child) {
    if (this.children) {
      var index = this.children.indexOf(child);
      if (index !== -1) {
        this.children.splice(index, 1);
        return true;
      }
    }
    return false;
  };
  Element3.prototype.addAttr = function(name, value) {
    if (!this.attributes) {
      this.attributes = [];
    }
    for (var _i = 0, _a2 = this.attributes; _i < _a2.length; _i++) {
      var attribute = _a2[_i];
      if (attribute.name === name) {
        attribute.value += " " + value;
        return;
      }
    }
    this.attributes.push({ name, value });
  };
  Element3.prototype.clone = function(cloneChildren) {
    if (cloneChildren === void 0) {
      cloneChildren = true;
    }
    var elem = new Element3();
    if (this.attributes) {
      elem.attributes = [];
      for (var _i = 0, _a2 = this.attributes; _i < _a2.length; _i++) {
        var attribute = _a2[_i];
        elem.addAttr(attribute.name, attribute.value);
      }
    }
    if (cloneChildren && this.children) {
      elem.children = [];
      for (var index = 0; index < this.children.length; index++) {
        elem.addChild(this.children[index].clone());
      }
    }
    return elem;
  };
  Element3.prototype.cloneWithParent = function() {
    var clone = this.clone(false);
    if (this.parent && !(this.parent instanceof RootElement)) {
      var parentClone = this.parent.cloneWithParent();
      parentClone.addChild(clone);
    }
    return clone;
  };
  return Element3;
}();
var RootElement = function(_super) {
  __extends3(RootElement2, _super);
  function RootElement2() {
    return _super !== null && _super.apply(this, arguments) || this;
  }
  return RootElement2;
}(Element);
var LabelElement = function(_super) {
  __extends3(LabelElement2, _super);
  function LabelElement2(label) {
    var _this = _super.call(this) || this;
    _this.addAttr("name", label);
    return _this;
  }
  return LabelElement2;
}(Element);
var MarkedStringPrinter = function() {
  function MarkedStringPrinter2(quote) {
    this.quote = quote;
    this.result = [];
  }
  MarkedStringPrinter2.prototype.print = function(element) {
    this.result = [];
    if (element instanceof RootElement) {
      if (element.children) {
        this.doPrint(element.children, 0);
      }
    } else {
      this.doPrint([element], 0);
    }
    var value = this.result.join("\n");
    return [{ language: "html", value }];
  };
  MarkedStringPrinter2.prototype.doPrint = function(elements, indent) {
    for (var _i = 0, elements_1 = elements; _i < elements_1.length; _i++) {
      var element = elements_1[_i];
      this.doPrintElement(element, indent);
      if (element.children) {
        this.doPrint(element.children, indent + 1);
      }
    }
  };
  MarkedStringPrinter2.prototype.writeLine = function(level, content) {
    var indent = new Array(level + 1).join("  ");
    this.result.push(indent + content);
  };
  MarkedStringPrinter2.prototype.doPrintElement = function(element, indent) {
    var name = element.findAttribute("name");
    if (element instanceof LabelElement || name === "\u2026") {
      this.writeLine(indent, name);
      return;
    }
    var content = ["<"];
    if (name) {
      content.push(name);
    } else {
      content.push("element");
    }
    if (element.attributes) {
      for (var _i = 0, _a2 = element.attributes; _i < _a2.length; _i++) {
        var attr = _a2[_i];
        if (attr.name !== "name") {
          content.push(" ");
          content.push(attr.name);
          var value = attr.value;
          if (value) {
            content.push("=");
            content.push(quotes.ensure(value, this.quote));
          }
        }
      }
    }
    content.push(">");
    this.writeLine(indent, content.join(""));
  };
  return MarkedStringPrinter2;
}();
var quotes;
(function(quotes2) {
  function ensure(value, which) {
    return which + remove(value) + which;
  }
  quotes2.ensure = ensure;
  function remove(value) {
    var match = value.match(/^['"](.*)["']$/);
    if (match) {
      return match[1];
    }
    return value;
  }
  quotes2.remove = remove;
})(quotes || (quotes = {}));
var Specificity = function() {
  function Specificity2() {
    this.id = 0;
    this.attr = 0;
    this.tag = 0;
  }
  return Specificity2;
}();
function toElement(node, parentElement) {
  var result = new Element();
  for (var _i = 0, _a2 = node.getChildren(); _i < _a2.length; _i++) {
    var child = _a2[_i];
    switch (child.type) {
      case NodeType.SelectorCombinator:
        if (parentElement) {
          var segments = child.getText().split("&");
          if (segments.length === 1) {
            result.addAttr("name", segments[0]);
            break;
          }
          result = parentElement.cloneWithParent();
          if (segments[0]) {
            var root = result.findRoot();
            root.prepend(segments[0]);
          }
          for (var i = 1; i < segments.length; i++) {
            if (i > 1) {
              var clone = parentElement.cloneWithParent();
              result.addChild(clone.findRoot());
              result = clone;
            }
            result.append(segments[i]);
          }
        }
        break;
      case NodeType.SelectorPlaceholder:
        if (child.matches("@at-root")) {
          return result;
        }
      case NodeType.ElementNameSelector:
        var text = child.getText();
        result.addAttr("name", text === "*" ? "element" : unescape(text));
        break;
      case NodeType.ClassSelector:
        result.addAttr("class", unescape(child.getText().substring(1)));
        break;
      case NodeType.IdentifierSelector:
        result.addAttr("id", unescape(child.getText().substring(1)));
        break;
      case NodeType.MixinDeclaration:
        result.addAttr("class", child.getName());
        break;
      case NodeType.PseudoSelector:
        result.addAttr(unescape(child.getText()), "");
        break;
      case NodeType.AttributeSelector:
        var selector = child;
        var identifier = selector.getIdentifier();
        if (identifier) {
          var expression = selector.getValue();
          var operator = selector.getOperator();
          var value = void 0;
          if (expression && operator) {
            switch (unescape(operator.getText())) {
              case "|=":
                value = "".concat(quotes.remove(unescape(expression.getText())), "-\u2026");
                break;
              case "^=":
                value = "".concat(quotes.remove(unescape(expression.getText())), "\u2026");
                break;
              case "$=":
                value = "\u2026".concat(quotes.remove(unescape(expression.getText())));
                break;
              case "~=":
                value = " \u2026 ".concat(quotes.remove(unescape(expression.getText())), " \u2026 ");
                break;
              case "*=":
                value = "\u2026".concat(quotes.remove(unescape(expression.getText())), "\u2026");
                break;
              default:
                value = quotes.remove(unescape(expression.getText()));
                break;
            }
          }
          result.addAttr(unescape(identifier.getText()), value);
        }
        break;
    }
  }
  return result;
}
function unescape(content) {
  var scanner = new Scanner();
  scanner.setSource(content);
  var token = scanner.scanUnquotedString();
  if (token) {
    return token.text;
  }
  return content;
}
var SelectorPrinting = function() {
  function SelectorPrinting2(cssDataManager) {
    this.cssDataManager = cssDataManager;
  }
  SelectorPrinting2.prototype.selectorToMarkedString = function(node) {
    var root = selectorToElement(node);
    if (root) {
      var markedStrings = new MarkedStringPrinter('"').print(root);
      markedStrings.push(this.selectorToSpecificityMarkedString(node));
      return markedStrings;
    } else {
      return [];
    }
  };
  SelectorPrinting2.prototype.simpleSelectorToMarkedString = function(node) {
    var element = toElement(node);
    var markedStrings = new MarkedStringPrinter('"').print(element);
    markedStrings.push(this.selectorToSpecificityMarkedString(node));
    return markedStrings;
  };
  SelectorPrinting2.prototype.isPseudoElementIdentifier = function(text) {
    var match = text.match(/^::?([\w-]+)/);
    if (!match) {
      return false;
    }
    return !!this.cssDataManager.getPseudoElement("::" + match[1]);
  };
  SelectorPrinting2.prototype.selectorToSpecificityMarkedString = function(node) {
    var _this = this;
    var calculateScore = function(node2) {
      var specificity2 = new Specificity();
      elementLoop:
        for (var _i = 0, _a2 = node2.getChildren(); _i < _a2.length; _i++) {
          var element = _a2[_i];
          switch (element.type) {
            case NodeType.IdentifierSelector:
              specificity2.id++;
              break;
            case NodeType.ClassSelector:
            case NodeType.AttributeSelector:
              specificity2.attr++;
              break;
            case NodeType.ElementNameSelector:
              if (element.matches("*")) {
                break;
              }
              specificity2.tag++;
              break;
            case NodeType.PseudoSelector:
              var text = element.getText();
              if (_this.isPseudoElementIdentifier(text)) {
                specificity2.tag++;
                break;
              }
              if (text.match(/^:where/i)) {
                continue elementLoop;
              }
              if (text.match(/^:(not|has|is)/i) && element.getChildren().length > 0) {
                var mostSpecificListItem = new Specificity();
                for (var _b = 0, _c = element.getChildren(); _b < _c.length; _b++) {
                  var containerElement = _c[_b];
                  var list = void 0;
                  if (containerElement.type === NodeType.Undefined) {
                    list = containerElement.getChildren();
                  } else {
                    list = [containerElement];
                  }
                  for (var _d = 0, _e = containerElement.getChildren(); _d < _e.length; _d++) {
                    var childElement = _e[_d];
                    var itemSpecificity = calculateScore(childElement);
                    if (itemSpecificity.id > mostSpecificListItem.id) {
                      mostSpecificListItem = itemSpecificity;
                      continue;
                    } else if (itemSpecificity.id < mostSpecificListItem.id) {
                      continue;
                    }
                    if (itemSpecificity.attr > mostSpecificListItem.attr) {
                      mostSpecificListItem = itemSpecificity;
                      continue;
                    } else if (itemSpecificity.attr < mostSpecificListItem.attr) {
                      continue;
                    }
                    if (itemSpecificity.tag > mostSpecificListItem.tag) {
                      mostSpecificListItem = itemSpecificity;
                      continue;
                    }
                  }
                }
                specificity2.id += mostSpecificListItem.id;
                specificity2.attr += mostSpecificListItem.attr;
                specificity2.tag += mostSpecificListItem.tag;
                continue elementLoop;
              }
              specificity2.attr++;
              break;
          }
          if (element.getChildren().length > 0) {
            var itemSpecificity = calculateScore(element);
            specificity2.id += itemSpecificity.id;
            specificity2.attr += itemSpecificity.attr;
            specificity2.tag += itemSpecificity.tag;
          }
        }
      return specificity2;
    };
    var specificity = calculateScore(node);
    ;
    return localize5("specificity", "[Selector Specificity](https://developer.mozilla.org/en-US/docs/Web/CSS/Specificity): ({0}, {1}, {2})", specificity.id, specificity.attr, specificity.tag);
  };
  return SelectorPrinting2;
}();
var SelectorElementBuilder = function() {
  function SelectorElementBuilder2(element) {
    this.prev = null;
    this.element = element;
  }
  SelectorElementBuilder2.prototype.processSelector = function(selector) {
    var parentElement = null;
    if (!(this.element instanceof RootElement)) {
      if (selector.getChildren().some(function(c) {
        return c.hasChildren() && c.getChild(0).type === NodeType.SelectorCombinator;
      })) {
        var curr = this.element.findRoot();
        if (curr.parent instanceof RootElement) {
          parentElement = this.element;
          this.element = curr.parent;
          this.element.removeChild(curr);
          this.prev = null;
        }
      }
    }
    for (var _i = 0, _a2 = selector.getChildren(); _i < _a2.length; _i++) {
      var selectorChild = _a2[_i];
      if (selectorChild instanceof SimpleSelector) {
        if (this.prev instanceof SimpleSelector) {
          var labelElement = new LabelElement("\u2026");
          this.element.addChild(labelElement);
          this.element = labelElement;
        } else if (this.prev && (this.prev.matches("+") || this.prev.matches("~")) && this.element.parent) {
          this.element = this.element.parent;
        }
        if (this.prev && this.prev.matches("~")) {
          this.element.addChild(new LabelElement("\u22EE"));
        }
        var thisElement = toElement(selectorChild, parentElement);
        var root = thisElement.findRoot();
        this.element.addChild(root);
        this.element = thisElement;
      }
      if (selectorChild instanceof SimpleSelector || selectorChild.type === NodeType.SelectorCombinatorParent || selectorChild.type === NodeType.SelectorCombinatorShadowPiercingDescendant || selectorChild.type === NodeType.SelectorCombinatorSibling || selectorChild.type === NodeType.SelectorCombinatorAllSiblings) {
        this.prev = selectorChild;
      }
    }
  };
  return SelectorElementBuilder2;
}();
function isNewSelectorContext(node) {
  switch (node.type) {
    case NodeType.MixinDeclaration:
    case NodeType.Stylesheet:
      return true;
  }
  return false;
}
function selectorToElement(node) {
  if (node.matches("@at-root")) {
    return null;
  }
  var root = new RootElement();
  var parentRuleSets = [];
  var ruleSet = node.getParent();
  if (ruleSet instanceof RuleSet) {
    var parent = ruleSet.getParent();
    while (parent && !isNewSelectorContext(parent)) {
      if (parent instanceof RuleSet) {
        if (parent.getSelectors().matches("@at-root")) {
          break;
        }
        parentRuleSets.push(parent);
      }
      parent = parent.getParent();
    }
  }
  var builder = new SelectorElementBuilder(root);
  for (var i = parentRuleSets.length - 1; i >= 0; i--) {
    var selector = parentRuleSets[i].getSelectors().getChild(0);
    if (selector) {
      builder.processSelector(selector);
    }
  }
  builder.processSelector(node);
  return root;
}

// node_modules/vscode-css-languageservice/lib/esm/services/cssHover.js
var CSSHover = function() {
  function CSSHover2(clientCapabilities, cssDataManager) {
    this.clientCapabilities = clientCapabilities;
    this.cssDataManager = cssDataManager;
    this.selectorPrinting = new SelectorPrinting(cssDataManager);
  }
  CSSHover2.prototype.configure = function(settings) {
    this.defaultSettings = settings;
  };
  CSSHover2.prototype.doHover = function(document, position, stylesheet, settings) {
    if (settings === void 0) {
      settings = this.defaultSettings;
    }
    function getRange2(node2) {
      return Range.create(document.positionAt(node2.offset), document.positionAt(node2.end));
    }
    var offset = document.offsetAt(position);
    var nodepath = getNodePath(stylesheet, offset);
    var hover = null;
    for (var i = 0; i < nodepath.length; i++) {
      var node = nodepath[i];
      if (node instanceof Selector) {
        hover = {
          contents: this.selectorPrinting.selectorToMarkedString(node),
          range: getRange2(node)
        };
        break;
      }
      if (node instanceof SimpleSelector) {
        if (!startsWith(node.getText(), "@")) {
          hover = {
            contents: this.selectorPrinting.simpleSelectorToMarkedString(node),
            range: getRange2(node)
          };
        }
        break;
      }
      if (node instanceof Declaration) {
        var propertyName = node.getFullPropertyName();
        var entry = this.cssDataManager.getProperty(propertyName);
        if (entry) {
          var contents = getEntryDescription(entry, this.doesSupportMarkdown(), settings);
          if (contents) {
            hover = {
              contents,
              range: getRange2(node)
            };
          } else {
            hover = null;
          }
        }
        continue;
      }
      if (node instanceof UnknownAtRule) {
        var atRuleName = node.getText();
        var entry = this.cssDataManager.getAtDirective(atRuleName);
        if (entry) {
          var contents = getEntryDescription(entry, this.doesSupportMarkdown(), settings);
          if (contents) {
            hover = {
              contents,
              range: getRange2(node)
            };
          } else {
            hover = null;
          }
        }
        continue;
      }
      if (node instanceof Node && node.type === NodeType.PseudoSelector) {
        var selectorName = node.getText();
        var entry = selectorName.slice(0, 2) === "::" ? this.cssDataManager.getPseudoElement(selectorName) : this.cssDataManager.getPseudoClass(selectorName);
        if (entry) {
          var contents = getEntryDescription(entry, this.doesSupportMarkdown(), settings);
          if (contents) {
            hover = {
              contents,
              range: getRange2(node)
            };
          } else {
            hover = null;
          }
        }
        continue;
      }
    }
    if (hover) {
      hover.contents = this.convertContents(hover.contents);
    }
    return hover;
  };
  CSSHover2.prototype.convertContents = function(contents) {
    if (!this.doesSupportMarkdown()) {
      if (typeof contents === "string") {
        return contents;
      } else if ("kind" in contents) {
        return {
          kind: "plaintext",
          value: contents.value
        };
      } else if (Array.isArray(contents)) {
        return contents.map(function(c) {
          return typeof c === "string" ? c : c.value;
        });
      } else {
        return contents.value;
      }
    }
    return contents;
  };
  CSSHover2.prototype.doesSupportMarkdown = function() {
    if (!isDefined(this.supportsMarkdown)) {
      if (!isDefined(this.clientCapabilities)) {
        this.supportsMarkdown = true;
        return this.supportsMarkdown;
      }
      var hover = this.clientCapabilities.textDocument && this.clientCapabilities.textDocument.hover;
      this.supportsMarkdown = hover && hover.contentFormat && Array.isArray(hover.contentFormat) && hover.contentFormat.indexOf(MarkupKind.Markdown) !== -1;
    }
    return this.supportsMarkdown;
  };
  return CSSHover2;
}();

// node_modules/vscode-css-languageservice/lib/esm/services/cssNavigation.js
var __awaiter3 = function(thisArg, _arguments, P, generator) {
  function adopt(value) {
    return value instanceof P ? value : new P(function(resolve) {
      resolve(value);
    });
  }
  return new (P || (P = Promise))(function(resolve, reject) {
    function fulfilled(value) {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    }
    function rejected(value) {
      try {
        step(generator["throw"](value));
      } catch (e) {
        reject(e);
      }
    }
    function step(result) {
      result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected);
    }
    step((generator = generator.apply(thisArg, _arguments || [])).next());
  });
};
var __generator3 = function(thisArg, body) {
  var _ = { label: 0, sent: function() {
    if (t[0] & 1)
      throw t[1];
    return t[1];
  }, trys: [], ops: [] }, f2, y, t, g;
  return g = { next: verb(0), "throw": verb(1), "return": verb(2) }, typeof Symbol === "function" && (g[Symbol.iterator] = function() {
    return this;
  }), g;
  function verb(n) {
    return function(v) {
      return step([n, v]);
    };
  }
  function step(op) {
    if (f2)
      throw new TypeError("Generator is already executing.");
    while (_)
      try {
        if (f2 = 1, y && (t = op[0] & 2 ? y["return"] : op[0] ? y["throw"] || ((t = y["return"]) && t.call(y), 0) : y.next) && !(t = t.call(y, op[1])).done)
          return t;
        if (y = 0, t)
          op = [op[0] & 2, t.value];
        switch (op[0]) {
          case 0:
          case 1:
            t = op;
            break;
          case 4:
            _.label++;
            return { value: op[1], done: false };
          case 5:
            _.label++;
            y = op[1];
            op = [0];
            continue;
          case 7:
            op = _.ops.pop();
            _.trys.pop();
            continue;
          default:
            if (!(t = _.trys, t = t.length > 0 && t[t.length - 1]) && (op[0] === 6 || op[0] === 2)) {
              _ = 0;
              continue;
            }
            if (op[0] === 3 && (!t || op[1] > t[0] && op[1] < t[3])) {
              _.label = op[1];
              break;
            }
            if (op[0] === 6 && _.label < t[1]) {
              _.label = t[1];
              t = op;
              break;
            }
            if (t && _.label < t[2]) {
              _.label = t[2];
              _.ops.push(op);
              break;
            }
            if (t[2])
              _.ops.pop();
            _.trys.pop();
            continue;
        }
        op = body.call(thisArg, _);
      } catch (e) {
        op = [6, e];
        y = 0;
      } finally {
        f2 = t = 0;
      }
    if (op[0] & 5)
      throw op[1];
    return { value: op[0] ? op[1] : void 0, done: true };
  }
};
var localize6 = loadMessageBundle();
var startsWithSchemeRegex = /^\w+:\/\//;
var startsWithData = /^data:/;
var CSSNavigation = function() {
  function CSSNavigation2(fileSystemProvider, resolveModuleReferences) {
    this.fileSystemProvider = fileSystemProvider;
    this.resolveModuleReferences = resolveModuleReferences;
  }
  CSSNavigation2.prototype.findDefinition = function(document, position, stylesheet) {
    var symbols = new Symbols(stylesheet);
    var offset = document.offsetAt(position);
    var node = getNodeAtOffset(stylesheet, offset);
    if (!node) {
      return null;
    }
    var symbol = symbols.findSymbolFromNode(node);
    if (!symbol) {
      return null;
    }
    return {
      uri: document.uri,
      range: getRange(symbol.node, document)
    };
  };
  CSSNavigation2.prototype.findReferences = function(document, position, stylesheet) {
    var highlights = this.findDocumentHighlights(document, position, stylesheet);
    return highlights.map(function(h) {
      return {
        uri: document.uri,
        range: h.range
      };
    });
  };
  CSSNavigation2.prototype.findDocumentHighlights = function(document, position, stylesheet) {
    var result = [];
    var offset = document.offsetAt(position);
    var node = getNodeAtOffset(stylesheet, offset);
    if (!node || node.type === NodeType.Stylesheet || node.type === NodeType.Declarations) {
      return result;
    }
    if (node.type === NodeType.Identifier && node.parent && node.parent.type === NodeType.ClassSelector) {
      node = node.parent;
    }
    var symbols = new Symbols(stylesheet);
    var symbol = symbols.findSymbolFromNode(node);
    var name = node.getText();
    stylesheet.accept(function(candidate) {
      if (symbol) {
        if (symbols.matchesSymbol(candidate, symbol)) {
          result.push({
            kind: getHighlightKind(candidate),
            range: getRange(candidate, document)
          });
          return false;
        }
      } else if (node && node.type === candidate.type && candidate.matches(name)) {
        result.push({
          kind: getHighlightKind(candidate),
          range: getRange(candidate, document)
        });
      }
      return true;
    });
    return result;
  };
  CSSNavigation2.prototype.isRawStringDocumentLinkNode = function(node) {
    return node.type === NodeType.Import;
  };
  CSSNavigation2.prototype.findDocumentLinks = function(document, stylesheet, documentContext) {
    var linkData = this.findUnresolvedLinks(document, stylesheet);
    var resolvedLinks = [];
    for (var _i = 0, linkData_1 = linkData; _i < linkData_1.length; _i++) {
      var data = linkData_1[_i];
      var link = data.link;
      var target = link.target;
      if (!target || startsWithData.test(target)) {
      } else if (startsWithSchemeRegex.test(target)) {
        resolvedLinks.push(link);
      } else {
        var resolved = documentContext.resolveReference(target, document.uri);
        if (resolved) {
          link.target = resolved;
        }
        resolvedLinks.push(link);
      }
    }
    return resolvedLinks;
  };
  CSSNavigation2.prototype.findDocumentLinks2 = function(document, stylesheet, documentContext) {
    return __awaiter3(this, void 0, void 0, function() {
      var linkData, resolvedLinks, _i, linkData_2, data, link, target, resolvedTarget;
      return __generator3(this, function(_a2) {
        switch (_a2.label) {
          case 0:
            linkData = this.findUnresolvedLinks(document, stylesheet);
            resolvedLinks = [];
            _i = 0, linkData_2 = linkData;
            _a2.label = 1;
          case 1:
            if (!(_i < linkData_2.length))
              return [3, 6];
            data = linkData_2[_i];
            link = data.link;
            target = link.target;
            if (!(!target || startsWithData.test(target)))
              return [3, 2];
            return [3, 5];
          case 2:
            if (!startsWithSchemeRegex.test(target))
              return [3, 3];
            resolvedLinks.push(link);
            return [3, 5];
          case 3:
            return [4, this.resolveRelativeReference(target, document.uri, documentContext, data.isRawLink)];
          case 4:
            resolvedTarget = _a2.sent();
            if (resolvedTarget !== void 0) {
              link.target = resolvedTarget;
              resolvedLinks.push(link);
            }
            _a2.label = 5;
          case 5:
            _i++;
            return [3, 1];
          case 6:
            return [2, resolvedLinks];
        }
      });
    });
  };
  CSSNavigation2.prototype.findUnresolvedLinks = function(document, stylesheet) {
    var _this = this;
    var result = [];
    var collect = function(uriStringNode) {
      var rawUri = uriStringNode.getText();
      var range = getRange(uriStringNode, document);
      if (range.start.line === range.end.line && range.start.character === range.end.character) {
        return;
      }
      if (startsWith(rawUri, "'") || startsWith(rawUri, '"')) {
        rawUri = rawUri.slice(1, -1);
      }
      var isRawLink = uriStringNode.parent ? _this.isRawStringDocumentLinkNode(uriStringNode.parent) : false;
      result.push({ link: { target: rawUri, range }, isRawLink });
    };
    stylesheet.accept(function(candidate) {
      if (candidate.type === NodeType.URILiteral) {
        var first = candidate.getChild(0);
        if (first) {
          collect(first);
        }
        return false;
      }
      if (candidate.parent && _this.isRawStringDocumentLinkNode(candidate.parent)) {
        var rawText = candidate.getText();
        if (startsWith(rawText, "'") || startsWith(rawText, '"')) {
          collect(candidate);
        }
        return false;
      }
      return true;
    });
    return result;
  };
  CSSNavigation2.prototype.findDocumentSymbols = function(document, stylesheet) {
    var result = [];
    stylesheet.accept(function(node) {
      var entry = {
        name: null,
        kind: SymbolKind.Class,
        location: null
      };
      var locationNode = node;
      if (node instanceof Selector) {
        entry.name = node.getText();
        locationNode = node.findAParent(NodeType.Ruleset, NodeType.ExtendsReference);
        if (locationNode) {
          entry.location = Location.create(document.uri, getRange(locationNode, document));
          result.push(entry);
        }
        return false;
      } else if (node instanceof VariableDeclaration) {
        entry.name = node.getName();
        entry.kind = SymbolKind.Variable;
      } else if (node instanceof MixinDeclaration) {
        entry.name = node.getName();
        entry.kind = SymbolKind.Method;
      } else if (node instanceof FunctionDeclaration) {
        entry.name = node.getName();
        entry.kind = SymbolKind.Function;
      } else if (node instanceof Keyframe) {
        entry.name = localize6("literal.keyframes", "@keyframes {0}", node.getName());
      } else if (node instanceof FontFace) {
        entry.name = localize6("literal.fontface", "@font-face");
      } else if (node instanceof Media) {
        var mediaList = node.getChild(0);
        if (mediaList instanceof Medialist) {
          entry.name = "@media " + mediaList.getText();
          entry.kind = SymbolKind.Module;
        }
      }
      if (entry.name) {
        entry.location = Location.create(document.uri, getRange(locationNode, document));
        result.push(entry);
      }
      return true;
    });
    return result;
  };
  CSSNavigation2.prototype.findDocumentColors = function(document, stylesheet) {
    var result = [];
    stylesheet.accept(function(node) {
      var colorInfo = getColorInformation(node, document);
      if (colorInfo) {
        result.push(colorInfo);
      }
      return true;
    });
    return result;
  };
  CSSNavigation2.prototype.getColorPresentations = function(document, stylesheet, color, range) {
    var result = [];
    var red256 = Math.round(color.red * 255), green256 = Math.round(color.green * 255), blue256 = Math.round(color.blue * 255);
    var label;
    if (color.alpha === 1) {
      label = "rgb(".concat(red256, ", ").concat(green256, ", ").concat(blue256, ")");
    } else {
      label = "rgba(".concat(red256, ", ").concat(green256, ", ").concat(blue256, ", ").concat(color.alpha, ")");
    }
    result.push({ label, textEdit: TextEdit.replace(range, label) });
    if (color.alpha === 1) {
      label = "#".concat(toTwoDigitHex(red256)).concat(toTwoDigitHex(green256)).concat(toTwoDigitHex(blue256));
    } else {
      label = "#".concat(toTwoDigitHex(red256)).concat(toTwoDigitHex(green256)).concat(toTwoDigitHex(blue256)).concat(toTwoDigitHex(Math.round(color.alpha * 255)));
    }
    result.push({ label, textEdit: TextEdit.replace(range, label) });
    var hsl = hslFromColor(color);
    if (hsl.a === 1) {
      label = "hsl(".concat(hsl.h, ", ").concat(Math.round(hsl.s * 100), "%, ").concat(Math.round(hsl.l * 100), "%)");
    } else {
      label = "hsla(".concat(hsl.h, ", ").concat(Math.round(hsl.s * 100), "%, ").concat(Math.round(hsl.l * 100), "%, ").concat(hsl.a, ")");
    }
    result.push({ label, textEdit: TextEdit.replace(range, label) });
    var hwb = hwbFromColor(color);
    if (hwb.a === 1) {
      label = "hwb(".concat(hwb.h, " ").concat(Math.round(hwb.w * 100), "% ").concat(Math.round(hwb.b * 100), "%)");
    } else {
      label = "hwb(".concat(hwb.h, " ").concat(Math.round(hwb.w * 100), "% ").concat(Math.round(hwb.b * 100), "% / ").concat(hwb.a, ")");
    }
    result.push({ label, textEdit: TextEdit.replace(range, label) });
    return result;
  };
  CSSNavigation2.prototype.doRename = function(document, position, newName, stylesheet) {
    var _a2;
    var highlights = this.findDocumentHighlights(document, position, stylesheet);
    var edits = highlights.map(function(h) {
      return TextEdit.replace(h.range, newName);
    });
    return {
      changes: (_a2 = {}, _a2[document.uri] = edits, _a2)
    };
  };
  CSSNavigation2.prototype.resolveModuleReference = function(ref, documentUri, documentContext) {
    return __awaiter3(this, void 0, void 0, function() {
      var moduleName, rootFolderUri, documentFolderUri, modulePath, pathWithinModule;
      return __generator3(this, function(_a2) {
        switch (_a2.label) {
          case 0:
            if (!startsWith(documentUri, "file://"))
              return [3, 2];
            moduleName = getModuleNameFromPath(ref);
            rootFolderUri = documentContext.resolveReference("/", documentUri);
            documentFolderUri = dirname(documentUri);
            return [4, this.resolvePathToModule(moduleName, documentFolderUri, rootFolderUri)];
          case 1:
            modulePath = _a2.sent();
            if (modulePath) {
              pathWithinModule = ref.substring(moduleName.length + 1);
              return [2, joinPath(modulePath, pathWithinModule)];
            }
            _a2.label = 2;
          case 2:
            return [2, void 0];
        }
      });
    });
  };
  CSSNavigation2.prototype.resolveRelativeReference = function(ref, documentUri, documentContext, isRawLink) {
    return __awaiter3(this, void 0, void 0, function() {
      var relativeReference, _a2;
      return __generator3(this, function(_b) {
        switch (_b.label) {
          case 0:
            relativeReference = documentContext.resolveReference(ref, documentUri);
            if (!(ref[0] === "~" && ref[1] !== "/" && this.fileSystemProvider))
              return [3, 2];
            ref = ref.substring(1);
            return [4, this.resolveModuleReference(ref, documentUri, documentContext)];
          case 1:
            return [2, _b.sent() || relativeReference];
          case 2:
            if (!this.resolveModuleReferences)
              return [3, 7];
            _a2 = relativeReference;
            if (!_a2)
              return [3, 4];
            return [4, this.fileExists(relativeReference)];
          case 3:
            _a2 = _b.sent();
            _b.label = 4;
          case 4:
            if (!_a2)
              return [3, 5];
            return [2, relativeReference];
          case 5:
            return [4, this.resolveModuleReference(ref, documentUri, documentContext)];
          case 6:
            return [2, _b.sent() || relativeReference];
          case 7:
            return [2, relativeReference];
        }
      });
    });
  };
  CSSNavigation2.prototype.resolvePathToModule = function(_moduleName, documentFolderUri, rootFolderUri) {
    return __awaiter3(this, void 0, void 0, function() {
      var packPath;
      return __generator3(this, function(_a2) {
        switch (_a2.label) {
          case 0:
            packPath = joinPath(documentFolderUri, "node_modules", _moduleName, "package.json");
            return [4, this.fileExists(packPath)];
          case 1:
            if (_a2.sent()) {
              return [2, dirname(packPath)];
            } else if (rootFolderUri && documentFolderUri.startsWith(rootFolderUri) && documentFolderUri.length !== rootFolderUri.length) {
              return [2, this.resolvePathToModule(_moduleName, dirname(documentFolderUri), rootFolderUri)];
            }
            return [2, void 0];
        }
      });
    });
  };
  CSSNavigation2.prototype.fileExists = function(uri) {
    return __awaiter3(this, void 0, void 0, function() {
      var stat, err_1;
      return __generator3(this, function(_a2) {
        switch (_a2.label) {
          case 0:
            if (!this.fileSystemProvider) {
              return [2, false];
            }
            _a2.label = 1;
          case 1:
            _a2.trys.push([1, 3, , 4]);
            return [4, this.fileSystemProvider.stat(uri)];
          case 2:
            stat = _a2.sent();
            if (stat.type === FileType.Unknown && stat.size === -1) {
              return [2, false];
            }
            return [2, true];
          case 3:
            err_1 = _a2.sent();
            return [2, false];
          case 4:
            return [2];
        }
      });
    });
  };
  return CSSNavigation2;
}();
function getColorInformation(node, document) {
  var color = getColorValue(node);
  if (color) {
    var range = getRange(node, document);
    return { color, range };
  }
  return null;
}
function getRange(node, document) {
  return Range.create(document.positionAt(node.offset), document.positionAt(node.end));
}
function getHighlightKind(node) {
  if (node.type === NodeType.Selector) {
    return DocumentHighlightKind.Write;
  }
  if (node instanceof Identifier) {
    if (node.parent && node.parent instanceof Property) {
      if (node.isCustomProperty) {
        return DocumentHighlightKind.Write;
      }
    }
  }
  if (node.parent) {
    switch (node.parent.type) {
      case NodeType.FunctionDeclaration:
      case NodeType.MixinDeclaration:
      case NodeType.Keyframe:
      case NodeType.VariableDeclaration:
      case NodeType.FunctionParameter:
        return DocumentHighlightKind.Write;
    }
  }
  return DocumentHighlightKind.Read;
}
function toTwoDigitHex(n) {
  var r = n.toString(16);
  return r.length !== 2 ? "0" + r : r;
}
function getModuleNameFromPath(path) {
  if (path[0] === "@") {
    return path.substring(0, path.indexOf("/", path.indexOf("/") + 1));
  }
  return path.substring(0, path.indexOf("/"));
}

// node_modules/vscode-css-languageservice/lib/esm/services/lintRules.js
var localize7 = loadMessageBundle();
var Warning = Level.Warning;
var Error2 = Level.Error;
var Ignore = Level.Ignore;
var Rule = function() {
  function Rule2(id, message, defaultValue) {
    this.id = id;
    this.message = message;
    this.defaultValue = defaultValue;
  }
  return Rule2;
}();
var Setting = function() {
  function Setting2(id, message, defaultValue) {
    this.id = id;
    this.message = message;
    this.defaultValue = defaultValue;
  }
  return Setting2;
}();
var Rules = {
  AllVendorPrefixes: new Rule("compatibleVendorPrefixes", localize7("rule.vendorprefixes.all", "When using a vendor-specific prefix make sure to also include all other vendor-specific properties"), Ignore),
  IncludeStandardPropertyWhenUsingVendorPrefix: new Rule("vendorPrefix", localize7("rule.standardvendorprefix.all", "When using a vendor-specific prefix also include the standard property"), Warning),
  DuplicateDeclarations: new Rule("duplicateProperties", localize7("rule.duplicateDeclarations", "Do not use duplicate style definitions"), Ignore),
  EmptyRuleSet: new Rule("emptyRules", localize7("rule.emptyRuleSets", "Do not use empty rulesets"), Warning),
  ImportStatemement: new Rule("importStatement", localize7("rule.importDirective", "Import statements do not load in parallel"), Ignore),
  BewareOfBoxModelSize: new Rule("boxModel", localize7("rule.bewareOfBoxModelSize", "Do not use width or height when using padding or border"), Ignore),
  UniversalSelector: new Rule("universalSelector", localize7("rule.universalSelector", "The universal selector (*) is known to be slow"), Ignore),
  ZeroWithUnit: new Rule("zeroUnits", localize7("rule.zeroWidthUnit", "No unit for zero needed"), Ignore),
  RequiredPropertiesForFontFace: new Rule("fontFaceProperties", localize7("rule.fontFaceProperties", "@font-face rule must define 'src' and 'font-family' properties"), Warning),
  HexColorLength: new Rule("hexColorLength", localize7("rule.hexColor", "Hex colors must consist of three, four, six or eight hex numbers"), Error2),
  ArgsInColorFunction: new Rule("argumentsInColorFunction", localize7("rule.colorFunction", "Invalid number of parameters"), Error2),
  UnknownProperty: new Rule("unknownProperties", localize7("rule.unknownProperty", "Unknown property."), Warning),
  UnknownAtRules: new Rule("unknownAtRules", localize7("rule.unknownAtRules", "Unknown at-rule."), Warning),
  IEStarHack: new Rule("ieHack", localize7("rule.ieHack", "IE hacks are only necessary when supporting IE7 and older"), Ignore),
  UnknownVendorSpecificProperty: new Rule("unknownVendorSpecificProperties", localize7("rule.unknownVendorSpecificProperty", "Unknown vendor specific property."), Ignore),
  PropertyIgnoredDueToDisplay: new Rule("propertyIgnoredDueToDisplay", localize7("rule.propertyIgnoredDueToDisplay", "Property is ignored due to the display."), Warning),
  AvoidImportant: new Rule("important", localize7("rule.avoidImportant", "Avoid using !important. It is an indication that the specificity of the entire CSS has gotten out of control and needs to be refactored."), Ignore),
  AvoidFloat: new Rule("float", localize7("rule.avoidFloat", "Avoid using 'float'. Floats lead to fragile CSS that is easy to break if one aspect of the layout changes."), Ignore),
  AvoidIdSelector: new Rule("idSelector", localize7("rule.avoidIdSelector", "Selectors should not contain IDs because these rules are too tightly coupled with the HTML."), Ignore)
};
var Settings = {
  ValidProperties: new Setting("validProperties", localize7("rule.validProperties", "A list of properties that are not validated against the `unknownProperties` rule."), [])
};
var LintConfigurationSettings = function() {
  function LintConfigurationSettings2(conf) {
    if (conf === void 0) {
      conf = {};
    }
    this.conf = conf;
  }
  LintConfigurationSettings2.prototype.getRule = function(rule) {
    if (this.conf.hasOwnProperty(rule.id)) {
      var level = toLevel(this.conf[rule.id]);
      if (level) {
        return level;
      }
    }
    return rule.defaultValue;
  };
  LintConfigurationSettings2.prototype.getSetting = function(setting) {
    return this.conf[setting.id];
  };
  return LintConfigurationSettings2;
}();
function toLevel(level) {
  switch (level) {
    case "ignore":
      return Level.Ignore;
    case "warning":
      return Level.Warning;
    case "error":
      return Level.Error;
  }
  return null;
}

// node_modules/vscode-css-languageservice/lib/esm/services/cssCodeActions.js
var localize8 = loadMessageBundle();
var CSSCodeActions = function() {
  function CSSCodeActions2(cssDataManager) {
    this.cssDataManager = cssDataManager;
  }
  CSSCodeActions2.prototype.doCodeActions = function(document, range, context, stylesheet) {
    return this.doCodeActions2(document, range, context, stylesheet).map(function(ca) {
      var textDocumentEdit = ca.edit && ca.edit.documentChanges && ca.edit.documentChanges[0];
      return Command.create(ca.title, "_css.applyCodeAction", document.uri, document.version, textDocumentEdit && textDocumentEdit.edits);
    });
  };
  CSSCodeActions2.prototype.doCodeActions2 = function(document, range, context, stylesheet) {
    var result = [];
    if (context.diagnostics) {
      for (var _i = 0, _a2 = context.diagnostics; _i < _a2.length; _i++) {
        var diagnostic = _a2[_i];
        this.appendFixesForMarker(document, stylesheet, diagnostic, result);
      }
    }
    return result;
  };
  CSSCodeActions2.prototype.getFixesForUnknownProperty = function(document, property, marker, result) {
    var propertyName = property.getName();
    var candidates = [];
    this.cssDataManager.getProperties().forEach(function(p) {
      var score = difference(propertyName, p.name);
      if (score >= propertyName.length / 2) {
        candidates.push({ property: p.name, score });
      }
    });
    candidates.sort(function(a2, b) {
      return b.score - a2.score || a2.property.localeCompare(b.property);
    });
    var maxActions = 3;
    for (var _i = 0, candidates_1 = candidates; _i < candidates_1.length; _i++) {
      var candidate = candidates_1[_i];
      var propertyName_1 = candidate.property;
      var title = localize8("css.codeaction.rename", "Rename to '{0}'", propertyName_1);
      var edit = TextEdit.replace(marker.range, propertyName_1);
      var documentIdentifier = VersionedTextDocumentIdentifier.create(document.uri, document.version);
      var workspaceEdit = { documentChanges: [TextDocumentEdit.create(documentIdentifier, [edit])] };
      var codeAction = CodeAction.create(title, workspaceEdit, CodeActionKind.QuickFix);
      codeAction.diagnostics = [marker];
      result.push(codeAction);
      if (--maxActions <= 0) {
        return;
      }
    }
  };
  CSSCodeActions2.prototype.appendFixesForMarker = function(document, stylesheet, marker, result) {
    if (marker.code !== Rules.UnknownProperty.id) {
      return;
    }
    var offset = document.offsetAt(marker.range.start);
    var end = document.offsetAt(marker.range.end);
    var nodepath = getNodePath(stylesheet, offset);
    for (var i = nodepath.length - 1; i >= 0; i--) {
      var node = nodepath[i];
      if (node instanceof Declaration) {
        var property = node.getProperty();
        if (property && property.offset === offset && property.end === end) {
          this.getFixesForUnknownProperty(document, property, marker, result);
          return;
        }
      }
    }
  };
  return CSSCodeActions2;
}();

// node_modules/vscode-css-languageservice/lib/esm/services/lintUtil.js
var Element2 = function() {
  function Element3(decl) {
    this.fullPropertyName = decl.getFullPropertyName().toLowerCase();
    this.node = decl;
  }
  return Element3;
}();
function setSide(model, side, value, property) {
  var state = model[side];
  state.value = value;
  if (value) {
    if (!includes(state.properties, property)) {
      state.properties.push(property);
    }
  }
}
function setAllSides(model, value, property) {
  setSide(model, "top", value, property);
  setSide(model, "right", value, property);
  setSide(model, "bottom", value, property);
  setSide(model, "left", value, property);
}
function updateModelWithValue(model, side, value, property) {
  if (side === "top" || side === "right" || side === "bottom" || side === "left") {
    setSide(model, side, value, property);
  } else {
    setAllSides(model, value, property);
  }
}
function updateModelWithList(model, values2, property) {
  switch (values2.length) {
    case 1:
      updateModelWithValue(model, void 0, values2[0], property);
      break;
    case 2:
      updateModelWithValue(model, "top", values2[0], property);
      updateModelWithValue(model, "bottom", values2[0], property);
      updateModelWithValue(model, "right", values2[1], property);
      updateModelWithValue(model, "left", values2[1], property);
      break;
    case 3:
      updateModelWithValue(model, "top", values2[0], property);
      updateModelWithValue(model, "right", values2[1], property);
      updateModelWithValue(model, "left", values2[1], property);
      updateModelWithValue(model, "bottom", values2[2], property);
      break;
    case 4:
      updateModelWithValue(model, "top", values2[0], property);
      updateModelWithValue(model, "right", values2[1], property);
      updateModelWithValue(model, "bottom", values2[2], property);
      updateModelWithValue(model, "left", values2[3], property);
      break;
  }
}
function matches(value, candidates) {
  for (var _i = 0, candidates_1 = candidates; _i < candidates_1.length; _i++) {
    var candidate = candidates_1[_i];
    if (value.matches(candidate)) {
      return true;
    }
  }
  return false;
}
function checkLineWidth(value, allowsKeywords) {
  if (allowsKeywords === void 0) {
    allowsKeywords = true;
  }
  if (allowsKeywords && matches(value, ["initial", "unset"])) {
    return false;
  }
  return parseFloat(value.getText()) !== 0;
}
function checkLineWidthList(nodes, allowsKeywords) {
  if (allowsKeywords === void 0) {
    allowsKeywords = true;
  }
  return nodes.map(function(node) {
    return checkLineWidth(node, allowsKeywords);
  });
}
function checkLineStyle(valueNode, allowsKeywords) {
  if (allowsKeywords === void 0) {
    allowsKeywords = true;
  }
  if (matches(valueNode, ["none", "hidden"])) {
    return false;
  }
  if (allowsKeywords && matches(valueNode, ["initial", "unset"])) {
    return false;
  }
  return true;
}
function checkLineStyleList(nodes, allowsKeywords) {
  if (allowsKeywords === void 0) {
    allowsKeywords = true;
  }
  return nodes.map(function(node) {
    return checkLineStyle(node, allowsKeywords);
  });
}
function checkBorderShorthand(node) {
  var children = node.getChildren();
  if (children.length === 1) {
    var value = children[0];
    return checkLineWidth(value) && checkLineStyle(value);
  }
  for (var _i = 0, children_1 = children; _i < children_1.length; _i++) {
    var child = children_1[_i];
    var value = child;
    if (!checkLineWidth(value, false) || !checkLineStyle(value, false)) {
      return false;
    }
  }
  return true;
}
function calculateBoxModel(propertyTable) {
  var model = {
    top: { value: false, properties: [] },
    right: { value: false, properties: [] },
    bottom: { value: false, properties: [] },
    left: { value: false, properties: [] }
  };
  for (var _i = 0, propertyTable_1 = propertyTable; _i < propertyTable_1.length; _i++) {
    var property = propertyTable_1[_i];
    var value = property.node.value;
    if (typeof value === "undefined") {
      continue;
    }
    switch (property.fullPropertyName) {
      case "box-sizing":
        return {
          top: { value: false, properties: [] },
          right: { value: false, properties: [] },
          bottom: { value: false, properties: [] },
          left: { value: false, properties: [] }
        };
      case "width":
        model.width = property;
        break;
      case "height":
        model.height = property;
        break;
      default:
        var segments = property.fullPropertyName.split("-");
        switch (segments[0]) {
          case "border":
            switch (segments[1]) {
              case void 0:
              case "top":
              case "right":
              case "bottom":
              case "left":
                switch (segments[2]) {
                  case void 0:
                    updateModelWithValue(model, segments[1], checkBorderShorthand(value), property);
                    break;
                  case "width":
                    updateModelWithValue(model, segments[1], checkLineWidth(value, false), property);
                    break;
                  case "style":
                    updateModelWithValue(model, segments[1], checkLineStyle(value, true), property);
                    break;
                }
                break;
              case "width":
                updateModelWithList(model, checkLineWidthList(value.getChildren(), false), property);
                break;
              case "style":
                updateModelWithList(model, checkLineStyleList(value.getChildren(), true), property);
                break;
            }
            break;
          case "padding":
            if (segments.length === 1) {
              updateModelWithList(model, checkLineWidthList(value.getChildren(), true), property);
            } else {
              updateModelWithValue(model, segments[1], checkLineWidth(value, true), property);
            }
            break;
        }
        break;
    }
  }
  return model;
}

// node_modules/vscode-css-languageservice/lib/esm/services/lint.js
var localize9 = loadMessageBundle();
var NodesByRootMap = function() {
  function NodesByRootMap2() {
    this.data = {};
  }
  NodesByRootMap2.prototype.add = function(root, name, node) {
    var entry = this.data[root];
    if (!entry) {
      entry = { nodes: [], names: [] };
      this.data[root] = entry;
    }
    entry.names.push(name);
    if (node) {
      entry.nodes.push(node);
    }
  };
  return NodesByRootMap2;
}();
var LintVisitor = function() {
  function LintVisitor2(document, settings, cssDataManager) {
    var _this = this;
    this.cssDataManager = cssDataManager;
    this.warnings = [];
    this.settings = settings;
    this.documentText = document.getText();
    this.keyframes = new NodesByRootMap();
    this.validProperties = {};
    var properties = settings.getSetting(Settings.ValidProperties);
    if (Array.isArray(properties)) {
      properties.forEach(function(p) {
        if (typeof p === "string") {
          var name = p.trim().toLowerCase();
          if (name.length) {
            _this.validProperties[name] = true;
          }
        }
      });
    }
  }
  LintVisitor2.entries = function(node, document, settings, cssDataManager, entryFilter) {
    var visitor = new LintVisitor2(document, settings, cssDataManager);
    node.acceptVisitor(visitor);
    visitor.completeValidations();
    return visitor.getEntries(entryFilter);
  };
  LintVisitor2.prototype.isValidPropertyDeclaration = function(element) {
    var propertyName = element.fullPropertyName;
    return this.validProperties[propertyName];
  };
  LintVisitor2.prototype.fetch = function(input, s) {
    var elements = [];
    for (var _i = 0, input_1 = input; _i < input_1.length; _i++) {
      var curr = input_1[_i];
      if (curr.fullPropertyName === s) {
        elements.push(curr);
      }
    }
    return elements;
  };
  LintVisitor2.prototype.fetchWithValue = function(input, s, v) {
    var elements = [];
    for (var _i = 0, input_2 = input; _i < input_2.length; _i++) {
      var inputElement = input_2[_i];
      if (inputElement.fullPropertyName === s) {
        var expression = inputElement.node.getValue();
        if (expression && this.findValueInExpression(expression, v)) {
          elements.push(inputElement);
        }
      }
    }
    return elements;
  };
  LintVisitor2.prototype.findValueInExpression = function(expression, v) {
    var found = false;
    expression.accept(function(node) {
      if (node.type === NodeType.Identifier && node.matches(v)) {
        found = true;
      }
      return !found;
    });
    return found;
  };
  LintVisitor2.prototype.getEntries = function(filter) {
    if (filter === void 0) {
      filter = Level.Warning | Level.Error;
    }
    return this.warnings.filter(function(entry) {
      return (entry.getLevel() & filter) !== 0;
    });
  };
  LintVisitor2.prototype.addEntry = function(node, rule, details) {
    var entry = new Marker(node, rule, this.settings.getRule(rule), details);
    this.warnings.push(entry);
  };
  LintVisitor2.prototype.getMissingNames = function(expected, actual) {
    var expectedClone = expected.slice(0);
    for (var i = 0; i < actual.length; i++) {
      var k = expectedClone.indexOf(actual[i]);
      if (k !== -1) {
        expectedClone[k] = null;
      }
    }
    var result = null;
    for (var i = 0; i < expectedClone.length; i++) {
      var curr = expectedClone[i];
      if (curr) {
        if (result === null) {
          result = localize9("namelist.single", "'{0}'", curr);
        } else {
          result = localize9("namelist.concatenated", "{0}, '{1}'", result, curr);
        }
      }
    }
    return result;
  };
  LintVisitor2.prototype.visitNode = function(node) {
    switch (node.type) {
      case NodeType.UnknownAtRule:
        return this.visitUnknownAtRule(node);
      case NodeType.Keyframe:
        return this.visitKeyframe(node);
      case NodeType.FontFace:
        return this.visitFontFace(node);
      case NodeType.Ruleset:
        return this.visitRuleSet(node);
      case NodeType.SimpleSelector:
        return this.visitSimpleSelector(node);
      case NodeType.Function:
        return this.visitFunction(node);
      case NodeType.NumericValue:
        return this.visitNumericValue(node);
      case NodeType.Import:
        return this.visitImport(node);
      case NodeType.HexColorValue:
        return this.visitHexColorValue(node);
      case NodeType.Prio:
        return this.visitPrio(node);
      case NodeType.IdentifierSelector:
        return this.visitIdentifierSelector(node);
    }
    return true;
  };
  LintVisitor2.prototype.completeValidations = function() {
    this.validateKeyframes();
  };
  LintVisitor2.prototype.visitUnknownAtRule = function(node) {
    var atRuleName = node.getChild(0);
    if (!atRuleName) {
      return false;
    }
    var atDirective = this.cssDataManager.getAtDirective(atRuleName.getText());
    if (atDirective) {
      return false;
    }
    this.addEntry(atRuleName, Rules.UnknownAtRules, "Unknown at rule ".concat(atRuleName.getText()));
    return true;
  };
  LintVisitor2.prototype.visitKeyframe = function(node) {
    var keyword = node.getKeyword();
    if (!keyword) {
      return false;
    }
    var text = keyword.getText();
    this.keyframes.add(node.getName(), text, text !== "@keyframes" ? keyword : null);
    return true;
  };
  LintVisitor2.prototype.validateKeyframes = function() {
    var expected = ["@-webkit-keyframes", "@-moz-keyframes", "@-o-keyframes"];
    for (var name in this.keyframes.data) {
      var actual = this.keyframes.data[name].names;
      var needsStandard = actual.indexOf("@keyframes") === -1;
      if (!needsStandard && actual.length === 1) {
        continue;
      }
      var missingVendorSpecific = this.getMissingNames(expected, actual);
      if (missingVendorSpecific || needsStandard) {
        for (var _i = 0, _a2 = this.keyframes.data[name].nodes; _i < _a2.length; _i++) {
          var node = _a2[_i];
          if (needsStandard) {
            var message = localize9("keyframes.standardrule.missing", "Always define standard rule '@keyframes' when defining keyframes.");
            this.addEntry(node, Rules.IncludeStandardPropertyWhenUsingVendorPrefix, message);
          }
          if (missingVendorSpecific) {
            var message = localize9("keyframes.vendorspecific.missing", "Always include all vendor specific rules: Missing: {0}", missingVendorSpecific);
            this.addEntry(node, Rules.AllVendorPrefixes, message);
          }
        }
      }
    }
    return true;
  };
  LintVisitor2.prototype.visitSimpleSelector = function(node) {
    var firstChar = this.documentText.charAt(node.offset);
    if (node.length === 1 && firstChar === "*") {
      this.addEntry(node, Rules.UniversalSelector);
    }
    return true;
  };
  LintVisitor2.prototype.visitIdentifierSelector = function(node) {
    this.addEntry(node, Rules.AvoidIdSelector);
    return true;
  };
  LintVisitor2.prototype.visitImport = function(node) {
    this.addEntry(node, Rules.ImportStatemement);
    return true;
  };
  LintVisitor2.prototype.visitRuleSet = function(node) {
    var declarations = node.getDeclarations();
    if (!declarations) {
      return false;
    }
    if (!declarations.hasChildren()) {
      this.addEntry(node.getSelectors(), Rules.EmptyRuleSet);
    }
    var propertyTable = [];
    for (var _i = 0, _a2 = declarations.getChildren(); _i < _a2.length; _i++) {
      var element = _a2[_i];
      if (element instanceof Declaration) {
        propertyTable.push(new Element2(element));
      }
    }
    var boxModel = calculateBoxModel(propertyTable);
    if (boxModel.width) {
      var properties = [];
      if (boxModel.right.value) {
        properties = union(properties, boxModel.right.properties);
      }
      if (boxModel.left.value) {
        properties = union(properties, boxModel.left.properties);
      }
      if (properties.length !== 0) {
        for (var _b = 0, properties_1 = properties; _b < properties_1.length; _b++) {
          var item = properties_1[_b];
          this.addEntry(item.node, Rules.BewareOfBoxModelSize);
        }
        this.addEntry(boxModel.width.node, Rules.BewareOfBoxModelSize);
      }
    }
    if (boxModel.height) {
      var properties = [];
      if (boxModel.top.value) {
        properties = union(properties, boxModel.top.properties);
      }
      if (boxModel.bottom.value) {
        properties = union(properties, boxModel.bottom.properties);
      }
      if (properties.length !== 0) {
        for (var _c = 0, properties_2 = properties; _c < properties_2.length; _c++) {
          var item = properties_2[_c];
          this.addEntry(item.node, Rules.BewareOfBoxModelSize);
        }
        this.addEntry(boxModel.height.node, Rules.BewareOfBoxModelSize);
      }
    }
    var displayElems = this.fetchWithValue(propertyTable, "display", "inline-block");
    if (displayElems.length > 0) {
      var elem = this.fetch(propertyTable, "float");
      for (var index = 0; index < elem.length; index++) {
        var node_1 = elem[index].node;
        var value = node_1.getValue();
        if (value && !value.matches("none")) {
          this.addEntry(node_1, Rules.PropertyIgnoredDueToDisplay, localize9("rule.propertyIgnoredDueToDisplayInlineBlock", "inline-block is ignored due to the float. If 'float' has a value other than 'none', the box is floated and 'display' is treated as 'block'"));
        }
      }
    }
    displayElems = this.fetchWithValue(propertyTable, "display", "block");
    if (displayElems.length > 0) {
      var elem = this.fetch(propertyTable, "vertical-align");
      for (var index = 0; index < elem.length; index++) {
        this.addEntry(elem[index].node, Rules.PropertyIgnoredDueToDisplay, localize9("rule.propertyIgnoredDueToDisplayBlock", "Property is ignored due to the display. With 'display: block', vertical-align should not be used."));
      }
    }
    var elements = this.fetch(propertyTable, "float");
    for (var index = 0; index < elements.length; index++) {
      var element = elements[index];
      if (!this.isValidPropertyDeclaration(element)) {
        this.addEntry(element.node, Rules.AvoidFloat);
      }
    }
    for (var i = 0; i < propertyTable.length; i++) {
      var element = propertyTable[i];
      if (element.fullPropertyName !== "background" && !this.validProperties[element.fullPropertyName]) {
        var value = element.node.getValue();
        if (value && this.documentText.charAt(value.offset) !== "-") {
          var elements_1 = this.fetch(propertyTable, element.fullPropertyName);
          if (elements_1.length > 1) {
            for (var k = 0; k < elements_1.length; k++) {
              var value_1 = elements_1[k].node.getValue();
              if (value_1 && this.documentText.charAt(value_1.offset) !== "-" && elements_1[k] !== element) {
                this.addEntry(element.node, Rules.DuplicateDeclarations);
              }
            }
          }
        }
      }
    }
    var isExportBlock = node.getSelectors().matches(":export");
    if (!isExportBlock) {
      var propertiesBySuffix = new NodesByRootMap();
      var containsUnknowns = false;
      for (var _d = 0, propertyTable_1 = propertyTable; _d < propertyTable_1.length; _d++) {
        var element = propertyTable_1[_d];
        var decl = element.node;
        if (this.isCSSDeclaration(decl)) {
          var name = element.fullPropertyName;
          var firstChar = name.charAt(0);
          if (firstChar === "-") {
            if (name.charAt(1) !== "-") {
              if (!this.cssDataManager.isKnownProperty(name) && !this.validProperties[name]) {
                this.addEntry(decl.getProperty(), Rules.UnknownVendorSpecificProperty);
              }
              var nonPrefixedName = decl.getNonPrefixedPropertyName();
              propertiesBySuffix.add(nonPrefixedName, name, decl.getProperty());
            }
          } else {
            var fullName = name;
            if (firstChar === "*" || firstChar === "_") {
              this.addEntry(decl.getProperty(), Rules.IEStarHack);
              name = name.substr(1);
            }
            if (!this.cssDataManager.isKnownProperty(fullName) && !this.cssDataManager.isKnownProperty(name)) {
              if (!this.validProperties[name]) {
                this.addEntry(decl.getProperty(), Rules.UnknownProperty, localize9("property.unknownproperty.detailed", "Unknown property: '{0}'", decl.getFullPropertyName()));
              }
            }
            propertiesBySuffix.add(name, name, null);
          }
        } else {
          containsUnknowns = true;
        }
      }
      if (!containsUnknowns) {
        for (var suffix in propertiesBySuffix.data) {
          var entry = propertiesBySuffix.data[suffix];
          var actual = entry.names;
          var needsStandard = this.cssDataManager.isStandardProperty(suffix) && actual.indexOf(suffix) === -1;
          if (!needsStandard && actual.length === 1) {
            continue;
          }
          var expected = [];
          for (var i = 0, len = LintVisitor2.prefixes.length; i < len; i++) {
            var prefix = LintVisitor2.prefixes[i];
            if (this.cssDataManager.isStandardProperty(prefix + suffix)) {
              expected.push(prefix + suffix);
            }
          }
          var missingVendorSpecific = this.getMissingNames(expected, actual);
          if (missingVendorSpecific || needsStandard) {
            for (var _e = 0, _f2 = entry.nodes; _e < _f2.length; _e++) {
              var node_2 = _f2[_e];
              if (needsStandard) {
                var message = localize9("property.standard.missing", "Also define the standard property '{0}' for compatibility", suffix);
                this.addEntry(node_2, Rules.IncludeStandardPropertyWhenUsingVendorPrefix, message);
              }
              if (missingVendorSpecific) {
                var message = localize9("property.vendorspecific.missing", "Always include all vendor specific properties: Missing: {0}", missingVendorSpecific);
                this.addEntry(node_2, Rules.AllVendorPrefixes, message);
              }
            }
          }
        }
      }
    }
    return true;
  };
  LintVisitor2.prototype.visitPrio = function(node) {
    this.addEntry(node, Rules.AvoidImportant);
    return true;
  };
  LintVisitor2.prototype.visitNumericValue = function(node) {
    var funcDecl = node.findParent(NodeType.Function);
    if (funcDecl && funcDecl.getName() === "calc") {
      return true;
    }
    var decl = node.findParent(NodeType.Declaration);
    if (decl) {
      var declValue = decl.getValue();
      if (declValue) {
        var value = node.getValue();
        if (!value.unit || units.length.indexOf(value.unit.toLowerCase()) === -1) {
          return true;
        }
        if (parseFloat(value.value) === 0 && !!value.unit && !this.validProperties[decl.getFullPropertyName()]) {
          this.addEntry(node, Rules.ZeroWithUnit);
        }
      }
    }
    return true;
  };
  LintVisitor2.prototype.visitFontFace = function(node) {
    var declarations = node.getDeclarations();
    if (!declarations) {
      return false;
    }
    var definesSrc = false, definesFontFamily = false;
    var containsUnknowns = false;
    for (var _i = 0, _a2 = declarations.getChildren(); _i < _a2.length; _i++) {
      var node_3 = _a2[_i];
      if (this.isCSSDeclaration(node_3)) {
        var name = node_3.getProperty().getName().toLowerCase();
        if (name === "src") {
          definesSrc = true;
        }
        if (name === "font-family") {
          definesFontFamily = true;
        }
      } else {
        containsUnknowns = true;
      }
    }
    if (!containsUnknowns && (!definesSrc || !definesFontFamily)) {
      this.addEntry(node, Rules.RequiredPropertiesForFontFace);
    }
    return true;
  };
  LintVisitor2.prototype.isCSSDeclaration = function(node) {
    if (node instanceof Declaration) {
      if (!node.getValue()) {
        return false;
      }
      var property = node.getProperty();
      if (!property) {
        return false;
      }
      var identifier = property.getIdentifier();
      if (!identifier || identifier.containsInterpolation()) {
        return false;
      }
      return true;
    }
    return false;
  };
  LintVisitor2.prototype.visitHexColorValue = function(node) {
    var length = node.length;
    if (length !== 9 && length !== 7 && length !== 5 && length !== 4) {
      this.addEntry(node, Rules.HexColorLength);
    }
    return false;
  };
  LintVisitor2.prototype.visitFunction = function(node) {
    var fnName = node.getName().toLowerCase();
    var expectedAttrCount = -1;
    var actualAttrCount = 0;
    switch (fnName) {
      case "rgb(":
      case "hsl(":
        expectedAttrCount = 3;
        break;
      case "rgba(":
      case "hsla(":
        expectedAttrCount = 4;
        break;
    }
    if (expectedAttrCount !== -1) {
      node.getArguments().accept(function(n) {
        if (n instanceof BinaryExpression) {
          actualAttrCount += 1;
          return false;
        }
        return true;
      });
      if (actualAttrCount !== expectedAttrCount) {
        this.addEntry(node, Rules.ArgsInColorFunction);
      }
    }
    return true;
  };
  LintVisitor2.prefixes = [
    "-ms-",
    "-moz-",
    "-o-",
    "-webkit-"
  ];
  return LintVisitor2;
}();

// node_modules/vscode-css-languageservice/lib/esm/services/cssValidation.js
var CSSValidation = function() {
  function CSSValidation2(cssDataManager) {
    this.cssDataManager = cssDataManager;
  }
  CSSValidation2.prototype.configure = function(settings) {
    this.settings = settings;
  };
  CSSValidation2.prototype.doValidation = function(document, stylesheet, settings) {
    if (settings === void 0) {
      settings = this.settings;
    }
    if (settings && settings.validate === false) {
      return [];
    }
    var entries = [];
    entries.push.apply(entries, ParseErrorCollector.entries(stylesheet));
    entries.push.apply(entries, LintVisitor.entries(stylesheet, document, new LintConfigurationSettings(settings && settings.lint), this.cssDataManager));
    var ruleIds = [];
    for (var r in Rules) {
      ruleIds.push(Rules[r].id);
    }
    function toDiagnostic(marker) {
      var range = Range.create(document.positionAt(marker.getOffset()), document.positionAt(marker.getOffset() + marker.getLength()));
      var source = document.languageId;
      return {
        code: marker.getRule().id,
        source,
        message: marker.getMessage(),
        severity: marker.getLevel() === Level.Warning ? DiagnosticSeverity.Warning : DiagnosticSeverity.Error,
        range
      };
    }
    return entries.filter(function(entry) {
      return entry.getLevel() !== Level.Ignore;
    }).map(toDiagnostic);
  };
  return CSSValidation2;
}();

// node_modules/vscode-css-languageservice/lib/esm/parser/scssScanner.js
var __extends4 = function() {
  var extendStatics = function(d, b) {
    extendStatics = Object.setPrototypeOf || { __proto__: [] } instanceof Array && function(d2, b2) {
      d2.__proto__ = b2;
    } || function(d2, b2) {
      for (var p in b2)
        if (Object.prototype.hasOwnProperty.call(b2, p))
          d2[p] = b2[p];
    };
    return extendStatics(d, b);
  };
  return function(d, b) {
    if (typeof b !== "function" && b !== null)
      throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
    extendStatics(d, b);
    function __() {
      this.constructor = d;
    }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
  };
}();
var _FSL2 = "/".charCodeAt(0);
var _NWL2 = "\n".charCodeAt(0);
var _CAR2 = "\r".charCodeAt(0);
var _LFD2 = "\f".charCodeAt(0);
var _DLR2 = "$".charCodeAt(0);
var _HSH2 = "#".charCodeAt(0);
var _CUL2 = "{".charCodeAt(0);
var _EQS2 = "=".charCodeAt(0);
var _BNG2 = "!".charCodeAt(0);
var _LAN2 = "<".charCodeAt(0);
var _RAN2 = ">".charCodeAt(0);
var _DOT2 = ".".charCodeAt(0);
var _ATS2 = "@".charCodeAt(0);
var customTokenValue = TokenType.CustomToken;
var VariableName = customTokenValue++;
var InterpolationFunction = customTokenValue++;
var Default = customTokenValue++;
var EqualsOperator = customTokenValue++;
var NotEqualsOperator = customTokenValue++;
var GreaterEqualsOperator = customTokenValue++;
var SmallerEqualsOperator = customTokenValue++;
var Ellipsis = customTokenValue++;
var Module2 = customTokenValue++;
var SCSSScanner = function(_super) {
  __extends4(SCSSScanner2, _super);
  function SCSSScanner2() {
    return _super !== null && _super.apply(this, arguments) || this;
  }
  SCSSScanner2.prototype.scanNext = function(offset) {
    if (this.stream.advanceIfChar(_DLR2)) {
      var content = ["$"];
      if (this.ident(content)) {
        return this.finishToken(offset, VariableName, content.join(""));
      } else {
        this.stream.goBackTo(offset);
      }
    }
    if (this.stream.advanceIfChars([_HSH2, _CUL2])) {
      return this.finishToken(offset, InterpolationFunction);
    }
    if (this.stream.advanceIfChars([_EQS2, _EQS2])) {
      return this.finishToken(offset, EqualsOperator);
    }
    if (this.stream.advanceIfChars([_BNG2, _EQS2])) {
      return this.finishToken(offset, NotEqualsOperator);
    }
    if (this.stream.advanceIfChar(_LAN2)) {
      if (this.stream.advanceIfChar(_EQS2)) {
        return this.finishToken(offset, SmallerEqualsOperator);
      }
      return this.finishToken(offset, TokenType.Delim);
    }
    if (this.stream.advanceIfChar(_RAN2)) {
      if (this.stream.advanceIfChar(_EQS2)) {
        return this.finishToken(offset, GreaterEqualsOperator);
      }
      return this.finishToken(offset, TokenType.Delim);
    }
    if (this.stream.advanceIfChars([_DOT2, _DOT2, _DOT2])) {
      return this.finishToken(offset, Ellipsis);
    }
    return _super.prototype.scanNext.call(this, offset);
  };
  SCSSScanner2.prototype.comment = function() {
    if (_super.prototype.comment.call(this)) {
      return true;
    }
    if (!this.inURL && this.stream.advanceIfChars([_FSL2, _FSL2])) {
      this.stream.advanceWhileChar(function(ch) {
        switch (ch) {
          case _NWL2:
          case _CAR2:
          case _LFD2:
            return false;
          default:
            return true;
        }
      });
      return true;
    } else {
      return false;
    }
  };
  return SCSSScanner2;
}(Scanner);

// node_modules/vscode-css-languageservice/lib/esm/parser/scssErrors.js
var localize10 = loadMessageBundle();
var SCSSIssueType = function() {
  function SCSSIssueType2(id, message) {
    this.id = id;
    this.message = message;
  }
  return SCSSIssueType2;
}();
var SCSSParseError = {
  FromExpected: new SCSSIssueType("scss-fromexpected", localize10("expected.from", "'from' expected")),
  ThroughOrToExpected: new SCSSIssueType("scss-throughexpected", localize10("expected.through", "'through' or 'to' expected")),
  InExpected: new SCSSIssueType("scss-fromexpected", localize10("expected.in", "'in' expected"))
};

// node_modules/vscode-css-languageservice/lib/esm/parser/scssParser.js
var __extends5 = function() {
  var extendStatics = function(d, b) {
    extendStatics = Object.setPrototypeOf || { __proto__: [] } instanceof Array && function(d2, b2) {
      d2.__proto__ = b2;
    } || function(d2, b2) {
      for (var p in b2)
        if (Object.prototype.hasOwnProperty.call(b2, p))
          d2[p] = b2[p];
    };
    return extendStatics(d, b);
  };
  return function(d, b) {
    if (typeof b !== "function" && b !== null)
      throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
    extendStatics(d, b);
    function __() {
      this.constructor = d;
    }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
  };
}();
var SCSSParser = function(_super) {
  __extends5(SCSSParser2, _super);
  function SCSSParser2() {
    return _super.call(this, new SCSSScanner()) || this;
  }
  SCSSParser2.prototype._parseStylesheetStatement = function(isNested) {
    if (isNested === void 0) {
      isNested = false;
    }
    if (this.peek(TokenType.AtKeyword)) {
      return this._parseWarnAndDebug() || this._parseControlStatement() || this._parseMixinDeclaration() || this._parseMixinContent() || this._parseMixinReference() || this._parseFunctionDeclaration() || this._parseForward() || this._parseUse() || this._parseRuleset(isNested) || _super.prototype._parseStylesheetAtStatement.call(this, isNested);
    }
    return this._parseRuleset(true) || this._parseVariableDeclaration();
  };
  SCSSParser2.prototype._parseImport = function() {
    if (!this.peekKeyword("@import")) {
      return null;
    }
    var node = this.create(Import);
    this.consumeToken();
    if (!node.addChild(this._parseURILiteral()) && !node.addChild(this._parseStringLiteral())) {
      return this.finish(node, ParseError.URIOrStringExpected);
    }
    while (this.accept(TokenType.Comma)) {
      if (!node.addChild(this._parseURILiteral()) && !node.addChild(this._parseStringLiteral())) {
        return this.finish(node, ParseError.URIOrStringExpected);
      }
    }
    if (!this.peek(TokenType.SemiColon) && !this.peek(TokenType.EOF)) {
      node.setMedialist(this._parseMediaQueryList());
    }
    return this.finish(node);
  };
  SCSSParser2.prototype._parseVariableDeclaration = function(panic) {
    if (panic === void 0) {
      panic = [];
    }
    if (!this.peek(VariableName)) {
      return null;
    }
    var node = this.create(VariableDeclaration);
    if (!node.setVariable(this._parseVariable())) {
      return null;
    }
    if (!this.accept(TokenType.Colon)) {
      return this.finish(node, ParseError.ColonExpected);
    }
    if (this.prevToken) {
      node.colonPosition = this.prevToken.offset;
    }
    if (!node.setValue(this._parseExpr())) {
      return this.finish(node, ParseError.VariableValueExpected, [], panic);
    }
    while (this.peek(TokenType.Exclamation)) {
      if (node.addChild(this._tryParsePrio())) {
      } else {
        this.consumeToken();
        if (!this.peekRegExp(TokenType.Ident, /^(default|global)$/)) {
          return this.finish(node, ParseError.UnknownKeyword);
        }
        this.consumeToken();
      }
    }
    if (this.peek(TokenType.SemiColon)) {
      node.semicolonPosition = this.token.offset;
    }
    return this.finish(node);
  };
  SCSSParser2.prototype._parseMediaCondition = function() {
    return this._parseInterpolation() || _super.prototype._parseMediaCondition.call(this);
  };
  SCSSParser2.prototype._parseMediaFeatureName = function() {
    return this._parseModuleMember() || this._parseFunction() || this._parseIdent() || this._parseVariable();
  };
  SCSSParser2.prototype._parseKeyframeSelector = function() {
    return this._tryParseKeyframeSelector() || this._parseControlStatement(this._parseKeyframeSelector.bind(this)) || this._parseVariableDeclaration() || this._parseMixinContent();
  };
  SCSSParser2.prototype._parseVariable = function() {
    if (!this.peek(VariableName)) {
      return null;
    }
    var node = this.create(Variable);
    this.consumeToken();
    return node;
  };
  SCSSParser2.prototype._parseModuleMember = function() {
    var pos = this.mark();
    var node = this.create(Module);
    if (!node.setIdentifier(this._parseIdent([ReferenceType.Module]))) {
      return null;
    }
    if (this.hasWhitespace() || !this.acceptDelim(".") || this.hasWhitespace()) {
      this.restoreAtMark(pos);
      return null;
    }
    if (!node.addChild(this._parseVariable() || this._parseFunction())) {
      return this.finish(node, ParseError.IdentifierOrVariableExpected);
    }
    return node;
  };
  SCSSParser2.prototype._parseIdent = function(referenceTypes) {
    var _this = this;
    if (!this.peek(TokenType.Ident) && !this.peek(InterpolationFunction) && !this.peekDelim("-")) {
      return null;
    }
    var node = this.create(Identifier);
    node.referenceTypes = referenceTypes;
    node.isCustomProperty = this.peekRegExp(TokenType.Ident, /^--/);
    var hasContent = false;
    var indentInterpolation = function() {
      var pos = _this.mark();
      if (_this.acceptDelim("-")) {
        if (!_this.hasWhitespace()) {
          _this.acceptDelim("-");
        }
        if (_this.hasWhitespace()) {
          _this.restoreAtMark(pos);
          return null;
        }
      }
      return _this._parseInterpolation();
    };
    while (this.accept(TokenType.Ident) || node.addChild(indentInterpolation()) || hasContent && this.acceptRegexp(/^[\w-]/)) {
      hasContent = true;
      if (this.hasWhitespace()) {
        break;
      }
    }
    return hasContent ? this.finish(node) : null;
  };
  SCSSParser2.prototype._parseTermExpression = function() {
    return this._parseModuleMember() || this._parseVariable() || this._parseSelectorCombinator() || _super.prototype._parseTermExpression.call(this);
  };
  SCSSParser2.prototype._parseInterpolation = function() {
    if (this.peek(InterpolationFunction)) {
      var node = this.create(Interpolation);
      this.consumeToken();
      if (!node.addChild(this._parseExpr()) && !this._parseSelectorCombinator()) {
        if (this.accept(TokenType.CurlyR)) {
          return this.finish(node);
        }
        return this.finish(node, ParseError.ExpressionExpected);
      }
      if (!this.accept(TokenType.CurlyR)) {
        return this.finish(node, ParseError.RightCurlyExpected);
      }
      return this.finish(node);
    }
    return null;
  };
  SCSSParser2.prototype._parseOperator = function() {
    if (this.peek(EqualsOperator) || this.peek(NotEqualsOperator) || this.peek(GreaterEqualsOperator) || this.peek(SmallerEqualsOperator) || this.peekDelim(">") || this.peekDelim("<") || this.peekIdent("and") || this.peekIdent("or") || this.peekDelim("%")) {
      var node = this.createNode(NodeType.Operator);
      this.consumeToken();
      return this.finish(node);
    }
    return _super.prototype._parseOperator.call(this);
  };
  SCSSParser2.prototype._parseUnaryOperator = function() {
    if (this.peekIdent("not")) {
      var node = this.create(Node);
      this.consumeToken();
      return this.finish(node);
    }
    return _super.prototype._parseUnaryOperator.call(this);
  };
  SCSSParser2.prototype._parseRuleSetDeclaration = function() {
    if (this.peek(TokenType.AtKeyword)) {
      return this._parseKeyframe() || this._parseImport() || this._parseMedia(true) || this._parseFontFace() || this._parseWarnAndDebug() || this._parseControlStatement() || this._parseFunctionDeclaration() || this._parseExtends() || this._parseMixinReference() || this._parseMixinContent() || this._parseMixinDeclaration() || this._parseRuleset(true) || this._parseSupports(true) || _super.prototype._parseRuleSetDeclarationAtStatement.call(this);
    }
    return this._parseVariableDeclaration() || this._tryParseRuleset(true) || _super.prototype._parseRuleSetDeclaration.call(this);
  };
  SCSSParser2.prototype._parseDeclaration = function(stopTokens) {
    var custonProperty = this._tryParseCustomPropertyDeclaration(stopTokens);
    if (custonProperty) {
      return custonProperty;
    }
    var node = this.create(Declaration);
    if (!node.setProperty(this._parseProperty())) {
      return null;
    }
    if (!this.accept(TokenType.Colon)) {
      return this.finish(node, ParseError.ColonExpected, [TokenType.Colon], stopTokens || [TokenType.SemiColon]);
    }
    if (this.prevToken) {
      node.colonPosition = this.prevToken.offset;
    }
    var hasContent = false;
    if (node.setValue(this._parseExpr())) {
      hasContent = true;
      node.addChild(this._parsePrio());
    }
    if (this.peek(TokenType.CurlyL)) {
      node.setNestedProperties(this._parseNestedProperties());
    } else {
      if (!hasContent) {
        return this.finish(node, ParseError.PropertyValueExpected);
      }
    }
    if (this.peek(TokenType.SemiColon)) {
      node.semicolonPosition = this.token.offset;
    }
    return this.finish(node);
  };
  SCSSParser2.prototype._parseNestedProperties = function() {
    var node = this.create(NestedProperties);
    return this._parseBody(node, this._parseDeclaration.bind(this));
  };
  SCSSParser2.prototype._parseExtends = function() {
    if (this.peekKeyword("@extend")) {
      var node = this.create(ExtendsReference);
      this.consumeToken();
      if (!node.getSelectors().addChild(this._parseSimpleSelector())) {
        return this.finish(node, ParseError.SelectorExpected);
      }
      while (this.accept(TokenType.Comma)) {
        node.getSelectors().addChild(this._parseSimpleSelector());
      }
      if (this.accept(TokenType.Exclamation)) {
        if (!this.acceptIdent("optional")) {
          return this.finish(node, ParseError.UnknownKeyword);
        }
      }
      return this.finish(node);
    }
    return null;
  };
  SCSSParser2.prototype._parseSimpleSelectorBody = function() {
    return this._parseSelectorCombinator() || this._parseSelectorPlaceholder() || _super.prototype._parseSimpleSelectorBody.call(this);
  };
  SCSSParser2.prototype._parseSelectorCombinator = function() {
    if (this.peekDelim("&")) {
      var node = this.createNode(NodeType.SelectorCombinator);
      this.consumeToken();
      while (!this.hasWhitespace() && (this.acceptDelim("-") || this.accept(TokenType.Num) || this.accept(TokenType.Dimension) || node.addChild(this._parseIdent()) || this.acceptDelim("&"))) {
      }
      return this.finish(node);
    }
    return null;
  };
  SCSSParser2.prototype._parseSelectorPlaceholder = function() {
    if (this.peekDelim("%")) {
      var node = this.createNode(NodeType.SelectorPlaceholder);
      this.consumeToken();
      this._parseIdent();
      return this.finish(node);
    } else if (this.peekKeyword("@at-root")) {
      var node = this.createNode(NodeType.SelectorPlaceholder);
      this.consumeToken();
      return this.finish(node);
    }
    return null;
  };
  SCSSParser2.prototype._parseElementName = function() {
    var pos = this.mark();
    var node = _super.prototype._parseElementName.call(this);
    if (node && !this.hasWhitespace() && this.peek(TokenType.ParenthesisL)) {
      this.restoreAtMark(pos);
      return null;
    }
    return node;
  };
  SCSSParser2.prototype._tryParsePseudoIdentifier = function() {
    return this._parseInterpolation() || _super.prototype._tryParsePseudoIdentifier.call(this);
  };
  SCSSParser2.prototype._parseWarnAndDebug = function() {
    if (!this.peekKeyword("@debug") && !this.peekKeyword("@warn") && !this.peekKeyword("@error")) {
      return null;
    }
    var node = this.createNode(NodeType.Debug);
    this.consumeToken();
    node.addChild(this._parseExpr());
    return this.finish(node);
  };
  SCSSParser2.prototype._parseControlStatement = function(parseStatement) {
    if (parseStatement === void 0) {
      parseStatement = this._parseRuleSetDeclaration.bind(this);
    }
    if (!this.peek(TokenType.AtKeyword)) {
      return null;
    }
    return this._parseIfStatement(parseStatement) || this._parseForStatement(parseStatement) || this._parseEachStatement(parseStatement) || this._parseWhileStatement(parseStatement);
  };
  SCSSParser2.prototype._parseIfStatement = function(parseStatement) {
    if (!this.peekKeyword("@if")) {
      return null;
    }
    return this._internalParseIfStatement(parseStatement);
  };
  SCSSParser2.prototype._internalParseIfStatement = function(parseStatement) {
    var node = this.create(IfStatement);
    this.consumeToken();
    if (!node.setExpression(this._parseExpr(true))) {
      return this.finish(node, ParseError.ExpressionExpected);
    }
    this._parseBody(node, parseStatement);
    if (this.acceptKeyword("@else")) {
      if (this.peekIdent("if")) {
        node.setElseClause(this._internalParseIfStatement(parseStatement));
      } else if (this.peek(TokenType.CurlyL)) {
        var elseNode = this.create(ElseStatement);
        this._parseBody(elseNode, parseStatement);
        node.setElseClause(elseNode);
      }
    }
    return this.finish(node);
  };
  SCSSParser2.prototype._parseForStatement = function(parseStatement) {
    if (!this.peekKeyword("@for")) {
      return null;
    }
    var node = this.create(ForStatement);
    this.consumeToken();
    if (!node.setVariable(this._parseVariable())) {
      return this.finish(node, ParseError.VariableNameExpected, [TokenType.CurlyR]);
    }
    if (!this.acceptIdent("from")) {
      return this.finish(node, SCSSParseError.FromExpected, [TokenType.CurlyR]);
    }
    if (!node.addChild(this._parseBinaryExpr())) {
      return this.finish(node, ParseError.ExpressionExpected, [TokenType.CurlyR]);
    }
    if (!this.acceptIdent("to") && !this.acceptIdent("through")) {
      return this.finish(node, SCSSParseError.ThroughOrToExpected, [TokenType.CurlyR]);
    }
    if (!node.addChild(this._parseBinaryExpr())) {
      return this.finish(node, ParseError.ExpressionExpected, [TokenType.CurlyR]);
    }
    return this._parseBody(node, parseStatement);
  };
  SCSSParser2.prototype._parseEachStatement = function(parseStatement) {
    if (!this.peekKeyword("@each")) {
      return null;
    }
    var node = this.create(EachStatement);
    this.consumeToken();
    var variables = node.getVariables();
    if (!variables.addChild(this._parseVariable())) {
      return this.finish(node, ParseError.VariableNameExpected, [TokenType.CurlyR]);
    }
    while (this.accept(TokenType.Comma)) {
      if (!variables.addChild(this._parseVariable())) {
        return this.finish(node, ParseError.VariableNameExpected, [TokenType.CurlyR]);
      }
    }
    this.finish(variables);
    if (!this.acceptIdent("in")) {
      return this.finish(node, SCSSParseError.InExpected, [TokenType.CurlyR]);
    }
    if (!node.addChild(this._parseExpr())) {
      return this.finish(node, ParseError.ExpressionExpected, [TokenType.CurlyR]);
    }
    return this._parseBody(node, parseStatement);
  };
  SCSSParser2.prototype._parseWhileStatement = function(parseStatement) {
    if (!this.peekKeyword("@while")) {
      return null;
    }
    var node = this.create(WhileStatement);
    this.consumeToken();
    if (!node.addChild(this._parseBinaryExpr())) {
      return this.finish(node, ParseError.ExpressionExpected, [TokenType.CurlyR]);
    }
    return this._parseBody(node, parseStatement);
  };
  SCSSParser2.prototype._parseFunctionBodyDeclaration = function() {
    return this._parseVariableDeclaration() || this._parseReturnStatement() || this._parseWarnAndDebug() || this._parseControlStatement(this._parseFunctionBodyDeclaration.bind(this));
  };
  SCSSParser2.prototype._parseFunctionDeclaration = function() {
    if (!this.peekKeyword("@function")) {
      return null;
    }
    var node = this.create(FunctionDeclaration);
    this.consumeToken();
    if (!node.setIdentifier(this._parseIdent([ReferenceType.Function]))) {
      return this.finish(node, ParseError.IdentifierExpected, [TokenType.CurlyR]);
    }
    if (!this.accept(TokenType.ParenthesisL)) {
      return this.finish(node, ParseError.LeftParenthesisExpected, [TokenType.CurlyR]);
    }
    if (node.getParameters().addChild(this._parseParameterDeclaration())) {
      while (this.accept(TokenType.Comma)) {
        if (this.peek(TokenType.ParenthesisR)) {
          break;
        }
        if (!node.getParameters().addChild(this._parseParameterDeclaration())) {
          return this.finish(node, ParseError.VariableNameExpected);
        }
      }
    }
    if (!this.accept(TokenType.ParenthesisR)) {
      return this.finish(node, ParseError.RightParenthesisExpected, [TokenType.CurlyR]);
    }
    return this._parseBody(node, this._parseFunctionBodyDeclaration.bind(this));
  };
  SCSSParser2.prototype._parseReturnStatement = function() {
    if (!this.peekKeyword("@return")) {
      return null;
    }
    var node = this.createNode(NodeType.ReturnStatement);
    this.consumeToken();
    if (!node.addChild(this._parseExpr())) {
      return this.finish(node, ParseError.ExpressionExpected);
    }
    return this.finish(node);
  };
  SCSSParser2.prototype._parseMixinDeclaration = function() {
    if (!this.peekKeyword("@mixin")) {
      return null;
    }
    var node = this.create(MixinDeclaration);
    this.consumeToken();
    if (!node.setIdentifier(this._parseIdent([ReferenceType.Mixin]))) {
      return this.finish(node, ParseError.IdentifierExpected, [TokenType.CurlyR]);
    }
    if (this.accept(TokenType.ParenthesisL)) {
      if (node.getParameters().addChild(this._parseParameterDeclaration())) {
        while (this.accept(TokenType.Comma)) {
          if (this.peek(TokenType.ParenthesisR)) {
            break;
          }
          if (!node.getParameters().addChild(this._parseParameterDeclaration())) {
            return this.finish(node, ParseError.VariableNameExpected);
          }
        }
      }
      if (!this.accept(TokenType.ParenthesisR)) {
        return this.finish(node, ParseError.RightParenthesisExpected, [TokenType.CurlyR]);
      }
    }
    return this._parseBody(node, this._parseRuleSetDeclaration.bind(this));
  };
  SCSSParser2.prototype._parseParameterDeclaration = function() {
    var node = this.create(FunctionParameter);
    if (!node.setIdentifier(this._parseVariable())) {
      return null;
    }
    if (this.accept(Ellipsis)) {
    }
    if (this.accept(TokenType.Colon)) {
      if (!node.setDefaultValue(this._parseExpr(true))) {
        return this.finish(node, ParseError.VariableValueExpected, [], [TokenType.Comma, TokenType.ParenthesisR]);
      }
    }
    return this.finish(node);
  };
  SCSSParser2.prototype._parseMixinContent = function() {
    if (!this.peekKeyword("@content")) {
      return null;
    }
    var node = this.create(MixinContentReference);
    this.consumeToken();
    if (this.accept(TokenType.ParenthesisL)) {
      if (node.getArguments().addChild(this._parseFunctionArgument())) {
        while (this.accept(TokenType.Comma)) {
          if (this.peek(TokenType.ParenthesisR)) {
            break;
          }
          if (!node.getArguments().addChild(this._parseFunctionArgument())) {
            return this.finish(node, ParseError.ExpressionExpected);
          }
        }
      }
      if (!this.accept(TokenType.ParenthesisR)) {
        return this.finish(node, ParseError.RightParenthesisExpected);
      }
    }
    return this.finish(node);
  };
  SCSSParser2.prototype._parseMixinReference = function() {
    if (!this.peekKeyword("@include")) {
      return null;
    }
    var node = this.create(MixinReference);
    this.consumeToken();
    var firstIdent = this._parseIdent([ReferenceType.Mixin]);
    if (!node.setIdentifier(firstIdent)) {
      return this.finish(node, ParseError.IdentifierExpected, [TokenType.CurlyR]);
    }
    if (!this.hasWhitespace() && this.acceptDelim(".") && !this.hasWhitespace()) {
      var secondIdent = this._parseIdent([ReferenceType.Mixin]);
      if (!secondIdent) {
        return this.finish(node, ParseError.IdentifierExpected, [TokenType.CurlyR]);
      }
      var moduleToken = this.create(Module);
      firstIdent.referenceTypes = [ReferenceType.Module];
      moduleToken.setIdentifier(firstIdent);
      node.setIdentifier(secondIdent);
      node.addChild(moduleToken);
    }
    if (this.accept(TokenType.ParenthesisL)) {
      if (node.getArguments().addChild(this._parseFunctionArgument())) {
        while (this.accept(TokenType.Comma)) {
          if (this.peek(TokenType.ParenthesisR)) {
            break;
          }
          if (!node.getArguments().addChild(this._parseFunctionArgument())) {
            return this.finish(node, ParseError.ExpressionExpected);
          }
        }
      }
      if (!this.accept(TokenType.ParenthesisR)) {
        return this.finish(node, ParseError.RightParenthesisExpected);
      }
    }
    if (this.peekIdent("using") || this.peek(TokenType.CurlyL)) {
      node.setContent(this._parseMixinContentDeclaration());
    }
    return this.finish(node);
  };
  SCSSParser2.prototype._parseMixinContentDeclaration = function() {
    var node = this.create(MixinContentDeclaration);
    if (this.acceptIdent("using")) {
      if (!this.accept(TokenType.ParenthesisL)) {
        return this.finish(node, ParseError.LeftParenthesisExpected, [TokenType.CurlyL]);
      }
      if (node.getParameters().addChild(this._parseParameterDeclaration())) {
        while (this.accept(TokenType.Comma)) {
          if (this.peek(TokenType.ParenthesisR)) {
            break;
          }
          if (!node.getParameters().addChild(this._parseParameterDeclaration())) {
            return this.finish(node, ParseError.VariableNameExpected);
          }
        }
      }
      if (!this.accept(TokenType.ParenthesisR)) {
        return this.finish(node, ParseError.RightParenthesisExpected, [TokenType.CurlyL]);
      }
    }
    if (this.peek(TokenType.CurlyL)) {
      this._parseBody(node, this._parseMixinReferenceBodyStatement.bind(this));
    }
    return this.finish(node);
  };
  SCSSParser2.prototype._parseMixinReferenceBodyStatement = function() {
    return this._tryParseKeyframeSelector() || this._parseRuleSetDeclaration();
  };
  SCSSParser2.prototype._parseFunctionArgument = function() {
    var node = this.create(FunctionArgument);
    var pos = this.mark();
    var argument = this._parseVariable();
    if (argument) {
      if (!this.accept(TokenType.Colon)) {
        if (this.accept(Ellipsis)) {
          node.setValue(argument);
          return this.finish(node);
        } else {
          this.restoreAtMark(pos);
        }
      } else {
        node.setIdentifier(argument);
      }
    }
    if (node.setValue(this._parseExpr(true))) {
      this.accept(Ellipsis);
      node.addChild(this._parsePrio());
      return this.finish(node);
    } else if (node.setValue(this._tryParsePrio())) {
      return this.finish(node);
    }
    return null;
  };
  SCSSParser2.prototype._parseURLArgument = function() {
    var pos = this.mark();
    var node = _super.prototype._parseURLArgument.call(this);
    if (!node || !this.peek(TokenType.ParenthesisR)) {
      this.restoreAtMark(pos);
      var node_1 = this.create(Node);
      node_1.addChild(this._parseBinaryExpr());
      return this.finish(node_1);
    }
    return node;
  };
  SCSSParser2.prototype._parseOperation = function() {
    if (!this.peek(TokenType.ParenthesisL)) {
      return null;
    }
    var node = this.create(Node);
    this.consumeToken();
    while (node.addChild(this._parseListElement())) {
      this.accept(TokenType.Comma);
    }
    if (!this.accept(TokenType.ParenthesisR)) {
      return this.finish(node, ParseError.RightParenthesisExpected);
    }
    return this.finish(node);
  };
  SCSSParser2.prototype._parseListElement = function() {
    var node = this.create(ListEntry);
    var child = this._parseBinaryExpr();
    if (!child) {
      return null;
    }
    if (this.accept(TokenType.Colon)) {
      node.setKey(child);
      if (!node.setValue(this._parseBinaryExpr())) {
        return this.finish(node, ParseError.ExpressionExpected);
      }
    } else {
      node.setValue(child);
    }
    return this.finish(node);
  };
  SCSSParser2.prototype._parseUse = function() {
    if (!this.peekKeyword("@use")) {
      return null;
    }
    var node = this.create(Use);
    this.consumeToken();
    if (!node.addChild(this._parseStringLiteral())) {
      return this.finish(node, ParseError.StringLiteralExpected);
    }
    if (!this.peek(TokenType.SemiColon) && !this.peek(TokenType.EOF)) {
      if (!this.peekRegExp(TokenType.Ident, /as|with/)) {
        return this.finish(node, ParseError.UnknownKeyword);
      }
      if (this.acceptIdent("as") && (!node.setIdentifier(this._parseIdent([ReferenceType.Module])) && !this.acceptDelim("*"))) {
        return this.finish(node, ParseError.IdentifierOrWildcardExpected);
      }
      if (this.acceptIdent("with")) {
        if (!this.accept(TokenType.ParenthesisL)) {
          return this.finish(node, ParseError.LeftParenthesisExpected, [TokenType.ParenthesisR]);
        }
        if (!node.getParameters().addChild(this._parseModuleConfigDeclaration())) {
          return this.finish(node, ParseError.VariableNameExpected);
        }
        while (this.accept(TokenType.Comma)) {
          if (this.peek(TokenType.ParenthesisR)) {
            break;
          }
          if (!node.getParameters().addChild(this._parseModuleConfigDeclaration())) {
            return this.finish(node, ParseError.VariableNameExpected);
          }
        }
        if (!this.accept(TokenType.ParenthesisR)) {
          return this.finish(node, ParseError.RightParenthesisExpected);
        }
      }
    }
    if (!this.accept(TokenType.SemiColon) && !this.accept(TokenType.EOF)) {
      return this.finish(node, ParseError.SemiColonExpected);
    }
    return this.finish(node);
  };
  SCSSParser2.prototype._parseModuleConfigDeclaration = function() {
    var node = this.create(ModuleConfiguration);
    if (!node.setIdentifier(this._parseVariable())) {
      return null;
    }
    if (!this.accept(TokenType.Colon) || !node.setValue(this._parseExpr(true))) {
      return this.finish(node, ParseError.VariableValueExpected, [], [TokenType.Comma, TokenType.ParenthesisR]);
    }
    if (this.accept(TokenType.Exclamation)) {
      if (this.hasWhitespace() || !this.acceptIdent("default")) {
        return this.finish(node, ParseError.UnknownKeyword);
      }
    }
    return this.finish(node);
  };
  SCSSParser2.prototype._parseForward = function() {
    if (!this.peekKeyword("@forward")) {
      return null;
    }
    var node = this.create(Forward);
    this.consumeToken();
    if (!node.addChild(this._parseStringLiteral())) {
      return this.finish(node, ParseError.StringLiteralExpected);
    }
    if (this.acceptIdent("with")) {
      if (!this.accept(TokenType.ParenthesisL)) {
        return this.finish(node, ParseError.LeftParenthesisExpected, [TokenType.ParenthesisR]);
      }
      if (!node.getParameters().addChild(this._parseModuleConfigDeclaration())) {
        return this.finish(node, ParseError.VariableNameExpected);
      }
      while (this.accept(TokenType.Comma)) {
        if (this.peek(TokenType.ParenthesisR)) {
          break;
        }
        if (!node.getParameters().addChild(this._parseModuleConfigDeclaration())) {
          return this.finish(node, ParseError.VariableNameExpected);
        }
      }
      if (!this.accept(TokenType.ParenthesisR)) {
        return this.finish(node, ParseError.RightParenthesisExpected);
      }
    }
    if (!this.peek(TokenType.SemiColon) && !this.peek(TokenType.EOF)) {
      if (!this.peekRegExp(TokenType.Ident, /as|hide|show/)) {
        return this.finish(node, ParseError.UnknownKeyword);
      }
      if (this.acceptIdent("as")) {
        var identifier = this._parseIdent([ReferenceType.Forward]);
        if (!node.setIdentifier(identifier)) {
          return this.finish(node, ParseError.IdentifierExpected);
        }
        if (this.hasWhitespace() || !this.acceptDelim("*")) {
          return this.finish(node, ParseError.WildcardExpected);
        }
      }
      if (this.peekIdent("hide") || this.peekIdent("show")) {
        if (!node.addChild(this._parseForwardVisibility())) {
          return this.finish(node, ParseError.IdentifierOrVariableExpected);
        }
      }
    }
    if (!this.accept(TokenType.SemiColon) && !this.accept(TokenType.EOF)) {
      return this.finish(node, ParseError.SemiColonExpected);
    }
    return this.finish(node);
  };
  SCSSParser2.prototype._parseForwardVisibility = function() {
    var node = this.create(ForwardVisibility);
    node.setIdentifier(this._parseIdent());
    while (node.addChild(this._parseVariable() || this._parseIdent())) {
      this.accept(TokenType.Comma);
    }
    return node.getChildren().length > 1 ? node : null;
  };
  SCSSParser2.prototype._parseSupportsCondition = function() {
    return this._parseInterpolation() || _super.prototype._parseSupportsCondition.call(this);
  };
  return SCSSParser2;
}(Parser);

// node_modules/vscode-css-languageservice/lib/esm/services/scssCompletion.js
var __extends6 = function() {
  var extendStatics = function(d, b) {
    extendStatics = Object.setPrototypeOf || { __proto__: [] } instanceof Array && function(d2, b2) {
      d2.__proto__ = b2;
    } || function(d2, b2) {
      for (var p in b2)
        if (Object.prototype.hasOwnProperty.call(b2, p))
          d2[p] = b2[p];
    };
    return extendStatics(d, b);
  };
  return function(d, b) {
    if (typeof b !== "function" && b !== null)
      throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
    extendStatics(d, b);
    function __() {
      this.constructor = d;
    }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
  };
}();
var localize11 = loadMessageBundle();
var SCSSCompletion = function(_super) {
  __extends6(SCSSCompletion2, _super);
  function SCSSCompletion2(lsServiceOptions, cssDataManager) {
    var _this = _super.call(this, "$", lsServiceOptions, cssDataManager) || this;
    addReferencesToDocumentation(SCSSCompletion2.scssModuleLoaders);
    addReferencesToDocumentation(SCSSCompletion2.scssModuleBuiltIns);
    return _this;
  }
  SCSSCompletion2.prototype.isImportPathParent = function(type) {
    return type === NodeType.Forward || type === NodeType.Use || _super.prototype.isImportPathParent.call(this, type);
  };
  SCSSCompletion2.prototype.getCompletionForImportPath = function(importPathNode, result) {
    var parentType = importPathNode.getParent().type;
    if (parentType === NodeType.Forward || parentType === NodeType.Use) {
      for (var _i = 0, _a2 = SCSSCompletion2.scssModuleBuiltIns; _i < _a2.length; _i++) {
        var p = _a2[_i];
        var item = {
          label: p.label,
          documentation: p.documentation,
          textEdit: TextEdit.replace(this.getCompletionRange(importPathNode), "'".concat(p.label, "'")),
          kind: CompletionItemKind.Module
        };
        result.items.push(item);
      }
    }
    return _super.prototype.getCompletionForImportPath.call(this, importPathNode, result);
  };
  SCSSCompletion2.prototype.createReplaceFunction = function() {
    var tabStopCounter = 1;
    return function(_match, p1) {
      return "\\" + p1 + ": ${" + tabStopCounter++ + ":" + (SCSSCompletion2.variableDefaults[p1] || "") + "}";
    };
  };
  SCSSCompletion2.prototype.createFunctionProposals = function(proposals, existingNode, sortToEnd, result) {
    for (var _i = 0, proposals_1 = proposals; _i < proposals_1.length; _i++) {
      var p = proposals_1[_i];
      var insertText = p.func.replace(/\[?(\$\w+)\]?/g, this.createReplaceFunction());
      var label = p.func.substr(0, p.func.indexOf("("));
      var item = {
        label,
        detail: p.func,
        documentation: p.desc,
        textEdit: TextEdit.replace(this.getCompletionRange(existingNode), insertText),
        insertTextFormat: InsertTextFormat.Snippet,
        kind: CompletionItemKind.Function
      };
      if (sortToEnd) {
        item.sortText = "z";
      }
      result.items.push(item);
    }
    return result;
  };
  SCSSCompletion2.prototype.getCompletionsForSelector = function(ruleSet, isNested, result) {
    this.createFunctionProposals(SCSSCompletion2.selectorFuncs, null, true, result);
    return _super.prototype.getCompletionsForSelector.call(this, ruleSet, isNested, result);
  };
  SCSSCompletion2.prototype.getTermProposals = function(entry, existingNode, result) {
    var functions = SCSSCompletion2.builtInFuncs;
    if (entry) {
      functions = functions.filter(function(f2) {
        return !f2.type || !entry.restrictions || entry.restrictions.indexOf(f2.type) !== -1;
      });
    }
    this.createFunctionProposals(functions, existingNode, true, result);
    return _super.prototype.getTermProposals.call(this, entry, existingNode, result);
  };
  SCSSCompletion2.prototype.getColorProposals = function(entry, existingNode, result) {
    this.createFunctionProposals(SCSSCompletion2.colorProposals, existingNode, false, result);
    return _super.prototype.getColorProposals.call(this, entry, existingNode, result);
  };
  SCSSCompletion2.prototype.getCompletionsForDeclarationProperty = function(declaration, result) {
    this.getCompletionForAtDirectives(result);
    this.getCompletionsForSelector(null, true, result);
    return _super.prototype.getCompletionsForDeclarationProperty.call(this, declaration, result);
  };
  SCSSCompletion2.prototype.getCompletionsForExtendsReference = function(_extendsRef, existingNode, result) {
    var symbols = this.getSymbolContext().findSymbolsAtOffset(this.offset, ReferenceType.Rule);
    for (var _i = 0, symbols_1 = symbols; _i < symbols_1.length; _i++) {
      var symbol = symbols_1[_i];
      var suggest = {
        label: symbol.name,
        textEdit: TextEdit.replace(this.getCompletionRange(existingNode), symbol.name),
        kind: CompletionItemKind.Function
      };
      result.items.push(suggest);
    }
    return result;
  };
  SCSSCompletion2.prototype.getCompletionForAtDirectives = function(result) {
    var _a2;
    (_a2 = result.items).push.apply(_a2, SCSSCompletion2.scssAtDirectives);
    return result;
  };
  SCSSCompletion2.prototype.getCompletionForTopLevel = function(result) {
    this.getCompletionForAtDirectives(result);
    this.getCompletionForModuleLoaders(result);
    _super.prototype.getCompletionForTopLevel.call(this, result);
    return result;
  };
  SCSSCompletion2.prototype.getCompletionForModuleLoaders = function(result) {
    var _a2;
    (_a2 = result.items).push.apply(_a2, SCSSCompletion2.scssModuleLoaders);
    return result;
  };
  SCSSCompletion2.variableDefaults = {
    "$red": "1",
    "$green": "2",
    "$blue": "3",
    "$alpha": "1.0",
    "$color": "#000000",
    "$weight": "0.5",
    "$hue": "0",
    "$saturation": "0%",
    "$lightness": "0%",
    "$degrees": "0",
    "$amount": "0",
    "$string": '""',
    "$substring": '"s"',
    "$number": "0",
    "$limit": "1"
  };
  SCSSCompletion2.colorProposals = [
    { func: "red($color)", desc: localize11("scss.builtin.red", "Gets the red component of a color.") },
    { func: "green($color)", desc: localize11("scss.builtin.green", "Gets the green component of a color.") },
    { func: "blue($color)", desc: localize11("scss.builtin.blue", "Gets the blue component of a color.") },
    { func: "mix($color, $color, [$weight])", desc: localize11("scss.builtin.mix", "Mixes two colors together.") },
    { func: "hue($color)", desc: localize11("scss.builtin.hue", "Gets the hue component of a color.") },
    { func: "saturation($color)", desc: localize11("scss.builtin.saturation", "Gets the saturation component of a color.") },
    { func: "lightness($color)", desc: localize11("scss.builtin.lightness", "Gets the lightness component of a color.") },
    { func: "adjust-hue($color, $degrees)", desc: localize11("scss.builtin.adjust-hue", "Changes the hue of a color.") },
    { func: "lighten($color, $amount)", desc: localize11("scss.builtin.lighten", "Makes a color lighter.") },
    { func: "darken($color, $amount)", desc: localize11("scss.builtin.darken", "Makes a color darker.") },
    { func: "saturate($color, $amount)", desc: localize11("scss.builtin.saturate", "Makes a color more saturated.") },
    { func: "desaturate($color, $amount)", desc: localize11("scss.builtin.desaturate", "Makes a color less saturated.") },
    { func: "grayscale($color)", desc: localize11("scss.builtin.grayscale", "Converts a color to grayscale.") },
    { func: "complement($color)", desc: localize11("scss.builtin.complement", "Returns the complement of a color.") },
    { func: "invert($color)", desc: localize11("scss.builtin.invert", "Returns the inverse of a color.") },
    { func: "alpha($color)", desc: localize11("scss.builtin.alpha", "Gets the opacity component of a color.") },
    { func: "opacity($color)", desc: "Gets the alpha component (opacity) of a color." },
    { func: "rgba($color, $alpha)", desc: localize11("scss.builtin.rgba", "Changes the alpha component for a color.") },
    { func: "opacify($color, $amount)", desc: localize11("scss.builtin.opacify", "Makes a color more opaque.") },
    { func: "fade-in($color, $amount)", desc: localize11("scss.builtin.fade-in", "Makes a color more opaque.") },
    { func: "transparentize($color, $amount)", desc: localize11("scss.builtin.transparentize", "Makes a color more transparent.") },
    { func: "fade-out($color, $amount)", desc: localize11("scss.builtin.fade-out", "Makes a color more transparent.") },
    { func: "adjust-color($color, [$red], [$green], [$blue], [$hue], [$saturation], [$lightness], [$alpha])", desc: localize11("scss.builtin.adjust-color", "Increases or decreases one or more components of a color.") },
    { func: "scale-color($color, [$red], [$green], [$blue], [$saturation], [$lightness], [$alpha])", desc: localize11("scss.builtin.scale-color", "Fluidly scales one or more properties of a color.") },
    { func: "change-color($color, [$red], [$green], [$blue], [$hue], [$saturation], [$lightness], [$alpha])", desc: localize11("scss.builtin.change-color", "Changes one or more properties of a color.") },
    { func: "ie-hex-str($color)", desc: localize11("scss.builtin.ie-hex-str", "Converts a color into the format understood by IE filters.") }
  ];
  SCSSCompletion2.selectorFuncs = [
    { func: "selector-nest($selectors\u2026)", desc: localize11("scss.builtin.selector-nest", "Nests selector beneath one another like they would be nested in the stylesheet.") },
    { func: "selector-append($selectors\u2026)", desc: localize11("scss.builtin.selector-append", "Appends selectors to one another without spaces in between.") },
    { func: "selector-extend($selector, $extendee, $extender)", desc: localize11("scss.builtin.selector-extend", "Extends $extendee with $extender within $selector.") },
    { func: "selector-replace($selector, $original, $replacement)", desc: localize11("scss.builtin.selector-replace", "Replaces $original with $replacement within $selector.") },
    { func: "selector-unify($selector1, $selector2)", desc: localize11("scss.builtin.selector-unify", "Unifies two selectors to produce a selector that matches elements matched by both.") },
    { func: "is-superselector($super, $sub)", desc: localize11("scss.builtin.is-superselector", "Returns whether $super matches all the elements $sub does, and possibly more.") },
    { func: "simple-selectors($selector)", desc: localize11("scss.builtin.simple-selectors", "Returns the simple selectors that comprise a compound selector.") },
    { func: "selector-parse($selector)", desc: localize11("scss.builtin.selector-parse", "Parses a selector into the format returned by &.") }
  ];
  SCSSCompletion2.builtInFuncs = [
    { func: "unquote($string)", desc: localize11("scss.builtin.unquote", "Removes quotes from a string.") },
    { func: "quote($string)", desc: localize11("scss.builtin.quote", "Adds quotes to a string.") },
    { func: "str-length($string)", desc: localize11("scss.builtin.str-length", "Returns the number of characters in a string.") },
    { func: "str-insert($string, $insert, $index)", desc: localize11("scss.builtin.str-insert", "Inserts $insert into $string at $index.") },
    { func: "str-index($string, $substring)", desc: localize11("scss.builtin.str-index", "Returns the index of the first occurance of $substring in $string.") },
    { func: "str-slice($string, $start-at, [$end-at])", desc: localize11("scss.builtin.str-slice", "Extracts a substring from $string.") },
    { func: "to-upper-case($string)", desc: localize11("scss.builtin.to-upper-case", "Converts a string to upper case.") },
    { func: "to-lower-case($string)", desc: localize11("scss.builtin.to-lower-case", "Converts a string to lower case.") },
    { func: "percentage($number)", desc: localize11("scss.builtin.percentage", "Converts a unitless number to a percentage."), type: "percentage" },
    { func: "round($number)", desc: localize11("scss.builtin.round", "Rounds a number to the nearest whole number.") },
    { func: "ceil($number)", desc: localize11("scss.builtin.ceil", "Rounds a number up to the next whole number.") },
    { func: "floor($number)", desc: localize11("scss.builtin.floor", "Rounds a number down to the previous whole number.") },
    { func: "abs($number)", desc: localize11("scss.builtin.abs", "Returns the absolute value of a number.") },
    { func: "min($numbers)", desc: localize11("scss.builtin.min", "Finds the minimum of several numbers.") },
    { func: "max($numbers)", desc: localize11("scss.builtin.max", "Finds the maximum of several numbers.") },
    { func: "random([$limit])", desc: localize11("scss.builtin.random", "Returns a random number.") },
    { func: "length($list)", desc: localize11("scss.builtin.length", "Returns the length of a list.") },
    { func: "nth($list, $n)", desc: localize11("scss.builtin.nth", "Returns a specific item in a list.") },
    { func: "set-nth($list, $n, $value)", desc: localize11("scss.builtin.set-nth", "Replaces the nth item in a list.") },
    { func: "join($list1, $list2, [$separator])", desc: localize11("scss.builtin.join", "Joins together two lists into one.") },
    { func: "append($list1, $val, [$separator])", desc: localize11("scss.builtin.append", "Appends a single value onto the end of a list.") },
    { func: "zip($lists)", desc: localize11("scss.builtin.zip", "Combines several lists into a single multidimensional list.") },
    { func: "index($list, $value)", desc: localize11("scss.builtin.index", "Returns the position of a value within a list.") },
    { func: "list-separator(#list)", desc: localize11("scss.builtin.list-separator", "Returns the separator of a list.") },
    { func: "map-get($map, $key)", desc: localize11("scss.builtin.map-get", "Returns the value in a map associated with a given key.") },
    { func: "map-merge($map1, $map2)", desc: localize11("scss.builtin.map-merge", "Merges two maps together into a new map.") },
    { func: "map-remove($map, $keys)", desc: localize11("scss.builtin.map-remove", "Returns a new map with keys removed.") },
    { func: "map-keys($map)", desc: localize11("scss.builtin.map-keys", "Returns a list of all keys in a map.") },
    { func: "map-values($map)", desc: localize11("scss.builtin.map-values", "Returns a list of all values in a map.") },
    { func: "map-has-key($map, $key)", desc: localize11("scss.builtin.map-has-key", "Returns whether a map has a value associated with a given key.") },
    { func: "keywords($args)", desc: localize11("scss.builtin.keywords", "Returns the keywords passed to a function that takes variable arguments.") },
    { func: "feature-exists($feature)", desc: localize11("scss.builtin.feature-exists", "Returns whether a feature exists in the current Sass runtime.") },
    { func: "variable-exists($name)", desc: localize11("scss.builtin.variable-exists", "Returns whether a variable with the given name exists in the current scope.") },
    { func: "global-variable-exists($name)", desc: localize11("scss.builtin.global-variable-exists", "Returns whether a variable with the given name exists in the global scope.") },
    { func: "function-exists($name)", desc: localize11("scss.builtin.function-exists", "Returns whether a function with the given name exists.") },
    { func: "mixin-exists($name)", desc: localize11("scss.builtin.mixin-exists", "Returns whether a mixin with the given name exists.") },
    { func: "inspect($value)", desc: localize11("scss.builtin.inspect", "Returns the string representation of a value as it would be represented in Sass.") },
    { func: "type-of($value)", desc: localize11("scss.builtin.type-of", "Returns the type of a value.") },
    { func: "unit($number)", desc: localize11("scss.builtin.unit", "Returns the unit(s) associated with a number.") },
    { func: "unitless($number)", desc: localize11("scss.builtin.unitless", "Returns whether a number has units.") },
    { func: "comparable($number1, $number2)", desc: localize11("scss.builtin.comparable", "Returns whether two numbers can be added, subtracted, or compared.") },
    { func: "call($name, $args\u2026)", desc: localize11("scss.builtin.call", "Dynamically calls a Sass function.") }
  ];
  SCSSCompletion2.scssAtDirectives = [
    {
      label: "@extend",
      documentation: localize11("scss.builtin.@extend", "Inherits the styles of another selector."),
      kind: CompletionItemKind.Keyword
    },
    {
      label: "@at-root",
      documentation: localize11("scss.builtin.@at-root", "Causes one or more rules to be emitted at the root of the document."),
      kind: CompletionItemKind.Keyword
    },
    {
      label: "@debug",
      documentation: localize11("scss.builtin.@debug", "Prints the value of an expression to the standard error output stream. Useful for debugging complicated Sass files."),
      kind: CompletionItemKind.Keyword
    },
    {
      label: "@warn",
      documentation: localize11("scss.builtin.@warn", "Prints the value of an expression to the standard error output stream. Useful for libraries that need to warn users of deprecations or recovering from minor mixin usage mistakes. Warnings can be turned off with the `--quiet` command-line option or the `:quiet` Sass option."),
      kind: CompletionItemKind.Keyword
    },
    {
      label: "@error",
      documentation: localize11("scss.builtin.@error", "Throws the value of an expression as a fatal error with stack trace. Useful for validating arguments to mixins and functions."),
      kind: CompletionItemKind.Keyword
    },
    {
      label: "@if",
      documentation: localize11("scss.builtin.@if", "Includes the body if the expression does not evaluate to `false` or `null`."),
      insertText: "@if ${1:expr} {\n	$0\n}",
      insertTextFormat: InsertTextFormat.Snippet,
      kind: CompletionItemKind.Keyword
    },
    {
      label: "@for",
      documentation: localize11("scss.builtin.@for", "For loop that repeatedly outputs a set of styles for each `$var` in the `from/through` or `from/to` clause."),
      insertText: "@for \\$${1:var} from ${2:start} ${3|to,through|} ${4:end} {\n	$0\n}",
      insertTextFormat: InsertTextFormat.Snippet,
      kind: CompletionItemKind.Keyword
    },
    {
      label: "@each",
      documentation: localize11("scss.builtin.@each", "Each loop that sets `$var` to each item in the list or map, then outputs the styles it contains using that value of `$var`."),
      insertText: "@each \\$${1:var} in ${2:list} {\n	$0\n}",
      insertTextFormat: InsertTextFormat.Snippet,
      kind: CompletionItemKind.Keyword
    },
    {
      label: "@while",
      documentation: localize11("scss.builtin.@while", "While loop that takes an expression and repeatedly outputs the nested styles until the statement evaluates to `false`."),
      insertText: "@while ${1:condition} {\n	$0\n}",
      insertTextFormat: InsertTextFormat.Snippet,
      kind: CompletionItemKind.Keyword
    },
    {
      label: "@mixin",
      documentation: localize11("scss.builtin.@mixin", "Defines styles that can be re-used throughout the stylesheet with `@include`."),
      insertText: "@mixin ${1:name} {\n	$0\n}",
      insertTextFormat: InsertTextFormat.Snippet,
      kind: CompletionItemKind.Keyword
    },
    {
      label: "@include",
      documentation: localize11("scss.builtin.@include", "Includes the styles defined by another mixin into the current rule."),
      kind: CompletionItemKind.Keyword
    },
    {
      label: "@function",
      documentation: localize11("scss.builtin.@function", "Defines complex operations that can be re-used throughout stylesheets."),
      kind: CompletionItemKind.Keyword
    }
  ];
  SCSSCompletion2.scssModuleLoaders = [
    {
      label: "@use",
      documentation: localize11("scss.builtin.@use", "Loads mixins, functions, and variables from other Sass stylesheets as 'modules', and combines CSS from multiple stylesheets together."),
      references: [{ name: "Sass documentation", url: "https://sass-lang.com/documentation/at-rules/use" }],
      insertText: "@use $0;",
      insertTextFormat: InsertTextFormat.Snippet,
      kind: CompletionItemKind.Keyword
    },
    {
      label: "@forward",
      documentation: localize11("scss.builtin.@forward", "Loads a Sass stylesheet and makes its mixins, functions, and variables available when this stylesheet is loaded with the @use rule."),
      references: [{ name: "Sass documentation", url: "https://sass-lang.com/documentation/at-rules/forward" }],
      insertText: "@forward $0;",
      insertTextFormat: InsertTextFormat.Snippet,
      kind: CompletionItemKind.Keyword
    }
  ];
  SCSSCompletion2.scssModuleBuiltIns = [
    {
      label: "sass:math",
      documentation: localize11("scss.builtin.sass:math", "Provides functions that operate on numbers."),
      references: [{ name: "Sass documentation", url: "https://sass-lang.com/documentation/modules/math" }]
    },
    {
      label: "sass:string",
      documentation: localize11("scss.builtin.sass:string", "Makes it easy to combine, search, or split apart strings."),
      references: [{ name: "Sass documentation", url: "https://sass-lang.com/documentation/modules/string" }]
    },
    {
      label: "sass:color",
      documentation: localize11("scss.builtin.sass:color", "Generates new colors based on existing ones, making it easy to build color themes."),
      references: [{ name: "Sass documentation", url: "https://sass-lang.com/documentation/modules/color" }]
    },
    {
      label: "sass:list",
      documentation: localize11("scss.builtin.sass:list", "Lets you access and modify values in lists."),
      references: [{ name: "Sass documentation", url: "https://sass-lang.com/documentation/modules/list" }]
    },
    {
      label: "sass:map",
      documentation: localize11("scss.builtin.sass:map", "Makes it possible to look up the value associated with a key in a map, and much more."),
      references: [{ name: "Sass documentation", url: "https://sass-lang.com/documentation/modules/map" }]
    },
    {
      label: "sass:selector",
      documentation: localize11("scss.builtin.sass:selector", "Provides access to Sass\u2019s powerful selector engine."),
      references: [{ name: "Sass documentation", url: "https://sass-lang.com/documentation/modules/selector" }]
    },
    {
      label: "sass:meta",
      documentation: localize11("scss.builtin.sass:meta", "Exposes the details of Sass\u2019s inner workings."),
      references: [{ name: "Sass documentation", url: "https://sass-lang.com/documentation/modules/meta" }]
    }
  ];
  return SCSSCompletion2;
}(CSSCompletion);
function addReferencesToDocumentation(items) {
  items.forEach(function(i) {
    if (i.documentation && i.references && i.references.length > 0) {
      var markdownDoc = typeof i.documentation === "string" ? { kind: "markdown", value: i.documentation } : { kind: "markdown", value: i.documentation.value };
      markdownDoc.value += "\n\n";
      markdownDoc.value += i.references.map(function(r) {
        return "[".concat(r.name, "](").concat(r.url, ")");
      }).join(" | ");
      i.documentation = markdownDoc;
    }
  });
}

// node_modules/vscode-css-languageservice/lib/esm/parser/lessScanner.js
var __extends7 = function() {
  var extendStatics = function(d, b) {
    extendStatics = Object.setPrototypeOf || { __proto__: [] } instanceof Array && function(d2, b2) {
      d2.__proto__ = b2;
    } || function(d2, b2) {
      for (var p in b2)
        if (Object.prototype.hasOwnProperty.call(b2, p))
          d2[p] = b2[p];
    };
    return extendStatics(d, b);
  };
  return function(d, b) {
    if (typeof b !== "function" && b !== null)
      throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
    extendStatics(d, b);
    function __() {
      this.constructor = d;
    }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
  };
}();
var _FSL3 = "/".charCodeAt(0);
var _NWL3 = "\n".charCodeAt(0);
var _CAR3 = "\r".charCodeAt(0);
var _LFD3 = "\f".charCodeAt(0);
var _TIC = "`".charCodeAt(0);
var _DOT3 = ".".charCodeAt(0);
var customTokenValue2 = TokenType.CustomToken;
var Ellipsis2 = customTokenValue2++;
var LESSScanner = function(_super) {
  __extends7(LESSScanner2, _super);
  function LESSScanner2() {
    return _super !== null && _super.apply(this, arguments) || this;
  }
  LESSScanner2.prototype.scanNext = function(offset) {
    var tokenType = this.escapedJavaScript();
    if (tokenType !== null) {
      return this.finishToken(offset, tokenType);
    }
    if (this.stream.advanceIfChars([_DOT3, _DOT3, _DOT3])) {
      return this.finishToken(offset, Ellipsis2);
    }
    return _super.prototype.scanNext.call(this, offset);
  };
  LESSScanner2.prototype.comment = function() {
    if (_super.prototype.comment.call(this)) {
      return true;
    }
    if (!this.inURL && this.stream.advanceIfChars([_FSL3, _FSL3])) {
      this.stream.advanceWhileChar(function(ch) {
        switch (ch) {
          case _NWL3:
          case _CAR3:
          case _LFD3:
            return false;
          default:
            return true;
        }
      });
      return true;
    } else {
      return false;
    }
  };
  LESSScanner2.prototype.escapedJavaScript = function() {
    var ch = this.stream.peekChar();
    if (ch === _TIC) {
      this.stream.advance(1);
      this.stream.advanceWhileChar(function(ch2) {
        return ch2 !== _TIC;
      });
      return this.stream.advanceIfChar(_TIC) ? TokenType.EscapedJavaScript : TokenType.BadEscapedJavaScript;
    }
    return null;
  };
  return LESSScanner2;
}(Scanner);

// node_modules/vscode-css-languageservice/lib/esm/parser/lessParser.js
var __extends8 = function() {
  var extendStatics = function(d, b) {
    extendStatics = Object.setPrototypeOf || { __proto__: [] } instanceof Array && function(d2, b2) {
      d2.__proto__ = b2;
    } || function(d2, b2) {
      for (var p in b2)
        if (Object.prototype.hasOwnProperty.call(b2, p))
          d2[p] = b2[p];
    };
    return extendStatics(d, b);
  };
  return function(d, b) {
    if (typeof b !== "function" && b !== null)
      throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
    extendStatics(d, b);
    function __() {
      this.constructor = d;
    }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
  };
}();
var LESSParser = function(_super) {
  __extends8(LESSParser2, _super);
  function LESSParser2() {
    return _super.call(this, new LESSScanner()) || this;
  }
  LESSParser2.prototype._parseStylesheetStatement = function(isNested) {
    if (isNested === void 0) {
      isNested = false;
    }
    if (this.peek(TokenType.AtKeyword)) {
      return this._parseVariableDeclaration() || this._parsePlugin() || _super.prototype._parseStylesheetAtStatement.call(this, isNested);
    }
    return this._tryParseMixinDeclaration() || this._tryParseMixinReference() || this._parseFunction() || this._parseRuleset(true);
  };
  LESSParser2.prototype._parseImport = function() {
    if (!this.peekKeyword("@import") && !this.peekKeyword("@import-once")) {
      return null;
    }
    var node = this.create(Import);
    this.consumeToken();
    if (this.accept(TokenType.ParenthesisL)) {
      if (!this.accept(TokenType.Ident)) {
        return this.finish(node, ParseError.IdentifierExpected, [TokenType.SemiColon]);
      }
      do {
        if (!this.accept(TokenType.Comma)) {
          break;
        }
      } while (this.accept(TokenType.Ident));
      if (!this.accept(TokenType.ParenthesisR)) {
        return this.finish(node, ParseError.RightParenthesisExpected, [TokenType.SemiColon]);
      }
    }
    if (!node.addChild(this._parseURILiteral()) && !node.addChild(this._parseStringLiteral())) {
      return this.finish(node, ParseError.URIOrStringExpected, [TokenType.SemiColon]);
    }
    if (!this.peek(TokenType.SemiColon) && !this.peek(TokenType.EOF)) {
      node.setMedialist(this._parseMediaQueryList());
    }
    return this.finish(node);
  };
  LESSParser2.prototype._parsePlugin = function() {
    if (!this.peekKeyword("@plugin")) {
      return null;
    }
    var node = this.createNode(NodeType.Plugin);
    this.consumeToken();
    if (!node.addChild(this._parseStringLiteral())) {
      return this.finish(node, ParseError.StringLiteralExpected);
    }
    if (!this.accept(TokenType.SemiColon)) {
      return this.finish(node, ParseError.SemiColonExpected);
    }
    return this.finish(node);
  };
  LESSParser2.prototype._parseMediaQuery = function() {
    var node = _super.prototype._parseMediaQuery.call(this);
    if (!node) {
      var node_1 = this.create(MediaQuery);
      if (node_1.addChild(this._parseVariable())) {
        return this.finish(node_1);
      }
      return null;
    }
    return node;
  };
  LESSParser2.prototype._parseMediaDeclaration = function(isNested) {
    if (isNested === void 0) {
      isNested = false;
    }
    return this._tryParseRuleset(isNested) || this._tryToParseDeclaration() || this._tryParseMixinDeclaration() || this._tryParseMixinReference() || this._parseDetachedRuleSetMixin() || this._parseStylesheetStatement(isNested);
  };
  LESSParser2.prototype._parseMediaFeatureName = function() {
    return this._parseIdent() || this._parseVariable();
  };
  LESSParser2.prototype._parseVariableDeclaration = function(panic) {
    if (panic === void 0) {
      panic = [];
    }
    var node = this.create(VariableDeclaration);
    var mark = this.mark();
    if (!node.setVariable(this._parseVariable(true))) {
      return null;
    }
    if (this.accept(TokenType.Colon)) {
      if (this.prevToken) {
        node.colonPosition = this.prevToken.offset;
      }
      if (node.setValue(this._parseDetachedRuleSet())) {
        node.needsSemicolon = false;
      } else if (!node.setValue(this._parseExpr())) {
        return this.finish(node, ParseError.VariableValueExpected, [], panic);
      }
      node.addChild(this._parsePrio());
    } else {
      this.restoreAtMark(mark);
      return null;
    }
    if (this.peek(TokenType.SemiColon)) {
      node.semicolonPosition = this.token.offset;
    }
    return this.finish(node);
  };
  LESSParser2.prototype._parseDetachedRuleSet = function() {
    var mark = this.mark();
    if (this.peekDelim("#") || this.peekDelim(".")) {
      this.consumeToken();
      if (!this.hasWhitespace() && this.accept(TokenType.ParenthesisL)) {
        var node = this.create(MixinDeclaration);
        if (node.getParameters().addChild(this._parseMixinParameter())) {
          while (this.accept(TokenType.Comma) || this.accept(TokenType.SemiColon)) {
            if (this.peek(TokenType.ParenthesisR)) {
              break;
            }
            if (!node.getParameters().addChild(this._parseMixinParameter())) {
              this.markError(node, ParseError.IdentifierExpected, [], [TokenType.ParenthesisR]);
            }
          }
        }
        if (!this.accept(TokenType.ParenthesisR)) {
          this.restoreAtMark(mark);
          return null;
        }
      } else {
        this.restoreAtMark(mark);
        return null;
      }
    }
    if (!this.peek(TokenType.CurlyL)) {
      return null;
    }
    var content = this.create(BodyDeclaration);
    this._parseBody(content, this._parseDetachedRuleSetBody.bind(this));
    return this.finish(content);
  };
  LESSParser2.prototype._parseDetachedRuleSetBody = function() {
    return this._tryParseKeyframeSelector() || this._parseRuleSetDeclaration();
  };
  LESSParser2.prototype._addLookupChildren = function(node) {
    if (!node.addChild(this._parseLookupValue())) {
      return false;
    }
    var expectsValue = false;
    while (true) {
      if (this.peek(TokenType.BracketL)) {
        expectsValue = true;
      }
      if (!node.addChild(this._parseLookupValue())) {
        break;
      }
      expectsValue = false;
    }
    return !expectsValue;
  };
  LESSParser2.prototype._parseLookupValue = function() {
    var node = this.create(Node);
    var mark = this.mark();
    if (!this.accept(TokenType.BracketL)) {
      this.restoreAtMark(mark);
      return null;
    }
    if ((node.addChild(this._parseVariable(false, true)) || node.addChild(this._parsePropertyIdentifier())) && this.accept(TokenType.BracketR) || this.accept(TokenType.BracketR)) {
      return node;
    }
    this.restoreAtMark(mark);
    return null;
  };
  LESSParser2.prototype._parseVariable = function(declaration, insideLookup) {
    if (declaration === void 0) {
      declaration = false;
    }
    if (insideLookup === void 0) {
      insideLookup = false;
    }
    var isPropertyReference = !declaration && this.peekDelim("$");
    if (!this.peekDelim("@") && !isPropertyReference && !this.peek(TokenType.AtKeyword)) {
      return null;
    }
    var node = this.create(Variable);
    var mark = this.mark();
    while (this.acceptDelim("@") || !declaration && this.acceptDelim("$")) {
      if (this.hasWhitespace()) {
        this.restoreAtMark(mark);
        return null;
      }
    }
    if (!this.accept(TokenType.AtKeyword) && !this.accept(TokenType.Ident)) {
      this.restoreAtMark(mark);
      return null;
    }
    if (!insideLookup && this.peek(TokenType.BracketL)) {
      if (!this._addLookupChildren(node)) {
        this.restoreAtMark(mark);
        return null;
      }
    }
    return node;
  };
  LESSParser2.prototype._parseTermExpression = function() {
    return this._parseVariable() || this._parseEscaped() || _super.prototype._parseTermExpression.call(this) || this._tryParseMixinReference(false);
  };
  LESSParser2.prototype._parseEscaped = function() {
    if (this.peek(TokenType.EscapedJavaScript) || this.peek(TokenType.BadEscapedJavaScript)) {
      var node = this.createNode(NodeType.EscapedValue);
      this.consumeToken();
      return this.finish(node);
    }
    if (this.peekDelim("~")) {
      var node = this.createNode(NodeType.EscapedValue);
      this.consumeToken();
      if (this.accept(TokenType.String) || this.accept(TokenType.EscapedJavaScript)) {
        return this.finish(node);
      } else {
        return this.finish(node, ParseError.TermExpected);
      }
    }
    return null;
  };
  LESSParser2.prototype._parseOperator = function() {
    var node = this._parseGuardOperator();
    if (node) {
      return node;
    } else {
      return _super.prototype._parseOperator.call(this);
    }
  };
  LESSParser2.prototype._parseGuardOperator = function() {
    if (this.peekDelim(">")) {
      var node = this.createNode(NodeType.Operator);
      this.consumeToken();
      this.acceptDelim("=");
      return node;
    } else if (this.peekDelim("=")) {
      var node = this.createNode(NodeType.Operator);
      this.consumeToken();
      this.acceptDelim("<");
      return node;
    } else if (this.peekDelim("<")) {
      var node = this.createNode(NodeType.Operator);
      this.consumeToken();
      this.acceptDelim("=");
      return node;
    }
    return null;
  };
  LESSParser2.prototype._parseRuleSetDeclaration = function() {
    if (this.peek(TokenType.AtKeyword)) {
      return this._parseKeyframe() || this._parseMedia(true) || this._parseImport() || this._parseSupports(true) || this._parseDetachedRuleSetMixin() || this._parseVariableDeclaration() || _super.prototype._parseRuleSetDeclarationAtStatement.call(this);
    }
    return this._tryParseMixinDeclaration() || this._tryParseRuleset(true) || this._tryParseMixinReference() || this._parseFunction() || this._parseExtend() || _super.prototype._parseRuleSetDeclaration.call(this);
  };
  LESSParser2.prototype._parseKeyframeIdent = function() {
    return this._parseIdent([ReferenceType.Keyframe]) || this._parseVariable();
  };
  LESSParser2.prototype._parseKeyframeSelector = function() {
    return this._parseDetachedRuleSetMixin() || _super.prototype._parseKeyframeSelector.call(this);
  };
  LESSParser2.prototype._parseSimpleSelectorBody = function() {
    return this._parseSelectorCombinator() || _super.prototype._parseSimpleSelectorBody.call(this);
  };
  LESSParser2.prototype._parseSelector = function(isNested) {
    var node = this.create(Selector);
    var hasContent = false;
    if (isNested) {
      hasContent = node.addChild(this._parseCombinator());
    }
    while (node.addChild(this._parseSimpleSelector())) {
      hasContent = true;
      var mark = this.mark();
      if (node.addChild(this._parseGuard()) && this.peek(TokenType.CurlyL)) {
        break;
      }
      this.restoreAtMark(mark);
      node.addChild(this._parseCombinator());
    }
    return hasContent ? this.finish(node) : null;
  };
  LESSParser2.prototype._parseSelectorCombinator = function() {
    if (this.peekDelim("&")) {
      var node = this.createNode(NodeType.SelectorCombinator);
      this.consumeToken();
      while (!this.hasWhitespace() && (this.acceptDelim("-") || this.accept(TokenType.Num) || this.accept(TokenType.Dimension) || node.addChild(this._parseIdent()) || this.acceptDelim("&"))) {
      }
      return this.finish(node);
    }
    return null;
  };
  LESSParser2.prototype._parseSelectorIdent = function() {
    if (!this.peekInterpolatedIdent()) {
      return null;
    }
    var node = this.createNode(NodeType.SelectorInterpolation);
    var hasContent = this._acceptInterpolatedIdent(node);
    return hasContent ? this.finish(node) : null;
  };
  LESSParser2.prototype._parsePropertyIdentifier = function(inLookup) {
    if (inLookup === void 0) {
      inLookup = false;
    }
    var propertyRegex = /^[\w-]+/;
    if (!this.peekInterpolatedIdent() && !this.peekRegExp(this.token.type, propertyRegex)) {
      return null;
    }
    var mark = this.mark();
    var node = this.create(Identifier);
    node.isCustomProperty = this.acceptDelim("-") && this.acceptDelim("-");
    var childAdded = false;
    if (!inLookup) {
      if (node.isCustomProperty) {
        childAdded = this._acceptInterpolatedIdent(node);
      } else {
        childAdded = this._acceptInterpolatedIdent(node, propertyRegex);
      }
    } else {
      if (node.isCustomProperty) {
        childAdded = node.addChild(this._parseIdent());
      } else {
        childAdded = node.addChild(this._parseRegexp(propertyRegex));
      }
    }
    if (!childAdded) {
      this.restoreAtMark(mark);
      return null;
    }
    if (!inLookup && !this.hasWhitespace()) {
      this.acceptDelim("+");
      if (!this.hasWhitespace()) {
        this.acceptIdent("_");
      }
    }
    return this.finish(node);
  };
  LESSParser2.prototype.peekInterpolatedIdent = function() {
    return this.peek(TokenType.Ident) || this.peekDelim("@") || this.peekDelim("$") || this.peekDelim("-");
  };
  LESSParser2.prototype._acceptInterpolatedIdent = function(node, identRegex) {
    var _this = this;
    var hasContent = false;
    var indentInterpolation = function() {
      var pos = _this.mark();
      if (_this.acceptDelim("-")) {
        if (!_this.hasWhitespace()) {
          _this.acceptDelim("-");
        }
        if (_this.hasWhitespace()) {
          _this.restoreAtMark(pos);
          return null;
        }
      }
      return _this._parseInterpolation();
    };
    var accept = identRegex ? function() {
      return _this.acceptRegexp(identRegex);
    } : function() {
      return _this.accept(TokenType.Ident);
    };
    while (accept() || node.addChild(this._parseInterpolation() || this.try(indentInterpolation))) {
      hasContent = true;
      if (this.hasWhitespace()) {
        break;
      }
    }
    return hasContent;
  };
  LESSParser2.prototype._parseInterpolation = function() {
    var mark = this.mark();
    if (this.peekDelim("@") || this.peekDelim("$")) {
      var node = this.createNode(NodeType.Interpolation);
      this.consumeToken();
      if (this.hasWhitespace() || !this.accept(TokenType.CurlyL)) {
        this.restoreAtMark(mark);
        return null;
      }
      if (!node.addChild(this._parseIdent())) {
        return this.finish(node, ParseError.IdentifierExpected);
      }
      if (!this.accept(TokenType.CurlyR)) {
        return this.finish(node, ParseError.RightCurlyExpected);
      }
      return this.finish(node);
    }
    return null;
  };
  LESSParser2.prototype._tryParseMixinDeclaration = function() {
    var mark = this.mark();
    var node = this.create(MixinDeclaration);
    if (!node.setIdentifier(this._parseMixinDeclarationIdentifier()) || !this.accept(TokenType.ParenthesisL)) {
      this.restoreAtMark(mark);
      return null;
    }
    if (node.getParameters().addChild(this._parseMixinParameter())) {
      while (this.accept(TokenType.Comma) || this.accept(TokenType.SemiColon)) {
        if (this.peek(TokenType.ParenthesisR)) {
          break;
        }
        if (!node.getParameters().addChild(this._parseMixinParameter())) {
          this.markError(node, ParseError.IdentifierExpected, [], [TokenType.ParenthesisR]);
        }
      }
    }
    if (!this.accept(TokenType.ParenthesisR)) {
      this.restoreAtMark(mark);
      return null;
    }
    node.setGuard(this._parseGuard());
    if (!this.peek(TokenType.CurlyL)) {
      this.restoreAtMark(mark);
      return null;
    }
    return this._parseBody(node, this._parseMixInBodyDeclaration.bind(this));
  };
  LESSParser2.prototype._parseMixInBodyDeclaration = function() {
    return this._parseFontFace() || this._parseRuleSetDeclaration();
  };
  LESSParser2.prototype._parseMixinDeclarationIdentifier = function() {
    var identifier;
    if (this.peekDelim("#") || this.peekDelim(".")) {
      identifier = this.create(Identifier);
      this.consumeToken();
      if (this.hasWhitespace() || !identifier.addChild(this._parseIdent())) {
        return null;
      }
    } else if (this.peek(TokenType.Hash)) {
      identifier = this.create(Identifier);
      this.consumeToken();
    } else {
      return null;
    }
    identifier.referenceTypes = [ReferenceType.Mixin];
    return this.finish(identifier);
  };
  LESSParser2.prototype._parsePseudo = function() {
    if (!this.peek(TokenType.Colon)) {
      return null;
    }
    var mark = this.mark();
    var node = this.create(ExtendsReference);
    this.consumeToken();
    if (this.acceptIdent("extend")) {
      return this._completeExtends(node);
    }
    this.restoreAtMark(mark);
    return _super.prototype._parsePseudo.call(this);
  };
  LESSParser2.prototype._parseExtend = function() {
    if (!this.peekDelim("&")) {
      return null;
    }
    var mark = this.mark();
    var node = this.create(ExtendsReference);
    this.consumeToken();
    if (this.hasWhitespace() || !this.accept(TokenType.Colon) || !this.acceptIdent("extend")) {
      this.restoreAtMark(mark);
      return null;
    }
    return this._completeExtends(node);
  };
  LESSParser2.prototype._completeExtends = function(node) {
    if (!this.accept(TokenType.ParenthesisL)) {
      return this.finish(node, ParseError.LeftParenthesisExpected);
    }
    var selectors = node.getSelectors();
    if (!selectors.addChild(this._parseSelector(true))) {
      return this.finish(node, ParseError.SelectorExpected);
    }
    while (this.accept(TokenType.Comma)) {
      if (!selectors.addChild(this._parseSelector(true))) {
        return this.finish(node, ParseError.SelectorExpected);
      }
    }
    if (!this.accept(TokenType.ParenthesisR)) {
      return this.finish(node, ParseError.RightParenthesisExpected);
    }
    return this.finish(node);
  };
  LESSParser2.prototype._parseDetachedRuleSetMixin = function() {
    if (!this.peek(TokenType.AtKeyword)) {
      return null;
    }
    var mark = this.mark();
    var node = this.create(MixinReference);
    if (node.addChild(this._parseVariable(true)) && (this.hasWhitespace() || !this.accept(TokenType.ParenthesisL))) {
      this.restoreAtMark(mark);
      return null;
    }
    if (!this.accept(TokenType.ParenthesisR)) {
      return this.finish(node, ParseError.RightParenthesisExpected);
    }
    return this.finish(node);
  };
  LESSParser2.prototype._tryParseMixinReference = function(atRoot) {
    if (atRoot === void 0) {
      atRoot = true;
    }
    var mark = this.mark();
    var node = this.create(MixinReference);
    var identifier = this._parseMixinDeclarationIdentifier();
    while (identifier) {
      this.acceptDelim(">");
      var nextId = this._parseMixinDeclarationIdentifier();
      if (nextId) {
        node.getNamespaces().addChild(identifier);
        identifier = nextId;
      } else {
        break;
      }
    }
    if (!node.setIdentifier(identifier)) {
      this.restoreAtMark(mark);
      return null;
    }
    var hasArguments = false;
    if (this.accept(TokenType.ParenthesisL)) {
      hasArguments = true;
      if (node.getArguments().addChild(this._parseMixinArgument())) {
        while (this.accept(TokenType.Comma) || this.accept(TokenType.SemiColon)) {
          if (this.peek(TokenType.ParenthesisR)) {
            break;
          }
          if (!node.getArguments().addChild(this._parseMixinArgument())) {
            return this.finish(node, ParseError.ExpressionExpected);
          }
        }
      }
      if (!this.accept(TokenType.ParenthesisR)) {
        return this.finish(node, ParseError.RightParenthesisExpected);
      }
      identifier.referenceTypes = [ReferenceType.Mixin];
    } else {
      identifier.referenceTypes = [ReferenceType.Mixin, ReferenceType.Rule];
    }
    if (this.peek(TokenType.BracketL)) {
      if (!atRoot) {
        this._addLookupChildren(node);
      }
    } else {
      node.addChild(this._parsePrio());
    }
    if (!hasArguments && !this.peek(TokenType.SemiColon) && !this.peek(TokenType.CurlyR) && !this.peek(TokenType.EOF)) {
      this.restoreAtMark(mark);
      return null;
    }
    return this.finish(node);
  };
  LESSParser2.prototype._parseMixinArgument = function() {
    var node = this.create(FunctionArgument);
    var pos = this.mark();
    var argument = this._parseVariable();
    if (argument) {
      if (!this.accept(TokenType.Colon)) {
        this.restoreAtMark(pos);
      } else {
        node.setIdentifier(argument);
      }
    }
    if (node.setValue(this._parseDetachedRuleSet() || this._parseExpr(true))) {
      return this.finish(node);
    }
    this.restoreAtMark(pos);
    return null;
  };
  LESSParser2.prototype._parseMixinParameter = function() {
    var node = this.create(FunctionParameter);
    if (this.peekKeyword("@rest")) {
      var restNode = this.create(Node);
      this.consumeToken();
      if (!this.accept(Ellipsis2)) {
        return this.finish(node, ParseError.DotExpected, [], [TokenType.Comma, TokenType.ParenthesisR]);
      }
      node.setIdentifier(this.finish(restNode));
      return this.finish(node);
    }
    if (this.peek(Ellipsis2)) {
      var varargsNode = this.create(Node);
      this.consumeToken();
      node.setIdentifier(this.finish(varargsNode));
      return this.finish(node);
    }
    var hasContent = false;
    if (node.setIdentifier(this._parseVariable())) {
      this.accept(TokenType.Colon);
      hasContent = true;
    }
    if (!node.setDefaultValue(this._parseDetachedRuleSet() || this._parseExpr(true)) && !hasContent) {
      return null;
    }
    return this.finish(node);
  };
  LESSParser2.prototype._parseGuard = function() {
    if (!this.peekIdent("when")) {
      return null;
    }
    var node = this.create(LessGuard);
    this.consumeToken();
    node.isNegated = this.acceptIdent("not");
    if (!node.getConditions().addChild(this._parseGuardCondition())) {
      return this.finish(node, ParseError.ConditionExpected);
    }
    while (this.acceptIdent("and") || this.accept(TokenType.Comma)) {
      if (!node.getConditions().addChild(this._parseGuardCondition())) {
        return this.finish(node, ParseError.ConditionExpected);
      }
    }
    return this.finish(node);
  };
  LESSParser2.prototype._parseGuardCondition = function() {
    if (!this.peek(TokenType.ParenthesisL)) {
      return null;
    }
    var node = this.create(GuardCondition);
    this.consumeToken();
    if (!node.addChild(this._parseExpr())) {
    }
    if (!this.accept(TokenType.ParenthesisR)) {
      return this.finish(node, ParseError.RightParenthesisExpected);
    }
    return this.finish(node);
  };
  LESSParser2.prototype._parseFunction = function() {
    var pos = this.mark();
    var node = this.create(Function);
    if (!node.setIdentifier(this._parseFunctionIdentifier())) {
      return null;
    }
    if (this.hasWhitespace() || !this.accept(TokenType.ParenthesisL)) {
      this.restoreAtMark(pos);
      return null;
    }
    if (node.getArguments().addChild(this._parseMixinArgument())) {
      while (this.accept(TokenType.Comma) || this.accept(TokenType.SemiColon)) {
        if (this.peek(TokenType.ParenthesisR)) {
          break;
        }
        if (!node.getArguments().addChild(this._parseMixinArgument())) {
          return this.finish(node, ParseError.ExpressionExpected);
        }
      }
    }
    if (!this.accept(TokenType.ParenthesisR)) {
      return this.finish(node, ParseError.RightParenthesisExpected);
    }
    return this.finish(node);
  };
  LESSParser2.prototype._parseFunctionIdentifier = function() {
    if (this.peekDelim("%")) {
      var node = this.create(Identifier);
      node.referenceTypes = [ReferenceType.Function];
      this.consumeToken();
      return this.finish(node);
    }
    return _super.prototype._parseFunctionIdentifier.call(this);
  };
  LESSParser2.prototype._parseURLArgument = function() {
    var pos = this.mark();
    var node = _super.prototype._parseURLArgument.call(this);
    if (!node || !this.peek(TokenType.ParenthesisR)) {
      this.restoreAtMark(pos);
      var node_2 = this.create(Node);
      node_2.addChild(this._parseBinaryExpr());
      return this.finish(node_2);
    }
    return node;
  };
  return LESSParser2;
}(Parser);

// node_modules/vscode-css-languageservice/lib/esm/services/lessCompletion.js
var __extends9 = function() {
  var extendStatics = function(d, b) {
    extendStatics = Object.setPrototypeOf || { __proto__: [] } instanceof Array && function(d2, b2) {
      d2.__proto__ = b2;
    } || function(d2, b2) {
      for (var p in b2)
        if (Object.prototype.hasOwnProperty.call(b2, p))
          d2[p] = b2[p];
    };
    return extendStatics(d, b);
  };
  return function(d, b) {
    if (typeof b !== "function" && b !== null)
      throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
    extendStatics(d, b);
    function __() {
      this.constructor = d;
    }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
  };
}();
var localize12 = loadMessageBundle();
var LESSCompletion = function(_super) {
  __extends9(LESSCompletion2, _super);
  function LESSCompletion2(lsOptions, cssDataManager) {
    return _super.call(this, "@", lsOptions, cssDataManager) || this;
  }
  LESSCompletion2.prototype.createFunctionProposals = function(proposals, existingNode, sortToEnd, result) {
    for (var _i = 0, proposals_1 = proposals; _i < proposals_1.length; _i++) {
      var p = proposals_1[_i];
      var item = {
        label: p.name,
        detail: p.example,
        documentation: p.description,
        textEdit: TextEdit.replace(this.getCompletionRange(existingNode), p.name + "($0)"),
        insertTextFormat: InsertTextFormat.Snippet,
        kind: CompletionItemKind.Function
      };
      if (sortToEnd) {
        item.sortText = "z";
      }
      result.items.push(item);
    }
    return result;
  };
  LESSCompletion2.prototype.getTermProposals = function(entry, existingNode, result) {
    var functions = LESSCompletion2.builtInProposals;
    if (entry) {
      functions = functions.filter(function(f2) {
        return !f2.type || !entry.restrictions || entry.restrictions.indexOf(f2.type) !== -1;
      });
    }
    this.createFunctionProposals(functions, existingNode, true, result);
    return _super.prototype.getTermProposals.call(this, entry, existingNode, result);
  };
  LESSCompletion2.prototype.getColorProposals = function(entry, existingNode, result) {
    this.createFunctionProposals(LESSCompletion2.colorProposals, existingNode, false, result);
    return _super.prototype.getColorProposals.call(this, entry, existingNode, result);
  };
  LESSCompletion2.prototype.getCompletionsForDeclarationProperty = function(declaration, result) {
    this.getCompletionsForSelector(null, true, result);
    return _super.prototype.getCompletionsForDeclarationProperty.call(this, declaration, result);
  };
  LESSCompletion2.builtInProposals = [
    {
      "name": "if",
      "example": "if(condition, trueValue [, falseValue]);",
      "description": localize12("less.builtin.if", "returns one of two values depending on a condition.")
    },
    {
      "name": "boolean",
      "example": "boolean(condition);",
      "description": localize12("less.builtin.boolean", '"store" a boolean test for later evaluation in a guard or if().')
    },
    {
      "name": "length",
      "example": "length(@list);",
      "description": localize12("less.builtin.length", "returns the number of elements in a value list")
    },
    {
      "name": "extract",
      "example": "extract(@list, index);",
      "description": localize12("less.builtin.extract", "returns a value at the specified position in the list")
    },
    {
      "name": "range",
      "example": "range([start, ] end [, step]);",
      "description": localize12("less.builtin.range", "generate a list spanning a range of values")
    },
    {
      "name": "each",
      "example": "each(@list, ruleset);",
      "description": localize12("less.builtin.each", "bind the evaluation of a ruleset to each member of a list.")
    },
    {
      "name": "escape",
      "example": "escape(@string);",
      "description": localize12("less.builtin.escape", "URL encodes a string")
    },
    {
      "name": "e",
      "example": "e(@string);",
      "description": localize12("less.builtin.e", "escape string content")
    },
    {
      "name": "replace",
      "example": "replace(@string, @pattern, @replacement[, @flags]);",
      "description": localize12("less.builtin.replace", "string replace")
    },
    {
      "name": "unit",
      "example": "unit(@dimension, [@unit: '']);",
      "description": localize12("less.builtin.unit", "remove or change the unit of a dimension")
    },
    {
      "name": "color",
      "example": "color(@string);",
      "description": localize12("less.builtin.color", "parses a string to a color"),
      "type": "color"
    },
    {
      "name": "convert",
      "example": "convert(@value, unit);",
      "description": localize12("less.builtin.convert", "converts numbers from one type into another")
    },
    {
      "name": "data-uri",
      "example": "data-uri([mimetype,] url);",
      "description": localize12("less.builtin.data-uri", "inlines a resource and falls back to `url()`"),
      "type": "url"
    },
    {
      "name": "abs",
      "description": localize12("less.builtin.abs", "absolute value of a number"),
      "example": "abs(number);"
    },
    {
      "name": "acos",
      "description": localize12("less.builtin.acos", "arccosine - inverse of cosine function"),
      "example": "acos(number);"
    },
    {
      "name": "asin",
      "description": localize12("less.builtin.asin", "arcsine - inverse of sine function"),
      "example": "asin(number);"
    },
    {
      "name": "ceil",
      "example": "ceil(@number);",
      "description": localize12("less.builtin.ceil", "rounds up to an integer")
    },
    {
      "name": "cos",
      "description": localize12("less.builtin.cos", "cosine function"),
      "example": "cos(number);"
    },
    {
      "name": "floor",
      "description": localize12("less.builtin.floor", "rounds down to an integer"),
      "example": "floor(@number);"
    },
    {
      "name": "percentage",
      "description": localize12("less.builtin.percentage", "converts to a %, e.g. 0.5 > 50%"),
      "example": "percentage(@number);",
      "type": "percentage"
    },
    {
      "name": "round",
      "description": localize12("less.builtin.round", "rounds a number to a number of places"),
      "example": "round(number, [places: 0]);"
    },
    {
      "name": "sqrt",
      "description": localize12("less.builtin.sqrt", "calculates square root of a number"),
      "example": "sqrt(number);"
    },
    {
      "name": "sin",
      "description": localize12("less.builtin.sin", "sine function"),
      "example": "sin(number);"
    },
    {
      "name": "tan",
      "description": localize12("less.builtin.tan", "tangent function"),
      "example": "tan(number);"
    },
    {
      "name": "atan",
      "description": localize12("less.builtin.atan", "arctangent - inverse of tangent function"),
      "example": "atan(number);"
    },
    {
      "name": "pi",
      "description": localize12("less.builtin.pi", "returns pi"),
      "example": "pi();"
    },
    {
      "name": "pow",
      "description": localize12("less.builtin.pow", "first argument raised to the power of the second argument"),
      "example": "pow(@base, @exponent);"
    },
    {
      "name": "mod",
      "description": localize12("less.builtin.mod", "first argument modulus second argument"),
      "example": "mod(number, number);"
    },
    {
      "name": "min",
      "description": localize12("less.builtin.min", "returns the lowest of one or more values"),
      "example": "min(@x, @y);"
    },
    {
      "name": "max",
      "description": localize12("less.builtin.max", "returns the lowest of one or more values"),
      "example": "max(@x, @y);"
    }
  ];
  LESSCompletion2.colorProposals = [
    {
      "name": "argb",
      "example": "argb(@color);",
      "description": localize12("less.builtin.argb", "creates a #AARRGGBB")
    },
    {
      "name": "hsl",
      "example": "hsl(@hue, @saturation, @lightness);",
      "description": localize12("less.builtin.hsl", "creates a color")
    },
    {
      "name": "hsla",
      "example": "hsla(@hue, @saturation, @lightness, @alpha);",
      "description": localize12("less.builtin.hsla", "creates a color")
    },
    {
      "name": "hsv",
      "example": "hsv(@hue, @saturation, @value);",
      "description": localize12("less.builtin.hsv", "creates a color")
    },
    {
      "name": "hsva",
      "example": "hsva(@hue, @saturation, @value, @alpha);",
      "description": localize12("less.builtin.hsva", "creates a color")
    },
    {
      "name": "hue",
      "example": "hue(@color);",
      "description": localize12("less.builtin.hue", "returns the `hue` channel of `@color` in the HSL space")
    },
    {
      "name": "saturation",
      "example": "saturation(@color);",
      "description": localize12("less.builtin.saturation", "returns the `saturation` channel of `@color` in the HSL space")
    },
    {
      "name": "lightness",
      "example": "lightness(@color);",
      "description": localize12("less.builtin.lightness", "returns the `lightness` channel of `@color` in the HSL space")
    },
    {
      "name": "hsvhue",
      "example": "hsvhue(@color);",
      "description": localize12("less.builtin.hsvhue", "returns the `hue` channel of `@color` in the HSV space")
    },
    {
      "name": "hsvsaturation",
      "example": "hsvsaturation(@color);",
      "description": localize12("less.builtin.hsvsaturation", "returns the `saturation` channel of `@color` in the HSV space")
    },
    {
      "name": "hsvvalue",
      "example": "hsvvalue(@color);",
      "description": localize12("less.builtin.hsvvalue", "returns the `value` channel of `@color` in the HSV space")
    },
    {
      "name": "red",
      "example": "red(@color);",
      "description": localize12("less.builtin.red", "returns the `red` channel of `@color`")
    },
    {
      "name": "green",
      "example": "green(@color);",
      "description": localize12("less.builtin.green", "returns the `green` channel of `@color`")
    },
    {
      "name": "blue",
      "example": "blue(@color);",
      "description": localize12("less.builtin.blue", "returns the `blue` channel of `@color`")
    },
    {
      "name": "alpha",
      "example": "alpha(@color);",
      "description": localize12("less.builtin.alpha", "returns the `alpha` channel of `@color`")
    },
    {
      "name": "luma",
      "example": "luma(@color);",
      "description": localize12("less.builtin.luma", "returns the `luma` value (perceptual brightness) of `@color`")
    },
    {
      "name": "saturate",
      "example": "saturate(@color, 10%);",
      "description": localize12("less.builtin.saturate", "return `@color` 10% points more saturated")
    },
    {
      "name": "desaturate",
      "example": "desaturate(@color, 10%);",
      "description": localize12("less.builtin.desaturate", "return `@color` 10% points less saturated")
    },
    {
      "name": "lighten",
      "example": "lighten(@color, 10%);",
      "description": localize12("less.builtin.lighten", "return `@color` 10% points lighter")
    },
    {
      "name": "darken",
      "example": "darken(@color, 10%);",
      "description": localize12("less.builtin.darken", "return `@color` 10% points darker")
    },
    {
      "name": "fadein",
      "example": "fadein(@color, 10%);",
      "description": localize12("less.builtin.fadein", "return `@color` 10% points less transparent")
    },
    {
      "name": "fadeout",
      "example": "fadeout(@color, 10%);",
      "description": localize12("less.builtin.fadeout", "return `@color` 10% points more transparent")
    },
    {
      "name": "fade",
      "example": "fade(@color, 50%);",
      "description": localize12("less.builtin.fade", "return `@color` with 50% transparency")
    },
    {
      "name": "spin",
      "example": "spin(@color, 10);",
      "description": localize12("less.builtin.spin", "return `@color` with a 10 degree larger in hue")
    },
    {
      "name": "mix",
      "example": "mix(@color1, @color2, [@weight: 50%]);",
      "description": localize12("less.builtin.mix", "return a mix of `@color1` and `@color2`")
    },
    {
      "name": "greyscale",
      "example": "greyscale(@color);",
      "description": localize12("less.builtin.greyscale", "returns a grey, 100% desaturated color")
    },
    {
      "name": "contrast",
      "example": "contrast(@color1, [@darkcolor: black], [@lightcolor: white], [@threshold: 43%]);",
      "description": localize12("less.builtin.contrast", "return `@darkcolor` if `@color1 is> 43% luma` otherwise return `@lightcolor`, see notes")
    },
    {
      "name": "multiply",
      "example": "multiply(@color1, @color2);"
    },
    {
      "name": "screen",
      "example": "screen(@color1, @color2);"
    },
    {
      "name": "overlay",
      "example": "overlay(@color1, @color2);"
    },
    {
      "name": "softlight",
      "example": "softlight(@color1, @color2);"
    },
    {
      "name": "hardlight",
      "example": "hardlight(@color1, @color2);"
    },
    {
      "name": "difference",
      "example": "difference(@color1, @color2);"
    },
    {
      "name": "exclusion",
      "example": "exclusion(@color1, @color2);"
    },
    {
      "name": "average",
      "example": "average(@color1, @color2);"
    },
    {
      "name": "negation",
      "example": "negation(@color1, @color2);"
    }
  ];
  return LESSCompletion2;
}(CSSCompletion);

// node_modules/vscode-css-languageservice/lib/esm/services/cssFolding.js
function getFoldingRanges(document, context) {
  var ranges = computeFoldingRanges(document);
  return limitFoldingRanges(ranges, context);
}
function computeFoldingRanges(document) {
  function getStartLine(t) {
    return document.positionAt(t.offset).line;
  }
  function getEndLine(t) {
    return document.positionAt(t.offset + t.len).line;
  }
  function getScanner() {
    switch (document.languageId) {
      case "scss":
        return new SCSSScanner();
      case "less":
        return new LESSScanner();
      default:
        return new Scanner();
    }
  }
  function tokenToRange(t, kind) {
    var startLine = getStartLine(t);
    var endLine = getEndLine(t);
    if (startLine !== endLine) {
      return {
        startLine,
        endLine,
        kind
      };
    } else {
      return null;
    }
  }
  var ranges = [];
  var delimiterStack = [];
  var scanner = getScanner();
  scanner.ignoreComment = false;
  scanner.setSource(document.getText());
  var token = scanner.scan();
  var prevToken = null;
  var _loop_1 = function() {
    switch (token.type) {
      case TokenType.CurlyL:
      case InterpolationFunction: {
        delimiterStack.push({ line: getStartLine(token), type: "brace", isStart: true });
        break;
      }
      case TokenType.CurlyR: {
        if (delimiterStack.length !== 0) {
          var prevDelimiter = popPrevStartDelimiterOfType(delimiterStack, "brace");
          if (!prevDelimiter) {
            break;
          }
          var endLine = getEndLine(token);
          if (prevDelimiter.type === "brace") {
            if (prevToken && getEndLine(prevToken) !== endLine) {
              endLine--;
            }
            if (prevDelimiter.line !== endLine) {
              ranges.push({
                startLine: prevDelimiter.line,
                endLine,
                kind: void 0
              });
            }
          }
        }
        break;
      }
      case TokenType.Comment: {
        var commentRegionMarkerToDelimiter_1 = function(marker) {
          if (marker === "#region") {
            return { line: getStartLine(token), type: "comment", isStart: true };
          } else {
            return { line: getEndLine(token), type: "comment", isStart: false };
          }
        };
        var getCurrDelimiter = function(token2) {
          var matches2 = token2.text.match(/^\s*\/\*\s*(#region|#endregion)\b\s*(.*?)\s*\*\//);
          if (matches2) {
            return commentRegionMarkerToDelimiter_1(matches2[1]);
          } else if (document.languageId === "scss" || document.languageId === "less") {
            var matches_1 = token2.text.match(/^\s*\/\/\s*(#region|#endregion)\b\s*(.*?)\s*/);
            if (matches_1) {
              return commentRegionMarkerToDelimiter_1(matches_1[1]);
            }
          }
          return null;
        };
        var currDelimiter = getCurrDelimiter(token);
        if (currDelimiter) {
          if (currDelimiter.isStart) {
            delimiterStack.push(currDelimiter);
          } else {
            var prevDelimiter = popPrevStartDelimiterOfType(delimiterStack, "comment");
            if (!prevDelimiter) {
              break;
            }
            if (prevDelimiter.type === "comment") {
              if (prevDelimiter.line !== currDelimiter.line) {
                ranges.push({
                  startLine: prevDelimiter.line,
                  endLine: currDelimiter.line,
                  kind: "region"
                });
              }
            }
          }
        } else {
          var range = tokenToRange(token, "comment");
          if (range) {
            ranges.push(range);
          }
        }
        break;
      }
    }
    prevToken = token;
    token = scanner.scan();
  };
  while (token.type !== TokenType.EOF) {
    _loop_1();
  }
  return ranges;
}
function popPrevStartDelimiterOfType(stack, type) {
  if (stack.length === 0) {
    return null;
  }
  for (var i = stack.length - 1; i >= 0; i--) {
    if (stack[i].type === type && stack[i].isStart) {
      return stack.splice(i, 1)[0];
    }
  }
  return null;
}
function limitFoldingRanges(ranges, context) {
  var maxRanges = context && context.rangeLimit || Number.MAX_VALUE;
  var sortedRanges = ranges.sort(function(r1, r2) {
    var diff = r1.startLine - r2.startLine;
    if (diff === 0) {
      diff = r1.endLine - r2.endLine;
    }
    return diff;
  });
  var validRanges = [];
  var prevEndLine = -1;
  sortedRanges.forEach(function(r) {
    if (!(r.startLine < prevEndLine && prevEndLine < r.endLine)) {
      validRanges.push(r);
      prevEndLine = r.endLine;
    }
  });
  if (validRanges.length < maxRanges) {
    return validRanges;
  } else {
    return validRanges.slice(0, maxRanges);
  }
}

// node_modules/vscode-css-languageservice/lib/esm/beautify/beautify-css.js
var legacy_beautify_css;
(function() {
  "use strict";
  var __webpack_modules__ = [
    ,
    ,
    function(module) {
      function OutputLine(parent) {
        this.__parent = parent;
        this.__character_count = 0;
        this.__indent_count = -1;
        this.__alignment_count = 0;
        this.__wrap_point_index = 0;
        this.__wrap_point_character_count = 0;
        this.__wrap_point_indent_count = -1;
        this.__wrap_point_alignment_count = 0;
        this.__items = [];
      }
      OutputLine.prototype.clone_empty = function() {
        var line = new OutputLine(this.__parent);
        line.set_indent(this.__indent_count, this.__alignment_count);
        return line;
      };
      OutputLine.prototype.item = function(index) {
        if (index < 0) {
          return this.__items[this.__items.length + index];
        } else {
          return this.__items[index];
        }
      };
      OutputLine.prototype.has_match = function(pattern) {
        for (var lastCheckedOutput = this.__items.length - 1; lastCheckedOutput >= 0; lastCheckedOutput--) {
          if (this.__items[lastCheckedOutput].match(pattern)) {
            return true;
          }
        }
        return false;
      };
      OutputLine.prototype.set_indent = function(indent, alignment) {
        if (this.is_empty()) {
          this.__indent_count = indent || 0;
          this.__alignment_count = alignment || 0;
          this.__character_count = this.__parent.get_indent_size(this.__indent_count, this.__alignment_count);
        }
      };
      OutputLine.prototype._set_wrap_point = function() {
        if (this.__parent.wrap_line_length) {
          this.__wrap_point_index = this.__items.length;
          this.__wrap_point_character_count = this.__character_count;
          this.__wrap_point_indent_count = this.__parent.next_line.__indent_count;
          this.__wrap_point_alignment_count = this.__parent.next_line.__alignment_count;
        }
      };
      OutputLine.prototype._should_wrap = function() {
        return this.__wrap_point_index && this.__character_count > this.__parent.wrap_line_length && this.__wrap_point_character_count > this.__parent.next_line.__character_count;
      };
      OutputLine.prototype._allow_wrap = function() {
        if (this._should_wrap()) {
          this.__parent.add_new_line();
          var next = this.__parent.current_line;
          next.set_indent(this.__wrap_point_indent_count, this.__wrap_point_alignment_count);
          next.__items = this.__items.slice(this.__wrap_point_index);
          this.__items = this.__items.slice(0, this.__wrap_point_index);
          next.__character_count += this.__character_count - this.__wrap_point_character_count;
          this.__character_count = this.__wrap_point_character_count;
          if (next.__items[0] === " ") {
            next.__items.splice(0, 1);
            next.__character_count -= 1;
          }
          return true;
        }
        return false;
      };
      OutputLine.prototype.is_empty = function() {
        return this.__items.length === 0;
      };
      OutputLine.prototype.last = function() {
        if (!this.is_empty()) {
          return this.__items[this.__items.length - 1];
        } else {
          return null;
        }
      };
      OutputLine.prototype.push = function(item) {
        this.__items.push(item);
        var last_newline_index = item.lastIndexOf("\n");
        if (last_newline_index !== -1) {
          this.__character_count = item.length - last_newline_index;
        } else {
          this.__character_count += item.length;
        }
      };
      OutputLine.prototype.pop = function() {
        var item = null;
        if (!this.is_empty()) {
          item = this.__items.pop();
          this.__character_count -= item.length;
        }
        return item;
      };
      OutputLine.prototype._remove_indent = function() {
        if (this.__indent_count > 0) {
          this.__indent_count -= 1;
          this.__character_count -= this.__parent.indent_size;
        }
      };
      OutputLine.prototype._remove_wrap_indent = function() {
        if (this.__wrap_point_indent_count > 0) {
          this.__wrap_point_indent_count -= 1;
        }
      };
      OutputLine.prototype.trim = function() {
        while (this.last() === " ") {
          this.__items.pop();
          this.__character_count -= 1;
        }
      };
      OutputLine.prototype.toString = function() {
        var result = "";
        if (this.is_empty()) {
          if (this.__parent.indent_empty_lines) {
            result = this.__parent.get_indent_string(this.__indent_count);
          }
        } else {
          result = this.__parent.get_indent_string(this.__indent_count, this.__alignment_count);
          result += this.__items.join("");
        }
        return result;
      };
      function IndentStringCache(options, baseIndentString) {
        this.__cache = [""];
        this.__indent_size = options.indent_size;
        this.__indent_string = options.indent_char;
        if (!options.indent_with_tabs) {
          this.__indent_string = new Array(options.indent_size + 1).join(options.indent_char);
        }
        baseIndentString = baseIndentString || "";
        if (options.indent_level > 0) {
          baseIndentString = new Array(options.indent_level + 1).join(this.__indent_string);
        }
        this.__base_string = baseIndentString;
        this.__base_string_length = baseIndentString.length;
      }
      IndentStringCache.prototype.get_indent_size = function(indent, column) {
        var result = this.__base_string_length;
        column = column || 0;
        if (indent < 0) {
          result = 0;
        }
        result += indent * this.__indent_size;
        result += column;
        return result;
      };
      IndentStringCache.prototype.get_indent_string = function(indent_level, column) {
        var result = this.__base_string;
        column = column || 0;
        if (indent_level < 0) {
          indent_level = 0;
          result = "";
        }
        column += indent_level * this.__indent_size;
        this.__ensure_cache(column);
        result += this.__cache[column];
        return result;
      };
      IndentStringCache.prototype.__ensure_cache = function(column) {
        while (column >= this.__cache.length) {
          this.__add_column();
        }
      };
      IndentStringCache.prototype.__add_column = function() {
        var column = this.__cache.length;
        var indent = 0;
        var result = "";
        if (this.__indent_size && column >= this.__indent_size) {
          indent = Math.floor(column / this.__indent_size);
          column -= indent * this.__indent_size;
          result = new Array(indent + 1).join(this.__indent_string);
        }
        if (column) {
          result += new Array(column + 1).join(" ");
        }
        this.__cache.push(result);
      };
      function Output(options, baseIndentString) {
        this.__indent_cache = new IndentStringCache(options, baseIndentString);
        this.raw = false;
        this._end_with_newline = options.end_with_newline;
        this.indent_size = options.indent_size;
        this.wrap_line_length = options.wrap_line_length;
        this.indent_empty_lines = options.indent_empty_lines;
        this.__lines = [];
        this.previous_line = null;
        this.current_line = null;
        this.next_line = new OutputLine(this);
        this.space_before_token = false;
        this.non_breaking_space = false;
        this.previous_token_wrapped = false;
        this.__add_outputline();
      }
      Output.prototype.__add_outputline = function() {
        this.previous_line = this.current_line;
        this.current_line = this.next_line.clone_empty();
        this.__lines.push(this.current_line);
      };
      Output.prototype.get_line_number = function() {
        return this.__lines.length;
      };
      Output.prototype.get_indent_string = function(indent, column) {
        return this.__indent_cache.get_indent_string(indent, column);
      };
      Output.prototype.get_indent_size = function(indent, column) {
        return this.__indent_cache.get_indent_size(indent, column);
      };
      Output.prototype.is_empty = function() {
        return !this.previous_line && this.current_line.is_empty();
      };
      Output.prototype.add_new_line = function(force_newline) {
        if (this.is_empty() || !force_newline && this.just_added_newline()) {
          return false;
        }
        if (!this.raw) {
          this.__add_outputline();
        }
        return true;
      };
      Output.prototype.get_code = function(eol) {
        this.trim(true);
        var last_item = this.current_line.pop();
        if (last_item) {
          if (last_item[last_item.length - 1] === "\n") {
            last_item = last_item.replace(/\n+$/g, "");
          }
          this.current_line.push(last_item);
        }
        if (this._end_with_newline) {
          this.__add_outputline();
        }
        var sweet_code = this.__lines.join("\n");
        if (eol !== "\n") {
          sweet_code = sweet_code.replace(/[\n]/g, eol);
        }
        return sweet_code;
      };
      Output.prototype.set_wrap_point = function() {
        this.current_line._set_wrap_point();
      };
      Output.prototype.set_indent = function(indent, alignment) {
        indent = indent || 0;
        alignment = alignment || 0;
        this.next_line.set_indent(indent, alignment);
        if (this.__lines.length > 1) {
          this.current_line.set_indent(indent, alignment);
          return true;
        }
        this.current_line.set_indent();
        return false;
      };
      Output.prototype.add_raw_token = function(token) {
        for (var x = 0; x < token.newlines; x++) {
          this.__add_outputline();
        }
        this.current_line.set_indent(-1);
        this.current_line.push(token.whitespace_before);
        this.current_line.push(token.text);
        this.space_before_token = false;
        this.non_breaking_space = false;
        this.previous_token_wrapped = false;
      };
      Output.prototype.add_token = function(printable_token) {
        this.__add_space_before_token();
        this.current_line.push(printable_token);
        this.space_before_token = false;
        this.non_breaking_space = false;
        this.previous_token_wrapped = this.current_line._allow_wrap();
      };
      Output.prototype.__add_space_before_token = function() {
        if (this.space_before_token && !this.just_added_newline()) {
          if (!this.non_breaking_space) {
            this.set_wrap_point();
          }
          this.current_line.push(" ");
        }
      };
      Output.prototype.remove_indent = function(index) {
        var output_length = this.__lines.length;
        while (index < output_length) {
          this.__lines[index]._remove_indent();
          index++;
        }
        this.current_line._remove_wrap_indent();
      };
      Output.prototype.trim = function(eat_newlines) {
        eat_newlines = eat_newlines === void 0 ? false : eat_newlines;
        this.current_line.trim();
        while (eat_newlines && this.__lines.length > 1 && this.current_line.is_empty()) {
          this.__lines.pop();
          this.current_line = this.__lines[this.__lines.length - 1];
          this.current_line.trim();
        }
        this.previous_line = this.__lines.length > 1 ? this.__lines[this.__lines.length - 2] : null;
      };
      Output.prototype.just_added_newline = function() {
        return this.current_line.is_empty();
      };
      Output.prototype.just_added_blankline = function() {
        return this.is_empty() || this.current_line.is_empty() && this.previous_line.is_empty();
      };
      Output.prototype.ensure_empty_line_above = function(starts_with, ends_with) {
        var index = this.__lines.length - 2;
        while (index >= 0) {
          var potentialEmptyLine = this.__lines[index];
          if (potentialEmptyLine.is_empty()) {
            break;
          } else if (potentialEmptyLine.item(0).indexOf(starts_with) !== 0 && potentialEmptyLine.item(-1) !== ends_with) {
            this.__lines.splice(index + 1, 0, new OutputLine(this));
            this.previous_line = this.__lines[this.__lines.length - 2];
            break;
          }
          index--;
        }
      };
      module.exports.Output = Output;
    },
    ,
    ,
    ,
    function(module) {
      function Options(options, merge_child_field) {
        this.raw_options = _mergeOpts(options, merge_child_field);
        this.disabled = this._get_boolean("disabled");
        this.eol = this._get_characters("eol", "auto");
        this.end_with_newline = this._get_boolean("end_with_newline");
        this.indent_size = this._get_number("indent_size", 4);
        this.indent_char = this._get_characters("indent_char", " ");
        this.indent_level = this._get_number("indent_level");
        this.preserve_newlines = this._get_boolean("preserve_newlines", true);
        this.max_preserve_newlines = this._get_number("max_preserve_newlines", 32786);
        if (!this.preserve_newlines) {
          this.max_preserve_newlines = 0;
        }
        this.indent_with_tabs = this._get_boolean("indent_with_tabs", this.indent_char === "	");
        if (this.indent_with_tabs) {
          this.indent_char = "	";
          if (this.indent_size === 1) {
            this.indent_size = 4;
          }
        }
        this.wrap_line_length = this._get_number("wrap_line_length", this._get_number("max_char"));
        this.indent_empty_lines = this._get_boolean("indent_empty_lines");
        this.templating = this._get_selection_list("templating", ["auto", "none", "django", "erb", "handlebars", "php", "smarty"], ["auto"]);
      }
      Options.prototype._get_array = function(name, default_value) {
        var option_value = this.raw_options[name];
        var result = default_value || [];
        if (typeof option_value === "object") {
          if (option_value !== null && typeof option_value.concat === "function") {
            result = option_value.concat();
          }
        } else if (typeof option_value === "string") {
          result = option_value.split(/[^a-zA-Z0-9_\/\-]+/);
        }
        return result;
      };
      Options.prototype._get_boolean = function(name, default_value) {
        var option_value = this.raw_options[name];
        var result = option_value === void 0 ? !!default_value : !!option_value;
        return result;
      };
      Options.prototype._get_characters = function(name, default_value) {
        var option_value = this.raw_options[name];
        var result = default_value || "";
        if (typeof option_value === "string") {
          result = option_value.replace(/\\r/, "\r").replace(/\\n/, "\n").replace(/\\t/, "	");
        }
        return result;
      };
      Options.prototype._get_number = function(name, default_value) {
        var option_value = this.raw_options[name];
        default_value = parseInt(default_value, 10);
        if (isNaN(default_value)) {
          default_value = 0;
        }
        var result = parseInt(option_value, 10);
        if (isNaN(result)) {
          result = default_value;
        }
        return result;
      };
      Options.prototype._get_selection = function(name, selection_list, default_value) {
        var result = this._get_selection_list(name, selection_list, default_value);
        if (result.length !== 1) {
          throw new Error("Invalid Option Value: The option '" + name + "' can only be one of the following values:\n" + selection_list + "\nYou passed in: '" + this.raw_options[name] + "'");
        }
        return result[0];
      };
      Options.prototype._get_selection_list = function(name, selection_list, default_value) {
        if (!selection_list || selection_list.length === 0) {
          throw new Error("Selection list cannot be empty.");
        }
        default_value = default_value || [selection_list[0]];
        if (!this._is_valid_selection(default_value, selection_list)) {
          throw new Error("Invalid Default Value!");
        }
        var result = this._get_array(name, default_value);
        if (!this._is_valid_selection(result, selection_list)) {
          throw new Error("Invalid Option Value: The option '" + name + "' can contain only the following values:\n" + selection_list + "\nYou passed in: '" + this.raw_options[name] + "'");
        }
        return result;
      };
      Options.prototype._is_valid_selection = function(result, selection_list) {
        return result.length && selection_list.length && !result.some(function(item) {
          return selection_list.indexOf(item) === -1;
        });
      };
      function _mergeOpts(allOptions, childFieldName) {
        var finalOpts = {};
        allOptions = _normalizeOpts(allOptions);
        var name;
        for (name in allOptions) {
          if (name !== childFieldName) {
            finalOpts[name] = allOptions[name];
          }
        }
        if (childFieldName && allOptions[childFieldName]) {
          for (name in allOptions[childFieldName]) {
            finalOpts[name] = allOptions[childFieldName][name];
          }
        }
        return finalOpts;
      }
      function _normalizeOpts(options) {
        var convertedOpts = {};
        var key;
        for (key in options) {
          var newKey = key.replace(/-/g, "_");
          convertedOpts[newKey] = options[key];
        }
        return convertedOpts;
      }
      module.exports.Options = Options;
      module.exports.normalizeOpts = _normalizeOpts;
      module.exports.mergeOpts = _mergeOpts;
    },
    ,
    function(module) {
      var regexp_has_sticky = RegExp.prototype.hasOwnProperty("sticky");
      function InputScanner(input_string) {
        this.__input = input_string || "";
        this.__input_length = this.__input.length;
        this.__position = 0;
      }
      InputScanner.prototype.restart = function() {
        this.__position = 0;
      };
      InputScanner.prototype.back = function() {
        if (this.__position > 0) {
          this.__position -= 1;
        }
      };
      InputScanner.prototype.hasNext = function() {
        return this.__position < this.__input_length;
      };
      InputScanner.prototype.next = function() {
        var val = null;
        if (this.hasNext()) {
          val = this.__input.charAt(this.__position);
          this.__position += 1;
        }
        return val;
      };
      InputScanner.prototype.peek = function(index) {
        var val = null;
        index = index || 0;
        index += this.__position;
        if (index >= 0 && index < this.__input_length) {
          val = this.__input.charAt(index);
        }
        return val;
      };
      InputScanner.prototype.__match = function(pattern, index) {
        pattern.lastIndex = index;
        var pattern_match = pattern.exec(this.__input);
        if (pattern_match && !(regexp_has_sticky && pattern.sticky)) {
          if (pattern_match.index !== index) {
            pattern_match = null;
          }
        }
        return pattern_match;
      };
      InputScanner.prototype.test = function(pattern, index) {
        index = index || 0;
        index += this.__position;
        if (index >= 0 && index < this.__input_length) {
          return !!this.__match(pattern, index);
        } else {
          return false;
        }
      };
      InputScanner.prototype.testChar = function(pattern, index) {
        var val = this.peek(index);
        pattern.lastIndex = 0;
        return val !== null && pattern.test(val);
      };
      InputScanner.prototype.match = function(pattern) {
        var pattern_match = this.__match(pattern, this.__position);
        if (pattern_match) {
          this.__position += pattern_match[0].length;
        } else {
          pattern_match = null;
        }
        return pattern_match;
      };
      InputScanner.prototype.read = function(starting_pattern, until_pattern, until_after) {
        var val = "";
        var match;
        if (starting_pattern) {
          match = this.match(starting_pattern);
          if (match) {
            val += match[0];
          }
        }
        if (until_pattern && (match || !starting_pattern)) {
          val += this.readUntil(until_pattern, until_after);
        }
        return val;
      };
      InputScanner.prototype.readUntil = function(pattern, until_after) {
        var val = "";
        var match_index = this.__position;
        pattern.lastIndex = this.__position;
        var pattern_match = pattern.exec(this.__input);
        if (pattern_match) {
          match_index = pattern_match.index;
          if (until_after) {
            match_index += pattern_match[0].length;
          }
        } else {
          match_index = this.__input_length;
        }
        val = this.__input.substring(this.__position, match_index);
        this.__position = match_index;
        return val;
      };
      InputScanner.prototype.readUntilAfter = function(pattern) {
        return this.readUntil(pattern, true);
      };
      InputScanner.prototype.get_regexp = function(pattern, match_from) {
        var result = null;
        var flags = "g";
        if (match_from && regexp_has_sticky) {
          flags = "y";
        }
        if (typeof pattern === "string" && pattern !== "") {
          result = new RegExp(pattern, flags);
        } else if (pattern) {
          result = new RegExp(pattern.source, flags);
        }
        return result;
      };
      InputScanner.prototype.get_literal_regexp = function(literal_string) {
        return RegExp(literal_string.replace(/[-\/\\^$*+?.()|[\]{}]/g, "\\$&"));
      };
      InputScanner.prototype.peekUntilAfter = function(pattern) {
        var start = this.__position;
        var val = this.readUntilAfter(pattern);
        this.__position = start;
        return val;
      };
      InputScanner.prototype.lookBack = function(testVal) {
        var start = this.__position - 1;
        return start >= testVal.length && this.__input.substring(start - testVal.length, start).toLowerCase() === testVal;
      };
      module.exports.InputScanner = InputScanner;
    },
    ,
    ,
    ,
    ,
    function(module) {
      function Directives(start_block_pattern, end_block_pattern) {
        start_block_pattern = typeof start_block_pattern === "string" ? start_block_pattern : start_block_pattern.source;
        end_block_pattern = typeof end_block_pattern === "string" ? end_block_pattern : end_block_pattern.source;
        this.__directives_block_pattern = new RegExp(start_block_pattern + / beautify( \w+[:]\w+)+ /.source + end_block_pattern, "g");
        this.__directive_pattern = / (\w+)[:](\w+)/g;
        this.__directives_end_ignore_pattern = new RegExp(start_block_pattern + /\sbeautify\signore:end\s/.source + end_block_pattern, "g");
      }
      Directives.prototype.get_directives = function(text) {
        if (!text.match(this.__directives_block_pattern)) {
          return null;
        }
        var directives = {};
        this.__directive_pattern.lastIndex = 0;
        var directive_match = this.__directive_pattern.exec(text);
        while (directive_match) {
          directives[directive_match[1]] = directive_match[2];
          directive_match = this.__directive_pattern.exec(text);
        }
        return directives;
      };
      Directives.prototype.readIgnored = function(input) {
        return input.readUntilAfter(this.__directives_end_ignore_pattern);
      };
      module.exports.Directives = Directives;
    },
    ,
    function(module, __unused_webpack_exports, __webpack_require__2) {
      var Beautifier = __webpack_require__2(16).Beautifier, Options = __webpack_require__2(17).Options;
      function css_beautify2(source_text, options) {
        var beautifier = new Beautifier(source_text, options);
        return beautifier.beautify();
      }
      module.exports = css_beautify2;
      module.exports.defaultOptions = function() {
        return new Options();
      };
    },
    function(module, __unused_webpack_exports, __webpack_require__2) {
      var Options = __webpack_require__2(17).Options;
      var Output = __webpack_require__2(2).Output;
      var InputScanner = __webpack_require__2(8).InputScanner;
      var Directives = __webpack_require__2(13).Directives;
      var directives_core = new Directives(/\/\*/, /\*\//);
      var lineBreak = /\r\n|[\r\n]/;
      var allLineBreaks = /\r\n|[\r\n]/g;
      var whitespaceChar = /\s/;
      var whitespacePattern = /(?:\s|\n)+/g;
      var block_comment_pattern = /\/\*(?:[\s\S]*?)((?:\*\/)|$)/g;
      var comment_pattern = /\/\/(?:[^\n\r\u2028\u2029]*)/g;
      function Beautifier(source_text, options) {
        this._source_text = source_text || "";
        this._options = new Options(options);
        this._ch = null;
        this._input = null;
        this.NESTED_AT_RULE = {
          "@page": true,
          "@font-face": true,
          "@keyframes": true,
          "@media": true,
          "@supports": true,
          "@document": true
        };
        this.CONDITIONAL_GROUP_RULE = {
          "@media": true,
          "@supports": true,
          "@document": true
        };
      }
      Beautifier.prototype.eatString = function(endChars) {
        var result = "";
        this._ch = this._input.next();
        while (this._ch) {
          result += this._ch;
          if (this._ch === "\\") {
            result += this._input.next();
          } else if (endChars.indexOf(this._ch) !== -1 || this._ch === "\n") {
            break;
          }
          this._ch = this._input.next();
        }
        return result;
      };
      Beautifier.prototype.eatWhitespace = function(allowAtLeastOneNewLine) {
        var result = whitespaceChar.test(this._input.peek());
        var newline_count = 0;
        while (whitespaceChar.test(this._input.peek())) {
          this._ch = this._input.next();
          if (allowAtLeastOneNewLine && this._ch === "\n") {
            if (newline_count === 0 || newline_count < this._options.max_preserve_newlines) {
              newline_count++;
              this._output.add_new_line(true);
            }
          }
        }
        return result;
      };
      Beautifier.prototype.foundNestedPseudoClass = function() {
        var openParen = 0;
        var i = 1;
        var ch = this._input.peek(i);
        while (ch) {
          if (ch === "{") {
            return true;
          } else if (ch === "(") {
            openParen += 1;
          } else if (ch === ")") {
            if (openParen === 0) {
              return false;
            }
            openParen -= 1;
          } else if (ch === ";" || ch === "}") {
            return false;
          }
          i++;
          ch = this._input.peek(i);
        }
        return false;
      };
      Beautifier.prototype.print_string = function(output_string) {
        this._output.set_indent(this._indentLevel);
        this._output.non_breaking_space = true;
        this._output.add_token(output_string);
      };
      Beautifier.prototype.preserveSingleSpace = function(isAfterSpace) {
        if (isAfterSpace) {
          this._output.space_before_token = true;
        }
      };
      Beautifier.prototype.indent = function() {
        this._indentLevel++;
      };
      Beautifier.prototype.outdent = function() {
        if (this._indentLevel > 0) {
          this._indentLevel--;
        }
      };
      Beautifier.prototype.beautify = function() {
        if (this._options.disabled) {
          return this._source_text;
        }
        var source_text = this._source_text;
        var eol = this._options.eol;
        if (eol === "auto") {
          eol = "\n";
          if (source_text && lineBreak.test(source_text || "")) {
            eol = source_text.match(lineBreak)[0];
          }
        }
        source_text = source_text.replace(allLineBreaks, "\n");
        var baseIndentString = source_text.match(/^[\t ]*/)[0];
        this._output = new Output(this._options, baseIndentString);
        this._input = new InputScanner(source_text);
        this._indentLevel = 0;
        this._nestedLevel = 0;
        this._ch = null;
        var parenLevel = 0;
        var insideRule = false;
        var insidePropertyValue = false;
        var enteringConditionalGroup = false;
        var insideAtExtend = false;
        var insideAtImport = false;
        var topCharacter = this._ch;
        var whitespace;
        var isAfterSpace;
        var previous_ch;
        while (true) {
          whitespace = this._input.read(whitespacePattern);
          isAfterSpace = whitespace !== "";
          previous_ch = topCharacter;
          this._ch = this._input.next();
          if (this._ch === "\\" && this._input.hasNext()) {
            this._ch += this._input.next();
          }
          topCharacter = this._ch;
          if (!this._ch) {
            break;
          } else if (this._ch === "/" && this._input.peek() === "*") {
            this._output.add_new_line();
            this._input.back();
            var comment = this._input.read(block_comment_pattern);
            var directives = directives_core.get_directives(comment);
            if (directives && directives.ignore === "start") {
              comment += directives_core.readIgnored(this._input);
            }
            this.print_string(comment);
            this.eatWhitespace(true);
            this._output.add_new_line();
          } else if (this._ch === "/" && this._input.peek() === "/") {
            this._output.space_before_token = true;
            this._input.back();
            this.print_string(this._input.read(comment_pattern));
            this.eatWhitespace(true);
          } else if (this._ch === "@") {
            this.preserveSingleSpace(isAfterSpace);
            if (this._input.peek() === "{") {
              this.print_string(this._ch + this.eatString("}"));
            } else {
              this.print_string(this._ch);
              var variableOrRule = this._input.peekUntilAfter(/[: ,;{}()[\]\/='"]/g);
              if (variableOrRule.match(/[ :]$/)) {
                variableOrRule = this.eatString(": ").replace(/\s$/, "");
                this.print_string(variableOrRule);
                this._output.space_before_token = true;
              }
              variableOrRule = variableOrRule.replace(/\s$/, "");
              if (variableOrRule === "extend") {
                insideAtExtend = true;
              } else if (variableOrRule === "import") {
                insideAtImport = true;
              }
              if (variableOrRule in this.NESTED_AT_RULE) {
                this._nestedLevel += 1;
                if (variableOrRule in this.CONDITIONAL_GROUP_RULE) {
                  enteringConditionalGroup = true;
                }
              } else if (!insideRule && parenLevel === 0 && variableOrRule.indexOf(":") !== -1) {
                insidePropertyValue = true;
                this.indent();
              }
            }
          } else if (this._ch === "#" && this._input.peek() === "{") {
            this.preserveSingleSpace(isAfterSpace);
            this.print_string(this._ch + this.eatString("}"));
          } else if (this._ch === "{") {
            if (insidePropertyValue) {
              insidePropertyValue = false;
              this.outdent();
            }
            if (enteringConditionalGroup) {
              enteringConditionalGroup = false;
              insideRule = this._indentLevel >= this._nestedLevel;
            } else {
              insideRule = this._indentLevel >= this._nestedLevel - 1;
            }
            if (this._options.newline_between_rules && insideRule) {
              if (this._output.previous_line && this._output.previous_line.item(-1) !== "{") {
                this._output.ensure_empty_line_above("/", ",");
              }
            }
            this._output.space_before_token = true;
            if (this._options.brace_style === "expand") {
              this._output.add_new_line();
              this.print_string(this._ch);
              this.indent();
              this._output.set_indent(this._indentLevel);
            } else {
              this.indent();
              this.print_string(this._ch);
            }
            this.eatWhitespace(true);
            this._output.add_new_line();
          } else if (this._ch === "}") {
            this.outdent();
            this._output.add_new_line();
            if (previous_ch === "{") {
              this._output.trim(true);
            }
            insideAtImport = false;
            insideAtExtend = false;
            if (insidePropertyValue) {
              this.outdent();
              insidePropertyValue = false;
            }
            this.print_string(this._ch);
            insideRule = false;
            if (this._nestedLevel) {
              this._nestedLevel--;
            }
            this.eatWhitespace(true);
            this._output.add_new_line();
            if (this._options.newline_between_rules && !this._output.just_added_blankline()) {
              if (this._input.peek() !== "}") {
                this._output.add_new_line(true);
              }
            }
          } else if (this._ch === ":") {
            if ((insideRule || enteringConditionalGroup) && !(this._input.lookBack("&") || this.foundNestedPseudoClass()) && !this._input.lookBack("(") && !insideAtExtend && parenLevel === 0) {
              this.print_string(":");
              if (!insidePropertyValue) {
                insidePropertyValue = true;
                this._output.space_before_token = true;
                this.eatWhitespace(true);
                this.indent();
              }
            } else {
              if (this._input.lookBack(" ")) {
                this._output.space_before_token = true;
              }
              if (this._input.peek() === ":") {
                this._ch = this._input.next();
                this.print_string("::");
              } else {
                this.print_string(":");
              }
            }
          } else if (this._ch === '"' || this._ch === "'") {
            this.preserveSingleSpace(isAfterSpace);
            this.print_string(this._ch + this.eatString(this._ch));
            this.eatWhitespace(true);
          } else if (this._ch === ";") {
            if (parenLevel === 0) {
              if (insidePropertyValue) {
                this.outdent();
                insidePropertyValue = false;
              }
              insideAtExtend = false;
              insideAtImport = false;
              this.print_string(this._ch);
              this.eatWhitespace(true);
              if (this._input.peek() !== "/") {
                this._output.add_new_line();
              }
            } else {
              this.print_string(this._ch);
              this.eatWhitespace(true);
              this._output.space_before_token = true;
            }
          } else if (this._ch === "(") {
            if (this._input.lookBack("url")) {
              this.print_string(this._ch);
              this.eatWhitespace();
              parenLevel++;
              this.indent();
              this._ch = this._input.next();
              if (this._ch === ")" || this._ch === '"' || this._ch === "'") {
                this._input.back();
              } else if (this._ch) {
                this.print_string(this._ch + this.eatString(")"));
                if (parenLevel) {
                  parenLevel--;
                  this.outdent();
                }
              }
            } else {
              this.preserveSingleSpace(isAfterSpace);
              this.print_string(this._ch);
              this.eatWhitespace();
              parenLevel++;
              this.indent();
            }
          } else if (this._ch === ")") {
            if (parenLevel) {
              parenLevel--;
              this.outdent();
            }
            this.print_string(this._ch);
          } else if (this._ch === ",") {
            this.print_string(this._ch);
            this.eatWhitespace(true);
            if (this._options.selector_separator_newline && !insidePropertyValue && parenLevel === 0 && !insideAtImport && !insideAtExtend) {
              this._output.add_new_line();
            } else {
              this._output.space_before_token = true;
            }
          } else if ((this._ch === ">" || this._ch === "+" || this._ch === "~") && !insidePropertyValue && parenLevel === 0) {
            if (this._options.space_around_combinator) {
              this._output.space_before_token = true;
              this.print_string(this._ch);
              this._output.space_before_token = true;
            } else {
              this.print_string(this._ch);
              this.eatWhitespace();
              if (this._ch && whitespaceChar.test(this._ch)) {
                this._ch = "";
              }
            }
          } else if (this._ch === "]") {
            this.print_string(this._ch);
          } else if (this._ch === "[") {
            this.preserveSingleSpace(isAfterSpace);
            this.print_string(this._ch);
          } else if (this._ch === "=") {
            this.eatWhitespace();
            this.print_string("=");
            if (whitespaceChar.test(this._ch)) {
              this._ch = "";
            }
          } else if (this._ch === "!" && !this._input.lookBack("\\")) {
            this.print_string(" ");
            this.print_string(this._ch);
          } else {
            this.preserveSingleSpace(isAfterSpace);
            this.print_string(this._ch);
          }
        }
        var sweetCode = this._output.get_code(eol);
        return sweetCode;
      };
      module.exports.Beautifier = Beautifier;
    },
    function(module, __unused_webpack_exports, __webpack_require__2) {
      var BaseOptions = __webpack_require__2(6).Options;
      function Options(options) {
        BaseOptions.call(this, options, "css");
        this.selector_separator_newline = this._get_boolean("selector_separator_newline", true);
        this.newline_between_rules = this._get_boolean("newline_between_rules", true);
        var space_around_selector_separator = this._get_boolean("space_around_selector_separator");
        this.space_around_combinator = this._get_boolean("space_around_combinator") || space_around_selector_separator;
        var brace_style_split = this._get_selection_list("brace_style", ["collapse", "expand", "end-expand", "none", "preserve-inline"]);
        this.brace_style = "collapse";
        for (var bs = 0; bs < brace_style_split.length; bs++) {
          if (brace_style_split[bs] !== "expand") {
            this.brace_style = "collapse";
          } else {
            this.brace_style = brace_style_split[bs];
          }
        }
      }
      Options.prototype = new BaseOptions();
      module.exports.Options = Options;
    }
  ];
  var __webpack_module_cache__ = {};
  function __webpack_require__(moduleId) {
    var cachedModule = __webpack_module_cache__[moduleId];
    if (cachedModule !== void 0) {
      return cachedModule.exports;
    }
    var module = __webpack_module_cache__[moduleId] = {
      exports: {}
    };
    __webpack_modules__[moduleId](module, module.exports, __webpack_require__);
    return module.exports;
  }
  var __webpack_exports__ = __webpack_require__(15);
  legacy_beautify_css = __webpack_exports__;
})();
var css_beautify = legacy_beautify_css;

// node_modules/vscode-css-languageservice/lib/esm/services/cssFormatter.js
function format2(document, range, options) {
  var value = document.getText();
  var includesEnd = true;
  var initialIndentLevel = 0;
  var inRule = false;
  var tabSize = options.tabSize || 4;
  if (range) {
    var startOffset = document.offsetAt(range.start);
    var extendedStart = startOffset;
    while (extendedStart > 0 && isWhitespace(value, extendedStart - 1)) {
      extendedStart--;
    }
    if (extendedStart === 0 || isEOL(value, extendedStart - 1)) {
      startOffset = extendedStart;
    } else {
      if (extendedStart < startOffset) {
        startOffset = extendedStart + 1;
      }
    }
    var endOffset = document.offsetAt(range.end);
    var extendedEnd = endOffset;
    while (extendedEnd < value.length && isWhitespace(value, extendedEnd)) {
      extendedEnd++;
    }
    if (extendedEnd === value.length || isEOL(value, extendedEnd)) {
      endOffset = extendedEnd;
    }
    range = Range.create(document.positionAt(startOffset), document.positionAt(endOffset));
    inRule = isInRule(value, startOffset);
    includesEnd = endOffset === value.length;
    value = value.substring(startOffset, endOffset);
    if (startOffset !== 0) {
      var startOfLineOffset = document.offsetAt(Position.create(range.start.line, 0));
      initialIndentLevel = computeIndentLevel(document.getText(), startOfLineOffset, options);
    }
    if (inRule) {
      value = "{\n".concat(trimLeft(value));
    }
  } else {
    range = Range.create(Position.create(0, 0), document.positionAt(value.length));
  }
  var cssOptions = {
    indent_size: tabSize,
    indent_char: options.insertSpaces ? " " : "	",
    end_with_newline: includesEnd && getFormatOption(options, "insertFinalNewline", false),
    selector_separator_newline: getFormatOption(options, "newlineBetweenSelectors", true),
    newline_between_rules: getFormatOption(options, "newlineBetweenRules", true),
    space_around_selector_separator: getFormatOption(options, "spaceAroundSelectorSeparator", false),
    brace_style: getFormatOption(options, "braceStyle", "collapse"),
    indent_empty_lines: getFormatOption(options, "indentEmptyLines", false),
    max_preserve_newlines: getFormatOption(options, "maxPreserveNewLines", void 0),
    preserve_newlines: getFormatOption(options, "preserveNewLines", true),
    wrap_line_length: getFormatOption(options, "wrapLineLength", void 0),
    eol: "\n"
  };
  var result = css_beautify(value, cssOptions);
  if (inRule) {
    result = trimLeft(result.substring(2));
  }
  if (initialIndentLevel > 0) {
    var indent = options.insertSpaces ? repeat(" ", tabSize * initialIndentLevel) : repeat("	", initialIndentLevel);
    result = result.split("\n").join("\n" + indent);
    if (range.start.character === 0) {
      result = indent + result;
    }
  }
  return [{
    range,
    newText: result
  }];
}
function trimLeft(str) {
  return str.replace(/^\s+/, "");
}
var _CUL3 = "{".charCodeAt(0);
var _CUR2 = "}".charCodeAt(0);
function isInRule(str, offset) {
  while (offset >= 0) {
    var ch = str.charCodeAt(offset);
    if (ch === _CUL3) {
      return true;
    } else if (ch === _CUR2) {
      return false;
    }
    offset--;
  }
  return false;
}
function getFormatOption(options, key, dflt) {
  if (options && options.hasOwnProperty(key)) {
    var value = options[key];
    if (value !== null) {
      return value;
    }
  }
  return dflt;
}
function computeIndentLevel(content, offset, options) {
  var i = offset;
  var nChars = 0;
  var tabSize = options.tabSize || 4;
  while (i < content.length) {
    var ch = content.charAt(i);
    if (ch === " ") {
      nChars++;
    } else if (ch === "	") {
      nChars += tabSize;
    } else {
      break;
    }
    i++;
  }
  return Math.floor(nChars / tabSize);
}
function isEOL(text, offset) {
  return "\r\n".indexOf(text.charAt(offset)) !== -1;
}
function isWhitespace(text, offset) {
  return " 	".indexOf(text.charAt(offset)) !== -1;
}

// node_modules/vscode-css-languageservice/lib/esm/data/webCustomData.js
var cssData = {
  "version": 1.1,
  "properties": [
    {
      "name": "additive-symbols",
      "browsers": [
        "FF33"
      ],
      "syntax": "[ <integer> && <symbol> ]#",
      "relevance": 50,
      "description": "@counter-style descriptor. Specifies the symbols used by the marker-construction algorithm specified by the system descriptor. Needs to be specified if the counter system is 'additive'.",
      "restrictions": [
        "integer",
        "string",
        "image",
        "identifier"
      ]
    },
    {
      "name": "align-content",
      "values": [
        {
          "name": "center",
          "description": "Lines are packed toward the center of the flex container."
        },
        {
          "name": "flex-end",
          "description": "Lines are packed toward the end of the flex container."
        },
        {
          "name": "flex-start",
          "description": "Lines are packed toward the start of the flex container."
        },
        {
          "name": "space-around",
          "description": "Lines are evenly distributed in the flex container, with half-size spaces on either end."
        },
        {
          "name": "space-between",
          "description": "Lines are evenly distributed in the flex container."
        },
        {
          "name": "stretch",
          "description": "Lines stretch to take up the remaining space."
        }
      ],
      "syntax": "normal | <baseline-position> | <content-distribution> | <overflow-position>? <content-position>",
      "relevance": 62,
      "description": "Aligns a flex container\u2019s lines within the flex container when there is extra space in the cross-axis, similar to how 'justify-content' aligns individual items within the main-axis.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "align-items",
      "values": [
        {
          "name": "baseline",
          "description": "If the flex item\u2019s inline axis is the same as the cross axis, this value is identical to 'flex-start'. Otherwise, it participates in baseline alignment."
        },
        {
          "name": "center",
          "description": "The flex item\u2019s margin box is centered in the cross axis within the line."
        },
        {
          "name": "flex-end",
          "description": "The cross-end margin edge of the flex item is placed flush with the cross-end edge of the line."
        },
        {
          "name": "flex-start",
          "description": "The cross-start margin edge of the flex item is placed flush with the cross-start edge of the line."
        },
        {
          "name": "stretch",
          "description": "If the cross size property of the flex item computes to auto, and neither of the cross-axis margins are auto, the flex item is stretched."
        }
      ],
      "syntax": "normal | stretch | <baseline-position> | [ <overflow-position>? <self-position> ]",
      "relevance": 85,
      "description": "Aligns flex items along the cross axis of the current line of the flex container.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "justify-items",
      "values": [
        {
          "name": "auto"
        },
        {
          "name": "normal"
        },
        {
          "name": "end"
        },
        {
          "name": "start"
        },
        {
          "name": "flex-end",
          "description": '"Flex items are packed toward the end of the line."'
        },
        {
          "name": "flex-start",
          "description": '"Flex items are packed toward the start of the line."'
        },
        {
          "name": "self-end",
          "description": "The item is packed flush to the edge of the alignment container of the end side of the item, in the appropriate axis."
        },
        {
          "name": "self-start",
          "description": "The item is packed flush to the edge of the alignment container of the start side of the item, in the appropriate axis.."
        },
        {
          "name": "center",
          "description": "The items are packed flush to each other toward the center of the of the alignment container."
        },
        {
          "name": "left"
        },
        {
          "name": "right"
        },
        {
          "name": "baseline"
        },
        {
          "name": "first baseline"
        },
        {
          "name": "last baseline"
        },
        {
          "name": "stretch",
          "description": "If the cross size property of the flex item computes to auto, and neither of the cross-axis margins are auto, the flex item is stretched."
        },
        {
          "name": "save"
        },
        {
          "name": "unsave"
        },
        {
          "name": "legacy"
        }
      ],
      "syntax": "normal | stretch | <baseline-position> | <overflow-position>? [ <self-position> | left | right ] | legacy | legacy && [ left | right | center ]",
      "relevance": 53,
      "description": "Defines the default justify-self for all items of the box, giving them the default way of justifying each box along the appropriate axis",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "justify-self",
      "values": [
        {
          "name": "auto"
        },
        {
          "name": "normal"
        },
        {
          "name": "end"
        },
        {
          "name": "start"
        },
        {
          "name": "flex-end",
          "description": '"Flex items are packed toward the end of the line."'
        },
        {
          "name": "flex-start",
          "description": '"Flex items are packed toward the start of the line."'
        },
        {
          "name": "self-end",
          "description": "The item is packed flush to the edge of the alignment container of the end side of the item, in the appropriate axis."
        },
        {
          "name": "self-start",
          "description": "The item is packed flush to the edge of the alignment container of the start side of the item, in the appropriate axis.."
        },
        {
          "name": "center",
          "description": "The items are packed flush to each other toward the center of the of the alignment container."
        },
        {
          "name": "left"
        },
        {
          "name": "right"
        },
        {
          "name": "baseline"
        },
        {
          "name": "first baseline"
        },
        {
          "name": "last baseline"
        },
        {
          "name": "stretch",
          "description": "If the cross size property of the flex item computes to auto, and neither of the cross-axis margins are auto, the flex item is stretched."
        },
        {
          "name": "save"
        },
        {
          "name": "unsave"
        }
      ],
      "syntax": "auto | normal | stretch | <baseline-position> | <overflow-position>? [ <self-position> | left | right ]",
      "relevance": 53,
      "description": "Defines the way of justifying a box inside its container along the appropriate axis.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "align-self",
      "values": [
        {
          "name": "auto",
          "description": "Computes to the value of 'align-items' on the element\u2019s parent, or 'stretch' if the element has no parent. On absolutely positioned elements, it computes to itself."
        },
        {
          "name": "baseline",
          "description": "If the flex item\u2019s inline axis is the same as the cross axis, this value is identical to 'flex-start'. Otherwise, it participates in baseline alignment."
        },
        {
          "name": "center",
          "description": "The flex item\u2019s margin box is centered in the cross axis within the line."
        },
        {
          "name": "flex-end",
          "description": "The cross-end margin edge of the flex item is placed flush with the cross-end edge of the line."
        },
        {
          "name": "flex-start",
          "description": "The cross-start margin edge of the flex item is placed flush with the cross-start edge of the line."
        },
        {
          "name": "stretch",
          "description": "If the cross size property of the flex item computes to auto, and neither of the cross-axis margins are auto, the flex item is stretched."
        }
      ],
      "syntax": "auto | normal | stretch | <baseline-position> | <overflow-position>? <self-position>",
      "relevance": 72,
      "description": "Allows the default alignment along the cross axis to be overridden for individual flex items.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "all",
      "browsers": [
        "E79",
        "FF27",
        "S9.1",
        "C37",
        "O24"
      ],
      "values": [],
      "syntax": "initial | inherit | unset | revert",
      "relevance": 53,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/all"
        }
      ],
      "description": "Shorthand that resets all properties except 'direction' and 'unicode-bidi'.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "alt",
      "browsers": [
        "S9"
      ],
      "values": [],
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/alt"
        }
      ],
      "description": "Provides alternative text for assistive technology to replace the generated content of a ::before or ::after element.",
      "restrictions": [
        "string",
        "enum"
      ]
    },
    {
      "name": "animation",
      "values": [
        {
          "name": "alternate",
          "description": "The animation cycle iterations that are odd counts are played in the normal direction, and the animation cycle iterations that are even counts are played in a reverse direction."
        },
        {
          "name": "alternate-reverse",
          "description": "The animation cycle iterations that are odd counts are played in the reverse direction, and the animation cycle iterations that are even counts are played in a normal direction."
        },
        {
          "name": "backwards",
          "description": "The beginning property value (as defined in the first @keyframes at-rule) is applied before the animation is displayed, during the period defined by 'animation-delay'."
        },
        {
          "name": "both",
          "description": "Both forwards and backwards fill modes are applied."
        },
        {
          "name": "forwards",
          "description": "The final property value (as defined in the last @keyframes at-rule) is maintained after the animation completes."
        },
        {
          "name": "infinite",
          "description": "Causes the animation to repeat forever."
        },
        {
          "name": "none",
          "description": "No animation is performed"
        },
        {
          "name": "normal",
          "description": "Normal playback."
        },
        {
          "name": "reverse",
          "description": "All iterations of the animation are played in the reverse direction from the way they were specified."
        }
      ],
      "syntax": "<single-animation>#",
      "relevance": 82,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/animation"
        }
      ],
      "description": "Shorthand property combines six of the animation properties into a single property.",
      "restrictions": [
        "time",
        "timing-function",
        "enum",
        "identifier",
        "number"
      ]
    },
    {
      "name": "animation-delay",
      "syntax": "<time>#",
      "relevance": 64,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/animation-delay"
        }
      ],
      "description": "Defines when the animation will start.",
      "restrictions": [
        "time"
      ]
    },
    {
      "name": "animation-direction",
      "values": [
        {
          "name": "alternate",
          "description": "The animation cycle iterations that are odd counts are played in the normal direction, and the animation cycle iterations that are even counts are played in a reverse direction."
        },
        {
          "name": "alternate-reverse",
          "description": "The animation cycle iterations that are odd counts are played in the reverse direction, and the animation cycle iterations that are even counts are played in a normal direction."
        },
        {
          "name": "normal",
          "description": "Normal playback."
        },
        {
          "name": "reverse",
          "description": "All iterations of the animation are played in the reverse direction from the way they were specified."
        }
      ],
      "syntax": "<single-animation-direction>#",
      "relevance": 57,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/animation-direction"
        }
      ],
      "description": "Defines whether or not the animation should play in reverse on alternate cycles.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "animation-duration",
      "syntax": "<time>#",
      "relevance": 68,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/animation-duration"
        }
      ],
      "description": "Defines the length of time that an animation takes to complete one cycle.",
      "restrictions": [
        "time"
      ]
    },
    {
      "name": "animation-fill-mode",
      "values": [
        {
          "name": "backwards",
          "description": "The beginning property value (as defined in the first @keyframes at-rule) is applied before the animation is displayed, during the period defined by 'animation-delay'."
        },
        {
          "name": "both",
          "description": "Both forwards and backwards fill modes are applied."
        },
        {
          "name": "forwards",
          "description": "The final property value (as defined in the last @keyframes at-rule) is maintained after the animation completes."
        },
        {
          "name": "none",
          "description": "There is no change to the property value between the time the animation is applied and the time the animation begins playing or after the animation completes."
        }
      ],
      "syntax": "<single-animation-fill-mode>#",
      "relevance": 63,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/animation-fill-mode"
        }
      ],
      "description": "Defines what values are applied by the animation outside the time it is executing.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "animation-iteration-count",
      "values": [
        {
          "name": "infinite",
          "description": "Causes the animation to repeat forever."
        }
      ],
      "syntax": "<single-animation-iteration-count>#",
      "relevance": 60,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/animation-iteration-count"
        }
      ],
      "description": "Defines the number of times an animation cycle is played. The default value is one, meaning the animation will play from beginning to end once.",
      "restrictions": [
        "number",
        "enum"
      ]
    },
    {
      "name": "animation-name",
      "values": [
        {
          "name": "none",
          "description": "No animation is performed"
        }
      ],
      "syntax": "[ none | <keyframes-name> ]#",
      "relevance": 68,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/animation-name"
        }
      ],
      "description": "Defines a list of animations that apply. Each name is used to select the keyframe at-rule that provides the property values for the animation.",
      "restrictions": [
        "identifier",
        "enum"
      ]
    },
    {
      "name": "animation-play-state",
      "values": [
        {
          "name": "paused",
          "description": "A running animation will be paused."
        },
        {
          "name": "running",
          "description": "Resume playback of a paused animation."
        }
      ],
      "syntax": "<single-animation-play-state>#",
      "relevance": 54,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/animation-play-state"
        }
      ],
      "description": "Defines whether the animation is running or paused.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "animation-timing-function",
      "syntax": "<easing-function>#",
      "relevance": 71,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/animation-timing-function"
        }
      ],
      "description": "Describes how the animation will progress over one cycle of its duration.",
      "restrictions": [
        "timing-function"
      ]
    },
    {
      "name": "backface-visibility",
      "values": [
        {
          "name": "hidden",
          "description": "Back side is hidden."
        },
        {
          "name": "visible",
          "description": "Back side is visible."
        }
      ],
      "syntax": "visible | hidden",
      "relevance": 59,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/backface-visibility"
        }
      ],
      "description": "Determines whether or not the 'back' side of a transformed element is visible when facing the viewer. With an identity transform, the front side of an element faces the viewer.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "background",
      "values": [
        {
          "name": "fixed",
          "description": "The background is fixed with regard to the viewport. In paged media where there is no viewport, a 'fixed' background is fixed with respect to the page box and therefore replicated on every page."
        },
        {
          "name": "local",
          "description": "The background is fixed with regard to the element's contents: if the element has a scrolling mechanism, the background scrolls with the element's contents."
        },
        {
          "name": "none",
          "description": "A value of 'none' counts as an image layer but draws nothing."
        },
        {
          "name": "scroll",
          "description": "The background is fixed with regard to the element itself and does not scroll with its contents. (It is effectively attached to the element's border.)"
        }
      ],
      "syntax": "[ <bg-layer> , ]* <final-bg-layer>",
      "relevance": 93,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/background"
        }
      ],
      "description": "Shorthand property for setting most background properties at the same place in the style sheet.",
      "restrictions": [
        "enum",
        "image",
        "color",
        "position",
        "length",
        "repeat",
        "percentage",
        "box"
      ]
    },
    {
      "name": "background-attachment",
      "values": [
        {
          "name": "fixed",
          "description": "The background is fixed with regard to the viewport. In paged media where there is no viewport, a 'fixed' background is fixed with respect to the page box and therefore replicated on every page."
        },
        {
          "name": "local",
          "description": "The background is fixed with regard to the element\u2019s contents: if the element has a scrolling mechanism, the background scrolls with the element\u2019s contents."
        },
        {
          "name": "scroll",
          "description": "The background is fixed with regard to the element itself and does not scroll with its contents. (It is effectively attached to the element\u2019s border.)"
        }
      ],
      "syntax": "<attachment>#",
      "relevance": 54,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/background-attachment"
        }
      ],
      "description": "Specifies whether the background images are fixed with regard to the viewport ('fixed') or scroll along with the element ('scroll') or its contents ('local').",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "background-blend-mode",
      "browsers": [
        "E79",
        "FF30",
        "S8",
        "C35",
        "O22"
      ],
      "values": [
        {
          "name": "normal",
          "description": "Default attribute which specifies no blending"
        },
        {
          "name": "multiply",
          "description": "The source color is multiplied by the destination color and replaces the destination."
        },
        {
          "name": "screen",
          "description": "Multiplies the complements of the backdrop and source color values, then complements the result."
        },
        {
          "name": "overlay",
          "description": "Multiplies or screens the colors, depending on the backdrop color value."
        },
        {
          "name": "darken",
          "description": "Selects the darker of the backdrop and source colors."
        },
        {
          "name": "lighten",
          "description": "Selects the lighter of the backdrop and source colors."
        },
        {
          "name": "color-dodge",
          "description": "Brightens the backdrop color to reflect the source color."
        },
        {
          "name": "color-burn",
          "description": "Darkens the backdrop color to reflect the source color."
        },
        {
          "name": "hard-light",
          "description": "Multiplies or screens the colors, depending on the source color value."
        },
        {
          "name": "soft-light",
          "description": "Darkens or lightens the colors, depending on the source color value."
        },
        {
          "name": "difference",
          "description": "Subtracts the darker of the two constituent colors from the lighter color.."
        },
        {
          "name": "exclusion",
          "description": "Produces an effect similar to that of the Difference mode but lower in contrast."
        },
        {
          "name": "hue",
          "browsers": [
            "E79",
            "FF30",
            "S8",
            "C35",
            "O22"
          ],
          "description": "Creates a color with the hue of the source color and the saturation and luminosity of the backdrop color."
        },
        {
          "name": "saturation",
          "browsers": [
            "E79",
            "FF30",
            "S8",
            "C35",
            "O22"
          ],
          "description": "Creates a color with the saturation of the source color and the hue and luminosity of the backdrop color."
        },
        {
          "name": "color",
          "browsers": [
            "E79",
            "FF30",
            "S8",
            "C35",
            "O22"
          ],
          "description": "Creates a color with the hue and saturation of the source color and the luminosity of the backdrop color."
        },
        {
          "name": "luminosity",
          "browsers": [
            "E79",
            "FF30",
            "S8",
            "C35",
            "O22"
          ],
          "description": "Creates a color with the luminosity of the source color and the hue and saturation of the backdrop color."
        }
      ],
      "syntax": "<blend-mode>#",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/background-blend-mode"
        }
      ],
      "description": "Defines the blending mode of each background layer.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "background-clip",
      "syntax": "<box>#",
      "relevance": 69,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/background-clip"
        }
      ],
      "description": "Determines the background painting area.",
      "restrictions": [
        "box"
      ]
    },
    {
      "name": "background-color",
      "syntax": "<color>",
      "relevance": 95,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/background-color"
        }
      ],
      "description": "Sets the background color of an element.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "background-image",
      "values": [
        {
          "name": "none",
          "description": "Counts as an image layer but draws nothing."
        }
      ],
      "syntax": "<bg-image>#",
      "relevance": 89,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/background-image"
        }
      ],
      "description": "Sets the background image(s) of an element.",
      "restrictions": [
        "image",
        "enum"
      ]
    },
    {
      "name": "background-origin",
      "syntax": "<box>#",
      "relevance": 53,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/background-origin"
        }
      ],
      "description": "For elements rendered as a single box, specifies the background positioning area. For elements rendered as multiple boxes (e.g., inline boxes on several lines, boxes on several pages) specifies which boxes 'box-decoration-break' operates on to determine the background positioning area(s).",
      "restrictions": [
        "box"
      ]
    },
    {
      "name": "background-position",
      "syntax": "<bg-position>#",
      "relevance": 88,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/background-position"
        }
      ],
      "description": "Specifies the initial position of the background image(s) (after any resizing) within their corresponding background positioning area.",
      "restrictions": [
        "position",
        "length",
        "percentage"
      ]
    },
    {
      "name": "background-position-x",
      "values": [
        {
          "name": "center",
          "description": "Equivalent to '50%' ('left 50%') for the horizontal position if the horizontal position is not otherwise specified, or '50%' ('top 50%') for the vertical position if it is."
        },
        {
          "name": "left",
          "description": "Equivalent to '0%' for the horizontal position if one or two values are given, otherwise specifies the left edge as the origin for the next offset."
        },
        {
          "name": "right",
          "description": "Equivalent to '100%' for the horizontal position if one or two values are given, otherwise specifies the right edge as the origin for the next offset."
        }
      ],
      "status": "experimental",
      "syntax": "[ center | [ [ left | right | x-start | x-end ]? <length-percentage>? ]! ]#",
      "relevance": 54,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/background-position-x"
        }
      ],
      "description": "If background images have been specified, this property specifies their initial position (after any resizing) within their corresponding background positioning area.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "background-position-y",
      "values": [
        {
          "name": "bottom",
          "description": "Equivalent to '100%' for the vertical position if one or two values are given, otherwise specifies the bottom edge as the origin for the next offset."
        },
        {
          "name": "center",
          "description": "Equivalent to '50%' ('left 50%') for the horizontal position if the horizontal position is not otherwise specified, or '50%' ('top 50%') for the vertical position if it is."
        },
        {
          "name": "top",
          "description": "Equivalent to '0%' for the vertical position if one or two values are given, otherwise specifies the top edge as the origin for the next offset."
        }
      ],
      "status": "experimental",
      "syntax": "[ center | [ [ top | bottom | y-start | y-end ]? <length-percentage>? ]! ]#",
      "relevance": 53,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/background-position-y"
        }
      ],
      "description": "If background images have been specified, this property specifies their initial position (after any resizing) within their corresponding background positioning area.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "background-repeat",
      "values": [],
      "syntax": "<repeat-style>#",
      "relevance": 85,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/background-repeat"
        }
      ],
      "description": "Specifies how background images are tiled after they have been sized and positioned.",
      "restrictions": [
        "repeat"
      ]
    },
    {
      "name": "background-size",
      "values": [
        {
          "name": "auto",
          "description": "Resolved by using the image\u2019s intrinsic ratio and the size of the other dimension, or failing that, using the image\u2019s intrinsic size, or failing that, treating it as 100%."
        },
        {
          "name": "contain",
          "description": "Scale the image, while preserving its intrinsic aspect ratio (if any), to the largest size such that both its width and its height can fit inside the background positioning area."
        },
        {
          "name": "cover",
          "description": "Scale the image, while preserving its intrinsic aspect ratio (if any), to the smallest size such that both its width and its height can completely cover the background positioning area."
        }
      ],
      "syntax": "<bg-size>#",
      "relevance": 85,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/background-size"
        }
      ],
      "description": "Specifies the size of the background images.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "behavior",
      "browsers": [
        "IE6"
      ],
      "relevance": 50,
      "description": "IE only. Used to extend behaviors of the browser.",
      "restrictions": [
        "url"
      ]
    },
    {
      "name": "block-size",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C57",
        "O44"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Depends on the values of other properties."
        }
      ],
      "syntax": "<'width'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/block-size"
        }
      ],
      "description": "Size of an element in the direction opposite that of the direction specified by 'writing-mode'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "border",
      "syntax": "<line-width> || <line-style> || <color>",
      "relevance": 96,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border"
        }
      ],
      "description": "Shorthand property for setting border width, style, and color.",
      "restrictions": [
        "length",
        "line-width",
        "line-style",
        "color"
      ]
    },
    {
      "name": "border-block-end",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'border-top-width'> || <'border-top-style'> || <color>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-block-end"
        }
      ],
      "description": "Logical 'border-bottom'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "line-width",
        "line-style",
        "color"
      ]
    },
    {
      "name": "border-block-start",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'border-top-width'> || <'border-top-style'> || <color>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-block-start"
        }
      ],
      "description": "Logical 'border-top'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "line-width",
        "line-style",
        "color"
      ]
    },
    {
      "name": "border-block-end-color",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'border-top-color'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-block-end-color"
        }
      ],
      "description": "Logical 'border-bottom-color'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "border-block-start-color",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'border-top-color'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-block-start-color"
        }
      ],
      "description": "Logical 'border-top-color'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "border-block-end-style",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'border-top-style'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-block-end-style"
        }
      ],
      "description": "Logical 'border-bottom-style'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "line-style"
      ]
    },
    {
      "name": "border-block-start-style",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'border-top-style'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-block-start-style"
        }
      ],
      "description": "Logical 'border-top-style'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "line-style"
      ]
    },
    {
      "name": "border-block-end-width",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'border-top-width'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-block-end-width"
        }
      ],
      "description": "Logical 'border-bottom-width'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "line-width"
      ]
    },
    {
      "name": "border-block-start-width",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'border-top-width'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-block-start-width"
        }
      ],
      "description": "Logical 'border-top-width'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "line-width"
      ]
    },
    {
      "name": "border-bottom",
      "syntax": "<line-width> || <line-style> || <color>",
      "relevance": 89,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-bottom"
        }
      ],
      "description": "Shorthand property for setting border width, style and color.",
      "restrictions": [
        "length",
        "line-width",
        "line-style",
        "color"
      ]
    },
    {
      "name": "border-bottom-color",
      "syntax": "<'border-top-color'>",
      "relevance": 72,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-bottom-color"
        }
      ],
      "description": "Sets the color of the bottom border.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "border-bottom-left-radius",
      "syntax": "<length-percentage>{1,2}",
      "relevance": 75,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-bottom-left-radius"
        }
      ],
      "description": "Defines the radii of the bottom left outer border edge.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "border-bottom-right-radius",
      "syntax": "<length-percentage>{1,2}",
      "relevance": 75,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-bottom-right-radius"
        }
      ],
      "description": "Defines the radii of the bottom right outer border edge.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "border-bottom-style",
      "syntax": "<line-style>",
      "relevance": 59,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-bottom-style"
        }
      ],
      "description": "Sets the style of the bottom border.",
      "restrictions": [
        "line-style"
      ]
    },
    {
      "name": "border-bottom-width",
      "syntax": "<line-width>",
      "relevance": 63,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-bottom-width"
        }
      ],
      "description": "Sets the thickness of the bottom border.",
      "restrictions": [
        "length",
        "line-width"
      ]
    },
    {
      "name": "border-collapse",
      "values": [
        {
          "name": "collapse",
          "description": "Selects the collapsing borders model."
        },
        {
          "name": "separate",
          "description": "Selects the separated borders border model."
        }
      ],
      "syntax": "collapse | separate",
      "relevance": 75,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-collapse"
        }
      ],
      "description": "Selects a table's border model.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "border-color",
      "values": [],
      "syntax": "<color>{1,4}",
      "relevance": 87,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-color"
        }
      ],
      "description": "The color of the border around all four edges of an element.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "border-image",
      "values": [
        {
          "name": "auto",
          "description": "If 'auto' is specified then the border image width is the intrinsic width or height (whichever is applicable) of the corresponding image slice. If the image does not have the required intrinsic dimension then the corresponding border-width is used instead."
        },
        {
          "name": "fill",
          "description": "Causes the middle part of the border-image to be preserved."
        },
        {
          "name": "none",
          "description": "Use the border styles."
        },
        {
          "name": "repeat",
          "description": "The image is tiled (repeated) to fill the area."
        },
        {
          "name": "round",
          "description": "The image is tiled (repeated) to fill the area. If it does not fill the area with a whole number of tiles, the image is rescaled so that it does."
        },
        {
          "name": "space",
          "description": "The image is tiled (repeated) to fill the area. If it does not fill the area with a whole number of tiles, the extra space is distributed around the tiles."
        },
        {
          "name": "stretch",
          "description": "The image is stretched to fill the area."
        },
        {
          "name": "url()"
        }
      ],
      "syntax": "<'border-image-source'> || <'border-image-slice'> [ / <'border-image-width'> | / <'border-image-width'>? / <'border-image-outset'> ]? || <'border-image-repeat'>",
      "relevance": 52,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-image"
        }
      ],
      "description": "Shorthand property for setting 'border-image-source', 'border-image-slice', 'border-image-width', 'border-image-outset' and 'border-image-repeat'. Omitted values are set to their initial values.",
      "restrictions": [
        "length",
        "percentage",
        "number",
        "url",
        "enum"
      ]
    },
    {
      "name": "border-image-outset",
      "syntax": "[ <length> | <number> ]{1,4}",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-image-outset"
        }
      ],
      "description": "The values specify the amount by which the border image area extends beyond the border box on the top, right, bottom, and left sides respectively. If the fourth value is absent, it is the same as the second. If the third one is also absent, it is the same as the first. If the second one is also absent, it is the same as the first. Numbers represent multiples of the corresponding border-width.",
      "restrictions": [
        "length",
        "number"
      ]
    },
    {
      "name": "border-image-repeat",
      "values": [
        {
          "name": "repeat",
          "description": "The image is tiled (repeated) to fill the area."
        },
        {
          "name": "round",
          "description": "The image is tiled (repeated) to fill the area. If it does not fill the area with a whole number of tiles, the image is rescaled so that it does."
        },
        {
          "name": "space",
          "description": "The image is tiled (repeated) to fill the area. If it does not fill the area with a whole number of tiles, the extra space is distributed around the tiles."
        },
        {
          "name": "stretch",
          "description": "The image is stretched to fill the area."
        }
      ],
      "syntax": "[ stretch | repeat | round | space ]{1,2}",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-image-repeat"
        }
      ],
      "description": "Specifies how the images for the sides and the middle part of the border image are scaled and tiled. If the second keyword is absent, it is assumed to be the same as the first.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "border-image-slice",
      "values": [
        {
          "name": "fill",
          "description": "Causes the middle part of the border-image to be preserved."
        }
      ],
      "syntax": "<number-percentage>{1,4} && fill?",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-image-slice"
        }
      ],
      "description": "Specifies inward offsets from the top, right, bottom, and left edges of the image, dividing it into nine regions: four corners, four edges and a middle.",
      "restrictions": [
        "number",
        "percentage"
      ]
    },
    {
      "name": "border-image-source",
      "values": [
        {
          "name": "none",
          "description": "Use the border styles."
        }
      ],
      "syntax": "none | <image>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-image-source"
        }
      ],
      "description": "Specifies an image to use instead of the border styles given by the 'border-style' properties and as an additional background layer for the element. If the value is 'none' or if the image cannot be displayed, the border styles will be used.",
      "restrictions": [
        "image"
      ]
    },
    {
      "name": "border-image-width",
      "values": [
        {
          "name": "auto",
          "description": "The border image width is the intrinsic width or height (whichever is applicable) of the corresponding image slice. If the image does not have the required intrinsic dimension then the corresponding border-width is used instead."
        }
      ],
      "syntax": "[ <length-percentage> | <number> | auto ]{1,4}",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-image-width"
        }
      ],
      "description": "The four values of 'border-image-width' specify offsets that are used to divide the border image area into nine parts. They represent inward distances from the top, right, bottom, and left sides of the area, respectively.",
      "restrictions": [
        "length",
        "percentage",
        "number"
      ]
    },
    {
      "name": "border-inline-end",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'border-top-width'> || <'border-top-style'> || <color>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-inline-end"
        }
      ],
      "description": "Logical 'border-right'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "line-width",
        "line-style",
        "color"
      ]
    },
    {
      "name": "border-inline-start",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'border-top-width'> || <'border-top-style'> || <color>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-inline-start"
        }
      ],
      "description": "Logical 'border-left'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "line-width",
        "line-style",
        "color"
      ]
    },
    {
      "name": "border-inline-end-color",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'border-top-color'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-inline-end-color"
        }
      ],
      "description": "Logical 'border-right-color'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "border-inline-start-color",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'border-top-color'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-inline-start-color"
        }
      ],
      "description": "Logical 'border-left-color'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "border-inline-end-style",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'border-top-style'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-inline-end-style"
        }
      ],
      "description": "Logical 'border-right-style'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "line-style"
      ]
    },
    {
      "name": "border-inline-start-style",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'border-top-style'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-inline-start-style"
        }
      ],
      "description": "Logical 'border-left-style'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "line-style"
      ]
    },
    {
      "name": "border-inline-end-width",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'border-top-width'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-inline-end-width"
        }
      ],
      "description": "Logical 'border-right-width'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "line-width"
      ]
    },
    {
      "name": "border-inline-start-width",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'border-top-width'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-inline-start-width"
        }
      ],
      "description": "Logical 'border-left-width'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "line-width"
      ]
    },
    {
      "name": "border-left",
      "syntax": "<line-width> || <line-style> || <color>",
      "relevance": 83,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-left"
        }
      ],
      "description": "Shorthand property for setting border width, style and color",
      "restrictions": [
        "length",
        "line-width",
        "line-style",
        "color"
      ]
    },
    {
      "name": "border-left-color",
      "syntax": "<color>",
      "relevance": 65,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-left-color"
        }
      ],
      "description": "Sets the color of the left border.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "border-left-style",
      "syntax": "<line-style>",
      "relevance": 53,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-left-style"
        }
      ],
      "description": "Sets the style of the left border.",
      "restrictions": [
        "line-style"
      ]
    },
    {
      "name": "border-left-width",
      "syntax": "<line-width>",
      "relevance": 59,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-left-width"
        }
      ],
      "description": "Sets the thickness of the left border.",
      "restrictions": [
        "length",
        "line-width"
      ]
    },
    {
      "name": "border-radius",
      "syntax": "<length-percentage>{1,4} [ / <length-percentage>{1,4} ]?",
      "relevance": 92,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-radius"
        }
      ],
      "description": "Defines the radii of the outer border edge.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "border-right",
      "syntax": "<line-width> || <line-style> || <color>",
      "relevance": 82,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-right"
        }
      ],
      "description": "Shorthand property for setting border width, style and color",
      "restrictions": [
        "length",
        "line-width",
        "line-style",
        "color"
      ]
    },
    {
      "name": "border-right-color",
      "syntax": "<color>",
      "relevance": 65,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-right-color"
        }
      ],
      "description": "Sets the color of the right border.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "border-right-style",
      "syntax": "<line-style>",
      "relevance": 53,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-right-style"
        }
      ],
      "description": "Sets the style of the right border.",
      "restrictions": [
        "line-style"
      ]
    },
    {
      "name": "border-right-width",
      "syntax": "<line-width>",
      "relevance": 59,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-right-width"
        }
      ],
      "description": "Sets the thickness of the right border.",
      "restrictions": [
        "length",
        "line-width"
      ]
    },
    {
      "name": "border-spacing",
      "syntax": "<length> <length>?",
      "relevance": 68,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-spacing"
        }
      ],
      "description": "The lengths specify the distance that separates adjoining cell borders. If one length is specified, it gives both the horizontal and vertical spacing. If two are specified, the first gives the horizontal spacing and the second the vertical spacing. Lengths may not be negative.",
      "restrictions": [
        "length"
      ]
    },
    {
      "name": "border-style",
      "values": [],
      "syntax": "<line-style>{1,4}",
      "relevance": 81,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-style"
        }
      ],
      "description": "The style of the border around edges of an element.",
      "restrictions": [
        "line-style"
      ]
    },
    {
      "name": "border-top",
      "syntax": "<line-width> || <line-style> || <color>",
      "relevance": 88,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-top"
        }
      ],
      "description": "Shorthand property for setting border width, style and color",
      "restrictions": [
        "length",
        "line-width",
        "line-style",
        "color"
      ]
    },
    {
      "name": "border-top-color",
      "syntax": "<color>",
      "relevance": 72,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-top-color"
        }
      ],
      "description": "Sets the color of the top border.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "border-top-left-radius",
      "syntax": "<length-percentage>{1,2}",
      "relevance": 76,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-top-left-radius"
        }
      ],
      "description": "Defines the radii of the top left outer border edge.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "border-top-right-radius",
      "syntax": "<length-percentage>{1,2}",
      "relevance": 75,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-top-right-radius"
        }
      ],
      "description": "Defines the radii of the top right outer border edge.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "border-top-style",
      "syntax": "<line-style>",
      "relevance": 57,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-top-style"
        }
      ],
      "description": "Sets the style of the top border.",
      "restrictions": [
        "line-style"
      ]
    },
    {
      "name": "border-top-width",
      "syntax": "<line-width>",
      "relevance": 61,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-top-width"
        }
      ],
      "description": "Sets the thickness of the top border.",
      "restrictions": [
        "length",
        "line-width"
      ]
    },
    {
      "name": "border-width",
      "values": [],
      "syntax": "<line-width>{1,4}",
      "relevance": 82,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/border-width"
        }
      ],
      "description": "Shorthand that sets the four 'border-*-width' properties. If it has four values, they set top, right, bottom and left in that order. If left is missing, it is the same as right; if bottom is missing, it is the same as top; if right is missing, it is the same as top.",
      "restrictions": [
        "length",
        "line-width"
      ]
    },
    {
      "name": "bottom",
      "values": [
        {
          "name": "auto",
          "description": "For non-replaced elements, the effect of this value depends on which of related properties have the value 'auto' as well"
        }
      ],
      "syntax": "<length> | <percentage> | auto",
      "relevance": 90,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/bottom"
        }
      ],
      "description": "Specifies how far an absolutely positioned box's bottom margin edge is offset above the bottom edge of the box's 'containing block'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "box-decoration-break",
      "browsers": [
        "E79",
        "FF32",
        "S7",
        "C22",
        "O15"
      ],
      "values": [
        {
          "name": "clone",
          "description": "Each box is independently wrapped with the border and padding."
        },
        {
          "name": "slice",
          "description": "The effect is as though the element were rendered with no breaks present, and then sliced by the breaks afterward."
        }
      ],
      "syntax": "slice | clone",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/box-decoration-break"
        }
      ],
      "description": "Specifies whether individual boxes are treated as broken pieces of one continuous box, or whether each box is individually wrapped with the border and padding.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "box-shadow",
      "values": [
        {
          "name": "inset",
          "description": "Changes the drop shadow from an outer shadow (one that shadows the box onto the canvas, as if it were lifted above the canvas) to an inner shadow (one that shadows the canvas onto the box, as if the box were cut out of the canvas and shifted behind it)."
        },
        {
          "name": "none",
          "description": "No shadow."
        }
      ],
      "syntax": "none | <shadow>#",
      "relevance": 90,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/box-shadow"
        }
      ],
      "description": "Attaches one or more drop-shadows to the box. The property is a comma-separated list of shadows, each specified by 2-4 length values, an optional color, and an optional 'inset' keyword. Omitted lengths are 0; omitted colors are a user agent chosen color.",
      "restrictions": [
        "length",
        "color",
        "enum"
      ]
    },
    {
      "name": "box-sizing",
      "values": [
        {
          "name": "border-box",
          "description": "The specified width and height (and respective min/max properties) on this element determine the border box of the element."
        },
        {
          "name": "content-box",
          "description": "Behavior of width and height as specified by CSS2.1. The specified width and height (and respective min/max properties) apply to the width and height respectively of the content box of the element."
        }
      ],
      "syntax": "content-box | border-box",
      "relevance": 93,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/box-sizing"
        }
      ],
      "description": "Specifies the behavior of the 'width' and 'height' properties.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "break-after",
      "values": [
        {
          "name": "always",
          "description": "Always force a page break before/after the generated box."
        },
        {
          "name": "auto",
          "description": "Neither force nor forbid a page/column break before/after the principal box."
        },
        {
          "name": "avoid",
          "description": "Avoid a break before/after the principal box."
        },
        {
          "name": "avoid-column",
          "description": "Avoid a column break before/after the principal box."
        },
        {
          "name": "avoid-page",
          "description": "Avoid a page break before/after the principal box."
        },
        {
          "name": "column",
          "description": "Always force a column break before/after the principal box."
        },
        {
          "name": "left",
          "description": "Force one or two page breaks before/after the generated box so that the next page is formatted as a left page."
        },
        {
          "name": "page",
          "description": "Always force a page break before/after the principal box."
        },
        {
          "name": "right",
          "description": "Force one or two page breaks before/after the generated box so that the next page is formatted as a right page."
        }
      ],
      "syntax": "auto | avoid | always | all | avoid-page | page | left | right | recto | verso | avoid-column | column | avoid-region | region",
      "relevance": 50,
      "description": "Describes the page/column/region break behavior after the generated box.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "break-before",
      "values": [
        {
          "name": "always",
          "description": "Always force a page break before/after the generated box."
        },
        {
          "name": "auto",
          "description": "Neither force nor forbid a page/column break before/after the principal box."
        },
        {
          "name": "avoid",
          "description": "Avoid a break before/after the principal box."
        },
        {
          "name": "avoid-column",
          "description": "Avoid a column break before/after the principal box."
        },
        {
          "name": "avoid-page",
          "description": "Avoid a page break before/after the principal box."
        },
        {
          "name": "column",
          "description": "Always force a column break before/after the principal box."
        },
        {
          "name": "left",
          "description": "Force one or two page breaks before/after the generated box so that the next page is formatted as a left page."
        },
        {
          "name": "page",
          "description": "Always force a page break before/after the principal box."
        },
        {
          "name": "right",
          "description": "Force one or two page breaks before/after the generated box so that the next page is formatted as a right page."
        }
      ],
      "syntax": "auto | avoid | always | all | avoid-page | page | left | right | recto | verso | avoid-column | column | avoid-region | region",
      "relevance": 50,
      "description": "Describes the page/column/region break behavior before the generated box.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "break-inside",
      "values": [
        {
          "name": "auto",
          "description": "Impose no additional breaking constraints within the box."
        },
        {
          "name": "avoid",
          "description": "Avoid breaks within the box."
        },
        {
          "name": "avoid-column",
          "description": "Avoid a column break within the box."
        },
        {
          "name": "avoid-page",
          "description": "Avoid a page break within the box."
        }
      ],
      "syntax": "auto | avoid | avoid-page | avoid-column | avoid-region",
      "relevance": 51,
      "description": "Describes the page/column/region break behavior inside the principal box.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "caption-side",
      "values": [
        {
          "name": "bottom",
          "description": "Positions the caption box below the table box."
        },
        {
          "name": "top",
          "description": "Positions the caption box above the table box."
        }
      ],
      "syntax": "top | bottom | block-start | block-end | inline-start | inline-end",
      "relevance": 52,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/caption-side"
        }
      ],
      "description": "Specifies the position of the caption box with respect to the table box.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "caret-color",
      "browsers": [
        "E79",
        "FF53",
        "S11.1",
        "C57",
        "O44"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The user agent selects an appropriate color for the caret. This is generally currentcolor, but the user agent may choose a different color to ensure good visibility and contrast with the surrounding content, taking into account the value of currentcolor, the background, shadows, and other factors."
        }
      ],
      "syntax": "auto | <color>",
      "relevance": 53,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/caret-color"
        }
      ],
      "description": "Controls the color of the text insertion indicator.",
      "restrictions": [
        "color",
        "enum"
      ]
    },
    {
      "name": "clear",
      "values": [
        {
          "name": "both",
          "description": "The clearance of the generated box is set to the amount necessary to place the top border edge below the bottom outer edge of any right-floating and left-floating boxes that resulted from elements earlier in the source document."
        },
        {
          "name": "left",
          "description": "The clearance of the generated box is set to the amount necessary to place the top border edge below the bottom outer edge of any left-floating boxes that resulted from elements earlier in the source document."
        },
        {
          "name": "none",
          "description": "No constraint on the box's position with respect to floats."
        },
        {
          "name": "right",
          "description": "The clearance of the generated box is set to the amount necessary to place the top border edge below the bottom outer edge of any right-floating boxes that resulted from elements earlier in the source document."
        }
      ],
      "syntax": "none | left | right | both | inline-start | inline-end",
      "relevance": 85,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/clear"
        }
      ],
      "description": "Indicates which sides of an element's box(es) may not be adjacent to an earlier floating box. The 'clear' property does not consider floats inside the element itself or in other block formatting contexts.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "clip",
      "values": [
        {
          "name": "auto",
          "description": "The element does not clip."
        },
        {
          "name": "rect()",
          "description": "Specifies offsets from the edges of the border box."
        }
      ],
      "syntax": "<shape> | auto",
      "relevance": 75,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/clip"
        }
      ],
      "description": "Deprecated. Use the 'clip-path' property when support allows. Defines the visible portion of an element\u2019s box.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "clip-path",
      "values": [
        {
          "name": "none",
          "description": "No clipping path gets created."
        },
        {
          "name": "url()",
          "description": "References a <clipPath> element to create a clipping path."
        }
      ],
      "syntax": "<clip-source> | [ <basic-shape> || <geometry-box> ] | none",
      "relevance": 62,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/clip-path"
        }
      ],
      "description": "Specifies a clipping path where everything inside the path is visible and everything outside is clipped out.",
      "restrictions": [
        "url",
        "shape",
        "geometry-box",
        "enum"
      ]
    },
    {
      "name": "clip-rule",
      "browsers": [
        "E",
        "C5",
        "FF3",
        "IE10",
        "O9",
        "S6"
      ],
      "values": [
        {
          "name": "evenodd",
          "description": "Determines the \u2018insideness\u2019 of a point on the canvas by drawing a ray from that point to infinity in any direction and counting the number of path segments from the given shape that the ray crosses."
        },
        {
          "name": "nonzero",
          "description": "Determines the \u2018insideness\u2019 of a point on the canvas by drawing a ray from that point to infinity in any direction and then examining the places where a segment of the shape crosses the ray."
        }
      ],
      "relevance": 50,
      "description": "Indicates the algorithm which is to be used to determine what parts of the canvas are included inside the shape.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "color",
      "syntax": "<color>",
      "relevance": 95,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/color"
        }
      ],
      "description": "Sets the color of an element's text",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "color-interpolation-filters",
      "browsers": [
        "E",
        "C5",
        "FF3",
        "IE10",
        "O9",
        "S6"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Color operations are not required to occur in a particular color space."
        },
        {
          "name": "linearRGB",
          "description": "Color operations should occur in the linearized RGB color space."
        },
        {
          "name": "sRGB",
          "description": "Color operations should occur in the sRGB color space."
        }
      ],
      "relevance": 50,
      "description": "Specifies the color space for imaging operations performed via filter effects.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "column-count",
      "values": [
        {
          "name": "auto",
          "description": "Determines the number of columns by the 'column-width' property and the element width."
        }
      ],
      "syntax": "<integer> | auto",
      "relevance": 53,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/column-count"
        }
      ],
      "description": "Describes the optimal number of columns into which the content of the element will be flowed.",
      "restrictions": [
        "integer",
        "enum"
      ]
    },
    {
      "name": "column-fill",
      "values": [
        {
          "name": "auto",
          "description": "Fills columns sequentially."
        },
        {
          "name": "balance",
          "description": "Balance content equally between columns, if possible."
        }
      ],
      "syntax": "auto | balance | balance-all",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/column-fill"
        }
      ],
      "description": "In continuous media, this property will only be consulted if the length of columns has been constrained. Otherwise, columns will automatically be balanced.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "column-gap",
      "values": [
        {
          "name": "normal",
          "description": "User agent specific and typically equivalent to 1em."
        }
      ],
      "syntax": "normal | <length-percentage>",
      "relevance": 54,
      "description": "Sets the gap between columns. If there is a column rule between columns, it will appear in the middle of the gap.",
      "restrictions": [
        "length",
        "enum"
      ]
    },
    {
      "name": "column-rule",
      "syntax": "<'column-rule-width'> || <'column-rule-style'> || <'column-rule-color'>",
      "relevance": 51,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/column-rule"
        }
      ],
      "description": "Shorthand for setting 'column-rule-width', 'column-rule-style', and 'column-rule-color' at the same place in the style sheet. Omitted values are set to their initial values.",
      "restrictions": [
        "length",
        "line-width",
        "line-style",
        "color"
      ]
    },
    {
      "name": "column-rule-color",
      "syntax": "<color>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/column-rule-color"
        }
      ],
      "description": "Sets the color of the column rule",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "column-rule-style",
      "syntax": "<'border-style'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/column-rule-style"
        }
      ],
      "description": "Sets the style of the rule between columns of an element.",
      "restrictions": [
        "line-style"
      ]
    },
    {
      "name": "column-rule-width",
      "syntax": "<'border-width'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/column-rule-width"
        }
      ],
      "description": "Sets the width of the rule between columns. Negative values are not allowed.",
      "restrictions": [
        "length",
        "line-width"
      ]
    },
    {
      "name": "columns",
      "values": [
        {
          "name": "auto",
          "description": "The width depends on the values of other properties."
        }
      ],
      "syntax": "<'column-width'> || <'column-count'>",
      "relevance": 51,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/columns"
        }
      ],
      "description": "A shorthand property which sets both 'column-width' and 'column-count'.",
      "restrictions": [
        "length",
        "integer",
        "enum"
      ]
    },
    {
      "name": "column-span",
      "values": [
        {
          "name": "all",
          "description": "The element spans across all columns. Content in the normal flow that appears before the element is automatically balanced across all columns before the element appear."
        },
        {
          "name": "none",
          "description": "The element does not span multiple columns."
        }
      ],
      "syntax": "none | all",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/column-span"
        }
      ],
      "description": "Describes the page/column break behavior after the generated box.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "column-width",
      "values": [
        {
          "name": "auto",
          "description": "The width depends on the values of other properties."
        }
      ],
      "syntax": "<length> | auto",
      "relevance": 51,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/column-width"
        }
      ],
      "description": "Describes the width of columns in multicol elements.",
      "restrictions": [
        "length",
        "enum"
      ]
    },
    {
      "name": "contain",
      "browsers": [
        "E79",
        "FF69",
        "S15.4",
        "C52",
        "O40"
      ],
      "values": [
        {
          "name": "none",
          "description": "Indicates that the property has no effect."
        },
        {
          "name": "strict",
          "description": "Turns on all forms of containment for the element."
        },
        {
          "name": "content",
          "description": "All containment rules except size are applied to the element."
        },
        {
          "name": "size",
          "description": "For properties that can have effects on more than just an element and its descendants, those effects don't escape the containing element."
        },
        {
          "name": "layout",
          "description": "Turns on layout containment for the element."
        },
        {
          "name": "style",
          "description": "Turns on style containment for the element."
        },
        {
          "name": "paint",
          "description": "Turns on paint containment for the element."
        }
      ],
      "syntax": "none | strict | content | [ size || layout || style || paint ]",
      "relevance": 59,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/contain"
        }
      ],
      "description": "Indicates that an element and its contents are, as much as possible, independent of the rest of the document tree.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "content",
      "values": [
        {
          "name": "attr()",
          "description": "The attr(n) function returns as a string the value of attribute n for the subject of the selector."
        },
        {
          "name": "counter(name)",
          "description": "Counters are denoted by identifiers (see the 'counter-increment' and 'counter-reset' properties)."
        },
        {
          "name": "icon",
          "description": "The (pseudo-)element is replaced in its entirety by the resource referenced by its 'icon' property, and treated as a replaced element."
        },
        {
          "name": "none",
          "description": "On elements, this inhibits the children of the element from being rendered as children of this element, as if the element was empty. On pseudo-elements it causes the pseudo-element to have no content."
        },
        {
          "name": "normal",
          "description": "See http://www.w3.org/TR/css3-content/#content for computation rules."
        },
        {
          "name": "url()"
        }
      ],
      "syntax": "normal | none | [ <content-replacement> | <content-list> ] [/ [ <string> | <counter> ]+ ]?",
      "relevance": 90,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/content"
        }
      ],
      "description": "Determines which page-based occurrence of a given element is applied to a counter or string value.",
      "restrictions": [
        "string",
        "url"
      ]
    },
    {
      "name": "counter-increment",
      "values": [
        {
          "name": "none",
          "description": "This element does not alter the value of any counters."
        }
      ],
      "syntax": "[ <counter-name> <integer>? ]+ | none",
      "relevance": 53,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/counter-increment"
        }
      ],
      "description": "Manipulate the value of existing counters.",
      "restrictions": [
        "identifier",
        "integer"
      ]
    },
    {
      "name": "counter-reset",
      "values": [
        {
          "name": "none",
          "description": "The counter is not modified."
        }
      ],
      "syntax": "[ <counter-name> <integer>? | <reversed-counter-name> <integer>? ]+ | none",
      "relevance": 53,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/counter-reset"
        }
      ],
      "description": "Property accepts one or more names of counters (identifiers), each one optionally followed by an integer. The integer gives the value that the counter is set to on each occurrence of the element.",
      "restrictions": [
        "identifier",
        "integer"
      ]
    },
    {
      "name": "cursor",
      "values": [
        {
          "name": "alias",
          "description": "Indicates an alias of/shortcut to something is to be created. Often rendered as an arrow with a small curved arrow next to it."
        },
        {
          "name": "all-scroll",
          "description": "Indicates that the something can be scrolled in any direction. Often rendered as arrows pointing up, down, left, and right with a dot in the middle."
        },
        {
          "name": "auto",
          "description": "The UA determines the cursor to display based on the current context."
        },
        {
          "name": "cell",
          "description": "Indicates that a cell or set of cells may be selected. Often rendered as a thick plus-sign with a dot in the middle."
        },
        {
          "name": "col-resize",
          "description": "Indicates that the item/column can be resized horizontally. Often rendered as arrows pointing left and right with a vertical bar separating them."
        },
        {
          "name": "context-menu",
          "description": "A context menu is available for the object under the cursor. Often rendered as an arrow with a small menu-like graphic next to it."
        },
        {
          "name": "copy",
          "description": "Indicates something is to be copied. Often rendered as an arrow with a small plus sign next to it."
        },
        {
          "name": "crosshair",
          "description": "A simple crosshair (e.g., short line segments resembling a '+' sign). Often used to indicate a two dimensional bitmap selection mode."
        },
        {
          "name": "default",
          "description": "The platform-dependent default cursor. Often rendered as an arrow."
        },
        {
          "name": "e-resize",
          "description": "Indicates that east edge is to be moved."
        },
        {
          "name": "ew-resize",
          "description": "Indicates a bidirectional east-west resize cursor."
        },
        {
          "name": "grab",
          "description": "Indicates that something can be grabbed."
        },
        {
          "name": "grabbing",
          "description": "Indicates that something is being grabbed."
        },
        {
          "name": "help",
          "description": "Help is available for the object under the cursor. Often rendered as a question mark or a balloon."
        },
        {
          "name": "move",
          "description": "Indicates something is to be moved."
        },
        {
          "name": "-moz-grab",
          "description": "Indicates that something can be grabbed."
        },
        {
          "name": "-moz-grabbing",
          "description": "Indicates that something is being grabbed."
        },
        {
          "name": "-moz-zoom-in",
          "description": "Indicates that something can be zoomed (magnified) in."
        },
        {
          "name": "-moz-zoom-out",
          "description": "Indicates that something can be zoomed (magnified) out."
        },
        {
          "name": "ne-resize",
          "description": "Indicates that movement starts from north-east corner."
        },
        {
          "name": "nesw-resize",
          "description": "Indicates a bidirectional north-east/south-west cursor."
        },
        {
          "name": "no-drop",
          "description": "Indicates that the dragged item cannot be dropped at the current cursor location. Often rendered as a hand or pointer with a small circle with a line through it."
        },
        {
          "name": "none",
          "description": "No cursor is rendered for the element."
        },
        {
          "name": "not-allowed",
          "description": "Indicates that the requested action will not be carried out. Often rendered as a circle with a line through it."
        },
        {
          "name": "n-resize",
          "description": "Indicates that north edge is to be moved."
        },
        {
          "name": "ns-resize",
          "description": "Indicates a bidirectional north-south cursor."
        },
        {
          "name": "nw-resize",
          "description": "Indicates that movement starts from north-west corner."
        },
        {
          "name": "nwse-resize",
          "description": "Indicates a bidirectional north-west/south-east cursor."
        },
        {
          "name": "pointer",
          "description": "The cursor is a pointer that indicates a link."
        },
        {
          "name": "progress",
          "description": "A progress indicator. The program is performing some processing, but is different from 'wait' in that the user may still interact with the program. Often rendered as a spinning beach ball, or an arrow with a watch or hourglass."
        },
        {
          "name": "row-resize",
          "description": "Indicates that the item/row can be resized vertically. Often rendered as arrows pointing up and down with a horizontal bar separating them."
        },
        {
          "name": "se-resize",
          "description": "Indicates that movement starts from south-east corner."
        },
        {
          "name": "s-resize",
          "description": "Indicates that south edge is to be moved."
        },
        {
          "name": "sw-resize",
          "description": "Indicates that movement starts from south-west corner."
        },
        {
          "name": "text",
          "description": "Indicates text that may be selected. Often rendered as a vertical I-beam."
        },
        {
          "name": "vertical-text",
          "description": "Indicates vertical-text that may be selected. Often rendered as a horizontal I-beam."
        },
        {
          "name": "wait",
          "description": "Indicates that the program is busy and the user should wait. Often rendered as a watch or hourglass."
        },
        {
          "name": "-webkit-grab",
          "description": "Indicates that something can be grabbed."
        },
        {
          "name": "-webkit-grabbing",
          "description": "Indicates that something is being grabbed."
        },
        {
          "name": "-webkit-zoom-in",
          "description": "Indicates that something can be zoomed (magnified) in."
        },
        {
          "name": "-webkit-zoom-out",
          "description": "Indicates that something can be zoomed (magnified) out."
        },
        {
          "name": "w-resize",
          "description": "Indicates that west edge is to be moved."
        },
        {
          "name": "zoom-in",
          "description": "Indicates that something can be zoomed (magnified) in."
        },
        {
          "name": "zoom-out",
          "description": "Indicates that something can be zoomed (magnified) out."
        }
      ],
      "syntax": "[ [ <url> [ <x> <y> ]? , ]* [ auto | default | none | context-menu | help | pointer | progress | wait | cell | crosshair | text | vertical-text | alias | copy | move | no-drop | not-allowed | e-resize | n-resize | ne-resize | nw-resize | s-resize | se-resize | sw-resize | w-resize | ew-resize | ns-resize | nesw-resize | nwse-resize | col-resize | row-resize | all-scroll | zoom-in | zoom-out | grab | grabbing ] ]",
      "relevance": 92,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/cursor"
        }
      ],
      "description": "Allows control over cursor appearance in an element",
      "restrictions": [
        "url",
        "number",
        "enum"
      ]
    },
    {
      "name": "direction",
      "values": [
        {
          "name": "ltr",
          "description": "Left-to-right direction."
        },
        {
          "name": "rtl",
          "description": "Right-to-left direction."
        }
      ],
      "syntax": "ltr | rtl",
      "relevance": 69,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/direction"
        }
      ],
      "description": "Specifies the inline base direction or directionality of any bidi paragraph, embedding, isolate, or override established by the box. Note: for HTML content use the 'dir' attribute and 'bdo' element rather than this property.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "display",
      "values": [
        {
          "name": "block",
          "description": "The element generates a block-level box"
        },
        {
          "name": "contents",
          "description": "The element itself does not generate any boxes, but its children and pseudo-elements still generate boxes as normal."
        },
        {
          "name": "flex",
          "description": "The element generates a principal flex container box and establishes a flex formatting context."
        },
        {
          "name": "flexbox",
          "description": "The element lays out its contents using flow layout (block-and-inline layout). Standardized as 'flex'."
        },
        {
          "name": "flow-root",
          "description": "The element generates a block container box, and lays out its contents using flow layout."
        },
        {
          "name": "grid",
          "description": "The element generates a principal grid container box, and establishes a grid formatting context."
        },
        {
          "name": "inline",
          "description": "The element generates an inline-level box."
        },
        {
          "name": "inline-block",
          "description": "A block box, which itself is flowed as a single inline box, similar to a replaced element. The inside of an inline-block is formatted as a block box, and the box itself is formatted as an inline box."
        },
        {
          "name": "inline-flex",
          "description": "Inline-level flex container."
        },
        {
          "name": "inline-flexbox",
          "description": "Inline-level flex container. Standardized as 'inline-flex'"
        },
        {
          "name": "inline-table",
          "description": "Inline-level table wrapper box containing table box."
        },
        {
          "name": "list-item",
          "description": "One or more block boxes and one marker box."
        },
        {
          "name": "-moz-box",
          "description": "The element lays out its contents using flow layout (block-and-inline layout). Standardized as 'flex'."
        },
        {
          "name": "-moz-deck"
        },
        {
          "name": "-moz-grid"
        },
        {
          "name": "-moz-grid-group"
        },
        {
          "name": "-moz-grid-line"
        },
        {
          "name": "-moz-groupbox"
        },
        {
          "name": "-moz-inline-box",
          "description": "Inline-level flex container. Standardized as 'inline-flex'"
        },
        {
          "name": "-moz-inline-grid"
        },
        {
          "name": "-moz-inline-stack"
        },
        {
          "name": "-moz-marker"
        },
        {
          "name": "-moz-popup"
        },
        {
          "name": "-moz-stack"
        },
        {
          "name": "-ms-flexbox",
          "description": "The element lays out its contents using flow layout (block-and-inline layout). Standardized as 'flex'."
        },
        {
          "name": "-ms-grid",
          "description": "The element generates a principal grid container box, and establishes a grid formatting context."
        },
        {
          "name": "-ms-inline-flexbox",
          "description": "Inline-level flex container. Standardized as 'inline-flex'"
        },
        {
          "name": "-ms-inline-grid",
          "description": "Inline-level grid container."
        },
        {
          "name": "none",
          "description": "The element and its descendants generates no boxes."
        },
        {
          "name": "ruby",
          "description": "The element generates a principal ruby container box, and establishes a ruby formatting context."
        },
        {
          "name": "ruby-base"
        },
        {
          "name": "ruby-base-container"
        },
        {
          "name": "ruby-text"
        },
        {
          "name": "ruby-text-container"
        },
        {
          "name": "run-in",
          "description": "The element generates a run-in box. Run-in elements act like inlines or blocks, depending on the surrounding elements."
        },
        {
          "name": "table",
          "description": "The element generates a principal table wrapper box containing an additionally-generated table box, and establishes a table formatting context."
        },
        {
          "name": "table-caption"
        },
        {
          "name": "table-cell"
        },
        {
          "name": "table-column"
        },
        {
          "name": "table-column-group"
        },
        {
          "name": "table-footer-group"
        },
        {
          "name": "table-header-group"
        },
        {
          "name": "table-row"
        },
        {
          "name": "table-row-group"
        },
        {
          "name": "-webkit-box",
          "description": "The element lays out its contents using flow layout (block-and-inline layout). Standardized as 'flex'."
        },
        {
          "name": "-webkit-flex",
          "description": "The element lays out its contents using flow layout (block-and-inline layout)."
        },
        {
          "name": "-webkit-inline-box",
          "description": "Inline-level flex container. Standardized as 'inline-flex'"
        },
        {
          "name": "-webkit-inline-flex",
          "description": "Inline-level flex container."
        }
      ],
      "syntax": "[ <display-outside> || <display-inside> ] | <display-listitem> | <display-internal> | <display-box> | <display-legacy>",
      "relevance": 96,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/display"
        }
      ],
      "description": "In combination with 'float' and 'position', determines the type of box or boxes that are generated for an element.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "empty-cells",
      "values": [
        {
          "name": "hide",
          "description": "No borders or backgrounds are drawn around/behind empty cells."
        },
        {
          "name": "-moz-show-background"
        },
        {
          "name": "show",
          "description": "Borders and backgrounds are drawn around/behind empty cells (like normal cells)."
        }
      ],
      "syntax": "show | hide",
      "relevance": 51,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/empty-cells"
        }
      ],
      "description": "In the separated borders model, this property controls the rendering of borders and backgrounds around cells that have no visible content.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "enable-background",
      "values": [
        {
          "name": "accumulate",
          "description": "If the ancestor container element has a property of new, then all graphics elements within the current container are rendered both on the parent's background image and onto the target."
        },
        {
          "name": "new",
          "description": "Create a new background image canvas. All children of the current container element can access the background, and they will be rendered onto both the parent's background image canvas in addition to the target device."
        }
      ],
      "relevance": 50,
      "description": "Deprecated. Use 'isolation' property instead when support allows. Specifies how the accumulation of the background image is managed.",
      "restrictions": [
        "integer",
        "length",
        "percentage",
        "enum"
      ]
    },
    {
      "name": "fallback",
      "browsers": [
        "FF33"
      ],
      "syntax": "<counter-style-name>",
      "relevance": 50,
      "description": "@counter-style descriptor. Specifies a fallback counter style to be used when the current counter style can\u2019t create a representation for a given counter value.",
      "restrictions": [
        "identifier"
      ]
    },
    {
      "name": "fill",
      "values": [
        {
          "name": "url()",
          "description": "A URL reference to a paint server element, which is an element that defines a paint server: \u2018hatch\u2019, \u2018linearGradient\u2019, \u2018mesh\u2019, \u2018pattern\u2019, \u2018radialGradient\u2019 and \u2018solidcolor\u2019."
        },
        {
          "name": "none",
          "description": "No paint is applied in this layer."
        }
      ],
      "relevance": 77,
      "description": "Paints the interior of the given graphical element.",
      "restrictions": [
        "color",
        "enum",
        "url"
      ]
    },
    {
      "name": "fill-opacity",
      "relevance": 52,
      "description": "Specifies the opacity of the painting operation used to paint the interior the current object.",
      "restrictions": [
        "number(0-1)"
      ]
    },
    {
      "name": "fill-rule",
      "values": [
        {
          "name": "evenodd",
          "description": "Determines the \u2018insideness\u2019 of a point on the canvas by drawing a ray from that point to infinity in any direction and counting the number of path segments from the given shape that the ray crosses."
        },
        {
          "name": "nonzero",
          "description": "Determines the \u2018insideness\u2019 of a point on the canvas by drawing a ray from that point to infinity in any direction and then examining the places where a segment of the shape crosses the ray."
        }
      ],
      "relevance": 50,
      "description": "Indicates the algorithm (or winding rule) which is to be used to determine what parts of the canvas are included inside the shape.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "filter",
      "browsers": [
        "E12",
        "FF35",
        "S9.1",
        "C53",
        "O40"
      ],
      "values": [
        {
          "name": "none",
          "description": "No filter effects are applied."
        },
        {
          "name": "blur()",
          "description": "Applies a Gaussian blur to the input image."
        },
        {
          "name": "brightness()",
          "description": "Applies a linear multiplier to input image, making it appear more or less bright."
        },
        {
          "name": "contrast()",
          "description": "Adjusts the contrast of the input."
        },
        {
          "name": "drop-shadow()",
          "description": "Applies a drop shadow effect to the input image."
        },
        {
          "name": "grayscale()",
          "description": "Converts the input image to grayscale."
        },
        {
          "name": "hue-rotate()",
          "description": "Applies a hue rotation on the input image. "
        },
        {
          "name": "invert()",
          "description": "Inverts the samples in the input image."
        },
        {
          "name": "opacity()",
          "description": "Applies transparency to the samples in the input image."
        },
        {
          "name": "saturate()",
          "description": "Saturates the input image."
        },
        {
          "name": "sepia()",
          "description": "Converts the input image to sepia."
        },
        {
          "name": "url()",
          "browsers": [
            "E12",
            "FF35",
            "S9.1",
            "C53",
            "O40"
          ],
          "description": "A filter reference to a <filter> element."
        }
      ],
      "syntax": "none | <filter-function-list>",
      "relevance": 66,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/filter"
        }
      ],
      "description": "Processes an element\u2019s rendering before it is displayed in the document, by applying one or more filter effects.",
      "restrictions": [
        "enum",
        "url"
      ]
    },
    {
      "name": "flex",
      "values": [
        {
          "name": "auto",
          "description": "Retrieves the value of the main size property as the used 'flex-basis'."
        },
        {
          "name": "content",
          "description": "Indicates automatic sizing, based on the flex item\u2019s content."
        },
        {
          "name": "none",
          "description": "Expands to '0 0 auto'."
        }
      ],
      "syntax": "none | [ <'flex-grow'> <'flex-shrink'>? || <'flex-basis'> ]",
      "relevance": 80,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/flex"
        }
      ],
      "description": "Specifies the components of a flexible length: the flex grow factor and flex shrink factor, and the flex basis.",
      "restrictions": [
        "length",
        "number",
        "percentage"
      ]
    },
    {
      "name": "flex-basis",
      "values": [
        {
          "name": "auto",
          "description": "Retrieves the value of the main size property as the used 'flex-basis'."
        },
        {
          "name": "content",
          "description": "Indicates automatic sizing, based on the flex item\u2019s content."
        }
      ],
      "syntax": "content | <'width'>",
      "relevance": 65,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/flex-basis"
        }
      ],
      "description": "Sets the flex basis.",
      "restrictions": [
        "length",
        "number",
        "percentage"
      ]
    },
    {
      "name": "flex-direction",
      "values": [
        {
          "name": "column",
          "description": "The flex container\u2019s main axis has the same orientation as the block axis of the current writing mode."
        },
        {
          "name": "column-reverse",
          "description": "Same as 'column', except the main-start and main-end directions are swapped."
        },
        {
          "name": "row",
          "description": "The flex container\u2019s main axis has the same orientation as the inline axis of the current writing mode."
        },
        {
          "name": "row-reverse",
          "description": "Same as 'row', except the main-start and main-end directions are swapped."
        }
      ],
      "syntax": "row | row-reverse | column | column-reverse",
      "relevance": 83,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/flex-direction"
        }
      ],
      "description": "Specifies how flex items are placed in the flex container, by setting the direction of the flex container\u2019s main axis.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "flex-flow",
      "values": [
        {
          "name": "column",
          "description": "The flex container\u2019s main axis has the same orientation as the block axis of the current writing mode."
        },
        {
          "name": "column-reverse",
          "description": "Same as 'column', except the main-start and main-end directions are swapped."
        },
        {
          "name": "nowrap",
          "description": "The flex container is single-line."
        },
        {
          "name": "row",
          "description": "The flex container\u2019s main axis has the same orientation as the inline axis of the current writing mode."
        },
        {
          "name": "row-reverse",
          "description": "Same as 'row', except the main-start and main-end directions are swapped."
        },
        {
          "name": "wrap",
          "description": "The flexbox is multi-line."
        },
        {
          "name": "wrap-reverse",
          "description": "Same as 'wrap', except the cross-start and cross-end directions are swapped."
        }
      ],
      "syntax": "<'flex-direction'> || <'flex-wrap'>",
      "relevance": 61,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/flex-flow"
        }
      ],
      "description": "Specifies how flexbox items are placed in the flexbox.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "flex-grow",
      "syntax": "<number>",
      "relevance": 75,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/flex-grow"
        }
      ],
      "description": "Sets the flex grow factor. Negative numbers are invalid.",
      "restrictions": [
        "number"
      ]
    },
    {
      "name": "flex-shrink",
      "syntax": "<number>",
      "relevance": 74,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/flex-shrink"
        }
      ],
      "description": "Sets the flex shrink factor. Negative numbers are invalid.",
      "restrictions": [
        "number"
      ]
    },
    {
      "name": "flex-wrap",
      "values": [
        {
          "name": "nowrap",
          "description": "The flex container is single-line."
        },
        {
          "name": "wrap",
          "description": "The flexbox is multi-line."
        },
        {
          "name": "wrap-reverse",
          "description": "Same as 'wrap', except the cross-start and cross-end directions are swapped."
        }
      ],
      "syntax": "nowrap | wrap | wrap-reverse",
      "relevance": 79,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/flex-wrap"
        }
      ],
      "description": "Controls whether the flex container is single-line or multi-line, and the direction of the cross-axis, which determines the direction new lines are stacked in.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "float",
      "values": [
        {
          "name": "inline-end",
          "description": "A keyword indicating that the element must float on the end side of its containing block. That is the right side with ltr scripts, and the left side with rtl scripts."
        },
        {
          "name": "inline-start",
          "description": "A keyword indicating that the element must float on the start side of its containing block. That is the left side with ltr scripts, and the right side with rtl scripts."
        },
        {
          "name": "left",
          "description": "The element generates a block box that is floated to the left. Content flows on the right side of the box, starting at the top (subject to the 'clear' property)."
        },
        {
          "name": "none",
          "description": "The box is not floated."
        },
        {
          "name": "right",
          "description": "Similar to 'left', except the box is floated to the right, and content flows on the left side of the box, starting at the top."
        }
      ],
      "syntax": "left | right | none | inline-start | inline-end",
      "relevance": 91,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/float"
        }
      ],
      "description": "Specifies how a box should be floated. It may be set for any element, but only applies to elements that generate boxes that are not absolutely positioned.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "flood-color",
      "browsers": [
        "E",
        "C5",
        "FF3",
        "IE10",
        "O9",
        "S6"
      ],
      "relevance": 50,
      "description": "Indicates what color to use to flood the current filter primitive subregion.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "flood-opacity",
      "browsers": [
        "E",
        "C5",
        "FF3",
        "IE10",
        "O9",
        "S6"
      ],
      "relevance": 50,
      "description": "Indicates what opacity to use to flood the current filter primitive subregion.",
      "restrictions": [
        "number(0-1)",
        "percentage"
      ]
    },
    {
      "name": "font",
      "values": [
        {
          "name": "100",
          "description": "Thin"
        },
        {
          "name": "200",
          "description": "Extra Light (Ultra Light)"
        },
        {
          "name": "300",
          "description": "Light"
        },
        {
          "name": "400",
          "description": "Normal"
        },
        {
          "name": "500",
          "description": "Medium"
        },
        {
          "name": "600",
          "description": "Semi Bold (Demi Bold)"
        },
        {
          "name": "700",
          "description": "Bold"
        },
        {
          "name": "800",
          "description": "Extra Bold (Ultra Bold)"
        },
        {
          "name": "900",
          "description": "Black (Heavy)"
        },
        {
          "name": "bold",
          "description": "Same as 700"
        },
        {
          "name": "bolder",
          "description": "Specifies the weight of the face bolder than the inherited value."
        },
        {
          "name": "caption",
          "description": "The font used for captioned controls (e.g., buttons, drop-downs, etc.)."
        },
        {
          "name": "icon",
          "description": "The font used to label icons."
        },
        {
          "name": "italic",
          "description": "Selects a font that is labeled 'italic', or, if that is not available, one labeled 'oblique'."
        },
        {
          "name": "large"
        },
        {
          "name": "larger"
        },
        {
          "name": "lighter",
          "description": "Specifies the weight of the face lighter than the inherited value."
        },
        {
          "name": "medium"
        },
        {
          "name": "menu",
          "description": "The font used in menus (e.g., dropdown menus and menu lists)."
        },
        {
          "name": "message-box",
          "description": "The font used in dialog boxes."
        },
        {
          "name": "normal",
          "description": "Specifies a face that is not labeled as a small-caps font."
        },
        {
          "name": "oblique",
          "description": "Selects a font that is labeled 'oblique'."
        },
        {
          "name": "small"
        },
        {
          "name": "small-caps",
          "description": "Specifies a font that is labeled as a small-caps font. If a genuine small-caps font is not available, user agents should simulate a small-caps font."
        },
        {
          "name": "small-caption",
          "description": "The font used for labeling small controls."
        },
        {
          "name": "smaller"
        },
        {
          "name": "status-bar",
          "description": "The font used in window status bars."
        },
        {
          "name": "x-large"
        },
        {
          "name": "x-small"
        },
        {
          "name": "xx-large"
        },
        {
          "name": "xx-small"
        }
      ],
      "syntax": "[ [ <'font-style'> || <font-variant-css21> || <'font-weight'> || <'font-stretch'> ]? <'font-size'> [ / <'line-height'> ]? <'font-family'> ] | caption | icon | menu | message-box | small-caption | status-bar",
      "relevance": 84,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/font"
        }
      ],
      "description": "Shorthand property for setting 'font-style', 'font-variant', 'font-weight', 'font-size', 'line-height', and 'font-family', at the same place in the style sheet. The syntax of this property is based on a traditional typographical shorthand notation to set multiple properties related to fonts.",
      "restrictions": [
        "font"
      ]
    },
    {
      "name": "font-family",
      "values": [
        {
          "name": "-apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif"
        },
        {
          "name": "Arial, Helvetica, sans-serif"
        },
        {
          "name": "Cambria, Cochin, Georgia, Times, 'Times New Roman', serif"
        },
        {
          "name": "'Courier New', Courier, monospace"
        },
        {
          "name": "cursive"
        },
        {
          "name": "fantasy"
        },
        {
          "name": "'Franklin Gothic Medium', 'Arial Narrow', Arial, sans-serif"
        },
        {
          "name": "Georgia, 'Times New Roman', Times, serif"
        },
        {
          "name": "'Gill Sans', 'Gill Sans MT', Calibri, 'Trebuchet MS', sans-serif"
        },
        {
          "name": "Impact, Haettenschweiler, 'Arial Narrow Bold', sans-serif"
        },
        {
          "name": "'Lucida Sans', 'Lucida Sans Regular', 'Lucida Grande', 'Lucida Sans Unicode', Geneva, Verdana, sans-serif"
        },
        {
          "name": "monospace"
        },
        {
          "name": "sans-serif"
        },
        {
          "name": "'Segoe UI', Tahoma, Geneva, Verdana, sans-serif"
        },
        {
          "name": "serif"
        },
        {
          "name": "'Times New Roman', Times, serif"
        },
        {
          "name": "'Trebuchet MS', 'Lucida Sans Unicode', 'Lucida Grande', 'Lucida Sans', Arial, sans-serif"
        },
        {
          "name": "Verdana, Geneva, Tahoma, sans-serif"
        }
      ],
      "syntax": "<family-name>",
      "relevance": 94,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/font-family"
        }
      ],
      "description": "Specifies a prioritized list of font family names or generic family names. A user agent iterates through the list of family names until it matches an available font that contains a glyph for the character to be rendered.",
      "restrictions": [
        "font"
      ]
    },
    {
      "name": "font-feature-settings",
      "values": [
        {
          "name": '"aalt"',
          "description": "Access All Alternates."
        },
        {
          "name": '"abvf"',
          "description": "Above-base Forms. Required in Khmer script."
        },
        {
          "name": '"abvm"',
          "description": "Above-base Mark Positioning. Required in Indic scripts."
        },
        {
          "name": '"abvs"',
          "description": "Above-base Substitutions. Required in Indic scripts."
        },
        {
          "name": '"afrc"',
          "description": "Alternative Fractions."
        },
        {
          "name": '"akhn"',
          "description": "Akhand. Required in most Indic scripts."
        },
        {
          "name": '"blwf"',
          "description": "Below-base Form. Required in a number of Indic scripts."
        },
        {
          "name": '"blwm"',
          "description": "Below-base Mark Positioning. Required in Indic scripts."
        },
        {
          "name": '"blws"',
          "description": "Below-base Substitutions. Required in Indic scripts."
        },
        {
          "name": '"calt"',
          "description": "Contextual Alternates."
        },
        {
          "name": '"case"',
          "description": "Case-Sensitive Forms. Applies only to European scripts; particularly prominent in Spanish-language setting."
        },
        {
          "name": '"ccmp"',
          "description": "Glyph Composition/Decomposition."
        },
        {
          "name": '"cfar"',
          "description": "Conjunct Form After Ro. Required in Khmer scripts."
        },
        {
          "name": '"cjct"',
          "description": "Conjunct Forms. Required in Indic scripts that show similarity to Devanagari."
        },
        {
          "name": '"clig"',
          "description": "Contextual Ligatures."
        },
        {
          "name": '"cpct"',
          "description": "Centered CJK Punctuation. Used primarily in Chinese fonts."
        },
        {
          "name": '"cpsp"',
          "description": "Capital Spacing. Should not be used in connecting scripts (e.g. most Arabic)."
        },
        {
          "name": '"cswh"',
          "description": "Contextual Swash."
        },
        {
          "name": '"curs"',
          "description": "Cursive Positioning. Can be used in any cursive script."
        },
        {
          "name": '"c2pc"',
          "description": "Petite Capitals From Capitals. Applies only to bicameral scripts."
        },
        {
          "name": '"c2sc"',
          "description": "Small Capitals From Capitals. Applies only to bicameral scripts."
        },
        {
          "name": '"dist"',
          "description": "Distances. Required in Indic scripts."
        },
        {
          "name": '"dlig"',
          "description": "Discretionary ligatures."
        },
        {
          "name": '"dnom"',
          "description": "Denominators."
        },
        {
          "name": '"dtls"',
          "description": "Dotless Forms. Applied to math formula layout."
        },
        {
          "name": '"expt"',
          "description": "Expert Forms. Applies only to Japanese."
        },
        {
          "name": '"falt"',
          "description": "Final Glyph on Line Alternates. Can be used in any cursive script."
        },
        {
          "name": '"fin2"',
          "description": "Terminal Form #2. Used only with the Syriac script."
        },
        {
          "name": '"fin3"',
          "description": "Terminal Form #3. Used only with the Syriac script."
        },
        {
          "name": '"fina"',
          "description": "Terminal Forms. Can be used in any alphabetic script."
        },
        {
          "name": '"flac"',
          "description": "Flattened ascent forms. Applied to math formula layout."
        },
        {
          "name": '"frac"',
          "description": "Fractions."
        },
        {
          "name": '"fwid"',
          "description": "Full Widths. Applies to any script which can use monospaced forms."
        },
        {
          "name": '"half"',
          "description": "Half Forms. Required in Indic scripts that show similarity to Devanagari."
        },
        {
          "name": '"haln"',
          "description": "Halant Forms. Required in Indic scripts."
        },
        {
          "name": '"halt"',
          "description": "Alternate Half Widths. Used only in CJKV fonts."
        },
        {
          "name": '"hist"',
          "description": "Historical Forms."
        },
        {
          "name": '"hkna"',
          "description": "Horizontal Kana Alternates. Applies only to fonts that support kana (hiragana and katakana)."
        },
        {
          "name": '"hlig"',
          "description": "Historical Ligatures."
        },
        {
          "name": '"hngl"',
          "description": "Hangul. Korean only."
        },
        {
          "name": '"hojo"',
          "description": "Hojo Kanji Forms (JIS X 0212-1990 Kanji Forms). Used only with Kanji script."
        },
        {
          "name": '"hwid"',
          "description": "Half Widths. Generally used only in CJKV fonts."
        },
        {
          "name": '"init"',
          "description": "Initial Forms. Can be used in any alphabetic script."
        },
        {
          "name": '"isol"',
          "description": "Isolated Forms. Can be used in any cursive script."
        },
        {
          "name": '"ital"',
          "description": "Italics. Applies mostly to Latin; note that many non-Latin fonts contain Latin as well."
        },
        {
          "name": '"jalt"',
          "description": "Justification Alternates. Can be used in any cursive script."
        },
        {
          "name": '"jp78"',
          "description": "JIS78 Forms. Applies only to Japanese."
        },
        {
          "name": '"jp83"',
          "description": "JIS83 Forms. Applies only to Japanese."
        },
        {
          "name": '"jp90"',
          "description": "JIS90 Forms. Applies only to Japanese."
        },
        {
          "name": '"jp04"',
          "description": "JIS2004 Forms. Applies only to Japanese."
        },
        {
          "name": '"kern"',
          "description": "Kerning."
        },
        {
          "name": '"lfbd"',
          "description": "Left Bounds."
        },
        {
          "name": '"liga"',
          "description": "Standard Ligatures."
        },
        {
          "name": '"ljmo"',
          "description": "Leading Jamo Forms. Required for Hangul script when Ancient Hangul writing system is supported."
        },
        {
          "name": '"lnum"',
          "description": "Lining Figures."
        },
        {
          "name": '"locl"',
          "description": "Localized Forms."
        },
        {
          "name": '"ltra"',
          "description": "Left-to-right glyph alternates."
        },
        {
          "name": '"ltrm"',
          "description": "Left-to-right mirrored forms."
        },
        {
          "name": '"mark"',
          "description": "Mark Positioning."
        },
        {
          "name": '"med2"',
          "description": "Medial Form #2. Used only with the Syriac script."
        },
        {
          "name": '"medi"',
          "description": "Medial Forms."
        },
        {
          "name": '"mgrk"',
          "description": "Mathematical Greek."
        },
        {
          "name": '"mkmk"',
          "description": "Mark to Mark Positioning."
        },
        {
          "name": '"nalt"',
          "description": "Alternate Annotation Forms."
        },
        {
          "name": '"nlck"',
          "description": "NLC Kanji Forms. Used only with Kanji script."
        },
        {
          "name": '"nukt"',
          "description": "Nukta Forms. Required in Indic scripts.."
        },
        {
          "name": '"numr"',
          "description": "Numerators."
        },
        {
          "name": '"onum"',
          "description": "Oldstyle Figures."
        },
        {
          "name": '"opbd"',
          "description": "Optical Bounds."
        },
        {
          "name": '"ordn"',
          "description": "Ordinals. Applies mostly to Latin script."
        },
        {
          "name": '"ornm"',
          "description": "Ornaments."
        },
        {
          "name": '"palt"',
          "description": "Proportional Alternate Widths. Used mostly in CJKV fonts."
        },
        {
          "name": '"pcap"',
          "description": "Petite Capitals."
        },
        {
          "name": '"pkna"',
          "description": "Proportional Kana. Generally used only in Japanese fonts."
        },
        {
          "name": '"pnum"',
          "description": "Proportional Figures."
        },
        {
          "name": '"pref"',
          "description": "Pre-base Forms. Required in Khmer and Myanmar (Burmese) scripts and southern Indic scripts that may display a pre-base form of Ra."
        },
        {
          "name": '"pres"',
          "description": "Pre-base Substitutions. Required in Indic scripts."
        },
        {
          "name": '"pstf"',
          "description": "Post-base Forms. Required in scripts of south and southeast Asia that have post-base forms for consonants eg: Gurmukhi, Malayalam, Khmer."
        },
        {
          "name": '"psts"',
          "description": "Post-base Substitutions."
        },
        {
          "name": '"pwid"',
          "description": "Proportional Widths."
        },
        {
          "name": '"qwid"',
          "description": "Quarter Widths. Generally used only in CJKV fonts."
        },
        {
          "name": '"rand"',
          "description": "Randomize."
        },
        {
          "name": '"rclt"',
          "description": "Required Contextual Alternates. May apply to any script, but is especially important for many styles of Arabic."
        },
        {
          "name": '"rlig"',
          "description": "Required Ligatures. Applies to Arabic and Syriac. May apply to some other scripts."
        },
        {
          "name": '"rkrf"',
          "description": "Rakar Forms. Required in Devanagari and Gujarati scripts."
        },
        {
          "name": '"rphf"',
          "description": "Reph Form. Required in Indic scripts. E.g. Devanagari, Kannada."
        },
        {
          "name": '"rtbd"',
          "description": "Right Bounds."
        },
        {
          "name": '"rtla"',
          "description": "Right-to-left alternates."
        },
        {
          "name": '"rtlm"',
          "description": "Right-to-left mirrored forms."
        },
        {
          "name": '"ruby"',
          "description": "Ruby Notation Forms. Applies only to Japanese."
        },
        {
          "name": '"salt"',
          "description": "Stylistic Alternates."
        },
        {
          "name": '"sinf"',
          "description": "Scientific Inferiors."
        },
        {
          "name": '"size"',
          "description": "Optical size."
        },
        {
          "name": '"smcp"',
          "description": "Small Capitals. Applies only to bicameral scripts."
        },
        {
          "name": '"smpl"',
          "description": "Simplified Forms. Applies only to Chinese and Japanese."
        },
        {
          "name": '"ssty"',
          "description": "Math script style alternates."
        },
        {
          "name": '"stch"',
          "description": "Stretching Glyph Decomposition."
        },
        {
          "name": '"subs"',
          "description": "Subscript."
        },
        {
          "name": '"sups"',
          "description": "Superscript."
        },
        {
          "name": '"swsh"',
          "description": "Swash. Does not apply to ideographic scripts."
        },
        {
          "name": '"titl"',
          "description": "Titling."
        },
        {
          "name": '"tjmo"',
          "description": "Trailing Jamo Forms. Required for Hangul script when Ancient Hangul writing system is supported."
        },
        {
          "name": '"tnam"',
          "description": "Traditional Name Forms. Applies only to Japanese."
        },
        {
          "name": '"tnum"',
          "description": "Tabular Figures."
        },
        {
          "name": '"trad"',
          "description": "Traditional Forms. Applies only to Chinese and Japanese."
        },
        {
          "name": '"twid"',
          "description": "Third Widths. Generally used only in CJKV fonts."
        },
        {
          "name": '"unic"',
          "description": "Unicase."
        },
        {
          "name": '"valt"',
          "description": "Alternate Vertical Metrics. Applies only to scripts with vertical writing modes."
        },
        {
          "name": '"vatu"',
          "description": "Vattu Variants. Used for Indic scripts. E.g. Devanagari."
        },
        {
          "name": '"vert"',
          "description": "Vertical Alternates. Applies only to scripts with vertical writing modes."
        },
        {
          "name": '"vhal"',
          "description": "Alternate Vertical Half Metrics. Used only in CJKV fonts."
        },
        {
          "name": '"vjmo"',
          "description": "Vowel Jamo Forms. Required for Hangul script when Ancient Hangul writing system is supported."
        },
        {
          "name": '"vkna"',
          "description": "Vertical Kana Alternates. Applies only to fonts that support kana (hiragana and katakana)."
        },
        {
          "name": '"vkrn"',
          "description": "Vertical Kerning."
        },
        {
          "name": '"vpal"',
          "description": "Proportional Alternate Vertical Metrics. Used mostly in CJKV fonts."
        },
        {
          "name": '"vrt2"',
          "description": "Vertical Alternates and Rotation. Applies only to scripts with vertical writing modes."
        },
        {
          "name": '"zero"',
          "description": "Slashed Zero."
        },
        {
          "name": "normal",
          "description": "No change in glyph substitution or positioning occurs."
        },
        {
          "name": "off",
          "description": "Disable feature."
        },
        {
          "name": "on",
          "description": "Enable feature."
        }
      ],
      "syntax": "normal | <feature-tag-value>#",
      "relevance": 57,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/font-feature-settings"
        }
      ],
      "description": "Provides low-level control over OpenType font features. It is intended as a way of providing access to font features that are not widely used but are needed for a particular use case.",
      "restrictions": [
        "string",
        "integer"
      ]
    },
    {
      "name": "font-kerning",
      "browsers": [
        "E79",
        "FF32",
        "S9",
        "C33",
        "O20"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Specifies that kerning is applied at the discretion of the user agent."
        },
        {
          "name": "none",
          "description": "Specifies that kerning is not applied."
        },
        {
          "name": "normal",
          "description": "Specifies that kerning is applied."
        }
      ],
      "syntax": "auto | normal | none",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/font-kerning"
        }
      ],
      "description": "Kerning is the contextual adjustment of inter-glyph spacing. This property controls metric kerning, kerning that utilizes adjustment data contained in the font.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "font-language-override",
      "browsers": [
        "FF34"
      ],
      "values": [
        {
          "name": "normal",
          "description": "Implies that when rendering with OpenType fonts the language of the document is used to infer the OpenType language system, used to select language specific features when rendering."
        }
      ],
      "syntax": "normal | <string>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/font-language-override"
        }
      ],
      "description": "The value of 'normal' implies that when rendering with OpenType fonts the language of the document is used to infer the OpenType language system, used to select language specific features when rendering.",
      "restrictions": [
        "string"
      ]
    },
    {
      "name": "font-size",
      "values": [
        {
          "name": "large"
        },
        {
          "name": "larger"
        },
        {
          "name": "medium"
        },
        {
          "name": "small"
        },
        {
          "name": "smaller"
        },
        {
          "name": "x-large"
        },
        {
          "name": "x-small"
        },
        {
          "name": "xx-large"
        },
        {
          "name": "xx-small"
        }
      ],
      "syntax": "<absolute-size> | <relative-size> | <length-percentage>",
      "relevance": 95,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/font-size"
        }
      ],
      "description": "Indicates the desired height of glyphs from the font. For scalable fonts, the font-size is a scale factor applied to the EM unit of the font. (Note that certain glyphs may bleed outside their EM box.) For non-scalable fonts, the font-size is converted into absolute units and matched against the declared font-size of the font, using the same absolute coordinate space for both of the matched values.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "font-size-adjust",
      "browsers": [
        "E79",
        "FF40",
        "C43",
        "O30"
      ],
      "values": [
        {
          "name": "none",
          "description": "Do not preserve the font\u2019s x-height."
        }
      ],
      "syntax": "none | [ ex-height | cap-height | ch-width | ic-width | ic-height ]? [ from-font | <number> ]",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/font-size-adjust"
        }
      ],
      "description": "Preserves the readability of text when font fallback occurs by adjusting the font-size so that the x-height is the same regardless of the font used.",
      "restrictions": [
        "number"
      ]
    },
    {
      "name": "font-stretch",
      "values": [
        {
          "name": "condensed"
        },
        {
          "name": "expanded"
        },
        {
          "name": "extra-condensed"
        },
        {
          "name": "extra-expanded"
        },
        {
          "name": "narrower",
          "description": "Indicates a narrower value relative to the width of the parent element."
        },
        {
          "name": "normal"
        },
        {
          "name": "semi-condensed"
        },
        {
          "name": "semi-expanded"
        },
        {
          "name": "ultra-condensed"
        },
        {
          "name": "ultra-expanded"
        },
        {
          "name": "wider",
          "description": "Indicates a wider value relative to the width of the parent element."
        }
      ],
      "syntax": "<font-stretch-absolute>{1,2}",
      "relevance": 56,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/font-stretch"
        }
      ],
      "description": "Selects a normal, condensed, or expanded face from a font family.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "font-style",
      "values": [
        {
          "name": "italic",
          "description": "Selects a font that is labeled as an 'italic' face, or an 'oblique' face if one is not"
        },
        {
          "name": "normal",
          "description": "Selects a face that is classified as 'normal'."
        },
        {
          "name": "oblique",
          "description": "Selects a font that is labeled as an 'oblique' face, or an 'italic' face if one is not."
        }
      ],
      "syntax": "normal | italic | oblique <angle>{0,2}",
      "relevance": 90,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/font-style"
        }
      ],
      "description": "Allows italic or oblique faces to be selected. Italic forms are generally cursive in nature while oblique faces are typically sloped versions of the regular face.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "font-synthesis",
      "browsers": [
        "E97",
        "FF34",
        "S9",
        "C97",
        "O83"
      ],
      "values": [
        {
          "name": "none",
          "description": "Disallow all synthetic faces."
        },
        {
          "name": "style",
          "description": "Allow synthetic italic faces."
        },
        {
          "name": "weight",
          "description": "Allow synthetic bold faces."
        }
      ],
      "syntax": "none | [ weight || style || small-caps ]",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/font-synthesis"
        }
      ],
      "description": "Controls whether user agents are allowed to synthesize bold or oblique font faces when a font family lacks bold or italic faces.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "font-variant",
      "values": [
        {
          "name": "normal",
          "description": "Specifies a face that is not labeled as a small-caps font."
        },
        {
          "name": "small-caps",
          "description": "Specifies a font that is labeled as a small-caps font. If a genuine small-caps font is not available, user agents should simulate a small-caps font."
        }
      ],
      "syntax": "normal | none | [ <common-lig-values> || <discretionary-lig-values> || <historical-lig-values> || <contextual-alt-values> || stylistic(<feature-value-name>) || historical-forms || styleset(<feature-value-name>#) || character-variant(<feature-value-name>#) || swash(<feature-value-name>) || ornaments(<feature-value-name>) || annotation(<feature-value-name>) || [ small-caps | all-small-caps | petite-caps | all-petite-caps | unicase | titling-caps ] || <numeric-figure-values> || <numeric-spacing-values> || <numeric-fraction-values> || ordinal || slashed-zero || <east-asian-variant-values> || <east-asian-width-values> || ruby ]",
      "relevance": 64,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/font-variant"
        }
      ],
      "description": "Specifies variant representations of the font",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "font-variant-alternates",
      "browsers": [
        "FF34",
        "S9.1"
      ],
      "values": [
        {
          "name": "annotation()",
          "description": "Enables display of alternate annotation forms."
        },
        {
          "name": "character-variant()",
          "description": "Enables display of specific character variants."
        },
        {
          "name": "historical-forms",
          "description": "Enables display of historical forms."
        },
        {
          "name": "normal",
          "description": "None of the features are enabled."
        },
        {
          "name": "ornaments()",
          "description": "Enables replacement of default glyphs with ornaments, if provided in the font."
        },
        {
          "name": "styleset()",
          "description": "Enables display with stylistic sets."
        },
        {
          "name": "stylistic()",
          "description": "Enables display of stylistic alternates."
        },
        {
          "name": "swash()",
          "description": "Enables display of swash glyphs."
        }
      ],
      "syntax": "normal | [ stylistic( <feature-value-name> ) || historical-forms || styleset( <feature-value-name># ) || character-variant( <feature-value-name># ) || swash( <feature-value-name> ) || ornaments( <feature-value-name> ) || annotation( <feature-value-name> ) ]",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/font-variant-alternates"
        }
      ],
      "description": "For any given character, fonts can provide a variety of alternate glyphs in addition to the default glyph for that character. This property provides control over the selection of these alternate glyphs.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "font-variant-caps",
      "browsers": [
        "E79",
        "FF34",
        "S9.1",
        "C52",
        "O39"
      ],
      "values": [
        {
          "name": "all-petite-caps",
          "description": "Enables display of petite capitals for both upper and lowercase letters."
        },
        {
          "name": "all-small-caps",
          "description": "Enables display of small capitals for both upper and lowercase letters."
        },
        {
          "name": "normal",
          "description": "None of the features are enabled."
        },
        {
          "name": "petite-caps",
          "description": "Enables display of petite capitals."
        },
        {
          "name": "small-caps",
          "description": "Enables display of small capitals. Small-caps glyphs typically use the form of uppercase letters but are reduced to the size of lowercase letters."
        },
        {
          "name": "titling-caps",
          "description": "Enables display of titling capitals."
        },
        {
          "name": "unicase",
          "description": "Enables display of mixture of small capitals for uppercase letters with normal lowercase letters."
        }
      ],
      "syntax": "normal | small-caps | all-small-caps | petite-caps | all-petite-caps | unicase | titling-caps",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/font-variant-caps"
        }
      ],
      "description": "Specifies control over capitalized forms.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "font-variant-east-asian",
      "browsers": [
        "E79",
        "FF34",
        "S9.1",
        "C63",
        "O50"
      ],
      "values": [
        {
          "name": "full-width",
          "description": "Enables rendering of full-width variants."
        },
        {
          "name": "jis04",
          "description": "Enables rendering of JIS04 forms."
        },
        {
          "name": "jis78",
          "description": "Enables rendering of JIS78 forms."
        },
        {
          "name": "jis83",
          "description": "Enables rendering of JIS83 forms."
        },
        {
          "name": "jis90",
          "description": "Enables rendering of JIS90 forms."
        },
        {
          "name": "normal",
          "description": "None of the features are enabled."
        },
        {
          "name": "proportional-width",
          "description": "Enables rendering of proportionally-spaced variants."
        },
        {
          "name": "ruby",
          "description": "Enables display of ruby variant glyphs."
        },
        {
          "name": "simplified",
          "description": "Enables rendering of simplified forms."
        },
        {
          "name": "traditional",
          "description": "Enables rendering of traditional forms."
        }
      ],
      "syntax": "normal | [ <east-asian-variant-values> || <east-asian-width-values> || ruby ]",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/font-variant-east-asian"
        }
      ],
      "description": "Allows control of glyph substitute and positioning in East Asian text.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "font-variant-ligatures",
      "browsers": [
        "E79",
        "FF34",
        "S9.1",
        "C34",
        "O21"
      ],
      "values": [
        {
          "name": "additional-ligatures",
          "description": "Enables display of additional ligatures."
        },
        {
          "name": "common-ligatures",
          "description": "Enables display of common ligatures."
        },
        {
          "name": "contextual",
          "browsers": [
            "E79",
            "FF34",
            "S9.1",
            "C34",
            "O21"
          ],
          "description": "Enables display of contextual alternates."
        },
        {
          "name": "discretionary-ligatures",
          "description": "Enables display of discretionary ligatures."
        },
        {
          "name": "historical-ligatures",
          "description": "Enables display of historical ligatures."
        },
        {
          "name": "no-additional-ligatures",
          "description": "Disables display of additional ligatures."
        },
        {
          "name": "no-common-ligatures",
          "description": "Disables display of common ligatures."
        },
        {
          "name": "no-contextual",
          "browsers": [
            "E79",
            "FF34",
            "S9.1",
            "C34",
            "O21"
          ],
          "description": "Disables display of contextual alternates."
        },
        {
          "name": "no-discretionary-ligatures",
          "description": "Disables display of discretionary ligatures."
        },
        {
          "name": "no-historical-ligatures",
          "description": "Disables display of historical ligatures."
        },
        {
          "name": "none",
          "browsers": [
            "E79",
            "FF34",
            "S9.1",
            "C34",
            "O21"
          ],
          "description": "Disables all ligatures."
        },
        {
          "name": "normal",
          "description": "Implies that the defaults set by the font are used."
        }
      ],
      "syntax": "normal | none | [ <common-lig-values> || <discretionary-lig-values> || <historical-lig-values> || <contextual-alt-values> ]",
      "relevance": 53,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/font-variant-ligatures"
        }
      ],
      "description": "Specifies control over which ligatures are enabled or disabled. A value of \u2018normal\u2019 implies that the defaults set by the font are used.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "font-variant-numeric",
      "browsers": [
        "E79",
        "FF34",
        "S9.1",
        "C52",
        "O39"
      ],
      "values": [
        {
          "name": "diagonal-fractions",
          "description": "Enables display of lining diagonal fractions."
        },
        {
          "name": "lining-nums",
          "description": "Enables display of lining numerals."
        },
        {
          "name": "normal",
          "description": "None of the features are enabled."
        },
        {
          "name": "oldstyle-nums",
          "description": "Enables display of old-style numerals."
        },
        {
          "name": "ordinal",
          "description": "Enables display of letter forms used with ordinal numbers."
        },
        {
          "name": "proportional-nums",
          "description": "Enables display of proportional numerals."
        },
        {
          "name": "slashed-zero",
          "description": "Enables display of slashed zeros."
        },
        {
          "name": "stacked-fractions",
          "description": "Enables display of lining stacked fractions."
        },
        {
          "name": "tabular-nums",
          "description": "Enables display of tabular numerals."
        }
      ],
      "syntax": "normal | [ <numeric-figure-values> || <numeric-spacing-values> || <numeric-fraction-values> || ordinal || slashed-zero ]",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/font-variant-numeric"
        }
      ],
      "description": "Specifies control over numerical forms.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "font-variant-position",
      "browsers": [
        "FF34",
        "S9.1"
      ],
      "values": [
        {
          "name": "normal",
          "description": "None of the features are enabled."
        },
        {
          "name": "sub",
          "description": "Enables display of subscript variants (OpenType feature: subs)."
        },
        {
          "name": "super",
          "description": "Enables display of superscript variants (OpenType feature: sups)."
        }
      ],
      "syntax": "normal | sub | super",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/font-variant-position"
        }
      ],
      "description": "Specifies the vertical position",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "font-weight",
      "values": [
        {
          "name": "100",
          "description": "Thin"
        },
        {
          "name": "200",
          "description": "Extra Light (Ultra Light)"
        },
        {
          "name": "300",
          "description": "Light"
        },
        {
          "name": "400",
          "description": "Normal"
        },
        {
          "name": "500",
          "description": "Medium"
        },
        {
          "name": "600",
          "description": "Semi Bold (Demi Bold)"
        },
        {
          "name": "700",
          "description": "Bold"
        },
        {
          "name": "800",
          "description": "Extra Bold (Ultra Bold)"
        },
        {
          "name": "900",
          "description": "Black (Heavy)"
        },
        {
          "name": "bold",
          "description": "Same as 700"
        },
        {
          "name": "bolder",
          "description": "Specifies the weight of the face bolder than the inherited value."
        },
        {
          "name": "lighter",
          "description": "Specifies the weight of the face lighter than the inherited value."
        },
        {
          "name": "normal",
          "description": "Same as 400"
        }
      ],
      "syntax": "<font-weight-absolute>{1,2}",
      "relevance": 94,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/font-weight"
        }
      ],
      "description": "Specifies weight of glyphs in the font, their degree of blackness or stroke thickness.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "glyph-orientation-horizontal",
      "relevance": 50,
      "description": "Controls glyph orientation when the inline-progression-direction is horizontal.",
      "restrictions": [
        "angle",
        "number"
      ]
    },
    {
      "name": "glyph-orientation-vertical",
      "values": [
        {
          "name": "auto",
          "description": "Sets the orientation based on the fullwidth or non-fullwidth characters and the most common orientation."
        }
      ],
      "relevance": 50,
      "description": "Controls glyph orientation when the inline-progression-direction is vertical.",
      "restrictions": [
        "angle",
        "number",
        "enum"
      ]
    },
    {
      "name": "grid-area",
      "browsers": [
        "E16",
        "FF52",
        "S10.1",
        "C57",
        "O44"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The property contributes nothing to the grid item\u2019s placement, indicating auto-placement, an automatic span, or a default span of one."
        },
        {
          "name": "span",
          "description": "Contributes a grid span to the grid item\u2019s placement such that the corresponding edge of the grid item\u2019s grid area is N lines from its opposite edge."
        }
      ],
      "syntax": "<grid-line> [ / <grid-line> ]{0,3}",
      "relevance": 53,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/grid-area"
        }
      ],
      "description": "Determine a grid item\u2019s size and location within the grid by contributing a line, a span, or nothing (automatic) to its grid placement. Shorthand for 'grid-row-start', 'grid-column-start', 'grid-row-end', and 'grid-column-end'.",
      "restrictions": [
        "identifier",
        "integer"
      ]
    },
    {
      "name": "grid",
      "browsers": [
        "E16",
        "FF52",
        "S10.1",
        "C57",
        "O44"
      ],
      "syntax": "<'grid-template'> | <'grid-template-rows'> / [ auto-flow && dense? ] <'grid-auto-columns'>? | [ auto-flow && dense? ] <'grid-auto-rows'>? / <'grid-template-columns'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/grid"
        }
      ],
      "description": "The grid CSS property is a shorthand property that sets all of the explicit grid properties ('grid-template-rows', 'grid-template-columns', and 'grid-template-areas'), and all the implicit grid properties ('grid-auto-rows', 'grid-auto-columns', and 'grid-auto-flow'), in a single declaration.",
      "restrictions": [
        "identifier",
        "length",
        "percentage",
        "string",
        "enum"
      ]
    },
    {
      "name": "grid-auto-columns",
      "values": [
        {
          "name": "min-content",
          "description": "Represents the largest min-content contribution of the grid items occupying the grid track."
        },
        {
          "name": "max-content",
          "description": "Represents the largest max-content contribution of the grid items occupying the grid track."
        },
        {
          "name": "auto",
          "description": "As a maximum, identical to 'max-content'. As a minimum, represents the largest minimum size (as specified by min-width/min-height) of the grid items occupying the grid track."
        },
        {
          "name": "minmax()",
          "description": "Defines a size range greater than or equal to min and less than or equal to max."
        }
      ],
      "syntax": "<track-size>+",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/grid-auto-columns"
        }
      ],
      "description": "Specifies the size of implicitly created columns.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "grid-auto-flow",
      "browsers": [
        "E16",
        "FF52",
        "S10.1",
        "C57",
        "O44"
      ],
      "values": [
        {
          "name": "row",
          "description": "The auto-placement algorithm places items by filling each row in turn, adding new rows as necessary."
        },
        {
          "name": "column",
          "description": "The auto-placement algorithm places items by filling each column in turn, adding new columns as necessary."
        },
        {
          "name": "dense",
          "description": "If specified, the auto-placement algorithm uses a \u201Cdense\u201D packing algorithm, which attempts to fill in holes earlier in the grid if smaller items come up later."
        }
      ],
      "syntax": "[ row | column ] || dense",
      "relevance": 52,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/grid-auto-flow"
        }
      ],
      "description": "Controls how the auto-placement algorithm works, specifying exactly how auto-placed items get flowed into the grid.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "grid-auto-rows",
      "values": [
        {
          "name": "min-content",
          "description": "Represents the largest min-content contribution of the grid items occupying the grid track."
        },
        {
          "name": "max-content",
          "description": "Represents the largest max-content contribution of the grid items occupying the grid track."
        },
        {
          "name": "auto",
          "description": "As a maximum, identical to 'max-content'. As a minimum, represents the largest minimum size (as specified by min-width/min-height) of the grid items occupying the grid track."
        },
        {
          "name": "minmax()",
          "description": "Defines a size range greater than or equal to min and less than or equal to max."
        }
      ],
      "syntax": "<track-size>+",
      "relevance": 51,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/grid-auto-rows"
        }
      ],
      "description": "Specifies the size of implicitly created rows.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "grid-column",
      "browsers": [
        "E16",
        "FF52",
        "S10.1",
        "C57",
        "O44"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The property contributes nothing to the grid item\u2019s placement, indicating auto-placement, an automatic span, or a default span of one."
        },
        {
          "name": "span",
          "description": "Contributes a grid span to the grid item\u2019s placement such that the corresponding edge of the grid item\u2019s grid area is N lines from its opposite edge."
        }
      ],
      "syntax": "<grid-line> [ / <grid-line> ]?",
      "relevance": 53,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/grid-column"
        }
      ],
      "description": "Shorthand for 'grid-column-start' and 'grid-column-end'.",
      "restrictions": [
        "identifier",
        "integer",
        "enum"
      ]
    },
    {
      "name": "grid-column-end",
      "browsers": [
        "E16",
        "FF52",
        "S10.1",
        "C57",
        "O44"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The property contributes nothing to the grid item\u2019s placement, indicating auto-placement, an automatic span, or a default span of one."
        },
        {
          "name": "span",
          "description": "Contributes a grid span to the grid item\u2019s placement such that the corresponding edge of the grid item\u2019s grid area is N lines from its opposite edge."
        }
      ],
      "syntax": "<grid-line>",
      "relevance": 51,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/grid-column-end"
        }
      ],
      "description": "Determine a grid item\u2019s size and location within the grid by contributing a line, a span, or nothing (automatic) to its grid placement.",
      "restrictions": [
        "identifier",
        "integer",
        "enum"
      ]
    },
    {
      "name": "grid-column-gap",
      "browsers": [
        "FF52",
        "C57",
        "S10.1",
        "O44"
      ],
      "status": "obsolete",
      "syntax": "<length-percentage>",
      "relevance": 2,
      "description": "Specifies the gutters between grid columns. Replaced by 'column-gap' property.",
      "restrictions": [
        "length"
      ]
    },
    {
      "name": "grid-column-start",
      "browsers": [
        "E16",
        "FF52",
        "S10.1",
        "C57",
        "O44"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The property contributes nothing to the grid item\u2019s placement, indicating auto-placement, an automatic span, or a default span of one."
        },
        {
          "name": "span",
          "description": "Contributes a grid span to the grid item\u2019s placement such that the corresponding edge of the grid item\u2019s grid area is N lines from its opposite edge."
        }
      ],
      "syntax": "<grid-line>",
      "relevance": 51,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/grid-column-start"
        }
      ],
      "description": "Determine a grid item\u2019s size and location within the grid by contributing a line, a span, or nothing (automatic) to its grid placement.",
      "restrictions": [
        "identifier",
        "integer",
        "enum"
      ]
    },
    {
      "name": "grid-gap",
      "browsers": [
        "FF52",
        "C57",
        "S10.1",
        "O44"
      ],
      "status": "obsolete",
      "syntax": "<'grid-row-gap'> <'grid-column-gap'>?",
      "relevance": 3,
      "description": "Shorthand that specifies the gutters between grid columns and grid rows in one declaration. Replaced by 'gap' property.",
      "restrictions": [
        "length"
      ]
    },
    {
      "name": "grid-row",
      "browsers": [
        "E16",
        "FF52",
        "S10.1",
        "C57",
        "O44"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The property contributes nothing to the grid item\u2019s placement, indicating auto-placement, an automatic span, or a default span of one."
        },
        {
          "name": "span",
          "description": "Contributes a grid span to the grid item\u2019s placement such that the corresponding edge of the grid item\u2019s grid area is N lines from its opposite edge."
        }
      ],
      "syntax": "<grid-line> [ / <grid-line> ]?",
      "relevance": 52,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/grid-row"
        }
      ],
      "description": "Shorthand for 'grid-row-start' and 'grid-row-end'.",
      "restrictions": [
        "identifier",
        "integer",
        "enum"
      ]
    },
    {
      "name": "grid-row-end",
      "browsers": [
        "E16",
        "FF52",
        "S10.1",
        "C57",
        "O44"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The property contributes nothing to the grid item\u2019s placement, indicating auto-placement, an automatic span, or a default span of one."
        },
        {
          "name": "span",
          "description": "Contributes a grid span to the grid item\u2019s placement such that the corresponding edge of the grid item\u2019s grid area is N lines from its opposite edge."
        }
      ],
      "syntax": "<grid-line>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/grid-row-end"
        }
      ],
      "description": "Determine a grid item\u2019s size and location within the grid by contributing a line, a span, or nothing (automatic) to its grid placement.",
      "restrictions": [
        "identifier",
        "integer",
        "enum"
      ]
    },
    {
      "name": "grid-row-gap",
      "browsers": [
        "FF52",
        "C57",
        "S10.1",
        "O44"
      ],
      "status": "obsolete",
      "syntax": "<length-percentage>",
      "relevance": 1,
      "description": "Specifies the gutters between grid rows. Replaced by 'row-gap' property.",
      "restrictions": [
        "length"
      ]
    },
    {
      "name": "grid-row-start",
      "browsers": [
        "E16",
        "FF52",
        "S10.1",
        "C57",
        "O44"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The property contributes nothing to the grid item\u2019s placement, indicating auto-placement, an automatic span, or a default span of one."
        },
        {
          "name": "span",
          "description": "Contributes a grid span to the grid item\u2019s placement such that the corresponding edge of the grid item\u2019s grid area is N lines from its opposite edge."
        }
      ],
      "syntax": "<grid-line>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/grid-row-start"
        }
      ],
      "description": "Determine a grid item\u2019s size and location within the grid by contributing a line, a span, or nothing (automatic) to its grid placement.",
      "restrictions": [
        "identifier",
        "integer",
        "enum"
      ]
    },
    {
      "name": "grid-template",
      "browsers": [
        "E16",
        "FF52",
        "S10.1",
        "C57",
        "O44"
      ],
      "values": [
        {
          "name": "none",
          "description": "Sets all three properties to their initial values."
        },
        {
          "name": "min-content",
          "description": "Represents the largest min-content contribution of the grid items occupying the grid track."
        },
        {
          "name": "max-content",
          "description": "Represents the largest max-content contribution of the grid items occupying the grid track."
        },
        {
          "name": "auto",
          "description": "As a maximum, identical to 'max-content'. As a minimum, represents the largest minimum size (as specified by min-width/min-height) of the grid items occupying the grid track."
        },
        {
          "name": "subgrid",
          "description": "Sets 'grid-template-rows' and 'grid-template-columns' to 'subgrid', and 'grid-template-areas' to its initial value."
        },
        {
          "name": "minmax()",
          "description": "Defines a size range greater than or equal to min and less than or equal to max."
        },
        {
          "name": "repeat()",
          "description": "Represents a repeated fragment of the track list, allowing a large number of columns or rows that exhibit a recurring pattern to be written in a more compact form."
        }
      ],
      "syntax": "none | [ <'grid-template-rows'> / <'grid-template-columns'> ] | [ <line-names>? <string> <track-size>? <line-names>? ]+ [ / <explicit-track-list> ]?",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/grid-template"
        }
      ],
      "description": "Shorthand for setting grid-template-columns, grid-template-rows, and grid-template-areas in a single declaration.",
      "restrictions": [
        "identifier",
        "length",
        "percentage",
        "string",
        "enum"
      ]
    },
    {
      "name": "grid-template-areas",
      "browsers": [
        "E16",
        "FF52",
        "S10.1",
        "C57",
        "O44"
      ],
      "values": [
        {
          "name": "none",
          "description": "The grid container doesn\u2019t define any named grid areas."
        }
      ],
      "syntax": "none | <string>+",
      "relevance": 52,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/grid-template-areas"
        }
      ],
      "description": "Specifies named grid areas, which are not associated with any particular grid item, but can be referenced from the grid-placement properties.",
      "restrictions": [
        "string"
      ]
    },
    {
      "name": "grid-template-columns",
      "values": [
        {
          "name": "none",
          "description": "There is no explicit grid; any rows/columns will be implicitly generated."
        },
        {
          "name": "min-content",
          "description": "Represents the largest min-content contribution of the grid items occupying the grid track."
        },
        {
          "name": "max-content",
          "description": "Represents the largest max-content contribution of the grid items occupying the grid track."
        },
        {
          "name": "auto",
          "description": "As a maximum, identical to 'max-content'. As a minimum, represents the largest minimum size (as specified by min-width/min-height) of the grid items occupying the grid track."
        },
        {
          "name": "subgrid",
          "description": "Indicates that the grid will align to its parent grid in that axis."
        },
        {
          "name": "minmax()",
          "description": "Defines a size range greater than or equal to min and less than or equal to max."
        },
        {
          "name": "repeat()",
          "description": "Represents a repeated fragment of the track list, allowing a large number of columns or rows that exhibit a recurring pattern to be written in a more compact form."
        }
      ],
      "syntax": "none | <track-list> | <auto-track-list> | subgrid <line-name-list>?",
      "relevance": 58,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/grid-template-columns"
        }
      ],
      "description": "specifies, as a space-separated track list, the line names and track sizing functions of the grid.",
      "restrictions": [
        "identifier",
        "length",
        "percentage",
        "enum"
      ]
    },
    {
      "name": "grid-template-rows",
      "values": [
        {
          "name": "none",
          "description": "There is no explicit grid; any rows/columns will be implicitly generated."
        },
        {
          "name": "min-content",
          "description": "Represents the largest min-content contribution of the grid items occupying the grid track."
        },
        {
          "name": "max-content",
          "description": "Represents the largest max-content contribution of the grid items occupying the grid track."
        },
        {
          "name": "auto",
          "description": "As a maximum, identical to 'max-content'. As a minimum, represents the largest minimum size (as specified by min-width/min-height) of the grid items occupying the grid track."
        },
        {
          "name": "subgrid",
          "description": "Indicates that the grid will align to its parent grid in that axis."
        },
        {
          "name": "minmax()",
          "description": "Defines a size range greater than or equal to min and less than or equal to max."
        },
        {
          "name": "repeat()",
          "description": "Represents a repeated fragment of the track list, allowing a large number of columns or rows that exhibit a recurring pattern to be written in a more compact form."
        }
      ],
      "syntax": "none | <track-list> | <auto-track-list> | subgrid <line-name-list>?",
      "relevance": 54,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/grid-template-rows"
        }
      ],
      "description": "specifies, as a space-separated track list, the line names and track sizing functions of the grid.",
      "restrictions": [
        "identifier",
        "length",
        "percentage",
        "string",
        "enum"
      ]
    },
    {
      "name": "height",
      "values": [
        {
          "name": "auto",
          "description": "The height depends on the values of other properties."
        },
        {
          "name": "fit-content",
          "description": "Use the fit-content inline size or fit-content block size, as appropriate to the writing mode."
        },
        {
          "name": "max-content",
          "description": "Use the max-content inline size or max-content block size, as appropriate to the writing mode."
        },
        {
          "name": "min-content",
          "description": "Use the min-content inline size or min-content block size, as appropriate to the writing mode."
        }
      ],
      "syntax": "<viewport-length>{1,2}",
      "relevance": 96,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/height"
        }
      ],
      "description": "Specifies the height of the content area, padding area or border area (depending on 'box-sizing') of certain boxes.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "hyphens",
      "values": [
        {
          "name": "auto",
          "description": "Conditional hyphenation characters inside a word, if present, take priority over automatic resources when determining hyphenation points within the word."
        },
        {
          "name": "manual",
          "description": "Words are only broken at line breaks where there are characters inside the word that suggest line break opportunities"
        },
        {
          "name": "none",
          "description": "Words are not broken at line breaks, even if characters inside the word suggest line break points."
        }
      ],
      "syntax": "none | manual | auto",
      "relevance": 55,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/hyphens"
        }
      ],
      "description": "Controls whether hyphenation is allowed to create more break opportunities within a line of text.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "image-orientation",
      "browsers": [
        "E81",
        "FF26",
        "S13.1",
        "C81",
        "O67"
      ],
      "values": [
        {
          "name": "flip",
          "description": "After rotating by the precededing angle, the image is flipped horizontally. Defaults to 0deg if the angle is ommitted."
        },
        {
          "name": "from-image",
          "description": "If the image has an orientation specified in its metadata, such as EXIF, this value computes to the angle that the metadata specifies is necessary to correctly orient the image."
        }
      ],
      "syntax": "from-image | <angle> | [ <angle>? flip ]",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/image-orientation"
        }
      ],
      "description": "Specifies an orthogonal rotation to be applied to an image before it is laid out.",
      "restrictions": [
        "angle"
      ]
    },
    {
      "name": "image-rendering",
      "browsers": [
        "E79",
        "FF3.6",
        "S6",
        "C13",
        "O15"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The image should be scaled with an algorithm that maximizes the appearance of the image."
        },
        {
          "name": "crisp-edges",
          "description": "The image must be scaled with an algorithm that preserves contrast and edges in the image, and which does not smooth colors or introduce blur to the image in the process."
        },
        {
          "name": "-moz-crisp-edges",
          "browsers": [
            "E79",
            "FF3.6",
            "S6",
            "C13",
            "O15"
          ]
        },
        {
          "name": "optimizeQuality",
          "description": "Deprecated."
        },
        {
          "name": "optimizeSpeed",
          "description": "Deprecated."
        },
        {
          "name": "pixelated",
          "description": "When scaling the image up, the 'nearest neighbor' or similar algorithm must be used, so that the image appears to be simply composed of very large pixels."
        }
      ],
      "syntax": "auto | crisp-edges | pixelated",
      "relevance": 54,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/image-rendering"
        }
      ],
      "description": "Provides a hint to the user-agent about what aspects of an image are most important to preserve when the image is scaled, to aid the user-agent in the choice of an appropriate scaling algorithm.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "ime-mode",
      "browsers": [
        "E12",
        "FF3",
        "IE5"
      ],
      "values": [
        {
          "name": "active",
          "description": "The input method editor is initially active; text entry is performed using it unless the user specifically dismisses it."
        },
        {
          "name": "auto",
          "description": "No change is made to the current input method editor state. This is the default."
        },
        {
          "name": "disabled",
          "description": "The input method editor is disabled and may not be activated by the user."
        },
        {
          "name": "inactive",
          "description": "The input method editor is initially inactive, but the user may activate it if they wish."
        },
        {
          "name": "normal",
          "description": "The IME state should be normal; this value can be used in a user style sheet to override the page setting."
        }
      ],
      "status": "obsolete",
      "syntax": "auto | normal | active | inactive | disabled",
      "relevance": 0,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/ime-mode"
        }
      ],
      "description": "Controls the state of the input method editor for text fields.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "inline-size",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C57",
        "O44"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Depends on the values of other properties."
        }
      ],
      "syntax": "<'width'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/inline-size"
        }
      ],
      "description": "Size of an element in the direction specified by 'writing-mode'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "isolation",
      "browsers": [
        "E79",
        "FF36",
        "S8",
        "C41",
        "O30"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Elements are not isolated unless an operation is applied that causes the creation of a stacking context."
        },
        {
          "name": "isolate",
          "description": "In CSS will turn the element into a stacking context."
        }
      ],
      "syntax": "auto | isolate",
      "relevance": 51,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/isolation"
        }
      ],
      "description": "In CSS setting to 'isolate' will turn the element into a stacking context. In SVG, it defines whether an element is isolated or not.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "justify-content",
      "values": [
        {
          "name": "center",
          "description": "Flex items are packed toward the center of the line."
        },
        {
          "name": "start",
          "description": "The items are packed flush to each other toward the start edge of the alignment container in the main axis."
        },
        {
          "name": "end",
          "description": "The items are packed flush to each other toward the end edge of the alignment container in the main axis."
        },
        {
          "name": "left",
          "description": "The items are packed flush to each other toward the left edge of the alignment container in the main axis."
        },
        {
          "name": "right",
          "description": "The items are packed flush to each other toward the right edge of the alignment container in the main axis."
        },
        {
          "name": "safe",
          "description": "If the size of the item overflows the alignment container, the item is instead aligned as if the alignment mode were start."
        },
        {
          "name": "unsafe",
          "description": "Regardless of the relative sizes of the item and alignment container, the given alignment value is honored."
        },
        {
          "name": "stretch",
          "description": "If the combined size of the alignment subjects is less than the size of the alignment container, any auto-sized alignment subjects have their size increased equally (not proportionally), while still respecting the constraints imposed by max-height/max-width (or equivalent functionality), so that the combined size exactly fills the alignment container."
        },
        {
          "name": "space-evenly",
          "description": "The items are evenly distributed within the alignment container along the main axis."
        },
        {
          "name": "flex-end",
          "description": "Flex items are packed toward the end of the line."
        },
        {
          "name": "flex-start",
          "description": "Flex items are packed toward the start of the line."
        },
        {
          "name": "space-around",
          "description": "Flex items are evenly distributed in the line, with half-size spaces on either end."
        },
        {
          "name": "space-between",
          "description": "Flex items are evenly distributed in the line."
        },
        {
          "name": "baseline",
          "description": "Specifies participation in first-baseline alignment."
        },
        {
          "name": "first baseline",
          "description": "Specifies participation in first-baseline alignment."
        },
        {
          "name": "last baseline",
          "description": "Specifies participation in last-baseline alignment."
        }
      ],
      "syntax": "normal | <content-distribution> | <overflow-position>? [ <content-position> | left | right ]",
      "relevance": 85,
      "description": "Aligns flex items along the main axis of the current line of the flex container.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "kerning",
      "values": [
        {
          "name": "auto",
          "description": "Indicates that the user agent should adjust inter-glyph spacing based on kerning tables that are included in the font that will be used."
        }
      ],
      "relevance": 50,
      "description": "Indicates whether the user agent should adjust inter-glyph spacing based on kerning tables that are included in the relevant font or instead disable auto-kerning and set inter-character spacing to a specific length.",
      "restrictions": [
        "length",
        "enum"
      ]
    },
    {
      "name": "left",
      "values": [
        {
          "name": "auto",
          "description": "For non-replaced elements, the effect of this value depends on which of related properties have the value 'auto' as well"
        }
      ],
      "syntax": "<length> | <percentage> | auto",
      "relevance": 95,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/left"
        }
      ],
      "description": "Specifies how far an absolutely positioned box's left margin edge is offset to the right of the left edge of the box's 'containing block'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "letter-spacing",
      "values": [
        {
          "name": "normal",
          "description": "The spacing is the normal spacing for the current font. It is typically zero-length."
        }
      ],
      "syntax": "normal | <length>",
      "relevance": 81,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/letter-spacing"
        }
      ],
      "description": "Specifies the minimum, maximum, and optimal spacing between grapheme clusters.",
      "restrictions": [
        "length"
      ]
    },
    {
      "name": "lighting-color",
      "browsers": [
        "E",
        "C5",
        "FF3",
        "IE10",
        "O9",
        "S6"
      ],
      "relevance": 50,
      "description": "Defines the color of the light source for filter primitives 'feDiffuseLighting' and 'feSpecularLighting'.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "line-break",
      "values": [
        {
          "name": "auto",
          "description": "The UA determines the set of line-breaking restrictions to use for CJK scripts, and it may vary the restrictions based on the length of the line; e.g., use a less restrictive set of line-break rules for short lines."
        },
        {
          "name": "loose",
          "description": "Breaks text using the least restrictive set of line-breaking rules. Typically used for short lines, such as in newspapers."
        },
        {
          "name": "normal",
          "description": "Breaks text using the most common set of line-breaking rules."
        },
        {
          "name": "strict",
          "description": "Breaks CJK scripts using a more restrictive set of line-breaking rules than 'normal'."
        }
      ],
      "syntax": "auto | loose | normal | strict | anywhere",
      "relevance": 51,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/line-break"
        }
      ],
      "description": "Specifies what set of line breaking restrictions are in effect within the element.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "line-height",
      "values": [
        {
          "name": "normal",
          "description": "Tells user agents to set the computed value to a 'reasonable' value based on the font size of the element."
        }
      ],
      "syntax": "normal | <number> | <length> | <percentage>",
      "relevance": 93,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/line-height"
        }
      ],
      "description": "Determines the block-progression dimension of the text content area of an inline box.",
      "restrictions": [
        "number",
        "length",
        "percentage"
      ]
    },
    {
      "name": "list-style",
      "values": [
        {
          "name": "armenian"
        },
        {
          "name": "circle",
          "description": "A hollow circle."
        },
        {
          "name": "decimal"
        },
        {
          "name": "decimal-leading-zero"
        },
        {
          "name": "disc",
          "description": "A filled circle."
        },
        {
          "name": "georgian"
        },
        {
          "name": "inside",
          "description": "The marker box is outside the principal block box, as described in the section on the ::marker pseudo-element below."
        },
        {
          "name": "lower-alpha"
        },
        {
          "name": "lower-greek"
        },
        {
          "name": "lower-latin"
        },
        {
          "name": "lower-roman"
        },
        {
          "name": "none"
        },
        {
          "name": "outside",
          "description": "The ::marker pseudo-element is an inline element placed immediately before all ::before pseudo-elements in the principal block box, after which the element's content flows."
        },
        {
          "name": "square",
          "description": "A filled square."
        },
        {
          "name": "symbols()",
          "description": "Allows a counter style to be defined inline."
        },
        {
          "name": "upper-alpha"
        },
        {
          "name": "upper-latin"
        },
        {
          "name": "upper-roman"
        },
        {
          "name": "url()"
        }
      ],
      "syntax": "<'list-style-type'> || <'list-style-position'> || <'list-style-image'>",
      "relevance": 85,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/list-style"
        }
      ],
      "description": "Shorthand for setting 'list-style-type', 'list-style-position' and 'list-style-image'",
      "restrictions": [
        "image",
        "enum",
        "url"
      ]
    },
    {
      "name": "list-style-image",
      "values": [
        {
          "name": "none",
          "description": "The default contents of the of the list item\u2019s marker are given by 'list-style-type' instead."
        }
      ],
      "syntax": "<image> | none",
      "relevance": 52,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/list-style-image"
        }
      ],
      "description": "Sets the image that will be used as the list item marker. When the image is available, it will replace the marker set with the 'list-style-type' marker.",
      "restrictions": [
        "image"
      ]
    },
    {
      "name": "list-style-position",
      "values": [
        {
          "name": "inside",
          "description": "The marker box is outside the principal block box, as described in the section on the ::marker pseudo-element below."
        },
        {
          "name": "outside",
          "description": "The ::marker pseudo-element is an inline element placed immediately before all ::before pseudo-elements in the principal block box, after which the element's content flows."
        }
      ],
      "syntax": "inside | outside",
      "relevance": 55,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/list-style-position"
        }
      ],
      "description": "Specifies the position of the '::marker' pseudo-element's box in the list item.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "list-style-type",
      "values": [
        {
          "name": "armenian",
          "description": "Traditional uppercase Armenian numbering."
        },
        {
          "name": "circle",
          "description": "A hollow circle."
        },
        {
          "name": "decimal",
          "description": "Western decimal numbers."
        },
        {
          "name": "decimal-leading-zero",
          "description": "Decimal numbers padded by initial zeros."
        },
        {
          "name": "disc",
          "description": "A filled circle."
        },
        {
          "name": "georgian",
          "description": "Traditional Georgian numbering."
        },
        {
          "name": "lower-alpha",
          "description": "Lowercase ASCII letters."
        },
        {
          "name": "lower-greek",
          "description": "Lowercase classical Greek."
        },
        {
          "name": "lower-latin",
          "description": "Lowercase ASCII letters."
        },
        {
          "name": "lower-roman",
          "description": "Lowercase ASCII Roman numerals."
        },
        {
          "name": "none",
          "description": "No marker"
        },
        {
          "name": "square",
          "description": "A filled square."
        },
        {
          "name": "symbols()",
          "description": "Allows a counter style to be defined inline."
        },
        {
          "name": "upper-alpha",
          "description": "Uppercase ASCII letters."
        },
        {
          "name": "upper-latin",
          "description": "Uppercase ASCII letters."
        },
        {
          "name": "upper-roman",
          "description": "Uppercase ASCII Roman numerals."
        }
      ],
      "syntax": "<counter-style> | <string> | none",
      "relevance": 75,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/list-style-type"
        }
      ],
      "description": "Used to construct the default contents of a list item\u2019s marker",
      "restrictions": [
        "enum",
        "string"
      ]
    },
    {
      "name": "margin",
      "values": [
        {
          "name": "auto"
        }
      ],
      "syntax": "[ <length> | <percentage> | auto ]{1,4}",
      "relevance": 96,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/margin"
        }
      ],
      "description": "Shorthand property to set values for the thickness of the margin area. If left is omitted, it is the same as right. If bottom is omitted it is the same as top, if right is omitted it is the same as top. Negative values for margin properties are allowed, but there may be implementation-specific limits.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "margin-block-end",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "values": [
        {
          "name": "auto"
        }
      ],
      "syntax": "<'margin-left'>",
      "relevance": 53,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/margin-block-end"
        }
      ],
      "description": "Logical 'margin-bottom'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "margin-block-start",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "values": [
        {
          "name": "auto"
        }
      ],
      "syntax": "<'margin-left'>",
      "relevance": 53,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/margin-block-start"
        }
      ],
      "description": "Logical 'margin-top'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "margin-bottom",
      "values": [
        {
          "name": "auto"
        }
      ],
      "syntax": "<length> | <percentage> | auto",
      "relevance": 92,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/margin-bottom"
        }
      ],
      "description": "Shorthand property to set values for the thickness of the margin area. If left is omitted, it is the same as right. If bottom is omitted it is the same as top, if right is omitted it is the same as top. Negative values for margin properties are allowed, but there may be implementation-specific limits..",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "margin-inline-end",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "values": [
        {
          "name": "auto"
        }
      ],
      "syntax": "<'margin-left'>",
      "relevance": 51,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/margin-inline-end"
        }
      ],
      "description": "Logical 'margin-right'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "margin-inline-start",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "values": [
        {
          "name": "auto"
        }
      ],
      "syntax": "<'margin-left'>",
      "relevance": 52,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/margin-inline-start"
        }
      ],
      "description": "Logical 'margin-left'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "margin-left",
      "values": [
        {
          "name": "auto"
        }
      ],
      "syntax": "<length> | <percentage> | auto",
      "relevance": 92,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/margin-left"
        }
      ],
      "description": "Shorthand property to set values for the thickness of the margin area. If left is omitted, it is the same as right. If bottom is omitted it is the same as top, if right is omitted it is the same as top. Negative values for margin properties are allowed, but there may be implementation-specific limits..",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "margin-right",
      "values": [
        {
          "name": "auto"
        }
      ],
      "syntax": "<length> | <percentage> | auto",
      "relevance": 91,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/margin-right"
        }
      ],
      "description": "Shorthand property to set values for the thickness of the margin area. If left is omitted, it is the same as right. If bottom is omitted it is the same as top, if right is omitted it is the same as top. Negative values for margin properties are allowed, but there may be implementation-specific limits..",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "margin-top",
      "values": [
        {
          "name": "auto"
        }
      ],
      "syntax": "<length> | <percentage> | auto",
      "relevance": 95,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/margin-top"
        }
      ],
      "description": "Shorthand property to set values for the thickness of the margin area. If left is omitted, it is the same as right. If bottom is omitted it is the same as top, if right is omitted it is the same as top. Negative values for margin properties are allowed, but there may be implementation-specific limits..",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "marker",
      "values": [
        {
          "name": "none",
          "description": "Indicates that no marker symbol will be drawn at the given vertex or vertices."
        },
        {
          "name": "url()",
          "description": "Indicates that the <marker> element referenced will be used."
        }
      ],
      "relevance": 50,
      "description": "Specifies the marker symbol that shall be used for all points on the sets the value for all vertices on the given \u2018path\u2019 element or basic shape.",
      "restrictions": [
        "url"
      ]
    },
    {
      "name": "marker-end",
      "values": [
        {
          "name": "none",
          "description": "Indicates that no marker symbol will be drawn at the given vertex or vertices."
        },
        {
          "name": "url()",
          "description": "Indicates that the <marker> element referenced will be used."
        }
      ],
      "relevance": 50,
      "description": "Specifies the marker that will be drawn at the last vertices of the given markable element.",
      "restrictions": [
        "url"
      ]
    },
    {
      "name": "marker-mid",
      "values": [
        {
          "name": "none",
          "description": "Indicates that no marker symbol will be drawn at the given vertex or vertices."
        },
        {
          "name": "url()",
          "description": "Indicates that the <marker> element referenced will be used."
        }
      ],
      "relevance": 50,
      "description": "Specifies the marker that will be drawn at all vertices except the first and last.",
      "restrictions": [
        "url"
      ]
    },
    {
      "name": "marker-start",
      "values": [
        {
          "name": "none",
          "description": "Indicates that no marker symbol will be drawn at the given vertex or vertices."
        },
        {
          "name": "url()",
          "description": "Indicates that the <marker> element referenced will be used."
        }
      ],
      "relevance": 50,
      "description": "Specifies the marker that will be drawn at the first vertices of the given markable element.",
      "restrictions": [
        "url"
      ]
    },
    {
      "name": "mask-image",
      "browsers": [
        "E79",
        "FF53",
        "S15.4",
        "C1",
        "O15"
      ],
      "values": [
        {
          "name": "none",
          "description": "Counts as a transparent black image layer."
        },
        {
          "name": "url()",
          "description": "Reference to a <mask element or to a CSS image."
        }
      ],
      "syntax": "<mask-reference>#",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/mask-image"
        }
      ],
      "description": "Sets the mask layer image of an element.",
      "restrictions": [
        "url",
        "image",
        "enum"
      ]
    },
    {
      "name": "mask-mode",
      "browsers": [
        "FF53",
        "S15.4"
      ],
      "values": [
        {
          "name": "alpha",
          "description": "Alpha values of the mask layer image should be used as the mask values."
        },
        {
          "name": "auto",
          "description": "Use alpha values if 'mask-image' is an image, luminance if a <mask> element or a CSS image."
        },
        {
          "name": "luminance",
          "description": "Luminance values of the mask layer image should be used as the mask values."
        }
      ],
      "syntax": "<masking-mode>#",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/mask-mode"
        }
      ],
      "description": "Indicates whether the mask layer image is treated as luminance mask or alpha mask.",
      "restrictions": [
        "url",
        "image",
        "enum"
      ]
    },
    {
      "name": "mask-origin",
      "browsers": [
        "E79",
        "FF53",
        "S15.4",
        "C1",
        "O15"
      ],
      "syntax": "<geometry-box>#",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/mask-origin"
        }
      ],
      "description": "Specifies the mask positioning area.",
      "restrictions": [
        "geometry-box",
        "enum"
      ]
    },
    {
      "name": "mask-position",
      "browsers": [
        "E79",
        "FF53",
        "S15.4",
        "C1",
        "O15"
      ],
      "syntax": "<position>#",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/mask-position"
        }
      ],
      "description": "Specifies how mask layer images are positioned.",
      "restrictions": [
        "position",
        "length",
        "percentage"
      ]
    },
    {
      "name": "mask-repeat",
      "browsers": [
        "E79",
        "FF53",
        "S15.4",
        "C1",
        "O15"
      ],
      "syntax": "<repeat-style>#",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/mask-repeat"
        }
      ],
      "description": "Specifies how mask layer images are tiled after they have been sized and positioned.",
      "restrictions": [
        "repeat"
      ]
    },
    {
      "name": "mask-size",
      "browsers": [
        "E79",
        "FF53",
        "S15.4",
        "C4",
        "O15"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Resolved by using the image\u2019s intrinsic ratio and the size of the other dimension, or failing that, using the image\u2019s intrinsic size, or failing that, treating it as 100%."
        },
        {
          "name": "contain",
          "description": "Scale the image, while preserving its intrinsic aspect ratio (if any), to the largest size such that both its width and its height can fit inside the background positioning area."
        },
        {
          "name": "cover",
          "description": "Scale the image, while preserving its intrinsic aspect ratio (if any), to the smallest size such that both its width and its height can completely cover the background positioning area."
        }
      ],
      "syntax": "<bg-size>#",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/mask-size"
        }
      ],
      "description": "Specifies the size of the mask layer images.",
      "restrictions": [
        "length",
        "percentage",
        "enum"
      ]
    },
    {
      "name": "mask-type",
      "browsers": [
        "E79",
        "FF35",
        "S7",
        "C24",
        "O15"
      ],
      "values": [
        {
          "name": "alpha",
          "description": "Indicates that the alpha values of the mask should be used."
        },
        {
          "name": "luminance",
          "description": "Indicates that the luminance values of the mask should be used."
        }
      ],
      "syntax": "luminance | alpha",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/mask-type"
        }
      ],
      "description": "Defines whether the content of the <mask> element is treated as as luminance mask or alpha mask.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "max-block-size",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C57",
        "O44"
      ],
      "values": [
        {
          "name": "none",
          "description": "No limit on the width of the box."
        }
      ],
      "syntax": "<'max-width'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/max-block-size"
        }
      ],
      "description": "Maximum size of an element in the direction opposite that of the direction specified by 'writing-mode'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "max-height",
      "values": [
        {
          "name": "none",
          "description": "No limit on the height of the box."
        },
        {
          "name": "fit-content",
          "description": "Use the fit-content inline size or fit-content block size, as appropriate to the writing mode."
        },
        {
          "name": "max-content",
          "description": "Use the max-content inline size or max-content block size, as appropriate to the writing mode."
        },
        {
          "name": "min-content",
          "description": "Use the min-content inline size or min-content block size, as appropriate to the writing mode."
        }
      ],
      "syntax": "<viewport-length>",
      "relevance": 85,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/max-height"
        }
      ],
      "description": "Allows authors to constrain content height to a certain range.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "max-inline-size",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C57",
        "O44"
      ],
      "values": [
        {
          "name": "none",
          "description": "No limit on the height of the box."
        }
      ],
      "syntax": "<'max-width'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/max-inline-size"
        }
      ],
      "description": "Maximum size of an element in the direction specified by 'writing-mode'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "max-width",
      "values": [
        {
          "name": "none",
          "description": "No limit on the width of the box."
        },
        {
          "name": "fit-content",
          "description": "Use the fit-content inline size or fit-content block size, as appropriate to the writing mode."
        },
        {
          "name": "max-content",
          "description": "Use the max-content inline size or max-content block size, as appropriate to the writing mode."
        },
        {
          "name": "min-content",
          "description": "Use the min-content inline size or min-content block size, as appropriate to the writing mode."
        }
      ],
      "syntax": "<viewport-length>",
      "relevance": 91,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/max-width"
        }
      ],
      "description": "Allows authors to constrain content width to a certain range.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "min-block-size",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C57",
        "O44"
      ],
      "syntax": "<'min-width'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/min-block-size"
        }
      ],
      "description": "Minimal size of an element in the direction opposite that of the direction specified by 'writing-mode'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "min-height",
      "values": [
        {
          "name": "auto"
        },
        {
          "name": "fit-content",
          "description": "Use the fit-content inline size or fit-content block size, as appropriate to the writing mode."
        },
        {
          "name": "max-content",
          "description": "Use the max-content inline size or max-content block size, as appropriate to the writing mode."
        },
        {
          "name": "min-content",
          "description": "Use the min-content inline size or min-content block size, as appropriate to the writing mode."
        }
      ],
      "syntax": "<viewport-length>",
      "relevance": 90,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/min-height"
        }
      ],
      "description": "Allows authors to constrain content height to a certain range.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "min-inline-size",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C57",
        "O44"
      ],
      "syntax": "<'min-width'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/min-inline-size"
        }
      ],
      "description": "Minimal size of an element in the direction specified by 'writing-mode'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "min-width",
      "values": [
        {
          "name": "auto"
        },
        {
          "name": "fit-content",
          "description": "Use the fit-content inline size or fit-content block size, as appropriate to the writing mode."
        },
        {
          "name": "max-content",
          "description": "Use the max-content inline size or max-content block size, as appropriate to the writing mode."
        },
        {
          "name": "min-content",
          "description": "Use the min-content inline size or min-content block size, as appropriate to the writing mode."
        }
      ],
      "syntax": "<viewport-length>",
      "relevance": 88,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/min-width"
        }
      ],
      "description": "Allows authors to constrain content width to a certain range.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "mix-blend-mode",
      "browsers": [
        "E79",
        "FF32",
        "S8",
        "C41",
        "O28"
      ],
      "values": [
        {
          "name": "normal",
          "description": "Default attribute which specifies no blending"
        },
        {
          "name": "multiply",
          "description": "The source color is multiplied by the destination color and replaces the destination."
        },
        {
          "name": "screen",
          "description": "Multiplies the complements of the backdrop and source color values, then complements the result."
        },
        {
          "name": "overlay",
          "description": "Multiplies or screens the colors, depending on the backdrop color value."
        },
        {
          "name": "darken",
          "description": "Selects the darker of the backdrop and source colors."
        },
        {
          "name": "lighten",
          "description": "Selects the lighter of the backdrop and source colors."
        },
        {
          "name": "color-dodge",
          "description": "Brightens the backdrop color to reflect the source color."
        },
        {
          "name": "color-burn",
          "description": "Darkens the backdrop color to reflect the source color."
        },
        {
          "name": "hard-light",
          "description": "Multiplies or screens the colors, depending on the source color value."
        },
        {
          "name": "soft-light",
          "description": "Darkens or lightens the colors, depending on the source color value."
        },
        {
          "name": "difference",
          "description": "Subtracts the darker of the two constituent colors from the lighter color.."
        },
        {
          "name": "exclusion",
          "description": "Produces an effect similar to that of the Difference mode but lower in contrast."
        },
        {
          "name": "hue",
          "browsers": [
            "E79",
            "FF32",
            "S8",
            "C41",
            "O28"
          ],
          "description": "Creates a color with the hue of the source color and the saturation and luminosity of the backdrop color."
        },
        {
          "name": "saturation",
          "browsers": [
            "E79",
            "FF32",
            "S8",
            "C41",
            "O28"
          ],
          "description": "Creates a color with the saturation of the source color and the hue and luminosity of the backdrop color."
        },
        {
          "name": "color",
          "browsers": [
            "E79",
            "FF32",
            "S8",
            "C41",
            "O28"
          ],
          "description": "Creates a color with the hue and saturation of the source color and the luminosity of the backdrop color."
        },
        {
          "name": "luminosity",
          "browsers": [
            "E79",
            "FF32",
            "S8",
            "C41",
            "O28"
          ],
          "description": "Creates a color with the luminosity of the source color and the hue and saturation of the backdrop color."
        }
      ],
      "syntax": "<blend-mode>",
      "relevance": 52,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/mix-blend-mode"
        }
      ],
      "description": "Defines the formula that must be used to mix the colors with the backdrop.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "motion",
      "browsers": [
        "C46",
        "O33"
      ],
      "values": [
        {
          "name": "none",
          "description": "No motion path gets created."
        },
        {
          "name": "path()",
          "description": "Defines an SVG path as a string, with optional 'fill-rule' as the first argument."
        },
        {
          "name": "auto",
          "description": "Indicates that the object is rotated by the angle of the direction of the motion path."
        },
        {
          "name": "reverse",
          "description": "Indicates that the object is rotated by the angle of the direction of the motion path plus 180 degrees."
        }
      ],
      "relevance": 50,
      "description": "Shorthand property for setting 'motion-path', 'motion-offset' and 'motion-rotation'.",
      "restrictions": [
        "url",
        "length",
        "percentage",
        "angle",
        "shape",
        "geometry-box",
        "enum"
      ]
    },
    {
      "name": "motion-offset",
      "browsers": [
        "C46",
        "O33"
      ],
      "relevance": 50,
      "description": "A distance that describes the position along the specified motion path.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "motion-path",
      "browsers": [
        "C46",
        "O33"
      ],
      "values": [
        {
          "name": "none",
          "description": "No motion path gets created."
        },
        {
          "name": "path()",
          "description": "Defines an SVG path as a string, with optional 'fill-rule' as the first argument."
        }
      ],
      "relevance": 50,
      "description": "Specifies the motion path the element gets positioned at.",
      "restrictions": [
        "url",
        "shape",
        "geometry-box",
        "enum"
      ]
    },
    {
      "name": "motion-rotation",
      "browsers": [
        "C46",
        "O33"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Indicates that the object is rotated by the angle of the direction of the motion path."
        },
        {
          "name": "reverse",
          "description": "Indicates that the object is rotated by the angle of the direction of the motion path plus 180 degrees."
        }
      ],
      "relevance": 50,
      "description": "Defines the direction of the element while positioning along the motion path.",
      "restrictions": [
        "angle"
      ]
    },
    {
      "name": "-moz-animation",
      "browsers": [
        "FF9"
      ],
      "values": [
        {
          "name": "alternate",
          "description": "The animation cycle iterations that are odd counts are played in the normal direction, and the animation cycle iterations that are even counts are played in a reverse direction."
        },
        {
          "name": "alternate-reverse",
          "description": "The animation cycle iterations that are odd counts are played in the reverse direction, and the animation cycle iterations that are even counts are played in a normal direction."
        },
        {
          "name": "backwards",
          "description": "The beginning property value (as defined in the first @keyframes at-rule) is applied before the animation is displayed, during the period defined by 'animation-delay'."
        },
        {
          "name": "both",
          "description": "Both forwards and backwards fill modes are applied."
        },
        {
          "name": "forwards",
          "description": "The final property value (as defined in the last @keyframes at-rule) is maintained after the animation completes."
        },
        {
          "name": "infinite",
          "description": "Causes the animation to repeat forever."
        },
        {
          "name": "none",
          "description": "No animation is performed"
        },
        {
          "name": "normal",
          "description": "Normal playback."
        },
        {
          "name": "reverse",
          "description": "All iterations of the animation are played in the reverse direction from the way they were specified."
        }
      ],
      "relevance": 50,
      "description": "Shorthand property combines six of the animation properties into a single property.",
      "restrictions": [
        "time",
        "enum",
        "timing-function",
        "identifier",
        "number"
      ]
    },
    {
      "name": "-moz-animation-delay",
      "browsers": [
        "FF9"
      ],
      "relevance": 50,
      "description": "Defines when the animation will start.",
      "restrictions": [
        "time"
      ]
    },
    {
      "name": "-moz-animation-direction",
      "browsers": [
        "FF9"
      ],
      "values": [
        {
          "name": "alternate",
          "description": "The animation cycle iterations that are odd counts are played in the normal direction, and the animation cycle iterations that are even counts are played in a reverse direction."
        },
        {
          "name": "alternate-reverse",
          "description": "The animation cycle iterations that are odd counts are played in the reverse direction, and the animation cycle iterations that are even counts are played in a normal direction."
        },
        {
          "name": "normal",
          "description": "Normal playback."
        },
        {
          "name": "reverse",
          "description": "All iterations of the animation are played in the reverse direction from the way they were specified."
        }
      ],
      "relevance": 50,
      "description": "Defines whether or not the animation should play in reverse on alternate cycles.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-moz-animation-duration",
      "browsers": [
        "FF9"
      ],
      "relevance": 50,
      "description": "Defines the length of time that an animation takes to complete one cycle.",
      "restrictions": [
        "time"
      ]
    },
    {
      "name": "-moz-animation-iteration-count",
      "browsers": [
        "FF9"
      ],
      "values": [
        {
          "name": "infinite",
          "description": "Causes the animation to repeat forever."
        }
      ],
      "relevance": 50,
      "description": "Defines the number of times an animation cycle is played. The default value is one, meaning the animation will play from beginning to end once.",
      "restrictions": [
        "number",
        "enum"
      ]
    },
    {
      "name": "-moz-animation-name",
      "browsers": [
        "FF9"
      ],
      "values": [
        {
          "name": "none",
          "description": "No animation is performed"
        }
      ],
      "relevance": 50,
      "description": "Defines a list of animations that apply. Each name is used to select the keyframe at-rule that provides the property values for the animation.",
      "restrictions": [
        "identifier",
        "enum"
      ]
    },
    {
      "name": "-moz-animation-play-state",
      "browsers": [
        "FF9"
      ],
      "values": [
        {
          "name": "paused",
          "description": "A running animation will be paused."
        },
        {
          "name": "running",
          "description": "Resume playback of a paused animation."
        }
      ],
      "relevance": 50,
      "description": "Defines whether the animation is running or paused.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-moz-animation-timing-function",
      "browsers": [
        "FF9"
      ],
      "relevance": 50,
      "description": "Describes how the animation will progress over one cycle of its duration. See the 'transition-timing-function'.",
      "restrictions": [
        "timing-function"
      ]
    },
    {
      "name": "-moz-appearance",
      "browsers": [
        "FF1"
      ],
      "values": [
        {
          "name": "button"
        },
        {
          "name": "button-arrow-down"
        },
        {
          "name": "button-arrow-next"
        },
        {
          "name": "button-arrow-previous"
        },
        {
          "name": "button-arrow-up"
        },
        {
          "name": "button-bevel"
        },
        {
          "name": "checkbox"
        },
        {
          "name": "checkbox-container"
        },
        {
          "name": "checkbox-label"
        },
        {
          "name": "dialog"
        },
        {
          "name": "groupbox"
        },
        {
          "name": "listbox"
        },
        {
          "name": "menuarrow"
        },
        {
          "name": "menuimage"
        },
        {
          "name": "menuitem"
        },
        {
          "name": "menuitemtext"
        },
        {
          "name": "menulist"
        },
        {
          "name": "menulist-button"
        },
        {
          "name": "menulist-text"
        },
        {
          "name": "menulist-textfield"
        },
        {
          "name": "menupopup"
        },
        {
          "name": "menuradio"
        },
        {
          "name": "menuseparator"
        },
        {
          "name": "-moz-mac-unified-toolbar"
        },
        {
          "name": "-moz-win-borderless-glass"
        },
        {
          "name": "-moz-win-browsertabbar-toolbox"
        },
        {
          "name": "-moz-win-communications-toolbox"
        },
        {
          "name": "-moz-win-glass"
        },
        {
          "name": "-moz-win-media-toolbox"
        },
        {
          "name": "none"
        },
        {
          "name": "progressbar"
        },
        {
          "name": "progresschunk"
        },
        {
          "name": "radio"
        },
        {
          "name": "radio-container"
        },
        {
          "name": "radio-label"
        },
        {
          "name": "radiomenuitem"
        },
        {
          "name": "resizer"
        },
        {
          "name": "resizerpanel"
        },
        {
          "name": "scrollbarbutton-down"
        },
        {
          "name": "scrollbarbutton-left"
        },
        {
          "name": "scrollbarbutton-right"
        },
        {
          "name": "scrollbarbutton-up"
        },
        {
          "name": "scrollbar-small"
        },
        {
          "name": "scrollbartrack-horizontal"
        },
        {
          "name": "scrollbartrack-vertical"
        },
        {
          "name": "separator"
        },
        {
          "name": "spinner"
        },
        {
          "name": "spinner-downbutton"
        },
        {
          "name": "spinner-textfield"
        },
        {
          "name": "spinner-upbutton"
        },
        {
          "name": "statusbar"
        },
        {
          "name": "statusbarpanel"
        },
        {
          "name": "tab"
        },
        {
          "name": "tabpanels"
        },
        {
          "name": "tab-scroll-arrow-back"
        },
        {
          "name": "tab-scroll-arrow-forward"
        },
        {
          "name": "textfield"
        },
        {
          "name": "textfield-multiline"
        },
        {
          "name": "toolbar"
        },
        {
          "name": "toolbox"
        },
        {
          "name": "tooltip"
        },
        {
          "name": "treeheadercell"
        },
        {
          "name": "treeheadersortarrow"
        },
        {
          "name": "treeitem"
        },
        {
          "name": "treetwistyopen"
        },
        {
          "name": "treeview"
        },
        {
          "name": "treewisty"
        },
        {
          "name": "window"
        }
      ],
      "status": "nonstandard",
      "syntax": "none | button | button-arrow-down | button-arrow-next | button-arrow-previous | button-arrow-up | button-bevel | button-focus | caret | checkbox | checkbox-container | checkbox-label | checkmenuitem | dualbutton | groupbox | listbox | listitem | menuarrow | menubar | menucheckbox | menuimage | menuitem | menuitemtext | menulist | menulist-button | menulist-text | menulist-textfield | menupopup | menuradio | menuseparator | meterbar | meterchunk | progressbar | progressbar-vertical | progresschunk | progresschunk-vertical | radio | radio-container | radio-label | radiomenuitem | range | range-thumb | resizer | resizerpanel | scale-horizontal | scalethumbend | scalethumb-horizontal | scalethumbstart | scalethumbtick | scalethumb-vertical | scale-vertical | scrollbarbutton-down | scrollbarbutton-left | scrollbarbutton-right | scrollbarbutton-up | scrollbarthumb-horizontal | scrollbarthumb-vertical | scrollbartrack-horizontal | scrollbartrack-vertical | searchfield | separator | sheet | spinner | spinner-downbutton | spinner-textfield | spinner-upbutton | splitter | statusbar | statusbarpanel | tab | tabpanel | tabpanels | tab-scroll-arrow-back | tab-scroll-arrow-forward | textfield | textfield-multiline | toolbar | toolbarbutton | toolbarbutton-dropdown | toolbargripper | toolbox | tooltip | treeheader | treeheadercell | treeheadersortarrow | treeitem | treeline | treetwisty | treetwistyopen | treeview | -moz-mac-unified-toolbar | -moz-win-borderless-glass | -moz-win-browsertabbar-toolbox | -moz-win-communicationstext | -moz-win-communications-toolbox | -moz-win-exclude-glass | -moz-win-glass | -moz-win-mediatext | -moz-win-media-toolbox | -moz-window-button-box | -moz-window-button-box-maximized | -moz-window-button-close | -moz-window-button-maximize | -moz-window-button-minimize | -moz-window-button-restore | -moz-window-frame-bottom | -moz-window-frame-left | -moz-window-frame-right | -moz-window-titlebar | -moz-window-titlebar-maximized",
      "relevance": 0,
      "description": "Used in Gecko (Firefox) to display an element using a platform-native styling based on the operating system's theme.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-moz-backface-visibility",
      "browsers": [
        "FF10"
      ],
      "values": [
        {
          "name": "hidden"
        },
        {
          "name": "visible"
        }
      ],
      "relevance": 50,
      "description": "Determines whether or not the 'back' side of a transformed element is visible when facing the viewer. With an identity transform, the front side of an element faces the viewer.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-moz-background-clip",
      "browsers": [
        "FF1-3.6"
      ],
      "values": [
        {
          "name": "padding"
        }
      ],
      "relevance": 50,
      "description": "Determines the background painting area.",
      "restrictions": [
        "box",
        "enum"
      ]
    },
    {
      "name": "-moz-background-inline-policy",
      "browsers": [
        "FF1"
      ],
      "values": [
        {
          "name": "bounding-box"
        },
        {
          "name": "continuous"
        },
        {
          "name": "each-box"
        }
      ],
      "relevance": 50,
      "description": "In Gecko-based applications like Firefox, the -moz-background-inline-policy CSS property specifies how the background image of an inline element is determined when the content of the inline element wraps onto multiple lines. The choice of position has significant effects on repetition.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-moz-background-origin",
      "browsers": [
        "FF1"
      ],
      "relevance": 50,
      "description": "For elements rendered as a single box, specifies the background positioning area. For elements rendered as multiple boxes (e.g., inline boxes on several lines, boxes on several pages) specifies which boxes 'box-decoration-break' operates on to determine the background positioning area(s).",
      "restrictions": [
        "box"
      ]
    },
    {
      "name": "-moz-border-bottom-colors",
      "browsers": [
        "FF1"
      ],
      "status": "nonstandard",
      "syntax": "<color>+ | none",
      "relevance": 0,
      "description": "Sets a list of colors for the bottom border.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "-moz-border-image",
      "browsers": [
        "FF3.6"
      ],
      "values": [
        {
          "name": "auto",
          "description": "If 'auto' is specified then the border image width is the intrinsic width or height (whichever is applicable) of the corresponding image slice. If the image does not have the required intrinsic dimension then the corresponding border-width is used instead."
        },
        {
          "name": "fill",
          "description": "Causes the middle part of the border-image to be preserved."
        },
        {
          "name": "none"
        },
        {
          "name": "repeat",
          "description": "The image is tiled (repeated) to fill the area."
        },
        {
          "name": "round",
          "description": "The image is tiled (repeated) to fill the area. If it does not fill the area with a whole number of tiles, the image is rescaled so that it does."
        },
        {
          "name": "space",
          "description": "The image is tiled (repeated) to fill the area. If it does not fill the area with a whole number of tiles, the extra space is distributed around the tiles."
        },
        {
          "name": "stretch",
          "description": "The image is stretched to fill the area."
        },
        {
          "name": "url()"
        }
      ],
      "relevance": 50,
      "description": "Shorthand property for setting 'border-image-source', 'border-image-slice', 'border-image-width', 'border-image-outset' and 'border-image-repeat'. Omitted values are set to their initial values.",
      "restrictions": [
        "length",
        "percentage",
        "number",
        "url",
        "enum"
      ]
    },
    {
      "name": "-moz-border-left-colors",
      "browsers": [
        "FF1"
      ],
      "status": "nonstandard",
      "syntax": "<color>+ | none",
      "relevance": 0,
      "description": "Sets a list of colors for the bottom border.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "-moz-border-right-colors",
      "browsers": [
        "FF1"
      ],
      "status": "nonstandard",
      "syntax": "<color>+ | none",
      "relevance": 0,
      "description": "Sets a list of colors for the bottom border.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "-moz-border-top-colors",
      "browsers": [
        "FF1"
      ],
      "status": "nonstandard",
      "syntax": "<color>+ | none",
      "relevance": 0,
      "description": "Ske Firefox, -moz-border-bottom-colors sets a list of colors for the bottom border.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "-moz-box-align",
      "browsers": [
        "FF1"
      ],
      "values": [
        {
          "name": "baseline",
          "description": "If this box orientation is inline-axis or horizontal, all children are placed with their baselines aligned, and extra space placed before or after as necessary. For block flows, the baseline of the first non-empty line box located within the element is used. For tables, the baseline of the first cell is used."
        },
        {
          "name": "center",
          "description": "Any extra space is divided evenly, with half placed above the child and the other half placed after the child."
        },
        {
          "name": "end",
          "description": "For normal direction boxes, the bottom edge of each child is placed along the bottom of the box. Extra space is placed above the element. For reverse direction boxes, the top edge of each child is placed along the top of the box. Extra space is placed below the element."
        },
        {
          "name": "start",
          "description": "For normal direction boxes, the top edge of each child is placed along the top of the box. Extra space is placed below the element. For reverse direction boxes, the bottom edge of each child is placed along the bottom of the box. Extra space is placed above the element."
        },
        {
          "name": "stretch",
          "description": "The height of each child is adjusted to that of the containing block."
        }
      ],
      "relevance": 50,
      "description": "Specifies how a XUL box aligns its contents across (perpendicular to) the direction of its layout. The effect of this is only visible if there is extra space in the box.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-moz-box-direction",
      "browsers": [
        "FF1"
      ],
      "values": [
        {
          "name": "normal",
          "description": "A box with a computed value of horizontal for box-orient displays its children from left to right. A box with a computed value of vertical displays its children from top to bottom."
        },
        {
          "name": "reverse",
          "description": "A box with a computed value of horizontal for box-orient displays its children from right to left. A box with a computed value of vertical displays its children from bottom to top."
        }
      ],
      "relevance": 50,
      "description": "Specifies whether a box lays out its contents normally (from the top or left edge), or in reverse (from the bottom or right edge).",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-moz-box-flex",
      "browsers": [
        "FF1"
      ],
      "relevance": 50,
      "description": "Specifies how a box grows to fill the box that contains it, in the direction of the containing box's layout.",
      "restrictions": [
        "number"
      ]
    },
    {
      "name": "-moz-box-flexgroup",
      "browsers": [
        "FF1"
      ],
      "relevance": 50,
      "description": "Flexible elements can be assigned to flex groups using the 'box-flex-group' property.",
      "restrictions": [
        "integer"
      ]
    },
    {
      "name": "-moz-box-ordinal-group",
      "browsers": [
        "FF1"
      ],
      "relevance": 50,
      "description": "Indicates the ordinal group the element belongs to. Elements with a lower ordinal group are displayed before those with a higher ordinal group.",
      "restrictions": [
        "integer"
      ]
    },
    {
      "name": "-moz-box-orient",
      "browsers": [
        "FF1"
      ],
      "values": [
        {
          "name": "block-axis",
          "description": "Elements are oriented along the box's axis."
        },
        {
          "name": "horizontal",
          "description": "The box displays its children from left to right in a horizontal line."
        },
        {
          "name": "inline-axis",
          "description": "Elements are oriented vertically."
        },
        {
          "name": "vertical",
          "description": "The box displays its children from stacked from top to bottom vertically."
        }
      ],
      "relevance": 50,
      "description": "In Mozilla applications, -moz-box-orient specifies whether a box lays out its contents horizontally or vertically.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-moz-box-pack",
      "browsers": [
        "FF1"
      ],
      "values": [
        {
          "name": "center",
          "description": "The extra space is divided evenly, with half placed before the first child and the other half placed after the last child."
        },
        {
          "name": "end",
          "description": "For normal direction boxes, the right edge of the last child is placed at the right side, with all extra space placed before the first child. For reverse direction boxes, the left edge of the first child is placed at the left side, with all extra space placed after the last child."
        },
        {
          "name": "justify",
          "description": "The space is divided evenly in-between each child, with none of the extra space placed before the first child or after the last child. If there is only one child, treat the pack value as if it were start."
        },
        {
          "name": "start",
          "description": "For normal direction boxes, the left edge of the first child is placed at the left side, with all extra space placed after the last child. For reverse direction boxes, the right edge of the last child is placed at the right side, with all extra space placed before the first child."
        }
      ],
      "relevance": 50,
      "description": "Specifies how a box packs its contents in the direction of its layout. The effect of this is only visible if there is extra space in the box.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-moz-box-sizing",
      "browsers": [
        "FF1"
      ],
      "values": [
        {
          "name": "border-box",
          "description": "The specified width and height (and respective min/max properties) on this element determine the border box of the element."
        },
        {
          "name": "content-box",
          "description": "Behavior of width and height as specified by CSS2.1. The specified width and height (and respective min/max properties) apply to the width and height respectively of the content box of the element."
        },
        {
          "name": "padding-box",
          "description": "The specified width and height (and respective min/max properties) on this element determine the padding box of the element."
        }
      ],
      "relevance": 50,
      "description": "Box Model addition in CSS3.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-moz-column-count",
      "browsers": [
        "FF3.5"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Determines the number of columns by the 'column-width' property and the element width."
        }
      ],
      "relevance": 50,
      "description": "Describes the optimal number of columns into which the content of the element will be flowed.",
      "restrictions": [
        "integer"
      ]
    },
    {
      "name": "-moz-column-gap",
      "browsers": [
        "FF3.5"
      ],
      "values": [
        {
          "name": "normal",
          "description": "User agent specific and typically equivalent to 1em."
        }
      ],
      "relevance": 50,
      "description": "Sets the gap between columns. If there is a column rule between columns, it will appear in the middle of the gap.",
      "restrictions": [
        "length"
      ]
    },
    {
      "name": "-moz-column-rule",
      "browsers": [
        "FF3.5"
      ],
      "relevance": 50,
      "description": "Shorthand for setting 'column-rule-width', 'column-rule-style', and 'column-rule-color' at the same place in the style sheet. Omitted values are set to their initial values.",
      "restrictions": [
        "length",
        "line-width",
        "line-style",
        "color"
      ]
    },
    {
      "name": "-moz-column-rule-color",
      "browsers": [
        "FF3.5"
      ],
      "relevance": 50,
      "description": "Sets the color of the column rule",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "-moz-column-rule-style",
      "browsers": [
        "FF3.5"
      ],
      "relevance": 50,
      "description": "Sets the style of the rule between columns of an element.",
      "restrictions": [
        "line-style"
      ]
    },
    {
      "name": "-moz-column-rule-width",
      "browsers": [
        "FF3.5"
      ],
      "relevance": 50,
      "description": "Sets the width of the rule between columns. Negative values are not allowed.",
      "restrictions": [
        "length",
        "line-width"
      ]
    },
    {
      "name": "-moz-columns",
      "browsers": [
        "FF9"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The width depends on the values of other properties."
        }
      ],
      "relevance": 50,
      "description": "A shorthand property which sets both 'column-width' and 'column-count'.",
      "restrictions": [
        "length",
        "integer"
      ]
    },
    {
      "name": "-moz-column-width",
      "browsers": [
        "FF3.5"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The width depends on the values of other properties."
        }
      ],
      "relevance": 50,
      "description": "This property describes the width of columns in multicol elements.",
      "restrictions": [
        "length"
      ]
    },
    {
      "name": "-moz-font-feature-settings",
      "browsers": [
        "FF4"
      ],
      "values": [
        {
          "name": '"c2cs"'
        },
        {
          "name": '"dlig"'
        },
        {
          "name": '"kern"'
        },
        {
          "name": '"liga"'
        },
        {
          "name": '"lnum"'
        },
        {
          "name": '"onum"'
        },
        {
          "name": '"smcp"'
        },
        {
          "name": '"swsh"'
        },
        {
          "name": '"tnum"'
        },
        {
          "name": "normal",
          "description": "No change in glyph substitution or positioning occurs."
        },
        {
          "name": "off",
          "browsers": [
            "FF4"
          ]
        },
        {
          "name": "on",
          "browsers": [
            "FF4"
          ]
        }
      ],
      "relevance": 50,
      "description": "Provides low-level control over OpenType font features. It is intended as a way of providing access to font features that are not widely used but are needed for a particular use case.",
      "restrictions": [
        "string",
        "integer"
      ]
    },
    {
      "name": "-moz-hyphens",
      "browsers": [
        "FF9"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Conditional hyphenation characters inside a word, if present, take priority over automatic resources when determining hyphenation points within the word."
        },
        {
          "name": "manual",
          "description": "Words are only broken at line breaks where there are characters inside the word that suggest line break opportunities"
        },
        {
          "name": "none",
          "description": "Words are not broken at line breaks, even if characters inside the word suggest line break points."
        }
      ],
      "relevance": 50,
      "description": "Controls whether hyphenation is allowed to create more break opportunities within a line of text.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-moz-perspective",
      "browsers": [
        "FF10"
      ],
      "values": [
        {
          "name": "none",
          "description": "No perspective transform is applied."
        }
      ],
      "relevance": 50,
      "description": "Applies the same transform as the perspective(<number>) transform function, except that it applies only to the positioned or transformed children of the element, not to the transform on the element itself.",
      "restrictions": [
        "length"
      ]
    },
    {
      "name": "-moz-perspective-origin",
      "browsers": [
        "FF10"
      ],
      "relevance": 50,
      "description": "Establishes the origin for the perspective property. It effectively sets the X and Y position at which the viewer appears to be looking at the children of the element.",
      "restrictions": [
        "position",
        "percentage",
        "length"
      ]
    },
    {
      "name": "-moz-text-align-last",
      "browsers": [
        "FF12"
      ],
      "values": [
        {
          "name": "auto"
        },
        {
          "name": "center",
          "description": "The inline contents are centered within the line box."
        },
        {
          "name": "justify",
          "description": "The text is justified according to the method specified by the 'text-justify' property."
        },
        {
          "name": "left",
          "description": "The inline contents are aligned to the left edge of the line box. In vertical text, 'left' aligns to the edge of the line box that would be the start edge for left-to-right text."
        },
        {
          "name": "right",
          "description": "The inline contents are aligned to the right edge of the line box. In vertical text, 'right' aligns to the edge of the line box that would be the end edge for left-to-right text."
        }
      ],
      "relevance": 50,
      "description": "Describes how the last line of a block or a line right before a forced line break is aligned when 'text-align' is set to 'justify'.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-moz-text-decoration-color",
      "browsers": [
        "FF6"
      ],
      "relevance": 50,
      "description": "Specifies the color of text decoration (underlines overlines, and line-throughs) set on the element with text-decoration-line.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "-moz-text-decoration-line",
      "browsers": [
        "FF6"
      ],
      "values": [
        {
          "name": "line-through",
          "description": "Each line of text has a line through the middle."
        },
        {
          "name": "none",
          "description": "Neither produces nor inhibits text decoration."
        },
        {
          "name": "overline",
          "description": "Each line of text has a line above it."
        },
        {
          "name": "underline",
          "description": "Each line of text is underlined."
        }
      ],
      "relevance": 50,
      "description": "Specifies what line decorations, if any, are added to the element.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-moz-text-decoration-style",
      "browsers": [
        "FF6"
      ],
      "values": [
        {
          "name": "dashed",
          "description": "Produces a dashed line style."
        },
        {
          "name": "dotted",
          "description": "Produces a dotted line."
        },
        {
          "name": "double",
          "description": "Produces a double line."
        },
        {
          "name": "none",
          "description": "Produces no line."
        },
        {
          "name": "solid",
          "description": "Produces a solid line."
        },
        {
          "name": "wavy",
          "description": "Produces a wavy line."
        }
      ],
      "relevance": 50,
      "description": "Specifies the line style for underline, line-through and overline text decoration.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-moz-text-size-adjust",
      "browsers": [
        "FF"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Renderers must use the default size adjustment when displaying on a small device."
        },
        {
          "name": "none",
          "description": "Renderers must not do size adjustment when displaying on a small device."
        }
      ],
      "relevance": 50,
      "description": "Specifies a size adjustment for displaying text content in mobile browsers.",
      "restrictions": [
        "enum",
        "percentage"
      ]
    },
    {
      "name": "-moz-transform",
      "browsers": [
        "FF3.5"
      ],
      "values": [
        {
          "name": "matrix()",
          "description": "Specifies a 2D transformation in the form of a transformation matrix of six values. matrix(a,b,c,d,e,f) is equivalent to applying the transformation matrix [a b c d e f]"
        },
        {
          "name": "matrix3d()",
          "description": "Specifies a 3D transformation as a 4x4 homogeneous matrix of 16 values in column-major order."
        },
        {
          "name": "none"
        },
        {
          "name": "perspective",
          "description": "Specifies a perspective projection matrix."
        },
        {
          "name": "rotate()",
          "description": "Specifies a 2D rotation by the angle specified in the parameter about the origin of the element, as defined by the transform-origin property."
        },
        {
          "name": "rotate3d()",
          "description": "Specifies a clockwise 3D rotation by the angle specified in last parameter about the [x,y,z] direction vector described by the first 3 parameters."
        },
        {
          "name": "rotateX('angle')",
          "description": "Specifies a clockwise rotation by the given angle about the X axis."
        },
        {
          "name": "rotateY('angle')",
          "description": "Specifies a clockwise rotation by the given angle about the Y axis."
        },
        {
          "name": "rotateZ('angle')",
          "description": "Specifies a clockwise rotation by the given angle about the Z axis."
        },
        {
          "name": "scale()",
          "description": "Specifies a 2D scale operation by the [sx,sy] scaling vector described by the 2 parameters. If the second parameter is not provided, it is takes a value equal to the first."
        },
        {
          "name": "scale3d()",
          "description": "Specifies a 3D scale operation by the [sx,sy,sz] scaling vector described by the 3 parameters."
        },
        {
          "name": "scaleX()",
          "description": "Specifies a scale operation using the [sx,1] scaling vector, where sx is given as the parameter."
        },
        {
          "name": "scaleY()",
          "description": "Specifies a scale operation using the [sy,1] scaling vector, where sy is given as the parameter."
        },
        {
          "name": "scaleZ()",
          "description": "Specifies a scale operation using the [1,1,sz] scaling vector, where sz is given as the parameter."
        },
        {
          "name": "skew()",
          "description": "Specifies a skew transformation along the X and Y axes. The first angle parameter specifies the skew on the X axis. The second angle parameter specifies the skew on the Y axis. If the second parameter is not given then a value of 0 is used for the Y angle (ie: no skew on the Y axis)."
        },
        {
          "name": "skewX()",
          "description": "Specifies a skew transformation along the X axis by the given angle."
        },
        {
          "name": "skewY()",
          "description": "Specifies a skew transformation along the Y axis by the given angle."
        },
        {
          "name": "translate()",
          "description": "Specifies a 2D translation by the vector [tx, ty], where tx is the first translation-value parameter and ty is the optional second translation-value parameter."
        },
        {
          "name": "translate3d()",
          "description": "Specifies a 3D translation by the vector [tx,ty,tz], with tx, ty and tz being the first, second and third translation-value parameters respectively."
        },
        {
          "name": "translateX()",
          "description": "Specifies a translation by the given amount in the X direction."
        },
        {
          "name": "translateY()",
          "description": "Specifies a translation by the given amount in the Y direction."
        },
        {
          "name": "translateZ()",
          "description": "Specifies a translation by the given amount in the Z direction. Note that percentage values are not allowed in the translateZ translation-value, and if present are evaluated as 0."
        }
      ],
      "relevance": 50,
      "description": "A two-dimensional transformation is applied to an element through the 'transform' property. This property contains a list of transform functions similar to those allowed by SVG.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-moz-transform-origin",
      "browsers": [
        "FF3.5"
      ],
      "relevance": 50,
      "description": "Establishes the origin of transformation for an element.",
      "restrictions": [
        "position",
        "length",
        "percentage"
      ]
    },
    {
      "name": "-moz-transition",
      "browsers": [
        "FF4"
      ],
      "values": [
        {
          "name": "all",
          "description": "Every property that is able to undergo a transition will do so."
        },
        {
          "name": "none",
          "description": "No property will transition."
        }
      ],
      "relevance": 50,
      "description": "Shorthand property combines four of the transition properties into a single property.",
      "restrictions": [
        "time",
        "property",
        "timing-function",
        "enum"
      ]
    },
    {
      "name": "-moz-transition-delay",
      "browsers": [
        "FF4"
      ],
      "relevance": 50,
      "description": "Defines when the transition will start. It allows a transition to begin execution some period of time from when it is applied.",
      "restrictions": [
        "time"
      ]
    },
    {
      "name": "-moz-transition-duration",
      "browsers": [
        "FF4"
      ],
      "relevance": 50,
      "description": "Specifies how long the transition from the old value to the new value should take.",
      "restrictions": [
        "time"
      ]
    },
    {
      "name": "-moz-transition-property",
      "browsers": [
        "FF4"
      ],
      "values": [
        {
          "name": "all",
          "description": "Every property that is able to undergo a transition will do so."
        },
        {
          "name": "none",
          "description": "No property will transition."
        }
      ],
      "relevance": 50,
      "description": "Specifies the name of the CSS property to which the transition is applied.",
      "restrictions": [
        "property"
      ]
    },
    {
      "name": "-moz-transition-timing-function",
      "browsers": [
        "FF4"
      ],
      "relevance": 50,
      "description": "Describes how the intermediate values used during a transition will be calculated.",
      "restrictions": [
        "timing-function"
      ]
    },
    {
      "name": "-moz-user-focus",
      "browsers": [
        "FF1"
      ],
      "values": [
        {
          "name": "ignore"
        },
        {
          "name": "normal"
        }
      ],
      "status": "nonstandard",
      "syntax": "ignore | normal | select-after | select-before | select-menu | select-same | select-all | none",
      "relevance": 0,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/-moz-user-focus"
        }
      ],
      "description": "Used to indicate whether the element can have focus."
    },
    {
      "name": "-moz-user-select",
      "browsers": [
        "FF1.5"
      ],
      "values": [
        {
          "name": "all"
        },
        {
          "name": "element"
        },
        {
          "name": "elements"
        },
        {
          "name": "-moz-all"
        },
        {
          "name": "-moz-none"
        },
        {
          "name": "none"
        },
        {
          "name": "text"
        },
        {
          "name": "toggle"
        }
      ],
      "relevance": 50,
      "description": "Controls the appearance of selection.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-accelerator",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "false",
          "description": "The element does not contain an accelerator key sequence."
        },
        {
          "name": "true",
          "description": "The element contains an accelerator key sequence."
        }
      ],
      "status": "nonstandard",
      "syntax": "false | true",
      "relevance": 0,
      "description": "IE only. Has the ability to turn off its system underlines for accelerator keys until the ALT key is pressed",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-behavior",
      "browsers": [
        "IE8"
      ],
      "relevance": 50,
      "description": "IE only. Used to extend behaviors of the browser",
      "restrictions": [
        "url"
      ]
    },
    {
      "name": "-ms-block-progression",
      "browsers": [
        "IE8"
      ],
      "values": [
        {
          "name": "bt",
          "description": "Bottom-to-top block flow. Layout is horizontal."
        },
        {
          "name": "lr",
          "description": "Left-to-right direction. The flow orientation is vertical."
        },
        {
          "name": "rl",
          "description": "Right-to-left direction. The flow orientation is vertical."
        },
        {
          "name": "tb",
          "description": "Top-to-bottom direction. The flow orientation is horizontal."
        }
      ],
      "status": "nonstandard",
      "syntax": "tb | rl | bt | lr",
      "relevance": 0,
      "description": "Sets the block-progression value and the flow orientation",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-content-zoom-chaining",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "chained",
          "description": "The nearest zoomable parent element begins zooming when the user hits a zoom limit during a manipulation. No bounce effect is shown."
        },
        {
          "name": "none",
          "description": "A bounce effect is shown when the user hits a zoom limit during a manipulation."
        }
      ],
      "status": "nonstandard",
      "syntax": "none | chained",
      "relevance": 0,
      "description": "Specifies the zoom behavior that occurs when a user hits the zoom limit during a manipulation."
    },
    {
      "name": "-ms-content-zooming",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "none",
          "description": "The element is not zoomable."
        },
        {
          "name": "zoom",
          "description": "The element is zoomable."
        }
      ],
      "status": "nonstandard",
      "syntax": "none | zoom",
      "relevance": 0,
      "description": "Specifies whether zooming is enabled.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-content-zoom-limit",
      "browsers": [
        "E",
        "IE10"
      ],
      "status": "nonstandard",
      "syntax": "<'-ms-content-zoom-limit-min'> <'-ms-content-zoom-limit-max'>",
      "relevance": 0,
      "description": "Shorthand property for the -ms-content-zoom-limit-min and -ms-content-zoom-limit-max properties.",
      "restrictions": [
        "percentage"
      ]
    },
    {
      "name": "-ms-content-zoom-limit-max",
      "browsers": [
        "E",
        "IE10"
      ],
      "status": "nonstandard",
      "syntax": "<percentage>",
      "relevance": 0,
      "description": "Specifies the maximum zoom factor.",
      "restrictions": [
        "percentage"
      ]
    },
    {
      "name": "-ms-content-zoom-limit-min",
      "browsers": [
        "E",
        "IE10"
      ],
      "status": "nonstandard",
      "syntax": "<percentage>",
      "relevance": 0,
      "description": "Specifies the minimum zoom factor.",
      "restrictions": [
        "percentage"
      ]
    },
    {
      "name": "-ms-content-zoom-snap",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "mandatory",
          "description": "Indicates that the motion of the content after the contact is picked up is always adjusted so that it lands on a snap-point."
        },
        {
          "name": "none",
          "description": "Indicates that zooming is unaffected by any defined snap-points."
        },
        {
          "name": "proximity",
          "description": 'Indicates that the motion of the content after the contact is picked up may be adjusted if the content would normally stop "close enough" to a snap-point.'
        },
        {
          "name": "snapInterval(100%, 100%)",
          "description": "Specifies where the snap-points will be placed."
        },
        {
          "name": "snapList()",
          "description": "Specifies the position of individual snap-points as a comma-separated list of zoom factors."
        }
      ],
      "status": "nonstandard",
      "syntax": "<'-ms-content-zoom-snap-type'> || <'-ms-content-zoom-snap-points'>",
      "relevance": 0,
      "description": "Shorthand property for the -ms-content-zoom-snap-type and -ms-content-zoom-snap-points properties."
    },
    {
      "name": "-ms-content-zoom-snap-points",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "snapInterval(100%, 100%)",
          "description": "Specifies where the snap-points will be placed."
        },
        {
          "name": "snapList()",
          "description": "Specifies the position of individual snap-points as a comma-separated list of zoom factors."
        }
      ],
      "status": "nonstandard",
      "syntax": "snapInterval( <percentage>, <percentage> ) | snapList( <percentage># )",
      "relevance": 0,
      "description": "Defines where zoom snap-points are located."
    },
    {
      "name": "-ms-content-zoom-snap-type",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "mandatory",
          "description": "Indicates that the motion of the content after the contact is picked up is always adjusted so that it lands on a snap-point."
        },
        {
          "name": "none",
          "description": "Indicates that zooming is unaffected by any defined snap-points."
        },
        {
          "name": "proximity",
          "description": 'Indicates that the motion of the content after the contact is picked up may be adjusted if the content would normally stop "close enough" to a snap-point.'
        }
      ],
      "status": "nonstandard",
      "syntax": "none | proximity | mandatory",
      "relevance": 0,
      "description": "Specifies how zooming is affected by defined snap-points.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-filter",
      "browsers": [
        "IE8-9"
      ],
      "status": "nonstandard",
      "syntax": "<string>",
      "relevance": 0,
      "description": "IE only. Used to produce visual effects.",
      "restrictions": [
        "string"
      ]
    },
    {
      "name": "-ms-flex",
      "browsers": [
        "IE10"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Retrieves the value of the main size property as the used 'flex-basis'."
        },
        {
          "name": "none",
          "description": "Expands to '0 0 auto'."
        }
      ],
      "relevance": 50,
      "description": "specifies the parameters of a flexible length: the positive and negative flexibility, and the preferred size.",
      "restrictions": [
        "length",
        "number",
        "percentage"
      ]
    },
    {
      "name": "-ms-flex-align",
      "browsers": [
        "IE10"
      ],
      "values": [
        {
          "name": "baseline",
          "description": "If the flex item\u2019s inline axis is the same as the cross axis, this value is identical to 'flex-start'. Otherwise, it participates in baseline alignment."
        },
        {
          "name": "center",
          "description": "The flex item\u2019s margin box is centered in the cross axis within the line."
        },
        {
          "name": "end",
          "description": "The cross-end margin edge of the flex item is placed flush with the cross-end edge of the line."
        },
        {
          "name": "start",
          "description": "The cross-start margin edge of the flexbox item is placed flush with the cross-start edge of the line."
        },
        {
          "name": "stretch",
          "description": "If the cross size property of the flexbox item is anything other than 'auto', this value is identical to 'start'."
        }
      ],
      "relevance": 50,
      "description": "Aligns flex items along the cross axis of the current line of the flex container.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-flex-direction",
      "browsers": [
        "IE10"
      ],
      "values": [
        {
          "name": "column",
          "description": "The flex container\u2019s main axis has the same orientation as the block axis of the current writing mode."
        },
        {
          "name": "column-reverse",
          "description": "Same as 'column', except the main-start and main-end directions are swapped."
        },
        {
          "name": "row",
          "description": "The flex container\u2019s main axis has the same orientation as the inline axis of the current writing mode."
        },
        {
          "name": "row-reverse",
          "description": "Same as 'row', except the main-start and main-end directions are swapped."
        }
      ],
      "relevance": 50,
      "description": "Specifies how flex items are placed in the flex container, by setting the direction of the flex container\u2019s main axis.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-flex-flow",
      "browsers": [
        "IE10"
      ],
      "values": [
        {
          "name": "column",
          "description": "The flex container\u2019s main axis has the same orientation as the block axis of the current writing mode."
        },
        {
          "name": "column-reverse",
          "description": "Same as 'column', except the main-start and main-end directions are swapped."
        },
        {
          "name": "nowrap",
          "description": "The flex container is single-line."
        },
        {
          "name": "row",
          "description": "The flex container\u2019s main axis has the same orientation as the inline axis of the current writing mode."
        },
        {
          "name": "wrap",
          "description": "The flexbox is multi-line."
        },
        {
          "name": "wrap-reverse",
          "description": "Same as 'wrap', except the cross-start and cross-end directions are swapped."
        }
      ],
      "relevance": 50,
      "description": "Specifies how flexbox items are placed in the flexbox.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-flex-item-align",
      "browsers": [
        "IE10"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Computes to the value of 'align-items' on the element\u2019s parent, or 'stretch' if the element has no parent. On absolutely positioned elements, it computes to itself."
        },
        {
          "name": "baseline",
          "description": "If the flex item\u2019s inline axis is the same as the cross axis, this value is identical to 'flex-start'. Otherwise, it participates in baseline alignment."
        },
        {
          "name": "center",
          "description": "The flex item\u2019s margin box is centered in the cross axis within the line."
        },
        {
          "name": "end",
          "description": "The cross-end margin edge of the flex item is placed flush with the cross-end edge of the line."
        },
        {
          "name": "start",
          "description": "The cross-start margin edge of the flex item is placed flush with the cross-start edge of the line."
        },
        {
          "name": "stretch",
          "description": "If the cross size property of the flex item computes to auto, and neither of the cross-axis margins are auto, the flex item is stretched."
        }
      ],
      "relevance": 50,
      "description": "Allows the default alignment along the cross axis to be overridden for individual flex items.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-flex-line-pack",
      "browsers": [
        "IE10"
      ],
      "values": [
        {
          "name": "center",
          "description": "Lines are packed toward the center of the flex container."
        },
        {
          "name": "distribute",
          "description": "Lines are evenly distributed in the flex container, with half-size spaces on either end."
        },
        {
          "name": "end",
          "description": "Lines are packed toward the end of the flex container."
        },
        {
          "name": "justify",
          "description": "Lines are evenly distributed in the flex container."
        },
        {
          "name": "start",
          "description": "Lines are packed toward the start of the flex container."
        },
        {
          "name": "stretch",
          "description": "Lines stretch to take up the remaining space."
        }
      ],
      "relevance": 50,
      "description": "Aligns a flex container\u2019s lines within the flex container when there is extra space in the cross-axis, similar to how 'justify-content' aligns individual items within the main-axis.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-flex-order",
      "browsers": [
        "IE10"
      ],
      "relevance": 50,
      "description": "Controls the order in which children of a flex container appear within the flex container, by assigning them to ordinal groups.",
      "restrictions": [
        "integer"
      ]
    },
    {
      "name": "-ms-flex-pack",
      "browsers": [
        "IE10"
      ],
      "values": [
        {
          "name": "center",
          "description": "Flex items are packed toward the center of the line."
        },
        {
          "name": "distribute",
          "description": "Flex items are evenly distributed in the line, with half-size spaces on either end."
        },
        {
          "name": "end",
          "description": "Flex items are packed toward the end of the line."
        },
        {
          "name": "justify",
          "description": "Flex items are evenly distributed in the line."
        },
        {
          "name": "start",
          "description": "Flex items are packed toward the start of the line."
        }
      ],
      "relevance": 50,
      "description": "Aligns flex items along the main axis of the current line of the flex container.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-flex-wrap",
      "browsers": [
        "IE10"
      ],
      "values": [
        {
          "name": "nowrap",
          "description": "The flex container is single-line."
        },
        {
          "name": "wrap",
          "description": "The flexbox is multi-line."
        },
        {
          "name": "wrap-reverse",
          "description": "Same as 'wrap', except the cross-start and cross-end directions are swapped."
        }
      ],
      "relevance": 50,
      "description": "Controls whether the flex container is single-line or multi-line, and the direction of the cross-axis, which determines the direction new lines are stacked in.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-flow-from",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "none",
          "description": "The block container is not a CSS Region."
        }
      ],
      "status": "nonstandard",
      "syntax": "[ none | <custom-ident> ]#",
      "relevance": 0,
      "description": "Makes a block container a region and associates it with a named flow.",
      "restrictions": [
        "identifier"
      ]
    },
    {
      "name": "-ms-flow-into",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "none",
          "description": "The element is not moved to a named flow and normal CSS processing takes place."
        }
      ],
      "status": "nonstandard",
      "syntax": "[ none | <custom-ident> ]#",
      "relevance": 0,
      "description": "Places an element or its contents into a named flow.",
      "restrictions": [
        "identifier"
      ]
    },
    {
      "name": "-ms-grid-column",
      "browsers": [
        "E12",
        "IE10"
      ],
      "values": [
        {
          "name": "auto"
        },
        {
          "name": "end"
        },
        {
          "name": "start"
        }
      ],
      "relevance": 50,
      "description": "Used to place grid items and explicitly defined grid cells in the Grid.",
      "restrictions": [
        "integer",
        "string",
        "enum"
      ]
    },
    {
      "name": "-ms-grid-column-align",
      "browsers": [
        "E12",
        "IE10"
      ],
      "values": [
        {
          "name": "center",
          "description": "Places the center of the Grid Item's margin box at the center of the Grid Item's column."
        },
        {
          "name": "end",
          "description": "Aligns the end edge of the Grid Item's margin box to the end edge of the Grid Item's column."
        },
        {
          "name": "start",
          "description": "Aligns the starting edge of the Grid Item's margin box to the starting edge of the Grid Item's column."
        },
        {
          "name": "stretch",
          "description": "Ensures that the Grid Item's margin box is equal to the size of the Grid Item's column."
        }
      ],
      "relevance": 50,
      "description": "Aligns the columns in a grid.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-grid-columns",
      "browsers": [
        "E",
        "IE10"
      ],
      "status": "nonstandard",
      "syntax": "none | <track-list> | <auto-track-list>",
      "relevance": 0,
      "description": "Lays out the columns of the grid."
    },
    {
      "name": "-ms-grid-column-span",
      "browsers": [
        "E12",
        "IE10"
      ],
      "relevance": 50,
      "description": "Specifies the number of columns to span.",
      "restrictions": [
        "integer"
      ]
    },
    {
      "name": "-ms-grid-layer",
      "browsers": [
        "E",
        "IE10"
      ],
      "relevance": 50,
      "description": "Grid-layer is similar in concept to z-index, but avoids overloading the meaning of the z-index property, which is applicable only to positioned elements.",
      "restrictions": [
        "integer"
      ]
    },
    {
      "name": "-ms-grid-row",
      "browsers": [
        "E12",
        "IE10"
      ],
      "values": [
        {
          "name": "auto"
        },
        {
          "name": "end"
        },
        {
          "name": "start"
        }
      ],
      "relevance": 50,
      "description": "grid-row is used to place grid items and explicitly defined grid cells in the Grid.",
      "restrictions": [
        "integer",
        "string",
        "enum"
      ]
    },
    {
      "name": "-ms-grid-row-align",
      "browsers": [
        "E12",
        "IE10"
      ],
      "values": [
        {
          "name": "center",
          "description": "Places the center of the Grid Item's margin box at the center of the Grid Item's row."
        },
        {
          "name": "end",
          "description": "Aligns the end edge of the Grid Item's margin box to the end edge of the Grid Item's row."
        },
        {
          "name": "start",
          "description": "Aligns the starting edge of the Grid Item's margin box to the starting edge of the Grid Item's row."
        },
        {
          "name": "stretch",
          "description": "Ensures that the Grid Item's margin box is equal to the size of the Grid Item's row."
        }
      ],
      "relevance": 50,
      "description": "Aligns the rows in a grid.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-grid-rows",
      "browsers": [
        "E",
        "IE10"
      ],
      "status": "nonstandard",
      "syntax": "none | <track-list> | <auto-track-list>",
      "relevance": 0,
      "description": "Lays out the columns of the grid."
    },
    {
      "name": "-ms-grid-row-span",
      "browsers": [
        "E12",
        "IE10"
      ],
      "relevance": 50,
      "description": "Specifies the number of rows to span.",
      "restrictions": [
        "integer"
      ]
    },
    {
      "name": "-ms-high-contrast-adjust",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Properties will be adjusted as applicable."
        },
        {
          "name": "none",
          "description": "No adjustments will be applied."
        }
      ],
      "status": "nonstandard",
      "syntax": "auto | none",
      "relevance": 0,
      "description": "Specifies if properties should be adjusted in high contrast mode.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-hyphenate-limit-chars",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The user agent chooses a value that adapts to the current layout."
        }
      ],
      "status": "nonstandard",
      "syntax": "auto | <integer>{1,3}",
      "relevance": 0,
      "description": "Specifies the minimum number of characters in a hyphenated word.",
      "restrictions": [
        "integer"
      ]
    },
    {
      "name": "-ms-hyphenate-limit-lines",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "no-limit",
          "description": "There is no limit."
        }
      ],
      "status": "nonstandard",
      "syntax": "no-limit | <integer>",
      "relevance": 0,
      "description": "Indicates the maximum number of successive hyphenated lines in an element.",
      "restrictions": [
        "integer"
      ]
    },
    {
      "name": "-ms-hyphenate-limit-zone",
      "browsers": [
        "E",
        "IE10"
      ],
      "status": "nonstandard",
      "syntax": "<percentage> | <length>",
      "relevance": 0,
      "description": "Specifies the maximum amount of unfilled space (before justification) that may be left in the line box before hyphenation is triggered to pull part of a word from the next line back up into the current line.",
      "restrictions": [
        "percentage",
        "length"
      ]
    },
    {
      "name": "-ms-hyphens",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Conditional hyphenation characters inside a word, if present, take priority over automatic resources when determining hyphenation points within the word."
        },
        {
          "name": "manual",
          "description": "Words are only broken at line breaks where there are characters inside the word that suggest line break opportunities"
        },
        {
          "name": "none",
          "description": "Words are not broken at line breaks, even if characters inside the word suggest line break points."
        }
      ],
      "relevance": 50,
      "description": "Controls whether hyphenation is allowed to create more break opportunities within a line of text.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-ime-mode",
      "browsers": [
        "IE10"
      ],
      "values": [
        {
          "name": "active",
          "description": "The input method editor is initially active; text entry is performed using it unless the user specifically dismisses it."
        },
        {
          "name": "auto",
          "description": "No change is made to the current input method editor state. This is the default."
        },
        {
          "name": "disabled",
          "description": "The input method editor is disabled and may not be activated by the user."
        },
        {
          "name": "inactive",
          "description": "The input method editor is initially inactive, but the user may activate it if they wish."
        },
        {
          "name": "normal",
          "description": "The IME state should be normal; this value can be used in a user style sheet to override the page setting."
        }
      ],
      "relevance": 50,
      "description": "Controls the state of the input method editor for text fields.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-interpolation-mode",
      "browsers": [
        "IE7"
      ],
      "values": [
        {
          "name": "bicubic"
        },
        {
          "name": "nearest-neighbor"
        }
      ],
      "relevance": 50,
      "description": "Gets or sets the interpolation (resampling) method used to stretch images.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-layout-grid",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "char",
          "description": "Any of the range of character values available to the -ms-layout-grid-char property."
        },
        {
          "name": "line",
          "description": "Any of the range of line values available to the -ms-layout-grid-line property."
        },
        {
          "name": "mode",
          "description": "Any of the range of mode values available to the -ms-layout-grid-mode property."
        },
        {
          "name": "type",
          "description": "Any of the range of type values available to the -ms-layout-grid-type property."
        }
      ],
      "relevance": 50,
      "description": "Sets or retrieves the composite document grid properties that specify the layout of text characters."
    },
    {
      "name": "-ms-layout-grid-char",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Largest character in the font of the element is used to set the character grid."
        },
        {
          "name": "none",
          "description": "Default. No character grid is set."
        }
      ],
      "relevance": 50,
      "description": "Sets or retrieves the size of the character grid used for rendering the text content of an element.",
      "restrictions": [
        "enum",
        "length",
        "percentage"
      ]
    },
    {
      "name": "-ms-layout-grid-line",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Largest character in the font of the element is used to set the character grid."
        },
        {
          "name": "none",
          "description": "Default. No grid line is set."
        }
      ],
      "relevance": 50,
      "description": "Sets or retrieves the gridline value used for rendering the text content of an element.",
      "restrictions": [
        "length"
      ]
    },
    {
      "name": "-ms-layout-grid-mode",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "both",
          "description": "Default. Both the char and line grid modes are enabled. This setting is necessary to fully enable the layout grid on an element."
        },
        {
          "name": "char",
          "description": "Only a character grid is used. This is recommended for use with block-level elements, such as a blockquote, where the line grid is intended to be disabled."
        },
        {
          "name": "line",
          "description": "Only a line grid is used. This is recommended for use with inline elements, such as a span, to disable the horizontal grid on runs of text that act as a single entity in the grid layout."
        },
        {
          "name": "none",
          "description": "No grid is used."
        }
      ],
      "relevance": 50,
      "description": "Gets or sets whether the text layout grid uses two dimensions.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-layout-grid-type",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "fixed",
          "description": "Grid used for monospaced layout. All noncursive characters are treated as equal; every character is centered within a single grid space by default."
        },
        {
          "name": "loose",
          "description": "Default. Grid used for Japanese and Korean characters."
        },
        {
          "name": "strict",
          "description": "Grid used for Chinese, as well as Japanese (Genko) and Korean characters. Only the ideographs, kanas, and wide characters are snapped to the grid."
        }
      ],
      "relevance": 50,
      "description": "Sets or retrieves the type of grid used for rendering the text content of an element.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-line-break",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The UA determines the set of line-breaking restrictions to use for CJK scripts, and it may vary the restrictions based on the length of the line; e.g., use a less restrictive set of line-break rules for short lines."
        },
        {
          "name": "keep-all",
          "description": "Sequences of CJK characters can no longer break on implied break points. This option should only be used where the presence of word separator characters still creates line-breaking opportunities, as in Korean."
        },
        {
          "name": "newspaper",
          "description": "Breaks CJK scripts using the least restrictive set of line-breaking rules. Typically used for short lines, such as in newspapers."
        },
        {
          "name": "normal",
          "description": "Breaks CJK scripts using a normal set of line-breaking rules."
        },
        {
          "name": "strict",
          "description": "Breaks CJK scripts using a more restrictive set of line-breaking rules than 'normal'."
        }
      ],
      "relevance": 50,
      "description": "Specifies what set of line breaking restrictions are in effect within the element.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-overflow-style",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "auto",
          "description": "No preference, UA should use the first scrolling method in the list that it supports."
        },
        {
          "name": "-ms-autohiding-scrollbar",
          "description": "Indicates the element displays auto-hiding scrollbars during mouse interactions and panning indicators during touch and keyboard interactions."
        },
        {
          "name": "none",
          "description": "Indicates the element does not display scrollbars or panning indicators, even when its content overflows."
        },
        {
          "name": "scrollbar",
          "description": 'Scrollbars are typically narrow strips inserted on one or two edges of an element and which often have arrows to click on and a "thumb" to drag up and down (or left and right) to move the contents of the element.'
        }
      ],
      "status": "nonstandard",
      "syntax": "auto | none | scrollbar | -ms-autohiding-scrollbar",
      "relevance": 0,
      "description": "Specify whether content is clipped when it overflows the element's content area.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-perspective",
      "browsers": [
        "IE10"
      ],
      "values": [
        {
          "name": "none",
          "description": "No perspective transform is applied."
        }
      ],
      "relevance": 50,
      "description": "Applies the same transform as the perspective(<number>) transform function, except that it applies only to the positioned or transformed children of the element, not to the transform on the element itself.",
      "restrictions": [
        "length"
      ]
    },
    {
      "name": "-ms-perspective-origin",
      "browsers": [
        "IE10"
      ],
      "relevance": 50,
      "description": "Establishes the origin for the perspective property. It effectively sets the X and Y position at which the viewer appears to be looking at the children of the element.",
      "restrictions": [
        "position",
        "percentage",
        "length"
      ]
    },
    {
      "name": "-ms-perspective-origin-x",
      "browsers": [
        "IE10"
      ],
      "relevance": 50,
      "description": "Establishes the origin for the perspective property. It effectively sets the X  position at which the viewer appears to be looking at the children of the element.",
      "restrictions": [
        "position",
        "percentage",
        "length"
      ]
    },
    {
      "name": "-ms-perspective-origin-y",
      "browsers": [
        "IE10"
      ],
      "relevance": 50,
      "description": "Establishes the origin for the perspective property. It effectively sets the Y position at which the viewer appears to be looking at the children of the element.",
      "restrictions": [
        "position",
        "percentage",
        "length"
      ]
    },
    {
      "name": "-ms-progress-appearance",
      "browsers": [
        "IE10"
      ],
      "values": [
        {
          "name": "bar"
        },
        {
          "name": "ring"
        }
      ],
      "relevance": 50,
      "description": "Gets or sets a value that specifies whether a progress control displays as a bar or a ring.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-scrollbar-3dlight-color",
      "browsers": [
        "IE8"
      ],
      "status": "nonstandard",
      "syntax": "<color>",
      "relevance": 0,
      "description": "Determines the color of the top and left edges of the scroll box and scroll arrows of a scroll bar.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "-ms-scrollbar-arrow-color",
      "browsers": [
        "IE8"
      ],
      "status": "nonstandard",
      "syntax": "<color>",
      "relevance": 0,
      "description": "Determines the color of the arrow elements of a scroll arrow.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "-ms-scrollbar-base-color",
      "browsers": [
        "IE8"
      ],
      "status": "nonstandard",
      "syntax": "<color>",
      "relevance": 0,
      "description": "Determines the color of the main elements of a scroll bar, which include the scroll box, track, and scroll arrows.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "-ms-scrollbar-darkshadow-color",
      "browsers": [
        "IE8"
      ],
      "status": "nonstandard",
      "syntax": "<color>",
      "relevance": 0,
      "description": "Determines the color of the gutter of a scroll bar.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "-ms-scrollbar-face-color",
      "browsers": [
        "IE8"
      ],
      "status": "nonstandard",
      "syntax": "<color>",
      "relevance": 0,
      "description": "Determines the color of the scroll box and scroll arrows of a scroll bar.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "-ms-scrollbar-highlight-color",
      "browsers": [
        "IE8"
      ],
      "status": "nonstandard",
      "syntax": "<color>",
      "relevance": 0,
      "description": "Determines the color of the top and left edges of the scroll box and scroll arrows of a scroll bar.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "-ms-scrollbar-shadow-color",
      "browsers": [
        "IE8"
      ],
      "status": "nonstandard",
      "syntax": "<color>",
      "relevance": 0,
      "description": "Determines the color of the bottom and right edges of the scroll box and scroll arrows of a scroll bar.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "-ms-scrollbar-track-color",
      "browsers": [
        "IE5"
      ],
      "status": "nonstandard",
      "syntax": "<color>",
      "relevance": 0,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/-ms-scrollbar-track-color"
        }
      ],
      "description": "Determines the color of the track element of a scroll bar.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "-ms-scroll-chaining",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "chained"
        },
        {
          "name": "none"
        }
      ],
      "status": "nonstandard",
      "syntax": "chained | none",
      "relevance": 0,
      "description": "Gets or sets a value that indicates the scrolling behavior that occurs when a user hits the content boundary during a manipulation.",
      "restrictions": [
        "enum",
        "length"
      ]
    },
    {
      "name": "-ms-scroll-limit",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "auto"
        }
      ],
      "status": "nonstandard",
      "syntax": "<'-ms-scroll-limit-x-min'> <'-ms-scroll-limit-y-min'> <'-ms-scroll-limit-x-max'> <'-ms-scroll-limit-y-max'>",
      "relevance": 0,
      "description": "Gets or sets a shorthand value that sets values for the -ms-scroll-limit-x-min, -ms-scroll-limit-y-min, -ms-scroll-limit-x-max, and -ms-scroll-limit-y-max properties.",
      "restrictions": [
        "length"
      ]
    },
    {
      "name": "-ms-scroll-limit-x-max",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "auto"
        }
      ],
      "status": "nonstandard",
      "syntax": "auto | <length>",
      "relevance": 0,
      "description": "Gets or sets a value that specifies the maximum value for the scrollLeft property.",
      "restrictions": [
        "length"
      ]
    },
    {
      "name": "-ms-scroll-limit-x-min",
      "browsers": [
        "E",
        "IE10"
      ],
      "status": "nonstandard",
      "syntax": "<length>",
      "relevance": 0,
      "description": "Gets or sets a value that specifies the minimum value for the scrollLeft property.",
      "restrictions": [
        "length"
      ]
    },
    {
      "name": "-ms-scroll-limit-y-max",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "auto"
        }
      ],
      "status": "nonstandard",
      "syntax": "auto | <length>",
      "relevance": 0,
      "description": "Gets or sets a value that specifies the maximum value for the scrollTop property.",
      "restrictions": [
        "length"
      ]
    },
    {
      "name": "-ms-scroll-limit-y-min",
      "browsers": [
        "E",
        "IE10"
      ],
      "status": "nonstandard",
      "syntax": "<length>",
      "relevance": 0,
      "description": "Gets or sets a value that specifies the minimum value for the scrollTop property.",
      "restrictions": [
        "length"
      ]
    },
    {
      "name": "-ms-scroll-rails",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "none"
        },
        {
          "name": "railed"
        }
      ],
      "status": "nonstandard",
      "syntax": "none | railed",
      "relevance": 0,
      "description": "Gets or sets a value that indicates whether or not small motions perpendicular to the primary axis of motion will result in either changes to both the scrollTop and scrollLeft properties or a change to the primary axis (for instance, either the scrollTop or scrollLeft properties will change, but not both).",
      "restrictions": [
        "enum",
        "length"
      ]
    },
    {
      "name": "-ms-scroll-snap-points-x",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "snapInterval(100%, 100%)"
        },
        {
          "name": "snapList()"
        }
      ],
      "status": "nonstandard",
      "syntax": "snapInterval( <length-percentage>, <length-percentage> ) | snapList( <length-percentage># )",
      "relevance": 0,
      "description": "Gets or sets a value that defines where snap-points will be located along the x-axis.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-scroll-snap-points-y",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "snapInterval(100%, 100%)"
        },
        {
          "name": "snapList()"
        }
      ],
      "status": "nonstandard",
      "syntax": "snapInterval( <length-percentage>, <length-percentage> ) | snapList( <length-percentage># )",
      "relevance": 0,
      "description": "Gets or sets a value that defines where snap-points will be located along the y-axis.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-scroll-snap-type",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "none",
          "description": "The visual viewport of this scroll container must ignore snap points, if any, when scrolled."
        },
        {
          "name": "mandatory",
          "description": "The visual viewport of this scroll container is guaranteed to rest on a snap point when there are no active scrolling operations."
        },
        {
          "name": "proximity",
          "description": "The visual viewport of this scroll container may come to rest on a snap point at the termination of a scroll at the discretion of the UA given the parameters of the scroll."
        }
      ],
      "status": "nonstandard",
      "syntax": "none | proximity | mandatory",
      "relevance": 0,
      "description": "Gets or sets a value that defines what type of snap-point should be used for the current element. There are two type of snap-points, with the primary difference being whether or not the user is guaranteed to always stop on a snap-point.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-scroll-snap-x",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "mandatory"
        },
        {
          "name": "none"
        },
        {
          "name": "proximity"
        },
        {
          "name": "snapInterval(100%, 100%)"
        },
        {
          "name": "snapList()"
        }
      ],
      "status": "nonstandard",
      "syntax": "<'-ms-scroll-snap-type'> <'-ms-scroll-snap-points-x'>",
      "relevance": 0,
      "description": "Gets or sets a shorthand value that sets values for the -ms-scroll-snap-type and -ms-scroll-snap-points-x properties.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-scroll-snap-y",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "mandatory"
        },
        {
          "name": "none"
        },
        {
          "name": "proximity"
        },
        {
          "name": "snapInterval(100%, 100%)"
        },
        {
          "name": "snapList()"
        }
      ],
      "status": "nonstandard",
      "syntax": "<'-ms-scroll-snap-type'> <'-ms-scroll-snap-points-y'>",
      "relevance": 0,
      "description": "Gets or sets a shorthand value that sets values for the -ms-scroll-snap-type and -ms-scroll-snap-points-y properties.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-scroll-translation",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "none"
        },
        {
          "name": "vertical-to-horizontal"
        }
      ],
      "status": "nonstandard",
      "syntax": "none | vertical-to-horizontal",
      "relevance": 0,
      "description": "Gets or sets a value that specifies whether vertical-to-horizontal scroll wheel translation occurs on the specified element.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-text-align-last",
      "browsers": [
        "E",
        "IE8"
      ],
      "values": [
        {
          "name": "auto"
        },
        {
          "name": "center",
          "description": "The inline contents are centered within the line box."
        },
        {
          "name": "justify",
          "description": "The text is justified according to the method specified by the 'text-justify' property."
        },
        {
          "name": "left",
          "description": "The inline contents are aligned to the left edge of the line box. In vertical text, 'left' aligns to the edge of the line box that would be the start edge for left-to-right text."
        },
        {
          "name": "right",
          "description": "The inline contents are aligned to the right edge of the line box. In vertical text, 'right' aligns to the edge of the line box that would be the end edge for left-to-right text."
        }
      ],
      "relevance": 50,
      "description": "Describes how the last line of a block or a line right before a forced line break is aligned when 'text-align' is set to 'justify'.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-text-autospace",
      "browsers": [
        "E",
        "IE8"
      ],
      "values": [
        {
          "name": "ideograph-alpha",
          "description": "Creates 1/4em extra spacing between runs of ideographic letters and non-ideographic letters, such as Latin-based, Cyrillic, Greek, Arabic or Hebrew."
        },
        {
          "name": "ideograph-numeric",
          "description": "Creates 1/4em extra spacing between runs of ideographic letters and numeric glyphs."
        },
        {
          "name": "ideograph-parenthesis",
          "description": "Creates extra spacing between normal (non wide) parenthesis and ideographs."
        },
        {
          "name": "ideograph-space",
          "description": "Extends the width of the space character while surrounded by ideographs."
        },
        {
          "name": "none",
          "description": "No extra space is created."
        },
        {
          "name": "punctuation",
          "description": "Creates extra non-breaking spacing around punctuation as required by language-specific typographic conventions."
        }
      ],
      "status": "nonstandard",
      "syntax": "none | ideograph-alpha | ideograph-numeric | ideograph-parenthesis | ideograph-space",
      "relevance": 0,
      "description": "Determines whether or not a full-width punctuation mark character should be trimmed if it appears at the beginning of a line, so that its 'ink' lines up with the first glyph in the line above and below.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-text-combine-horizontal",
      "browsers": [
        "E",
        "IE11"
      ],
      "values": [
        {
          "name": "all",
          "description": "Attempt to typeset horizontally all consecutive characters within the box such that they take up the space of a single character within the vertical line box."
        },
        {
          "name": "digits",
          "description": "Attempt to typeset horizontally each maximal sequence of consecutive ASCII digits (U+0030\u2013U+0039) that has as many or fewer characters than the specified integer such that it takes up the space of a single character within the vertical line box."
        },
        {
          "name": "none",
          "description": "No special processing."
        }
      ],
      "relevance": 50,
      "description": "This property specifies the combination of multiple characters into the space of a single character.",
      "restrictions": [
        "enum",
        "integer"
      ]
    },
    {
      "name": "-ms-text-justify",
      "browsers": [
        "E",
        "IE8"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The UA determines the justification algorithm to follow, based on a balance between performance and adequate presentation quality."
        },
        {
          "name": "distribute",
          "description": "Justification primarily changes spacing both at word separators and at grapheme cluster boundaries in all scripts except those in the connected and cursive groups. This value is sometimes used in e.g. Japanese, often with the 'text-align-last' property."
        },
        {
          "name": "inter-cluster",
          "description": "Justification primarily changes spacing at word separators and at grapheme cluster boundaries in clustered scripts. This value is typically used for Southeast Asian scripts such as Thai."
        },
        {
          "name": "inter-ideograph",
          "description": "Justification primarily changes spacing at word separators and at inter-graphemic boundaries in scripts that use no word spaces. This value is typically used for CJK languages."
        },
        {
          "name": "inter-word",
          "description": "Justification primarily changes spacing at word separators. This value is typically used for languages that separate words using spaces, like English or (sometimes) Korean."
        },
        {
          "name": "kashida",
          "description": "Justification primarily stretches Arabic and related scripts through the use of kashida or other calligraphic elongation."
        }
      ],
      "relevance": 50,
      "description": "Selects the justification algorithm used when 'text-align' is set to 'justify'. The property applies to block containers, but the UA may (but is not required to) also support it on inline elements.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-text-kashida-space",
      "browsers": [
        "E",
        "IE10"
      ],
      "relevance": 50,
      "description": "Sets or retrieves the ratio of kashida expansion to white space expansion when justifying lines of text in the object.",
      "restrictions": [
        "percentage"
      ]
    },
    {
      "name": "-ms-text-overflow",
      "browsers": [
        "IE10"
      ],
      "values": [
        {
          "name": "clip",
          "description": "Clip inline content that overflows. Characters may be only partially rendered."
        },
        {
          "name": "ellipsis",
          "description": "Render an ellipsis character (U+2026) to represent clipped inline content."
        }
      ],
      "relevance": 50,
      "description": "Text can overflow for example when it is prevented from wrapping",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-text-size-adjust",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Renderers must use the default size adjustment when displaying on a small device."
        },
        {
          "name": "none",
          "description": "Renderers must not do size adjustment when displaying on a small device."
        }
      ],
      "relevance": 50,
      "description": "Specifies a size adjustment for displaying text content in mobile browsers.",
      "restrictions": [
        "enum",
        "percentage"
      ]
    },
    {
      "name": "-ms-text-underline-position",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "alphabetic",
          "description": "The underline is aligned with the alphabetic baseline. In this case the underline is likely to cross some descenders."
        },
        {
          "name": "auto",
          "description": "The user agent may use any algorithm to determine the underline's position. In horizontal line layout, the underline should be aligned as for alphabetic. In vertical line layout, if the language is set to Japanese or Korean, the underline should be aligned as for over."
        },
        {
          "name": "over",
          "description": "The underline is aligned with the 'top' (right in vertical writing) edge of the element's em-box. In this mode, an overline also switches sides."
        },
        {
          "name": "under",
          "description": "The underline is aligned with the 'bottom' (left in vertical writing) edge of the element's em-box. In this case the underline usually does not cross the descenders. This is sometimes called 'accounting' underline."
        }
      ],
      "relevance": 50,
      "description": "Sets the position of an underline specified on the same element: it does not affect underlines specified by ancestor elements.This property is typically used in vertical writing contexts such as in Japanese documents where it often desired to have the underline appear 'over' (to the right of) the affected run of text",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-touch-action",
      "browsers": [
        "IE10"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The element is a passive element, with several exceptions."
        },
        {
          "name": "double-tap-zoom",
          "description": "The element will zoom on double-tap."
        },
        {
          "name": "manipulation",
          "description": "The element is a manipulation-causing element."
        },
        {
          "name": "none",
          "description": "The element is a manipulation-blocking element."
        },
        {
          "name": "pan-x",
          "description": "The element permits touch-driven panning on the horizontal axis. The touch pan is performed on the nearest ancestor with horizontally scrollable content."
        },
        {
          "name": "pan-y",
          "description": "The element permits touch-driven panning on the vertical axis. The touch pan is performed on the nearest ancestor with vertically scrollable content."
        },
        {
          "name": "pinch-zoom",
          "description": "The element permits pinch-zooming. The pinch-zoom is performed on the nearest ancestor with zoomable content."
        }
      ],
      "relevance": 50,
      "description": "Gets or sets a value that indicates whether and how a given region can be manipulated by the user.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-touch-select",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "grippers",
          "description": "Grippers are always on."
        },
        {
          "name": "none",
          "description": "Grippers are always off."
        }
      ],
      "status": "nonstandard",
      "syntax": "grippers | none",
      "relevance": 0,
      "description": "Gets or sets a value that toggles the 'gripper' visual elements that enable touch text selection.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-transform",
      "browsers": [
        "IE9-9"
      ],
      "values": [
        {
          "name": "matrix()",
          "description": "Specifies a 2D transformation in the form of a transformation matrix of six values. matrix(a,b,c,d,e,f) is equivalent to applying the transformation matrix [a b c d e f]"
        },
        {
          "name": "matrix3d()",
          "description": "Specifies a 3D transformation as a 4x4 homogeneous matrix of 16 values in column-major order."
        },
        {
          "name": "none"
        },
        {
          "name": "rotate()",
          "description": "Specifies a 2D rotation by the angle specified in the parameter about the origin of the element, as defined by the transform-origin property."
        },
        {
          "name": "rotate3d()",
          "description": "Specifies a clockwise 3D rotation by the angle specified in last parameter about the [x,y,z] direction vector described by the first 3 parameters."
        },
        {
          "name": "rotateX('angle')",
          "description": "Specifies a clockwise rotation by the given angle about the X axis."
        },
        {
          "name": "rotateY('angle')",
          "description": "Specifies a clockwise rotation by the given angle about the Y axis."
        },
        {
          "name": "rotateZ('angle')",
          "description": "Specifies a clockwise rotation by the given angle about the Z axis."
        },
        {
          "name": "scale()",
          "description": "Specifies a 2D scale operation by the [sx,sy] scaling vector described by the 2 parameters. If the second parameter is not provided, it is takes a value equal to the first."
        },
        {
          "name": "scale3d()",
          "description": "Specifies a 3D scale operation by the [sx,sy,sz] scaling vector described by the 3 parameters."
        },
        {
          "name": "scaleX()",
          "description": "Specifies a scale operation using the [sx,1] scaling vector, where sx is given as the parameter."
        },
        {
          "name": "scaleY()",
          "description": "Specifies a scale operation using the [sy,1] scaling vector, where sy is given as the parameter."
        },
        {
          "name": "scaleZ()",
          "description": "Specifies a scale operation using the [1,1,sz] scaling vector, where sz is given as the parameter."
        },
        {
          "name": "skew()",
          "description": "Specifies a skew transformation along the X and Y axes. The first angle parameter specifies the skew on the X axis. The second angle parameter specifies the skew on the Y axis. If the second parameter is not given then a value of 0 is used for the Y angle (ie: no skew on the Y axis)."
        },
        {
          "name": "skewX()",
          "description": "Specifies a skew transformation along the X axis by the given angle."
        },
        {
          "name": "skewY()",
          "description": "Specifies a skew transformation along the Y axis by the given angle."
        },
        {
          "name": "translate()",
          "description": "Specifies a 2D translation by the vector [tx, ty], where tx is the first translation-value parameter and ty is the optional second translation-value parameter."
        },
        {
          "name": "translate3d()",
          "description": "Specifies a 3D translation by the vector [tx,ty,tz], with tx, ty and tz being the first, second and third translation-value parameters respectively."
        },
        {
          "name": "translateX()",
          "description": "Specifies a translation by the given amount in the X direction."
        },
        {
          "name": "translateY()",
          "description": "Specifies a translation by the given amount in the Y direction."
        },
        {
          "name": "translateZ()",
          "description": "Specifies a translation by the given amount in the Z direction. Note that percentage values are not allowed in the translateZ translation-value, and if present are evaluated as 0."
        }
      ],
      "relevance": 50,
      "description": "A two-dimensional transformation is applied to an element through the 'transform' property. This property contains a list of transform functions similar to those allowed by SVG.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-transform-origin",
      "browsers": [
        "IE9-9"
      ],
      "relevance": 50,
      "description": "Establishes the origin of transformation for an element.",
      "restrictions": [
        "position",
        "length",
        "percentage"
      ]
    },
    {
      "name": "-ms-transform-origin-x",
      "browsers": [
        "IE10"
      ],
      "relevance": 50,
      "description": "The x coordinate of the origin for transforms applied to an element with respect to its border box.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "-ms-transform-origin-y",
      "browsers": [
        "IE10"
      ],
      "relevance": 50,
      "description": "The y coordinate of the origin for transforms applied to an element with respect to its border box.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "-ms-transform-origin-z",
      "browsers": [
        "IE10"
      ],
      "relevance": 50,
      "description": "The z coordinate of the origin for transforms applied to an element with respect to its border box.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "-ms-user-select",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "element"
        },
        {
          "name": "none"
        },
        {
          "name": "text"
        }
      ],
      "status": "nonstandard",
      "syntax": "none | element | text",
      "relevance": 0,
      "description": "Controls the appearance of selection.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-word-break",
      "browsers": [
        "IE8"
      ],
      "values": [
        {
          "name": "break-all",
          "description": "Lines may break between any two grapheme clusters for non-CJK scripts."
        },
        {
          "name": "keep-all",
          "description": "Block characters can no longer create implied break points."
        },
        {
          "name": "normal",
          "description": "Breaks non-CJK scripts according to their own rules."
        }
      ],
      "relevance": 50,
      "description": "Specifies line break opportunities for non-CJK scripts.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-word-wrap",
      "browsers": [
        "IE8"
      ],
      "values": [
        {
          "name": "break-word",
          "description": "An unbreakable 'word' may be broken at an arbitrary point if there are no otherwise-acceptable break points in the line."
        },
        {
          "name": "normal",
          "description": "Lines may break only at allowed break points."
        }
      ],
      "relevance": 50,
      "description": "Specifies whether the UA may break within a word to prevent overflow when an otherwise-unbreakable string is too long to fit.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-wrap-flow",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "auto",
          "description": "For floats an exclusion is created, for all other elements an exclusion is not created."
        },
        {
          "name": "both",
          "description": "Inline flow content can flow on all sides of the exclusion."
        },
        {
          "name": "clear",
          "description": "Inline flow content can only wrap on top and bottom of the exclusion and must leave the areas to the start and end edges of the exclusion box empty."
        },
        {
          "name": "end",
          "description": "Inline flow content can wrap on the end side of the exclusion area but must leave the area to the start edge of the exclusion area empty."
        },
        {
          "name": "maximum",
          "description": "Inline flow content can wrap on the side of the exclusion with the largest available space for the given line, and must leave the other side of the exclusion empty."
        },
        {
          "name": "minimum",
          "description": "Inline flow content can flow around the edge of the exclusion with the smallest available space within the flow content\u2019s containing block, and must leave the other edge of the exclusion empty."
        },
        {
          "name": "start",
          "description": "Inline flow content can wrap on the start edge of the exclusion area but must leave the area to end edge of the exclusion area empty."
        }
      ],
      "status": "nonstandard",
      "syntax": "auto | both | start | end | maximum | clear",
      "relevance": 0,
      "description": "An element becomes an exclusion when its 'wrap-flow' property has a computed value other than 'auto'.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-wrap-margin",
      "browsers": [
        "E",
        "IE10"
      ],
      "status": "nonstandard",
      "syntax": "<length>",
      "relevance": 0,
      "description": "Gets or sets a value that is used to offset the inner wrap shape from other shapes.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "-ms-wrap-through",
      "browsers": [
        "E",
        "IE10"
      ],
      "values": [
        {
          "name": "none",
          "description": "The exclusion element does not inherit its parent node's wrapping context. Its descendants are only subject to exclusion shapes defined inside the element."
        },
        {
          "name": "wrap",
          "description": "The exclusion element inherits its parent node's wrapping context. Its descendant inline content wraps around exclusions defined outside the element."
        }
      ],
      "status": "nonstandard",
      "syntax": "wrap | none",
      "relevance": 0,
      "description": "Specifies if an element inherits its parent wrapping context. In other words if it is subject to the exclusions defined outside the element.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-writing-mode",
      "browsers": [
        "IE8"
      ],
      "values": [
        {
          "name": "bt-lr"
        },
        {
          "name": "bt-rl"
        },
        {
          "name": "lr-bt"
        },
        {
          "name": "lr-tb"
        },
        {
          "name": "rl-bt"
        },
        {
          "name": "rl-tb"
        },
        {
          "name": "tb-lr"
        },
        {
          "name": "tb-rl"
        }
      ],
      "relevance": 50,
      "description": "Shorthand property for both 'direction' and 'block-progression'.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-ms-zoom",
      "browsers": [
        "IE8"
      ],
      "values": [
        {
          "name": "normal"
        }
      ],
      "relevance": 50,
      "description": "Sets or retrieves the magnification scale of the object.",
      "restrictions": [
        "enum",
        "integer",
        "number",
        "percentage"
      ]
    },
    {
      "name": "-ms-zoom-animation",
      "browsers": [
        "IE10"
      ],
      "values": [
        {
          "name": "default"
        },
        {
          "name": "none"
        }
      ],
      "relevance": 50,
      "description": "Gets or sets a value that indicates whether an animation is used when zooming.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "nav-down",
      "browsers": [
        "O9.5"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The user agent automatically determines which element to navigate the focus to in response to directional navigational input."
        },
        {
          "name": "current",
          "description": "Indicates that the user agent should target the frame that the element is in."
        },
        {
          "name": "root",
          "description": "Indicates that the user agent should target the full window."
        }
      ],
      "relevance": 50,
      "description": "Provides an way to control directional focus navigation.",
      "restrictions": [
        "enum",
        "identifier",
        "string"
      ]
    },
    {
      "name": "nav-index",
      "browsers": [
        "O9.5"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The element's sequential navigation order is assigned automatically by the user agent."
        }
      ],
      "relevance": 50,
      "description": "Provides an input-method-neutral way of specifying the sequential navigation order (also known as 'tabbing order').",
      "restrictions": [
        "number"
      ]
    },
    {
      "name": "nav-left",
      "browsers": [
        "O9.5"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The user agent automatically determines which element to navigate the focus to in response to directional navigational input."
        },
        {
          "name": "current",
          "description": "Indicates that the user agent should target the frame that the element is in."
        },
        {
          "name": "root",
          "description": "Indicates that the user agent should target the full window."
        }
      ],
      "relevance": 50,
      "description": "Provides an way to control directional focus navigation.",
      "restrictions": [
        "enum",
        "identifier",
        "string"
      ]
    },
    {
      "name": "nav-right",
      "browsers": [
        "O9.5"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The user agent automatically determines which element to navigate the focus to in response to directional navigational input."
        },
        {
          "name": "current",
          "description": "Indicates that the user agent should target the frame that the element is in."
        },
        {
          "name": "root",
          "description": "Indicates that the user agent should target the full window."
        }
      ],
      "relevance": 50,
      "description": "Provides an way to control directional focus navigation.",
      "restrictions": [
        "enum",
        "identifier",
        "string"
      ]
    },
    {
      "name": "nav-up",
      "browsers": [
        "O9.5"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The user agent automatically determines which element to navigate the focus to in response to directional navigational input."
        },
        {
          "name": "current",
          "description": "Indicates that the user agent should target the frame that the element is in."
        },
        {
          "name": "root",
          "description": "Indicates that the user agent should target the full window."
        }
      ],
      "relevance": 50,
      "description": "Provides an way to control directional focus navigation.",
      "restrictions": [
        "enum",
        "identifier",
        "string"
      ]
    },
    {
      "name": "negative",
      "browsers": [
        "FF33"
      ],
      "syntax": "<symbol> <symbol>?",
      "relevance": 50,
      "description": "@counter-style descriptor. Defines how to alter the representation when the counter value is negative.",
      "restrictions": [
        "image",
        "identifier",
        "string"
      ]
    },
    {
      "name": "-o-animation",
      "browsers": [
        "O12"
      ],
      "values": [
        {
          "name": "alternate",
          "description": "The animation cycle iterations that are odd counts are played in the normal direction, and the animation cycle iterations that are even counts are played in a reverse direction."
        },
        {
          "name": "alternate-reverse",
          "description": "The animation cycle iterations that are odd counts are played in the reverse direction, and the animation cycle iterations that are even counts are played in a normal direction."
        },
        {
          "name": "backwards",
          "description": "The beginning property value (as defined in the first @keyframes at-rule) is applied before the animation is displayed, during the period defined by 'animation-delay'."
        },
        {
          "name": "both",
          "description": "Both forwards and backwards fill modes are applied."
        },
        {
          "name": "forwards",
          "description": "The final property value (as defined in the last @keyframes at-rule) is maintained after the animation completes."
        },
        {
          "name": "infinite",
          "description": "Causes the animation to repeat forever."
        },
        {
          "name": "none",
          "description": "No animation is performed"
        },
        {
          "name": "normal",
          "description": "Normal playback."
        },
        {
          "name": "reverse",
          "description": "All iterations of the animation are played in the reverse direction from the way they were specified."
        }
      ],
      "relevance": 50,
      "description": "Shorthand property combines six of the animation properties into a single property.",
      "restrictions": [
        "time",
        "enum",
        "timing-function",
        "identifier",
        "number"
      ]
    },
    {
      "name": "-o-animation-delay",
      "browsers": [
        "O12"
      ],
      "relevance": 50,
      "description": "Defines when the animation will start.",
      "restrictions": [
        "time"
      ]
    },
    {
      "name": "-o-animation-direction",
      "browsers": [
        "O12"
      ],
      "values": [
        {
          "name": "alternate",
          "description": "The animation cycle iterations that are odd counts are played in the normal direction, and the animation cycle iterations that are even counts are played in a reverse direction."
        },
        {
          "name": "alternate-reverse",
          "description": "The animation cycle iterations that are odd counts are played in the reverse direction, and the animation cycle iterations that are even counts are played in a normal direction."
        },
        {
          "name": "normal",
          "description": "Normal playback."
        },
        {
          "name": "reverse",
          "description": "All iterations of the animation are played in the reverse direction from the way they were specified."
        }
      ],
      "relevance": 50,
      "description": "Defines whether or not the animation should play in reverse on alternate cycles.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-o-animation-duration",
      "browsers": [
        "O12"
      ],
      "relevance": 50,
      "description": "Defines the length of time that an animation takes to complete one cycle.",
      "restrictions": [
        "time"
      ]
    },
    {
      "name": "-o-animation-fill-mode",
      "browsers": [
        "O12"
      ],
      "values": [
        {
          "name": "backwards",
          "description": "The beginning property value (as defined in the first @keyframes at-rule) is applied before the animation is displayed, during the period defined by 'animation-delay'."
        },
        {
          "name": "both",
          "description": "Both forwards and backwards fill modes are applied."
        },
        {
          "name": "forwards",
          "description": "The final property value (as defined in the last @keyframes at-rule) is maintained after the animation completes."
        },
        {
          "name": "none",
          "description": "There is no change to the property value between the time the animation is applied and the time the animation begins playing or after the animation completes."
        }
      ],
      "relevance": 50,
      "description": "Defines what values are applied by the animation outside the time it is executing.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-o-animation-iteration-count",
      "browsers": [
        "O12"
      ],
      "values": [
        {
          "name": "infinite",
          "description": "Causes the animation to repeat forever."
        }
      ],
      "relevance": 50,
      "description": "Defines the number of times an animation cycle is played. The default value is one, meaning the animation will play from beginning to end once.",
      "restrictions": [
        "number",
        "enum"
      ]
    },
    {
      "name": "-o-animation-name",
      "browsers": [
        "O12"
      ],
      "values": [
        {
          "name": "none",
          "description": "No animation is performed"
        }
      ],
      "relevance": 50,
      "description": "Defines a list of animations that apply. Each name is used to select the keyframe at-rule that provides the property values for the animation.",
      "restrictions": [
        "identifier",
        "enum"
      ]
    },
    {
      "name": "-o-animation-play-state",
      "browsers": [
        "O12"
      ],
      "values": [
        {
          "name": "paused",
          "description": "A running animation will be paused."
        },
        {
          "name": "running",
          "description": "Resume playback of a paused animation."
        }
      ],
      "relevance": 50,
      "description": "Defines whether the animation is running or paused.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-o-animation-timing-function",
      "browsers": [
        "O12"
      ],
      "relevance": 50,
      "description": "Describes how the animation will progress over one cycle of its duration. See the 'transition-timing-function'.",
      "restrictions": [
        "timing-function"
      ]
    },
    {
      "name": "object-fit",
      "browsers": [
        "E79",
        "FF36",
        "S10",
        "C32",
        "O19"
      ],
      "values": [
        {
          "name": "contain",
          "description": "The replaced content is sized to maintain its aspect ratio while fitting within the element\u2019s content box: its concrete object size is resolved as a contain constraint against the element's used width and height."
        },
        {
          "name": "cover",
          "description": "The replaced content is sized to maintain its aspect ratio while filling the element's entire content box: its concrete object size is resolved as a cover constraint against the element\u2019s used width and height."
        },
        {
          "name": "fill",
          "description": "The replaced content is sized to fill the element\u2019s content box: the object's concrete object size is the element's used width and height."
        },
        {
          "name": "none",
          "description": "The replaced content is not resized to fit inside the element's content box"
        },
        {
          "name": "scale-down",
          "description": "Size the content as if \u2018none\u2019 or \u2018contain\u2019 were specified, whichever would result in a smaller concrete object size."
        }
      ],
      "syntax": "fill | contain | cover | none | scale-down",
      "relevance": 68,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/object-fit"
        }
      ],
      "description": "Specifies how the contents of a replaced element should be scaled relative to the box established by its used height and width.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "object-position",
      "browsers": [
        "E79",
        "FF36",
        "S10",
        "C32",
        "O19"
      ],
      "syntax": "<position>",
      "relevance": 53,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/object-position"
        }
      ],
      "description": "Determines the alignment of the replaced element inside its box.",
      "restrictions": [
        "position",
        "length",
        "percentage"
      ]
    },
    {
      "name": "-o-border-image",
      "browsers": [
        "O11.6"
      ],
      "values": [
        {
          "name": "auto",
          "description": "If 'auto' is specified then the border image width is the intrinsic width or height (whichever is applicable) of the corresponding image slice. If the image does not have the required intrinsic dimension then the corresponding border-width is used instead."
        },
        {
          "name": "fill",
          "description": "Causes the middle part of the border-image to be preserved."
        },
        {
          "name": "none"
        },
        {
          "name": "repeat",
          "description": "The image is tiled (repeated) to fill the area."
        },
        {
          "name": "round",
          "description": "The image is tiled (repeated) to fill the area. If it does not fill the area with a whole number of tiles, the image is rescaled so that it does."
        },
        {
          "name": "space",
          "description": "The image is tiled (repeated) to fill the area. If it does not fill the area with a whole number of tiles, the extra space is distributed around the tiles."
        },
        {
          "name": "stretch",
          "description": "The image is stretched to fill the area."
        }
      ],
      "relevance": 50,
      "description": "Shorthand property for setting 'border-image-source', 'border-image-slice', 'border-image-width', 'border-image-outset' and 'border-image-repeat'. Omitted values are set to their initial values.",
      "restrictions": [
        "length",
        "percentage",
        "number",
        "image",
        "enum"
      ]
    },
    {
      "name": "-o-object-fit",
      "browsers": [
        "O10.6"
      ],
      "values": [
        {
          "name": "contain",
          "description": "The replaced content is sized to maintain its aspect ratio while fitting within the element\u2019s content box: its concrete object size is resolved as a contain constraint against the element's used width and height."
        },
        {
          "name": "cover",
          "description": "The replaced content is sized to maintain its aspect ratio while filling the element's entire content box: its concrete object size is resolved as a cover constraint against the element\u2019s used width and height."
        },
        {
          "name": "fill",
          "description": "The replaced content is sized to fill the element\u2019s content box: the object's concrete object size is the element's used width and height."
        },
        {
          "name": "none",
          "description": "The replaced content is not resized to fit inside the element's content box"
        },
        {
          "name": "scale-down",
          "description": "Size the content as if \u2018none\u2019 or \u2018contain\u2019 were specified, whichever would result in a smaller concrete object size."
        }
      ],
      "relevance": 50,
      "description": "Specifies how the contents of a replaced element should be scaled relative to the box established by its used height and width.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-o-object-position",
      "browsers": [
        "O10.6"
      ],
      "relevance": 50,
      "description": "Determines the alignment of the replaced element inside its box.",
      "restrictions": [
        "position",
        "length",
        "percentage"
      ]
    },
    {
      "name": "opacity",
      "syntax": "<alpha-value>",
      "relevance": 93,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/opacity"
        }
      ],
      "description": "Opacity of an element's text, where 1 is opaque and 0 is entirely transparent.",
      "restrictions": [
        "number(0-1)"
      ]
    },
    {
      "name": "order",
      "syntax": "<integer>",
      "relevance": 63,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/order"
        }
      ],
      "description": "Controls the order in which children of a flex container appear within the flex container, by assigning them to ordinal groups.",
      "restrictions": [
        "integer"
      ]
    },
    {
      "name": "orphans",
      "browsers": [
        "E12",
        "S1.3",
        "C25",
        "IE8",
        "O9.2"
      ],
      "syntax": "<integer>",
      "relevance": 51,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/orphans"
        }
      ],
      "description": "Specifies the minimum number of line boxes in a block container that must be left in a fragment before a fragmentation break.",
      "restrictions": [
        "integer"
      ]
    },
    {
      "name": "-o-table-baseline",
      "browsers": [
        "O9.6"
      ],
      "relevance": 50,
      "description": "Determines which row of a inline-table should be used as baseline of inline-table.",
      "restrictions": [
        "integer"
      ]
    },
    {
      "name": "-o-tab-size",
      "browsers": [
        "O10.6"
      ],
      "relevance": 50,
      "description": "This property determines the width of the tab character (U+0009), in space characters (U+0020), when rendered.",
      "restrictions": [
        "integer",
        "length"
      ]
    },
    {
      "name": "-o-text-overflow",
      "browsers": [
        "O10"
      ],
      "values": [
        {
          "name": "clip",
          "description": "Clip inline content that overflows. Characters may be only partially rendered."
        },
        {
          "name": "ellipsis",
          "description": "Render an ellipsis character (U+2026) to represent clipped inline content."
        }
      ],
      "relevance": 50,
      "description": "Text can overflow for example when it is prevented from wrapping",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-o-transform",
      "browsers": [
        "O10.5"
      ],
      "values": [
        {
          "name": "matrix()",
          "description": "Specifies a 2D transformation in the form of a transformation matrix of six values. matrix(a,b,c,d,e,f) is equivalent to applying the transformation matrix [a b c d e f]"
        },
        {
          "name": "matrix3d()",
          "description": "Specifies a 3D transformation as a 4x4 homogeneous matrix of 16 values in column-major order."
        },
        {
          "name": "none"
        },
        {
          "name": "rotate()",
          "description": "Specifies a 2D rotation by the angle specified in the parameter about the origin of the element, as defined by the transform-origin property."
        },
        {
          "name": "rotate3d()",
          "description": "Specifies a clockwise 3D rotation by the angle specified in last parameter about the [x,y,z] direction vector described by the first 3 parameters."
        },
        {
          "name": "rotateX('angle')",
          "description": "Specifies a clockwise rotation by the given angle about the X axis."
        },
        {
          "name": "rotateY('angle')",
          "description": "Specifies a clockwise rotation by the given angle about the Y axis."
        },
        {
          "name": "rotateZ('angle')",
          "description": "Specifies a clockwise rotation by the given angle about the Z axis."
        },
        {
          "name": "scale()",
          "description": "Specifies a 2D scale operation by the [sx,sy] scaling vector described by the 2 parameters. If the second parameter is not provided, it is takes a value equal to the first."
        },
        {
          "name": "scale3d()",
          "description": "Specifies a 3D scale operation by the [sx,sy,sz] scaling vector described by the 3 parameters."
        },
        {
          "name": "scaleX()",
          "description": "Specifies a scale operation using the [sx,1] scaling vector, where sx is given as the parameter."
        },
        {
          "name": "scaleY()",
          "description": "Specifies a scale operation using the [sy,1] scaling vector, where sy is given as the parameter."
        },
        {
          "name": "scaleZ()",
          "description": "Specifies a scale operation using the [1,1,sz] scaling vector, where sz is given as the parameter."
        },
        {
          "name": "skew()",
          "description": "Specifies a skew transformation along the X and Y axes. The first angle parameter specifies the skew on the X axis. The second angle parameter specifies the skew on the Y axis. If the second parameter is not given then a value of 0 is used for the Y angle (ie: no skew on the Y axis)."
        },
        {
          "name": "skewX()",
          "description": "Specifies a skew transformation along the X axis by the given angle."
        },
        {
          "name": "skewY()",
          "description": "Specifies a skew transformation along the Y axis by the given angle."
        },
        {
          "name": "translate()",
          "description": "Specifies a 2D translation by the vector [tx, ty], where tx is the first translation-value parameter and ty is the optional second translation-value parameter."
        },
        {
          "name": "translate3d()",
          "description": "Specifies a 3D translation by the vector [tx,ty,tz], with tx, ty and tz being the first, second and third translation-value parameters respectively."
        },
        {
          "name": "translateX()",
          "description": "Specifies a translation by the given amount in the X direction."
        },
        {
          "name": "translateY()",
          "description": "Specifies a translation by the given amount in the Y direction."
        },
        {
          "name": "translateZ()",
          "description": "Specifies a translation by the given amount in the Z direction. Note that percentage values are not allowed in the translateZ translation-value, and if present are evaluated as 0."
        }
      ],
      "relevance": 50,
      "description": "A two-dimensional transformation is applied to an element through the 'transform' property. This property contains a list of transform functions similar to those allowed by SVG.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-o-transform-origin",
      "browsers": [
        "O10.5"
      ],
      "relevance": 50,
      "description": "Establishes the origin of transformation for an element.",
      "restrictions": [
        "positon",
        "length",
        "percentage"
      ]
    },
    {
      "name": "-o-transition",
      "browsers": [
        "O11.5"
      ],
      "values": [
        {
          "name": "all",
          "description": "Every property that is able to undergo a transition will do so."
        },
        {
          "name": "none",
          "description": "No property will transition."
        }
      ],
      "relevance": 50,
      "description": "Shorthand property combines four of the transition properties into a single property.",
      "restrictions": [
        "time",
        "property",
        "timing-function",
        "enum"
      ]
    },
    {
      "name": "-o-transition-delay",
      "browsers": [
        "O11.5"
      ],
      "relevance": 50,
      "description": "Defines when the transition will start. It allows a transition to begin execution some period of time from when it is applied.",
      "restrictions": [
        "time"
      ]
    },
    {
      "name": "-o-transition-duration",
      "browsers": [
        "O11.5"
      ],
      "relevance": 50,
      "description": "Specifies how long the transition from the old value to the new value should take.",
      "restrictions": [
        "time"
      ]
    },
    {
      "name": "-o-transition-property",
      "browsers": [
        "O11.5"
      ],
      "values": [
        {
          "name": "all",
          "description": "Every property that is able to undergo a transition will do so."
        },
        {
          "name": "none",
          "description": "No property will transition."
        }
      ],
      "relevance": 50,
      "description": "Specifies the name of the CSS property to which the transition is applied.",
      "restrictions": [
        "property"
      ]
    },
    {
      "name": "-o-transition-timing-function",
      "browsers": [
        "O11.5"
      ],
      "relevance": 50,
      "description": "Describes how the intermediate values used during a transition will be calculated.",
      "restrictions": [
        "timing-function"
      ]
    },
    {
      "name": "offset-block-end",
      "browsers": [
        "FF41"
      ],
      "values": [
        {
          "name": "auto",
          "description": "For non-replaced elements, the effect of this value depends on which of related properties have the value 'auto' as well."
        }
      ],
      "relevance": 50,
      "description": "Logical 'bottom'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "offset-block-start",
      "browsers": [
        "FF41"
      ],
      "values": [
        {
          "name": "auto",
          "description": "For non-replaced elements, the effect of this value depends on which of related properties have the value 'auto' as well."
        }
      ],
      "relevance": 50,
      "description": "Logical 'top'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "offset-inline-end",
      "browsers": [
        "FF41"
      ],
      "values": [
        {
          "name": "auto",
          "description": "For non-replaced elements, the effect of this value depends on which of related properties have the value 'auto' as well."
        }
      ],
      "relevance": 50,
      "description": "Logical 'right'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "offset-inline-start",
      "browsers": [
        "FF41"
      ],
      "values": [
        {
          "name": "auto",
          "description": "For non-replaced elements, the effect of this value depends on which of related properties have the value 'auto' as well."
        }
      ],
      "relevance": 50,
      "description": "Logical 'left'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "outline",
      "values": [
        {
          "name": "auto",
          "description": "Permits the user agent to render a custom outline style, typically the default platform style."
        },
        {
          "name": "invert",
          "description": "Performs a color inversion on the pixels on the screen."
        }
      ],
      "syntax": "[ <'outline-color'> || <'outline-style'> || <'outline-width'> ]",
      "relevance": 88,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/outline"
        }
      ],
      "description": "Shorthand property for 'outline-style', 'outline-width', and 'outline-color'.",
      "restrictions": [
        "length",
        "line-width",
        "line-style",
        "color",
        "enum"
      ]
    },
    {
      "name": "outline-color",
      "values": [
        {
          "name": "invert",
          "description": "Performs a color inversion on the pixels on the screen."
        }
      ],
      "syntax": "<color> | invert",
      "relevance": 55,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/outline-color"
        }
      ],
      "description": "The color of the outline.",
      "restrictions": [
        "enum",
        "color"
      ]
    },
    {
      "name": "outline-offset",
      "browsers": [
        "E15",
        "FF1.5",
        "S1.2",
        "C1",
        "O9.5"
      ],
      "syntax": "<length>",
      "relevance": 69,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/outline-offset"
        }
      ],
      "description": "Offset the outline and draw it beyond the border edge.",
      "restrictions": [
        "length"
      ]
    },
    {
      "name": "outline-style",
      "values": [
        {
          "name": "auto",
          "description": "Permits the user agent to render a custom outline style, typically the default platform style."
        }
      ],
      "syntax": "auto | <'border-style'>",
      "relevance": 61,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/outline-style"
        }
      ],
      "description": "Style of the outline.",
      "restrictions": [
        "line-style",
        "enum"
      ]
    },
    {
      "name": "outline-width",
      "syntax": "<line-width>",
      "relevance": 61,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/outline-width"
        }
      ],
      "description": "Width of the outline.",
      "restrictions": [
        "length",
        "line-width"
      ]
    },
    {
      "name": "overflow",
      "values": [
        {
          "name": "auto",
          "description": "The behavior of the 'auto' value is UA-dependent, but should cause a scrolling mechanism to be provided for overflowing boxes."
        },
        {
          "name": "hidden",
          "description": "Content is clipped and no scrolling mechanism should be provided to view the content outside the clipping region."
        },
        {
          "name": "-moz-hidden-unscrollable",
          "description": "Same as the standardized 'clip', except doesn\u2019t establish a block formatting context."
        },
        {
          "name": "scroll",
          "description": "Content is clipped and if the user agent uses a scrolling mechanism that is visible on the screen (such as a scroll bar or a panner), that mechanism should be displayed for a box whether or not any of its content is clipped."
        },
        {
          "name": "visible",
          "description": "Content is not clipped, i.e., it may be rendered outside the content box."
        }
      ],
      "syntax": "[ visible | hidden | clip | scroll | auto ]{1,2}",
      "relevance": 93,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/overflow"
        }
      ],
      "description": "Shorthand for setting 'overflow-x' and 'overflow-y'.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "overflow-wrap",
      "values": [
        {
          "name": "break-word",
          "description": "An otherwise unbreakable sequence of characters may be broken at an arbitrary point if there are no otherwise-acceptable break points in the line."
        },
        {
          "name": "normal",
          "description": "Lines may break only at allowed break points."
        }
      ],
      "syntax": "normal | break-word | anywhere",
      "relevance": 66,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/overflow-wrap"
        }
      ],
      "description": "Specifies whether the UA may break within a word to prevent overflow when an otherwise-unbreakable string is too long to fit within the line box.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "overflow-x",
      "values": [
        {
          "name": "auto",
          "description": "The behavior of the 'auto' value is UA-dependent, but should cause a scrolling mechanism to be provided for overflowing boxes."
        },
        {
          "name": "hidden",
          "description": "Content is clipped and no scrolling mechanism should be provided to view the content outside the clipping region."
        },
        {
          "name": "scroll",
          "description": "Content is clipped and if the user agent uses a scrolling mechanism that is visible on the screen (such as a scroll bar or a panner), that mechanism should be displayed for a box whether or not any of its content is clipped."
        },
        {
          "name": "visible",
          "description": "Content is not clipped, i.e., it may be rendered outside the content box."
        }
      ],
      "syntax": "visible | hidden | clip | scroll | auto",
      "relevance": 81,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/overflow-x"
        }
      ],
      "description": "Specifies the handling of overflow in the horizontal direction.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "overflow-y",
      "values": [
        {
          "name": "auto",
          "description": "The behavior of the 'auto' value is UA-dependent, but should cause a scrolling mechanism to be provided for overflowing boxes."
        },
        {
          "name": "hidden",
          "description": "Content is clipped and no scrolling mechanism should be provided to view the content outside the clipping region."
        },
        {
          "name": "scroll",
          "description": "Content is clipped and if the user agent uses a scrolling mechanism that is visible on the screen (such as a scroll bar or a panner), that mechanism should be displayed for a box whether or not any of its content is clipped."
        },
        {
          "name": "visible",
          "description": "Content is not clipped, i.e., it may be rendered outside the content box."
        }
      ],
      "syntax": "visible | hidden | clip | scroll | auto",
      "relevance": 83,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/overflow-y"
        }
      ],
      "description": "Specifies the handling of overflow in the vertical direction.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "pad",
      "browsers": [
        "FF33"
      ],
      "syntax": "<integer> && <symbol>",
      "relevance": 50,
      "description": "@counter-style descriptor. Specifies a \u201Cfixed-width\u201D counter style, where representations shorter than the pad value are padded with a particular <symbol>",
      "restrictions": [
        "integer",
        "image",
        "string",
        "identifier"
      ]
    },
    {
      "name": "padding",
      "values": [],
      "syntax": "[ <length> | <percentage> ]{1,4}",
      "relevance": 96,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/padding"
        }
      ],
      "description": "Shorthand property to set values for the thickness of the padding area. If left is omitted, it is the same as right. If bottom is omitted it is the same as top, if right is omitted it is the same as top. The value may not be negative.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "padding-bottom",
      "syntax": "<length> | <percentage>",
      "relevance": 89,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/padding-bottom"
        }
      ],
      "description": "Shorthand property to set values for the thickness of the padding area. If left is omitted, it is the same as right. If bottom is omitted it is the same as top, if right is omitted it is the same as top. The value may not be negative.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "padding-block-end",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'padding-left'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/padding-block-end"
        }
      ],
      "description": "Logical 'padding-bottom'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "padding-block-start",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'padding-left'>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/padding-block-start"
        }
      ],
      "description": "Logical 'padding-top'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "padding-inline-end",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'padding-left'>",
      "relevance": 51,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/padding-inline-end"
        }
      ],
      "description": "Logical 'padding-right'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "padding-inline-start",
      "browsers": [
        "E79",
        "FF41",
        "S12.1",
        "C69",
        "O56"
      ],
      "syntax": "<'padding-left'>",
      "relevance": 52,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/padding-inline-start"
        }
      ],
      "description": "Logical 'padding-left'. Mapping depends on the parent element\u2019s 'writing-mode', 'direction', and 'text-orientation'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "padding-left",
      "syntax": "<length> | <percentage>",
      "relevance": 91,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/padding-left"
        }
      ],
      "description": "Shorthand property to set values for the thickness of the padding area. If left is omitted, it is the same as right. If bottom is omitted it is the same as top, if right is omitted it is the same as top. The value may not be negative.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "padding-right",
      "syntax": "<length> | <percentage>",
      "relevance": 90,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/padding-right"
        }
      ],
      "description": "Shorthand property to set values for the thickness of the padding area. If left is omitted, it is the same as right. If bottom is omitted it is the same as top, if right is omitted it is the same as top. The value may not be negative.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "padding-top",
      "syntax": "<length> | <percentage>",
      "relevance": 90,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/padding-top"
        }
      ],
      "description": "Shorthand property to set values for the thickness of the padding area. If left is omitted, it is the same as right. If bottom is omitted it is the same as top, if right is omitted it is the same as top. The value may not be negative.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "page-break-after",
      "values": [
        {
          "name": "always",
          "description": "Always force a page break after the generated box."
        },
        {
          "name": "auto",
          "description": "Neither force nor forbid a page break after generated box."
        },
        {
          "name": "avoid",
          "description": "Avoid a page break after the generated box."
        },
        {
          "name": "left",
          "description": "Force one or two page breaks after the generated box so that the next page is formatted as a left page."
        },
        {
          "name": "right",
          "description": "Force one or two page breaks after the generated box so that the next page is formatted as a right page."
        }
      ],
      "syntax": "auto | always | avoid | left | right | recto | verso",
      "relevance": 52,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/page-break-after"
        }
      ],
      "description": "Defines rules for page breaks after an element.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "page-break-before",
      "values": [
        {
          "name": "always",
          "description": "Always force a page break before the generated box."
        },
        {
          "name": "auto",
          "description": "Neither force nor forbid a page break before the generated box."
        },
        {
          "name": "avoid",
          "description": "Avoid a page break before the generated box."
        },
        {
          "name": "left",
          "description": "Force one or two page breaks before the generated box so that the next page is formatted as a left page."
        },
        {
          "name": "right",
          "description": "Force one or two page breaks before the generated box so that the next page is formatted as a right page."
        }
      ],
      "syntax": "auto | always | avoid | left | right | recto | verso",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/page-break-before"
        }
      ],
      "description": "Defines rules for page breaks before an element.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "page-break-inside",
      "values": [
        {
          "name": "auto",
          "description": "Neither force nor forbid a page break inside the generated box."
        },
        {
          "name": "avoid",
          "description": "Avoid a page break inside the generated box."
        }
      ],
      "syntax": "auto | avoid",
      "relevance": 53,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/page-break-inside"
        }
      ],
      "description": "Defines rules for page breaks inside an element.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "paint-order",
      "browsers": [
        "E17",
        "FF60",
        "S8",
        "C35",
        "O22"
      ],
      "values": [
        {
          "name": "fill"
        },
        {
          "name": "markers"
        },
        {
          "name": "normal",
          "description": "The element is painted with the standard order of painting operations: the 'fill' is painted first, then its 'stroke' and finally its markers."
        },
        {
          "name": "stroke"
        }
      ],
      "syntax": "normal | [ fill || stroke || markers ]",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/paint-order"
        }
      ],
      "description": "Controls the order that the three paint operations that shapes and text are rendered with: their fill, their stroke and any markers they might have.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "perspective",
      "values": [
        {
          "name": "none",
          "description": "No perspective transform is applied."
        }
      ],
      "syntax": "none | <length>",
      "relevance": 55,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/perspective"
        }
      ],
      "description": "Applies the same transform as the perspective(<number>) transform function, except that it applies only to the positioned or transformed children of the element, not to the transform on the element itself.",
      "restrictions": [
        "length",
        "enum"
      ]
    },
    {
      "name": "perspective-origin",
      "syntax": "<position>",
      "relevance": 51,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/perspective-origin"
        }
      ],
      "description": "Establishes the origin for the perspective property. It effectively sets the X and Y position at which the viewer appears to be looking at the children of the element.",
      "restrictions": [
        "position",
        "percentage",
        "length"
      ]
    },
    {
      "name": "pointer-events",
      "values": [
        {
          "name": "all",
          "description": "The given element can be the target element for pointer events whenever the pointer is over either the interior or the perimeter of the element."
        },
        {
          "name": "fill",
          "description": "The given element can be the target element for pointer events whenever the pointer is over the interior of the element."
        },
        {
          "name": "none",
          "description": "The given element does not receive pointer events."
        },
        {
          "name": "painted",
          "description": 'The given element can be the target element for pointer events when the pointer is over a "painted" area. '
        },
        {
          "name": "stroke",
          "description": "The given element can be the target element for pointer events whenever the pointer is over the perimeter of the element."
        },
        {
          "name": "visible",
          "description": "The given element can be the target element for pointer events when the \u2018visibility\u2019 property is set to visible and the pointer is over either the interior or the perimeter of the element."
        },
        {
          "name": "visibleFill",
          "description": "The given element can be the target element for pointer events when the \u2018visibility\u2019 property is set to visible and when the pointer is over the interior of the element."
        },
        {
          "name": "visiblePainted",
          "description": "The given element can be the target element for pointer events when the \u2018visibility\u2019 property is set to visible and when the pointer is over a \u2018painted\u2019 area."
        },
        {
          "name": "visibleStroke",
          "description": "The given element can be the target element for pointer events when the \u2018visibility\u2019 property is set to visible and when the pointer is over the perimeter of the element."
        }
      ],
      "syntax": "auto | none | visiblePainted | visibleFill | visibleStroke | visible | painted | fill | stroke | all | inherit",
      "relevance": 82,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/pointer-events"
        }
      ],
      "description": "Specifies under what circumstances a given element can be the target element for a pointer event.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "position",
      "values": [
        {
          "name": "absolute",
          "description": "The box's position (and possibly size) is specified with the 'top', 'right', 'bottom', and 'left' properties. These properties specify offsets with respect to the box's 'containing block'."
        },
        {
          "name": "fixed",
          "description": "The box's position is calculated according to the 'absolute' model, but in addition, the box is fixed with respect to some reference. As with the 'absolute' model, the box's margins do not collapse with any other margins."
        },
        {
          "name": "-ms-page",
          "description": "The box's position is calculated according to the 'absolute' model."
        },
        {
          "name": "relative",
          "description": "The box's position is calculated according to the normal flow (this is called the position in normal flow). Then the box is offset relative to its normal position."
        },
        {
          "name": "static",
          "description": "The box is a normal box, laid out according to the normal flow. The 'top', 'right', 'bottom', and 'left' properties do not apply."
        },
        {
          "name": "sticky",
          "description": "The box's position is calculated according to the normal flow. Then the box is offset relative to its flow root and containing block and in all cases, including table elements, does not affect the position of any following boxes."
        },
        {
          "name": "-webkit-sticky",
          "description": "The box's position is calculated according to the normal flow. Then the box is offset relative to its flow root and containing block and in all cases, including table elements, does not affect the position of any following boxes."
        }
      ],
      "syntax": "static | relative | absolute | sticky | fixed",
      "relevance": 96,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/position"
        }
      ],
      "description": "The position CSS property sets how an element is positioned in a document. The top, right, bottom, and left properties determine the final location of positioned elements.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "prefix",
      "browsers": [
        "FF33"
      ],
      "syntax": "<symbol>",
      "relevance": 50,
      "description": "@counter-style descriptor. Specifies a <symbol> that is prepended to the marker representation.",
      "restrictions": [
        "image",
        "string",
        "identifier"
      ]
    },
    {
      "name": "quotes",
      "values": [
        {
          "name": "none",
          "description": "The 'open-quote' and 'close-quote' values of the 'content' property produce no quotations marks, as if they were 'no-open-quote' and 'no-close-quote' respectively."
        }
      ],
      "syntax": "none | auto | [ <string> <string> ]+",
      "relevance": 53,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/quotes"
        }
      ],
      "description": "Specifies quotation marks for any number of embedded quotations.",
      "restrictions": [
        "string"
      ]
    },
    {
      "name": "range",
      "browsers": [
        "FF33"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The range depends on the counter system."
        },
        {
          "name": "infinite",
          "description": "If used as the first value in a range, it represents negative infinity; if used as the second value, it represents positive infinity."
        }
      ],
      "syntax": "[ [ <integer> | infinite ]{2} ]# | auto",
      "relevance": 50,
      "description": "@counter-style descriptor. Defines the ranges over which the counter style is defined.",
      "restrictions": [
        "integer",
        "enum"
      ]
    },
    {
      "name": "resize",
      "browsers": [
        "E79",
        "FF4",
        "S3",
        "C1",
        "O12.1"
      ],
      "values": [
        {
          "name": "both",
          "description": "The UA presents a bidirectional resizing mechanism to allow the user to adjust both the height and the width of the element."
        },
        {
          "name": "horizontal",
          "description": "The UA presents a unidirectional horizontal resizing mechanism to allow the user to adjust only the width of the element."
        },
        {
          "name": "none",
          "description": "The UA does not present a resizing mechanism on the element, and the user is given no direct manipulation mechanism to resize the element."
        },
        {
          "name": "vertical",
          "description": "The UA presents a unidirectional vertical resizing mechanism to allow the user to adjust only the height of the element."
        }
      ],
      "syntax": "none | both | horizontal | vertical | block | inline",
      "relevance": 61,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/resize"
        }
      ],
      "description": "Specifies whether or not an element is resizable by the user, and if so, along which axis/axes.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "right",
      "values": [
        {
          "name": "auto",
          "description": "For non-replaced elements, the effect of this value depends on which of related properties have the value 'auto' as well"
        }
      ],
      "syntax": "<length> | <percentage> | auto",
      "relevance": 91,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/right"
        }
      ],
      "description": "Specifies how far an absolutely positioned box's right margin edge is offset to the left of the right edge of the box's 'containing block'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "ruby-align",
      "browsers": [
        "FF38"
      ],
      "values": [
        {
          "name": "auto",
          "browsers": [
            "FF38"
          ],
          "description": "The user agent determines how the ruby contents are aligned. This is the initial value."
        },
        {
          "name": "center",
          "description": "The ruby content is centered within its box."
        },
        {
          "name": "distribute-letter",
          "browsers": [
            "FF38"
          ],
          "description": "If the width of the ruby text is smaller than that of the base, then the ruby text contents are evenly distributed across the width of the base, with the first and last ruby text glyphs lining up with the corresponding first and last base glyphs. If the width of the ruby text is at least the width of the base, then the letters of the base are evenly distributed across the width of the ruby text."
        },
        {
          "name": "distribute-space",
          "browsers": [
            "FF38"
          ],
          "description": "If the width of the ruby text is smaller than that of the base, then the ruby text contents are evenly distributed across the width of the base, with a certain amount of white space preceding the first and following the last character in the ruby text. That amount of white space is normally equal to half the amount of inter-character space of the ruby text."
        },
        {
          "name": "left",
          "description": "The ruby text content is aligned with the start edge of the base."
        },
        {
          "name": "line-edge",
          "browsers": [
            "FF38"
          ],
          "description": "If the ruby text is not adjacent to a line edge, it is aligned as in 'auto'. If it is adjacent to a line edge, then it is still aligned as in auto, but the side of the ruby text that touches the end of the line is lined up with the corresponding edge of the base."
        },
        {
          "name": "right",
          "browsers": [
            "FF38"
          ],
          "description": "The ruby text content is aligned with the end edge of the base."
        },
        {
          "name": "start",
          "browsers": [
            "FF38"
          ],
          "description": "The ruby text content is aligned with the start edge of the base."
        },
        {
          "name": "space-between",
          "browsers": [
            "FF38"
          ],
          "description": "The ruby content expands as defined for normal text justification (as defined by 'text-justify'),"
        },
        {
          "name": "space-around",
          "browsers": [
            "FF38"
          ],
          "description": "As for 'space-between' except that there exists an extra justification opportunities whose space is distributed half before and half after the ruby content."
        }
      ],
      "status": "experimental",
      "syntax": "start | center | space-between | space-around",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/ruby-align"
        }
      ],
      "description": "Specifies how text is distributed within the various ruby boxes when their contents do not exactly fill their respective boxes.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "ruby-overhang",
      "browsers": [
        "FF10",
        "IE5"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The ruby text can overhang text adjacent to the base on either side. This is the initial value."
        },
        {
          "name": "end",
          "description": "The ruby text can overhang the text that follows it."
        },
        {
          "name": "none",
          "description": "The ruby text cannot overhang any text adjacent to its base, only its own base."
        },
        {
          "name": "start",
          "description": "The ruby text can overhang the text that precedes it."
        }
      ],
      "relevance": 50,
      "description": "Determines whether, and on which side, ruby text is allowed to partially overhang any adjacent text in addition to its own base, when the ruby text is wider than the ruby base.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "ruby-position",
      "browsers": [
        "E84",
        "FF38",
        "S7",
        "C84",
        "O70"
      ],
      "values": [
        {
          "name": "after",
          "description": "The ruby text appears after the base. This is a relatively rare setting used in ideographic East Asian writing systems, most easily found in educational text."
        },
        {
          "name": "before",
          "description": "The ruby text appears before the base. This is the most common setting used in ideographic East Asian writing systems."
        },
        {
          "name": "inline"
        },
        {
          "name": "right",
          "description": "The ruby text appears on the right of the base. Unlike 'before' and 'after', this value is not relative to the text flow direction."
        }
      ],
      "status": "experimental",
      "syntax": "[ alternate || [ over | under ] ] | inter-character",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/ruby-position"
        }
      ],
      "description": "Used by the parent of elements with display: ruby-text to control the position of the ruby text with respect to its base.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "ruby-span",
      "browsers": [
        "FF10"
      ],
      "values": [
        {
          "name": "attr(x)",
          "description": "The value of attribute 'x' is a string value. The string value is evaluated as a <number> to determine the number of ruby base elements to be spanned by the annotation element."
        },
        {
          "name": "none",
          "description": "No spanning. The computed value is '1'."
        }
      ],
      "relevance": 50,
      "description": "Determines whether, and on which side, ruby text is allowed to partially overhang any adjacent text in addition to its own base, when the ruby text is wider than the ruby base.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "scrollbar-3dlight-color",
      "browsers": [
        "IE5"
      ],
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/scrollbar-3dlight-color"
        }
      ],
      "description": "Determines the color of the top and left edges of the scroll box and scroll arrows of a scroll bar.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "scrollbar-arrow-color",
      "browsers": [
        "IE5"
      ],
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/scrollbar-arrow-color"
        }
      ],
      "description": "Determines the color of the arrow elements of a scroll arrow.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "scrollbar-base-color",
      "browsers": [
        "IE5"
      ],
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/scrollbar-base-color"
        }
      ],
      "description": "Determines the color of the main elements of a scroll bar, which include the scroll box, track, and scroll arrows.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "scrollbar-darkshadow-color",
      "browsers": [
        "IE5"
      ],
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/scrollbar-darkshadow-color"
        }
      ],
      "description": "Determines the color of the gutter of a scroll bar.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "scrollbar-face-color",
      "browsers": [
        "IE5"
      ],
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/scrollbar-face-color"
        }
      ],
      "description": "Determines the color of the scroll box and scroll arrows of a scroll bar.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "scrollbar-highlight-color",
      "browsers": [
        "IE5"
      ],
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/scrollbar-highlight-color"
        }
      ],
      "description": "Determines the color of the top and left edges of the scroll box and scroll arrows of a scroll bar.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "scrollbar-shadow-color",
      "browsers": [
        "IE5"
      ],
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/scrollbar-shadow-color"
        }
      ],
      "description": "Determines the color of the bottom and right edges of the scroll box and scroll arrows of a scroll bar.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "scrollbar-track-color",
      "browsers": [
        "IE6"
      ],
      "relevance": 50,
      "description": "Determines the color of the track element of a scroll bar.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "scroll-behavior",
      "browsers": [
        "E79",
        "FF36",
        "S15.4",
        "C61",
        "O48"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Scrolls in an instant fashion."
        },
        {
          "name": "smooth",
          "description": "Scrolls in a smooth fashion using a user-agent-defined timing function and time period."
        }
      ],
      "syntax": "auto | smooth",
      "relevance": 52,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/scroll-behavior"
        }
      ],
      "description": "Specifies the scrolling behavior for a scrolling box, when scrolling happens due to navigation or CSSOM scrolling APIs.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "scroll-snap-coordinate",
      "browsers": [
        "FF39"
      ],
      "values": [
        {
          "name": "none",
          "description": "Specifies that this element does not contribute a snap point."
        }
      ],
      "status": "obsolete",
      "syntax": "none | <position>#",
      "relevance": 0,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/scroll-snap-coordinate"
        }
      ],
      "description": "Defines the x and y coordinate within the element which will align with the nearest ancestor scroll container\u2019s snap-destination for the respective axis.",
      "restrictions": [
        "position",
        "length",
        "percentage",
        "enum"
      ]
    },
    {
      "name": "scroll-snap-destination",
      "browsers": [
        "FF39"
      ],
      "status": "obsolete",
      "syntax": "<position>",
      "relevance": 0,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/scroll-snap-destination"
        }
      ],
      "description": "Define the x and y coordinate within the scroll container\u2019s visual viewport which element snap points will align with.",
      "restrictions": [
        "position",
        "length",
        "percentage"
      ]
    },
    {
      "name": "scroll-snap-points-x",
      "browsers": [
        "FF39",
        "S9"
      ],
      "values": [
        {
          "name": "none",
          "description": "No snap points are defined by this scroll container."
        },
        {
          "name": "repeat()",
          "description": "Defines an interval at which snap points are defined, starting from the container\u2019s relevant start edge."
        }
      ],
      "status": "obsolete",
      "syntax": "none | repeat( <length-percentage> )",
      "relevance": 0,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/scroll-snap-points-x"
        }
      ],
      "description": "Defines the positioning of snap points along the x axis of the scroll container it is applied to.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "scroll-snap-points-y",
      "browsers": [
        "FF39",
        "S9"
      ],
      "values": [
        {
          "name": "none",
          "description": "No snap points are defined by this scroll container."
        },
        {
          "name": "repeat()",
          "description": "Defines an interval at which snap points are defined, starting from the container\u2019s relevant start edge."
        }
      ],
      "status": "obsolete",
      "syntax": "none | repeat( <length-percentage> )",
      "relevance": 0,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/scroll-snap-points-y"
        }
      ],
      "description": "Defines the positioning of snap points along the y axis of the scroll container it is applied to.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "scroll-snap-type",
      "values": [
        {
          "name": "none",
          "description": "The visual viewport of this scroll container must ignore snap points, if any, when scrolled."
        },
        {
          "name": "mandatory",
          "description": "The visual viewport of this scroll container is guaranteed to rest on a snap point when there are no active scrolling operations."
        },
        {
          "name": "proximity",
          "description": "The visual viewport of this scroll container may come to rest on a snap point at the termination of a scroll at the discretion of the UA given the parameters of the scroll."
        }
      ],
      "syntax": "none | [ x | y | block | inline | both ] [ mandatory | proximity ]?",
      "relevance": 52,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/scroll-snap-type"
        }
      ],
      "description": "Defines how strictly snap points are enforced on the scroll container.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "shape-image-threshold",
      "browsers": [
        "E79",
        "FF62",
        "S10.1",
        "C37",
        "O24"
      ],
      "syntax": "<alpha-value>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/shape-image-threshold"
        }
      ],
      "description": "Defines the alpha channel threshold used to extract the shape using an image. A value of 0.5 means that the shape will enclose all the pixels that are more than 50% opaque.",
      "restrictions": [
        "number"
      ]
    },
    {
      "name": "shape-margin",
      "browsers": [
        "E79",
        "FF62",
        "S10.1",
        "C37",
        "O24"
      ],
      "syntax": "<length-percentage>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/shape-margin"
        }
      ],
      "description": "Adds a margin to a 'shape-outside'. This defines a new shape that is the smallest contour that includes all the points that are the 'shape-margin' distance outward in the perpendicular direction from a point on the underlying shape.",
      "restrictions": [
        "url",
        "length",
        "percentage"
      ]
    },
    {
      "name": "shape-outside",
      "browsers": [
        "E79",
        "FF62",
        "S10.1",
        "C37",
        "O24"
      ],
      "values": [
        {
          "name": "margin-box",
          "description": "The background is painted within (clipped to) the margin box."
        },
        {
          "name": "none",
          "description": "The float area is unaffected."
        }
      ],
      "syntax": "none | [ <shape-box> || <basic-shape> ] | <image>",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/shape-outside"
        }
      ],
      "description": "Specifies an orthogonal rotation to be applied to an image before it is laid out.",
      "restrictions": [
        "image",
        "box",
        "shape",
        "enum"
      ]
    },
    {
      "name": "shape-rendering",
      "values": [
        {
          "name": "auto",
          "description": "Suppresses aural rendering."
        },
        {
          "name": "crispEdges",
          "description": "Emphasize the contrast between clean edges of artwork over rendering speed and geometric precision."
        },
        {
          "name": "geometricPrecision",
          "description": "Emphasize geometric precision over speed and crisp edges."
        },
        {
          "name": "optimizeSpeed",
          "description": "Emphasize rendering speed over geometric precision and crisp edges."
        }
      ],
      "relevance": 50,
      "description": "Provides hints about what tradeoffs to make as it renders vector graphics elements such as <path> elements and basic shapes such as circles and rectangles.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "size",
      "browsers": [
        "C",
        "O8"
      ],
      "syntax": "<length>{1,2} | auto | [ <page-size> || [ portrait | landscape ] ]",
      "relevance": 53,
      "description": "The size CSS at-rule descriptor, used with the @page at-rule, defines the size and orientation of the box which is used to represent a page. Most of the time, this size corresponds to the target size of the printed page if applicable.",
      "restrictions": [
        "length"
      ]
    },
    {
      "name": "src",
      "values": [
        {
          "name": "url()",
          "description": "Reference font by URL"
        },
        {
          "name": "format()",
          "description": "Optional hint describing the format of the font resource."
        },
        {
          "name": "local()",
          "description": "Format-specific string that identifies a locally available copy of a given font."
        }
      ],
      "syntax": "[ <url> [ format( <string># ) ]? | local( <family-name> ) ]#",
      "relevance": 87,
      "description": "@font-face descriptor. Specifies the resource containing font data. It is required, whether the font is downloadable or locally installed.",
      "restrictions": [
        "enum",
        "url",
        "identifier"
      ]
    },
    {
      "name": "stop-color",
      "relevance": 51,
      "description": "Indicates what color to use at that gradient stop.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "stop-opacity",
      "relevance": 50,
      "description": "Defines the opacity of a given gradient stop.",
      "restrictions": [
        "number(0-1)"
      ]
    },
    {
      "name": "stroke",
      "values": [
        {
          "name": "url()",
          "description": "A URL reference to a paint server element, which is an element that defines a paint server: \u2018hatch\u2019, \u2018linearGradient\u2019, \u2018mesh\u2019, \u2018pattern\u2019, \u2018radialGradient\u2019 and \u2018solidcolor\u2019."
        },
        {
          "name": "none",
          "description": "No paint is applied in this layer."
        }
      ],
      "relevance": 65,
      "description": "Paints along the outline of the given graphical element.",
      "restrictions": [
        "color",
        "enum",
        "url"
      ]
    },
    {
      "name": "stroke-dasharray",
      "values": [
        {
          "name": "none",
          "description": "Indicates that no dashing is used."
        }
      ],
      "relevance": 58,
      "description": "Controls the pattern of dashes and gaps used to stroke paths.",
      "restrictions": [
        "length",
        "percentage",
        "number",
        "enum"
      ]
    },
    {
      "name": "stroke-dashoffset",
      "relevance": 59,
      "description": "Specifies the distance into the dash pattern to start the dash.",
      "restrictions": [
        "percentage",
        "length"
      ]
    },
    {
      "name": "stroke-linecap",
      "values": [
        {
          "name": "butt",
          "description": "Indicates that the stroke for each subpath does not extend beyond its two endpoints."
        },
        {
          "name": "round",
          "description": "Indicates that at each end of each subpath, the shape representing the stroke will be extended by a half circle with a radius equal to the stroke width."
        },
        {
          "name": "square",
          "description": "Indicates that at the end of each subpath, the shape representing the stroke will be extended by a rectangle with the same width as the stroke width and whose length is half of the stroke width."
        }
      ],
      "relevance": 53,
      "description": "Specifies the shape to be used at the end of open subpaths when they are stroked.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "stroke-linejoin",
      "values": [
        {
          "name": "bevel",
          "description": "Indicates that a bevelled corner is to be used to join path segments."
        },
        {
          "name": "miter",
          "description": "Indicates that a sharp corner is to be used to join path segments."
        },
        {
          "name": "round",
          "description": "Indicates that a round corner is to be used to join path segments."
        }
      ],
      "relevance": 50,
      "description": "Specifies the shape to be used at the corners of paths or basic shapes when they are stroked.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "stroke-miterlimit",
      "relevance": 51,
      "description": "When two line segments meet at a sharp angle and miter joins have been specified for 'stroke-linejoin', it is possible for the miter to extend far beyond the thickness of the line stroking the path.",
      "restrictions": [
        "number"
      ]
    },
    {
      "name": "stroke-opacity",
      "relevance": 52,
      "description": "Specifies the opacity of the painting operation used to stroke the current object.",
      "restrictions": [
        "number(0-1)"
      ]
    },
    {
      "name": "stroke-width",
      "relevance": 61,
      "description": "Specifies the width of the stroke on the current object.",
      "restrictions": [
        "percentage",
        "length"
      ]
    },
    {
      "name": "suffix",
      "browsers": [
        "FF33"
      ],
      "syntax": "<symbol>",
      "relevance": 50,
      "description": "@counter-style descriptor. Specifies a <symbol> that is appended to the marker representation.",
      "restrictions": [
        "image",
        "string",
        "identifier"
      ]
    },
    {
      "name": "system",
      "browsers": [
        "FF33"
      ],
      "values": [
        {
          "name": "additive",
          "description": "Represents \u201Csign-value\u201D numbering systems, which, rather than using reusing digits in different positions to change their value, define additional digits with much larger values, so that the value of the number can be obtained by adding all the digits together."
        },
        {
          "name": "alphabetic",
          "description": 'Interprets the list of counter symbols as digits to an alphabetic numbering system, similar to the default lower-alpha counter style, which wraps from "a", "b", "c", to "aa", "ab", "ac".'
        },
        {
          "name": "cyclic",
          "description": "Cycles repeatedly through its provided symbols, looping back to the beginning when it reaches the end of the list."
        },
        {
          "name": "extends",
          "description": "Use the algorithm of another counter style, but alter other aspects."
        },
        {
          "name": "fixed",
          "description": "Runs through its list of counter symbols once, then falls back."
        },
        {
          "name": "numeric",
          "description": `interprets the list of counter symbols as digits to a "place-value" numbering system, similar to the default 'decimal' counter style.`
        },
        {
          "name": "symbolic",
          "description": "Cycles repeatedly through its provided symbols, doubling, tripling, etc. the symbols on each successive pass through the list."
        }
      ],
      "syntax": "cyclic | numeric | alphabetic | symbolic | additive | [ fixed <integer>? ] | [ extends <counter-style-name> ]",
      "relevance": 50,
      "description": "@counter-style descriptor. Specifies which algorithm will be used to construct the counter\u2019s representation based on the counter value.",
      "restrictions": [
        "enum",
        "integer"
      ]
    },
    {
      "name": "symbols",
      "browsers": [
        "FF33"
      ],
      "syntax": "<symbol>+",
      "relevance": 50,
      "description": "@counter-style descriptor. Specifies the symbols used by the marker-construction algorithm specified by the system descriptor.",
      "restrictions": [
        "image",
        "string",
        "identifier"
      ]
    },
    {
      "name": "table-layout",
      "values": [
        {
          "name": "auto",
          "description": "Use any automatic table layout algorithm."
        },
        {
          "name": "fixed",
          "description": "Use the fixed table layout algorithm."
        }
      ],
      "syntax": "auto | fixed",
      "relevance": 60,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/table-layout"
        }
      ],
      "description": "Controls the algorithm used to lay out the table cells, rows, and columns.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "tab-size",
      "browsers": [
        "E79",
        "FF91",
        "S7",
        "C21",
        "O15"
      ],
      "syntax": "<integer> | <length>",
      "relevance": 51,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/tab-size"
        }
      ],
      "description": "Determines the width of the tab character (U+0009), in space characters (U+0020), when rendered.",
      "restrictions": [
        "integer",
        "length"
      ]
    },
    {
      "name": "text-align",
      "values": [
        {
          "name": "center",
          "description": "The inline contents are centered within the line box."
        },
        {
          "name": "end",
          "description": "The inline contents are aligned to the end edge of the line box."
        },
        {
          "name": "justify",
          "description": "The text is justified according to the method specified by the 'text-justify' property."
        },
        {
          "name": "left",
          "description": "The inline contents are aligned to the left edge of the line box. In vertical text, 'left' aligns to the edge of the line box that would be the start edge for left-to-right text."
        },
        {
          "name": "right",
          "description": "The inline contents are aligned to the right edge of the line box. In vertical text, 'right' aligns to the edge of the line box that would be the end edge for left-to-right text."
        },
        {
          "name": "start",
          "description": "The inline contents are aligned to the start edge of the line box."
        }
      ],
      "syntax": "start | end | left | right | center | justify | match-parent",
      "relevance": 94,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/text-align"
        }
      ],
      "description": "Describes how inline contents of a block are horizontally aligned if the contents do not completely fill the line box.",
      "restrictions": [
        "string"
      ]
    },
    {
      "name": "text-align-last",
      "browsers": [
        "E12",
        "FF49",
        "C47",
        "IE5.5",
        "O34"
      ],
      "values": [
        {
          "name": "auto",
          "description": "Content on the affected line is aligned per 'text-align' unless 'text-align' is set to 'justify', in which case it is 'start-aligned'."
        },
        {
          "name": "center",
          "description": "The inline contents are centered within the line box."
        },
        {
          "name": "justify",
          "description": "The text is justified according to the method specified by the 'text-justify' property."
        },
        {
          "name": "left",
          "description": "The inline contents are aligned to the left edge of the line box. In vertical text, 'left' aligns to the edge of the line box that would be the start edge for left-to-right text."
        },
        {
          "name": "right",
          "description": "The inline contents are aligned to the right edge of the line box. In vertical text, 'right' aligns to the edge of the line box that would be the end edge for left-to-right text."
        }
      ],
      "syntax": "auto | start | end | left | right | center | justify",
      "relevance": 51,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/text-align-last"
        }
      ],
      "description": "Describes how the last line of a block or a line right before a forced line break is aligned when 'text-align' is set to 'justify'.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "text-anchor",
      "values": [
        {
          "name": "end",
          "description": "The rendered characters are aligned such that the end of the resulting rendered text is at the initial current text position."
        },
        {
          "name": "middle",
          "description": "The rendered characters are aligned such that the geometric middle of the resulting rendered text is at the initial current text position."
        },
        {
          "name": "start",
          "description": "The rendered characters are aligned such that the start of the resulting rendered text is at the initial current text position."
        }
      ],
      "relevance": 50,
      "description": "Used to align (start-, middle- or end-alignment) a string of text relative to a given point.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "text-decoration",
      "values": [
        {
          "name": "dashed",
          "description": "Produces a dashed line style."
        },
        {
          "name": "dotted",
          "description": "Produces a dotted line."
        },
        {
          "name": "double",
          "description": "Produces a double line."
        },
        {
          "name": "line-through",
          "description": "Each line of text has a line through the middle."
        },
        {
          "name": "none",
          "description": "Produces no line."
        },
        {
          "name": "overline",
          "description": "Each line of text has a line above it."
        },
        {
          "name": "solid",
          "description": "Produces a solid line."
        },
        {
          "name": "underline",
          "description": "Each line of text is underlined."
        },
        {
          "name": "wavy",
          "description": "Produces a wavy line."
        }
      ],
      "syntax": "<'text-decoration-line'> || <'text-decoration-style'> || <'text-decoration-color'> || <'text-decoration-thickness'>",
      "relevance": 92,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/text-decoration"
        }
      ],
      "description": "Decorations applied to font used for an element's text.",
      "restrictions": [
        "enum",
        "color"
      ]
    },
    {
      "name": "text-decoration-color",
      "browsers": [
        "E79",
        "FF36",
        "S12.1",
        "C57",
        "O44"
      ],
      "syntax": "<color>",
      "relevance": 52,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/text-decoration-color"
        }
      ],
      "description": "Specifies the color of text decoration (underlines overlines, and line-throughs) set on the element with text-decoration-line.",
      "restrictions": [
        "color"
      ]
    },
    {
      "name": "text-decoration-line",
      "browsers": [
        "E79",
        "FF36",
        "S12.1",
        "C57",
        "O44"
      ],
      "values": [
        {
          "name": "line-through",
          "description": "Each line of text has a line through the middle."
        },
        {
          "name": "none",
          "description": "Neither produces nor inhibits text decoration."
        },
        {
          "name": "overline",
          "description": "Each line of text has a line above it."
        },
        {
          "name": "underline",
          "description": "Each line of text is underlined."
        }
      ],
      "syntax": "none | [ underline || overline || line-through || blink ] | spelling-error | grammar-error",
      "relevance": 52,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/text-decoration-line"
        }
      ],
      "description": "Specifies what line decorations, if any, are added to the element.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "text-decoration-style",
      "browsers": [
        "E79",
        "FF36",
        "S12.1",
        "C57",
        "O44"
      ],
      "values": [
        {
          "name": "dashed",
          "description": "Produces a dashed line style."
        },
        {
          "name": "dotted",
          "description": "Produces a dotted line."
        },
        {
          "name": "double",
          "description": "Produces a double line."
        },
        {
          "name": "none",
          "description": "Produces no line."
        },
        {
          "name": "solid",
          "description": "Produces a solid line."
        },
        {
          "name": "wavy",
          "description": "Produces a wavy line."
        }
      ],
      "syntax": "solid | double | dotted | dashed | wavy",
      "relevance": 51,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/text-decoration-style"
        }
      ],
      "description": "Specifies the line style for underline, line-through and overline text decoration.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "text-indent",
      "values": [],
      "syntax": "<length-percentage> && hanging? && each-line?",
      "relevance": 68,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/text-indent"
        }
      ],
      "description": "Specifies the indentation applied to lines of inline content in a block. The indentation only affects the first line of inline content in the block unless the 'hanging' keyword is specified, in which case it affects all lines except the first.",
      "restrictions": [
        "percentage",
        "length"
      ]
    },
    {
      "name": "text-justify",
      "browsers": [
        "E12",
        "FF55",
        "C32",
        "IE11",
        "O19"
      ],
      "values": [
        {
          "name": "auto",
          "description": "The UA determines the justification algorithm to follow, based on a balance between performance and adequate presentation quality."
        },
        {
          "name": "distribute",
          "description": "Justification primarily changes spacing both at word separators and at grapheme cluster boundaries in all scripts except those in the connected and cursive groups. This value is sometimes used in e.g. Japanese, often with the 'text-align-last' property."
        },
        {
          "name": "distribute-all-lines"
        },
        {
          "name": "inter-cluster",
          "description": "Justification primarily changes spacing at word separators and at grapheme cluster boundaries in clustered scripts. This value is typically used for Southeast Asian scripts such as Thai."
        },
        {
          "name": "inter-ideograph",
          "description": "Justification primarily changes spacing at word separators and at inter-graphemic boundaries in scripts that use no word spaces. This value is typically used for CJK languages."
        },
        {
          "name": "inter-word",
          "description": "Justification primarily changes spacing at word separators. This value is typically used for languages that separate words using spaces, like English or (sometimes) Korean."
        },
        {
          "name": "kashida",
          "description": "Justification primarily stretches Arabic and related scripts through the use of kashida or other calligraphic elongation."
        },
        {
          "name": "newspaper"
        }
      ],
      "syntax": "auto | inter-character | inter-word | none",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/text-justify"
        }
      ],
      "description": "Selects the justification algorithm used when 'text-align' is set to 'justify'. The property applies to block containers, but the UA may (but is not required to) also support it on inline elements.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "text-orientation",
      "browsers": [
        "E79",
        "FF41",
        "S14",
        "C48",
        "O35"
      ],
      "values": [
        {
          "name": "sideways",
          "browsers": [
            "E79",
            "FF41",
            "S14",
            "C48",
            "O35"
          ],
          "description": "This value is equivalent to 'sideways-right' in 'vertical-rl' writing mode and equivalent to 'sideways-left' in 'vertical-lr' writing mode."
        },
        {
          "name": "sideways-right",
          "browsers": [
            "E79",
            "FF41",
            "S14",
            "C48",
            "O35"
          ],
          "description": "In vertical writing modes, this causes text to be set as if in a horizontal layout, but rotated 90\xB0 clockwise."
        },
        {
          "name": "upright",
          "description": "In vertical writing modes, characters from horizontal-only scripts are rendered upright, i.e. in their standard horizontal orientation."
        }
      ],
      "syntax": "mixed | upright | sideways",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/text-orientation"
        }
      ],
      "description": "Specifies the orientation of text within a line.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "text-overflow",
      "values": [
        {
          "name": "clip",
          "description": "Clip inline content that overflows. Characters may be only partially rendered."
        },
        {
          "name": "ellipsis",
          "description": "Render an ellipsis character (U+2026) to represent clipped inline content."
        }
      ],
      "syntax": "[ clip | ellipsis | <string> ]{1,2}",
      "relevance": 82,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/text-overflow"
        }
      ],
      "description": "Text can overflow for example when it is prevented from wrapping.",
      "restrictions": [
        "enum",
        "string"
      ]
    },
    {
      "name": "text-rendering",
      "browsers": [
        "E79",
        "FF1",
        "S5",
        "C4",
        "O15"
      ],
      "values": [
        {
          "name": "auto"
        },
        {
          "name": "geometricPrecision",
          "description": "Indicates that the user agent shall emphasize geometric precision over legibility and rendering speed."
        },
        {
          "name": "optimizeLegibility",
          "description": "Indicates that the user agent shall emphasize legibility over rendering speed and geometric precision."
        },
        {
          "name": "optimizeSpeed",
          "description": "Indicates that the user agent shall emphasize rendering speed over legibility and geometric precision."
        }
      ],
      "syntax": "auto | optimizeSpeed | optimizeLegibility | geometricPrecision",
      "relevance": 70,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/text-rendering"
        }
      ],
      "description": "The creator of SVG content might want to provide a hint to the implementation about what tradeoffs to make as it renders text. The \u2018text-rendering\u2019 property provides these hints.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "text-shadow",
      "values": [
        {
          "name": "none",
          "description": "No shadow."
        }
      ],
      "syntax": "none | <shadow-t>#",
      "relevance": 74,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/text-shadow"
        }
      ],
      "description": "Enables shadow effects to be applied to the text of the element.",
      "restrictions": [
        "length",
        "color"
      ]
    },
    {
      "name": "text-transform",
      "values": [
        {
          "name": "capitalize",
          "description": "Puts the first typographic letter unit of each word in titlecase."
        },
        {
          "name": "lowercase",
          "description": "Puts all letters in lowercase."
        },
        {
          "name": "none",
          "description": "No effects."
        },
        {
          "name": "uppercase",
          "description": "Puts all letters in uppercase."
        }
      ],
      "syntax": "none | capitalize | uppercase | lowercase | full-width | full-size-kana",
      "relevance": 86,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/text-transform"
        }
      ],
      "description": "Controls capitalization effects of an element\u2019s text.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "text-underline-position",
      "values": [
        {
          "name": "above"
        },
        {
          "name": "auto",
          "description": "The user agent may use any algorithm to determine the underline\u2019s position. In horizontal line layout, the underline should be aligned as for alphabetic. In vertical line layout, if the language is set to Japanese or Korean, the underline should be aligned as for over."
        },
        {
          "name": "below",
          "description": "The underline is aligned with the under edge of the element\u2019s content box."
        }
      ],
      "syntax": "auto | from-font | [ under || [ left | right ] ]",
      "relevance": 50,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/text-underline-position"
        }
      ],
      "description": "Sets the position of an underline specified on the same element: it does not affect underlines specified by ancestor elements. This property is typically used in vertical writing contexts such as in Japanese documents where it often desired to have the underline appear 'over' (to the right of) the affected run of text",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "top",
      "values": [
        {
          "name": "auto",
          "description": "For non-replaced elements, the effect of this value depends on which of related properties have the value 'auto' as well"
        }
      ],
      "syntax": "<length> | <percentage> | auto",
      "relevance": 95,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/top"
        }
      ],
      "description": "Specifies how far an absolutely positioned box's top margin edge is offset below the top edge of the box's 'containing block'.",
      "restrictions": [
        "length",
        "percentage"
      ]
    },
    {
      "name": "touch-action",
      "values": [
        {
          "name": "auto",
          "description": "The user agent may determine any permitted touch behaviors for touches that begin on the element."
        },
        {
          "name": "cross-slide-x"
        },
        {
          "name": "cross-slide-y"
        },
        {
          "name": "double-tap-zoom"
        },
        {
          "name": "manipulation",
          "description": "The user agent may consider touches that begin on the element only for the purposes of scrolling and continuous zooming."
        },
        {
          "name": "none",
          "description": "Touches that begin on the element must not trigger default touch behaviors."
        },
        {
          "name": "pan-x",
          "description": "The user agent may consider touches that begin on the element only for the purposes of horizontally scrolling the element\u2019s nearest ancestor with horizontally scrollable content."
        },
        {
          "name": "pan-y",
          "description": "The user agent may consider touches that begin on the element only for the purposes of vertically scrolling the element\u2019s nearest ancestor with vertically scrollable content."
        },
        {
          "name": "pinch-zoom"
        }
      ],
      "syntax": "auto | none | [ [ pan-x | pan-left | pan-right ] || [ pan-y | pan-up | pan-down ] || pinch-zoom ] | manipulation",
      "relevance": 67,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/touch-action"
        }
      ],
      "description": "Determines whether touch input may trigger default behavior supplied by user agent.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "transform",
      "values": [
        {
          "name": "matrix()",
          "description": "Specifies a 2D transformation in the form of a transformation matrix of six values. matrix(a,b,c,d,e,f) is equivalent to applying the transformation matrix [a b c d e f]"
        },
        {
          "name": "matrix3d()",
          "description": "Specifies a 3D transformation as a 4x4 homogeneous matrix of 16 values in column-major order."
        },
        {
          "name": "none"
        },
        {
          "name": "perspective()",
          "description": "Specifies a perspective projection matrix."
        },
        {
          "name": "rotate()",
          "description": "Specifies a 2D rotation by the angle specified in the parameter about the origin of the element, as defined by the transform-origin property."
        },
        {
          "name": "rotate3d()",
          "description": "Specifies a clockwise 3D rotation by the angle specified in last parameter about the [x,y,z] direction vector described by the first 3 parameters."
        },
        {
          "name": "rotateX('angle')",
          "description": "Specifies a clockwise rotation by the given angle about the X axis."
        },
        {
          "name": "rotateY('angle')",
          "description": "Specifies a clockwise rotation by the given angle about the Y axis."
        },
        {
          "name": "rotateZ('angle')",
          "description": "Specifies a clockwise rotation by the given angle about the Z axis."
        },
        {
          "name": "scale()",
          "description": "Specifies a 2D scale operation by the [sx,sy] scaling vector described by the 2 parameters. If the second parameter is not provided, it is takes a value equal to the first."
        },
        {
          "name": "scale3d()",
          "description": "Specifies a 3D scale operation by the [sx,sy,sz] scaling vector described by the 3 parameters."
        },
        {
          "name": "scaleX()",
          "description": "Specifies a scale operation using the [sx,1] scaling vector, where sx is given as the parameter."
        },
        {
          "name": "scaleY()",
          "description": "Specifies a scale operation using the [sy,1] scaling vector, where sy is given as the parameter."
        },
        {
          "name": "scaleZ()",
          "description": "Specifies a scale operation using the [1,1,sz] scaling vector, where sz is given as the parameter."
        },
        {
          "name": "skew()",
          "description": "Specifies a skew transformation along the X and Y axes. The first angle parameter specifies the skew on the X axis. The second angle parameter specifies the skew on the Y axis. If the second parameter is not given then a value of 0 is used for the Y angle (ie: no skew on the Y axis)."
        },
        {
          "name": "skewX()",
          "description": "Specifies a skew transformation along the X axis by the given angle."
        },
        {
          "name": "skewY()",
          "description": "Specifies a skew transformation along the Y axis by the given angle."
        },
        {
          "name": "translate()",
          "description": "Specifies a 2D translation by the vector [tx, ty], where tx is the first translation-value parameter and ty is the optional second translation-value parameter."
        },
        {
          "name": "translate3d()",
          "description": "Specifies a 3D translation by the vector [tx,ty,tz], with tx, ty and tz being the first, second and third translation-value parameters respectively."
        },
        {
          "name": "translateX()",
          "description": "Specifies a translation by the given amount in the X direction."
        },
        {
          "name": "translateY()",
          "description": "Specifies a translation by the given amount in the Y direction."
        },
        {
          "name": "translateZ()",
          "description": "Specifies a translation by the given amount in the Z direction. Note that percentage values are not allowed in the translateZ translation-value, and if present are evaluated as 0."
        }
      ],
      "syntax": "none | <transform-list>",
      "relevance": 90,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/transform"
        }
      ],
      "description": "A two-dimensional transformation is applied to an element through the 'transform' property. This property contains a list of transform functions similar to those allowed by SVG.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "transform-origin",
      "syntax": "[ <length-percentage> | left | center | right | top | bottom ] | [ [ <length-percentage> | left | center | right ] && [ <length-percentage> | top | center | bottom ] ] <length>?",
      "relevance": 77,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/transform-origin"
        }
      ],
      "description": "Establishes the origin of transformation for an element.",
      "restrictions": [
        "position",
        "length",
        "percentage"
      ]
    },
    {
      "name": "transform-style",
      "browsers": [
        "E12",
        "FF16",
        "S9",
        "C36",
        "O23"
      ],
      "values": [
        {
          "name": "flat",
          "description": "All children of this element are rendered flattened into the 2D plane of the element."
        },
        {
          "name": "preserve-3d",
          "browsers": [
            "E12",
            "FF16",
            "S9",
            "C36",
            "O23"
          ],
          "description": "Flattening is not performed, so children maintain their position in 3D space."
        }
      ],
      "syntax": "flat | preserve-3d",
      "relevance": 55,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/transform-style"
        }
      ],
      "description": "Defines how nested elements are rendered in 3D space.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "transition",
      "values": [
        {
          "name": "all",
          "description": "Every property that is able to undergo a transition will do so."
        },
        {
          "name": "none",
          "description": "No property will transition."
        }
      ],
      "syntax": "<single-transition>#",
      "relevance": 88,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/transition"
        }
      ],
      "description": "Shorthand property combines four of the transition properties into a single property.",
      "restrictions": [
        "time",
        "property",
        "timing-function",
        "enum"
      ]
    },
    {
      "name": "transition-delay",
      "syntax": "<time>#",
      "relevance": 64,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/transition-delay"
        }
      ],
      "description": "Defines when the transition will start. It allows a transition to begin execution some period of time from when it is applied.",
      "restrictions": [
        "time"
      ]
    },
    {
      "name": "transition-duration",
      "syntax": "<time>#",
      "relevance": 64,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/transition-duration"
        }
      ],
      "description": "Specifies how long the transition from the old value to the new value should take.",
      "restrictions": [
        "time"
      ]
    },
    {
      "name": "transition-property",
      "values": [
        {
          "name": "all",
          "description": "Every property that is able to undergo a transition will do so."
        },
        {
          "name": "none",
          "description": "No property will transition."
        }
      ],
      "syntax": "none | <single-transition-property>#",
      "relevance": 64,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/transition-property"
        }
      ],
      "description": "Specifies the name of the CSS property to which the transition is applied.",
      "restrictions": [
        "property"
      ]
    },
    {
      "name": "transition-timing-function",
      "syntax": "<easing-function>#",
      "relevance": 64,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/transition-timing-function"
        }
      ],
      "description": "Describes how the intermediate values used during a transition will be calculated.",
      "restrictions": [
        "timing-function"
      ]
    },
    {
      "name": "unicode-bidi",
      "values": [
        {
          "name": "bidi-override",
          "description": "Inside the element, reordering is strictly in sequence according to the 'direction' property; the implicit part of the bidirectional algorithm is ignored."
        },
        {
          "name": "embed",
          "description": "If the element is inline-level, this value opens an additional level of embedding with respect to the bidirectional algorithm. The direction of this embedding level is given by the 'direction' property."
        },
        {
          "name": "isolate",
          "description": "The contents of the element are considered to be inside a separate, independent paragraph."
        },
        {
          "name": "isolate-override",
          "description": "This combines the isolation behavior of 'isolate' with the directional override behavior of 'bidi-override'"
        },
        {
          "name": "normal",
          "description": "The element does not open an additional level of embedding with respect to the bidirectional algorithm. For inline-level elements, implicit reordering works across element boundaries."
        },
        {
          "name": "plaintext",
          "description": "For the purposes of the Unicode bidirectional algorithm, the base directionality of each bidi paragraph for which the element forms the containing block is determined not by the element's computed 'direction'."
        }
      ],
      "syntax": "normal | embed | isolate | bidi-override | isolate-override | plaintext",
      "relevance": 57,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/unicode-bidi"
        }
      ],
      "description": "The level of embedding with respect to the bidirectional algorithm.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "unicode-range",
      "values": [
        {
          "name": "U+26",
          "description": "Ampersand."
        },
        {
          "name": "U+20-24F, U+2B0-2FF, U+370-4FF, U+1E00-1EFF, U+2000-20CF, U+2100-23FF, U+2500-26FF, U+E000-F8FF, U+FB00\u2013FB4F",
          "description": "WGL4 character set (Pan-European)."
        },
        {
          "name": "U+20-17F, U+2B0-2FF, U+2000-206F, U+20A0-20CF, U+2100-21FF, U+2600-26FF",
          "description": "The Multilingual European Subset No. 1. Latin. Covers ~44 languages."
        },
        {
          "name": "U+20-2FF, U+370-4FF, U+1E00-20CF, U+2100-23FF, U+2500-26FF, U+FB00-FB4F, U+FFF0-FFFD",
          "description": "The Multilingual European Subset No. 2. Latin, Greek, and Cyrillic. Covers ~128 language."
        },
        {
          "name": "U+20-4FF, U+530-58F, U+10D0-10FF, U+1E00-23FF, U+2440-245F, U+2500-26FF, U+FB00-FB4F, U+FE20-FE2F, U+FFF0-FFFD",
          "description": "The Multilingual European Subset No. 3. Covers all characters belonging to European scripts."
        },
        {
          "name": "U+00-7F",
          "description": "Basic Latin (ASCII)."
        },
        {
          "name": "U+80-FF",
          "description": "Latin-1 Supplement. Accented characters for Western European languages, common punctuation characters, multiplication and division signs."
        },
        {
          "name": "U+100-17F",
          "description": "Latin Extended-A. Accented characters for for Czech, Dutch, Polish, and Turkish."
        },
        {
          "name": "U+180-24F",
          "description": "Latin Extended-B. Croatian, Slovenian, Romanian, Non-European and historic latin, Khoisan, Pinyin, Livonian, Sinology."
        },
        {
          "name": "U+1E00-1EFF",
          "description": "Latin Extended Additional. Vietnamese, German captial sharp s, Medievalist, Latin general use."
        },
        {
          "name": "U+250-2AF",
          "description": "International Phonetic Alphabet Extensions."
        },
        {
          "name": "U+370-3FF",
          "description": "Greek and Coptic."
        },
        {
          "name": "U+1F00-1FFF",
          "description": "Greek Extended. Accented characters for polytonic Greek."
        },
        {
          "name": "U+400-4FF",
          "description": "Cyrillic."
        },
        {
          "name": "U+500-52F",
          "description": "Cyrillic Supplement. Extra letters for Komi, Khanty, Chukchi, Mordvin, Kurdish, Aleut, Chuvash, Abkhaz, Azerbaijani, and Orok."
        },
        {
          "name": "U+00-52F, U+1E00-1FFF, U+2200\u201322FF",
          "description": "Latin, Greek, Cyrillic, some punctuation and symbols."
        },
        {
          "name": "U+530\u201358F",
          "description": "Armenian."
        },
        {
          "name": "U+590\u20135FF",
          "description": "Hebrew."
        },
        {
          "name": "U+600\u20136FF",
          "description": "Arabic."
        },
        {
          "name": "U+750\u201377F",
          "description": "Arabic Supplement. Additional letters for African languages, Khowar, Torwali, Burushaski, and early Persian."
        },
        {
          "name": "U+8A0\u20138FF",
          "description": "Arabic Extended-A. Additional letters for African languages, European and Central Asian languages, Rohingya, Tamazight, Arwi, and Koranic annotation signs."
        },
        {
          "name": "U+700\u201374F",
          "description": "Syriac."
        },
        {
          "name": "U+900\u201397F",
          "description": "Devanagari."
        },
        {
          "name": "U+980\u20139FF",
          "description": "Bengali."
        },
        {
          "name": "U+A00\u2013A7F",
          "description": "Gurmukhi."
        },
        {
          "name": "U+A80\u2013AFF",
          "description": "Gujarati."
        },
        {
          "name": "U+B00\u2013B7F",
          "description": "Oriya."
        },
        {
          "name": "U+B80\u2013BFF",
          "description": "Tamil."
        },
        {
          "name": "U+C00\u2013C7F",
          "description": "Telugu."
        },
        {
          "name": "U+C80\u2013CFF",
          "description": "Kannada."
        },
        {
          "name": "U+D00\u2013D7F",
          "description": "Malayalam."
        },
        {
          "name": "U+D80\u2013DFF",
          "description": "Sinhala."
        },
        {
          "name": "U+118A0\u2013118FF",
          "description": "Warang Citi."
        },
        {
          "name": "U+E00\u2013E7F",
          "description": "Thai."
        },
        {
          "name": "U+1A20\u20131AAF",
          "description": "Tai Tham."
        },
        {
          "name": "U+AA80\u2013AADF",
          "description": "Tai Viet."
        },
        {
          "name": "U+E80\u2013EFF",
          "description": "Lao."
        },
        {
          "name": "U+F00\u2013FFF",
          "description": "Tibetan."
        },
        {
          "name": "U+1000\u2013109F",
          "description": "Myanmar (Burmese)."
        },
        {
          "name": "U+10A0\u201310FF",
          "description": "Georgian."
        },
        {
          "name": "U+1200\u2013137F",
          "description": "Ethiopic."
        },
        {
          "name": "U+1380\u2013139F",
          "description": "Ethiopic Supplement. Extra Syllables for Sebatbeit, and Tonal marks"
        },
        {
          "name": "U+2D80\u20132DDF",
          "description": "Ethiopic Extended. Extra Syllables for Me'en, Blin, and Sebatbeit."
        },
        {
          "name": "U+AB00\u2013AB2F",
          "description": "Ethiopic Extended-A. Extra characters for Gamo-Gofa-Dawro, Basketo, and Gumuz."
        },
        {
          "name": "U+1780\u201317FF",
          "description": "Khmer."
        },
        {
          "name": "U+1800\u201318AF",
          "description": "Mongolian."
        },
        {
          "name": "U+1B80\u20131BBF",
          "description": "Sundanese."
        },
        {
          "name": "U+1CC0\u20131CCF",
          "description": "Sundanese Supplement. Punctuation."
        },
        {
          "name": "U+4E00\u20139FD5",
          "description": "CJK (Chinese, Japanese, Korean) Unified Ideographs. Most common ideographs for modern Chinese and Japanese."
        },
        {
          "name": "U+3400\u20134DB5",
          "description": "CJK Unified Ideographs Extension A. Rare ideographs."
        },
        {
          "name": "U+2F00\u20132FDF",
          "description": "Kangxi Radicals."
        },
        {
          "name": "U+2E80\u20132EFF",
          "description": "CJK Radicals Supplement. Alternative forms of Kangxi Radicals."
        },
        {
          "name": "U+1100\u201311FF",
          "description": "Hangul Jamo."
        },
        {
          "name": "U+AC00\u2013D7AF",
          "description": "Hangul Syllables."
        },
        {
          "name": "U+3040\u2013309F",
          "description": "Hiragana."
        },
        {
          "name": "U+30A0\u201330FF",
          "description": "Katakana."
        },
        {
          "name": "U+A5, U+4E00-9FFF, U+30??, U+FF00-FF9F",
          "description": "Japanese Kanji, Hiragana and Katakana characters plus Yen/Yuan symbol."
        },
        {
          "name": "U+A4D0\u2013A4FF",
          "description": "Lisu."
        },
        {
          "name": "U+A000\u2013A48F",
          "description": "Yi Syllables."
        },
        {
          "name": "U+A490\u2013A4CF",
          "description": "Yi Radicals."
        },
        {
          "name": "U+2000-206F",
          "description": "General Punctuation."
        },
        {
          "name": "U+3000\u2013303F",
          "description": "CJK Symbols and Punctuation."
        },
        {
          "name": "U+2070\u2013209F",
          "description": "Superscripts and Subscripts."
        },
        {
          "name": "U+20A0\u201320CF",
          "description": "Currency Symbols."
        },
        {
          "name": "U+2100\u2013214F",
          "description": "Letterlike Symbols."
        },
        {
          "name": "U+2150\u2013218F",
          "description": "Number Forms."
        },
        {
          "name": "U+2190\u201321FF",
          "description": "Arrows."
        },
        {
          "name": "U+2200\u201322FF",
          "description": "Mathematical Operators."
        },
        {
          "name": "U+2300\u201323FF",
          "description": "Miscellaneous Technical."
        },
        {
          "name": "U+E000-F8FF",
          "description": "Private Use Area."
        },
        {
          "name": "U+FB00\u2013FB4F",
          "description": "Alphabetic Presentation Forms. Ligatures for latin, Armenian, and Hebrew."
        },
        {
          "name": "U+FB50\u2013FDFF",
          "description": "Arabic Presentation Forms-A. Contextual forms / ligatures for Persian, Urdu, Sindhi, Central Asian languages, etc, Arabic pedagogical symbols, word ligatures."
        },
        {
          "name": "U+1F600\u20131F64F",
          "description": "Emoji: Emoticons."
        },
        {
          "name": "U+2600\u201326FF",
          "description": "Emoji: Miscellaneous Symbols."
        },
        {
          "name": "U+1F300\u20131F5FF",
          "description": "Emoji: Miscellaneous Symbols and Pictographs."
        },
        {
          "name": "U+1F900\u20131F9FF",
          "description": "Emoji: Supplemental Symbols and Pictographs."
        },
        {
          "name": "U+1F680\u20131F6FF",
          "description": "Emoji: Transport and Map Symbols."
        }
      ],
      "syntax": "<unicode-range>#",
      "relevance": 73,
      "description": "@font-face descriptor. Defines the set of Unicode codepoints that may be supported by the font face for which it is declared.",
      "restrictions": [
        "unicode-range"
      ]
    },
    {
      "name": "user-select",
      "values": [
        {
          "name": "all",
          "description": "The content of the element must be selected atomically"
        },
        {
          "name": "auto"
        },
        {
          "name": "contain",
          "description": "UAs must not allow a selection which is started in this element to be extended outside of this element."
        },
        {
          "name": "none",
          "description": "The UA must not allow selections to be started in this element."
        },
        {
          "name": "text",
          "description": "The element imposes no constraint on the selection."
        }
      ],
      "syntax": "auto | text | none | contain | all",
      "relevance": 78,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/user-select"
        }
      ],
      "description": "Controls the appearance of selection.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "vertical-align",
      "values": [
        {
          "name": "auto",
          "description": "Align the dominant baseline of the parent box with the equivalent, or heuristically reconstructed, baseline of the element inline box."
        },
        {
          "name": "baseline",
          "description": "Align the 'alphabetic' baseline of the element with the 'alphabetic' baseline of the parent element."
        },
        {
          "name": "bottom",
          "description": "Align the after edge of the extended inline box with the after-edge of the line box."
        },
        {
          "name": "middle",
          "description": "Align the 'middle' baseline of the inline element with the middle baseline of the parent."
        },
        {
          "name": "sub",
          "description": "Lower the baseline of the box to the proper position for subscripts of the parent's box. (This value has no effect on the font size of the element's text.)"
        },
        {
          "name": "super",
          "description": "Raise the baseline of the box to the proper position for superscripts of the parent's box. (This value has no effect on the font size of the element's text.)"
        },
        {
          "name": "text-bottom",
          "description": "Align the bottom of the box with the after-edge of the parent element's font."
        },
        {
          "name": "text-top",
          "description": "Align the top of the box with the before-edge of the parent element's font."
        },
        {
          "name": "top",
          "description": "Align the before edge of the extended inline box with the before-edge of the line box."
        },
        {
          "name": "-webkit-baseline-middle"
        }
      ],
      "syntax": "baseline | sub | super | text-top | text-bottom | middle | top | bottom | <percentage> | <length>",
      "relevance": 92,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/vertical-align"
        }
      ],
      "description": "Affects the vertical positioning of the inline boxes generated by an inline-level element inside a line box.",
      "restrictions": [
        "percentage",
        "length"
      ]
    },
    {
      "name": "visibility",
      "values": [
        {
          "name": "collapse",
          "description": "Table-specific. If used on elements other than rows, row groups, columns, or column groups, 'collapse' has the same meaning as 'hidden'."
        },
        {
          "name": "hidden",
          "description": "The generated box is invisible (fully transparent, nothing is drawn), but still affects layout."
        },
        {
          "name": "visible",
          "description": "The generated box is visible."
        }
      ],
      "syntax": "visible | hidden | collapse",
      "relevance": 88,
      "references": [
        {
          "name": "MDN Reference",
          "url": "https://developer.mozilla.org/docs/Web/CSS/visibility"
        }
      ],
      "description": "Specifies whether the boxes generated by an element are rendered. Invisible boxes still affect layout (set the \u2018display\u2019 property to \u2018none\u2019 to suppress box generation altogether).",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-webkit-animation",
      "browsers": [
        "C",
        "S5"
      ],
      "values": [
        {
          "name": "alternate",
          "description": "The animation cycle iterations that are odd counts are played in the normal direction, and the animation cycle iterations that are even counts are played in a reverse direction."
        },
        {
          "name": "alternate-reverse",
          "description": "The animation cycle iterations that are odd counts are played in the reverse direction, and the animation cycle iterations that are even counts are played in a normal direction."
        },
        {
          "name": "backwards",
          "description": "The beginning property value (as defined in the first @keyframes at-rule) is applied before the animation is displayed, during the period defined by 'animation-delay'."
        },
        {
          "name": "both",
          "description": "Both forwards and backwards fill modes are applied."
        },
        {
          "name": "forwards",
          "description": "The final property value (as defined in the last @keyframes at-rule) is maintained after the animation completes."
        },
        {
          "name": "infinite",
          "description": "Causes the animation to repeat forever."
        },
        {
          "name": "none",
          "description": "No animation is performed"
        },
        {
          "name": "normal",
          "description": "Normal playback."
        },
        {
          "name": "reverse",
          "description": "All iterations of the animation are played in the reverse direction from the way they were specified."
        }
      ],
      "relevance": 50,
      "description": "Shorthand property combines six of the animation properties into a single property.",
      "restrictions": [
        "time",
        "enum",
        "timing-function",
        "identifier",
        "number"
      ]
    },
    {
      "name": "-webkit-animation-delay",
      "browsers": [
        "C",
        "S5"
      ],
      "relevance": 50,
      "description": "Defines when the animation will start.",
      "restrictions": [
        "time"
      ]
    },
    {
      "name": "-webkit-animation-direction",
      "browsers": [
        "C",
        "S5"
      ],
      "values": [
        {
          "name": "alternate",
          "description": "The animation cycle iterations that are odd counts are played in the normal direction, and the animation cycle iterations that are even counts are played in a reverse direction."
        },
        {
          "name": "alternate-reverse",
          "description": "The animation cycle iterations that are odd counts are played in the reverse direction, and the animation cycle iterations that are even counts are played in a normal direction."
        },
        {
          "name": "normal",
          "description": "Normal playback."
        },
        {
          "name": "reverse",
          "description": "All iterations of the animation are played in the reverse direction from the way they were specified."
        }
      ],
      "relevance": 50,
      "description": "Defines whether or not the animation should play in reverse on alternate cycles.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-webkit-animation-duration",
      "browsers": [
        "C",
        "S5"
      ],
      "relevance": 50,
      "description": "Defines the length of time that an animation takes to complete one cycle.",
      "restrictions": [
        "time"
      ]
    },
    {
      "name": "-webkit-animation-fill-mode",
      "browsers": [
        "C",
        "S5"
      ],
      "values": [
        {
          "name": "backwards",
          "description": "The beginning property value (as defined in the first @keyframes at-rule) is applied before the animation is displayed, during the period defined by 'animation-delay'."
        },
        {
          "name": "both",
          "description": "Both forwards and backwards fill modes are applied."
        },
        {
          "name": "forwards",
          "description": "The final property value (as defined in the last @keyframes at-rule) is maintained after the animation completes."
        },
        {
          "name": "none",
          "description": "There is no change to the property value between the time the animation is applied and the time the animation begins playing or after the animation completes."
        }
      ],
      "relevance": 50,
      "description": "Defines what values are applied by the animation outside the time it is executing.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-webkit-animation-iteration-count",
      "browsers": [
        "C",
        "S5"
      ],
      "values": [
        {
          "name": "infinite",
          "description": "Causes the animation to repeat forever."
        }
      ],
      "relevance": 50,
      "description": "Defines the number of times an animation cycle is played. The default value is one, meaning the animation will play from beginning to end once.",
      "restrictions": [
        "number",
        "enum"
      ]
    },
    {
      "name": "-webkit-animation-name",
      "browsers": [
        "C",
        "S5"
      ],
      "values": [
        {
          "name": "none",
          "description": "No animation is performed"
        }
      ],
      "relevance": 50,
      "description": "Defines a list of animations that apply. Each name is used to select the keyframe at-rule that provides the property values for the animation.",
      "restrictions": [
        "identifier",
        "enum"
      ]
    },
    {
      "name": "-webkit-animation-play-state",
      "browsers": [
        "C",
        "S5"
      ],
      "values": [
        {
          "name": "paused",
          "description": "A running animation will be paused."
        },
        {
          "name": "running",
          "description": "Resume playback of a paused animation."
        }
      ],
      "relevance": 50,
      "description": "Defines whether the animation is running or paused.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-webkit-animation-timing-function",
      "browsers": [
        "C",
        "S5"
      ],
      "relevance": 50,
      "description": "Describes how the animation will progress over one cycle of its duration. See the 'transition-timing-function'.",
      "restrictions": [
        "timing-function"
      ]
    },
    {
      "name": "-webkit-appearance",
      "browsers": [
        "C",
        "S3"
      ],
      "values": [
        {
          "name": "button"
        },
        {
          "name": "button-bevel"
        },
        {
          "name": "caps-lock-indicator"
        },
        {
          "name": "caret"
        },
        {
          "name": "checkbox"
        },
        {
          "name": "default-button"
        },
        {
          "name": "listbox"
        },
        {
          "name": "listitem"
        },
        {
          "name": "media-fullscreen-button"
        },
        {
          "name": "media-mute-button"
        },
        {
          "name": "media-play-button"
        },
        {
          "name": "media-seek-back-button"
        },
        {
          "name": "media-seek-forward-button"
        },
        {
          "name": "media-slider"
        },
        {
          "name": "media-sliderthumb"
        },
        {
          "name": "menulist"
        },
        {
          "name": "menulist-button"
        },
        {
          "name": "menulist-text"
        },
        {
          "name": "menulist-textfield"
        },
        {
          "name": "none"
        },
        {
          "name": "push-button"
        },
        {
          "name": "radio"
        },
        {
          "name": "scrollbarbutton-down"
        },
        {
          "name": "scrollbarbutton-left"
        },
        {
          "name": "scrollbarbutton-right"
        },
        {
          "name": "scrollbarbutton-up"
        },
        {
          "name": "scrollbargripper-horizontal"
        },
        {
          "name": "scrollbargripper-vertical"
        },
        {
          "name": "scrollbarthumb-horizontal"
        },
        {
          "name": "scrollbarthumb-vertical"
        },
        {
          "name": "scrollbartrack-horizontal"
        },
        {
          "name": "scrollbartrack-vertical"
        },
        {
          "name": "searchfield"
        },
        {
          "name": "searchfield-cancel-button"
        },
        {
          "name": "searchfield-decoration"
        },
        {
          "name": "searchfield-results-button"
        },
        {
          "name": "searchfield-results-decoration"
        },
        {
          "name": "slider-horizontal"
        },
        {
          "name": "sliderthumb-horizontal"
        },
        {
          "name": "sliderthumb-vertical"
        },
        {
          "name": "slider-vertical"
        },
        {
          "name": "square-button"
        },
        {
          "name": "textarea"
        },
        {
          "name": "textfield"
        }
      ],
      "status": "nonstandard",
      "syntax": "none | button | button-bevel | caret | checkbox | default-button | inner-spin-button | listbox | listitem | media-controls-background | media-controls-fullscreen-background | media-current-time-display | media-enter-fullscreen-button | media-exit-fullscreen-button | media-fullscreen-button | media-mute-button | media-overlay-play-button | media-play-button | media-seek-back-button | media-seek-forward-button | media-slider | media-sliderthumb | media-time-remaining-display | media-toggle-closed-captions-button | media-volume-slider | media-volume-slider-container | media-volume-sliderthumb | menulist | menulist-button | menulist-text | menulist-textfield | meter | progress-bar | progress-bar-value | push-button | radio | searchfield | searchfield-cancel-button | searchfield-decoration | searchfield-results-button | searchfield-results-decoration | slider-horizontal | slider-vertical | sliderthumb-horizontal | sliderthumb-vertical | square-button | textarea | textfield | -apple-pay-button",
      "relevance": 0,
      "description": "Changes the appearance of buttons and other controls to resemble native controls.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-webkit-backdrop-filter",
      "browsers": [
        "S9"
      ],
      "values": [
        {
          "name": "none",
          "description": "No filter effects are applied."
        },
        {
          "name": "blur()",
          "description": "Applies a Gaussian blur to the input image."
        },
        {
          "name": "brightness()",
          "description": "Applies a linear multiplier to input image, making it appear more or less bright."
        },
        {
          "name": "contrast()",
          "description": "Adjusts the contrast of the input."
        },
        {
          "name": "drop-shadow()",
          "description": "Applies a drop shadow effect to the input image."
        },
        {
          "name": "grayscale()",
          "description": "Converts the input image to grayscale."
        },
        {
          "name": "hue-rotate()",
          "description": "Applies a hue rotation on the input image. "
        },
        {
          "name": "invert()",
          "description": "Inverts the samples in the input image."
        },
        {
          "name": "opacity()",
          "description": "Applies transparency to the samples in the input image."
        },
        {
          "name": "saturate()",
          "description": "Saturates the input image."
        },
        {
          "name": "sepia()",
          "description": "Converts the input image to sepia."
        },
        {
          "name": "url()",
          "description": "A filter reference to a <filter> element."
        }
      ],
      "relevance": 50,
      "description": "Applies a filter effect where the first filter in the list takes the element's background image as the input image.",
      "restrictions": [
        "enum",
        "url"
      ]
    },
    {
      "name": "-webkit-backface-visibility",
      "browsers": [
        "C",
        "S5"
      ],
      "values": [
        {
          "name": "hidden"
        },
        {
          "name": "visible"
        }
      ],
      "relevance": 50,
      "description": "Determines whether or not the 'back' side of a transformed element is visible when facing the viewer. With an identity transform, the front side of an element faces the viewer.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-webkit-background-clip",
      "browsers": [
        "C",
        "S3"
      ],
      "relevance": 50,
      "description": "Determines the background painting area.",
      "restrictions": [
        "box"
      ]
    },
    {
      "name": "-webkit-background-composite",
      "browsers": [
        "C",
        "S3"
      ],
      "values": [
        {
          "name": "border"
        },
        {
          "name": "padding"
        }
      ],
      "relevance": 50,
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-webkit-background-origin",
      "browsers": [
        "C",
        "S3"
      ],
      "relevance": 50,
      "description": "For elements rendered as a single box, specifies the background positioning area. For elements rendered as multiple boxes (e.g., inline boxes on several lines, boxes on several pages) specifies which boxes 'box-decoration-break' operates on to determine the background positioning area(s).",
      "restrictions": [
        "box"
      ]
    },
    {
      "name": "-webkit-border-image",
      "browsers": [
        "C",
        "S5"
      ],
      "values": [
        {
          "name": "auto",
          "description": "If 'auto' is specified then the border image width is the intrinsic width or height (whichever is applicable) of the corresponding image slice. If the image does not have the required intrinsic dimension then the corresponding border-width is used instead."
        },
        {
          "name": "fill",
          "description": "Causes the middle part of the border-image to be preserved."
        },
        {
          "name": "none"
        },
        {
          "name": "repeat",
          "description": "The image is tiled (repeated) to fill the area."
        },
        {
          "name": "round",
          "description": "The image is tiled (repeated) to fill the area. If it does not fill the area with a whole number of tiles, the image is rescaled so that it does."
        },
        {
          "name": "space",
          "description": "The image is tiled (repeated) to fill the area. If it does not fill the area with a whole number of tiles, the extra space is distributed around the tiles."
        },
        {
          "name": "stretch",
          "description": "The image is stretched to fill the area."
        },
        {
          "name": "url()"
        }
      ],
      "relevance": 50,
      "description": "Shorthand property for setting 'border-image-source', 'border-image-slice', 'border-image-width', 'border-image-outset' and 'border-image-repeat'. Omitted values are set to their initial values.",
      "restrictions": [
        "length",
        "percentage",
        "number",
        "url",
        "enum"
      ]
    },
    {
      "name": "-webkit-box-align",
      "browsers": [
        "C",
        "S3"
      ],
      "values": [
        {
          "name": "baseline",
          "description": "If this box orientation is inline-axis or horizontal, all children are placed with their baselines aligned, and extra space placed before or after as necessary. For block flows, the baseline of the first non-empty line box located within the element is used. For tables, the baseline of the first cell is used."
        },
        {
          "name": "center",
          "description": "Any extra space is divided evenly, with half placed above the child and the other half placed after the child."
        },
        {
          "name": "end",
          "description": "For normal direction boxes, the bottom edge of each child is placed along the bottom of the box. Extra space is placed above the element. For reverse direction boxes, the top edge of each child is placed along the top of the box. Extra space is placed below the element."
        },
        {
          "name": "start",
          "description": "For normal direction boxes, the top edge of each child is placed along the top of the box. Extra space is placed below the element. For reverse direction boxes, the bottom edge of each child is placed along the bottom of the box. Extra space is placed above the element."
        },
        {
          "name": "stretch",
          "description": "The height of each child is adjusted to that of the containing block."
        }
      ],
      "relevance": 50,
      "description": "Specifies the alignment of nested elements within an outer flexible box element.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-webkit-box-direction",
      "browsers": [
        "C",
        "S3"
      ],
      "values": [
        {
          "name": "normal",
          "description": "A box with a computed value of horizontal for box-orient displays its children from left to right. A box with a computed value of vertical displays its children from top to bottom."
        },
        {
          "name": "reverse",
          "description": "A box with a computed value of horizontal for box-orient displays its children from right to left. A box with a computed value of vertical displays its children from bottom to top."
        }
      ],
      "relevance": 50,
      "description": "In webkit applications, -webkit-box-direction specifies whether a box lays out its contents normally (from the top or left edge), or in reverse (from the bottom or right edge).",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-webkit-box-flex",
      "browsers": [
        "C",
        "S3"
      ],
      "relevance": 50,
      "description": "Specifies an element's flexibility.",
      "restrictions": [
        "number"
      ]
    },
    {
      "name": "-webkit-box-flex-group",
      "browsers": [
        "C",
        "S3"
      ],
      "relevance": 50,
      "description": "Flexible elements can be assigned to flex groups using the 'box-flex-group' property.",
      "restrictions": [
        "integer"
      ]
    },
    {
      "name": "-webkit-box-ordinal-group",
      "browsers": [
        "C",
        "S3"
      ],
      "relevance": 50,
      "description": "Indicates the ordinal group the element belongs to. Elements with a lower ordinal group are displayed before those with a higher ordinal group.",
      "restrictions": [
        "integer"
      ]
    },
    {
      "name": "-webkit-box-orient",
      "browsers": [
        "C",
        "S3"
      ],
      "values": [
        {
          "name": "block-axis",
          "description": "Elements are oriented along the box's axis."
        },
        {
          "name": "horizontal",
          "description": "The box displays its children from left to right in a horizontal line."
        },
        {
          "name": "inline-axis",
          "description": "Elements are oriented vertically."
        },
        {
          "name": "vertical",
          "description": "The box displays its children from stacked from top to bottom vertically."
        }
      ],
      "relevance": 50,
      "description": "In webkit applications, -webkit-box-orient specifies whether a box lays out its contents horizontally or vertically.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-webkit-box-pack",
      "browsers": [
        "C",
        "S3"
      ],
      "values": [
        {
          "name": "center",
          "description": "The extra space is divided evenly, with half placed before the first child and the other half placed after the last child."
        },
        {
          "name": "end",
          "description": "For normal direction boxes, the right edge of the last child is placed at the right side, with all extra space placed before the first child. For reverse direction boxes, the left edge of the first child is placed at the left side, with all extra space placed after the last child."
        },
        {
          "name": "justify",
          "description": "The space is divided evenly in-between each child, with none of the extra space placed before the first child or after the last child. If there is only one child, treat the pack value as if it were start."
        },
        {
          "name": "start",
          "description": "For normal direction boxes, the left edge of the first child is placed at the left side, with all extra space placed after the last child. For reverse direction boxes, the right edge of the last child is placed at the right side, with all extra space placed before the first child."
        }
      ],
      "relevance": 50,
      "description": "Specifies alignment of child elements within the current element in the direction of orientation.",
      "restrictions": [
        "enum"
      ]
    },
    {
      "name": "-webkit-box-reflect",
      "browsers": [
        "E79",
        "S4",
        "C4",
        "O15"
      ],
      "values": [
        {
          "name": "above",
          "description": "The reflection appears above the border box."
        },
        {
          "name": "below",
          "description": "The reflection appears below the border box."
        },
        {
          "name": "left",
          "description": "The reflection appears to the left of the border box."
        },
        {
          "name": "right",
          "description": "The reflection 