import { h } from "@stencil/core";
export class Autocomplete {
    constructor() {
        this.activeIndex = -1;
        this.data = [];
        this.active = false;
        /**
         * The text is displayed by the form field for users
         */
        this.text = "";
        /**
         * The actual value of the form field
         */
        this.value = "";
        /**
         * The placeholder for the input field
         */
        this.placeholder = "";
        /**
         * Enable/Disable the input field
         */
        this.disabled = false;
        /**
         * The minimum input size for generating suggestions
         */
        this.minInput = 0;
        /**
         * The maximally shown suggestions in the list
         */
        this.maxSuggestions = 5;
        /**
         * Form validation: is the form input required
         */
        this.required = false;
        /**
         * id of the input field
         */
        this.inputId = "";
        /**
         * The class names, which should be set on the rendered html elements
         */
        this.cssClasses = {
            wrapper: "",
            input: "",
            suggestions: "suggestions",
            suggestion: "suggestion",
            active: "active"
        };
    }
    /**
     * Returns the `value` of the selected item
     */
    async getValue() {
        return this.value;
    }
    /**
     * Returns the `text` of the selected item
     */
    async getText() {
        return this.text;
    }
    /**
     * Clears the form field (suggestions and selection)
     */
    async clear() {
        this.handleClose();
    }
    handleKeyDown(keyCode) {
        if (keyCode == 40 || keyCode == 38) { // up/down arrows
            event.preventDefault();
            this.handleActivation(keyCode == 40);
        }
        else if (keyCode == 13 || keyCode == 9) { // enter/tab
            event.preventDefault();
            this.handleSelection(this.activeIndex);
        }
        else if (keyCode == 27) { // esc
            this.handleClose();
        }
    }
    handleKeyUp(keyCode, text) {
        if ([40, 38, 13, 9, 27].indexOf(keyCode) < 0) {
            this.clearSelection(true);
            this.prepareSuggestions(text);
        }
        this.active = true;
        this.text = text;
    }
    handleFocus(e) {
        e.preventDefault();
        this.active = true;
    }
    handleBlur(e) {
        e.preventDefault();
        setTimeout(() => {
            if (this.active) {
                if (this.value) {
                    this.clearData();
                }
                else {
                    this.handleClose();
                }
            }
        }, 250);
    }
    handleClose() {
        this.clearSelection();
        this.clearData();
    }
    handleActivation(next = true) {
        if (this.data.length > 0) {
            if (next && (this.activeIndex + 1) < this.data.length) {
                this.activeIndex += 1;
            }
            else if (next) {
                this.activeIndex = 0;
            }
            else if (!next && (this.activeIndex) > 0) {
                this.activeIndex -= 1;
            }
            else if (!next) {
                this.activeIndex = this.data.length - 1;
            }
        }
    }
    handleSelection(index) {
        if (index >= 0 && index < this.data.length) {
            this.text = this.data[index].text;
            this.value = this.data[index].value;
            this.selected.emit(this.data[index]);
            this.clearData();
        }
    }
    clearData() {
        this.data = [];
        this.activeIndex = -1;
        this.active = false;
    }
    clearSelection(clearOnlyValue = false) {
        if (this.value != "") {
            this.unselected.emit({
                text: this.text,
                value: this.value
            });
            this.value = "";
        }
        if (!clearOnlyValue) {
            this.text = "";
        }
    }
    async prepareSuggestions(text) {
        if (this.suggestionGenerator && text.length >= this.minInput) {
            let suggestions = await this.suggestionGenerator(text);
            suggestions.splice(this.maxSuggestions);
            this.data = suggestions;
        }
        else {
            this.data = [];
        }
    }
    render() {
        return (h("div", { class: this.cssClasses.wrapper },
            h("input", { value: this.text, class: this.cssClasses.input, onKeyDown: (e) => this.handleKeyDown(e.keyCode), onKeyUp: (e) => this.handleKeyUp(e.keyCode, e.target['value']), onBlur: (e) => { this.handleBlur(e); }, onFocus: (e) => { this.handleFocus(e); }, type: "text", id: this.inputId, required: this.required, autocomplete: "off", disabled: this.disabled, placeholder: this.placeholder }),
            this.data && this.data.length > 0
                ? h("div", { class: this.cssClasses.suggestions }, this.data.map((suggestion, index) => {
                    return h("button", { onClick: () => this.handleSelection(index), type: "button", class: this.cssClasses.suggestion + (this.activeIndex == index ? (" " + this.cssClasses.active) : ""), "data-value": suggestion.value }, suggestion.suggestion ? suggestion.suggestion : suggestion.text);
                }))
                : ""));
    }
    static get is() { return "web-complete"; }
    static get properties() { return {
        "text": {
            "type": "string",
            "mutable": true,
            "complexType": {
                "original": "string",
                "resolved": "string",
                "references": {}
            },
            "required": false,
            "optional": false,
            "docs": {
                "tags": [],
                "text": "The text is displayed by the form field for users"
            },
            "attribute": "text",
            "reflect": false,
            "defaultValue": "\"\""
        },
        "value": {
            "type": "string",
            "mutable": true,
            "complexType": {
                "original": "string",
                "resolved": "string",
                "references": {}
            },
            "required": false,
            "optional": false,
            "docs": {
                "tags": [],
                "text": "The actual value of the form field"
            },
            "attribute": "value",
            "reflect": false,
            "defaultValue": "\"\""
        },
        "placeholder": {
            "type": "string",
            "mutable": false,
            "complexType": {
                "original": "string",
                "resolved": "string",
                "references": {}
            },
            "required": false,
            "optional": false,
            "docs": {
                "tags": [],
                "text": "The placeholder for the input field"
            },
            "attribute": "placeholder",
            "reflect": false,
            "defaultValue": "\"\""
        },
        "disabled": {
            "type": "boolean",
            "mutable": false,
            "complexType": {
                "original": "boolean",
                "resolved": "boolean",
                "references": {}
            },
            "required": false,
            "optional": false,
            "docs": {
                "tags": [],
                "text": "Enable/Disable the input field"
            },
            "attribute": "disabled",
            "reflect": false,
            "defaultValue": "false"
        },
        "minInput": {
            "type": "number",
            "mutable": false,
            "complexType": {
                "original": "number",
                "resolved": "number",
                "references": {}
            },
            "required": false,
            "optional": false,
            "docs": {
                "tags": [],
                "text": "The minimum input size for generating suggestions"
            },
            "attribute": "min-input",
            "reflect": false,
            "defaultValue": "0"
        },
        "maxSuggestions": {
            "type": "number",
            "mutable": false,
            "complexType": {
                "original": "number",
                "resolved": "number",
                "references": {}
            },
            "required": false,
            "optional": false,
            "docs": {
                "tags": [],
                "text": "The maximally shown suggestions in the list"
            },
            "attribute": "max-suggestions",
            "reflect": false,
            "defaultValue": "5"
        },
        "required": {
            "type": "boolean",
            "mutable": false,
            "complexType": {
                "original": "boolean",
                "resolved": "boolean",
                "references": {}
            },
            "required": false,
            "optional": false,
            "docs": {
                "tags": [],
                "text": "Form validation: is the form input required"
            },
            "attribute": "required",
            "reflect": false,
            "defaultValue": "false"
        },
        "inputId": {
            "type": "string",
            "mutable": false,
            "complexType": {
                "original": "string",
                "resolved": "string",
                "references": {}
            },
            "required": false,
            "optional": false,
            "docs": {
                "tags": [],
                "text": "id of the input field"
            },
            "attribute": "input-id",
            "reflect": false,
            "defaultValue": "\"\""
        },
        "suggestionGenerator": {
            "type": "unknown",
            "mutable": false,
            "complexType": {
                "original": "(text:string) => Promise<Array<{text:string, value:string, suggestion?:string}>>",
                "resolved": "(text: string) => Promise<{ text: string; value: string; suggestion?: string; }[]>",
                "references": {
                    "Promise": {
                        "location": "global"
                    },
                    "Array": {
                        "location": "global"
                    }
                }
            },
            "required": false,
            "optional": false,
            "docs": {
                "tags": [],
                "text": "Async suggestion generator:\r\n`text` is the displayed for users in the form after selection (if no `suggesion` also as suggesion)\r\n`value` is the actual value of the form field\r\noptional `suggesion` if the autocomplete suggestion item should be formatted differently than `text`"
            }
        },
        "cssClasses": {
            "type": "unknown",
            "mutable": false,
            "complexType": {
                "original": "{ wrapper: string; input: string; suggestions: string; suggestion: string; active: string; }",
                "resolved": "{ wrapper: string; input: string; suggestions: string; suggestion: string; active: string; }",
                "references": {}
            },
            "required": false,
            "optional": false,
            "docs": {
                "tags": [],
                "text": "The class names, which should be set on the rendered html elements"
            },
            "defaultValue": "{\r\n    wrapper: \"\",\r\n    input: \"\",\r\n    suggestions: \"suggestions\",\r\n    suggestion: \"suggestion\",\r\n    active: \"active\"\r\n  }"
        }
    }; }
    static get states() { return {
        "activeIndex": {},
        "data": {},
        "active": {}
    }; }
    static get events() { return [{
            "method": "selected",
            "name": "selected",
            "bubbles": true,
            "cancelable": true,
            "composed": true,
            "docs": {
                "tags": [],
                "text": "Emitted when an item from suggestions was selected"
            },
            "complexType": {
                "original": "any",
                "resolved": "any",
                "references": {}
            }
        }, {
            "method": "unselected",
            "name": "unselected",
            "bubbles": true,
            "cancelable": true,
            "composed": true,
            "docs": {
                "tags": [],
                "text": "Emitted when item was cleared/unselected"
            },
            "complexType": {
                "original": "any",
                "resolved": "any",
                "references": {}
            }
        }]; }
    static get methods() { return {
        "getValue": {
            "complexType": {
                "signature": "() => Promise<string>",
                "parameters": [],
                "references": {
                    "Promise": {
                        "location": "global"
                    }
                },
                "return": "Promise<string>"
            },
            "docs": {
                "text": "Returns the `value` of the selected item",
                "tags": []
            }
        },
        "getText": {
            "complexType": {
                "signature": "() => Promise<string>",
                "parameters": [],
                "references": {
                    "Promise": {
                        "location": "global"
                    }
                },
                "return": "Promise<string>"
            },
            "docs": {
                "text": "Returns the `text` of the selected item",
                "tags": []
            }
        },
        "clear": {
            "complexType": {
                "signature": "() => Promise<void>",
                "parameters": [],
                "references": {
                    "Promise": {
                        "location": "global"
                    }
                },
                "return": "Promise<void>"
            },
            "docs": {
                "text": "Clears the form field (suggestions and selection)",
                "tags": []
            }
        }
    }; }
}
