import { IFCBUILDING, IFCBUILDINGSTOREY, IFCUNITASSIGNMENT } from 'web-ifc';
export class StoreyManager {
    constructor() {
        this.list = [];
        this.unitsFactor = {
            MILLI: 0.001,
            CENTI: 0.01,
            DECI: 0.1
        };
        this.loaderError = 'Loader must be defined!';
    }
    dispose() {
        this.list = null;
        this.unitsFactor = null;
    }
    async getAbsoluteElevation(modelID) {
        if (!this.loader)
            throw new Error(this.loaderError);
        await this.getCurrentStoreys(modelID);
        const unitsScale = await this.getUnitsFactor(modelID);
        const siteCoords = await this.getSiteCoords(modelID);
        const transformHeight = await this.getTransformHeight(modelID);
        const storeys = this.list[modelID];
        const result = {};
        for (let i = 0; i < storeys.length; i++) {
            const storey = storeys[i];
            const baseHeight = storey.Elevation.value;
            const name = this.getStoreyName(storey);
            result[name] = (baseHeight + siteCoords[2]) * unitsScale + transformHeight;
        }
        return result;
    }
    async getCurrentStoreys(modelID) {
        if (!this.list[modelID]) {
            this.list[modelID] = await this.loader.ifcManager.getAllItemsOfType(modelID, IFCBUILDINGSTOREY, true);
        }
    }
    async getSiteCoords(modelID) {
        try {
            const building = await this.getBuilding(modelID);
            const sitePlace = building.ObjectPlacement.PlacementRelTo.RelativePlacement.Location;
            return sitePlace.Coordinates.map((coord) => coord.value);
        }
        catch (e) {
            return [0, 0, 0];
        }
    }
    async getBuilding(modelID) {
        const allBuildingsIDs = await this.loader.ifcManager.getAllItemsOfType(modelID, IFCBUILDING, false);
        const buildingID = allBuildingsIDs[0];
        return this.loader.ifcManager.getItemProperties(modelID, buildingID, true);
    }
    async getTransformHeight(modelID) {
        const transformMatrix = await this.loader.ifcManager.ifcAPI.GetCoordinationMatrix(modelID);
        return transformMatrix[13];
    }
    getStoreyName(storey) {
        if (storey.Name)
            return storey.Name.value;
        if (storey.LongName)
            return storey.LongName.value;
        return storey.GlobalId;
    }
    // TODO: This assumes the first unit is the length, which is true in most cases
    // Might need to fix this in the future
    async getUnitsFactor(modelID) {
        var _a;
        const allUnitsIDs = await this.loader.ifcManager.getAllItemsOfType(modelID, IFCUNITASSIGNMENT, false);
        const unitsID = allUnitsIDs[0];
        const unitsProps = await this.loader.ifcManager.getItemProperties(modelID, unitsID);
        const lengthUnitID = unitsProps.Units[0].value;
        const lengthUnit = await this.loader.ifcManager.getItemProperties(modelID, lengthUnitID);
        const prefix = (_a = lengthUnit.Prefix) === null || _a === void 0 ? void 0 : _a.value;
        return this.unitsFactor[prefix] || 1;
    }
}
//# sourceMappingURL=storey-manager.js.map