///<reference path="qunit.d.ts"/>
///<reference path="../src/layout.ts"/>
///<reference path="../src/layout3d.ts"/>
QUnit.module("Headless API");
test('strongly connected components', function () {
    var la = {
        getSourceIndex: function (_a) {
            var source = _a[0], target = _a[1];
            return source;
        },
        getTargetIndex: function (_a) {
            var source = _a[0], target = _a[1];
            return target;
        }
    };
    var links = [[0, 1]];
    var components = cola.stronglyConnectedComponents(2, links, la);
    equal(components.length, 2);
    links = [[0, 1], [1, 2], [2, 0]];
    components = cola.stronglyConnectedComponents(3, links, la);
    equal(components.length, 1);
    links = [[0, 1], [1, 2], [2, 0], [2, 3], [3, 4], [4, 5], [5, 3]];
    components = cola.stronglyConnectedComponents(6, links, la);
    equal(components.length, 2);
    links = [[0, 1], [1, 2], [2, 0], [2, 3], [3, 4], [4, 2]];
    components = cola.stronglyConnectedComponents(5, links, la);
    equal(components.length, 1);
});
test("Basic headless layout", function () {
    // layout a triangular graph
    // should have no trouble finding an arrangement with all lengths close to the ideal length
    var layout = new cola.Layout()
        .links([
        { source: 0, target: 1 },
        { source: 1, target: 2 },
        { source: 2, target: 0 }])
        .start(10);
    // that's it!
    var vs = layout.nodes();
    equal(vs.length, 3, 'node array created');
    ok(layout.alpha() <= layout.convergenceThreshold(), 'converged to alpha=' + layout.alpha());
    var checkLengths = function (idealLength) {
        return layout.links().forEach(function (e) {
            var dx = e.source.x - e.target.x, dy = e.source.y - e.target.y;
            var length = Math.sqrt(dx * dx + dy * dy);
            ok(Math.abs(length - idealLength) < 0.01, 'correct link length: ' + length);
        });
    };
    checkLengths(layout.linkDistance());
    // rerun layout with a new ideal link length
    layout.linkDistance(10).start(10);
    checkLengths(10);
});
test("Layout events", function () {
    // layout a little star graph with listeners counting the different events
    var starts = 0, ticks = 0, ends = 0;
    var layout = new cola.Layout()
        .links([
        { source: 0, target: 1 },
        { source: 1, target: 2 },
        { source: 1, target: 3 }])
        .on(cola.EventType.start, function (e) { return starts++; })
        .on(cola.EventType.tick, function (e) { return ticks++; })
        .on(cola.EventType.end, function (e) { return ends++; })
        .start();
    ok(layout.alpha() <= layout.convergenceThreshold(), 'converged to alpha=' + layout.alpha());
    equal(starts, 1, 'started once');
    ok(ticks >= 1 && ticks < 50, "ticked " + ticks + " times");
    equal(ends, 1, 'ended once');
});
QUnit.module("3D Layout");
test("single link", function () {
    // single link with non-zero coords only in z-axis.
    // should relax to ideal length, nodes moving only in z-axis
    var nodes = [new cola.Node3D(0, 0, -1), new cola.Node3D(0, 0, 1)];
    var links = [new cola.Link3D(0, 1)];
    var desiredLength = 10;
    var layout = new cola.Layout3D(nodes, links, desiredLength).start();
    var linkLength = layout.linkLength(links[0]);
    nodes.forEach(function (_a) {
        var x = _a.x, y = _a.y;
        return ok(Math.abs(x) < 1e-5 && Math.abs(y) < 1e-5);
    });
    ok(Math.abs(linkLength - desiredLength) < 1e-4, "length = " + linkLength);
    // test per-link desiredLength:
    var smallerLength = 5;
    links[0].length = smallerLength;
    layout = new cola.Layout3D(nodes, links);
    layout.useJaccardLinkLengths = false;
    layout.start();
    linkLength = layout.linkLength(links[0]);
    ok(Math.abs(linkLength - smallerLength) < 1e-4, "length = " + linkLength);
});
function graph(links) {
    var N = links.reduce(function (n, _a) {
        var u = _a[0], v = _a[1];
        return Math.max(n, u, v);
    }, -1) + 1, nodes = new Array(N);
    for (var i = N; i--;)
        nodes[i] = new cola.Node3D;
    return { nodes: nodes, links: links.map(function (_a) {
            var u = _a[0], v = _a[1];
            return new cola.Link3D(u, v);
        }) };
}
test("Pyramid", function () {
    // k4 should relax to a 3D pyramid with all edges the same length
    var _a = graph([[0, 1], [1, 2], [2, 0], [0, 3], [1, 3], [2, 3]]), nodes = _a.nodes, links = _a.links;
    var layout = new cola.Layout3D(nodes, links, 10).start(0);
    var d = layout.descent;
    var x = layout.result;
    var s = d.computeStress();
    var takeDescentStep = function (alpha) {
        for (var i = 0; i < 3; ++i) {
            d.takeDescentStep(d.x[i], d.g[i], alpha);
        }
    };
    var reduceStress = function () {
        d.computeDerivatives(d.x);
        var alpha = 2 * d.computeStepSize(d.g);
        var f = 5;
        takeDescentStep(f * alpha);
        var sOver = d.computeStress();
        takeDescentStep(-f * alpha);
        f = 0.8;
        takeDescentStep(f * alpha);
        var sUnder = d.computeStress();
        takeDescentStep(-f * alpha);
        takeDescentStep(alpha);
        var s = d.computeStress();
        ok(sOver >= s, "  overshoot'=" + sOver + ", s=" + s);
        ok(sUnder >= s, "  undershoot'=" + sUnder + ", s=" + s);
        return [s, alpha];
    };
    for (var i = 0; i < 10; i++) {
        var _b = reduceStress(), s2 = _b[0], alpha = _b[1];
        ok(s2 <= s, "s'=" + s2 + ", s=" + s + ", alpha=" + alpha);
        s = s2;
    }
    layout = new cola.Layout3D(nodes, links, 10).start();
    var lengths = links.map(function (l) { return layout.linkLength(l); });
    lengths.forEach(function (l) { return ok(Math.abs(l - lengths[0]) < 1e-4, "length = " + l); });
});
test("Fixed nodes", function () {
    var _a = graph([[0, 1], [1, 2], [2, 3], [3, 4]]), nodes = _a.nodes, links = _a.links;
    var lock = function (i, x) {
        nodes[i].fixed = true;
        nodes[i].x = x;
    };
    var closeEnough = function (a, b, t) { return Math.abs(a - b) < t; };
    var layout = new cola.Layout3D(nodes, links, 10);
    var check = function () {
        // locked nodes should be at their initial position
        for (var i = 0; i < nodes.length; i++)
            if (nodes[i].fixed)
                cola.Layout3D.dims.forEach(function (d, j) {
                    return ok(closeEnough(layout.result[j][i], nodes[i][d], 1), "nodes[" + i + "] lock in " + d + "-axis at " + nodes[i][d] + ", actual=" + layout.result[j][i]);
                });
        var lengths = links.map(function (l) { return layout.linkLength(l); });
        var meanLength = lengths.reduce(function (s, l) { return s + l; }, 0) / lengths.length;
        // check all edge-lengths are within 5% of the mean
        lengths.forEach(function (l) { return ok(closeEnough(l, meanLength, meanLength / 20), "edge length = " + l); });
    };
    // nodes 0 and 4 will be locked at (-5,0,0) and (5,0,0) respectively
    // with these locks and ideal edge length at 10, unfixed nodes will arc around in a horse-shoe shape
    lock(0, -5);
    lock(4, 5);
    layout.start();
    check();
    // move the lock positions
    lock(0, -10);
    lock(4, 10);
    // run layout incrementally
    for (var i = 0; i < 100; i++)
        layout.tick();
    check();
});
//# sourceMappingURL=apitests.js.map