"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const loglevel = require("loglevel");
const path = require("path");
const utils_1 = require("./utils");
const log = loglevel.getLogger('webdriver-manager');
/**
 * Starts the selenium server standalone with browser drivers. Also handles
 * the SIGINT event when the server is stopped.
 * @param argv The argv from yargs.
 */
async function handler(argv) {
    log.setLevel(argv.log_level);
    const options = utils_1.convertArgs2Options(argv);
    if (options.server.runAsDetach) {
        await start(options);
        process.exit();
    }
    else {
        process.stdin.resume();
        process.on('SIGINT', () => {
            const optionsBinary = utils_1.addOptionsBinary(options);
            const seleniumServer = optionsBinary.server.binary;
            process.kill(seleniumServer.seleniumProcess.pid);
            process.exit(process.exitCode);
        });
        start(options).then(() => { });
    }
}
exports.handler = handler;
/**
 * Goes through all the option providers and creates a set of java options
 * to pass to java when starting the selenium server standalone.
 * @param options The constructed options.
 * @returns Promise starting the server with the resolved exit code.
 */
function start(options) {
    const optionsBinary = utils_1.addOptionsBinary(options);
    return startBinary(optionsBinary);
}
exports.start = start;
/**
 * Goes through all the option providers and creates a set of java options
 * to pass to java when starting the selenium server standalone.
 * @param optionsBinary The constructed options with binaries.
 * @returns Promise starting the server with the resolved exit code.
 */
function startBinary(optionsBinary) {
    const javaOpts = {};
    for (const browserDriver of optionsBinary.browserDrivers) {
        if (browserDriver.binary) {
            javaOpts[browserDriver.binary.seleniumFlag] =
                browserDriver.binary.getBinaryPath(browserDriver.version);
        }
    }
    if (optionsBinary.server) {
        if (optionsBinary.server.chromeLogs) {
            const chromeLogs = optionsBinary.server.chromeLogs.replace('"', '').replace('\'', '');
            javaOpts['-Dwebdriver.chrome.logfile'] = path.resolve(chromeLogs);
        }
        if (optionsBinary.server.edge) {
            const edge = optionsBinary.server.edge.replace('"', '').replace('\'', '');
            javaOpts['-Dwebdriver.edge.driver'] = path.resolve(edge);
        }
        if (optionsBinary.server.binary) {
            return optionsBinary.server.binary
                .startServer(javaOpts, optionsBinary.server.version);
        }
    }
    return Promise.reject('Could not start the server');
}
exports.startBinary = startBinary;
//# sourceMappingURL=start.js.map