"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chromedriver_1 = require("../provider/chromedriver");
const geckodriver_1 = require("../provider/geckodriver");
const iedriver_1 = require("../provider/iedriver");
const selenium_server_1 = require("../provider/selenium_server");
/**
 * Converts an options object into an options binary object.
 * @param options
 */
function addOptionsBinary(options) {
    if (!options) {
        return null;
    }
    const providerConfig = {
        ignoreSSL: options.ignoreSSL,
        outDir: options.outDir,
        proxy: options.proxy
    };
    const optionsBinary = options;
    if (optionsBinary.browserDrivers) {
        for (const browserDriver of optionsBinary.browserDrivers) {
            if (browserDriver.name === 'chromedriver') {
                browserDriver.binary = new chromedriver_1.ChromeDriver(providerConfig);
            }
            else if (browserDriver.name === 'geckodriver') {
                const geckoProviderConfig = providerConfig;
                geckoProviderConfig.oauthToken = optionsBinary.githubToken;
                browserDriver.binary = new geckodriver_1.GeckoDriver(geckoProviderConfig);
            }
            else if (browserDriver.name === 'iedriver') {
                browserDriver.binary = new iedriver_1.IEDriver(providerConfig);
            }
        }
    }
    if (optionsBinary.server) {
        const seleniumProviderConfig = providerConfig;
        seleniumProviderConfig.outDir = optionsBinary.outDir;
        seleniumProviderConfig.port = optionsBinary.server.port;
        seleniumProviderConfig.runAsDetach = optionsBinary.server.runAsDetach;
        seleniumProviderConfig.runAsNode = optionsBinary.server.runAsNode;
        optionsBinary.server.binary = new selenium_server_1.SeleniumServer(seleniumProviderConfig);
    }
    return optionsBinary;
}
exports.addOptionsBinary = addOptionsBinary;
/**
 * Create the options with all providers. Used for clean and status commands.
 * @param argv
 */
function convertArgs2AllOptions(argv) {
    let versionsChrome, versionsGecko, versionsIe, versionsStandalone = undefined;
    if (argv.versions) {
        versionsChrome = argv.versions.chrome;
        versionsGecko = argv.versions.gecko;
        versionsIe = argv.versions.ie;
        versionsStandalone = argv.versions.standalone;
    }
    return {
        browserDrivers: [
            { name: 'chromedriver', version: versionsChrome },
            { name: 'geckodriver', version: versionsGecko },
            { name: 'iedriver', version: versionsIe }
        ],
        server: {
            name: 'selenium',
            version: versionsStandalone,
            runAsNode: argv.standalone_node,
            runAsDetach: argv.detach,
            chromeLogs: argv.chrome_logs,
            edge: argv.edge,
        },
        ignoreSSL: argv.ignore_ssl,
        outDir: argv.out_dir,
        proxy: argv.proxy,
        githubToken: argv.github_token,
    };
}
exports.convertArgs2AllOptions = convertArgs2AllOptions;
/**
 * Create the options with providers depending on argv's. Used for update and
 * start commands.
 * @param argv
 */
function convertArgs2Options(argv) {
    const options = {
        browserDrivers: [],
        server: null,
        ignoreSSL: argv.ignore_ssl,
        outDir: argv.out_dir,
        proxy: argv.proxy,
        githubToken: argv.github_token,
    };
    let versionsChrome, versionsGecko, versionsIe, versionsStandalone = undefined;
    if (argv.versions) {
        versionsChrome = argv.versions.chrome;
        versionsGecko = argv.versions.gecko;
        versionsIe = argv.versions.ie;
        versionsStandalone = argv.versions.standalone;
    }
    if (argv.chrome) {
        options.browserDrivers.push({ name: 'chromedriver', version: versionsChrome });
    }
    if (argv.gecko) {
        options.browserDrivers.push({ name: 'geckodriver', version: versionsGecko });
    }
    if (argv.iedriver) {
        options.browserDrivers.push({ name: 'iedriver', version: versionsIe });
    }
    if (argv.standalone) {
        options.server = {};
        options.server.name = 'selenium';
        options.server.runAsNode = argv.standalone_node;
        options.server.runAsDetach = argv.detach;
        options.server.version = versionsStandalone;
        options.server.chromeLogs = argv.chrome_logs;
        options.server.edge = argv.edge;
    }
    return options;
}
exports.convertArgs2Options = convertArgs2Options;
//# sourceMappingURL=utils.js.map