"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const os = require("os");
const path = require("path");
const provider_1 = require("./provider");
const file_utils_1 = require("./utils/file_utils");
const github_json_1 = require("./utils/github_json");
const http_utils_1 = require("./utils/http_utils");
const version_list_1 = require("./utils/version_list");
class GeckoDriver {
    constructor(providerConfig) {
        this.cacheFileName = 'geckodriver.json';
        this.configFileName = 'geckodriver.config.json';
        this.ignoreSSL = false;
        this.osType = os.type();
        this.osArch = os.arch();
        this.outDir = provider_1.OUT_DIR;
        this.proxy = null;
        this.requestUrl = 'https://api.github.com/repos/mozilla/geckodriver/releases';
        this.seleniumFlag = '-Dwebdriver.gecko.driver';
        if (providerConfig) {
            if (providerConfig.cacheFileName) {
                this.cacheFileName = providerConfig.cacheFileName;
            }
            if (providerConfig.configFileName) {
                this.configFileName = providerConfig.configFileName;
            }
            this.ignoreSSL = providerConfig.ignoreSSL;
            if (providerConfig.osArch) {
                this.osArch = providerConfig.osArch;
            }
            if (providerConfig.osType) {
                this.osType = providerConfig.osType;
            }
            if (providerConfig.outDir) {
                this.outDir = providerConfig.outDir;
            }
            if (providerConfig.proxy) {
                this.proxy = providerConfig.proxy;
            }
            if (providerConfig.requestUrl) {
                this.requestUrl = providerConfig.requestUrl;
            }
            if (providerConfig.oauthToken) {
                this.oauthToken = providerConfig.oauthToken;
            }
        }
    }
    /**
     * Should update the cache and download, find the version to download,
     * then download that binary.
     * @param version Optional to provide the version number or latest.
     */
    async updateBinary(version) {
        await github_json_1.updateJson(this.requestUrl, {
            fileName: path.resolve(this.outDir, this.cacheFileName),
            ignoreSSL: this.ignoreSSL,
            proxy: this.proxy
        }, this.oauthToken);
        const versionList = github_json_1.convertJsonToVersionList(path.resolve(this.outDir, this.cacheFileName));
        const versionObj = version_list_1.getVersion(versionList, osHelper(this.osType, this.osArch), version);
        const geckoDriverUrl = versionObj.url;
        const geckoDriverCompressed = path.resolve(this.outDir, versionObj.name);
        // We should check the zip file size if it exists. The size will
        // be used to either make the request, or quit the request if the file
        // size matches.
        let fileSize = 0;
        try {
            fileSize = fs.statSync(geckoDriverCompressed).size;
        }
        catch (err) {
        }
        await http_utils_1.requestBinary(geckoDriverUrl, {
            fileName: geckoDriverCompressed,
            fileSize,
            ignoreSSL: this.ignoreSSL,
            proxy: this.proxy
        });
        // Uncompress tarball (for linux and mac) or unzip the file for Windows.
        // Rename all the files (a grand total of 1) and set the permissions.
        let fileList;
        if (this.osType === 'Windows_NT') {
            fileList = file_utils_1.zipFileList(geckoDriverCompressed);
        }
        else {
            fileList = await file_utils_1.tarFileList(geckoDriverCompressed);
        }
        const fileItem = path.resolve(this.outDir, fileList[0]);
        if (this.osType === 'Windows_NT') {
            file_utils_1.unzipFile(geckoDriverCompressed, this.outDir);
        }
        else {
            await file_utils_1.uncompressTarball(geckoDriverCompressed, this.outDir);
        }
        const renamedFileName = file_utils_1.renameFileWithVersion(fileItem, '_' + versionObj.version);
        file_utils_1.changeFilePermissions(renamedFileName, '0755', this.osType);
        file_utils_1.generateConfigFile(this.outDir, path.resolve(this.outDir, this.configFileName), matchBinaries(this.osType), renamedFileName);
        return Promise.resolve();
    }
    /**
     * Gets the binary file path.
     * @param version Optional to provide the version number or latest.
     */
    getBinaryPath(version) {
        try {
            const configFilePath = path.resolve(this.outDir, this.configFileName);
            return file_utils_1.getBinaryPathFromConfig(configFilePath, version);
        }
        catch (_) {
            return null;
        }
    }
    /**
     * Gets a comma delimited list of versions downloaded. Also has the "latest"
     * downloaded noted.
     */
    getStatus() {
        try {
            const configFilePath = path.resolve(this.outDir, this.configFileName);
            const configJson = JSON.parse(fs.readFileSync(configFilePath).toString());
            const versions = [];
            for (const binaryPath of configJson['all']) {
                let version = '';
                let regex = /.*geckodriver_(\d+.\d+.\d+.*)/g;
                if (this.osType === 'Windows_NT') {
                    regex = /.*geckodriver_(\d+.\d+.\d+.*).exe/g;
                }
                try {
                    const exec = regex.exec(binaryPath);
                    if (exec && exec[1]) {
                        version = exec[1];
                    }
                }
                catch (_) {
                }
                if (configJson['last'] === binaryPath) {
                    version += ' (latest)';
                }
                versions.push(version);
            }
            return versions.join(', ');
        }
        catch (_) {
            return null;
        }
    }
    /**
     * Get a line delimited list of files removed.
     */
    cleanFiles() {
        return file_utils_1.removeFiles(this.outDir, [/geckodriver.*/g]);
    }
}
exports.GeckoDriver = GeckoDriver;
/**
 * Helps translate the os type and arch to the download name associated
 * with composing the download link.
 * @param ostype The operating stystem type.
 * @param osarch The chip architecture.
 * @returns The download name associated with composing the download link.
 */
function osHelper(ostype, osarch) {
    if (ostype === 'Darwin') {
        return 'macos';
    }
    else if (ostype === 'Windows_NT') {
        if (osarch === 'x64') {
            return 'win64';
        }
        else if (osarch === 'x32') {
            return 'win32';
        }
    }
    else if (ostype === 'Linux') {
        if (osarch === 'x64') {
            return 'linux64';
        }
        else if (osarch === 'x32') {
            return 'linux32';
        }
    }
    return null;
}
exports.osHelper = osHelper;
/**
 * Matches the installed binaries depending on the operating system.
 * @param ostype The operating stystem type.
 */
function matchBinaries(ostype) {
    if (ostype === 'Darwin' || ostype === 'Linux') {
        return /geckodriver_\d+.\d+.\d+/g;
    }
    else if (ostype === 'Windows_NT') {
        return /geckodriver_\d+.\d+.\d+.exe/g;
    }
    return null;
}
exports.matchBinaries = matchBinaries;
//# sourceMappingURL=geckodriver.js.map