/// <reference types="node" />
import * as childProcess from 'child_process';
import { ProviderConfig, ProviderInterface } from './provider';
export interface SeleniumServerProviderConfig extends ProviderConfig {
    port?: number;
    runAsNode?: boolean;
    runAsDetach?: boolean;
}
export declare class SeleniumServer implements ProviderInterface {
    cacheFileName: string;
    configFileName: string;
    ignoreSSL: boolean;
    osType: string;
    osArch: string;
    outDir: string;
    port: number;
    proxy: string;
    requestUrl: string;
    seleniumProcess: childProcess.ChildProcess;
    runAsNode: boolean;
    runAsDetach: boolean;
    constructor(providerConfig?: SeleniumServerProviderConfig);
    /**
     * Should update the cache and download, find the version to download,
     * then download that binary.
     * @param version Optional to provide the version number or latest.
     */
    updateBinary(version?: string): Promise<void>;
    /**
     * Starts selenium standalone server and handles emitted exit events.
     * @param opts The options to pass to the jar file.
     * @param version The optional version of the selenium jar file.
     * @returns A promise so the server can run while awaiting its completion.
     */
    startServer(opts: {
        [key: string]: string;
    }, version?: string): Promise<number>;
    /**
     * Get the binary file path.
     * @param version Optional to provide the version number or the latest.
     */
    getBinaryPath(version?: string): string | null;
    /**
     * Get the selenium server start command (not including the java command)
     * @param opts The options to pass to the jar file.
     * @param version The optional version of the selenium jar file.
     * @returns The spawn arguments array.
     */
    getCmdStartServer(opts: {
        [key: string]: string;
    }, version?: string): string[];
    /**
     * Gets the java command either by the JAVA_HOME environment variable or
     * just the java command.
     */
    getJava(): string;
    /**
     * If we are running the selenium server role = node, send
     * the command to stop the server via http get request. Reference:
     * https://github.com/SeleniumHQ/selenium/issues/2852#issuecomment-268324091
     *
     * If we are not running as the selenium server role = node, kill the
     * process with pid.
     *
     * @param host The protocol and ip address, default http://127.0.0.1
     * @param port The port number, default 4444
     * @returns A promise of the http get request completing.
     */
    stopServer(host?: string, port?: number): Promise<void>;
    /**
     * Gets a comma delimited list of versions downloaded. Also has the "latest"
     * downloaded noted.
     */
    getStatus(): string | null;
    /**
     * Get a line delimited list of files removed.
     */
    cleanFiles(): string;
}
/**
 * Captures the version name which includes the semantic version and extra
 * metadata. So an example for 12.34/selenium-server-standalone-12.34.56.jar,
 * the version is 12.34.56. For metadata,
 * 12.34/selenium-server-standalone-12.34.56-beta.jar is 12.34.56-beta.
 * @param xmlKey The xml key including the partial url.
 */
export declare function versionParser(xmlKey: string): string;
/**
 * Captures the version name which includes the semantic version and extra
 * metadata. So an example for 12.34/selenium-server-standalone-12.34.56.jar,
 * the version is 12.34.56. For metadata,
 * 12.34/selenium-server-standalone-12.34.56-beta.jar is still 12.34.56.
 * @param xmlKey The xml key including the partial url.
 */
export declare function semanticVersionParser(xmlKey: string): string;
/**
 * Matches the installed binaries.
 */
export declare function matchBinaries(): RegExp | null;
