"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const AdmZip = require("adm-zip");
const fs = require("fs");
const path = require("path");
const tar = require("tar");
const xml2js = require("xml2js");
/**
 * Check to see if the modified timestamp is expired.
 * @param fileName THe xml filename.
 */
function isExpired(fileName) {
    try {
        const timestamp = new Date(fs.statSync(fileName).mtime).getTime();
        const size = fs.statSync(fileName).size;
        const now = Date.now();
        if (size > 0 && (now - (60 * 60 * 1000) < timestamp)) {
            return false;
        }
        else {
            return true;
        }
    }
    catch (err) {
        return true;
    }
}
exports.isExpired = isExpired;
/**
 * Reads the json file from file.
 * @param fileName The json filename to read.
 * @returns
 */
function readJson(fileName) {
    try {
        const contents = fs.readFileSync(fileName).toString();
        return JSON.parse(contents);
    }
    catch (err) {
        return null;
    }
}
exports.readJson = readJson;
/**
 * Reads the xml file.
 * @param fileName The xml filename to read.
 */
function readXml(fileName) {
    try {
        const contents = fs.readFileSync(fileName).toString();
        return convertXml2js(contents);
    }
    catch (err) {
        return null;
    }
}
exports.readXml = readXml;
/**
 * Convert the xml file to an object.
 * @param content The xml contents.
 */
function convertXml2js(content) {
    let retResult = null;
    xml2js.parseString(content, (err, result) => {
        retResult = result;
    });
    return retResult;
}
exports.convertXml2js = convertXml2js;
/**
 * Renames a file with a semantic version.
 * @param srcFileName The full path to the original file name.
 * @param versionNumber The semver number.
 * @returns The renamed file name.
 */
function renameFileWithVersion(srcFileName, versionNumber) {
    const dirName = path.dirname(srcFileName);
    const extName = path.extname(srcFileName);
    const baseName = path.basename(srcFileName, extName);
    const dstFileName = path.resolve(dirName, baseName + versionNumber + extName);
    fs.renameSync(srcFileName, dstFileName);
    return dstFileName;
}
exports.renameFileWithVersion = renameFileWithVersion;
/**
 * Gets a list of files in the zip file.
 * @param zipFileName The zip file.
 * @returns A list of files in the zip file.
 */
function zipFileList(zipFileName) {
    const fileList = [];
    const zip = new AdmZip(zipFileName);
    zip.getEntries().forEach(entry => {
        fileList.push(entry.name);
    });
    return fileList;
}
exports.zipFileList = zipFileList;
/**
 * Uncompress the zip file to a destination directory.
 * @param zipFileName The zip file.
 * @param dstDir The destination directory for the contents of the zip file.
 * @returns A list of uncompressed files.
 */
function unzipFile(zipFileName, dstDir) {
    const fileList = [];
    const zip = new AdmZip(zipFileName);
    zip.extractAllTo(dstDir, true);
    for (const fileItem of zipFileList(zipFileName)) {
        fileList.push(path.resolve(dstDir, fileItem));
    }
    return fileList;
}
exports.unzipFile = unzipFile;
/**
 * Gets a list of files in the tarball file.
 * @param tarball The tarball file.
 * @returns A lsit of files in the tarball file.
 */
function tarFileList(tarball) {
    const fileList = [];
    return tar
        .list({
        file: tarball,
        onentry: entry => {
            fileList.push(entry['path'].toString());
        }
    })
        .then(() => {
        return fileList;
    });
}
exports.tarFileList = tarFileList;
/**
 * Uncompress the tar file to a destination directory.
 * @param tarball The tarball file.
 * @param dstDir The destination directory for the contents of the zip file.
 * @returns A list of uncompressed files.
 */
async function uncompressTarball(tarball, dstDir) {
    try {
        fs.mkdirSync(path.resolve(dstDir));
    }
    catch (err) {
    }
    const fileList = await tarFileList(tarball);
    return tar.extract({ file: tarball }).then(() => {
        const dstFiles = [];
        for (const fileItem of fileList) {
            const dstFileName = path.resolve(dstDir, fileItem);
            fs.renameSync(path.resolve(fileItem), dstFileName);
            dstFiles.push(dstFileName);
        }
        return dstFiles;
    });
}
exports.uncompressTarball = uncompressTarball;
/**
 * Change the permissions for Linux and MacOS with chmod.
 * @param fileName The full path to the filename to change permissions.
 * @param mode The number to modify.
 * @param osType The OS type to decide if we need to change permissions on the
 *     file.
 */
function changeFilePermissions(fileName, mode, osType) {
    if (osType === 'Darwin' || osType === 'Linux') {
        fs.chmodSync(path.resolve(fileName), mode);
    }
}
exports.changeFilePermissions = changeFilePermissions;
/**
 * Writes a config file that matches the regex pattern.
 * @param outDir The output directory.
 * @param fileName The full path to the file name.
 * @param fileBinaryPathRegex The regExp to match files in the outDir.
 * @param lastFileBinaryPath The full path to the last binary file downloaded.
 */
function generateConfigFile(outDir, fileName, fileBinaryPathRegex, lastFileBinaryPath) {
    const configData = {};
    if (lastFileBinaryPath) {
        configData['last'] = lastFileBinaryPath;
    }
    configData['all'] = getMatchingFiles(outDir, fileBinaryPathRegex);
    fs.writeFileSync(fileName, JSON.stringify(configData));
}
exports.generateConfigFile = generateConfigFile;
/**
 * Gets matching files form the outDir and returns it as an array.
 * @param outDir The output directory.
 * @param fileBinaryPathRegex The regExp to match files in the outDir.
 */
function getMatchingFiles(outDir, fileBinaryPathRegex) {
    const existFiles = fs.readdirSync(outDir);
    const matchingFiles = [];
    for (const existFile of existFiles) {
        if (existFile.match(fileBinaryPathRegex)) {
            matchingFiles.push(path.resolve(outDir, existFile));
        }
    }
    return matchingFiles;
}
exports.getMatchingFiles = getMatchingFiles;
/**
 * Get the binary path from the configuration file. The configuration file
 * should be formatted as { 'last': string, 'all': string[] }. In the 'all'
 * array, we should match the 'version'. The version does not necessarily have
 * to be a valid semantic version.
 * @param cacheFilePath The cache file path.
 * @param version An optional version that is not necessarily semver.
 */
function getBinaryPathFromConfig(cacheFilePath, version) {
    const cacheJson = JSON.parse(fs.readFileSync(cacheFilePath).toString());
    let binaryPath = null;
    if (!version) {
        binaryPath = cacheJson['last'];
    }
    else {
        for (const cachePath of cacheJson['all']) {
            if (cachePath.match(version)) {
                binaryPath = cachePath;
            }
        }
    }
    return binaryPath;
}
exports.getBinaryPathFromConfig = getBinaryPathFromConfig;
/**
 * Removes the files that match the regular expressions and returns a string
 * of removed files.
 * @param outDir The output directory.
 * @param fileRegexes The regExp to match files to remove in the outDir.
 */
function removeFiles(outDir, fileRegexes) {
    try {
        const existFiles = fs.readdirSync(outDir);
        const removedFiles = [];
        for (const fileRegex of fileRegexes) {
            for (const existFile of existFiles) {
                if (existFile.match(fileRegex)) {
                    removedFiles.push(existFile);
                    fs.unlinkSync(path.resolve(outDir, existFile));
                }
            }
        }
        return (removedFiles.sort()).join('\n');
    }
    catch (_) {
        return null;
    }
}
exports.removeFiles = removeFiles;
//# sourceMappingURL=file_utils.js.map