import * as request from 'request';
/**
 * The request options that extend the request. This is not exported
 * in preference to build an HttpOptions object with extra metadata
 * and the http_utils methods will help build this object.
 */
export interface RequestOptionsValue extends request.OptionsWithUrl {
    proxy?: string;
    ignoreSSL?: boolean;
}
/**
 * A json object interface.
 */
export interface JsonObject {
    [key: string]: any;
}
/**
 * The http option interface to build the request.
 */
export interface HttpOptions {
    fileName?: string;
    fileSize?: number;
    headers?: {
        [key: string]: string | number | string[];
    };
    ignoreSSL?: boolean;
    proxy?: string;
}
/**
 * Initialize the request options.
 * @param requestUrl The request url.
 * @param httpOptions The http options for the request.
 */
export declare function initOptions(requestUrl: string, httpOptions: HttpOptions): RequestOptionsValue;
/**
 * Set ignore SSL option.
 * @param options The HTTP options
 * @param optIgnoreSSL The ignore SSL option.
 */
export declare function optionsSSL(options: RequestOptionsValue, optIgnoreSSL: boolean): RequestOptionsValue;
export declare function optionsProxy(options: RequestOptionsValue, requestUrl: string, optProxy: string): RequestOptionsValue;
/**
 * Resolves proxy based on values set.
 * @param requestUrl The url to download the file.
 * @param optProxy The proxy to connect to to download files.
 * @return Either undefined or the proxy.
 */
export declare function resolveProxy(requestUrl: string, optProxy: string): string;
/**
 * Builds a curl command for logging purposes.
 * @param requestOptions The request options.
 * @param fileName The file name path.
 * @returns The curl command.
 */
export declare function curlCommand(requestOptions: RequestOptionsValue, fileName?: string): string;
/**
 * Add a header to the request.
 * @param options The options to add a header.
 * @param name The key name of the header.
 * @param value The value of the header.
 * @returns The modified options object.
 */
export declare function addHeader(options: RequestOptionsValue, name: string, value: string | number | string[]): RequestOptionsValue;
/**
 * The request to download the binary.
 * @param binaryUrl The download url for the binary.
 * @param httpOptions The http options for the request.
 * @param isLogInfo Log info or debug
 */
export declare function requestBinary(binaryUrl: string, httpOptions: HttpOptions, isLogInfo?: boolean): Promise<boolean>;
/**
 * Request the body from the url and log the curl.
 * @param requestUrl The request url.
 * @param httpOptions The http options for the request.
 * @returns A promise string of the response body.
 */
export declare function requestBody(requestUrl: string, httpOptions: HttpOptions): Promise<string>;
