"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

var _asyncToGenerator2 = _interopRequireDefault(require("@babel/runtime/helpers/asyncToGenerator"));

var _objectSpread2 = _interopRequireDefault(require("@babel/runtime/helpers/objectSpread"));

const bfj = require('bfj');

const path = require('path');

const mkdir = require('mkdirp');

const _require = require('chalk'),
      bold = _require.bold;

const Logger = require('./Logger');

const viewer = require('./viewer');

class BundleAnalyzerPlugin {
  constructor(opts) {
    this.opts = (0, _objectSpread2.default)({
      analyzerMode: 'server',
      analyzerHost: '127.0.0.1',
      analyzerPort: 8888,
      reportFilename: 'report.html',
      defaultSizes: 'parsed',
      openAnalyzer: true,
      generateStatsFile: false,
      statsFilename: 'stats.json',
      statsOptions: null,
      excludeAssets: null,
      logLevel: 'info',
      // deprecated
      startAnalyzer: true
    }, opts);
    this.server = null;
    this.logger = new Logger(this.opts.logLevel);
  }

  apply(compiler) {
    this.compiler = compiler;

    const done = stats => {
      stats = stats.toJson(this.opts.statsOptions);
      const actions = [];

      if (this.opts.generateStatsFile) {
        actions.push(() => this.generateStatsFile(stats));
      } // Handling deprecated `startAnalyzer` flag


      if (this.opts.analyzerMode === 'server' && !this.opts.startAnalyzer) {
        this.opts.analyzerMode = 'disabled';
      }

      if (this.opts.analyzerMode === 'server') {
        actions.push(() => this.startAnalyzerServer(stats));
      } else if (this.opts.analyzerMode === 'static') {
        actions.push(() => this.generateStaticReport(stats));
      }

      if (actions.length) {
        // Making analyzer logs to be after all webpack logs in the console
        setImmediate(() => {
          actions.forEach(action => action());
        });
      }
    };

    if (compiler.hooks) {
      compiler.hooks.done.tap('webpack-bundle-analyzer', done);
    } else {
      compiler.plugin('done', done);
    }
  }

  generateStatsFile(stats) {
    var _this = this;

    return (0, _asyncToGenerator2.default)(function* () {
      const statsFilepath = path.resolve(_this.compiler.outputPath, _this.opts.statsFilename);
      mkdir.sync(path.dirname(statsFilepath));

      try {
        yield bfj.write(statsFilepath, stats, {
          space: 2,
          promises: 'ignore',
          buffers: 'ignore',
          maps: 'ignore',
          iterables: 'ignore',
          circular: 'ignore'
        });

        _this.logger.info(`${bold('Webpack Bundle Analyzer')} saved stats file to ${bold(statsFilepath)}`);
      } catch (error) {
        _this.logger.error(`${bold('Webpack Bundle Analyzer')} error saving stats file to ${bold(statsFilepath)}: ${error}`);
      }
    })();
  }

  startAnalyzerServer(stats) {
    var _this2 = this;

    return (0, _asyncToGenerator2.default)(function* () {
      if (_this2.server) {
        (yield _this2.server).updateChartData(stats);
      } else {
        _this2.server = viewer.startServer(stats, {
          openBrowser: _this2.opts.openAnalyzer,
          host: _this2.opts.analyzerHost,
          port: _this2.opts.analyzerPort,
          bundleDir: _this2.getBundleDirFromCompiler(),
          logger: _this2.logger,
          defaultSizes: _this2.opts.defaultSizes,
          excludeAssets: _this2.opts.excludeAssets
        });
      }
    })();
  }

  generateStaticReport(stats) {
    viewer.generateReport(stats, {
      openBrowser: this.opts.openAnalyzer,
      reportFilename: path.resolve(this.compiler.outputPath, this.opts.reportFilename),
      bundleDir: this.getBundleDirFromCompiler(),
      logger: this.logger,
      defaultSizes: this.opts.defaultSizes,
      excludeAssets: this.opts.excludeAssets
    });
  }

  getBundleDirFromCompiler() {
    return this.compiler.outputFileSystem.constructor.name === 'MemoryFileSystem' ? null : this.compiler.outputPath;
  }

}

module.exports = BundleAnalyzerPlugin;