"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

var _asyncToGenerator2 = _interopRequireDefault(require("@babel/runtime/helpers/asyncToGenerator"));

const path = require('path');

const fs = require('fs');

const http = require('http');

const WebSocket = require('ws');

const _ = require('lodash');

const express = require('express');

const ejs = require('ejs');

const opener = require('opener');

const mkdir = require('mkdirp');

const _require = require('chalk'),
      bold = _require.bold;

const Logger = require('./Logger');

const analyzer = require('./analyzer');

const projectRoot = path.resolve(__dirname, '..');
module.exports = {
  startServer,
  generateReport,
  // deprecated
  start: startServer
};

function startServer(_x, _x2) {
  return _startServer.apply(this, arguments);
}

function _startServer() {
  _startServer = (0, _asyncToGenerator2.default)(function* (bundleStats, opts) {
    const _ref2 = opts || {},
          _ref2$port = _ref2.port,
          port = _ref2$port === void 0 ? 8888 : _ref2$port,
          _ref2$host = _ref2.host,
          host = _ref2$host === void 0 ? '127.0.0.1' : _ref2$host,
          _ref2$openBrowser = _ref2.openBrowser,
          openBrowser = _ref2$openBrowser === void 0 ? true : _ref2$openBrowser,
          _ref2$bundleDir = _ref2.bundleDir,
          bundleDir = _ref2$bundleDir === void 0 ? null : _ref2$bundleDir,
          _ref2$logger = _ref2.logger,
          logger = _ref2$logger === void 0 ? new Logger() : _ref2$logger,
          _ref2$defaultSizes = _ref2.defaultSizes,
          defaultSizes = _ref2$defaultSizes === void 0 ? 'parsed' : _ref2$defaultSizes,
          _ref2$excludeAssets = _ref2.excludeAssets,
          excludeAssets = _ref2$excludeAssets === void 0 ? null : _ref2$excludeAssets;

    const analyzerOpts = {
      logger,
      excludeAssets
    };
    let chartData = getChartData(analyzerOpts, bundleStats, bundleDir);
    if (!chartData) return;
    const app = express(); // Explicitly using our `ejs` dependency to render templates
    // Fixes #17

    app.engine('ejs', require('ejs').renderFile);
    app.set('view engine', 'ejs');
    app.set('views', `${projectRoot}/views`);
    app.use(express.static(`${projectRoot}/public`));
    app.use('/', (req, res) => {
      res.render('viewer', {
        mode: 'server',

        get chartData() {
          return JSON.stringify(chartData);
        },

        defaultSizes: JSON.stringify(defaultSizes)
      });
    });
    const server = http.createServer(app);
    yield new Promise(resolve => {
      server.listen(port, host, () => {
        resolve();
        const url = `http://${host}:${server.address().port}`;
        logger.info(`${bold('Webpack Bundle Analyzer')} is started at ${bold(url)}\n` + `Use ${bold('Ctrl+C')} to close it`);

        if (openBrowser) {
          opener(url);
        }
      });
    });
    const wss = new WebSocket.Server({
      server
    });
    wss.on('connection', ws => {
      ws.on('error', err => {
        // Ignore network errors like `ECONNRESET`, `EPIPE`, etc.
        if (err.errno) return;
        logger.info(err.message);
      });
    });
    return {
      ws: wss,
      http: server,
      updateChartData
    };

    function updateChartData(bundleStats) {
      const newChartData = getChartData(analyzerOpts, bundleStats, bundleDir);
      if (!newChartData) return;
      chartData = newChartData;
      wss.clients.forEach(client => {
        if (client.readyState === WebSocket.OPEN) {
          client.send(JSON.stringify({
            event: 'chartDataUpdated',
            data: newChartData
          }));
        }
      });
    }
  });
  return _startServer.apply(this, arguments);
}

function generateReport(bundleStats, opts) {
  const _ref = opts || {},
        _ref$openBrowser = _ref.openBrowser,
        openBrowser = _ref$openBrowser === void 0 ? true : _ref$openBrowser,
        _ref$reportFilename = _ref.reportFilename,
        reportFilename = _ref$reportFilename === void 0 ? 'report.html' : _ref$reportFilename,
        _ref$bundleDir = _ref.bundleDir,
        bundleDir = _ref$bundleDir === void 0 ? null : _ref$bundleDir,
        _ref$logger = _ref.logger,
        logger = _ref$logger === void 0 ? new Logger() : _ref$logger,
        _ref$defaultSizes = _ref.defaultSizes,
        defaultSizes = _ref$defaultSizes === void 0 ? 'parsed' : _ref$defaultSizes,
        _ref$excludeAssets = _ref.excludeAssets,
        excludeAssets = _ref$excludeAssets === void 0 ? null : _ref$excludeAssets;

  const chartData = getChartData({
    logger,
    excludeAssets
  }, bundleStats, bundleDir);
  if (!chartData) return;
  ejs.renderFile(`${projectRoot}/views/viewer.ejs`, {
    mode: 'static',
    chartData: JSON.stringify(chartData),
    assetContent: getAssetContent,
    defaultSizes: JSON.stringify(defaultSizes)
  }, (err, reportHtml) => {
    if (err) return logger.error(err);
    const reportFilepath = path.resolve(bundleDir || process.cwd(), reportFilename);
    mkdir.sync(path.dirname(reportFilepath));
    fs.writeFileSync(reportFilepath, reportHtml);
    logger.info(`${bold('Webpack Bundle Analyzer')} saved report to ${bold(reportFilepath)}`);

    if (openBrowser) {
      opener(`file://${reportFilepath}`);
    }
  });
}

function getAssetContent(filename) {
  return fs.readFileSync(`${projectRoot}/public/${filename}`, 'utf8');
}

function getChartData(analyzerOpts, ...args) {
  let chartData;
  const logger = analyzerOpts.logger;

  try {
    chartData = analyzer.getViewerData(...args, analyzerOpts);
  } catch (err) {
    logger.error(`Could't analyze webpack bundle:\n${err}`);
    logger.debug(err.stack);
    chartData = null;
  }

  if (_.isPlainObject(chartData) && _.isEmpty(chartData)) {
    logger.error("Could't find any javascript bundles in provided stats file");
    chartData = null;
  }

  return chartData;
}