/*
	MIT License http://www.opensource.org/licenses/mit-license.php
	Author Tobias Koppers @sokra
*/
"use strict";

const Template = require("../Template");

class WebWorkerMainTemplatePlugin {
	apply(mainTemplate) {
		const needChunkOnDemandLoadingCode = chunk => {
			for(const chunkGroup of chunk.groupsIterable) {
				if(chunkGroup.getNumberOfChildren() > 0) return true;
			}
			return false;
		};
		mainTemplate.hooks.localVars.tap("WebWorkerMainTemplatePlugin", (source, chunk) => {
			if(needChunkOnDemandLoadingCode(chunk)) {
				return Template.asString([
					source,
					"",
					"// object to store loaded chunks",
					"// \"1\" means \"already loaded\"",
					"var installedChunks = {",
					Template.indent(
						chunk.ids.map((id) => `${id}: 1`).join(",\n")
					),
					"};"
				]);
			}
			return source;
		});
		mainTemplate.hooks.requireEnsure.tap("WebWorkerMainTemplatePlugin", (_, chunk, hash) => {
			const chunkFilename = mainTemplate.outputOptions.chunkFilename;
			return Template.asString([
				"promises.push(Promise.resolve().then(function() {",
				Template.indent([
					"// \"1\" is the signal for \"already loaded\"",
					"if(!installedChunks[chunkId]) {",
					Template.indent([
						"importScripts(" +
						mainTemplate.getAssetPath(JSON.stringify(chunkFilename), {
							hash: `" + ${mainTemplate.renderCurrentHashCode(hash)} + "`,
							hashWithLength: (length) => `" + ${mainTemplate.renderCurrentHashCode(hash, length)} + "`,
							chunk: {
								id: "\" + chunkId + \""
							}
						}) + ");"
					]),
					"}",
				]),
				"}));"
			]);
		});
		mainTemplate.hooks.bootstrap.tap("WebWorkerMainTemplatePlugin", (source, chunk, hash) => {
			if(needChunkOnDemandLoadingCode(chunk)) {
				const chunkCallbackName = mainTemplate.outputOptions.chunkCallbackName;
				const globalObject = mainTemplate.outputOptions.globalObject;
				return Template.asString([
					source,
					`${globalObject}[${JSON.stringify(chunkCallbackName)}] = function webpackChunkCallback(chunkIds, moreModules) {`,
					Template.indent([
						"for(var moduleId in moreModules) {",
						Template.indent(mainTemplate.renderAddModule(hash, chunk, "moduleId", "moreModules[moduleId]")),
						"}",
						"while(chunkIds.length)",
						Template.indent("installedChunks[chunkIds.pop()] = 1;")
					]),
					"};"
				]);
			}
			return source;
		});
		mainTemplate.hooks.hotBootstrap.tap("WebWorkerMainTemplatePlugin", (source, chunk, hash) => {
			const hotUpdateChunkFilename = mainTemplate.outputOptions.hotUpdateChunkFilename;
			const hotUpdateMainFilename = mainTemplate.outputOptions.hotUpdateMainFilename;
			const hotUpdateFunction = mainTemplate.outputOptions.hotUpdateFunction;
			const globalObject = mainTemplate.outputOptions.globalObject;
			const currentHotUpdateChunkFilename = mainTemplate.getAssetPath(JSON.stringify(hotUpdateChunkFilename), {
				hash: `" + ${mainTemplate.renderCurrentHashCode(hash)} + "`,
				hashWithLength: (length) => `" + ${mainTemplate.renderCurrentHashCode(hash, length)} + "`,
				chunk: {
					id: "\" + chunkId + \""
				}
			});
			const currentHotUpdateMainFilename = mainTemplate.getAssetPath(JSON.stringify(hotUpdateMainFilename), {
				hash: `" + ${mainTemplate.renderCurrentHashCode(hash)} + "`,
				hashWithLength: (length) => `" + ${mainTemplate.renderCurrentHashCode(hash, length)} + "`,
			});

			return source + "\n" +
				`var parentHotUpdateCallback = ${globalObject}[${JSON.stringify(hotUpdateFunction)}];\n` +
				`${globalObject}[${JSON.stringify(hotUpdateFunction)}] = ` +
				Template.getFunctionContent(require("./WebWorkerMainTemplate.runtime.js"))
				.replace(/\/\/\$semicolon/g, ";")
				.replace(/\$require\$/g, mainTemplate.requireFn)
				.replace(/\$hotMainFilename\$/g, currentHotUpdateMainFilename)
				.replace(/\$hotChunkFilename\$/g, currentHotUpdateChunkFilename)
				.replace(/\$hash\$/g, JSON.stringify(hash));
		});
		mainTemplate.hooks.hash.tap("WebWorkerMainTemplatePlugin", hash => {
			hash.update("webworker");
			hash.update("3");
			hash.update(`${mainTemplate.outputOptions.publicPath}`);
			hash.update(`${mainTemplate.outputOptions.filename}`);
			hash.update(`${mainTemplate.outputOptions.chunkFilename}`);
			hash.update(`${mainTemplate.outputOptions.chunkCallbackName}`);
			hash.update(`${mainTemplate.outputOptions.globalObject}`);
		});
	}
}
module.exports = WebWorkerMainTemplatePlugin;
