/*
	MIT License http://www.opensource.org/licenses/mit-license.php
	Author Tobias Koppers @sokra
*/
"use strict";

const path = require("path");

const OptionsDefaulter = require("./OptionsDefaulter");
const Template = require("./Template");

const isProductionLikeMode = options => {
	return options.mode === "production" || !options.mode;
};

class WebpackOptionsDefaulter extends OptionsDefaulter {
	constructor() {
		super();

		this.set("entry", "./src");

		this.set("devtool", "make", options => (options.mode === "development" ? "eval" : false));
		this.set("cache", "make", options => options.mode === "development");

		this.set("context", process.cwd());
		this.set("target", "web");

		this.set("module", "call", value => Object.assign({}, value));
		this.set("module.unknownContextRequest", ".");
		this.set("module.unknownContextRegExp", false);
		this.set("module.unknownContextRecursive", true);
		this.set("module.unknownContextCritical", true);
		this.set("module.exprContextRequest", ".");
		this.set("module.exprContextRegExp", false);
		this.set("module.exprContextRecursive", true);
		this.set("module.exprContextCritical", true);
		this.set("module.wrappedContextRegExp", /.*/);
		this.set("module.wrappedContextRecursive", true);
		this.set("module.wrappedContextCritical", false);
		this.set("module.strictExportPresence", false);
		this.set("module.strictThisContextOnImports", false);
		this.set("module.unsafeCache", "make", options => !!options.cache);
		this.set("module.rules", []);
		this.set("module.defaultRules", "make", options => [{
				type: "javascript/auto",
				resolve: {}
			},
			{
				test: /\.mjs$/i,
				type: "javascript/esm",
				resolve: {
					mainFields: options.target === "web" || options.target === "webworker" ? ["browser", "main"] : ["main"]
				}
			},
			{
				test: /\.json$/i,
				type: "json"
			},
			{
				test: /\.wasm$/i,
				type: "webassembly/experimental"
			}
		]);

		this.set("output", "call", (value, options) => {
			if(typeof value === "string") {
				return {
					filename: value
				};
			} else if(typeof value !== "object") {
				return {};
			} else {
				return Object.assign({}, value);
			}
		});

		this.set("output.filename", "[name].js");
		this.set("output.chunkFilename", "make", options => {
			const filename = options.output.filename;
			if(typeof filename === "function") return filename;
			const hasName = filename.includes("[name]");
			const hasId = filename.includes("[id]");
			const hasChunkHash = filename.includes("[chunkhash]");
			// Anything changing depending on chunk is fine
			if(hasChunkHash || hasName || hasId) return filename;
			// Elsewise prefix "[id]." in front of the basename to make it changing
			return filename.replace(/(^|\/)([^/]*(?:\?|$))/, "$1[id].$2");
		});
		this.set("output.webassemblyModuleFilename", "[modulehash].module.wasm");
		this.set("output.library", "");
		this.set("output.hotUpdateFunction", "make", options => {
			return Template.toIdentifier("webpackHotUpdate" + Template.toIdentifier(options.output.library));
		});
		this.set("output.jsonpFunction", "make", options => {
			return Template.toIdentifier("webpackJsonp" + Template.toIdentifier(options.output.library));
		});
		this.set("output.chunkCallbackName", "make", options => {
			return Template.toIdentifier("webpackChunk" + Template.toIdentifier(options.output.library));
		});
		this.set("output.globalObject", "make", options => {
			switch(options.target) {
				case "web":
				case "electron-renderer":
					return "window";
				case "webworker":
					return "self";
				case "node":
				case "async-node":
				case "node-webkit":
				case "electron-main":
					return "global";
				default:
					return "self";
			}
		});
		this.set("output.devtoolNamespace", "make", options => {
			return options.output.library || "";
		});
		this.set("output.libraryTarget", "var");
		this.set("output.path", path.join(process.cwd(), "dist"));
		this.set("output.pathinfo", "make", options => options.mode === "development");
		this.set("output.sourceMapFilename", "[file].map[query]");
		this.set("output.hotUpdateChunkFilename", "[id].[hash].hot-update.js");
		this.set("output.hotUpdateMainFilename", "[hash].hot-update.json");
		this.set("output.crossOriginLoading", false);
		this.set("output.jsonpScriptType", false);
		this.set("output.chunkLoadTimeout", 120000);
		this.set("output.hashFunction", "md4");
		this.set("output.hashDigest", "hex");
		this.set("output.hashDigestLength", 20);
		this.set("output.devtoolLineToLine", false);
		this.set("output.strictModuleExceptionHandling", false);

		this.set("node", "call", value => {
			if(typeof value === "boolean") {
				return value;
			} else {
				return Object.assign({}, value);
			}
		});
		this.set("node.console", false);
		this.set("node.process", true);
		this.set("node.global", true);
		this.set("node.Buffer", true);
		this.set("node.setImmediate", true);
		this.set("node.__filename", "mock");
		this.set("node.__dirname", "mock");

		this.set("performance", "make", options => (isProductionLikeMode(options) ? false : undefined));
		this.set("performance", "call", value => {
			if(typeof value === "boolean") {
				return value;
			} else {
				return Object.assign({}, value);
			}
		});
		this.set("performance.maxAssetSize", 250000);
		this.set("performance.maxEntrypointSize", 250000);
		this.set("performance.hints", "make", options => (isProductionLikeMode(options) ? "warning" : false));

		this.set("optimization.removeAvailableModules", true);
		this.set("optimization.removeEmptyChunks", true);
		this.set("optimization.mergeDuplicateChunks", true);
		this.set("optimization.flagIncludedChunks", "make", options => isProductionLikeMode(options));
		this.set("optimization.occurrenceOrder", "make", options => isProductionLikeMode(options));
		this.set("optimization.sideEffects", "make", options => isProductionLikeMode(options));
		this.set("optimization.providedExports", true);
		this.set("optimization.usedExports", "make", options => isProductionLikeMode(options));
		this.set("optimization.concatenateModules", "make", options => isProductionLikeMode(options));
		this.set("optimization.splitChunks", {});
		this.set("optimization.splitChunks.chunks", "async");
		this.set("optimization.splitChunks.minSize", 30000);
		this.set("optimization.splitChunks.minChunks", 1);
		this.set("optimization.splitChunks.maxAsyncRequests", 5);
		this.set("optimization.splitChunks.maxInitialRequests", 3);
		this.set("optimization.splitChunks.name", true);
		this.set("optimization.splitChunks.cacheGroups", {});
		this.set("optimization.splitChunks.cacheGroups.default", {
			reuseExistingChunk: true,
			minChunks: 2,
			priority: -20
		});
		this.set("optimization.splitChunks.cacheGroups.vendors", {
			test: /[\\/]node_modules[\\/]/,
			priority: -10
		});
		this.set("optimization.runtimeChunk", "call", value => {
			if(value === "single") {
				return {
					name: "runtime"
				};
			}
			if(value === true || value === "multiple") {
				return {
					name: entrypoint => `runtime~${entrypoint.name}`
				};
			}
			return value;
		});
		this.set("optimization.noEmitOnErrors", "make", options => isProductionLikeMode(options));
		this.set("optimization.namedModules", "make", options => options.mode === "development");
		this.set("optimization.namedChunks", "make", options => options.mode === "development");
		this.set("optimization.portableRecords", "make", options => !!(options.recordsInputPath || options.recordsOutputPath || options.recordsPath));
		this.set("optimization.minimize", "make", options => isProductionLikeMode(options));
		this.set("optimization.minimizer", "make", options => [{
			apply: compiler => {
				// Lazy load the uglifyjs plugin
				const UglifyJsPlugin = require("uglifyjs-webpack-plugin");
				new UglifyJsPlugin({
					cache: true,
					parallel: true,
					sourceMap: options.devtool && /source-?map/.test(options.devtool)
				}).apply(compiler);
			}
		}]);
		this.set("optimization.nodeEnv", "make", options => options.mode || "production");

		this.set("resolve", "call", value => Object.assign({}, value));
		this.set("resolve.unsafeCache", true);
		this.set("resolve.modules", ["node_modules"]);
		this.set("resolve.extensions", [".wasm", ".mjs", ".js", ".json"]);
		this.set("resolve.mainFiles", ["index"]);
		this.set("resolve.aliasFields", "make", options => {
			if(options.target === "web" || options.target === "webworker") return ["browser"];
			else return [];
		});
		this.set("resolve.mainFields", "make", options => {
			if(options.target === "web" || options.target === "webworker") return ["browser", "module", "main"];
			else return ["module", "main"];
		});
		this.set("resolve.cacheWithContext", "make", options => {
			return Array.isArray(options.resolve.plugins) && options.resolve.plugins.length > 0;
		});

		this.set("resolveLoader", "call", value => Object.assign({}, value));
		this.set("resolveLoader.unsafeCache", true);
		this.set("resolveLoader.mainFields", ["loader", "main"]);
		this.set("resolveLoader.extensions", [".js", ".json"]);
		this.set("resolveLoader.mainFiles", ["index"]);
		this.set("resolveLoader.cacheWithContext", "make", options => {
			return Array.isArray(options.resolveLoader.plugins) && options.resolveLoader.plugins.length > 0;
		});
	}
}

module.exports = WebpackOptionsDefaulter;
