(function (root, factory) {
    if (typeof exports === "object") {
        // CommonJS-like
        module.exports = factory();
    } else if (typeof define === 'function' && define.amd) {
        // AMD.
        define([], factory);
    } else {
        // Browser globals
        root.WorldWind = factory();
    }
}(this, function () {

/**
 * @license almond 0.3.0 Copyright (c) 2011-2014, The Dojo Foundation All Rights Reserved.
 * Available via the MIT or new BSD license.
 * see: http://github.com/jrburke/almond for details
 */
//Going sloppy to avoid 'use strict' string cost, but strict practices should
//be followed.
/*jslint sloppy: true */
/*global setTimeout: false */

var requirejs, require, define;
(function (undef) {
    var main, req, makeMap, handlers,
        defined = {},
        waiting = {},
        config = {},
        defining = {},
        hasOwn = Object.prototype.hasOwnProperty,
        aps = [].slice,
        jsSuffixRegExp = /\.js$/;

    function hasProp(obj, prop) {
        return hasOwn.call(obj, prop);
    }

    /**
     * Given a relative module name, like ./something, normalize it to
     * a real name that can be mapped to a path.
     * @param {String} name the relative name
     * @param {String} baseName a real name that the name arg is relative
     * to.
     * @returns {String} normalized name
     */
    function normalize(name, baseName) {
        var nameParts, nameSegment, mapValue, foundMap, lastIndex,
            foundI, foundStarMap, starI, i, j, part,
            baseParts = baseName && baseName.split("/"),
            map = config.map,
            starMap = (map && map['*']) || {};

        //Adjust any relative paths.
        if (name && name.charAt(0) === ".") {
            //If have a base name, try to normalize against it,
            //otherwise, assume it is a top-level require that will
            //be relative to baseUrl in the end.
            if (baseName) {
                //Convert baseName to array, and lop off the last part,
                //so that . matches that "directory" and not name of the baseName's
                //module. For instance, baseName of "one/two/three", maps to
                //"one/two/three.js", but we want the directory, "one/two" for
                //this normalization.
                baseParts = baseParts.slice(0, baseParts.length - 1);
                name = name.split('/');
                lastIndex = name.length - 1;

                // Node .js allowance:
                if (config.nodeIdCompat && jsSuffixRegExp.test(name[lastIndex])) {
                    name[lastIndex] = name[lastIndex].replace(jsSuffixRegExp, '');
                }

                name = baseParts.concat(name);

                //start trimDots
                for (i = 0; i < name.length; i += 1) {
                    part = name[i];
                    if (part === ".") {
                        name.splice(i, 1);
                        i -= 1;
                    } else if (part === "..") {
                        if (i === 1 && (name[2] === '..' || name[0] === '..')) {
                            //End of the line. Keep at least one non-dot
                            //path segment at the front so it can be mapped
                            //correctly to disk. Otherwise, there is likely
                            //no path mapping for a path starting with '..'.
                            //This can still fail, but catches the most reasonable
                            //uses of ..
                            break;
                        } else if (i > 0) {
                            name.splice(i - 1, 2);
                            i -= 2;
                        }
                    }
                }
                //end trimDots

                name = name.join("/");
            } else if (name.indexOf('./') === 0) {
                // No baseName, so this is ID is resolved relative
                // to baseUrl, pull off the leading dot.
                name = name.substring(2);
            }
        }

        //Apply map config if available.
        if ((baseParts || starMap) && map) {
            nameParts = name.split('/');

            for (i = nameParts.length; i > 0; i -= 1) {
                nameSegment = nameParts.slice(0, i).join("/");

                if (baseParts) {
                    //Find the longest baseName segment match in the config.
                    //So, do joins on the biggest to smallest lengths of baseParts.
                    for (j = baseParts.length; j > 0; j -= 1) {
                        mapValue = map[baseParts.slice(0, j).join('/')];

                        //baseName segment has  config, find if it has one for
                        //this name.
                        if (mapValue) {
                            mapValue = mapValue[nameSegment];
                            if (mapValue) {
                                //Match, update name to the new value.
                                foundMap = mapValue;
                                foundI = i;
                                break;
                            }
                        }
                    }
                }

                if (foundMap) {
                    break;
                }

                //Check for a star map match, but just hold on to it,
                //if there is a shorter segment match later in a matching
                //config, then favor over this star map.
                if (!foundStarMap && starMap && starMap[nameSegment]) {
                    foundStarMap = starMap[nameSegment];
                    starI = i;
                }
            }

            if (!foundMap && foundStarMap) {
                foundMap = foundStarMap;
                foundI = starI;
            }

            if (foundMap) {
                nameParts.splice(0, foundI, foundMap);
                name = nameParts.join('/');
            }
        }

        return name;
    }

    function makeRequire(relName, forceSync) {
        return function () {
            //A version of a require function that passes a moduleName
            //value for items that may need to
            //look up paths relative to the moduleName
            var args = aps.call(arguments, 0);

            //If first arg is not require('string'), and there is only
            //one arg, it is the array form without a callback. Insert
            //a null so that the following concat is correct.
            if (typeof args[0] !== 'string' && args.length === 1) {
                args.push(null);
            }
            return req.apply(undef, args.concat([relName, forceSync]));
        };
    }

    function makeNormalize(relName) {
        return function (name) {
            return normalize(name, relName);
        };
    }

    function makeLoad(depName) {
        return function (value) {
            defined[depName] = value;
        };
    }

    function callDep(name) {
        if (hasProp(waiting, name)) {
            var args = waiting[name];
            delete waiting[name];
            defining[name] = true;
            main.apply(undef, args);
        }

        if (!hasProp(defined, name) && !hasProp(defining, name)) {
            throw new Error('No ' + name);
        }
        return defined[name];
    }

    //Turns a plugin!resource to [plugin, resource]
    //with the plugin being undefined if the name
    //did not have a plugin prefix.
    function splitPrefix(name) {
        var prefix,
            index = name ? name.indexOf('!') : -1;
        if (index > -1) {
            prefix = name.substring(0, index);
            name = name.substring(index + 1, name.length);
        }
        return [prefix, name];
    }

    /**
     * Makes a name map, normalizing the name, and using a plugin
     * for normalization if necessary. Grabs a ref to plugin
     * too, as an optimization.
     */
    makeMap = function (name, relName) {
        var plugin,
            parts = splitPrefix(name),
            prefix = parts[0];

        name = parts[1];

        if (prefix) {
            prefix = normalize(prefix, relName);
            plugin = callDep(prefix);
        }

        //Normalize according
        if (prefix) {
            if (plugin && plugin.normalize) {
                name = plugin.normalize(name, makeNormalize(relName));
            } else {
                name = normalize(name, relName);
            }
        } else {
            name = normalize(name, relName);
            parts = splitPrefix(name);
            prefix = parts[0];
            name = parts[1];
            if (prefix) {
                plugin = callDep(prefix);
            }
        }

        //Using ridiculous property names for space reasons
        return {
            f: prefix ? prefix + '!' + name : name, //fullName
            n: name,
            pr: prefix,
            p: plugin
        };
    };

    function makeConfig(name) {
        return function () {
            return (config && config.config && config.config[name]) || {};
        };
    }

    handlers = {
        require: function (name) {
            return makeRequire(name);
        },
        exports: function (name) {
            var e = defined[name];
            if (typeof e !== 'undefined') {
                return e;
            } else {
                return (defined[name] = {});
            }
        },
        module: function (name) {
            return {
                id: name,
                uri: '',
                exports: defined[name],
                config: makeConfig(name)
            };
        }
    };

    main = function (name, deps, callback, relName) {
        var cjsModule, depName, ret, map, i,
            args = [],
            callbackType = typeof callback,
            usingExports;

        //Use name if no relName
        relName = relName || name;

        //Call the callback to define the module, if necessary.
        if (callbackType === 'undefined' || callbackType === 'function') {
            //Pull out the defined dependencies and pass the ordered
            //values to the callback.
            //Default to [require, exports, module] if no deps
            deps = !deps.length && callback.length ? ['require', 'exports', 'module'] : deps;
            for (i = 0; i < deps.length; i += 1) {
                map = makeMap(deps[i], relName);
                depName = map.f;

                //Fast path CommonJS standard dependencies.
                if (depName === "require") {
                    args[i] = handlers.require(name);
                } else if (depName === "exports") {
                    //CommonJS module spec 1.1
                    args[i] = handlers.exports(name);
                    usingExports = true;
                } else if (depName === "module") {
                    //CommonJS module spec 1.1
                    cjsModule = args[i] = handlers.module(name);
                } else if (hasProp(defined, depName) ||
                           hasProp(waiting, depName) ||
                           hasProp(defining, depName)) {
                    args[i] = callDep(depName);
                } else if (map.p) {
                    map.p.load(map.n, makeRequire(relName, true), makeLoad(depName), {});
                    args[i] = defined[depName];
                } else {
                    throw new Error(name + ' missing ' + depName);
                }
            }

            ret = callback ? callback.apply(defined[name], args) : undefined;

            if (name) {
                //If setting exports via "module" is in play,
                //favor that over return value and exports. After that,
                //favor a non-undefined return value over exports use.
                if (cjsModule && cjsModule.exports !== undef &&
                        cjsModule.exports !== defined[name]) {
                    defined[name] = cjsModule.exports;
                } else if (ret !== undef || !usingExports) {
                    //Use the return value from the function.
                    defined[name] = ret;
                }
            }
        } else if (name) {
            //May just be an object definition for the module. Only
            //worry about defining if have a module name.
            defined[name] = callback;
        }
    };

    requirejs = require = req = function (deps, callback, relName, forceSync, alt) {
        if (typeof deps === "string") {
            if (handlers[deps]) {
                //callback in this case is really relName
                return handlers[deps](callback);
            }
            //Just return the module wanted. In this scenario, the
            //deps arg is the module name, and second arg (if passed)
            //is just the relName.
            //Normalize module name, if it contains . or ..
            return callDep(makeMap(deps, callback).f);
        } else if (!deps.splice) {
            //deps is a config object, not an array.
            config = deps;
            if (config.deps) {
                req(config.deps, config.callback);
            }
            if (!callback) {
                return;
            }

            if (callback.splice) {
                //callback is an array, which means it is a dependency list.
                //Adjust args if there are dependencies
                deps = callback;
                callback = relName;
                relName = null;
            } else {
                deps = undef;
            }
        }

        //Support require(['a'])
        callback = callback || function () {};

        //If relName is a function, it is an errback handler,
        //so remove it.
        if (typeof relName === 'function') {
            relName = forceSync;
            forceSync = alt;
        }

        //Simulate async callback;
        if (forceSync) {
            main(undef, deps, callback, relName);
        } else {
            //Using a non-zero value because of concern for what old browsers
            //do, and latest browsers "upgrade" to 4 if lower value is used:
            //http://www.whatwg.org/specs/web-apps/current-work/multipage/timers.html#dom-windowtimers-settimeout:
            //If want a value immediately, use require('id') instead -- something
            //that works in almond on the global level, but not guaranteed and
            //unlikely to work in other AMD implementations.
            setTimeout(function () {
                main(undef, deps, callback, relName);
            }, 4);
        }

        return req;
    };

    /**
     * Just drops the config on the floor, but returns req in case
     * the config return value is used.
     */
    req.config = function (cfg) {
        return req(cfg);
    };

    /**
     * Expose module registry for debugging and tooling
     */
    requirejs._defined = defined;

    define = function (name, deps, callback) {

        //This module may not have dependencies
        if (!deps.splice) {
            //deps is not an array, so probably means
            //an object literal or factory function for
            //the value. Adjust args.
            callback = deps;
            deps = [];
        }

        if (!hasProp(defined, name) && !hasProp(waiting, name)) {
            waiting[name] = [name, deps, callback];
        }
    };

    define.amd = {
        jQuery: true
    };
}());

define("../tools/almond", function(){});

/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
define('formats/aaigrid/AAIGridConstants',[],
    function () {
        'use strict';

        /**
         * Provides constants for the AAIGridReader.
         * @exports AAIGridConstants
         */
        var AAIGridConstants = {
            N_COLS: 'ncols',
            N_ROWS:'nrows',
            X_LL_CORNER: 'xllcorner',
            Y_LL_CORNER: 'yllcorner',
            CELL_SIZE: 'cellsize',
            NO_DATA_VALUE: 'NODATA_value'
        };

        return AAIGridConstants;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports AAIGridMetadata
 */
define('formats/aaigrid/AAIGridMetadata',[],
    function () {
        'use strict';

        /**
         * Constructs a new container for AAIGrid metadata.
         * @alias AAIGridMetadata
         * @constructor
         * @classdesc Contains the metadata for an AAIGrid file.
         */
        var AAIGridMetadata = function () {
            // Documented in defineProperties below.
            this._ncols = null;

            // Documented in defineProperties below.
            this._nrows = null;

            // Documented in defineProperties below.
            this._xllcorner = null;

            // Documented in defineProperties below.
            this._yllcorner = null;

            // Documented in defineProperties below.
            this._cellsize = null;

            // Documented in defineProperties below.
            this._NODATA_value = undefined;
        };

        Object.defineProperties(AAIGridMetadata.prototype, {

            /**
             * Number of columns in the grid.
             * @memberof AAIGridMetadata.prototype
             * @type {Number}
             */
            ncols: {
                get: function () {
                    return this._ncols;
                },
                set: function (value) {
                    this._ncols = value;
                }
            },

            /**
             * Number of rows in the grid.
             * @memberof AAIGridMetadata.prototype
             * @type {Number}
             */
            nrows: {
                get: function () {
                    return this._nrows;
                },
                set: function (value) {
                    this._nrows = value;
                }
            },

            /**
             * The western edge (left), x-coordinate
             * @memberof AAIGridMetadata.prototype
             * @type {Number}
             */
            xllcorner: {
                get: function () {
                    return this._xllcorner;
                },
                set: function (value) {
                    this._xllcorner = value;
                }
            },

            /**
             * The southern edge (bottom), y-coordinate
             * @memberof AAIGridMetadata.prototype
             * @type {Number}
             */
            yllcorner: {
                get: function () {
                    return this._yllcorner;
                },
                set: function (value) {
                    this._yllcorner = value;
                }
            },

            /**
             * The length of one side of a square cell (resolution of the grid).
             * @memberof AAIGridMetadata.prototype
             * @type {Number}
             */
            cellsize: {
                get: function () {
                    return this._cellsize;
                },
                set: function (value) {
                    this._cellsize = value;
                }
            },

            /**
             * The value that is regarded as "missing" or "not applicable".
             * This value is optional. If not present, its value is <code>undefined</code>.
             * @memberof AAIGridMetadata.prototype
             * @type {Number|undefined}
             */
            NODATA_value: {
                get: function () {
                    return this._NODATA_value;
                },
                set: function (value) {
                    this._NODATA_value = value;
                }
            }
        });

        return AAIGridMetadata;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports AbstractError
 */
define('error/AbstractError',[],function () {
    "use strict";

    /**
     * Constructs an error with a specified name and message.
     * @alias AbstractError
     * @constructor
     * @abstract
     * @classdesc Provides an abstract base class for error classes. This class is not meant to be instantiated
     * directly but used only by subclasses.
     * @param {String} name The error's name.
     * @param {String} message The message.
     */
    var AbstractError = function (name, message) {
        this.name = name;
        this.message = message;
    };

    /**
     * Returns the message and stack trace associated with this error.
     * @returns {String} The message and stack trace associated with this error.
     */
    AbstractError.prototype.toString = function () {
        var str = this.name + ': ' + this.message;

        if (this.stack) {
            str += '\n' + this.stack.toString();
        }

        return str;
    };

    return AbstractError;

});
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports ArgumentError
 */
define('error/ArgumentError',['../error/AbstractError'],
    function (AbstractError) {
        "use strict";

        /**
         * Constructs an argument error with a specified message.
         * @alias ArgumentError
         * @constructor
         * @classdesc Represents an error associated with invalid function arguments.
         * @augments AbstractError
         * @param {String} message The message.
         */
        var ArgumentError = function (message) {
            AbstractError.call(this, "ArgumentError", message);

            var stack;
            try {
                //noinspection ExceptionCaughtLocallyJS
                throw new Error();
            } catch (e) {
                stack = e.stack;
            }
            this.stack = stack;
        };

        ArgumentError.prototype = Object.create(AbstractError.prototype);

        return ArgumentError;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
define('util/Logger',[],function () {
    "use strict";
    /**
     * Logs selected message types to the console.
     * @exports Logger
     */

    var Logger = {
        /**
         * Log no messages.
         * @constant
         */
        LEVEL_NONE: 0,
        /**
         * Log messages marked as severe.
         * @constant
         */
        LEVEL_SEVERE: 1,
        /**
         * Log messages marked as warnings and messages marked as severe.
         * @constant
         */
        LEVEL_WARNING: 2,
        /**
         * Log messages marked as information, messages marked as warnings and messages marked as severe.
         * @constant
         */
        LEVEL_INFO: 3,

        /**
         * Set the logging level used by subsequent invocations of the logger.
         * @param {Number} level The logging level, one of Logger.LEVEL_NONE, Logger.LEVEL_SEVERE, Logger.LEVEL_WARNING,
         * or Logger.LEVEL_INFO.
         */
        setLoggingLevel: function (level) {
            loggingLevel = level;
        },

        /**
         * Indicates the current logging level.
         * @returns {Number} The current logging level.
         */
        getLoggingLevel: function () {
            return loggingLevel;
        },

        /**
         * Logs a specified message at a specified level.
         * @param {Number} level The logging level of the message. If the current logging level allows this message to be
         * logged it is written to the console.
         * @param {String} message The message to log. Nothing is logged if the message is null or undefined.
         */
        log: function (level, message) {
            if (message && level > 0 && level <= loggingLevel) {
                if (level === Logger.LEVEL_SEVERE) {
                    console.error(message);
                } else if (level === Logger.LEVEL_WARNING) {
                    console.warn(message);
                } else if (level === Logger.LEVEL_INFO) {
                    console.info(message);
                } else {
                    console.log(message);
                }
            }
        },

        // Intentionally not documented.
        makeMessage: function (className, functionName, message) {
            var msg = this.messageTable[message] ? this.messageTable[message] : message;

            return className + "." + functionName + ": " + msg;
        },

        // Intentionally not documented.
        logMessage: function (level, className, functionName, message) {
            var msg = this.makeMessage(className, functionName, message);
            this.log(level, msg);

            return msg;
        },

        // Intentionally not documented.
        messageTable: { // KEEP THIS TABLE IN ALPHABETICAL ORDER
            abstractInvocation: "The function called is abstract and must be overridden in a subclass.",
            indexOutOfRange: "The specified index is out of range.",
            invalidColumn: "The specified column is out of range.",
            invalidHeight: "The specified height is zero or negative.",
            invalidWidth: "The specified width is zero or negative.",
            invalidRow: "The specified row is out of range.",
            invalidSize: "The specified size is zero or negative.",
            missingAltitudeMode: "The specified altitude mode is null or undefined.",
            missingArrayBuffer: "The specified array buffer is null or undefined",
            missingAttributeName: "The specified DBase attribute file name is null or undefined.",
            missingArray: "The specified array is null, undefined or of insufficient length.",
            missingBoundaries: "The specified boundaries array is null or undefined.",
            missingBuffer: "The specified buffer descriptor is null or undefined.",
            missingColor: "The specified color is null or undefined.",
            missingConfig: "The specified config is null or undefined.",
            missingDc: "The specified draw context is null or undefined.",
            missingDomElement: "The specified DOM element is null or undefined.",
            missingEntry: "The specified entry is null or undefined.",
            missingFont: "The specified font is null or undefined.",
            missingFrustum: "The specified frustum is null or undefined.",
            missingFunction: "The specified function is null or undefined.",
            missingGlContext: "The specified WebGL rendering context is null or undefined.",
            missingGlobe: "The specified globe is null or undefined.",
            missingId: "The specified id is null or undefined.",
            missingImage: "The specified image is null or undefined.",
            missingImageFormat: "The specified image format is null or undefined.",
            missingIndices: "The specified indices array is null or undefined.",
            missingKey: "The specified key is null or undefined.",
            missingLevel: "The specified level is null or undefined.",
            missingLine: "The specified line is null or undefined.",
            missingList: "The specified list is null or undefined.",
            missingListener: "The specified listener is null or undefined",
            missingLocation: "The specified location is null or undefined.",
            missingMatrix: "The specified matrix is null or undefined.",
            missingOffset: "The specified offset is null or undefined.",
            missingPath: "The specified path is null or undefined.",
            missingPlacename: "The specified place name is null or undefined.",
            missingPlane: "The specified plane is null or undefined.",
            missingPoint: "The specified point is null or undefined.",
            missingPoints: "The specified points array is null or undefined.",
            missingPosition: "The specified position is null or undefined.",
            missingPositions: "The specified positions array is null or undefined.",
            missingProgram: "The specified program is null or undefined.",
            missingProjection: "The specified projection is null or undefined.",
            missingRectangle: "The specified rectangle is null or undefined.",
            missingRenderable: "The specified renderable is null or undefined.",
            missingResolution: "The specified resolution is null, undefined, or zero.",
            missingResource: "The specified resource is null or undefined.",
            missingResult: "The specified result variable is null or undefined.",
            missingResults: "The specified results array is null or undefined.",
            missingSector: "The specified sector is null or undefined.",
            missingShapeType: "The specified shape type is null or undefined.",
            missingSize: "The specified size is null or undefined.",
            missingText: "The specified text is null or undefined.",
            missingTexture: "The specified texture is null or undefined.",
            missingTile: "The specified tile is null or undefined.",
            missingType: "The specified type is null or undefined.",
            missingUrl: "The specified URL is null or undefined",
            missingVector: "The specified vector is null or undefined.",
            missingVertex: "The specified vertex is null or undefined.",
            missingViewport: "The specified viewport is null or undefined.",
            missingWebCoverageService: "The specified WebCoverageService is null or undefined.",
            missingWorldWindow: "The specified WorldWindow is null or undefined.",
            notYetImplemented: "This function is not yet implemented",
            unsupportedVersion: "The specified version is not supported.",
            webglNotSupported: "The browser does not support WebGL, or WebGL is disabled."
        }
    };

    var loggingLevel = 1; // log severe messages by default

    return Logger;
});
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports AAIGridReader
 */
define('formats/aaigrid/AAIGridReader',[
        './AAIGridConstants',
        './AAIGridMetadata',
        '../../error/ArgumentError',
        '../../util/Logger'
    ],
    function (AAIGridConstants,
              AAIGridMetadata,
              ArgumentError,
              Logger) {
        'use strict';

        /**
         * Constructs an AAIGrid reader object for the specified data source.
         * Call [getImageData]{@link AAIGridReader#getImageData} to retrieve the data as a typed array.
         * Use [metadata]{@link AAIGridReader#metadata} to access the metadata of this AAIGrid.
         * @alias AAIGridReader
         * @constructor
         * @classdesc Parses an AAIGrid and creates a typed array with its values.
         * @param {String|ArrayBuffer} dataSource The data source for the AAIGrid.
         * @throws {ArgumentError} If the specified data source is not a string or an array buffer.
         */
        var AAIGridReader = function (dataSource) {
            if (!dataSource) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "AAIGridReader", "constructor", "missingDataSource")
                );
            }

            if (typeof dataSource !== 'string' && !(dataSource instanceof ArrayBuffer)) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "AAIGridReader", "constructor", "invalidDataSource")
                );
            }

            //Internal. Contains the numerical values of the AAIGrid.
            this._values = null;

            //Documented in defineProperties below.
            this._metadata = new AAIGridMetadata();

            var dataString = this.decodeData(dataSource);
            this.parse(dataString);
        };

        Object.defineProperties(AAIGridReader.prototype, {
            /**
             * An object containing the metadata of the AAIGrid file.
             * @memberof AAIGridReader.prototype
             * @type {AAIGridMetadata}
             * @readonly
             */
            metadata: {
                get: function () {
                    return this._metadata;
                }
            }
        });

        /**
         * Returns the content of the AAIGrid as an Int16Array or Float32Array.
         *
         * @return {Int16Array|Float32Array} The content of the AAIGrid.
         */
        AAIGridReader.prototype.getImageData = function () {
            return this._values;
        };

        /**
         * Internal. Applications should not call this method.
         * Decodes an arrayBuffer as a string.
         * If the dataSource is a string, no decoding takes place, the string is immediately returned.
         *
         * @param {String|ArrayBuffer} dataSource The data source to decode.
         * @return {String} The decoded array buffer.
         * @throws {ArgumentError} If the specified data source is not a string or an array buffer.
         */
        AAIGridReader.prototype.decodeData = function (dataSource) {
            if (typeof dataSource !== 'string' && !(dataSource instanceof ArrayBuffer)) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "AAIGridReader", "decodeData", "invalidDataSource")
                );
            }

            if (typeof dataSource === 'string') {
                return dataSource;
            }

            if (typeof window.TextDecoder === 'function') {
                var decoder = new TextDecoder('utf-8');
                return decoder.decode(dataSource);
            }

            // The procedure below is for old browsers.
            var dataString = '';
            var bufferView = new Uint8Array(dataSource);

            var step = 65535; // The maximum number of arguments a function can take
            for (var i = 0, len = bufferView.length; i < len; i += step) {
                if (i + step > len) {
                    step = len - i;
                }
                dataString += String.fromCharCode.apply(null, bufferView.subarray(i, i + step));
            }

            return dataString;
        };

        /**
         * Internal. Applications should not call this method.
         * Parses the AAIGrid.
         *
         * @param {String} dataString The string to parse.
         * @throws {ArgumentError} If the specified data source is not a string.
         */
        AAIGridReader.prototype.parse = function (dataString) {
            if (typeof dataString !== 'string') {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "AAIGridReader", "parse", "invalidDataString")
                );
            }

            var lines = dataString.replace(/\r?\n|\r/g, '\n').split('\n');

            var values = [];
            var isInt = true;
            for (var i = 0, len = lines.length; i < len; i++) {
                var line = lines[i];

                if (!line) {
                    continue;
                }

                var words = line
                    .trim()
                    .split(' ')
                    .map(function (word) {
                        return word.trim();
                    })
                    .filter(function (word) {
                        return word !== '';
                    });

                if (words[0] === AAIGridConstants.N_COLS) {
                    this.metadata.ncols = +words[1];
                }
                else if (words[0] === AAIGridConstants.N_ROWS) {
                    this.metadata.nrows = +words[1];
                }
                else if (words[0] === AAIGridConstants.X_LL_CORNER) {
                    this.metadata.xllcorner = +words[1];
                }
                else if (words[0] === AAIGridConstants.Y_LL_CORNER) {
                    this.metadata.yllcorner = +words[1];
                }
                else if (words[0] === AAIGridConstants.CELL_SIZE) {
                    this.metadata.cellsize = +words[1];
                }
                else if (words[0] === AAIGridConstants.NO_DATA_VALUE) {
                    this.metadata.NODATA_value = +words[1];
                }
                else if (isFinite(+words[0])) {
                    var numbers = words.map(function (word) {
                        var number = +word;
                        if (Math.floor(number) !== number) {
                            isInt = false;
                        }
                        return number;
                    });
                    values = values.concat(numbers);
                }
            }

            this._values = isInt ? new Int16Array(values) : new Float32Array(values);
        };

        /**
         * Attempts to retrieve the AAIGrid data from the provided URL, parse the data and return an AAIGridReader
         * using the provided parserCompletionCallback.
         *
         * @param {String} url An URL pointing to an external resource.
         * @param {Function} parserCompletionCallback A function to execute when the retrieval finishes, taking two
         * arguments:
         * <ul>
         *     <li>AAIGridReader instance in case of success, otherwise <code>null</code></li>
         *     <li>XMLHttpRequest instance</li>
         * </ul>
         */
        AAIGridReader.retrieveFromUrl = function (url, parserCompletionCallback) {
            if (!url) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "AAIGridReader", "retrieveFromUrl", "missingUrl"));
            }

            if (!parserCompletionCallback) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "AAIGridReader", "retrieveFromUrl",
                        "The specified callback is null or undefined."));
            }

            var xhr = new XMLHttpRequest();

            xhr.onload = function () {
                if (this.status >= 200 && this.status < 300) {
                    parserCompletionCallback(new AAIGridReader(this.response), xhr);
                }
                else {
                    Logger.log(Logger.LEVEL_WARNING, "AAIGrid retrieval failed (" + xhr.statusText + "): " + url);
                    parserCompletionCallback(null, xhr);
                }
            };

            xhr.onerror = function () {
                Logger.log(Logger.LEVEL_WARNING, "AAIGrid retrieval failed: " + url);
                parserCompletionCallback(null, xhr);
            };

            xhr.ontimeout = function () {
                Logger.log(Logger.LEVEL_WARNING, "AAIGrid retrieval timed out: " + url);
                parserCompletionCallback(null, xhr);
            };

            xhr.open("GET", url, true);
            xhr.responseType = 'arraybuffer';
            xhr.send(null);
        };

        return AAIGridReader;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
define('geom/Angle',[], function () {
    "use strict";
    /**
     * Provides constants and functions for working with angles.
     * @exports Angle
     */

    var Angle = {
        /**
         * Conversion factor for degrees to radians.
         * @constant
         */
        DEGREES_TO_RADIANS: Math.PI / 180.0,
        /**
         * Conversion factor for radians to degrees.
         * @constant
         */
        RADIANS_TO_DEGREES: 180.0 / Math.PI,
        /**
         * 2 pi.
         * @constant
         */
        TWO_PI: 2 * Math.PI,
        /**
         * pi / 2
         * @constant
         */
        HALF_PI: Math.PI / 2,

        /**
         * Normalizes a specified value to be within the range of [-180, 180] degrees.
         * @param {Number} degrees The value to normalize, in degrees.
         * @returns {Number} The specified value normalized to [-180, 180] degrees.
         */
        normalizedDegrees: function (degrees) {
            var angle = degrees % 360;

            return angle > 180 ? angle - 360 : angle < -180 ? 360 + angle : angle;
        },

        /**
         * Normalizes a specified value to be within the range of [-90, 90] degrees.
         * @param {Number} degrees The value to normalize, in degrees.
         * @returns {Number} The specified value normalized to the normal range of latitude.
         */
        normalizedDegreesLatitude: function (degrees) {
            var lat = degrees % 180;

            return lat > 90 ? 180 - lat : lat < -90 ? -180 - lat : lat;
        },

        /**
         * Normalizes a specified value to be within the range of [-180, 180] degrees.
         * @param {Number} degrees The value to normalize, in degrees.
         * @returns {Number} The specified value normalized to the normal range of longitude.
         */
        normalizedDegreesLongitude: function (degrees) {
            var lon = degrees % 360;

            return lon > 180 ? lon - 360 : lon < -180 ? 360 + lon : lon;
        },

        /**
         * Normalizes a specified value to be within the range of [-Pi, Pi] radians.
         * @param {Number} radians The value to normalize, in radians.
         * @returns {Number} The specified value normalized to [-Pi, Pi] radians.
         */
        normalizedRadians: function (radians) {
            var angle = radians % this.TWO_PI;

            return angle > Math.PI ? angle - this.TWO_PI : angle < -Math.PI ? this.TWO_PI + angle : angle;
        },

        /**
         * Normalizes a specified value to be within the range of [-Pi/2, Pi/2] radians.
         * @param {Number} radians The value to normalize, in radians.
         * @returns {Number} The specified value normalized to the normal range of latitude.
         */
        normalizedRadiansLatitude: function (radians) {
            var lat = radians % Math.PI;

            return lat > this.HALF_PI ? Math.PI - lat : lat < -this.HALF_PI ? -Math.PI - lat : lat;
        },

        /**
         * Normalizes a specified value to be within the range of [-Pi, Pi] radians.
         * @param {Number} radians The value to normalize, in radians.
         * @returns {Number} The specified value normalized to the normal range of longitude.
         */
        normalizedRadiansLongitude: function (radians) {
            var lon = radians % this.TWO_PI;

            return lon > Math.PI ? lon - this.TWO_PI : lon < -Math.PI ? this.TWO_PI + lon : lon;
        },

        /**
         * Indicates whether a specified value is within the normal range of latitude, [-90, 90].
         * @param {Number} degrees The value to test, in degrees.
         * @returns {Boolean} true if the value is within the normal range of latitude, otherwise false.
         */
        isValidLatitude: function (degrees) {
            return degrees >= -90 && degrees <= 90;
        },

        /**
         * Indicates whether a specified value is within the normal range of longitude, [-180, 180].
         * @param {Number} degrees The value to test, in degrees.
         * @returns {boolean} true if the value is within the normal range of longitude, otherwise false.
         */
        isValidLongitude: function (degrees) {
            return degrees >= -180 && degrees <= 180;
        },

        /**
         * Returns a string representation of a specified value in degrees.
         * @param {Number} degrees The value for which to compute the string.
         * @returns {String} The computed string, which is a decimal degrees value followed by the degree symbol.
         */
        toString: function (degrees) {
            return degrees.toString() + '\u00B0';
        },

        /**
         * Returns a decimal degrees string representation of a specified value in degrees.
         * @param {Number} degrees The value for which to compute the string.
         * @returns {String} The computed string, which is a decimal degrees value followed by the degree symbol.
         */
        toDecimalDegreesString: function (degrees) {
            return degrees.toString() + '\u00B0';
        },

        /**
         * Returns a degrees-minutes-seconds string representation of a specified value in degrees.
         * @param {Number} degrees The value for which to compute the string.
         * @returns {String} The computed string in degrees, minutes and decimal seconds.
         */
        toDMSString: function (degrees) {
            var sign,
                temp,
                d,
                m,
                s;

            sign = degrees < 0 ? -1 : 1;
            temp = sign * degrees;
            d = Math.floor(temp);
            temp = (temp - d) * 60;
            m = Math.floor(temp);
            temp = (temp - m) * 60;
            s = Math.round(temp);

            if (s == 60) {
                m++;
                s = 0;
            }
            if (m == 60) {
                d++;
                m = 0;
            }

            return (sign == -1 ? "-" : "") + d + "\u00B0" + " " + m + "\u2019" + " " + s + "\u201D";
        },

        /**
         * Returns a degrees-minutes string representation of a specified value in degrees.
         * @param {Number} degrees The value for which to compute the string.
         * @returns {String} The computed string in degrees and decimal minutes.
         */
        toDMString: function (degrees) {
            var sign,
                temp,
                d,
                m,
                s,
                mf;

            sign = degrees < 0 ? -1 : 1;
            temp = sign * degrees;
            d = Math.floor(temp);
            temp = (temp - d) * 60;
            m = Math.floor(temp);
            temp = (temp - m) * 60;
            s = Math.round(temp);

            if (s == 60) {
                m++;
                s = 0;
            }
            if (m == 60) {
                d++;
                m = 0;
            }

            mf = s == 0 ? m : m + s / 60;

            return (sign == -1 ? "-" : "") + d + "\u00B0" + " " + mf + "\u2019";
        }
    };

    return Angle;
});
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports Color
 */
define('util/Color',[
        '../util/Logger'
    ],
    function (Logger) {
        "use strict";

        /**
         * Constructs a color from red, green, blue and alpha values.
         * @alias Color
         * @constructor
         * @classdesc Represents a red, green, blue, alpha, color.
         * @param {Number} red The red component, a number between 0 and 1.
         * @param {Number} green The green component, a number between 0 and 1.
         * @param {Number} blue The blue component, a number between 0 and 1.
         * @param {Number} alpha The alpha component, a number between 0 and 1.
         */
        var Color = function (red, green, blue, alpha) {

            /**
             * This color's red component, a number between 0 and 1.
             * @type {Number}
             */
            this.red = red;

            /**
             * This color's green component, a number between 0 and 1.
             * @type {Number}
             */
            this.green = green;

            /**
             * This color's blue component, a number between 0 and 1.
             * @type {Number}
             */
            this.blue = blue;

            /**
             * This color's alpha component, a number between 0 and 1.
             * @type {Number}
             */
            this.alpha = alpha;
        };

        /**
         * The color white.
         * @type {Color}
         * @constant
         */
        Color.WHITE = new Color(1, 1, 1, 1);

        /**
         * The color black.
         * @type {Color}
         * @constant
         */
        Color.BLACK = new Color(0, 0, 0, 1);

        /**
         * The color red.
         * @type {Color}
         * @constant
         */
        Color.RED = new Color(1, 0, 0, 1);

        /**
         * The color green.
         * @type {Color}
         * @constant
         */
        Color.GREEN = new Color(0, 1, 0, 1);

        /**
         * The color blue.
         * @type {Color}
         * @constant
         */
        Color.BLUE = new Color(0, 0, 1, 1);

        /**
         * The color cyan.
         * @type {Color}
         * @constant
         */
        Color.CYAN = new Color(0, 1, 1, 1);

        /**
         * The color yellow.
         * @type {Color}
         * @constant
         */
        Color.YELLOW = new Color(1, 1, 0, 1);

        /**
         * The color magenta.
         * @type {Color}
         * @constant
         */
        Color.MAGENTA = new Color(1, 0, 1, 1);

        /**
         * A light gray (75% white).
         * @type {Color}
         */
        Color.LIGHT_GRAY = new Color(0.75, 0.75, 0.75, 1);

        /**
         * A medium gray (50% white).
         * @type {Color}
         */
        Color.MEDIUM_GRAY = new Color(0.5, 0.5, 0.5, 1);

        /**
         * A dark gray (25% white).
         * @type {Color}
         */
        Color.DARK_GRAY = new Color(0.25, 0.25, 0.25, 1);

        /**
         * A transparent color.
         * @type {Color}
         */
        Color.TRANSPARENT = new Color(0, 0, 0, 0);

        /**
         * Assigns the components of this color.
         * @param {Number} red The red component, a number between 0 and 1.
         * @param {Number} green The green component, a number between 0 and 1.
         * @param {Number} blue The blue component, a number between 0 and 1.
         * @param {Number} alpha The alpha component, a number between 0 and 1.
         * @returns {Color} This color with the specified components assigned.
         */
        Color.prototype.set = function (red, green, blue, alpha) {
            this.red = red;
            this.green = green;
            this.blue = blue;
            this.alpha = alpha;

            return this;
        };

        /**
         * Copies the components of a specified color to this color.
         * @param {Color} color The color to copy.
         * @returns {Color} This color set to the red, green, blue and alpha values of the specified color.
         * @throws {ArgumentError} If the specified color is null or undefined.
         */
        Color.prototype.copy = function (color) {
            if (!color) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Color", "copy", "missingColor"));
            }

            this.red = color.red;
            this.green = color.green;
            this.blue = color.blue;
            this.alpha = color.alpha;

            return this;
        };

        /**
         * Create a copy of this color.
         * @returns {Color} A new instance containing the color components of this color.
         */
        Color.prototype.clone = function () {
            return new Color(this.red, this.green, this.blue, this.alpha);
        };

        /**
         * Returns this color's components premultiplied by this color's alpha component.
         * @param {Float32Array} array A pre-allocated array in which to return the color components.
         * @returns {Float32Array} This colors premultiplied components as an array, in the order RGBA.
         */
        Color.prototype.premultipliedComponents = function (array) {
            var a = this.alpha;

            array[0] = this.red * a;
            array[1] = this.green * a;
            array[2] = this.blue * a;
            array[3] = a;

            return array;
        };

        /**
         * Construct a color from an array of color components expressed as byte values.
         * @param {Uint8Array} bytes A four-element array containing the red, green, blue and alpha color
         * components each in the range [0, 255];
         * @returns {Color} The constructed color.
         */
        Color.colorFromByteArray = function (bytes) {
            return new Color(bytes[0] / 255, bytes[1] / 255, bytes[2] / 255, bytes[3] / 255);
        };

        /**
         * Construct a color from specified color components expressed as byte values.
         * @param {number} redByte The red component in the range [0, 255].
         * @param {number} greenByte The green component in the range [0, 255].
         * @param {number} blueByte The blue component in the range [0, 255].
         * @param {number} alphaByte The alpha component in the range [0, 255].
         * @returns {Color} The constructed color.
         */
        Color.colorFromBytes = function (redByte, greenByte, blueByte, alphaByte) {
            return new Color(redByte / 255, greenByte / 255, blueByte / 255, alphaByte / 255);
        };

        Color.colorFromHex = function (color) {
            var red = parseInt(color.substring(0, 2), 16);
            var green = parseInt(color.substring(2, 4), 16);
            var blue = parseInt(color.substring(4, 6), 16);
            var alpha = parseInt(color.substring(6, 8), 16);
            return Color.colorFromBytes(red, green, blue, alpha);
        };

        Color.colorFromKmlHex = function (color) {
            var alpha = parseInt(color.substring(0, 2), 16);
            var blue = parseInt(color.substring(2, 4), 16);
            var green = parseInt(color.substring(4, 6), 16);
            var red = parseInt(color.substring(6, 8), 16);
            return Color.colorFromBytes(red, green, blue, alpha);
        };

        /**
         * Computes and sets this color to the next higher RBG color. If the color overflows, this color is set to
         * (1 / 255, 0, 0, *), where * indicates the current alpha value.
         * @returns {Color} This color, set to the next possible color.
         */
        Color.prototype.nextColor = function () {
            var rb = Math.round(this.red * 255),
                gb = Math.round(this.green * 255),
                bb = Math.round(this.blue * 255);

            if (rb < 255) {
                this.red = (rb + 1) / 255;
            } else if (gb < 255) {
                this.red = 0;
                this.green = (gb + 1) / 255;
            } else if (bb < 255) {
                this.red = 0;
                this.green = 0;
                this.blue = (bb + 1) / 255;
            } else {
                this.red = 1 / 255;
                this.green = 0;
                this.blue = 0;
            }

            return this;
        };

        /**
         * Indicates whether this color is equal to a specified color after converting the floating-point component
         * values of each color to byte values.
         * @param {Color} color The color to test,
         * @returns {Boolean} true if the colors are equal, otherwise false.
         */
        Color.prototype.equals = function (color) {
            var rbA = Math.round(this.red * 255),
                gbA = Math.round(this.green * 255),
                bbA = Math.round(this.blue * 255),
                abA = Math.round(this.alpha * 255),
                rbB = Math.round(color.red * 255),
                gbB = Math.round(color.green * 255),
                bbB = Math.round(color.blue * 255),
                abB = Math.round(color.alpha * 255);

            return rbA === rbB && gbA === gbB && bbA === bbB && abA === abB;
        };

        /**
         * Indicates whether this color is equal to another color expressed as an array of bytes.
         * @param {Uint8Array} bytes The red, green, blue and alpha color components.
         * @returns {Boolean} true if the colors are equal, otherwise false.
         */
        Color.prototype.equalsBytes = function (bytes) {
            var rb = Math.round(this.red * 255),
                gb = Math.round(this.green * 255),
                bb = Math.round(this.blue * 255),
                ab = Math.round(this.alpha * 255);

            return rb === bytes[0] && gb === bytes[1] && bb === bytes[2] && ab === bytes[3];
        };

        /**
         * Returns a string representation of this color, indicating the byte values corresponding to this color's
         * floating-point component values.
         * @returns {String}
         */
        Color.prototype.toByteString = function () {
            var rb = Math.round(this.red * 255),
                gb = Math.round(this.green * 255),
                bb = Math.round(this.blue * 255),
                ab = Math.round(this.alpha * 255);

            return "(" + rb + "," + gb + "," + bb + "," + ab + ")";
        };

        /**
         * Create a hex color string that CSS can use. Optionally, inhibit capturing alpha,
         * because some uses reject a four-component color specification.
         * @param {Boolean} isUsingAlpha Enable the use of an alpha component.
         * @returns {string} A color string suitable for CSS.
         * @deprecated since version 0.10.0, use toCssColorString for valid CSS color strings
         */
        Color.prototype.toHexString = function (isUsingAlpha) {
            // Use Math.ceil() to get 0.75 to map to 0xc0. This is important if the display is dithering.
            var redHex = Math.ceil(this.red * 255).toString(16),
                greenHex = Math.ceil(this.green * 255).toString(16),
                blueHex = Math.ceil(this.blue * 255).toString(16),
                alphaHex = Math.ceil(this.alpha * 255).toString(16);

            var result = "#";
            result += (redHex.length < 2) ? ('0' + redHex) : redHex;
            result += (greenHex.length < 2) ? ('0' + greenHex) : greenHex;
            result += (blueHex.length < 2) ? ('0' + blueHex) : blueHex;
            if (isUsingAlpha) {
                result += (alphaHex.length < 2) ? ('0' + alphaHex) : alphaHex;
            }

            return result;
        };

        /**
         * Create a rgba color string that conforms to CSS Color Module Level 3 specification.
         * @returns {string} A color string suitable for CSS.
         */
        Color.prototype.toCssColorString = function () {
            var red = Math.round(this.red * 255),
                green = Math.round(this.green * 255),
                blue = Math.round(this.blue * 255);

            // Per the CSS Color Module Level 3 specification, alpha is expressed as floating point value between 0 - 1
            return 'rgba(' + red + ', ' + green + ', ' + blue + ', ' + this.alpha + ')';
        };

        return Color;
    });

/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports Font
 */
define('util/Font',[
        '../error/ArgumentError',
        '../util/Color',
        '../util/Logger'
    ],
    function (ArgumentError,
              Color,
              Logger) {
        "use strict";

        /**
         * Construct a font descriptor. See the individual attribute descriptions below for possible parameter values.
         * @param {Number} size The size of font.
         * @param {String} style The style of the font.
         * @param {String} variant The variant of the font.
         * @param {String} weight The weight of the font.
         * @param {String} family The family of the font.
         * @param {String} horizontalAlignment The vertical alignment of the font.
         * @alias Font
         * @constructor
         * @classdesc Holds attributes controlling the style, size and other attributes of {@link Text} shapes and
         * the textual features of {@link Placemark} and other shapes. The values used for these attributes are those
         * defined by the [CSS Font property]{@link http://www.w3schools.com/cssref/pr_font_font.asp}.
         */
        var Font = function (size, style, variant, weight, family, horizontalAlignment) {
            /*
             * All properties of Font are intended to be private and must be accessed via public getters and setters.
             */

            if (!size) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Font", "constructor",
                    "missingSize"));
            }
            else if (size <= 0) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Font", "constructor",
                    "invalidSize"));
            }
            else {
                this._size = size;
            }

            this.style = style || "normal";
            this.variant = variant || "normal";
            this.weight = weight || "normal";
            this.family = family || "sans-serif";
            this.horizontalAlignment = horizontalAlignment || "center";
        };

        Object.defineProperties(Font.prototype, {
            /**
             * The font size.
             * @memberof Font.prototype
             * @type Number
             */
            size: {
                get: function () {
                    return this._size;
                },
                set: function (value) {
                    this._fontString = null;
                    this._size = value;
                }
            },
            /**
             * The font style.
             * See [CSS font-style]{@link http://www.w3schools.com/cssref/pr_font_font-style.asp} for defined values.
             * @memberof Font.prototype
             * @type {String}
             * @default "normal"
             */
            style: {
                get: function () {
                    return this._style;
                },
                set: function (value) {
                    this._fontString = null;
                    this._style = value;
                }
            },
            /**
             * The font variant.
             * See [CSS font-variant]{@link http://www.w3schools.com/cssref/pr_font_font-variant.asp} for defined values.
             * @memberof Font.prototype
             * @type {String}
             * @default "normal"
             */
            variant: {
                get: function () {
                    return this._variant;
                },
                set: function (value) {
                    this._fontString = null;
                    this._variant = value;
                }
            },
            /**
             * The font weight.
             * See [CSS font-weight]{@link http://www.w3schools.com/cssref/pr_font_weight.asp} for defined values.
             * @memberof Font.prototype
             * @type {String}
             * @default "normal"
             */
            weight: {
                get: function () {
                    return this._weight;
                },
                set: function (value) {
                    this._fontString = null;
                    this._weight = value;
                }
            },
            /**
             * The font family.
             * See [CSS font-family]{@link http://www.w3schools.com/cssref/pr_font_font-family.asp} for defined values.
             * @memberof Font.prototype
             * @type {String}
             * @default "sans-serif"
             */
            family: {
                get: function () {
                    return this._family;
                },
                set: function (value) {
                    this._fontString = null;
                    this._family = value;
                }
            },
            /**
             * The horizontal alignment of the font.
             * Recognized values are "left", "center" and "right".
             * @memberof Font.prototype
             * @type {String}
             * @default "center"
             */
            horizontalAlignment: {
                get: function () {
                    return this._horizontalAlignment;
                },
                set: function (value) {
                    this._toString = null;
                    this._horizontalAlignment = value;
                }
            },

            /**
             * A string representing this font's style, weight, size and family properties, suitable for
             * passing directly to a 2D canvas context.
             * @memberof Font.prototype
             */
            fontString: {
                get: function () {
                    if (!this._fontString) {
                        this._fontString =
                            this._style + " " +
                            this.variant + " " +
                            this._weight + " " +
                            this._size.toString() + "px " +
                            this._family;
                    }
                    return this._fontString;
                }
            }
        });

        /**
         * Returns a string representation of this object.
         * @returns {String} A string representation of this object.
         */
        Font.prototype.toString = function () {
            if (!this._toString || !this._fontString) {
                this._toString = this.fontString + " " + this.horizontalAlignment;
            }
            return this._toString;
        };

        return Font;
    }
);
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
define('util/Insets',['../error/ArgumentError',
        '../util/Logger'
    ],
    function (ArgumentError,
              Logger) {
        "use strict";

        /**
         * Constructs an Insets object that is a representation of the borders of a container.
         * It specifies the space that a container must leave at each of its edges.
         * @alias Insets
         * @param {Number} top The inset from the top.
         * @param {Number} left The inset from the left.
         * @param {Number} bottom The inset from the bottom.
         * @param {Number} right The inset from the right.
         * @constructor
         */
        var Insets = function (top, left, bottom, right) {

            if (arguments.length !== 4) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Insets", "constructor", "invalidArgumentCount"));
            }

            // These are all documented with their property accessors below.
            this._top = top;
            this._left = left;
            this._bottom = bottom;
            this._right = right;
        };

        /**
         * Set top, left, bottom, and right to the specified values.
         * @param {Number} top The inset from the top.
         * @param {Number} left The inset from the left.
         * @param {Number} bottom The inset from the bottom.
         * @param {Number} right The inset from the right.
         */
        Insets.prototype.set = function (top, left, bottom, right) {
            this._top = top;
            this._left = left;
            this._bottom = bottom;
            this._right = right;
        };

        /**
         * Creates a new copy of this insets with identical property values.
         * @returns {Insets} A new insets instance with its property values the same as this one's.
         */
        Insets.prototype.clone = function () {
            return new Insets(this._top, this._left, this._bottom, this._right);
        };

        /**
         * Returns a string representation of this object.
         * @returns {String} A string representation of this object.
         */
        Insets.prototype.toString = function () {
            return this._top + " " + this._left + " " + this._bottom + " " + this._right;
        };

        Object.defineProperties(Insets.prototype, {

            /**
             * Indicates the the inset from the top.
             * @type {Number}
             * @memberof Insets.prototype
             */
            top: {
                get: function () {
                    return this._top;
                },
                set: function (value) {
                    this._top = value;
                }
            },

            /**
             * Indicates the the inset from the left.
             * @type {Number}
             * @memberof Insets.prototype
             */
            left: {
                get: function () {
                    return this._left;
                },
                set: function (value) {
                    this._left = value;
                }
            },

            /**
             * Indicates the the inset from the bottom.
             * @type {Number}
             * @memberof Insets.prototype
             */
            bottom: {
                get: function () {
                    return this._bottom;
                },
                set: function (value) {
                    this._bottom = value;
                }
            },

            /**
             * Indicates the the inset from the right.
             * @type {Number}
             * @memberof Insets.prototype
             */
            right: {
                get: function () {
                    return this._right;
                },
                set: function (value) {
                    this._right = value;
                }
            }

        });

        return Insets;
    }
);
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
define('geom/Vec3',[
        '../util/Logger',
        '../error/ArgumentError'
    ],
    function (Logger,
              ArgumentError) {
        "use strict";

        /**
         * Constructs a three-component vector.
         * @alias Vec3
         * @classdesc Represents a three-component vector. Access the X component of the vector as v[0], the Y
         * component as v[1] and the Z component as v[2].
         * @augments Float64Array
         * @param {Number} x X component of vector.
         * @param {Number} y Y component of vector.
         * @param {Number} z Z component of vector.
         * @constructor
         */
        var Vec3 = function Vec3(x, y, z) {
            this[0] = x;
            this[1] = y;
            this[2] = z;
        };

        // Vec3 extends Float64Array.
        Vec3.prototype = new Float64Array(3);

        /**
         * A vector corresponding to the origin.
         * @type {Vec3}
         */
        Vec3.ZERO = new Vec3(0, 0, 0);

        /**
         * Computes the average of a specified array of vectors.
         * @param {Vec3[]} vectors The vectors whose average to compute.
         * @param {Vec3} result A pre-allocated Vec3 in which to return the computed average.
         * @returns {Vec3} The result argument set to the average of the specified array of vectors.
         * @throws {ArgumentError} If the specified array of vectors is null, undefined or empty or the specified
         * result argument is null or undefined.
         */
        Vec3.average = function (vectors, result) {
            if (!vectors || vectors.length < 1) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec3", "average", "missingArray"));
            }

            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec3", "average", "missingResult"));
            }

            var count = vectors.length,
                vec;

            result[0] = 0;
            result[1] = 0;
            result[2] = 0;

            for (var i = 0, len = vectors.length; i < len; i++) {
                vec = vectors[i];

                result[0] += vec[0] / count;
                result[1] += vec[1] / count;
                result[2] += vec[2] / count;
            }

            return result;
        };

        /**
         * Computes the average of a specified array of points packed into a single array.
         * @param {Float32Array | Float64Array | Number[]} points The points whose average to compute.
         * @param {Vec3} result A pre-allocated Vec3 in which to return the computed average.
         * @returns {Vec3} The result argument set to the average of the specified array of points.
         * @throws {ArgumentError} If the specified array of points is null, undefined or empty or the result argument
         * is null or undefined.
         */
        Vec3.averageOfBuffer = function (points, result) {
            if (!points || points.length < 1) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec3", "averageBuffer", "missingArray"));
            }

            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec3", "averageBuffer", "missingResult"));
            }

            var count = points.length / 3;

            result[0] = 0;
            result[1] = 0;
            result[2] = 0;

            for (var i = 0; i < count; i++) {
                result[0] += points[i * 3] / count;
                result[1] += points[i * 3 + 1] / count;
                result[2] += points[i * 3 + 2] / count;
            }

            return result;
        };

        /**
         * Indicates whether three vectors are colinear.
         * @param {Vec3} a The first vector.
         * @param {Vec3} b The second vector.
         * @param {Vec3} c The third vector.
         * @returns {Boolean} true if the vectors are colinear, otherwise false.
         * @throws {ArgumentError} If any of the specified vectors are null or undefined.
         */
        Vec3.areColinear = function (a, b, c) {
            if (!a || !b || !c) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec3", "areColinear", "missingVector"));
            }

           var ab = new Vec3(a[0]-b[0],a[1]-b[1],a[2]-b[2]).normalize(),
               bc = new Vec3(c[0]-b[0],c[1]-b[1],c[2]-b[2]).normalize();

            // ab and bc are considered colinear if their dot product is near +/-1.
            return Math.abs(ab.dot(bc)) > 0.999;
        };

        /**
         * Computes the normal vector of a specified triangle.
         *
         * @param {Vec3} a The triangle's first vertex.
         * @param {Vec3} b The triangle's second vertex.
         * @param {Vec3} c The triangle's third vertex.
         * @returns {Vec3} The triangle's unit-normal vector.
         * @throws {ArgumentError} If any of the specified vectors are null or undefined.
         */
        Vec3.computeTriangleNormal = function (a, b, c) {
            if (!a || !b || !c) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec3", "areColinear", "missingVector"));
            }

            var x = ((b[1] - a[1]) * (c[2] - a[2])) - ((b[2] - a[2]) * (c[1] - a[1])),
                y = ((b[2] - a[2]) * (c[0] - a[0])) - ((b[0] - a[0]) * (c[2] - a[2])),
                z = ((b[0] - a[0]) * (c[1] - a[1])) - ((b[1] - a[1]) * (c[0] - a[0])),
                length = (x * x) + (y * y) + (z * z);

            if (length === 0) {
                return new Vec3(x, y, z);
            }

            length = Math.sqrt(length);

            return new Vec3(x / length, y / length, z / length);
        };

        /**
         * Finds three non-colinear points in an array of coordinates.
         *
         * @param {Number[]} coords The coordinates, in the order x0, y0, z0, x1, y1, z1, ...
         * @param {Number} stride The number of numbers between successive points. 0 indicates that the points
         * are arranged one immediately after the other, as would the value 3.
         * @returns {Vec3[]} Three non-colinear points from the input array of coordinates, or null if three
         * non-colinear points could not be found or the specified coordinates array is null, undefined or
         * contains fewer than three points.
         */
        Vec3.findThreeIndependentVertices = function (coords, stride) {
            var xstride = (stride && stride > 0) ? stride : 3;

            if (!coords || coords.length < 3 * xstride) {
                return null;
            }

            var a = new Vec3(coords[0], coords[1], coords[2]),
                b = null,
                c = null,
                k = xstride;

            for (; k < coords.length; k += xstride) {
                b = new Vec3(coords[k], coords[k + 1], coords[k + 2]);
                if (!(b[0] === a[0] && b[1] === a[1] && b[2] === a[2])) {
                    break;
                }
                b = null;
            }

            if (!b) {
                return null;
            }

            for (k += xstride; k < coords.length; k += xstride) {
                c = new Vec3(coords[k], coords[k + 1], coords[k + 2]);

                // if c is not coincident with a or b, and the vectors ab and bc are not colinear, break and
                // return a, b, c.
                if (!((c[0] === a[0] && c[1] === a[1] && c[2] === a[2])
                    || (c[0] === b[0] && c[1] === b[1] && c[2] === b[2]))) {
                    if (!Vec3.areColinear(a, b, c))
                        break;
                }

                c = null;
            }

            return c ? [a, b, c] : null;
        };

        /**
         * Computes a unit-normal vector for a buffer of coordinate triples. The normal vector is computed from the
         * first three non-colinear points in the buffer.
         *
         * @param {Number[]} coords The coordinates, in the order x0, y0, z0, x1, y1, z1, ...
         * @param {Number} stride The number of numbers between successive points. 0 indicates that the points
         * are arranged one immediately after the other, as would the value 3.
         * @returns {Vec3} The computed unit-length normal vector.
         */
        Vec3.computeBufferNormal = function (coords, stride) {
            var vertices = Vec3.findThreeIndependentVertices(coords, stride);

            return vertices ? Vec3.computeTriangleNormal(vertices[0], vertices[1], vertices[2]) : null;
        };

        /**
         * Creates a new Vec3 vector from a Vec3 src.
         * @param {Vec3} src The Vec3 src vector top copy values from.
         */
        Vec3.fromVec3 = function(src) {
            return new Vec3(src[0], src[1], src[2]);
        };

        Vec3.zero = function() {
            return new Vec3(0, 0, 0);
        };

        /**
         * Assigns the components of this vector.
         * @param {Number} x The X component of the vector.
         * @param {Number} y The Y component of the vector.
         * @param {Number} z The Z component of the vector.
         * @returns {Vec3} This vector with the specified components assigned.
         */
        Vec3.prototype.set = function (x, y, z) {
            this[0] = x;
            this[1] = y;
            this[2] = z;

            return this;
        };

        /**
         * Copies the components of a specified vector to this vector.
         * @param {Vec3} vector The vector to copy.
         * @returns {Vec3} This vector set to the X, Y and Z values of the specified vector.
         * @throws {ArgumentError} If the specified vector is null or undefined.
         */
        Vec3.prototype.copy = function (vector) {
            if (!vector) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec3", "copy", "missingVector"));
            }

            this[0] = vector[0];
            this[1] = vector[1];
            this[2] = vector[2];

            return this;
        };

        /**
         * Indicates whether the components of this vector are identical to those of a specified vector.
         * @param {Vec3} vector The vector to test.
         * @returns {Boolean} true if the components of this vector are equal to those of the specified one,
         * otherwise false.
         */
        Vec3.prototype.equals = function (vector) {
            return this[0] === vector[0] && this[1] === vector[1] && this[2] === vector[2];
        };

        /**
         * Adds a specified vector to this vector.
         * @param {Vec3} addend The vector to add.
         * @returns {Vec3} This vector after adding the specified vector to it.
         * @throws {ArgumentError} If the addend is null or undefined.
         */
        Vec3.prototype.add = function (addend) {
            if (!addend) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec3", "add", "missingVector"));
            }

            this[0] += addend[0];
            this[1] += addend[1];
            this[2] += addend[2];

            return this;
        };

        /**
         * Subtracts a specified vector from this vector.
         * @param {Vec3} subtrahend The vector to subtract
         * @returns {Vec3} This vector after subtracting the specified vector from it.
         * @throws {ArgumentError} If the subtrahend is null or undefined.
         */
        Vec3.prototype.subtract = function (subtrahend) {
            if (!subtrahend) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec3", "subtract", "missingVector"));
            }

            this[0] -= subtrahend[0];
            this[1] -= subtrahend[1];
            this[2] -= subtrahend[2];

            return this;
        };

        /**
         * Multiplies this vector by a scalar.
         * @param {Number} scalar The scalar to multiply this vector by.
         * @returns {Vec3} This vector multiplied by the specified scalar.
         */
        Vec3.prototype.multiply = function (scalar) {
            this[0] *= scalar;
            this[1] *= scalar;
            this[2] *= scalar;

            return this;
        };

        /**
         * Divides this vector by a scalar.
         * @param {Number} divisor The scalar to divide this vector by.
         * @returns {Vec3} This vector divided by the specified scalar.
         */
        Vec3.prototype.divide = function (divisor) {
            this[0] /= divisor;
            this[1] /= divisor;
            this[2] /= divisor;

            return this;
        };

        /**
         * Multiplies this vector by a 4x4 matrix. The multiplication is performed with an implicit W component of 1.
         * The resultant W component of the product is then divided through the X, Y, and Z components.
         *
         * @param {Matrix} matrix The matrix to multiply this vector by.
         * @returns {Vec3} This vector multiplied by the specified matrix.
         * @throws ArgumentError If the specified matrix is null or undefined.
         */
        Vec3.prototype.multiplyByMatrix = function (matrix) {
            if (!matrix) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec3", "multiplyByMatrix", "missingMatrix"));
            }

            var x = matrix[0] * this[0] + matrix[1] * this[1] + matrix[2] * this[2] + matrix[3],
                y = matrix[4] * this[0] + matrix[5] * this[1] + matrix[6] * this[2] + matrix[7],
                z = matrix[8] * this[0] + matrix[9] * this[1] + matrix[10] * this[2] + matrix[11],
                w = matrix[12] * this[0] + matrix[13] * this[1] + matrix[14] * this[2] + matrix[15];

            this[0] = x / w;
            this[1] = y / w;
            this[2] = z / w;

            return this;
        };

        /**
         * Mixes (interpolates) a specified vector with this vector, modifying this vector.
         * @param {Vec3} vector The vector to mix with this one.
         * @param {Number} weight The relative weight of this vector.
         * @returns {Vec3} This vector modified to the mix of itself and the specified vector.
         * @throws {ArgumentError} If the specified vector is null or undefined.
         */
        Vec3.prototype.mix = function (vector, weight) {
            if (!vector) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec3", "mix", "missingVector"));
            }

            var w0 = 1 - weight,
                w1 = weight;

            this[0] = this[0] * w0 + vector[0] * w1;
            this[1] = this[1] * w0 + vector[1] * w1;
            this[2] = this[2] * w0 + vector[2] * w1;

            return this;
        };

        /**
         * Negates the components of this vector.
         * @returns {Vec3} This vector, negated.
         */
        Vec3.prototype.negate = function () {
            this[0] = -this[0];
            this[1] = -this[1];
            this[2] = -this[2];

            return this;
        };

        /**
         * Computes the scalar dot product of this vector and a specified vector.
         * @param {Vec3} vector The vector to multiply.
         * @returns {Number} The dot product of the two vectors.
         * @throws {ArgumentError} If the specified vector is null or undefined.
         */
        Vec3.prototype.dot = function (vector) {
            if (!vector) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec3", "dot", "missingVector"));
            }

            return this[0] * vector[0] +
                this[1] * vector[1] +
                this[2] * vector[2];
        };

        /**
         * Computes the cross product of this vector and a specified vector, modifying this vector.
         * @param {Vec3} vector The vector to cross with this vector.
         * @returns {Vec3} This vector set to the cross product of itself and the specified vector.
         * @throws {ArgumentError} If the specified vector is null or undefined.
         */
        Vec3.prototype.cross = function (vector) {
            if (!vector) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec3", "cross", "missingVector"));
            }

            var x = this[1] * vector[2] - this[2] * vector[1],
                y = this[2] * vector[0] - this[0] * vector[2],
                z = this[0] * vector[1] - this[1] * vector[0];

            this[0] = x;
            this[1] = y;
            this[2] = z;

            return this;
        };

        /**
         * Computes the squared magnitude of this vector.
         * @returns {Number} The squared magnitude of this vector.
         */
        Vec3.prototype.magnitudeSquared = function () {
            return this.dot(this);
        };

        /**
         * Computes the magnitude of this vector.
         * @returns {Number} The magnitude of this vector.
         */
        Vec3.prototype.magnitude = function () {
            return Math.sqrt(this.magnitudeSquared());
        };

        /**
         * Normalizes this vector to a unit vector.
         * @returns {Vec3} This vector, normalized.
         */
        Vec3.prototype.normalize = function () {
            var magnitude = this.magnitude(),
                magnitudeInverse = 1 / magnitude;

            this[0] *= magnitudeInverse;
            this[1] *= magnitudeInverse;
            this[2] *= magnitudeInverse;

            return this;
        };

        /**
         * Computes the squared distance from this vector to a specified vector.
         * @param {Vec3} vector The vector to compute the distance to.
         * @returns {Number} The squared distance between the vectors.
         * @throws {ArgumentError} If the specified vector is null or undefined.
         */
        Vec3.prototype.distanceToSquared = function (vector) {
            if (!vector) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec3", "distanceToSquared", "missingVector"));
            }

            var dx = this[0] - vector[0],
                dy = this[1] - vector[1],
                dz = this[2] - vector[2];

            return dx * dx + dy * dy + dz * dz;
        };

        /**
         * Computes the distance from this vector to another vector.
         * @param {Vec3} vector The vector to compute the distance to.
         * @returns {number} The distance between the vectors.
         * @throws {ArgumentError} If the specified vector is null or undefined.
         */
        Vec3.prototype.distanceTo = function (vector) {
            if (!vector) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec3", "distanceTo", "missingVector"));
            }

            return Math.sqrt(this.distanceToSquared(vector));
        };

        /**
         * Swaps this vector with that vector. This vector's components are set to the values of the specified
         * vector's components, and the specified vector's components are set to the values of this vector's components.
         * @param {Vec3} that The vector to swap.
         * @returns {Vec3} This vector set to the values of the specified vector.
         */
        Vec3.prototype.swap = function (that) {
            var tmp = this[0];
            this[0] = that[0];
            that[0] = tmp;

            tmp = this[1];
            this[1] = that[1];
            that[1] = tmp;

            tmp = this[2];
            this[2] = that[2];
            that[2] = tmp;

            return this;
        };

        /**
         * Returns a string representation of this vector.
         * @returns {String} A string representation of this vector, in the form "(x, y, z)".
         */
        Vec3.prototype.toString = function () {
            return "(" + this[0] + ", " + this[1] + ", " + this[2] + ")";
        };

        return Vec3;
    }
);
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
define('geom/Vec2',[
        '../util/Logger',
        '../error/ArgumentError',
        '../geom/Vec3'
    ],
    function (Logger,
              ArgumentError,
              Vec3) {
        "use strict";

        /**
         * Constructs a two-component vector.
         * @alias Vec2
         * @classdesc Represents a two-component vector. Access the X component of the vector as v[0] and the Y
         * component as v[1].
         * @augments Float64Array
         * @param {Number} x X component of vector.
         * @param {Number} y Y component of vector.
         * @constructor
         */
        var Vec2 = function Vec2(x, y) {
            this[0] = x;
            this[1] = y;
        };

        // Vec2 inherits from Float64Array.
        Vec2.prototype = new Float64Array(2);

        /**
         * Assigns the components of this vector.
         * @param {Number} x The X component of the vector.
         * @param {Number} y The Y component of the vector.
         * @returns {Vec2} This vector with the specified components assigned.
         */
        Vec2.prototype.set = function (x, y) {
            this[0] = x;
            this[1] = y;

            return this;
        };

        /**
         * Copies the components of a specified vector to this vector.
         * @param {Vec2} vector The vector to copy.
         * @returns {Vec2} This vector set to the values of the specified vector.
         * @throws {ArgumentError} If the specified vector is null or undefined.
         */
        Vec2.prototype.copy = function (vector) {
            if (!vector) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec2", "copy", "missingVector"));
            }

            this[0] = vector[0];
            this[1] = vector[1];

            return this;
        };

        /**
         * Indicates whether the X and Y components of this vector are identical to those of a specified vector.
         * @param {Vec2} vector The vector to test.
         * @returns {Boolean} true if this vector's components are equal to those of the specified vector,
         * otherwise false.
         */
        Vec2.prototype.equals = function (vector) {
            return this[0] === vector[0] && this[1] === vector[1];
        };

        /**
         * Computes the average of a specified array of vectors.
         * @param {Vec2[]} vectors The vectors whose average to compute.
         * @param {Vec2} result A pre-allocated Vec2 in which to return the computed average.
         * @returns {Vec2} The result argument set to the average of the specified lists of vectors.
         * @throws {ArgumentError} If the specified array of vectors is null, undefined or empty, or the specified
         * result argument is null or undefined.
         */
        Vec2.average = function (vectors, result) {
            if (!vectors || vectors.length < 1) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec2", "average", "missingArray"));
            }

            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec2", "average", "missingResult"));
            }

            var count = vectors.length,
                vec;

            result[0] = 0;
            result[1] = 0;

            for (var i = 0, len = vectors.length; i < len; i++) {
                vec = vectors[i];

                result[0] += vec[0] / count;
                result[1] += vec[1] / count;
            }

            return result;
        };

        /**
         * Adds a vector to this vector.
         * @param {Vec2} addend The vector to add to this one.
         * @returns {Vec2} This vector after adding the specified vector to it.
         * @throws {ArgumentError} If the specified addend is null or undefined.
         */
        Vec2.prototype.add = function (addend) {
            if (!addend) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec2", "add", "missingVector"));
            }

            this[0] += addend[0];
            this[1] += addend[1];

            return this;
        };

        /**
         * Subtracts a vector from this vector.
         * @param {Vec2} subtrahend The vector to subtract from this one.
         * @returns {Vec2} This vector after subtracting the specified vector from it.
         * @throws {ArgumentError} If the subtrahend is null or undefined.
         */
        Vec2.prototype.subtract = function (subtrahend) {
            if (!subtrahend) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec2", "subtract", "missingVector"));
            }

            this[0] -= subtrahend[0];
            this[1] -= subtrahend[1];

            return this;
        };

        /**
         * Multiplies this vector by a scalar.
         * @param {Number} scalar The scalar to multiply this vector by.
         * @returns {Vec2} This vector multiplied by the specified scalar.
         */
        Vec2.prototype.multiply = function (scalar) {
            this[0] *= scalar;
            this[1] *= scalar;

            return this;
        };

        /**
         * Divide this vector by a scalar.
         * @param {Number} divisor The scalar to divide this vector by.
         * @returns {Vec2} This vector divided by the specified scalar.
         */
        Vec2.prototype.divide = function (divisor) {
            this[0] /= divisor;
            this[1] /= divisor;

            return this;
        };

        /**
         * Mixes (interpolates) a specified vector with this vector, modifying this vector.
         * @param {Vec2} vector The vector to mix.
         * @param {Number} weight The relative weight of this vector.
         * @returns {Vec2} This vector modified to the mix of itself and the specified vector.
         * @throws {ArgumentError} If the specified vector is null or undefined.
         */
        Vec2.prototype.mix = function (vector, weight) {
            if (!vector) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec2", "mix", "missingVector"));
            }

            var w0 = 1 - weight,
                w1 = weight;

            this[0] = this[0] * w0 + vector[0] * w1;
            this[1] = this[1] * w0 + vector[1] * w1;

            return this;
        };

        /**
         * Negates this vector.
         * @returns {Vec2} This vector, negated.
         */
        Vec2.prototype.negate = function () {
            this[0] = -this[0];
            this[1] = -this[1];

            return this;
        };

        /**
         * Computes the scalar dot product of this vector and a specified vector.
         * @param {Vec2} vector The vector to multiply.
         * @returns {Number} The scalar dot product of the vectors.
         * @throws {ArgumentError} If the specified vector is null or undefined.
         */
        Vec2.prototype.dot = function (vector) {
            if (!vector) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec2", "dot", "missingVector"));
            }

            return this[0] * vector[0] + this[1] * vector[1];
        };

        /**
         * Computes the squared magnitude of this vector.
         * @returns {Number} The squared magnitude of this vector.
         */
        Vec2.prototype.magnitudeSquared = function () {
            return this.dot(this);
        };

        /**
         * Computes the magnitude of this vector.
         * @returns {Number} The magnitude of this vector.
         */
        Vec2.prototype.magnitude = function () {
            return Math.sqrt(this.magnitudeSquared());
        };

        /**
         * Normalizes this vector to a unit vector.
         * @returns {Vec2} This vector, normalized.
         */
        Vec2.prototype.normalize = function () {
            var magnitude = this.magnitude(),
                magnitudeInverse = 1 / magnitude;

            this[0] *= magnitudeInverse;
            this[1] *= magnitudeInverse;

            return this;
        };

        /**
         * Computes the squared distance from this vector to a specified vector.
         * @param {Vec2} vector The vector to compute the distance to.
         * @returns {Number} The squared distance between the vectors.
         * @throws {ArgumentError} If the specified vector is null or undefined.
         */
        Vec2.prototype.distanceToSquared = function (vector) {
            if (!vector) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec2", "distanceToSquared", "missingVector"));
            }

            var dx = this[0] - vector[0],
                dy = this[1] - vector[1];

            return dx * dx + dy * dy;
        };

        /**
         * Computes the distance from this vector to a specified vector.
         * @param {Vec2} vector The vector to compute the distance to.
         * @returns {Number} The distance between the vectors.
         * @throws {ArgumentError} If the specified vector is null or undefined.
         */
        Vec2.prototype.distanceTo = function (vector) {
            if (!vector) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Vec2", "distanceTo", "missingVector"));
            }

            return Math.sqrt(this.distanceToSquared(vector));
        };

        /**
         * Creates a {@link Vec3} using this vector's X and Y components and a Z component of 0.
         * @returns {Vec3} A new vector whose X and Y components are those of this vector and whose Z component is 0.
         */
        Vec2.prototype.toVec3 = function () {
            return new Vec3(this[0], this[1], 0);
        };

        /**
         * Swaps the components of this vector with those of another vector. This vector is set to the values of the
         * specified vector, and the specified vector's components are set to the values of this vector.
         * @param {Vec2} that The vector to swap.
         * @returns {Vec2} This vector set to the values of the specified vector.
         */
        Vec2.prototype.swap = function (that) {
            var tmp = this[0];
            this[0] = that[0];
            that[0] = tmp;

            tmp = this[1];
            this[1] = that[1];
            that[1] = tmp;

            return this;
        };

        /**
         * Returns a string representation of this vector.
         * @returns {String} A string representation of this vector, in the form "(x, y)".
         */
        Vec2.prototype.toString = function () {
            return "(" + this[0] + ", " + this[1] + ")";
        };

        return Vec2;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports Offset
 */
define('util/Offset',['../geom/Vec2'
    ],
    function (Vec2) {
        "use strict";

        /**
         * Constructs an offset instance given specified units and offsets.
         * @alias Offset
         * @constructor
         * @classdesc Specifies an offset relative to a rectangle. Used by [Placemark]{@link Placemark} and
         * other shapes.
         * @param {String} xUnits The type of units specified for the X dimension. May be one of the following:
         * <ul>
         *     <li>[WorldWind.OFFSET_FRACTION]{@link WorldWind#OFFSET_FRACTION}</li>
         *     <li>[WorldWind.OFFSET_INSET_PIXELS]{@link WorldWind#OFFSET_INSET_PIXELS}</li>
         *     <li>[WorldWind.OFFSET_PIXELS]{@link WorldWind#OFFSET_PIXELS}</li>
         * </ul>
         * @param {Number} x The offset in the X dimension.
         * @param {String} yUnits The type of units specified for the Y dimension, assuming a lower-left Y origin.
         * May be one of the following:
         * <ul>
         *     <li>[WorldWind.OFFSET_FRACTION]{@link WorldWind#OFFSET_FRACTION}</li>
         *     <li>[WorldWind.OFFSET_INSET_PIXELS]{@link WorldWind#OFFSET_INSET_PIXELS}</li>
         *     <li>[WorldWind.OFFSET_PIXELS]{@link WorldWind#OFFSET_PIXELS}</li>
         * </ul>
         * @param {Number} y The offset in the Y dimension.
         */
        var Offset = function (xUnits, x, yUnits, y) {

            /**
             * The offset in the X dimension, interpreted according to this instance's xUnits argument.
             * @type {Number}
             */
            this.x = x;

            /**
             * The offset in the Y dimension, interpreted according to this instance's yUnits argument.
             * @type {Number}
             */
            this.y = y;

            /**
             * The units of this instance's X offset. See this class' constructor description for a list of the
             * possible values.
             * @type {String}
             */
            this.xUnits = xUnits;

            /**
             * The units of this instance's Y offset. See this class' constructor description for a list of the
             * possible values.
             * @type {String}
             */
            this.yUnits = yUnits;
        };

        /**
         * Creates a new copy of this offset with identical property values.
         * @returns {Offset} A new offset instance with its property values the same as this one's.
         */
        Offset.prototype.clone = function () {
            return new Offset(this.xUnits, this.x, this.yUnits, this.y);
        };

        /**
         * Returns this offset's absolute X and Y coordinates in pixels for a rectangle of a specified size in pixels.
         * The returned offset is in pixels relative to the rectangle's origin, and is defined in the coordinate
         * system used by the caller.
         * @param {Number} width The rectangle's width in pixels.
         * @param {Number} height The rectangles height in pixels.
         * @returns {Vec2} The computed offset relative to the rectangle's origin.
         */
        Offset.prototype.offsetForSize = function (width, height) {
            var x, y;

            if (this.xUnits === WorldWind.OFFSET_FRACTION) {
                x = width * this.x;
            } else if (this.xUnits === WorldWind.OFFSET_INSET_PIXELS) {
                x = width - this.x;
            } else { // default to OFFSET_PIXELS
                x = this.x;
            }

            if (this.yUnits === WorldWind.OFFSET_FRACTION) {
                y = height * this.y;
            } else if (this.yUnits === WorldWind.OFFSET_INSET_PIXELS) {
                y = height - this.y;
            } else { // default to OFFSET_PIXELS
                y = this.y;
            }

            return new Vec2(x, y);
        };

        /**
         * Returns a string representation of this object.
         * @returns {String} A string representation of this object.
         */
        Offset.prototype.toString = function () {
            return this.xUnits + " " + this.x + " " + this.yUnits + " " + this.y;
        };

        return Offset;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports TextAttributes
 */
define('shapes/TextAttributes',[
        '../util/Color',
        '../util/Font',
        '../util/Offset'
    ],
    function (Color,
              Font,
              Offset) {
        "use strict";

        /**
         * Constructs a text attributes bundle.
         * @alias TextAttributes
         * @constructor
         * @classdesc Holds attributes applied to [Text]{@link Text} shapes and [Placemark]{@link Placemark} labels.
         *
         * @param {TextAttributes} attributes Attributes to initialize this attributes instance to. May be null,
         * in which case the new instance contains default attributes.
         */
        var TextAttributes = function (attributes) {
            this._color = attributes ? attributes._color.clone() : Color.WHITE.clone();
            this._font = attributes ? attributes._font : new Font(14);
            this._offset = attributes ? attributes._offset
                : new Offset(WorldWind.OFFSET_FRACTION, 0.5, WorldWind.OFFSET_FRACTION, 0.0);
            this._scale = attributes ? attributes._scale : 1;
            this._depthTest = attributes ? attributes._depthTest : false;
            this._enableOutline = attributes ? attributes._enableOutline : true;
            this._outlineWidth = attributes ? attributes._outlineWidth : 4;
            this._outlineColor = attributes ? attributes._outlineColor : new Color(0, 0, 0, 0.5);

            /**
             * Indicates whether this object's state key is invalid. Subclasses must set this value to true when their
             * attributes change. The state key will be automatically computed the next time it's requested. This flag
             * will be set to false when that occurs.
             * @type {boolean}
             * @protected
             */
            this.stateKeyInvalid = true;
        };

        /**
         * Computes the state key for this attributes object. Subclasses that define additional attributes must
         * override this method, call it from that method, and append the state of their attributes to its
         * return value.
         * @returns {String} The state key for this object.
         * @protected
         */
        TextAttributes.prototype.computeStateKey = function () {
            return "c " + this._color.toHexString(true) +
                " f " + this._font.toString() +
                " o " + this._offset.toString() +
                " s " + this._scale +
                " dt " + this._depthTest +
                " eo " + this._enableOutline +
                " ow " + this._outlineWidth +
                " oc " + this._outlineColor.toHexString(true);
        };

        Object.defineProperties(TextAttributes.prototype, {
            /**
             * A string identifying the state of this attributes object. The string encodes the current values of all
             * this object's properties. It's typically used to validate cached representations of shapes associated
             * with this attributes object.
             * @type {String}
             * @readonly
             * @memberof TextAttributes.prototype
             */
            stateKey: {
                get: function () {
                    if (this.stateKeyInvalid) {
                        this._stateKey = this.computeStateKey();
                        this.stateKeyInvalid = false;
                    }
                    return this._stateKey;
                }
            },

            /**
             * The text color.
             * @type {Color}
             * @default White (1, 1, 1, 1)
             * @memberof TextAttributes.prototype
             */
            color: {
                get: function () {
                    return this._color;
                },
                set: function (value) {
                    this._color = value;
                    this.stateKeyInvalid = true;
                }
            },

            /**
             * The text size, face and other characteristics, as described in [Font]{@link Font}.
             * @type {Font}
             * @default Those of [Font]{@link Font}, but with a font size of 14.
             * @memberof TextAttributes.prototype
             */
            font: {
                get: function () {
                    return this._font;
                },
                set: function (value) {
                    this._font = value;
                    this.stateKeyInvalid = true;
                }
            },

            /**
             * Indicates the location of the text relative to its specified position.
             * May be null, in which case the text's bottom-left corner is placed at the specified position.
             * @type {Offset}
             * @default 0.5, 0.0, both fractional (Places the text's horizontal center and vertical bottom at the
             * specified position.)
             * @memberof TextAttributes.prototype
             */
            offset: {
                get: function () {
                    return this._offset;
                },
                set: function (value) {
                    this._offset = value;
                    this.stateKeyInvalid = true;
                }
            },

            /**
             * Indicates the amount to scale the text. A value of 0 makes the text disappear.
             * @type {Number}
             * @default 1.0
             * @memberof TextAttributes.prototype
             */
            scale: {
                get: function () {
                    return this._scale;
                },
                set: function (value) {
                    this._scale = value;
                    this.stateKeyInvalid = true;
                }
            },

            /**
             * Indicates whether the text should be depth-tested against other objects in the scene. If true,
             * the text may be occluded by terrain and other objects in certain viewing situations. If false,
             * the text will not be occluded by terrain and other objects.
             * @type {Boolean}
             * @default false
             * @memberof TextAttributes.prototype
             */
            depthTest: {
                get: function () {
                    return this._depthTest;
                },
                set: function (value) {
                    this._depthTest = value;
                    this.stateKeyInvalid = true;
                }
            },

            /**
             * Indicates if the text will feature an outline around its characters.
             * @type {Boolean}
             * @default true
             * @memberof TextAttributes.prototype
             */
            enableOutline: {
                get: function () {
                    return this._enableOutline;
                },
                set: function (value) {
                    this._enableOutline = value;
                    this.stateKeyInvalid = true;
                }
            },

            /**
             * Indicates the text outline width (or thickness) in pixels.
             * @type {Number}
             * @default 4
             * @memberof TextAttributes.prototype
             */
            outlineWidth: {
                get: function () {
                    return this._outlineWidth;
                },
                set: function (value) {
                    this._outlineWidth = value;
                    this.stateKeyInvalid = true;
                }
            },

            /**
             * The color of the outline.
             * @type {Color}
             * @default Half-transparent black (0, 0, 0, 0.5)
             * @memberof TextAttributes.prototype
             */
            outlineColor: {
                get: function () {
                    return this._outlineColor;
                },
                set: function (value) {
                    this._outlineColor = value;
                    this.stateKeyInvalid = true;
                }
            }
        });

        return TextAttributes;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
define('shapes/AnnotationAttributes',[
        '../util/Color',
        '../util/Font',
        '../util/Insets',
        '../shapes/TextAttributes'
    ],
    function (Color,
              Font,
              Insets,
              TextAttributes) {
        "use strict";

        /**
         * Constructs an annotation attributes bundle.
         * @alias AnnotationAttributes
         * @constructor
         * @classdesc Holds attributes applied to {@link Annotation} shapes.
         * @param {AnnotationAttributes} attributes Attributes to initialize this attributes instance to. May be null,
         * in which case the new instance contains default attributes.
         */
        var AnnotationAttributes = function (attributes) {

            // These are all documented with their property accessors below.
            this._cornerRadius = attributes ? attributes._cornerRadius : 0;
            this._insets = attributes ? attributes._insets : new Insets(0, 0, 0, 0);
            this._backgroundColor = attributes ? attributes._backgroundColor.clone() : Color.WHITE.clone();
            this._leaderGapWidth = attributes ? attributes._leaderGapWidth : 40;
            this._leaderGapHeight = attributes ? attributes._leaderGapHeight : 30;
            this._opacity = attributes ? attributes._opacity : 1;
            this._scale = attributes ? attributes._scale : 1;
            this._drawLeader = attributes ? attributes._drawLeader : true;
            this._width = attributes ? attributes._width : 200;
            this._height = attributes ? attributes._height : 100;
            this._textAttributes = attributes ? attributes._textAttributes : this.createDefaultTextAttributes();

            /**
             * Indicates whether this object's state key is invalid. Subclasses must set this value to true when their
             * attributes change. The state key will be automatically computed the next time it's requested. This flag
             * will be set to false when that occurs.
             * @type {Boolean}
             * @protected
             */
            this.stateKeyInvalid = true;
        };

        /**
         * Computes the state key for this attributes object. Subclasses that define additional attributes must
         * override this method, call it from that method, and append the state of their attributes to its
         * return value.
         * @returns {String} The state key for this object.
         * @protected
         */
        AnnotationAttributes.prototype.computeStateKey = function () {
            return "wi " + this._width
                + " he " + this._height
                + " cr " + this._cornerRadius
                + " in " + this._insets.toString()
                + " bg " + this.backgroundColor.toHexString(true)
                + " dl " + this.drawLeader
                + " lgw " + this.leaderGapWidth
                + " lgh " + this.leaderGapHeight
                + " op " + this.opacity
                + " ta " + this._textAttributes.stateKey
                + " sc " + this.scale;
        };

        // Internal use only. Intentionally not documented.
        AnnotationAttributes.prototype.createDefaultTextAttributes = function() {
            var attributes = new TextAttributes(null);
            attributes.enableOutline = false; // Annotations display text without an outline by default
            return attributes;
        };

        Object.defineProperties(AnnotationAttributes.prototype, {

            /**
             * Indicates the width of the callout.
             * @type {Number}
             * @default 200
             * @memberof AnnotationAttributes.prototype
             */
            width: {
                get: function () {
                    return this._width;
                },
                set: function (value) {
                    this._width = value;
                    this.stateKeyInvalid = true;
                }
            },

            /**
             * Indicates height of the callout.
             * @type {Number}
             * @default 100
             * @memberof AnnotationAttributes.prototype
             */
            height: {
                get: function () {
                    return this._height;
                },
                set: function (value) {
                    this._height = value;
                    this.stateKeyInvalid = true;
                }
            },

            /**
             * Indicates the radius for the corners.
             * @type {Number}
             * @default 0
             * @memberof AnnotationAttributes.prototype
             */
            cornerRadius: {
                get: function () {
                    return this._cornerRadius;
                },
                set: function (value) {
                    this._cornerRadius = value;
                    this.stateKeyInvalid = true;
                }
            },

            /**
             * Indicates the insets instance of this object.
             * Insets adjusts top, bottom, left, right padding for the text.
             * @type {Insets}
             * @default 0, 0, 0, 0
             * @memberof AnnotationAttributes.prototype
             */
            insets: {
                get: function () {
                    return this._insets;
                },
                set: function (value) {
                    this._insets = value;
                    this.stateKeyInvalid = true;
                }
            },

            /**
             * Indicates the background color of the callout.
             * @type {Color}
             * @default White
             * @memberof AnnotationAttributes.prototype
             */
            backgroundColor: {
                get: function () {
                    return this._backgroundColor;
                },
                set: function (value) {
                    this._backgroundColor = value;
                    this.stateKeyInvalid = true;
                }
            },


            /**
             * Indicates the attributes to apply to the annotation's text.
             * @type {TextAttributes}
             * @default The defaults of {@link TextAttributes}.
             * @memberof AnnotationAttributes.prototype
             */
            textAttributes: {
                get: function () {
                    return this._textAttributes;
                },
                set: function (value) {
                    this._textAttributes = value;
                    this.stateKeyInvalid = true;
                }
            },

            /**
             * Indicates whether to draw a leader pointing to the annotation's geographic position.
             * @type {Boolean}
             * @default true
             * @memberof AnnotationAttributes.prototype
             */
            drawLeader: {
                get: function () {
                    return this._drawLeader;
                },
                set: function (value) {
                    this._drawLeader = value;
                    this.stateKeyInvalid = true;
                }
            },

            /**
             * Indicates the gap width of the leader in pixels.
             * @type {Number}
             * @default 40
             * @memberof AnnotationAttributes.prototype
             */
            leaderGapWidth: {
                get: function () {
                    return this._leaderGapWidth;
                },
                set: function (value) {
                    this._leaderGapWidth = value;
                    this.stateKeyInvalid = true;
                }
            },

            /**
             * Indicates the gap height of the leader in pixels.
             * @type {Number}
             * @default 30
             * @memberof AnnotationAttributes.prototype
             */
            leaderGapHeight: {
                get: function () {
                    return this._leaderGapHeight;
                },
                set: function (value) {
                    this._leaderGapHeight = value;
                    this.stateKeyInvalid = true;
                }
            },

            /**
             * Indicates the opacity of the annotation.
             * The value ranges from 0 to 1.
             * Opacity affects both callout and text.
             * @type {Number}
             * @default 1
             * @memberof AnnotationAttributes.prototype
             */
            opacity: {
                get: function () {
                    return this._opacity;
                },
                set: function (value) {
                    this._opacity = value;
                    this.stateKeyInvalid = true;
                }
            },

            /**
             * Indicates the scale multiplier.
             * @type {Number}
             * @default 1
             * @memberof AnnotationAttributes.prototype
             */
            scale: {
                get: function () {
                    return this._scale;
                },
                set: function (value) {
                    this._scale = value;
                    this.stateKeyInvalid = true;
                }
            },

            /**
             * A string identifying the state of this attributes object. The string encodes the current values of all
             * this object's properties. It's typically used to validate cached representations of shapes associated
             * with this attributes object.
             * @type {String}
             * @readonly
             * @memberof AnnotationAttributes.prototype
             */
            stateKey: {
                get: function () {
                    if (this.stateKeyInvalid) {
                        this._stateKey = this.computeStateKey();
                        this.stateKeyInvalid = false;
                    }
                    return this._stateKey;
                }
            }
        });

        return AnnotationAttributes;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports GpuShader
 */
define('shaders/GpuShader',[
        '../error/ArgumentError',
        '../util/Logger'
    ],
    function (ArgumentError,
              Logger) {
        "use strict";

        /**
         * Constructs a GPU shader of a specified type with specified GLSL source code.
         *
         * @alias GpuShader
         * @constructor
         * @classdesc
         * Represents an OpenGL shading language (GLSL) shader and provides methods for compiling and disposing
         * of them.
         *
         * @param {WebGLRenderingContext} gl The current WebGL context.
         * @param {Number} shaderType The type of shader, either WebGLRenderingContext.VERTEX_SHADER
         * or WebGLRenderingContext.FRAGMENT_SHADER.
         * @param {String} shaderSource The shader's source code.
         * @throws {ArgumentError} If the shader type is unrecognized, the shader source is null or undefined or shader
         * compilation fails. If the compilation fails the error thrown contains any compilation messages.
         */
        var GpuShader = function (gl, shaderType, shaderSource) {
            if (!(shaderType === gl.VERTEX_SHADER
                || shaderType === gl.FRAGMENT_SHADER)) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "GpuShader", "constructor",
                    "The specified shader type is unrecognized."));
            }

            if (!shaderSource) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "GpuShader", "constructor",
                    "The specified shader source is null or undefined."));
            }

            var shader = gl.createShader(shaderType);
            if (!shader) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "GpuShader", "constructor",
                    "Unable to create shader of type " +
                    (shaderType == gl.VERTEX_SHADER ? "VERTEX_SHADER." : "FRAGMENT_SHADER.")));
            }

            if (!this.compile(gl, shader, shaderType, shaderSource)) {
                var infoLog = gl.getShaderInfoLog(shader);

                gl.deleteShader(shader);

                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "GpuShader", "constructor",
                    "Unable to compile shader: " + infoLog));
            }

            this.shaderId = shader;
        };

        /**
         * Compiles the source code for this shader. This method is not meant to be invoked by applications. It is
         * invoked internally as needed.
         * @param {WebGLRenderingContext} gl The current WebGL rendering context.
         * @param {WebGLShader} shaderId The shader ID.
         * @param {Number} shaderType The type of shader, either WebGLRenderingContext.VERTEX_SHADER
         * or WebGLRenderingContext.FRAGMENT_SHADER.
         * @param {String} shaderSource The shader's source code.
         * @returns {boolean} <code>true</code> if the shader compiled successfully, otherwise <code>false</code>.
         */
        GpuShader.prototype.compile = function (gl, shaderId, shaderType, shaderSource) {
            gl.shaderSource(shaderId, shaderSource);
            gl.compileShader(shaderId);

            return gl.getShaderParameter(shaderId, gl.COMPILE_STATUS);
        };

        /**
         * Releases this shader's WebGL shader.
         * @param {WebGLRenderingContext} gl The current WebGL rendering context.
         */
        GpuShader.prototype.dispose = function (gl) {
            if (this.shaderId) {
                gl.deleteShader(this.shaderId);
                delete this.shaderId;
            }
        };

        return GpuShader;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports GpuProgram
 */
define('shaders/GpuProgram',[
        '../error/ArgumentError',
        '../util/Color',
        '../shaders/GpuShader',
        '../util/Logger'
    ],
    function (ArgumentError,
              Color,
              GpuShader,
              Logger) {
        "use strict";

        /**
         * Constructs a GPU program with specified source code for vertex and fragment shaders.
         * This constructor is intended to be called only by subclasses.
         * <p>
         * This constructor creates WebGL shaders for the specified shader sources and attaches them to a new GLSL
         * program. The method compiles the shaders and then links the program if compilation is successful. Use the
         * [DrawContext.bindProgram]{@link DrawContext#bindProgram} function to make the program current during rendering.
         *
         * @alias GpuProgram
         * @constructor
         * @classdesc
         * Represents an OpenGL shading language (GLSL) shader program and provides methods for identifying and
         * accessing shader variables. Shader programs are created by instances of this class and made current when the
         * DrawContext.bindProgram function is invoked.
         * <p>
         * This is an abstract class and not intended to be created directly.
         *
         * @param {WebGLRenderingContext} gl The current WebGL context.
         * @param {String} vertexShaderSource The source code for the vertex shader.
         * @param {String} fragmentShaderSource The source code for the fragment shader.
         * @param {String[]} attributeBindings An array of attribute variable names whose bindings are to be explicitly
         * specified. Each name is bound to its corresponding index in the array. May be null, in which case the
         * linker determines all the bindings.
         * @throws {ArgumentError} If either source is null or undefined, the shaders cannot be compiled, or linking of
         * the compiled shaders into a program fails.
         */
        var GpuProgram = function (gl, vertexShaderSource, fragmentShaderSource, attributeBindings) {
            if (!vertexShaderSource || !fragmentShaderSource) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "GpuProgram", "constructor",
                    "The specified shader source is null or undefined."));
            }

            var program, vShader, fShader;

            try {
                vShader = new GpuShader(gl, gl.VERTEX_SHADER, vertexShaderSource);
                fShader = new GpuShader(gl, gl.FRAGMENT_SHADER, fragmentShaderSource);
            } catch (e) {
                if (vShader)
                    vShader.dispose(gl);
                if (fShader)
                    fShader.dispose(gl);

                throw e;
            }

            program = gl.createProgram();
            if (!program) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "GpuProgram", "constructor",
                    "Unable to create shader program."));
            }

            gl.attachShader(program, vShader.shaderId);
            gl.attachShader(program, fShader.shaderId);

            if (attributeBindings) {
                for (var i = 0, len = attributeBindings.length; i < len; i++) {
                    gl.bindAttribLocation(program, i, attributeBindings[i]);
                }
            }

            if (!this.link(gl, program)) {
                // Get the info log before deleting the program.
                var infoLog = gl.getProgramInfoLog(program);

                gl.detachShader(program, vShader.shaderId);
                gl.detachShader(program, fShader.shaderId);
                gl.deleteProgram(program);
                vShader.dispose(gl);
                fShader.dispose(gl);

                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "GpuProgram", "constructor",
                    "Unable to link shader program: " + infoLog));
            }

            /**
             * Indicates the WebGL program object associated with this GPU program.
             * @type {WebGLProgram}
             * @readonly
             */
            this.programId = program;

            // Internal. Intentionally not documented. These will be filled in as attribute locations are requested.
            this.attributeLocations = {};
            this.uniformLocations = {};

            // Internal. Intentionally not documented.
            this.vertexShader = vShader;

            // Internal. Intentionally not documented.
            this.fragmentShader = fShader;

            // Internal. Intentionally not documented.
            this.size = vertexShaderSource.length + fragmentShaderSource.length;

            // Internal. Intentionally not documented.
            this.scratchArray = new Float32Array(16);
        };

        /**
         * Releases this GPU program's WebGL program and associated shaders. Upon return this GPU program's WebGL
         * program ID is 0 as is that of the associated shaders.
         *
         * @param {WebGLRenderingContext} gl The current WebGL context.
         */
        GpuProgram.prototype.dispose = function (gl) {
            if (this.programId) {
                if (this.vertexShader) {
                    gl.detachShader(this.programId, this.vertexShader.shaderId);
                }
                if (this.fragmentShader) {
                    gl.detachShader(this.programId, this.fragmentShader.shaderId);
                }

                gl.deleteProgram(this.programId);
                delete this.programId;
            }

            if (this.vertexShader) {
                this.vertexShader.dispose(gl);
                delete this.vertexShader;
            }

            if (this.fragmentShader) {
                this.fragmentShader.dispose(gl);
                delete this.fragmentShader;
            }

            this.attributeLocations = {};
            this.uniformLocations = {};
        };

        /**
         * Returns the GLSL attribute location of a specified attribute name.
         *
         * @param {WebGLRenderingContext} gl The current WebGL context.
         * @param {String} attributeName The name of the attribute whose location is determined.
         * @returns {Number} The WebGL attribute location of the specified attribute, or -1 if the attribute is not
         * found.
         * @throws {ArgumentError} If the specified attribute name is null, empty or undefined.
         */
        GpuProgram.prototype.attributeLocation = function (gl, attributeName) {
            if (!attributeName || attributeName.length == 0) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "GpuProgram", "attributeLocation",
                    "The specified attribute name is null, undefined or empty."));
            }

            var location = this.attributeLocations[attributeName];
            if (!location) {
                location = gl.getAttribLocation(this.programId, attributeName);
                this.attributeLocations[attributeName] = location;
            }

            return location;
        };

        /**
         * Returns the GLSL uniform location of a specified uniform name.
         *
         * @param {WebGLRenderingContext} gl The current WebGL context.
         * @param {String} uniformName The name of the uniform variable whose location is determined.
         * @returns {WebGLUniformLocation} The WebGL uniform location of the specified uniform variable,
         * or -1 if the uniform is not found.
         * @throws {ArgumentError} If the specified uniform name is null, empty or undefined.
         */
        GpuProgram.prototype.uniformLocation = function (gl, uniformName) {
            if (!uniformName || uniformName.length == 0) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "GpuProgram", "uniformLocation",
                    "The specified uniform name is null, undefined or empty."));
            }

            var location = this.uniformLocations[uniformName];
            if (!location) {
                location = gl.getUniformLocation(this.programId, uniformName);
                this.uniformLocations[uniformName] = location;
            }

            return location;
        };

        /**
         * Links a specified GLSL program. This method is not meant to be called by applications. It is called
         * internally as needed.
         *
         * @param {WebGLRenderingContext} gl The current WebGL context.
         * @param {WebGLProgram} program The WebGL program.
         * @returns {Boolean} true if linking was successful, otherwise false.
         * @protected
         */
        GpuProgram.prototype.link = function (gl, program) {
            gl.linkProgram(program);

            return gl.getProgramParameter(program, gl.LINK_STATUS);
        };

        /**
         * Loads a specified matrix as the value of a GLSL 4x4 matrix uniform variable with the specified location.
         * <p>
         * This functions converts the matrix into column-major order prior to loading its components into the GLSL
         * uniform variable, but does not modify the specified matrix.
         *
         * @param {WebGLRenderingContext} gl The current WebGL context.
         * @param {Matrix} matrix The matrix to load.
         * @param {WebGLUniformLocation} location The location of the uniform variable in the currently bound GLSL program.
         * @throws {ArgumentError} If the specified matrix is null or undefined.
         */
        GpuProgram.prototype.loadUniformMatrix = function (gl, matrix, location) {
            if (!matrix) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "GpuProgram", "loadUniformMatrix",
                    "missingMatrix"));
            }

            var columnMajorArray = matrix.columnMajorComponents(this.scratchArray);
            gl.uniformMatrix4fv(location, false, columnMajorArray);
        };

        /**
         * Loads a specified color as the value of a GLSL vec4 uniform variable with the specified location.
         * <p>
         * This function multiplies the red, green and blue components by the alpha component prior to loading the color
         * in the GLSL uniform variable, but does not modify the specified color.
         *
         * @param {WebGLRenderingContext} gl The current WebGL context.
         * @param {Color} color The color to load.
         * @param {WebGLUniformLocation} location The location of the uniform variable in the currently bound GLSL program.
         * @throws {ArgumentError} If the specified color is null or undefined.
         */
        GpuProgram.prototype.loadUniformColor = function (gl, color, location) {
            if (!color) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "GpuProgram", "loadUniformColor",
                    "missingColor"));
            }

            var premul = color.premultipliedComponents(this.scratchArray);
            gl.uniform4f(location, premul[0], premul[1], premul[2], premul[3]);
        };

        /**
         * Loads the specified RGBA color components as the value of a GLSL vec4 uniform variable with the specified
         * location.
         * <p>
         * This function multiplies the red, green and blue components by the alpha component prior to loading the color
         * in the GLSL uniform variable.
         *
         * @param {WebGLRenderingContext} gl The current WebGL context.
         * @param {Number} red The red component, a number between 0 and 1.
         * @param {Number} green The green component, a number between 0 and 1.
         * @param {Number} blue The blue component, a number between 0 and 1.
         * @param {Number} alpha The alpha component, a number between 0 and 1.
         * @param {WebGLUniformLocation} location The location of the uniform variable in the currently bound GLSL program.
         */
        GpuProgram.prototype.loadUniformColorComponents = function (gl, red, green, blue, alpha, location) {
            gl.uniform4f(location, red * alpha, green * alpha, blue * alpha, alpha);
        };

        return GpuProgram;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports BasicTextureProgram
 */
define('shaders/BasicTextureProgram',[
    '../error/ArgumentError',
    '../util/Color',
    '../shaders/GpuProgram',
    '../util/Logger'
],
    function (ArgumentError,
        Color,
        GpuProgram,
        Logger) {
        "use strict";

        /**
         * Constructs a new program.
         * Initializes, compiles and links this GLSL program with the source code for its vertex and fragment shaders.
         * <p>
         * This method creates WebGL shaders for the program's shader sources and attaches them to a new GLSL program. This
         * method then compiles the shaders and then links the program if compilation is successful. Use the bind method to make the
         * program current during rendering.
         *
         * @alias BasicTextureProgram
         * @constructor
         * @augments GpuProgram
         * @classdesc BasicTextureProgram is a GLSL program that draws textured or untextured geometry.
         * @param {WebGLRenderingContext} gl The current WebGL context.
         * @throws {ArgumentError} If the shaders cannot be compiled, or if linking of
         * the compiled shaders into a program fails.
         */
        var BasicTextureProgram = function (gl) {
            var vertexShaderSource =
                'attribute vec4 vertexPoint;\n' +
                'attribute vec4 vertexTexCoord;\n' +
                'attribute vec4 normalVector;\n' +
                'uniform mat4 mvpMatrix;\n' +
                'uniform mat4 mvInverseMatrix;\n' +
                'uniform mat4 texCoordMatrix;\n' +
                'uniform bool applyLighting;\n' +
                'varying vec2 texCoord;\n' +
                'varying vec4 normal;\n' +
                'void main() {gl_Position = mvpMatrix * vertexPoint;\n' +
                'texCoord = (texCoordMatrix * vertexTexCoord).st;\n' +
                'if (applyLighting) {normal = mvInverseMatrix * normalVector;}\n' +
                '}',
                fragmentShaderSource =
                    'precision mediump float;\n' +
                    'uniform float opacity;\n' +
                    'uniform vec4 color;\n' +
                    'uniform bool enableTexture;\n' +
                    'uniform bool modulateColor;\n' +
                    'uniform sampler2D textureSampler;\n' +
                    'uniform bool applyLighting;\n' +
                    'varying vec2 texCoord;\n' +
                    'varying vec4 normal;\n' +
                    'void main() {\n' +
                    'vec4 textureColor = texture2D(textureSampler, texCoord);\n' +
                    'float ambient = 0.15; vec4 lightDirection = vec4(0, 0, 1, 0);\n' +
                    'if (enableTexture && !modulateColor)\n' +
                    '    gl_FragColor = textureColor * color * opacity;\n' +
                    'else if (enableTexture && modulateColor)\n' +
                    '    gl_FragColor = color * ceil(textureColor.a);\n' +
                    'else\n' +
                    '    gl_FragColor = color * opacity;\n' +
                    'if (gl_FragColor.a == 0.0) {discard;}\n' +
                    'if (applyLighting) {\n' +
                    '    vec4 n = normal * (gl_FrontFacing ? 1.0 : -1.0);\n' +
                    '    gl_FragColor.rgb *= clamp(ambient + dot(lightDirection, n), 0.0, 1.0);\n' +
                    '}\n' +
                    '}';

            // Specify bindings to avoid the WebGL performance warning that's generated when normalVector gets
            // bound to location 0.
            var bindings = ["vertexPoint", "normalVector", "vertexTexCoord"];

            // Call to the superclass, which performs shader program compiling and linking.
            GpuProgram.call(this, gl, vertexShaderSource, fragmentShaderSource, bindings);

            /**
             * The WebGL location for this program's 'vertexPoint' attribute.
             * @type {Number}
             * @readonly
             */
            this.vertexPointLocation = this.attributeLocation(gl, "vertexPoint");

            /**
             * The WebGL location for this program's 'normalVector' attribute.
             * @type {Number}
             * @readonly
             */
            this.normalVectorLocation = this.attributeLocation(gl, "normalVector");

            /**
             * The WebGL location for this program's 'vertexTexCoord' attribute.
             * @type {Number}
             * @readonly
             */
            this.vertexTexCoordLocation = this.attributeLocation(gl, "vertexTexCoord");

            /**
             * The WebGL location for this program's 'mvpMatrix' uniform.
             * @type {WebGLUniformLocation}
             * @readonly
             */
            this.mvpMatrixLocation = this.uniformLocation(gl, "mvpMatrix");

            /**
             * The WebGL location for this program's 'mvInverseMatrix' uniform.
             * @type {WebGLUniformLocation}
             * @readonly
             */
            this.mvInverseMatrixLocation = this.uniformLocation(gl, "mvInverseMatrix");

            /**
             * The WebGL location for this program's 'color' uniform.
             * @type {WebGLUniformLocation}
             * @readonly
             */
            this.colorLocation = this.uniformLocation(gl, "color");

            /**
             * The WebGL location for this program's 'enableTexture' uniform.
             * @type {WebGLUniformLocation}
             * @readonly
             */
            this.textureEnabledLocation = this.uniformLocation(gl, "enableTexture");

            /**
             * The WebGL location for this program's 'modulateColor' uniform.
             * @type {WebGLUniformLocation}
             * @readonly
             */
            this.modulateColorLocation = this.uniformLocation(gl, "modulateColor");

            /**
             * The WebGL location for this program's 'textureSampler' uniform.
             * @type {WebGLUniformLocation}
             * @readonly
             */
            this.textureUnitLocation = this.uniformLocation(gl, "textureSampler");

            /**
             * The WebGL location for this program's 'texCoordMatrix' uniform.
             * @type {WebGLUniformLocation}
             * @readonly
             */
            this.textureMatrixLocation = this.uniformLocation(gl, "texCoordMatrix");

            /**
             * The WebGL location for this program's 'opacity' uniform.
             * @type {WebGLUniformLocation}
             * @readonly
             */
            this.opacityLocation = this.uniformLocation(gl, "opacity");

            /**
             * The WegGL location for this program's 'enableLighting' uniform.
             * @type {WebGLUniformLocation}
             * @readonly
             */
            this.applyLightingLocation = this.uniformLocation(gl, "applyLighting");
        };

        /**
         * A string that uniquely identifies this program.
         * @type {string}
         * @readonly
         */
        BasicTextureProgram.key = "WorldWindGpuBasicTextureProgram";

        // Inherit from GpuProgram.
        BasicTextureProgram.prototype = Object.create(GpuProgram.prototype);

        /**
         * Loads the specified matrix as the value of this program's 'mvInverseMatrix' uniform variable.
         *
         * @param {WebGLRenderingContext} gl The current WebGL context.
         * @param {Matrix} matrix The matrix to load.
         * @throws {ArgumentError} If the specified matrix is null or undefined.
         */
        BasicTextureProgram.prototype.loadModelviewInverse = function (gl, matrix) {
            if (!matrix) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "BasicTextureProgram", "loadModelviewInverse", "missingMatrix"));
            }

            this.loadUniformMatrix(gl, matrix, this.mvInverseMatrixLocation);
        };

        /**
         * Loads the specified matrix as the value of this program's 'mvpMatrix' uniform variable.
         *
         * @param {WebGLRenderingContext} gl The current WebGL context.
         * @param {Matrix} matrix The matrix to load.
         * @throws {ArgumentError} If the specified matrix is null or undefined.
         */
        BasicTextureProgram.prototype.loadModelviewProjection = function (gl, matrix) {
            if (!matrix) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "BasicTextureProgram", "loadModelviewProjection", "missingMatrix"));
            }

            this.loadUniformMatrix(gl, matrix, this.mvpMatrixLocation);
        };

        /**
         * Loads the specified color as the value of this program's 'color' uniform variable.
         *
         * @param {WebGLRenderingContext} gl The current WebGL context.
         * @param {Color} color The color to load.
         * @throws {ArgumentError} If the specified color is null or undefined.
         */
        BasicTextureProgram.prototype.loadColor = function (gl, color) {
            if (!color) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "BasicTextureProgram", "loadColor", "missingColor"));
            }

            this.loadUniformColor(gl, color, this.colorLocation);
        };

        /**
         * Loads the specified boolean as the value of this program's 'enableTexture' uniform variable.
         * @param {WebGLRenderingContext} gl The current WebGL context.
         * @param {Boolean} enable true to enable texturing, false to disable texturing.
         */
        BasicTextureProgram.prototype.loadTextureEnabled = function (gl, enable) {
            gl.uniform1i(this.textureEnabledLocation, enable ? 1 : 0);
        };

        /**
         * Loads the specified boolean as the value of this program's 'modulateColor' uniform variable. When this
         * value is true and the value of the textureEnabled variable is true, the color uniform of this shader is
         * multiplied by the rounded alpha component of the texture color at each fragment. This causes the color
         * to be either fully opaque or fully transparent depending on the value of the texture color's alpha value.
         * This is used during picking to replace opaque or mostly opaque texture colors with the pick color, and
         * to make all other texture colors transparent.
         * @param {WebGLRenderingContext} gl The current WebGL context.
         * @param {Boolean} enable true to enable modulation, false to disable modulation.
         */
        BasicTextureProgram.prototype.loadModulateColor = function (gl, enable) {
            gl.uniform1i(this.modulateColorLocation, enable ? 1 : 0);
        };

        /**
         * Loads the specified number as the value of this program's 'textureSampler' uniform variable.
         * @param {WebGLRenderingContext} gl The current WebGL context.
         * @param {Number} unit The texture unit.
         */
        BasicTextureProgram.prototype.loadTextureUnit = function (gl, unit) {
            gl.uniform1i(this.textureUnitLocation, unit - gl.TEXTURE0);
        };

        /**
         * Loads the specified matrix as the value of this program's 'texCoordMatrix' uniform variable.
         * @param {WebGLRenderingContext} gl The current WebGL context.
         * @param {Matrix} matrix The texture coordinate matrix.
         */
        BasicTextureProgram.prototype.loadTextureMatrix = function (gl, matrix) {
            this.loadUniformMatrix(gl, matrix, this.textureMatrixLocation);
        };

        /**
         * Loads the specified number as the value of this program's 'opacity' uniform variable.
         * @param {WebGLRenderingContext} gl The current WebGL context.
         * @param {Number} opacity The opacity in the range [0, 1].
         */
        BasicTextureProgram.prototype.loadOpacity = function (gl, opacity) {
            gl.uniform1f(this.opacityLocation, opacity);
        };

        /**
         * Loads the specified boolean as the value of this program's 'applyLighting' uniform variable.
         * @param {WebGLRenderingContext} gl The current WebGL context.
         * @param {Number} applyLighting true to apply lighting, otherwise false.
         */
        BasicTextureProgram.prototype.loadApplyLighting = function (gl, applyLighting) {
            gl.uniform1i(this.applyLightingLocation, applyLighting);
        };

        return BasicTextureProgram;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports Line
 */
define('geom/Line',[
        '../error/ArgumentError',
        '../util/Logger',
        '../geom/Vec3'],
    function (ArgumentError,
              Logger,
              Vec3) {
        "use strict";

        /**
         * Constructs a line from a specified origin and direction.
         * @alias Line
         * @constructor
         * @classdesc Represents a line in Cartesian coordinates.
         * @param {Vec3} origin The line's origin.
         * @param {Vec3} direction The line's direction.
         * @throws {ArgumentError} If either the origin or the direction are null or undefined.
         */
        var Line = function (origin, direction) {
            if (!origin) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Line", "constructor",
                    "Origin is null or undefined."));
            }

            if (!direction) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Line", "constructor",
                    "Direction is null or undefined."));
            }

            /**
             * This line's origin.
             * @type {Vec3}
             */
            this.origin = origin;

            /**
             * This line's direction.
             * @type {Vec3}
             */
            this.direction = direction;
        };

        /**
         * Creates a line given two specified endpoints.
         * @param {Vec3} pointA The first endpoint.
         * @param {Vec3} pointB The second endpoint.
         * @return {Line} The new line.
         * @throws {ArgumentError} If either endpoint is null or undefined.
         */
        Line.fromSegment = function (pointA, pointB) {
            if (!pointA || !pointB) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Line", "fromSegment", "missingPoint"));
            }

            var origin = new Vec3(pointA[0], pointA[1], pointA[2]),
                direction = new Vec3(pointB[0] - pointA[0], pointB[1] - pointA[1], pointB[2] - pointA[2]);

            return new Line(origin, direction);
        };

        /**
         * Computes a Cartesian point a specified distance along this line.
         * @param {Number} distance The distance from this line's origin at which to compute the point.
         * @param {Vec3} result A pre-allocated {@Link Vec3} instance in which to return the computed point.
         * @return {Vec3} The specified result argument containing the computed point.
         * @throws {ArgumentError} If the specified result argument is null or undefined.
         */
        Line.prototype.pointAt = function (distance, result) {
            if (!result) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Line", "pointAt", "missingResult."));
            }

            result[0] = this.origin[0] + this.direction[0] * distance;
            result[1] = this.origin[1] + this.direction[1] * distance;
            result[2] = this.origin[2] + this.direction[2] * distance;

            return result;
        };

        /**
         * Indicates whether the components of this line are equal to those of a specified line.
         * @param {Line} otherLine The line to test equality with. May be null or undefined, in which case this
         * function returns false.
         * @returns {boolean} true if all components of this line are equal to the corresponding
         * components of the specified line, otherwise false.
         */
        Line.prototype.equals = function (otherLine) {
            if (otherLine) {
                return this.origin.equals(otherLine.origin) && this.direction.equals(otherLine.direction);
            }

            return false;
        };

        /**
         * Creates a new line that is a copy of this line.
         * @returns {Line} The new line.
         */
        Line.prototype.clone = function () {
            var clone = new Line(new Vec3(0, 0, 0), new Vec3(0, 0, 0));
            clone.copy(this);

            return clone;
        };

        /**
         * Copies the components of a specified line to this line.
         * @param {Line} copyLine The line to copy.
         * @returns {Line} A copy of this line equal to otherLine.
         * @throws {ArgumentError} If the specified line is null or undefined.
         */
        Line.prototype.copy = function (copyLine) {
            if (!copyLine) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Line", "copy", "missingLine"));
            }

            this.origin.copy(copyLine.origin);
            this.direction.copy(copyLine.direction);

            return this;
        };

        return Line;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports Plane
 */
define('geom/Plane',[
        '../error/ArgumentError',
        '../geom/Line',
        '../util/Logger',
        '../geom/Vec3'
    ],
    function (ArgumentError,
              Line,
              Logger,
              Vec3) {
        "use strict";

        /**
         * Constructs a plane.
         * This constructor does not normalize the components. It assumes that a unit normal vector is provided.
         * @alias Plane
         * @constructor
         * @classdesc Represents a plane in Cartesian coordinates.
         * The plane's X, Y and Z components indicate the plane's normal vector. The distance component
         * indicates the plane's distance from the origin relative to its unit normal.
         * The components are expected to be normalized.
         * @param {Number} x The X coordinate of the plane's unit normal vector.
         * @param {Number} y The Y coordinate of the plane's unit normal vector.
         * @param {Number} z The Z coordinate of the plane's unit normal vector.
         * @param {Number} distance The plane's distance from the origin.
         */
        var Plane = function (x, y, z, distance) {
            /**
             * The normal vector to the plane.
             * @type {Vec3}
             */
            this.normal = new Vec3(x, y, z);

            /**
             * The plane's distance from the origin.
             * @type {Number}
             */
            this.distance = distance;
        };

        /**
         * Computes a plane that passes through the specified three points.
         * The plane's normal is the cross product of the
         * two vectors from pb to pa and pc to pa, respectively. The
         * returned plane is undefined if any of the specified points are colinear.
         *
         * @param {Vec3} pa The first point.
         * @param {Vec3} pb The second point.
         * @param {Vec3} pc The third point.
         *
         * @return {Plane} A plane passing through the specified points.
         *
         * @throws {ArgumentError} if pa, pb, or pc is null or undefined.
         */
        Plane.fromPoints = function(pa, pb, pc) {
            if (!pa || !pb || !pc) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Plane", "fromPoints", "missingVector"));
            }

            var vab = new Vec3(pb[0], pb[1], pb[2]);
            vab.subtract(pa);
            var vac = new Vec3(pc[0], pc[1], pc[2]);
            vac.subtract(pa);
            vab.cross(vac);
            vab.normalize();
            var d = -vab.dot(pa);

            return new Plane(vab[0], vab[1], vab[2], d);
        };

        /**
         * Computes the dot product of this plane's normal vector with a specified vector.
         * Since the plane was defined with a unit normal vector, this function returns the distance of the vector from
         * the plane.
         * @param {Vec3} vector The vector to dot with this plane's normal vector.
         * @returns {Number} The computed dot product.
         * @throws {ArgumentError} If the specified vector is null or undefined.
         */
        Plane.prototype.dot = function (vector) {
            if (!vector) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Plane", "dot", "missingVector"));
            }

            return this.normal.dot(vector) + this.distance;
        };

        /**
         * Computes the distance between this plane and a point.
         * @param {Vec3} point The point whose distance to compute.
         * @returns {Number} The computed distance.
         * @throws {ArgumentError} If the specified point is null or undefined.
         */
        Plane.prototype.distanceToPoint = function (point) {
            return this.dot(point);
        };

        /**
         * Transforms this plane by a specified matrix.
         * @param {Matrix} matrix The matrix to apply to this plane.
         * @returns {Plane} This plane transformed by the specified matrix.
         * @throws {ArgumentError} If the specified matrix is null or undefined.
         */
        Plane.prototype.transformByMatrix = function (matrix){
            if (!matrix) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Plane", "transformByMatrix", "missingMatrix"));
            }

            var x = matrix[0] * this.normal[0] + matrix[1] * this.normal[1] + matrix[2] * this.normal[2] + matrix[3] * this.distance,
                y = matrix[4] * this.normal[0] + matrix[5] * this.normal[1] + matrix[6] * this.normal[2] + matrix[7] * this.distance,
                z = matrix[8] * this.normal[0] + matrix[9] * this.normal[1] + matrix[10] * this.normal[2] + matrix[11] * this.distance,
                distance = matrix[12] * this.normal[0] + matrix[13] * this.normal[1] + matrix[14] * this.normal[2] + matrix[15] * this.distance;

            this.normal[0] = x;
            this.normal[1] = y;
            this.normal[2] = z;
            this.distance = distance;
            
            return this;
        };

        /**
         * Normalizes the components of this plane.
         * @returns {Plane} This plane with its components normalized.
         */
        Plane.prototype.normalize = function () {
            var magnitude = this.normal.magnitude();

            if (magnitude === 0)
                return this;

            this.normal.divide(magnitude);
            this.distance /= magnitude;

            return this;
        };

        /**
         * Determines whether a specified line segment intersects this plane.
         *
         * @param {Vec3} endPoint1 The first end point of the line segment.
         * @param {Vec3} endPoint2 The second end point of the line segment.
         * @returns {Boolean} true if the line segment intersects this plane, otherwise false.
         */
        Plane.prototype.intersectsSegment = function(endPoint1, endPoint2) {
            var distance1 = this.dot(endPoint1),
                distance2 = this.dot(endPoint2);

            return distance1 * distance2 <= 0;
        };

        /**
         * Computes the intersection point of this plane with a specified line segment.
         *
         * @param {Vec3} endPoint1 The first end point of the line segment.
         * @param {Vec3} endPoint2 The second end point of the line segment.
         * @param {Vec3} result A variable in which to return the intersection point of the line segment with this plane.
         * @returns {Boolean} true If the line segment intersects this plane, otherwise false.
         */
        Plane.prototype.intersectsSegmentAt = function (endPoint1, endPoint2, result) {
            // Compute the distance from the end-points.
            var distance1 = this.dot(endPoint1),
                distance2 = this.dot(endPoint2);

            // If both points points lie on the plane, ...
            if (distance1 === 0 && distance2 === 0) {
                // Choose an arbitrary endpoint as the intersection.
                result[0] = endPoint1[0];
                result[1] = endPoint1[1];
                result[2] = endPoint1[2];

                return true;
            }
            else if (distance1 === distance2) {
                // The intersection is undefined.
                return false;
            }

            var weight1 = -distance1 / (distance2 - distance1),
                weight2 = 1 - weight1;

            result[0] = weight1 * endPoint1[0] + weight2 * endPoint2[0];
            result[1] = weight1 * endPoint1[1] + weight2 * endPoint2[1];
            result[2] = weight1 * endPoint1[2] + weight2 * endPoint2[2];

            return distance1 * distance2 <= 0;
        };

        /**
         * Determines whether two points are on the same side of this plane.
         *
         * @param {Vec3} pointA the first point.
         * @param {Vec3} pointB the second point.
         *
         * @return {Number} -1 If both points are on the negative side of this plane, +1 if both points are on the
         * positive side of this plane, 0 if the points are on opposite sides of this plane.
         *
         * @throws {ArgumentError} If either point is null or undefined.
         */
        Plane.prototype.onSameSide = function (pointA, pointB) {
            if (!pointA || !pointB) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Plane", "onSameSide", "missingPoint"));
            }

            var da = this.distanceToPoint(pointA),
                db = this.distanceToPoint(pointB);

            if (da < 0 && db < 0)
                return -1;

            if (da > 0 && db > 0)
                return 1;

            return 0;
        };

        /**
         * Clips a line segment to this plane.
         * @param {Vec3} pointA The first line segment endpoint.
         * @param {Vec3} pointB The second line segment endpoint.
         *
         * @returns {Vec3[]}  An array of two points both on the positive side of the plane. If the direction of the line formed by the
         *         two points is positive with respect to this plane's normal vector, the first point in the array will be
         *         the intersection point on the plane, and the second point will be the original segment end point. If the
         *         direction of the line is negative with respect to this plane's normal vector, the first point in the
         *         array will be the original segment's begin point, and the second point will be the intersection point on
         *         the plane. If the segment does not intersect the plane, null is returned. If the segment is coincident
         *         with the plane, the input points are returned, in their input order.
         *
         * @throws {ArgumentError} If either point is null or undefined.
         */
        Plane.prototype.clip = function (pointA, pointB) {
            if (!pointA || !pointB) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Plane", "clip", "missingPoint"));
            }

            if (pointA.equals(pointB)) {
                return null;
            }

            // Get the projection of the segment onto the plane.
            var line = Line.fromSegment(pointA, pointB),
                lDotV = this.normal.dot(line.direction),
                lDotS, t, p;

            // Are the line and plane parallel?
            if (lDotV === 0) { // line and plane are parallel and may be coincident.
                lDotS = this.dot(line.origin);
                if (lDotS === 0) {
                    return [pointA, pointB]; // line is coincident with the plane
                } else {
                    return null; // line is not coincident with the plane.
                }
            }

            // Not parallel so the line intersects. But does the segment intersect?
            t = -this.dot(line.origin) / lDotV; // lDotS / lDotV
            if (t < 0 || t > 1) { // segment does not intersect
                return null;
            }

            p = line.pointAt(t, new Vec3(0, 0, 0));
            if (lDotV > 0) {
                return [p, pointB];
            } else {
                return [pointA, p];
            }
        };

        return Plane;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports Rectangle
 */
define('geom/Rectangle',[
        '../util/Logger'
    ],
    function (Logger) {
        "use strict";

        /**
         * Constructs a rectangle with a specified origin and size.
         * @alias Rectangle
         * @constructor
         * @classdesc Represents a rectangle in 2D Cartesian coordinates.
         * @param {Number} x The X coordinate of the rectangle's origin.
         * @param {Number} y The Y coordinate of the rectangle's origin.
         * @param {Number} width The rectangle's width.
         * @param {Number} height The rectangle's height.
         */
        var Rectangle = function (x, y, width, height) {

            /**
             * The X coordinate of this rectangle's origin.
             * @type {Number}
             */
            this.x = x;

            /**
             * The Y coordinate of this rectangle's origin.
             * @type {Number}
             */
            this.y = y;

            /**
             * This rectangle's width.
             * @type {Number}
             */
            this.width = width;

            /**
             * This rectangle's height.
             * @type {Number}
             */
            this.height = height;
        };

        /**
         * Sets all this rectangle's properties.
         * @param {Number} x The X coordinate of the rectangle's origin.
         * @param {Number} y The Y coordinate of the rectangle's origin.
         * @param {Number} width The rectangle's width.
         * @param {Number} height The rectangle's height.
         */
        Rectangle.prototype.set = function(x, y, width, height) {
            this.x = x;
            this.y = y;
            this.width = width;
            this.height = height;
        };

        /**
         * Returns the minimum X value of this rectangle.
         * @returns {Number} The rectangle's minimum X value.
         */
        Rectangle.prototype.getMinX = function () {
            return this.x;
        };

        /**
         * Returns the minimum Y value of this rectangle.
         * @returns {Number} The rectangle's minimum Y value.
         */
        Rectangle.prototype.getMinY = function () {
            return this.y;
        };

        /**
         * Returns the maximum X value of this rectangle.
         * @returns {Number} The rectangle's maximum X value.
         */
        Rectangle.prototype.getMaxX = function () {
            return this.x + this.width;
        };

        /**
         * Returns the maximum Y value of this rectangle.
         * @returns {Number} The rectangle's maximum Y value.
         */
        Rectangle.prototype.getMaxY = function () {
            return this.y + this.height;
        };

        /**
         * Indicates whether this rectangle contains a specified point.
         * @param {Vec2} point The point to test.
         * @returns {Boolean} true if this rectangle contains the specified point, otherwise false.
         */
        Rectangle.prototype.containsPoint = function (point) {
            return point[0] >= this.x && point[0] <= (this.x + this.width)
                && point[1] >= this.y && point[1] <= (this.y + this.height);
        };
        /**
         *
         * Indicates whether this rectangle intersects a specified one.
         * @param {Rectangle} that The rectangle to test.
         * @returns {Boolean} true if this triangle and the specified one intersect, otherwise false.
         */
        Rectangle.prototype.intersects = function (that) {
            if ((that.x + that.width) < this.x)
                return false;

            if (that.x > (this.x + this.width))
                return false;

            if ((that.y + that.height) < this.y)
                return false;

            //noinspection RedundantIfStatementJS
            if (that.y > (this.y + this.height))
                return false;

            return true;
        };

        /**
         * Indicates whether this rectangle intersects any rectangle in a specified array of rectangles.
         * @param {Rectangle[]} rectangles The rectangles to test intersection with.
         * @returns {Boolean} true if this rectangle intersects any rectangle in the array, otherwise false.
         */
        Rectangle.prototype.intersectsRectangles = function (rectangles) {
            if (rectangles) {
                for (var i = 0; i < rectangles.length; i++){
                    if (this.intersects(rectangles[i])) {
                        return true;
                    }
                }
            }

            return false;
        };

        /**
         * Returns a string representation of this object.
         * @returns {String} A string representation of this object.
         */
        Rectangle.prototype.toString = function () {
            return this.x + ", " + this.y + ", " + this.width + ", " + this.height;
        };

        return Rectangle;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
define('util/WWMath',[
        '../geom/Angle',
        '../error/ArgumentError',
        '../geom/Line',
        '../util/Logger',
        '../geom/Rectangle',
        '../geom/Vec3'
    ],
    function (Angle,
              ArgumentError,
              Line,
              Logger,
              Rectangle,
              Vec3) {
        "use strict";
        /**
         * Provides math constants and functions.
         * @exports WWMath
         */
        var WWMath = {

            /**
             * Returns a number within the range of a specified minimum and maximum.
             * @param {Number} value The value to clamp.
             * @param {Number} minimum The minimum value to return.
             * @param {Number} maximum The maximum value to return.
             * @returns {Number} The minimum value if the specified value is less than the minimum, the maximum value if
             * the specified value is greater than the maximum, otherwise the value specified is returned.
             */
            clamp: function (value, minimum, maximum) {
                return value < minimum ? minimum : value > maximum ? maximum : value;
            },

            /**
             * Computes a number between two numbers.
             * @param amount {Number} The relative distance between the numbers at which to compute the new number. This
             * should normally be a number between 0 and 1 but whatever number is specified is applied.
             * @param {Number} value1 The first number.
             * @param {Number} value2 The second number.
             * @returns {Number} the computed value.
             */
            interpolate: function (amount, value1, value2) {
                return (1 - amount) * value1 + amount * value2;
            },

            /**
             * Returns the cube root of a specified value.
             * @param {Number} x The value whose cube root is computed.
             * @returns {Number} The cube root of the specified number.
             */
            cbrt: function (x) {
                // Use the built-in version if it exists. cbrt() is defined in ECMA6.
                if (typeof Math.cbrt == 'function') {
                    return Math.cbrt(x);
                } else {
                    return Math.pow(x, 1 / 3);
                }
            },

            /**
             * Computes the Cartesian intersection point of a specified line with an ellipsoid.
             * @param {Line} line The line for which to compute the intersection.
             * @param {Number} equatorialRadius The ellipsoid's major radius.
             * @param {Number} polarRadius The ellipsoid's minor radius.
             * @param {Vec3} result A pre-allocated Vec3 instance in which to return the computed point.
             * @returns {boolean} true if the line intersects the ellipsoid, otherwise false
             * @throws {ArgumentError} If the specified line or result is null or undefined.
             * @deprecated utilize the Globe.intersectsLine method attached implementation
             */
            computeEllipsoidalGlobeIntersection: function (line, equatorialRadius, polarRadius, result) {
                if (!line) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "computeEllipsoidalGlobeIntersection", "missingLine"));
                }

                if (!result) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "computeEllipsoidalGlobeIntersection", "missingResult"));
                }

                // Taken from "Mathematics for 3D Game Programming and Computer Graphics, Second Edition", Section 5.2.3.
                //
                // Note that the parameter n from in equations 5.70 and 5.71 is omitted here. For an ellipsoidal globe this
                // parameter is always 1, so its square and its product with any other value simplifies to the identity.

                var vx = line.direction[0],
                    vy = line.direction[1],
                    vz = line.direction[2],
                    sx = line.origin[0],
                    sy = line.origin[1],
                    sz = line.origin[2],
                    m = equatorialRadius / polarRadius, // ratio of the x semi-axis length to the y semi-axis length
                    m2 = m * m,
                    r2 = equatorialRadius * equatorialRadius, // nominal radius squared
                    a = vx * vx + m2 * vy * vy + vz * vz,
                    b = 2 * (sx * vx + m2 * sy * vy + sz * vz),
                    c = sx * sx + m2 * sy * sy + sz * sz - r2,
                    d = b * b - 4 * a * c, // discriminant
                    t;

                if (d < 0) {
                    return false;
                } else {
                    t = (-b - Math.sqrt(d)) / (2 * a);
                    result[0] = sx + vx * t;
                    result[1] = sy + vy * t;
                    result[2] = sz + vz * t;
                    return true;
                }
            },

            /**
             * Returns the normal vector corresponding to the triangle defined by three vertices (a, b, c).
             *
             * @param {Vec3} a The triangle's first vertex.
             * @param {Vec3} b The triangle's second vertex.
             * @param {Vec3} c The triangle's third vertex.
             *
             * @return {Vec3} the triangle's unit-length normal vector.
             *
             * @throws {ArgumentError} If the specified vertices are null or undefined.
             */
            computeTriangleNormal: function (a, b, c) {
                if (!a || !b || !c) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "computeTriangleNormal", "missingVertex"));
                }

                var x = ((b[1] - a[1]) * (c[2] - a[2])) - ((b[2] - a[2]) * (c[1] - a[1]));
                var y = ((b[2] - a[2]) * (c[0] - a[0])) - ((b[0] - a[0]) * (c[2] - a[2]));
                var z = ((b[0] - a[0]) * (c[1] - a[1])) - ((b[1] - a[1]) * (c[0] - a[0]));

                var length = (x * x) + (y * y) + (z * z);
                if (length == 0) {
                    return new Vec3(x, y, z);
                }

                length = Math.sqrt(length);
                return new Vec3(x / length, y / length, z / length);
            },

            /**
             * Computes the Cartesian intersection point of a specified line with a triangle.
             * @param {Line} line The line for which to compute the intersection.
             * @param {Vec3} vertex0 The triangle's first vertex.
             * @param {Vec3} vertex1 The triangle's second vertex.
             * @param {Vec3} vertex2 The triangle's third vertex.
             * @param {Vec3} result A pre-allocated Vec3 instance in which to return the computed point.
             * @returns {boolean} true if the line intersects the triangle, otherwise false
             * @throws {ArgumentError} If the specified line, vertex or result is null or undefined.
             */
            computeTriangleIntersection: function (line, vertex0, vertex1, vertex2, result) {
                if (!line) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "computeTriangleIntersection", "missingLine"));
                }

                if (!vertex0 || !vertex1 || !vertex2) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "computeTriangleIntersection", "missingVertex"));
                }

                if (!result) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "computeTriangleIntersection", "missingResult"));
                }

                // Taken from Moller and Trumbore
                // https://www.cs.virginia.edu/~gfx/Courses/2003/ImageSynthesis/papers/Acceleration/Fast%20MinimumStorage%20RayTriangle%20Intersection.pdf

                var vx = line.direction[0],
                    vy = line.direction[1],
                    vz = line.direction[2],
                    sx = line.origin[0],
                    sy = line.origin[1],
                    sz = line.origin[2],
                    EPSILON = 0.00001;

                // find vectors for two edges sharing point a: vertex1 - vertex0 and vertex2 - vertex0
                var edge1x = vertex1[0] - vertex0[0],
                    edge1y = vertex1[1] - vertex0[1],
                    edge1z = vertex1[2] - vertex0[2],
                    edge2x = vertex2[0] - vertex0[0],
                    edge2y = vertex2[1] - vertex0[1],
                    edge2z = vertex2[2] - vertex0[2];

                // Compute cross product of line direction and edge2
                var px = (vy * edge2z) - (vz * edge2y),
                    py = (vz * edge2x) - (vx * edge2z),
                    pz = (vx * edge2y) - (vy * edge2x);

                // Get determinant
                var det = edge1x * px + edge1y * py + edge1z * pz; // edge1 dot p
                if (det > -EPSILON && det < EPSILON) { // if det is near zero then ray lies in plane of triangle
                    return false;
                }

                var inv_det = 1.0 / det;

                // Compute distance for vertex A to ray origin: origin - vertex0
                var tx = sx - vertex0[0],
                    ty = sy - vertex0[1],
                    tz = sz - vertex0[2];

                // Calculate u parameter and test bounds: 1/det * t dot p
                var u = inv_det * (tx * px + ty * py + tz * pz);
                if (u < -EPSILON || u > 1 + EPSILON) {
                    return false;
                }

                // Prepare to test v parameter: t cross edge1
                var qx = (ty * edge1z) - (tz * edge1y),
                    qy = (tz * edge1x) - (tx * edge1z),
                    qz = (tx * edge1y) - (ty * edge1x);

                // Calculate v parameter and test bounds: 1/det * dir dot q
                var v = inv_det * (vx * qx + vy * qy + vz * qz);
                if (v < -EPSILON || u + v > 1 + EPSILON) {
                    return false;
                }

                // Calculate the point of intersection on the line: t = 1/det * edge2 dot q
                var t = inv_det * (edge2x * qx + edge2y * qy + edge2z * qz);
                if (t < 0) {
                    return false;
                } else {
                    result[0] = sx + vx * t;
                    result[1] = sy + vy * t;
                    result[2] = sz + vz * t;
                    return true;
                }
            },

            /**
             * Computes the Cartesian intersection point(s) of a specified line with a non-indexed list of
             * triangle vertices.
             * @param {Line} line The line for which to compute the intersection(s).
             * @param {Vec3[]} points The list of triangle vertices arranged such that each
             * 3-tuple, (i,i+1,i+2), specifies a triangle.
             * @param {Vec3[]} results The Cartesian intersection point(s) if any.
             * @returns {boolean} true if the line intersects any triangle, otherwise false
             * @throws {ArgumentError} If any of the arguments is not supplied.
             */
            computeTriangleListIntersection: function (line, points, results) {
                if (!line) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "computeIndexedTrianglesIntersection", "missingLine"));
                }

                if (!points) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "computeIndexedTrianglesIntersection", "missingPoints"));
                }

                if (!results) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "computeIndexedTrianglesIntersection", "missingResults"));
                }
                var iPoint = new Vec3(0, 0, 0);
                for (var i = 0, len = points.length; i < len; i += 3) {
                    if (WWMath.computeTriangleIntersection(line, points[i], points[i + 1], points[i + 2], iPoint)) {
                        results.push(iPoint);
                        iPoint = new Vec3(0, 0, 0);
                    }
                }

                return results.length > 0;
            },

            computeIndexedTrianglesIntersection: function (line, points, indices, results) {
                if (!line) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "computeIndexedTrianglesIntersection", "missingLine"));
                }

                if (!points) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "computeIndexedTrianglesIntersection", "missingPoints"));
                }

                if (!indices) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "computeIndexedTrianglesIntersection", "missingIndices"));
                }

                if (!results) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "computeIndexedTrianglesIntersection", "missingResults"));
                }

                var v0 = new Vec3(0, 0, 0),
                    v1 = new Vec3(0, 0, 0),
                    v2 = new Vec3(0, 0, 0),
                    iPoint = new Vec3(0, 0, 0);

                for (var i = 0, len = indices.length; i < len; i += 3) {
                    var i0 = 3 * indices[i],
                        i1 = 3 * indices[i + 1],
                        i2 = 3 * indices[i + 2];

                    v0[0] = points[i0];
                    v0[1] = points[i0 + 1];
                    v0[2] = points[i0 + 2];

                    v1[0] = points[i1];
                    v1[1] = points[i1 + 1];
                    v1[2] = points[i1 + 2];

                    v2[0] = points[i2];
                    v2[1] = points[i2 + 1];
                    v2[2] = points[i2 + 2];

                    if (WWMath.computeTriangleIntersection(line, v0, v1, v2, iPoint)) {
                        results.push(iPoint);
                        iPoint = new Vec3(0, 0, 0);
                    }
                }

                return results.length > 0;
            },

            /**
             * Computes the Cartesian intersection points of a specified line with a triangle strip. The triangle strip
             * is specified by a list of vertex points and a list of indices indicating the triangle strip tessellation
             * of those vertices. The triangle strip indices are interpreted in the same manner as WebGL, where each
             * index indicates a vertex position rather than an actual index into the points array (e.g. a triangle
             * strip index of 1 indicates the XYZ tuple starting at array index 3). This is equivalent to calling
             * computeTriangleIntersection for each individual triangle in the triangle strip, but is potentially much
             * more efficient.
             * @param {Line} line The line for which to compute the intersection.
             * @param {Array} points The list of vertex points, organized as a list of tightly-packed XYZ tuples.
             * @param {Array} indices The list of triangle strip indices, organized as a list of vertex positions.
             * @param {Array} results A pre-allocated array instance in which to return the intersection points as
             * {@link Vec3} instances.
             * @throws {ArgumentError} If the specified line, points, indices or results is null or undefined.
             */
            computeTriStripIntersections: function (line, points, indices, results) {
                if (!line) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "computeTriStripIntersections", "missingLine"));
                }

                if (!points) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "computeTriStripIntersections", "missingPoints"));
                }

                if (!indices) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "computeTriStripIntersections", "missingIndices"));
                }

                if (!results) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "computeTriStripIntersections", "missingResults"));
                }

                // Taken from Moller and Trumbore
                // https://www.cs.virginia.edu/~gfx/Courses/2003/ImageSynthesis/papers/Acceleration/Fast%20MinimumStorage%20RayTriangle%20Intersection.pdf

                // Adapted from the original ray-triangle intersection algorithm to optimize for ray-triangle strip
                // intersection. We optimize by reusing constant terms, replacing use of Vec3 with inline primitives,
                // and exploiting the triangle strip organization to reuse computations common to adjacent triangles.
                // These optimizations reduce worst-case terrain picking performance by approximately 50% in Chrome on a
                // 2010 iMac and a Nexus 9.

                var vx = line.direction[0],
                    vy = line.direction[1],
                    vz = line.direction[2],
                    sx = line.origin[0],
                    sy = line.origin[1],
                    sz = line.origin[2],
                    vert0x, vert0y, vert0z,
                    vert1x, vert1y, vert1z,
                    vert2x, vert2y, vert2z,
                    edge1x, edge1y, edge1z,
                    edge2x, edge2y, edge2z,
                    px, py, pz,
                    tx, ty, tz,
                    qx, qy, qz,
                    u, v, t,
                    det, inv_det,
                    index,
                    EPSILON = 0.00001;

                // Get the triangle strip's first vertex.
                index = 3 * indices[0];
                vert1x = points[index++];
                vert1y = points[index++];
                vert1z = points[index];

                // Get the triangle strip's second vertex.
                index = 3 * indices[1];
                vert2x = points[index++];
                vert2y = points[index++];
                vert2z = points[index];

                // Compute the intersection of each triangle with the specified ray.
                for (var i = 2, len = indices.length; i < len; i++) {
                    // Move the last two vertices into the first two vertices. This takes advantage of the triangle
                    // strip's structure and avoids redundant reads from points and indices. During the first
                    // iteration this places the triangle strip's first three vertices in vert0, vert1 and vert2,
                    // respectively.
                    vert0x = vert1x;
                    vert0y = vert1y;
                    vert0z = vert1z;
                    vert1x = vert2x;
                    vert1y = vert2y;
                    vert1z = vert2z;

                    // Get the triangle strip's next vertex.
                    index = 3 * indices[i];
                    vert2x = points[index++];
                    vert2y = points[index++];
                    vert2z = points[index];

                    // find vectors for two edges sharing point a: vert1 - vert0 and vert2 - vert0
                    edge1x = vert1x - vert0x;
                    edge1y = vert1y - vert0y;
                    edge1z = vert1z - vert0z;
                    edge2x = vert2x - vert0x;
                    edge2y = vert2y - vert0y;
                    edge2z = vert2z - vert0z;

                    // Compute cross product of line direction and edge2
                    px = (vy * edge2z) - (vz * edge2y);
                    py = (vz * edge2x) - (vx * edge2z);
                    pz = (vx * edge2y) - (vy * edge2x);

                    // Get determinant
                    det = edge1x * px + edge1y * py + edge1z * pz; // edge1 dot p
                    if (det > -EPSILON && det < EPSILON) { // if det is near zero then ray lies in plane of triangle
                        continue;
                    }

                    inv_det = 1.0 / det;

                    // Compute distance for vertex A to ray origin: origin - vert0
                    tx = sx - vert0x;
                    ty = sy - vert0y;
                    tz = sz - vert0z;

                    // Calculate u parameter and test bounds: 1/det * t dot p
                    u = inv_det * (tx * px + ty * py + tz * pz);
                    if (u < -EPSILON || u > 1 + EPSILON) {
                        continue;
                    }

                    // Prepare to test v parameter: tvec cross edge1
                    qx = (ty * edge1z) - (tz * edge1y);
                    qy = (tz * edge1x) - (tx * edge1z);
                    qz = (tx * edge1y) - (ty * edge1x);

                    // Calculate v parameter and test bounds: 1/det * dir dot q
                    v = inv_det * (vx * qx + vy * qy + vz * qz);
                    if (v < -EPSILON || u + v > 1 + EPSILON) {
                        continue;
                    }

                    // Calculate the point of intersection on the line: t = 1/det * edge2 dot q
                    t = inv_det * (edge2x * qx + edge2y * qy + edge2z * qz);
                    if (t >= 0) {
                        results.push(new Vec3(sx + vx * t, sy + vy * t, sz + vz * t));
                    }
                }
            },

            /**
             * Computes the absolute value of a specified value.
             * @param {Number} a The value whose absolute value to compute.
             * @returns {Number} The absolute value of the specified number.
             */
            fabs: function (a) {
                return a >= 0 ? a : -a;
            },

            /**
             * Computes the floating-point modulus of a specified number.
             * @param {Number} number The number whose modulus to compute.
             * @param {Number} modulus The modulus.
             * @returns {Number} The remainder after dividing the number by the modulus: number % modulus.
             */
            fmod: function (number, modulus) {
                return modulus === 0 ? 0 : number - Math.floor(number / modulus) * modulus;
            },

            /**
             * Returns the fractional part of a specified number
             * @param {Number} number The number whose fractional part to compute.
             * @returns {Number} The fractional part of the specified number: number - floor(number).
             */
            fract: function (number) {
                return number - Math.floor(number);
            },

            /**
             * Returns the integer modulus of a specified number. This differs from the % operator in that
             * the result is always positive when the modulus is positive. For example -1 % 10 = -1,
             * whereas mod(-1, 10) = 1.
             * @param {Number} number The number whose modulus to compute.
             * @param {Number} modulus The modulus.
             * @returns {Number} The remainder after dividing the number by the modulus.
             */
            mod: function (number, modulus) {
                return ((number % modulus) + modulus) % modulus;
            },

            /**
             * Returns the maximum of two specified numbers.
             * @param {Number} value1 The first value to compare.
             * @param {Number} value2 The second value to compare.
             * @returns {Number} The maximum of the two specified values.
             */
            max: function (value1, value2) {
                return value1 > value2 ? value1 : value2;
            },

            /**
             * Computes the axes of a local coordinate system on the specified globe, placing the resultant axes in the specified
             * axis arguments.
             *
             * Upon return the specified axis arguments contain three orthogonal axes identifying the X, Y, and Z axes. Each
             * axis has unit length.
             *
             * The local coordinate system is defined such that the Z axis maps to the globe's surface normal at the point, the
             * Y axis maps to the north pointing tangent, and the X axis maps to the east pointing tangent.
             *
             * @param {Vec3} origin The local coordinate system origin, in model coordinates.
             * @param {Globe} globe The globe the coordinate system is relative to.
             * @param {Vec3} xAxisResult A pre-allocated Vec3 in which to return the computed X axis.
             * @param {Vec3} yAxisResult A pre-allocated Vec3 in which to return the computed Y axis.
             * @param {Vec3} zAxisResult A pre-allocated Vec3 in which to return the computed Z axis.
             * @throws {ArgumentError} If any argument is null or undefined.
             */
            localCoordinateAxesAtPoint: function (origin, globe, xAxisResult, yAxisResult, zAxisResult) {
                if (!origin) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "localCoordinateAxesAtPoint", "missingVector"));
                }

                if (!globe) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "localCoordinateAxesAtPoint", "missingGlobe"));
                }

                if (!xAxisResult || !yAxisResult || !zAxisResult) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath",
                        "localCoordinateAxesAtPoint", "missingResult"));
                }

                var x = origin[0],
                    y = origin[1],
                    z = origin[2];

                // Compute the z axis from the surface normal in model coordinates. This axis is used to determine the other two
                // axes, and is the only constant in the computations below.
                globe.surfaceNormalAtPoint(x, y, z, zAxisResult);

                // Compute the y axis from the north pointing tangent in model coordinates. This axis is known to be orthogonal to
                // the z axis, and is therefore used to compute the x axis.
                globe.northTangentAtPoint(x, y, z, yAxisResult);

                // Compute the x axis as the cross product of the y and z axes. This ensures that the x and z axes are orthogonal.
                xAxisResult.set(yAxisResult[0], yAxisResult[1], yAxisResult[2]);
                xAxisResult.cross(zAxisResult);
                xAxisResult.normalize();

                // Re-compute the y axis as the cross product of the z and x axes. This ensures that all three axes are orthogonal.
                // Though the initial y axis computed above is likely to be very nearly orthogonal, we re-compute it using cross
                // products to reduce the effect of floating point rounding errors caused by working with Earth sized coordinates.
                yAxisResult.set(zAxisResult[0], zAxisResult[1], zAxisResult[2]);
                yAxisResult.cross(xAxisResult);
                yAxisResult.normalize();
            },

            /**
             * Computes the near clip distance that corresponds to a specified far clip distance and resolution at the far clip
             * plane.
             *
             * This computes a near clip distance appropriate for use in [perspectiveFrustumRect]{@link WWMath#perspectiveFrustumRectangle}
             * and [setToPerspectiveProjection]{@link Matrix#setToPerspectiveProjection}. This returns zero if either the distance or the
             * resolution are zero.
             *
             * @param {Number} farDistance The far clip distance, in meters.
             * @param {Number} farResolution The depth resolution at the far clip plane, in meters.
             * @param {Number} depthBits The number of bit-planes in the depth buffer.
             * @returns {Number} The near clip distance, in meters.
             * @throws {ArgumentError} If either the distance or resolution is negative, or if the depth bits is less
             * than one.
             */
            perspectiveNearDistanceForFarDistance: function (farDistance, farResolution, depthBits) {
                if (farDistance < 0) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath", "perspectiveNearDistanceForFarDistance",
                        "The specified distance is negative."));
                }

                if (farResolution < 0) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath", "perspectiveNearDistanceForFarDistance",
                        "The specified resolution is negative."));
                }

                if (depthBits < 1) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath", "perspectiveNearDistanceForFarDistance",
                        "The specified depth bits is negative."));
                }

                var maxDepthValue = (1 << depthBits) - 1;

                return farDistance / (maxDepthValue / (1 - farResolution / farDistance) - maxDepthValue + 1);
            },

            /**
             * Computes the maximum near clip distance for a perspective projection that avoids clipping an object at a
             * given distance from the eye point.
             * <p/>
             * This computes a near clip distance appropriate for use in perspectiveFrustumRect and
             * Matrix.setToPerspectiveProjection. The given distance should specify the smallest distance between the
             * eye and the object being viewed, but may be an approximation if an exact distance is not required.
             *
             * @param {Number} fovyDegrees The camera vertical field of view.
             * @param {Number} distanceToSurface The distance from the perspective eye point to the nearest object, in
             * meters.
             * @returns {Number} The maximum near clip distance, in meters.
             * @throws {ArgumentError} If the specified width or height is less than or equal to zero, or if the
             * specified distance is negative.
             */
            perspectiveNearDistance: function (fovyDegrees, distanceToSurface) {
                if (fovyDegrees <= 0 || fovyDegrees >= 180) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath", "perspectiveNearDistance",
                        "invalidFieldOfView"));
                }

                if (distanceToSurface < 0) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath", "perspectiveNearDistance",
                        "The specified distance is negative."));
                }

                var tanHalfFov = Math.tan(0.5 * fovyDegrees / 180 * Math.PI);
                return distanceToSurface / (2 * Math.sqrt(2 * tanHalfFov * tanHalfFov + 1));
            },

            /**
             * Computes the coordinates of a rectangle carved out of a perspective projection's frustum at a given
             * distance in model coordinates. This returns an empty rectangle if the specified distance is zero.
             *
             * @param {Number} viewportWidth The viewport width, in screen coordinates.
             * @param {Number} viewportHeight The viewport height, in screen coordinates.
             * @param {Number} distance The distance along the negative Z axis, in model coordinates.
             * @returns {Rectangle} The frustum rectangle, in model coordinates.
             * @throws {ArgumentError} If the specified width or height is less than or equal to zero, or if the
             * specified distance is negative.
             */
            perspectiveFrustumRectangle: function (viewportWidth, viewportHeight, distance) {
                if (viewportWidth <= 0) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath", "perspectiveFrustumRectangle",
                        "invalidWidth"));
                }

                if (viewportHeight <= 0) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath", "perspectiveFrustumRectangle",
                        "invalidHeight"));
                }

                if (distance < 0) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath", "perspectiveFrustumRectangle",
                        "The specified distance is negative."));
                }

                // Assumes a 45 degree horizontal field of view.
                var width = distance,
                    height = distance * viewportHeight / viewportWidth;

                return new Rectangle(-width / 2, -height / 2, width, height);
            },

            /**
             * Computes the vertical size of a pixel in model coordinates at a given distance from the eye point in a
             * perspective projection. This returns zero if the specified distance is zero. The returned size is
             * undefined if the distance is less than zero.
             * <p/>
             * This method assumes the model of a screen composed of rectangular pixels, where pixel coordinates denote
             * infinitely thin space between pixels. The units of the returned size are in model coordinates per pixel
             * (usually meters per pixel).
             *
             * @param {Number} viewportWidth The viewport width, in screen coordinates.
             * @param {Number} viewportHeight The viewport height, in screen coordinates.
             * @param {Number} distance The distance from the perspective eye point at which to determine pixel size, in
             * model coordinates.
             * @returns {Number} The pixel size at the specified distance from the eye point, in model coordinates per
             * pixel.
             * @throws {ArgumentError} If the specified width or height is less than or equal to zero, or if the
             * specified distance is negative.
             */
            perspectivePixelSize: function (viewportWidth, viewportHeight, distance) {
                if (viewportWidth <= 0) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath", "perspectivePixelSize",
                        "invalidWidth"));
                }

                if (viewportHeight <= 0) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath", "perspectivePixelSize",
                        "invalidHeight"));
                }

                if (distance < 0) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath", "perspectivePixelSize",
                        "The specified distance is negative."));
                }

                var frustumHeight = WWMath.perspectiveFrustumRectangle(viewportWidth, viewportHeight, distance).height;
                return frustumHeight / viewportHeight;
            },

            /**
             * Computes the bounding rectangle for a unit quadrilateral after applying a transformation matrix to that
             * quadrilateral.
             * @param {Matrix} transformMatrix The matrix to apply to the unit quadrilateral.
             * @returns {Rectangle} The computed bounding rectangle.
             */
            boundingRectForUnitQuad: function (transformMatrix) {
                if (!transformMatrix) {
                    throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "WWMath", "boundingRectForUnitQuad",
                        "missingMatrix"));
                }

                var m = transformMatrix,
                    // transform of (0, 0)
                    x1 = m[3],
                    y1 = m[7],
                    // transform of (1, 0)
                    x2 = m[0] + m[3],
                    y2 = m[4] + m[7],
                    // transform of (0, 1)
                    x3 = m[1] + m[3],
                    y3 = m[5] + m[7],
                    // transform of (1, 1)
                    x4 = m[0] + m[1] + m[3],
                    y4 = m[4] + m[5] + m[7],
                    minX = Math.min(Math.min(x1, x2), Math.min(x3, x4)),
                    maxX = Math.max(Math.max(x1, x2), Math.max(x3, x4)),
                    minY = Math.min(Math.min(y1, y2), Math.min(y3, y4)),
                    maxY = Math.max(Math.max(y1, y2), Math.max(y3, y4));

                return new Rectangle(minX, minY, maxX - minX, maxY - minY);
            },

            /**
             * Indicates whether a specified value is a power of two.
             * @param {Number} value The value to test.
             * @returns {boolean} <code>true</code> if the specified value is a power of two,
             * otherwise <code>false</code>.
             */
            isPowerOfTwo: function (value) {
                return value != 0 && (value & (value - 1)) === 0;
            },

            /**
             * Determine the sign of a number.
             * @param {Number} value The value to determine the sign of.
             * @returns {Number} 1, -1, or 0, depending on the sign of the value.
             */
            signum: function (value) {
                return value > 0 ? 1 : value < 0 ? -1 : 0;
            },

            /**
             * Calculates the Gudermannian inverse used to unproject Mercator projections.
             * @param {Number} latitude The latitude in degrees.
             * @returns {Number} The Gudermannian inverse for the specified latitude.
             */
            gudermannianInverse: function (latitude) {
                return Math.log(Math.tan(Math.PI / 4 + (latitude * Angle.DEGREES_TO_RADIANS) / 2)) / Math.PI;
            },

            epsg3857ToEpsg4326: function (easting, northing) {
                var r = 6.378137e6,
                    latRadians = (Math.PI / 2) - 2 * Math.atan(Math.exp(-northing / r)),
                    lonRadians = easting / r;

                return [
                    WWMath.clamp(latRadians * Angle.RADIANS_TO_DEGREES, -90, 90),
                    WWMath.clamp(lonRadians * Angle.RADIANS_TO_DEGREES, -180, 180)
                ];
            },

            /**
             * Returns the value that is the nearest power of 2 less than or equal to the given value.
             * @param {Number} value the reference value. The power of 2 returned is less than or equal to this value.
             * @returns {Number} the value that is the nearest power of 2 less than or equal to the reference value
             */
            powerOfTwoFloor: function (value) {
                var power = Math.floor(Math.log(value) / Math.log(2));
                return Math.pow(2, power);
            },

            /**
             * Restricts an angle to the range [0, 360] degrees, wrapping angles outside the range.
             * Wrapping takes place as though traversing the edge of a unit circle;
             * angles less than 0 wrap back to 360, while angles greater than 360 wrap back to 0.
             *
             * @param {Number} degrees the angle to wrap in degrees
             *
             * @return {Number} the specified angle wrapped to [0, 360] degrees
             */
            normalizeAngle360: function (degrees) {
                var angle = degrees % 360;
                return angle >= 0 ? angle : (angle < 0 ? 360 + angle : 360 - angle);
            }
        };

        return WWMath;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports Location
 */
define('geom/Location',[
        '../geom/Angle',
        '../error/ArgumentError',
        '../util/Logger',
        '../geom/Plane',
        '../geom/Vec3',
        '../util/WWMath'
    ],
    function (Angle,
              ArgumentError,
              Logger,
              Plane,
              Vec3,
              WWMath) {
        "use strict";

        /**
         * Constructs a location from a specified latitude and longitude in degrees.
         * @alias Location
         * @constructor
         * @classdesc Represents a latitude, longitude pair in degrees.
         * @param {Number} latitude The latitude in degrees.
         * @param {Number} longitude The longitude in degrees.
         */
        var Location = function (latitude, longitude) {
            /**
             * The latitude in degrees.
             * @type {Number}
             */
            this.latitude = latitude;
            /**
             * The longitude in degrees.
             * @type {Number}
             */
            this.longitude = longitude;
        };

        /**
         * A location with latitude and longitude both 0.
         * @constant
         * @type {Location}
         */
        Location.ZERO = new Location(0, 0);

        /**
         * Creates a location from angles specified in radians.
         * @param {Number} latitudeRadians The latitude in radians.
         * @param {Number} longitudeRadians The longitude in radians.
         * @returns {Location} The new location with latitude and longitude in degrees.
         */
        Location.fromRadians = function (latitudeRadians, longitudeRadians) {
            return new Location(latitudeRadians * Angle.RADIANS_TO_DEGREES, longitudeRadians * Angle.RADIANS_TO_DEGREES);
        };

        /**
         * Copies this location to the latitude and longitude of a specified location.
         * @param {Location} location The location to copy.
         * @returns {Location} This location, set to the values of the specified location.
         * @throws {ArgumentError} If the specified location is null or undefined.
         */
        Location.prototype.copy = function (location) {
            if (!location) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "copy", "missingLocation"));
            }

            this.latitude = location.latitude;
            this.longitude = location.longitude;

            return this;
        };

        /**
         * Sets this location to the latitude and longitude.
         * @param {Number} latitude The latitude to set.
         * @param {Number} longitude The longitude to set.
         * @returns {Location} This location, set to the values of the specified latitude and longitude.
         */
        Location.prototype.set = function (latitude, longitude) {
            this.latitude = latitude;
            this.longitude = longitude;

            return this;
        };

        /**
         * Indicates whether this location is equal to a specified location.
         * @param {Location} location The location to compare this one to.
         * @returns {Boolean} <code>true</code> if this location is equal to the specified location, otherwise
         * <code>false</code>.
         */
        Location.prototype.equals = function (location) {
            return location
                && location.latitude === this.latitude && location.longitude === this.longitude;
        };

        /**
         * Compute a location along a path at a specified distance between two specified locations.
         * @param {String} pathType The type of path to assume. Recognized values are
         * [WorldWind.GREAT_CIRCLE]{@link WorldWind#GREAT_CIRCLE},
         * [WorldWind.RHUMB_LINE]{@link WorldWind#RHUMB_LINE} and
         * [WorldWind.LINEAR]{@link WorldWind#LINEAR}.
         * If the path type is not recognized then WorldWind.LINEAR is used.
         * @param {Number} amount The fraction of the path between the two locations at which to compute the new
         * location. This number should be between 0 and 1. If not, it is clamped to the nearest of those values.
         * @param {Location} location1 The starting location.
         * @param {Location} location2 The ending location.
         * @param {Location} result A Location in which to return the result.
         * @returns {Location} The specified result location.
         * @throws {ArgumentError} If either specified location or the result argument is null or undefined.
         */
        Location.interpolateAlongPath = function (pathType, amount, location1, location2, result) {
            if (!location1 || !location2) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "interpolateAlongPath", "missingLocation"));
            }

            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "interpolateAlongPath", "missingResult"));
            }

            if (pathType === WorldWind.GREAT_CIRCLE) {
                return this.interpolateGreatCircle(amount, location1, location2, result);
            } else if (pathType && pathType === WorldWind.RHUMB_LINE) {
                return this.interpolateRhumb(amount, location1, location2, result);
            } else {
                return this.interpolateLinear(amount, location1, location2, result);
            }
        };

        /**
         * Compute a location along a great circle path at a specified distance between two specified locations.
         * @param {Number} amount The fraction of the path between the two locations at which to compute the new
         * location. This number should be between 0 and 1. If not, it is clamped to the nearest of those values.
         * This function uses a spherical model, not elliptical.
         * @param {Location} location1 The starting location.
         * @param {Location} location2 The ending location.
         * @param {Location} result A Location in which to return the result.
         * @returns {Location} The specified result location.
         * @throws {ArgumentError} If either specified location or the result argument is null or undefined.
         */
        Location.interpolateGreatCircle = function (amount, location1, location2, result) {
            if (!location1 || !location2) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "interpolateGreatCircle", "missingLocation"));
            }
            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "interpolateGreatCircle", "missingResult"));
            }

            if (location1.equals(location2)) {
                result.latitude = location1.latitude;
                result.longitude = location1.longitude;
                return result;
            }

            var t = WWMath.clamp(amount, 0, 1),
                azimuthDegrees = this.greatCircleAzimuth(location1, location2),
                distanceRadians = this.greatCircleDistance(location1, location2);

            return this.greatCircleLocation(location1, azimuthDegrees, t * distanceRadians, result);
        };

        /**
         * Computes the azimuth angle (clockwise from North) that points from the first location to the second location.
         * This angle can be used as the starting azimuth for a great circle arc that begins at the first location, and
         * passes through the second location.
         * This function uses a spherical model, not elliptical.
         * @param {Location} location1 The starting location.
         * @param {Location} location2 The ending location.
         * @returns {Number} The computed azimuth, in degrees.
         * @throws {ArgumentError} If either specified location is null or undefined.
         */
        Location.greatCircleAzimuth = function (location1, location2) {
            if (!location1 || !location2) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "greatCircleAzimuth", "missingLocation"));
            }

            var lat1 = location1.latitude * Angle.DEGREES_TO_RADIANS,
                lat2 = location2.latitude * Angle.DEGREES_TO_RADIANS,
                lon1 = location1.longitude * Angle.DEGREES_TO_RADIANS,
                lon2 = location2.longitude * Angle.DEGREES_TO_RADIANS,
                x,
                y,
                azimuthRadians;

            if (lat1 == lat2 && lon1 == lon2) {
                return 0;
            }

            if (lon1 == lon2) {
                return lat1 > lat2 ? 180 : 0;
            }

            // Taken from "Map Projections - A Working Manual", page 30, equation 5-4b.
            // The atan2() function is used in place of the traditional atan(y/x) to simplify the case when x == 0.
            y = Math.cos(lat2) * Math.sin(lon2 - lon1);
            x = Math.cos(lat1) * Math.sin(lat2) - Math.sin(lat1) * Math.cos(lat2) * Math.cos(lon2 - lon1);
            azimuthRadians = Math.atan2(y, x);

            return isNaN(azimuthRadians) ? 0 : azimuthRadians * Angle.RADIANS_TO_DEGREES;
        };

        /**
         * Computes the great circle angular distance between two locations. The return value gives the distance as the
         * angle between the two positions. In radians, this angle is the arc length of the segment between the two
         * positions. To compute a distance in meters from this value, multiply the return value by the radius of the
         * globe.
         * This function uses a spherical model, not elliptical.
         *
         * @param {Location} location1 The starting location.
         * @param {Location} location2 The ending location.
         * @returns {Number} The computed distance, in radians.
         * @throws {ArgumentError} If either specified location is null or undefined.
         */
        Location.greatCircleDistance = function (location1, location2) {
            if (!location1 || !location2) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "greatCircleDistance", "missingLocation"));
            }

            var lat1Radians = location1.latitude * Angle.DEGREES_TO_RADIANS,
                lat2Radians = location2.latitude * Angle.DEGREES_TO_RADIANS,
                lon1Radians = location1.longitude * Angle.DEGREES_TO_RADIANS,
                lon2Radians = location2.longitude * Angle.DEGREES_TO_RADIANS,
                a,
                b,
                c,
                distanceRadians;

            if (lat1Radians == lat2Radians && lon1Radians == lon2Radians) {
                return 0;
            }

            // "Haversine formula," taken from https://en.wikipedia.org/wiki/Great-circle_distance#Formul.C3.A6
            a = Math.sin((lat2Radians - lat1Radians) / 2.0);
            b = Math.sin((lon2Radians - lon1Radians) / 2.0);
            c = a * a + Math.cos(lat1Radians) * Math.cos(lat2Radians) * b * b;
            distanceRadians = 2.0 * Math.asin(Math.sqrt(c));

            return isNaN(distanceRadians) ? 0 : distanceRadians;
        };

        /**
         * Computes the location on a great circle path corresponding to a given starting location, azimuth, and
         * arc distance.
         * This function uses a spherical model, not elliptical.
         *
         * @param {Location} location The starting location.
         * @param {Number} greatCircleAzimuthDegrees The azimuth in degrees.
         * @param {Number} pathLengthRadians The radian distance along the path at which to compute the end location.
         * @param {Location} result A Location in which to return the result.
         * @returns {Location} The specified result location.
         * @throws {ArgumentError} If the specified location or the result argument is null or undefined.
         */
        Location.greatCircleLocation = function (location, greatCircleAzimuthDegrees, pathLengthRadians, result) {
            if (!location) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "greatCircleLocation", "missingLocation"));
            }
            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "greatCircleLocation", "missingResult"));
            }

            if (pathLengthRadians == 0) {
                result.latitude = location.latitude;
                result.longitude = location.longitude;
                return result;
            }

            var latRadians = location.latitude * Angle.DEGREES_TO_RADIANS,
                lonRadians = location.longitude * Angle.DEGREES_TO_RADIANS,
                azimuthRadians = greatCircleAzimuthDegrees * Angle.DEGREES_TO_RADIANS,
                endLatRadians,
                endLonRadians;

            // Taken from "Map Projections - A Working Manual", page 31, equation 5-5 and 5-6.
            endLatRadians = Math.asin(Math.sin(latRadians) * Math.cos(pathLengthRadians) +
                Math.cos(latRadians) * Math.sin(pathLengthRadians) * Math.cos(azimuthRadians));
            endLonRadians = lonRadians + Math.atan2(
                    Math.sin(pathLengthRadians) * Math.sin(azimuthRadians),
                    Math.cos(latRadians) * Math.cos(pathLengthRadians) -
                    Math.sin(latRadians) * Math.sin(pathLengthRadians) * Math.cos(azimuthRadians));

            if (isNaN(endLatRadians) || isNaN(endLonRadians)) {
                result.latitude = location.latitude;
                result.longitude = location.longitude;
            } else {
                result.latitude = Angle.normalizedDegreesLatitude(endLatRadians * Angle.RADIANS_TO_DEGREES);
                result.longitude = Angle.normalizedDegreesLongitude(endLonRadians * Angle.RADIANS_TO_DEGREES);
            }

            return result;
        };

        /**
         * Compute a location along a rhumb path at a specified distance between two specified locations.
         * This function uses a spherical model, not elliptical.
         * @param {Number} amount The fraction of the path between the two locations at which to compute the new
         * location. This number should be between 0 and 1. If not, it is clamped to the nearest of those values.
         * @param {Location} location1 The starting location.
         * @param {Location} location2 The ending location.
         * @param {Location} result A Location in which to return the result.
         * @returns {Location} The specified result location.
         * @throws {ArgumentError} If either specified location or the result argument is null or undefined.
         */
        Location.interpolateRhumb = function (amount, location1, location2, result) {
            if (!location1 || !location2) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "interpolateRhumb", "missingLocation"));
            }
            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "interpolateRhumb", "missingResult"));
            }

            if (location1.equals(location2)) {
                result.latitude = location1.latitude;
                result.longitude = location1.longitude;
                return result;
            }

            var t = WWMath.clamp(amount, 0, 1),
                azimuthDegrees = this.rhumbAzimuth(location1, location2),
                distanceRadians = this.rhumbDistance(location1, location2);

            return this.rhumbLocation(location1, azimuthDegrees, t * distanceRadians, result);
        };

        /**
         * Computes the azimuth angle (clockwise from North) that points from the first location to the second location.
         * This angle can be used as the azimuth for a rhumb arc that begins at the first location, and
         * passes through the second location.
         * This function uses a spherical model, not elliptical.
         * @param {Location} location1 The starting location.
         * @param {Location} location2 The ending location.
         * @returns {Number} The computed azimuth, in degrees.
         * @throws {ArgumentError} If either specified location is null or undefined.
         */
        Location.rhumbAzimuth = function (location1, location2) {
            if (!location1 || !location2) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "rhumbAzimuth", "missingLocation"));
            }

            var lat1 = location1.latitude * Angle.DEGREES_TO_RADIANS,
                lat2 = location2.latitude * Angle.DEGREES_TO_RADIANS,
                lon1 = location1.longitude * Angle.DEGREES_TO_RADIANS,
                lon2 = location2.longitude * Angle.DEGREES_TO_RADIANS,
                dLon,
                dPhi,
                azimuthRadians;

            if (lat1 == lat2 && lon1 == lon2) {
                return 0;
            }

            dLon = lon2 - lon1;
            dPhi = Math.log(Math.tan(lat2 / 2.0 + Math.PI / 4) / Math.tan(lat1 / 2.0 + Math.PI / 4));

            // If lonChange over 180 take shorter rhumb across 180 meridian.
            if (WWMath.fabs(dLon) > Math.PI) {
                dLon = dLon > 0 ? -(2 * Math.PI - dLon) : (2 * Math.PI + dLon);
            }

            azimuthRadians = Math.atan2(dLon, dPhi);

            return isNaN(azimuthRadians) ? 0 : azimuthRadians * Angle.RADIANS_TO_DEGREES;
        };

        /**
         * Computes the rhumb angular distance between two locations. The return value gives the distance as the
         * angle between the two positions in radians. This angle is the arc length of the segment between the two
         * positions. To compute a distance in meters from this value, multiply the return value by the radius of the
         * globe.
         * This function uses a spherical model, not elliptical.
         *
         * @param {Location} location1 The starting location.
         * @param {Location} location2 The ending location.
         * @returns {Number} The computed distance, in radians.
         * @throws {ArgumentError} If either specified location is null or undefined.
         */
        Location.rhumbDistance = function (location1, location2) {
            if (!location1 || !location2) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "rhumbDistance", "missingLocation"));
            }

            var lat1 = location1.latitude * Angle.DEGREES_TO_RADIANS,
                lat2 = location2.latitude * Angle.DEGREES_TO_RADIANS,
                lon1 = location1.longitude * Angle.DEGREES_TO_RADIANS,
                lon2 = location2.longitude * Angle.DEGREES_TO_RADIANS,
                dLat,
                dLon,
                dPhi,
                q,
                distanceRadians;

            if (lat1 == lat2 && lon1 == lon2) {
                return 0;
            }

            dLat = lat2 - lat1;
            dLon = lon2 - lon1;
            dPhi = Math.log(Math.tan(lat2 / 2.0 + Math.PI / 4) / Math.tan(lat1 / 2.0 + Math.PI / 4));
            q = dLat / dPhi;

            if (isNaN(dPhi) || isNaN(q)) {
                q = Math.cos(lat1);
            }

            // If lonChange over 180 take shorter rhumb across 180 meridian.
            if (WWMath.fabs(dLon) > Math.PI) {
                dLon = dLon > 0 ? -(2 * Math.PI - dLon) : (2 * Math.PI + dLon);
            }

            distanceRadians = Math.sqrt(dLat * dLat + q * q * dLon * dLon);

            return isNaN(distanceRadians) ? 0 : distanceRadians;
        };

        /**
         * Computes the location on a rhumb arc with the given starting location, azimuth, and arc distance.
         * This function uses a spherical model, not elliptical.
         *
         * @param {Location} location The starting location.
         * @param {Number} azimuthDegrees The azimuth in degrees.
         * @param {Number} pathLengthRadians The radian distance along the path at which to compute the location.
         * @param {Location} result A Location in which to return the result.
         * @returns {Location} The specified result location.
         * @throws {ArgumentError} If the specified location or the result argument is null or undefined.
         */
        Location.rhumbLocation = function (location, azimuthDegrees, pathLengthRadians, result) {
            if (!location) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "rhumbLocation", "missingLocation"));
            }
            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "rhumbLocation", "missingResult"));
            }

            if (pathLengthRadians == 0) {
                result.latitude = location.latitude;
                result.longitude = location.longitude;
                return result;
            }

            var latRadians = location.latitude * Angle.DEGREES_TO_RADIANS,
                lonRadians = location.longitude * Angle.DEGREES_TO_RADIANS,
                azimuthRadians = azimuthDegrees * Angle.DEGREES_TO_RADIANS,
                endLatRadians = latRadians + pathLengthRadians * Math.cos(azimuthRadians),
                dPhi = Math.log(Math.tan(endLatRadians / 2 + Math.PI / 4) / Math.tan(latRadians / 2 + Math.PI / 4)),
                q = (endLatRadians - latRadians) / dPhi,
                dLon,
                endLonRadians;

            if (isNaN(dPhi) || isNaN(q) || !isFinite(q)) {
                q = Math.cos(latRadians);
            }

            dLon = pathLengthRadians * Math.sin(azimuthRadians) / q;

            // Handle latitude passing over either pole.
            if (WWMath.fabs(endLatRadians) > Math.PI / 2) {
                endLatRadians = endLatRadians > 0 ? Math.PI - endLatRadians : -Math.PI - endLatRadians;
            }

            endLonRadians = WWMath.fmod(lonRadians + dLon + Math.PI, 2 * Math.PI) - Math.PI;

            if (isNaN(endLatRadians) || isNaN(endLonRadians)) {
                result.latitude = location.latitude;
                result.longitude = location.longitude;
            } else {
                result.latitude = Angle.normalizedDegreesLatitude(endLatRadians * Angle.RADIANS_TO_DEGREES);
                result.longitude = Angle.normalizedDegreesLongitude(endLonRadians * Angle.RADIANS_TO_DEGREES);
            }

            return result;
        };

        /**
         * Compute a location along a linear path at a specified distance between two specified locations.
         * @param {Number} amount The fraction of the path between the two locations at which to compute the new
         * location. This number should be between 0 and 1. If not, it is clamped to the nearest of those values.
         * @param {Location} location1 The starting location.
         * @param {Location} location2 The ending location.
         * @param {Location} result A Location in which to return the result.
         * @returns {Location} The specified result location.
         * @throws {ArgumentError} If either specified location or the result argument is null or undefined.
         */
        Location.interpolateLinear = function (amount, location1, location2, result) {
            if (!location1 || !location2) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "interpolateLinear", "missingLocation"));
            }
            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "interpolateLinear", "missingResult"));
            }

            if (location1.equals(location2)) {
                result.latitude = location1.latitude;
                result.longitude = location1.longitude;
                return result;
            }

            var t = WWMath.clamp(amount, 0, 1),
                azimuthDegrees = this.linearAzimuth(location1, location2),
                distanceRadians = this.linearDistance(location1, location2);

            return this.linearLocation(location1, azimuthDegrees, t * distanceRadians, result);
        };

        /**
         * Computes the azimuth angle (clockwise from North) that points from the first location to the second location.
         * This angle can be used as the azimuth for a linear arc that begins at the first location, and
         * passes through the second location.
         * @param {Location} location1 The starting location.
         * @param {Location} location2 The ending location.
         * @returns {Number} The computed azimuth, in degrees.
         * @throws {ArgumentError} If either specified location is null or undefined.
         */
        Location.linearAzimuth = function (location1, location2) {
            if (!location1 || !location2) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "linearAzimuth", "missingLocation"));
            }

            var lat1 = location1.latitude * Angle.DEGREES_TO_RADIANS,
                lat2 = location2.latitude * Angle.DEGREES_TO_RADIANS,
                lon1 = location1.longitude * Angle.DEGREES_TO_RADIANS,
                lon2 = location2.longitude * Angle.DEGREES_TO_RADIANS,
                dLon,
                dPhi,
                azimuthRadians;

            if (lat1 == lat2 && lon1 == lon2) {
                return 0;
            }

            dLon = lon2 - lon1;
            dPhi = lat2 - lat1;

            // If longitude change is over 180 take shorter path across 180 meridian.
            if (WWMath.fabs(dLon) > Math.PI) {
                dLon = dLon > 0 ? -(2 * Math.PI - dLon) : (2 * Math.PI + dLon);
            }

            azimuthRadians = Math.atan2(dLon, dPhi);

            return isNaN(azimuthRadians) ? 0 : azimuthRadians * Angle.RADIANS_TO_DEGREES;
        };

        /**
         * Computes the linear angular distance between two locations. The return value gives the distance as the
         * angle between the two positions in radians. This angle is the arc length of the segment between the two
         * positions. To compute a distance in meters from this value, multiply the return value by the radius of the
         * globe.
         *
         * @param {Location} location1 The starting location.
         * @param {Location} location2 The ending location.
         * @returns {Number} The computed distance, in radians.
         * @throws {ArgumentError} If either specified location is null or undefined.
         */
        Location.linearDistance = function (location1, location2) {
            if (!location1 || !location2) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "linearDistance", "missingLocation"));
            }

            var lat1 = location1.latitude * Angle.DEGREES_TO_RADIANS,
                lat2 = location2.latitude * Angle.DEGREES_TO_RADIANS,
                lon1 = location1.longitude * Angle.DEGREES_TO_RADIANS,
                lon2 = location2.longitude * Angle.DEGREES_TO_RADIANS,
                dLat,
                dLon,
                distanceRadians;

            if (lat1 == lat2 && lon1 == lon2) {
                return 0;
            }

            dLat = lat2 - lat1;
            dLon = lon2 - lon1;

            // If lonChange over 180 take shorter path across 180 meridian.
            if (WWMath.fabs(dLon) > Math.PI) {
                dLon = dLon > 0 ? -(2 * Math.PI - dLon) : (2 * Math.PI + dLon);
            }

            distanceRadians = Math.sqrt(dLat * dLat + dLon * dLon);

            return isNaN(distanceRadians) ? 0 : distanceRadians;
        };

        /**
         * Computes the location on a linear path with the given starting location, azimuth, and arc distance.
         *
         * @param {Location} location The starting location.
         * @param {Number} azimuthDegrees The azimuth in degrees.
         * @param {Number} pathLengthRadians The radian distance along the path at which to compute the location.
         * @param {Location} result A Location in which to return the result.
         * @returns {Location} The specified result location.
         * @throws {ArgumentError} If the specified location or the result argument is null or undefined.
         */
        Location.linearLocation = function (location, azimuthDegrees, pathLengthRadians, result) {
            if (!location) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "linearLocation", "missingLocation"));
            }
            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "linearLocation", "missingResult"));
            }

            if (pathLengthRadians == 0) {
                result.latitude = location.latitude;
                result.longitude = location.longitude;
                return result;
            }

            var latRadians = location.latitude * Angle.DEGREES_TO_RADIANS,
                lonRadians = location.longitude * Angle.DEGREES_TO_RADIANS,
                azimuthRadians = azimuthDegrees * Angle.DEGREES_TO_RADIANS,
                endLatRadians = latRadians + pathLengthRadians * Math.cos(azimuthRadians),
                endLonRadians;

            // Handle latitude passing over either pole.
            if (WWMath.fabs(endLatRadians) > Math.PI / 2) {
                endLatRadians = endLatRadians > 0 ? Math.PI - endLatRadians : -Math.PI - endLatRadians;
            }

            endLonRadians =
                WWMath.fmod(lonRadians + pathLengthRadians * Math.sin(azimuthRadians) + Math.PI, 2 * Math.PI) - Math.PI;

            if (isNaN(endLatRadians) || isNaN(endLonRadians)) {
                result.latitude = location.latitude;
                result.longitude = location.longitude;
            } else {
                result.latitude = Angle.normalizedDegreesLatitude(endLatRadians * Angle.RADIANS_TO_DEGREES);
                result.longitude = Angle.normalizedDegreesLongitude(endLonRadians * Angle.RADIANS_TO_DEGREES);
            }

            return result;
        };

        /**
         * Determine whether a list of locations crosses the dateline.
         * @param {Location[]} locations The locations to test.
         * @returns {boolean} True if the dateline is crossed, else false.
         * @throws {ArgumentError} If the locations list is null.
         */
        Location.locationsCrossDateLine = function (locations) {
            if (!locations) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "locationsCrossDateline", "missingLocation"));
            }

            var pos = null;
            for (var idx = 0, len = locations.length; idx < len; idx += 1) {
                var posNext = locations[idx];

                if (pos != null) {
                    // A segment cross the line if end pos have different longitude signs
                    // and are more than 180 degrees longitude apart
                    if (WWMath.signum(pos.longitude) != WWMath.signum(posNext.longitude)) {
                        var delta = Math.abs(pos.longitude - posNext.longitude);
                        if (delta > 180 && delta < 360)
                            return true;
                    }
                }
                pos = posNext;
            }

            return false;
        };

        /**
         * Returns two locations with the most extreme latitudes on the sequence of great circle arcs defined by each pair
         * of locations in the specified iterable.
         *
         * @param {Location[]} locations The pairs of locations defining a sequence of great circle arcs.
         *
         * @return {Location[]} Two locations with the most extreme latitudes on the great circle arcs.
         *
         * @throws IllegalArgumentException if locations is null.
         */
        Location.greatCircleArcExtremeLocations = function (locations) {
            if (!locations) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "greatCircleArcExtremeLocations", "missingLocation"));
            }

            var minLatLocation = null;
            var maxLatLocation = null;

            var lastLocation = null;

            for (var idx = 0, len = locations.length; idx < len; idx += 1) {
                var location = locations[idx];

                if (lastLocation != null) {
                    var extremes = Location.greatCircleArcExtremeForTwoLocations(lastLocation, location);
                    if (extremes == null) {
                        continue;
                    }

                    if (minLatLocation == null || minLatLocation.latitude > extremes[0].latitude) {
                        minLatLocation = extremes[0];
                    }
                    if (maxLatLocation == null || maxLatLocation.latitude < extremes[1].latitude) {
                        maxLatLocation = extremes[1];
                    }
                }

                lastLocation = location;
            }

            return [minLatLocation, maxLatLocation];
        };

        /**
         * Returns two locations with the most extreme latitudes on the great circle arc defined by, and limited to, the two
         * locations.
         *
         * @param {Location} begin Beginning location on the great circle arc.
         * @param {Location} end   Ending location on the great circle arc.
         *
         * @return {Location[]} Two locations with the most extreme latitudes on the great circle arc.
         *
         * @throws {ArgumentError} If either begin or end are null.
         */
        Location.greatCircleArcExtremeForTwoLocations = function (begin, end) {
            if (!begin || !end) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "greatCircleArcExtremeForTwoLocations", "missingLocation"));
            }

            var idx, len, location; // Iteration variables.
            var minLatLocation = null;
            var maxLatLocation = null;
            var minLat = 90;
            var maxLat = -90;

            // Compute the min and max latitude and associated locations from the arc endpoints.
            var locations = [begin, end];
            for (idx = 0, len = locations.length; idx < len; idx += 1) {
                location = locations[idx];

                if (minLat >= location.latitude) {
                    minLat = location.latitude;
                    minLatLocation = location;
                }
                if (maxLat <= location.latitude) {
                    maxLat = location.latitude;
                    maxLatLocation = location;
                }
            }
            // The above could be written for greater clarity, simplicity, and speed:
            // minLat = Math.min(begin.latitude, end.latitude);
            // maxLat = Math.max(begin.latitude, end.latitude);
            // minLatLocation = minLat == begin.latitude ? begin : end;
            // maxLatLocation = maxLat == begin.latitude ? begin : end;

            // Compute parameters for the great circle arc defined by begin and end. Then compute the locations of extreme
            // latitude on entire the great circle which that arc is part of.
            var greatArcAzimuth = Location.greatCircleAzimuth(begin, end);
            var greatArcDistance = Location.greatCircleDistance(begin, end);
            var greatCircleExtremes = Location.greatCircleExtremeLocationsUsingAzimuth(begin, greatArcAzimuth);

            // Determine whether either of the extreme locations are inside the arc defined by begin and end. If so,
            // adjust the min and max latitude accordingly.
            for (idx = 0, len = greatCircleExtremes.length; idx < len; idx += 1) {
                location = greatCircleExtremes[idx];

                var az = Location.greatCircleAzimuth(begin, location);
                var d = Location.greatCircleDistance(begin, location);

                // The extreme location must be between the begin and end locations. Therefore its azimuth relative to
                // the begin location should have the same signum, and its distance relative to the begin location should
                // be between 0 and greatArcDistance, inclusive.
                if (WWMath.signum(az) == WWMath.signum(greatArcAzimuth)) {
                    if (d >= 0 && d <= greatArcDistance) {
                        if (minLat >= location.latitude) {
                            minLat = location.latitude;
                            minLatLocation = location;
                        }
                        if (maxLat <= location.latitude) {
                            maxLat = location.latitude;
                            maxLatLocation = location;
                        }
                    }
                }
            }

            return [minLatLocation, maxLatLocation];
        };

        /**
         * Returns two locations with the most extreme latitudes on the great circle with the given starting location and
         * azimuth.
         *
         * @param {Location} location Location on the great circle.
         * @param {number} azimuth  Great circle azimuth angle (clockwise from North).
         *
         * @return {Location[]} Two locations where the great circle has its extreme latitudes.
         *
         * @throws {ArgumentError} If location is null.
         */
        Location.greatCircleExtremeLocationsUsingAzimuth = function (location, azimuth) {
            if (!location) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Location", "greatCircleArcExtremeLocationsUsingAzimuth", "missingLocation"));
            }

            var lat0 = location.latitude;
            var az = azimuth * Angle.DEGREES_TO_RADIANS;

            // Derived by solving the function for longitude on a great circle against the desired longitude. We start
            // with the equation in "Map Projections - A Working Manual", page 31, equation 5-5:
            //
            //     lat = asin( sin(lat0) * cos(C) + cos(lat0) * sin(C) * cos(Az) )
            //
            // Where (lat0, lon) are the starting coordinates, c is the angular distance along the great circle from the
            // starting coordinate, and Az is the azimuth. All values are in radians. Solving for angular distance gives
            // distance to the equator:
            //
            //     tan(C) = -tan(lat0) / cos(Az)
            //
            // The great circle is by definition centered about the Globe's origin. Therefore intersections with the
            // equator will be antipodal (exactly 180 degrees opposite each other), as will be the extreme latitudes.
            // By observing the symmetry of a great circle, it is also apparent that the extreme latitudes will be 90
            // degrees from either intersection with the equator.
            //
            // d1 = c + 90
            // d2 = c - 90

            var tanDistance = -Math.tan(lat0) / Math.cos(az);
            var distance = Math.atan(tanDistance);

            var extremeDistance1 = distance + (Math.PI / 2.0);
            var extremeDistance2 = distance - (Math.PI / 2.0);

            return [
                Location.greatCircleLocation(location, azimuth, extremeDistance1, new Location(0, 0)),
                Location.greatCircleLocation(location, azimuth, extremeDistance2, new Location(0, 0))
            ];
        };

        /**
         * Determine where a line between two positions crosses a given meridian. The intersection test is performed by
         * intersecting a line in Cartesian space between the two positions with a plane through the meridian. Thus, it is
         * most suitable for working with positions that are fairly close together as the calculation does not take into
         * account great circle or rhumb paths.
         *
         * @param {Location} p1         First position.
         * @param {Location} p2         Second position.
         * @param {number} meridian     Longitude line to intersect with.
         * @param {Globe} globe         Globe used to compute intersection.
         *
         * @return {number} latitude The intersection latitude along the meridian
         *
         * TODO: this code allocates 4 new Vec3 and 1 new Position; use scratch variables???
         * TODO: Why not? Every location created would then allocated those variables as well, even if they aren't needed :(.
         */
        Location.intersectionWithMeridian = function (p1, p2, meridian, globe) {
            // TODO: add support for 2D
            //if (globe instanceof Globe2D)
            //{
            //    // y = mx + b case after normalizing negative angles.
            //    double lon1 = p1.getLongitude().degrees < 0 ? p1.getLongitude().degrees + 360 : p1.getLongitude().degrees;
            //    double lon2 = p2.getLongitude().degrees < 0 ? p2.getLongitude().degrees + 360 : p2.getLongitude().degrees;
            //    if (lon1 == lon2)
            //        return null;
            //
            //    double med = meridian.degrees < 0 ? meridian.degrees + 360 : meridian.degrees;
            //    double slope = (p2.latitude.degrees - p1.latitude.degrees) / (lon2 - lon1);
            //    double lat = p1.latitude.degrees + slope * (med - lon1);
            //
            //    return LatLon.fromDegrees(lat, meridian.degrees);
            //}

            var pt1 = globe.computePointFromLocation(p1.latitude, p1.longitude, new Vec3(0, 0, 0));
            var pt2 = globe.computePointFromLocation(p2.latitude, p2.longitude, new Vec3(0, 0, 0));

            // Compute a plane through the origin, North Pole, and the desired meridian.
            var northPole = globe.computePointFromLocation(90, meridian, new Vec3(0, 0, 0));
            var pointOnEquator = globe.computePointFromLocation(0, meridian, new Vec3(0, 0, 0));

            var plane = Plane.fromPoints(northPole, pointOnEquator, Vec3.ZERO);

            var intersectionPoint = new Vec3(0, 0, 0);
            if (!plane.intersectsSegmentAt(pt1, pt2, intersectionPoint)) {
                return null;
            }

            // TODO: unable to simply create a new Position(0, 0, 0)
            var pos = new WorldWind.Position(0, 0, 0);
            globe.computePositionFromPoint(intersectionPoint[0], intersectionPoint[1], intersectionPoint[2], pos);

            return pos.latitude;
        };

        /**
         * Determine where a line between two positions crosses a given meridian. The intersection test is performed by
         * intersecting a line in Cartesian space. Thus, it is most suitable for working with positions that are fairly
         * close together as the calculation does not take into account great circle or rhumb paths.
         *
         * @param {Location | Position} p1 First position.
         * @param {Location | Position} p2 Second position.
         * @param {number} meridian Longitude line to intersect with.
         *
         * @return {number | null} latitude The intersection latitude along the meridian
         * or null if the line is collinear with the meridian
         */
        Location.meridianIntersection = function(p1, p2, meridian){
                // y = mx + b case after normalizing negative angles.
                var lon1 = p1.longitude < 0 ? p1.longitude + 360 : p1.longitude;
                var lon2 = p2.longitude < 0 ? p2.longitude + 360 : p2.longitude;
                if (lon1 === lon2) {
                    //infinite solutions, the line is collinear with the anti-meridian
                    return null;
                }

                var med = meridian < 0 ? meridian + 360 : meridian;
                var slope = (p2.latitude - p1.latitude) / (lon2 - lon1);
                var lat = p1.latitude + slope * (med - lon1);

                return lat;
        };

        /**
         * A bit mask indicating which if any pole is being referenced.
         * This corresponds to Java WW's AVKey.NORTH and AVKey.SOUTH,
         * although this encoding can capture both poles simultaneously, which was
         * a 'to do' item in the Java implementation.
         * @type {{NONE: number, NORTH: number, SOUTH: number}}
         */
        Location.poles = {
            'NONE': 0,
            'NORTH': 1,
            'SOUTH': 2
        };

        return Location;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports Position
 */
define('geom/Position',[
        '../geom/Angle',
        '../error/ArgumentError',
        '../geom/Location',
        '../util/Logger',
        '../util/WWMath'
    ],
    function (Angle,
              ArgumentError,
              Location,
              Logger,
              WWMath) {
        "use strict";

        /**
         * Constructs a position from a specified latitude and longitude in degrees and altitude in meters.
         * @alias Position
         * @constructor
         * @classdesc Represents a latitude, longitude, altitude triple, with latitude and longitude in degrees and
         * altitude in meters.
         * @param {Number} latitude The latitude in degrees.
         * @param {Number} longitude The longitude in degrees.
         * @param {Number} altitude The altitude in meters.
         */
        var Position = function (latitude, longitude, altitude) {
            /**
             * The latitude in degrees.
             * @type {Number}
             */
            this.latitude = latitude;
            /**
             * The longitude in degrees.
             * @type {Number}
             */
            this.longitude = longitude;
            /**
             * The altitude in meters.
             * @type {Number}
             */
            this.altitude = altitude;
        };

        /**
         * A Position with latitude, longitude and altitude all 0.
         * @constant
         * @type {Position}
         */
        Position.ZERO = new Position(0, 0, 0);

        /**
         * Creates a position from angles specified in radians.
         * @param {Number} latitudeRadians The latitude in radians.
         * @param {Number} longitudeRadians The longitude in radians.
         * @param {Number} altitude The altitude in meters.
         * @returns {Position} The new position with latitude and longitude in degrees.
         */
        Position.fromRadians = function (latitudeRadians, longitudeRadians, altitude) {
            return new Position(
                latitudeRadians * Angle.RADIANS_TO_DEGREES,
                longitudeRadians * Angle.RADIANS_TO_DEGREES,
                altitude);
        };

        /**
         * Sets this position to the latitude, longitude and altitude of a specified position.
         * @param {Position} position The position to copy.
         * @returns {Position} This position, set to the values of the specified position.
         * @throws {ArgumentError} If the specified position is null or undefined.
         */
        Position.prototype.copy = function (position) {
            if (!position) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Position", "copy", "missingPosition"));
            }

            this.latitude = position.latitude;
            this.longitude = position.longitude;
            this.altitude = position.altitude;

            return this;
        };

        /**
         * Indicates whether this position has the same latitude, longitude and altitude as a specified position.
         * @param {Position} position The position to compare with this one.
         * @returns {Boolean} true if this position is equal to the specified one, otherwise false.
         */
        Position.prototype.equals = function (position) {
            return position
                && position.latitude === this.latitude
                && position.longitude === this.longitude
                && position.altitude === this.altitude;
        };

        /**
         * Computes a position along a great circle path at a specified distance between two specified positions.
         * @param {Number} amount The fraction of the path between the two positions at which to compute the new
         * position. This number should be between 0 and 1. If not, it is clamped to the nearest of those values.
         * @param {Position} position1 The starting position.
         * @param {Position} position2 The ending position.
         * @param {Position} result A Position in which to return the result.
         * @returns {Position} The specified result position.
         * @throws {ArgumentError} If either specified position or the result argument is null or undefined.
         */
        Position.interpolateGreatCircle = function (amount, position1, position2, result) {
            if (!position1 || !position2) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Position", "interpolateGreatCircle", "missingPosition"));
            }

            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Position", "interpolateGreatCircle", "missingResult"));
            }

            var t = WWMath.clamp(amount, 0, 1);
            result.altitude = WWMath.interpolate(t, position1.altitude, position2.altitude);

            //noinspection JSCheckFunctionSignatures
            Location.interpolateGreatCircle(t, position1, position2, result);

            return result;
        };

        /**
         * Computes a position along a rhumb path at a specified distance between two specified positions.
         * @param {Number} amount The fraction of the path between the two positions at which to compute the new
         * position. This number should be between 0 and 1. If not, it is clamped to the nearest of those values.
         * @param {Position} position1 The starting position.
         * @param {Position} position2 The ending position.
         * @param {Position} result A Position in which to return the result.
         * @returns {Position} The specified result position.
         * @throws {ArgumentError} If either specified position or the result argument is null or undefined.
         */
        Position.interpolateRhumb = function (amount, position1, position2, result) {
            if (!position1 || !position2) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Position", "interpolateRhumb", "missingPosition"));
            }

            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Position", "interpolateRhumb", "missingResult"));
            }

            var t = WWMath.clamp(amount, 0, 1);
            result.altitude = WWMath.interpolate(t, position1.altitude, position2.altitude);

            //noinspection JSCheckFunctionSignatures
            Location.interpolateRhumb(t, position1, position2, result);

            return result;
        };

        /**
         * Computes a position along a linear path at a specified distance between two specified positions.
         * @param {Number} amount The fraction of the path between the two positions at which to compute the new
         * position. This number should be between 0 and 1. If not, it is clamped to the nearest of those values.
         * @param {Position} position1 The starting position.
         * @param {Position} position2 The ending position.
         * @param {Position} result A Position in which to return the result.
         * @returns {Position} The specified result position.
         * @throws {ArgumentError} If either specified position or the result argument is null or undefined.
         */
        Position.interpolateLinear = function (amount, position1, position2, result) {
            if (!position1 || !position2) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Position", "interpolateLinear", "missingPosition"));
            }

            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Position", "interpolateLinear", "missingResult"));
            }

            var t = WWMath.clamp(amount, 0, 1);
            result.altitude = WWMath.interpolate(t, position1.altitude, position2.altitude);

            //noinspection JSCheckFunctionSignatures
            Location.interpolateLinear(t, position1, position2, result);

            return result;
        };

        /**
         * Returns a string representation of this position.
         * @returns {String}
         */
        Position.prototype.toString = function () {
            return "(" + this.latitude.toString() + "\u00b0, " + this.longitude.toString() + "\u00b0, "
                + this.altitude.toString() + ")";
        };

        return Position;
    });

/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports Texture
 */
define('render/Texture',[
        '../error/ArgumentError',
        '../util/Logger',
        '../util/WWMath'
    ],
    function (ArgumentError,
              Logger,
              WWMath) {
        "use strict";

        /**
         * Constructs a texture for a specified image.
         * @alias Texture
         * @constructor
         * @classdesc Represents a WebGL texture. Applications typically do not interact with this class.
         * @param {WebGLRenderingContext} gl The current WebGL rendering context.
         * @param {Image} image The texture's image.
         * @param {GLenum} wrapMode Optional. Specifies the wrap mode of the texture. Defaults to gl.CLAMP_TO_EDGE
         * @throws {ArgumentError} If the specified WebGL context or image is null or undefined.
         */
        var Texture = function (gl, image, wrapMode) {

            if (!gl) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Texture", "constructor",
                    "missingGlContext"));
            }

            if (!image) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Texture", "constructor",
                    "missingImage"));
            }

            if (!wrapMode) {
                wrapMode = gl.CLAMP_TO_EDGE;
            }

            var textureId = gl.createTexture(),
                isPowerOfTwo = (WWMath.isPowerOfTwo(image.width) && WWMath.isPowerOfTwo(image.height));

            this.originalImageWidth = image.width;
            this.originalImageHeight = image.height;

            if (wrapMode === gl.REPEAT && !isPowerOfTwo) {
                image = this.resizeImage(image);
                isPowerOfTwo = true;
            }

            this.imageWidth = image.width;
            this.imageHeight = image.height;
            this.size = image.width * image.height * 4;

            gl.bindTexture(gl.TEXTURE_2D, textureId);

            gl.pixelStorei(gl.UNPACK_PREMULTIPLY_ALPHA_WEBGL, 1);
            gl.texImage2D(gl.TEXTURE_2D, 0,
                gl.RGBA, gl.RGBA, gl.UNSIGNED_BYTE, image);
            gl.pixelStorei(gl.UNPACK_PREMULTIPLY_ALPHA_WEBGL, 0);

            if (isPowerOfTwo) {
                gl.generateMipmap(gl.TEXTURE_2D);
            }

            this.textureId = textureId;

            /**
             * The time at which this texture was created.
             * @type {Date}
             */
            this.creationTime = new Date();

            // Internal use only. Intentionally not documented.
            this.texParameters = {};
            this.texParameters[gl.TEXTURE_MIN_FILTER] = isPowerOfTwo ? gl.LINEAR_MIPMAP_LINEAR : gl.LINEAR;
            this.texParameters[gl.TEXTURE_WRAP_S] = wrapMode;
            this.texParameters[gl.TEXTURE_WRAP_T] = wrapMode;

            // Internal use only. Intentionally not documented.
            // https://www.khronos.org/registry/webgl/extensions/EXT_texture_filter_anisotrop
            this.anisotropicFilterExt = (gl.getExtension("EXT_texture_filter_anisotropic") ||
                                         gl.getExtension("WEBKIT_EXT_texture_filter_anisotropic"));
        };

        /**
         * Sets a texture parameter to apply when binding this texture.
         *
         * Currently only gl.TEXTURE_MAG_FILTER has an effect.
         *
         * @param {Glenum} name The name of the parameter
         * @param {GLint} value The value for this parameter
         */
        Texture.prototype.setTexParameter = function (name, value) {
            this.texParameters[name] = value;
        };

        /**
         * Returns the value of a texture parameter to be assigned to this texture.
         * @param {Glenum} name The name of the parameter
         * @returns {GLint} The value for this parameter
         */
        Texture.prototype.getTexParameter = function (name) {
            return this.texParameters[name];
        };

        /**
         * Clears the list of texture parameters to apply when binding this texture.
         */
        Texture.prototype.clearTexParameters = function () {
            this.texParameters = {};
        };

        /**
         * Disposes of the WebGL texture object associated with this texture.
         * @param gl
         */
        Texture.prototype.dispose = function (gl) {
            gl.deleteTexture(this.textureId);
            delete this.textureId;
        };

        /**
         * Binds this texture in the current WebGL graphics context.
         * @param {DrawContext} dc The current draw context.
         */
        Texture.prototype.bind = function (dc) {
            var gl = dc.currentGlContext;

            gl.bindTexture(gl.TEXTURE_2D, this.textureId);

            this.applyTexParameters(dc);

            dc.frameStatistics.incrementTextureLoadCount(1);
            return true;
        };

        /**
         * Applies the configured texture parameters to the OpenGL context.
         * @param {DrawContext} dc The current draw context.
         */
        Texture.prototype.applyTexParameters = function (dc) {
            var gl = dc.currentGlContext;

            // Configure the OpenGL texture minification function. Use nearest in pickingMode or linear by default.
            var textureMinFilter = dc.pickingMode ? gl.NEAREST : this.texParameters[gl.TEXTURE_MIN_FILTER] || gl.LINEAR;
            gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, textureMinFilter);

            // Configure the OpenGL texture magnification function. Use nearest in pickingMode or linear by default.
            var textureMagFilter = dc.pickingMode ? gl.NEAREST : this.texParameters[gl.TEXTURE_MAG_FILTER] || gl.LINEAR;
            gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, textureMagFilter);

            gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, this.texParameters[gl.TEXTURE_WRAP_S] || gl.CLAMP_TO_EDGE);
            gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, this.texParameters[gl.TEXTURE_WRAP_T] || gl.CLAMP_TO_EDGE);

            // Try to enable the anisotropic texture filtering only if we have a linear magnification filter.
            // This can't be enabled all the time because Windows seems to ignore the TEXTURE_MAG_FILTER parameter when
            // this extension is enabled.
            if (textureMagFilter === gl.LINEAR) {
                // Setup 4x anisotropic texture filtering when this feature is available.
                if (this.anisotropicFilterExt) {
                    gl.texParameteri(gl.TEXTURE_2D, this.anisotropicFilterExt.TEXTURE_MAX_ANISOTROPY_EXT, 4);
                }
            }
        };

        /**
         * Resizes an image to a power of two.
         * @param {Image} image The image to resize.
         */
        Texture.prototype.resizeImage = function (image) {
            var canvas = document.createElement("canvas");
            canvas.width = WWMath.powerOfTwoFloor(image.width);
            canvas.height = WWMath.powerOfTwoFloor(image.height);
            var ctx = canvas.getContext("2d");
            ctx.drawImage(image, 0, 0, canvas.width, canvas.height);
            return canvas;
        };


        return Texture;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports Matrix
 */
define('geom/Matrix',[
        '../geom/Angle',
        '../error/ArgumentError',
        '../util/Logger',
        '../geom/Plane',
        '../geom/Position',
        '../geom/Rectangle',
        '../render/Texture',
        '../geom/Vec3',
        '../util/WWMath'
    ],
    function (Angle,
              ArgumentError,
              Logger,
              Plane,
              Position,
              Rectangle,
              Texture,
              Vec3,
              WWMath) {
        "use strict";

        /**
         * Constructs a matrix.
         * @alias Matrix
         * @constructor
         * @classdesc Represents a 4 x 4 double precision matrix stored in a Float64Array in row-major order.
         * @param {Number} m11 matrix element at row 1, column 1.
         * @param {Number} m12 matrix element at row 1, column 2.
         * @param {Number} m13 matrix element at row 1, column 3.
         * @param {Number} m14 matrix element at row 1, column 4.
         * @param {Number} m21 matrix element at row 2, column 1.
         * @param {Number} m22 matrix element at row 2, column 2.
         * @param {Number} m23 matrix element at row 2, column 3.
         * @param {Number} m24 matrix element at row 2, column 4.
         * @param {Number} m31 matrix element at row 3, column 1.
         * @param {Number} m32 matrix element at row 3, column 2.
         * @param {Number} m33 matrix element at row 3, column 3.
         * @param {Number} m34 matrix element at row 3, column 4.
         * @param {Number} m41 matrix element at row 4, column 1.
         * @param {Number} m42 matrix element at row 4, column 2.
         * @param {Number} m43 matrix element at row 4, column 3.
         * @param {Number} m44 matrix element at row 4, column 4.
         */
        var Matrix = function (m11, m12, m13, m14,
                               m21, m22, m23, m24,
                               m31, m32, m33, m34,
                               m41, m42, m43, m44) {
            this[0] = m11;
            this[1] = m12;
            this[2] = m13;
            this[3] = m14;
            this[4] = m21;
            this[5] = m22;
            this[6] = m23;
            this[7] = m24;
            this[8] = m31;
            this[9] = m32;
            this[10] = m33;
            this[11] = m34;
            this[12] = m41;
            this[13] = m42;
            this[14] = m43;
            this[15] = m44;
        };

        // Derives from Float64Array.
        Matrix.prototype = new Float64Array(16);

        /**
         * Creates an identity matrix.
         * @returns {Matrix} A new identity matrix.
         */
        Matrix.fromIdentity = function () {
            return new Matrix(
                1, 0, 0, 0,
                0, 1, 0, 0,
                0, 0, 1, 0,
                0, 0, 0, 1
            );
        };

        /**
         * Computes the principal axes of a point collection expressed in a typed array.
         * @param {Float32Array} points The points for which to compute the axes,
         * expressed as X0, Y0, Z0, X1, Y1, Z1, ...
         * @param {Vec3} axis1 A vector in which to return the first (longest) principal axis.
         * @param {Vec3} axis2 A vector in which to return the second (mid-length) principal axis.
         * @param {Vec3} axis3 A vector in which to return the third (shortest) principal axis.
         * @throws {ArgumentError} If the specified points array is null, undefined or empty, or one of the
         * specified axes arguments is null or undefined.
         */
        Matrix.principalAxesFromPoints = function (points, axis1, axis2, axis3) {
            if (!points || points.length < 1) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "principalAxesFromPoints",
                    "missingPoints"));
            }

            if (!axis1 || !axis2 || !axis3) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "principalAxesFromPoints",
                    "An axis argument is null or undefined."));
            }

            // Compute the covariance matrix.
            var covariance = Matrix.fromIdentity();
            covariance.setToCovarianceOfPoints(points);

            // Compute the eigenvectors from the covariance matrix. Since the covariance matrix is symmetric by
            // definition, we can safely use the "symmetric" method below.
            covariance.eigensystemFromSymmetricMatrix(axis1, axis2, axis3);

            // Normalize the eigenvectors, which are already sorted in order from most prominent to least prominent.
            axis1.normalize();
            axis2.normalize();
            axis3.normalize();
        };

        /**
         * Sets the components of this matrix to specified values.
         * @param {Number} m11 matrix element at row 1, column 1.
         * @param {Number} m12 matrix element at row 1, column 2.
         * @param {Number} m13 matrix element at row 1, column 3.
         * @param {Number} m14 matrix element at row 1, column 4.
         * @param {Number} m21 matrix element at row 2, column 1.
         * @param {Number} m22 matrix element at row 2, column 2.
         * @param {Number} m23 matrix element at row 2, column 3.
         * @param {Number} m24 matrix element at row 2, column 4.
         * @param {Number} m31 matrix element at row 3, column 1.
         * @param {Number} m32 matrix element at row 3, column 2.
         * @param {Number} m33 matrix element at row 3, column 3.
         * @param {Number} m34 matrix element at row 3, column 4.
         * @param {Number} m41 matrix element at row 4, column 1.
         * @param {Number} m42 matrix element at row 4, column 2.
         * @param {Number} m43 matrix element at row 4, column 3.
         * @param {Number} m44 matrix element at row 4, column 4.
         * @returns {Matrix} This matrix with its components set to the specified values.
         */
        Matrix.prototype.set = function (m11, m12, m13, m14,
                                         m21, m22, m23, m24,
                                         m31, m32, m33, m34,
                                         m41, m42, m43, m44) {
            this[0] = m11;
            this[1] = m12;
            this[2] = m13;
            this[3] = m14;
            this[4] = m21;
            this[5] = m22;
            this[6] = m23;
            this[7] = m24;
            this[8] = m31;
            this[9] = m32;
            this[10] = m33;
            this[11] = m34;
            this[12] = m41;
            this[13] = m42;
            this[14] = m43;
            this[15] = m44;

            return this;
        };

        /**
         * Sets this matrix to the identity matrix.
         * @returns {Matrix} This matrix set to the identity matrix.
         */
        Matrix.prototype.setToIdentity = function () {
            this[0] = 1;
            this[1] = 0;
            this[2] = 0;
            this[3] = 0;
            this[4] = 0;
            this[5] = 1;
            this[6] = 0;
            this[7] = 0;
            this[8] = 0;
            this[9] = 0;
            this[10] = 1;
            this[11] = 0;
            this[12] = 0;
            this[13] = 0;
            this[14] = 0;
            this[15] = 1;
        };

        /**
         * Copies the components of a specified matrix to this matrix.
         * @param {Matrix} matrix The matrix to copy.
         * @returns {Matrix} This matrix set to the values of the specified matrix.
         * @throws {ArgumentError} If the specified matrix is null or undefined.
         */
        Matrix.prototype.copy = function (matrix) {
            if (!matrix) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "copy", "missingMatrix"));
            }

            this[0] = matrix[0];
            this[1] = matrix[1];
            this[2] = matrix[2];
            this[3] = matrix[3];
            this[4] = matrix[4];
            this[5] = matrix[5];
            this[6] = matrix[6];
            this[7] = matrix[7];
            this[8] = matrix[8];
            this[9] = matrix[9];
            this[10] = matrix[10];
            this[11] = matrix[11];
            this[12] = matrix[12];
            this[13] = matrix[13];
            this[14] = matrix[14];
            this[15] = matrix[15];
        };

        /**
         * Creates a new matrix that is a copy of this matrix.
         * @returns {Matrix} The new matrix.
         */
        Matrix.prototype.clone = function () {
            var clone = Matrix.fromIdentity();
            clone.copy(this);

            return clone;
        };

        /**
         * Indicates whether the components of this matrix are equal to those of a specified matrix.
         * @param {Matrix} matrix The matrix to test equality with. May be null or undefined, in which case this
         * function returns false.
         * @returns {boolean} true if all components of this matrix are equal to the corresponding
         * components of the specified matrix, otherwise false.
         */
        Matrix.prototype.equals = function (matrix) {
            return matrix
                && this[0] == matrix[0]
                && this[1] == matrix[1]
                && this[2] == matrix[2]
                && this[3] == matrix[3]
                && this[4] == matrix[4]
                && this[5] == matrix[5]
                && this[6] == matrix[6]
                && this[7] == matrix[7]
                && this[8] == matrix[8]
                && this[9] == matrix[9]
                && this[10] == matrix[10]
                && this[11] == matrix[11]
                && this[12] == matrix[12]
                && this[13] == matrix[13]
                && this[14] == matrix[14]
                && this[15] == matrix[15];
        };

        /**
         * Stores this matrix's components in column-major order in a specified array.
         * <p>
         * The array must have space for at least 16 elements. This matrix's components are stored in the array
         * starting with row 0 column 0 in index 0, row 1 column 0 in index 1, row 2 column 0 in index 2, and so on.
         *
         * @param {Float32Array | Float64Array | Number[]} result An array of at least 16 elements. Upon return,
         * contains this matrix's components in column-major.
         * @returns {Float32Array} The specified result array.
         * @throws {ArgumentError} If the specified result array in null or undefined.
         */
        Matrix.prototype.columnMajorComponents = function (result) {
            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "columnMajorComponents", "missingResult"));
            }

            // Column 1
            result[0] = this[0];
            result[1] = this[4];
            result[2] = this[8];
            result[3] = this[12];
            // Column 2
            result[4] = this[1];
            result[5] = this[5];
            result[6] = this[9];
            result[7] = this[13];
            // Column 3
            result[8] = this[2];
            result[9] = this[6];
            result[10] = this[10];
            result[11] = this[14];
            // Column 4
            result[12] = this[3];
            result[13] = this[7];
            result[14] = this[11];
            result[15] = this[15];

            return result;
        };

        /**
         * Sets this matrix to a translation matrix with specified translation components.
         * @param {Number} x The X translation component.
         * @param {Number} y The Y translation component.
         * @param {Number} z The Z translation component.
         * @returns {Matrix} This matrix with its translation components set to those specified and all other
         * components set to that of an identity matrix.
         */
        Matrix.prototype.setToTranslation = function (x, y, z) {
            this[0] = 1;
            this[1] = 0;
            this[2] = 0;
            this[3] = x;
            this[4] = 0;
            this[5] = 1;
            this[6] = 0;
            this[7] = y;
            this[8] = 0;
            this[9] = 0;
            this[10] = 1;
            this[11] = z;
            this[12] = 0;
            this[13] = 0;
            this[14] = 0;
            this[15] = 1;

            return this;
        };

        /**
         * Sets the translation components of this matrix to specified values.
         * @param {Number} x The X translation component.
         * @param {Number} y The Y translation component.
         * @param {Number} z The Z translation component.
         * @returns {Matrix} This matrix with its translation components set to the specified values and all other
         * components unmodified.
         */
        Matrix.prototype.setTranslation = function (x, y, z) {
            this[3] = x;
            this[7] = y;
            this[11] = z;

            return this;
        };

        /**
         * Sets this matrix to a scale matrix with specified scale components.
         * @param {Number} xScale The X scale component.
         * @param {Number} yScale The Y scale component.
         * @param {Number} zScale The Z scale component.
         * @returns {Matrix} This matrix with its scale components set to those specified and all other
         * components set to that of an identity matrix.
         */
        Matrix.prototype.setToScale = function (xScale, yScale, zScale) {
            this[0] = xScale;
            this[1] = 0;
            this[2] = 0;
            this[3] = 0;
            this[4] = 0;
            this[5] = yScale;
            this[6] = 0;
            this[7] = 0;
            this[8] = 0;
            this[9] = 0;
            this[10] = zScale;
            this[11] = 0;
            this[12] = 0;
            this[13] = 0;
            this[14] = 0;
            this[15] = 1;

            return this;
        };

        /**
         * Sets the scale components of this matrix to specified values.
         * @param {Number} xScale The X scale component.
         * @param {Number} yScale The Y scale component.
         * @param {Number} zScale The Z scale component.
         * @returns {Matrix} This matrix with its scale components set to the specified values and all other
         * components unmodified.
         */
        Matrix.prototype.setScale = function (xScale, yScale, zScale) {
            this[0] = xScale;
            this[5] = yScale;
            this[10] = zScale;

            return this;
        };

        /**
         * Sets this matrix to the transpose of a specified matrix.
         * @param {Matrix} matrix The matrix whose transpose is to be copied.
         * @returns {Matrix} This matrix, with its values set to the transpose of the specified matrix.
         * @throws {ArgumentError} If the specified matrix in null or undefined.
         */
        Matrix.prototype.setToTransposeOfMatrix = function (matrix) {
            if (!matrix) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "setToTransposeOfMatrix", "missingMatrix"));
            }

            this[0] = matrix[0];
            this[1] = matrix[4];
            this[2] = matrix[8];
            this[3] = matrix[12];
            this[4] = matrix[1];
            this[5] = matrix[5];
            this[6] = matrix[9];
            this[7] = matrix[13];
            this[8] = matrix[2];
            this[9] = matrix[6];
            this[10] = matrix[10];
            this[11] = matrix[14];
            this[12] = matrix[3];
            this[13] = matrix[7];
            this[14] = matrix[11];
            this[15] = matrix[15];

            return this;
        };

        /**
         * Sets this matrix to the matrix product of two specified matrices.
         * @param {Matrix} matrixA The first matrix multiplicand.
         * @param {Matrix} matrixB The second matrix multiplicand.
         * @returns {Matrix} This matrix set to the product of matrixA x matrixB.
         * @throws {ArgumentError} If either specified matrix is null or undefined.
         */
        Matrix.prototype.setToMultiply = function (matrixA, matrixB) {
            if (!matrixA || !matrixB) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "setToMultiply", "missingMatrix"));
            }

            var ma = matrixA,
                mb = matrixB;

            this[0] = ma[0] * mb[0] + ma[1] * mb[4] + ma[2] * mb[8] + ma[3] * mb[12];
            this[1] = ma[0] * mb[1] + ma[1] * mb[5] + ma[2] * mb[9] + ma[3] * mb[13];
            this[2] = ma[0] * mb[2] + ma[1] * mb[6] + ma[2] * mb[10] + ma[3] * mb[14];
            this[3] = ma[0] * mb[3] + ma[1] * mb[7] + ma[2] * mb[11] + ma[3] * mb[15];

            this[4] = ma[4] * mb[0] + ma[5] * mb[4] + ma[6] * mb[8] + ma[7] * mb[12];
            this[5] = ma[4] * mb[1] + ma[5] * mb[5] + ma[6] * mb[9] + ma[7] * mb[13];
            this[6] = ma[4] * mb[2] + ma[5] * mb[6] + ma[6] * mb[10] + ma[7] * mb[14];
            this[7] = ma[4] * mb[3] + ma[5] * mb[7] + ma[6] * mb[11] + ma[7] * mb[15];

            this[8] = ma[8] * mb[0] + ma[9] * mb[4] + ma[10] * mb[8] + ma[11] * mb[12];
            this[9] = ma[8] * mb[1] + ma[9] * mb[5] + ma[10] * mb[9] + ma[11] * mb[13];
            this[10] = ma[8] * mb[2] + ma[9] * mb[6] + ma[10] * mb[10] + ma[11] * mb[14];
            this[11] = ma[8] * mb[3] + ma[9] * mb[7] + ma[10] * mb[11] + ma[11] * mb[15];

            this[12] = ma[12] * mb[0] + ma[13] * mb[4] + ma[14] * mb[8] + ma[15] * mb[12];
            this[13] = ma[12] * mb[1] + ma[13] * mb[5] + ma[14] * mb[9] + ma[15] * mb[13];
            this[14] = ma[12] * mb[2] + ma[13] * mb[6] + ma[14] * mb[10] + ma[15] * mb[14];
            this[15] = ma[12] * mb[3] + ma[13] * mb[7] + ma[14] * mb[11] + ma[15] * mb[15];

            return this;
        };

        /**
         * Sets this matrix to the symmetric covariance Matrix computed from the x, y, z coordinates of a specified
         * points array.
         * <p/>
         * The computed covariance matrix represents the correlation between each pair of x-, y-, and z-coordinates as
         * they're distributed about the point array's arithmetic mean. Its layout is as follows:
         * <p/>
         * <code> C(x, x)  C(x, y)  C(x, z) <br/> C(x, y)  C(y, y)  C(y, z) <br/> C(x, z)  C(y, z)  C(z, z) </code>
         * <p/>
         * C(i, j) is the covariance of coordinates i and j, where i or j are a coordinate's dispersion about its mean
         * value. If any entry is zero, then there's no correlation between the two coordinates defining that entry. If the
         * returned matrix is diagonal, then all three coordinates are uncorrelated, and the specified point is
         * distributed evenly about its mean point.
         * @param {Float32Array | Float64Array | Number[]} points The points to consider.
         * @returns {Matrix} This matrix set to the covariance matrix for the specified list of points.
         * @throws {ArgumentError} If the specified array of points is null, undefined or empty.
         */
        Matrix.prototype.setToCovarianceOfPoints = function (points) {
            if (!points || points.length < 1) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "setToCovarianceOfPoints", "missingArray"));
            }

            var mean,
                dx,
                dy,
                dz,
                count = 0,
                c11 = 0,
                c22 = 0,
                c33 = 0,
                c12 = 0,
                c13 = 0,
                c23 = 0,
                vec = new Vec3(0, 0, 0);

            mean = Vec3.averageOfBuffer(points, new Vec3(0, 0, 0));

            for (var i = 0, len = points.length / 3; i < len; i++) {
                vec[0] = points[i * 3];
                vec[1] = points[i * 3 + 1];
                vec[2] = points[i * 3 + 2];

                dx = vec[0] - mean[0];
                dy = vec[1] - mean[1];
                dz = vec[2] - mean[2];

                ++count;
                c11 += dx * dx;
                c22 += dy * dy;
                c33 += dz * dz;
                c12 += dx * dy; // c12 = c21
                c13 += dx * dz; // c13 = c31
                c23 += dy * dz; // c23 = c32
            }

            // Row 1
            this[0] = c11 / count;
            this[1] = c12 / count;
            this[2] = c13 / count;
            this[3] = 0;

            // Row 2
            this[4] = c12 / count;
            this[5] = c22 / count;
            this[6] = c23 / count;
            this[7] = 0;

            // Row 3
            this[8] = c13 / count;
            this[9] = c23 / count;
            this[10] = c33 / count;
            this[11] = 0;

            // Row 4
            this[12] = 0;
            this[13] = 0;
            this[14] = 0;
            this[15] = 0;

            return this;
        };

        /**
         * Multiplies this matrix by a translation matrix with specified translation values.
         * @param {Number} x The X translation component.
         * @param {Number} y The Y translation component.
         * @param {Number} z The Z translation component.
         * @returns {Matrix} This matrix multiplied by the translation matrix implied by the specified values.
         */
        Matrix.prototype.multiplyByTranslation = function (x, y, z) {

            this.multiply(
                1, 0, 0, x,
                0, 1, 0, y,
                0, 0, 1, z,
                0, 0, 0, 1);

            return this;
        };

        /**
         * Multiplies this matrix by a rotation matrix about a specified axis and angle.
         * @param {Number} x The X component of the rotation axis.
         * @param {Number} y The Y component of the rotation axis.
         * @param {Number} z The Z component of the rotation axis.
         * @param {Number} angleDegrees The angle to rotate, in degrees.
         * @returns {Matrix} This matrix multiplied by the rotation matrix implied by the specified values.
         */
        Matrix.prototype.multiplyByRotation = function (x, y, z, angleDegrees) {

            var c = Math.cos(angleDegrees * Angle.DEGREES_TO_RADIANS),
                s = Math.sin(angleDegrees * Angle.DEGREES_TO_RADIANS);

            this.multiply(
                c + (1 - c) * x * x, (1 - c) * x * y - s * z, (1 - c) * x * z + s * y, 0,
                (1 - c) * x * y + s * z, c + (1 - c) * y * y, (1 - c) * y * z - s * x, 0,
                (1 - c) * x * z - s * y, (1 - c) * y * z + s * x, c + (1 - c) * z * z, 0,
                0, 0, 0, 1);

            return this;
        };

        /**
         * Multiplies this matrix by a scale matrix with specified values.
         * @param {Number} xScale The X scale component.
         * @param {Number} yScale The Y scale component.
         * @param {Number} zScale The Z scale component.
         * @returns {Matrix} This matrix multiplied by the scale matrix implied by the specified values.
         */
        Matrix.prototype.multiplyByScale = function (xScale, yScale, zScale) {

            this.multiply(
                xScale, 0, 0, 0,
                0, yScale, 0, 0,
                0, 0, zScale, 0,
                0, 0, 0, 1);

            return this;
        };

        /**
         * Sets this matrix to one that flips and shifts the y-axis.
         * <p>
         * The resultant matrix maps Y=0 to Y=1 and Y=1 to Y=0. All existing values are overwritten. This matrix is
         * usually used to change the coordinate origin from an upper left coordinate origin to a lower left coordinate
         * origin. This is typically necessary to align the coordinate system of images (top-left origin) with that of
         * OpenGL (bottom-left origin).
         * @returns {Matrix} This matrix set to values described above.
         */
        Matrix.prototype.setToUnitYFlip = function () {

            this[0] = 1;
            this[1] = 0;
            this[2] = 0;
            this[3] = 0;
            this[4] = 0;
            this[5] = -1;
            this[6] = 0;
            this[7] = 1;
            this[8] = 0;
            this[9] = 0;
            this[10] = 1;
            this[11] = 0;
            this[12] = 0;
            this[13] = 0;
            this[14] = 0;
            this[15] = 1;

            return this;
        };

        /**
         * Multiplies this matrix by a local coordinate system transform for the specified globe.
         * <p>
         * The local coordinate system is defined such that the local origin (0, 0, 0) maps to the specified origin
         * point, the z axis maps to the globe's surface normal at the point, the y-axis maps to the north pointing
         * tangent, and the x-axis maps to the east pointing tangent.
         *
         * @param {Vec3} origin The local coordinate system origin, in model coordinates.
         * @param {Globe} globe The globe the coordinate system is relative to.
         *
         * @throws {ArgumentError} If either argument is null or undefined.
         */
        Matrix.prototype.multiplyByLocalCoordinateTransform = function (origin, globe) {
            if (!origin) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "multiplyByLocalCoordinateTransform",
                        "Origin vector is null or undefined"));
            }

            if (!globe) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "multiplyByLocalCoordinateTransform",
                        "missingGlobe"));
            }

            var xAxis = new Vec3(0, 0, 0),
                yAxis = new Vec3(0, 0, 0),
                zAxis = new Vec3(0, 0, 0);

            WWMath.localCoordinateAxesAtPoint(origin, globe, xAxis, yAxis, zAxis);

            this.multiply(
                xAxis[0], yAxis[0], zAxis[0], origin[0],
                xAxis[1], yAxis[1], zAxis[1], origin[1],
                xAxis[2], yAxis[2], zAxis[2], origin[2],
                0, 0, 0, 1);

            return this;
        };

        /**
         * Multiplies this matrix by a texture transform for the specified texture.
         * <p>
         * A texture image transform maps the bottom-left corner of the texture's image data to coordinate [0,0] and maps the
         * top-right of the texture's image data to coordinate [1,1]. This correctly handles textures whose image data has
         * non-power-of-two dimensions, and correctly orients textures whose image data has its origin in the upper-left corner.
         *
         * @param {Texture} texture The texture to multiply a transform for.
         *
         * @throws {ArgumentError} If the texture is null or undefined.
         */
        Matrix.prototype.multiplyByTextureTransform = function (texture) {
            if (!texture) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "multiplyByTextureTransform",
                        "missingTexture"));
            }

            // Compute the scale necessary to map the edge of the image data to the range [0,1]. When the texture contains
            // power-of-two image data the scale is 1 and has no effect. Otherwise, the scale is computed such that the portion
            // of the texture containing image data maps to the range [0,1].
            var sx = texture.originalImageWidth / texture.imageWidth,
                sy = texture.originalImageHeight / texture.imageHeight;

            // Multiply this by a scaling matrix that maps the texture's image data to the range [0,1] and inverts the y axis.
            // We have precomputed the result here in order to avoid an unnecessary matrix multiplication.
            this.multiply(
                sx, 0, 0, 0,
                0, -sy, 0, sy,
                0, 0, 1, 0,
                0, 0, 0, 1);

            return this;
        };

        /**
         * Returns the translation components of this matrix.
         * @param {Vec3} result A pre-allocated {@link Vec3} in which to return the translation components.
         * @returns {Vec3} The specified result argument set to the translation components of this matrix.
         * @throws {ArgumentError} If the specified result argument is null or undefined.
         */
        Matrix.prototype.extractTranslation = function (result) {
            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "extractTranslation", "missingResult"));
            }

            result[0] = this[3];
            result[1] = this[7];
            result[2] = this[11];

            return result;
        };

        /**
         * Returns the rotation angles of this matrix.
         * @param {Vec3} result A pre-allocated {@link Vec3} in which to return the rotation angles.
         * @returns {Vec3} The specified result argument set to the rotation angles of this matrix. The angles are in
         * degrees.
         * @throws {ArgumentError} If the specified result argument is null or undefined.
         */
        Matrix.prototype.extractRotationAngles = function (result) {
            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "extractRotationAngles", "missingResult"));
            }

            // Taken from Extracting Euler Angles from a Rotation Matrix by Mike Day, Insomniac Games.
            // http://www.insomniacgames.com/mike-day-extracting-euler-angles-from-a-rotation-matrix/

            var x = Math.atan2(this[6], this[10]),
                y = Math.atan2(-this[2], Math.sqrt(this[0] * this[0] + this[1] * this[1])),
                cx = Math.cos(x),
                sx = Math.sin(x),
                z = Math.atan2(sx * this[8] - cx * this[4], cx * this[5] - sx * this[9]);

            result[0] = x * Angle.RADIANS_TO_DEGREES;
            result[1] = y * Angle.RADIANS_TO_DEGREES;
            result[2] = z * Angle.RADIANS_TO_DEGREES;

            return result;
        };

        /**
         * Multiplies this matrix by a first person viewing matrix for the specified globe.
         * <p>
         * A first person viewing matrix places the viewer's eye at the specified eyePosition. By default the viewer is looking
         * straight down at the globe's surface from the eye position, with the globe's normal vector coming out of the screen
         * and north pointing toward the top of the screen.
         * <p>
         * Heading specifies the viewer's azimuth, or its angle relative to North. Heading values range from -180 degrees to 180
         * degrees. A heading of 0 degrees looks North, 90 degrees looks East, +-180 degrees looks South, and -90 degrees looks
         * West.
         * <p>
         * Tilt specifies the viewer's angle relative to the surface. Tilt values range from -180 degrees to 180 degrees. A tilt
         * of 0 degrees looks straight down at the globe's surface, 90 degrees looks at the horizon, and 180 degrees looks
         * straight up. Tilt values greater than 180 degrees cause the viewer to turn upside down, and are therefore rarely used.
         * <p>
         * Roll specifies the viewer's angle relative to the horizon. Roll values range from -180 degrees to 180 degrees. A roll
         * of 0 degrees orients the viewer so that up is pointing to the top of the screen, at 90 degrees up is pointing to the
         * right, at +-180 degrees up is pointing to the bottom, and at -90 up is pointing to the left.
         *
         * @param {Position} eyePosition The viewer's geographic eye position relative to the specified globe.
         * @param {Number} heading The viewer's angle relative to north, in degrees.
         * @param {Number} tilt The viewer's angle relative to the surface, in degrees.
         * @param {Number} roll The viewer's angle relative to the horizon, in degrees.
         * @param {Globe} globe The globe the viewer is looking at.
         *
         * @throws {ArgumentError} If the specified position or globe is null or undefined.
         */
        Matrix.prototype.multiplyByFirstPersonModelview = function (eyePosition, heading, tilt, roll, globe) {
            if (!eyePosition) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "multiplyByFirstPersonModelview", "missingPosition"));
            }

            if (!globe) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "multiplyByFirstPersonModelview", "missingGlobe"));
            }

            var c,
                s,
                ex, ey, ez,
                xx, xy, xz,
                yx, yy, yz,
                zx, zy, zz,
                eyePoint = new Vec3(0, 0, 0),
                xAxis = new Vec3(0, 0, 0),
                yAxis = new Vec3(0, 0, 0),
                zAxis = new Vec3(0, 0, 0);

            // Roll. Rotate the eye point in a counter-clockwise direction about the z axis. Note that we invert the sines used
            // in the rotation matrix in order to produce the counter-clockwise rotation. We invert only the cosines since
            // sin(-a) = -sin(a) and cos(-a) = cos(a).
            c = Math.cos(roll * Angle.DEGREES_TO_RADIANS);
            s = Math.sin(roll * Angle.DEGREES_TO_RADIANS);
            this.multiply(
                c, s, 0, 0,
                -s, c, 0, 0,
                0, 0, 1, 0,
                0, 0, 0, 1);

            // Tilt. Rotate the eye point in a counter-clockwise direction about the x axis. Note that we invert the sines used
            // in the rotation matrix in order to produce the counter-clockwise rotation. We invert only the cosines since
            // sin(-a) = -sin(a) and cos(-a) = cos(a).
            c = Math.cos(tilt * Angle.DEGREES_TO_RADIANS);
            s = Math.sin(tilt * Angle.DEGREES_TO_RADIANS);
            this.multiply(1, 0, 0, 0,
                0, c, s, 0,
                0, -s, c, 0,
                0, 0, 0, 1);

            // Heading. Rotate the eye point in a clockwise direction about the z axis again. This has a different effect than
            // roll when tilt is non-zero because the viewer is no longer looking down the z axis.
            c = Math.cos(heading * Angle.DEGREES_TO_RADIANS);
            s = Math.sin(heading * Angle.DEGREES_TO_RADIANS);
            this.multiply(c, -s, 0, 0,
                s, c, 0, 0,
                0, 0, 1, 0,
                0, 0, 0, 1);

            // Compute the eye point in model coordinates. This point is mapped to the origin in the look at transform below.
            globe.computePointFromPosition(eyePosition.latitude, eyePosition.longitude, eyePosition.altitude, eyePoint);
            ex = eyePoint[0];
            ey = eyePoint[1];
            ez = eyePoint[2];

            // Transform the origin to the local coordinate system at the eye point.
            WWMath.localCoordinateAxesAtPoint(eyePoint, globe, xAxis, yAxis, zAxis);
            xx = xAxis[0];
            xy = xAxis[1];
            xz = xAxis[2];
            yx = yAxis[0];
            yy = yAxis[1];
            yz = yAxis[2];
            zx = zAxis[0];
            zy = zAxis[1];
            zz = zAxis[2];

            this.multiply(xx, xy, xz, -xx * ex - xy * ey - xz * ez,
                yx, yy, yz, -yx * ex - yy * ey - yz * ez,
                zx, zy, zz, -zx * ex - zy * ey - zz * ez,
                0, 0, 0, 1);

            return this;
        };

        /**
         * Multiplies this matrix by a look at viewing matrix for the specified globe.
         * <p>
         * A look at viewing matrix places the center of the screen at the specified lookAtPosition. By default the viewer is
         * looking straight down at the look at position from the specified range, with the globe's normal vector coming out of
         * the screen and north pointing toward the top of the screen.
         * <p>
         * Range specifies the distance between the look at position and the viewer's eye point. Range values may be any positive
         * real number. A range of 0 places the eye point at the look at point, while a positive range moves the eye point away
         * from but still looking at the look at point.
         * <p>
         * Heading specifies the viewer's azimuth, or its angle relative to North. Heading values range from -180 degrees to 180
         * degrees. A heading of 0 degrees looks North, 90 degrees looks East, +-180 degrees looks South, and -90 degrees looks
         * West.
         * <p>
         * Tilt specifies the viewer's angle relative to the surface. Tilt values range from -180 degrees to 180 degrees. A tilt
         * of 0 degrees looks straight down at the globe's surface, 90 degrees looks at the horizon, and 180 degrees looks
         * straight up. Tilt values greater than 180 degrees cause the viewer to turn upside down, and are therefore rarely used.
         * <p>
         * Roll specifies the viewer's angle relative to the horizon. Roll values range from -180 degrees to 180 degrees. A roll
         * of 0 degrees orients the viewer so that up is pointing to the top of the screen, at 90 degrees up is pointing to the
         * right, at +-180 degrees up is pointing to the bottom, and at -90 up is pointing to the left.
         *
         * @param {Position} lookAtPosition The viewer's geographic look at position relative to the specified globe.
         * @param {Number} range The distance between the eye point and the look at point, in model coordinates.
         * @param {Number} heading The viewer's angle relative to north, in degrees.
         * @param {Number} tilt The viewer's angle relative to the surface, in degrees.
         * @param {Number} roll The viewer's angle relative to the horizon, in degrees.
         * @param {Globe} globe The globe the viewer is looking at.
         *
         * @throws {ArgumentError} If either the specified look-at position or globe is null or undefined, or the
         * specified range is less than zero.
         */
        Matrix.prototype.multiplyByLookAtModelview = function (lookAtPosition, range, heading, tilt, roll, globe) {
            if (!lookAtPosition) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "multiplyByLookAtModelview", "missingPosition"));
            }

            if (range < 0) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "multiplyByLookAtModelview",
                        "Range is less than zero"));
            }

            if (!globe) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "multiplyByLookAtModelview", "missingGlobe"));
            }

            // Translate the eye point along the positive z axis while keeping the look at point in the center of the viewport.
            this.multiplyByTranslation(0, 0, -range);

            // Transform the origin to the local coordinate system at the look at position, and rotate the viewer by the
            // specified heading, tilt and roll.
            this.multiplyByFirstPersonModelview(lookAtPosition, heading, tilt, roll, globe);

            return this;
        };

        /**
         * Sets this matrix to a perspective projection matrix for the specified viewport dimensions and clip distances.
         * <p>
         * A perspective projection matrix maps points in eye coordinates into clip coordinates in a way that causes
         * distant objects to appear smaller, and preserves the appropriate depth information for each point. In model
         * coordinates, a perspective projection is defined by frustum originating at the eye position and extending
         * outward in the viewer's direction. The near distance and the far distance identify the minimum and maximum
         * distance, respectively, at which an object in the scene is visible. Near and far distances must be positive
         * and may not be equal.
         *
         * @param {Number} viewportWidth The viewport width, in screen coordinates.
         * @param {Number} viewportHeight The viewport height, in screen coordinates.
         * @param {Number} fovyDegrees The camera vertical field of view.
         * @param {Number} nearDistance The near clip plane distance, in model coordinates.
         * @param {Number} farDistance The far clip plane distance, in model coordinates.
         * @throws {ArgumentError} If the specified width or height is less than or equal to zero, if the near and far
         * distances are equal, or if either the near or far distance are less than or equal to zero.
         */
        Matrix.prototype.setToPerspectiveProjection = function (viewportWidth, viewportHeight, fovyDegrees, nearDistance, farDistance) {
            if (viewportWidth <= 0) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "setToPerspectiveProjection",
                    "invalidWidth"));
            }

            if (viewportHeight <= 0) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "setToPerspectiveProjection",
                    "invalidHeight"));
            }

            if (fovyDegrees <= 0 || fovyDegrees >= 180) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "setToPerspectiveProjection",
                    "invalidFieldOfView"));
            }

            if (nearDistance === farDistance) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "setToPerspectiveProjection",
                    "Near and far distance are the same."));
            }

            if (nearDistance <= 0 || farDistance <= 0) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "setToPerspectiveProjection",
                    "Near or far distance is less than or equal to zero."));
            }

            // Compute the dimensions of the near rectangle given the specified parameters.
            var aspect = viewportWidth / viewportHeight,
                tanfovy_2 = Math.tan(fovyDegrees * 0.5 / 180.0 * Math.PI),
                nearHeight = 2 * nearDistance * tanfovy_2,
                nearWidth = nearHeight * aspect;

            // Taken from Mathematics for 3D Game Programming and Computer Graphics, Second Edition, equation 4.52.

            // Row 1
            this[0] = 2 * nearDistance / nearWidth;
            this[1] = 0;
            this[2] = 0;
            this[3] = 0;
            // Row 2
            this[4] = 0;
            this[5] = 2 * nearDistance / nearHeight;
            this[6] = 0;
            this[7] = 0;
            // Row 3
            this[8] = 0;
            this[9] = 0;
            this[10] = -(farDistance + nearDistance) / (farDistance - nearDistance);
            this[11] = -2 * nearDistance * farDistance / (farDistance - nearDistance);
            // Row 4
            this[12] = 0;
            this[13] = 0;
            this[14] = -1;
            this[15] = 0;

            return this;
        };

        /**
         * Sets this matrix to a screen projection matrix for the specified viewport dimensions.
         * <p>
         * A screen projection matrix is an orthographic projection that assumes that points in model coordinates
         * represent a screen point and a depth. Screen projection matrices therefore map model coordinates directly
         * into screen coordinates without modification. A point's xy coordinates are interpreted as literal screen
         * coordinates and must be in the viewport to be visible. A point's z coordinate is interpreted as a depth value
         * that ranges from 0 to 1. Additionally, the screen projection matrix preserves the depth value returned by
         * [DrawContext.project]{@link DrawContext#project}.
         *
         * @param {Number} viewportWidth The viewport width, in screen coordinates.
         * @param {Number} viewportHeight The viewport height, in screen coordinates.
         * @throws {ArgumentError} If the specified width or height is less than or equal to zero.
         */
        Matrix.prototype.setToScreenProjection = function (viewportWidth, viewportHeight) {
            if (viewportWidth <= 0) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "setToScreenProjection",
                    "invalidWidth"));
            }

            if (viewportHeight <= 0) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "setToScreenProjection",
                    "invalidHeight"));
            }

            // Taken from Mathematics for 3D Game Programming and Computer Graphics, Second Edition, equation 4.57.
            // Simplified to assume that the viewport origin is (0, 0).
            //
            // The third row of this projection matrix is configured so that points with z coordinates representing
            // depth values ranging from 0 to 1 are not modified after transformation into window coordinates. This
            // projection matrix maps z values in the range [0, 1] to the range [-1, 1] by applying the following
            // function to incoming z coordinates:
            //
            // zp = z0 * 2 - 1
            //
            // Where 'z0' is the point's z coordinate and 'zp' is the projected z coordinate. The GPU then maps the
            // projected z coordinate into window coordinates in the range [0, 1] by applying the following function:
            //
            // zw = zp * 0.5 + 0.5
            //
            // The result is that a point's z coordinate is effectively passed to the GPU without modification.

            // Row 1
            this[0] = 2 / viewportWidth;
            this[1] = 0;
            this[2] = 0;
            this[3] = -1;
            // Row 2
            this[4] = 0;
            this[5] = 2 / viewportHeight;
            this[6] = 0;
            this[7] = -1;
            // Row 3
            this[8] = 0;
            this[9] = 0;
            this[10] = 2;
            this[11] = -1;
            // Row 4
            this[12] = 0;
            this[13] = 0;
            this[14] = 0;
            this[15] = 1;

            return this;
        };

        /**
         * Returns this viewing matrix's eye point.
         * <p>
         * This method assumes that this matrix represents a viewing matrix. If this does not represent a viewing matrix the
         * results are undefined.
         * <p>
         * In model coordinates, a viewing matrix's eye point is the point the viewer is looking from and maps to the center of
         * the screen.
         *
         * @param {Vec3} result A pre-allocated {@link Vec3} in which to return the extracted values.
         * @return {Vec3} The specified result argument containing the viewing matrix's eye point, in model coordinates.
         * @throws {ArgumentError} If the specified result argument is null or undefined.
         */
        Matrix.prototype.extractEyePoint = function (result) {
            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "extractEyePoint", "missingResult"));
            }

            // The eye point of a modelview matrix is computed by transforming the origin (0, 0, 0, 1) by the matrix's inverse.
            // This is equivalent to transforming the inverse of this matrix's translation components in the rightmost column by
            // the transpose of its upper 3x3 components.
            result[0] = -(this[0] * this[3]) - (this[4] * this[7]) - (this[8] * this[11]);
            result[1] = -(this[1] * this[3]) - (this[5] * this[7]) - (this[9] * this[11]);
            result[2] = -(this[2] * this[3]) - (this[6] * this[7]) - (this[10] * this[11]);

            return result;
        };

        /**
         * Returns this viewing matrix's heading angle in degrees. The roll argument enables the caller to disambiguate
         * heading and roll when the two rotation axes for heading and roll are parallel, causing gimbal lock.
         * <p>
         * The result of this method is undefined if this matrix is not a viewing matrix.
         *
         * @param {Number} roll the viewing matrix's roll angle in degrees, or 0 if the roll angle is unknown
         *
         * @return {Number} the extracted heading angle in degrees
         */
        Matrix.prototype.extractHeading = function (roll) {
            var rad = roll * Angle.DEGREES_TO_RADIANS;
            var cr = Math.cos(rad);
            var sr = Math.sin(rad);

            var ch = (cr * this[0]) - (sr * this[4]);
            var sh = (sr * this[5]) - (cr * this[1]);
            return Math.atan2(sh, ch) * Angle.RADIANS_TO_DEGREES;
        };

        /**
         * Returns this viewing matrix's tilt angle in degrees.
         * <p>
         * The result of this method is undefined if this matrix is not a viewing matrix.
         *
         * @return {Number} the extracted heading angle in degrees
         */
        Matrix.prototype.extractTilt = function () {
            var ct = this[10];
            var st = Math.sqrt(this[2] * this[2] + this[6] * this[6]);
            return Math.atan2(st, ct) * Angle.RADIANS_TO_DEGREES;
        };

        /**
         * Returns this viewing matrix's forward vector.
         * <p>
         * This method assumes that this matrix represents a viewing matrix. If this does not represent a viewing matrix the
         * results are undefined.
         *
         * @param {Vec3} result A pre-allocated {@link Vec3} in which to return the extracted values.
         * @return {Vec3} The specified result argument containing the viewing matrix's forward vector, in model coordinates.
         * @throws {ArgumentError} If the specified result argument is null or undefined.
         */
        Matrix.prototype.extractForwardVector = function (result) {
            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "extractForwardVector", "missingResult"));
            }

            // The forward vector of a modelview matrix is computed by transforming the negative Z axis (0, 0, -1, 0) by the
            // matrix's inverse. We have pre-computed the result inline here to simplify this computation.
            result[0] = -this[8];
            result[1] = -this[9];
            result[2] = -this[10];

            return result;
        };

        /**
         * Extracts this viewing matrix's parameters given a viewing origin and a globe.
         * <p>
         * This method assumes that this matrix represents a viewing matrix. If this does not represent a viewing matrix the
         * results are undefined.
         * <p>
         * This returns a parameterization of this viewing matrix based on the specified origin and globe. The origin indicates
         * the model coordinate point that the view's orientation is relative to, while the globe provides the necessary model
         * coordinate context for the origin and the orientation. The origin should be either the view's eye point or a point on
         * the view's forward vector. The view's roll must be specified in order to disambiguate heading and roll when the view's
         * tilt is zero.
         * <p>
         * The following list outlines the returned key-value pairs and their meanings:
         * <ul>
         * <li> 'origin' - The geographic position corresponding to the origin point.</li>
         * <li> 'range' - The distance between the specified origin point and the view's eye point, in model coordinates.</li>
         * <li> 'heading' - The view's heading angle relative to the globe's north pointing tangent at the origin point, in degrees.</li>
         * <li> 'tilt' - The view's tilt angle relative to the globe's normal vector at the origin point, in degrees.</li>
         * <li> 'roll' - The view's roll relative to the globe's normal vector at the origin point, in degrees.</li>
         * </ul>
         * @param {Vec3} origin The origin of the viewing parameters, in model coordinates.
         * @param {Number} roll The view's roll, in degrees.
         * @param {Globe} globe The globe the viewer is looking at.
         * @param {Object} result A pre-allocated object in which to return the viewing parameters.
         *
         * @return {Object} The specified result argument containing a parameterization of this viewing matrix.
         *
         * @throws {ArgumentError} If either the specified origin or globe are null or undefined or the specified
         * result argument is null or undefined.
         */
        Matrix.prototype.extractViewingParameters = function (origin, roll, globe, result) {
            if (!origin) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "extractViewingParameters",
                        "The specified origin is null or undefined."));
            }

            if (!globe) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "extractViewingParameters", "missingGlobe"));
            }

            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "extractViewingParameters", "missingResult"));
            }

            var originPos = new Position(0, 0, 0),
                modelviewLocal = Matrix.fromIdentity(),
                range,
                ct,
                st,
                tilt,
                cr, sr,
                ch, sh,
                heading;

            globe.computePositionFromPoint(origin[0], origin[1], origin[2], originPos);

            // Transform the modelview matrix to a local coordinate system at the origin. This eliminates the geographic
            // transform contained in the modelview matrix while maintaining rotation and translation relative to the origin.
            modelviewLocal.copy(this);
            modelviewLocal.multiplyByLocalCoordinateTransform(origin, globe);

            range = -modelviewLocal[11];
            ct = modelviewLocal[10];
            st = Math.sqrt(modelviewLocal[2] * modelviewLocal[2] + modelviewLocal[6] * modelviewLocal[6]);
            tilt = Math.atan2(st, ct) * Angle.RADIANS_TO_DEGREES;

            cr = Math.cos(roll * Angle.DEGREES_TO_RADIANS);
            sr = Math.sin(roll * Angle.DEGREES_TO_RADIANS);
            ch = cr * modelviewLocal[0] - sr * modelviewLocal[4];
            sh = sr * modelviewLocal[5] - cr * modelviewLocal[1];
            heading = Math.atan2(sh, ch) * Angle.RADIANS_TO_DEGREES;

            result['origin'] = originPos;
            result['range'] = range;
            result['heading'] = heading;
            result['tilt'] = tilt;
            result['roll'] = roll;

            return result;
        };

        /**
         * Applies a specified depth offset to this projection matrix.
         * <p>
         * This method assumes that this matrix represents a projection matrix. If this does not represent a projection
         * matrix the results are undefined. Projection matrices can be created by calling
         * [setToPerspectiveProjection]{@link Matrix#setToPerspectiveProjection} or [setToScreenProjection]{@link Matrix#setToScreenProjection}.
         * <p>
         * The depth offset may be any real number and is typically used to draw geometry slightly closer to the user's
         * eye in order to give those shapes visual priority over nearby or geometry. An offset of zero has no effect.
         * An offset less than zero brings depth values closer to the eye, while an offset greater than zero pushes
         * depth values away from the eye.
         * <p>
         * Depth offset may be applied to both perspective and orthographic projection matrices. The effect on each
         * projection type is outlined here:
         * <p>
         * <strong>Perspective Projection</strong>
         * <p>
         * The effect of depth offset on a perspective projection increases exponentially with distance from the eye.
         * This has the effect of adjusting the offset for the loss in depth precision with geometry drawn further from
         * the eye. Distant geometry requires a greater offset to differentiate itself from nearby geometry, while close
         * geometry does not.
         * <p>
         * <strong>Orthographic Projection</strong>
         * <p>
         * The effect of depth offset on an orthographic projection increases linearly with distance from the eye. While
         * it is reasonable to apply a depth offset to an orthographic projection, the effect is most appropriate when
         * applied to the projection used to draw the scene. For example, when an object's coordinates are projected by
         * a perspective projection into screen coordinates then drawn using an orthographic projection, it is best to
         * apply the offset to the original perspective projection. The method [DrawContext.project]{@link DrawContext#project} performs the
         * correct behavior for the projection type used to draw the scene.
         *
         * @param {Number} depthOffset The amount of offset to apply.
         * @returns {Matrix} This matrix with it's depth offset set to the specified offset.
         */
        Matrix.prototype.offsetProjectionDepth = function (depthOffset) {

            this[10] *= 1 + depthOffset;

            return this;
        };

        /**
         * Multiplies this matrix by a specified matrix.
         *
         * @param {Matrix} matrix The matrix to multiply with this matrix.
         * @returns {Matrix} This matrix after multiplying it by the specified matrix.
         * @throws {ArgumentError} if the specified matrix is null or undefined.
         */
        Matrix.prototype.multiplyMatrix = function (matrix) {
            if (!matrix) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "multiplyMatrix", "missingMatrix"));
            }

            var ma = this,
                mb = matrix,
                ma0, ma1, ma2, ma3;

            // Row 1
            ma0 = ma[0];
            ma1 = ma[1];
            ma2 = ma[2];
            ma3 = ma[3];
            ma[0] = (ma0 * mb[0]) + (ma1 * mb[4]) + (ma2 * mb[8]) + (ma3 * mb[12]);
            ma[1] = (ma0 * mb[1]) + (ma1 * mb[5]) + (ma2 * mb[9]) + (ma3 * mb[13]);
            ma[2] = (ma0 * mb[2]) + (ma1 * mb[6]) + (ma2 * mb[10]) + (ma3 * mb[14]);
            ma[3] = (ma0 * mb[3]) + (ma1 * mb[7]) + (ma2 * mb[11]) + (ma3 * mb[15]);

            // Row 2
            ma0 = ma[4];
            ma1 = ma[5];
            ma2 = ma[6];
            ma3 = ma[7];
            ma[4] = (ma0 * mb[0]) + (ma1 * mb[4]) + (ma2 * mb[8]) + (ma3 * mb[12]);
            ma[5] = (ma0 * mb[1]) + (ma1 * mb[5]) + (ma2 * mb[9]) + (ma3 * mb[13]);
            ma[6] = (ma0 * mb[2]) + (ma1 * mb[6]) + (ma2 * mb[10]) + (ma3 * mb[14]);
            ma[7] = (ma0 * mb[3]) + (ma1 * mb[7]) + (ma2 * mb[11]) + (ma3 * mb[15]);

            // Row 3
            ma0 = ma[8];
            ma1 = ma[9];
            ma2 = ma[10];
            ma3 = ma[11];
            ma[8] = (ma0 * mb[0]) + (ma1 * mb[4]) + (ma2 * mb[8]) + (ma3 * mb[12]);
            ma[9] = (ma0 * mb[1]) + (ma1 * mb[5]) + (ma2 * mb[9]) + (ma3 * mb[13]);
            ma[10] = (ma0 * mb[2]) + (ma1 * mb[6]) + (ma2 * mb[10]) + (ma3 * mb[14]);
            ma[11] = (ma0 * mb[3]) + (ma1 * mb[7]) + (ma2 * mb[11]) + (ma3 * mb[15]);

            // Row 4
            ma0 = ma[12];
            ma1 = ma[13];
            ma2 = ma[14];
            ma3 = ma[15];
            ma[12] = (ma0 * mb[0]) + (ma1 * mb[4]) + (ma2 * mb[8]) + (ma3 * mb[12]);
            ma[13] = (ma0 * mb[1]) + (ma1 * mb[5]) + (ma2 * mb[9]) + (ma3 * mb[13]);
            ma[14] = (ma0 * mb[2]) + (ma1 * mb[6]) + (ma2 * mb[10]) + (ma3 * mb[14]);
            ma[15] = (ma0 * mb[3]) + (ma1 * mb[7]) + (ma2 * mb[11]) + (ma3 * mb[15]);

            return this;
        };

        /**
         * Multiplies this matrix by a matrix specified by individual components.
         *
         * @param {Number} m00 matrix element at row 1, column 1.
         * @param {Number} m01 matrix element at row 1, column 2.
         * @param {Number} m02 matrix element at row 1, column 3.
         * @param {Number} m03 matrix element at row 1, column 4.
         * @param {Number} m10 matrix element at row 2, column 1.
         * @param {Number} m11 matrix element at row 2, column 2.
         * @param {Number} m12 matrix element at row 2, column 3.
         * @param {Number} m13 matrix element at row 2, column 4.
         * @param {Number} m20 matrix element at row 3, column 1.
         * @param {Number} m21 matrix element at row 3, column 2.
         * @param {Number} m22 matrix element at row 3, column 3.
         * @param {Number} m23 matrix element at row 3, column 4.
         * @param {Number} m30 matrix element at row 4, column 1.
         * @param {Number} m31 matrix element at row 4, column 2.
         * @param {Number} m32 matrix element at row 4, column 3.
         * @param {Number} m33 matrix element at row 4, column 4.
         * @returns {Matrix} This matrix with its components multiplied by the specified values.
         */
        Matrix.prototype.multiply = function (m00, m01, m02, m03,
                                              m10, m11, m12, m13,
                                              m20, m21, m22, m23,
                                              m30, m31, m32, m33) {

            var ma = this,
                ma0, ma1, ma2, ma3;

            // Row 1
            ma0 = ma[0];
            ma1 = ma[1];
            ma2 = ma[2];
            ma3 = ma[3];
            ma[0] = (ma0 * m00) + (ma1 * m10) + (ma2 * m20) + (ma3 * m30);
            ma[1] = (ma0 * m01) + (ma1 * m11) + (ma2 * m21) + (ma3 * m31);
            ma[2] = (ma0 * m02) + (ma1 * m12) + (ma2 * m22) + (ma3 * m32);
            ma[3] = (ma0 * m03) + (ma1 * m13) + (ma2 * m23) + (ma3 * m33);

            // Row 2
            ma0 = ma[4];
            ma1 = ma[5];
            ma2 = ma[6];
            ma3 = ma[7];
            ma[4] = (ma0 * m00) + (ma1 * m10) + (ma2 * m20) + (ma3 * m30);
            ma[5] = (ma0 * m01) + (ma1 * m11) + (ma2 * m21) + (ma3 * m31);
            ma[6] = (ma0 * m02) + (ma1 * m12) + (ma2 * m22) + (ma3 * m32);
            ma[7] = (ma0 * m03) + (ma1 * m13) + (ma2 * m23) + (ma3 * m33);

            // Row 3
            ma0 = ma[8];
            ma1 = ma[9];
            ma2 = ma[10];
            ma3 = ma[11];
            ma[8] = (ma0 * m00) + (ma1 * m10) + (ma2 * m20) + (ma3 * m30);
            ma[9] = (ma0 * m01) + (ma1 * m11) + (ma2 * m21) + (ma3 * m31);
            ma[10] = (ma0 * m02) + (ma1 * m12) + (ma2 * m22) + (ma3 * m32);
            ma[11] = (ma0 * m03) + (ma1 * m13) + (ma2 * m23) + (ma3 * m33);

            // Row 4
            ma0 = ma[12];
            ma1 = ma[13];
            ma2 = ma[14];
            ma3 = ma[15];
            ma[12] = (ma0 * m00) + (ma1 * m10) + (ma2 * m20) + (ma3 * m30);
            ma[13] = (ma0 * m01) + (ma1 * m11) + (ma2 * m21) + (ma3 * m31);
            ma[14] = (ma0 * m02) + (ma1 * m12) + (ma2 * m22) + (ma3 * m32);
            ma[15] = (ma0 * m03) + (ma1 * m13) + (ma2 * m23) + (ma3 * m33);

            return this;
        };

        /**
         * Inverts the specified matrix and stores the result in this matrix.
         * <p>
         * This throws an exception if the specified matrix is singular.
         * <p>
         * The result of this method is undefined if this matrix is passed in as the matrix to invert.
         *
         * @param {Matrix} matrix The matrix whose inverse is computed.
         * @returns {Matrix} This matrix set to the inverse of the specified matrix.
         *
         * @throws {ArgumentError} If the specified matrix is null, undefined or cannot be inverted.
         */
        Matrix.prototype.invertMatrix = function (matrix) {
            if (!matrix) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "invertMatrix", "missingMatrix"));
            }

            // Copy the specified matrix into a mutable two-dimensional array.
            var A = [[], [], [], []];
            A[0][0] = matrix[0];
            A[0][1] = matrix[1];
            A[0][2] = matrix[2];
            A[0][3] = matrix[3];
            A[1][0] = matrix[4];
            A[1][1] = matrix[5];
            A[1][2] = matrix[6];
            A[1][3] = matrix[7];
            A[2][0] = matrix[8];
            A[2][1] = matrix[9];
            A[2][2] = matrix[10];
            A[2][3] = matrix[11];
            A[3][0] = matrix[12];
            A[3][1] = matrix[13];
            A[3][2] = matrix[14];
            A[3][3] = matrix[15];

            var index = [],
                d = Matrix.ludcmp(A, index),
                i,
                j;

            // Compute the matrix's determinant.
            for (i = 0; i < 4; i += 1) {
                d *= A[i][i];
            }

            // The matrix is singular if its determinant is zero or very close to zero.
            if (Math.abs(d) < 1.0e-8)
                return null;

            var Y = [[], [], [], []],
                col = [];
            for (j = 0; j < 4; j += 1) {
                for (i = 0; i < 4; i += 1) {
                    col[i] = 0.0;
                }

                col[j] = 1.0;
                Matrix.lubksb(A, index, col);

                for (i = 0; i < 4; i += 1) {
                    Y[i][j] = col[i];
                }
            }

            this[0] = Y[0][0];
            this[1] = Y[0][1];
            this[2] = Y[0][2];
            this[3] = Y[0][3];
            this[4] = Y[1][0];
            this[5] = Y[1][1];
            this[6] = Y[1][2];
            this[7] = Y[1][3];
            this[8] = Y[2][0];
            this[9] = Y[2][1];
            this[10] = Y[2][2];
            this[11] = Y[2][3];
            this[12] = Y[3][0];
            this[13] = Y[3][1];
            this[14] = Y[3][2];
            this[15] = Y[3][3];

            return this;
        };

        /* Internal. Intentionally not documented.
         * Utility method to solve a linear system with an LU factorization of a matrix.
         * Solves Ax=b, where A is in LU factorized form.
         * Algorithm derived from "Numerical Recipes in C", Press et al., 1988.
         *
         * @param {Number[]} A An LU factorization of a matrix.
         * @param {Number[]} index Permutation vector of that LU factorization.
         * @param {Number[]} b Vector to be solved.
         */
        // Method "lubksb" derived from "Numerical Recipes in C", Press et al., 1988
        Matrix.lubksb = function (A, index, b) {
            var ii = -1,
                i,
                j,
                sum;
            for (i = 0; i < 4; i += 1) {
                var ip = index[i];
                sum = b[ip];
                b[ip] = b[i];

                if (ii != -1) {
                    for (j = ii; j <= i - 1; j += 1) {
                        sum -= A[i][j] * b[j];
                    }
                } else if (sum != 0.0) {
                    ii = i;
                }

                b[i] = sum;
            }

            for (i = 3; i >= 0; i -= 1) {
                sum = b[i];
                for (j = i + 1; j < 4; j += 1) {
                    sum -= A[i][j] * b[j];
                }

                b[i] = sum / A[i][i];
            }
        };

        /* Internal. Intentionally not documented.
         * Utility method to perform an LU factorization of a matrix.
         * "ludcmp" is derived from "Numerical Recipes in C", Press et al., 1988.
         *
         * @param {Number[]} A matrix to be factored
         * @param {Number[]} index permutation vector
         * @returns {Number} Condition number of matrix.
         */
        Matrix.ludcmp = function (A, index) {
            var TINY = 1.0e-20,
                vv = [], /* new double[4]; */
                d = 1.0,
                temp,
                i,
                j,
                k,
                big,
                sum,
                imax,
                dum;
            for (i = 0; i < 4; i += 1) {
                big = 0.0;
                for (j = 0; j < 4; j += 1) {
                    if ((temp = Math.abs(A[i][j])) > big) {
                        big = temp;
                    }
                }

                if (big == 0.0) {
                    return 0.0; // Matrix is singular if the entire row contains zero.
                } else {
                    vv[i] = 1.0 / big;
                }
            }

            for (j = 0; j < 4; j += 1) {
                for (i = 0; i < j; i += 1) {
                    sum = A[i][j];
                    for (k = 0; k < i; k += 1) {
                        sum -= A[i][k] * A[k][j];
                    }

                    A[i][j] = sum;
                }

                big = 0.0;
                imax = -1;
                for (i = j; i < 4; i += 1) {
                    sum = A[i][j];
                    for (k = 0; k < j; k++) {
                        sum -= A[i][k] * A[k][j];
                    }

                    A[i][j] = sum;

                    if ((dum = vv[i] * Math.abs(sum)) >= big) {
                        big = dum;
                        imax = i;
                    }
                }

                if (j != imax) {
                    for (k = 0; k < 4; k += 1) {
                        dum = A[imax][k];
                        A[imax][k] = A[j][k];
                        A[j][k] = dum;
                    }

                    d = -d;
                    vv[imax] = vv[j];
                }

                index[j] = imax;
                if (A[j][j] == 0.0)
                    A[j][j] = TINY;

                if (j != 3) {
                    dum = 1.0 / A[j][j];
                    for (i = j + 1; i < 4; i += 1) {
                        A[i][j] *= dum;
                    }
                }
            }

            return d;
        };

        /**
         * Inverts the specified matrix and stores the result in this matrix.
         * <p>
         * The specified matrix is assumed to represent an orthonormal transform matrix. This matrix's upper 3x3 is
         * transposed, then its fourth column is transformed by the transposed upper 3x3 and negated.
         * <p>
         * The result of this method is undefined if this matrix is passed in as the matrix to invert.
         *
         * @param {Matrix} matrix The matrix whose inverse is computed. This matrix is assumed to represent an
         * orthonormal transform matrix.
         * @returns {Matrix} This matrix set to the inverse of the specified matrix.
         *
         * @throws {ArgumentError} If the specified matrix is null or undefined.
         */
        Matrix.prototype.invertOrthonormalMatrix = function (matrix) {
            if (!matrix) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "invertOrthonormalMatrix", "missingMatrix"));
            }

            // 'a' is assumed to contain a 3D transformation matrix.
            // Upper-3x3 is inverted, translation is transformed by inverted-upper-3x3 and negated.

            var a = matrix;

            this[0] = a[0];
            this[1] = a[4];
            this[2] = a[8];
            this[3] = 0.0 - (a[0] * a[3]) - (a[4] * a[7]) - (a[8] * a[11]);

            this[4] = a[1];
            this[5] = a[5];
            this[6] = a[9];
            this[7] = 0.0 - (a[1] * a[3]) - (a[5] * a[7]) - (a[9] * a[11]);

            this[8] = a[2];
            this[9] = a[6];
            this[10] = a[10];
            this[11] = 0.0 - (a[2] * a[3]) - (a[6] * a[7]) - (a[10] * a[11]);

            this[12] = 0;
            this[13] = 0;
            this[14] = 0;
            this[15] = 1;

            return this;
        };

        /**
         * Computes the eigenvectors of this matrix.
         * <p>
         * The eigenvectors are returned sorted from the most prominent vector to the least prominent vector.
         * Each eigenvector has length equal to its corresponding eigenvalue.
         *
         * @param {Vec3} result1 A pre-allocated vector in which to return the most prominent eigenvector.
         * @param {Vec3} result2 A pre-allocated vector in which to return the second most prominent eigenvector.
         * @param {Vec3} result3 A pre-allocated vector in which to return the least prominent eigenvector.
         *
         * @throws {ArgumentError} if any argument is null or undefined or if this matrix is not symmetric.
         */
        Matrix.prototype.eigensystemFromSymmetricMatrix = function (result1, result2, result3) {
            if (!result1 || !result2 || !result3) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "eigensystemFromSymmetricMatrix", "missingResult"));
            }

            if (this[1] != this[4] || this[2] != this[8] || this[6] != this[9]) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "eigensystemFromSymmetricMatrix",
                        "Matrix is not symmetric"));
            }

            // Taken from Mathematics for 3D Game Programming and Computer Graphics, Second Edition, listing 14.6.

            var epsilon = 1.0e-10,
                // Since the matrix is symmetric m12=m21, m13=m31 and m23=m32, therefore we can ignore the values m21,
                // m32 and m32.
                m11 = this[0],
                m12 = this[1],
                m13 = this[2],
                m22 = this[5],
                m23 = this[6],
                m33 = this[10],
                r = [
                    [1, 0, 0],
                    [0, 1, 0],
                    [0, 0, 1]
                ],
                maxSweeps = 32,
                u, u2, u2p1, t, c, s, temp, i, i1, i2, i3;

            for (var a = 0; a < maxSweeps; a++) {
                // Exit if off-diagonal entries small enough
                if (WWMath.fabs(m12) < epsilon && WWMath.fabs(m13) < epsilon && WWMath.fabs(m23) < epsilon)
                    break;

                // Annihilate (1,2) entry.
                if (m12 != 0) {
                    u = (m22 - m11) * 0.5 / m12;
                    u2 = u * u;
                    u2p1 = u2 + 1;
                    t = (u2p1 != u2) ? ((u < 0) ? -1 : 1) * (Math.sqrt(u2p1) - WWMath.fabs(u)) : 0.5 / u;
                    c = 1 / Math.sqrt(t * t + 1);
                    s = c * t;

                    m11 -= t * m12;
                    m22 += t * m12;
                    m12 = 0;

                    temp = c * m13 - s * m23;
                    m23 = s * m13 + c * m23;
                    m13 = temp;

                    for (i = 0; i < 3; i++) {
                        temp = c * r[i][0] - s * r[i][1];
                        r[i][1] = s * r[i][0] + c * r[i][1];
                        r[i][0] = temp;
                    }
                }

                // Annihilate (1,3) entry.
                if (m13 != 0) {
                    u = (m33 - m11) * 0.5 / m13;
                    u2 = u * u;
                    u2p1 = u2 + 1;
                    t = (u2p1 != u2) ? ((u < 0) ? -1 : 1) * (Math.sqrt(u2p1) - WWMath.fabs(u)) : 0.5 / u;
                    c = 1 / Math.sqrt(t * t + 1);
                    s = c * t;

                    m11 -= t * m13;
                    m33 += t * m13;
                    m13 = 0;

                    temp = c * m12 - s * m23;
                    m23 = s * m12 + c * m23;
                    m12 = temp;

                    for (i = 0; i < 3; i++) {
                        temp = c * r[i][0] - s * r[i][2];
                        r[i][2] = s * r[i][0] + c * r[i][2];
                        r[i][0] = temp;
                    }
                }

                // Annihilate (2,3) entry.
                if (m23 != 0) {
                    u = (m33 - m22) * 0.5 / m23;
                    u2 = u * u;
                    u2p1 = u2 + 1;
                    t = (u2p1 != u2) ? ((u < 0) ? -1 : 1) * (Math.sqrt(u2p1) - WWMath.fabs(u)) : 0.5 / u;
                    c = 1 / Math.sqrt(t * t + 1);
                    s = c * t;

                    m22 -= t * m23;
                    m33 += t * m23;
                    m23 = 0;

                    temp = c * m12 - s * m13;
                    m13 = s * m12 + c * m13;
                    m12 = temp;

                    for (i = 0; i < 3; i++) {
                        temp = c * r[i][1] - s * r[i][2];
                        r[i][2] = s * r[i][1] + c * r[i][2];
                        r[i][1] = temp;
                    }
                }
            }

            i1 = 0;
            i2 = 1;
            i3 = 2;

            if (m11 < m22) {
                temp = m11;
                m11 = m22;
                m22 = temp;

                temp = i1;
                i1 = i2;
                i2 = temp;
            }

            if (m22 < m33) {
                temp = m22;
                m22 = m33;
                m33 = temp;

                temp = i2;
                i2 = i3;
                i3 = temp;
            }

            if (m11 < m22) {
                temp = m11;
                m11 = m22;
                m22 = temp;

                temp = i1;
                i1 = i2;
                i2 = temp;
            }

            result1[0] = r[0][i1];
            result1[1] = r[1][i1];
            result1[2] = r[2][i1];

            result2[0] = r[0][i2];
            result2[1] = r[1][i2];
            result2[2] = r[2][i2];

            result3[0] = r[0][i3];
            result3[1] = r[1][i3];
            result3[2] = r[2][i3];

            result1.normalize();
            result2.normalize();
            result3.normalize();

            result1.multiply(m11);
            result2.multiply(m22);
            result3.multiply(m33);
        };

        /**
         * Extracts and returns a new matrix whose upper 3x3 entries are identical to those of this matrix,
         * and whose fourth row and column are 0 except for a 1 in the diagonal position.
         * @returns {Matrix} The upper 3x3 matrix of this matrix.
         */
        Matrix.prototype.upper3By3 = function () {
            var result = Matrix.fromIdentity();

            result[0] = this[0];
            result[1] = this[1];
            result[2] = this[2];

            result[4] = this[4];
            result[5] = this[5];
            result[6] = this[6];

            result[8] = this[8];
            result[9] = this[9];
            result[10] = this[10];

            return result;
        };

        /**
         * Projects a Cartesian point to screen coordinates. This method assumes this matrix represents an inverse
         * modelview-projection matrix. The result of this method is undefined if this matrix is not an inverse
         * modelview-projection matrix.
         * <p/>
         * The resultant screen point is in OpenGL screen coordinates, with the origin in the bottom-left corner and axes
         * that extend up and to the right from the origin.
         * <p/>
         * This stores the projected point in the result argument, and returns a boolean value indicating whether or not the
         * projection is successful. This returns false if the Cartesian point is clipped by the near clipping plane or the
         * far clipping plane.
         *
         * @param {Number}    x the Cartesian point's X component
         * @param {Number}    y the Cartesian point's y component
         * @param {Number}    z the Cartesian point's z component
         * @param {Rectangle} viewport the viewport defining the screen point's coordinate system
         * @param {Vec3}    result a pre-allocated {@link Vec3} in which to return the projected point
         *
         * @return {boolean} true if the transformation is successful, otherwise false
         *
         * @throws {ArgumentError} If any argument is null
         */
        Matrix.prototype.project = function (x, y, z, viewport, result) {
            if (!viewport) {
                throw new ArgumentError(Logger.logMessage(Logger.ERROR, "Matrix", "project",
                    "missingViewport"));
            }

            if (!result) {
                throw new ArgumentError(Logger.logMessage(Logger.ERROR, "Matrix", "project",
                    "missingResult"));
            }

            // Transform the model point from model coordinates to eye coordinates then to clip coordinates. This inverts
            // the Z axis and stores the negative of the eye coordinate Z value in the W coordinate.
            var sx = this[0] * x + this[1] * y + this[2] * z + this[3];
            var sy = this[4] * x + this[5] * y + this[6] * z + this[7];
            var sz = this[8] * x + this[9] * y + this[10] * z + this[11];
            var sw = this[12] * x + this[13] * y + this[14] * z + this[15];

            if (sw === 0) {
                return false;
            }

            // Complete the conversion from model coordinates to clip coordinates by dividing by W. The resultant X, Y
            // and Z coordinates are in the range [-1,1].
            sx /= sw;
            sy /= sw;
            sz /= sw;

            // Clip the point against the near and far clip planes.
            if (sz < -1 || sz > 1) {
                return false;
            }

            // Convert the point from clip coordinate to the range [0,1]. This enables the X and Y coordinates to be
            // converted to screen coordinates, and the Z coordinate to represent a depth value in the range[0,1].
            sx = sx * 0.5 + 0.5;
            sy = sy * 0.5 + 0.5;
            sz = sz * 0.5 + 0.5;

            // Convert the X and Y coordinates from the range [0,1] to screen coordinates.
            sx = sx * viewport.width + viewport.x;
            sy = sy * viewport.height + viewport.y;

            result[0] = sx;
            result[1] = sy;
            result[2] = sz;

            return true;
        };

        /**
         * Transforms the specified screen point from WebGL screen coordinates to model coordinates. This method assumes
         * this matrix represents an inverse modelview-projection matrix. The result of this method is
         * undefined if this matrix is not an inverse modelview-projection matrix.
         * <p>
         * The screen point is understood to be in WebGL screen coordinates, with the origin in the bottom-left corner
         * and axes that extend up and to the right from the origin.
         * <p>
         * This function stores the transformed point in the result argument, and returns true or false to indicate whether the
         * transformation is successful. It returns false if the modelview or projection matrices
         * are malformed, or if the screenPoint is clipped by the near clipping plane or the far clipping plane.
         *
         * @param {Vec3} screenPoint The screen coordinate point to un-project.
         * @param {Rectangle} viewport The viewport defining the screen point's coordinate system
         * @param {Vec3} result A pre-allocated vector in which to return the unprojected point.
         * @returns {boolean} true if the transformation is successful, otherwise false.
         * @throws {ArgumentError} If either the specified point or result argument is null or undefined.
         */
        Matrix.prototype.unProject = function (screenPoint, viewport, result) {
            if (!screenPoint) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "unProject",
                    "missingPoint"));
            }

            if (!viewport) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "unProject",
                    "missingViewport"));
            }

            if (!result) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Matrix", "unProject",
                    "missingResult"));
            }

            var sx = screenPoint[0],
                sy = screenPoint[1],
                sz = screenPoint[2];

            // Convert the XY screen coordinates to coordinates in the range [0, 1]. This enables the XY coordinates to
            // be converted to clip coordinates.
            sx = (sx - viewport.x) / viewport.width;
            sy = (sy - viewport.y) / viewport.height;

            // Convert from coordinates in the range [0, 1] to clip coordinates in the range [-1, 1].
            sx = sx * 2 - 1;
            sy = sy * 2 - 1;
            sz = sz * 2 - 1;

            // Clip the point against the near and far clip planes. In clip coordinates the near and far clip planes are
            // perpendicular to the Z axis and are located at -1 and 1, respectively.
            if (sz < -1 || sz > 1) {
                return false;
            }

            // Transform the screen point from clip coordinates to model coordinates. This inverts the Z axis and stores
            // the negative of the eye coordinate Z value in the W coordinate.
            var
                x = this[0] * sx + this[1] * sy + this[2] * sz + this[3],
                y = this[4] * sx + this[5] * sy + this[6] * sz + this[7],
                z = this[8] * sx + this[9] * sy + this[10] * sz + this[11],
                w = this[12] * sx + this[13] * sy + this[14] * sz + this[15];

            if (w === 0) {
                return false;
            }

            // Complete the conversion from model coordinates to clip coordinates by dividing by W.
            result[0] = x / w;
            result[1] = y / w;
            result[2] = z / w;

            return true;
        };

        /**
         * Returns a string representation of this matrix.
         * @returns {String} A string representation of this matrix.
         */
        Matrix.prototype.toString = function () {
            return "(" + this[0] + ", " + this[1] + ", " + this[2] + ", " + this[3] + ")\n" +
                "(" + this[4] + ", " + this[5] + ", " + this[6] + ", " + this[7] + ")\n" +
                "(" + this[8] + ", " + this[9] + ", " + this[10] + ", " + this[11] + ")\n" +
                "(" + this[12] + ", " + this[13] + ", " + this[14] + ", " + this[15] + ")";
        };

        return Matrix;
    });


/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports PickedObject
 */
define('pick/PickedObject',[],
    function () {
        "use strict";

        /**
         * Constructs a picked object.
         * @alias PickedObject
         * @constructor
         * @classdesc Represents a picked object.
         * @param {Color} color The pick color identifying the object.
         * @param {Object} userObject An object to associate with this picked object, usually the picked shape.
         * @param {Position} position The picked object's geographic position. May be null if unknown.
         * @param {Layer} parentLayer The layer containing the picked object.
         * @param {Boolean} isTerrain true if the picked object is terrain, otherwise false.
         */
        var PickedObject = function (color, userObject, position, parentLayer, isTerrain) {

            /**
             * This picked object's pick color.
             * @type {Color}
             * @readonly
             */
            this.color = color;

            /**
             * The picked shape.
             * @type {Object}
             * @readonly
             */
            this.userObject = userObject;

            /**
             * This picked object's geographic position.
             * @type {Position}
             * @readonly
             */
            this.position = position;

            /**
             * The layer containing this picked object.
             * @type {Layer}
             * @readonly
             */
            this.parentLayer = parentLayer;

            /**
             * Indicates whether this picked object is terrain.
             * @type {Boolean}
             * @readonly
             */
            this.isTerrain = isTerrain;

            /**
             * Indicates whether this picked object is the top object.
             * @type {boolean}
             */
            this.isOnTop = false;
        };

        return PickedObject;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports UnsupportedOperationError
 */
define('error/UnsupportedOperationError',['../error/AbstractError'],
    function (AbstractError) {
        "use strict";

        /**
         * Constructs an unsupported-operation error with a specified message.
         * @alias UnsupportedOperationError
         * @constructor
         * @classdesc Represents an error associated with an operation that is not available or should not be invoked.
         * Typically raised when an abstract function of an abstract base class is called because a subclass has not
         * implemented the function.
         * @augments AbstractError
         * @param {String} message The message.
         */
        var UnsupportedOperationError = function (message) {
            AbstractError.call(this, "UnsupportedOperationError", message);

            var stack;
            try {
                //noinspection ExceptionCaughtLocallyJS
                throw new Error();
            } catch (e) {
                stack = e.stack;
            }
            this.stack = stack;
        };

        UnsupportedOperationError.prototype = Object.create(AbstractError.prototype);

        return UnsupportedOperationError;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports Renderable
 */
define('render/Renderable',[
        '../util/Logger',
        '../error/UnsupportedOperationError'
    ],
    function (Logger,
              UnsupportedOperationError) {
        "use strict";

        /**
         * Constructs a base renderable.
         * @alias Renderable
         * @constructor
         * @classdesc Represents a shape or other object that can be rendered. This is an abstract class and is not
         * meant to be instantiated directly.
         */
        var Renderable = function () {

            /**
             * The display name of the renderable.
             * @type {String}
             * @default "Renderable"
             */
            this.displayName = "Renderable";

            /**
             * Indicates whether to display this renderable.
             * @type {Boolean}
             * @default true
             */
            this.enabled = true;

            /**
             * Indicates the object to return as the userObject of this shape when picked. If null,
             * then this shape is returned as the userObject.
             * @type {Object}
             * @default null
             * @see  [PickedObject.userObject]{@link PickedObject#userObject}
             */
            this.pickDelegate = null;

            /**
             * An application defined object associated with this renderable. A typical use case is to associate
             * application defined data with a picked renderable.
             * @type {Object}
             * @default An empty object
             */
            this.userProperties = {};
        };

        /**
         * Render this renderable. Some shapes actually draw themselves during this call, others only add themselves
         * to the draw context's ordered rendering list for subsequent drawing when their renderOrdered method is called.
         * This method is intended to be called by layers such as {@link RenderableLayer} and not by applications.
         * @param {DrawContext} dc The current draw context.
         */
        Renderable.prototype.render = function (dc) {
            throw new UnsupportedOperationError(
                Logger.logMessage(Logger.LEVEL_SEVERE, "Renderable", "render", "abstractInvocation"));
        };

        return Renderable;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
define('shapes/Annotation',[
        '../shapes/AnnotationAttributes',
        '../error/ArgumentError',
        '../shaders/BasicTextureProgram',
        '../util/Color',
        '../util/Font',
        '../util/Insets',
        '../util/Logger',
        '../geom/Matrix',
        '../util/Offset',
        '../pick/PickedObject',
        '../render/Renderable',
        '../shapes/TextAttributes',
        '../geom/Vec2',
        '../geom/Vec3',
        '../util/WWMath'
    ],
    function (AnnotationAttributes,
              ArgumentError,
              BasicTextureProgram,
              Color,
              Font,
              Insets,
              Logger,
              Matrix,
              Offset,
              PickedObject,
              Renderable,
              TextAttributes,
              Vec2,
              Vec3,
              WWMath) {
        "use strict";

        /**
         * Constructs an annotation.
         * @alias Annotation
         * @constructor
         * @augments Renderable
         * @classdesc Represents an Annotation shape. An annotation displays a callout, a text and a leader pointing
         * the annotation's geographic position to the ground.
         * @param {Position} position The annotations's geographic position.
         * @param {AnnotationAttributes} attributes The attributes to associate with this annotation.
         * @throws {ArgumentError} If the specified position is null or undefined.
         */
        var Annotation = function (position, attributes) {

            if (!position) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Annotation", "constructor", "missingPosition"));
            }

            Renderable.call(this);

            /**
             * This annotation's geographic position.
             * @type {Position}
             */
            this.position = position;

            /**
             * The annotation's attributes.
             * @type {AnnotationAttributes}
             * @default see [AnnotationAttributes]{@link AnnotationAttributes}
             */
            this.attributes = attributes ? attributes : new AnnotationAttributes(null);

            /**
             * This annotation's altitude mode. May be one of
             * <ul>
             *  <li>[WorldWind.ABSOLUTE]{@link WorldWind#ABSOLUTE}</li>
             *  <li>[WorldWind.RELATIVE_TO_GROUND]{@link WorldWind#RELATIVE_TO_GROUND}</li>
             *  <li>[WorldWind.CLAMP_TO_GROUND]{@link WorldWind#CLAMP_TO_GROUND}</li>
             * </ul>
             * @default WorldWind.ABSOLUTE
             */
            this.altitudeMode = WorldWind.ABSOLUTE;

            // Internal use only. Intentionally not documented.
            this.layer = null;

            // Internal use only. Intentionally not documented.
            this.lastStateKey = null;

            // Internal use only. Intentionally not documented.
            this.calloutTransform = Matrix.fromIdentity();

            // Internal use only. Intentionally not documented.
            this.calloutOffset = new WorldWind.Offset(
                WorldWind.OFFSET_FRACTION, 0.5,
                WorldWind.OFFSET_FRACTION, 0);

            // Internal use only. Intentionally not documented.
            this.label = "";

            // Internal use only. Intentionally not documented.
            this.labelTexture = null;

            // Internal use only. Intentionally not documented.
            this.labelTransform = Matrix.fromIdentity();

            // Internal use only. Intentionally not documented.
            this.placePoint = new Vec3(0, 0, 0);

            // Internal use only. Intentionally not documented.
            this.depthOffset = -2.05;

            // Internal use only. Intentionally not documented.
            this.calloutPoints = null;
        };

        Annotation.matrix = Matrix.fromIdentity();
        Annotation.screenPoint = new Vec3(0, 0, 0);
        Annotation.scratchPoint = new Vec3(0, 0, 0);

        Annotation.prototype = Object.create(Renderable.prototype);

        Object.defineProperties(Annotation.prototype, {

            /**
             * The text for this annotation.
             * @type {String}
             * @memberof Annotation.prototype
             */
            text: {
                get: function () {
                    return this.label;
                },
                set: function (value) {
                    this.label = value;
                    this.lastStateKey = null;
                }
            }
        });

        /**
         * Draws this shape as an ordered renderable. Applications do not call this function. It is called by
         * [WorldWindow]{@link WorldWindow} during rendering.
         * @param {DrawContext} dc The current draw context.
         */
        Annotation.prototype.renderOrdered = function (dc) {

            this.drawOrderedAnnotation(dc);

            if (dc.pickingMode) {

                var po = new PickedObject(this.pickColor.clone(), this,
                    this.position, this.layer, false);

                if (dc.pickPoint) {
                    if (this.labelBounds.containsPoint(
                            dc.convertPointToViewport(dc.pickPoint, Annotation.scratchPoint))) {
                        po.labelPicked = true;
                    }
                }

                dc.resolvePick(po);
            }
        };

        /**
         * Creates a new annotation that is a copy of this annotation.
         * @returns {Annotation} The new annotation.
         */
        Annotation.prototype.clone = function () {
            var clone = new Annotation(this.position);

            clone.copy(this);
            clone.pickDelegate = this.pickDelegate ? this.pickDelegate : this;

            return clone;
        };

        /**
         * Copies the contents of a specified annotation to this annotation.
         * @param {Annotation} that The Annotation to copy.
         */
        Annotation.prototype.copy = function (that) {
            this.position = that.position;
            this.enabled = that.enabled;
            this.attributes = that.attributes;
            this.label = that.label;
            this.altitudeMode = that.altitudeMode;
            this.pickDelegate = that.pickDelegate;
            this.depthOffset = that.depthOffset;

            return this;
        };

        /**
         * Renders this annotation. This method is typically not called by applications but is called by
         * {@link RenderableLayer} during rendering. For this shape this method creates and
         * enques an ordered renderable with the draw context and does not actually draw the annotation.
         * @param {DrawContext} dc The current draw context.
         */
        Annotation.prototype.render = function (dc) {

            if (!this.enabled) {
                return;
            }

            if (!dc.accumulateOrderedRenderables) {
                return;
            }

            if (dc.globe.projectionLimits
                && !dc.globe.projectionLimits.containsLocation(this.position.latitude, this.position.longitude)) {
                return;
            }

            var orderedAnnotation;
            if (this.lastFrameTime !== dc.timestamp) {
                orderedAnnotation = this.makeOrderedRenderable(dc);
            } else {
                var annotationCopy = this.clone();
                orderedAnnotation = annotationCopy.makeOrderedRenderable(dc);
            }

            if (!orderedAnnotation) {
                return;
            }

            orderedAnnotation.layer = dc.currentLayer;

            this.lastFrameTime = dc.timestamp;
            dc.addOrderedRenderable(orderedAnnotation);
        };

        // Internal. Intentionally not documented.
        Annotation.prototype.drawOrderedAnnotation = function (dc) {
            this.beginDrawing(dc);

            try {
                this.doDrawOrderedAnnotation(dc);
            } finally {
                this.endDrawing(dc);
            }
        };

        /* Intentionally not documented
         * Creates an ordered renderable for this shape.
         * @protected
         * @param {DrawContext} dc The current draw context.
         * @returns {OrderedRenderable} The ordered renderable. May be null, in which case an ordered renderable
         * cannot be created or should not be created at the time this method is called.
         */
        Annotation.prototype.makeOrderedRenderable = function (dc) {

            var w, h, s, iLeft, iRight, iTop, iBottom,
                offset, leaderGapHeight;

            // Conditionally wraps the text to the width defined in the attributes.
            // Also conditionally truncates the wrapped text to the height, if defined. 
            if (this.attributes.width > 0) {
                this.label = dc.textRenderer.wrap(
                    this.label,
                    this.attributes.width, this.attributes.height);
            }

            // Compute the annotation's model point.
            dc.surfacePointForMode(this.position.latitude, this.position.longitude, this.position.altitude,
                this.altitudeMode, this.placePoint);

            this.eyeDistance = dc.eyePoint.distanceTo(this.placePoint);

            // Compute the annotation's screen point in the OpenGL coordinate system of the WorldWindow
            // by projecting its model coordinate point onto the viewport. Apply a depth offset in order
            // to cause the annotation to appear above nearby terrain.
            if (!dc.projectWithDepth(this.placePoint, this.depthOffset, Annotation.screenPoint)) {
                return null;
            }

            this.labelTexture = dc.createTextTexture(this.label, this.attributes.textAttributes);

            w = this.labelTexture.imageWidth;
            h = this.labelTexture.imageHeight;
            s = this.attributes.scale;
            iLeft = this.attributes.insets.left;
            iRight = this.attributes.insets.right;
            iTop = this.attributes.insets.top;
            iBottom = this.attributes.insets.bottom;
            leaderGapHeight = this.attributes.leaderGapHeight;

            offset = this.calloutOffset.offsetForSize((w + iLeft + iRight) * s, (h + iTop + iBottom) * s);

            this.calloutTransform.setTranslation(
                Annotation.screenPoint[0] - offset[0],
                Annotation.screenPoint[1] + leaderGapHeight,
                Annotation.screenPoint[2]);

            this.labelTransform.setTranslation(
                Annotation.screenPoint[0] - offset[0] + iLeft * s,
                Annotation.screenPoint[1] + leaderGapHeight + iBottom * s,
                Annotation.screenPoint[2]);

            this.labelTransform.setScale(w * s, h * s, 1);

            this.labelBounds = WWMath.boundingRectForUnitQuad(this.labelTransform);

            // Compute dimensions of the callout taking in consideration the insets
            var width = (w + iLeft + iRight) * s;
            var height = (h + iTop + iBottom) * s;

            var leaderOffsetX = (width / 2);

            var leaderOffsetY = -leaderGapHeight;

            if (!this.attributes.drawLeader) {
                leaderOffsetY = 0;
            }

            if (this.attributes.stateKey !== this.lastStateKey) {
                this.calloutPoints = this.createCallout(
                    width, height,
                    leaderOffsetX, leaderOffsetY,
                    this.attributes.leaderGapWidth, this.attributes.cornerRadius);
            }

            return this;
        };

        // Internal. Intentionally not documented.
        Annotation.prototype.beginDrawing = function (dc) {
            var gl = dc.currentGlContext,
                program;

            dc.findAndBindProgram(BasicTextureProgram);

            program = dc.currentProgram;

            gl.enableVertexAttribArray(program.vertexPointLocation);
            gl.enableVertexAttribArray(program.vertexTexCoordLocation);

            program.loadModulateColor(gl, dc.pickingMode);
        };

        // Internal. Intentionally not documented.
        Annotation.prototype.endDrawing = function (dc) {
            var gl = dc.currentGlContext,
                program = dc.currentProgram;

            // Clear the vertex attribute state.
            gl.disableVertexAttribArray(program.vertexPointLocation);
            gl.disableVertexAttribArray(program.vertexTexCoordLocation);

            // Clear GL bindings.
            dc.bindProgram(null);
        };

        // Internal. Intentionally not documented.
        Annotation.prototype.drawCorner = function (x0, y0, cornerRadius, start, end, steps, buffer, startIdx) {
            if (cornerRadius < 1) {
                return startIdx;
            }

            var step = (end - start) / (steps - 1);
            for (var i = 1; i < steps - 1; i++) {
                var a = start + step * i;
                var x = x0 + Math.cos(a) * cornerRadius;
                var y = y0 + Math.sin(a) * cornerRadius;
                buffer[startIdx++] = x;
                buffer[startIdx++] = y;
            }

            return startIdx;
        };

        // Internal. Intentionally not documented.
        Annotation.prototype.createCallout = function (width, height, leaderOffsetX, leaderOffsetY, leaderGapWidth,
                                                       cornerRadius) {

            var cornerSteps = 16;

            var numVertices = 2 * (12 + (cornerRadius < 1 ? 0 : 4 * (cornerSteps - 2)));

            var buffer = new Float32Array(numVertices);

            var idx = 0;

            //Bottom right
            buffer[idx++] = width / 2 + leaderGapWidth / 2;
            buffer[idx++] = 0;
            buffer[idx++] = width - cornerRadius;
            buffer[idx++] = 0;
            idx = this.drawCorner(width - cornerRadius, cornerRadius, cornerRadius, -Math.PI / 2, 0,
                cornerSteps, buffer, idx);

            //Right
            buffer[idx++] = width;
            buffer[idx++] = cornerRadius;
            buffer[idx++] = width;
            buffer[idx++] = height - cornerRadius;
            idx = this.drawCorner(width - cornerRadius, height - cornerRadius, cornerRadius, 0, Math.PI / 2,
                cornerSteps, buffer, idx);

            //Top
            buffer[idx++] = width - cornerRadius;
            buffer[idx++] = height;
            buffer[idx++] = cornerRadius;
            buffer[idx++] = height;
            idx = this.drawCorner(cornerRadius, height - cornerRadius, cornerRadius, Math.PI / 2, Math.PI,
                cornerSteps, buffer, idx);

            //Left
            buffer[idx++] = 0;
            buffer[idx++] = height - cornerRadius;
            buffer[idx++] = 0;
            buffer[idx++] = cornerRadius;
            idx = this.drawCorner(cornerRadius, cornerRadius, cornerRadius, Math.PI, Math.PI * 1.5,
                cornerSteps, buffer, idx);

            //Bottom left
            buffer[idx++] = cornerRadius;
            buffer[idx++] = 0;
            buffer[idx++] = width / 2 - leaderGapWidth / 2;
            buffer[idx++] = 0;

            //Draw leader
            buffer[idx++] = leaderOffsetX;
            buffer[idx++] = leaderOffsetY;

            buffer[idx++] = width / 2 + leaderGapWidth / 2;
            buffer[idx] = 0;

            return buffer;
        };

        // Internal. Intentionally not documented.
        Annotation.prototype.doDrawOrderedAnnotation = function (dc) {

            var gl = dc.currentGlContext,
                program = dc.currentProgram,
                textureBound;

            var refreshBuffers = false;

            if (dc.pickingMode) {
                this.pickColor = dc.uniquePickColor();
            }

            program.loadOpacity(gl, dc.pickingMode ? 1 : this.attributes.opacity * this.layer.opacity);

            // Attributes have changed. We need to track this because the callout vbo data may
            // have changed if scaled or text wrapping changes callout dimensions
            var calloutAttributesChanged = (this.attributes.stateKey !== this.lastStateKey);

            // Create new cache key if callout drawing points have changed
            if (!this.calloutCacheKey || calloutAttributesChanged) {
                this.calloutCacheKey = dc.gpuResourceCache.generateCacheKey();
            }

            var calloutVboId = dc.gpuResourceCache.resourceForKey(this.calloutCacheKey);

            if (!calloutVboId) {
                calloutVboId = gl.createBuffer();
                dc.gpuResourceCache.putResource(this.calloutCacheKey, calloutVboId,
                    this.calloutPoints.length * 4);

                refreshBuffers = true;
            }

            // Remove the last generated vbo data if attributes changed
            if (calloutAttributesChanged && this.calloutCacheKey) {
                dc.gpuResourceCache.removeResource(this.calloutCacheKey);
            }

            // Store current statekey because we are no longer using it
            // in this iteration
            this.lastStateKey = this.attributes.stateKey;

            // Compute and specify the MVP matrix.
            Annotation.matrix.copy(dc.screenProjection);
            Annotation.matrix.multiplyMatrix(this.calloutTransform);
            program.loadModelviewProjection(gl, Annotation.matrix);

            gl.bindBuffer(gl.ARRAY_BUFFER, calloutVboId);

            if (refreshBuffers) {
                gl.bufferData(gl.ARRAY_BUFFER,
                    this.calloutPoints, gl.STATIC_DRAW);

                dc.frameStatistics.incrementVboLoadCount(1);
            }

            program.loadColor(gl, dc.pickingMode ? this.pickColor : this.attributes.backgroundColor);
            program.loadTextureEnabled(gl, false);

            gl.vertexAttribPointer(program.vertexPointLocation, 2, gl.FLOAT, false, 0, 0);
            gl.vertexAttribPointer(program.vertexTexCoordLocation, 2, gl.FLOAT, false, 0, 0);

            gl.drawArrays(gl.TRIANGLE_FAN, 0, this.calloutPoints.length / 2);

            // Draw text
            Annotation.matrix.copy(dc.screenProjection);
            Annotation.matrix.multiplyMatrix(this.labelTransform);
            program.loadModelviewProjection(gl, Annotation.matrix);

            Annotation.matrix.setToIdentity();
            Annotation.matrix.multiplyByTextureTransform(this.labelTexture);
            program.loadTextureMatrix(gl, Annotation.matrix);

            program.loadColor(gl, dc.pickingMode ? this.pickColor : Color.WHITE);
            textureBound = this.labelTexture.bind(dc);
            program.loadTextureEnabled(gl, textureBound);

            // Configure GL to use the draw context's unit quad VBOs for both model coordinates and texture coordinates.
            // Most browsers can share the same buffer for vertex and texture coordinates, but Internet Explorer requires
            // that they be in separate buffers, so the code below uses the 3D buffer for vertex coords and the 2D
            // buffer for texture coords.
            gl.bindBuffer(gl.ARRAY_BUFFER, dc.unitQuadBuffer3());
            gl.vertexAttribPointer(program.vertexPointLocation, 3, gl.FLOAT, false, 0, 0);
            gl.bindBuffer(gl.ARRAY_BUFFER, dc.unitQuadBuffer());
            gl.vertexAttribPointer(program.vertexTexCoordLocation, 2, gl.FLOAT, false, 0, 0);

            gl.drawArrays(gl.TRIANGLE_STRIP, 0, 4);
        };

        return Annotation;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */

define('util/measure/MeasurerUtils',[
        '../../geom/Location',
        '../../geom/Position'
    ],
    function (Location,
              Position) {
        'use strict';

        /**
         * Provides utilities for Measurements.
         * @exports MeasurerUtils
         */
        var MeasurerUtils = {

            /**
             * Subdivide a list of positions so that no segment is longer then the provided maxLength.
             * <p>If needed, new intermediate positions will be created along lines that follow the given pathType one
             * of WorldWind.LINEAR, WorldWind.RHUMB_LINE or WorldWind.GREAT_CIRCLE.
             * All position elevations will be either at the terrain surface if followTerrain is true, or interpolated
             * according to the original elevations.</p>
             *
             * @param {Globe} globe
             * @param {Position[]} positions
             * @param {Boolean} followTerrain
             * @param {String} pathType One of WorldWind.LINEAR, WorldWind.RHUMB_LINE or WorldWind.GREAT_CIRCLE
             * @param {Number} maxLength The maximum length for one segment
             *
             * @return {Position[]} a list of positions with no segment longer then maxLength and elevations following
             * terrain or not.
             */
            subdividePositions: function (globe, positions, followTerrain, pathType, maxLength) {
                var subdividedPositions = [];
                var loc = new Location(0, 0);
                var destLatLon = new Location(0, 0);
                var pos1 = positions[0];
                var elevation;

                this.addPosition(globe, subdividedPositions, pos1, followTerrain);

                for (var i = 1; i < positions.length; i++) {
                    var pos2 = positions[i];
                    var arcLengthRadians = Location.greatCircleDistance(pos1, pos2);
                    loc = Location.interpolateAlongPath(pathType, 0.5, pos1, pos2, loc);
                    var arcLength = arcLengthRadians * globe.radiusAt(loc.latitude, loc.longitude);
                    if (arcLength > maxLength) {
                        // if necessary subdivide segment at regular intervals smaller then maxLength
                        var segmentAzimuth = null;
                        var segmentDistance = null;
                        var steps = Math.ceil(arcLength / maxLength); // number of intervals - at least two
                        for (var j = 1; j < steps; j++) {
                            var s = j / steps;
                            if (pathType === WorldWind.LINEAR) {
                                destLatLon = Location.interpolateLinear(s, pos1, pos2, destLatLon);
                            }
                            else if (pathType === WorldWind.RHUMB_LINE) {
                                if (segmentAzimuth == null) {
                                    segmentAzimuth = Location.rhumbAzimuth(pos1, pos2);
                                    segmentDistance = Location.rhumbDistance(pos1, pos2);
                                }
                                destLatLon = Location.rhumbLocation(pos1, segmentAzimuth, s * segmentDistance,
                                    destLatLon);
                            }
                            else {
                                //GREAT_CIRCLE
                                if (segmentAzimuth == null) {
                                    segmentAzimuth = Location.greatCircleAzimuth(pos1, pos2); //degrees
                                    segmentDistance = Location.greatCircleDistance(pos1, pos2); //radians
                                }
                                //Location, degrees, radians, Location
                                destLatLon = Location.greatCircleLocation(pos1, segmentAzimuth, s * segmentDistance,
                                    destLatLon);
                            }

                            // Set elevation
                            if (followTerrain) {
                                elevation = globe.elevationAtLocation(destLatLon.latitude, destLatLon.longitude);
                            }
                            else {
                                elevation = pos1.altitude * (1 - s) + pos2.altitude * s;
                            }

                            subdividedPositions.push(new Position(destLatLon.latitude, destLatLon.longitude, elevation));
                        }
                    }

                    // Finally add the segment end position
                    this.addPosition(globe, subdividedPositions, pos2, followTerrain);

                    // Prepare for next segment
                    pos1 = pos2;
                }

                return subdividedPositions;
            },

            /**
             * Adds a position to a list of positions.
             * If the path is following the terrain the elevation is also computed.
             *
             * @param {Globe} globe
             * @param {Position[]} positions The list of positions to add to
             * @param {Position} position The position to add to the list
             * @param {Boolean} followTerrain
             *
             * @return {Position[]} The list of positions
             */
            addPosition: function (globe, positions, position, followTerrain) {
                var elevation = position.altitude;
                if (followTerrain) {
                    elevation = globe.elevationAtLocation(position.latitude, position.longitude);
                }
                positions.push(new Position(position.latitude, position.longitude, elevation));
                return positions;
            },

            /**
             * Determines whether a location is located inside a given polygon.
             *
             * @param {Location} location
             * @param {Location[]}locations The list of positions describing the polygon.
             * Last one should be the same as the first one.
             *
             * @return {Boolean} true if the location is inside the polygon.
             */
            isLocationInside: function (location, locations) {
                var result = false;
                var p1 = locations[0];
                for (var i = 1, len = locations.length; i < len; i++) {
                    var p2 = locations[i];
                    if (((p2.latitude <= location.latitude && location.latitude < p1.latitude) ||
                        (p1.latitude <= location.latitude && location.latitude < p2.latitude)) &&
                        (location.longitude < (p1.longitude - p2.longitude) * (location.latitude - p2.latitude) /
                        (p1.latitude - p2.latitude) + p2.longitude)) {
                        result = !result;
                    }
                    p1 = p2;
                }
                return result;
            },

            /**
             * Computes the angle between two Vec3 in radians.
             *
             * @param {Vec3} v1
             * @param {Vec3} v2
             *
             * @return {Number} The ange in radians
             */
            angleBetweenVectors: function (v1, v2) {
                var dot = v1.dot(v2);
                // Compute the sum of magnitudes.
                var length = v1.magnitude() * v2.magnitude();
                // Normalize the dot product, if necessary.
                if (!(length === 0) && (length !== 1.0)) {
                    dot /= length;
                }

                // The normalized dot product should be in the range [-1, 1]. Otherwise the result is an error from
                // floating point roundoff. So if dot is less than -1 or greater than +1, we treat it as -1 and +1
                // respectively.
                if (dot < -1.0) {
                    dot = -1.0;
                }
                else if (dot > 1.0) {
                    dot = 1.0;
                }

                // Angle is arc-cosine of normalized dot product.
                return Math.acos(dot);
            }

        };

        return MeasurerUtils;

    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports Sector
 */
define('geom/Sector',[
    '../geom/Angle',
    '../error/ArgumentError',
    '../geom/Location',
    '../util/Logger',
    '../geom/Vec3',
    '../util/WWMath'
],
    function (Angle,
        ArgumentError,
        Location,
        Logger,
        Vec3,
        WWMath) {
        "use strict";

        /**
         * Constructs a Sector from specified minimum and maximum latitudes and longitudes in degrees.
         * @alias Sector
         * @constructor
         * @classdesc Represents a rectangular region in geographic coordinates in degrees.
         * @param {Number} minLatitude The sector's minimum latitude in degrees.
         * @param {Number} maxLatitude The sector's maximum latitude in degrees.
         * @param {Number} minLongitude The sector's minimum longitude in degrees.
         * @param {Number} maxLongitude The sector's maximum longitude in degrees.
         */
        var Sector = function (minLatitude, maxLatitude, minLongitude, maxLongitude) {
            /**
             * This sector's minimum latitude in degrees.
             * @type {Number}
             */
            this.minLatitude = minLatitude;
            /**
             * This sector's maximum latitude in degrees.
             * @type {Number}
             */
            this.maxLatitude = maxLatitude;
            /**
             * This sector's minimum longitude in degrees.
             * @type {Number}
             */
            this.minLongitude = minLongitude;
            /**
             * This sector's maximum longitude in degrees.
             * @type {Number}
             */
            this.maxLongitude = maxLongitude;
        };

        /**
         * A sector with minimum and maximum latitudes and minimum and maximum longitudes all zero.
         * @constant
         * @type {Sector}
         */
        Sector.ZERO = new Sector(0, 0, 0, 0);

        /**
         * A sector that encompasses the full range of latitude ([-90, 90]) and longitude ([-180, 180]).
         * @constant
         * @type {Sector}
         */
        Sector.FULL_SPHERE = new Sector(-90, 90, -180, 180);

        /**
         * Sets this sector's latitudes and longitudes to those of a specified sector.
         * @param {Sector} sector The sector to copy.
         * @returns {Sector} This sector, set to the values of the specified sector.
         * @throws {ArgumentError} If the specified sector is null or undefined.
         */
        Sector.prototype.copy = function (sector) {
            if (!sector) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Sector", "copy", "missingSector"));
            }

            this.minLatitude = sector.minLatitude;
            this.maxLatitude = sector.maxLatitude;
            this.minLongitude = sector.minLongitude;
            this.maxLongitude = sector.maxLongitude;

            return this;
        };

        /**
         * Indicates whether this sector has width or height.
         * @returns {Boolean} true if this sector's minimum and maximum latitudes or minimum and maximum
         * longitudes do not differ, otherwise false.
         */
        Sector.prototype.isEmpty = function () {
            return this.minLatitude === this.maxLatitude && this.minLongitude === this.maxLongitude;
        };

        /**
         * Returns the angle between this sector's minimum and maximum latitudes, in degrees.
         * @returns {Number} The difference between this sector's minimum and maximum latitudes, in degrees.
         */
        Sector.prototype.deltaLatitude = function () {
            return this.maxLatitude - this.minLatitude;
        };

        /**
         * Returns the angle between this sector's minimum and maximum longitudes, in degrees.
         * @returns {Number} The difference between this sector's minimum and maximum longitudes, in degrees.
         */
        Sector.prototype.deltaLongitude = function () {
            return this.maxLongitude - this.minLongitude;
        };

        /**
         * Returns the angle midway between this sector's minimum and maximum latitudes.
         * @returns {Number} The mid-angle of this sector's minimum and maximum latitudes, in degrees.
         */
        Sector.prototype.centroidLatitude = function () {
            return 0.5 * (this.minLatitude + this.maxLatitude);
        };

        /**
         * Returns the angle midway between this sector's minimum and maximum longitudes.
         * @returns {Number} The mid-angle of this sector's minimum and maximum longitudes, in degrees.
         */
        Sector.prototype.centroidLongitude = function () {
            return 0.5 * (this.minLongitude + this.maxLongitude);
        };

        /**
         * Computes the location of the angular center of this sector, which is the mid-angle of each of this sector's
         * latitude and longitude dimensions.
         * @param {Location} result A pre-allocated {@link Location} in which to return the computed centroid.
         * @returns {Location} The specified result argument containing the computed centroid.
         * @throws {ArgumentError} If the result argument is null or undefined.
         */
        Sector.prototype.centroid = function (result) {
            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Sector", "centroid", "missingResult"));
            }

            result.latitude = this.centroidLatitude();
            result.longitude = this.centroidLongitude();

            return result;
        };

        /**
         * Returns this sector's minimum latitude in radians.
         * @returns {Number} This sector's minimum latitude in radians.
         */
        Sector.prototype.minLatitudeRadians = function () {
            return this.minLatitude * Angle.DEGREES_TO_RADIANS;
        };

        /**
         * Returns this sector's maximum latitude in radians.
         * @returns {Number} This sector's maximum latitude in radians.
         */
        Sector.prototype.maxLatitudeRadians = function () {
            return this.maxLatitude * Angle.DEGREES_TO_RADIANS;
        };

        /**
         * Returns this sector's minimum longitude in radians.
         * @returns {Number} This sector's minimum longitude in radians.
         */
        Sector.prototype.minLongitudeRadians = function () {
            return this.minLongitude * Angle.DEGREES_TO_RADIANS;
        };

        /**
         * Returns this sector's maximum longitude in radians.
         * @returns {Number} This sector's maximum longitude in radians.
         */
        Sector.prototype.maxLongitudeRadians = function () {
            return this.maxLongitude * Angle.DEGREES_TO_RADIANS;
        };

        /**
         * Modifies this sector to encompass an array of specified locations.
         * @param {Location[]} locations An array of locations. The array may be sparse.
         * @returns {Sector} This sector, modified to encompass all locations in the specified array.
         * @throws {ArgumentError} If the specified array is null, undefined or empty or has fewer than two locations.
         */
        Sector.prototype.setToBoundingSector = function (locations) {
            if (!locations || locations.length < 2) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Sector", "setToBoundingSector",
                    "missingArray"));
            }

            var minLatitude = 90,
                maxLatitude = -90,
                minLongitude = 180,
                maxLongitude = -180;

            for (var idx = 0, len = locations.length; idx < len; idx += 1) {
                var location = locations[idx];

                if (!location) {
                    continue;
                }

                minLatitude = Math.min(minLatitude, location.latitude);
                maxLatitude = Math.max(maxLatitude, location.latitude);
                minLongitude = Math.min(minLongitude, location.longitude);
                maxLongitude = Math.max(maxLongitude, location.longitude);
            }

            this.minLatitude = minLatitude;
            this.maxLatitude = maxLatitude;
            this.minLongitude = minLongitude;
            this.maxLongitude = maxLongitude;

            return this;
        };

        /**
         * Computes bounding sectors from a list of locations that span the dateline.
         * @param {Location[]} locations The locations to bound.
         * @returns {Sector[]} Two sectors, one in the eastern hemisphere and one in the western hemisphere.
         * Returns null if the computed bounding sector has zero width or height.
         * @throws {ArgumentError} If the specified array is null, undefined or empty or the number of locations
         * is less than 2.
         */
        Sector.splitBoundingSectors = function (locations) {
            if (!locations || locations.length < 2) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "Sector", "splitBoundingSectors",
                    "missingArray"));
            }

            var minLat = 90;
            var minLon = 180;
            var maxLat = -90;
            var maxLon = -180;

            var lastLocation = null;

            for (var idx = 0, len = locations.length; idx < len; idx += 1) {
                var location = locations[idx];

                var lat = location.latitude;
                if (lat < minLat) {
                    minLat = lat;
                }
                if (lat > maxLat) {
                    maxLat = lat;
                }

                var lon = location.longitude;
                if (lon >= 0 && lon < minLon) {
                    minLon = lon;
                }
                if (lon <= 0 && lon > maxLon) {
                    maxLon = lon;
                }

                if (lastLocation != null) {
                    var lastLon = lastLocation.longitude;
                    if (WWMath.signum(lon) != WWMath.signum(lastLon)) {
                        if (Math.abs(lon - lastLon) < 180) {
                            // Crossing the zero longitude line too
                            maxLon = 0;
                            minLon = 0;
                        }
                    }
                }
                lastLocation = location;
            }

            if (minLat === maxLat && minLon === maxLon) {
                return null;
            }

            return [
                new Sector(minLat, maxLat, minLon, 180), // Sector on eastern hemisphere.
                new Sector(minLat, maxLat, -180, maxLon) // Sector on western hemisphere.
            ];
        };

        /**
         * Indicates whether this sector intersects a specified sector.
         * This sector intersects the specified sector when each sector's boundaries either overlap with the specified
         * sector or are adjacent to the specified sector.
         * The sectors are assumed to have normalized angles (angles within the range [-90, 90] latitude and
         * [-180, 180] longitude).
         * @param {Sector} sector The sector to test intersection with. May be null or undefined, in which case this
         * function returns false.
         * @returns {Boolean} true if the specifies sector intersections this sector, otherwise false.
         */
        Sector.prototype.intersects = function (sector) {
            // Assumes normalized angles: [-90, 90], [-180, 180].
            return sector
                && this.minLongitude <= sector.maxLongitude
                && this.maxLongitude >= sector.minLongitude
                && this.minLatitude <= sector.maxLatitude
                && this.maxLatitude >= sector.minLatitude;
        };

        /**
         * Indicates whether this sector intersects a specified sector exclusive of the sector boundaries.
         * This sector overlaps the specified sector when the union of the two sectors defines a non-empty sector.
         * The sectors are assumed to have normalized angles (angles within the range [-90, 90] latitude and
         * [-180, 180] longitude).
         * @param {Sector} sector The sector to test overlap with. May be null or undefined, in which case this
         * function returns false.
         * @returns {Boolean} true if the specified sector overlaps this sector, otherwise false.
         */
        Sector.prototype.overlaps = function (sector) {
            // Assumes normalized angles: [-90, 90], [-180, 180].
            return sector
                && this.minLongitude < sector.maxLongitude
                && this.maxLongitude > sector.minLongitude
                && this.minLatitude < sector.maxLatitude
                && this.maxLatitude > sector.minLatitude;
        };

        /**
         * Indicates whether this sector fully contains a specified sector.
         * This sector contains the specified sector when the specified sector's boundaries are completely contained
         * within this sector's boundaries, or are equal to this sector's boundaries.
         * The sectors are assumed to have normalized angles (angles within the range [-90, 90] latitude and
         * [-180, 180] longitude).
         * @param {Sector} sector The sector to test containment with. May be null or undefined, in which case this
         * function returns false.
         * @returns {Boolean} true if the specified sector contains this sector, otherwise false.
         */
        Sector.prototype.contains = function (sector) {
            // Assumes normalized angles: [-90, 90], [-180, 180].
            return sector
                && this.minLatitude <= sector.minLatitude
                && this.maxLatitude >= sector.maxLatitude
                && this.minLongitude <= sector.minLongitude
                && this.maxLongitude >= sector.maxLongitude;
        };

        /**
         * Indicates whether this sector contains a specified geographic location.
         * @param {Number} latitude The location's latitude in degrees.
         * @param {Number} longitude The location's longitude in degrees.
         * @returns {Boolean} true if this sector contains the location, otherwise false.
         */
        Sector.prototype.containsLocation = function (latitude, longitude) {
            // Assumes normalized angles: [-90, 90], [-180, 180].
            return this.minLatitude <= latitude
                && this.maxLatitude >= latitude
                && this.minLongitude <= longitude
                && this.maxLongitude >= longitude;
        };

        /**
         * Sets this sector to the intersection of itself and a specified sector.
         * @param {Sector} sector The sector to intersect with this one.
         * @returns {Sector} This sector, set to its intersection with the specified sector.
         * @throws {ArgumentError} If the specified sector is null or undefined.
         */
        Sector.prototype.intersection = function (sector) {
            if (!sector instanceof Sector) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Sector", "intersection", "missingSector"));
            }

            // Assumes normalized angles: [-180, 180], [-90, 90].
            if (this.minLatitude < sector.minLatitude)
                this.minLatitude = sector.minLatitude;
            if (this.maxLatitude > sector.maxLatitude)
                this.maxLatitude = sector.maxLatitude;
            if (this.minLongitude < sector.minLongitude)
                this.minLongitude = sector.minLongitude;
            if (this.maxLongitude > sector.maxLongitude)
                this.maxLongitude = sector.maxLongitude;

            // If the sectors do not overlap in either latitude or longitude, then the result of the above logic results in
            // the max being greater than the min. In this case, set the max to indicate that the sector is empty in
            // that dimension.
            if (this.maxLatitude < this.minLatitude)
                this.maxLatitude = this.minLatitude;
            if (this.maxLongitude < this.minLongitude)
                this.maxLongitude = this.minLongitude;

            return this;
        };

        /**
         * Returns a list of the Lat/Lon coordinates of a Sector's corners.
         *
         * @returns {Array} an array of the four corner locations, in the order SW, SE, NE, NW
         */
        Sector.prototype.getCorners = function () {
            var corners = [];

            corners.push(new Location(this.minLatitude, this.minLongitude));
            corners.push(new Location(this.minLatitude, this.maxLongitude));
            corners.push(new Location(this.maxLatitude, this.maxLongitude));
            corners.push(new Location(this.maxLatitude, this.minLongitude));

            return corners;
        };

        /**
         * Returns an array of {@link Vec3} that bounds the specified sector on the surface of the specified
         * {@link Globe}. The returned points enclose the globe's surface terrain in the sector,
         * according to the specified vertical exaggeration, minimum elevation, and maximum elevation. If the minimum and
         * maximum elevation are equal, this assumes a maximum elevation of 10 + the minimum.
         *
         * @param {Globe} globe the globe the extent relates to.
         * @param {Number} verticalExaggeration the globe's vertical surface exaggeration.
         *
         * @returns {Vec3} a set of points that enclose the globe's surface on the specified sector. Can be turned into a {@link BoundingBox}
         * with the setToVec3Points method.
         *
         * @throws {ArgumentError} if the globe is null.
         */
        Sector.prototype.computeBoundingPoints = function (globe, verticalExaggeration) {
            // TODO: Refactor this method back to computeBoundingBox.
            // This method was originally computeBoundingBox and returned a BoundingBox. This created a circular dependency between
            // Sector and BoundingBox that the Karma unit test suite doesn't appear to like. If we discover a way to make Karma handle this
            // situation, we should refactor this method.
            if (globe === null) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Sector", "computeBoundingBox", "missingGlobe"));
            }

            var minAndMaxElevations = globe.minAndMaxElevationsForSector(this);

            // Compute the exaggerated minimum and maximum heights.
            var minHeight = minAndMaxElevations[0] * verticalExaggeration;
            var maxHeight = minAndMaxElevations[1] * verticalExaggeration;

            if (minHeight === maxHeight)
                maxHeight = minHeight + 10; // Ensure the top and bottom heights are not equal.

            var points = [];
            var corners = this.getCorners();
            for (var i = 0; i < corners.length; i++) {
                points.push(globe.computePointFromPosition(corners[i].latitude, corners[i].longitude, minHeight, new Vec3(0, 0, 0)));
                points.push(globe.computePointFromPosition(corners[i].latitude, corners[i].longitude, maxHeight, new Vec3(0, 0, 0)));
            }

            // A point at the centroid captures the maximum vertical dimension.
            var centroid = this.centroid(new Location(0, 0));
            points.push(globe.computePointFromPosition(centroid.latitude, centroid.longitude, maxHeight, new Vec3(0, 0, 0)));

            // If the sector spans the equator, then the curvature of all four edges need to be taken into account. The
            // extreme points along the top and bottom edges are located at their mid-points, and the extreme points along
            // the left and right edges are on the equator. Add points with the longitude of the sector's centroid but with
            // the sector's min and max latitude, and add points with the sector's min and max longitude but with latitude
            // at the equator. See WWJINT-225.
            if (this.minLatitude < 0 && this.maxLatitude > 0) {
                points.push(globe.computePointFromPosition(this.minLatitude, centroid.longitude, maxHeight, new Vec3(0, 0, 0)));
                points.push(globe.computePointFromPosition(this.maxLatitude, centroid.longitude, maxHeight, new Vec3(0, 0, 0)));
                points.push(globe.computePointFromPosition(0, this.minLongitude, maxHeight, new Vec3(0, 0, 0)));
                points.push(globe.computePointFromPosition(0, this.maxLongitude, maxHeight, new Vec3(0, 0, 0)));
            }
            // If the sector is located entirely in the southern hemisphere, then the curvature of its top edge needs to be
            // taken into account. The extreme point along the top edge is located at its mid-point. Add a point with the
            // longitude of the sector's centroid but with the sector's max latitude. See WWJINT-225.
            else if (this.minLatitude < 0) {
                points.push(globe.computePointFromPosition(this.maxLatitude, centroid.longitude, maxHeight, new Vec3(0, 0, 0)));
            }
            // If the sector is located entirely in the northern hemisphere, then the curvature of its bottom edge needs to
            // be taken into account. The extreme point along the bottom edge is located at its mid-point. Add a point with
            // the longitude of the sector's centroid but with the sector's min latitude. See WWJINT-225.
            else {
                points.push(globe.computePointFromPosition(this.minLatitude, centroid.longitude, maxHeight, new Vec3(0, 0, 0)));
            }

            // If the sector spans 360 degrees of longitude then is a band around the entire globe. (If one edge is a pole
            // then the sector looks like a circle around the pole.) Add points at the min and max latitudes and longitudes
            // 0, 180, 90, and -90 to capture full extent of the band.
            if (this.deltaLongitude() >= 360) {
                var minLat = this.minLatitude;
                points.push(globe.computePointFromPosition(minLat, 0, maxHeight, new Vec3(0, 0, 0)));
                points.push(globe.computePointFromPosition(minLat, 90, maxHeight, new Vec3(0, 0, 0)));
                points.push(globe.computePointFromPosition(minLat, -90, maxHeight, new Vec3(0, 0, 0)));
                points.push(globe.computePointFromPosition(minLat, 180, maxHeight, new Vec3(0, 0, 0)));

                var maxLat = this.maxLatitude;
                points.push(globe.computePointFromPosition(maxLat, 0, maxHeight, new Vec3(0, 0, 0)));
                points.push(globe.computePointFromPosition(maxLat, 90, maxHeight, new Vec3(0, 0, 0)));
                points.push(globe.computePointFromPosition(maxLat, -90, maxHeight, new Vec3(0, 0, 0)));
                points.push(globe.computePointFromPosition(maxLat, 180, maxHeight, new Vec3(0, 0, 0)));
            }
            else if (this.deltaLongitude() > 180) {
                // Need to compute more points to ensure the box encompasses the full sector.
                var cLon = centroid.longitude;
                var cLat = centroid.latitude;

                // centroid latitude, longitude midway between min longitude and centroid longitude
                var lon = (this.minLongitude + cLon) / 2;
                points.push(globe.computePointFromPosition(cLat, lon, maxHeight, new Vec3(0, 0, 0)));

                // centroid latitude, longitude midway between centroid longitude and max longitude
                lon = (cLon + this.maxLongitude) / 2;
                points.push(globe.computePointFromPosition(cLat, lon, maxHeight, new Vec3(0, 0, 0)));

                // centroid latitude, longitude at min longitude and max longitude
                points.push(globe.computePointFromPosition(cLat, this.minLongitude, maxHeight, new Vec3(0, 0, 0)));
                points.push(globe.computePointFromPosition(cLat, this.maxLongitude, maxHeight, new Vec3(0, 0, 0)));
            }

            return points;
        };

        /**
         * Sets this sector to the union of itself and a specified sector.
         * @param {Sector} sector The sector to union with this one.
         * @returns {Sector} This sector, set to its union with the specified sector.
         * @throws {ArgumentError} if the specified sector is null or undefined.
         */
        Sector.prototype.union = function (sector) {
            if (!sector instanceof Sector) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Sector", "union", "missingSector"));
            }

            // Assumes normalized angles: [-180, 180], [-90, 90].
            if (this.minLatitude > sector.minLatitude)
                this.minLatitude = sector.minLatitude;
            if (this.maxLatitude < sector.maxLatitude)
                this.maxLatitude = sector.maxLatitude;
            if (this.minLongitude > sector.minLongitude)
                this.minLongitude = sector.minLongitude;
            if (this.maxLongitude < sector.maxLongitude)
                this.maxLongitude = sector.maxLongitude;

            return this;
        };

        /**
         * Computes the Cartesian coordinates of a Sector's center.
         *
         * @param globe The globe associated with the sector.
         * @param exaggeration The vertical exaggeration to apply.
         *
         * @return the Cartesian coordinates of the sector's center.
         *
         * @throws IllegalArgumentException if <code>globe</code> is null.
         */
        Sector.prototype.computeCenterPoint = function (globe, exaggeration) {
            if (globe == null) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.Level.LEVEL_SEVERE, "Sector", "computeCornerPoints", "missingGlobe"));
            }

            var lat = 0.5 * (this.minLatitude + this.maxLatitude);
            var lon = 0.5 * (this.minLongitude + this.maxLongitude);
            return globe.computePointFromPosition(lat, lon, exaggeration * globe.elevationAtLocation(lat, lon), Vec3.zero());
        };

        /**
         * Computes the Cartesian coordinates of a Sector's corners.
         *
         * @param globe The globe associated with the sector.
         * @param exaggeration The vertical exaggeration to apply.
         *
         * @return an array of four Cartesian points.
         *
         * @throws IllegalArgumentException if <code>globe</code> is null.
         */
        Sector.prototype.computeCornerPoints = function (globe, exaggeration) {
            if (!globe) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.Level.LEVEL_SEVERE, "Sector", "computeCornerPoints", "missingGlobe"));
            }
            var corners = new Array(4);

            var minLat = this.minLatitude;
            var maxLat = this.maxLatitude;
            var minLon = this.minLongitude;
            var maxLon = this.maxLongitude;

            corners[0] = globe.computePointFromPosition(minLat, minLon, exaggeration * globe.elevationAtLocation(minLat, minLon), Vec3.zero());
            corners[1] = globe.computePointFromPosition(minLat, maxLon, exaggeration * globe.elevationAtLocation(minLat, maxLon), Vec3.zero());
            corners[2] = globe.computePointFromPosition(maxLat, maxLon, exaggeration * globe.elevationAtLocation(maxLat, maxLon), Vec3.zero());
            corners[3] = globe.computePointFromPosition(maxLat, minLon, exaggeration * globe.elevationAtLocation(maxLat, minLon), Vec3.zero());
            this.cornerPoints = new Array(corners.length);
            for (var i = 0, len = corners.length; i < len; i++) {
                this.cornerPoints[i] = Vec3.fromVec3(corners[i]);
            }

            return corners;
        };

        /**
         * Returns an approximation of the distance in model coordinates between the
         * surface geometry defined by this sector and the specified model
         * coordinate point. The returned value represents the shortest distance
         * between the specified point and this sector's corner points or its center
         * point. The draw context defines the globe and the elevations that are
         * used to compute the corner points and the center point.
         *
         * @param dc The draw context defining the surface geometry.
         * @param point The model coordinate point to compute a distance to.
         *
         * @return The distance between this sector's surface geometry and the
         * specified point, in model coordinates.
         *
         * @throws IllegalArgumentException if any argument is null.
         */
        Sector.prototype.distanceTo = function (dc, point) {
            if (!dc) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.Level.LEVEL_SEVERE, "Sector", "distanceTo", "missingDc"));
            }

            if (!point) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.Level.LEVEL_SEVERE, "Sector", "distanceTo", "missingPoint"));
            }

            var corners = this.computeCornerPoints(dc.globe, dc.verticalExaggeration);
            var centerPoint = this.computeCenterPoint(dc.globe, dc.verticalExaggeration);

            // Get the distance for each of the sector's corners and its center.
            var d1 = point.distanceTo(corners[0]);
            var d2 = point.distanceTo(corners[1]);
            var d3 = point.distanceTo(corners[2]);
            var d4 = point.distanceTo(corners[3]);
            var d5 = point.distanceTo(centerPoint);

            // Find the minimum distance.
            var minDistance = d1;
            if (minDistance > d2) {
                minDistance = d2;
            }
            if (minDistance > d3) {
                minDistance = d3;
            }
            if (minDistance > d4) {
                minDistance = d4;
            }
            if (minDistance > d5) {
                minDistance = d5;
            }

            return minDistance;
        };

        return Sector;
    });
/**
 * @license
 * Copyright 2000, Silicon Graphics, Inc. All Rights Reserved.
 * Copyright 2015, Google Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice including the dates of first publication and
 * either this permission notice or a reference to http://oss.sgi.com/projects/FreeB/
 * shall be included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * SILICON GRAPHICS, INC. BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR
 * IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Original Code. The Original Code is: OpenGL Sample Implementation,
 * Version 1.2.1, released January 26, 2000, developed by Silicon Graphics,
 * Inc. The Original Code is Copyright (c) 1991-2000 Silicon Graphics, Inc.
 * Copyright in any portions created by third parties is as indicated
 * elsewhere herein. All Rights Reserved.
 */
/**
 * @author ericv@cs.stanford.edu (Eric Veach)
 * @author bckenny@google.com (Brendan Kenny)
 */

/**
 * Base namespace.
 * @const
 */
var libtess = {};

/**
 * Whether to run asserts and extra debug checks.
 * @define {boolean}
 */
libtess.DEBUG = false;

/**
 * Checks if the condition evaluates to true if libtess.DEBUG is true.
 * @param {*} condition The condition to check.
 * @param {string=} opt_message Error message in case of failure.
 * @throws {Error} Assertion failed, the condition evaluates to false.
 */
libtess.assert = function(condition, opt_message) {
  if (libtess.DEBUG && !condition) {
    throw new Error('Assertion failed' +
        (opt_message ? ': ' + opt_message : ''));
  }
};

/**
 * The maximum vertex coordinate size, 1e150. Anything larger will trigger a
 * GLU_TESS_COORD_TOO_LARGE error callback and the vertex will be clamped to
 * this value for all tessellation calculations.
 * @const {number}
 */
libtess.GLU_TESS_MAX_COORD = 1e150;
// NOTE(bckenny): value from glu.pl generator

/**
 * Normally the polygon is projected to a plane perpendicular to one of the
 * three coordinate axes before tessellating in 2d. This helps numerical
 * accuracy by forgoing a transformation step by simply dropping one coordinate
 * dimension.
 *
 * However, this can affect the placement of intersection points for non-axis-
 * aligned polygons. Setting TRUE_PROJECT to true will instead project onto a
 * plane actually perpendicular to the polygon's normal.
 *
 * NOTE(bckenny): I can find no instances on the internet in which this mode has
 * been used, but it's difficult to search for. This was a compile-time setting
 * in the original, so setting this as constant. If this is exposed in the
 * public API, remove the ignore coverage directives on
 * libtess.normal.projectPolygon and libtess.normal.normalize_.
 * @const {boolean}
 */
libtess.TRUE_PROJECT = false;

/**
 * The default tolerance for merging features, 0, meaning vertices are only
 * merged if they are exactly coincident
 * If a higher tolerance is needed, significant rewriting will need to occur.
 * See libtess.sweep.TOLERANCE_NONZERO_ as a starting place.
 * @const {number}
 */
libtess.GLU_TESS_DEFAULT_TOLERANCE = 0;

/**
 * The input contours parition the plane into regions. A winding
 * rule determines which of these regions are inside the polygon.
 *
 * For a single contour C, the winding number of a point x is simply
 * the signed number of revolutions we make around x as we travel
 * once around C (where CCW is positive). When there are several
 * contours, the individual winding numbers are summed. This
 * procedure associates a signed integer value with each point x in
 * the plane. Note that the winding number is the same for all
 * points in a single region.
 *
 * The winding rule classifies a region as "inside" if its winding
 * number belongs to the chosen category (odd, nonzero, positive,
 * negative, or absolute value of at least two). The current GLU
 * tesselator implements the "odd" rule. The "nonzero" rule is another
 * common way to define the interior. The other three rules are
 * useful for polygon CSG operations.
 * @enum {number}
 */
libtess.windingRule = {
  // NOTE(bckenny): values from enumglu.spec
  GLU_TESS_WINDING_ODD: 100130,
  GLU_TESS_WINDING_NONZERO: 100131,
  GLU_TESS_WINDING_POSITIVE: 100132,
  GLU_TESS_WINDING_NEGATIVE: 100133,
  GLU_TESS_WINDING_ABS_GEQ_TWO: 100134
};

/**
 * The type of primitive return from a "begin" callback. GL_LINE_LOOP is only
 * returned when GLU_TESS_BOUNDARY_ONLY is true. GL_TRIANGLE_STRIP and
 * GL_TRIANGLE_FAN are no longer returned since 1.1.0 (see release notes).
 * @enum {number}
 */
libtess.primitiveType = {
  GL_LINE_LOOP: 2,
  GL_TRIANGLES: 4,
  GL_TRIANGLE_STRIP: 5,
  GL_TRIANGLE_FAN: 6
};

/**
 * The types of errors provided in the error callback.
 * @enum {number}
 */
libtess.errorType = {
  // TODO(bckenny) doc types
  // NOTE(bckenny): values from enumglu.spec
  GLU_TESS_MISSING_BEGIN_POLYGON: 100151,
  GLU_TESS_MISSING_END_POLYGON: 100153,
  GLU_TESS_MISSING_BEGIN_CONTOUR: 100152,
  GLU_TESS_MISSING_END_CONTOUR: 100154,
  GLU_TESS_COORD_TOO_LARGE: 100155,
  GLU_TESS_NEED_COMBINE_CALLBACK: 100156
};

/**
 * Enum values necessary for providing settings and callbacks. See the readme
 * for details.
 * @enum {number}
 */
libtess.gluEnum = {
  // TODO(bckenny): rename so not always typing libtess.gluEnum.*?

  // NOTE(bckenny): values from enumglu.spec
  GLU_TESS_BEGIN: 100100,
  GLU_TESS_VERTEX: 100101,
  GLU_TESS_END: 100102,
  GLU_TESS_ERROR: 100103,
  GLU_TESS_EDGE_FLAG: 100104,
  GLU_TESS_COMBINE: 100105,
  GLU_TESS_BEGIN_DATA: 100106,
  GLU_TESS_VERTEX_DATA: 100107,
  GLU_TESS_END_DATA: 100108,
  GLU_TESS_ERROR_DATA: 100109,
  GLU_TESS_EDGE_FLAG_DATA: 100110,
  GLU_TESS_COMBINE_DATA: 100111,

  GLU_TESS_MESH: 100112,  //  NOTE(bckenny): from tess.c
  GLU_TESS_TOLERANCE: 100142,
  GLU_TESS_WINDING_RULE: 100140,
  GLU_TESS_BOUNDARY_ONLY: 100141,

  // TODO(bckenny): move this to libtess.errorType?
  GLU_INVALID_ENUM: 100900,
  GLU_INVALID_VALUE: 100901
};

/** @typedef {number} */
libtess.PQHandle;

/* global libtess */

/** @const */
libtess.geom = {};

/**
 * Returns whether vertex u and vertex v are equal.
 * @param {libtess.GluVertex} u
 * @param {libtess.GluVertex} v
 * @return {boolean}
 */
libtess.geom.vertEq = function(u, v) {
  return u.s === v.s && u.t === v.t;
};

/**
 * Returns whether vertex u is lexicographically less than or equal to vertex v.
 * @param {libtess.GluVertex} u
 * @param {libtess.GluVertex} v
 * @return {boolean}
 */
libtess.geom.vertLeq = function(u, v) {
  return (u.s < v.s) || (u.s === v.s && u.t <= v.t);
};

/**
 * Given three vertices u,v,w such that geom.vertLeq(u,v) && geom.vertLeq(v,w),
 * evaluates the t-coord of the edge uw at the s-coord of the vertex v.
 * Returns v.t - (uw)(v.s), ie. the signed distance from uw to v.
 * If uw is vertical (and thus passes thru v), the result is zero.
 *
 * The calculation is extremely accurate and stable, even when v
 * is very close to u or w.  In particular if we set v.t = 0 and
 * let r be the negated result (this evaluates (uw)(v.s)), then
 * r is guaranteed to satisfy MIN(u.t,w.t) <= r <= MAX(u.t,w.t).
 * @param {libtess.GluVertex} u
 * @param {libtess.GluVertex} v
 * @param {libtess.GluVertex} w
 * @return {number}
 */
libtess.geom.edgeEval = function(u, v, w) {

  var gapL = v.s - u.s;
  var gapR = w.s - v.s;

  if (gapL + gapR > 0) {
    if (gapL < gapR) {
      return (v.t - u.t) + (u.t - w.t) * (gapL / (gapL + gapR));
    } else {
      return (v.t - w.t) + (w.t - u.t) * (gapR / (gapL + gapR));
    }
  }

  // vertical line
  return 0;
};

/**
 * Returns a number whose sign matches geom.edgeEval(u,v,w) but which
 * is cheaper to evaluate.  Returns > 0, == 0 , or < 0
 * as v is above, on, or below the edge uw.
 * @param {libtess.GluVertex} u
 * @param {libtess.GluVertex} v
 * @param {libtess.GluVertex} w
 * @return {number}
 */
libtess.geom.edgeSign = function(u, v, w) {

  var gapL = v.s - u.s;
  var gapR = w.s - v.s;

  if (gapL + gapR > 0) {
    return (v.t - w.t) * gapL + (v.t - u.t) * gapR;
  }

  // vertical line
  return 0;
};

/**
 * Version of VertLeq with s and t transposed.
 * Returns whether vertex u is lexicographically less than or equal to vertex v.
 * @param {libtess.GluVertex} u
 * @param {libtess.GluVertex} v
 * @return {boolean}
 */
libtess.geom.transLeq = function(u, v) {
  return (u.t < v.t) || (u.t === v.t && u.s <= v.s);
};

/**
 * Version of geom.edgeEval with s and t transposed.
 * Given three vertices u,v,w such that geom.transLeq(u,v) &&
 * geom.transLeq(v,w), evaluates the t-coord of the edge uw at the s-coord of
 * the vertex v. Returns v.s - (uw)(v.t), ie. the signed distance from uw to v.
 * If uw is vertical (and thus passes thru v), the result is zero.
 *
 * The calculation is extremely accurate and stable, even when v
 * is very close to u or w.  In particular if we set v.s = 0 and
 * let r be the negated result (this evaluates (uw)(v.t)), then
 * r is guaranteed to satisfy MIN(u.s,w.s) <= r <= MAX(u.s,w.s).
 * @param {libtess.GluVertex} u
 * @param {libtess.GluVertex} v
 * @param {libtess.GluVertex} w
 * @return {number}
 */
libtess.geom.transEval = function(u, v, w) {

  var gapL = v.t - u.t;
  var gapR = w.t - v.t;

  if (gapL + gapR > 0) {
    if (gapL < gapR) {
      return (v.s - u.s) + (u.s - w.s) * (gapL / (gapL + gapR));
    } else {
      return (v.s - w.s) + (w.s - u.s) * (gapR / (gapL + gapR));
    }
  }

  // vertical line
  return 0;
};

/**
 * Version of geom.edgeSign with s and t transposed.
 * Returns a number whose sign matches geom.transEval(u,v,w) but which
 * is cheaper to evaluate.  Returns > 0, == 0 , or < 0
 * as v is above, on, or below the edge uw.
 * @param {libtess.GluVertex} u
 * @param {libtess.GluVertex} v
 * @param {libtess.GluVertex} w
 * @return {number}
 */
libtess.geom.transSign = function(u, v, w) {

  var gapL = v.t - u.t;
  var gapR = w.t - v.t;

  if (gapL + gapR > 0) {
    return (v.s - w.s) * gapL + (v.s - u.s) * gapR;
  }

  // vertical line
  return 0;
};

/**
 * Returns whether edge is directed from right to left.
 * @param {libtess.GluHalfEdge} e
 * @return {boolean}
 */
libtess.geom.edgeGoesLeft = function(e) {
  return libtess.geom.vertLeq(e.dst(), e.org);
};

/**
 * Returns whether edge is directed from left to right.
 * @param {libtess.GluHalfEdge} e
 * @return {boolean}
 */
libtess.geom.edgeGoesRight = function(e) {
  return libtess.geom.vertLeq(e.org, e.dst());
};

/**
 * Calculates the L1 distance between vertices u and v.
 * @param {libtess.GluVertex} u
 * @param {libtess.GluVertex} v
 * @return {number}
 */
libtess.geom.vertL1dist = function(u, v) {
  return Math.abs(u.s - v.s) + Math.abs(u.t - v.t);
};

// NOTE(bckenny): vertCCW is called nowhere in libtess and isn't part of the
// public API.
/* istanbul ignore next */
/**
 * For almost-degenerate situations, the results are not reliable.
 * Unless the floating-point arithmetic can be performed without
 * rounding errors, *any* implementation will give incorrect results
 * on some degenerate inputs, so the client must have some way to
 * handle this situation.
 * @param {!libtess.GluVertex} u
 * @param {!libtess.GluVertex} v
 * @param {!libtess.GluVertex} w
 * @return {boolean}
 */
libtess.geom.vertCCW = function(u, v, w) {
  return (u.s * (v.t - w.t) + v.s * (w.t - u.t) + w.s * (u.t - v.t)) >= 0;
};

/**
 * Given parameters a,x,b,y returns the value (b*x+a*y)/(a+b),
 * or (x+y)/2 if a==b==0. It requires that a,b >= 0, and enforces
 * this in the rare case that one argument is slightly negative.
 * The implementation is extremely stable numerically.
 * In particular it guarantees that the result r satisfies
 * MIN(x,y) <= r <= MAX(x,y), and the results are very accurate
 * even when a and b differ greatly in magnitude.
 * @private
 * @param {number} a
 * @param {number} x
 * @param {number} b
 * @param {number} y
 * @return {number}
 */
libtess.geom.interpolate_ = function(a, x, b, y) {
  // from Macro RealInterpolate:
  //(a = (a < 0) ? 0 : a, b = (b < 0) ? 0 : b, ((a <= b) ? ((b == 0) ? ((x+y) / 2) : (x + (y-x) * (a/(a+b)))) : (y + (x-y) * (b/(a+b)))))
  a = (a < 0) ? 0 : a;
  b = (b < 0) ? 0 : b;

  if (a <= b) {
    if (b === 0) {
      return (x + y) / 2;
    } else {
      return x + (y - x) * (a / (a + b));
    }
  } else {
    return y + (x - y) * (b / (a + b));
  }
};

/**
 * Given edges (o1,d1) and (o2,d2), compute their point of intersection.
 * The computed point is guaranteed to lie in the intersection of the
 * bounding rectangles defined by each edge.
 * @param {!libtess.GluVertex} o1
 * @param {!libtess.GluVertex} d1
 * @param {!libtess.GluVertex} o2
 * @param {!libtess.GluVertex} d2
 * @param {!libtess.GluVertex} v
 */
libtess.geom.edgeIntersect = function(o1, d1, o2, d2, v) {
  // This is certainly not the most efficient way to find the intersection
  // of two line segments, but it is very numerically stable.

  // Strategy: find the two middle vertices in the VertLeq ordering,
  // and interpolate the intersection s-value from these. Then repeat
  // using the TransLeq ordering to find the intersection t-value.
  var z1;
  var z2;
  var tmp;
  if (!libtess.geom.vertLeq(o1, d1)) {
    // Swap(o1, d1);
    tmp = o1;
    o1 = d1;
    d1 = tmp;
  }
  if (!libtess.geom.vertLeq(o2, d2)) {
    // Swap(o2, d2);
    tmp = o2;
    o2 = d2;
    d2 = tmp;
  }
  if (!libtess.geom.vertLeq(o1, o2)) {
    // Swap(o1, o2);
    tmp = o1;
    o1 = o2;
    o2 = tmp;
    // Swap(d1, d2);
    tmp = d1;
    d1 = d2;
    d2 = tmp;
  }

  if (!libtess.geom.vertLeq(o2, d1)) {
    // Technically, no intersection -- do our best
    v.s = (o2.s + d1.s) / 2;

  } else if (libtess.geom.vertLeq(d1, d2)) {
    // Interpolate between o2 and d1
    z1 = libtess.geom.edgeEval(o1, o2, d1);
    z2 = libtess.geom.edgeEval(o2, d1, d2);
    if (z1 + z2 < 0) { z1 = -z1; z2 = -z2; }
    v.s = libtess.geom.interpolate_(z1, o2.s, z2, d1.s);

  } else {
    // Interpolate between o2 and d2
    z1 = libtess.geom.edgeSign(o1, o2, d1);
    z2 = -libtess.geom.edgeSign(o1, d2, d1);
    if (z1 + z2 < 0) { z1 = -z1; z2 = -z2; }
    v.s = libtess.geom.interpolate_(z1, o2.s, z2, d2.s);
  }

  // Now repeat the process for t
  if (!libtess.geom.transLeq(o1, d1)) {
    // Swap(o1, d1);
    tmp = o1;
    o1 = d1;
    d1 = tmp;
  }
  if (!libtess.geom.transLeq(o2, d2)) {
    // Swap(o2, d2);
    tmp = o2;
    o2 = d2;
    d2 = tmp;
  }
  if (!libtess.geom.transLeq(o1, o2)) {
    // Swap(o1, o2);
    tmp = o1;
    o1 = o2;
    o2 = tmp;
    // Swap(d1, d2);
    tmp = d1;
    d1 = d2;
    d2 = tmp;
  }

  if (!libtess.geom.transLeq(o2, d1)) {
    // Technically, no intersection -- do our best
    v.t = (o2.t + d1.t) / 2;

  } else if (libtess.geom.transLeq(d1, d2)) {
    // Interpolate between o2 and d1
    z1 = libtess.geom.transEval(o1, o2, d1);
    z2 = libtess.geom.transEval(o2, d1, d2);
    if (z1 + z2 < 0) { z1 = -z1; z2 = -z2; }
    v.t = libtess.geom.interpolate_(z1, o2.t, z2, d1.t);

  } else {
    // Interpolate between o2 and d2
    z1 = libtess.geom.transSign(o1, o2, d1);
    z2 = -libtess.geom.transSign(o1, d2, d1);
    if (z1 + z2 < 0) { z1 = -z1; z2 = -z2; }
    v.t = libtess.geom.interpolate_(z1, o2.t, z2, d2.t);
  }
};

/* global libtess */

// TODO(bckenny): could maybe merge GluMesh and mesh.js since these are
// operations on the mesh

/** @const */
libtess.mesh = {};

/****************** Basic Edge Operations **********************/


/**
 * makeEdge creates one edge, two vertices, and a loop (face).
 * The loop consists of the two new half-edges.
 *
 * @param {libtess.GluMesh} mesh [description].
 * @return {libtess.GluHalfEdge} [description].
 */
libtess.mesh.makeEdge = function(mesh) {
  // TODO(bckenny): probably move to GluMesh, but needs Make* methods with it

  var e = libtess.mesh.makeEdgePair_(mesh.eHead);

  // complete edge with vertices and face (see mesh.makeEdgePair_)
  libtess.mesh.makeVertex_(e, mesh.vHead);
  libtess.mesh.makeVertex_(e.sym, mesh.vHead);
  libtess.mesh.makeFace_(e, mesh.fHead);

  return e;
};


/**
 * meshSplice(eOrg, eDst) is the basic operation for changing the
 * mesh connectivity and topology. It changes the mesh so that
 *  eOrg.oNext <- OLD( eDst.oNext )
 *  eDst.oNext <- OLD( eOrg.oNext )
 * where OLD(...) means the value before the meshSplice operation.
 *
 * This can have two effects on the vertex structure:
 *  - if eOrg.org != eDst.org, the two vertices are merged together
 *  - if eOrg.org == eDst.org, the origin is split into two vertices
 * In both cases, eDst.org is changed and eOrg.org is untouched.
 *
 * Similarly (and independently) for the face structure,
 *  - if eOrg.lFace == eDst.lFace, one loop is split into two
 *  - if eOrg.lFace != eDst.lFace, two distinct loops are joined into one
 * In both cases, eDst.lFace is changed and eOrg.lFace is unaffected.
 *
 * Some special cases:
 * If eDst == eOrg, the operation has no effect.
 * If eDst == eOrg.lNext, the new face will have a single edge.
 * If eDst == eOrg.lPrev(), the old face will have a single edge.
 * If eDst == eOrg.oNext, the new vertex will have a single edge.
 * If eDst == eOrg.oPrev(), the old vertex will have a single edge.
 *
 * @param {libtess.GluHalfEdge} eOrg [description].
 * @param {libtess.GluHalfEdge} eDst [description].
 */
libtess.mesh.meshSplice = function(eOrg, eDst) {
  // TODO: more descriptive name?

  var joiningLoops = false;
  var joiningVertices = false;

  if (eOrg === eDst) {
    return;
  }

  if (eDst.org !== eOrg.org) {
    // We are merging two disjoint vertices -- destroy eDst.org
    joiningVertices = true;
    libtess.mesh.killVertex_(eDst.org, eOrg.org);
  }

  if (eDst.lFace !== eOrg.lFace) {
    // We are connecting two disjoint loops -- destroy eDst.lFace
    joiningLoops = true;
    libtess.mesh.killFace_(eDst.lFace, eOrg.lFace);
  }

  // Change the edge structure
  libtess.mesh.splice_(eDst, eOrg);

  if (!joiningVertices) {
    // We split one vertex into two -- the new vertex is eDst.org.
    // Make sure the old vertex points to a valid half-edge.
    libtess.mesh.makeVertex_(eDst, eOrg.org);
    eOrg.org.anEdge = eOrg;
  }

  if (!joiningLoops) {
    // We split one loop into two -- the new loop is eDst.lFace.
    // Make sure the old face points to a valid half-edge.
    libtess.mesh.makeFace_(eDst, eOrg.lFace);
    eOrg.lFace.anEdge = eOrg;
  }
};


/**
 * deleteEdge(eDel) removes the edge eDel. There are several cases:
 * if (eDel.lFace != eDel.rFace()), we join two loops into one; the loop
 * eDel.lFace is deleted. Otherwise, we are splitting one loop into two;
 * the newly created loop will contain eDel.dst(). If the deletion of eDel
 * would create isolated vertices, those are deleted as well.
 *
 * This function could be implemented as two calls to __gl_meshSplice
 * plus a few calls to memFree, but this would allocate and delete
 * unnecessary vertices and faces.
 *
 * @param {libtess.GluHalfEdge} eDel [description].
 */
libtess.mesh.deleteEdge = function(eDel) {
  var eDelSym = eDel.sym;
  var joiningLoops = false;

  // First step: disconnect the origin vertex eDel.org.  We make all
  // changes to get a consistent mesh in this "intermediate" state.
  if (eDel.lFace !== eDel.rFace()) {
    // We are joining two loops into one -- remove the left face
    joiningLoops = true;
    libtess.mesh.killFace_(eDel.lFace, eDel.rFace());
  }

  if (eDel.oNext === eDel) {
    libtess.mesh.killVertex_(eDel.org, null);

  } else {
    // Make sure that eDel.org and eDel.rFace() point to valid half-edges
    eDel.rFace().anEdge = eDel.oPrev();
    eDel.org.anEdge = eDel.oNext;

    libtess.mesh.splice_(eDel, eDel.oPrev());

    if (!joiningLoops) {
      // We are splitting one loop into two -- create a new loop for eDel.
      libtess.mesh.makeFace_(eDel, eDel.lFace);
    }
  }

  // Claim: the mesh is now in a consistent state, except that eDel.org
  // may have been deleted.  Now we disconnect eDel.dst().
  if (eDelSym.oNext === eDelSym) {
    libtess.mesh.killVertex_(eDelSym.org, null);
    libtess.mesh.killFace_(eDelSym.lFace, null);

  } else {
    // Make sure that eDel.dst() and eDel.lFace point to valid half-edges
    eDel.lFace.anEdge = eDelSym.oPrev();
    eDelSym.org.anEdge = eDelSym.oNext;
    libtess.mesh.splice_(eDelSym, eDelSym.oPrev());
  }

  // Any isolated vertices or faces have already been freed.
  libtess.mesh.killEdge_(eDel);
};

/******************** Other Edge Operations **********************/

/* All these routines can be implemented with the basic edge
 * operations above.  They are provided for convenience and efficiency.
 */


/**
 * addEdgeVertex(eOrg) creates a new edge eNew such that
 * eNew == eOrg.lNext, and eNew.dst() is a newly created vertex.
 * eOrg and eNew will have the same left face.
 *
 * @param {libtess.GluHalfEdge} eOrg [description].
 * @return {libtess.GluHalfEdge} [description].
 */
libtess.mesh.addEdgeVertex = function(eOrg) {
  // TODO(bckenny): why is it named this?

  var eNew = libtess.mesh.makeEdgePair_(eOrg);
  var eNewSym = eNew.sym;

  // Connect the new edge appropriately
  libtess.mesh.splice_(eNew, eOrg.lNext);

  // Set the vertex and face information
  eNew.org = eOrg.dst();

  libtess.mesh.makeVertex_(eNewSym, eNew.org);

  eNew.lFace = eNewSym.lFace = eOrg.lFace;

  return eNew;
};


/**
 * splitEdge(eOrg) splits eOrg into two edges eOrg and eNew,
 * such that eNew == eOrg.lNext. The new vertex is eOrg.dst() == eNew.org.
 * eOrg and eNew will have the same left face.
 *
 * @param {libtess.GluHalfEdge} eOrg [description].
 * @return {!libtess.GluHalfEdge} [description].
 */
libtess.mesh.splitEdge = function(eOrg) {
  var tempHalfEdge = libtess.mesh.addEdgeVertex(eOrg);
  var eNew = tempHalfEdge.sym;

  // Disconnect eOrg from eOrg.dst() and connect it to eNew.org
  libtess.mesh.splice_(eOrg.sym, eOrg.sym.oPrev());
  libtess.mesh.splice_(eOrg.sym, eNew);

  // Set the vertex and face information
  eOrg.sym.org = eNew.org; // NOTE(bckenny): assignment to dst
  eNew.dst().anEdge = eNew.sym;  // may have pointed to eOrg.sym
  eNew.sym.lFace = eOrg.rFace(); // NOTE(bckenny): assignment to rFace
  eNew.winding = eOrg.winding;  // copy old winding information
  eNew.sym.winding = eOrg.sym.winding;

  return eNew;
};


/**
 * connect(eOrg, eDst) creates a new edge from eOrg.dst()
 * to eDst.org, and returns the corresponding half-edge eNew.
 * If eOrg.lFace == eDst.lFace, this splits one loop into two,
 * and the newly created loop is eNew.lFace. Otherwise, two disjoint
 * loops are merged into one, and the loop eDst.lFace is destroyed.
 *
 * If (eOrg == eDst), the new face will have only two edges.
 * If (eOrg.lNext == eDst), the old face is reduced to a single edge.
 * If (eOrg.lNext.lNext == eDst), the old face is reduced to two edges.
 *
 * @param {libtess.GluHalfEdge} eOrg [description].
 * @param {libtess.GluHalfEdge} eDst [description].
 * @return {!libtess.GluHalfEdge} [description].
 */
libtess.mesh.connect = function(eOrg, eDst) {
  var joiningLoops = false;
  var eNew = libtess.mesh.makeEdgePair_(eOrg);
  var eNewSym = eNew.sym;

  if (eDst.lFace !== eOrg.lFace) {
    // We are connecting two disjoint loops -- destroy eDst.lFace
    joiningLoops = true;
    libtess.mesh.killFace_(eDst.lFace, eOrg.lFace);
  }

  // Connect the new edge appropriately
  libtess.mesh.splice_(eNew, eOrg.lNext);
  libtess.mesh.splice_(eNewSym, eDst);

  // Set the vertex and face information
  eNew.org = eOrg.dst();
  eNewSym.org = eDst.org;
  eNew.lFace = eNewSym.lFace = eOrg.lFace;

  // Make sure the old face points to a valid half-edge
  eOrg.lFace.anEdge = eNewSym;

  if (!joiningLoops) {
    // We split one loop into two -- the new loop is eNew.lFace
    libtess.mesh.makeFace_(eNew, eOrg.lFace);
  }
  return eNew;
};

/******************** Other Operations **********************/


/**
 * zapFace(fZap) destroys a face and removes it from the
 * global face list. All edges of fZap will have a null pointer as their
 * left face. Any edges which also have a null pointer as their right face
 * are deleted entirely (along with any isolated vertices this produces).
 * An entire mesh can be deleted by zapping its faces, one at a time,
 * in any order. Zapped faces cannot be used in further mesh operations!
 *
 * @param {libtess.GluFace} fZap [description].
 */
libtess.mesh.zapFace = function(fZap) {
  var eStart = fZap.anEdge;

  // walk around face, deleting edges whose right face is also NULL
  var eNext = eStart.lNext;
  var e;
  do {
    e = eNext;
    eNext = e.lNext;

    e.lFace = null;
    if (e.rFace() === null) {
      // delete the edge -- see mesh.deleteEdge above
      if (e.oNext === e) {
        libtess.mesh.killVertex_(e.org, null);

      } else {
        // Make sure that e.org points to a valid half-edge
        e.org.anEdge = e.oNext;
        libtess.mesh.splice_(e, e.oPrev());
      }

      var eSym = e.sym;

      if (eSym.oNext === eSym) {
        libtess.mesh.killVertex_(eSym.org, null);

      } else {
        // Make sure that eSym.org points to a valid half-edge
        eSym.org.anEdge = eSym.oNext;
        libtess.mesh.splice_(eSym, eSym.oPrev());
      }
      libtess.mesh.killEdge_(e);
    }
  } while (e !== eStart);

  // delete from circular doubly-linked list
  var fPrev = fZap.prev;
  var fNext = fZap.next;
  fNext.prev = fPrev;
  fPrev.next = fNext;

  // TODO(bckenny): memFree( fZap );
  // TODO(bckenny): probably null at callsite
};

// TODO(bckenny): meshUnion isn't called within libtess and isn't part of the
// public API. Could be useful if more mesh manipulation functions are exposed.
/* istanbul ignore next */
/**
 * meshUnion() forms the union of all structures in
 * both meshes, and returns the new mesh (the old meshes are destroyed).
 *
 * @param {!libtess.GluMesh} mesh1
 * @param {!libtess.GluMesh} mesh2
 * @return {!libtess.GluMesh}
 */
libtess.mesh.meshUnion = function(mesh1, mesh2) {
  // TODO(bceknny): probably move to GluMesh method
  var f1 = mesh1.fHead;
  var v1 = mesh1.vHead;
  var e1 = mesh1.eHead;

  var f2 = mesh2.fHead;
  var v2 = mesh2.vHead;
  var e2 = mesh2.eHead;

  // Add the faces, vertices, and edges of mesh2 to those of mesh1
  if (f2.next !== f2) {
    f1.prev.next = f2.next;
    f2.next.prev = f1.prev;
    f2.prev.next = f1;
    f1.prev = f2.prev;
  }

  if (v2.next !== v2) {
    v1.prev.next = v2.next;
    v2.next.prev = v1.prev;
    v2.prev.next = v1;
    v1.prev = v2.prev;
  }

  if (e2.next !== e2) {
    e1.sym.next.sym.next = e2.next;
    e2.next.sym.next = e1.sym.next;
    e2.sym.next.sym.next = e1;
    e1.sym.next = e2.sym.next;
  }

  // TODO(bckenny): memFree(mesh2);
  // TODO(bckenny): If function is kept, remove mesh2's data to enforce.
  return mesh1;
};


/**
 * deleteMesh(mesh) will free all storage for any valid mesh.
 * @param {libtess.GluMesh} mesh [description].
 */
libtess.mesh.deleteMesh = function(mesh) {
  // TODO(bckenny): unnecessary, I think.
  // TODO(bckenny): might want to explicitly null at callsite
  // lots of memFrees. see also DELETE_BY_ZAPPING
};

/************************ Utility Routines ************************/


/**
 * Creates a new pair of half-edges which form their own loop.
 * No vertex or face structures are allocated, but these must be assigned
 * before the current edge operation is completed.
 *
 * TODO(bckenny): warning about eNext strictly being first of pair? (see code)
 *
 * @private
 * @param {libtess.GluHalfEdge} eNext [description].
 * @return {libtess.GluHalfEdge} [description].
 */
libtess.mesh.makeEdgePair_ = function(eNext) {
  var e = new libtess.GluHalfEdge();
  var eSym = new libtess.GluHalfEdge();

  // TODO(bckenny): how do we ensure this? see above comment in jsdoc
  // Make sure eNext points to the first edge of the edge pair
  // if (eNext->Sym < eNext ) { eNext = eNext->Sym; }

  // NOTE(bckenny): check this for bugs in current implementation!

  // Insert in circular doubly-linked list before eNext.
  // Note that the prev pointer is stored in sym.next.
  var ePrev = eNext.sym.next;
  eSym.next = ePrev;
  ePrev.sym.next = e;
  e.next = eNext;
  eNext.sym.next = eSym;

  e.sym = eSym;
  e.oNext = e;
  e.lNext = eSym;

  eSym.sym = e;
  eSym.oNext = eSym;
  eSym.lNext = e;

  return e;
};


/**
 * splice_ is best described by the Guibas/Stolfi paper or the
 * CS348a notes. Basically, it modifies the mesh so that
 * a.oNext and b.oNext are exchanged. This can have various effects
 * depending on whether a and b belong to different face or vertex rings.
 * For more explanation see mesh.meshSplice below.
 *
 * @private
 * @param {libtess.GluHalfEdge} a [description].
 * @param {libtess.GluHalfEdge} b [description].
 */
libtess.mesh.splice_ = function(a, b) {
  var aONext = a.oNext;
  var bONext = b.oNext;

  aONext.sym.lNext = b;
  bONext.sym.lNext = a;
  a.oNext = bONext;
  b.oNext = aONext;
};


/**
 * makeVertex_(eOrig, vNext) attaches a new vertex and makes it the
 * origin of all edges in the vertex loop to which eOrig belongs. "vNext" gives
 * a place to insert the new vertex in the global vertex list.  We insert
 * the new vertex *before* vNext so that algorithms which walk the vertex
 * list will not see the newly created vertices.
 *
 * NOTE: unlike original, acutally allocates new vertex.
 *
 * @private
 * @param {libtess.GluHalfEdge} eOrig [description].
 * @param {libtess.GluVertex} vNext [description].
 */
libtess.mesh.makeVertex_ = function(eOrig, vNext) {
  // insert in circular doubly-linked list before vNext
  var vPrev = vNext.prev;
  var vNew = new libtess.GluVertex(vNext, vPrev);
  vPrev.next = vNew;
  vNext.prev = vNew;

  vNew.anEdge = eOrig;
  // leave coords, s, t undefined
  // TODO(bckenny): does above line mean 0 specifically, or does it matter?

  // fix other edges on this vertex loop
  var e = eOrig;
  do {
    e.org = vNew;
    e = e.oNext;
  } while (e !== eOrig);
};


/**
 * makeFace_(eOrig, fNext) attaches a new face and makes it the left
 * face of all edges in the face loop to which eOrig belongs. "fNext" gives
 * a place to insert the new face in the global face list.  We insert
 * the new face *before* fNext so that algorithms which walk the face
 * list will not see the newly created faces.
 *
 * NOTE: unlike original, acutally allocates new face.
 *
 * @private
 * @param {libtess.GluHalfEdge} eOrig [description].
 * @param {libtess.GluFace} fNext [description].
 */
libtess.mesh.makeFace_ = function(eOrig, fNext) {
  // insert in circular doubly-linked list before fNext
  var fPrev = fNext.prev;
  var fNew = new libtess.GluFace(fNext, fPrev);
  fPrev.next = fNew;
  fNext.prev = fNew;

  fNew.anEdge = eOrig;

  // The new face is marked "inside" if the old one was.  This is a
  // convenience for the common case where a face has been split in two.
  fNew.inside = fNext.inside;

  // fix other edges on this face loop
  var e = eOrig;
  do {
    e.lFace = fNew;
    e = e.lNext;
  } while (e !== eOrig);
};


/**
 * killEdge_ destroys an edge (the half-edges eDel and eDel.sym),
 * and removes from the global edge list.
 *
 * @private
 * @param {libtess.GluHalfEdge} eDel [description].
 */
libtess.mesh.killEdge_ = function(eDel) {
  // TODO(bckenny): in this case, no need to worry(?), but check when checking mesh.makeEdgePair_
  // Half-edges are allocated in pairs, see EdgePair above
  // if (eDel->Sym < eDel ) { eDel = eDel->Sym; }

  // delete from circular doubly-linked list
  var eNext = eDel.next;
  var ePrev = eDel.sym.next;
  eNext.sym.next = ePrev;
  ePrev.sym.next = eNext;

  // TODO(bckenny): memFree( eDel ); (which also frees eDel.sym)
  // TODO(bckenny): need to null at callsites?
};


/**
 * killVertex_ destroys a vertex and removes it from the global
 * vertex list. It updates the vertex loop to point to a given new vertex.
 *
 * @private
 * @param {libtess.GluVertex} vDel [description].
 * @param {libtess.GluVertex} newOrg [description].
 */
libtess.mesh.killVertex_ = function(vDel, newOrg) {
  var eStart = vDel.anEdge;

  // change the origin of all affected edges
  var e = eStart;
  do {
    e.org = newOrg;
    e = e.oNext;
  } while (e !== eStart);

  // delete from circular doubly-linked list
  var vPrev = vDel.prev;
  var vNext = vDel.next;
  vNext.prev = vPrev;
  vPrev.next = vNext;

  // TODO(bckenny): memFree( vDel );
  // TODO(bckenny): need to null at callsites?
};


/**
 * killFace_ destroys a face and removes it from the global face
 * list. It updates the face loop to point to a given new face.
 *
 * @private
 * @param {libtess.GluFace} fDel [description].
 * @param {libtess.GluFace} newLFace [description].
 */
libtess.mesh.killFace_ = function(fDel, newLFace) {
  var eStart = fDel.anEdge;

  // change the left face of all affected edges
  var e = eStart;
  do {
    e.lFace = newLFace;
    e = e.lNext;
  } while (e !== eStart);

  // delete from circular doubly-linked list
  var fPrev = fDel.prev;
  var fNext = fDel.next;
  fNext.prev = fPrev;
  fPrev.next = fNext;

  // TODO(bckenny): memFree( fDel );
  // TODO(bckenny): need to null at callsites?
};

/* global libtess */

/** @const */
libtess.normal = {};

// TODO(bckenny): Integrate SLANTED_SWEEP somehow?
/* The "feature merging" is not intended to be complete. There are
 * special cases where edges are nearly parallel to the sweep line
 * which are not implemented. The algorithm should still behave
 * robustly (ie. produce a reasonable tesselation) in the presence
 * of such edges, however it may miss features which could have been
 * merged. We could minimize this effect by choosing the sweep line
 * direction to be something unusual (ie. not parallel to one of the
 * coordinate axes).
 * #if defined(SLANTED_SWEEP)
 * #define S_UNIT_X  0.50941539564955385 // Pre-normalized
 * #define S_UNIT_Y  0.86052074622010633
 * #endif
 */

/**
 * X coordinate of local basis for polygon projection.
 * @private {number}
 * @const
 */
libtess.normal.S_UNIT_X_ = 1.0;

/**
 * Y coordinate of local basis for polygon projection.
 * @private {number}
 * @const
 */
libtess.normal.S_UNIT_Y_ = 0.0;

/**
 * Determines a polygon normal and projects vertices onto the plane of the
 * polygon. A pre-computed normal for the data may be provided, or set to the
 * zero vector if one should be computed from it.
 * @param {!libtess.GluTesselator} tess
 * @param {number} normalX
 * @param {number} normalY
 * @param {number} normalZ
 */
libtess.normal.projectPolygon = function(tess, normalX, normalY, normalZ) {
  var computedNormal = false;

  var norm = [
    normalX,
    normalY,
    normalZ
  ];
  if (normalX === 0 && normalY === 0 && normalZ === 0) {
    libtess.normal.computeNormal_(tess, norm);
    computedNormal = true;
  }

  var i = libtess.normal.longAxis_(norm);
  var vHead = tess.mesh.vHead;
  var v;

  // NOTE(bckenny): This branch is never taken. See comment on
  // libtess.TRUE_PROJECT.
  /* istanbul ignore if */
  if (libtess.TRUE_PROJECT) {
    // Choose the initial sUnit vector to be approximately perpendicular
    // to the normal.
    libtess.normal.normalize_(norm);

    var sUnit = [0, 0, 0];
    var tUnit = [0, 0, 0];

    sUnit[i] = 0;
    sUnit[(i + 1) % 3] = libtess.normal.S_UNIT_X_;
    sUnit[(i + 2) % 3] = libtess.normal.S_UNIT_Y_;

    // Now make it exactly perpendicular
    var w = libtess.normal.dot_(sUnit, norm);
    sUnit[0] -= w * norm[0];
    sUnit[1] -= w * norm[1];
    sUnit[2] -= w * norm[2];
    libtess.normal.normalize_(sUnit);

    // Choose tUnit so that (sUnit,tUnit,norm) form a right-handed frame
    tUnit[0] = norm[1] * sUnit[2] - norm[2] * sUnit[1];
    tUnit[1] = norm[2] * sUnit[0] - norm[0] * sUnit[2];
    tUnit[2] = norm[0] * sUnit[1] - norm[1] * sUnit[0];
    libtess.normal.normalize_(tUnit);

    // Project the vertices onto the sweep plane
    for (v = vHead.next; v !== vHead; v = v.next) {
      v.s = libtess.normal.dot_(v.coords, sUnit);
      v.t = libtess.normal.dot_(v.coords, tUnit);
    }

  } else {
    // Project perpendicular to a coordinate axis -- better numerically
    var sAxis = (i + 1) % 3;
    var tAxis = (i + 2) % 3;
    var tNegate = norm[i] > 0 ? 1 : -1;

    // Project the vertices onto the sweep plane
    for (v = vHead.next; v !== vHead; v = v.next) {
      v.s = v.coords[sAxis];
      v.t = tNegate * v.coords[tAxis];
    }
  }

  if (computedNormal) {
    libtess.normal.checkOrientation_(tess);
  }
};

// NOTE(bckenny): libtess.normal.dot_ is no longer called in code without
// libtess.TRUE_PROJECT defined.
/* istanbul ignore next */
/**
 * Computes the dot product of vectors u and v.
 * @private
 * @param {!Array<number>} u
 * @param {!Array<number>} v
 * @return {number}
 */
libtess.normal.dot_ = function(u, v) {
  return u[0] * v[0] + u[1] * v[1] + u[2] * v[2];
};

// NOTE(bckenny): only called from within libtess.normal.projectPolygon's
// TRUE_PROJECT branch, so ignoring for code coverage.
/* istanbul ignore next */
/**
 * Normalize vector v.
 * @private
 * @param {!Array<number>} v
 */
libtess.normal.normalize_ = function(v) {
  var len = v[0] * v[0] + v[1] * v[1] + v[2] * v[2];

  len = Math.sqrt(len);
  v[0] /= len;
  v[1] /= len;
  v[2] /= len;
};

/**
 * Returns the index of the longest component of vector v.
 * @private
 * @param {!Array<number>} v
 * @return {number}
 */
libtess.normal.longAxis_ = function(v) {
  var i = 0;

  if (Math.abs(v[1]) > Math.abs(v[0])) {
    i = 1;
  }
  if (Math.abs(v[2]) > Math.abs(v[i])) {
    i = 2;
  }

  return i;
};

/**
 * Compute an approximate normal of the polygon from the vertices themselves.
 * Result returned in norm.
 * @private
 * @param {!libtess.GluTesselator} tess
 * @param {!Array<number>} norm
 */
libtess.normal.computeNormal_ = function(tess, norm) {
  var maxVal = [
    -2 * libtess.GLU_TESS_MAX_COORD,
    -2 * libtess.GLU_TESS_MAX_COORD,
    -2 * libtess.GLU_TESS_MAX_COORD
  ];
  var minVal = [
    2 * libtess.GLU_TESS_MAX_COORD,
    2 * libtess.GLU_TESS_MAX_COORD,
    2 * libtess.GLU_TESS_MAX_COORD
  ];
  var maxVert = [];
  var minVert = [];

  var v;
  var vHead = tess.mesh.vHead;
  for (v = vHead.next; v !== vHead; v = v.next) {
    for (var i = 0; i < 3; ++i) {
      var c = v.coords[i];
      if (c < minVal[i]) { minVal[i] = c; minVert[i] = v; }
      if (c > maxVal[i]) { maxVal[i] = c; maxVert[i] = v; }
    }
  }

  // Find two vertices separated by at least 1/sqrt(3) of the maximum
  // distance between any two vertices
  var index = 0;
  if (maxVal[1] - minVal[1] > maxVal[0] - minVal[0]) { index = 1; }
  if (maxVal[2] - minVal[2] > maxVal[index] - minVal[index]) { index = 2; }
  if (minVal[index] >= maxVal[index]) {
    // All vertices are the same -- normal doesn't matter
    norm[0] = 0; norm[1] = 0; norm[2] = 1;
    return;
  }

  // Look for a third vertex which forms the triangle with maximum area
  // (Length of normal == twice the triangle area)
  var maxLen2 = 0;
  var v1 = minVert[index];
  var v2 = maxVert[index];
  var tNorm = [0, 0, 0];
  var d1 = [
    v1.coords[0] - v2.coords[0],
    v1.coords[1] - v2.coords[1],
    v1.coords[2] - v2.coords[2]
  ];
  var d2 = [0, 0, 0];
  for (v = vHead.next; v !== vHead; v = v.next) {
    d2[0] = v.coords[0] - v2.coords[0];
    d2[1] = v.coords[1] - v2.coords[1];
    d2[2] = v.coords[2] - v2.coords[2];
    tNorm[0] = d1[1] * d2[2] - d1[2] * d2[1];
    tNorm[1] = d1[2] * d2[0] - d1[0] * d2[2];
    tNorm[2] = d1[0] * d2[1] - d1[1] * d2[0];
    var tLen2 = tNorm[0] * tNorm[0] + tNorm[1] * tNorm[1] + tNorm[2] * tNorm[2];
    if (tLen2 > maxLen2) {
      maxLen2 = tLen2;
      norm[0] = tNorm[0];
      norm[1] = tNorm[1];
      norm[2] = tNorm[2];
    }
  }

  if (maxLen2 <= 0) {
    // All points lie on a single line -- any decent normal will do
    norm[0] = norm[1] = norm[2] = 0;
    norm[libtess.normal.longAxis_(d1)] = 1;
  }
};

/**
 * Check that the sum of the signed area of all projected contours is
 * non-negative. If not, negate the t-coordinates to reverse the orientation and
 * make it so.
 * @private
 * @param {!libtess.GluTesselator} tess
 */
libtess.normal.checkOrientation_ = function(tess) {
  var area = 0;
  var fHead = tess.mesh.fHead;
  for (var f = fHead.next; f !== fHead; f = f.next) {
    var e = f.anEdge;
    if (e.winding <= 0) { continue; }
    do {
      area += (e.org.s - e.dst().s) * (e.org.t + e.dst().t);
      e = e.lNext;
    } while (e !== f.anEdge);
  }

  if (area < 0) {
    // Reverse the orientation by flipping all the t-coordinates
    var vHead = tess.mesh.vHead;
    for (var v = vHead.next; v !== vHead; v = v.next) {
      v.t = -v.t;
    }
  }
};

/* global libtess */

/** @const */
libtess.render = {};

/**
 * Takes a mesh, breaks it into separate triangles, and renders them. The
 * rendering output is provided as callbacks (see the API). Set flagEdges to
 * true to get edgeFlag callbacks (tess.flagBoundary in original libtess).
 * @param {!libtess.GluTesselator} tess
 * @param {!libtess.GluMesh} mesh
 * @param {boolean} flagEdges
 */
libtess.render.renderMesh = function(tess, mesh, flagEdges) {
  var beginCallbackCalled = false;

  // TODO(bckenny): edgeState needs to be boolean, but !== on first call
  // force edge state output for first vertex
  var edgeState = -1;

  // We examine all faces in an arbitrary order. Whenever we find
  // an inside triangle f, we render f.
  // NOTE(bckenny): go backwards through face list to match original libtess
  // triangle order
  for (var f = mesh.fHead.prev; f !== mesh.fHead; f = f.prev) {
    if (f.inside) {
      // We're going to emit a triangle, so call begin callback once
      if (!beginCallbackCalled) {
        tess.callBeginCallback(libtess.primitiveType.GL_TRIANGLES);
        beginCallbackCalled = true;
      }

      // check that face has only three edges
      var e = f.anEdge;
      // Loop once for each edge (there will always be 3 edges)
      do {
        if (flagEdges) {
          // Set the "edge state" to true just before we output the
          // first vertex of each edge on the polygon boundary.
          var newState = !e.rFace().inside ? 1 : 0; // TODO(bckenny): total hack to get edgeState working. fix me.
          if (edgeState !== newState) {
            edgeState = newState;
            // TODO(bckenny): edgeState should be boolean now
            tess.callEdgeFlagCallback(!!edgeState);
          }
        }

        // emit vertex
        tess.callVertexCallback(e.org.data);

        e = e.lNext;
      } while (e !== f.anEdge);
    }
  }

  // only call end callback if begin was called
  if (beginCallbackCalled) {
    tess.callEndCallback();
  }
};

/**
 * Takes a mesh, and outputs one contour for each face marked "inside". The
 * rendering output is provided as callbacks (see the API).
 * @param {!libtess.GluTesselator} tess
 * @param {!libtess.GluMesh} mesh
 */
libtess.render.renderBoundary = function(tess, mesh) {
  for (var f = mesh.fHead.next; f !== mesh.fHead; f = f.next) {
    if (f.inside) {
      tess.callBeginCallback(libtess.primitiveType.GL_LINE_LOOP);

      var e = f.anEdge;
      do {
        tess.callVertexCallback(e.org.data);
        e = e.lNext;
      } while (e !== f.anEdge);

      tess.callEndCallback();
    }
  }
};

/* global libtess */

// TODO(bckenny): a number of these never return null (as opposed to original) and should be typed appropriately

/*
 * Invariants for the Edge Dictionary.
 * - each pair of adjacent edges e2=succ(e1) satisfies edgeLeq_(e1,e2)
 *   at any valid location of the sweep event
 * - if edgeLeq_(e2,e1) as well (at any valid sweep event), then e1 and e2
 *   share a common endpoint
 * - for each e, e.dst() has been processed, but not e.org
 * - each edge e satisfies vertLeq(e.dst(),event) && vertLeq(event,e.org)
 *   where "event" is the current sweep line event.
 * - no edge e has zero length
 *
 * Invariants for the Mesh (the processed portion).
 * - the portion of the mesh left of the sweep line is a planar graph,
 *   ie. there is *some* way to embed it in the plane
 * - no processed edge has zero length
 * - no two processed vertices have identical coordinates
 * - each "inside" region is monotone, ie. can be broken into two chains
 *   of monotonically increasing vertices according to VertLeq(v1,v2)
 *   - a non-invariant: these chains may intersect (very slightly)
 *
 * Invariants for the Sweep.
 * - if none of the edges incident to the event vertex have an activeRegion
 *   (ie. none of these edges are in the edge dictionary), then the vertex
 *   has only right-going edges.
 * - if an edge is marked "fixUpperEdge" (it is a temporary edge introduced
 *   by ConnectRightVertex), then it is the only right-going edge from
 *   its associated vertex.  (This says that these edges exist only
 *   when it is necessary.)
 */

/** @const */
libtess.sweep = {};


/**
 * Make the sentinel coordinates big enough that they will never be
 * merged with real input features.  (Even with the largest possible
 * input contour and the maximum tolerance of 1.0, no merging will be
 * done with coordinates larger than 3 * libtess.GLU_TESS_MAX_COORD).
 * @private
 * @const
 * @type {number}
 */
libtess.sweep.SENTINEL_COORD_ = 4 * libtess.GLU_TESS_MAX_COORD;


/**
 * Because vertices at exactly the same location are merged together
 * before we process the sweep event, some degenerate cases can't occur.
 * However if someone eventually makes the modifications required to
 * merge features which are close together, the cases below marked
 * TOLERANCE_NONZERO will be useful.  They were debugged before the
 * code to merge identical vertices in the main loop was added.
 * @private
 * @const
 * @type {boolean}
 */
libtess.sweep.TOLERANCE_NONZERO_ = false;


/**
 * computeInterior(tess) computes the planar arrangement specified
 * by the given contours, and further subdivides this arrangement
 * into regions. Each region is marked "inside" if it belongs
 * to the polygon, according to the rule given by tess.windingRule.
 * Each interior region is guaranteed be monotone.
 *
 * @param {libtess.GluTesselator} tess [description].
 */
libtess.sweep.computeInterior = function(tess) {
  tess.fatalError = false;

  // Each vertex defines an event for our sweep line. Start by inserting
  // all the vertices in a priority queue. Events are processed in
  // lexicographic order, ie.
  // e1 < e2  iff  e1.x < e2.x || (e1.x == e2.x && e1.y < e2.y)
  libtess.sweep.removeDegenerateEdges_(tess);
  libtess.sweep.initPriorityQ_(tess);
  libtess.sweep.initEdgeDict_(tess);

  var v;
  while ((v = tess.pq.extractMin()) !== null) {
    for (;;) {
      var vNext = tess.pq.minimum();
      if (vNext === null || !libtess.geom.vertEq(vNext, v)) {
        break;
      }

      /* Merge together all vertices at exactly the same location.
       * This is more efficient than processing them one at a time,
       * simplifies the code (see connectLeftDegenerate), and is also
       * important for correct handling of certain degenerate cases.
       * For example, suppose there are two identical edges A and B
       * that belong to different contours (so without this code they would
       * be processed by separate sweep events).  Suppose another edge C
       * crosses A and B from above.  When A is processed, we split it
       * at its intersection point with C.  However this also splits C,
       * so when we insert B we may compute a slightly different
       * intersection point.  This might leave two edges with a small
       * gap between them.  This kind of error is especially obvious
       * when using boundary extraction (GLU_TESS_BOUNDARY_ONLY).
       */
      vNext = tess.pq.extractMin();
      libtess.sweep.spliceMergeVertices_(tess, v.anEdge, vNext.anEdge);
    }
    libtess.sweep.sweepEvent_(tess, v);
  }

  // TODO(bckenny): what does the next comment mean? can we eliminate event except when debugging?
  // Set tess.event for debugging purposes
  var minRegion = tess.dict.getMin().getKey();
  tess.event = minRegion.eUp.org;
  libtess.sweep.doneEdgeDict_(tess);
  libtess.sweep.donePriorityQ_(tess);

  libtess.sweep.removeDegenerateFaces_(tess.mesh);
  tess.mesh.checkMesh();
};


/**
 * When we merge two edges into one, we need to compute the combined
 * winding of the new edge.
 * @private
 * @param {libtess.GluHalfEdge} eDst [description].
 * @param {libtess.GluHalfEdge} eSrc [description].
 */
libtess.sweep.addWinding_ = function(eDst, eSrc) {
  // NOTE(bckenny): from AddWinding macro
  eDst.winding += eSrc.winding;
  eDst.sym.winding += eSrc.sym.winding;
};


/**
 * Both edges must be directed from right to left (this is the canonical
 * direction for the upper edge of each region).
 *
 * The strategy is to evaluate a "t" value for each edge at the
 * current sweep line position, given by tess.event.  The calculations
 * are designed to be very stable, but of course they are not perfect.
 *
 * Special case: if both edge destinations are at the sweep event,
 * we sort the edges by slope (they would otherwise compare equally).
 *
 * @private
 * @param {!libtess.GluTesselator} tess
 * @param {!libtess.ActiveRegion} reg1
 * @param {!libtess.ActiveRegion} reg2
 * @return {boolean}
 */
libtess.sweep.edgeLeq_ = function(tess, reg1, reg2) {
  var event = tess.event;
  var e1 = reg1.eUp;
  var e2 = reg2.eUp;

  if (e1.dst() === event) {
    if (e2.dst() === event) {
      // Two edges right of the sweep line which meet at the sweep event.
      // Sort them by slope.
      if (libtess.geom.vertLeq(e1.org, e2.org)) {
        return libtess.geom.edgeSign(e2.dst(), e1.org, e2.org) <= 0;
      }

      return libtess.geom.edgeSign(e1.dst(), e2.org, e1.org) >= 0;
    }

    return libtess.geom.edgeSign(e2.dst(), event, e2.org) <= 0;
  }

  if (e2.dst() === event) {
    return libtess.geom.edgeSign(e1.dst(), event, e1.org) >= 0;
  }

  // General case - compute signed distance *from* e1, e2 to event
  var t1 = libtess.geom.edgeEval(e1.dst(), event, e1.org);
  var t2 = libtess.geom.edgeEval(e2.dst(), event, e2.org);
  return (t1 >= t2);
};


/**
 * [deleteRegion_ description]
 * @private
 * @param {libtess.GluTesselator} tess [description].
 * @param {libtess.ActiveRegion} reg [description].
 */
libtess.sweep.deleteRegion_ = function(tess, reg) {
  if (reg.fixUpperEdge) {
    // It was created with zero winding number, so it better be
    // deleted with zero winding number (ie. it better not get merged
    // with a real edge).
  }

  reg.eUp.activeRegion = null;

  tess.dict.deleteNode(reg.nodeUp);
  reg.nodeUp = null;

  // memFree( reg ); TODO(bckenny)
  // TODO(bckenny): may need to null at callsite
};


/**
 * Replace an upper edge which needs fixing (see connectRightVertex).
 * @private
 * @param {libtess.ActiveRegion} reg [description].
 * @param {libtess.GluHalfEdge} newEdge [description].
 */
libtess.sweep.fixUpperEdge_ = function(reg, newEdge) {
  libtess.mesh.deleteEdge(reg.eUp);

  reg.fixUpperEdge = false;
  reg.eUp = newEdge;
  newEdge.activeRegion = reg;
};


/**
 * Find the region above the uppermost edge with the same origin.
 * @private
 * @param {libtess.ActiveRegion} reg [description].
 * @return {libtess.ActiveRegion} [description].
 */
libtess.sweep.topLeftRegion_ = function(reg) {
  var org = reg.eUp.org;

  // Find the region above the uppermost edge with the same origin
  do {
    reg = reg.regionAbove();
  } while (reg.eUp.org === org);

  // If the edge above was a temporary edge introduced by connectRightVertex,
  // now is the time to fix it.
  if (reg.fixUpperEdge) {
    var e = libtess.mesh.connect(reg.regionBelow().eUp.sym, reg.eUp.lNext);
    libtess.sweep.fixUpperEdge_(reg, e);
    reg = reg.regionAbove();
  }

  return reg;
};


/**
 * Find the region above the uppermost edge with the same destination.
 * @private
 * @param {libtess.ActiveRegion} reg [description].
 * @return {libtess.ActiveRegion} [description].
 */
libtess.sweep.topRightRegion_ = function(reg) {
  var dst = reg.eUp.dst();

  do {
    reg = reg.regionAbove();
  } while (reg.eUp.dst() === dst);

  return reg;
};


/**
 * Add a new active region to the sweep line, *somewhere* below "regAbove"
 * (according to where the new edge belongs in the sweep-line dictionary).
 * The upper edge of the new region will be "eNewUp".
 * Winding number and "inside" flag are not updated.
 *
 * @private
 * @param {libtess.GluTesselator} tess [description].
 * @param {libtess.ActiveRegion} regAbove [description].
 * @param {libtess.GluHalfEdge} eNewUp [description].
 * @return {libtess.ActiveRegion} regNew.
 */
libtess.sweep.addRegionBelow_ = function(tess, regAbove, eNewUp) {
  var regNew = new libtess.ActiveRegion();

  regNew.eUp = eNewUp;
  regNew.nodeUp = tess.dict.insertBefore(regAbove.nodeUp, regNew);
  eNewUp.activeRegion = regNew;

  return regNew;
};


/**
 * [isWindingInside_ description]
 * @private
 * @param {libtess.GluTesselator} tess [description].
 * @param {number} n int.
 * @return {boolean} [description].
 */
libtess.sweep.isWindingInside_ = function(tess, n) {
  switch (tess.windingRule) {
    case libtess.windingRule.GLU_TESS_WINDING_ODD:
      return ((n & 1) !== 0);
    case libtess.windingRule.GLU_TESS_WINDING_NONZERO:
      return (n !== 0);
    case libtess.windingRule.GLU_TESS_WINDING_POSITIVE:
      return (n > 0);
    case libtess.windingRule.GLU_TESS_WINDING_NEGATIVE:
      return (n < 0);
    case libtess.windingRule.GLU_TESS_WINDING_ABS_GEQ_TWO:
      return (n >= 2) || (n <= -2);
  }

  // TODO(bckenny): not reached
  return false;
};


/**
 * [computeWinding_ description]
 * @private
 * @param {libtess.GluTesselator} tess [description].
 * @param {libtess.ActiveRegion} reg [description].
 */
libtess.sweep.computeWinding_ = function(tess, reg) {
  reg.windingNumber = reg.regionAbove().windingNumber + reg.eUp.winding;
  reg.inside = libtess.sweep.isWindingInside_(tess, reg.windingNumber);
};


/**
 * Delete a region from the sweep line. This happens when the upper
 * and lower chains of a region meet (at a vertex on the sweep line).
 * The "inside" flag is copied to the appropriate mesh face (we could
 * not do this before -- since the structure of the mesh is always
 * changing, this face may not have even existed until now).
 *
 * @private
 * @param {libtess.GluTesselator} tess [description].
 * @param {libtess.ActiveRegion} reg [description].
 */
libtess.sweep.finishRegion_ = function(tess, reg) {
  // TODO(bckenny): may need to null reg at callsite

  var e = reg.eUp;
  var f = e.lFace;

  f.inside = reg.inside;
  f.anEdge = e;   // optimization for tessmono.tessellateMonoRegion() // TODO(bckenny): how so?
  libtess.sweep.deleteRegion_(tess, reg);
};


/**
 * We are given a vertex with one or more left-going edges. All affected
 * edges should be in the edge dictionary. Starting at regFirst.eUp,
 * we walk down deleting all regions where both edges have the same
 * origin vOrg. At the same time we copy the "inside" flag from the
 * active region to the face, since at this point each face will belong
 * to at most one region (this was not necessarily true until this point
 * in the sweep). The walk stops at the region above regLast; if regLast
 * is null we walk as far as possible. At the same time we relink the
 * mesh if necessary, so that the ordering of edges around vOrg is the
 * same as in the dictionary.
 *
 * @private
 * @param {libtess.GluTesselator} tess [description].
 * @param {libtess.ActiveRegion} regFirst [description].
 * @param {libtess.ActiveRegion} regLast [description].
 * @return {libtess.GluHalfEdge} [description].
 */
libtess.sweep.finishLeftRegions_ = function(tess, regFirst, regLast) {
  var regPrev = regFirst;
  var ePrev = regFirst.eUp;
  while (regPrev !== regLast) {
    // placement was OK
    regPrev.fixUpperEdge = false;
    var reg = regPrev.regionBelow();
    var e = reg.eUp;
    if (e.org !== ePrev.org) {
      if (!reg.fixUpperEdge) {
        /* Remove the last left-going edge. Even though there are no further
         * edges in the dictionary with this origin, there may be further
         * such edges in the mesh (if we are adding left edges to a vertex
         * that has already been processed). Thus it is important to call
         * finishRegion rather than just deleteRegion.
         */
        libtess.sweep.finishRegion_(tess, regPrev);
        break;
      }

      // If the edge below was a temporary edge introduced by
      // connectRightVertex, now is the time to fix it.
      e = libtess.mesh.connect(ePrev.lPrev(), e.sym);
      libtess.sweep.fixUpperEdge_(reg, e);
    }

    // Relink edges so that ePrev.oNext === e
    if (ePrev.oNext !== e) {
      libtess.mesh.meshSplice(e.oPrev(), e);
      libtess.mesh.meshSplice(ePrev, e);
    }

    // may change reg.eUp
    libtess.sweep.finishRegion_(tess, regPrev);
    ePrev = reg.eUp;
    regPrev = reg;
  }

  return ePrev;
};


/**
 * Purpose: insert right-going edges into the edge dictionary, and update
 * winding numbers and mesh connectivity appropriately. All right-going
 * edges share a common origin vOrg. Edges are inserted CCW starting at
 * eFirst; the last edge inserted is eLast.oPrev. If vOrg has any
 * left-going edges already processed, then eTopLeft must be the edge
 * such that an imaginary upward vertical segment from vOrg would be
 * contained between eTopLeft.oPrev and eTopLeft; otherwise eTopLeft
 * should be null.
 *
 * @private
 * @param {libtess.GluTesselator} tess [description].
 * @param {libtess.ActiveRegion} regUp [description].
 * @param {libtess.GluHalfEdge} eFirst [description].
 * @param {libtess.GluHalfEdge} eLast [description].
 * @param {libtess.GluHalfEdge} eTopLeft [description].
 * @param {boolean} cleanUp [description].
 */
libtess.sweep.addRightEdges_ = function(tess, regUp, eFirst, eLast, eTopLeft,
                                        cleanUp) {

  var firstTime = true;

  // Insert the new right-going edges in the dictionary
  var e = eFirst;
  do {
    libtess.sweep.addRegionBelow_(tess, regUp, e.sym);
    e = e.oNext;
  } while (e !== eLast);

  // Walk *all* right-going edges from e.org, in the dictionary order,
  // updating the winding numbers of each region, and re-linking the mesh
  // edges to match the dictionary ordering (if necessary).
  if (eTopLeft === null) {
    eTopLeft = regUp.regionBelow().eUp.rPrev();
  }
  var regPrev = regUp;
  var ePrev = eTopLeft;
  var reg;
  for (;;) {
    reg = regPrev.regionBelow();
    e = reg.eUp.sym;
    if (e.org !== ePrev.org) {
      break;
    }

    if (e.oNext !== ePrev) {
      // Unlink e from its current position, and relink below ePrev
      libtess.mesh.meshSplice(e.oPrev(), e);
      libtess.mesh.meshSplice(ePrev.oPrev(), e);
    }
    // Compute the winding number and "inside" flag for the new regions
    reg.windingNumber = regPrev.windingNumber - e.winding;
    reg.inside = libtess.sweep.isWindingInside_(tess, reg.windingNumber);

    // Check for two outgoing edges with same slope -- process these
    // before any intersection tests (see example in libtess.sweep.computeInterior).
    regPrev.dirty = true;
    if (!firstTime && libtess.sweep.checkForRightSplice_(tess, regPrev)) {
      libtess.sweep.addWinding_(e, ePrev);
      libtess.sweep.deleteRegion_(tess, regPrev); // TODO(bckenny): need to null regPrev anywhere else?
      libtess.mesh.deleteEdge(ePrev);
    }
    firstTime = false;
    regPrev = reg;
    ePrev = e;
  }

  regPrev.dirty = true;

  if (cleanUp) {
    // Check for intersections between newly adjacent edges.
    libtess.sweep.walkDirtyRegions_(tess, regPrev);
  }
};


/**
 * Set up data for and call GLU_TESS_COMBINE callback on GluTesselator.
 * @private
 * @param {!libtess.GluTesselator} tess
 * @param {!libtess.GluVertex} isect A raw vertex at the intersection.
 * @param {!Array<Object>} data The vertices of the intersecting edges.
 * @param {!Array<number>} weights The linear combination coefficients for this intersection.
 * @param {boolean} needed Whether a returned vertex is necessary in this case.
 */
libtess.sweep.callCombine_ = function(tess, isect, data, weights, needed) {
  // Copy coord data in case the callback changes it.
  var coords = [
    isect.coords[0],
    isect.coords[1],
    isect.coords[2]
  ];

  isect.data = null;
  isect.data = tess.callCombineCallback(coords, data, weights);
  if (isect.data === null) {
    if (!needed) {
      // not needed, so just use data from first vertex
      isect.data = data[0];

    } else if (!tess.fatalError) {
      // The only way fatal error is when two edges are found to intersect,
      // but the user has not provided the callback necessary to handle
      // generated intersection points.
      tess.callErrorCallback(libtess.errorType.GLU_TESS_NEED_COMBINE_CALLBACK);
      tess.fatalError = true;
    }
  }
};


/**
 * Two vertices with idential coordinates are combined into one.
 * e1.org is kept, while e2.org is discarded.
 * @private
 * @param {!libtess.GluTesselator} tess
 * @param {libtess.GluHalfEdge} e1 [description].
 * @param {libtess.GluHalfEdge} e2 [description].
 */
libtess.sweep.spliceMergeVertices_ = function(tess, e1, e2) {
  // TODO(bckenny): better way to init these? save them?
  var data = [null, null, null, null];
  var weights = [0.5, 0.5, 0, 0];

  data[0] = e1.org.data;
  data[1] = e2.org.data;
  libtess.sweep.callCombine_(tess, e1.org, data, weights, false);
  libtess.mesh.meshSplice(e1, e2);
};


/**
 * Find some weights which describe how the intersection vertex is
 * a linear combination of org and dst. Each of the two edges
 * which generated "isect" is allocated 50% of the weight; each edge
 * splits the weight between its org and dst according to the
 * relative distance to "isect".
 *
 * @private
 * @param {libtess.GluVertex} isect [description].
 * @param {libtess.GluVertex} org [description].
 * @param {libtess.GluVertex} dst [description].
 * @param {Array.<number>} weights [description].
 * @param {number} weightIndex Index into weights for first weight to supply.
 */
libtess.sweep.vertexWeights_ = function(isect, org, dst, weights, weightIndex) {
  // TODO(bckenny): think through how we can use L1dist here and be correct for coords
  var t1 = libtess.geom.vertL1dist(org, isect);
  var t2 = libtess.geom.vertL1dist(dst, isect);

  // TODO(bckenny): introduced weightIndex to mimic addressing in original
  // 1) document (though it is private and only used from getIntersectData)
  // 2) better way? manually inline into getIntersectData? supply two two-length tmp arrays?
  var i0 = weightIndex;
  var i1 = weightIndex + 1;
  weights[i0] = 0.5 * t2 / (t1 + t2);
  weights[i1] = 0.5 * t1 / (t1 + t2);
  isect.coords[0] += weights[i0] * org.coords[0] + weights[i1] * dst.coords[0];
  isect.coords[1] += weights[i0] * org.coords[1] + weights[i1] * dst.coords[1];
  isect.coords[2] += weights[i0] * org.coords[2] + weights[i1] * dst.coords[2];
};


/**
 * We've computed a new intersection point, now we need a "data" pointer
 * from the user so that we can refer to this new vertex in the
 * rendering callbacks.
 * @private
 * @param {!libtess.GluTesselator} tess
 * @param {libtess.GluVertex} isect [description].
 * @param {libtess.GluVertex} orgUp [description].
 * @param {libtess.GluVertex} dstUp [description].
 * @param {libtess.GluVertex} orgLo [description].
 * @param {libtess.GluVertex} dstLo [description].
 */
libtess.sweep.getIntersectData_ = function(tess, isect, orgUp, dstUp, orgLo,
                                           dstLo) {

  // TODO(bckenny): called for every intersection event, should these be from a pool?
  // TODO(bckenny): better way to init these?
  var weights = [0, 0, 0, 0];
  var data = [
    orgUp.data,
    dstUp.data,
    orgLo.data,
    dstLo.data
  ];

  // TODO(bckenny): it appears isect is a reappropriated vertex, so does need to be zeroed.
  // double check this.
  isect.coords[0] = isect.coords[1] = isect.coords[2] = 0;

  // TODO(bckenny): see note in libtess.sweep.vertexWeights_ for explanation of weightIndex. fix?
  libtess.sweep.vertexWeights_(isect, orgUp, dstUp, weights, 0);
  libtess.sweep.vertexWeights_(isect, orgLo, dstLo, weights, 2);

  libtess.sweep.callCombine_(tess, isect, data, weights, true);
};


/**
 * Check the upper and lower edge of regUp, to make sure that the
 * eUp.org is above eLo, or eLo.org is below eUp (depending on which
 * origin is leftmost).
 *
 * The main purpose is to splice right-going edges with the same
 * dest vertex and nearly identical slopes (ie. we can't distinguish
 * the slopes numerically). However the splicing can also help us
 * to recover from numerical errors. For example, suppose at one
 * point we checked eUp and eLo, and decided that eUp.org is barely
 * above eLo. Then later, we split eLo into two edges (eg. from
 * a splice operation like this one). This can change the result of
 * our test so that now eUp.org is incident to eLo, or barely below it.
 * We must correct this condition to maintain the dictionary invariants.
 *
 * One possibility is to check these edges for intersection again
 * (i.e. checkForIntersect). This is what we do if possible. However
 * checkForIntersect requires that tess.event lies between eUp and eLo,
 * so that it has something to fall back on when the intersection
 * calculation gives us an unusable answer. So, for those cases where
 * we can't check for intersection, this routine fixes the problem
 * by just splicing the offending vertex into the other edge.
 * This is a guaranteed solution, no matter how degenerate things get.
 * Basically this is a combinatorial solution to a numerical problem.
 *
 * @private
 * @param {libtess.GluTesselator} tess [description].
 * @param {libtess.ActiveRegion} regUp [description].
 * @return {boolean} [description].
 */
libtess.sweep.checkForRightSplice_ = function(tess, regUp) {
  // TODO(bckenny): fully learn how these two checks work

  var regLo = regUp.regionBelow();
  var eUp = regUp.eUp;
  var eLo = regLo.eUp;

  if (libtess.geom.vertLeq(eUp.org, eLo.org)) {
    if (libtess.geom.edgeSign(eLo.dst(), eUp.org, eLo.org) > 0) {
      return false;
    }

    // eUp.org appears to be below eLo
    if (!libtess.geom.vertEq(eUp.org, eLo.org)) {
      // Splice eUp.org into eLo
      libtess.mesh.splitEdge(eLo.sym);
      libtess.mesh.meshSplice(eUp, eLo.oPrev());
      regUp.dirty = regLo.dirty = true;

    } else if (eUp.org !== eLo.org) {
      // merge the two vertices, discarding eUp.org
      tess.pq.remove(eUp.org.pqHandle);
      libtess.sweep.spliceMergeVertices_(tess, eLo.oPrev(), eUp);
    }

  } else {
    if (libtess.geom.edgeSign(eUp.dst(), eLo.org, eUp.org) < 0) {
      return false;
    }

    // eLo.org appears to be above eUp, so splice eLo.org into eUp
    regUp.regionAbove().dirty = regUp.dirty = true;
    libtess.mesh.splitEdge(eUp.sym);
    libtess.mesh.meshSplice(eLo.oPrev(), eUp);
  }

  return true;
};


/**
 * Check the upper and lower edge of regUp to make sure that the
 * eUp.dst() is above eLo, or eLo.dst() is below eUp (depending on which
 * destination is rightmost).
 *
 * Theoretically, this should always be true. However, splitting an edge
 * into two pieces can change the results of previous tests. For example,
 * suppose at one point we checked eUp and eLo, and decided that eUp.dst()
 * is barely above eLo. Then later, we split eLo into two edges (eg. from
 * a splice operation like this one). This can change the result of
 * the test so that now eUp.dst() is incident to eLo, or barely below it.
 * We must correct this condition to maintain the dictionary invariants
 * (otherwise new edges might get inserted in the wrong place in the
 * dictionary, and bad stuff will happen).
 *
 * We fix the problem by just splicing the offending vertex into the
 * other edge.
 *
 * @private
 * @param {libtess.GluTesselator} tess description].
 * @param {libtess.ActiveRegion} regUp [description].
 * @return {boolean} [description].
 */
libtess.sweep.checkForLeftSplice_ = function(tess, regUp) {
  var regLo = regUp.regionBelow();
  var eUp = regUp.eUp;
  var eLo = regLo.eUp;
  var e;

  if (libtess.geom.vertLeq(eUp.dst(), eLo.dst())) {
    if (libtess.geom.edgeSign(eUp.dst(), eLo.dst(), eUp.org) < 0) {
      return false;
    }

    // eLo.dst() is above eUp, so splice eLo.dst() into eUp
    regUp.regionAbove().dirty = regUp.dirty = true;
    e = libtess.mesh.splitEdge(eUp);
    libtess.mesh.meshSplice(eLo.sym, e);
    e.lFace.inside = regUp.inside;

  } else {
    if (libtess.geom.edgeSign(eLo.dst(), eUp.dst(), eLo.org) > 0) {
      return false;
    }

    // eUp.dst() is below eLo, so splice eUp.dst() into eLo
    regUp.dirty = regLo.dirty = true;
    e = libtess.mesh.splitEdge(eLo);
    libtess.mesh.meshSplice(eUp.lNext, eLo.sym);
    e.rFace().inside = regUp.inside;
  }

  return true;
};


/**
 * Check the upper and lower edges of the given region to see if
 * they intersect. If so, create the intersection and add it
 * to the data structures.
 *
 * Returns true if adding the new intersection resulted in a recursive
 * call to addRightEdges_(); in this case all "dirty" regions have been
 * checked for intersections, and possibly regUp has been deleted.
 *
 * @private
 * @param {libtess.GluTesselator} tess [description].
 * @param {libtess.ActiveRegion} regUp [description].
 * @return {boolean} [description].
 */
libtess.sweep.checkForIntersect_ = function(tess, regUp) {
  var regLo = regUp.regionBelow();
  var eUp = regUp.eUp;
  var eLo = regLo.eUp;
  var orgUp = eUp.org;
  var orgLo = eLo.org;
  var dstUp = eUp.dst();
  var dstLo = eLo.dst();

  var isect = new libtess.GluVertex();

  if (orgUp === orgLo) {
    // right endpoints are the same
    return false;
  }

  var tMinUp = Math.min(orgUp.t, dstUp.t);
  var tMaxLo = Math.max(orgLo.t, dstLo.t);
  if (tMinUp > tMaxLo) {
    // t ranges do not overlap
    return false;
  }

  if (libtess.geom.vertLeq(orgUp, orgLo)) {
    if (libtess.geom.edgeSign(dstLo, orgUp, orgLo) > 0) {
      return false;
    }
  } else {
    if (libtess.geom.edgeSign(dstUp, orgLo, orgUp) < 0) {
      return false;
    }
  }

  // At this point the edges intersect, at least marginally
  libtess.geom.edgeIntersect(dstUp, orgUp, dstLo, orgLo, isect);

  // The following properties are guaranteed:

  if (libtess.geom.vertLeq(isect, tess.event)) {
    /* The intersection point lies slightly to the left of the sweep line,
     * so move it until it's slightly to the right of the sweep line.
     * (If we had perfect numerical precision, this would never happen
     * in the first place). The easiest and safest thing to do is
     * replace the intersection by tess.event.
     */
    isect.s = tess.event.s;
    isect.t = tess.event.t;
  }

  // TODO(bckenny): try to find test54.d
  /* Similarly, if the computed intersection lies to the right of the
   * rightmost origin (which should rarely happen), it can cause
   * unbelievable inefficiency on sufficiently degenerate inputs.
   * (If you have the test program, try running test54.d with the
   * "X zoom" option turned on).
   */
  var orgMin = libtess.geom.vertLeq(orgUp, orgLo) ? orgUp : orgLo;
  if (libtess.geom.vertLeq(orgMin, isect)) {
    isect.s = orgMin.s;
    isect.t = orgMin.t;
  }

  if (libtess.geom.vertEq(isect, orgUp) || libtess.geom.vertEq(isect, orgLo)) {
    // Easy case -- intersection at one of the right endpoints
    libtess.sweep.checkForRightSplice_(tess, regUp);
    return false;
  }

  // TODO(bckenny): clean this up; length is distracting
  if ((!libtess.geom.vertEq(dstUp, tess.event) &&
      libtess.geom.edgeSign(dstUp, tess.event, isect) >= 0) ||
      (!libtess.geom.vertEq(dstLo, tess.event) &&
          libtess.geom.edgeSign(dstLo, tess.event, isect) <= 0)) {

    /* Very unusual -- the new upper or lower edge would pass on the
     * wrong side of the sweep event, or through it. This can happen
     * due to very small numerical errors in the intersection calculation.
     */
    if (dstLo === tess.event) {
      // Splice dstLo into eUp, and process the new region(s)
      libtess.mesh.splitEdge(eUp.sym);
      libtess.mesh.meshSplice(eLo.sym, eUp);
      regUp = libtess.sweep.topLeftRegion_(regUp);
      eUp = regUp.regionBelow().eUp;
      libtess.sweep.finishLeftRegions_(tess, regUp.regionBelow(), regLo);
      libtess.sweep.addRightEdges_(tess, regUp, eUp.oPrev(), eUp, eUp, true);
      return true;
    }

    if (dstUp === tess.event) {
      // Splice dstUp into eLo, and process the new region(s)
      libtess.mesh.splitEdge(eLo.sym);
      libtess.mesh.meshSplice(eUp.lNext, eLo.oPrev());
      regLo = regUp;
      regUp = libtess.sweep.topRightRegion_(regUp);
      var e = regUp.regionBelow().eUp.rPrev();
      regLo.eUp = eLo.oPrev();
      eLo = libtess.sweep.finishLeftRegions_(tess, regLo, null);
      libtess.sweep.addRightEdges_(tess, regUp, eLo.oNext, eUp.rPrev(), e,
          true);
      return true;
    }

    /* Special case: called from connectRightVertex. If either
     * edge passes on the wrong side of tess.event, split it
     * (and wait for connectRightVertex to splice it appropriately).
     */
    if (libtess.geom.edgeSign(dstUp, tess.event, isect) >= 0) {
      regUp.regionAbove().dirty = regUp.dirty = true;
      libtess.mesh.splitEdge(eUp.sym);
      eUp.org.s = tess.event.s;
      eUp.org.t = tess.event.t;
    }

    if (libtess.geom.edgeSign(dstLo, tess.event, isect) <= 0) {
      regUp.dirty = regLo.dirty = true;
      libtess.mesh.splitEdge(eLo.sym);
      eLo.org.s = tess.event.s;
      eLo.org.t = tess.event.t;
    }

    // leave the rest for connectRightVertex
    return false;
  }

  /* General case -- split both edges, splice into new vertex.
   * When we do the splice operation, the order of the arguments is
   * arbitrary as far as correctness goes. However, when the operation
   * creates a new face, the work done is proportional to the size of
   * the new face. We expect the faces in the processed part of
   * the mesh (ie. eUp.lFace) to be smaller than the faces in the
   * unprocessed original contours (which will be eLo.oPrev.lFace).
   */
  libtess.mesh.splitEdge(eUp.sym);
  libtess.mesh.splitEdge(eLo.sym);
  libtess.mesh.meshSplice(eLo.oPrev(), eUp);
  eUp.org.s = isect.s;
  eUp.org.t = isect.t;
  eUp.org.pqHandle = tess.pq.insert(eUp.org);
  libtess.sweep.getIntersectData_(tess, eUp.org, orgUp, dstUp, orgLo, dstLo);
  regUp.regionAbove().dirty = regUp.dirty = regLo.dirty = true;

  return false;
};


/**
 * When the upper or lower edge of any region changes, the region is
 * marked "dirty". This routine walks through all the dirty regions
 * and makes sure that the dictionary invariants are satisfied
 * (see the comments at the beginning of this file). Of course,
 * new dirty regions can be created as we make changes to restore
 * the invariants.
 * @private
 * @param {libtess.GluTesselator} tess [description].
 * @param {libtess.ActiveRegion} regUp [description].
 */
libtess.sweep.walkDirtyRegions_ = function(tess, regUp) {
  var regLo = regUp.regionBelow();

  for (;;) {
    // Find the lowest dirty region (we walk from the bottom up).
    while (regLo.dirty) {
      regUp = regLo;
      regLo = regLo.regionBelow();
    }
    if (!regUp.dirty) {
      regLo = regUp;
      regUp = regUp.regionAbove();
      if (regUp === null || !regUp.dirty) {
        // We've walked all the dirty regions
        return;
      }
    }

    regUp.dirty = false;
    var eUp = regUp.eUp;
    var eLo = regLo.eUp;

    if (eUp.dst() !== eLo.dst()) {
      // Check that the edge ordering is obeyed at the dst vertices.
      if (libtess.sweep.checkForLeftSplice_(tess, regUp)) {
        // If the upper or lower edge was marked fixUpperEdge, then
        // we no longer need it (since these edges are needed only for
        // vertices which otherwise have no right-going edges).
        if (regLo.fixUpperEdge) {
          libtess.sweep.deleteRegion_(tess, regLo);
          libtess.mesh.deleteEdge(eLo);
          regLo = regUp.regionBelow();
          eLo = regLo.eUp;

        } else if (regUp.fixUpperEdge) {
          libtess.sweep.deleteRegion_(tess, regUp);
          libtess.mesh.deleteEdge(eUp);
          regUp = regLo.regionAbove();
          eUp = regUp.eUp;
        }
      }
    }

    if (eUp.org !== eLo.org) {
      if (eUp.dst() !== eLo.dst() && !regUp.fixUpperEdge &&
          !regLo.fixUpperEdge &&
          (eUp.dst() === tess.event || eLo.dst() === tess.event)) {
        /* When all else fails in checkForIntersect(), it uses tess.event
         * as the intersection location. To make this possible, it requires
         * that tess.event lie between the upper and lower edges, and also
         * that neither of these is marked fixUpperEdge (since in the worst
         * case it might splice one of these edges into tess.event, and
         * violate the invariant that fixable edges are the only right-going
         * edge from their associated vertex).
         */
        if (libtess.sweep.checkForIntersect_(tess, regUp)) {
          // walkDirtyRegions() was called recursively; we're done
          return;
        }

      } else {
        // Even though we can't use checkForIntersect(), the org vertices
        // may violate the dictionary edge ordering. Check and correct this.
        libtess.sweep.checkForRightSplice_(tess, regUp);
      }
    }

    if (eUp.org === eLo.org && eUp.dst() === eLo.dst()) {
      // A degenerate loop consisting of only two edges -- delete it.
      libtess.sweep.addWinding_(eLo, eUp);
      libtess.sweep.deleteRegion_(tess, regUp);
      libtess.mesh.deleteEdge(eUp);
      regUp = regLo.regionAbove();
    }
  }
};


/**
 * Purpose: connect a "right" vertex vEvent (one where all edges go left)
 * to the unprocessed portion of the mesh. Since there are no right-going
 * edges, two regions (one above vEvent and one below) are being merged
 * into one. regUp is the upper of these two regions.
 *
 * There are two reasons for doing this (adding a right-going edge):
 *  - if the two regions being merged are "inside", we must add an edge
 *    to keep them separated (the combined region would not be monotone).
 *  - in any case, we must leave some record of vEvent in the dictionary,
 *    so that we can merge vEvent with features that we have not seen yet.
 *    For example, maybe there is a vertical edge which passes just to
 *    the right of vEvent; we would like to splice vEvent into this edge.
 *
 * However, we don't want to connect vEvent to just any vertex. We don't
 * want the new edge to cross any other edges; otherwise we will create
 * intersection vertices even when the input data had no self-intersections.
 * (This is a bad thing; if the user's input data has no intersections,
 * we don't want to generate any false intersections ourselves.)
 *
 * Our eventual goal is to connect vEvent to the leftmost unprocessed
 * vertex of the combined region (the union of regUp and regLo).
 * But because of unseen vertices with all right-going edges, and also
 * new vertices which may be created by edge intersections, we don't
 * know where that leftmost unprocessed vertex is. In the meantime, we
 * connect vEvent to the closest vertex of either chain, and mark the region
 * as "fixUpperEdge". This flag says to delete and reconnect this edge
 * to the next processed vertex on the boundary of the combined region.
 * Quite possibly the vertex we connected to will turn out to be the
 * closest one, in which case we won't need to make any changes.
 *
 * @private
 * @param {libtess.GluTesselator} tess [description].
 * @param {libtess.ActiveRegion} regUp [description].
 * @param {libtess.GluHalfEdge} eBottomLeft [description].
 */
libtess.sweep.connectRightVertex_ = function(tess, regUp, eBottomLeft) {
  var eTopLeft = eBottomLeft.oNext;
  var regLo = regUp.regionBelow();
  var eUp = regUp.eUp;
  var eLo = regLo.eUp;
  var degenerate = false;

  if (eUp.dst() !== eLo.dst()) {
    libtess.sweep.checkForIntersect_(tess, regUp);
  }

  // Possible new degeneracies: upper or lower edge of regUp may pass
  // through vEvent, or may coincide with new intersection vertex
  if (libtess.geom.vertEq(eUp.org, tess.event)) {
    libtess.mesh.meshSplice(eTopLeft.oPrev(), eUp);
    regUp = libtess.sweep.topLeftRegion_(regUp);
    eTopLeft = regUp.regionBelow().eUp;
    libtess.sweep.finishLeftRegions_(tess, regUp.regionBelow(), regLo);
    degenerate = true;
  }
  if (libtess.geom.vertEq(eLo.org, tess.event)) {
    libtess.mesh.meshSplice(eBottomLeft, eLo.oPrev());
    eBottomLeft = libtess.sweep.finishLeftRegions_(tess, regLo, null);
    degenerate = true;
  }
  if (degenerate) {
    libtess.sweep.addRightEdges_(tess, regUp, eBottomLeft.oNext, eTopLeft,
        eTopLeft, true);
    return;
  }

  // Non-degenerate situation -- need to add a temporary, fixable edge.
  // Connect to the closer of eLo.org, eUp.org.
  var eNew;
  if (libtess.geom.vertLeq(eLo.org, eUp.org)) {
    eNew = eLo.oPrev();
  } else {
    eNew = eUp;
  }
  eNew = libtess.mesh.connect(eBottomLeft.lPrev(), eNew);

  // Prevent cleanup, otherwise eNew might disappear before we've even
  // had a chance to mark it as a temporary edge.
  libtess.sweep.addRightEdges_(tess, regUp, eNew, eNew.oNext, eNew.oNext,
      false);
  eNew.sym.activeRegion.fixUpperEdge = true;
  libtess.sweep.walkDirtyRegions_(tess, regUp);
};


/**
 * The event vertex lies exacty on an already-processed edge or vertex.
 * Adding the new vertex involves splicing it into the already-processed
 * part of the mesh.
 * @private
 * @param {!libtess.GluTesselator} tess
 * @param {libtess.ActiveRegion} regUp [description].
 * @param {libtess.GluVertex} vEvent [description].
 */
libtess.sweep.connectLeftDegenerate_ = function(tess, regUp, vEvent) {
  var e = regUp.eUp;
  /* istanbul ignore if */
  if (libtess.geom.vertEq(e.org, vEvent)) {
    // NOTE(bckenny): this code is unreachable but remains for a hypothetical
    // future extension of libtess. See docs on libtess.sweep.TOLERANCE_NONZERO_
    // for more information. Conditional on TOLERANCE_NONZERO_ to help Closure
    // Compiler eliminate dead code.
    // e.org is an unprocessed vertex - just combine them, and wait
    // for e.org to be pulled from the queue
    if (libtess.sweep.TOLERANCE_NONZERO_) {
      libtess.sweep.spliceMergeVertices_(tess, e, vEvent.anEdge);
    }
    return;
  }

  if (!libtess.geom.vertEq(e.dst(), vEvent)) {
    // General case -- splice vEvent into edge e which passes through it
    libtess.mesh.splitEdge(e.sym);

    if (regUp.fixUpperEdge) {
      // This edge was fixable -- delete unused portion of original edge
      libtess.mesh.deleteEdge(e.oNext);
      regUp.fixUpperEdge = false;
    }

    libtess.mesh.meshSplice(vEvent.anEdge, e);

    // recurse
    libtess.sweep.sweepEvent_(tess, vEvent);
    return;
  }

  // NOTE(bckenny): this code is unreachable but remains for a hypothetical
  // future extension of libtess. See docs on libtess.sweep.TOLERANCE_NONZERO_
  // for more information. Conditional on TOLERANCE_NONZERO_ to help Closure
  // Compiler eliminate dead code.
  // vEvent coincides with e.dst(), which has already been processed.
  // Splice in the additional right-going edges.
  /* istanbul ignore next */

  /* istanbul ignore next */
  if (libtess.sweep.TOLERANCE_NONZERO_) {
    regUp = libtess.sweep.topRightRegion_(regUp);
    var reg = regUp.regionBelow();
    var eTopRight = reg.eUp.sym;
    var eTopLeft = eTopRight.oNext;
    var eLast = eTopLeft;

    if (reg.fixUpperEdge) {
      // Here e.dst() has only a single fixable edge going right.
      // We can delete it since now we have some real right-going edges.

      // there are some left edges too
      libtess.sweep.deleteRegion_(tess, reg); // TODO(bckenny): something to null?
      libtess.mesh.deleteEdge(eTopRight);
      eTopRight = eTopLeft.oPrev();
    }

    libtess.mesh.meshSplice(vEvent.anEdge, eTopRight);
    if (!libtess.geom.edgeGoesLeft(eTopLeft)) {
      // e.dst() had no left-going edges -- indicate this to addRightEdges()
      eTopLeft = null;
    }

    libtess.sweep.addRightEdges_(tess, regUp, eTopRight.oNext, eLast, eTopLeft,
        true);
  }
};


/**
 * Connect a "left" vertex (one where both edges go right)
 * to the processed portion of the mesh. Let R be the active region
 * containing vEvent, and let U and L be the upper and lower edge
 * chains of R. There are two possibilities:
 *
 * - the normal case: split R into two regions, by connecting vEvent to
 *   the rightmost vertex of U or L lying to the left of the sweep line
 *
 * - the degenerate case: if vEvent is close enough to U or L, we
 *   merge vEvent into that edge chain. The subcases are:
 *  - merging with the rightmost vertex of U or L
 *  - merging with the active edge of U or L
 *  - merging with an already-processed portion of U or L
 *
 * @private
 * @param {libtess.GluTesselator} tess   [description].
 * @param {libtess.GluVertex} vEvent [description].
 */
libtess.sweep.connectLeftVertex_ = function(tess, vEvent) {
  // TODO(bckenny): tmp only used for sweep. better to keep tmp across calls?
  var tmp = new libtess.ActiveRegion();

  // NOTE(bckenny): this was commented out in the original
  // libtess.assert(vEvent.anEdge.oNext.oNext === vEvent.anEdge);

  // Get a pointer to the active region containing vEvent
  tmp.eUp = vEvent.anEdge.sym;
  var regUp = tess.dict.search(tmp).getKey();
  var regLo = regUp.regionBelow();
  var eUp = regUp.eUp;
  var eLo = regLo.eUp;

  // try merging with U or L first
  if (libtess.geom.edgeSign(eUp.dst(), vEvent, eUp.org) === 0) {
    libtess.sweep.connectLeftDegenerate_(tess, regUp, vEvent);
    return;
  }

  // Connect vEvent to rightmost processed vertex of either chain.
  // e.dst() is the vertex that we will connect to vEvent.
  var reg = libtess.geom.vertLeq(eLo.dst(), eUp.dst()) ? regUp : regLo;
  var eNew;
  if (regUp.inside || reg.fixUpperEdge) {
    if (reg === regUp) {
      eNew = libtess.mesh.connect(vEvent.anEdge.sym, eUp.lNext);

    } else {
      var tempHalfEdge = libtess.mesh.connect(eLo.dNext(), vEvent.anEdge);
      eNew = tempHalfEdge.sym;
    }

    if (reg.fixUpperEdge) {
      libtess.sweep.fixUpperEdge_(reg, eNew);

    } else {
      libtess.sweep.computeWinding_(tess,
          libtess.sweep.addRegionBelow_(tess, regUp, eNew));
    }
    libtess.sweep.sweepEvent_(tess, vEvent);

  } else {
    // The new vertex is in a region which does not belong to the polygon.
    // We don''t need to connect this vertex to the rest of the mesh.
    libtess.sweep.addRightEdges_(tess, regUp, vEvent.anEdge, vEvent.anEdge,
        null, true);
  }
};


/**
 * Does everything necessary when the sweep line crosses a vertex.
 * Updates the mesh and the edge dictionary.
 * @private
 * @param {libtess.GluTesselator} tess [description].
 * @param {libtess.GluVertex} vEvent [description].
 */
libtess.sweep.sweepEvent_ = function(tess, vEvent) {
  tess.event = vEvent; // for access in edgeLeq_ // TODO(bckenny): wuh?

  /* Check if this vertex is the right endpoint of an edge that is
   * already in the dictionary.  In this case we don't need to waste
   * time searching for the location to insert new edges.
   */
  var e = vEvent.anEdge;
  while (e.activeRegion === null) {
    e = e.oNext;
    if (e === vEvent.anEdge) {
      // All edges go right -- not incident to any processed edges
      libtess.sweep.connectLeftVertex_(tess, vEvent);
      return;
    }
  }

  /* Processing consists of two phases: first we "finish" all the
   * active regions where both the upper and lower edges terminate
   * at vEvent (ie. vEvent is closing off these regions).
   * We mark these faces "inside" or "outside" the polygon according
   * to their winding number, and delete the edges from the dictionary.
   * This takes care of all the left-going edges from vEvent.
   */
  var regUp = libtess.sweep.topLeftRegion_(e.activeRegion);
  var reg = regUp.regionBelow();
  var eTopLeft = reg.eUp;
  var eBottomLeft = libtess.sweep.finishLeftRegions_(tess, reg, null);

  /* Next we process all the right-going edges from vEvent. This
   * involves adding the edges to the dictionary, and creating the
   * associated "active regions" which record information about the
   * regions between adjacent dictionary edges.
   */
  if (eBottomLeft.oNext === eTopLeft) {
    // No right-going edges -- add a temporary "fixable" edge
    libtess.sweep.connectRightVertex_(tess, regUp, eBottomLeft);

  } else {
    libtess.sweep.addRightEdges_(tess, regUp, eBottomLeft.oNext, eTopLeft,
        eTopLeft, true);
  }
};


/**
 * We add two sentinel edges above and below all other edges,
 * to avoid special cases at the top and bottom.
 * @private
 * @param {libtess.GluTesselator} tess [description].
 * @param {number} t [description].
 */
libtess.sweep.addSentinel_ = function(tess, t) {
  var reg = new libtess.ActiveRegion();

  var e = libtess.mesh.makeEdge(tess.mesh);

  e.org.s = libtess.sweep.SENTINEL_COORD_;
  e.org.t = t;
  e.dst().s = -libtess.sweep.SENTINEL_COORD_;
  e.dst().t = t;
  tess.event = e.dst(); //initialize it

  reg.eUp = e;
  reg.windingNumber = 0;
  reg.inside = false;
  reg.fixUpperEdge = false;
  reg.sentinel = true;
  reg.dirty = false;
  reg.nodeUp = tess.dict.insert(reg);
};


/**
 * We maintain an ordering of edge intersections with the sweep line.
 * This order is maintained in a dynamic dictionary.
 * @private
 * @param {libtess.GluTesselator} tess [description].
 */
libtess.sweep.initEdgeDict_ = function(tess) {
  tess.dict = new libtess.Dict(tess, libtess.sweep.edgeLeq_);

  libtess.sweep.addSentinel_(tess, -libtess.sweep.SENTINEL_COORD_);
  libtess.sweep.addSentinel_(tess, libtess.sweep.SENTINEL_COORD_);
};


/**
 * [doneEdgeDict_ description]
 * @private
 * @param {libtess.GluTesselator} tess [description].
 */
libtess.sweep.doneEdgeDict_ = function(tess) {
  // NOTE(bckenny): fixedEdges is only used in the assert below, so ignore so
  // when asserts are removed jshint won't error.
  /* jshint unused:false */
  var fixedEdges = 0;

  var reg;
  while ((reg = tess.dict.getMin().getKey()) !== null) {
    // At the end of all processing, the dictionary should contain
    // only the two sentinel edges, plus at most one "fixable" edge
    // created by connectRightVertex().
    if (!reg.sentinel) {
    }
    libtess.sweep.deleteRegion_(tess, reg);
  }

  // NOTE(bckenny): see tess.dict.deleteDict_() for old delete dict function
  tess.dict = null;
};


/**
 * Remove zero-length edges, and contours with fewer than 3 vertices.
 * @private
 * @param {libtess.GluTesselator} tess [description].
 */
libtess.sweep.removeDegenerateEdges_ = function(tess) {
  var eHead = tess.mesh.eHead;

  var eNext;
  for (var e = eHead.next; e !== eHead; e = eNext) {
    eNext = e.next;
    var eLNext = e.lNext;

    if (libtess.geom.vertEq(e.org, e.dst()) && e.lNext.lNext !== e) {
      // Zero-length edge, contour has at least 3 edges
      libtess.sweep.spliceMergeVertices_(tess, eLNext, e); // deletes e.org
      libtess.mesh.deleteEdge(e); // e is a self-loop TODO(bckenny): does this comment really apply here?
      e = eLNext;
      eLNext = e.lNext;
    }

    if (eLNext.lNext === e) {
      // Degenerate contour (one or two edges)
      if (eLNext !== e) {
        if (eLNext === eNext || eLNext === eNext.sym) {
          eNext = eNext.next;
        }
        libtess.mesh.deleteEdge(eLNext);
      }

      if (e === eNext || e === eNext.sym) {
        eNext = eNext.next;
      }
      libtess.mesh.deleteEdge(e);
    }
  }
};


/**
 * Construct priority queue and insert all vertices into it, which determines
 * the order in which vertices cross the sweep line.
 * @private
 * @param {libtess.GluTesselator} tess [description].
 */
libtess.sweep.initPriorityQ_ = function(tess) {
  var pq = new libtess.PriorityQ();
  tess.pq = pq;

  var vHead = tess.mesh.vHead;
  var v;
  for (v = vHead.next; v !== vHead; v = v.next) {
    v.pqHandle = pq.insert(v);
  }

  pq.init();
};


/**
 * [donePriorityQ_ description]
 * @private
 * @param {libtess.GluTesselator} tess [description].
 */
libtess.sweep.donePriorityQ_ = function(tess) {
  // TODO(bckenny): probably don't need deleteQ. check that function for comment
  tess.pq.deleteQ();
  tess.pq = null;
};


/**
 * Delete any degenerate faces with only two edges. walkDirtyRegions()
 * will catch almost all of these, but it won't catch degenerate faces
 * produced by splice operations on already-processed edges.
 * The two places this can happen are in finishLeftRegions(), when
 * we splice in a "temporary" edge produced by connectRightVertex(),
 * and in checkForLeftSplice(), where we splice already-processed
 * edges to ensure that our dictionary invariants are not violated
 * by numerical errors.
 *
 * In both these cases it is *very* dangerous to delete the offending
 * edge at the time, since one of the routines further up the stack
 * will sometimes be keeping a pointer to that edge.
 *
 * @private
 * @param {libtess.GluMesh} mesh [description].
 */
libtess.sweep.removeDegenerateFaces_ = function(mesh) {
  var fNext;
  for (var f = mesh.fHead.next; f !== mesh.fHead; f = fNext) {
    fNext = f.next;
    var e = f.anEdge;

    if (e.lNext.lNext === e) {
      // A face with only two edges
      libtess.sweep.addWinding_(e.oNext, e);
      libtess.mesh.deleteEdge(e);
    }
  }
};

/* global libtess */

/** @const */
libtess.tessmono = {};

/**
 * Tessellates a monotone region (what else would it do??). The region must
 * consist of a single loop of half-edges (see mesh.js) oriented CCW. "Monotone"
 * in this case means that any vertical line intersects the interior of the
 * region in a single interval.
 *
 * Tessellation consists of adding interior edges (actually pairs of
 * half-edges), to split the region into non-overlapping triangles.
 * @private
 * @param {!libtess.GluFace} face
 */
libtess.tessmono.tessellateMonoRegion_ = function(face) {
  /* The basic idea is explained in Preparata and Shamos (which I don't
   * have handy right now), although their implementation is more
   * complicated than this one. The are two edge chains, an upper chain
   * and a lower chain. We process all vertices from both chains in order,
   * from right to left.
   *
   * The algorithm ensures that the following invariant holds after each
   * vertex is processed: the untessellated region consists of two
   * chains, where one chain (say the upper) is a single edge, and
   * the other chain is concave. The left vertex of the single edge
   * is always to the left of all vertices in the concave chain.
   *
   * Each step consists of adding the rightmost unprocessed vertex to one
   * of the two chains, and forming a fan of triangles from the rightmost
   * of two chain endpoints. Determining whether we can add each triangle
   * to the fan is a simple orientation test. By making the fan as large
   * as possible, we restore the invariant (check it yourself).
   *
   * All edges are oriented CCW around the boundary of the region.
   * First, find the half-edge whose origin vertex is rightmost.
   * Since the sweep goes from left to right, face.anEdge should
   * be close to the edge we want.
   */
  var up = face.anEdge;

  for (; libtess.geom.vertLeq(up.dst(), up.org); up = up.lPrev()) { }
  for (; libtess.geom.vertLeq(up.org, up.dst()); up = up.lNext) { }

  var lo = up.lPrev();

  var tempHalfEdge;
  while (up.lNext !== lo) {
    if (libtess.geom.vertLeq(up.dst(), lo.org)) {
      // up.dst() is on the left. It is safe to form triangles from lo.org.
      // The edgeGoesLeft test guarantees progress even when some triangles
      // are CW, given that the upper and lower chains are truly monotone.
      while (lo.lNext !== up && (libtess.geom.edgeGoesLeft(lo.lNext) ||
          libtess.geom.edgeSign(lo.org, lo.dst(), lo.lNext.dst()) <= 0)) {

        tempHalfEdge = libtess.mesh.connect(lo.lNext, lo);
        lo = tempHalfEdge.sym;
      }
      lo = lo.lPrev();

    } else {
      // lo.org is on the left. We can make CCW triangles from up.dst().
      while (lo.lNext !== up && (libtess.geom.edgeGoesRight(up.lPrev()) ||
          libtess.geom.edgeSign(up.dst(), up.org, up.lPrev().org) >= 0)) {

        tempHalfEdge = libtess.mesh.connect(up, up.lPrev());
        up = tempHalfEdge.sym;
      }
      up = up.lNext;
    }
  }

  // Now lo.org == up.dst() == the leftmost vertex. The remaining region
  // can be tessellated in a fan from this leftmost vertex.
  while (lo.lNext.lNext !== up) {
    tempHalfEdge = libtess.mesh.connect(lo.lNext, lo);
    lo = tempHalfEdge.sym;
  }
};

/**
 * Tessellates each region of the mesh which is marked "inside" the polygon.
 * Each such region must be monotone.
 * @param {!libtess.GluMesh} mesh
 */
libtess.tessmono.tessellateInterior = function(mesh) {
  var next;
  for (var f = mesh.fHead.next; f !== mesh.fHead; f = next) {
    // Make sure we don't try to tessellate the new triangles.
    next = f.next;
    if (f.inside) {
      libtess.tessmono.tessellateMonoRegion_(f);
    }
  }
};

/**
 * Zaps (i.e. sets to null) all faces which are not marked "inside" the polygon.
 * Since further mesh operations on null faces are not allowed, the main purpose
 * is to clean up the mesh so that exterior loops are not represented in the
 * data structure.
 * @param {!libtess.GluMesh} mesh
 */
libtess.tessmono.discardExterior = function(mesh) {
  var next;
  for (var f = mesh.fHead.next; f !== mesh.fHead; f = next) {
    // Since f will be destroyed, save its next pointer.
    next = f.next;
    if (!f.inside) {
      libtess.mesh.zapFace(f);
    }
  }
};

/**
 * Resets the winding numbers on all edges so that regions marked "inside" the
 * polygon have a winding number of "value", and regions outside have a winding
 * number of 0.
 *
 * If keepOnlyBoundary is true, it also deletes all edges which do not separate
 * an interior region from an exterior one.
 *
 * @param {!libtess.GluMesh} mesh
 * @param {number} value
 * @param {boolean} keepOnlyBoundary
 */
libtess.tessmono.setWindingNumber = function(mesh, value, keepOnlyBoundary) {
  var eNext;
  for (var e = mesh.eHead.next; e !== mesh.eHead; e = eNext) {
    eNext = e.next;

    if (e.rFace().inside !== e.lFace.inside) {
      // This is a boundary edge (one side is interior, one is exterior).
      e.winding = (e.lFace.inside) ? value : -value;

    } else {
      // Both regions are interior, or both are exterior.
      if (!keepOnlyBoundary) {
        e.winding = 0;

      } else {
        libtess.mesh.deleteEdge(e);
      }
    }
  }
};

/* global libtess */

/**
 * A list of edges crossing the sweep line, sorted from top to bottom.
 * Implementation is a doubly-linked list, sorted by the injected edgeLeq
 * comparator function. Here it is a simple ordering, but see libtess.sweep for
 * the list of invariants on the edge dictionary this ordering creates.
 * @constructor
 * @struct
 * @param {!libtess.GluTesselator} frame
 * @param {function(!libtess.GluTesselator, !libtess.ActiveRegion, !libtess.ActiveRegion): boolean} leq
 */
libtess.Dict = function(frame, leq) {

  /**
   * The head of the doubly-linked DictNode list. At creation time, links back
   * and forward only to itself.
   * @private {!libtess.DictNode}
   */
  this.head_ = new libtess.DictNode();

  /**
   * The GluTesselator used as the frame for edge/event comparisons.
   * @private {!libtess.GluTesselator}
   */
  this.frame_ = frame;

  /**
   * Comparison function to maintain the invariants of the Dict. See
   * libtess.sweep.edgeLeq_ for source.
   * @private
   * @type {function(!libtess.GluTesselator, !libtess.ActiveRegion, !libtess.ActiveRegion): boolean}
   */
  this.leq_ = leq;
};

/* istanbul ignore next */
/**
 * Formerly used to delete the dict.
 * NOTE(bckenny): No longer called but left for memFree documentation. Nulled at
 * former callsite instead (sweep.doneEdgeDict_)
 * @private
 */
libtess.Dict.prototype.deleteDict_ = function() {
  // for (var node = this.head_.next; node !== this.head_; node = node.next) {
  //   memFree(node);
  // }
  // memFree(dict);
};

/**
 * Insert the supplied key into the edge list and return its new node.
 * @param {libtess.DictNode} node
 * @param {!libtess.ActiveRegion} key
 * @return {!libtess.DictNode}
 */
libtess.Dict.prototype.insertBefore = function(node, key) {
  do {
    node = node.prev;
  } while (node.key !== null && !this.leq_(this.frame_, node.key, key));

  // insert the new node and update the surrounding nodes to point to it
  var newNode = new libtess.DictNode(key, node.next, node);
  node.next.prev = newNode;
  node.next = newNode;

  return newNode;
};

/**
 * Insert key into the dict and return the new node that contains it.
 * @param {!libtess.ActiveRegion} key
 * @return {!libtess.DictNode}
 */
libtess.Dict.prototype.insert = function(key) {
  // NOTE(bckenny): from a macro in dict.h/dict-list.h
  return this.insertBefore(this.head_, key);
};

/**
 * Remove node from the list.
 * @param {libtess.DictNode} node
 */
libtess.Dict.prototype.deleteNode = function(node) {
  node.next.prev = node.prev;
  node.prev.next = node.next;

  // NOTE(bckenny): nulled at callsite (sweep.deleteRegion_)
  // memFree( node );
};

/**
 * Search returns the node with the smallest key greater than or equal
 * to the given key. If there is no such key, returns a node whose
 * key is null. Similarly, max(d).getSuccessor() has a null key, etc.
 * @param {!libtess.ActiveRegion} key
 * @return {!libtess.DictNode}
 */
libtess.Dict.prototype.search = function(key) {
  var node = this.head_;

  do {
    node = node.next;
  } while (node.key !== null && !this.leq_(this.frame_, key, node.key));

  return node;
};

/**
 * Return the node with the smallest key.
 * @return {!libtess.DictNode}
 */
libtess.Dict.prototype.getMin = function() {
  // NOTE(bckenny): from a macro in dict.h/dict-list.h
  return this.head_.next;
};

// NOTE(bckenny): libtess.Dict.getMax isn't called within libtess and isn't part
// of the public API. For now, leaving in but ignoring for coverage.
/* istanbul ignore next */
/**
 * Returns the node with the greatest key.
 * @return {!libtess.DictNode}
 */
libtess.Dict.prototype.getMax = function() {
  // NOTE(bckenny): from a macro in dict.h/dict-list.h
  return this.head_.prev;
};

/* global libtess */

/**
 * A doubly-linked-list node with a libtess.ActiveRegion payload.
 * The key for this node and the next and previous nodes in the parent Dict list
 * can be provided to insert it into an existing list (or all can be omitted if
 * this is to be the founding node of the list).
 * @param {!libtess.ActiveRegion=} opt_key
 * @param {!libtess.DictNode=} opt_nextNode
 * @param {!libtess.DictNode=} opt_prevNode
 * @constructor
 * @struct
 */
libtess.DictNode = function(opt_key, opt_nextNode, opt_prevNode) {
  /**
   * The ActiveRegion key for this node, or null if the head of the list.
   * @type {libtess.ActiveRegion}
   */
  this.key = opt_key || null;

  /**
   * Link to next DictNode in parent list or to self if this is the first node.
   * @type {!libtess.DictNode}
   */
  this.next = opt_nextNode || this;

  /**
   * Link to previous DictNode in parent list or to self if this is the first
   * node.
   * @type {!libtess.DictNode}
   */
  this.prev = opt_prevNode || this;
};

/**
 * Get the key from this node.
 * @return {libtess.ActiveRegion}
 */
libtess.DictNode.prototype.getKey = function() {
  return this.key;
};

/**
 * Get the successor node to this one.
 * @return {!libtess.DictNode}
 */
libtess.DictNode.prototype.getSuccessor = function() {
  return this.next;
};

/**
 * Get the predecessor node to this one.
 * @return {!libtess.DictNode}
 */
libtess.DictNode.prototype.getPredecessor = function() {
  return this.prev;
};

/* global libtess */

// TODO(bckenny): create more javascript-y API, e.g. make gluTessEndPolygon
// async, don't require so many temp objects created

/**
 * The tesselator main class, providing the public API.
 * @constructor
 * @struct
 */
libtess.GluTesselator = function() {
  // Only initialize fields which can be changed by the api. Other fields
  // are initialized where they are used.

  /*** state needed for collecting the input data ***/

  /**
   * Tesselator state, tracking what begin/end calls have been seen.
   * @private {libtess.GluTesselator.tessState_}
   */
  this.state_ = libtess.GluTesselator.tessState_.T_DORMANT;

  /**
   * lastEdge_.org is the most recent vertex
   * @private {libtess.GluHalfEdge}
   */
  this.lastEdge_ = null;

  /**
   * stores the input contours, and eventually the tessellation itself
   * @type {libtess.GluMesh}
   */
  this.mesh = null;

  /**
   * Error callback.
   * @private {?function((libtess.errorType|libtess.gluEnum), Object=)}
   */
  this.errorCallback_ = null;

  /*** state needed for projecting onto the sweep plane ***/

  /**
   * user-specified normal (if provided)
   * @private {!Array<number>}
   */
  this.normal_ = [0, 0, 0];

  /*** state needed for the line sweep ***/

  /**
   * rule for determining polygon interior
   * @type {libtess.windingRule}
   */
  this.windingRule = libtess.windingRule.GLU_TESS_WINDING_ODD;

  /**
   * fatal error: needed combine callback
   * @type {boolean}
   */
  this.fatalError = false;

  /**
   * edge dictionary for sweep line
   * @type {libtess.Dict}
   */
  this.dict = null;
  // NOTE(bckenny): dict initialized in sweep.initEdgeDict_, removed in sweep.doneEdgeDict_

  /**
   * priority queue of vertex events
   * @type {libtess.PriorityQ}
   */
  this.pq = null;
  // NOTE(bckenny): pq initialized in sweep.initPriorityQ

  /**
   * current sweep event being processed
   * @type {libtess.GluVertex}
   */
  this.event = null;

  /**
   * Combine callback.
   * @private {?function(Array<number>, Array<Object>, Array<number>, Object=): Object}
   */
  this.combineCallback_ = null;

  /*** state needed for rendering callbacks (see render.js) ***/

  /**
   * Extract contours, not triangles
   * @private {boolean}
   */
  this.boundaryOnly_ = false;

  /**
   * Begin callback.
   * @private {?function(libtess.primitiveType, Object=)}
   */
  this.beginCallback_ = null;

  /**
   * Edge flag callback.
   * @private {?function(boolean, Object=)}
   */
  this.edgeFlagCallback_ = null;

  /**
   * Vertex callback.
   * @private {?function(Object, Object=)}
   */
  this.vertexCallback_ = null;

  /**
   * End callback.
   * @private {?function(Object=)}
   */
  this.endCallback_ = null;

  /**
   * Mesh callback.
   * @private {?function(libtess.GluMesh)}
   */
  this.meshCallback_ = null;

  /**
   * client data for current polygon
   * @private {Object}
   */
  this.polygonData_ = null;
};

/**
 * The begin/end calls must be properly nested. We keep track of the current
 * state to enforce the ordering.
 * @enum {number}
 * @private
 */
libtess.GluTesselator.tessState_ = {
  T_DORMANT: 0,
  T_IN_POLYGON: 1,
  T_IN_CONTOUR: 2
};

/**
 * Destory the tesselator object. See README.
 */
libtess.GluTesselator.prototype.gluDeleteTess = function() {
  // TODO(bckenny): This does nothing but assert that it isn't called while
  // building the polygon since we rely on GC to handle memory. *If* the public
  // API changes, this should go.
  this.requireState_(libtess.GluTesselator.tessState_.T_DORMANT);
  // memFree(tess); TODO(bckenny)
};

/**
 * Set properties for control over tesselation. See README.
 * @param {libtess.gluEnum} which [description].
 * @param {number|boolean} value [description].
 */
libtess.GluTesselator.prototype.gluTessProperty = function(which, value) {
  // TODO(bckenny): split into more setters?
  // TODO(bckenny): in any case, we can do better than this switch statement

  switch (which) {
    case libtess.gluEnum.GLU_TESS_TOLERANCE:
      // NOTE(bckenny): libtess has never supported any tolerance but 0.
      return;

    case libtess.gluEnum.GLU_TESS_WINDING_RULE:
      var windingRule = /** @type {libtess.windingRule} */(value);

      switch (windingRule) {
        case libtess.windingRule.GLU_TESS_WINDING_ODD:
        case libtess.windingRule.GLU_TESS_WINDING_NONZERO:
        case libtess.windingRule.GLU_TESS_WINDING_POSITIVE:
        case libtess.windingRule.GLU_TESS_WINDING_NEGATIVE:
        case libtess.windingRule.GLU_TESS_WINDING_ABS_GEQ_TWO:
          this.windingRule = windingRule;
          return;
        default:
      }
      break;

    case libtess.gluEnum.GLU_TESS_BOUNDARY_ONLY:
      this.boundaryOnly_ = !!value;
      return;

    default:
      this.callErrorCallback(libtess.gluEnum.GLU_INVALID_ENUM);
      return;
  }
  this.callErrorCallback(libtess.gluEnum.GLU_INVALID_VALUE);
};

/**
 * Returns tessellator property
 * @param {libtess.gluEnum} which [description].
 * @return {number|boolean} [description].
 */
libtess.GluTesselator.prototype.gluGetTessProperty = function(which) {
  // TODO(bckenny): as above, split into more getters? and improve on switch statement
  // why are these being asserted in getter but not setter?

  switch (which) {
    case libtess.gluEnum.GLU_TESS_TOLERANCE:
      return 0;

    case libtess.gluEnum.GLU_TESS_WINDING_RULE:
      var rule = this.windingRule;
      return rule;

    case libtess.gluEnum.GLU_TESS_BOUNDARY_ONLY:
      return this.boundaryOnly_;

    default:
      this.callErrorCallback(libtess.gluEnum.GLU_INVALID_ENUM);
      break;
  }
  return false;
};

/**
 * Lets the user supply the polygon normal, if known. All input data is
 * projected into a plane perpendicular to the normal before tesselation. All
 * output triangles are oriented CCW with respect to the normal (CW orientation
 * can be obtained by reversing the sign of the supplied normal). For example,
 * if you know that all polygons lie in the x-y plane, call
 * `tess.gluTessNormal(0.0, 0.0, 1.0)` before rendering any polygons.
 * @param {number} x
 * @param {number} y
 * @param {number} z
 */
libtess.GluTesselator.prototype.gluTessNormal = function(x, y, z) {
  this.normal_[0] = x;
  this.normal_[1] = y;
  this.normal_[2] = z;
};

/**
 * Specify callbacks. See README for callback descriptions. A null or undefined
 * opt_fn removes current callback.
 * @param {libtess.gluEnum} which The callback-type gluEnum value.
 * @param {?Function=} opt_fn
 */
libtess.GluTesselator.prototype.gluTessCallback = function(which, opt_fn) {
  var fn = !opt_fn ? null : opt_fn;
  // TODO(bckenny): better opt_fn typing?
  // TODO(bckenny): should add documentation that references in callback are volatile (or make a copy)

  switch (which) {
    case libtess.gluEnum.GLU_TESS_BEGIN:
    case libtess.gluEnum.GLU_TESS_BEGIN_DATA:
      this.beginCallback_ = /** @type {?function(libtess.primitiveType, Object=)} */ (fn);
      return;

    case libtess.gluEnum.GLU_TESS_EDGE_FLAG:
    case libtess.gluEnum.GLU_TESS_EDGE_FLAG_DATA:
      this.edgeFlagCallback_ = /** @type {?function(boolean, Object=)} */ (fn);
      return;

    case libtess.gluEnum.GLU_TESS_VERTEX:
    case libtess.gluEnum.GLU_TESS_VERTEX_DATA:
      this.vertexCallback_ = /** @type {?function(Object, Object=)} */ (fn);
      return;

    case libtess.gluEnum.GLU_TESS_END:
    case libtess.gluEnum.GLU_TESS_END_DATA:
      this.endCallback_ = /** @type {?function(Object=)} */ (fn);
      return;

    case libtess.gluEnum.GLU_TESS_ERROR:
    case libtess.gluEnum.GLU_TESS_ERROR_DATA:
      this.errorCallback_ = /** @type {?function((libtess.errorType|libtess.gluEnum), Object=)} */ (fn);
      return;

    case libtess.gluEnum.GLU_TESS_COMBINE:
    case libtess.gluEnum.GLU_TESS_COMBINE_DATA:
      this.combineCallback_ = /** @type {?function(Array<number>, Array<Object>, Array<number>, Object=): Object} */ (fn);
      return;

    case libtess.gluEnum.GLU_TESS_MESH:
      this.meshCallback_ = /** @type {?function(libtess.GluMesh)} */ (fn);
      return;

    default:
      this.callErrorCallback(libtess.gluEnum.GLU_INVALID_ENUM);
      return;
  }
};

/**
 * Specify a vertex and associated data. Must be within calls to
 * beginContour/endContour. See README.
 * @param {!Array<number>} coords
 * @param {Object} data
 */
libtess.GluTesselator.prototype.gluTessVertex = function(coords, data) {
  var tooLarge = false;

  // TODO(bckenny): pool allocation?
  var clamped = [0, 0, 0];

  this.requireState_(libtess.GluTesselator.tessState_.T_IN_CONTOUR);

  for (var i = 0; i < 3; ++i) {
    var x = coords[i];
    if (x < -libtess.GLU_TESS_MAX_COORD) {
      x = -libtess.GLU_TESS_MAX_COORD;
      tooLarge = true;
    }
    if (x > libtess.GLU_TESS_MAX_COORD) {
      x = libtess.GLU_TESS_MAX_COORD;
      tooLarge = true;
    }
    clamped[i] = x;
  }

  if (tooLarge) {
    this.callErrorCallback(libtess.errorType.GLU_TESS_COORD_TOO_LARGE);
  }

  this.addVertex_(clamped, data);
};

/**
 * [gluTessBeginPolygon description]
 * @param {Object} data Client data for current polygon.
 */
libtess.GluTesselator.prototype.gluTessBeginPolygon = function(data) {
  this.requireState_(libtess.GluTesselator.tessState_.T_DORMANT);

  this.state_ = libtess.GluTesselator.tessState_.T_IN_POLYGON;

  this.mesh = new libtess.GluMesh();

  this.polygonData_ = data;
};

/**
 * [gluTessBeginContour description]
 */
libtess.GluTesselator.prototype.gluTessBeginContour = function() {
  this.requireState_(libtess.GluTesselator.tessState_.T_IN_POLYGON);

  this.state_ = libtess.GluTesselator.tessState_.T_IN_CONTOUR;
  this.lastEdge_ = null;
};

/**
 * [gluTessEndContour description]
 */
libtess.GluTesselator.prototype.gluTessEndContour = function() {
  this.requireState_(libtess.GluTesselator.tessState_.T_IN_CONTOUR);
  this.state_ = libtess.GluTesselator.tessState_.T_IN_POLYGON;
};

/**
 * [gluTessEndPolygon description]
 */
libtess.GluTesselator.prototype.gluTessEndPolygon = function() {
  this.requireState_(libtess.GluTesselator.tessState_.T_IN_POLYGON);
  this.state_ = libtess.GluTesselator.tessState_.T_DORMANT;

  // Determine the polygon normal and project vertices onto the plane
  // of the polygon.
  libtess.normal.projectPolygon(this, this.normal_[0], this.normal_[1],
      this.normal_[2]);

  // computeInterior(tess) computes the planar arrangement specified
  // by the given contours, and further subdivides this arrangement
  // into regions. Each region is marked "inside" if it belongs
  // to the polygon, according to the rule given by this.windingRule.
  // Each interior region is guaranteed be monotone.
  libtess.sweep.computeInterior(this);

  if (!this.fatalError) {
    // If the user wants only the boundary contours, we throw away all edges
    // except those which separate the interior from the exterior.
    // Otherwise we tessellate all the regions marked "inside".
    // NOTE(bckenny): we know this.mesh has been initialized, so help closure out.
    var mesh = /** @type {!libtess.GluMesh} */(this.mesh);
    if (this.boundaryOnly_) {
      libtess.tessmono.setWindingNumber(mesh, 1, true);
    } else {
      libtess.tessmono.tessellateInterior(mesh);
    }

    this.mesh.checkMesh();

    if (this.beginCallback_ || this.endCallback_ || this.vertexCallback_ ||
        this.edgeFlagCallback_) {

      if (this.boundaryOnly_) {
        // output boundary contours
        libtess.render.renderBoundary(this, this.mesh);

      } else {
        // output triangles (with edge callback if one is set)
        var flagEdges = !!this.edgeFlagCallback_;
        libtess.render.renderMesh(this, this.mesh, flagEdges);
      }
    }

    if (this.meshCallback_) {
      // Throw away the exterior faces, so that all faces are interior.
      // This way the user doesn't have to check the "inside" flag,
      // and we don't need to even reveal its existence. It also leaves
      // the freedom for an implementation to not generate the exterior
      // faces in the first place.
      libtess.tessmono.discardExterior(this.mesh);
      // user wants the mesh itself
      this.meshCallback_(this.mesh);

      this.mesh = null;
      this.polygonData_ = null;
      return;
    }
  }

  libtess.mesh.deleteMesh(this.mesh);
  this.polygonData_ = null;
  this.mesh = null;
};

/**
 * Change the tesselator state.
 * @private
 * @param {libtess.GluTesselator.tessState_} state
 */
libtess.GluTesselator.prototype.requireState_ = function(state) {
  if (this.state_ !== state) {
    this.gotoState_(state);
  }
};

/**
 * Change the current tesselator state one level at a time to get to the
 * desired state. Only triggered when the API is not called in the correct order
 * so an error callback is made, however the tesselator will always attempt to
 * recover afterwards (see README).
 * @private
 * @param {libtess.GluTesselator.tessState_} newState
 */
libtess.GluTesselator.prototype.gotoState_ = function(newState) {
  while (this.state_ !== newState) {
    if (this.state_ < newState) {
      switch (this.state_) {
        case libtess.GluTesselator.tessState_.T_DORMANT:
          this.callErrorCallback(
              libtess.errorType.GLU_TESS_MISSING_BEGIN_POLYGON);
          this.gluTessBeginPolygon(null);
          break;

        case libtess.GluTesselator.tessState_.T_IN_POLYGON:
          this.callErrorCallback(
              libtess.errorType.GLU_TESS_MISSING_BEGIN_CONTOUR);
          this.gluTessBeginContour();
          break;
      }

    } else {
      switch (this.state_) {
        case libtess.GluTesselator.tessState_.T_IN_CONTOUR:
          this.callErrorCallback(
              libtess.errorType.GLU_TESS_MISSING_END_CONTOUR);
          this.gluTessEndContour();
          break;

        case libtess.GluTesselator.tessState_.T_IN_POLYGON:
          this.callErrorCallback(
              libtess.errorType.GLU_TESS_MISSING_END_POLYGON);
          // NOTE(bckenny): libtess originally reset the tesselator, even though
          // the README claims it should spit out the tessellated results at
          // this point.
          // (see http://cgit.freedesktop.org/mesa/glu/tree/src/libtess/tess.c#n180)
          this.gluTessEndPolygon();
          break;
      }
    }
  }
};

/**
 * [addVertex_ description]
 * @private
 * @param {!Array<number>} coords [description].
 * @param {Object} data [description].
 */
libtess.GluTesselator.prototype.addVertex_ = function(coords, data) {
  var e = this.lastEdge_;
  if (e === null) {
    // Make a self-loop (one vertex, one edge).
    e = libtess.mesh.makeEdge(this.mesh);
    libtess.mesh.meshSplice(e, e.sym);

  } else {
    // Create a new vertex and edge which immediately follow e
    // in the ordering around the left face.
    libtess.mesh.splitEdge(e);
    e = e.lNext;
  }

  // The new vertex is now e.org.
  e.org.data = data;
  e.org.coords[0] = coords[0];
  e.org.coords[1] = coords[1];
  e.org.coords[2] = coords[2];

  // The winding of an edge says how the winding number changes as we
  // cross from the edge''s right face to its left face.  We add the
  // vertices in such an order that a CCW contour will add +1 to
  // the winding number of the region inside the contour.
  e.winding = 1;
  e.sym.winding = -1;

  this.lastEdge_ = e;
};

/**
 * Call callback to indicate the start of a primitive, to be followed by emitted
 * vertices, if any. In libtess.js, `type` will always be `GL_TRIANGLES`.
 * @param {libtess.primitiveType} type
 */
libtess.GluTesselator.prototype.callBeginCallback = function(type) {
  if (this.beginCallback_) {
    this.beginCallback_(type, this.polygonData_);
  }
};

/**
 * Call callback to emit a vertex of the tessellated polygon.
 * @param {Object} data
 */
libtess.GluTesselator.prototype.callVertexCallback = function(data) {
  if (this.vertexCallback_) {
    this.vertexCallback_(data, this.polygonData_);
  }
};

/**
 * Call callback to indicate whether the vertices to follow begin edges which
 * lie on a polygon boundary.
 * @param {boolean} flag
 */
libtess.GluTesselator.prototype.callEdgeFlagCallback = function(flag) {
  if (this.edgeFlagCallback_) {
    this.edgeFlagCallback_(flag, this.polygonData_);
  }
};

/**
 * Call callback to indicate the end of tessellation.
 */
libtess.GluTesselator.prototype.callEndCallback = function() {
  if (this.endCallback_) {
    this.endCallback_(this.polygonData_);
  }
};

/* jscs:disable maximumLineLength */
/**
 * Call callback for combining vertices at edge intersection requiring the
 * creation of a new vertex.
 * @param {!Array<number>} coords Intersection coordinates.
 * @param {!Array<Object>} data Array of vertex data, one per edge vertices.
 * @param {!Array<number>} weight Coefficients used for the linear combination of vertex coordinates that gives coords.
 * @return {?Object} Interpolated vertex.
 */
libtess.GluTesselator.prototype.callCombineCallback = function(coords, data, weight) {
  if (this.combineCallback_) {
    return this.combineCallback_(coords, data, weight, this.polygonData_) ||
        null;
  }

  return null;
};
/* jscs:enable maximumLineLength */

/**
 * Call error callback, if specified, with errno.
 * @param {(libtess.errorType|libtess.gluEnum)} errno
 */
libtess.GluTesselator.prototype.callErrorCallback = function(errno) {
  if (this.errorCallback_) {
    this.errorCallback_(errno, this.polygonData_);
  }
};

/* global libtess */

/**
 * Each face has a pointer to the next and previous faces in the
 * circular list, and a pointer to a half-edge with this face as
 * the left face (null if this is the dummy header). There is also
 * a field "data" for client data.
 *
 * @param {libtess.GluFace=} opt_nextFace
 * @param {libtess.GluFace=} opt_prevFace
 * @constructor
 * @struct
 */
libtess.GluFace = function(opt_nextFace, opt_prevFace) {
  // TODO(bckenny): reverse order of params?

  /**
   * next face (never null)
   * @type {!libtess.GluFace}
   */
  this.next = opt_nextFace || this;

  /**
   * previous face (never NULL)
   * @type {!libtess.GluFace}
   */
  this.prev = opt_prevFace || this;

  /**
   * A half edge with this left face.
   * @type {libtess.GluHalfEdge}
   */
  this.anEdge = null;

  /**
   * room for client's data
   * @type {Object}
   */
  this.data = null;

  /**
   * This face is in the polygon interior.
   * @type {boolean}
   */
  this.inside = false;
};

/* global libtess */

/**
 * The fundamental data structure is the "half-edge". Two half-edges
 * go together to make an edge, but they point in opposite directions.
 * Each half-edge has a pointer to its mate (the "symmetric" half-edge sym),
 * its origin vertex (org), the face on its left side (lFace), and the
 * adjacent half-edges in the CCW direction around the origin vertex
 * (oNext) and around the left face (lNext). There is also a "next"
 * pointer for the global edge list (see below).
 *
 * The notation used for mesh navigation:
 *  sym   = the mate of a half-edge (same edge, but opposite direction)
 *  oNext = edge CCW around origin vertex (keep same origin)
 *  dNext = edge CCW around destination vertex (keep same dest)
 *  lNext = edge CCW around left face (dest becomes new origin)
 *  rNext = edge CCW around right face (origin becomes new dest)
 *
 * "prev" means to substitute CW for CCW in the definitions above.
 *
 * The circular edge list is special; since half-edges always occur
 * in pairs (e and e.sym), each half-edge stores a pointer in only
 * one direction. Starting at eHead and following the e.next pointers
 * will visit each *edge* once (ie. e or e.sym, but not both).
 * e.sym stores a pointer in the opposite direction, thus it is
 * always true that e.sym.next.sym.next === e.
 *
 * @param {libtess.GluHalfEdge=} opt_nextEdge
 * @constructor
 * @struct
 */
libtess.GluHalfEdge = function(opt_nextEdge) {
  // TODO(bckenny): are these the right defaults? (from gl_meshNewMesh requirements)

  /**
   * doubly-linked list (prev==sym->next)
   * @type {!libtess.GluHalfEdge}
   */
  this.next = opt_nextEdge || this;

  // TODO(bckenny): how can this be required if created in pairs? move to factory creation only?
  /**
   * same edge, opposite direction
   * @type {libtess.GluHalfEdge}
   */
  this.sym = null;

  /**
   * next edge CCW around origin
   * @type {libtess.GluHalfEdge}
   */
  this.oNext = null;

  /**
   * next edge CCW around left face
   * @type {libtess.GluHalfEdge}
   */
  this.lNext = null;

  /**
   * origin vertex (oVertex too long)
   * @type {libtess.GluVertex}
   */
  this.org = null;

  /**
   * left face
   * @type {libtess.GluFace}
   */
  this.lFace = null;

  // Internal data (keep hidden)
  // NOTE(bckenny): can't be private, though...

  /**
   * a region with this upper edge (see sweep.js)
   * @type {libtess.ActiveRegion}
   */
  this.activeRegion = null;

  /**
   * change in winding number when crossing from the right face to the left face
   * @type {number}
   */
  this.winding = 0;
};

// NOTE(bckenny): the following came from macros in mesh
// TODO(bckenny): using methods as aliases for sym connections for now.
// not sure about this approach. getters? renames?


/**
 * [rFace description]
 * @return {libtess.GluFace} [description].
 */
libtess.GluHalfEdge.prototype.rFace = function() {
  return this.sym.lFace;
};


/**
 * [dst description]
 * @return {libtess.GluVertex} [description].
 */
libtess.GluHalfEdge.prototype.dst = function() {
  return this.sym.org;
};


/**
 * [oPrev description]
 * @return {libtess.GluHalfEdge} [description].
 */
libtess.GluHalfEdge.prototype.oPrev = function() {
  return this.sym.lNext;
};


/**
 * [lPrev description]
 * @return {libtess.GluHalfEdge} [description].
 */
libtess.GluHalfEdge.prototype.lPrev = function() {
  return this.oNext.sym;
};

// NOTE(bckenny): libtess.GluHalfEdge.dPrev is called nowhere in libtess and
// isn't part of the current public API. It could be useful for mesh traversal
// and manipulation if made public, however.
/* istanbul ignore next */
/**
 * The edge clockwise around destination vertex (keep same dest).
 * @return {libtess.GluHalfEdge}
 */
libtess.GluHalfEdge.prototype.dPrev = function() {
  return this.lNext.sym;
};


/**
 * [rPrev description]
 * @return {libtess.GluHalfEdge} [description].
 */
libtess.GluHalfEdge.prototype.rPrev = function() {
  return this.sym.oNext;
};


/**
 * [dNext description]
 * @return {libtess.GluHalfEdge} [description].
 */
libtess.GluHalfEdge.prototype.dNext = function() {
  return this.rPrev().sym;
};


// NOTE(bckenny): libtess.GluHalfEdge.rNext is called nowhere in libtess and
// isn't part of the current public API. It could be useful for mesh traversal
// and manipulation if made public, however.
/* istanbul ignore next */
/**
 * The edge CCW around the right face (origin of this becomes new dest).
 * @return {libtess.GluHalfEdge}
 */
libtess.GluHalfEdge.prototype.rNext = function() {
  return this.oPrev().sym;
};

/* global libtess */

/**
 * Creates a new mesh with no edges, no vertices,
 * and no loops (what we usually call a "face").
 *
 * @constructor
 * @struct
 */
libtess.GluMesh = function() {
  /**
   * dummy header for vertex list
   * @type {libtess.GluVertex}
   */
  this.vHead = new libtess.GluVertex();

  /**
   * dummy header for face list
   * @type {libtess.GluFace}
   */
  this.fHead = new libtess.GluFace();

  /**
   * dummy header for edge list
   * @type {libtess.GluHalfEdge}
   */
  this.eHead = new libtess.GluHalfEdge();

  /**
   * and its symmetric counterpart
   * @type {libtess.GluHalfEdge}
   */
  this.eHeadSym = new libtess.GluHalfEdge();

  // TODO(bckenny): better way to pair these?
  this.eHead.sym = this.eHeadSym;
  this.eHeadSym.sym = this.eHead;
};


// TODO(bckenny): #ifndef NDEBUG
/**
 * Checks mesh for self-consistency.
 */
libtess.GluMesh.prototype.checkMesh = function() {
  if (!libtess.DEBUG) {
    return;
  }

  var fHead = this.fHead;
  var vHead = this.vHead;
  var eHead = this.eHead;

  var e;

  // faces
  var f;
  var fPrev = fHead;
  for (fPrev = fHead; (f = fPrev.next) !== fHead; fPrev = f) {
    e = f.anEdge;
    do {
      e = e.lNext;
    } while (e !== f.anEdge);
  }

  // vertices
  var v;
  var vPrev = vHead;
  for (vPrev = vHead; (v = vPrev.next) !== vHead; vPrev = v) {
    e = v.anEdge;
    do {
      e = e.oNext;
    } while (e !== v.anEdge);
  }

  // edges
  var ePrev = eHead;
  for (ePrev = eHead; (e = ePrev.next) !== eHead; ePrev = e) {
  }
};

/* global libtess */

/**
 * Each vertex has a pointer to next and previous vertices in the
 * circular list, and a pointer to a half-edge with this vertex as
 * the origin (null if this is the dummy header). There is also a
 * field "data" for client data.
 * @param {libtess.GluVertex=} opt_nextVertex Optional reference to next vertex in the vertex list.
 * @param {libtess.GluVertex=} opt_prevVertex Optional reference to previous vertex in the vertex list.
 * @constructor
 * @struct
 */
libtess.GluVertex = function(opt_nextVertex, opt_prevVertex) {
  /**
   * Next vertex (never null).
   * @type {!libtess.GluVertex}
   */
  this.next = opt_nextVertex || this;

  /**
   * Previous vertex (never null).
   * @type {!libtess.GluVertex}
   */
  this.prev = opt_prevVertex || this;

  /**
   * A half-edge with this origin.
   * @type {libtess.GluHalfEdge}
   */
  this.anEdge = null;

  /**
   * The client's data.
   * @type {Object}
   */
  this.data = null;

  /**
   * The vertex location in 3D.
   * @type {!Array.<number>}
   */
  this.coords = [0, 0, 0];
  // TODO(bckenny): we may want to rethink coords, either eliminate (using s
  // and t and user data) or index into contiguous storage?

  /**
   * Component of projection onto the sweep plane.
   * @type {number}
   */
  this.s = 0;

  /**
   * Component of projection onto the sweep plane.
   * @type {number}
   */
  this.t = 0;

  /**
   * Handle to allow deletion from priority queue, or 0 if not yet inserted into
   * queue.
   * @type {libtess.PQHandle}
   */
  this.pqHandle = 0;
};

/* global libtess */

/**
 * A priority queue of vertices, ordered by libtess.geom.vertLeq, implemented
 * with a sorted array. Used for initial insertion of vertices (see
 * libtess.sweep.initPriorityQ_), sorted once, then it uses an internal
 * libtess.PriorityQHeap for any subsequently created vertices from
 * intersections.
 * @constructor
 * @struct
 */
libtess.PriorityQ = function() {
  /**
   * An unordered list of vertices that have been inserted in the queue, with
   * null in empty slots.
   * @private {Array<libtess.GluVertex>}
   */
  this.verts_ = [];

  /**
   * Array of indices into this.verts_, sorted by vertLeq over the addressed
   * vertices.
   * @private {Array<number>}
   */
  this.order_ = null;

  /**
   * The size of this queue, not counting any vertices stored in heap_.
   * @private {number}
   */
  this.size_ = 0;

  /**
   * Indicates that the queue has been initialized via init. If false, inserts
   * are fast insertions at the end of the verts_ array. If true, the verts_
   * array is sorted and subsequent inserts are done in the heap.
   * @private {boolean}
   */
  this.initialized_ = false;

  /**
   * A priority queue heap, used for faster insertions of vertices after verts_
   * has been sorted.
   * @private {libtess.PriorityQHeap}
   */
  this.heap_ = new libtess.PriorityQHeap();
};

/**
 * Release major storage memory used by priority queue.
 */
libtess.PriorityQ.prototype.deleteQ = function() {
  // TODO(bckenny): could instead clear most of these.
  this.heap_ = null;
  this.order_ = null;
  this.verts_ = null;
  // NOTE(bckenny): nulled at callsite (sweep.donePriorityQ_)
};

/**
 * Sort vertices by libtess.geom.vertLeq. Must be called before any method other
 * than insert is called to ensure correctness when removing or querying.
 */
libtess.PriorityQ.prototype.init = function() {
  // TODO(bckenny): reuse. in theory, we don't have to empty this, as access is
  // dictated by this.size_, but array.sort doesn't know that
  this.order_ = [];

  // Create an array of indirect pointers to the verts, so that
  // the handles we have returned are still valid.
  // TODO(bckenny): valid for when? it appears we can just store indexes into
  // verts_, but what did this mean?
  for (var i = 0; i < this.size_; i++) {
    this.order_[i] = i;
  }

  // sort the indirect pointers in descending order of the verts themselves
  // TODO(bckenny): make sure it's ok that verts[a] === verts[b] returns 1
  // TODO(bckenny): unstable sort means we may get slightly different polys in
  // different browsers, but only when passing in equal points
  // TODO(bckenny): make less awkward closure?
  var comparator = (function(verts) {
    return function(a, b) {
      return libtess.geom.vertLeq(verts[a], verts[b]) ? 1 : -1;
    };
  })(this.verts_);
  this.order_.sort(comparator);

  this.initialized_ = true;
  this.heap_.init();

  // NOTE(bckenny): debug assert of ordering of the verts_ array.
  if (libtess.DEBUG) {
    var p = 0;
    var r = p + this.size_ - 1;
    for (i = p; i < r; ++i) {
    }
  }
};

/**
 * Insert a vertex into the priority queue. Returns a PQHandle to refer to it,
 * which will never be 0.
 * @param {libtess.GluVertex} vert
 * @return {libtess.PQHandle}
 */
libtess.PriorityQ.prototype.insert = function(vert) {
  // NOTE(bckenny): originally returned LONG_MAX as alloc failure signal. no
  // longer does.
  if (this.initialized_) {
    return this.heap_.insert(vert);
  }

  var curr = this.size_++;

  this.verts_[curr] = vert;

  // Negative handles index the sorted array.
  return -(curr + 1);
};

/**
 * Removes the minimum vertex from the queue and returns it. If the queue is
 * empty, null will be returned.
 * @return {libtess.GluVertex}
 */
libtess.PriorityQ.prototype.extractMin = function() {
  if (this.size_ === 0) {
    return this.heap_.extractMin();
  }

  var sortMin = this.verts_[this.order_[this.size_ - 1]];
  if (!this.heap_.isEmpty()) {
    var heapMin = this.heap_.minimum();
    if (libtess.geom.vertLeq(heapMin, sortMin)) {
      return this.heap_.extractMin();
    }
  }

  do {
    --this.size_;
  } while (this.size_ > 0 && this.verts_[this.order_[this.size_ - 1]] === null);

  return sortMin;
};

/**
 * Returns the minimum vertex in the queue. If the queue is empty, null will be
 * returned.
 * @return {libtess.GluVertex}
 */
libtess.PriorityQ.prototype.minimum = function() {
  if (this.size_ === 0) {
    return this.heap_.minimum();
  }

  var sortMin = this.verts_[this.order_[this.size_ - 1]];
  if (!this.heap_.isEmpty()) {
    var heapMin = this.heap_.minimum();
    if (libtess.geom.vertLeq(heapMin, sortMin)) {
      return heapMin;
    }
  }

  return sortMin;
};

/**
 * Remove vertex with handle removeHandle from queue.
 * @param {libtess.PQHandle} removeHandle
 */
libtess.PriorityQ.prototype.remove = function(removeHandle) {
  if (removeHandle >= 0) {
    this.heap_.remove(removeHandle);
    return;
  }
  removeHandle = -(removeHandle + 1);

  this.verts_[removeHandle] = null;
  while (this.size_ > 0 && this.verts_[this.order_[this.size_ - 1]] === null) {
    --this.size_;
  }
};

/* global libtess */

/**
 * A priority queue of vertices, ordered by libtess.geom.vertLeq, implemented
 * with a binary heap. Used only within libtess.PriorityQ for prioritizing
 * vertices created by intersections (see libtess.sweep.checkForIntersect_).
 * @constructor
 * @struct
 */
libtess.PriorityQHeap = function() {
  /**
   * The heap itself. Active nodes are stored in the range 1..size, with the
   * minimum at 1. Each node stores only an index into verts_ and handles_.
   * @private {!Array<number>}
   */
  this.heap_ = libtess.PriorityQHeap.reallocNumeric_([0],
      libtess.PriorityQHeap.INIT_SIZE_ + 1);

  /**
   * An unordered list of vertices in the heap, with null in empty slots.
   * @private {!Array<libtess.GluVertex>}
   */
  this.verts_ = [null, null];

  /**
   * An unordered list of indices mapping vertex handles into the heap. An entry
   * at index i will map the vertex at i in verts_ to its place in the heap
   * (i.e. heap_[handles_[i]] === i).
   * Empty slots below size_ are a free list chain starting at freeList_.
   * @private {!Array<number>}
   */
  this.handles_ = [0, 0];

  /**
   * The size of the queue.
   * @private {number}
   */
  this.size_ = 0;

  /**
   * The queue's current allocated space.
   * @private {number}
   */
  this.max_ = libtess.PriorityQHeap.INIT_SIZE_;

  /**
   * The index of the next free hole in the verts_ array. That slot in handles_
   * has the next index in the free list. If there are no holes, freeList_ === 0
   * and a new vertex must be appended to the list.
   * @private {libtess.PQHandle}
   */
  this.freeList_ = 0;

  /**
   * Indicates that the heap has been initialized via init. If false, inserts
   * are fast insertions at the end of a list. If true, all inserts will now be
   * correctly ordered in the queue before returning.
   * @private {boolean}
   */
  this.initialized_ = false;

  // Point the first index at the first (currently null) vertex.
  this.heap_[1] = 1;
};

/**
 * The initial allocated space for the queue.
 * @const
 * @private {number}
 */
libtess.PriorityQHeap.INIT_SIZE_ = 32;

/**
 * Allocate a numeric index array of size size. oldArray's contents are copied
 * to the beginning of the new array. The rest of the array is filled with
 * zeroes.
 * @private
 * @param {!Array<number>} oldArray
 * @param {number} size
 * @return {!Array<number>}
 */
libtess.PriorityQHeap.reallocNumeric_ = function(oldArray, size) {
  var newArray = new Array(size);

  // NOTE(bckenny): V8 likes this significantly more than simply growing the
  // array element-by-element or expanding the existing array all at once, so,
  // for now, emulating realloc.
  for (var index = 0; index < oldArray.length; index++) {
    newArray[index] = oldArray[index];
  }

  for (; index < size; index++) {
    newArray[index] = 0;
  }

  return newArray;
};

/**
 * Initializing ordering of the heap. Must be called before any method other
 * than insert is called to ensure correctness when removing or querying.
 */
libtess.PriorityQHeap.prototype.init = function() {
  // This method of building a heap is O(n), rather than O(n lg n).
  for (var i = this.size_; i >= 1; --i) {
    // TODO(bckenny): since init is called before anything is inserted (see
    // PriorityQ.init), this will always be empty. Better to lazily init?
    this.floatDown_(i);
  }

  this.initialized_ = true;
};

/**
 * Insert a new vertex into the heap.
 * @param {libtess.GluVertex} vert The vertex to insert.
 * @return {libtess.PQHandle} A handle that can be used to remove the vertex.
 */
libtess.PriorityQHeap.prototype.insert = function(vert) {
  var endIndex = ++this.size_;

  // If the heap overflows, double its size.
  if ((endIndex * 2) > this.max_) {
    this.max_ *= 2;

    this.handles_ = libtess.PriorityQHeap.reallocNumeric_(this.handles_,
        this.max_ + 1);
  }

  var newVertSlot;
  if (this.freeList_ === 0) {
    // No free slots, append vertex.
    newVertSlot = endIndex;
  } else {
    // Put vertex in free slot, update freeList_ to next free slot.
    newVertSlot = this.freeList_;
    this.freeList_ = this.handles_[this.freeList_];
  }

  this.verts_[newVertSlot] = vert;
  this.handles_[newVertSlot] = endIndex;
  this.heap_[endIndex] = newVertSlot;

  if (this.initialized_) {
    this.floatUp_(endIndex);
  }
  return newVertSlot;
};

/**
 * @return {boolean} Whether the heap is empty.
 */
libtess.PriorityQHeap.prototype.isEmpty = function() {
  return this.size_ === 0;
};

/**
 * Returns the minimum vertex in the heap. If the heap is empty, null will be
 * returned.
 * @return {libtess.GluVertex}
 */
libtess.PriorityQHeap.prototype.minimum = function() {
  return this.verts_[this.heap_[1]];
};

/**
 * Removes the minimum vertex from the heap and returns it. If the heap is
 * empty, null will be returned.
 * @return {libtess.GluVertex}
 */
libtess.PriorityQHeap.prototype.extractMin = function() {
  var heap = this.heap_;
  var verts = this.verts_;
  var handles = this.handles_;

  var minHandle = heap[1];
  var minVertex = verts[minHandle];

  if (this.size_ > 0) {
    // Replace min with last vertex.
    heap[1] = heap[this.size_];
    handles[heap[1]] = 1;

    // Clear min vertex and put slot at front of freeList_.
    verts[minHandle] = null;
    handles[minHandle] = this.freeList_;
    this.freeList_ = minHandle;

    // Restore heap.
    if (--this.size_ > 0) {
      this.floatDown_(1);
    }
  }

  return minVertex;
};

/**
 * Remove vertex with handle removeHandle from heap.
 * @param {libtess.PQHandle} removeHandle
 */
libtess.PriorityQHeap.prototype.remove = function(removeHandle) {
  var heap = this.heap_;
  var verts = this.verts_;
  var handles = this.handles_;

  var heapIndex = handles[removeHandle];

  // Replace with last vertex.
  heap[heapIndex] = heap[this.size_];
  handles[heap[heapIndex]] = heapIndex;

  // Restore heap.
  if (heapIndex <= --this.size_) {
    if (heapIndex <= 1) {
      this.floatDown_(heapIndex);
    } else {
      var vert = verts[heap[heapIndex]];
      var parentVert = verts[heap[heapIndex >> 1]];
      if (libtess.geom.vertLeq(parentVert, vert)) {
        this.floatDown_(heapIndex);
      } else {
        this.floatUp_(heapIndex);
      }
    }
  }

  // Clear vertex and put slot at front of freeList_.
  verts[removeHandle] = null;
  handles[removeHandle] = this.freeList_;
  this.freeList_ = removeHandle;
};

/**
 * Restore heap by moving the vertex at index in the heap downwards to a valid
 * slot.
 * @private
 * @param {libtess.PQHandle} index
 */
libtess.PriorityQHeap.prototype.floatDown_ = function(index) {
  var heap = this.heap_;
  var verts = this.verts_;
  var handles = this.handles_;

  var currIndex = index;
  var currHandle = heap[currIndex];
  for (;;) {
    // The children of node i are nodes 2i and 2i+1.
    var childIndex = currIndex << 1;
    if (childIndex < this.size_) {
      // Set child to the index of the child with the minimum vertex.
      if (libtess.geom.vertLeq(verts[heap[childIndex + 1]],
          verts[heap[childIndex]])) {
        childIndex = childIndex + 1;
      }
    }

    var childHandle = heap[childIndex];
    if (childIndex > this.size_ ||
        libtess.geom.vertLeq(verts[currHandle], verts[childHandle])) {
      // Heap restored.
      heap[currIndex] = currHandle;
      handles[currHandle] = currIndex;
      return;
    }

    // Swap current node and child; repeat from childIndex.
    heap[currIndex] = childHandle;
    handles[childHandle] = currIndex;
    currIndex = childIndex;
  }
};

/**
 * Restore heap by moving the vertex at index in the heap upwards to a valid
 * slot.
 * @private
 * @param {libtess.PQHandle} index
 */
libtess.PriorityQHeap.prototype.floatUp_ = function(index) {
  var heap = this.heap_;
  var verts = this.verts_;
  var handles = this.handles_;

  var currIndex = index;
  var currHandle = heap[currIndex];
  for (;;) {
    // The parent of node i is node floor(i/2).
    var parentIndex = currIndex >> 1;
    var parentHandle = heap[parentIndex];

    if (parentIndex === 0 ||
        libtess.geom.vertLeq(verts[parentHandle], verts[currHandle])) {
      // Heap restored.
      heap[currIndex] = currHandle;
      handles[currHandle] = currIndex;
      return;
    }

    // Swap current node and parent; repeat from parentIndex.
    heap[currIndex] = parentHandle;
    handles[parentHandle] = currIndex;
    currIndex = parentIndex;
  }
};

/* global libtess */

// TODO(bckenny): apparently only visible outside of sweep for debugging routines.
// find out if we can hide

/**
 * For each pair of adjacent edges crossing the sweep line, there is
 * an ActiveRegion to represent the region between them. The active
 * regions are kept in sorted order in a dynamic dictionary. As the
 * sweep line crosses each vertex, we update the affected regions.
 * @constructor
 * @struct
 */
libtess.ActiveRegion = function() {
  // TODO(bckenny): I *think* eUp and nodeUp could be passed in as constructor params

  /**
   * The upper edge of the region, directed right to left
   * @type {libtess.GluHalfEdge}
   */
  this.eUp = null;

  /**
   * Dictionary node corresponding to eUp edge.
   * @type {libtess.DictNode}
   */
  this.nodeUp = null;

  /**
   * Used to determine which regions are inside the polygon.
   * @type {number}
   */
  this.windingNumber = 0;

  /**
   * Whether this region is inside the polygon.
   * @type {boolean}
   */
  this.inside = false;

  /**
   * Marks fake edges at t = +/-infinity.
   * @type {boolean}
   */
  this.sentinel = false;

  /**
   * Marks regions where the upper or lower edge has changed, but we haven't
   * checked whether they intersect yet.
   * @type {boolean}
   */
  this.dirty = false;

  /**
   * marks temporary edges introduced when we process a "right vertex" (one
   * without any edges leaving to the right)
   * @type {boolean}
   */
  this.fixUpperEdge = false;
};

/**
 * Returns the ActiveRegion below this one.
 * @return {libtess.ActiveRegion}
 */
libtess.ActiveRegion.prototype.regionBelow = function() {
  return this.nodeUp.getPredecessor().getKey();
};

/**
 * Returns the ActiveRegion above this one.
 * @return {libtess.ActiveRegion}
 */
libtess.ActiveRegion.prototype.regionAbove = function() {
  return this.nodeUp.getSuccessor().getKey();
};

/* global libtess, module */

/**
 * node.js export for non-compiled source
 */
if (typeof module !== 'undefined') {
  module.exports = libtess;
}
;
define("util/libtess", function(){});

/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports AreaMeasurer
 */

define('util/measure/AreaMeasurer',[
        '../../geom/Angle',
        '../../error/ArgumentError',
        '../../geom/Location',
        '../Logger',
        './MeasurerUtils',
        '../../geom/Sector',
        '../../geom/Vec3',
        '../libtess'
    ],
    function (Angle,
              ArgumentError,
              Location,
              Logger,
              MeasurerUtils,
              Sector,
              Vec3,
              libtessDummy) {
        'use strict';

        /**
         * Utility class to compute approximations of projected and surface (terrain following) area on a globe.
         *
         * <p>To properly compute surface area the measurer must be provided with a list of positions that describe a
         * closed path - one which last position is equal to the first.</p>
         *
         * <p>Segments which are longer then the current maxSegmentLength will be subdivided along lines following the
         * current pathType - WorldWind.LINEAR, WorldWind.RHUMB_LINE or WorldWind.GREAT_CIRCLE.</p>
         *
         * <p>Projected or non terrain following area is computed in a sinusoidal projection which is equivalent or
         * equal area.
         * Surface or terrain following area is approximated by sampling the path bounding sector with square cells
         * along a grid. Cells which center is inside the path  have their area estimated and summed according to the
         * overall slope at the cell south-west corner.</p>
         *
         * @alias AreaMeasurer
         * @constructor
         * @param {WorldWindow} wwd The WorldWindow associated with AreaMeasurer.
         * @throws {ArgumentError} If the specified WorldWindow is null or undefined.
         */
        var AreaMeasurer = function (wwd) {

            if (!wwd) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "AreaMeasurer", "constructor", "missingWorldWindow"));
            }

            this.wwd = wwd;

            // Private. Sampling grid max rows or cols
            this.DEFAULT_AREA_SAMPLING_STEPS = 32;

            // Private. Documentation is with the defined property below.
            this._areaTerrainSamplingSteps = this.DEFAULT_AREA_SAMPLING_STEPS;

            // Private. Documentation is with the defined property below.
            this._maxSegmentLength = 100e3;

            // Private. A list of positions with no segment longer then maxLength and elevations following terrain or not.
            this.subdividedPositions = null;

            // Private.
            this.vecZ = new Vec3(0, 0, 1);

            // Private. Reusable Location.
            this.scratchLocation = new Location(0, 0);

        };

        Object.defineProperties(AreaMeasurer.prototype, {
            /**
             * The sampling grid maximum number of rows or columns for terrain following surface area approximation.
             * @type {Number}
             * @memberof AreaMeasurer.prototype
             */
            areaTerrainSamplingSteps: {
                get: function () {
                    return this._areaTerrainSamplingSteps;
                },
                set: function (value) {
                    this._areaTerrainSamplingSteps = value;
                }
            },

            /**
             * The maximum length a segment can have before being subdivided along a line following the current pathType.
             * @type {Number}
             * @memberof AreaMeasurer.prototype
             */
            maxSegmentLength: {
                get: function () {
                    return this._maxSegmentLength;
                },
                set: function (value) {
                    this._maxSegmentLength = value;
                }
            }
        });

        /**
         * Get the sampling grid maximum number of rows or columns for terrain following surface area approximation.
         *
         * @param {Position[]} positions A list of positions describing a polygon
         * @param {Boolean} followTerrain If true, the computed length will account for terrain deformations as if
         * someone was walking along that path
         * @param {String} pathType One of WorldWind.LINEAR, WorldWind.RHUMB_LINE or WorldWind.GREAT_CIRCLE
         *
         * @return {Number} area in square meters
         */
        AreaMeasurer.prototype.getArea = function (positions, followTerrain, pathType) {
            var globe = this.wwd.globe;
            if (followTerrain) {
                return this.computeSurfaceAreaSampling(globe, positions, pathType);
            }
            return this.computeProjectedAreaGeometry(globe, positions, pathType);
        };

        /**
         * Sample the path bounding sector with square cells which area are approximated according to the surface normal
         * at the cell south-west corner.
         *
         * @param {Globe} globe
         * @param {Position[]} positions
         * @param {String} pathType One of WorldWind.LINEAR, WorldWind.RHUMB_LINE or WorldWind.GREAT_CIRCLE
         *
         * @return {Number} area in square meters
         */
        AreaMeasurer.prototype.computeSurfaceAreaSampling = function (globe, positions, pathType) {
            var sector = new Sector(0, 0, 0, 0);
            sector.setToBoundingSector(positions);

            // Subdivide long segments if needed
            this.subdividedPositions = MeasurerUtils.subdividePositions(globe, positions, true, pathType,
                this._maxSegmentLength);

            // Sample the bounding sector with cells about the same length in side - squares
            var steps = Math.max(this.DEFAULT_AREA_SAMPLING_STEPS, this._areaTerrainSamplingSteps);
            var deltaLatRadians = sector.deltaLatitude() * Angle.DEGREES_TO_RADIANS;
            var deltaLonRadians = sector.deltaLongitude() * Angle.DEGREES_TO_RADIANS;
            var stepsRadians = Math.max(deltaLatRadians / steps, deltaLonRadians / steps);
            var latSteps = Math.round(deltaLatRadians / stepsRadians);
            var lonSteps = Math.round(deltaLonRadians / stepsRadians *
                Math.cos(sector.centroidLatitude() * Angle.DEGREES_TO_RADIANS));
            var latStepRadians = deltaLatRadians / latSteps;
            var lonStepRadians = deltaLonRadians / lonSteps;

            var area = 0;
            for (var i = 0; i < latSteps; i++) {
                var lat = sector.minLatitude * Angle.DEGREES_TO_RADIANS + latStepRadians * i;
                // Compute this latitude row cells area
                var radius = globe.radiusAt((lat + latStepRadians / 2) * Angle.RADIANS_TO_DEGREES, sector.centroidLongitude());
                var cellWidth = lonStepRadians * radius * Math.cos(lat + latStepRadians / 2);
                var cellHeight = latStepRadians * radius;
                var cellArea = cellWidth * cellHeight;

                for (var j = 0; j < lonSteps; j++) {
                    var lon = sector.minLongitude * Angle.DEGREES_TO_RADIANS + lonStepRadians * j;
                    var minLat = lat * Angle.RADIANS_TO_DEGREES;
                    var maxLat = (lat + latStepRadians) * Angle.RADIANS_TO_DEGREES;
                    var minLon = lon * Angle.RADIANS_TO_DEGREES;
                    var maxLon = (lon + lonStepRadians) * Angle.RADIANS_TO_DEGREES;
                    var cellSector = new Sector(minLat, maxLat, minLon, maxLon);
                    var isLocationInside = MeasurerUtils.isLocationInside(cellSector.centroid(this.scratchLocation),
                        this.subdividedPositions);
                    if (isLocationInside) {
                        // Compute suface area using terrain normal in SW corner
                        // Corners elevation
                        var eleSW = globe.elevationAtLocation(minLat, minLon);
                        var eleSE = globe.elevationAtLocation(minLat, maxLon);
                        var eleNW = globe.elevationAtLocation(maxLat, minLon);

                        // Compute normal
                        var vx = new Vec3(cellWidth, 0, eleSE - eleSW);
                        var vy = new Vec3(0, cellHeight, eleNW - eleSW);
                        vx.normalize();
                        vy.normalize();
                        var normalSW = vx.cross(vy).normalize(); // point toward positive Z

                        // Compute slope factor
                        var tan = Math.tan(MeasurerUtils.angleBetweenVectors(this.vecZ, normalSW));
                        var slopeFactor = Math.sqrt(1 + tan * tan);

                        // Add cell area
                        area += (cellArea * slopeFactor);
                    }
                }
            }

            return area;
        };

        /**
         * Tessellate the path in lat-lon space, then sum each triangle area.
         *
         * @param {Globe} globe
         * @param {Position[]} positions
         * @param {String} pathType One of WorldWind.LINEAR, WorldWind.RHUMB_LINE or WorldWind.GREAT_CIRCLE
         *
         * @return {Number} area in square meters
         */
        AreaMeasurer.prototype.computeProjectedAreaGeometry = function (globe, positions, pathType) {

            // Subdivide long segments if needed
            this.subdividedPositions = MeasurerUtils.subdividePositions(globe, positions, false, pathType,
                this._maxSegmentLength);

            // First: tessellate polygon
            var verticesCount = this.subdividedPositions.length;
            var firstPos = this.subdividedPositions[0];
            var lastPos = this.subdividedPositions[verticesCount - 1];
            if (firstPos.equals(lastPos)) {
                verticesCount--;
            }

            var verts = [];
            var idx = 0;
            for (var i = 0; i < verticesCount; i++) {
                var pos = this.subdividedPositions[i];
                verts[idx++] = pos.longitude * Angle.DEGREES_TO_RADIANS;
                verts[idx++] = pos.latitude * Angle.DEGREES_TO_RADIANS;
                verts[idx++] = 0;
            }

            var triangles = this.tessellatePolygon(verticesCount, verts);

            // Second: sum triangles area
            var area = 0;
            var triangleCount = triangles.length / 9;
            for (i = 0; i < triangleCount; i++) {
                idx = i * 9;
                var triangle = [
                    triangles[idx + 0], triangles[idx + 1], triangles[idx + 2],
                    triangles[idx + 3], triangles[idx + 4], triangles[idx + 5],
                    triangles[idx + 6], triangles[idx + 7], triangles[idx + 8]
                ];
                area += this.computeTriangleProjectedArea(globe, triangle);
            }

            return area;

        };

        /**
         * Compute triangle area in a sinusoidal projection centered at the triangle center.
         * Note sinusoidal projection is equivalent or equal area.
         *
         * @param {Globe} globe
         * @param {Number[]} verts A list of 9 positions in radians describing a triangle
         *
         * @return {Number} area in square meters
         */
        AreaMeasurer.prototype.computeTriangleProjectedArea = function (globe, verts) {
            // http://www.mathopenref.com/coordtrianglearea.html
            var ax = verts[0];
            var ay = verts[1];
            var bx = verts[3];
            var by = verts[4];
            var cx = verts[6];
            var cy = verts[7];

            var area = Math.abs(
                ax * (by - cy) +
                bx * (cy - ay) +
                cx * (ay - by)
            );
            area /= 2;

            var centerLon = (ax + bx + cx) / 3;
            var centerLat = (ay + by + cy) / 3;

            // Apply globe radius at triangle center and scale down area according to center latitude cosine
            var radius = globe.radiusAt(centerLat * Angle.RADIANS_TO_DEGREES, centerLon * Angle.RADIANS_TO_DEGREES);
            area *= Math.cos(centerLat) * radius * radius; // Square meter

            return area;
        };

        /**
         * Tessellate a Polygon
         *
         * @param {Number} count the number of vertices
         * @param {Number[]} vertices A list of positions in radians
         *
         * @return {Number[]} a list of tessellated vertices
         */
        AreaMeasurer.prototype.tessellatePolygon = function (count, vertices) {
            var tess = new libtess.GluTesselator();
            var triangles = [];
            var coords;

            tess.gluTessCallback(libtess.gluEnum.GLU_TESS_BEGIN, function (prim) {
                if (prim !== libtess.primitiveType.GL_TRIANGLES) {
                    Logger.logMessage(Logger.LEVEL_WARNING, "AreaMeasurer", "tessellatePolygon",
                        "Tessellation error, primitive is not TRIANGLES.");
                }
            });

            tess.gluTessCallback(libtess.gluEnum.GLU_TESS_VERTEX_DATA, function (data, tris) {
                tris.push(data[0]);
                tris.push(data[1]);
                tris.push(data[2]);
            });

            //prevents triangle fans and strips
            tess.gluTessCallback(libtess.gluEnum.GLU_TESS_EDGE_FLAG, function () {
            });

            tess.gluTessCallback(libtess.gluEnum.GLU_TESS_ERROR, function (errno) {
                Logger.logMessage(Logger.LEVEL_WARNING, "AreaMeasurer", "tessellatePolygon",
                    "Tessellation error " + errno + ".");
            });

            // Tessellate the polygon.
            tess.gluTessBeginPolygon(triangles);
            tess.gluTessBeginContour();
            for (var i = 0; i < count; i++) {
                coords = vertices.slice(3 * i, 3 * i + 3);
                tess.gluTessVertex(coords, coords);
            }
            tess.gluTessEndContour();
            tess.gluTessEndPolygon();

            return triangles;
        };

        return AreaMeasurer;

    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports AbsentResourceList
 */
define('util/AbsentResourceList',[],
    function () {
        "use strict";

        /**
         * Constructs an absent resource list.
         * @alias AbsentResourceList
         * @constructor
         * @classdesc Provides a collection to keep track of resources whose retrieval failed and when retrieval
         * may be tried again. Applications typically do not use this class directly.
         * @param {Number} maxTrys The number of attempts to make before the resource is marked as absent.
         * @param {Number} minCheckInterval The amount of time to wait between attempts, in milliseconds.
         * @constructor
         */
        var AbsentResourceList = function (maxTrys, minCheckInterval) {

            /**
             * The number  of attempts to make before the resource is marked as absent.
             * @type {Number}
             */
            this.maxTrys = maxTrys;

            /**
             * The amount of time to wait before each attempt.
             * @type {Number}
             */
            this.minCheckInterval = minCheckInterval;

            /**
             * The amount of time, in milliseconds, beyond which retrieval attempts should again be allowed.
             * When this time has elapsed from the most recent failed attempt the number of trys attempted is
             * reset to 0. This prevents the resource from being permanently blocked.
             * @type {number}
             * @default 60,000 milliseconds (one minute)
             */
            this.tryAgainInterval = 60e3; // 60 seconds

            this.possiblyAbsent = {};
        };

        /**
         * Indicates whether a specified resource is marked as absent.
         * @param {String} resourceId The resource identifier.
         * @returns {Boolean} true if the resource is marked as absent, otherwise false.
         */
        AbsentResourceList.prototype.isResourceAbsent = function (resourceId) {
            var entry = this.possiblyAbsent[resourceId];

            if (!entry) {
                return false;
            }

            if (entry.permanent) {
                return true;
            }

            var timeSinceLastMark = Date.now() - entry.timeOfLastMark;

            if (timeSinceLastMark > this.tryAgainInterval) {
                delete this.possiblyAbsent[resourceId];
                return false;
            }

            return timeSinceLastMark < this.minCheckInterval || entry.numTrys > this.maxTrys;
        };

        /**
         * Marks a resource attempt as having failed. This increments the number-of-tries counter and sets the time
         * of the last attempt. When this method has been called [this.maxTrys]{@link AbsentResourceList#maxTrys}
         * times the resource is marked as absent until this absent resource list's
         * [try-again-interval]{@link AbsentResourceList#tryAgainInterval} is reached.
         * @param {String} resourceId The resource identifier.
         */
        AbsentResourceList.prototype.markResourceAbsent = function (resourceId) {
            var entry = this.possiblyAbsent[resourceId];

            if (!entry) {
                entry = {
                    timeOfLastMark: Date.now(),
                    numTrys: 0
                };
                this.possiblyAbsent[resourceId] = entry;
            }

            entry.numTrys = entry.numTrys + 1;
            entry.timeOfLastMark = Date.now();
        };

        /**
         * Marks a resource attempt as having failed permanently. No attempt will ever again be made to retrieve
         * the resource.
         * @param {String} resourceId The resource identifier.
         */
        AbsentResourceList.prototype.markResourceAbsentPermanently = function (resourceId) {
            var entry = this.possiblyAbsent[resourceId];

            if (!entry) {
                entry = {
                    timeOfLastMark: Date.now(),
                    numTrys: 0
                };
                this.possiblyAbsent[resourceId] = entry;
            }

            entry.numTrys = entry.numTrys + 1;
            entry.timeOfLastMark = Date.now();
            entry.permanent = true;
        };

        /**
         * Removes the specified resource from this absent resource list. Call this method when retrieval attempts
         * succeed.
         * @param {String} resourceId The resource identifier.
         */
        AbsentResourceList.prototype.unmarkResourceAbsent = function (resourceId) {
            var entry = this.possiblyAbsent[resourceId];

            if (entry) {
                delete this.possiblyAbsent[resourceId];
            }
        };

        return AbsentResourceList;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports ElevationCoverage
 */
define('globe/ElevationCoverage',['../error/ArgumentError',
        '../util/Logger',
        '../geom/Sector'],
    function (ArgumentError,
              Logger,
              Sector) {
        "use strict";

        /**
         * Constructs an ElevationCoverage
         * @alias ElevationCoverage
         * @constructor
         * @classdesc When used directly and not through a subclass, this class represents an elevation coverage
         * whose elevations are zero at all locations.
         * @param {Number} resolution The resolution of the coverage, in degrees. (To compute degrees from
         * meters, divide the number of meters by the globe's radius to obtain radians and convert the result to degrees.)
         * @throws {ArgumentError} If the resolution argument is null, undefined, or zero.
         */
        var ElevationCoverage = function (resolution) {
            if (!resolution) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "ElevationCoverage", "constructor",
                        "missingResolution"));
            }

            /**
             * Indicates the last time this coverage changed, in milliseconds since midnight Jan 1, 1970.
             * @type {Number}
             * @readonly
             * @default Date.now() at construction
             */
            this.timestamp = Date.now();

            /**
             * Indicates this coverage's display name.
             * @type {String}
             * @default "Coverage"
             */
            this.displayName = "Coverage";

            /**
             * Indicates whether or not to use this coverage.
             * @type {Boolean}
             * @default true
             */
            this._enabled = true;

            /**
             * The resolution of this coverage in degrees.
             * @type {Number}
             */
            this.resolution = resolution;

            /**
             * The sector this coverage spans.
             * @type {Sector}
             * @readonly
             */
            this.coverageSector = Sector.FULL_SPHERE;
        };

        Object.defineProperties(ElevationCoverage.prototype, {
            /**
             * Indicates whether or not to use this coverage.
             * @type {Boolean}
             * @default true
             */
            enabled: {
                get: function () {
                    return this._enabled;
                },
                set: function (value) {
                    this._enabled = value;
                    this.timestamp = Date.now();
                }
            }
        });

        /**
         * Returns the minimum and maximum elevations for the elevation tile(s) enclosing a specified sector. Note that the elevations 
         * returned may not accurately portray the elevation extremes solely within the specified sector. 
         * 
         * @param {Sector} sector The sector for which to determine extreme elevations.
         * @param {Number[]} result An array in which to return the requested minimum and maximum elevations.
         * @returns {Boolean} true if the coverage completely fills the sector with data, false otherwise.
         * @throws {ArgumentError} If any argument is null or undefined
         */
        ElevationCoverage.prototype.minAndMaxElevationsForSector = function (sector, result) {
            if (!sector) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "ElevationCoverage", "minAndMaxElevationsForSector", "missingSector"));
            }

            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "ElevationCoverage", "minAndMaxElevationsForSector", "missingResult"));
            }

            if (result[0] > 0) { // min elevation
                result[0] = 0;
            }

            if (result[1] < 0) { // max elevation
                result[1] = 0;
            }

            return true;
        };

        /**
         * Returns the elevation at a specified location.
         * @param {Number} latitude The location's latitude in degrees.
         * @param {Number} longitude The location's longitude in degrees.
         * @returns {Number} The elevation at the specified location, in meters. Returns null if the location is
         * outside the coverage area of this coverage.
         */
        ElevationCoverage.prototype.elevationAtLocation = function (latitude, longitude) {
            return 0;
        };

        /**
         * Returns the elevations at locations within a specified sector.
         * @param {Sector} sector The sector for which to determine the elevations.
         * @param {Number} numLat The number of latitudinal sample locations within the sector.
         * @param {Number} numLon The number of longitudinal sample locations within the sector.
         * @param {Number[]} result An array in which to return the requested elevations.
         * @returns {Boolean} true if the result array was completely filled with elevation data, false otherwise.
         * @throws {ArgumentError} If the specified sector or result array is null or undefined, or if either of the
         * specified numLat or numLon values is less than one.
         */
        ElevationCoverage.prototype.elevationsForGrid = function (sector, numLat, numLon, result) {
            if (!sector) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "ElevationCoverage", "elevationsForGrid", "missingSector"));
            }

            if (numLat <= 0 || numLon <= 0) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "ElevationCoverage",
                    "elevationsForGrid", "numLat or numLon is less than 1"));
            }

            if (!result || result.length < numLat * numLon) {
                throw new ArgumentError(Logger.logMessage(Logger.LEVEL_SEVERE, "ElevationCoverage",
                    "elevationsForGrid", "missingArray"));
            }

            for (var i = 0, len = result.length; i < len; i++) {
                result[i] = 0;
            }

            return true;
        };

        return ElevationCoverage;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports ElevationImage
 */
define('globe/ElevationImage',[
        '../error/ArgumentError',
        '../util/Logger',
        '../util/WWMath'
    ],
    function (ArgumentError,
              Logger,
              WWMath) {
        "use strict";

        /**
         * Constructs an elevation image.
         * @alias ElevationImage
         * @constructor
         * @classdesc Holds elevation values for an elevation tile.
         * This class is typically not used directly by applications.
         * @param {Sector} sector The sector spanned by this elevation image.
         * @param {Number} imageWidth The number of longitudinal sample points in this elevation image.
         * @param {Number} imageHeight The number of latitudinal sample points in this elevation image.
         * @throws {ArgumentError} If the sector is null or undefined
         */
        var ElevationImage = function (sector, imageWidth, imageHeight) {
            if (!sector) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "ElevationImage", "constructor", "missingSector"));
            }

            /**
             * The sector spanned by this elevation image.
             * @type {Sector}
             * @readonly
             */
            this.sector = sector;

            /**
             * The number of longitudinal sample points in this elevation image.
             * @type {Number}
             * @readonly
             */
            this.imageWidth = imageWidth;

            /**
             * The number of latitudinal sample points in this elevation image.
             * @type {Number}
             * @readonly
             */
            this.imageHeight = imageHeight;

            /**
             * The size in bytes of this elevation image.
             * @type {number}
             * @readonly
             */
            this.size = this.imageWidth * this.imageHeight;

            /**
             * Internal use only
             * false if the entire image consists of NO_DATA values, true otherwise.
             * @ignore
             */
            this.hasData = true;

            /**
             * Internal use only
             * true if any pixel in the image has a NO_DATA value, false otherwise.
             * @ignore
             */
            this.hasMissingData = false;
        };

        /**
         * Internal use only
         * The value that indicates a pixel contains no data.
         * TODO: This will eventually need to become an instance property
         * @ignore
         */
        ElevationImage.NO_DATA = 0;

        /**
         * Internal use only
         * Returns true if a set of elevation pixels represents the NO_DATA value.
         * @ignore
         */
        ElevationImage.isNoData = function (x0y0, x1y0, x0y1, x1y1) {
            // TODO: Change this logic once proper NO_DATA value handling is in place.
            var v = ElevationImage.NO_DATA;
            return x0y0 === v &&
                x1y0 === v &&
                x0y1 === v &&
                x1y1 === v;
        };

        /**
         * Returns the pixel value at a specified coordinate in this elevation image. The coordinate origin is the
         * image's lower left corner, so (0, 0) indicates the lower left pixel and (imageWidth-1, imageHeight-1)
         * indicates the upper right pixel. This returns 0 if the coordinate indicates a pixel outside of this elevation
         * image.
         * @param x The pixel's X coordinate.
         * @param y The pixel's Y coordinate.
         * @returns {Number} The pixel value at the specified coordinate in this elevation image.
         * Returns 0 if the coordinate indicates a pixel outside of this elevation image.
         */
        ElevationImage.prototype.pixel = function (x, y) {
            if (x < 0 || x >= this.imageWidth) {
                return 0;
            }

            if (y < 0 || y >= this.imageHeight) {
                return 0;
            }

            y = this.imageHeight - y - 1; // flip the y coordinate origin to the lower left corner
            return this.imageData[x + y * this.imageWidth];
        };

        /**
         * Returns the elevation at a specified geographic location.
         * @param {Number} latitude The location's latitude.
         * @param {Number} longitude The location's longitude.
         * @returns {Number} The elevation at the specified location.
         */
        ElevationImage.prototype.elevationAtLocation = function (latitude, longitude) {
            var maxLat = this.sector.maxLatitude,
                minLon = this.sector.minLongitude,
                deltaLat = this.sector.deltaLatitude(),
                deltaLon = this.sector.deltaLongitude(),
                x = (this.imageWidth - 1) * (longitude - minLon) / deltaLon,
                y = (this.imageHeight - 1) * (maxLat - latitude) / deltaLat,
                x0 = Math.floor(WWMath.clamp(x, 0, this.imageWidth - 1)),
                x1 = Math.floor(WWMath.clamp(x0 + 1, 0, this.imageWidth - 1)),
                y0 = Math.floor(WWMath.clamp(y, 0, this.imageHeight - 1)),
                y1 = Math.floor(WWMath.clamp(y0 + 1, 0, this.imageHeight - 1)),
                pixels = this.imageData,
                x0y0 = pixels[x0 + y0 * this.imageWidth],
                x1y0 = pixels[x1 + y0 * this.imageWidth],
                x0y1 = pixels[x0 + y1 * this.imageWidth],
                x1y1 = pixels[x1 + y1 * this.imageWidth],
                xf = x - x0,
                yf = y - y0;

            if (ElevationImage.isNoData(x0y0, x1y0, x0y1, x1y1)) {
                return NaN;
            }

            return (1 - xf) * (1 - yf) * x0y0 +
                xf * (1 - yf) * x1y0 +
                (1 - xf) * yf * x0y1 +
                xf * yf * x1y1;
        };

        /**
         * Returns elevations for a specified sector.
         * @param {Sector} sector The sector for which to return the elevations.
         * @param {Number} numLat The number of sample points in the longitudinal direction.
         * @param {Number} numLon The number of sample points in the latitudinal direction.
         * @param {Number[]} result An array in which to return the computed elevations.
         * @throws {ArgumentError} If either the specified sector or result argument is null or undefined, or if the
         * specified number of sample points in either direction is less than 1.
         */
        ElevationImage.prototype.elevationsForGrid = function (sector, numLat, numLon, result) {
            if (!sector) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "ElevationImage", "elevationsForGrid", "missingSector"));
            }

            if (numLat < 1 || numLon < 1) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "ElevationImage", "elevationsForGrid",
                        "The specified number of sample points is less than 1."));
            }

            if (!result) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "ElevationImage", "elevationsForGrid", "missingResult"));
            }

            var minLatSelf = this.sector.minLatitude,
                maxLatSelf = this.sector.maxLatitude,
                minLonSelf = this.sector.minLongitude,
                maxLonSelf = this.sector.maxLongitude,
                deltaLatSelf = maxLatSelf - minLatSelf,
                deltaLonSelf = maxLonSelf - minLonSelf,
                minLat = sector.minLatitude,
                maxLat = sector.maxLatitude,
                minLon = sector.minLongitude,
                maxLon = sector.maxLongitude,
                deltaLat = (maxLat - minLat) / (numLat > 1 ? numLat - 1 : 1),
                deltaLon = (maxLon - minLon) / (numLon > 1 ? numLon - 1 : 1),
                lat, lon,
                i, j, index = 0,
                pixels = this.imageData;

            for (j = 0, lat = minLat; j < numLat; j += 1, lat += deltaLat) {
                if (j === numLat - 1) {
                    lat = maxLat; // explicitly set the last lat to the max latitude to ensure alignment
                }

                if (lat >= minLatSelf && lat <= maxLatSelf) {
                    // Image y-coordinate of the specified location, given an image origin in the top-left corner.
                    var y = (this.imageHeight - 1) * (maxLatSelf - lat) / deltaLatSelf,
                        y0 = Math.floor(WWMath.clamp(y, 0, this.imageHeight - 1)),
                        y1 = Math.floor(WWMath.clamp(y0 + 1, 0, this.imageHeight - 1)),
                        yf = y - y0;

                    for (i = 0, lon = minLon; i < numLon; i += 1, lon += deltaLon) {
                        if (i === numLon - 1) {
                            lon = maxLon; // explicitly set the last lon to the max longitude to ensure alignment
                        }

                        if (lon >= minLonSelf && lon <= maxLonSelf && isNaN(result[index])) {
                            // Image x-coordinate of the specified location, given an image origin in the top-left corner.
                            var x = (this.imageWidth - 1) * (lon - minLonSelf) / deltaLonSelf,
                                x0 = Math.floor(WWMath.clamp(x, 0, this.imageWidth - 1)),
                                x1 = Math.floor(WWMath.clamp(x0 + 1, 0, this.imageWidth - 1)),
                                xf = x - x0;

                            var x0y0 = pixels[x0 + y0 * this.imageWidth],
                                x1y0 = pixels[x1 + y0 * this.imageWidth],
                                x0y1 = pixels[x0 + y1 * this.imageWidth],
                                x1y1 = pixels[x1 + y1 * this.imageWidth];

                            if (ElevationImage.isNoData(x0y0, x1y0, x0y1, x1y1)) {
                                result[index] = NaN;
                            }
                            else {
                                result[index] = (1 - xf) * (1 - yf) * x0y0 +
                                    xf * (1 - yf) * x1y0 +
                                    (1 - xf) * yf * x0y1 +
                                    xf * yf * x1y1;
                            }
                        }

                        index++;
                    }
                } else {
                    index += numLon; // skip this row
                }
            }
        };

        /**
         * Returns the minimum and maximum elevations within a specified sector.
         * @param {Sector} sector The sector of interest. If null or undefined, the minimum and maximum elevations
         * for the sector associated with this tile are returned.
         * @returns {Number[]} An array containing the minimum and maximum elevations within the specified sector,
         * or null if the specified sector does not include this elevation image's coverage sector or the image is filled with
         * NO_DATA values.
         */
        ElevationImage.prototype.minAndMaxElevationsForSector = function (sector) {
            if (!this.hasData) {
                return null;
            }

            var result = [];
            if (!sector) { // the sector is this sector
                result[0] = this.minElevation;
                result[1] = this.maxElevation;
            } else if (sector.contains(this.sector)) { // The specified sector completely contains this image; return the image min and max.
                if (result[0] > this.minElevation) {
                    result[0] = this.minElevation;
                }

                if (result[1] < this.maxElevation) {
                    result[1] = this.maxElevation;
                }
            } else { // The specified sector intersects a portion of this image; compute the min and max from intersecting pixels.
                var maxLatSelf = this.sector.maxLatitude,
                    minLonSelf = this.sector.minLongitude,
                    deltaLatSelf = this.sector.deltaLatitude(),
                    deltaLonSelf = this.sector.deltaLongitude(),
                    minLatOther = sector.minLatitude,
                    maxLatOther = sector.maxLatitude,
                    minLonOther = sector.minLongitude,
                    maxLonOther = sector.maxLongitude;

                // Image coordinates of the specified sector, given an image origin in the top-left corner. We take the floor and
                // ceiling of the min and max coordinates, respectively, in order to capture all pixels that would contribute to
                // elevations computed for the specified sector in a call to elevationsForSector.
                var minY = Math.floor((this.imageHeight - 1) * (maxLatSelf - maxLatOther) / deltaLatSelf),
                    maxY = Math.ceil((this.imageHeight - 1) * (maxLatSelf - minLatOther) / deltaLatSelf),
                    minX = Math.floor((this.imageWidth - 1) * (minLonOther - minLonSelf) / deltaLonSelf),
                    maxX = Math.ceil((this.imageWidth - 1) * (maxLonOther - minLonSelf) / deltaLonSelf);

                minY = WWMath.clamp(minY, 0, this.imageHeight - 1);
                maxY = WWMath.clamp(maxY, 0, this.imageHeight - 1);
                minX = WWMath.clamp(minX, 0, this.imageWidth - 1);
                maxX = WWMath.clamp(maxX, 0, this.imageWidth - 1);

                var pixels = this.imageData,
                    min = Number.MAX_VALUE,
                    max = -min;

                for (var y = minY; y <= maxY; y++) {
                    for (var x = minX; x <= maxX; x++) {
                        var p = pixels[Math.floor(x + y * this.imageWidth)];
                        if (min > p) {
                            min = p;
                        }

                        if (max < p) {
                            max = p;
                        }
                    }
                }

                if (result[0] > min) {
                    result[0] = min;
                }

                if (result[1] < max) {
                    result[1] = max;
                }
            }

            return result;
        };

        /**
         * Determines the minimum and maximum elevations within this elevation image and stores those values within
         * this object. See [minAndMaxElevationsForSector]{@link ElevationImage#minAndMaxElevationsForSector}
         */
        ElevationImage.prototype.findMinAndMaxElevation = function () {
            this.hasData = false;
            this.hasMissingData = false;

            if (this.imageData && (this.imageData.length > 0)) {
                this.minElevation = Number.MAX_VALUE;
                this.maxElevation = -Number.MAX_VALUE;

                var pixels = this.imageData,
                    pixelCount = this.imageWidth * this.imageHeight;

                for (var i = 0; i < pixelCount; i++) {
                    var p = pixels[i];
                    if (p !== ElevationImage.NO_DATA) {
                        this.hasData = true;
                        if (this.minElevation > p) {
                            this.minElevation = p;
                        }

                        if (this.maxElevation < p) {
                            this.maxElevation = p;
                        }
                    } else {
                        this.hasMissingData = true;
                    }
                }
            }

            if (!this.hasData) {
                this.minElevation = 0;
                this.maxElevation = 0;
            }
        };

        return ElevationImage;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports GeoTiffConstants
 */
define('formats/geotiff/GeoTiffConstants',[],
    function () {
        "use strict";

        /**
         * Provides all of the GeoTIFF tag constants.
         * @alias GeoTiff
         * @constructor
         * @classdesc Contains all of the TIFF tags that are used to store GeoTIFF information of any type.
         */
        var GeoTiffConstants = {
            /**
             * An object containing all GeoTiff specific tags.
             * @memberof GeoTiff
             * @type {Object}
             */
            Tag: {
                'MODEL_PIXEL_SCALE': 33550,
                'MODEL_TRANSFORMATION': 34264,
                'MODEL_TIEPOINT': 33922,
                'GEO_KEY_DIRECTORY': 34735,
                'GEO_DOUBLE_PARAMS': 34736,
                'GEO_ASCII_PARAMS': 34737,

                //gdal
                'GDAL_NODATA': 42113,
                'GDAL_METADATA': 42112
            },

            /**
             * An object containing all GeoTiff specific keys.
             * @memberof GeoTiff
             * @type {Object}
             */
            Key: {
                // GeoTIFF Configuration Keys
                'GTModelTypeGeoKey': 1024,
                'GTRasterTypeGeoKey': 1025,
                'GTCitationGeoKey': 1026,

                // Geographic CS Parameter Keys
                'GeographicTypeGeoKey': 2048,
                'GeogCitationGeoKey': 2049,
                'GeogGeodeticDatumGeoKey': 2050,
                'GeogPrimeMeridianGeoKey': 2051,
                'GeogLinearUnitsGeoKey': 2052,
                'GeogLinearUnitSizeGeoKey': 2053,
                'GeogAngularUnitsGeoKey': 2054,
                'GeogAngularUnitSizeGeoKey': 2055,
                'GeogEllipsoidGeoKey': 2056,
                'GeogSemiMajorAxisGeoKey': 2057,
                'GeogSemiMinorAxisGeoKey': 2058,
                'GeogInvFlatteningGeoKey': 2059,
                'GeogAzimuthUnitsGeoKey': 2060,
                'GeogPrimeMeridianLongGeoKey': 2061,
                'GeogTOWGS84GeoKey': 2062,

                // Projected CS Parameter Keys
                'ProjectedCSTypeGeoKey': 3072,
                'PCSCitationGeoKey': 3073,
                'ProjectionGeoKey': 3074,
                'ProjCoordTransGeoKey': 3075,
                'ProjLinearUnitsGeoKey': 3076,
                'ProjLinearUnitSizeGeoKey': 3077,
                'ProjStdParallel1GeoKey': 3078,
                'ProjStdParallel2GeoKey': 3079,
                'ProjNatOriginLongGeoKey': 3080,
                'ProjNatOriginLatGeoKey': 3081,
                'ProjFalseEastingGeoKey': 3082,
                'ProjFalseNorthingGeoKey': 3083,
                'ProjFalseOriginLongGeoKey': 3084,
                'ProjFalseOriginLatGeoKey': 3085,
                'ProjFalseOriginEastingGeoKey': 3086,
                'ProjFalseOriginNorthingGeoKey': 3087,
                'ProjCenterLongGeoKey': 3088,
                'ProjCenterLatGeoKey': 3089,
                'ProjCenterEastingGeoKey': 3090,
                'ProjCenterNorthingGeoKey': 3091,
                'ProjScaleAtNatOriginGeoKey': 3092,
                'ProjScaleAtCenterGeoKey': 3093,
                'ProjAzimuthAngleGeoKey': 3094,
                'ProjStraightVertPoleLongGeoKey': 3095,

                // Vertical CS Keys
                'VerticalCSTypeGeoKey': 4096,
                'VerticalCitationGeoKey': 4097,
                'VerticalDatumGeoKey': 4098,
                'VerticalUnitsGeoKey': 4099
            }
        };

        return GeoTiffConstants;
    }
);
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports GeoTiffKeyEntry
 */
define('formats/geotiff/GeoTiffKeyEntry',[
        '../../error/AbstractError',
        '../../error/ArgumentError',
        './GeoTiffConstants',
        '../../util/Logger'
    ],
    function (AbstractError,
              ArgumentError,
              GeoTiffConstants,
              Logger) {
        "use strict";

        /**
         * Constructs a geotiff KeyEntry. Applications typically do not call this constructor. It is called
         * by {@link GeoTiffReader} as GeoTIFF geo keys are read.
         * @alias GeoTiffKeyEntry
         * @constructor
         * @classdesc Contains the data associated with a GeoTIFF KeyEntry. Each KeyEntry is modeled on the
         * "TiffIFDEntry" format of the TIFF directory header.
         * @param {Number} keyId Gives the key-ID value of the Key (identical in function
         * to TIFF tag ID, but completely independent of TIFF tag-space).
         * @param {Number} tiffTagLocation Indicates which TIFF tag contains the value(s) of the Key.
         * @param {Number} count Indicates the number of values in this key.
         * @param {Number} valueOffset  Indicates the index offset into the TagArray indicated by tiffTagLocation.
         * @throws {ArgumentError} If either the specified keyId, tiffTagLocation, count or valueOffset
         * are null or undefined.
         */
        var GeoTiffKeyEntry = function (keyId, tiffTagLocation, count, valueOffset) {
            if (!keyId) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "GeoTiffKeyEntry", "constructor", "missingKeyId"));
            }

            if (tiffTagLocation === null || tiffTagLocation === undefined) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "GeoTiffKeyEntry", "constructor", "missingTiffTagLocation"));
            }

            if (!count) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "GeoTiffKeyEntry", "constructor", "missingCount"));
            }

            if (valueOffset === null || valueOffset === undefined) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "GeoTiffKeyEntry", "constructor", "missingValueOffset"));
            }

            // Documented in defineProperties below.
            this._keyId = keyId;

            // Documented in defineProperties below.
            this._tiffTagLocation = tiffTagLocation;

            // Documented in defineProperties below.
            this._count = count;

            // Documented in defineProperties below.
            this._valueOffset = valueOffset;
        };

        Object.defineProperties(GeoTiffKeyEntry.prototype, {

            /**
             * The key-ID value of the Key as specified to this GeoTiffKeyEntry's constructor.
             * @memberof GeoTiffKeyEntry.prototype
             * @type {Number}
             * @readonly
             */
            keyId: {
                get: function () {
                    return this._keyId;
                }
            },

            /**
             * The location of the TIFF tag as specified to this GeoTiffKeyEntry's constructor.
             * @memberof GeoTiffKeyEntry.prototype
             * @type {Number}
             * @readonly
             */
            tiffTagLocation: {
                get: function () {
                    return this._tiffTagLocation;
                }
            },

            /**
             * The number of values in the key as specified to this GeoTiffKeyEntry's constructor.
             * @memberof GeoTiffKeyEntry.prototype
             * @type {Number}
             * @readonly
             */
            count: {
                get: function () {
                    return this._count;
                }
            },

            /**
             * The index offset into the TagArray as specified to this GeoTiffKeyEntry's constructor.
             * @memberof GeoTiffKeyEntry.prototype
             * @type {Number}
             * @readonly
             */
            valueOffset: {
                get: function () {
                    return this._valueOffset;
                }
            }
        });

        /**
         * Get the value of a GeoKey.
         * @returns {Number}
         */
        GeoTiffKeyEntry.prototype.getGeoKeyValue = function (geoDoubleParamsValue, geoAsciiParamsValue) {
            var keyValue;

            if (this.tiffTagLocation === 0){
                keyValue = this.valueOffset
            }
            else if (this.tiffTagLocation === GeoTiffConstants.Tag.GEO_ASCII_PARAMS) {
                var retVal = "";

                if (geoAsciiParamsValue){
                    for (var i=this.valueOffset; i < this.count - 1; i++){
                        retVal += geoAsciiParamsValue[i];
                    }
                    if (geoAsciiParamsValue[this.count - 1] != '|'){
                        retVal += geoAsciiParamsValue[this.count - 1];
                    }

                    keyValue = retVal;
                }
            }
            else if (this.tiffTagLocation === GeoTiffConstants.Tag.GEO_DOUBLE_PARAMS) {
                if (geoDoubleParamsValue){
                    keyValue = geoDoubleParamsValue[this.valueOffset];
                }
            }

            return keyValue;
        };

        return GeoTiffKeyEntry;
    }
);

/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports GeoTiffMetadata
 */
define('formats/geotiff/GeoTiffMetadata',[
    ],
    function () {
        "use strict";

        /**
         * Provides GeoTIFF metadata.
         * @alias GeoTiffMetadata
         * @constructor
         * @classdesc Contains all of the TIFF and GeoTIFF metadata for a geotiff file.
         */
        var GeoTiffMetadata = function () {

            // Documented in defineProperties below.
            this._bitsPerSample = null;

            // Documented in defineProperties below.
            this._colorMap = null;

            // Documented in defineProperties below.
            this._compression = null;

            // Documented in defineProperties below.
            this._extraSamples = null;

            // Documented in defineProperties below.
            this._imageDescription = null;

            // Documented in defineProperties below.
            this._imageLength = null;

            // Documented in defineProperties below.
            this._imageWidth = null;

            // Documented in defineProperties below.
            this._maxSampleValue = null;

            // Documented in defineProperties below.
            this._minSampleValue = null;

            // Documented in defineProperties below.
            this._orientation = 0;

            // Documented in defineProperties below.
            this._photometricInterpretation = null;

            // Documented in defineProperties below.
            this._planarConfiguration = null;

            // Documented in defineProperties below.
            this._resolutionUnit = null;

            // Documented in defineProperties below.
            this._rowsPerStrip = null;

            // Documented in defineProperties below.
            this._samplesPerPixel = null;

            // Documented in defineProperties below.
            this._sampleFormat = null;

            // Documented in defineProperties below.
            this._software = null;

            // Documented in defineProperties below.
            this._stripByteCounts = null;

            // Documented in defineProperties below.
            this._stripOffsets = null;

            // Documented in defineProperties below.
            this._tileByteCounts = null;

            // Documented in defineProperties below.
            this._tileOffsets = null;

            // Documented in defineProperties below.
            this._tileLength = null;

            // Documented in defineProperties below.
            this._tileWidth = null;

            // Documented in defineProperties below.
            this._xResolution = null;

            // Documented in defineProperties below.
            this._yResolution = null;

            // Documented in defineProperties below.
            this._geoAsciiParams = null;

            // Documented in defineProperties below.
            this._geoDoubleParams = null;

            // Documented in defineProperties below.
            this._geoKeyDirectory = null;

            // Documented in defineProperties below.
            this._modelPixelScale = null;

            // Documented in defineProperties below.
            this._modelTiepoint = null;

            // Documented in defineProperties below.
            this._modelTransformation = null;

            // Documented in defineProperties below.
            this._noData = null;

            // Documented in defineProperties below.
            this._metaData = null;

            // Documented in defineProperties below.
            this._bbox = null;

            // Documented in defineProperties below.
            this._gtModelTypeGeoKey = null;

            // Documented in defineProperties below.
            this._gtRasterTypeGeoKey = null;

            // Documented in defineProperties below.
            this._gtCitationGeoKey = null;

            // Documented in defineProperties below.
            this._geographicTypeGeoKey = null;

            // Documented in defineProperties below.
            this._geogCitationGeoKey = null;

            // Documented in defineProperties below.
            this._geogAngularUnitsGeoKey = null;

            // Documented in defineProperties below.
            this._geogAngularUnitSizeGeoKey = null;

            // Documented in defineProperties below.
            this._geogSemiMajorAxisGeoKey = null;

            // Documented in defineProperties below.
            this._geogInvFlatteningGeoKey = null;

            // Documented in defineProperties below.
            this._projectedCSType = null;

            // Documented in defineProperties below.
            this._projLinearUnits = null;
        };

        Object.defineProperties(GeoTiffMetadata.prototype, {

            /**
             * Contains the number of bits per component.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number[]}
             */
            bitsPerSample: {
                get: function () {
                    return this._bitsPerSample;
                },

                set: function(value){
                    this._bitsPerSample = value;
                }
            },

            /**
             * Defines a Red-Green-Blue color map (often called a lookup table) for palette color images.
             * In a palette-color image, a pixel value is used to index into an RGB-lookup table.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number[]}
             */
            colorMap: {
                get: function () {
                    return this._colorMap;
                },

                set: function(value){
                    this._colorMap = value;
                }
            },

            /**
             * Contains the compression type of geotiff data.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            compression: {
                get: function () {
                    return this._compression;
                },

                set: function(value){
                    this._compression = value;
                }
            },

            /**
             * Contains the description of extra components.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number[]}
             */
            extraSamples: {
                get: function () {
                    return this._extraSamples;
                },

                set: function(value){
                    this._extraSamples = value;
                }
            },

            /**
             * Contains the image description.
             * @memberof GeoTiffMetadata.prototype
             * @type {String}
             */
            imageDescription: {
                get: function () {
                    return this._imageDescription;
                },

                set: function(value){
                    this._imageDescription = value;
                }
            },

            /**
             * Contains the number of rows in the image.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            imageLength: {
                get: function () {
                    return this._imageLength;
                },

                set: function(value){
                    this._imageLength = value;
                }
            },

            /**
             * Contains the number of columns in the image.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            imageWidth: {
                get: function () {
                    return this._imageWidth;
                },

                set: function(value){
                    this._imageWidth = value;
                }
            },

            /**
             * Contains the maximum component value used.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            maxSampleValue: {
                get: function () {
                    return this._maxSampleValue;
                },

                set: function(value){
                    this._maxSampleValue = value;
                }
            },

            /**
             * Contains the minimum component value used.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            minSampleValue: {
                get: function () {
                    return this._minSampleValue;
                },

                set: function(value){
                    this._minSampleValue = value;
                }
            },

            /**
             * Contains the orientation of the image with respect to the rows and columns.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            orientation: {
                get: function () {
                    return this._orientation;
                },

                set: function(value){
                    this._orientation = value;
                }
            },

            /**
             * Contains the photometric interpretation type of the geotiff file.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            photometricInterpretation: {
                get: function () {
                    return this._photometricInterpretation;
                },

                set: function(value){
                    this._photometricInterpretation = value;
                }
            },

            /**
             * Contains the planar configuration type of the geotiff file.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            planarConfiguration: {
                get: function () {
                    return this._planarConfiguration;
                },

                set: function(value){
                    this._planarConfiguration = value;
                }
            },

            /**
             * Contains the unit of measurement for XResolution and YResolution. The specified values are:
             * <ul>
             *     <li>1 = No absolute unit of measurement</li>
             *     <li>2 = Inch</li>
             *     <li>3 = Centimeter</li>
             * </ul>
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            resolutionUnit: {
                get: function () {
                    return this._resolutionUnit;
                },

                set: function(value){
                    this._resolutionUnit = value;
                }
            },

            /**
             * Contains the number of rows per strip.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            rowsPerStrip: {
                get: function () {
                    return this._rowsPerStrip;
                },

                set: function(value){
                    this._rowsPerStrip = value;
                }
            },

            /**
             * Contains the number of components per pixel.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            samplesPerPixel: {
                get: function () {
                    return this._samplesPerPixel;
                },

                set: function(value){
                    this._samplesPerPixel = value;
                }
            },

            /**
             * This field specifies how to interpret each data sample in a pixel. Possible values are:
             * <ul>
             *     <li>unsigned integer data</li>
             *     <li>two's complement signed integer data</li>
             *     <li>IEEE floating point data</li>
             *     <li>undefined data format</li>
             * </ul>
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            sampleFormat: {
                get: function () {
                    return this._sampleFormat;
                },

                set: function(value){
                    this._sampleFormat = value;
                }
            },

            software: {
                get: function () {
                    return this._software;
                },

                set: function(value){
                    this._software = value;
                }
            },

            /**
             * Contains the number of bytes in that strip after any compression, for each strip.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number[]}
             */
            stripByteCounts: {
                get: function () {
                    return this._stripByteCounts;
                },

                set: function(value){
                    this._stripByteCounts = value;
                }
            },

            /**
             * Contains the byte offset of that strip, for each strip.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number[]}
             */
            stripOffsets: {
                get: function () {
                    return this._stripOffsets;
                },

                set: function(value){
                    this._stripOffsets = value;
                }
            },

            tileByteCounts: {
                get: function () {
                    return this._tileByteCounts;
                },

                set: function(value){
                    this._tileByteCounts = value;
                }
            },

            /**
             * Contains the byte offset of that tile, for each tile.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number[]}
             */
            tileOffsets: {
                get: function () {
                    return this._tileOffsets;
                },

                set: function(value){
                    this._tileOffsets = value;
                }
            },

            tileLength: {
                get: function () {
                    return this._tileLength;
                },

                set: function(value){
                    this._tileLength = value;
                }
            },

            tileWidth: {
                get: function () {
                    return this._tileWidth;
                },

                set: function(value){
                    this._tileWidth = value;
                }
            },

            //geotiff
            /**
             * Contains all of the ASCII valued GeoKeys, referenced by the GeoKeyDirectoryTag.
             * @memberof GeoTiffMetadata.prototype
             * @type {String[]}
             */
            geoAsciiParams: {
                get: function () {
                    return this._geoAsciiParams;
                },

                set: function(value){
                    this._geoAsciiParams = value;
                }
            },

            /**
             * Contains all of the DOUBLE valued GeoKeys, referenced by the GeoKeyDirectoryTag.
             * @memberof GeoTiffMetadata.prototype
             * @type {Nmber[]}
             */
            geoDoubleParams: {
                get: function () {
                    return this._geoDoubleParams;
                },

                set: function(value){
                    this._geoDoubleParams = value;
                }
            },

            /**
             * Contains the values of GeoKeyDirectoryTag.
             * @memberof GeoTiffMetadata.prototype
             * @type {Nmber[]}
             */
            geoKeyDirectory: {
                get: function () {
                    return this._geoKeyDirectory;
                },

                set: function(value){
                    this._geoKeyDirectory = value;
                }
            },

            /**
             * Contains the values of ModelPixelScaleTag. The ModelPixelScaleTag tag may be used to specify the size
             * of raster pixel spacing in the model space units, when the raster space can be embedded in the model
             * space coordinate system without rotation
             * @memberof GeoTiffMetadata.prototype
             * @type {Nmber[]}
             */
            modelPixelScale: {
                get: function () {
                    return this._modelPixelScale;
                },

                set: function(value){
                    this._modelPixelScale = value;
                }
            },

            /**
             * Stores raster->model tiepoint pairs in the order ModelTiepointTag = (...,I,J,K, X,Y,Z...),
             * where (I,J,K) is the point at location (I,J) in raster space with pixel-value K,
             * and (X,Y,Z) is a vector in model space.
             * @memberof GeoTiffMetadata.prototype
             * @type {Nmber[]}
             */
            modelTiepoint: {
                get: function () {
                    return this._modelTiepoint;
                },

                set: function(value){
                    this._modelTiepoint = value;
                }
            },

            /**
             * Contains the information that may be used to specify the transformation matrix between the raster space
             * (and its dependent pixel-value space) and the model space.
             * @memberof GeoTiffMetadata.prototype
             * @type {Nmber[]}
             */
            modelTransformation: {
                get: function () {
                    return this._modelTransformation;
                },

                set: function(value){
                    this._modelTransformation = value;
                }
            },

            /**
             * Contains the NODATA value.
             * @memberof GeoTiffMetadata.prototype
             * @type {String}
             */
            noData: {
                get: function () {
                    return this._noData;
                },

                set: function(value){
                    this._noData = value;
                }
            },

            /**
             * Contains the METADATA value.
             * @memberof GeoTiffMetadata.prototype
             * @type {String}
             */
            metaData: {
                get: function () {
                    return this._metaData;
                },

                set: function(value){
                    this._metaData = value;
                }
            },

            /**
             * Contains the extent of the geotiff.
             * @memberof GeoTiffMetadata.prototype
             * @type {Sector}
             */
            bbox: {
                get: function () {
                    return this._bbox;
                },

                set: function(value){
                    this._bbox = value;
                }
            },

            //geokeys
            /**
             * Contains an ID defining the crs model.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            gtModelTypeGeoKey: {
                get: function () {
                    return this._gtModelTypeGeoKey;
                },

                set: function(value){
                    this._gtModelTypeGeoKey = value;
                }
            },

            /**
             * Contains an ID defining the raster sample type.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            gtRasterTypeGeoKey: {
                get: function () {
                    return this._gtRasterTypeGeoKey;
                },

                set: function(value){
                    this._gtRasterTypeGeoKey = value;
                }
            },

            /**
             * Contains an ASCII reference to the overall configuration of the geotiff file.
             * @memberof GeoTiffMetadata.prototype
             * @type {String}
             */
            gtCitationGeoKey: {
                get: function () {
                    return this._gtCitationGeoKey;
                },

                set: function(value){
                    this._gtCitationGeoKey = value;
                }
            },

            /**
             * Contains a value to specify the code for geographic coordinate system used to map lat-long to a specific
             * ellipsoid over the earth
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            geographicTypeGeoKey: {
                get: function () {
                    return this._geographicTypeGeoKey;
                },

                set: function(value){
                    this._geographicTypeGeoKey = value;
                }
            },

            /**
             * Contains a value to specify the code for geographic coordinate system used to map lat-long to a specific
             * ellipsoid over the earth
             * @memberof GeoTiffMetadata.prototype
             * @type {String}
             */
            geogCitationGeoKey: {
                get: function () {
                    return this._geogCitationGeoKey;
                },

                set: function(value){
                    this._geogCitationGeoKey = value;
                }
            },

            /**
             * Allows the definition of geocentric CS Linear units for used-defined GCS and for ellipsoids
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            geogAngularUnitsGeoKey: {
                get: function () {
                    return this._geogAngularUnitsGeoKey;
                },

                set: function(value){
                    this._geogAngularUnitsGeoKey = value;
                }
            },

            /**
             * Allows the definition of user-defined angular geographic units, as measured in radians
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            geogAngularUnitSizeGeoKey: {
                get: function () {
                    return this._geogAngularUnitSizeGeoKey;
                },

                set: function(value){
                    this._geogAngularUnitSizeGeoKey = value;
                }
            },

            /**
             * Allows the specification of user-defined Ellipsoidal Semi-Major Axis
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            geogSemiMajorAxisGeoKey: {
                get: function () {
                    return this._geogSemiMajorAxisGeoKey;
                },

                set: function(value){
                    this._geogSemiMajorAxisGeoKey = value;
                }
            },

            /**
             * Allows the specification of the inverse of user-defined Ellipsoid's flattening parameter f.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            geogInvFlatteningGeoKey: {
                get: function () {
                    return this._geogInvFlatteningGeoKey;
                },

                set: function(value){
                    this._geogInvFlatteningGeoKey = value;
                }
            },

            /**
             * Contains the EPSG code of the geotiff.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            projectedCSType: {
                get: function () {
                    return this._projectedCSType;
                },

                set: function(value){
                    this._projectedCSType = value;
                }
            },

            /**
             * Contains the linear units of the geotiff.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            projLinearUnits: {
                get: function () {
                    return this._projLinearUnits;
                },

                set: function(value){
                    this._projLinearUnits = value;
                }
            },

            /**
             * Contains the number of pixels per resolution unit in the image width direction.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            xResolution: {
                get: function () {
                    return this._xResolution;
                },

                set: function(value){
                    this._xResolution = value;
                }
            },

            /**
             * Contains the number of pixels per resolution unit in the image length direction.
             * @memberof GeoTiffMetadata.prototype
             * @type {Number}
             */
            yResolution: {
                get: function () {
                    return this._yResolution;
                },

                set: function(value){
                    this._yResolution = value;
                }
            }
        });

        return GeoTiffMetadata;
    }
);
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports Tiff
 */
define('formats/geotiff/TiffConstants',[],
    function () {
        "use strict";

        /**
         * Provides all of the TIFF tag and subtag constants.
         * @alias TiffConstants
         * @constructor
         * @classdesc Contains all of the TIFF tags that are used to store TIFF information of any type.
         */
        var TiffConstants = {
            /**
             * An object containing all TIFF specific tags.
             * @memberof Tiff
             * @type {Object}
             */
            Tag: {
                'NEW_SUBFILE_TYPE': 254,
                'SUBFILE_TYPE': 255,
                'IMAGE_WIDTH': 256,
                'IMAGE_LENGTH': 257,
                'BITS_PER_SAMPLE': 258,
                'COMPRESSION': 259,
                'PHOTOMETRIC_INTERPRETATION': 262,
                'THRESHHOLDING': 263,
                'CELL_WIDTH': 264,
                'CELL_LENGTH': 265,
                'FILL_ORDER': 266,
                'DOCUMENT_NAME': 269,
                'IMAGE_DESCRIPTION': 270,
                'MAKE': 271,
                'MODEL': 272,
                'STRIP_OFFSETS': 273,
                'ORIENTATION': 274,
                'SAMPLES_PER_PIXEL': 277,
                'ROWS_PER_STRIP': 278,
                'STRIP_BYTE_COUNTS': 279,
                'MIN_SAMPLE_VALUE': 280,
                'MAX_SAMPLE_VALUE': 281,
                'X_RESOLUTION': 282,
                'Y_RESOLUTION': 283,
                'PLANAR_CONFIGURATION': 284,
                'PAGE_NAME': 285,
                'X_POSITION': 286,
                'Y_POSITION': 287,
                'FREE_OFFSETS': 288,
                'FREE_BYTE_COUNTS': 289,
                'GRAY_RESPONSE_UNIT': 290,
                'GRAY_RESPONSE_CURVE': 291,
                'T4_OPTIONS': 292,
                'T6_PTIONS': 293,
                'RESOLUTION_UNIT': 296,
                'PAGE_NUMBER': 297,
                'TRANSFER_FUNCTION': 301,
                'SOFTWARE': 305,
                'DATE_TIME': 306,
                'ARTIST': 315,
                'HOST_COMPUTER': 316,
                'PREDICTOR': 317,
                'WHITE_POINT': 318,
                'PRIMARY_CHROMATICITIES': 319,
                'COLOR_MAP': 320,
                'HALFTONE_HINTS': 321,
                'TILE_WIDTH': 322,
                'TILE_LENGTH': 323,
                'TILE_OFFSETS': 324,
                'TILE_BYTE_COUNTS': 325,
                'INK_SET': 332,
                'INK_NAMES': 333,
                'NUMBER_OF_INKS': 334,
                'DOT_RANGE': 336,
                'TARGET_PRINTER': 337,
                'EXTRA_SAMPLES': 338,
                'SAMPLE_FORMAT': 339,
                'S_MIN_SAMPLE_VALUE': 340,
                'S_MAX_SAMPLE_VALUE': 341,
                'TRANSFER_RANGE': 342,
                'JPEG_PROC': 512,
                'JPEG_INTERCHANGE_FORMAT': 513,
                'JPEG_INTERCHANGE_FORMAT_LENGTH': 514,
                'JPEG_RESTART_INTERVAL': 515,
                'JPEG_LOSSLESS_PREDICTORS': 517,
                'JPEG_POINT_TRANSFORMS': 518,
                'JPEG_Q_TABLES': 519,
                'JPEG_DC_TABLES': 520,
                'JPEG_AC_TABLES': 521,
                'Y_Cb_Cr_COEFFICIENTS': 529,
                'Y_Cb_Cr_SUB_SAMPLING': 530,
                'Y_Cb_Cr_POSITIONING': 531,
                'REFERENCE_BLACK_WHITE': 532,
                'COPYRIGHT': 33432
            },

            /**
             * An object containing all TIFF compression types.
             * @memberof Tiff
             * @type {Object}
             */
            Compression: {
                'UNCOMPRESSED': 1,
                'CCITT_1D': 2,
                'GROUP_3_FAX': 3,
                'GROUP_4_FAX': 4,
                'LZW': 5,
                'JPEG': 6,
                'PACK_BITS': 32773
            },

            /**
             * An object containing all TIFF orientation types.
             * @memberof Tiff
             * @type {Object}
             */
            Orientation: {
                'Row0_IS_TOP__Col0_IS_LHS': 1,
                'Row0_IS_TOP__Col0_IS_RHS': 2,
                'Row0_IS_BOTTOM__Col0_IS_RHS': 3,
                'Row0_IS_BOTTOM__Col0_IS_LHS': 4,
                'Row0_IS_LHS__Col0_IS_TOP': 5,
                'Row0_IS_RHS__Col0_IS_TOP': 6,
                'Row0_IS_RHS__Col0_IS_BOTTOM': 7,
                'Row0_IS_LHS__Col0_IS_BOTTOM': 8
            },

            /**
             * An object containing all TIFF photometric interpretation types.
             * @memberof Tiff
             * @type {Object}
             */
            PhotometricInterpretation: {
                'WHITE_IS_ZERO': 0,
                'BLACK_IS_ZERO': 1,
                'RGB': 2,
                'RGB_PALETTE': 3,
                'TRANSPARENCY_MASK': 4,
                'CMYK': 5,
                'Y_Cb_Cr': 6,
                'CIE_LAB': 7
            },

            /**
             * An object containing all TIFF planar configuration types.
             * @memberof Tiff
             * @type {Object}
             */
            PlanarConfiguration: {
                'CHUNKY': 1,
                'PLANAR': 2
            },

            /**
             * An object containing all TIFF resolution unit types.
             * @memberof Tiff
             * @type {Object}
             */
            ResolutionUnit: {
                'NONE': 1,
                'INCH': 2,
                'CENTIMETER': 3
            },

            /**
             * An object containing all TIFF sample format types.
             * @memberof Tiff
             * @type {Object}
             */
            SampleFormat: {
                'UNSIGNED': 1,
                'SIGNED': 2,
                'IEEE_FLOAT': 3,
                'UNDEFINED': 4,
                'DEFAULT': 1
            },

            /**
             * An object containing all TIFF field types.
             * @memberof Tiff
             * @type {Object}
             */
            Type: {
                'BYTE': 1,
                'ASCII': 2,
                'SHORT': 3,
                'LONG': 4,
                'RATIONAL': 5,
                'SBYTE': 6,
                'UNDEFINED': 7,
                'SSHORT': 8,
                'SLONG': 9,
                'SRATIONAL': 10,
                'FLOAT': 11,
                'DOUBLE': 12
            }
        };

        return TiffConstants;
    }
);
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports GeoTiffUtil
 */
define('formats/geotiff/GeoTiffUtil',[
        '../../error/ArgumentError',
        '../../util/Logger',
        './TiffConstants'
    ],
    function (ArgumentError,
              Logger,
              TiffConstants) {
        "use strict";

        var GeoTiffUtil = {

            // Get bytes from an arraybuffer depending on the size. Internal use only.
            getBytes: function (geoTiffData, byteOffset, numOfBytes, isLittleEndian, isSigned) {
                if (numOfBytes <= 0) {
                    throw new ArgumentError(
                        Logger.logMessage(Logger.LEVEL_SEVERE, "GeoTiffReader", "getBytes", "noBytesRequested"));
                } else if (numOfBytes <= 1) {
                    if (isSigned) {
                        return geoTiffData.getInt8(byteOffset, isLittleEndian);
                    }
                    else {
                        return geoTiffData.getUint8(byteOffset, isLittleEndian);
                    }
                } else if (numOfBytes <= 2) {
                    if (isSigned) {
                        return geoTiffData.getInt16(byteOffset, isLittleEndian);
                    }
                    else {
                        return geoTiffData.getUint16(byteOffset, isLittleEndian);
                    }
                } else if (numOfBytes <= 3) {
                    if (isSigned) {
                        return geoTiffData.getInt32(byteOffset, isLittleEndian) >>> 8;
                    }
                    else {
                        return geoTiffData.getUint32(byteOffset, isLittleEndian) >>> 8;
                    }
                } else if (numOfBytes <= 4) {
                    if (isSigned) {
                        return geoTiffData.getInt32(byteOffset, isLittleEndian);
                    }
                    else {
                        return geoTiffData.getUint32(byteOffset, isLittleEndian);
                    }
                } else if (numOfBytes <= 8) {
                    return geoTiffData.getFloat64(byteOffset, isLittleEndian);
                } else {
                    throw new ArgumentError(
                        Logger.logMessage(Logger.LEVEL_SEVERE, "GeoTiffReader", "getBytes", "tooManyBytesRequested"));
                }
            },

            // Get sample value from an arraybuffer depending on the sample format. Internal use only.
            getSampleBytes: function (geoTiffData, byteOffset, numOfBytes, sampleFormat, isLittleEndian) {
                var res;

                switch (sampleFormat) {
                    case TiffConstants.SampleFormat.UNSIGNED:
                        res = this.getBytes(geoTiffData, byteOffset, numOfBytes, isLittleEndian, false);
                        break;
                    case TiffConstants.SampleFormat.SIGNED:
                        res = this.getBytes(geoTiffData, byteOffset, numOfBytes, isLittleEndian, true);
                        break;
                    case TiffConstants.SampleFormat.IEEE_FLOAT:
                        if (numOfBytes == 3) {
                            res = geoTiffData.getFloat32(byteOffset, isLittleEndian) >>> 8;
                        } else if (numOfBytes == 4) {
                            res = geoTiffData.getFloat32(byteOffset, isLittleEndian);
                        } else if (numOfBytes == 8) {
                            res = geoTiffData.getFloat64(byteOffset, isLittleEndian);
                        }
                        else {
                            Logger.log(Logger.LEVEL_WARNING, "Do not attempt to parse the data  not handled: " +
                                numOfBytes);
                        }
                        break;
                    case TiffConstants.SampleFormat.UNDEFINED:
                    default:
                        res = this.getBytes(geoTiffData, byteOffset, numOfBytes, isLittleEndian, false);
                        break;
                }

                return res;
            },

            // Get RGBA fill style for a canvas context as a string. Internal use only.
            getRGBAFillValue: function (r, g, b, a) {
                if (typeof a === 'undefined') {
                    a = 1.0;
                }
                return "rgba(" + r + ", " + g + ", " + b + ", " + a + ")";
            },

            // Clamp color sample from color sample value and number of bits per sample. Internal use only.
            clampColorSample: function (colorSample, bitsPerSample) {
                var multiplier = Math.pow(2, 8 - bitsPerSample);
                return Math.floor((colorSample * multiplier) + (multiplier - 1));
            }
        };

        return GeoTiffUtil;
    }
);
define('util/proj4-src',[], function () {
    'use strict';

    var globals = function(defs) {
        defs('EPSG:4326', "+title=WGS 84 (long/lat) +proj=longlat +ellps=WGS84 +datum=WGS84 +units=degrees");
        defs('EPSG:4269', "+title=NAD83 (long/lat) +proj=longlat +a=6378137.0 +b=6356752.31414036 +ellps=GRS80 +datum=NAD83 +units=degrees");
        defs('EPSG:3857', "+title=WGS 84 / Pseudo-Mercator +proj=merc +a=6378137 +b=6378137 +lat_ts=0.0 +lon_0=0.0 +x_0=0.0 +y_0=0 +k=1.0 +units=m +nadgrids=@null +no_defs");

        defs.WGS84 = defs['EPSG:4326'];
        defs['EPSG:3785'] = defs['EPSG:3857']; // maintain backward compat, official code is 3857
        defs.GOOGLE = defs['EPSG:3857'];
        defs['EPSG:900913'] = defs['EPSG:3857'];
        defs['EPSG:102113'] = defs['EPSG:3857'];
    };

    var PJD_3PARAM = 1;
    var PJD_7PARAM = 2;
    var PJD_WGS84 = 4; // WGS84 or equivalent
    var PJD_NODATUM = 5; // WGS84 or equivalent
    var SEC_TO_RAD = 4.84813681109535993589914102357e-6;
    var HALF_PI = Math.PI/2;
    // ellipoid pj_set_ell.c
    var SIXTH = 0.1666666666666666667;
    /* 1/6 */
    var RA4 = 0.04722222222222222222;
    /* 17/360 */
    var RA6 = 0.02215608465608465608;
    var EPSLN = 1.0e-10;
    // you'd think you could use Number.EPSILON above but that makes
    // Mollweide get into an infinate loop.

    var D2R = 0.01745329251994329577;
    var R2D = 57.29577951308232088;
    var FORTPI = Math.PI/4;
    var TWO_PI = Math.PI * 2;
    // SPI is slightly greater than Math.PI, so values that exceed the -180..180
    // degree range by a tiny amount don't get wrapped. This prevents points that
    // have drifted from their original location along the 180th meridian (due to
    // floating point error) from changing their sign.
    var SPI = 3.14159265359;

    var exports$1 = {};
    exports$1.greenwich = 0.0; //"0dE",
    exports$1.lisbon = -9.131906111111; //"9d07'54.862\"W",
    exports$1.paris = 2.337229166667; //"2d20'14.025\"E",
    exports$1.bogota = -74.080916666667; //"74d04'51.3\"W",
    exports$1.madrid = -3.687938888889; //"3d41'16.58\"W",
    exports$1.rome = 12.452333333333; //"12d27'8.4\"E",
    exports$1.bern = 7.439583333333; //"7d26'22.5\"E",
    exports$1.jakarta = 106.807719444444; //"106d48'27.79\"E",
    exports$1.ferro = -17.666666666667; //"17d40'W",
    exports$1.brussels = 4.367975; //"4d22'4.71\"E",
    exports$1.stockholm = 18.058277777778; //"18d3'29.8\"E",
    exports$1.athens = 23.7163375; //"23d42'58.815\"E",
    exports$1.oslo = 10.722916666667; //"10d43'22.5\"E"

    var units = {
        ft: {to_meter: 0.3048},
        'us-ft': {to_meter: 1200 / 3937}
    };

    var ignoredChar = /[\s_\-\/\(\)]/g;
    function match(obj, key) {
        if (obj[key]) {
            return obj[key];
        }
        var keys = Object.keys(obj);
        var lkey = key.toLowerCase().replace(ignoredChar, '');
        var i = -1;
        var testkey, processedKey;
        while (++i < keys.length) {
            testkey = keys[i];
            processedKey = testkey.toLowerCase().replace(ignoredChar, '');
            if (processedKey === lkey) {
                return obj[testkey];
            }
        }
    }

    var parseProj = function(defData) {
        var self = {};
        var paramObj = defData.split('+').map(function(v) {
            return v.trim();
        }).filter(function(a) {
            return a;
        }).reduce(function(p, a) {
            var split = a.split('=');
            split.push(true);
            p[split[0].toLowerCase()] = split[1];
            return p;
        }, {});
        var paramName, paramVal, paramOutname;
        var params = {
            proj: 'projName',
            datum: 'datumCode',
            rf: function(v) {
                self.rf = parseFloat(v);
            },
            lat_0: function(v) {
                self.lat0 = v * D2R;
            },
            lat_1: function(v) {
                self.lat1 = v * D2R;
            },
            lat_2: function(v) {
                self.lat2 = v * D2R;
            },
            lat_ts: function(v) {
                self.lat_ts = v * D2R;
            },
            lon_0: function(v) {
                self.long0 = v * D2R;
            },
            lon_1: function(v) {
                self.long1 = v * D2R;
            },
            lon_2: function(v) {
                self.long2 = v * D2R;
            },
            alpha: function(v) {
                self.alpha = parseFloat(v) * D2R;
            },
            lonc: function(v) {
                self.longc = v * D2R;
            },
            x_0: function(v) {
                self.x0 = parseFloat(v);
            },
            y_0: function(v) {
                self.y0 = parseFloat(v);
            },
            k_0: function(v) {
                self.k0 = parseFloat(v);
            },
            k: function(v) {
                self.k0 = parseFloat(v);
            },
            a: function(v) {
                self.a = parseFloat(v);
            },
            b: function(v) {
                self.b = parseFloat(v);
            },
            r_a: function() {
                self.R_A = true;
            },
            zone: function(v) {
                self.zone = parseInt(v, 10);
            },
            south: function() {
                self.utmSouth = true;
            },
            towgs84: function(v) {
                self.datum_params = v.split(",").map(function(a) {
                    return parseFloat(a);
                });
            },
            to_meter: function(v) {
                self.to_meter = parseFloat(v);
            },
            units: function(v) {
                self.units = v;
                var unit = match(units, v);
                if (unit) {
                    self.to_meter = unit.to_meter;
                }
            },
            from_greenwich: function(v) {
                self.from_greenwich = v * D2R;
            },
            pm: function(v) {
                var pm = match(exports$1, v);
                self.from_greenwich = (pm ? pm : parseFloat(v)) * D2R;
            },
            nadgrids: function(v) {
                if (v === '@null') {
                    self.datumCode = 'none';
                }
                else {
                    self.nadgrids = v;
                }
            },
            axis: function(v) {
                var legalAxis = "ewnsud";
                if (v.length === 3 && legalAxis.indexOf(v.substr(0, 1)) !== -1 && legalAxis.indexOf(v.substr(1, 1)) !== -1 && legalAxis.indexOf(v.substr(2, 1)) !== -1) {
                    self.axis = v;
                }
            }
        };
        for (paramName in paramObj) {
            paramVal = paramObj[paramName];
            if (paramName in params) {
                paramOutname = params[paramName];
                if (typeof paramOutname === 'function') {
                    paramOutname(paramVal);
                }
                else {
                    self[paramOutname] = paramVal;
                }
            }
            else {
                self[paramName] = paramVal;
            }
        }
        if(typeof self.datumCode === 'string' && self.datumCode !== "WGS84"){
            self.datumCode = self.datumCode.toLowerCase();
        }
        return self;
    };

    var NEUTRAL = 1;
    var KEYWORD = 2;
    var NUMBER = 3;
    var QUOTED = 4;
    var AFTERQUOTE = 5;
    var ENDED = -1;
    var whitespace = /\s/;
    var latin = /[A-Za-z]/;
    var keyword = /[A-Za-z84]/;
    var endThings = /[,\]]/;
    var digets = /[\d\.E\-\+]/;
    // const ignoredChar = /[\s_\-\/\(\)]/g;
    function Parser(text) {
        if (typeof text !== 'string') {
            throw new Error('not a string');
        }
        this.text = text.trim();
        this.level = 0;
        this.place = 0;
        this.root = null;
        this.stack = [];
        this.currentObject = null;
        this.state = NEUTRAL;
    }
    Parser.prototype.readCharicter = function() {
        var char = this.text[this.place++];
        if (this.state !== QUOTED) {
            while (whitespace.test(char)) {
                if (this.place >= this.text.length) {
                    return;
                }
                char = this.text[this.place++];
            }
        }
        switch (this.state) {
            case NEUTRAL:
                return this.neutral(char);
            case KEYWORD:
                return this.keyword(char)
            case QUOTED:
                return this.quoted(char);
            case AFTERQUOTE:
                return this.afterquote(char);
            case NUMBER:
                return this.number(char);
            case ENDED:
                return;
        }
    };
    Parser.prototype.afterquote = function(char) {
        if (char === '"') {
            this.word += '"';
            this.state = QUOTED;
            return;
        }
        if (endThings.test(char)) {
            this.word = this.word.trim();
            this.afterItem(char);
            return;
        }
        throw new Error('havn\'t handled "' +char + '" in afterquote yet, index ' + this.place);
    };
    Parser.prototype.afterItem = function(char) {
        if (char === ',') {
            if (this.word !== null) {
                this.currentObject.push(this.word);
            }
            this.word = null;
            this.state = NEUTRAL;
            return;
        }
        if (char === ']') {
            this.level--;
            if (this.word !== null) {
                this.currentObject.push(this.word);
                this.word = null;
            }
            this.state = NEUTRAL;
            this.currentObject = this.stack.pop();
            if (!this.currentObject) {
                this.state = ENDED;
            }

            return;
        }
    };
    Parser.prototype.number = function(char) {
        if (digets.test(char)) {
            this.word += char;
            return;
        }
        if (endThings.test(char)) {
            this.word = parseFloat(this.word);
            this.afterItem(char);
            return;
        }
        throw new Error('havn\'t handled "' +char + '" in number yet, index ' + this.place);
    };
    Parser.prototype.quoted = function(char) {
        if (char === '"') {
            this.state = AFTERQUOTE;
            return;
        }
        this.word += char;
        return;
    };
    Parser.prototype.keyword = function(char) {
        if (keyword.test(char)) {
            this.word += char;
            return;
        }
        if (char === '[') {
            var newObjects = [];
            newObjects.push(this.word);
            this.level++;
            if (this.root === null) {
                this.root = newObjects;
            } else {
                this.currentObject.push(newObjects);
            }
            this.stack.push(this.currentObject);
            this.currentObject = newObjects;
            this.state = NEUTRAL;
            return;
        }
        if (endThings.test(char)) {
            this.afterItem(char);
            return;
        }
        throw new Error('havn\'t handled "' +char + '" in keyword yet, index ' + this.place);
    };
    Parser.prototype.neutral = function(char) {
        if (latin.test(char)) {
            this.word = char;
            this.state = KEYWORD;
            return;
        }
        if (char === '"') {
            this.word = '';
            this.state = QUOTED;
            return;
        }
        if (digets.test(char)) {
            this.word = char;
            this.state = NUMBER;
            return;
        }
        if (endThings.test(char)) {
            this.afterItem(char);
            return;
        }
        throw new Error('havn\'t handled "' +char + '" in neutral yet, index ' + this.place);
    };
    Parser.prototype.output = function() {
        while (this.place < this.text.length) {
            this.readCharicter();
        }
        if (this.state === ENDED) {
            return this.root;
        }
        throw new Error('unable to parse string "' +this.text + '". State is ' + this.state);
    };

    function parseString(txt) {
        var parser = new Parser(txt);
        return parser.output();
    }

    function mapit(obj, key, value) {
        if (Array.isArray(key)) {
            value.unshift(key);
            key = null;
        }
        var thing = key ? {} : obj;

        var out = value.reduce(function(newObj, item) {
            sExpr(item, newObj);
            return newObj
        }, thing);
        if (key) {
            obj[key] = out;
        }
    }

    function sExpr(v, obj) {
        if (!Array.isArray(v)) {
            obj[v] = true;
            return;
        }
        var key = v.shift();
        if (key === 'PARAMETER') {
            key = v.shift();
        }
        if (v.length === 1) {
            if (Array.isArray(v[0])) {
                obj[key] = {};
                sExpr(v[0], obj[key]);
                return;
            }
            obj[key] = v[0];
            return;
        }
        if (!v.length) {
            obj[key] = true;
            return;
        }
        if (key === 'TOWGS84') {
            obj[key] = v;
            return;
        }
        if (!Array.isArray(key)) {
            obj[key] = {};
        }

        var i;
        switch (key) {
            case 'UNIT':
            case 'PRIMEM':
            case 'VERT_DATUM':
                obj[key] = {
                    name: v[0].toLowerCase(),
                    convert: v[1]
                };
                if (v.length === 3) {
                    sExpr(v[2], obj[key]);
                }
                return;
            case 'SPHEROID':
            case 'ELLIPSOID':
                obj[key] = {
                    name: v[0],
                    a: v[1],
                    rf: v[2]
                };
                if (v.length === 4) {
                    sExpr(v[3], obj[key]);
                }
                return;
            case 'PROJECTEDCRS':
            case 'PROJCRS':
            case 'GEOGCS':
            case 'GEOCCS':
            case 'PROJCS':
            case 'LOCAL_CS':
            case 'GEODCRS':
            case 'GEODETICCRS':
            case 'GEODETICDATUM':
            case 'EDATUM':
            case 'ENGINEERINGDATUM':
            case 'VERT_CS':
            case 'VERTCRS':
            case 'VERTICALCRS':
            case 'COMPD_CS':
            case 'COMPOUNDCRS':
            case 'ENGINEERINGCRS':
            case 'ENGCRS':
            case 'FITTED_CS':
            case 'LOCAL_DATUM':
            case 'DATUM':
                v[0] = ['name', v[0]];
                mapit(obj, key, v);
                return;
            default:
                i = -1;
                while (++i < v.length) {
                    if (!Array.isArray(v[i])) {
                        return sExpr(v, obj[key]);
                    }
                }
                return mapit(obj, key, v);
        }
    }

    var D2R$1 = 0.01745329251994329577;
    function rename(obj, params) {
        var outName = params[0];
        var inName = params[1];
        if (!(outName in obj) && (inName in obj)) {
            obj[outName] = obj[inName];
            if (params.length === 3) {
                obj[outName] = params[2](obj[outName]);
            }
        }
    }

    function d2r(input) {
        return input * D2R$1;
    }

    function cleanWKT(wkt) {
        if (wkt.type === 'GEOGCS') {
            wkt.projName = 'longlat';
        } else if (wkt.type === 'LOCAL_CS') {
            wkt.projName = 'identity';
            wkt.local = true;
        } else {
            if (typeof wkt.PROJECTION === 'object') {
                wkt.projName = Object.keys(wkt.PROJECTION)[0];
            } else {
                wkt.projName = wkt.PROJECTION;
            }
        }
        if (wkt.UNIT) {
            wkt.units = wkt.UNIT.name.toLowerCase();
            if (wkt.units === 'metre') {
                wkt.units = 'meter';
            }
            if (wkt.UNIT.convert) {
                if (wkt.type === 'GEOGCS') {
                    if (wkt.DATUM && wkt.DATUM.SPHEROID) {
                        wkt.to_meter = wkt.UNIT.convert*wkt.DATUM.SPHEROID.a;
                    }
                } else {
                    wkt.to_meter = wkt.UNIT.convert;
                }
            }
        }
        var geogcs = wkt.GEOGCS;
        if (wkt.type === 'GEOGCS') {
            geogcs = wkt;
        }
        if (geogcs) {
            //if(wkt.GEOGCS.PRIMEM&&wkt.GEOGCS.PRIMEM.convert){
            //  wkt.from_greenwich=wkt.GEOGCS.PRIMEM.convert*D2R;
            //}
            if (geogcs.DATUM) {
                wkt.datumCode = geogcs.DATUM.name.toLowerCase();
            } else {
                wkt.datumCode = geogcs.name.toLowerCase();
            }
            if (wkt.datumCode.slice(0, 2) === 'd_') {
                wkt.datumCode = wkt.datumCode.slice(2);
            }
            if (wkt.datumCode === 'new_zealand_geodetic_datum_1949' || wkt.datumCode === 'new_zealand_1949') {
                wkt.datumCode = 'nzgd49';
            }
            if (wkt.datumCode === 'wgs_1984') {
                if (wkt.PROJECTION === 'Mercator_Auxiliary_Sphere') {
                    wkt.sphere = true;
                }
                wkt.datumCode = 'wgs84';
            }
            if (wkt.datumCode.slice(-6) === '_ferro') {
                wkt.datumCode = wkt.datumCode.slice(0, - 6);
            }
            if (wkt.datumCode.slice(-8) === '_jakarta') {
                wkt.datumCode = wkt.datumCode.slice(0, - 8);
            }
            if (~wkt.datumCode.indexOf('belge')) {
                wkt.datumCode = 'rnb72';
            }
            if (geogcs.DATUM && geogcs.DATUM.SPHEROID) {
                wkt.ellps = geogcs.DATUM.SPHEROID.name.replace('_19', '').replace(/[Cc]larke\_18/, 'clrk');
                if (wkt.ellps.toLowerCase().slice(0, 13) === 'international') {
                    wkt.ellps = 'intl';
                }

                wkt.a = geogcs.DATUM.SPHEROID.a;
                wkt.rf = parseFloat(geogcs.DATUM.SPHEROID.rf, 10);
            }

            if (geogcs.DATUM && geogcs.DATUM.TOWGS84) {
                wkt.datum_params = geogcs.DATUM.TOWGS84;
            }
            if (~wkt.datumCode.indexOf('osgb_1936')) {
                wkt.datumCode = 'osgb36';
            }
            if (~wkt.datumCode.indexOf('osni_1952')) {
                wkt.datumCode = 'osni52';
            }
            if (~wkt.datumCode.indexOf('tm65')
                || ~wkt.datumCode.indexOf('geodetic_datum_of_1965')) {
                wkt.datumCode = 'ire65';
            }
            if (wkt.datumCode === 'ch1903+') {
                wkt.datumCode = 'ch1903';
            }
            if (~wkt.datumCode.indexOf('israel')) {
                wkt.datumCode = 'isr93';
            }
        }
        if (wkt.b && !isFinite(wkt.b)) {
            wkt.b = wkt.a;
        }

        function toMeter(input) {
            var ratio = wkt.to_meter || 1;
            return input * ratio;
        }
        var renamer = function(a) {
            return rename(wkt, a);
        };
        var list = [
            ['standard_parallel_1', 'Standard_Parallel_1'],
            ['standard_parallel_2', 'Standard_Parallel_2'],
            ['false_easting', 'False_Easting'],
            ['false_northing', 'False_Northing'],
            ['central_meridian', 'Central_Meridian'],
            ['latitude_of_origin', 'Latitude_Of_Origin'],
            ['latitude_of_origin', 'Central_Parallel'],
            ['scale_factor', 'Scale_Factor'],
            ['k0', 'scale_factor'],
            ['latitude_of_center', 'Latitude_Of_Center'],
            ['latitude_of_center', 'Latitude_of_center'],
            ['lat0', 'latitude_of_center', d2r],
            ['longitude_of_center', 'Longitude_Of_Center'],
            ['longitude_of_center', 'Longitude_of_center'],
            ['longc', 'longitude_of_center', d2r],
            ['x0', 'false_easting', toMeter],
            ['y0', 'false_northing', toMeter],
            ['long0', 'central_meridian', d2r],
            ['lat0', 'latitude_of_origin', d2r],
            ['lat0', 'standard_parallel_1', d2r],
            ['lat1', 'standard_parallel_1', d2r],
            ['lat2', 'standard_parallel_2', d2r],
            ['azimuth', 'Azimuth'],
            ['alpha', 'azimuth', d2r],
            ['srsCode', 'name']
        ];
        list.forEach(renamer);
        if (!wkt.long0 && wkt.longc && (wkt.projName === 'Albers_Conic_Equal_Area' || wkt.projName === 'Lambert_Azimuthal_Equal_Area')) {
            wkt.long0 = wkt.longc;
        }
        if (!wkt.lat_ts && wkt.lat1 && (wkt.projName === 'Stereographic_South_Pole' || wkt.projName === 'Polar Stereographic (variant B)')) {
            wkt.lat0 = d2r(wkt.lat1 > 0 ? 90 : -90);
            wkt.lat_ts = wkt.lat1;
        }
    }
    var wkt = function(wkt) {
        var lisp = parseString(wkt);
        var type = lisp.shift();
        var name = lisp.shift();
        lisp.unshift(['name', name]);
        lisp.unshift(['type', type]);
        var obj = {};
        sExpr(lisp, obj);
        cleanWKT(obj);
        return obj;
    };

    function defs(name) {
        /*global console*/
        var that = this;
        if (arguments.length === 2) {
            var def = arguments[1];
            if (typeof def === 'string') {
                if (def.charAt(0) === '+') {
                    defs[name] = parseProj(arguments[1]);
                }
                else {
                    defs[name] = wkt(arguments[1]);
                }
            } else {
                defs[name] = def;
            }
        }
        else if (arguments.length === 1) {
            if (Array.isArray(name)) {
                return name.map(function(v) {
                    if (Array.isArray(v)) {
                        defs.apply(that, v);
                    }
                    else {
                        defs(v);
                    }
                });
            }
            else if (typeof name === 'string') {
                if (name in defs) {
                    return defs[name];
                }
            }
            else if ('EPSG' in name) {
                defs['EPSG:' + name.EPSG] = name;
            }
            else if ('ESRI' in name) {
                defs['ESRI:' + name.ESRI] = name;
            }
            else if ('IAU2000' in name) {
                defs['IAU2000:' + name.IAU2000] = name;
            }
            else {
                console.log(name);
            }
            return;
        }


    }
    globals(defs);

    function testObj(code){
        return typeof code === 'string';
    }
    function testDef(code){
        return code in defs;
    }
    var codeWords = ['PROJECTEDCRS', 'PROJCRS', 'GEOGCS','GEOCCS','PROJCS','LOCAL_CS', 'GEODCRS', 'GEODETICCRS', 'GEODETICDATUM', 'ENGCRS', 'ENGINEERINGCRS'];
    function testWKT(code){
        return codeWords.some(function (word) {
            return code.indexOf(word) > -1;
        });
    }
    var codes = ['3857', '900913', '3785', '102113'];
    function checkMercator(item) {
        var auth = match(item, 'authority');
        if (!auth) {
            return;
        }
        var code = match(auth, 'epsg');
        return code && codes.indexOf(code) > -1;
    }
    function checkProjStr(item) {
        var ext = match(item, 'extension');
        if (!ext) {
            return;
        }
        return match(ext, 'proj4');
    }
    function testProj(code){
        return code[0] === '+';
    }
    function parse(code){
        if (testObj(code)) {
            //check to see if this is a WKT string
            if (testDef(code)) {
                return defs[code];
            }
            if (testWKT(code)) {
                var out = wkt(code);
                // test of spetial case, due to this being a very common and often malformed
                if (checkMercator(out)) {
                    return defs['EPSG:3857'];
                }
                var maybeProjStr = checkProjStr(out);
                if (maybeProjStr) {
                    return parseProj(maybeProjStr);
                }
                return out;
            }
            if (testProj(code)) {
                return parseProj(code);
            }
        }else{
            return code;
        }
    }

    var extend = function(destination, source) {
        destination = destination || {};
        var value, property;
        if (!source) {
            return destination;
        }
        for (property in source) {
            value = source[property];
            if (value !== undefined) {
                destination[property] = value;
            }
        }
        return destination;
    };

    var msfnz = function(eccent, sinphi, cosphi) {
        var con = eccent * sinphi;
        return cosphi / (Math.sqrt(1 - con * con));
    };

    var sign = function(x) {
        return x<0 ? -1 : 1;
    };

    var adjust_lon = function(x) {
        return (Math.abs(x) <= SPI) ? x : (x - (sign(x) * TWO_PI));
    };

    var tsfnz = function(eccent, phi, sinphi) {
        var con = eccent * sinphi;
        var com = 0.5 * eccent;
        con = Math.pow(((1 - con) / (1 + con)), com);
        return (Math.tan(0.5 * (HALF_PI - phi)) / con);
    };

    var phi2z = function(eccent, ts) {
        var eccnth = 0.5 * eccent;
        var con, dphi;
        var phi = HALF_PI - 2 * Math.atan(ts);
        for (var i = 0; i <= 15; i++) {
            con = eccent * Math.sin(phi);
            dphi = HALF_PI - 2 * Math.atan(ts * (Math.pow(((1 - con) / (1 + con)), eccnth))) - phi;
            phi += dphi;
            if (Math.abs(dphi) <= 0.0000000001) {
                return phi;
            }
        }
        //console.log("phi2z has NoConvergence");
        return -9999;
    };

    function init() {
        var con = this.b / this.a;
        this.es = 1 - con * con;
        if(!('x0' in this)){
            this.x0 = 0;
        }
        if(!('y0' in this)){
            this.y0 = 0;
        }
        this.e = Math.sqrt(this.es);
        if (this.lat_ts) {
            if (this.sphere) {
                this.k0 = Math.cos(this.lat_ts);
            }
            else {
                this.k0 = msfnz(this.e, Math.sin(this.lat_ts), Math.cos(this.lat_ts));
            }
        }
        else {
            if (!this.k0) {
                if (this.k) {
                    this.k0 = this.k;
                }
                else {
                    this.k0 = 1;
                }
            }
        }
    }

    /* Mercator forward equations--mapping lat,long to x,y
      --------------------------------------------------*/

    function forward(p) {
        var lon = p.x;
        var lat = p.y;
        // convert to radians
        if (lat * R2D > 90 && lat * R2D < -90 && lon * R2D > 180 && lon * R2D < -180) {
            return null;
        }

        var x, y;
        if (Math.abs(Math.abs(lat) - HALF_PI) <= EPSLN) {
            return null;
        }
        else {
            if (this.sphere) {
                x = this.x0 + this.a * this.k0 * adjust_lon(lon - this.long0);
                y = this.y0 + this.a * this.k0 * Math.log(Math.tan(FORTPI + 0.5 * lat));
            }
            else {
                var sinphi = Math.sin(lat);
                var ts = tsfnz(this.e, lat, sinphi);
                x = this.x0 + this.a * this.k0 * adjust_lon(lon - this.long0);
                y = this.y0 - this.a * this.k0 * Math.log(ts);
            }
            p.x = x;
            p.y = y;
            return p;
        }
    }

    /* Mercator inverse equations--mapping x,y to lat/long
      --------------------------------------------------*/
    function inverse(p) {

        var x = p.x - this.x0;
        var y = p.y - this.y0;
        var lon, lat;

        if (this.sphere) {
            lat = HALF_PI - 2 * Math.atan(Math.exp(-y / (this.a * this.k0)));
        }
        else {
            var ts = Math.exp(-y / (this.a * this.k0));
            lat = phi2z(this.e, ts);
            if (lat === -9999) {
                return null;
            }
        }
        lon = adjust_lon(this.long0 + x / (this.a * this.k0));

        p.x = lon;
        p.y = lat;
        return p;
    }

    var names$1 = ["Mercator", "Popular Visualisation Pseudo Mercator", "Mercator_1SP", "Mercator_Auxiliary_Sphere", "merc"];
    var merc = {
        init: init,
        forward: forward,
        inverse: inverse,
        names: names$1
    };

    function init$1() {
        //no-op for longlat
    }

    function identity(pt) {
        return pt;
    }
    var names$2 = ["longlat", "identity"];
    var longlat = {
        init: init$1,
        forward: identity,
        inverse: identity,
        names: names$2
    };

    var projs = [merc, longlat];
    var names = {};
    var projStore = [];

    function add(proj, i) {
        var len = projStore.length;
        if (!proj.names) {
            console.log(i);
            return true;
        }
        projStore[len] = proj;
        proj.names.forEach(function(n) {
            names[n.toLowerCase()] = len;
        });
        return this;
    }

    function get(name) {
        if (!name) {
            return false;
        }
        var n = name.toLowerCase();
        if (typeof names[n] !== 'undefined' && projStore[names[n]]) {
            return projStore[names[n]];
        }
    }

    function start() {
        projs.forEach(add);
    }
    var projections = {
        start: start,
        add: add,
        get: get
    };

    var exports$2 = {};
    exports$2.MERIT = {
        a: 6378137.0,
        rf: 298.257,
        ellipseName: "MERIT 1983"
    };

    exports$2.SGS85 = {
        a: 6378136.0,
        rf: 298.257,
        ellipseName: "Soviet Geodetic System 85"
    };

    exports$2.GRS80 = {
        a: 6378137.0,
        rf: 298.257222101,
        ellipseName: "GRS 1980(IUGG, 1980)"
    };

    exports$2.IAU76 = {
        a: 6378140.0,
        rf: 298.257,
        ellipseName: "IAU 1976"
    };

    exports$2.airy = {
        a: 6377563.396,
        b: 6356256.910,
        ellipseName: "Airy 1830"
    };

    exports$2.APL4 = {
        a: 6378137,
        rf: 298.25,
        ellipseName: "Appl. Physics. 1965"
    };

    exports$2.NWL9D = {
        a: 6378145.0,
        rf: 298.25,
        ellipseName: "Naval Weapons Lab., 1965"
    };

    exports$2.mod_airy = {
        a: 6377340.189,
        b: 6356034.446,
        ellipseName: "Modified Airy"
    };

    exports$2.andrae = {
        a: 6377104.43,
        rf: 300.0,
        ellipseName: "Andrae 1876 (Den., Iclnd.)"
    };

    exports$2.aust_SA = {
        a: 6378160.0,
        rf: 298.25,
        ellipseName: "Australian Natl & S. Amer. 1969"
    };

    exports$2.GRS67 = {
        a: 6378160.0,
        rf: 298.2471674270,
        ellipseName: "GRS 67(IUGG 1967)"
    };

    exports$2.bessel = {
        a: 6377397.155,
        rf: 299.1528128,
        ellipseName: "Bessel 1841"
    };

    exports$2.bess_nam = {
        a: 6377483.865,
        rf: 299.1528128,
        ellipseName: "Bessel 1841 (Namibia)"
    };

    exports$2.clrk66 = {
        a: 6378206.4,
        b: 6356583.8,
        ellipseName: "Clarke 1866"
    };

    exports$2.clrk80 = {
        a: 6378249.145,
        rf: 293.4663,
        ellipseName: "Clarke 1880 mod."
    };

    exports$2.clrk58 = {
        a: 6378293.645208759,
        rf: 294.2606763692654,
        ellipseName: "Clarke 1858"
    };

    exports$2.CPM = {
        a: 6375738.7,
        rf: 334.29,
        ellipseName: "Comm. des Poids et Mesures 1799"
    };

    exports$2.delmbr = {
        a: 6376428.0,
        rf: 311.5,
        ellipseName: "Delambre 1810 (Belgium)"
    };

    exports$2.engelis = {
        a: 6378136.05,
        rf: 298.2566,
        ellipseName: "Engelis 1985"
    };

    exports$2.evrst30 = {
        a: 6377276.345,
        rf: 300.8017,
        ellipseName: "Everest 1830"
    };

    exports$2.evrst48 = {
        a: 6377304.063,
        rf: 300.8017,
        ellipseName: "Everest 1948"
    };

    exports$2.evrst56 = {
        a: 6377301.243,
        rf: 300.8017,
        ellipseName: "Everest 1956"
    };

    exports$2.evrst69 = {
        a: 6377295.664,
        rf: 300.8017,
        ellipseName: "Everest 1969"
    };

    exports$2.evrstSS = {
        a: 6377298.556,
        rf: 300.8017,
        ellipseName: "Everest (Sabah & Sarawak)"
    };

    exports$2.fschr60 = {
        a: 6378166.0,
        rf: 298.3,
        ellipseName: "Fischer (Mercury Datum) 1960"
    };

    exports$2.fschr60m = {
        a: 6378155.0,
        rf: 298.3,
        ellipseName: "Fischer 1960"
    };

    exports$2.fschr68 = {
        a: 6378150.0,
        rf: 298.3,
        ellipseName: "Fischer 1968"
    };

    exports$2.helmert = {
        a: 6378200.0,
        rf: 298.3,
        ellipseName: "Helmert 1906"
    };

    exports$2.hough = {
        a: 6378270.0,
        rf: 297.0,
        ellipseName: "Hough"
    };

    exports$2.intl = {
        a: 6378388.0,
        rf: 297.0,
        ellipseName: "International 1909 (Hayford)"
    };

    exports$2.kaula = {
        a: 6378163.0,
        rf: 298.24,
        ellipseName: "Kaula 1961"
    };

    exports$2.lerch = {
        a: 6378139.0,
        rf: 298.257,
        ellipseName: "Lerch 1979"
    };

    exports$2.mprts = {
        a: 6397300.0,
        rf: 191.0,
        ellipseName: "Maupertius 1738"
    };

    exports$2.new_intl = {
        a: 6378157.5,
        b: 6356772.2,
        ellipseName: "New International 1967"
    };

    exports$2.plessis = {
        a: 6376523.0,
        rf: 6355863.0,
        ellipseName: "Plessis 1817 (France)"
    };

    exports$2.krass = {
        a: 6378245.0,
        rf: 298.3,
        ellipseName: "Krassovsky, 1942"
    };

    exports$2.SEasia = {
        a: 6378155.0,
        b: 6356773.3205,
        ellipseName: "Southeast Asia"
    };

    exports$2.walbeck = {
        a: 6376896.0,
        b: 6355834.8467,
        ellipseName: "Walbeck"
    };

    exports$2.WGS60 = {
        a: 6378165.0,
        rf: 298.3,
        ellipseName: "WGS 60"
    };

    exports$2.WGS66 = {
        a: 6378145.0,
        rf: 298.25,
        ellipseName: "WGS 66"
    };

    exports$2.WGS7 = {
        a: 6378135.0,
        rf: 298.26,
        ellipseName: "WGS 72"
    };

    var WGS84 = exports$2.WGS84 = {
        a: 6378137.0,
        rf: 298.257223563,
        ellipseName: "WGS 84"
    };

    exports$2.sphere = {
        a: 6370997.0,
        b: 6370997.0,
        ellipseName: "Normal Sphere (r=6370997)"
    };

    function eccentricity(a, b, rf, R_A) {
        var a2 = a * a; // used in geocentric
        var b2 = b * b; // used in geocentric
        var es = (a2 - b2) / a2; // e ^ 2
        var e = 0;
        if (R_A) {
            a *= 1 - es * (SIXTH + es * (RA4 + es * RA6));
            a2 = a * a;
            es = 0;
        } else {
            e = Math.sqrt(es); // eccentricity
        }
        var ep2 = (a2 - b2) / b2; // used in geocentric
        return {
            es: es,
            e: e,
            ep2: ep2
        };
    }
    function sphere(a, b, rf, ellps, sphere) {
        if (!a) { // do we have an ellipsoid?
            var ellipse = match(exports$2, ellps);
            if (!ellipse) {
                ellipse = WGS84;
            }
            a = ellipse.a;
            b = ellipse.b;
            rf = ellipse.rf;
        }

        if (rf && !b) {
            b = (1.0 - 1.0 / rf) * a;
        }
        if (rf === 0 || Math.abs(a - b) < EPSLN) {
            sphere = true;
            b = a;
        }
        return {
            a: a,
            b: b,
            rf: rf,
            sphere: sphere
        };
    }

    var exports$3 = {};
    exports$3.wgs84 = {
        towgs84: "0,0,0",
        ellipse: "WGS84",
        datumName: "WGS84"
    };

    exports$3.ch1903 = {
        towgs84: "674.374,15.056,405.346",
        ellipse: "bessel",
        datumName: "swiss"
    };

    exports$3.ggrs87 = {
        towgs84: "-199.87,74.79,246.62",
        ellipse: "GRS80",
        datumName: "Greek_Geodetic_Reference_System_1987"
    };

    exports$3.nad83 = {
        towgs84: "0,0,0",
        ellipse: "GRS80",
        datumName: "North_American_Datum_1983"
    };

    exports$3.nad27 = {
        nadgrids: "@conus,@alaska,@ntv2_0.gsb,@ntv1_can.dat",
        ellipse: "clrk66",
        datumName: "North_American_Datum_1927"
    };

    exports$3.potsdam = {
        towgs84: "606.0,23.0,413.0",
        ellipse: "bessel",
        datumName: "Potsdam Rauenberg 1950 DHDN"
    };

    exports$3.carthage = {
        towgs84: "-263.0,6.0,431.0",
        ellipse: "clark80",
        datumName: "Carthage 1934 Tunisia"
    };

    exports$3.hermannskogel = {
        towgs84: "653.0,-212.0,449.0",
        ellipse: "bessel",
        datumName: "Hermannskogel"
    };

    exports$3.osni52 = {
        towgs84: "482.530,-130.596,564.557,-1.042,-0.214,-0.631,8.15",
        ellipse: "airy",
        datumName: "Irish National"
    };

    exports$3.ire65 = {
        towgs84: "482.530,-130.596,564.557,-1.042,-0.214,-0.631,8.15",
        ellipse: "mod_airy",
        datumName: "Ireland 1965"
    };

    exports$3.rassadiran = {
        towgs84: "-133.63,-157.5,-158.62",
        ellipse: "intl",
        datumName: "Rassadiran"
    };

    exports$3.nzgd49 = {
        towgs84: "59.47,-5.04,187.44,0.47,-0.1,1.024,-4.5993",
        ellipse: "intl",
        datumName: "New Zealand Geodetic Datum 1949"
    };

    exports$3.osgb36 = {
        towgs84: "446.448,-125.157,542.060,0.1502,0.2470,0.8421,-20.4894",
        ellipse: "airy",
        datumName: "Airy 1830"
    };

    exports$3.s_jtsk = {
        towgs84: "589,76,480",
        ellipse: 'bessel',
        datumName: 'S-JTSK (Ferro)'
    };

    exports$3.beduaram = {
        towgs84: '-106,-87,188',
        ellipse: 'clrk80',
        datumName: 'Beduaram'
    };

    exports$3.gunung_segara = {
        towgs84: '-403,684,41',
        ellipse: 'bessel',
        datumName: 'Gunung Segara Jakarta'
    };

    exports$3.rnb72 = {
        towgs84: "106.869,-52.2978,103.724,-0.33657,0.456955,-1.84218,1",
        ellipse: "intl",
        datumName: "Reseau National Belge 1972"
    };

    function datum(datumCode, datum_params, a, b, es, ep2) {
        var out = {};

        if (datumCode === undefined || datumCode === 'none') {
            out.datum_type = PJD_NODATUM;
        } else {
            out.datum_type = PJD_WGS84;
        }

        if (datum_params) {
            out.datum_params = datum_params.map(parseFloat);
            if (out.datum_params[0] !== 0 || out.datum_params[1] !== 0 || out.datum_params[2] !== 0) {
                out.datum_type = PJD_3PARAM;
            }
            if (out.datum_params.length > 3) {
                if (out.datum_params[3] !== 0 || out.datum_params[4] !== 0 || out.datum_params[5] !== 0 || out.datum_params[6] !== 0) {
                    out.datum_type = PJD_7PARAM;
                    out.datum_params[3] *= SEC_TO_RAD;
                    out.datum_params[4] *= SEC_TO_RAD;
                    out.datum_params[5] *= SEC_TO_RAD;
                    out.datum_params[6] = (out.datum_params[6] / 1000000.0) + 1.0;
                }
            }
        }

        out.a = a; //datum object also uses these values
        out.b = b;
        out.es = es;
        out.ep2 = ep2;
        return out;
    }

    function Projection(srsCode,callback) {
        if (!(this instanceof Projection)) {
            return new Projection(srsCode);
        }
        callback = callback || function(error){
            if(error){
                throw error;
            }
        };
        var json = parse(srsCode);
        if(typeof json !== 'object'){
            callback(srsCode);
            return;
        }
        var ourProj = Projection.projections.get(json.projName);
        if(!ourProj){
            callback(srsCode);
            return;
        }
        if (json.datumCode && json.datumCode !== 'none') {
            var datumDef = match(exports$3, json.datumCode);
            if (datumDef) {
                json.datum_params = datumDef.towgs84 ? datumDef.towgs84.split(',') : null;
                json.ellps = datumDef.ellipse;
                json.datumName = datumDef.datumName ? datumDef.datumName : json.datumCode;
            }
        }
        json.k0 = json.k0 || 1.0;
        json.axis = json.axis || 'enu';
        json.ellps = json.ellps || 'wgs84';
        var sphere_ = sphere(json.a, json.b, json.rf, json.ellps, json.sphere);
        var ecc = eccentricity(sphere_.a, sphere_.b, sphere_.rf, json.R_A);
        var datumObj = json.datum || datum(json.datumCode, json.datum_params, sphere_.a, sphere_.b, ecc.es, ecc.ep2);

        extend(this, json); // transfer everything over from the projection because we don't know what we'll need
        extend(this, ourProj); // transfer all the methods from the projection

        // copy the 4 things over we calulated in deriveConstants.sphere
        this.a = sphere_.a;
        this.b = sphere_.b;
        this.rf = sphere_.rf;
        this.sphere = sphere_.sphere;

        // copy the 3 things we calculated in deriveConstants.eccentricity
        this.es = ecc.es;
        this.e = ecc.e;
        this.ep2 = ecc.ep2;

        // add in the datum object
        this.datum = datumObj;

        // init the projection
        this.init();

        // legecy callback from back in the day when it went to spatialreference.org
        callback(null, this);

    }
    Projection.projections = projections;
    Projection.projections.start();

    'use strict';
    function compareDatums(source, dest) {
        if (source.datum_type !== dest.datum_type) {
            return false; // false, datums are not equal
        } else if (source.a !== dest.a || Math.abs(source.es - dest.es) > 0.000000000050) {
            // the tolerance for es is to ensure that GRS80 and WGS84
            // are considered identical
            return false;
        } else if (source.datum_type === PJD_3PARAM) {
            return (source.datum_params[0] === dest.datum_params[0] && source.datum_params[1] === dest.datum_params[1] && source.datum_params[2] === dest.datum_params[2]);
        } else if (source.datum_type === PJD_7PARAM) {
            return (source.datum_params[0] === dest.datum_params[0] && source.datum_params[1] === dest.datum_params[1] && source.datum_params[2] === dest.datum_params[2] && source.datum_params[3] === dest.datum_params[3] && source.datum_params[4] === dest.datum_params[4] && source.datum_params[5] === dest.datum_params[5] && source.datum_params[6] === dest.datum_params[6]);
        } else {
            return true; // datums are equal
        }
    } // cs_compare_datums()

    /*
     * The function Convert_Geodetic_To_Geocentric converts geodetic coordinates
     * (latitude, longitude, and height) to geocentric coordinates (X, Y, Z),
     * according to the current ellipsoid parameters.
     *
     *    Latitude  : Geodetic latitude in radians                     (input)
     *    Longitude : Geodetic longitude in radians                    (input)
     *    Height    : Geodetic height, in meters                       (input)
     *    X         : Calculated Geocentric X coordinate, in meters    (output)
     *    Y         : Calculated Geocentric Y coordinate, in meters    (output)
     *    Z         : Calculated Geocentric Z coordinate, in meters    (output)
     *
     */
    function geodeticToGeocentric(p, es, a) {
        var Longitude = p.x;
        var Latitude = p.y;
        var Height = p.z ? p.z : 0; //Z value not always supplied

        var Rn; /*  Earth radius at location  */
        var Sin_Lat; /*  Math.sin(Latitude)  */
        var Sin2_Lat; /*  Square of Math.sin(Latitude)  */
        var Cos_Lat; /*  Math.cos(Latitude)  */

        /*
       ** Don't blow up if Latitude is just a little out of the value
       ** range as it may just be a rounding issue.  Also removed longitude
       ** test, it should be wrapped by Math.cos() and Math.sin().  NFW for PROJ.4, Sep/2001.
       */
        if (Latitude < -HALF_PI && Latitude > -1.001 * HALF_PI) {
            Latitude = -HALF_PI;
        } else if (Latitude > HALF_PI && Latitude < 1.001 * HALF_PI) {
            Latitude = HALF_PI;
        } else if (Latitude < -HALF_PI) {
            /* Latitude out of range */
            //..reportError('geocent:lat out of range:' + Latitude);
            return { x: -Infinity, y: -Infinity, z: p.z };
        } else if (Latitude > HALF_PI) {
            /* Latitude out of range */
            return { x: Infinity, y: Infinity, z: p.z };
        }

        if (Longitude > Math.PI) {
            Longitude -= (2 * Math.PI);
        }
        Sin_Lat = Math.sin(Latitude);
        Cos_Lat = Math.cos(Latitude);
        Sin2_Lat = Sin_Lat * Sin_Lat;
        Rn = a / (Math.sqrt(1.0e0 - es * Sin2_Lat));
        return {
            x: (Rn + Height) * Cos_Lat * Math.cos(Longitude),
            y: (Rn + Height) * Cos_Lat * Math.sin(Longitude),
            z: ((Rn * (1 - es)) + Height) * Sin_Lat
        };
    } // cs_geodetic_to_geocentric()

    function geocentricToGeodetic(p, es, a, b) {
        /* local defintions and variables */
        /* end-criterium of loop, accuracy of sin(Latitude) */
        var genau = 1e-12;
        var genau2 = (genau * genau);
        var maxiter = 30;

        var P; /* distance between semi-minor axis and location */
        var RR; /* distance between center and location */
        var CT; /* sin of geocentric latitude */
        var ST; /* cos of geocentric latitude */
        var RX;
        var RK;
        var RN; /* Earth radius at location */
        var CPHI0; /* cos of start or old geodetic latitude in iterations */
        var SPHI0; /* sin of start or old geodetic latitude in iterations */
        var CPHI; /* cos of searched geodetic latitude */
        var SPHI; /* sin of searched geodetic latitude */
        var SDPHI; /* end-criterium: addition-theorem of sin(Latitude(iter)-Latitude(iter-1)) */
        var iter; /* # of continous iteration, max. 30 is always enough (s.a.) */

        var X = p.x;
        var Y = p.y;
        var Z = p.z ? p.z : 0.0; //Z value not always supplied
        var Longitude;
        var Latitude;
        var Height;

        P = Math.sqrt(X * X + Y * Y);
        RR = Math.sqrt(X * X + Y * Y + Z * Z);

        /*      special cases for latitude and longitude */
        if (P / a < genau) {

            /*  special case, if P=0. (X=0., Y=0.) */
            Longitude = 0.0;

            /*  if (X,Y,Z)=(0.,0.,0.) then Height becomes semi-minor axis
         *  of ellipsoid (=center of mass), Latitude becomes PI/2 */
            if (RR / a < genau) {
                Latitude = HALF_PI;
                Height = -b;
                return {
                    x: p.x,
                    y: p.y,
                    z: p.z
                };
            }
        } else {
            /*  ellipsoidal (geodetic) longitude
         *  interval: -PI < Longitude <= +PI */
            Longitude = Math.atan2(Y, X);
        }

        /* --------------------------------------------------------------
       * Following iterative algorithm was developped by
       * "Institut for Erdmessung", University of Hannover, July 1988.
       * Internet: www.ife.uni-hannover.de
       * Iterative computation of CPHI,SPHI and Height.
       * Iteration of CPHI and SPHI to 10**-12 radian resp.
       * 2*10**-7 arcsec.
       * --------------------------------------------------------------
       */
        CT = Z / RR;
        ST = P / RR;
        RX = 1.0 / Math.sqrt(1.0 - es * (2.0 - es) * ST * ST);
        CPHI0 = ST * (1.0 - es) * RX;
        SPHI0 = CT * RX;
        iter = 0;

        /* loop to find sin(Latitude) resp. Latitude
       * until |sin(Latitude(iter)-Latitude(iter-1))| < genau */
        do {
            iter++;
            RN = a / Math.sqrt(1.0 - es * SPHI0 * SPHI0);

            /*  ellipsoidal (geodetic) height */
            Height = P * CPHI0 + Z * SPHI0 - RN * (1.0 - es * SPHI0 * SPHI0);

            RK = es * RN / (RN + Height);
            RX = 1.0 / Math.sqrt(1.0 - RK * (2.0 - RK) * ST * ST);
            CPHI = ST * (1.0 - RK) * RX;
            SPHI = CT * RX;
            SDPHI = SPHI * CPHI0 - CPHI * SPHI0;
            CPHI0 = CPHI;
            SPHI0 = SPHI;
        }
        while (SDPHI * SDPHI > genau2 && iter < maxiter);

        /*      ellipsoidal (geodetic) latitude */
        Latitude = Math.atan(SPHI / Math.abs(CPHI));
        return {
            x: Longitude,
            y: Latitude,
            z: Height
        };
    } // cs_geocentric_to_geodetic()

    /****************************************************************/
    // pj_geocentic_to_wgs84( p )
    //  p = point to transform in geocentric coordinates (x,y,z)


    /** point object, nothing fancy, just allows values to be
     passed back and forth by reference rather than by value.
     Other point classes may be used as long as they have
     x and y properties, which will get modified in the transform method.
     */
    function geocentricToWgs84(p, datum_type, datum_params) {

        if (datum_type === PJD_3PARAM) {
            // if( x[io] === HUGE_VAL )
            //    continue;
            return {
                x: p.x + datum_params[0],
                y: p.y + datum_params[1],
                z: p.z + datum_params[2],
            };
        } else if (datum_type === PJD_7PARAM) {
            var Dx_BF = datum_params[0];
            var Dy_BF = datum_params[1];
            var Dz_BF = datum_params[2];
            var Rx_BF = datum_params[3];
            var Ry_BF = datum_params[4];
            var Rz_BF = datum_params[5];
            var M_BF = datum_params[6];
            // if( x[io] === HUGE_VAL )
            //    continue;
            return {
                x: M_BF * (p.x - Rz_BF * p.y + Ry_BF * p.z) + Dx_BF,
                y: M_BF * (Rz_BF * p.x + p.y - Rx_BF * p.z) + Dy_BF,
                z: M_BF * (-Ry_BF * p.x + Rx_BF * p.y + p.z) + Dz_BF
            };
        }
    } // cs_geocentric_to_wgs84

    /****************************************************************/
    // pj_geocentic_from_wgs84()
    //  coordinate system definition,
    //  point to transform in geocentric coordinates (x,y,z)
    function geocentricFromWgs84(p, datum_type, datum_params) {

        if (datum_type === PJD_3PARAM) {
            //if( x[io] === HUGE_VAL )
            //    continue;
            return {
                x: p.x - datum_params[0],
                y: p.y - datum_params[1],
                z: p.z - datum_params[2],
            };

        } else if (datum_type === PJD_7PARAM) {
            var Dx_BF = datum_params[0];
            var Dy_BF = datum_params[1];
            var Dz_BF = datum_params[2];
            var Rx_BF = datum_params[3];
            var Ry_BF = datum_params[4];
            var Rz_BF = datum_params[5];
            var M_BF = datum_params[6];
            var x_tmp = (p.x - Dx_BF) / M_BF;
            var y_tmp = (p.y - Dy_BF) / M_BF;
            var z_tmp = (p.z - Dz_BF) / M_BF;
            //if( x[io] === HUGE_VAL )
            //    continue;

            return {
                x: x_tmp + Rz_BF * y_tmp - Ry_BF * z_tmp,
                y: -Rz_BF * x_tmp + y_tmp + Rx_BF * z_tmp,
                z: Ry_BF * x_tmp - Rx_BF * y_tmp + z_tmp
            };
        } //cs_geocentric_from_wgs84()
    }

    function checkParams(type) {
        return (type === PJD_3PARAM || type === PJD_7PARAM);
    }

    var datum_transform = function(source, dest, point) {
        // Short cut if the datums are identical.
        if (compareDatums(source, dest)) {
            return point; // in this case, zero is sucess,
            // whereas cs_compare_datums returns 1 to indicate TRUE
            // confusing, should fix this
        }

        // Explicitly skip datum transform by setting 'datum=none' as parameter for either source or dest
        if (source.datum_type === PJD_NODATUM || dest.datum_type === PJD_NODATUM) {
            return point;
        }

        // If this datum requires grid shifts, then apply it to geodetic coordinates.

        // Do we need to go through geocentric coordinates?
        if (source.es === dest.es && source.a === dest.a && !checkParams(source.datum_type) &&  !checkParams(dest.datum_type)) {
            return point;
        }

        // Convert to geocentric coordinates.
        point = geodeticToGeocentric(point, source.es, source.a);
        // Convert between datums
        if (checkParams(source.datum_type)) {
            point = geocentricToWgs84(point, source.datum_type, source.datum_params);
        }
        if (checkParams(dest.datum_type)) {
            point = geocentricFromWgs84(point, dest.datum_type, dest.datum_params);
        }
        return geocentricToGeodetic(point, dest.es, dest.a, dest.b);

    };

    var adjust_axis = function(crs, denorm, point) {
        var xin = point.x,
            yin = point.y,
            zin = point.z || 0.0;
        var v, t, i;
        var out = {};
        for (i = 0; i < 3; i++) {
            if (denorm && i === 2 && point.z === undefined) {
                continue;
            }
            if (i === 0) {
                v = xin;
                t = 'x';
            }
            else if (i === 1) {
                v = yin;
                t = 'y';
            }
            else {
                v = zin;
                t = 'z';
            }
            switch (crs.axis[i]) {
                case 'e':
                    out[t] = v;
                    break;
                case 'w':
                    out[t] = -v;
                    break;
                case 'n':
                    out[t] = v;
                    break;
                case 's':
                    out[t] = -v;
                    break;
                case 'u':
                    if (point[t] !== undefined) {
                        out.z = v;
                    }
                    break;
                case 'd':
                    if (point[t] !== undefined) {
                        out.z = -v;
                    }
                    break;
                default:
                    //console.log("ERROR: unknow axis ("+crs.axis[i]+") - check definition of "+crs.projName);
                    return null;
            }
        }
        return out;
    };

    var toPoint = function (array){
        var out = {
            x: array[0],
            y: array[1]
        };
        if (array.length>2) {
            out.z = array[2];
        }
        if (array.length>3) {
            out.m = array[3];
        }
        return out;
    };

    var checkSanity = function (point) {
        checkCoord(point.x);
        checkCoord(point.y);
    };
    function checkCoord(num) {
        if (typeof Number.isFinite === 'function') {
            if (Number.isFinite(num)) {
                return;
            }
            throw new TypeError('coordinates must be finite numbers');
        }
        if (typeof num !== 'number' || num !== num || !isFinite(num)) {
            throw new TypeError('coordinates must be finite numbers');
        }
    }

    function checkNotWGS(source, dest) {
        return ((source.datum.datum_type === PJD_3PARAM || source.datum.datum_type === PJD_7PARAM) && dest.datumCode !== 'WGS84') || ((dest.datum.datum_type === PJD_3PARAM || dest.datum.datum_type === PJD_7PARAM) && source.datumCode !== 'WGS84');
    }

    function transform(source, dest, point) {
        var wgs84;
        if (Array.isArray(point)) {
            point = toPoint(point);
        }
        checkSanity(point);
        // Workaround for datum shifts towgs84, if either source or destination projection is not wgs84
        if (source.datum && dest.datum && checkNotWGS(source, dest)) {
            wgs84 = new Projection('WGS84');
            point = transform(source, wgs84, point);
            source = wgs84;
        }
        // DGR, 2010/11/12
        if (source.axis !== 'enu') {
            point = adjust_axis(source, false, point);
        }
        // Transform source points to long/lat, if they aren't already.
        if (source.projName === 'longlat') {
            point = {
                x: point.x * D2R,
                y: point.y * D2R,
                z: point.z || 0
            };
        } else {
            if (source.to_meter) {
                point = {
                    x: point.x * source.to_meter,
                    y: point.y * source.to_meter,
                    z: point.z || 0
                };
            }
            point = source.inverse(point); // Convert Cartesian to longlat
        }
        // Adjust for the prime meridian if necessary
        if (source.from_greenwich) {
            point.x += source.from_greenwich;
        }

        // Convert datums if needed, and if possible.
        point = datum_transform(source.datum, dest.datum, point);

        // Adjust for the prime meridian if necessary
        if (dest.from_greenwich) {
            point = {
                x: point.x - dest.from_greenwich,
                y: point.y,
                z: point.z || 0
            };
        }

        if (dest.projName === 'longlat') {
            // convert radians to decimal degrees
            point = {
                x: point.x * R2D,
                y: point.y * R2D,
                z: point.z || 0
            };
        } else { // else project
            point = dest.forward(point);
            if (dest.to_meter) {
                point = {
                    x: point.x / dest.to_meter,
                    y: point.y / dest.to_meter,
                    z: point.z || 0
                };
            }
        }

        // DGR, 2010/11/12
        if (dest.axis !== 'enu') {
            return adjust_axis(dest, true, point);
        }

        return point;
    }

    var wgs84 = Projection('WGS84');

    function transformer(from, to, coords) {
        var transformedArray, out, keys;
        if (Array.isArray(coords)) {
            transformedArray = transform(from, to, coords) || {x: NaN, y: NaN};
            if (coords.length > 2) {
                if ((typeof from.name !== 'undefined' && from.name === 'geocent') || (typeof to.name !== 'undefined' && to.name === 'geocent')) {
                    if (typeof transformedArray.z === 'number') {
                        return [transformedArray.x, transformedArray.y, transformedArray.z].concat(coords.splice(3));
                    } else {
                        return [transformedArray.x, transformedArray.y, coords[2]].concat(coords.splice(3));
                    }
                } else {
                    return [transformedArray.x, transformedArray.y].concat(coords.splice(2));
                }
            } else {
                return [transformedArray.x, transformedArray.y];
            }
        } else {
            out = transform(from, to, coords);
            keys = Object.keys(coords);
            if (keys.length === 2) {
                return out;
            }
            keys.forEach(function (key) {
                if ((typeof from.name !== 'undefined' && from.name === 'geocent') || (typeof to.name !== 'undefined' && to.name === 'geocent')) {
                    if (key === 'x' || key === 'y' || key === 'z') {
                        return;
                    }
                } else {
                    if (key === 'x' || key === 'y') {
                        return;
                    }
                }
                out[key] = coords[key];
            });
            return out;
        }
    }

    function checkProj(item) {
        if (item instanceof Projection) {
            return item;
        }
        if (item.oProj) {
            return item.oProj;
        }
        return Projection(item);
    }

    function proj4$1(fromProj, toProj, coord) {
        fromProj = checkProj(fromProj);
        var single = false;
        var obj;
        if (typeof toProj === 'undefined') {
            toProj = fromProj;
            fromProj = wgs84;
            single = true;
        } else if (typeof toProj.x !== 'undefined' || Array.isArray(toProj)) {
            coord = toProj;
            toProj = fromProj;
            fromProj = wgs84;
            single = true;
        }
        toProj = checkProj(toProj);
        if (coord) {
            return transformer(fromProj, toProj, coord);
        } else {
            obj = {
                forward: function (coords) {
                    return transformer(fromProj, toProj, coords);
                },
                inverse: function (coords) {
                    return transformer(toProj, fromProj, coords);
                }
            };
            if (single) {
                obj.oProj = toProj;
            }
            return obj;
        }
    }

    /**
     * UTM zones are grouped, and assigned to one of a group of 6
     * sets.
     *
     * {int} @private
     */
    var NUM_100K_SETS = 6;

    /**
     * The column letters (for easting) of the lower left value, per
     * set.
     *
     * {string} @private
     */
    var SET_ORIGIN_COLUMN_LETTERS = 'AJSAJS';

    /**
     * The row letters (for northing) of the lower left value, per
     * set.
     *
     * {string} @private
     */
    var SET_ORIGIN_ROW_LETTERS = 'AFAFAF';

    var A = 65; // A
    var I = 73; // I
    var O = 79; // O
    var V = 86; // V
    var Z = 90; // Z
    var mgrs = {
        forward: forward$1,
        inverse: inverse$1,
        toPoint: toPoint$1
    };
    /**
     * Conversion of lat/lon to MGRS.
     *
     * @param {object} ll Object literal with lat and lon properties on a
     *     WGS84 ellipsoid.
     * @param {int} accuracy Accuracy in digits (5 for 1 m, 4 for 10 m, 3 for
     *      100 m, 2 for 1000 m or 1 for 10000 m). Optional, default is 5.
     * @return {string} the MGRS string for the given location and accuracy.
     */
    function forward$1(ll, accuracy) {
        accuracy = accuracy || 5; // default accuracy 1m
        return encode(LLtoUTM({
            lat: ll[1],
            lon: ll[0]
        }), accuracy);
    }

    /**
     * Conversion of MGRS to lat/lon.
     *
     * @param {string} mgrs MGRS string.
     * @return {array} An array with left (longitude), bottom (latitude), right
     *     (longitude) and top (latitude) values in WGS84, representing the
     *     bounding box for the provided MGRS reference.
     */
    function inverse$1(mgrs) {
        var bbox = UTMtoLL(decode(mgrs.toUpperCase()));
        if (bbox.lat && bbox.lon) {
            return [bbox.lon, bbox.lat, bbox.lon, bbox.lat];
        }
        return [bbox.left, bbox.bottom, bbox.right, bbox.top];
    }

    function toPoint$1(mgrs) {
        var bbox = UTMtoLL(decode(mgrs.toUpperCase()));
        if (bbox.lat && bbox.lon) {
            return [bbox.lon, bbox.lat];
        }
        return [(bbox.left + bbox.right) / 2, (bbox.top + bbox.bottom) / 2];
    }
    /**
     * Conversion from degrees to radians.
     *
     * @private
     * @param {number} deg the angle in degrees.
     * @return {number} the angle in radians.
     */
    function degToRad(deg) {
        return (deg * (Math.PI / 180.0));
    }

    /**
     * Conversion from radians to degrees.
     *
     * @private
     * @param {number} rad the angle in radians.
     * @return {number} the angle in degrees.
     */
    function radToDeg(rad) {
        return (180.0 * (rad / Math.PI));
    }

    /**
     * Converts a set of Longitude and Latitude co-ordinates to UTM
     * using the WGS84 ellipsoid.
     *
     * @private
     * @param {object} ll Object literal with lat and lon properties
     *     representing the WGS84 coordinate to be converted.
     * @return {object} Object literal containing the UTM value with easting,
     *     northing, zoneNumber and zoneLetter properties, and an optional
     *     accuracy property in digits. Returns null if the conversion failed.
     */
    function LLtoUTM(ll) {
        var Lat = ll.lat;
        var Long = ll.lon;
        var a = 6378137.0; //ellip.radius;
        var eccSquared = 0.00669438; //ellip.eccsq;
        var k0 = 0.9996;
        var LongOrigin;
        var eccPrimeSquared;
        var N, T, C, A, M;
        var LatRad = degToRad(Lat);
        var LongRad = degToRad(Long);
        var LongOriginRad;
        var ZoneNumber;
        // (int)
        ZoneNumber = Math.floor((Long + 180) / 6) + 1;

        //Make sure the longitude 180.00 is in Zone 60
        if (Long === 180) {
            ZoneNumber = 60;
        }

        // Special zone for Norway
        if (Lat >= 56.0 && Lat < 64.0 && Long >= 3.0 && Long < 12.0) {
            ZoneNumber = 32;
        }

        // Special zones for Svalbard
        if (Lat >= 72.0 && Lat < 84.0) {
            if (Long >= 0.0 && Long < 9.0) {
                ZoneNumber = 31;
            }
            else if (Long >= 9.0 && Long < 21.0) {
                ZoneNumber = 33;
            }
            else if (Long >= 21.0 && Long < 33.0) {
                ZoneNumber = 35;
            }
            else if (Long >= 33.0 && Long < 42.0) {
                ZoneNumber = 37;
            }
        }

        LongOrigin = (ZoneNumber - 1) * 6 - 180 + 3; //+3 puts origin
        // in middle of
        // zone
        LongOriginRad = degToRad(LongOrigin);

        eccPrimeSquared = (eccSquared) / (1 - eccSquared);

        N = a / Math.sqrt(1 - eccSquared * Math.sin(LatRad) * Math.sin(LatRad));
        T = Math.tan(LatRad) * Math.tan(LatRad);
        C = eccPrimeSquared * Math.cos(LatRad) * Math.cos(LatRad);
        A = Math.cos(LatRad) * (LongRad - LongOriginRad);

        M = a * ((1 - eccSquared / 4 - 3 * eccSquared * eccSquared / 64 - 5 * eccSquared * eccSquared * eccSquared / 256) * LatRad - (3 * eccSquared / 8 + 3 * eccSquared * eccSquared / 32 + 45 * eccSquared * eccSquared * eccSquared / 1024) * Math.sin(2 * LatRad) + (15 * eccSquared * eccSquared / 256 + 45 * eccSquared * eccSquared * eccSquared / 1024) * Math.sin(4 * LatRad) - (35 * eccSquared * eccSquared * eccSquared / 3072) * Math.sin(6 * LatRad));

        var UTMEasting = (k0 * N * (A + (1 - T + C) * A * A * A / 6.0 + (5 - 18 * T + T * T + 72 * C - 58 * eccPrimeSquared) * A * A * A * A * A / 120.0) + 500000.0);

        var UTMNorthing = (k0 * (M + N * Math.tan(LatRad) * (A * A / 2 + (5 - T + 9 * C + 4 * C * C) * A * A * A * A / 24.0 + (61 - 58 * T + T * T + 600 * C - 330 * eccPrimeSquared) * A * A * A * A * A * A / 720.0)));
        if (Lat < 0.0) {
            UTMNorthing += 10000000.0; //10000000 meter offset for
            // southern hemisphere
        }

        return {
            northing: Math.round(UTMNorthing),
            easting: Math.round(UTMEasting),
            zoneNumber: ZoneNumber,
            zoneLetter: getLetterDesignator(Lat)
        };
    }

    /**
     * Converts UTM coords to lat/long, using the WGS84 ellipsoid. This is a convenience
     * class where the Zone can be specified as a single string eg."60N" which
     * is then broken down into the ZoneNumber and ZoneLetter.
     *
     * @private
     * @param {object} utm An object literal with northing, easting, zoneNumber
     *     and zoneLetter properties. If an optional accuracy property is
     *     provided (in meters), a bounding box will be returned instead of
     *     latitude and longitude.
     * @return {object} An object literal containing either lat and lon values
     *     (if no accuracy was provided), or top, right, bottom and left values
     *     for the bounding box calculated according to the provided accuracy.
     *     Returns null if the conversion failed.
     */
    function UTMtoLL(utm) {

        var UTMNorthing = utm.northing;
        var UTMEasting = utm.easting;
        var zoneLetter = utm.zoneLetter;
        var zoneNumber = utm.zoneNumber;
        // check the ZoneNummber is valid
        if (zoneNumber < 0 || zoneNumber > 60) {
            return null;
        }

        var k0 = 0.9996;
        var a = 6378137.0; //ellip.radius;
        var eccSquared = 0.00669438; //ellip.eccsq;
        var eccPrimeSquared;
        var e1 = (1 - Math.sqrt(1 - eccSquared)) / (1 + Math.sqrt(1 - eccSquared));
        var N1, T1, C1, R1, D, M;
        var LongOrigin;
        var mu, phi1Rad;

        // remove 500,000 meter offset for longitude
        var x = UTMEasting - 500000.0;
        var y = UTMNorthing;

        // We must know somehow if we are in the Northern or Southern
        // hemisphere, this is the only time we use the letter So even
        // if the Zone letter isn't exactly correct it should indicate
        // the hemisphere correctly
        if (zoneLetter < 'N') {
            y -= 10000000.0; // remove 10,000,000 meter offset used
            // for southern hemisphere
        }

        // There are 60 zones with zone 1 being at West -180 to -174
        LongOrigin = (zoneNumber - 1) * 6 - 180 + 3; // +3 puts origin
        // in middle of
        // zone

        eccPrimeSquared = (eccSquared) / (1 - eccSquared);

        M = y / k0;
        mu = M / (a * (1 - eccSquared / 4 - 3 * eccSquared * eccSquared / 64 - 5 * eccSquared * eccSquared * eccSquared / 256));

        phi1Rad = mu + (3 * e1 / 2 - 27 * e1 * e1 * e1 / 32) * Math.sin(2 * mu) + (21 * e1 * e1 / 16 - 55 * e1 * e1 * e1 * e1 / 32) * Math.sin(4 * mu) + (151 * e1 * e1 * e1 / 96) * Math.sin(6 * mu);
        // double phi1 = ProjMath.radToDeg(phi1Rad);

        N1 = a / Math.sqrt(1 - eccSquared * Math.sin(phi1Rad) * Math.sin(phi1Rad));
        T1 = Math.tan(phi1Rad) * Math.tan(phi1Rad);
        C1 = eccPrimeSquared * Math.cos(phi1Rad) * Math.cos(phi1Rad);
        R1 = a * (1 - eccSquared) / Math.pow(1 - eccSquared * Math.sin(phi1Rad) * Math.sin(phi1Rad), 1.5);
        D = x / (N1 * k0);

        var lat = phi1Rad - (N1 * Math.tan(phi1Rad) / R1) * (D * D / 2 - (5 + 3 * T1 + 10 * C1 - 4 * C1 * C1 - 9 * eccPrimeSquared) * D * D * D * D / 24 + (61 + 90 * T1 + 298 * C1 + 45 * T1 * T1 - 252 * eccPrimeSquared - 3 * C1 * C1) * D * D * D * D * D * D / 720);
        lat = radToDeg(lat);

        var lon = (D - (1 + 2 * T1 + C1) * D * D * D / 6 + (5 - 2 * C1 + 28 * T1 - 3 * C1 * C1 + 8 * eccPrimeSquared + 24 * T1 * T1) * D * D * D * D * D / 120) / Math.cos(phi1Rad);
        lon = LongOrigin + radToDeg(lon);

        var result;
        if (utm.accuracy) {
            var topRight = UTMtoLL({
                northing: utm.northing + utm.accuracy,
                easting: utm.easting + utm.accuracy,
                zoneLetter: utm.zoneLetter,
                zoneNumber: utm.zoneNumber
            });
            result = {
                top: topRight.lat,
                right: topRight.lon,
                bottom: lat,
                left: lon
            };
        }
        else {
            result = {
                lat: lat,
                lon: lon
            };
        }
        return result;
    }

    /**
     * Calculates the MGRS letter designator for the given latitude.
     *
     * @private
     * @param {number} lat The latitude in WGS84 to get the letter designator
     *     for.
     * @return {char} The letter designator.
     */
    function getLetterDesignator(lat) {
        //This is here as an error flag to show that the Latitude is
        //outside MGRS limits
        var LetterDesignator = 'Z';

        if ((84 >= lat) && (lat >= 72)) {
            LetterDesignator = 'X';
        }
        else if ((72 > lat) && (lat >= 64)) {
            LetterDesignator = 'W';
        }
        else if ((64 > lat) && (lat >= 56)) {
            LetterDesignator = 'V';
        }
        else if ((56 > lat) && (lat >= 48)) {
            LetterDesignator = 'U';
        }
        else if ((48 > lat) && (lat >= 40)) {
            LetterDesignator = 'T';
        }
        else if ((40 > lat) && (lat >= 32)) {
            LetterDesignator = 'S';
        }
        else if ((32 > lat) && (lat >= 24)) {
            LetterDesignator = 'R';
        }
        else if ((24 > lat) && (lat >= 16)) {
            LetterDesignator = 'Q';
        }
        else if ((16 > lat) && (lat >= 8)) {
            LetterDesignator = 'P';
        }
        else if ((8 > lat) && (lat >= 0)) {
            LetterDesignator = 'N';
        }
        else if ((0 > lat) && (lat >= -8)) {
            LetterDesignator = 'M';
        }
        else if ((-8 > lat) && (lat >= -16)) {
            LetterDesignator = 'L';
        }
        else if ((-16 > lat) && (lat >= -24)) {
            LetterDesignator = 'K';
        }
        else if ((-24 > lat) && (lat >= -32)) {
            LetterDesignator = 'J';
        }
        else if ((-32 > lat) && (lat >= -40)) {
            LetterDesignator = 'H';
        }
        else if ((-40 > lat) && (lat >= -48)) {
            LetterDesignator = 'G';
        }
        else if ((-48 > lat) && (lat >= -56)) {
            LetterDesignator = 'F';
        }
        else if ((-56 > lat) && (lat >= -64)) {
            LetterDesignator = 'E';
        }
        else if ((-64 > lat) && (lat >= -72)) {
            LetterDesignator = 'D';
        }
        else if ((-72 > lat) && (lat >= -80)) {
            LetterDesignator = 'C';
        }
        return LetterDesignator;
    }

    /**
     * Encodes a UTM location as MGRS string.
     *
     * @private
     * @param {object} utm An object literal with easting, northing,
     *     zoneLetter, zoneNumber
     * @param {number} accuracy Accuracy in digits (1-5).
     * @return {string} MGRS string for the given UTM location.
     */
    function encode(utm, accuracy) {
        // prepend with leading zeroes
        var seasting = "00000" + utm.easting,
            snorthing = "00000" + utm.northing;

        return utm.zoneNumber + utm.zoneLetter + get100kID(utm.easting, utm.northing, utm.zoneNumber) + seasting.substr(seasting.length - 5, accuracy) + snorthing.substr(snorthing.length - 5, accuracy);
    }

    /**
     * Get the two letter 100k designator for a given UTM easting,
     * northing and zone number value.
     *
     * @private
     * @param {number} easting
     * @param {number} northing
     * @param {number} zoneNumber
     * @return the two letter 100k designator for the given UTM location.
     */
    function get100kID(easting, northing, zoneNumber) {
        var setParm = get100kSetForZone(zoneNumber);
        var setColumn = Math.floor(easting / 100000);
        var setRow = Math.floor(northing / 100000) % 20;
        return getLetter100kID(setColumn, setRow, setParm);
    }

    /**
     * Given a UTM zone number, figure out the MGRS 100K set it is in.
     *
     * @private
     * @param {number} i An UTM zone number.
     * @return {number} the 100k set the UTM zone is in.
     */
    function get100kSetForZone(i) {
        var setParm = i % NUM_100K_SETS;
        if (setParm === 0) {
            setParm = NUM_100K_SETS;
        }

        return setParm;
    }

    /**
     * Get the two-letter MGRS 100k designator given information
     * translated from the UTM northing, easting and zone number.
     *
     * @private
     * @param {number} column the column index as it relates to the MGRS
     *        100k set spreadsheet, created from the UTM easting.
     *        Values are 1-8.
     * @param {number} row the row index as it relates to the MGRS 100k set
     *        spreadsheet, created from the UTM northing value. Values
     *        are from 0-19.
     * @param {number} parm the set block, as it relates to the MGRS 100k set
     *        spreadsheet, created from the UTM zone. Values are from
     *        1-60.
     * @return two letter MGRS 100k code.
     */
    function getLetter100kID(column, row, parm) {
        // colOrigin and rowOrigin are the letters at the origin of the set
        var index = parm - 1;
        var colOrigin = SET_ORIGIN_COLUMN_LETTERS.charCodeAt(index);
        var rowOrigin = SET_ORIGIN_ROW_LETTERS.charCodeAt(index);

        // colInt and rowInt are the letters to build to return
        var colInt = colOrigin + column - 1;
        var rowInt = rowOrigin + row;
        var rollover = false;

        if (colInt > Z) {
            colInt = colInt - Z + A - 1;
            rollover = true;
        }

        if (colInt === I || (colOrigin < I && colInt > I) || ((colInt > I || colOrigin < I) && rollover)) {
            colInt++;
        }

        if (colInt === O || (colOrigin < O && colInt > O) || ((colInt > O || colOrigin < O) && rollover)) {
            colInt++;

            if (colInt === I) {
                colInt++;
            }
        }

        if (colInt > Z) {
            colInt = colInt - Z + A - 1;
        }

        if (rowInt > V) {
            rowInt = rowInt - V + A - 1;
            rollover = true;
        }
        else {
            rollover = false;
        }

        if (((rowInt === I) || ((rowOrigin < I) && (rowInt > I))) || (((rowInt > I) || (rowOrigin < I)) && rollover)) {
            rowInt++;
        }

        if (((rowInt === O) || ((rowOrigin < O) && (rowInt > O))) || (((rowInt > O) || (rowOrigin < O)) && rollover)) {
            rowInt++;

            if (rowInt === I) {
                rowInt++;
            }
        }

        if (rowInt > V) {
            rowInt = rowInt - V + A - 1;
        }

        var twoLetter = String.fromCharCode(colInt) + String.fromCharCode(rowInt);
        return twoLetter;
    }

    /**
     * Decode the UTM parameters from a MGRS string.
     *
     * @private
     * @param {string} mgrsString an UPPERCASE coordinate string is expected.
     * @return {object} An object literal with easting, northing, zoneLetter,
     *     zoneNumber and accuracy (in meters) properties.
     */
    function decode(mgrsString) {

        if (mgrsString && mgrsString.length === 0) {
            throw ("MGRSPoint coverting from nothing");
        }

        var length = mgrsString.length;

        var hunK = null;
        var sb = "";
        var testChar;
        var i = 0;

        // get Zone number
        while (!(/[A-Z]/).test(testChar = mgrsString.charAt(i))) {
            if (i >= 2) {
                throw ("MGRSPoint bad conversion from: " + mgrsString);
            }
            sb += testChar;
            i++;
        }

        var zoneNumber = parseInt(sb, 10);

        if (i === 0 || i + 3 > length) {
            // A good MGRS string has to be 4-5 digits long,
            // ##AAA/#AAA at least.
            throw ("MGRSPoint bad conversion from: " + mgrsString);
        }

        var zoneLetter = mgrsString.charAt(i++);

        // Should we check the zone letter here? Why not.
        if (zoneLetter <= 'A' || zoneLetter === 'B' || zoneLetter === 'Y' || zoneLetter >= 'Z' || zoneLetter === 'I' || zoneLetter === 'O') {
            throw ("MGRSPoint zone letter " + zoneLetter + " not handled: " + mgrsString);
        }

        hunK = mgrsString.substring(i, i += 2);

        var set = get100kSetForZone(zoneNumber);

        var east100k = getEastingFromChar(hunK.charAt(0), set);
        var north100k = getNorthingFromChar(hunK.charAt(1), set);

        // We have a bug where the northing may be 2000000 too low.
        // How
        // do we know when to roll over?

        while (north100k < getMinNorthing(zoneLetter)) {
            north100k += 2000000;
        }

        // calculate the char index for easting/northing separator
        var remainder = length - i;

        if (remainder % 2 !== 0) {
            throw ("MGRSPoint has to have an even number \nof digits after the zone letter and two 100km letters - front \nhalf for easting meters, second half for \nnorthing meters" + mgrsString);
        }

        var sep = remainder / 2;

        var sepEasting = 0.0;
        var sepNorthing = 0.0;
        var accuracyBonus, sepEastingString, sepNorthingString, easting, northing;
        if (sep > 0) {
            accuracyBonus = 100000.0 / Math.pow(10, sep);
            sepEastingString = mgrsString.substring(i, i + sep);
            sepEasting = parseFloat(sepEastingString) * accuracyBonus;
            sepNorthingString = mgrsString.substring(i + sep);
            sepNorthing = parseFloat(sepNorthingString) * accuracyBonus;
        }

        easting = sepEasting + east100k;
        northing = sepNorthing + north100k;

        return {
            easting: easting,
            northing: northing,
            zoneLetter: zoneLetter,
            zoneNumber: zoneNumber,
            accuracy: accuracyBonus
        };
    }

    /**
     * Given the first letter from a two-letter MGRS 100k zone, and given the
     * MGRS table set for the zone number, figure out the easting value that
     * should be added to the other, secondary easting value.
     *
     * @private
     * @param {char} e The first letter from a two-letter MGRS 100´k zone.
     * @param {number} set The MGRS table set for the zone number.
     * @return {number} The easting value for the given letter and set.
     */
    function getEastingFromChar(e, set) {
        // colOrigin is the letter at the origin of the set for the
        // column
        var curCol = SET_ORIGIN_COLUMN_LETTERS.charCodeAt(set - 1);
        var eastingValue = 100000.0;
        var rewindMarker = false;

        while (curCol !== e.charCodeAt(0)) {
            curCol++;
            if (curCol === I) {
                curCol++;
            }
            if (curCol === O) {
                curCol++;
            }
            if (curCol > Z) {
                if (rewindMarker) {
                    throw ("Bad character: " + e);
                }
                curCol = A;
                rewindMarker = true;
            }
            eastingValue += 100000.0;
        }

        return eastingValue;
    }

    /**
     * Given the second letter from a two-letter MGRS 100k zone, and given the
     * MGRS table set for the zone number, figure out the northing value that
     * should be added to the other, secondary northing value. You have to
     * remember that Northings are determined from the equator, and the vertical
     * cycle of letters mean a 2000000 additional northing meters. This happens
     * approx. every 18 degrees of latitude. This method does *NOT* count any
     * additional northings. You have to figure out how many 2000000 meters need
     * to be added for the zone letter of the MGRS coordinate.
     *
     * @private
     * @param {char} n Second letter of the MGRS 100k zone
     * @param {number} set The MGRS table set number, which is dependent on the
     *     UTM zone number.
     * @return {number} The northing value for the given letter and set.
     */
    function getNorthingFromChar(n, set) {

        if (n > 'V') {
            throw ("MGRSPoint given invalid Northing " + n);
        }

        // rowOrigin is the letter at the origin of the set for the
        // column
        var curRow = SET_ORIGIN_ROW_LETTERS.charCodeAt(set - 1);
        var northingValue = 0.0;
        var rewindMarker = false;

        while (curRow !== n.charCodeAt(0)) {
            curRow++;
            if (curRow === I) {
                curRow++;
            }
            if (curRow === O) {
                curRow++;
            }
            // fixing a bug making whole application hang in this loop
            // when 'n' is a wrong character
            if (curRow > V) {
                if (rewindMarker) { // making sure that this loop ends
                    throw ("Bad character: " + n);
                }
                curRow = A;
                rewindMarker = true;
            }
            northingValue += 100000.0;
        }

        return northingValue;
    }

    /**
     * The function getMinNorthing returns the minimum northing value of a MGRS
     * zone.
     *
     * Ported from Geotrans' c Lattitude_Band_Value structure table.
     *
     * @private
     * @param {char} zoneLetter The MGRS zone to get the min northing for.
     * @return {number}
     */
    function getMinNorthing(zoneLetter) {
        var northing;
        switch (zoneLetter) {
            case 'C':
                northing = 1100000.0;
                break;
            case 'D':
                northing = 2000000.0;
                break;
            case 'E':
                northing = 2800000.0;
                break;
            case 'F':
                northing = 3700000.0;
                break;
            case 'G':
                northing = 4600000.0;
                break;
            case 'H':
                northing = 5500000.0;
                break;
            case 'J':
                northing = 6400000.0;
                break;
            case 'K':
                northing = 7300000.0;
                break;
            case 'L':
                northing = 8200000.0;
                break;
            case 'M':
                northing = 9100000.0;
                break;
            case 'N':
                northing = 0.0;
                break;
            case 'P':
                northing = 800000.0;
                break;
            case 'Q':
                northing = 1700000.0;
                break;
            case 'R':
                northing = 2600000.0;
                break;
            case 'S':
                northing = 3500000.0;
                break;
            case 'T':
                northing = 4400000.0;
                break;
            case 'U':
                northing = 5300000.0;
                break;
            case 'V':
                northing = 6200000.0;
                break;
            case 'W':
                northing = 7000000.0;
                break;
            case 'X':
                northing = 7900000.0;
                break;
            default:
                northing = -1.0;
        }
        if (northing >= 0.0) {
            return northing;
        }
        else {
            throw ("Invalid zone letter: " + zoneLetter);
        }

    }

    function Point(x, y, z) {
        if (!(this instanceof Point)) {
            return new Point(x, y, z);
        }
        if (Array.isArray(x)) {
            this.x = x[0];
            this.y = x[1];
            this.z = x[2] || 0.0;
        } else if(typeof x === 'object') {
            this.x = x.x;
            this.y = x.y;
            this.z = x.z || 0.0;
        } else if (typeof x === 'string' && typeof y === 'undefined') {
            var coords = x.split(',');
            this.x = parseFloat(coords[0], 10);
            this.y = parseFloat(coords[1], 10);
            this.z = parseFloat(coords[2], 10) || 0.0;
        } else {
            this.x = x;
            this.y = y;
            this.z = z || 0.0;
        }
        console.warn('proj4.Point will be removed in version 3, use proj4.toPoint');
    }

    Point.fromMGRS = function(mgrsStr) {
        return new Point(toPoint$1(mgrsStr));
    };
    Point.prototype.toMGRS = function(accuracy) {
        return forward$1([this.x, this.y], accuracy);
    };

    var C00 = 1;
    var C02 = 0.25;
    var C04 = 0.046875;
    var C06 = 0.01953125;
    var C08 = 0.01068115234375;
    var C22 = 0.75;
    var C44 = 0.46875;
    var C46 = 0.01302083333333333333;
    var C48 = 0.00712076822916666666;
    var C66 = 0.36458333333333333333;
    var C68 = 0.00569661458333333333;
    var C88 = 0.3076171875;

    var pj_enfn = function(es) {
        var en = [];
        en[0] = C00 - es * (C02 + es * (C04 + es * (C06 + es * C08)));
        en[1] = es * (C22 - es * (C04 + es * (C06 + es * C08)));
        var t = es * es;
        en[2] = t * (C44 - es * (C46 + es * C48));
        t *= es;
        en[3] = t * (C66 - es * C68);
        en[4] = t * es * C88;
        return en;
    };

    var pj_mlfn = function(phi, sphi, cphi, en) {
        cphi *= sphi;
        sphi *= sphi;
        return (en[0] * phi - cphi * (en[1] + sphi * (en[2] + sphi * (en[3] + sphi * en[4]))));
    };

    var MAX_ITER = 20;

    var pj_inv_mlfn = function(arg, es, en) {
        var k = 1 / (1 - es);
        var phi = arg;
        for (var i = MAX_ITER; i; --i) { /* rarely goes over 2 iterations */
            var s = Math.sin(phi);
            var t = 1 - es * s * s;
            //t = this.pj_mlfn(phi, s, Math.cos(phi), en) - arg;
            //phi -= t * (t * Math.sqrt(t)) * k;
            t = (pj_mlfn(phi, s, Math.cos(phi), en) - arg) * (t * Math.sqrt(t)) * k;
            phi -= t;
            if (Math.abs(t) < EPSLN) {
                return phi;
            }
        }
        //..reportError("cass:pj_inv_mlfn: Convergence error");
        return phi;
    };

    // Heavily based on this tmerc projection implementation
    // https://github.com/mbloch/mapshaper-proj/blob/master/src/projections/tmerc.js

    function init$2() {
        this.x0 = this.x0 !== undefined ? this.x0 : 0;
        this.y0 = this.y0 !== undefined ? this.y0 : 0;
        this.long0 = this.long0 !== undefined ? this.long0 : 0;
        this.lat0 = this.lat0 !== undefined ? this.lat0 : 0;

        if (this.es) {
            this.en = pj_enfn(this.es);
            this.ml0 = pj_mlfn(this.lat0, Math.sin(this.lat0), Math.cos(this.lat0), this.en);
        }
    }

    /**
     Transverse Mercator Forward  - long/lat to x/y
     long/lat in radians
     */
    function forward$2(p) {
        var lon = p.x;
        var lat = p.y;

        var delta_lon = adjust_lon(lon - this.long0);
        var con;
        var x, y;
        var sin_phi = Math.sin(lat);
        var cos_phi = Math.cos(lat);

        if (!this.es) {
            var b = cos_phi * Math.sin(delta_lon);

            if ((Math.abs(Math.abs(b) - 1)) < EPSLN) {
                return (93);
            }
            else {
                x = 0.5 * this.a * this.k0 * Math.log((1 + b) / (1 - b)) + this.x0;
                y = cos_phi * Math.cos(delta_lon) / Math.sqrt(1 - Math.pow(b, 2));
                b = Math.abs(y);

                if (b >= 1) {
                    if ((b - 1) > EPSLN) {
                        return (93);
                    }
                    else {
                        y = 0;
                    }
                }
                else {
                    y = Math.acos(y);
                }

                if (lat < 0) {
                    y = -y;
                }

                y = this.a * this.k0 * (y - this.lat0) + this.y0;
            }
        }
        else {
            var al = cos_phi * delta_lon;
            var als = Math.pow(al, 2);
            var c = this.ep2 * Math.pow(cos_phi, 2);
            var cs = Math.pow(c, 2);
            var tq = Math.abs(cos_phi) > EPSLN ? Math.tan(lat) : 0;
            var t = Math.pow(tq, 2);
            var ts = Math.pow(t, 2);
            con = 1 - this.es * Math.pow(sin_phi, 2);
            al = al / Math.sqrt(con);
            var ml = pj_mlfn(lat, sin_phi, cos_phi, this.en);

            x = this.a * (this.k0 * al * (1 +
                als / 6 * (1 - t + c +
                    als / 20 * (5 - 18 * t + ts + 14 * c - 58 * t * c +
                        als / 42 * (61 + 179 * ts - ts * t - 479 * t))))) +
                this.x0;

            y = this.a * (this.k0 * (ml - this.ml0 +
                sin_phi * delta_lon * al / 2 * (1 +
                    als / 12 * (5 - t + 9 * c + 4 * cs +
                        als / 30 * (61 + ts - 58 * t + 270 * c - 330 * t * c +
                            als / 56 * (1385 + 543 * ts - ts * t - 3111 * t)))))) +
                this.y0;
        }

        p.x = x;
        p.y = y;

        return p;
    }

    /**
     Transverse Mercator Inverse  -  x/y to long/lat
     */
    function inverse$2(p) {
        var con, phi;
        var lat, lon;
        var x = (p.x - this.x0) * (1 / this.a);
        var y = (p.y - this.y0) * (1 / this.a);

        if (!this.es) {
            var f = Math.exp(x / this.k0);
            var g = 0.5 * (f - 1 / f);
            var temp = this.lat0 + y / this.k0;
            var h = Math.cos(temp);
            con = Math.sqrt((1 - Math.pow(h, 2)) / (1 + Math.pow(g, 2)));
            lat = Math.asin(con);

            if (y < 0) {
                lat = -lat;
            }

            if ((g === 0) && (h === 0)) {
                lon = 0;
            }
            else {
                lon = adjust_lon(Math.atan2(g, h) + this.long0);
            }
        }
        else { // ellipsoidal form
            con = this.ml0 + y / this.k0;
            phi = pj_inv_mlfn(con, this.es, this.en);

            if (Math.abs(phi) < HALF_PI) {
                var sin_phi = Math.sin(phi);
                var cos_phi = Math.cos(phi);
                var tan_phi = Math.abs(cos_phi) > EPSLN ? Math.tan(phi) : 0;
                var c = this.ep2 * Math.pow(cos_phi, 2);
                var cs = Math.pow(c, 2);
                var t = Math.pow(tan_phi, 2);
                var ts = Math.pow(t, 2);
                con = 1 - this.es * Math.pow(sin_phi, 2);
                var d = x * Math.sqrt(con) / this.k0;
                var ds = Math.pow(d, 2);
                con = con * tan_phi;

                lat = phi - (con * ds / (1 - this.es)) * 0.5 * (1 -
                    ds / 12 * (5 + 3 * t - 9 * c * t + c - 4 * cs -
                        ds / 30 * (61 + 90 * t - 252 * c * t + 45 * ts + 46 * c -
                            ds / 56 * (1385 + 3633 * t + 4095 * ts + 1574 * ts * t))));

                lon = adjust_lon(this.long0 + (d * (1 -
                    ds / 6 * (1 + 2 * t + c -
                        ds / 20 * (5 + 28 * t + 24 * ts + 8 * c * t + 6 * c -
                            ds / 42 * (61 + 662 * t + 1320 * ts + 720 * ts * t)))) / cos_phi));
            }
            else {
                lat = HALF_PI * sign(y);
                lon = 0;
            }
        }

        p.x = lon;
        p.y = lat;

        return p;
    }

    var names$3 = ["Transverse_Mercator", "Transverse Mercator", "tmerc"];
    var tmerc = {
        init: init$2,
        forward: forward$2,
        inverse: inverse$2,
        names: names$3
    };

    var sinh = function(x) {
        var r = Math.exp(x);
        r = (r - 1 / r) / 2;
        return r;
    };

    var hypot = function(x, y) {
        x = Math.abs(x);
        y = Math.abs(y);
        var a = Math.max(x, y);
        var b = Math.min(x, y) / (a ? a : 1);

        return a * Math.sqrt(1 + Math.pow(b, 2));
    };

    var log1py = function(x) {
        var y = 1 + x;
        var z = y - 1;

        return z === 0 ? x : x * Math.log(y) / z;
    };

    var asinhy = function(x) {
        var y = Math.abs(x);
        y = log1py(y * (1 + y / (hypot(1, y) + 1)));

        return x < 0 ? -y : y;
    };

    var gatg = function(pp, B) {
        var cos_2B = 2 * Math.cos(2 * B);
        var i = pp.length - 1;
        var h1 = pp[i];
        var h2 = 0;
        var h;

        while (--i >= 0) {
            h = -h2 + cos_2B * h1 + pp[i];
            h2 = h1;
            h1 = h;
        }

        return (B + h * Math.sin(2 * B));
    };

    var clens = function(pp, arg_r) {
        var r = 2 * Math.cos(arg_r);
        var i = pp.length - 1;
        var hr1 = pp[i];
        var hr2 = 0;
        var hr;

        while (--i >= 0) {
            hr = -hr2 + r * hr1 + pp[i];
            hr2 = hr1;
            hr1 = hr;
        }

        return Math.sin(arg_r) * hr;
    };

    var cosh = function(x) {
        var r = Math.exp(x);
        r = (r + 1 / r) / 2;
        return r;
    };

    var clens_cmplx = function(pp, arg_r, arg_i) {
        var sin_arg_r = Math.sin(arg_r);
        var cos_arg_r = Math.cos(arg_r);
        var sinh_arg_i = sinh(arg_i);
        var cosh_arg_i = cosh(arg_i);
        var r = 2 * cos_arg_r * cosh_arg_i;
        var i = -2 * sin_arg_r * sinh_arg_i;
        var j = pp.length - 1;
        var hr = pp[j];
        var hi1 = 0;
        var hr1 = 0;
        var hi = 0;
        var hr2;
        var hi2;

        while (--j >= 0) {
            hr2 = hr1;
            hi2 = hi1;
            hr1 = hr;
            hi1 = hi;
            hr = -hr2 + r * hr1 - i * hi1 + pp[j];
            hi = -hi2 + i * hr1 + r * hi1;
        }

        r = sin_arg_r * cosh_arg_i;
        i = cos_arg_r * sinh_arg_i;

        return [r * hr - i * hi, r * hi + i * hr];
    };

    // Heavily based on this etmerc projection implementation
    // https://github.com/mbloch/mapshaper-proj/blob/master/src/projections/etmerc.js

    function init$3() {
        if (this.es === undefined || this.es <= 0) {
            throw new Error('incorrect elliptical usage');
        }

        this.x0 = this.x0 !== undefined ? this.x0 : 0;
        this.y0 = this.y0 !== undefined ? this.y0 : 0;
        this.long0 = this.long0 !== undefined ? this.long0 : 0;
        this.lat0 = this.lat0 !== undefined ? this.lat0 : 0;

        this.cgb = [];
        this.cbg = [];
        this.utg = [];
        this.gtu = [];

        var f = this.es / (1 + Math.sqrt(1 - this.es));
        var n = f / (2 - f);
        var np = n;

        this.cgb[0] = n * (2 + n * (-2 / 3 + n * (-2 + n * (116 / 45 + n * (26 / 45 + n * (-2854 / 675 ))))));
        this.cbg[0] = n * (-2 + n * ( 2 / 3 + n * ( 4 / 3 + n * (-82 / 45 + n * (32 / 45 + n * (4642 / 4725))))));

        np = np * n;
        this.cgb[1] = np * (7 / 3 + n * (-8 / 5 + n * (-227 / 45 + n * (2704 / 315 + n * (2323 / 945)))));
        this.cbg[1] = np * (5 / 3 + n * (-16 / 15 + n * ( -13 / 9 + n * (904 / 315 + n * (-1522 / 945)))));

        np = np * n;
        this.cgb[2] = np * (56 / 15 + n * (-136 / 35 + n * (-1262 / 105 + n * (73814 / 2835))));
        this.cbg[2] = np * (-26 / 15 + n * (34 / 21 + n * (8 / 5 + n * (-12686 / 2835))));

        np = np * n;
        this.cgb[3] = np * (4279 / 630 + n * (-332 / 35 + n * (-399572 / 14175)));
        this.cbg[3] = np * (1237 / 630 + n * (-12 / 5 + n * ( -24832 / 14175)));

        np = np * n;
        this.cgb[4] = np * (4174 / 315 + n * (-144838 / 6237));
        this.cbg[4] = np * (-734 / 315 + n * (109598 / 31185));

        np = np * n;
        this.cgb[5] = np * (601676 / 22275);
        this.cbg[5] = np * (444337 / 155925);

        np = Math.pow(n, 2);
        this.Qn = this.k0 / (1 + n) * (1 + np * (1 / 4 + np * (1 / 64 + np / 256)));

        this.utg[0] = n * (-0.5 + n * ( 2 / 3 + n * (-37 / 96 + n * ( 1 / 360 + n * (81 / 512 + n * (-96199 / 604800))))));
        this.gtu[0] = n * (0.5 + n * (-2 / 3 + n * (5 / 16 + n * (41 / 180 + n * (-127 / 288 + n * (7891 / 37800))))));

        this.utg[1] = np * (-1 / 48 + n * (-1 / 15 + n * (437 / 1440 + n * (-46 / 105 + n * (1118711 / 3870720)))));
        this.gtu[1] = np * (13 / 48 + n * (-3 / 5 + n * (557 / 1440 + n * (281 / 630 + n * (-1983433 / 1935360)))));

        np = np * n;
        this.utg[2] = np * (-17 / 480 + n * (37 / 840 + n * (209 / 4480 + n * (-5569 / 90720 ))));
        this.gtu[2] = np * (61 / 240 + n * (-103 / 140 + n * (15061 / 26880 + n * (167603 / 181440))));

        np = np * n;
        this.utg[3] = np * (-4397 / 161280 + n * (11 / 504 + n * (830251 / 7257600)));
        this.gtu[3] = np * (49561 / 161280 + n * (-179 / 168 + n * (6601661 / 7257600)));

        np = np * n;
        this.utg[4] = np * (-4583 / 161280 + n * (108847 / 3991680));
        this.gtu[4] = np * (34729 / 80640 + n * (-3418889 / 1995840));

        np = np * n;
        this.utg[5] = np * (-20648693 / 638668800);
        this.gtu[5] = np * (212378941 / 319334400);

        var Z = gatg(this.cbg, this.lat0);
        this.Zb = -this.Qn * (Z + clens(this.gtu, 2 * Z));
    }

    function forward$3(p) {
        var Ce = adjust_lon(p.x - this.long0);
        var Cn = p.y;

        Cn = gatg(this.cbg, Cn);
        var sin_Cn = Math.sin(Cn);
        var cos_Cn = Math.cos(Cn);
        var sin_Ce = Math.sin(Ce);
        var cos_Ce = Math.cos(Ce);

        Cn = Math.atan2(sin_Cn, cos_Ce * cos_Cn);
        Ce = Math.atan2(sin_Ce * cos_Cn, hypot(sin_Cn, cos_Cn * cos_Ce));
        Ce = asinhy(Math.tan(Ce));

        var tmp = clens_cmplx(this.gtu, 2 * Cn, 2 * Ce);

        Cn = Cn + tmp[0];
        Ce = Ce + tmp[1];

        var x;
        var y;

        if (Math.abs(Ce) <= 2.623395162778) {
            x = this.a * (this.Qn * Ce) + this.x0;
            y = this.a * (this.Qn * Cn + this.Zb) + this.y0;
        }
        else {
            x = Infinity;
            y = Infinity;
        }

        p.x = x;
        p.y = y;

        return p;
    }

    function inverse$3(p) {
        var Ce = (p.x - this.x0) * (1 / this.a);
        var Cn = (p.y - this.y0) * (1 / this.a);

        Cn = (Cn - this.Zb) / this.Qn;
        Ce = Ce / this.Qn;

        var lon;
        var lat;

        if (Math.abs(Ce) <= 2.623395162778) {
            var tmp = clens_cmplx(this.utg, 2 * Cn, 2 * Ce);

            Cn = Cn + tmp[0];
            Ce = Ce + tmp[1];
            Ce = Math.atan(sinh(Ce));

            var sin_Cn = Math.sin(Cn);
            var cos_Cn = Math.cos(Cn);
            var sin_Ce = Math.sin(Ce);
            var cos_Ce = Math.cos(Ce);

            Cn = Math.atan2(sin_Cn * cos_Ce, hypot(sin_Ce, cos_Ce * cos_Cn));
            Ce = Math.atan2(sin_Ce, cos_Ce * cos_Cn);

            lon = adjust_lon(Ce + this.long0);
            lat = gatg(this.cgb, Cn);
        }
        else {
            lon = Infinity;
            lat = Infinity;
        }

        p.x = lon;
        p.y = lat;

        return p;
    }

    var names$4 = ["Extended_Transverse_Mercator", "Extended Transverse Mercator", "etmerc"];
    var etmerc = {
        init: init$3,
        forward: forward$3,
        inverse: inverse$3,
        names: names$4
    };

    var adjust_zone = function(zone, lon) {
        if (zone === undefined) {
            zone = Math.floor((adjust_lon(lon) + Math.PI) * 30 / Math.PI) + 1;

            if (zone < 0) {
                return 0;
            } else if (zone > 60) {
                return 60;
            }
        }
        return zone;
    };

    var dependsOn = 'etmerc';
    function init$4() {
        var zone = adjust_zone(this.zone, this.long0);
        if (zone === undefined) {
            throw new Error('unknown utm zone');
        }
        this.lat0 = 0;
        this.long0 =  ((6 * Math.abs(zone)) - 183) * D2R;
        this.x0 = 500000;
        this.y0 = this.utmSouth ? 10000000 : 0;
        this.k0 = 0.9996;

        etmerc.init.apply(this);
        this.forward = etmerc.forward;
        this.inverse = etmerc.inverse;
    }

    var names$5 = ["Universal Transverse Mercator System", "utm"];
    var utm = {
        init: init$4,
        names: names$5,
        dependsOn: dependsOn
    };

    var srat = function(esinp, exp) {
        return (Math.pow((1 - esinp) / (1 + esinp), exp));
    };

    var MAX_ITER$1 = 20;
    function init$6() {
        var sphi = Math.sin(this.lat0);
        var cphi = Math.cos(this.lat0);
        cphi *= cphi;
        this.rc = Math.sqrt(1 - this.es) / (1 - this.es * sphi * sphi);
        this.C = Math.sqrt(1 + this.es * cphi * cphi / (1 - this.es));
        this.phic0 = Math.asin(sphi / this.C);
        this.ratexp = 0.5 * this.C * this.e;
        this.K = Math.tan(0.5 * this.phic0 + FORTPI) / (Math.pow(Math.tan(0.5 * this.lat0 + FORTPI), this.C) * srat(this.e * sphi, this.ratexp));
    }

    function forward$5(p) {
        var lon = p.x;
        var lat = p.y;

        p.y = 2 * Math.atan(this.K * Math.pow(Math.tan(0.5 * lat + FORTPI), this.C) * srat(this.e * Math.sin(lat), this.ratexp)) - HALF_PI;
        p.x = this.C * lon;
        return p;
    }

    function inverse$5(p) {
        var DEL_TOL = 1e-14;
        var lon = p.x / this.C;
        var lat = p.y;
        var num = Math.pow(Math.tan(0.5 * lat + FORTPI) / this.K, 1 / this.C);
        for (var i = MAX_ITER$1; i > 0; --i) {
            lat = 2 * Math.atan(num * srat(this.e * Math.sin(p.y), - 0.5 * this.e)) - HALF_PI;
            if (Math.abs(lat - p.y) < DEL_TOL) {
                break;
            }
            p.y = lat;
        }
        /* convergence failed */
        if (!i) {
            return null;
        }
        p.x = lon;
        p.y = lat;
        return p;
    }

    var names$7 = ["gauss"];
    var gauss = {
        init: init$6,
        forward: forward$5,
        inverse: inverse$5,
        names: names$7
    };

    function init$5() {
        gauss.init.apply(this);
        if (!this.rc) {
            return;
        }
        this.sinc0 = Math.sin(this.phic0);
        this.cosc0 = Math.cos(this.phic0);
        this.R2 = 2 * this.rc;
        if (!this.title) {
            this.title = "Oblique Stereographic Alternative";
        }
    }

    function forward$4(p) {
        var sinc, cosc, cosl, k;
        p.x = adjust_lon(p.x - this.long0);
        gauss.forward.apply(this, [p]);
        sinc = Math.sin(p.y);
        cosc = Math.cos(p.y);
        cosl = Math.cos(p.x);
        k = this.k0 * this.R2 / (1 + this.sinc0 * sinc + this.cosc0 * cosc * cosl);
        p.x = k * cosc * Math.sin(p.x);
        p.y = k * (this.cosc0 * sinc - this.sinc0 * cosc * cosl);
        p.x = this.a * p.x + this.x0;
        p.y = this.a * p.y + this.y0;
        return p;
    }

    function inverse$4(p) {
        var sinc, cosc, lon, lat, rho;
        p.x = (p.x - this.x0) / this.a;
        p.y = (p.y - this.y0) / this.a;

        p.x /= this.k0;
        p.y /= this.k0;
        if ((rho = Math.sqrt(p.x * p.x + p.y * p.y))) {
            var c = 2 * Math.atan2(rho, this.R2);
            sinc = Math.sin(c);
            cosc = Math.cos(c);
            lat = Math.asin(cosc * this.sinc0 + p.y * sinc * this.cosc0 / rho);
            lon = Math.atan2(p.x * sinc, rho * this.cosc0 * cosc - p.y * this.sinc0 * sinc);
        }
        else {
            lat = this.phic0;
            lon = 0;
        }

        p.x = lon;
        p.y = lat;
        gauss.inverse.apply(this, [p]);
        p.x = adjust_lon(p.x + this.long0);
        return p;
    }

    var names$6 = ["Stereographic_North_Pole", "Oblique_Stereographic", "Polar_Stereographic", "sterea","Oblique Stereographic Alternative","Double_Stereographic"];
    var sterea = {
        init: init$5,
        forward: forward$4,
        inverse: inverse$4,
        names: names$6
    };

    function ssfn_(phit, sinphi, eccen) {
        sinphi *= eccen;
        return (Math.tan(0.5 * (HALF_PI + phit)) * Math.pow((1 - sinphi) / (1 + sinphi), 0.5 * eccen));
    }

    function init$7() {
        this.coslat0 = Math.cos(this.lat0);
        this.sinlat0 = Math.sin(this.lat0);
        if (this.sphere) {
            if (this.k0 === 1 && !isNaN(this.lat_ts) && Math.abs(this.coslat0) <= EPSLN) {
                this.k0 = 0.5 * (1 + sign(this.lat0) * Math.sin(this.lat_ts));
            }
        }
        else {
            if (Math.abs(this.coslat0) <= EPSLN) {
                if (this.lat0 > 0) {
                    //North pole
                    //trace('stere:north pole');
                    this.con = 1;
                }
                else {
                    //South pole
                    //trace('stere:south pole');
                    this.con = -1;
                }
            }
            this.cons = Math.sqrt(Math.pow(1 + this.e, 1 + this.e) * Math.pow(1 - this.e, 1 - this.e));
            if (this.k0 === 1 && !isNaN(this.lat_ts) && Math.abs(this.coslat0) <= EPSLN) {
                this.k0 = 0.5 * this.cons * msfnz(this.e, Math.sin(this.lat_ts), Math.cos(this.lat_ts)) / tsfnz(this.e, this.con * this.lat_ts, this.con * Math.sin(this.lat_ts));
            }
            this.ms1 = msfnz(this.e, this.sinlat0, this.coslat0);
            this.X0 = 2 * Math.atan(this.ssfn_(this.lat0, this.sinlat0, this.e)) - HALF_PI;
            this.cosX0 = Math.cos(this.X0);
            this.sinX0 = Math.sin(this.X0);
        }
    }

    // Stereographic forward equations--mapping lat,long to x,y
    function forward$6(p) {
        var lon = p.x;
        var lat = p.y;
        var sinlat = Math.sin(lat);
        var coslat = Math.cos(lat);
        var A, X, sinX, cosX, ts, rh;
        var dlon = adjust_lon(lon - this.long0);

        if (Math.abs(Math.abs(lon - this.long0) - Math.PI) <= EPSLN && Math.abs(lat + this.lat0) <= EPSLN) {
            //case of the origine point
            //trace('stere:this is the origin point');
            p.x = NaN;
            p.y = NaN;
            return p;
        }
        if (this.sphere) {
            //trace('stere:sphere case');
            A = 2 * this.k0 / (1 + this.sinlat0 * sinlat + this.coslat0 * coslat * Math.cos(dlon));
            p.x = this.a * A * coslat * Math.sin(dlon) + this.x0;
            p.y = this.a * A * (this.coslat0 * sinlat - this.sinlat0 * coslat * Math.cos(dlon)) + this.y0;
            return p;
        }
        else {
            X = 2 * Math.atan(this.ssfn_(lat, sinlat, this.e)) - HALF_PI;
            cosX = Math.cos(X);
            sinX = Math.sin(X);
            if (Math.abs(this.coslat0) <= EPSLN) {
                ts = tsfnz(this.e, lat * this.con, this.con * sinlat);
                rh = 2 * this.a * this.k0 * ts / this.cons;
                p.x = this.x0 + rh * Math.sin(lon - this.long0);
                p.y = this.y0 - this.con * rh * Math.cos(lon - this.long0);
                //trace(p.toString());
                return p;
            }
            else if (Math.abs(this.sinlat0) < EPSLN) {
                //Eq
                //trace('stere:equateur');
                A = 2 * this.a * this.k0 / (1 + cosX * Math.cos(dlon));
                p.y = A * sinX;
            }
            else {
                //other case
                //trace('stere:normal case');
                A = 2 * this.a * this.k0 * this.ms1 / (this.cosX0 * (1 + this.sinX0 * sinX + this.cosX0 * cosX * Math.cos(dlon)));
                p.y = A * (this.cosX0 * sinX - this.sinX0 * cosX * Math.cos(dlon)) + this.y0;
            }
            p.x = A * cosX * Math.sin(dlon) + this.x0;
        }
        //trace(p.toString());
        return p;
    }

    //* Stereographic inverse equations--mapping x,y to lat/long
    function inverse$6(p) {
        p.x -= this.x0;
        p.y -= this.y0;
        var lon, lat, ts, ce, Chi;
        var rh = Math.sqrt(p.x * p.x + p.y * p.y);
        if (this.sphere) {
            var c = 2 * Math.atan(rh / (2 * this.a * this.k0));
            lon = this.long0;
            lat = this.lat0;
            if (rh <= EPSLN) {
                p.x = lon;
                p.y = lat;
                return p;
            }
            lat = Math.asin(Math.cos(c) * this.sinlat0 + p.y * Math.sin(c) * this.coslat0 / rh);
            if (Math.abs(this.coslat0) < EPSLN) {
                if (this.lat0 > 0) {
                    lon = adjust_lon(this.long0 + Math.atan2(p.x, - 1 * p.y));
                }
                else {
                    lon = adjust_lon(this.long0 + Math.atan2(p.x, p.y));
                }
            }
            else {
                lon = adjust_lon(this.long0 + Math.atan2(p.x * Math.sin(c), rh * this.coslat0 * Math.cos(c) - p.y * this.sinlat0 * Math.sin(c)));
            }
            p.x = lon;
            p.y = lat;
            return p;
        }
        else {
            if (Math.abs(this.coslat0) <= EPSLN) {
                if (rh <= EPSLN) {
                    lat = this.lat0;
                    lon = this.long0;
                    p.x = lon;
                    p.y = lat;
                    //trace(p.toString());
                    return p;
                }
                p.x *= this.con;
                p.y *= this.con;
                ts = rh * this.cons / (2 * this.a * this.k0);
                lat = this.con * phi2z(this.e, ts);
                lon = this.con * adjust_lon(this.con * this.long0 + Math.atan2(p.x, - 1 * p.y));
            }
            else {
                ce = 2 * Math.atan(rh * this.cosX0 / (2 * this.a * this.k0 * this.ms1));
                lon = this.long0;
                if (rh <= EPSLN) {
                    Chi = this.X0;
                }
                else {
                    Chi = Math.asin(Math.cos(ce) * this.sinX0 + p.y * Math.sin(ce) * this.cosX0 / rh);
                    lon = adjust_lon(this.long0 + Math.atan2(p.x * Math.sin(ce), rh * this.cosX0 * Math.cos(ce) - p.y * this.sinX0 * Math.sin(ce)));
                }
                lat = -1 * phi2z(this.e, Math.tan(0.5 * (HALF_PI + Chi)));
            }
        }
        p.x = lon;
        p.y = lat;

        //trace(p.toString());
        return p;

    }

    var names$8 = ["stere", "Stereographic_South_Pole", "Polar Stereographic (variant B)"];
    var stere = {
        init: init$7,
        forward: forward$6,
        inverse: inverse$6,
        names: names$8,
        ssfn_: ssfn_
    };

    /*
      references:
        Formules et constantes pour le Calcul pour la
        projection cylindrique conforme à axe oblique et pour la transformation entre
        des systèmes de référence.
        http://www.swisstopo.admin.ch/internet/swisstopo/fr/home/topics/survey/sys/refsys/switzerland.parsysrelated1.31216.downloadList.77004.DownloadFile.tmp/swissprojectionfr.pdf
      */

    function init$8() {
        var phy0 = this.lat0;
        this.lambda0 = this.long0;
        var sinPhy0 = Math.sin(phy0);
        var semiMajorAxis = this.a;
        var invF = this.rf;
        var flattening = 1 / invF;
        var e2 = 2 * flattening - Math.pow(flattening, 2);
        var e = this.e = Math.sqrt(e2);
        this.R = this.k0 * semiMajorAxis * Math.sqrt(1 - e2) / (1 - e2 * Math.pow(sinPhy0, 2));
        this.alpha = Math.sqrt(1 + e2 / (1 - e2) * Math.pow(Math.cos(phy0), 4));
        this.b0 = Math.asin(sinPhy0 / this.alpha);
        var k1 = Math.log(Math.tan(Math.PI / 4 + this.b0 / 2));
        var k2 = Math.log(Math.tan(Math.PI / 4 + phy0 / 2));
        var k3 = Math.log((1 + e * sinPhy0) / (1 - e * sinPhy0));
        this.K = k1 - this.alpha * k2 + this.alpha * e / 2 * k3;
    }

    function forward$7(p) {
        var Sa1 = Math.log(Math.tan(Math.PI / 4 - p.y / 2));
        var Sa2 = this.e / 2 * Math.log((1 + this.e * Math.sin(p.y)) / (1 - this.e * Math.sin(p.y)));
        var S = -this.alpha * (Sa1 + Sa2) + this.K;

        // spheric latitude
        var b = 2 * (Math.atan(Math.exp(S)) - Math.PI / 4);

        // spheric longitude
        var I = this.alpha * (p.x - this.lambda0);

        // psoeudo equatorial rotation
        var rotI = Math.atan(Math.sin(I) / (Math.sin(this.b0) * Math.tan(b) + Math.cos(this.b0) * Math.cos(I)));

        var rotB = Math.asin(Math.cos(this.b0) * Math.sin(b) - Math.sin(this.b0) * Math.cos(b) * Math.cos(I));

        p.y = this.R / 2 * Math.log((1 + Math.sin(rotB)) / (1 - Math.sin(rotB))) + this.y0;
        p.x = this.R * rotI + this.x0;
        return p;
    }

    function inverse$7(p) {
        var Y = p.x - this.x0;
        var X = p.y - this.y0;

        var rotI = Y / this.R;
        var rotB = 2 * (Math.atan(Math.exp(X / this.R)) - Math.PI / 4);

        var b = Math.asin(Math.cos(this.b0) * Math.sin(rotB) + Math.sin(this.b0) * Math.cos(rotB) * Math.cos(rotI));
        var I = Math.atan(Math.sin(rotI) / (Math.cos(this.b0) * Math.cos(rotI) - Math.sin(this.b0) * Math.tan(rotB)));

        var lambda = this.lambda0 + I / this.alpha;

        var S = 0;
        var phy = b;
        var prevPhy = -1000;
        var iteration = 0;
        while (Math.abs(phy - prevPhy) > 0.0000001) {
            if (++iteration > 20) {
                //...reportError("omercFwdInfinity");
                return;
            }
            //S = Math.log(Math.tan(Math.PI / 4 + phy / 2));
            S = 1 / this.alpha * (Math.log(Math.tan(Math.PI / 4 + b / 2)) - this.K) + this.e * Math.log(Math.tan(Math.PI / 4 + Math.asin(this.e * Math.sin(phy)) / 2));
            prevPhy = phy;
            phy = 2 * Math.atan(Math.exp(S)) - Math.PI / 2;
        }

        p.x = lambda;
        p.y = phy;
        return p;
    }

    var names$9 = ["somerc"];
    var somerc = {
        init: init$8,
        forward: forward$7,
        inverse: inverse$7,
        names: names$9
    };

    /* Initialize the Oblique Mercator  projection
        ------------------------------------------*/
    function init$9() {
        this.no_off = this.no_off || false;
        this.no_rot = this.no_rot || false;

        if (isNaN(this.k0)) {
            this.k0 = 1;
        }
        var sinlat = Math.sin(this.lat0);
        var coslat = Math.cos(this.lat0);
        var con = this.e * sinlat;

        this.bl = Math.sqrt(1 + this.es / (1 - this.es) * Math.pow(coslat, 4));
        this.al = this.a * this.bl * this.k0 * Math.sqrt(1 - this.es) / (1 - con * con);
        var t0 = tsfnz(this.e, this.lat0, sinlat);
        var dl = this.bl / coslat * Math.sqrt((1 - this.es) / (1 - con * con));
        if (dl * dl < 1) {
            dl = 1;
        }
        var fl;
        var gl;
        if (!isNaN(this.longc)) {
            //Central point and azimuth method

            if (this.lat0 >= 0) {
                fl = dl + Math.sqrt(dl * dl - 1);
            }
            else {
                fl = dl - Math.sqrt(dl * dl - 1);
            }
            this.el = fl * Math.pow(t0, this.bl);
            gl = 0.5 * (fl - 1 / fl);
            this.gamma0 = Math.asin(Math.sin(this.alpha) / dl);
            this.long0 = this.longc - Math.asin(gl * Math.tan(this.gamma0)) / this.bl;

        }
        else {
            //2 points method
            var t1 = tsfnz(this.e, this.lat1, Math.sin(this.lat1));
            var t2 = tsfnz(this.e, this.lat2, Math.sin(this.lat2));
            if (this.lat0 >= 0) {
                this.el = (dl + Math.sqrt(dl * dl - 1)) * Math.pow(t0, this.bl);
            }
            else {
                this.el = (dl - Math.sqrt(dl * dl - 1)) * Math.pow(t0, this.bl);
            }
            var hl = Math.pow(t1, this.bl);
            var ll = Math.pow(t2, this.bl);
            fl = this.el / hl;
            gl = 0.5 * (fl - 1 / fl);
            var jl = (this.el * this.el - ll * hl) / (this.el * this.el + ll * hl);
            var pl = (ll - hl) / (ll + hl);
            var dlon12 = adjust_lon(this.long1 - this.long2);
            this.long0 = 0.5 * (this.long1 + this.long2) - Math.atan(jl * Math.tan(0.5 * this.bl * (dlon12)) / pl) / this.bl;
            this.long0 = adjust_lon(this.long0);
            var dlon10 = adjust_lon(this.long1 - this.long0);
            this.gamma0 = Math.atan(Math.sin(this.bl * (dlon10)) / gl);
            this.alpha = Math.asin(dl * Math.sin(this.gamma0));
        }

        if (this.no_off) {
            this.uc = 0;
        }
        else {
            if (this.lat0 >= 0) {
                this.uc = this.al / this.bl * Math.atan2(Math.sqrt(dl * dl - 1), Math.cos(this.alpha));
            }
            else {
                this.uc = -1 * this.al / this.bl * Math.atan2(Math.sqrt(dl * dl - 1), Math.cos(this.alpha));
            }
        }

    }

    /* Oblique Mercator forward equations--mapping lat,long to x,y
        ----------------------------------------------------------*/
    function forward$8(p) {
        var lon = p.x;
        var lat = p.y;
        var dlon = adjust_lon(lon - this.long0);
        var us, vs;
        var con;
        if (Math.abs(Math.abs(lat) - HALF_PI) <= EPSLN) {
            if (lat > 0) {
                con = -1;
            }
            else {
                con = 1;
            }
            vs = this.al / this.bl * Math.log(Math.tan(FORTPI + con * this.gamma0 * 0.5));
            us = -1 * con * HALF_PI * this.al / this.bl;
        }
        else {
            var t = tsfnz(this.e, lat, Math.sin(lat));
            var ql = this.el / Math.pow(t, this.bl);
            var sl = 0.5 * (ql - 1 / ql);
            var tl = 0.5 * (ql + 1 / ql);
            var vl = Math.sin(this.bl * (dlon));
            var ul = (sl * Math.sin(this.gamma0) - vl * Math.cos(this.gamma0)) / tl;
            if (Math.abs(Math.abs(ul) - 1) <= EPSLN) {
                vs = Number.POSITIVE_INFINITY;
            }
            else {
                vs = 0.5 * this.al * Math.log((1 - ul) / (1 + ul)) / this.bl;
            }
            if (Math.abs(Math.cos(this.bl * (dlon))) <= EPSLN) {
                us = this.al * this.bl * (dlon);
            }
            else {
                us = this.al * Math.atan2(sl * Math.cos(this.gamma0) + vl * Math.sin(this.gamma0), Math.cos(this.bl * dlon)) / this.bl;
            }
        }

        if (this.no_rot) {
            p.x = this.x0 + us;
            p.y = this.y0 + vs;
        }
        else {

            us -= this.uc;
            p.x = this.x0 + vs * Math.cos(this.alpha) + us * Math.sin(this.alpha);
            p.y = this.y0 + us * Math.cos(this.alpha) - vs * Math.sin(this.alpha);
        }
        return p;
    }

    function inverse$8(p) {
        var us, vs;
        if (this.no_rot) {
            vs = p.y - this.y0;
            us = p.x - this.x0;
        }
        else {
            vs = (p.x - this.x0) * Math.cos(this.alpha) - (p.y - this.y0) * Math.sin(this.alpha);
            us = (p.y - this.y0) * Math.cos(this.alpha) + (p.x - this.x0) * Math.sin(this.alpha);
            us += this.uc;
        }
        var qp = Math.exp(-1 * this.bl * vs / this.al);
        var sp = 0.5 * (qp - 1 / qp);
        var tp = 0.5 * (qp + 1 / qp);
        var vp = Math.sin(this.bl * us / this.al);
        var up = (vp * Math.cos(this.gamma0) + sp * Math.sin(this.gamma0)) / tp;
        var ts = Math.pow(this.el / Math.sqrt((1 + up) / (1 - up)), 1 / this.bl);
        if (Math.abs(up - 1) < EPSLN) {
            p.x = this.long0;
            p.y = HALF_PI;
        }
        else if (Math.abs(up + 1) < EPSLN) {
            p.x = this.long0;
            p.y = -1 * HALF_PI;
        }
        else {
            p.y = phi2z(this.e, ts);
            p.x = adjust_lon(this.long0 - Math.atan2(sp * Math.cos(this.gamma0) - vp * Math.sin(this.gamma0), Math.cos(this.bl * us / this.al)) / this.bl);
        }
        return p;
    }

    var names$10 = ["Hotine_Oblique_Mercator", "Hotine Oblique Mercator", "Hotine_Oblique_Mercator_Azimuth_Natural_Origin", "Hotine_Oblique_Mercator_Azimuth_Center", "omerc"];
    var omerc = {
        init: init$9,
        forward: forward$8,
        inverse: inverse$8,
        names: names$10
    };

    function init$10() {

        // array of:  r_maj,r_min,lat1,lat2,c_lon,c_lat,false_east,false_north
        //double c_lat;                   /* center latitude                      */
        //double c_lon;                   /* center longitude                     */
        //double lat1;                    /* first standard parallel              */
        //double lat2;                    /* second standard parallel             */
        //double r_maj;                   /* major axis                           */
        //double r_min;                   /* minor axis                           */
        //double false_east;              /* x offset in meters                   */
        //double false_north;             /* y offset in meters                   */

        if (!this.lat2) {
            this.lat2 = this.lat1;
        } //if lat2 is not defined
        if (!this.k0) {
            this.k0 = 1;
        }
        this.x0 = this.x0 || 0;
        this.y0 = this.y0 || 0;
        // Standard Parallels cannot be equal and on opposite sides of the equator
        if (Math.abs(this.lat1 + this.lat2) < EPSLN) {
            return;
        }

        var temp = this.b / this.a;
        this.e = Math.sqrt(1 - temp * temp);

        var sin1 = Math.sin(this.lat1);
        var cos1 = Math.cos(this.lat1);
        var ms1 = msfnz(this.e, sin1, cos1);
        var ts1 = tsfnz(this.e, this.lat1, sin1);

        var sin2 = Math.sin(this.lat2);
        var cos2 = Math.cos(this.lat2);
        var ms2 = msfnz(this.e, sin2, cos2);
        var ts2 = tsfnz(this.e, this.lat2, sin2);

        var ts0 = tsfnz(this.e, this.lat0, Math.sin(this.lat0));

        if (Math.abs(this.lat1 - this.lat2) > EPSLN) {
            this.ns = Math.log(ms1 / ms2) / Math.log(ts1 / ts2);
        }
        else {
            this.ns = sin1;
        }
        if (isNaN(this.ns)) {
            this.ns = sin1;
        }
        this.f0 = ms1 / (this.ns * Math.pow(ts1, this.ns));
        this.rh = this.a * this.f0 * Math.pow(ts0, this.ns);
        if (!this.title) {
            this.title = "Lambert Conformal Conic";
        }
    }

    // Lambert Conformal conic forward equations--mapping lat,long to x,y
    // -----------------------------------------------------------------
    function forward$9(p) {

        var lon = p.x;
        var lat = p.y;

        // singular cases :
        if (Math.abs(2 * Math.abs(lat) - Math.PI) <= EPSLN) {
            lat = sign(lat) * (HALF_PI - 2 * EPSLN);
        }

        var con = Math.abs(Math.abs(lat) - HALF_PI);
        var ts, rh1;
        if (con > EPSLN) {
            ts = tsfnz(this.e, lat, Math.sin(lat));
            rh1 = this.a * this.f0 * Math.pow(ts, this.ns);
        }
        else {
            con = lat * this.ns;
            if (con <= 0) {
                return null;
            }
            rh1 = 0;
        }
        var theta = this.ns * adjust_lon(lon - this.long0);
        p.x = this.k0 * (rh1 * Math.sin(theta)) + this.x0;
        p.y = this.k0 * (this.rh - rh1 * Math.cos(theta)) + this.y0;

        return p;
    }

    // Lambert Conformal Conic inverse equations--mapping x,y to lat/long
    // -----------------------------------------------------------------
    function inverse$9(p) {

        var rh1, con, ts;
        var lat, lon;
        var x = (p.x - this.x0) / this.k0;
        var y = (this.rh - (p.y - this.y0) / this.k0);
        if (this.ns > 0) {
            rh1 = Math.sqrt(x * x + y * y);
            con = 1;
        }
        else {
            rh1 = -Math.sqrt(x * x + y * y);
            con = -1;
        }
        var theta = 0;
        if (rh1 !== 0) {
            theta = Math.atan2((con * x), (con * y));
        }
        if ((rh1 !== 0) || (this.ns > 0)) {
            con = 1 / this.ns;
            ts = Math.pow((rh1 / (this.a * this.f0)), con);
            lat = phi2z(this.e, ts);
            if (lat === -9999) {
                return null;
            }
        }
        else {
            lat = -HALF_PI;
        }
        lon = adjust_lon(theta / this.ns + this.long0);

        p.x = lon;
        p.y = lat;
        return p;
    }

    var names$11 = ["Lambert Tangential Conformal Conic Projection", "Lambert_Conformal_Conic", "Lambert_Conformal_Conic_2SP", "lcc"];
    var lcc = {
        init: init$10,
        forward: forward$9,
        inverse: inverse$9,
        names: names$11
    };

    function init$11() {
        this.a = 6377397.155;
        this.es = 0.006674372230614;
        this.e = Math.sqrt(this.es);
        if (!this.lat0) {
            this.lat0 = 0.863937979737193;
        }
        if (!this.long0) {
            this.long0 = 0.7417649320975901 - 0.308341501185665;
        }
        /* if scale not set default to 0.9999 */
        if (!this.k0) {
            this.k0 = 0.9999;
        }
        this.s45 = 0.785398163397448; /* 45 */
        this.s90 = 2 * this.s45;
        this.fi0 = this.lat0;
        this.e2 = this.es;
        this.e = Math.sqrt(this.e2);
        this.alfa = Math.sqrt(1 + (this.e2 * Math.pow(Math.cos(this.fi0), 4)) / (1 - this.e2));
        this.uq = 1.04216856380474;
        this.u0 = Math.asin(Math.sin(this.fi0) / this.alfa);
        this.g = Math.pow((1 + this.e * Math.sin(this.fi0)) / (1 - this.e * Math.sin(this.fi0)), this.alfa * this.e / 2);
        this.k = Math.tan(this.u0 / 2 + this.s45) / Math.pow(Math.tan(this.fi0 / 2 + this.s45), this.alfa) * this.g;
        this.k1 = this.k0;
        this.n0 = this.a * Math.sqrt(1 - this.e2) / (1 - this.e2 * Math.pow(Math.sin(this.fi0), 2));
        this.s0 = 1.37008346281555;
        this.n = Math.sin(this.s0);
        this.ro0 = this.k1 * this.n0 / Math.tan(this.s0);
        this.ad = this.s90 - this.uq;
    }

    /* ellipsoid */
    /* calculate xy from lat/lon */
    /* Constants, identical to inverse transform function */
    function forward$10(p) {
        var gfi, u, deltav, s, d, eps, ro;
        var lon = p.x;
        var lat = p.y;
        var delta_lon = adjust_lon(lon - this.long0);
        /* Transformation */
        gfi = Math.pow(((1 + this.e * Math.sin(lat)) / (1 - this.e * Math.sin(lat))), (this.alfa * this.e / 2));
        u = 2 * (Math.atan(this.k * Math.pow(Math.tan(lat / 2 + this.s45), this.alfa) / gfi) - this.s45);
        deltav = -delta_lon * this.alfa;
        s = Math.asin(Math.cos(this.ad) * Math.sin(u) + Math.sin(this.ad) * Math.cos(u) * Math.cos(deltav));
        d = Math.asin(Math.cos(u) * Math.sin(deltav) / Math.cos(s));
        eps = this.n * d;
        ro = this.ro0 * Math.pow(Math.tan(this.s0 / 2 + this.s45), this.n) / Math.pow(Math.tan(s / 2 + this.s45), this.n);
        p.y = ro * Math.cos(eps) / 1;
        p.x = ro * Math.sin(eps) / 1;

        if (!this.czech) {
            p.y *= -1;
            p.x *= -1;
        }
        return (p);
    }

    /* calculate lat/lon from xy */
    function inverse$10(p) {
        var u, deltav, s, d, eps, ro, fi1;
        var ok;

        /* Transformation */
        /* revert y, x*/
        var tmp = p.x;
        p.x = p.y;
        p.y = tmp;
        if (!this.czech) {
            p.y *= -1;
            p.x *= -1;
        }
        ro = Math.sqrt(p.x * p.x + p.y * p.y);
        eps = Math.atan2(p.y, p.x);
        d = eps / Math.sin(this.s0);
        s = 2 * (Math.atan(Math.pow(this.ro0 / ro, 1 / this.n) * Math.tan(this.s0 / 2 + this.s45)) - this.s45);
        u = Math.asin(Math.cos(this.ad) * Math.sin(s) - Math.sin(this.ad) * Math.cos(s) * Math.cos(d));
        deltav = Math.asin(Math.cos(s) * Math.sin(d) / Math.cos(u));
        p.x = this.long0 - deltav / this.alfa;
        fi1 = u;
        ok = 0;
        var iter = 0;
        do {
            p.y = 2 * (Math.atan(Math.pow(this.k, - 1 / this.alfa) * Math.pow(Math.tan(u / 2 + this.s45), 1 / this.alfa) * Math.pow((1 + this.e * Math.sin(fi1)) / (1 - this.e * Math.sin(fi1)), this.e / 2)) - this.s45);
            if (Math.abs(fi1 - p.y) < 0.0000000001) {
                ok = 1;
            }
            fi1 = p.y;
            iter += 1;
        } while (ok === 0 && iter < 15);
        if (iter >= 15) {
            return null;
        }

        return (p);
    }

    var names$12 = ["Krovak", "krovak"];
    var krovak = {
        init: init$11,
        forward: forward$10,
        inverse: inverse$10,
        names: names$12
    };

    var mlfn = function(e0, e1, e2, e3, phi) {
        return (e0 * phi - e1 * Math.sin(2 * phi) + e2 * Math.sin(4 * phi) - e3 * Math.sin(6 * phi));
    };

    var e0fn = function(x) {
        return (1 - 0.25 * x * (1 + x / 16 * (3 + 1.25 * x)));
    };

    var e1fn = function(x) {
        return (0.375 * x * (1 + 0.25 * x * (1 + 0.46875 * x)));
    };

    var e2fn = function(x) {
        return (0.05859375 * x * x * (1 + 0.75 * x));
    };

    var e3fn = function(x) {
        return (x * x * x * (35 / 3072));
    };

    var gN = function(a, e, sinphi) {
        var temp = e * sinphi;
        return a / Math.sqrt(1 - temp * temp);
    };

    var adjust_lat = function(x) {
        return (Math.abs(x) < HALF_PI) ? x : (x - (sign(x) * Math.PI));
    };

    var imlfn = function(ml, e0, e1, e2, e3) {
        var phi;
        var dphi;

        phi = ml / e0;
        for (var i = 0; i < 15; i++) {
            dphi = (ml - (e0 * phi - e1 * Math.sin(2 * phi) + e2 * Math.sin(4 * phi) - e3 * Math.sin(6 * phi))) / (e0 - 2 * e1 * Math.cos(2 * phi) + 4 * e2 * Math.cos(4 * phi) - 6 * e3 * Math.cos(6 * phi));
            phi += dphi;
            if (Math.abs(dphi) <= 0.0000000001) {
                return phi;
            }
        }

        //..reportError("IMLFN-CONV:Latitude failed to converge after 15 iterations");
        return NaN;
    };

    function init$12() {
        if (!this.sphere) {
            this.e0 = e0fn(this.es);
            this.e1 = e1fn(this.es);
            this.e2 = e2fn(this.es);
            this.e3 = e3fn(this.es);
            this.ml0 = this.a * mlfn(this.e0, this.e1, this.e2, this.e3, this.lat0);
        }
    }

    /* Cassini forward equations--mapping lat,long to x,y
      -----------------------------------------------------------------------*/
    function forward$11(p) {

        /* Forward equations
          -----------------*/
        var x, y;
        var lam = p.x;
        var phi = p.y;
        lam = adjust_lon(lam - this.long0);

        if (this.sphere) {
            x = this.a * Math.asin(Math.cos(phi) * Math.sin(lam));
            y = this.a * (Math.atan2(Math.tan(phi), Math.cos(lam)) - this.lat0);
        }
        else {
            //ellipsoid
            var sinphi = Math.sin(phi);
            var cosphi = Math.cos(phi);
            var nl = gN(this.a, this.e, sinphi);
            var tl = Math.tan(phi) * Math.tan(phi);
            var al = lam * Math.cos(phi);
            var asq = al * al;
            var cl = this.es * cosphi * cosphi / (1 - this.es);
            var ml = this.a * mlfn(this.e0, this.e1, this.e2, this.e3, phi);

            x = nl * al * (1 - asq * tl * (1 / 6 - (8 - tl + 8 * cl) * asq / 120));
            y = ml - this.ml0 + nl * sinphi / cosphi * asq * (0.5 + (5 - tl + 6 * cl) * asq / 24);


        }

        p.x = x + this.x0;
        p.y = y + this.y0;
        return p;
    }

    /* Inverse equations
      -----------------*/
    function inverse$11(p) {
        p.x -= this.x0;
        p.y -= this.y0;
        var x = p.x / this.a;
        var y = p.y / this.a;
        var phi, lam;

        if (this.sphere) {
            var dd = y + this.lat0;
            phi = Math.asin(Math.sin(dd) * Math.cos(x));
            lam = Math.atan2(Math.tan(x), Math.cos(dd));
        }
        else {
            /* ellipsoid */
            var ml1 = this.ml0 / this.a + y;
            var phi1 = imlfn(ml1, this.e0, this.e1, this.e2, this.e3);
            if (Math.abs(Math.abs(phi1) - HALF_PI) <= EPSLN) {
                p.x = this.long0;
                p.y = HALF_PI;
                if (y < 0) {
                    p.y *= -1;
                }
                return p;
            }
            var nl1 = gN(this.a, this.e, Math.sin(phi1));

            var rl1 = nl1 * nl1 * nl1 / this.a / this.a * (1 - this.es);
            var tl1 = Math.pow(Math.tan(phi1), 2);
            var dl = x * this.a / nl1;
            var dsq = dl * dl;
            phi = phi1 - nl1 * Math.tan(phi1) / rl1 * dl * dl * (0.5 - (1 + 3 * tl1) * dl * dl / 24);
            lam = dl * (1 - dsq * (tl1 / 3 + (1 + 3 * tl1) * tl1 * dsq / 15)) / Math.cos(phi1);

        }

        p.x = adjust_lon(lam + this.long0);
        p.y = adjust_lat(phi);
        return p;

    }

    var names$13 = ["Cassini", "Cassini_Soldner", "cass"];
    var cass = {
        init: init$12,
        forward: forward$11,
        inverse: inverse$11,
        names: names$13
    };

    var qsfnz = function(eccent, sinphi) {
        var con;
        if (eccent > 1.0e-7) {
            con = eccent * sinphi;
            return ((1 - eccent * eccent) * (sinphi / (1 - con * con) - (0.5 / eccent) * Math.log((1 - con) / (1 + con))));
        }
        else {
            return (2 * sinphi);
        }
    };

    /*
      reference
        "New Equal-Area Map Projections for Noncircular Regions", John P. Snyder,
        The American Cartographer, Vol 15, No. 4, October 1988, pp. 341-355.
      */

    var S_POLE = 1;

    var N_POLE = 2;
    var EQUIT = 3;
    var OBLIQ = 4;

    /* Initialize the Lambert Azimuthal Equal Area projection
      ------------------------------------------------------*/
    function init$13() {
        var t = Math.abs(this.lat0);
        if (Math.abs(t - HALF_PI) < EPSLN) {
            this.mode = this.lat0 < 0 ? this.S_POLE : this.N_POLE;
        }
        else if (Math.abs(t) < EPSLN) {
            this.mode = this.EQUIT;
        }
        else {
            this.mode = this.OBLIQ;
        }
        if (this.es > 0) {
            var sinphi;

            this.qp = qsfnz(this.e, 1);
            this.mmf = 0.5 / (1 - this.es);
            this.apa = authset(this.es);
            switch (this.mode) {
                case this.N_POLE:
                    this.dd = 1;
                    break;
                case this.S_POLE:
                    this.dd = 1;
                    break;
                case this.EQUIT:
                    this.rq = Math.sqrt(0.5 * this.qp);
                    this.dd = 1 / this.rq;
                    this.xmf = 1;
                    this.ymf = 0.5 * this.qp;
                    break;
                case this.OBLIQ:
                    this.rq = Math.sqrt(0.5 * this.qp);
                    sinphi = Math.sin(this.lat0);
                    this.sinb1 = qsfnz(this.e, sinphi) / this.qp;
                    this.cosb1 = Math.sqrt(1 - this.sinb1 * this.sinb1);
                    this.dd = Math.cos(this.lat0) / (Math.sqrt(1 - this.es * sinphi * sinphi) * this.rq * this.cosb1);
                    this.ymf = (this.xmf = this.rq) / this.dd;
                    this.xmf *= this.dd;
                    break;
            }
        }
        else {
            if (this.mode === this.OBLIQ) {
                this.sinph0 = Math.sin(this.lat0);
                this.cosph0 = Math.cos(this.lat0);
            }
        }
    }

    /* Lambert Azimuthal Equal Area forward equations--mapping lat,long to x,y
      -----------------------------------------------------------------------*/
    function forward$12(p) {

        /* Forward equations
          -----------------*/
        var x, y, coslam, sinlam, sinphi, q, sinb, cosb, b, cosphi;
        var lam = p.x;
        var phi = p.y;

        lam = adjust_lon(lam - this.long0);
        if (this.sphere) {
            sinphi = Math.sin(phi);
            cosphi = Math.cos(phi);
            coslam = Math.cos(lam);
            if (this.mode === this.OBLIQ || this.mode === this.EQUIT) {
                y = (this.mode === this.EQUIT) ? 1 + cosphi * coslam : 1 + this.sinph0 * sinphi + this.cosph0 * cosphi * coslam;
                if (y <= EPSLN) {
                    return null;
                }
                y = Math.sqrt(2 / y);
                x = y * cosphi * Math.sin(lam);
                y *= (this.mode === this.EQUIT) ? sinphi : this.cosph0 * sinphi - this.sinph0 * cosphi * coslam;
            }
            else if (this.mode === this.N_POLE || this.mode === this.S_POLE) {
                if (this.mode === this.N_POLE) {
                    coslam = -coslam;
                }
                if (Math.abs(phi + this.phi0) < EPSLN) {
                    return null;
                }
                y = FORTPI - phi * 0.5;
                y = 2 * ((this.mode === this.S_POLE) ? Math.cos(y) : Math.sin(y));
                x = y * Math.sin(lam);
                y *= coslam;
            }
        }
        else {
            sinb = 0;
            cosb = 0;
            b = 0;
            coslam = Math.cos(lam);
            sinlam = Math.sin(lam);
            sinphi = Math.sin(phi);
            q = qsfnz(this.e, sinphi);
            if (this.mode === this.OBLIQ || this.mode === this.EQUIT) {
                sinb = q / this.qp;
                cosb = Math.sqrt(1 - sinb * sinb);
            }
            switch (this.mode) {
                case this.OBLIQ:
                    b = 1 + this.sinb1 * sinb + this.cosb1 * cosb * coslam;
                    break;
                case this.EQUIT:
                    b = 1 + cosb * coslam;
                    break;
                case this.N_POLE:
                    b = HALF_PI + phi;
                    q = this.qp - q;
                    break;
                case this.S_POLE:
                    b = phi - HALF_PI;
                    q = this.qp + q;
                    break;
            }
            if (Math.abs(b) < EPSLN) {
                return null;
            }
            switch (this.mode) {
                case this.OBLIQ:
                case this.EQUIT:
                    b = Math.sqrt(2 / b);
                    if (this.mode === this.OBLIQ) {
                        y = this.ymf * b * (this.cosb1 * sinb - this.sinb1 * cosb * coslam);
                    }
                    else {
                        y = (b = Math.sqrt(2 / (1 + cosb * coslam))) * sinb * this.ymf;
                    }
                    x = this.xmf * b * cosb * sinlam;
                    break;
                case this.N_POLE:
                case this.S_POLE:
                    if (q >= 0) {
                        x = (b = Math.sqrt(q)) * sinlam;
                        y = coslam * ((this.mode === this.S_POLE) ? b : -b);
                    }
                    else {
                        x = y = 0;
                    }
                    break;
            }
        }

        p.x = this.a * x + this.x0;
        p.y = this.a * y + this.y0;
        return p;
    }

    /* Inverse equations
      -----------------*/
    function inverse$12(p) {
        p.x -= this.x0;
        p.y -= this.y0;
        var x = p.x / this.a;
        var y = p.y / this.a;
        var lam, phi, cCe, sCe, q, rho, ab;
        if (this.sphere) {
            var cosz = 0,
                rh, sinz = 0;

            rh = Math.sqrt(x * x + y * y);
            phi = rh * 0.5;
            if (phi > 1) {
                return null;
            }
            phi = 2 * Math.asin(phi);
            if (this.mode === this.OBLIQ || this.mode === this.EQUIT) {
                sinz = Math.sin(phi);
                cosz = Math.cos(phi);
            }
            switch (this.mode) {
                case this.EQUIT:
                    phi = (Math.abs(rh) <= EPSLN) ? 0 : Math.asin(y * sinz / rh);
                    x *= sinz;
                    y = cosz * rh;
                    break;
                case this.OBLIQ:
                    phi = (Math.abs(rh) <= EPSLN) ? this.phi0 : Math.asin(cosz * this.sinph0 + y * sinz * this.cosph0 / rh);
                    x *= sinz * this.cosph0;
                    y = (cosz - Math.sin(phi) * this.sinph0) * rh;
                    break;
                case this.N_POLE:
                    y = -y;
                    phi = HALF_PI - phi;
                    break;
                case this.S_POLE:
                    phi -= HALF_PI;
                    break;
            }
            lam = (y === 0 && (this.mode === this.EQUIT || this.mode === this.OBLIQ)) ? 0 : Math.atan2(x, y);
        }
        else {
            ab = 0;
            if (this.mode === this.OBLIQ || this.mode === this.EQUIT) {
                x /= this.dd;
                y *= this.dd;
                rho = Math.sqrt(x * x + y * y);
                if (rho < EPSLN) {
                    p.x = 0;
                    p.y = this.phi0;
                    return p;
                }
                sCe = 2 * Math.asin(0.5 * rho / this.rq);
                cCe = Math.cos(sCe);
                x *= (sCe = Math.sin(sCe));
                if (this.mode === this.OBLIQ) {
                    ab = cCe * this.sinb1 + y * sCe * this.cosb1 / rho;
                    q = this.qp * ab;
                    y = rho * this.cosb1 * cCe - y * this.sinb1 * sCe;
                }
                else {
                    ab = y * sCe / rho;
                    q = this.qp * ab;
                    y = rho * cCe;
                }
            }
            else if (this.mode === this.N_POLE || this.mode === this.S_POLE) {
                if (this.mode === this.N_POLE) {
                    y = -y;
                }
                q = (x * x + y * y);
                if (!q) {
                    p.x = 0;
                    p.y = this.phi0;
                    return p;
                }
                ab = 1 - q / this.qp;
                if (this.mode === this.S_POLE) {
                    ab = -ab;
                }
            }
            lam = Math.atan2(x, y);
            phi = authlat(Math.asin(ab), this.apa);
        }

        p.x = adjust_lon(this.long0 + lam);
        p.y = phi;
        return p;
    }

    /* determine latitude from authalic latitude */
    var P00 = 0.33333333333333333333;

    var P01 = 0.17222222222222222222;
    var P02 = 0.10257936507936507936;
    var P10 = 0.06388888888888888888;
    var P11 = 0.06640211640211640211;
    var P20 = 0.01641501294219154443;

    function authset(es) {
        var t;
        var APA = [];
        APA[0] = es * P00;
        t = es * es;
        APA[0] += t * P01;
        APA[1] = t * P10;
        t *= es;
        APA[0] += t * P02;
        APA[1] += t * P11;
        APA[2] = t * P20;
        return APA;
    }

    function authlat(beta, APA) {
        var t = beta + beta;
        return (beta + APA[0] * Math.sin(t) + APA[1] * Math.sin(t + t) + APA[2] * Math.sin(t + t + t));
    }

    var names$14 = ["Lambert Azimuthal Equal Area", "Lambert_Azimuthal_Equal_Area", "laea"];
    var laea = {
        init: init$13,
        forward: forward$12,
        inverse: inverse$12,
        names: names$14,
        S_POLE: S_POLE,
        N_POLE: N_POLE,
        EQUIT: EQUIT,
        OBLIQ: OBLIQ
    };

    var asinz = function(x) {
        if (Math.abs(x) > 1) {
            x = (x > 1) ? 1 : -1;
        }
        return Math.asin(x);
    };

    function init$14() {

        if (Math.abs(this.lat1 + this.lat2) < EPSLN) {
            return;
        }
        this.temp = this.b / this.a;
        this.es = 1 - Math.pow(this.temp, 2);
        this.e3 = Math.sqrt(this.es);

        this.sin_po = Math.sin(this.lat1);
        this.cos_po = Math.cos(this.lat1);
        this.t1 = this.sin_po;
        this.con = this.sin_po;
        this.ms1 = msfnz(this.e3, this.sin_po, this.cos_po);
        this.qs1 = qsfnz(this.e3, this.sin_po, this.cos_po);

        this.sin_po = Math.sin(this.lat2);
        this.cos_po = Math.cos(this.lat2);
        this.t2 = this.sin_po;
        this.ms2 = msfnz(this.e3, this.sin_po, this.cos_po);
        this.qs2 = qsfnz(this.e3, this.sin_po, this.cos_po);

        this.sin_po = Math.sin(this.lat0);
        this.cos_po = Math.cos(this.lat0);
        this.t3 = this.sin_po;
        this.qs0 = qsfnz(this.e3, this.sin_po, this.cos_po);

        if (Math.abs(this.lat1 - this.lat2) > EPSLN) {
            this.ns0 = (this.ms1 * this.ms1 - this.ms2 * this.ms2) / (this.qs2 - this.qs1);
        }
        else {
            this.ns0 = this.con;
        }
        this.c = this.ms1 * this.ms1 + this.ns0 * this.qs1;
        this.rh = this.a * Math.sqrt(this.c - this.ns0 * this.qs0) / this.ns0;
    }

    /* Albers Conical Equal Area forward equations--mapping lat,long to x,y
      -------------------------------------------------------------------*/
    function forward$13(p) {

        var lon = p.x;
        var lat = p.y;

        this.sin_phi = Math.sin(lat);
        this.cos_phi = Math.cos(lat);

        var qs = qsfnz(this.e3, this.sin_phi, this.cos_phi);
        var rh1 = this.a * Math.sqrt(this.c - this.ns0 * qs) / this.ns0;
        var theta = this.ns0 * adjust_lon(lon - this.long0);
        var x = rh1 * Math.sin(theta) + this.x0;
        var y = this.rh - rh1 * Math.cos(theta) + this.y0;

        p.x = x;
        p.y = y;
        return p;
    }

    function inverse$13(p) {
        var rh1, qs, con, theta, lon, lat;

        p.x -= this.x0;
        p.y = this.rh - p.y + this.y0;
        if (this.ns0 >= 0) {
            rh1 = Math.sqrt(p.x * p.x + p.y * p.y);
            con = 1;
        }
        else {
            rh1 = -Math.sqrt(p.x * p.x + p.y * p.y);
            con = -1;
        }
        theta = 0;
        if (rh1 !== 0) {
            theta = Math.atan2(con * p.x, con * p.y);
        }
        con = rh1 * this.ns0 / this.a;
        if (this.sphere) {
            lat = Math.asin((this.c - con * con) / (2 * this.ns0));
        }
        else {
            qs = (this.c - con * con) / this.ns0;
            lat = this.phi1z(this.e3, qs);
        }

        lon = adjust_lon(theta / this.ns0 + this.long0);
        p.x = lon;
        p.y = lat;
        return p;
    }

    /* Function to compute phi1, the latitude for the inverse of the
       Albers Conical Equal-Area projection.
    -------------------------------------------*/
    function phi1z(eccent, qs) {
        var sinphi, cosphi, con, com, dphi;
        var phi = asinz(0.5 * qs);
        if (eccent < EPSLN) {
            return phi;
        }

        var eccnts = eccent * eccent;
        for (var i = 1; i <= 25; i++) {
            sinphi = Math.sin(phi);
            cosphi = Math.cos(phi);
            con = eccent * sinphi;
            com = 1 - con * con;
            dphi = 0.5 * com * com / cosphi * (qs / (1 - eccnts) - sinphi / com + 0.5 / eccent * Math.log((1 - con) / (1 + con)));
            phi = phi + dphi;
            if (Math.abs(dphi) <= 1e-7) {
                return phi;
            }
        }
        return null;
    }

    var names$15 = ["Albers_Conic_Equal_Area", "Albers", "aea"];
    var aea = {
        init: init$14,
        forward: forward$13,
        inverse: inverse$13,
        names: names$15,
        phi1z: phi1z
    };

    /*
      reference:
        Wolfram Mathworld "Gnomonic Projection"
        http://mathworld.wolfram.com/GnomonicProjection.html
        Accessed: 12th November 2009
      */
    function init$15() {

        /* Place parameters in static storage for common use
          -------------------------------------------------*/
        this.sin_p14 = Math.sin(this.lat0);
        this.cos_p14 = Math.cos(this.lat0);
        // Approximation for projecting points to the horizon (infinity)
        this.infinity_dist = 1000 * this.a;
        this.rc = 1;
    }

    /* Gnomonic forward equations--mapping lat,long to x,y
        ---------------------------------------------------*/
    function forward$14(p) {
        var sinphi, cosphi; /* sin and cos value        */
        var dlon; /* delta longitude value      */
        var coslon; /* cos of longitude        */
        var ksp; /* scale factor          */
        var g;
        var x, y;
        var lon = p.x;
        var lat = p.y;
        /* Forward equations
          -----------------*/
        dlon = adjust_lon(lon - this.long0);

        sinphi = Math.sin(lat);
        cosphi = Math.cos(lat);

        coslon = Math.cos(dlon);
        g = this.sin_p14 * sinphi + this.cos_p14 * cosphi * coslon;
        ksp = 1;
        if ((g > 0) || (Math.abs(g) <= EPSLN)) {
            x = this.x0 + this.a * ksp * cosphi * Math.sin(dlon) / g;
            y = this.y0 + this.a * ksp * (this.cos_p14 * sinphi - this.sin_p14 * cosphi * coslon) / g;
        }
        else {

            // Point is in the opposing hemisphere and is unprojectable
            // We still need to return a reasonable point, so we project
            // to infinity, on a bearing
            // equivalent to the northern hemisphere equivalent
            // This is a reasonable approximation for short shapes and lines that
            // straddle the horizon.

            x = this.x0 + this.infinity_dist * cosphi * Math.sin(dlon);
            y = this.y0 + this.infinity_dist * (this.cos_p14 * sinphi - this.sin_p14 * cosphi * coslon);

        }
        p.x = x;
        p.y = y;
        return p;
    }

    function inverse$14(p) {
        var rh; /* Rho */
        var sinc, cosc;
        var c;
        var lon, lat;

        /* Inverse equations
          -----------------*/
        p.x = (p.x - this.x0) / this.a;
        p.y = (p.y - this.y0) / this.a;

        p.x /= this.k0;
        p.y /= this.k0;

        if ((rh = Math.sqrt(p.x * p.x + p.y * p.y))) {
            c = Math.atan2(rh, this.rc);
            sinc = Math.sin(c);
            cosc = Math.cos(c);

            lat = asinz(cosc * this.sin_p14 + (p.y * sinc * this.cos_p14) / rh);
            lon = Math.atan2(p.x * sinc, rh * this.cos_p14 * cosc - p.y * this.sin_p14 * sinc);
            lon = adjust_lon(this.long0 + lon);
        }
        else {
            lat = this.phic0;
            lon = 0;
        }

        p.x = lon;
        p.y = lat;
        return p;
    }

    var names$16 = ["gnom"];
    var gnom = {
        init: init$15,
        forward: forward$14,
        inverse: inverse$14,
        names: names$16
    };

    var iqsfnz = function(eccent, q) {
        var temp = 1 - (1 - eccent * eccent) / (2 * eccent) * Math.log((1 - eccent) / (1 + eccent));
        if (Math.abs(Math.abs(q) - temp) < 1.0E-6) {
            if (q < 0) {
                return (-1 * HALF_PI);
            }
            else {
                return HALF_PI;
            }
        }
        //var phi = 0.5* q/(1-eccent*eccent);
        var phi = Math.asin(0.5 * q);
        var dphi;
        var sin_phi;
        var cos_phi;
        var con;
        for (var i = 0; i < 30; i++) {
            sin_phi = Math.sin(phi);
            cos_phi = Math.cos(phi);
            con = eccent * sin_phi;
            dphi = Math.pow(1 - con * con, 2) / (2 * cos_phi) * (q / (1 - eccent * eccent) - sin_phi / (1 - con * con) + 0.5 / eccent * Math.log((1 - con) / (1 + con)));
            phi += dphi;
            if (Math.abs(dphi) <= 0.0000000001) {
                return phi;
            }
        }

        //console.log("IQSFN-CONV:Latitude failed to converge after 30 iterations");
        return NaN;
    };

    /*
      reference:
        "Cartographic Projection Procedures for the UNIX Environment-
        A User's Manual" by Gerald I. Evenden,
        USGS Open File Report 90-284and Release 4 Interim Reports (2003)
    */
    function init$16() {
        //no-op
        if (!this.sphere) {
            this.k0 = msfnz(this.e, Math.sin(this.lat_ts), Math.cos(this.lat_ts));
        }
    }

    /* Cylindrical Equal Area forward equations--mapping lat,long to x,y
        ------------------------------------------------------------*/
    function forward$15(p) {
        var lon = p.x;
        var lat = p.y;
        var x, y;
        /* Forward equations
          -----------------*/
        var dlon = adjust_lon(lon - this.long0);
        if (this.sphere) {
            x = this.x0 + this.a * dlon * Math.cos(this.lat_ts);
            y = this.y0 + this.a * Math.sin(lat) / Math.cos(this.lat_ts);
        }
        else {
            var qs = qsfnz(this.e, Math.sin(lat));
            x = this.x0 + this.a * this.k0 * dlon;
            y = this.y0 + this.a * qs * 0.5 / this.k0;
        }

        p.x = x;
        p.y = y;
        return p;
    }

    /* Cylindrical Equal Area inverse equations--mapping x,y to lat/long
        ------------------------------------------------------------*/
    function inverse$15(p) {
        p.x -= this.x0;
        p.y -= this.y0;
        var lon, lat;

        if (this.sphere) {
            lon = adjust_lon(this.long0 + (p.x / this.a) / Math.cos(this.lat_ts));
            lat = Math.asin((p.y / this.a) * Math.cos(this.lat_ts));
        }
        else {
            lat = iqsfnz(this.e, 2 * p.y * this.k0 / this.a);
            lon = adjust_lon(this.long0 + p.x / (this.a * this.k0));
        }

        p.x = lon;
        p.y = lat;
        return p;
    }

    var names$17 = ["cea"];
    var cea = {
        init: init$16,
        forward: forward$15,
        inverse: inverse$15,
        names: names$17
    };

    function init$17() {

        this.x0 = this.x0 || 0;
        this.y0 = this.y0 || 0;
        this.lat0 = this.lat0 || 0;
        this.long0 = this.long0 || 0;
        this.lat_ts = this.lat_ts || 0;
        this.title = this.title || "Equidistant Cylindrical (Plate Carre)";

        this.rc = Math.cos(this.lat_ts);
    }

    // forward equations--mapping lat,long to x,y
    // -----------------------------------------------------------------
    function forward$16(p) {

        var lon = p.x;
        var lat = p.y;

        var dlon = adjust_lon(lon - this.long0);
        var dlat = adjust_lat(lat - this.lat0);
        p.x = this.x0 + (this.a * dlon * this.rc);
        p.y = this.y0 + (this.a * dlat);
        return p;
    }

    // inverse equations--mapping x,y to lat/long
    // -----------------------------------------------------------------
    function inverse$16(p) {

        var x = p.x;
        var y = p.y;

        p.x = adjust_lon(this.long0 + ((x - this.x0) / (this.a * this.rc)));
        p.y = adjust_lat(this.lat0 + ((y - this.y0) / (this.a)));
        return p;
    }

    var names$18 = ["Equirectangular", "Equidistant_Cylindrical", "eqc"];
    var eqc = {
        init: init$17,
        forward: forward$16,
        inverse: inverse$16,
        names: names$18
    };

    var MAX_ITER$2 = 20;

    function init$18() {
        /* Place parameters in static storage for common use
          -------------------------------------------------*/
        this.temp = this.b / this.a;
        this.es = 1 - Math.pow(this.temp, 2); // devait etre dans tmerc.js mais n y est pas donc je commente sinon retour de valeurs nulles
        this.e = Math.sqrt(this.es);
        this.e0 = e0fn(this.es);
        this.e1 = e1fn(this.es);
        this.e2 = e2fn(this.es);
        this.e3 = e3fn(this.es);
        this.ml0 = this.a * mlfn(this.e0, this.e1, this.e2, this.e3, this.lat0); //si que des zeros le calcul ne se fait pas
    }

    /* Polyconic forward equations--mapping lat,long to x,y
        ---------------------------------------------------*/
    function forward$17(p) {
        var lon = p.x;
        var lat = p.y;
        var x, y, el;
        var dlon = adjust_lon(lon - this.long0);
        el = dlon * Math.sin(lat);
        if (this.sphere) {
            if (Math.abs(lat) <= EPSLN) {
                x = this.a * dlon;
                y = -1 * this.a * this.lat0;
            }
            else {
                x = this.a * Math.sin(el) / Math.tan(lat);
                y = this.a * (adjust_lat(lat - this.lat0) + (1 - Math.cos(el)) / Math.tan(lat));
            }
        }
        else {
            if (Math.abs(lat) <= EPSLN) {
                x = this.a * dlon;
                y = -1 * this.ml0;
            }
            else {
                var nl = gN(this.a, this.e, Math.sin(lat)) / Math.tan(lat);
                x = nl * Math.sin(el);
                y = this.a * mlfn(this.e0, this.e1, this.e2, this.e3, lat) - this.ml0 + nl * (1 - Math.cos(el));
            }

        }
        p.x = x + this.x0;
        p.y = y + this.y0;
        return p;
    }

    /* Inverse equations
      -----------------*/
    function inverse$17(p) {
        var lon, lat, x, y, i;
        var al, bl;
        var phi, dphi;
        x = p.x - this.x0;
        y = p.y - this.y0;

        if (this.sphere) {
            if (Math.abs(y + this.a * this.lat0) <= EPSLN) {
                lon = adjust_lon(x / this.a + this.long0);
                lat = 0;
            }
            else {
                al = this.lat0 + y / this.a;
                bl = x * x / this.a / this.a + al * al;
                phi = al;
                var tanphi;
                for (i = MAX_ITER$2; i; --i) {
                    tanphi = Math.tan(phi);
                    dphi = -1 * (al * (phi * tanphi + 1) - phi - 0.5 * (phi * phi + bl) * tanphi) / ((phi - al) / tanphi - 1);
                    phi += dphi;
                    if (Math.abs(dphi) <= EPSLN) {
                        lat = phi;
                        break;
                    }
                }
                lon = adjust_lon(this.long0 + (Math.asin(x * Math.tan(phi) / this.a)) / Math.sin(lat));
            }
        }
        else {
            if (Math.abs(y + this.ml0) <= EPSLN) {
                lat = 0;
                lon = adjust_lon(this.long0 + x / this.a);
            }
            else {

                al = (this.ml0 + y) / this.a;
                bl = x * x / this.a / this.a + al * al;
                phi = al;
                var cl, mln, mlnp, ma;
                var con;
                for (i = MAX_ITER$2; i; --i) {
                    con = this.e * Math.sin(phi);
                    cl = Math.sqrt(1 - con * con) * Math.tan(phi);
                    mln = this.a * mlfn(this.e0, this.e1, this.e2, this.e3, phi);
                    mlnp = this.e0 - 2 * this.e1 * Math.cos(2 * phi) + 4 * this.e2 * Math.cos(4 * phi) - 6 * this.e3 * Math.cos(6 * phi);
                    ma = mln / this.a;
                    dphi = (al * (cl * ma + 1) - ma - 0.5 * cl * (ma * ma + bl)) / (this.es * Math.sin(2 * phi) * (ma * ma + bl - 2 * al * ma) / (4 * cl) + (al - ma) * (cl * mlnp - 2 / Math.sin(2 * phi)) - mlnp);
                    phi -= dphi;
                    if (Math.abs(dphi) <= EPSLN) {
                        lat = phi;
                        break;
                    }
                }

                //lat=phi4z(this.e,this.e0,this.e1,this.e2,this.e3,al,bl,0,0);
                cl = Math.sqrt(1 - this.es * Math.pow(Math.sin(lat), 2)) * Math.tan(lat);
                lon = adjust_lon(this.long0 + Math.asin(x * cl / this.a) / Math.sin(lat));
            }
        }

        p.x = lon;
        p.y = lat;
        return p;
    }

    var names$19 = ["Polyconic", "poly"];
    var poly = {
        init: init$18,
        forward: forward$17,
        inverse: inverse$17,
        names: names$19
    };

    /*
      reference
        Department of Land and Survey Technical Circular 1973/32
          http://www.linz.govt.nz/docs/miscellaneous/nz-map-definition.pdf
        OSG Technical Report 4.1
          http://www.linz.govt.nz/docs/miscellaneous/nzmg.pdf
      */

    /**
     * iterations: Number of iterations to refine inverse transform.
     *     0 -> km accuracy
     *     1 -> m accuracy -- suitable for most mapping applications
     *     2 -> mm accuracy
     */


    function init$19() {
        this.A = [];
        this.A[1] = 0.6399175073;
        this.A[2] = -0.1358797613;
        this.A[3] = 0.063294409;
        this.A[4] = -0.02526853;
        this.A[5] = 0.0117879;
        this.A[6] = -0.0055161;
        this.A[7] = 0.0026906;
        this.A[8] = -0.001333;
        this.A[9] = 0.00067;
        this.A[10] = -0.00034;

        this.B_re = [];
        this.B_im = [];
        this.B_re[1] = 0.7557853228;
        this.B_im[1] = 0;
        this.B_re[2] = 0.249204646;
        this.B_im[2] = 0.003371507;
        this.B_re[3] = -0.001541739;
        this.B_im[3] = 0.041058560;
        this.B_re[4] = -0.10162907;
        this.B_im[4] = 0.01727609;
        this.B_re[5] = -0.26623489;
        this.B_im[5] = -0.36249218;
        this.B_re[6] = -0.6870983;
        this.B_im[6] = -1.1651967;

        this.C_re = [];
        this.C_im = [];
        this.C_re[1] = 1.3231270439;
        this.C_im[1] = 0;
        this.C_re[2] = -0.577245789;
        this.C_im[2] = -0.007809598;
        this.C_re[3] = 0.508307513;
        this.C_im[3] = -0.112208952;
        this.C_re[4] = -0.15094762;
        this.C_im[4] = 0.18200602;
        this.C_re[5] = 1.01418179;
        this.C_im[5] = 1.64497696;
        this.C_re[6] = 1.9660549;
        this.C_im[6] = 2.5127645;

        this.D = [];
        this.D[1] = 1.5627014243;
        this.D[2] = 0.5185406398;
        this.D[3] = -0.03333098;
        this.D[4] = -0.1052906;
        this.D[5] = -0.0368594;
        this.D[6] = 0.007317;
        this.D[7] = 0.01220;
        this.D[8] = 0.00394;
        this.D[9] = -0.0013;
    }

    /**
     New Zealand Map Grid Forward  - long/lat to x/y
     long/lat in radians
     */
    function forward$18(p) {
        var n;
        var lon = p.x;
        var lat = p.y;

        var delta_lat = lat - this.lat0;
        var delta_lon = lon - this.long0;

        // 1. Calculate d_phi and d_psi    ...                          // and d_lambda
        // For this algorithm, delta_latitude is in seconds of arc x 10-5, so we need to scale to those units. Longitude is radians.
        var d_phi = delta_lat / SEC_TO_RAD * 1E-5;
        var d_lambda = delta_lon;
        var d_phi_n = 1; // d_phi^0

        var d_psi = 0;
        for (n = 1; n <= 10; n++) {
            d_phi_n = d_phi_n * d_phi;
            d_psi = d_psi + this.A[n] * d_phi_n;
        }

        // 2. Calculate theta
        var th_re = d_psi;
        var th_im = d_lambda;

        // 3. Calculate z
        var th_n_re = 1;
        var th_n_im = 0; // theta^0
        var th_n_re1;
        var th_n_im1;

        var z_re = 0;
        var z_im = 0;
        for (n = 1; n <= 6; n++) {
            th_n_re1 = th_n_re * th_re - th_n_im * th_im;
            th_n_im1 = th_n_im * th_re + th_n_re * th_im;
            th_n_re = th_n_re1;
            th_n_im = th_n_im1;
            z_re = z_re + this.B_re[n] * th_n_re - this.B_im[n] * th_n_im;
            z_im = z_im + this.B_im[n] * th_n_re + this.B_re[n] * th_n_im;
        }

        // 4. Calculate easting and northing
        p.x = (z_im * this.a) + this.x0;
        p.y = (z_re * this.a) + this.y0;

        return p;
    }

    /**
     New Zealand Map Grid Inverse  -  x/y to long/lat
     */
    function inverse$18(p) {
        var n;
        var x = p.x;
        var y = p.y;

        var delta_x = x - this.x0;
        var delta_y = y - this.y0;

        // 1. Calculate z
        var z_re = delta_y / this.a;
        var z_im = delta_x / this.a;

        // 2a. Calculate theta - first approximation gives km accuracy
        var z_n_re = 1;
        var z_n_im = 0; // z^0
        var z_n_re1;
        var z_n_im1;

        var th_re = 0;
        var th_im = 0;
        for (n = 1; n <= 6; n++) {
            z_n_re1 = z_n_re * z_re - z_n_im * z_im;
            z_n_im1 = z_n_im * z_re + z_n_re * z_im;
            z_n_re = z_n_re1;
            z_n_im = z_n_im1;
            th_re = th_re + this.C_re[n] * z_n_re - this.C_im[n] * z_n_im;
            th_im = th_im + this.C_im[n] * z_n_re + this.C_re[n] * z_n_im;
        }

        // 2b. Iterate to refine the accuracy of the calculation
        //        0 iterations gives km accuracy
        //        1 iteration gives m accuracy -- good enough for most mapping applications
        //        2 iterations bives mm accuracy
        for (var i = 0; i < this.iterations; i++) {
            var th_n_re = th_re;
            var th_n_im = th_im;
            var th_n_re1;
            var th_n_im1;

            var num_re = z_re;
            var num_im = z_im;
            for (n = 2; n <= 6; n++) {
                th_n_re1 = th_n_re * th_re - th_n_im * th_im;
                th_n_im1 = th_n_im * th_re + th_n_re * th_im;
                th_n_re = th_n_re1;
                th_n_im = th_n_im1;
                num_re = num_re + (n - 1) * (this.B_re[n] * th_n_re - this.B_im[n] * th_n_im);
                num_im = num_im + (n - 1) * (this.B_im[n] * th_n_re + this.B_re[n] * th_n_im);
            }

            th_n_re = 1;
            th_n_im = 0;
            var den_re = this.B_re[1];
            var den_im = this.B_im[1];
            for (n = 2; n <= 6; n++) {
                th_n_re1 = th_n_re * th_re - th_n_im * th_im;
                th_n_im1 = th_n_im * th_re + th_n_re * th_im;
                th_n_re = th_n_re1;
                th_n_im = th_n_im1;
                den_re = den_re + n * (this.B_re[n] * th_n_re - this.B_im[n] * th_n_im);
                den_im = den_im + n * (this.B_im[n] * th_n_re + this.B_re[n] * th_n_im);
            }

            // Complex division
            var den2 = den_re * den_re + den_im * den_im;
            th_re = (num_re * den_re + num_im * den_im) / den2;
            th_im = (num_im * den_re - num_re * den_im) / den2;
        }

        // 3. Calculate d_phi              ...                                    // and d_lambda
        var d_psi = th_re;
        var d_lambda = th_im;
        var d_psi_n = 1; // d_psi^0

        var d_phi = 0;
        for (n = 1; n <= 9; n++) {
            d_psi_n = d_psi_n * d_psi;
            d_phi = d_phi + this.D[n] * d_psi_n;
        }

        // 4. Calculate latitude and longitude
        // d_phi is calcuated in second of arc * 10^-5, so we need to scale back to radians. d_lambda is in radians.
        var lat = this.lat0 + (d_phi * SEC_TO_RAD * 1E5);
        var lon = this.long0 + d_lambda;

        p.x = lon;
        p.y = lat;

        return p;
    }

    var names$20 = ["New_Zealand_Map_Grid", "nzmg"];
    var nzmg = {
        init: init$19,
        forward: forward$18,
        inverse: inverse$18,
        names: names$20
    };

    /*
      reference
        "New Equal-Area Map Projections for Noncircular Regions", John P. Snyder,
        The American Cartographer, Vol 15, No. 4, October 1988, pp. 341-355.
      */


    /* Initialize the Miller Cylindrical projection
      -------------------------------------------*/
    function init$20() {
        //no-op
    }

    /* Miller Cylindrical forward equations--mapping lat,long to x,y
        ------------------------------------------------------------*/
    function forward$19(p) {
        var lon = p.x;
        var lat = p.y;
        /* Forward equations
          -----------------*/
        var dlon = adjust_lon(lon - this.long0);
        var x = this.x0 + this.a * dlon;
        var y = this.y0 + this.a * Math.log(Math.tan((Math.PI / 4) + (lat / 2.5))) * 1.25;

        p.x = x;
        p.y = y;
        return p;
    }

    /* Miller Cylindrical inverse equations--mapping x,y to lat/long
        ------------------------------------------------------------*/
    function inverse$19(p) {
        p.x -= this.x0;
        p.y -= this.y0;

        var lon = adjust_lon(this.long0 + p.x / this.a);
        var lat = 2.5 * (Math.atan(Math.exp(0.8 * p.y / this.a)) - Math.PI / 4);

        p.x = lon;
        p.y = lat;
        return p;
    }

    var names$21 = ["Miller_Cylindrical", "mill"];
    var mill = {
        init: init$20,
        forward: forward$19,
        inverse: inverse$19,
        names: names$21
    };

    var MAX_ITER$3 = 20;
    function init$21() {
        /* Place parameters in static storage for common use
        -------------------------------------------------*/


        if (!this.sphere) {
            this.en = pj_enfn(this.es);
        }
        else {
            this.n = 1;
            this.m = 0;
            this.es = 0;
            this.C_y = Math.sqrt((this.m + 1) / this.n);
            this.C_x = this.C_y / (this.m + 1);
        }

    }

    /* Sinusoidal forward equations--mapping lat,long to x,y
      -----------------------------------------------------*/
    function forward$20(p) {
        var x, y;
        var lon = p.x;
        var lat = p.y;
        /* Forward equations
        -----------------*/
        lon = adjust_lon(lon - this.long0);

        if (this.sphere) {
            if (!this.m) {
                lat = this.n !== 1 ? Math.asin(this.n * Math.sin(lat)) : lat;
            }
            else {
                var k = this.n * Math.sin(lat);
                for (var i = MAX_ITER$3; i; --i) {
                    var V = (this.m * lat + Math.sin(lat) - k) / (this.m + Math.cos(lat));
                    lat -= V;
                    if (Math.abs(V) < EPSLN) {
                        break;
                    }
                }
            }
            x = this.a * this.C_x * lon * (this.m + Math.cos(lat));
            y = this.a * this.C_y * lat;

        }
        else {

            var s = Math.sin(lat);
            var c = Math.cos(lat);
            y = this.a * pj_mlfn(lat, s, c, this.en);
            x = this.a * lon * c / Math.sqrt(1 - this.es * s * s);
        }

        p.x = x;
        p.y = y;
        return p;
    }

    function inverse$20(p) {
        var lat, temp, lon, s;

        p.x -= this.x0;
        lon = p.x / this.a;
        p.y -= this.y0;
        lat = p.y / this.a;

        if (this.sphere) {
            lat /= this.C_y;
            lon = lon / (this.C_x * (this.m + Math.cos(lat)));
            if (this.m) {
                lat = asinz((this.m * lat + Math.sin(lat)) / this.n);
            }
            else if (this.n !== 1) {
                lat = asinz(Math.sin(lat) / this.n);
            }
            lon = adjust_lon(lon + this.long0);
            lat = adjust_lat(lat);
        }
        else {
            lat = pj_inv_mlfn(p.y / this.a, this.es, this.en);
            s = Math.abs(lat);
            if (s < HALF_PI) {
                s = Math.sin(lat);
                temp = this.long0 + p.x * Math.sqrt(1 - this.es * s * s) / (this.a * Math.cos(lat));
                //temp = this.long0 + p.x / (this.a * Math.cos(lat));
                lon = adjust_lon(temp);
            }
            else if ((s - EPSLN) < HALF_PI) {
                lon = this.long0;
            }
        }
        p.x = lon;
        p.y = lat;
        return p;
    }

    var names$22 = ["Sinusoidal", "sinu"];
    var sinu = {
        init: init$21,
        forward: forward$20,
        inverse: inverse$20,
        names: names$22
    };

    function init$22() {}
    /* Mollweide forward equations--mapping lat,long to x,y
        ----------------------------------------------------*/
    function forward$21(p) {

        /* Forward equations
          -----------------*/
        var lon = p.x;
        var lat = p.y;

        var delta_lon = adjust_lon(lon - this.long0);
        var theta = lat;
        var con = Math.PI * Math.sin(lat);

        /* Iterate using the Newton-Raphson method to find theta
          -----------------------------------------------------*/
        while (true) {
            var delta_theta = -(theta + Math.sin(theta) - con) / (1 + Math.cos(theta));
            theta += delta_theta;
            if (Math.abs(delta_theta) < EPSLN) {
                break;
            }
        }
        theta /= 2;

        /* If the latitude is 90 deg, force the x coordinate to be "0 + false easting"
           this is done here because of precision problems with "cos(theta)"
           --------------------------------------------------------------------------*/
        if (Math.PI / 2 - Math.abs(lat) < EPSLN) {
            delta_lon = 0;
        }
        var x = 0.900316316158 * this.a * delta_lon * Math.cos(theta) + this.x0;
        var y = 1.4142135623731 * this.a * Math.sin(theta) + this.y0;

        p.x = x;
        p.y = y;
        return p;
    }

    function inverse$21(p) {
        var theta;
        var arg;

        /* Inverse equations
          -----------------*/
        p.x -= this.x0;
        p.y -= this.y0;
        arg = p.y / (1.4142135623731 * this.a);

        /* Because of division by zero problems, 'arg' can not be 1.  Therefore
           a number very close to one is used instead.
           -------------------------------------------------------------------*/
        if (Math.abs(arg) > 0.999999999999) {
            arg = 0.999999999999;
        }
        theta = Math.asin(arg);
        var lon = adjust_lon(this.long0 + (p.x / (0.900316316158 * this.a * Math.cos(theta))));
        if (lon < (-Math.PI)) {
            lon = -Math.PI;
        }
        if (lon > Math.PI) {
            lon = Math.PI;
        }
        arg = (2 * theta + Math.sin(2 * theta)) / Math.PI;
        if (Math.abs(arg) > 1) {
            arg = 1;
        }
        var lat = Math.asin(arg);

        p.x = lon;
        p.y = lat;
        return p;
    }

    var names$23 = ["Mollweide", "moll"];
    var moll = {
        init: init$22,
        forward: forward$21,
        inverse: inverse$21,
        names: names$23
    };

    function init$23() {

        /* Place parameters in static storage for common use
          -------------------------------------------------*/
        // Standard Parallels cannot be equal and on opposite sides of the equator
        if (Math.abs(this.lat1 + this.lat2) < EPSLN) {
            return;
        }
        this.lat2 = this.lat2 || this.lat1;
        this.temp = this.b / this.a;
        this.es = 1 - Math.pow(this.temp, 2);
        this.e = Math.sqrt(this.es);
        this.e0 = e0fn(this.es);
        this.e1 = e1fn(this.es);
        this.e2 = e2fn(this.es);
        this.e3 = e3fn(this.es);

        this.sinphi = Math.sin(this.lat1);
        this.cosphi = Math.cos(this.lat1);

        this.ms1 = msfnz(this.e, this.sinphi, this.cosphi);
        this.ml1 = mlfn(this.e0, this.e1, this.e2, this.e3, this.lat1);

        if (Math.abs(this.lat1 - this.lat2) < EPSLN) {
            this.ns = this.sinphi;
        }
        else {
            this.sinphi = Math.sin(this.lat2);
            this.cosphi = Math.cos(this.lat2);
            this.ms2 = msfnz(this.e, this.sinphi, this.cosphi);
            this.ml2 = mlfn(this.e0, this.e1, this.e2, this.e3, this.lat2);
            this.ns = (this.ms1 - this.ms2) / (this.ml2 - this.ml1);
        }
        this.g = this.ml1 + this.ms1 / this.ns;
        this.ml0 = mlfn(this.e0, this.e1, this.e2, this.e3, this.lat0);
        this.rh = this.a * (this.g - this.ml0);
    }

    /* Equidistant Conic forward equations--mapping lat,long to x,y
      -----------------------------------------------------------*/
    function forward$22(p) {
        var lon = p.x;
        var lat = p.y;
        var rh1;

        /* Forward equations
          -----------------*/
        if (this.sphere) {
            rh1 = this.a * (this.g - lat);
        }
        else {
            var ml = mlfn(this.e0, this.e1, this.e2, this.e3, lat);
            rh1 = this.a * (this.g - ml);
        }
        var theta = this.ns * adjust_lon(lon - this.long0);
        var x = this.x0 + rh1 * Math.sin(theta);
        var y = this.y0 + this.rh - rh1 * Math.cos(theta);
        p.x = x;
        p.y = y;
        return p;
    }

    /* Inverse equations
      -----------------*/
    function inverse$22(p) {
        p.x -= this.x0;
        p.y = this.rh - p.y + this.y0;
        var con, rh1, lat, lon;
        if (this.ns >= 0) {
            rh1 = Math.sqrt(p.x * p.x + p.y * p.y);
            con = 1;
        }
        else {
            rh1 = -Math.sqrt(p.x * p.x + p.y * p.y);
            con = -1;
        }
        var theta = 0;
        if (rh1 !== 0) {
            theta = Math.atan2(con * p.x, con * p.y);
        }

        if (this.sphere) {
            lon = adjust_lon(this.long0 + theta / this.ns);
            lat = adjust_lat(this.g - rh1 / this.a);
            p.x = lon;
            p.y = lat;
            return p;
        }
        else {
            var ml = this.g - rh1 / this.a;
            lat = imlfn(ml, this.e0, this.e1, this.e2, this.e3);
            lon = adjust_lon(this.long0 + theta / this.ns);
            p.x = lon;
            p.y = lat;
            return p;
        }

    }

    var names$24 = ["Equidistant_Conic", "eqdc"];
    var eqdc = {
        init: init$23,
        forward: forward$22,
        inverse: inverse$22,
        names: names$24
    };

    /* Initialize the Van Der Grinten projection
      ----------------------------------------*/
    function init$24() {
        //this.R = 6370997; //Radius of earth
        this.R = this.a;
    }

    function forward$23(p) {

        var lon = p.x;
        var lat = p.y;

        /* Forward equations
        -----------------*/
        var dlon = adjust_lon(lon - this.long0);
        var x, y;

        if (Math.abs(lat) <= EPSLN) {
            x = this.x0 + this.R * dlon;
            y = this.y0;
        }
        var theta = asinz(2 * Math.abs(lat / Math.PI));
        if ((Math.abs(dlon) <= EPSLN) || (Math.abs(Math.abs(lat) - HALF_PI) <= EPSLN)) {
            x = this.x0;
            if (lat >= 0) {
                y = this.y0 + Math.PI * this.R * Math.tan(0.5 * theta);
            }
            else {
                y = this.y0 + Math.PI * this.R * -Math.tan(0.5 * theta);
            }
            //  return(OK);
        }
        var al = 0.5 * Math.abs((Math.PI / dlon) - (dlon / Math.PI));
        var asq = al * al;
        var sinth = Math.sin(theta);
        var costh = Math.cos(theta);

        var g = costh / (sinth + costh - 1);
        var gsq = g * g;
        var m = g * (2 / sinth - 1);
        var msq = m * m;
        var con = Math.PI * this.R * (al * (g - msq) + Math.sqrt(asq * (g - msq) * (g - msq) - (msq + asq) * (gsq - msq))) / (msq + asq);
        if (dlon < 0) {
            con = -con;
        }
        x = this.x0 + con;
        //con = Math.abs(con / (Math.PI * this.R));
        var q = asq + g;
        con = Math.PI * this.R * (m * q - al * Math.sqrt((msq + asq) * (asq + 1) - q * q)) / (msq + asq);
        if (lat >= 0) {
            //y = this.y0 + Math.PI * this.R * Math.sqrt(1 - con * con - 2 * al * con);
            y = this.y0 + con;
        }
        else {
            //y = this.y0 - Math.PI * this.R * Math.sqrt(1 - con * con - 2 * al * con);
            y = this.y0 - con;
        }
        p.x = x;
        p.y = y;
        return p;
    }

    /* Van Der Grinten inverse equations--mapping x,y to lat/long
      ---------------------------------------------------------*/
    function inverse$23(p) {
        var lon, lat;
        var xx, yy, xys, c1, c2, c3;
        var a1;
        var m1;
        var con;
        var th1;
        var d;

        /* inverse equations
        -----------------*/
        p.x -= this.x0;
        p.y -= this.y0;
        con = Math.PI * this.R;
        xx = p.x / con;
        yy = p.y / con;
        xys = xx * xx + yy * yy;
        c1 = -Math.abs(yy) * (1 + xys);
        c2 = c1 - 2 * yy * yy + xx * xx;
        c3 = -2 * c1 + 1 + 2 * yy * yy + xys * xys;
        d = yy * yy / c3 + (2 * c2 * c2 * c2 / c3 / c3 / c3 - 9 * c1 * c2 / c3 / c3) / 27;
        a1 = (c1 - c2 * c2 / 3 / c3) / c3;
        m1 = 2 * Math.sqrt(-a1 / 3);
        con = ((3 * d) / a1) / m1;
        if (Math.abs(con) > 1) {
            if (con >= 0) {
                con = 1;
            }
            else {
                con = -1;
            }
        }
        th1 = Math.acos(con) / 3;
        if (p.y >= 0) {
            lat = (-m1 * Math.cos(th1 + Math.PI / 3) - c2 / 3 / c3) * Math.PI;
        }
        else {
            lat = -(-m1 * Math.cos(th1 + Math.PI / 3) - c2 / 3 / c3) * Math.PI;
        }

        if (Math.abs(xx) < EPSLN) {
            lon = this.long0;
        }
        else {
            lon = adjust_lon(this.long0 + Math.PI * (xys - 1 + Math.sqrt(1 + 2 * (xx * xx - yy * yy) + xys * xys)) / 2 / xx);
        }

        p.x = lon;
        p.y = lat;
        return p;
    }

    var names$25 = ["Van_der_Grinten_I", "VanDerGrinten", "vandg"];
    var vandg = {
        init: init$24,
        forward: forward$23,
        inverse: inverse$23,
        names: names$25
    };

    function init$25() {
        this.sin_p12 = Math.sin(this.lat0);
        this.cos_p12 = Math.cos(this.lat0);
    }

    function forward$24(p) {
        var lon = p.x;
        var lat = p.y;
        var sinphi = Math.sin(p.y);
        var cosphi = Math.cos(p.y);
        var dlon = adjust_lon(lon - this.long0);
        var e0, e1, e2, e3, Mlp, Ml, tanphi, Nl1, Nl, psi, Az, G, H, GH, Hs, c, kp, cos_c, s, s2, s3, s4, s5;
        if (this.sphere) {
            if (Math.abs(this.sin_p12 - 1) <= EPSLN) {
                //North Pole case
                p.x = this.x0 + this.a * (HALF_PI - lat) * Math.sin(dlon);
                p.y = this.y0 - this.a * (HALF_PI - lat) * Math.cos(dlon);
                return p;
            }
            else if (Math.abs(this.sin_p12 + 1) <= EPSLN) {
                //South Pole case
                p.x = this.x0 + this.a * (HALF_PI + lat) * Math.sin(dlon);
                p.y = this.y0 + this.a * (HALF_PI + lat) * Math.cos(dlon);
                return p;
            }
            else {
                //default case
                cos_c = this.sin_p12 * sinphi + this.cos_p12 * cosphi * Math.cos(dlon);
                c = Math.acos(cos_c);
                kp = c / Math.sin(c);
                p.x = this.x0 + this.a * kp * cosphi * Math.sin(dlon);
                p.y = this.y0 + this.a * kp * (this.cos_p12 * sinphi - this.sin_p12 * cosphi * Math.cos(dlon));
                return p;
            }
        }
        else {
            e0 = e0fn(this.es);
            e1 = e1fn(this.es);
            e2 = e2fn(this.es);
            e3 = e3fn(this.es);
            if (Math.abs(this.sin_p12 - 1) <= EPSLN) {
                //North Pole case
                Mlp = this.a * mlfn(e0, e1, e2, e3, HALF_PI);
                Ml = this.a * mlfn(e0, e1, e2, e3, lat);
                p.x = this.x0 + (Mlp - Ml) * Math.sin(dlon);
                p.y = this.y0 - (Mlp - Ml) * Math.cos(dlon);
                return p;
            }
            else if (Math.abs(this.sin_p12 + 1) <= EPSLN) {
                //South Pole case
                Mlp = this.a * mlfn(e0, e1, e2, e3, HALF_PI);
                Ml = this.a * mlfn(e0, e1, e2, e3, lat);
                p.x = this.x0 + (Mlp + Ml) * Math.sin(dlon);
                p.y = this.y0 + (Mlp + Ml) * Math.cos(dlon);
                return p;
            }
            else {
                //Default case
                tanphi = sinphi / cosphi;
                Nl1 = gN(this.a, this.e, this.sin_p12);
                Nl = gN(this.a, this.e, sinphi);
                psi = Math.atan((1 - this.es) * tanphi + this.es * Nl1 * this.sin_p12 / (Nl * cosphi));
                Az = Math.atan2(Math.sin(dlon), this.cos_p12 * Math.tan(psi) - this.sin_p12 * Math.cos(dlon));
                if (Az === 0) {
                    s = Math.asin(this.cos_p12 * Math.sin(psi) - this.sin_p12 * Math.cos(psi));
                }
                else if (Math.abs(Math.abs(Az) - Math.PI) <= EPSLN) {
                    s = -Math.asin(this.cos_p12 * Math.sin(psi) - this.sin_p12 * Math.cos(psi));
                }
                else {
                    s = Math.asin(Math.sin(dlon) * Math.cos(psi) / Math.sin(Az));
                }
                G = this.e * this.sin_p12 / Math.sqrt(1 - this.es);
                H = this.e * this.cos_p12 * Math.cos(Az) / Math.sqrt(1 - this.es);
                GH = G * H;
                Hs = H * H;
                s2 = s * s;
                s3 = s2 * s;
                s4 = s3 * s;
                s5 = s4 * s;
                c = Nl1 * s * (1 - s2 * Hs * (1 - Hs) / 6 + s3 / 8 * GH * (1 - 2 * Hs) + s4 / 120 * (Hs * (4 - 7 * Hs) - 3 * G * G * (1 - 7 * Hs)) - s5 / 48 * GH);
                p.x = this.x0 + c * Math.sin(Az);
                p.y = this.y0 + c * Math.cos(Az);
                return p;
            }
        }


    }

    function inverse$24(p) {
        p.x -= this.x0;
        p.y -= this.y0;
        var rh, z, sinz, cosz, lon, lat, con, e0, e1, e2, e3, Mlp, M, N1, psi, Az, cosAz, tmp, A, B, D, Ee, F;
        if (this.sphere) {
            rh = Math.sqrt(p.x * p.x + p.y * p.y);
            if (rh > (2 * HALF_PI * this.a)) {
                return;
            }
            z = rh / this.a;

            sinz = Math.sin(z);
            cosz = Math.cos(z);

            lon = this.long0;
            if (Math.abs(rh) <= EPSLN) {
                lat = this.lat0;
            }
            else {
                lat = asinz(cosz * this.sin_p12 + (p.y * sinz * this.cos_p12) / rh);
                con = Math.abs(this.lat0) - HALF_PI;
                if (Math.abs(con) <= EPSLN) {
                    if (this.lat0 >= 0) {
                        lon = adjust_lon(this.long0 + Math.atan2(p.x, - p.y));
                    }
                    else {
                        lon = adjust_lon(this.long0 - Math.atan2(-p.x, p.y));
                    }
                }
                else {
                    /*con = cosz - this.sin_p12 * Math.sin(lat);
            if ((Math.abs(con) < EPSLN) && (Math.abs(p.x) < EPSLN)) {
              //no-op, just keep the lon value as is
            } else {
              var temp = Math.atan2((p.x * sinz * this.cos_p12), (con * rh));
              lon = adjust_lon(this.long0 + Math.atan2((p.x * sinz * this.cos_p12), (con * rh)));
            }*/
                    lon = adjust_lon(this.long0 + Math.atan2(p.x * sinz, rh * this.cos_p12 * cosz - p.y * this.sin_p12 * sinz));
                }
            }

            p.x = lon;
            p.y = lat;
            return p;
        }
        else {
            e0 = e0fn(this.es);
            e1 = e1fn(this.es);
            e2 = e2fn(this.es);
            e3 = e3fn(this.es);
            if (Math.abs(this.sin_p12 - 1) <= EPSLN) {
                //North pole case
                Mlp = this.a * mlfn(e0, e1, e2, e3, HALF_PI);
                rh = Math.sqrt(p.x * p.x + p.y * p.y);
                M = Mlp - rh;
                lat = imlfn(M / this.a, e0, e1, e2, e3);
                lon = adjust_lon(this.long0 + Math.atan2(p.x, - 1 * p.y));
                p.x = lon;
                p.y = lat;
                return p;
            }
            else if (Math.abs(this.sin_p12 + 1) <= EPSLN) {
                //South pole case
                Mlp = this.a * mlfn(e0, e1, e2, e3, HALF_PI);
                rh = Math.sqrt(p.x * p.x + p.y * p.y);
                M = rh - Mlp;

                lat = imlfn(M / this.a, e0, e1, e2, e3);
                lon = adjust_lon(this.long0 + Math.atan2(p.x, p.y));
                p.x = lon;
                p.y = lat;
                return p;
            }
            else {
                //default case
                rh = Math.sqrt(p.x * p.x + p.y * p.y);
                Az = Math.atan2(p.x, p.y);
                N1 = gN(this.a, this.e, this.sin_p12);
                cosAz = Math.cos(Az);
                tmp = this.e * this.cos_p12 * cosAz;
                A = -tmp * tmp / (1 - this.es);
                B = 3 * this.es * (1 - A) * this.sin_p12 * this.cos_p12 * cosAz / (1 - this.es);
                D = rh / N1;
                Ee = D - A * (1 + A) * Math.pow(D, 3) / 6 - B * (1 + 3 * A) * Math.pow(D, 4) / 24;
                F = 1 - A * Ee * Ee / 2 - D * Ee * Ee * Ee / 6;
                psi = Math.asin(this.sin_p12 * Math.cos(Ee) + this.cos_p12 * Math.sin(Ee) * cosAz);
                lon = adjust_lon(this.long0 + Math.asin(Math.sin(Az) * Math.sin(Ee) / Math.cos(psi)));
                lat = Math.atan((1 - this.es * F * this.sin_p12 / Math.sin(psi)) * Math.tan(psi) / (1 - this.es));
                p.x = lon;
                p.y = lat;
                return p;
            }
        }

    }

    var names$26 = ["Azimuthal_Equidistant", "aeqd"];
    var aeqd = {
        init: init$25,
        forward: forward$24,
        inverse: inverse$24,
        names: names$26
    };

    function init$26() {
        //double temp;      /* temporary variable    */

        /* Place parameters in static storage for common use
          -------------------------------------------------*/
        this.sin_p14 = Math.sin(this.lat0);
        this.cos_p14 = Math.cos(this.lat0);
    }

    /* Orthographic forward equations--mapping lat,long to x,y
        ---------------------------------------------------*/
    function forward$25(p) {
        var sinphi, cosphi; /* sin and cos value        */
        var dlon; /* delta longitude value      */
        var coslon; /* cos of longitude        */
        var ksp; /* scale factor          */
        var g, x, y;
        var lon = p.x;
        var lat = p.y;
        /* Forward equations
          -----------------*/
        dlon = adjust_lon(lon - this.long0);

        sinphi = Math.sin(lat);
        cosphi = Math.cos(lat);

        coslon = Math.cos(dlon);
        g = this.sin_p14 * sinphi + this.cos_p14 * cosphi * coslon;
        ksp = 1;
        if ((g > 0) || (Math.abs(g) <= EPSLN)) {
            x = this.a * ksp * cosphi * Math.sin(dlon);
            y = this.y0 + this.a * ksp * (this.cos_p14 * sinphi - this.sin_p14 * cosphi * coslon);
        }
        p.x = x;
        p.y = y;
        return p;
    }

    function inverse$25(p) {
        var rh; /* height above ellipsoid      */
        var z; /* angle          */
        var sinz, cosz; /* sin of z and cos of z      */
        var con;
        var lon, lat;
        /* Inverse equations
          -----------------*/
        p.x -= this.x0;
        p.y -= this.y0;
        rh = Math.sqrt(p.x * p.x + p.y * p.y);
        z = asinz(rh / this.a);

        sinz = Math.sin(z);
        cosz = Math.cos(z);

        lon = this.long0;
        if (Math.abs(rh) <= EPSLN) {
            lat = this.lat0;
            p.x = lon;
            p.y = lat;
            return p;
        }
        lat = asinz(cosz * this.sin_p14 + (p.y * sinz * this.cos_p14) / rh);
        con = Math.abs(this.lat0) - HALF_PI;
        if (Math.abs(con) <= EPSLN) {
            if (this.lat0 >= 0) {
                lon = adjust_lon(this.long0 + Math.atan2(p.x, - p.y));
            }
            else {
                lon = adjust_lon(this.long0 - Math.atan2(-p.x, p.y));
            }
            p.x = lon;
            p.y = lat;
            return p;
        }
        lon = adjust_lon(this.long0 + Math.atan2((p.x * sinz), rh * this.cos_p14 * cosz - p.y * this.sin_p14 * sinz));
        p.x = lon;
        p.y = lat;
        return p;
    }

    var names$27 = ["ortho"];
    var ortho = {
        init: init$26,
        forward: forward$25,
        inverse: inverse$25,
        names: names$27
    };

    // QSC projection rewritten from the original PROJ4
    // https://github.com/OSGeo/proj.4/blob/master/src/PJ_qsc.c

    /* constants */
    var FACE_ENUM = {
        FRONT: 1,
        RIGHT: 2,
        BACK: 3,
        LEFT: 4,
        TOP: 5,
        BOTTOM: 6
    };

    var AREA_ENUM = {
        AREA_0: 1,
        AREA_1: 2,
        AREA_2: 3,
        AREA_3: 4
    };

    function init$27() {

        this.x0 = this.x0 || 0;
        this.y0 = this.y0 || 0;
        this.lat0 = this.lat0 || 0;
        this.long0 = this.long0 || 0;
        this.lat_ts = this.lat_ts || 0;
        this.title = this.title || "Quadrilateralized Spherical Cube";

        /* Determine the cube face from the center of projection. */
        if (this.lat0 >= HALF_PI - FORTPI / 2.0) {
            this.face = FACE_ENUM.TOP;
        } else if (this.lat0 <= -(HALF_PI - FORTPI / 2.0)) {
            this.face = FACE_ENUM.BOTTOM;
        } else if (Math.abs(this.long0) <= FORTPI) {
            this.face = FACE_ENUM.FRONT;
        } else if (Math.abs(this.long0) <= HALF_PI + FORTPI) {
            this.face = this.long0 > 0.0 ? FACE_ENUM.RIGHT : FACE_ENUM.LEFT;
        } else {
            this.face = FACE_ENUM.BACK;
        }

        /* Fill in useful values for the ellipsoid <-> sphere shift
       * described in [LK12]. */
        if (this.es !== 0) {
            this.one_minus_f = 1 - (this.a - this.b) / this.a;
            this.one_minus_f_squared = this.one_minus_f * this.one_minus_f;
        }
    }

    // QSC forward equations--mapping lat,long to x,y
    // -----------------------------------------------------------------
    function forward$26(p) {
        var xy = {x: 0, y: 0};
        var lat, lon;
        var theta, phi;
        var t, mu;
        /* nu; */
        var area = {value: 0};

        // move lon according to projection's lon
        p.x -= this.long0;

        /* Convert the geodetic latitude to a geocentric latitude.
       * This corresponds to the shift from the ellipsoid to the sphere
       * described in [LK12]. */
        if (this.es !== 0) {//if (P->es != 0) {
            lat = Math.atan(this.one_minus_f_squared * Math.tan(p.y));
        } else {
            lat = p.y;
        }

        /* Convert the input lat, lon into theta, phi as used by QSC.
       * This depends on the cube face and the area on it.
       * For the top and bottom face, we can compute theta and phi
       * directly from phi, lam. For the other faces, we must use
       * unit sphere cartesian coordinates as an intermediate step. */
        lon = p.x; //lon = lp.lam;
        if (this.face === FACE_ENUM.TOP) {
            phi = HALF_PI - lat;
            if (lon >= FORTPI && lon <= HALF_PI + FORTPI) {
                area.value = AREA_ENUM.AREA_0;
                theta = lon - HALF_PI;
            } else if (lon > HALF_PI + FORTPI || lon <= -(HALF_PI + FORTPI)) {
                area.value = AREA_ENUM.AREA_1;
                theta = (lon > 0.0 ? lon - SPI : lon + SPI);
            } else if (lon > -(HALF_PI + FORTPI) && lon <= -FORTPI) {
                area.value = AREA_ENUM.AREA_2;
                theta = lon + HALF_PI;
            } else {
                area.value = AREA_ENUM.AREA_3;
                theta = lon;
            }
        } else if (this.face === FACE_ENUM.BOTTOM) {
            phi = HALF_PI + lat;
            if (lon >= FORTPI && lon <= HALF_PI + FORTPI) {
                area.value = AREA_ENUM.AREA_0;
                theta = -lon + HALF_PI;
            } else if (lon < FORTPI && lon >= -FORTPI) {
                area.value = AREA_ENUM.AREA_1;
                theta = -lon;
            } else if (lon < -FORTPI && lon >= -(HALF_PI + FORTPI)) {
                area.value = AREA_ENUM.AREA_2;
                theta = -lon - HALF_PI;
            } else {
                area.value = AREA_ENUM.AREA_3;
                theta = (lon > 0.0 ? -lon + SPI : -lon - SPI);
            }
        } else {
            var q, r, s;
            var sinlat, coslat;
            var sinlon, coslon;

            if (this.face === FACE_ENUM.RIGHT) {
                lon = qsc_shift_lon_origin(lon, +HALF_PI);
            } else if (this.face === FACE_ENUM.BACK) {
                lon = qsc_shift_lon_origin(lon, +SPI);
            } else if (this.face === FACE_ENUM.LEFT) {
                lon = qsc_shift_lon_origin(lon, -HALF_PI);
            }
            sinlat = Math.sin(lat);
            coslat = Math.cos(lat);
            sinlon = Math.sin(lon);
            coslon = Math.cos(lon);
            q = coslat * coslon;
            r = coslat * sinlon;
            s = sinlat;

            if (this.face === FACE_ENUM.FRONT) {
                phi = Math.acos(q);
                theta = qsc_fwd_equat_face_theta(phi, s, r, area);
            } else if (this.face === FACE_ENUM.RIGHT) {
                phi = Math.acos(r);
                theta = qsc_fwd_equat_face_theta(phi, s, -q, area);
            } else if (this.face === FACE_ENUM.BACK) {
                phi = Math.acos(-q);
                theta = qsc_fwd_equat_face_theta(phi, s, -r, area);
            } else if (this.face === FACE_ENUM.LEFT) {
                phi = Math.acos(-r);
                theta = qsc_fwd_equat_face_theta(phi, s, q, area);
            } else {
                /* Impossible */
                phi = theta = 0;
                area.value = AREA_ENUM.AREA_0;
            }
        }

        /* Compute mu and nu for the area of definition.
       * For mu, see Eq. (3-21) in [OL76], but note the typos:
       * compare with Eq. (3-14). For nu, see Eq. (3-38). */
        mu = Math.atan((12 / SPI) * (theta + Math.acos(Math.sin(theta) * Math.cos(FORTPI)) - HALF_PI));
        t = Math.sqrt((1 - Math.cos(phi)) / (Math.cos(mu) * Math.cos(mu)) / (1 - Math.cos(Math.atan(1 / Math.cos(theta)))));

        /* Apply the result to the real area. */
        if (area.value === AREA_ENUM.AREA_1) {
            mu += HALF_PI;
        } else if (area.value === AREA_ENUM.AREA_2) {
            mu += SPI;
        } else if (area.value === AREA_ENUM.AREA_3) {
            mu += 1.5 * SPI;
        }

        /* Now compute x, y from mu and nu */
        xy.x = t * Math.cos(mu);
        xy.y = t * Math.sin(mu);
        xy.x = xy.x * this.a + this.x0;
        xy.y = xy.y * this.a + this.y0;

        p.x = xy.x;
        p.y = xy.y;
        return p;
    }

    // QSC inverse equations--mapping x,y to lat/long
    // -----------------------------------------------------------------
    function inverse$26(p) {
        var lp = {lam: 0, phi: 0};
        var mu, nu, cosmu, tannu;
        var tantheta, theta, cosphi, phi;
        var t;
        var area = {value: 0};

        /* de-offset */
        p.x = (p.x - this.x0) / this.a;
        p.y = (p.y - this.y0) / this.a;

        /* Convert the input x, y to the mu and nu angles as used by QSC.
       * This depends on the area of the cube face. */
        nu = Math.atan(Math.sqrt(p.x * p.x + p.y * p.y));
        mu = Math.atan2(p.y, p.x);
        if (p.x >= 0.0 && p.x >= Math.abs(p.y)) {
            area.value = AREA_ENUM.AREA_0;
        } else if (p.y >= 0.0 && p.y >= Math.abs(p.x)) {
            area.value = AREA_ENUM.AREA_1;
            mu -= HALF_PI;
        } else if (p.x < 0.0 && -p.x >= Math.abs(p.y)) {
            area.value = AREA_ENUM.AREA_2;
            mu = (mu < 0.0 ? mu + SPI : mu - SPI);
        } else {
            area.value = AREA_ENUM.AREA_3;
            mu += HALF_PI;
        }

        /* Compute phi and theta for the area of definition.
       * The inverse projection is not described in the original paper, but some
       * good hints can be found here (as of 2011-12-14):
       * http://fits.gsfc.nasa.gov/fitsbits/saf.93/saf.9302
       * (search for "Message-Id: <9302181759.AA25477 at fits.cv.nrao.edu>") */
        t = (SPI / 12) * Math.tan(mu);
        tantheta = Math.sin(t) / (Math.cos(t) - (1 / Math.sqrt(2)));
        theta = Math.atan(tantheta);
        cosmu = Math.cos(mu);
        tannu = Math.tan(nu);
        cosphi = 1 - cosmu * cosmu * tannu * tannu * (1 - Math.cos(Math.atan(1 / Math.cos(theta))));
        if (cosphi < -1) {
            cosphi = -1;
        } else if (cosphi > +1) {
            cosphi = +1;
        }

        /* Apply the result to the real area on the cube face.
       * For the top and bottom face, we can compute phi and lam directly.
       * For the other faces, we must use unit sphere cartesian coordinates
       * as an intermediate step. */
        if (this.face === FACE_ENUM.TOP) {
            phi = Math.acos(cosphi);
            lp.phi = HALF_PI - phi;
            if (area.value === AREA_ENUM.AREA_0) {
                lp.lam = theta + HALF_PI;
            } else if (area.value === AREA_ENUM.AREA_1) {
                lp.lam = (theta < 0.0 ? theta + SPI : theta - SPI);
            } else if (area.value === AREA_ENUM.AREA_2) {
                lp.lam = theta - HALF_PI;
            } else /* area.value == AREA_ENUM.AREA_3 */ {
                lp.lam = theta;
            }
        } else if (this.face === FACE_ENUM.BOTTOM) {
            phi = Math.acos(cosphi);
            lp.phi = phi - HALF_PI;
            if (area.value === AREA_ENUM.AREA_0) {
                lp.lam = -theta + HALF_PI;
            } else if (area.value === AREA_ENUM.AREA_1) {
                lp.lam = -theta;
            } else if (area.value === AREA_ENUM.AREA_2) {
                lp.lam = -theta - HALF_PI;
            } else /* area.value == AREA_ENUM.AREA_3 */ {
                lp.lam = (theta < 0.0 ? -theta - SPI : -theta + SPI);
            }
        } else {
            /* Compute phi and lam via cartesian unit sphere coordinates. */
            var q, r, s;
            q = cosphi;
            t = q * q;
            if (t >= 1) {
                s = 0;
            } else {
                s = Math.sqrt(1 - t) * Math.sin(theta);
            }
            t += s * s;
            if (t >= 1) {
                r = 0;
            } else {
                r = Math.sqrt(1 - t);
            }
            /* Rotate q,r,s into the correct area. */
            if (area.value === AREA_ENUM.AREA_1) {
                t = r;
                r = -s;
                s = t;
            } else if (area.value === AREA_ENUM.AREA_2) {
                r = -r;
                s = -s;
            } else if (area.value === AREA_ENUM.AREA_3) {
                t = r;
                r = s;
                s = -t;
            }
            /* Rotate q,r,s into the correct cube face. */
            if (this.face === FACE_ENUM.RIGHT) {
                t = q;
                q = -r;
                r = t;
            } else if (this.face === FACE_ENUM.BACK) {
                q = -q;
                r = -r;
            } else if (this.face === FACE_ENUM.LEFT) {
                t = q;
                q = r;
                r = -t;
            }
            /* Now compute phi and lam from the unit sphere coordinates. */
            lp.phi = Math.acos(-s) - HALF_PI;
            lp.lam = Math.atan2(r, q);
            if (this.face === FACE_ENUM.RIGHT) {
                lp.lam = qsc_shift_lon_origin(lp.lam, -HALF_PI);
            } else if (this.face === FACE_ENUM.BACK) {
                lp.lam = qsc_shift_lon_origin(lp.lam, -SPI);
            } else if (this.face === FACE_ENUM.LEFT) {
                lp.lam = qsc_shift_lon_origin(lp.lam, +HALF_PI);
            }
        }

        /* Apply the shift from the sphere to the ellipsoid as described
       * in [LK12]. */
        if (this.es !== 0) {
            var invert_sign;
            var tanphi, xa;
            invert_sign = (lp.phi < 0 ? 1 : 0);
            tanphi = Math.tan(lp.phi);
            xa = this.b / Math.sqrt(tanphi * tanphi + this.one_minus_f_squared);
            lp.phi = Math.atan(Math.sqrt(this.a * this.a - xa * xa) / (this.one_minus_f * xa));
            if (invert_sign) {
                lp.phi = -lp.phi;
            }
        }

        lp.lam += this.long0;
        p.x = lp.lam;
        p.y = lp.phi;
        return p;
    }

    /* Helper function for forward projection: compute the theta angle
     * and determine the area number. */
    function qsc_fwd_equat_face_theta(phi, y, x, area) {
        var theta;
        if (phi < EPSLN) {
            area.value = AREA_ENUM.AREA_0;
            theta = 0.0;
        } else {
            theta = Math.atan2(y, x);
            if (Math.abs(theta) <= FORTPI) {
                area.value = AREA_ENUM.AREA_0;
            } else if (theta > FORTPI && theta <= HALF_PI + FORTPI) {
                area.value = AREA_ENUM.AREA_1;
                theta -= HALF_PI;
            } else if (theta > HALF_PI + FORTPI || theta <= -(HALF_PI + FORTPI)) {
                area.value = AREA_ENUM.AREA_2;
                theta = (theta >= 0.0 ? theta - SPI : theta + SPI);
            } else {
                area.value = AREA_ENUM.AREA_3;
                theta += HALF_PI;
            }
        }
        return theta;
    }

    /* Helper function: shift the longitude. */
    function qsc_shift_lon_origin(lon, offset) {
        var slon = lon + offset;
        if (slon < -SPI) {
            slon += TWO_PI;
        } else if (slon > +SPI) {
            slon -= TWO_PI;
        }
        return slon;
    }

    var names$28 = ["Quadrilateralized Spherical Cube", "Quadrilateralized_Spherical_Cube", "qsc"];
    var qsc = {
        init: init$27,
        forward: forward$26,
        inverse: inverse$26,
        names: names$28
    };

    // Robinson projection
    // Based on https://github.com/OSGeo/proj.4/blob/master/src/PJ_robin.c
    // Polynomial coeficients from http://article.gmane.org/gmane.comp.gis.proj-4.devel/6039

    var COEFS_X = [
        [1.0000, 2.2199e-17, -7.15515e-05, 3.1103e-06],
        [0.9986, -0.000482243, -2.4897e-05, -1.3309e-06],
        [0.9954, -0.00083103, -4.48605e-05, -9.86701e-07],
        [0.9900, -0.00135364, -5.9661e-05, 3.6777e-06],
        [0.9822, -0.00167442, -4.49547e-06, -5.72411e-06],
        [0.9730, -0.00214868, -9.03571e-05, 1.8736e-08],
        [0.9600, -0.00305085, -9.00761e-05, 1.64917e-06],
        [0.9427, -0.00382792, -6.53386e-05, -2.6154e-06],
        [0.9216, -0.00467746, -0.00010457, 4.81243e-06],
        [0.8962, -0.00536223, -3.23831e-05, -5.43432e-06],
        [0.8679, -0.00609363, -0.000113898, 3.32484e-06],
        [0.8350, -0.00698325, -6.40253e-05, 9.34959e-07],
        [0.7986, -0.00755338, -5.00009e-05, 9.35324e-07],
        [0.7597, -0.00798324, -3.5971e-05, -2.27626e-06],
        [0.7186, -0.00851367, -7.01149e-05, -8.6303e-06],
        [0.6732, -0.00986209, -0.000199569, 1.91974e-05],
        [0.6213, -0.010418, 8.83923e-05, 6.24051e-06],
        [0.5722, -0.00906601, 0.000182, 6.24051e-06],
        [0.5322, -0.00677797, 0.000275608, 6.24051e-06]
    ];

    var COEFS_Y = [
        [-5.20417e-18, 0.0124, 1.21431e-18, -8.45284e-11],
        [0.0620, 0.0124, -1.26793e-09, 4.22642e-10],
        [0.1240, 0.0124, 5.07171e-09, -1.60604e-09],
        [0.1860, 0.0123999, -1.90189e-08, 6.00152e-09],
        [0.2480, 0.0124002, 7.10039e-08, -2.24e-08],
        [0.3100, 0.0123992, -2.64997e-07, 8.35986e-08],
        [0.3720, 0.0124029, 9.88983e-07, -3.11994e-07],
        [0.4340, 0.0123893, -3.69093e-06, -4.35621e-07],
        [0.4958, 0.0123198, -1.02252e-05, -3.45523e-07],
        [0.5571, 0.0121916, -1.54081e-05, -5.82288e-07],
        [0.6176, 0.0119938, -2.41424e-05, -5.25327e-07],
        [0.6769, 0.011713, -3.20223e-05, -5.16405e-07],
        [0.7346, 0.0113541, -3.97684e-05, -6.09052e-07],
        [0.7903, 0.0109107, -4.89042e-05, -1.04739e-06],
        [0.8435, 0.0103431, -6.4615e-05, -1.40374e-09],
        [0.8936, 0.00969686, -6.4636e-05, -8.547e-06],
        [0.9394, 0.00840947, -0.000192841, -4.2106e-06],
        [0.9761, 0.00616527, -0.000256, -4.2106e-06],
        [1.0000, 0.00328947, -0.000319159, -4.2106e-06]
    ];

    var FXC = 0.8487;
    var FYC = 1.3523;
    var C1 = R2D/5; // rad to 5-degree interval
    var RC1 = 1/C1;
    var NODES = 18;

    var poly3_val = function(coefs, x) {
        return coefs[0] + x * (coefs[1] + x * (coefs[2] + x * coefs[3]));
    };

    var poly3_der = function(coefs, x) {
        return coefs[1] + x * (2 * coefs[2] + x * 3 * coefs[3]);
    };

    function newton_rapshon(f_df, start, max_err, iters) {
        var x = start;
        for (; iters; --iters) {
            var upd = f_df(x);
            x -= upd;
            if (Math.abs(upd) < max_err) {
                break;
            }
        }
        return x;
    }

    function init$28() {
        this.x0 = this.x0 || 0;
        this.y0 = this.y0 || 0;
        this.long0 = this.long0 || 0;
        this.es = 0;
        this.title = this.title || "Robinson";
    }

    function forward$27(ll) {
        var lon = adjust_lon(ll.x - this.long0);

        var dphi = Math.abs(ll.y);
        var i = Math.floor(dphi * C1);
        if (i < 0) {
            i = 0;
        } else if (i >= NODES) {
            i = NODES - 1;
        }
        dphi = R2D * (dphi - RC1 * i);
        var xy = {
            x: poly3_val(COEFS_X[i], dphi) * lon,
            y: poly3_val(COEFS_Y[i], dphi)
        };
        if (ll.y < 0) {
            xy.y = -xy.y;
        }

        xy.x = xy.x * this.a * FXC + this.x0;
        xy.y = xy.y * this.a * FYC + this.y0;
        return xy;
    }

    function inverse$27(xy) {
        var ll = {
            x: (xy.x - this.x0) / (this.a * FXC),
            y: Math.abs(xy.y - this.y0) / (this.a * FYC)
        };

        if (ll.y >= 1) { // pathologic case
            ll.x /= COEFS_X[NODES][0];
            ll.y = xy.y < 0 ? -HALF_PI : HALF_PI;
        } else {
            // find table interval
            var i = Math.floor(ll.y * NODES);
            if (i < 0) {
                i = 0;
            } else if (i >= NODES) {
                i = NODES - 1;
            }
            for (;;) {
                if (COEFS_Y[i][0] > ll.y) {
                    --i;
                } else if (COEFS_Y[i+1][0] <= ll.y) {
                    ++i;
                } else {
                    break;
                }
            }
            // linear interpolation in 5 degree interval
            var coefs = COEFS_Y[i];
            var t = 5 * (ll.y - coefs[0]) / (COEFS_Y[i+1][0] - coefs[0]);
            // find t so that poly3_val(coefs, t) = ll.y
            t = newton_rapshon(function(x) {
                return (poly3_val(coefs, x) - ll.y) / poly3_der(coefs, x);
            }, t, EPSLN, 100);

            ll.x /= poly3_val(COEFS_X[i], t);
            ll.y = (5 * i + t) * D2R;
            if (xy.y < 0) {
                ll.y = -ll.y;
            }
        }

        ll.x = adjust_lon(ll.x + this.long0);
        return ll;
    }

    var names$29 = ["Robinson", "robin"];
    var robin = {
        init: init$28,
        forward: forward$27,
        inverse: inverse$27,
        names: names$29
    };

    function init$29() {
        this.name = 'geocent';

    }

    function forward$28(p) {
        var point = geodeticToGeocentric(p, this.es, this.a);
        return point;
    }

    function inverse$28(p) {
        var point = geocentricToGeodetic(p, this.es, this.a, this.b);
        return point;
    }

    var names$30 = ["Geocentric", 'geocentric', "geocent", "Geocent"];
    var geocent = {
        init: init$29,
        forward: forward$28,
        inverse: inverse$28,
        names: names$30
    };

    var includedProjections = function(proj4){
        proj4.Proj.projections.add(tmerc);
        proj4.Proj.projections.add(etmerc);
        proj4.Proj.projections.add(utm);
        proj4.Proj.projections.add(sterea);
        proj4.Proj.projections.add(stere);
        proj4.Proj.projections.add(somerc);
        proj4.Proj.projections.add(omerc);
        proj4.Proj.projections.add(lcc);
        proj4.Proj.projections.add(krovak);
        proj4.Proj.projections.add(cass);
        proj4.Proj.projections.add(laea);
        proj4.Proj.projections.add(aea);
        proj4.Proj.projections.add(gnom);
        proj4.Proj.projections.add(cea);
        proj4.Proj.projections.add(eqc);
        proj4.Proj.projections.add(poly);
        proj4.Proj.projections.add(nzmg);
        proj4.Proj.projections.add(mill);
        proj4.Proj.projections.add(sinu);
        proj4.Proj.projections.add(moll);
        proj4.Proj.projections.add(eqdc);
        proj4.Proj.projections.add(vandg);
        proj4.Proj.projections.add(aeqd);
        proj4.Proj.projections.add(ortho);
        proj4.Proj.projections.add(qsc);
        proj4.Proj.projections.add(robin);
        proj4.Proj.projections.add(geocent);
    };

    proj4$1.defaultDatum = 'WGS84'; //default datum
    proj4$1.Proj = Projection;
    proj4$1.WGS84 = new proj4$1.Proj('WGS84');
    proj4$1.Point = Point;
    proj4$1.toPoint = toPoint;
    proj4$1.defs = defs;
    proj4$1.transform = transform;
    proj4$1.mgrs = mgrs;
    proj4$1.version = '2.6.0';
    includedProjections(proj4$1);

    return proj4$1;

});

/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports TiffIFDEntry
 */
define('formats/geotiff/TiffIFDEntry',[
        '../../error/AbstractError',
        '../../error/ArgumentError',
        './GeoTiffUtil',
        '../../util/Logger',
        './TiffConstants'
    ],
    function (AbstractError,
              ArgumentError,
              GeoTiffUtil,
              Logger,
              TiffConstants) {
        "use strict";

        /**
         * Constructs an image file directory entry. Applications typically do not call this constructor. It is called
         * by {@link GeoTiffReader} as GeoTIFF image file directories are read.
         * @alias TiffIFDEntry
         * @constructor
         * @classdesc Contains the data associated with a GeoTIFF image file directory. An image file directory
         * contains information about the image, as well as pointers to the actual image data.
         * @param {Number} tag The TIFF tag that identifies the field.
         * @param {Number} type The type of the field.
         * @param {Number} count The number of values, count of the indicated type.
         * @param {Number} valueOffset  The file offset (in bytes) of the Value for the field. This file offset may
         * point anywhere in the file, even after the image data.
         * @param {ArrayBuffer} geoTiffData The buffer descriptor of the geotiff file's content.
         * @param {Boolean} isLittleEndian Indicates whether the geotiff byte order is little endian.
         * @throws {ArgumentError} If either the specified tag, type, count, valueOffset, geoTiffData or isLittleEndian
         * are null or undefined.
         */
        var TiffIFDEntry = function (tag, type, count, valueOffset, geoTiffData, isLittleEndian) {
            if (!tag) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "TiffIFDEntry", "constructor", "missingTag"));
            }

            if (!type) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "TiffIFDEntry", "constructor", "missingType"));
            }

            if (!count) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "TiffIFDEntry", "constructor", "missingCount"));
            }

            if (valueOffset === null || valueOffset === undefined) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "TiffIFDEntry", "constructor", "missingValueOffset"));
            }

            if (!geoTiffData) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "TiffIFDEntry", "constructor", "missingGeoTiffData"));
            }

            if (isLittleEndian === null || isLittleEndian === undefined) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "TiffIFDEntry", "constructor", "missingIsLittleEndian"));
            }

            // Documented in defineProperties below.
            this._tag = tag;

            // Documented in defineProperties below.
            this._type = type;

            // Documented in defineProperties below.
            this._count = count;

            // Documented in defineProperties below.
            this._valueOffset = valueOffset;

            // Documented in defineProperties below.
            this._geoTiffData = geoTiffData;

            // Documented in defineProperties below.
            this._isLittleEndian = isLittleEndian;
        };

        Object.defineProperties(TiffIFDEntry.prototype, {

            /**
             * The tag that identifies the field as specified to this TiffIFDEntry's constructor.
             * @memberof TiffIFDEntry.prototype
             * @type {Number}
             * @readonly
             */
            tag: {
                get: function () {
                    return this._tag;
                }
            },

            /**
             * The field type as specified to this TiffIFDEntry's constructor.
             * @memberof TiffIFDEntry.prototype
             * @type {Number}
             * @readonly
             */
            type: {
                get: function () {
                    return this._type;
                }
            },

            /**
             * The number of the values as specified to this TiffIFDEntry's constructor.
             * @memberof TiffIFDEntry.prototype
             * @type {Number}
             * @readonly
             */
            count: {
                get: function () {
                    return this._count;
                }
            },

            /**
             * The file offset as specified to this TiffIFDEntry's constructor.
             * @memberof TiffIFDEntry.prototype
             * @type {Number}
             * @readonly
             */
            valueOffset: {
                get: function () {
                    return this._valueOffset;
                }
            },

            /**
             * The geotiff buffer data as specified to this TiffIFDEntry's constructor.
             * @memberof TiffIFDEntry.prototype
             * @type {ArrayBuffer}
             * @readonly
             */
            geoTiffData: {
                get: function () {
                    return this._geoTiffData;
                }
            },

            /**
             * The little endian byte order flag as specified to this TiffIFDEntry's constructor.
             * @memberof TiffIFDEntry.prototype
             * @type {Boolean}
             * @readonly
             */
            isLittleEndian: {
                get: function () {
                    return this._isLittleEndian;
                }
            }
        });

        /**
         * Get the number of bytes of an image file directory depending on its type.
         * @returns {Number}
         */
        TiffIFDEntry.prototype.getIFDTypeLength = function () {
            switch(this.type){
                case TiffConstants.Type.BYTE:
                case TiffConstants.Type.ASCII:
                case TiffConstants.Type.SBYTE:
                case TiffConstants.Type.UNDEFINED:
                    return 1;
                case TiffConstants.Type.SHORT:
                case TiffConstants.Type.SSHORT:
                    return 2;
                case TiffConstants.Type.LONG:
                case TiffConstants.Type.SLONG:
                case TiffConstants.Type.FLOAT:
                    return 4;
                case TiffConstants.Type.RATIONAL:
                case TiffConstants.Type.SRATIONAL:
                case TiffConstants.Type.DOUBLE:
                    return 8;
                default:
                    return -1;
            }
        }

        /**
         * Get the value of an image file directory.
         * @returns {Number[]}
         */
        TiffIFDEntry.prototype.getIFDEntryValue = function () {
            var ifdValues = [];
            var value = null;
            var ifdTypeLength = this.getIFDTypeLength();
            var ifdValueSize = ifdTypeLength * this.count;

            if (ifdValueSize <= 4) {
                if (this.isLittleEndian === false) {
                    value = this.valueOffset >>> ((4 - ifdTypeLength) * 8);
                } else {
                    value = this.valueOffset;
                }
                ifdValues.push(value);
            } else {
                for (var i = 0; i < this.count; i++) {
                    var indexOffset = ifdTypeLength * i;

                    if (ifdTypeLength >= 8) {
                        if (this.type === TiffConstants.Type.RATIONAL || this.type === TiffConstants.Type.SRATIONAL) {
                            // Numerator
                            ifdValues.push(GeoTiffUtil.getBytes(this.geoTiffData, this.valueOffset + indexOffset, 4,
                                this.isLittleEndian));
                            // Denominator
                            ifdValues.push(GeoTiffUtil.getBytes(this.geoTiffData, this.valueOffset + indexOffset + 4, 4,
                                this.isLittleEndian));
                        } else if (this.type === TiffConstants.Type.DOUBLE) {
                            ifdValues.push(GeoTiffUtil.getBytes(this.geoTiffData, this.valueOffset + indexOffset, 8,
                                this.isLittleEndian));
                        } else {
                            throw new AbstractError(
                                Logger.logMessage(Logger.LEVEL_SEVERE, "TiffIFDEntry", "parse", "invalidTypeOfIFD"));
                        }
                    } else {
                        ifdValues.push(GeoTiffUtil.getBytes(this.geoTiffData, this.valueOffset + indexOffset,
                            ifdTypeLength, this.isLittleEndian));
                    }
                }
            }

            if (this.type === TiffConstants.Type.ASCII) {
                ifdValues.forEach(function (element, index, array) {
                    if (element === 0){
                        array.splice(index, 1);
                    }
                    else{
                        array[index] = String.fromCharCode(element);
                    }
                });

                return ifdValues.join("");
            }

            return ifdValues;
        };

        return TiffIFDEntry;
    }
);
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
define('util/WWUtil',[
        '../error/ArgumentError',
        '../geom/Line',
        '../util/Logger',
        '../geom/Rectangle',
        '../geom/Vec3'],
    function (ArgumentError,
              Line,
              Logger,
              Rectangle,
              Vec3) {
        "use strict";
        /**
         * Provides math constants and functions.
         * @exports WWUtil
         */
        var WWUtil = {
            // A regular expression that matches latitude followed by a comma and possible white space followed by
            // longitude. Latitude and longitude ranges are not considered.
            latLonRegex: /^(\-?\d+(\.\d+)?),\s*(\-?\d+(\.\d+)?)$/,

            /**
             * Returns the suffix for a specified mime type.
             * @param {String} mimeType The mime type to determine a suffix for.
             * @returns {String} The suffix for the specified mime type, or null if the mime type is not recognized.
             */
            suffixForMimeType: function (mimeType) {
                if (mimeType === "image/png")
                    return "png";

                if (mimeType === "image/jpeg")
                    return "jpg";

                if (mimeType === "application/bil16")
                    return "bil";

                if (mimeType === "application/bil32")
                    return "bil";

                return null;
            },

            /**
             * Returns the current location URL as obtained from window.location with the last path component
             * removed.
             * @returns {String} The current location URL with the last path component removed.
             */
            currentUrlSansFilePart: function () {
                var protocol = window.location.protocol,
                    host = window.location.host,
                    path = window.location.pathname,
                    pathParts = path.split("/"),
                    newPath = "";

                for (var i = 0, len = pathParts.length; i < len - 1; i++) {
                    if (pathParts[i].length > 0) {
                        newPath = newPath + "/" + pathParts[i];
                    }
                }

                return protocol + "//" + host + newPath;
            },

            /**
             * Returns the URL of the directory containing the WorldWind library.
             * @returns {String} The URL of the directory containing the WorldWind library, or null if that directory
             * cannot be determined.
             */
            worldwindlibLocation: function () {
                var scripts = document.getElementsByTagName("script"),
                    libraryName = "/worldwind.";

                for (var i = 0; i < scripts.length; i++) {
                    var index = scripts[i].src.indexOf(libraryName);
                    if (index >= 0) {
                        return scripts[i].src.substring(0, index) + "/";
                    }
                }

                return null;
            },

            /**
             * Returns the path component of a specified URL.
             * @param {String} url The URL from which to determine the path component.
             * @returns {String} The path component, or the empty string if the specified URL is null, undefined
             * or empty.
             */
            urlPath: function (url) {
                if (!url)
                    return "";

                var urlParts = url.split("/"),
                    newPath = "";

                for (var i = 0, len = urlParts.length; i < len; i++) {
                    var part = urlParts[i];

                    if (!part || part.length === 0
                        || part.indexOf(":") != -1
                        || part === "."
                        || part === ".."
                        || part === "null"
                        || part === "undefined") {
                        continue;
                    }

                    if (newPath.length !== 0) {
                        newPath = newPath + "/";
                    }

                    newPath = newPath + part;
                }

                return newPath;
            },

            /**
             * Sets each element of an array to a specified value. This function is intentionally generic, and works
             * with any data structure with a length property whose elements may be referenced using array index syntax.
             * @param array The array to fill.
             * @param {*} value The value to assign to each array element.
             */
            fillArray: function (array, value) {
                if (!array) {
                    return;
                }

                for (var i = 0, len = array.length; i < len; i++) {
                    array[i] = value;
                }
            },

            /**
             * Multiplies each element of an array by a specified value and assigns each element to the result. This
             * function is intentionally generic, and works with any data structure with a length property whose
             * elements may be referenced using array index syntax.
             * @param array The array to fill.
             * @param {*} value The value to multiply by each array element.
             */
            multiplyArray: function (array, value) {
                if (!array) {
                    return;
                }

                for (var i = 0, len = array.length; i < len; i++) {
                    array[i] *= value;
                }
            },

            // Used to form unique function names for JSONP callback functions.
            jsonpCounter: 0,

            /**
             * Request a resource using JSONP.
             * @param {String} url The url to receive the request.
             * @param {String} parameterName The JSONP callback function key required by the server. Typically
             * "jsonp" or "callback".
             * @param {Function} callback The function to invoke when the request succeeds. The function receives
             * one argument, the JSON payload of the JSONP request.
             */
            jsonp: function (url, parameterName, callback) {

                // Generate a unique function name for the JSONP callback.
                var functionName = "gov_nasa_worldwind_jsonp_" + WWUtil.jsonpCounter++;

                // Define a JSONP callback function. Assign it to global scope the browser can find it.
                window[functionName] = function (jsonData) {
                    // Remove the JSONP callback from global scope.
                    delete window[functionName];

                    // Call the client's callback function.
                    callback(jsonData);
                };

                // Append the callback query parameter to the URL.
                var jsonpUrl = url + (url.indexOf('?') === -1 ? '?' : '&');
                jsonpUrl += parameterName + "=" + functionName;

                // Create a script element for the browser to invoke.
                var script = document.createElement('script');
                script.async = true;
                script.src = jsonpUrl;

                // Prepare to add the script to the document's head.
                var head = document.getElementsByTagName('head')[0];

                // Set up to remove the script element once it's invoked.
                var cleanup = function () {
                    script.onload = undefined;
                    script.onerror = undefined;
                    head.removeChild(script);
                };

                script.onload = cleanup;
                script.onerror = cleanup;

                // Add the script element to the document, causing the browser to invoke it.
                head.appendChild(script);
            },

            arrayEquals: function (array1, array2) {
                return (array1.length == array2.length) && array1.every(function (element, index) {
                        return element === array2[index] || element.equals && element.equals(array2[index]);
                    });
            },

            /**
             * It transforms given item to the boolean. It respects that 0, "0" and "false" are percieved as false
             * on top of the standard Boolean function.
             * @param item {String} Item to transform
             * @returns {boolean} Value transformed to the boolean.
             */
            transformToBoolean: function (item) {
                if (item == 0 || item == "0" || item == "false") {
                    return false;
                } else {
                    return Boolean(item);
                }
            },

            /**
             * It clones original object into the new one. It is necessary to retain the options information valid
             * for all nodes.
             * @param original Object to clone
             * @returns {Object} Cloned object
             */
            clone: function (original) {
                var clone = {};
                var i, keys = Object.keys(original);

                for (i = 0; i < keys.length; i++) {
                    // copy each property into the clone
                    clone[keys[i]] = original[keys[i]];
                }

                return clone;
            },

            /**
             * It returns unique GUID.
             * @returns {string} String representing unique identifier in the application.
             */
            guid: function () {
                function s4() {
                    return Math.floor((1 + Math.random()) * 0x10000)
                        .toString(16)
                        .substring(1);
                }

                return s4() + s4() + '-' + s4() + '-' + s4() + '-' +
                    s4() + '-' + s4() + s4() + s4();
            },

            /**
             * Transforms item to date. It accepts ISO-8601 format.
             * @param item {String} To transform.
             * @returns {Date} Date extracted from the current information.
             */
            date: function(item) {
                return new Date(item);
            },

            /**
             * Determines whether subjectString begins with the characters of searchString.
             * @param {String} subjectString The string to analyse.
             * @param {String} searchString The characters to be searched for at the start of subjectString.
             * @param {Number} position The position in subjectString at which to begin searching for searchString; defaults to 0.
             * @return {Boolean} true if the given characters are found at the beginning of the string; otherwise, false.
             */
            startsWith: function(subjectString, searchString, position) {
                position = position || 0;
                return subjectString.substr(position, searchString.length) === searchString;
            },

            /**
             * Determines whether subjectString ends with the characters of searchString.
             * @param {String} subjectString The string to analyse.
             * @param {String} searchString The characters to be searched for at the end of subjectString.
             * @param {Number} length Optional. If provided overwrites the considered length of the string to search in. If omitted, the default value is the length of the string.
             * @return {Boolean} true if the given characters are found at the end of the string; otherwise, false.
             */
            endsWith: function(subjectString, searchString, length) {
                if (typeof length !== 'number' || !isFinite(length) || Math.floor(length) !== length || length > subjectString.length) {
                    length = subjectString.length;
                }
                length -= searchString.length;
                var lastIndex = subjectString.lastIndexOf(searchString, length);
                return lastIndex !== -1 && lastIndex === length;
            }
        };

        return WWUtil;
    });
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports GeoTiffReader
 */
define('formats/geotiff/GeoTiffReader',[
        '../../error/AbstractError',
        '../../error/ArgumentError',
        './GeoTiffConstants',
        './GeoTiffKeyEntry',
        './GeoTiffMetadata',
        './GeoTiffUtil',
        '../../geom/Location',
        '../../geom/Sector',
        '../../util/Logger',
        '../../util/proj4-src',
        './TiffConstants',
        './TiffIFDEntry',
        '../../util/WWUtil'
    ],
    function (AbstractError,
              ArgumentError,
              GeoTiffConstants,
              GeoTiffKeyEntry,
              GeoTiffMetadata,
              GeoTiffUtil,
              Location,
              Sector,
              Logger,
              Proj4,
              TiffConstants,
              TiffIFDEntry,
              WWUtil) {
        "use strict";

        /**
         * Constructs a geotiff reader object for a specified geotiff URL.
         * Call [readAsImage]{@link GeoTiffReader#readAsImage} to retrieve the image as a canvas or
         * [readAsData]{@link GeoTiffReader#readAsData} to retrieve the elevations as an array of elevation values.
         * @alias GeoTiffReader
         * @constructor
         * @classdesc Parses a geotiff and creates an image or an elevation array representing its contents.
         * @param {String} ArrayBuffer The ArrayBuffer of the GeoTiff
         * @throws {ArgumentError} If the specified URL is null or undefined.
         */
        var GeoTiffReader = function (arrayBuffer) {
            if (!arrayBuffer) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "GeoTiffReader", "constructor", "missingArrayBuffer"));
            }

            // Documented in defineProperties below.
            this._isLittleEndian = false;

            // Documented in defineProperties below.
            this._imageFileDirectories = [];

            // Documented in defineProperties below.
            this._geoTiffData = new DataView(arrayBuffer);

            // Documented in defineProperties below.
            this._metadata = new GeoTiffMetadata();

            this.parse();
        };

        Object.defineProperties(GeoTiffReader.prototype, {

            /**
             *Indicates whether the geotiff byte order is little endian..
             * @memberof GeoTiffReader.prototype
             * @type {Boolean}
             * @readonly
             */
            isLittleEndian: {
                get: function () {
                    return this._isLittleEndian;
                }
            },

            /**
             * An array containing all the image file directories of the geotiff file.
             * @memberof GeoTiffReader.prototype
             * @type {TiffIFDEntry[]}
             * @readonly
             */
            imageFileDirectories: {
                get: function () {
                    return this._imageFileDirectories;
                }
            },

            /**
             * The buffer descriptor of the geotiff file's content.
             * @memberof GeoTiffReader.prototype
             * @type {ArrayBuffer}
             * @readonly
             */
            geoTiffData: {
                get: function () {
                    return this._geoTiffData;
                }
            },

            /**
             * An objct containing all tiff and geotiff metadata of the geotiff file.
             * @memberof GeoTiffReader.prototype
             * @type {GeoTiffMetadata}
             * @readonly
             */
            metadata: {
                get: function () {
                    return this._metadata;
                }
            }
        });

        /**
         * Attempts to retrieve the GeoTiff data from the provided URL, parse the data and return a GeoTiffReader
         * using the provided parserCompletionCallback.
         *
         * @param url the URL source for the GeoTiff
         * @param parserCompletionCallback a callback wher
         */
        GeoTiffReader.retrieveFromUrl = function (url, parserCompletionCallback) {
            if (!url) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "GeoTiffReader", "retrieveFromUrl",
                        "missingUrl"));
            }

            if (!parserCompletionCallback) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "GeoTiffReader", "retrieveFromUrl",
                        "The specified callback is null or undefined."));
            }

            var xhr = new XMLHttpRequest();

            xhr.open("GET", url, true);
            xhr.responseType = 'arraybuffer';
            xhr.onreadystatechange = (function () {
                if (xhr.readyState === 4) {
                    if (xhr.status === 200) {
                        var arrayBuffer = xhr.response;
                        if (arrayBuffer) {
                            parserCompletionCallback(new GeoTiffReader(arrayBuffer), xhr);
                        }
                    }
                    else {
                        Logger.log(Logger.LEVEL_WARNING, "GeoTiff retrieval failed (" + xhr.statusText + "): " + url);
                        parserCompletionCallback(null, xhr);
                    }
                }
            }).bind(this);

            xhr.onerror = function () {
                Logger.log(Logger.LEVEL_WARNING, "GeoTiff retrieval failed: " + url);
                parserCompletionCallback(null, xhr);
            };

            xhr.ontimeout = function () {
                Logger.log(Logger.LEVEL_WARNING, "GeoTiff retrieval timed out: " + url);
                parserCompletionCallback(null, xhr);
            };

            xhr.send(null);
        };

        // Parse geotiff file. Internal use only
        GeoTiffReader.prototype.parse = function () {

            // check if it's been parsed before
            if (this._imageFileDirectories.length) {
                return;
            }

            this.initEndiannessFromFile();

            if (!this.isTiffFileType()) {
                throw new AbstractError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "GeoTiffReader", "parse", "invalidTiffFileType"));
            }

            var firstIFDOffset = GeoTiffUtil.getBytes(this.geoTiffData, 4, 4, this.isLittleEndian);

            this.parseImageFileDirectory(firstIFDOffset);
            this.getMetadataFromImageFileDirectory();
            this.parseGeoKeys();
            this.setBBox();
        };

        // Get byte order of the geotiff file. Internal use only.
        GeoTiffReader.prototype.initEndiannessFromFile = function () {
            var byteOrderValue = GeoTiffUtil.getBytes(this.geoTiffData, 0, 2, this.isLittleEndian);
            if (byteOrderValue === 0x4949) {
                this._isLittleEndian = true;
            }
            else if (byteOrderValue === 0x4D4D) {
                this._isLittleEndian = false;
            }
            else {
                throw new AbstractError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "GeoTiffReader", "initEndiannessFromFile", "invalidByteOrderValue"));
            }
        };

        /**
         * Creates an RGB canvas from the GeoTiff image data.
         *
         * @return {Canvas}
         *
         */
        GeoTiffReader.prototype.getImage = function () {
            return this.createImage();
        };

        /**
         * Creates a typed array based on the contents of the GeoTiff.
         *
         * @return {TypedArray}
         */
        GeoTiffReader.prototype.getImageData = function () {
            return this.createTypedElevationArray();
        };

        /**
         * Indicates whether this geotiff is a tiff file type.
         *
         * @return {Boolean} True if this geotiff file is a tiff file type.
         */
        GeoTiffReader.prototype.isTiffFileType = function () {
            var fileTypeValue = GeoTiffUtil.getBytes(this.geoTiffData, 2, 2, this.isLittleEndian);
            if (fileTypeValue === 42) {
                return true;
            }
            else {
                return false;
            }
        };

        /**
         * Indicates whether this geotiff is a geotiff file type.
         *
         * @return {Boolean} True if this geotiff file is a geotiff file type.
         */
        GeoTiffReader.prototype.isGeoTiff = function () {
            if (this.getIFDByTag(GeoTiffConstants.Tag.GEO_KEY_DIRECTORY)) {
                return true;
            }
            else {
                return false;
            }
        };

        // Generate a canvas image. Internal use only.
        GeoTiffReader.prototype.createImage = function () {
            var bitsPerSample = this.metadata.bitsPerSample;
            var samplesPerPixel = this.metadata.samplesPerPixel;
            var photometricInterpretation = this.metadata.photometricInterpretation;
            var imageLength = this.metadata.imageLength;
            var imageWidth = this.metadata.imageWidth;

            if (this.metadata.colorMap) {
                var colorMapValues = this.metadata.colorMap;
                var colorMapSampleSize = Math.pow(2, bitsPerSample[0]);
            }

            var canvas = document.createElement('canvas');
            canvas.width = imageWidth;
            canvas.height = imageLength;
            var ctx = canvas.getContext("2d");

            if (this.metadata.stripOffsets) {
                var strips = this.parseStrips(false);
                if (this.metadata.rowsPerStrip) {
                    var rowsPerStrip = this.metadata.rowsPerStrip;
                } else {
                    var rowsPerStrip = imageLength;
                }
                var numOfStrips = strips.length;
                var numRowsInPreviousStrip = 0;
                var numRowsInStrip = rowsPerStrip;
                var imageLengthModRowsPerStrip = imageLength % rowsPerStrip;
                var rowsInLastStrip = (imageLengthModRowsPerStrip === 0) ? rowsPerStrip :
                    imageLengthModRowsPerStrip;

                for (var i = 0; i < numOfStrips; i++) {
                    if ((i + 1) === numOfStrips) {
                        numRowsInStrip = rowsInLastStrip;
                    }

                    var numOfPixels = strips[i].length;
                    var yPadding = numRowsInPreviousStrip * i;

                    for (var y = 0, j = 0; y < numRowsInStrip, j < numOfPixels; y++) {
                        for (var x = 0; x < imageWidth; x++, j++) {
                            var pixelSamples = strips[i][j];

                            ctx.fillStyle = this.getFillStyle(
                                pixelSamples,
                                photometricInterpretation,
                                bitsPerSample,
                                samplesPerPixel,
                                colorMapValues,
                                colorMapSampleSize
                            );
                            ctx.fillRect(x, yPadding + y, 1, 1);
                        }
                    }
                    numRowsInPreviousStrip = rowsPerStrip;
                }
            }
            else if (this.metadata.tileOffsets) {
                var tiles = this.parseTiles(false);
                var tileWidth = this.metadata.tileWidth;
                var tileLength = this.metadata.tileLength;
                var tilesAcross = Math.ceil(imageWidth / tileWidth);

                for (var y = 0; y < imageLength; y++) {
                    for (var x = 0; x < imageWidth; x++) {
                        var tileAcross = Math.floor(x / tileWidth);
                        var tileDown = Math.floor(y / tileLength);
                        var tileIndex = tileDown * tilesAcross + tileAcross;
                        var xInTile = x % tileWidth;
                        var yInTile = y % tileLength;
                        var sampleIndex = yInTile * tileWidth + xInTile;
                        var pixelSamples = tiles[tileIndex][sampleIndex];
                        ctx.fillStyle = this.getFillStyle(
                            pixelSamples,
                            photometricInterpretation,
                            bitsPerSample,
                            samplesPerPixel,
                            colorMapValues,
                            colorMapSampleSize
                        );
                        ctx.fillRect(x, y, 1, 1);
                    }
                }
            }

            this._geoTiffData = null;

            return canvas;
        };

        // Get pixel fill style. Internal use only.
        GeoTiffReader.prototype.getFillStyle = function (pixelSamples, photometricInterpretation, bitsPerSample,
                                                         samplesPerPixel, colorMapValues, colorMapSampleSize) {
            var red = 0.0;
            var green = 0.0;
            var blue = 0.0;
            var opacity = 1.0;

            if (this.metadata.noData && pixelSamples[0] == this.metadata.noData) {
                opacity = 0.0;
            }

            switch (photometricInterpretation) {
                case TiffConstants.PhotometricInterpretation.WHITE_IS_ZERO:
                    var invertValue = Math.pow(2, bitsPerSample) - 1;
                    pixelSamples[0] = invertValue - pixelSamples[0];
                case TiffConstants.PhotometricInterpretation.BLACK_IS_ZERO:
                    red = green = blue = GeoTiffUtil.clampColorSample(
                        pixelSamples[0],
                        bitsPerSample[0]);
                    break;
                case TiffConstants.PhotometricInterpretation.RGB:
                    red = GeoTiffUtil.clampColorSample(pixelSamples[0], bitsPerSample[0]);
                    green = GeoTiffUtil.clampColorSample(pixelSamples[1], bitsPerSample[1]);
                    blue = GeoTiffUtil.clampColorSample(pixelSamples[2], bitsPerSample[2]);

                    if (samplesPerPixel === 4 && this.metadata.extraSamples[0] === 2) {
                        var maxValue = Math.pow(2, bitsPerSample[3]);
                        opacity = pixelSamples[3] / maxValue;
                    }
                    break;
                case TiffConstants.PhotometricInterpretation.RGB_PALETTE:
                    if (colorMapValues) {
                        var colorMapIndex = pixelSamples[0];

                        red = GeoTiffUtil.clampColorSample(
                            colorMapValues[colorMapIndex],
                            16);
                        green = GeoTiffUtil.clampColorSample(
                            colorMapValues[colorMapSampleSize + colorMapIndex],
                            16);
                        blue = GeoTiffUtil.clampColorSample(
                            colorMapValues[(2 * colorMapSampleSize) + colorMapIndex],
                            16);
                    }
                    break;
                case TiffConstants.PhotometricInterpretation.TRANSPARENCY_MASK:
                    //todo
                    Logger.log(Logger.LEVEL_WARNING, "Photometric interpretation not yet implemented: " +
                        "TRANSPARENCY_MASK");
                    break;
                case TiffConstants.PhotometricInterpretation.CMYK:
                    //todo
                    Logger.log(Logger.LEVEL_WARNING, "Photometric interpretation not yet implemented: CMYK");
                    break;
                case TiffConstants.PhotometricInterpretation.Y_Cb_Cr:
                    //todo
                    Logger.log(Logger.LEVEL_WARNING, "Photometric interpretation not yet implemented: Y_Cb_Cr");
                    break;
                case TiffConstants.PhotometricInterpretation.CIE_LAB:
                    //todo
                    Logger.log(Logger.LEVEL_WARNING, "Photometric interpretation not yet implemented: CIE_LAB");
                    break;
                default:
                    //todo
                    Logger.log("Unknown photometric interpretation: " + photometricInterpretation);
                    break;
            }

            return GeoTiffUtil.getRGBAFillValue(red, green, blue, opacity);
        }

        GeoTiffReader.prototype.createTypedElevationArray = function () {
            var elevationArray = [], typedElevationArray;
            var bitsPerSample = this.metadata.bitsPerSample[0];

            if (this.metadata.stripOffsets) {
                var strips = this.parseStrips(true);

                for (var i = 0; i < strips.length; i++) {
                    elevationArray = elevationArray.concat(strips[i]);
                }
            }
            else if (this.metadata.tileOffsets) {
                var tiles = this.parseTiles(true);
                var imageWidth = this.metadata.imageWidth;
                var imageLength = this.metadata.imageLength;
                var tileWidth = this.metadata.tileWidth;
                var tileLength = this.metadata.tileLength;
                var tilesAcross = Math.ceil(imageWidth / tileWidth);

                for (var y = 0; y < imageLength; y++) {
                    for (var x = 0; x < imageWidth; x++) {
                        var tileAcross = Math.floor(x / tileWidth);
                        var tileDown = Math.floor(y / tileLength);
                        var tileIndex = tileDown * tilesAcross + tileAcross;
                        var xInTile = x % tileWidth;
                        var yInTile = y % tileLength;
                        var sampleIndex = yInTile * tileWidth + xInTile;
                        var pixelSamples = tiles[tileIndex][sampleIndex];
                        elevationArray.push(pixelSamples);
                    }
                }
            }

            if (this.metadata.sampleFormat) {
                var sampleFormat = this.metadata.sampleFormat[0];
            }
            else {
                var sampleFormat = TiffConstants.SampleFormat.UNSIGNED;
            }

            switch (bitsPerSample) {
                case 8:
                    if (sampleFormat === TiffConstants.SampleFormat.SIGNED) {
                        typedElevationArray = new Int8Array(elevationArray);
                    }
                    else {
                        typedElevationArray = new Uint8Array(elevationArray);
                    }
                    break
                case 16:
                    if (sampleFormat === TiffConstants.SampleFormat.SIGNED) {
                        typedElevationArray = new Int16Array(elevationArray);
                    }
                    else {
                        typedElevationArray = new Uint16Array(elevationArray);
                    }
                    break;
                case 32:
                    if (sampleFormat === TiffConstants.SampleFormat.SIGNED) {
                        typedElevationArray = new Int32Array(elevationArray);
                    }
                    else if (sampleFormat === TiffConstants.SampleFormat.IEEE_FLOAT) {
                        typedElevationArray = new Float32Array(elevationArray);
                    }
                    else {
                        typedElevationArray = new Uint32Array(elevationArray);
                    }
                    break;
                case 64:
                    typedElevationArray = new Float64Array(elevationArray);
                    break;
                default:
                    break;
            }

            return typedElevationArray;
        }

        // Parse geotiff strips. Internal use only
        GeoTiffReader.prototype.parseStrips = function (returnElevation) {
            var samplesPerPixel = this.metadata.samplesPerPixel;
            var bitsPerSample = this.metadata.bitsPerSample;
            var stripOffsets = this.metadata.stripOffsets;
            var stripByteCounts = this.metadata.stripByteCounts;
            var compression = this.metadata.compression;
            if (this.metadata.sampleFormat) {
                var sampleFormat = this.metadata.sampleFormat;
            }
            else {
                var sampleFormat = TiffConstants.SampleFormat.UNSIGNED;
            }

            var bitsPerPixel = samplesPerPixel * bitsPerSample[0];
            var bytesPerPixel = bitsPerPixel / 8;

            var strips = [];
            // Loop through strips
            for (var i = 0; i < stripOffsets.length; i++) {
                var stripOffset = stripOffsets[i];
                var stripByteCount = stripByteCounts[i];

                strips[i] = this.parseBlock(returnElevation, compression, bytesPerPixel, stripByteCount, stripOffset,
                    bitsPerSample, sampleFormat);
            }

            return strips;
        }

        // Parse geotiff block. A block may be a strip or a tile. Internal use only.
        GeoTiffReader.prototype.parseBlock = function (returnElevation, compression, bytesPerPixel, blockByteCount,
                                                       blockOffset, bitsPerSample, sampleFormat) {
            var block = [];
            switch (compression) {
                case TiffConstants.Compression.UNCOMPRESSED:
                    // Loop through pixels.
                    for (var byteOffset = 0, increment = bytesPerPixel;
                         byteOffset < blockByteCount; byteOffset += increment) {
                        // Loop through samples (sub-pixels).
                        for (var m = 0, pixel = []; m < bitsPerSample.length; m++) {
                            var bytesPerSample = bitsPerSample[m] / 8;
                            var sampleOffset = m * bytesPerSample;

                            pixel.push(GeoTiffUtil.getSampleBytes(
                                this.geoTiffData,
                                blockOffset + byteOffset + sampleOffset,
                                bytesPerSample,
                                sampleFormat[m],
                                this.isLittleEndian));
                        }
                        if (returnElevation) {
                            block.push(pixel[0]);
                        }
                        else {
                            block.push(pixel);
                        }
                    }
                    break;
                case TiffConstants.Compression.CCITT_1D:
                    //todo
                    Logger.log(Logger.LEVEL_WARNING, "Compression type not yet implemented: CCITT_1D");
                    break;
                case TiffConstants.Compression.GROUP_3_FAX:
                    //todo
                    Logger.log(Logger.LEVEL_WARNING, "Compression type not yet implemented: GROUP_3_FAX");
                    break;
                case TiffConstants.Compression.GROUP_4_FAX:
                    //todo
                    Logger.log(Logger.LEVEL_WARNING, "Compression type not yet implemented: GROUP_4_FAX");
                    break;
                case TiffConstants.Compression.LZW:
                    //todo
                    Logger.log(Logger.LEVEL_WARNING, "Compression type not yet implemented: LZW");
                    break;
                case TiffConstants.Compression.JPEG:
                    //todo
                    Logger.log(Logger.LEVEL_WARNING, "Compression type not yet implemented: JPEG");
                    break;
                case TiffConstants.Compression.PACK_BITS:
                    if (this.metadata.tileOffsets) {
                        var tileWidth = this.metadata.tileWidth;
                        var tileLength = this.metadata.tileWidth;
                        var arrayBuffer = new ArrayBuffer(tileWidth * tileLength * bytesPerPixel);
                    }
                    else {
                        var rowsPerStrip = this.metadata.rowsPerStrip;
                        var imageWidth = this.metadata.imageWidth;
                        var arrayBuffer = new ArrayBuffer(rowsPerStrip * imageWidth * bytesPerPixel);
                    }

                    var uncompressedDataView = new DataView(arrayBuffer);

                    var newBlock = true;
                    var pixel = [];
                    var blockLength = 0;
                    var numOfIterations = 0;
                    var uncompressedOffset = 0;


                    for (var byteOffset = 0; byteOffset < blockByteCount; byteOffset += 1) {

                        if (newBlock) {
                            blockLength = 1;
                            numOfIterations = 1;
                            newBlock = false;

                            var nextSourceByte = this.geoTiffData.getInt8(blockOffset + byteOffset,
                                this.isLittleEndian);

                            if (nextSourceByte >= 0 && nextSourceByte <= 127) {
                                blockLength = nextSourceByte + 1;
                            }
                            else if (nextSourceByte >= -127 && nextSourceByte <= -1) {
                                numOfIterations = -nextSourceByte + 1;
                            }
                            else {
                                newBlock = true;
                            }
                        }
                        else {
                            var currentByte = GeoTiffUtil.getBytes(
                                this.geoTiffData,
                                blockOffset + byteOffset,
                                1,
                                this.isLittleEndian);

                            for (var currentIteration = 0; currentIteration < numOfIterations; currentIteration++) {
                                uncompressedDataView.setInt8(uncompressedOffset, currentByte);
                                uncompressedOffset++;
                            }

                            blockLength--;

                            if (blockLength === 0) {
                                newBlock = true;
                            }
                        }
                    }

                    for (var byteOffset = 0, increment = bytesPerPixel;
                         byteOffset < arrayBuffer.byteLength; byteOffset += increment) {
                        // Loop through samples (sub-pixels).
                        for (var m = 0, pixel = []; m < bitsPerSample.length; m++) {
                            var bytesPerSample = bitsPerSample[m] / 8;
                            var sampleOffset = m * bytesPerSample;

                            pixel.push(GeoTiffUtil.getSampleBytes(
                                uncompressedDataView,
                                byteOffset + sampleOffset,
                                bytesPerSample,
                                sampleFormat[m],
                                this.isLittleEndian));
                        }
                        if (returnElevation) {
                            block.push(pixel[0]);
                        }
                        else {
                            block.push(pixel);
                        }
                    }
                    break;
            }

            return block;
        }

        // Parse geotiff tiles. Internal use only
        GeoTiffReader.prototype.parseTiles = function (returnElevation) {
            var samplesPerPixel = this.metadata.samplesPerPixel;
            var bitsPerSample = this.metadata.bitsPerSample;
            var compression = this.metadata.compression;
            if (this.metadata.sampleFormat) {
                var sampleFormat = this.metadata.sampleFormat;
            }
            else {
                var sampleFormat = new Array(samplesPerPixel);
                WWUtil.fillArray(sampleFormat, TiffConstants.SampleFormat.UNSIGNED);
            }
            var bitsPerPixel = samplesPerPixel * bitsPerSample[0];
            var bytesPerPixel = bitsPerPixel / 8;
            var tileWidth = this.metadata.tileWidth;
            var tileLength = this.metadata.tileLength;
            var tileOffsets = this.metadata.tileOffsets;
            var tileByteCounts = this.metadata.tileByteCounts;
            var imageLength = this.metadata.imageLength;
            var imageWidth = this.metadata.imageWidth;

            var tilesAcross = Math.ceil(imageWidth / tileWidth);
            var tilesDown = Math.ceil(imageLength / tileLength);

            var tiles = [];

            for (var i = 0; i < tilesDown; i++) {
                for (var j = 0; j < tilesAcross; j++) {
                    var index = tilesAcross * i + j;
                    var tileOffset = tileOffsets[index];
                    var tileByteCount = tileByteCounts[index];
                    tiles[index] = this.parseBlock(returnElevation, compression, bytesPerPixel, tileByteCount,
                        tileOffset, bitsPerSample, sampleFormat);
                }
            }

            return tiles;
        }

        // Translate a pixel/line coordinates to projection coordinate. Internal use only.
        GeoTiffReader.prototype.geoTiffImageToPCS = function (xValue, yValue) {
            if (xValue === null || xValue === undefined) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "GeoTiffReader", "geoTiffImageToPCS", "missingXValue"));
            }
            if (yValue === null || yValue === undefined) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "GeoTiffReader", "geoTiffImageToPCS", "missingYValue"));
            }

            var res = [xValue, yValue];

            var tiePointValues = this.metadata.modelTiepoint;
            var modelPixelScaleValues = this.metadata.modelPixelScale;
            var modelTransformationValues = this.metadata.modelTransformation;

            var tiePointCount = tiePointValues ? tiePointValues.length : 0;
            var modelPixelScaleCount = modelPixelScaleValues ? modelPixelScaleValues.length : 0;
            var modelTransformationCount = modelTransformationValues ? modelTransformationValues.length : 0;

            if (tiePointCount > 6 && modelPixelScaleCount === 0) {
                //todo
            }
            else if (modelTransformationCount === 16) {
                var x_in = xValue;
                var y_in = yValue;

                xValue = x_in * modelTransformationValues[0] + y_in * modelTransformationValues[1] +
                    modelTransformationValues[3];
                yValue = x_in * modelTransformationValues[4] + y_in * modelTransformationValues[5] +
                    modelTransformationValues[7];

                res = [xValue, yValue];
            }
            else if (modelPixelScaleCount < 3 || tiePointCount < 6) {
                res = [xValue, yValue];
            }
            else {
                xValue = (xValue - tiePointValues[0]) * modelPixelScaleValues[0] + tiePointValues[3];
                yValue = (yValue - tiePointValues[1]) * (-1 * modelPixelScaleValues[1]) + tiePointValues[4];

                res = [xValue, yValue];
            }

            Proj4.defs([
                [
                    'EPSG:26771',
                    '+proj=tmerc +lat_0=36.66666666666666 +lon_0=-88.33333333333333 +k=0.9999749999999999 +' +
                    'x_0=152400.3048006096 +y_0=0 +ellps=clrk66 +datum=NAD27 +to_meter=0.3048006096012192 +no_defs '
                ],
                [
                    'EPSG:32633',
                    '+proj=utm +zone=33 +datum=WGS84 +units=m +no_defs'
                ]
            ]);

            if (this.metadata.projectedCSType) {
                res = Proj4('EPSG:' + this.metadata.projectedCSType, 'EPSG:4326', res);
            }

            return new Location(res[1], res[0]);
        };

        /**
         * Set the bounding box of the geotiff file. Internal use only.
         */
        GeoTiffReader.prototype.setBBox = function () {
            var upperLeft = this.geoTiffImageToPCS(0, 0);
            var upperRight = this.geoTiffImageToPCS(this.metadata.imageWidth, 0);
            var lowerLeft = this.geoTiffImageToPCS(0, this.metadata.imageLength);
            var lowerRight = this.geoTiffImageToPCS(
                this.metadata.imageWidth, this.metadata.imageLength);

            this.metadata.bbox = new Sector(
                lowerLeft.latitude,
                upperLeft.latitude,
                upperLeft.longitude,
                upperRight.longitude
            );
        }

        // Get metadata from image file directory. Internal use only.
        GeoTiffReader.prototype.getMetadataFromImageFileDirectory = function () {
            for (var i = 0; i < this.imageFileDirectories[0].length; i++) {

                switch (this.imageFileDirectories[0][i].tag) {
                    case TiffConstants.Tag.BITS_PER_SAMPLE:
                        this.metadata.bitsPerSample = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    case TiffConstants.Tag.COLOR_MAP:
                        this.metadata.colorMap = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    case TiffConstants.Tag.COMPRESSION:
                        this.metadata.compression = this.imageFileDirectories[0][i].getIFDEntryValue()[0];
                        break;
                    case TiffConstants.Tag.EXTRA_SAMPLES:
                        this.metadata.extraSamples = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    case TiffConstants.Tag.IMAGE_DESCRIPTION:
                        this.metadata.imageDescription = this.imageFileDirectories[0][i].getIFDEntryValue()[0];
                        break;
                    case TiffConstants.Tag.IMAGE_LENGTH:
                        this.metadata.imageLength = this.imageFileDirectories[0][i].getIFDEntryValue()[0];
                        break;
                    case TiffConstants.Tag.IMAGE_WIDTH:
                        this.metadata.imageWidth = this.imageFileDirectories[0][i].getIFDEntryValue()[0];
                        break;
                    case TiffConstants.Tag.MAX_SAMPLE_VALUE:
                        this.metadata.maxSampleValue = this.imageFileDirectories[0][i].getIFDEntryValue()[0];
                        break;
                    case TiffConstants.Tag.MIN_SAMPLE_VALUE:
                        this.metadata.minSampleValue = this.imageFileDirectories[0][i].getIFDEntryValue()[0];
                        break;
                    case TiffConstants.Tag.ORIENTATION:
                        this.metadata.orientation = this.imageFileDirectories[0][i].getIFDEntryValue()[0];
                        break;
                    case TiffConstants.Tag.PHOTOMETRIC_INTERPRETATION:
                        this.metadata.photometricInterpretation = this.imageFileDirectories[0][i].getIFDEntryValue()[0];
                        break;
                    case TiffConstants.Tag.PLANAR_CONFIGURATION:
                        this.metadata.planarConfiguration = this.imageFileDirectories[0][i].getIFDEntryValue()[0];
                        break;
                    case TiffConstants.Tag.ROWS_PER_STRIP:
                        this.metadata.rowsPerStrip = this.imageFileDirectories[0][i].getIFDEntryValue()[0];
                        break;
                    case TiffConstants.Tag.RESOLUTION_UNIT:
                        this.metadata.resolutionUnit = this.imageFileDirectories[0][i].getIFDEntryValue()[0];
                        break;
                    case TiffConstants.Tag.SAMPLES_PER_PIXEL:
                        this.metadata.samplesPerPixel = this.imageFileDirectories[0][i].getIFDEntryValue()[0];
                        break;
                    case TiffConstants.Tag.SAMPLE_FORMAT:
                        this.metadata.sampleFormat = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    case TiffConstants.Tag.SOFTWARE:
                        this.metadata.software = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    case TiffConstants.Tag.STRIP_BYTE_COUNTS:
                        this.metadata.stripByteCounts = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    case TiffConstants.Tag.STRIP_OFFSETS:
                        this.metadata.stripOffsets = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    case TiffConstants.Tag.TILE_BYTE_COUNTS:
                        this.metadata.tileByteCounts = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    case TiffConstants.Tag.TILE_OFFSETS:
                        this.metadata.tileOffsets = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    case TiffConstants.Tag.TILE_LENGTH:
                        this.metadata.tileLength = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    case TiffConstants.Tag.TILE_WIDTH:
                        this.metadata.tileWidth = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    case TiffConstants.Tag.X_RESOLUTION:
                        this.metadata.xResolution = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    case TiffConstants.Tag.Y_RESOLUTION:
                        this.metadata.tileWidth = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;

                    //geotiff
                    case GeoTiffConstants.Tag.GEO_ASCII_PARAMS:
                        this.metadata.geoAsciiParams = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    case GeoTiffConstants.Tag.GEO_DOUBLE_PARAMS:
                        this.metadata.geoDubleParams = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    case GeoTiffConstants.Tag.GEO_KEY_DIRECTORY:
                        this.metadata.geoKeyDirectory = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    case GeoTiffConstants.Tag.MODEL_PIXEL_SCALE:
                        this.metadata.modelPixelScale = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    case GeoTiffConstants.Tag.MODEL_TIEPOINT:
                        this.metadata.modelTiepoint = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    case GeoTiffConstants.Tag.GDAL_METADATA:
                        this.metadata.metaData = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    case GeoTiffConstants.Tag.GDAL_NODATA:
                        this.metadata.noData = this.imageFileDirectories[0][i].getIFDEntryValue();
                        break;
                    default:
                        Logger.log(Logger.LEVEL_WARNING, "Ignored GeoTiff tag: " + this.imageFileDirectories[0][i].tag);
                }
            }
        }

        // Parse GeoKeys. Internal use only.
        GeoTiffReader.prototype.parseGeoKeys = function () {
            if (!this.isGeoTiff()) {
                throw new AbstractError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "GeoTiffReader", "parse", "invalidGeoTiffFile"));
            }

            var geoKeyDirectory = this.metadata.geoKeyDirectory;
            if (geoKeyDirectory) {
                var keyDirectoryVersion = geoKeyDirectory[0];
                var keyRevision = geoKeyDirectory[1];
                var minorRevision = geoKeyDirectory[2];
                var numberOfKeys = geoKeyDirectory[3];

                for (var i = 0; i < numberOfKeys; i++) {
                    var keyId = geoKeyDirectory[4 + i * 4];
                    var tiffTagLocation = geoKeyDirectory[5 + i * 4];
                    var count = geoKeyDirectory[6 + i * 4];
                    var valueOffset = geoKeyDirectory[7 + i * 4];

                    switch (keyId) {
                        case GeoTiffConstants.Key.GTModelTypeGeoKey:
                            this.metadata.gtModelTypeGeoKey =
                                new GeoTiffKeyEntry(keyId, tiffTagLocation, count, valueOffset).getGeoKeyValue(
                                    this.metadata.geoDoubleParams,
                                    this.metadata.geoAsciiParams);
                            break;
                        case GeoTiffConstants.Key.GTRasterTypeGeoKey:
                            this.metadata.gtRasterTypeGeoKey =
                                new GeoTiffKeyEntry(keyId, tiffTagLocation, count, valueOffset).getGeoKeyValue(
                                    this.metadata.geoDoubleParams,
                                    this.metadata.geoAsciiParams);
                            break;
                        case GeoTiffConstants.Key.GTCitationGeoKey:
                            this.metadata.gtCitationGeoKey =
                                new GeoTiffKeyEntry(keyId, tiffTagLocation, count, valueOffset).getGeoKeyValue(
                                    this.metadata.geoDoubleParams,
                                    this.metadata.geoAsciiParams);
                            break;
                        case GeoTiffConstants.Key.GeographicTypeGeoKey:
                            this.metadata.geographicTypeGeoKey =
                                new GeoTiffKeyEntry(keyId, tiffTagLocation, count, valueOffset).getGeoKeyValue(
                                    this.metadata.geoDoubleParams,
                                    this.metadata.geoAsciiParams);
                            break;
                        case GeoTiffConstants.Key.GeogCitationGeoKey:
                            this.metadata.geogCitationGeoKey =
                                new GeoTiffKeyEntry(keyId, tiffTagLocation, count, valueOffset).getGeoKeyValue(
                                    this.metadata.geoDoubleParams,
                                    this.metadata.geoAsciiParams);
                            break;
                        case GeoTiffConstants.Key.GeogAngularUnitsGeoKey:
                            this.metadata.geogAngularUnitsGeoKey =
                                new GeoTiffKeyEntry(keyId, tiffTagLocation, count, valueOffset).getGeoKeyValue(
                                    this.metadata.geoDoubleParams,
                                    this.metadata.geoAsciiParams);
                            break;
                        case GeoTiffConstants.Key.GeogAngularUnitSizeGeoKey:
                            this.metadata.geogAngularUnitSizeGeoKey =
                                new GeoTiffKeyEntry(keyId, tiffTagLocation, count, valueOffset).getGeoKeyValue(
                                    this.metadata.geoDoubleParams,
                                    this.metadata.geoAsciiParams);
                            break;
                        case GeoTiffConstants.Key.GeogSemiMajorAxisGeoKey:
                            this.metadata.geogSemiMajorAxisGeoKey =
                                new GeoTiffKeyEntry(keyId, tiffTagLocation, count, valueOffset).getGeoKeyValue(
                                    this.metadata.geoDoubleParams,
                                    this.metadata.geoAsciiParams);
                            break;
                        case GeoTiffConstants.Key.GeogInvFlatteningGeoKey:
                            this.metadata.geogInvFlatteningGeoKey =
                                new GeoTiffKeyEntry(keyId, tiffTagLocation, count, valueOffset).getGeoKeyValue(
                                    this.metadata.geoDoubleParams,
                                    this.metadata.geoAsciiParams);
                            break;
                        case GeoTiffConstants.Key.ProjectedCSTypeGeoKey:
                            this.metadata.projectedCSType =
                                new GeoTiffKeyEntry(keyId, tiffTagLocation, count, valueOffset).getGeoKeyValue(
                                    this.metadata.geoDoubleParams,
                                    this.metadata.geoAsciiParams);
                            break;
                        case GeoTiffConstants.Key.ProjLinearUnitsGeoKey:
                            this.metadata.projLinearUnits =
                                new GeoTiffKeyEntry(keyId, tiffTagLocation, count, valueOffset).getGeoKeyValue(
                                    this.metadata.geoDoubleParams,
                                    this.metadata.geoAsciiParams);
                            break;
                        default:
                            Logger.log(Logger.LEVEL_WARNING, "Ignored GeoTiff key: " + keyId);
                            break;

                    }
                }
            }
            else {
                throw new AbstractError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "GeoTiffReader", "parseGeoKeys",
                        "missingGeoKeyDirectoryTag"));
            }
        };

        // Parse image file directory. Internal use only.
        GeoTiffReader.prototype.parseImageFileDirectory = function (offset) {
            if (!offset) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "GeoTiffReader", "parseImageFileDirectory",
                        "missingOffset"));
            }

            var noOfDirectoryEntries = GeoTiffUtil.getBytes(this.geoTiffData, offset, 2, this.isLittleEndian);

            var directoryEntries = [];

            for (var i = offset + 2, directoryEntryCounter = 0; directoryEntryCounter < noOfDirectoryEntries;
                 i += 12, directoryEntryCounter++) {
                var tag = GeoTiffUtil.getBytes(this.geoTiffData, i, 2, this.isLittleEndian);
                var type = GeoTiffUtil.getBytes(this.geoTiffData, i + 2, 2, this.isLittleEndian);
                var count = GeoTiffUtil.getBytes(this.geoTiffData, i + 4, 4, this.isLittleEndian);
                var valueOffset = GeoTiffUtil.getBytes(this.geoTiffData, i + 8, 4, this.isLittleEndian);

                directoryEntries.push(new TiffIFDEntry(
                    tag,
                    type,
                    count,
                    valueOffset,
                    this.geoTiffData,
                    this.isLittleEndian));
            }

            this._imageFileDirectories.push(directoryEntries);

            var nextIFDOffset = GeoTiffUtil.getBytes(this.geoTiffData, i, 4, this.isLittleEndian);

            if (nextIFDOffset === 0) {
                return;
            }
            else {
                this.parseImageFileDirectory(nextIFDOffset);
            }
        };

        // Get image file directory by tag value. Internal use only.
        GeoTiffReader.prototype.getIFDByTag = function (tag) {
            if (!tag) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "GeoTiffReader", "getIFDByTag", "missingTag"));
            }

            for (var i = 0; i < this.imageFileDirectories[0].length; i++) {
                if (this.imageFileDirectories[0][i].tag === tag) {
                    return this.imageFileDirectories[0][i];
                }
            }

            return null;
        };

        return GeoTiffReader;
    }
);
/*
 * Copyright 2003-2006, 2009, 2017, 2020 United States Government, as represented
 * by the Administrator of the National Aeronautics and Space Administration.
 * All rights reserved.
 *
 * The NASAWorldWind/WebWorldWind platform is licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License
 * at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed
 * under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * NASAWorldWind/WebWorldWind also contains the following 3rd party Open Source
 * software:
 *
 *    ES6-Promise – under MIT License
 *    libtess.js – SGI Free Software License B
 *    Proj4 – under MIT License
 *    JSZip – under MIT License
 *
 * A complete listing of 3rd Party software notices and licenses included in
 * WebWorldWind can be found in the WebWorldWind 3rd-party notices and licenses
 * PDF found in code  directory.
 */
/**
 * @exports Level
 */
define('util/Level',[
        '../geom/Angle',
        '../error/ArgumentError',
        '../geom/Location',
        '../util/Logger'
    ],
    function (Angle,
              ArgumentError,
              Location,
              Logger) {
        "use strict";

        /**
         * Constructs a Level within a [LevelSet]{@link LevelSet}. Applications typically do not interact with this
         * class.
         * @alias Level
         * @constructor
         * @classdesc Represents a level in a tile pyramid.
         * @throws {ArgumentError} If either the specified tile delta or parent level set is null or undefined.
         */
        var Level = function (levelNumber, tileDelta, parent) {
            if (!tileDelta) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Level", "constructor",
                        "The specified tile delta is null or undefined"));
            }

            if (!parent) {
                throw new ArgumentError(
                    Logger.logMessage(Logger.LEVEL_SEVERE, "Level", "constructor",
                        "The specified parent level set is null or undefined"));
            }

            /**
             * The level's ordinal in its parent level set.
             * @type {Number}
             */
            this.levelNumber = levelNumber;

            /**
             * The geographic size of tiles within this level.
             * @type {Location}
             */
            this.tileDelta = tileDelta;

            /**
             * The level set that this level is a member of.
             * @type {LevelSet}
             */
            this.parent = parent;

            /**
             * The size of pixels or elevation cells within this level, in radians per pixel or per cell.
             * @type {Number}
             */
            this.texelSize = (tileDelta.latitude * Angle.DEGREES_TO_RADIANS) / parent.tileHeight;

            /**
             * The width in pixels or cells of the resource associated with tiles within this level.
             * @type {Number}
             */
            this.tileWidth = parent.tileWidth;

            /**
             * The height in pixels or cells of the resource associated with tiles within this level.
             * @type {Number}
             */
            this.tileHeight = parent.tileHeight;

            /**
             * The sector spanned by this level.
             * @type {Sector}
             */
            this.sector = parent.sector;
        };

        /**
         * Indicates whether this level is the lowest resolution level (level 0) within its parent's level set.
         * @returns {Boolean} true If this tile is the lowest resolution in the parent level set,
         * otherwise false.
         */
        Level.prototype.isFirstLevel = function () {
            return this.parent.firstLevel() == this;
        };

        /**
         * Indicates whether this level is the highest resolution level within its parent's level set.
         * @returns {Boolean} true If this tile is the highest resolution in the parent level set,
         * otherwise false.
         */
        Level.prototype.isLastLevel = function () {
            return this.parent.lastLevel() == this;
        };

        /**
         * Returns the level whose ordinal occurs immediately before this level's ordinal in the parent level set, or
         * null if this is the fist level.
         * @returns {Level} The previous level, or null if this is the first level.
         */
        Level.prototype.previousLevel = function () {
            return this.parent.level(this.levelNumber - 1);
        };

        /**
  