"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.core = void 0;
const utils_js_1 = require("./utils.js");
const constants_js_1 = require("./constants.js");
const resolver_js_1 = require("./resolver.js");
const config_js_1 = require("./config.js");
exports.core = {
    _url: "",
    _options: {},
    _config: config_js_1.default,
    _catchers: new Map(),
    _resolvers: [],
    _deferred: [],
    _middlewares: [],
    _addons: [],
    addon(addon) {
        return { ...this, _addons: [...this._addons, addon], ...addon.wretch };
    },
    errorType(errorType) {
        return {
            ...this,
            _config: {
                ...this._config,
                errorType
            }
        };
    },
    polyfills(polyfills, replace = false) {
        return {
            ...this,
            _config: {
                ...this._config,
                polyfills: replace ? polyfills : (0, utils_js_1.mix)(this._config.polyfills, polyfills)
            }
        };
    },
    url(_url, replace = false) {
        if (replace)
            return { ...this, _url };
        const split = this._url.split("?");
        return {
            ...this,
            _url: split.length > 1 ?
                split[0] + _url + "?" + split[1] :
                this._url + _url
        };
    },
    options(options, replace = false) {
        return { ...this, _options: replace ? options : (0, utils_js_1.mix)(this._options, options) };
    },
    headers(headerValues) {
        const headers = !headerValues ? {} :
            Array.isArray(headerValues) ? Object.fromEntries(headerValues) :
                "entries" in headerValues ? Object.fromEntries(headerValues.entries()) :
                    headerValues;
        return { ...this, _options: (0, utils_js_1.mix)(this._options, { headers }) };
    },
    accept(headerValue) {
        return this.headers({ Accept: headerValue });
    },
    content(headerValue) {
        return this.headers({ [constants_js_1.CONTENT_TYPE_HEADER]: headerValue });
    },
    auth(headerValue) {
        return this.headers({ Authorization: headerValue });
    },
    catcher(errorId, catcher) {
        const newMap = new Map(this._catchers);
        newMap.set(errorId, catcher);
        return { ...this, _catchers: newMap };
    },
    catcherFallback(catcher) {
        return this.catcher(constants_js_1.CATCHER_FALLBACK, catcher);
    },
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    resolve(resolver, clear = false) {
        return { ...this, _resolvers: clear ? [resolver] : [...this._resolvers, resolver] };
    },
    defer(callback, clear = false) {
        return {
            ...this,
            _deferred: clear ? [callback] : [...this._deferred, callback]
        };
    },
    middlewares(middlewares, clear = false) {
        return {
            ...this,
            _middlewares: clear ? middlewares : [...this._middlewares, ...middlewares]
        };
    },
    fetch(method = this._options.method, url = "", body = null) {
        let base = this.url(url).options({ method });
        // "Jsonify" the body if it is an object and if it is likely that the content type targets json.
        const contentType = (0, utils_js_1.extractContentType)(base._options.headers);
        const formDataClass = this._config.polyfill("FormData", false);
        const jsonify = typeof body === "object" &&
            !(formDataClass && body instanceof formDataClass) &&
            (!base._options.headers || !contentType || (0, utils_js_1.isLikelyJsonMime)(contentType));
        base =
            !body ? base :
                jsonify ? base.json(body, contentType) :
                    base.body(body);
        return (0, resolver_js_1.resolver)(base
            ._deferred
            .reduce((acc, curr) => curr(acc, acc._url, acc._options), base));
    },
    get(url = "") {
        return this.fetch("GET", url);
    },
    delete(url = "") {
        return this.fetch("DELETE", url);
    },
    put(body, url = "") {
        return this.fetch("PUT", url, body);
    },
    post(body, url = "") {
        return this.fetch("POST", url, body);
    },
    patch(body, url = "") {
        return this.fetch("PATCH", url, body);
    },
    head(url = "") {
        return this.fetch("HEAD", url);
    },
    opts(url = "") {
        return this.fetch("OPTIONS", url);
    },
    body(contents) {
        return { ...this, _options: { ...this._options, body: contents } };
    },
    json(jsObject, contentType) {
        const currentContentType = (0, utils_js_1.extractContentType)(this._options.headers);
        return this.content(contentType ||
            (0, utils_js_1.isLikelyJsonMime)(currentContentType) && currentContentType ||
            constants_js_1.JSON_MIME).body(JSON.stringify(jsObject));
    }
};
//# sourceMappingURL=core.js.map