"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolver = exports.WretchError = void 0;
const middleware_js_1 = require("./middleware.js");
const utils_js_1 = require("./utils.js");
const constants_js_1 = require("./constants.js");
/**
 * This class inheriting from Error is thrown when the fetch response is not "ok".
 * It extends Error and adds status, text and body fields.
 */
class WretchError extends Error {
}
exports.WretchError = WretchError;
const resolver = (wretch) => {
    const sharedState = Object.create(null);
    wretch = wretch._addons.reduce((w, addon) => addon.beforeRequest &&
        addon.beforeRequest(w, wretch._options, sharedState)
        || w, wretch);
    const { _url: url, _options: opts, _config: config, _catchers: _catchers, _resolvers: resolvers, _middlewares: middlewares, _addons: addons } = wretch;
    const catchers = new Map(_catchers);
    const finalOptions = (0, utils_js_1.mix)(config.options, opts);
    // The generated fetch request
    let finalUrl = url;
    const _fetchReq = (0, middleware_js_1.middlewareHelper)(middlewares)((url, options) => {
        finalUrl = url;
        return config.polyfill("fetch")(url, options);
    })(url, finalOptions);
    // Throws on an http error
    const referenceError = new Error();
    const throwingPromise = _fetchReq
        .catch(error => {
        throw { [constants_js_1.FETCH_ERROR]: error };
    })
        .then(response => {
        var _a;
        if (!response.ok) {
            const err = new WretchError();
            // Enhance the error object
            err["cause"] = referenceError;
            err.stack = err.stack + "\nCAUSE: " + referenceError.stack;
            err.response = response;
            err.status = response.status;
            err.url = finalUrl;
            if (response.type === "opaque") {
                throw err;
            }
            const jsonErrorType = config.errorType === "json" || ((_a = response.headers.get("Content-Type")) === null || _a === void 0 ? void 0 : _a.split(";")[0]) === "application/json";
            const bodyPromise = !config.errorType ? Promise.resolve(response.body) :
                jsonErrorType ? response.text() :
                    response[config.errorType]();
            return bodyPromise.then((body) => {
                err.message = typeof body === "string" ? body : response.statusText;
                if (body) {
                    if (jsonErrorType && typeof body === "string") {
                        err.text = body;
                        err.json = JSON.parse(body);
                    }
                    else {
                        err[config.errorType] = body;
                    }
                }
                throw err;
            });
        }
        return response;
    });
    // Wraps the Promise in order to dispatch the error to a matching catcher
    const catchersWrapper = (promise) => {
        return promise.catch(err => {
            const fetchErrorFlag = Object.prototype.hasOwnProperty.call(err, constants_js_1.FETCH_ERROR);
            const error = fetchErrorFlag ? err[constants_js_1.FETCH_ERROR] : err;
            const catcher = ((error === null || error === void 0 ? void 0 : error.status) && catchers.get(error.status)) ||
                catchers.get(error === null || error === void 0 ? void 0 : error.name) || (fetchErrorFlag && catchers.has(constants_js_1.FETCH_ERROR) && catchers.get(constants_js_1.FETCH_ERROR));
            if (catcher)
                return catcher(error, wretch);
            const catcherFallback = catchers.get(constants_js_1.CATCHER_FALLBACK);
            if (catcherFallback)
                return catcherFallback(error, wretch);
            throw error;
        });
    };
    const bodyParser = funName => cb => funName ?
        // If a callback is provided, then callback with the body result otherwise return the parsed body itself.
        catchersWrapper(throwingPromise.then(_ => _ && _[funName]()).then(_ => cb ? cb(_) : _)) :
        // No body parsing method - return the response
        catchersWrapper(throwingPromise.then(_ => cb ? cb(_) : _));
    const responseChain = {
        _wretchReq: wretch,
        _fetchReq,
        _sharedState: sharedState,
        res: bodyParser(null),
        json: bodyParser("json"),
        blob: bodyParser("blob"),
        formData: bodyParser("formData"),
        arrayBuffer: bodyParser("arrayBuffer"),
        text: bodyParser("text"),
        error(errorId, cb) {
            catchers.set(errorId, cb);
            return this;
        },
        badRequest(cb) { return this.error(400, cb); },
        unauthorized(cb) { return this.error(401, cb); },
        forbidden(cb) { return this.error(403, cb); },
        notFound(cb) { return this.error(404, cb); },
        timeout(cb) { return this.error(408, cb); },
        internalError(cb) { return this.error(500, cb); },
        fetchError(cb) { return this.error(constants_js_1.FETCH_ERROR, cb); },
    };
    const enhancedResponseChain = addons.reduce((chain, addon) => ({
        ...chain,
        ...(typeof addon.resolver === "function" ? addon.resolver(chain) : addon.resolver)
    }), responseChain);
    return resolvers.reduce((chain, r) => r(chain, wretch), enhancedResponseChain);
};
exports.resolver = resolver;
//# sourceMappingURL=resolver.js.map