/**
AbstractInput - base class for all editable inputs.
It defines interface to be implemented by any input type.
To create your own input you can inherit from this class.

@class abstractinput
**/
(function ($) {
    "use strict";

    //types
    $.fn.editabletypes = {};

    var AbstractInput = function () { };

    AbstractInput.prototype = {
       /**
        Initializes input

        @method init()
        **/
       init: function(type, options, defaults) {
           this.type = type;
           this.options = $.extend({}, defaults, options);
       },

       /*
       this method called before render to init $tpl that is inserted in DOM
       */
       prerender: function() {
           this.$tpl = $(this.options.tpl); //whole tpl as jquery object
           this.$input = this.$tpl;         //control itself, can be changed in render method
           this.$clear = null;              //clear button
           this.error = null;               //error message, if input cannot be rendered
       },

       /**
        Renders input from tpl. Can return jQuery deferred object.
        Can be overwritten in child objects

        @method render()
       **/
       render: function() {

       },

       /**
        Sets element's html by value.

        @method value2html(value, element)
        @param {mixed} value
        @param {DOMElement} element
       **/
       value2html: function(value, element) {
           $(element)[this.options.escape ? 'text' : 'html']($.trim(value));
       },

       /**
        Converts element's html to value

        @method html2value(html)
        @param {string} html
        @returns {mixed}
       **/
       html2value: function(html) {
           return $('<div>').html(html).text();
       },

       /**
        Converts value to string (for internal compare). For submitting to server used value2submit().

        @method value2str(value)
        @param {mixed} value
        @returns {string}
       **/
       value2str: function(value) {
           return String(value);
       },

       /**
        Converts string received from server into value. Usually from `data-value` attribute.

        @method str2value(str)
        @param {string} str
        @returns {mixed}
       **/
       str2value: function(str) {
           return str;
       },

       /**
        Converts value for submitting to server. Result can be string or object.

        @method value2submit(value)
        @param {mixed} value
        @returns {mixed}
       **/
       value2submit: function(value) {
           return value;
       },

       /**
        Sets value of input.

        @method value2input(value)
        @param {mixed} value
       **/
       value2input: function(value) {
           this.$input.val(value);
       },

       /**
        Returns value of input. Value can be object (e.g. datepicker)

        @method input2value()
       **/
       input2value: function() {
           return this.$input.val();
       },

       /**
        Activates input. For text it sets focus.

        @method activate()
       **/
       activate: function() {
           if(this.$input.is(':visible')) {
               this.$input.focus();
           }
       },

       /**
        Creates input.

        @method clear()
       **/
       clear: function() {
           this.$input.val(null);
       },

       /**
        method to escape html.
       **/
       escape: function(str) {
           return $('<div>').text(str).html();
       },

       /**
        attach handler to automatically submit form when value changed (useful when buttons not shown)
       **/
       autosubmit: function() {

       },

       /**
       Additional actions when destroying element
       **/
       destroy: function() {
       },

       // -------- helper functions --------
       setClass: function() {
           if(this.options.inputclass) {
               this.$input.addClass(this.options.inputclass);
           }
       },

       setAttr: function(attr) {
           if (this.options[attr] !== undefined && this.options[attr] !== null) {
               this.$input.attr(attr, this.options[attr]);
           }
       },

       option: function(key, value) {
            this.options[key] = value;
       }

    };

    AbstractInput.defaults = {
        /**
        HTML template of input. Normally you should not change it.

        @property tpl
        @type string
        @default ''
        **/
        tpl: '',
        /**
        CSS class automatically applied to input

        @property inputclass
        @type string
        @default null
        **/
        inputclass: null,

        /**
        Size of the input - implemented for bootstrap5 only
        Allowed values 'sm', 'md', 'lg'

        @property inputsize
        @type string
        @default null
        **/
        inputsize: null,

        /**
        If `true` - html will be escaped in content of element via $.text() method.
        If `false` - html will not be escaped, $.html() used.
        When you use own `display` function, this option obviosly has no effect.

        @property escape
        @type boolean
        @since 1.5.0
        @default true
        **/
        escape: true,

        //scope for external methods (e.g. source defined as function)
        //for internal use only
        scope: null,

        //need to re-declare showbuttons here to get it's value from common config (passed only options existing in defaults)
        showbuttons: true
    };

    $.extend($.fn.editabletypes, {abstractinput: AbstractInput});

}(window.jQuery));
