/* -*- Mode: JavaScript; coding: utf-8; tab-width: 3; indent-tabs-mode: tab; c-basic-offset: 3 -*-
 *******************************************************************************
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * Copyright create3000, Scheffelstraße 31a, Leipzig, Germany 2011.
 *
 * All rights reserved. Holger Seelig <holger.seelig@yahoo.de>.
 *
 * The copyright notice above does not evidence any actual of intended
 * publication of such source code, and is an unpublished work by create3000.
 * This material contains CONFIDENTIAL INFORMATION that is the property of
 * create3000.
 *
 * No permission is granted to copy, distribute, or create derivative works from
 * the contents of this software, in whole or in part, without the prior written
 * permission of create3000.
 *
 * NON-MILITARY USE ONLY
 *
 * All create3000 software are effectively free software with a non-military use
 * restriction. It is free. Well commented source is provided. You may reuse the
 * source in any way you please with the exception anything that uses it must be
 * marked to indicate is contains 'non-military use only' components.
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * Copyright 2015, 2016 Holger Seelig <holger.seelig@yahoo.de>.
 *
 * This file is part of the X_ITE Project.
 *
 * X_ITE is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License version 3 only, as published by the
 * Free Software Foundation.
 *
 * X_ITE is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License version 3 for more
 * details (a copy is included in the LICENSE file that accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version 3
 * along with X_ITE.  If not, see <http://www.gnu.org/licenses/gpl.html> for a
 * copy of the GPLv3 License.
 *
 * For Silvio, Joy and Adi.
 *
 ******************************************************************************/


define ([
	"x_ite/Basic/X3DField",
	"x_ite/Bits/X3DConstants",
	"x_ite/InputOutput/Generator",
	"x_ite/Fields/SFNodeCache",
],
function (X3DField,
          X3DConstants,
          Generator,
          SFNodeCache)
{
"use strict";

	const handler =
	{
		get: function (target, key)
		{
			try
			{
				const value = target [key];

				if (value !== undefined)
					return value;

				const
					field      = target .getValue () .getField (key),
					accessType = field .getAccessType ();

				// Specification conform would be: accessType & X3DConstants .outputOnly.
				// But we allow read access to plain fields, too.
				if (accessType === X3DConstants .inputOnly)
					return undefined;

				return field .valueOf ();
			}
			catch (error)
			{
				return undefined;
			}
 		},
		set: function (target, key, value)
		{
			if (key in target)
			{
				target [key] = value;
				return true;
			}

			try
			{
				const
					field      = target .getValue () .getField (key),
					accessType = field .getAccessType ();

				if (accessType !== X3DConstants .outputOnly)
					field .setValue (value);

	 			return true;
			}
			catch (error)
			{
				console .error (target, key, error);
				return false;
			}
		},
		has: function (target, key)
		{
			try
			{
				return Boolean (target .getValue () .getField (key));
			}
			catch (error)
			{
				return key in target;
			}
		},
	};

	function SFNode (value)
	{
		// Node need to test for X3DBaseNode, because there is a special version of SFNode in Script.

		const proxy = new Proxy (this, handler);

		this ._target = this;
		this ._proxy  = proxy;

		if (value)
		{
			value .addParent (proxy);

			X3DField .call (this, value);
		}
		else
		{
			X3DField .call (this, null);
		}

		return proxy;
	}

	SFNode .prototype = Object .assign (Object .create (X3DField .prototype),
	{
		constructor: SFNode,
		_cloneCount: 0,
		clone: function ()
		{
			const target = this ._target;

			return new SFNode (target .getValue ());
		},
		copy: function (instance)
		{
			const
				target = this ._target,
				value  = target .getValue ();

			if (value)
				return new SFNode (value .copy (instance));

			return new SFNode ();
		},
		getTypeName: function ()
		{
			return "SFNode";
		},
		getType: function ()
		{
			return X3DConstants .SFNode;
		},
		equals: function (node)
		{
			const target = this ._target;

			if (node)
				return target .getValue () === node .getValue ();

			return target .getValue () === null;
		},
		isDefaultValue: function ()
		{
			const target = this ._target;

			return target .getValue () === null;
		},
		set: function (value)
		{
			const
				target  = this ._target,
				current = target .getValue ();

			if (current)
			{
				current .removeCloneCount (target ._cloneCount);
				current .removeParent (target ._proxy);
			}

			// No need to test for X3DBaseNode, because there is a special version of SFNode in Script.

			if (value)
			{
				value .addParent (target ._proxy);
				value .addCloneCount (target ._cloneCount);

				X3DField .prototype .set .call (target, value);
			}
			else
			{
				X3DField .prototype .set .call (target, null);
			}
		},
		getNodeTypeName: function ()
		{
			const
				target = this ._target,
			 	value  = target .getValue ();

			if (value)
				return value .getTypeName ();

			throw new Error ("SFNode.getNodeTypeName: node is null.");
		},
		getNodeName: function ()
		{
			const
				target = this ._target,
			 	value  = target .getValue ();

			if (value)
				return value .getName ();

			throw new Error ("SFNode.getNodeName: node is null.");
		},
		getNodeType: function ()
		{
			const
				target = this ._target,
			 	value  = target .getValue ();

			if (value)
				return value .getType () .slice ();

			throw new Error ("SFNode.getNodeType: node is null.");
		},
		getFieldDefinitions: function ()
		{
			const
				target = this ._target,
			 	value  = target .getValue ();

			if (value)
				return value .getFieldDefinitions ();

			throw new Error ("SFNode.getFieldDefinitions: node is null.");
		},
		addFieldCallback: function (name, string, object)
		{
			const target = this ._target;

			switch (arguments .length)
			{
				case 2:
				{
					return X3DField .prototype .addFieldCallback .apply (target, arguments);
				}
				case 3:
				{
					const value = target .getValue ();

					if (value)
						return value .getField (name) .addFieldCallback (string, object);

					throw new Error ("SFNode.addFieldCallback: node is null.");
				}
			}
		},
		removeFieldCallback: function (name, string)
		{
			const target = this ._target;

			switch (arguments .length)
			{
				case 1:
				{
					return X3DField .prototype .removeFieldCallback .apply (target, arguments);
				}
				case 2:
				{
					const value = target .getValue ();

					if (value)
						return value .getField (name) .removeFieldCallback (string);

					throw new Error ("SFNode.removeFieldCallback: node is null.");
				}
			}
		},
		addCloneCount: function (count)
		{
			const
				target = this ._target,
				value  = target .getValue ();

			target ._cloneCount += count;

			if (value)
				value .addCloneCount (count);
		},
		removeCloneCount: function (count)
		{
			const
				target = this ._target,
				value  = target .getValue ();

			target ._cloneCount -= count;

			if (value)
				value .removeCloneCount (count);
		},
		valueOf: function ()
		{
			const
				target = this ._target,
			 	value  = target .getValue ();

			if (value)
				return SFNodeCache .get (value);

			return null;
		},
		toStream: function (stream)
		{
			const
				target = this ._target,
			 	value  = target .getValue ();

			if (value)
				value .toStream (stream);
			else
				stream .string += "NULL";
		},
		toVRMLStream: function (stream)
		{
			const
				target = this ._target,
			 	value  = target .getValue ();

			if (value)
				value .toVRMLStream (stream);
			else
				stream .string += "NULL";
		},
		toXMLString: function ()
		{
			const
				target    = this ._target,
				stream    = { string: "" },
				generator = Generator .Get (stream),
				value     = target .getValue ();

			generator .PushExecutionContext (value .getExecutionContext ());

			target .toXMLStream (stream);

			generator .PopExecutionContext ();

			return stream .string;
		},
		toXMLStream: function (stream)
		{
			const
				target = this ._target,
			 	value  = target .getValue ();

			if (value)
				value .toXMLStream (stream);
			else
				stream .string += "<!-- NULL -->";
		},
		dispose: function ()
		{
			const target = this ._target;

			target .set (null);

			X3DField .prototype .dispose .call (target);
		},
	});

	return SFNode;
});
