import { InterpreterStatus } from './types';
export { InterpreterStatus };
const INIT_EVENT = { type: 'xstate.init' };
const ASSIGN_ACTION = 'xstate.assign';
function toArray(item) {
    return item === undefined ? [] : [].concat(item);
}
export function assign(assignment) {
    return {
        type: ASSIGN_ACTION,
        assignment
    };
}
function toActionObject(
// tslint:disable-next-line:ban-types
action, actionMap) {
    action =
        typeof action === 'string' && actionMap && actionMap[action]
            ? actionMap[action]
            : action;
    return typeof action === 'string'
        ? {
            type: action
        }
        : typeof action === 'function'
            ? {
                type: action.name,
                exec: action
            }
            : action;
}
const IS_PRODUCTION = process.env.NODE_ENV === 'production';
function createMatcher(value) {
    return (stateValue) => value === stateValue;
}
function toEventObject(event) {
    return (typeof event === 'string' ? { type: event } : event);
}
function createUnchangedState(value, context) {
    return {
        value,
        context,
        actions: [],
        changed: false,
        matches: createMatcher(value)
    };
}
export function createMachine(fsmConfig, options = {}) {
    const machine = {
        config: fsmConfig,
        _options: options,
        initialState: {
            value: fsmConfig.initial,
            actions: toArray(fsmConfig.states[fsmConfig.initial].entry).map((action) => toActionObject(action, options.actions)),
            context: fsmConfig.context,
            matches: createMatcher(fsmConfig.initial)
        },
        transition: (state, event) => {
            const { value, context } = typeof state === 'string'
                ? { value: state, context: fsmConfig.context }
                : state;
            const eventObject = toEventObject(event);
            const stateConfig = fsmConfig.states[value];
            if (!IS_PRODUCTION) {
                if (!stateConfig) {
                    throw new Error(`State '${value}' not found on machine${fsmConfig.id ? ` '${fsmConfig.id}'` : ''}.`);
                }
            }
            if (stateConfig.on) {
                const transitions = toArray(stateConfig.on[eventObject.type]);
                for (const transition of transitions) {
                    if (transition === undefined) {
                        return createUnchangedState(value, context);
                    }
                    const { target = value, actions = [], cond = () => true } = typeof transition === 'string'
                        ? { target: transition }
                        : transition;
                    let nextContext = context;
                    if (cond(context, eventObject)) {
                        const nextStateConfig = fsmConfig.states[target];
                        let assigned = false;
                        const allActions = []
                            .concat(stateConfig.exit, actions, nextStateConfig.entry)
                            .filter((a) => a)
                            .map((action) => toActionObject(action, machine._options.actions))
                            .filter((action) => {
                            if (action.type === ASSIGN_ACTION) {
                                assigned = true;
                                let tmpContext = Object.assign({}, nextContext);
                                if (typeof action.assignment === 'function') {
                                    tmpContext = action.assignment(nextContext, eventObject);
                                }
                                else {
                                    Object.keys(action.assignment).forEach((key) => {
                                        tmpContext[key] =
                                            typeof action.assignment[key] === 'function'
                                                ? action.assignment[key](nextContext, eventObject)
                                                : action.assignment[key];
                                    });
                                }
                                nextContext = tmpContext;
                                return false;
                            }
                            return true;
                        });
                        return {
                            value: target,
                            context: nextContext,
                            actions: allActions,
                            changed: target !== value || allActions.length > 0 || assigned,
                            matches: createMatcher(target)
                        };
                    }
                }
            }
            // No transitions match
            return createUnchangedState(value, context);
        }
    };
    return machine;
}
const executeStateActions = (state, event) => state.actions.forEach(({ exec }) => exec && exec(state.context, event));
export function interpret(machine) {
    let state = machine.initialState;
    let status = InterpreterStatus.NotStarted;
    const listeners = new Set();
    const service = {
        _machine: machine,
        send: (event) => {
            if (status !== InterpreterStatus.Running) {
                return;
            }
            state = machine.transition(state, event);
            executeStateActions(state, toEventObject(event));
            listeners.forEach((listener) => listener(state));
        },
        subscribe: (listener) => {
            listeners.add(listener);
            listener(state);
            return {
                unsubscribe: () => listeners.delete(listener)
            };
        },
        start: () => {
            status = InterpreterStatus.Running;
            executeStateActions(state, INIT_EVENT);
            return service;
        },
        stop: () => {
            status = InterpreterStatus.Stopped;
            listeners.clear();
            return service;
        },
        get state() {
            return state;
        },
        get status() {
            return status;
        }
    };
    return service;
}
