"use strict";
var core_1 = require('../core');
var DOMEventProducer = (function () {
    function DOMEventProducer(node, eventType, useCapture) {
        this.node = node;
        this.eventType = eventType;
        this.useCapture = useCapture;
        this.type = 'fromEvent';
    }
    DOMEventProducer.prototype._start = function (out) {
        this.listener = function (e) { return out._n(e); };
        var _a = this, node = _a.node, eventType = _a.eventType, useCapture = _a.useCapture;
        node.addEventListener(eventType, this.listener, useCapture);
    };
    DOMEventProducer.prototype._stop = function () {
        var _a = this, node = _a.node, eventType = _a.eventType, listener = _a.listener, useCapture = _a.useCapture;
        node.removeEventListener(eventType, listener, useCapture);
        this.listener = null;
    };
    return DOMEventProducer;
}());
exports.DOMEventProducer = DOMEventProducer;
/**
 * Creates a stream based on DOM events of type `eventType` from the target
 * node.
 *
 * Marble diagram:
 *
 * ```text
 *   fromEvent(node, eventType)
 * ---ev--ev----ev---------------
 * ```
 *
 * Example:
 *
 * ```js
 * import fromEvent from 'xstream/extra/fromEvent'
 *
 * const stream = fromEvent(document.querySelector('.button'), 'click')
 *   .mapTo('Button clicked!')
 *
 * stream.addListener({
 *   next: i => console.log(i),
 *   error: err => console.error(err),
 *   complete: () => console.log('completed')
 * })
 * ```
 *
 * ```text
 * > 'Button clicked!'
 * > 'Button clicked!'
 * > 'Button clicked!'
 * ```
 *
 * @param {EventTarget} node The element we want to listen to.
 * @param {string} eventType The type of events we want to listen to.
 * @param {boolean} useCapture An optional boolean that indicates that events of
 * this type will be dispatched to the registered listener before being
 * dispatched to any EventTarget beneath it in the DOM tree. Defaults to false.
 * @return {Stream}
 */
function fromEvent(node, eventType, useCapture) {
    if (useCapture === void 0) { useCapture = false; }
    return new core_1.Stream(new DOMEventProducer(node, eventType, useCapture));
}
Object.defineProperty(exports, "__esModule", { value: true });
exports.default = fromEvent;
//# sourceMappingURL=fromEvent.js.map